# -*- coding: utf-8 -*-

# Credits:
# XRay generation code based on code at http://www.mobileread.com/forums/showthread.php?t=157770
# Plug-in structure based on APNX Generator
# Icon from http://openclipart.org/image/128px/svg_to_png/32149/HandXray.png

from __future__ import (unicode_literals, division, absolute_import, print_function)

__license__ = 'GPL 3'
__copyright__ = '2012, Matthew Wilson <matthew@mjwilson.demon.co.uk>'
__docformat__ = 'restructuredtext en'

import os, time, re
import cStringIO
from array import *
import html5lib
from lxml.cssselect import CSSSelector
from threading import Thread
from Queue import Queue

from PyQt4.Qt import Qt, QMenu, QFileDialog, QIcon, QPixmap, QMessageBox, QInputDialog

from calibre import sanitize_file_name
from calibre.gui2 import Dispatcher, warning_dialog
from calibre.gui2.actions import InterfaceAction
from calibre.library.save_to_disk import get_components
from calibre.library.save_to_disk import config
from calibre.ptempfile import PersistentTemporaryFile
from calibre.utils.filenames import shorten_components_to
from calibre.utils.ipc.job import BaseJob

class XRayAction(InterfaceAction):

    name = 'XRay'
    action_spec = (_('XRay'), None, None, None)
    
    def genesis(self):
        self.xray_mixin = XRayMixin(self.gui)
        # Read the icons and assign to our global for potential sharing with the configuration dialog
        # Assign our menu to this action and an icon
        self.qaction.setIcon(get_icons('images/plugin_xray_xray.png'))
        self.qaction.triggered.connect(self.generate_selected)
        self.xray_menu = QMenu()
        self.load_menu()
        
    def load_menu(self):
        self.xray_menu.clear()
        self.xray_menu.addAction(_('Generate from selected books...'), self.generate_selected)
        self.xray_menu.addAction(_('Generate from file...'), self.generate_file)
        self.qaction.setMenu(self.xray_menu)

    def generate_selected(self):
        self.xray_mixin.genesis()
        
        xraydir = unicode(QFileDialog.getExistingDirectory(self.gui, _('Directory to save XRay file'), self.gui.library_path, QFileDialog.ShowDirsOnly | QFileDialog.DontResolveSymlinks))
        if not xraydir:
            return
        
        self._generate_selected(xraydir)
        
    def _generate_selected(self, xraydir, ids=None, do_auto_convert=False):
        if not ids:
            ids = [self.gui.library_view.model().id(r) for r in self.gui.library_view.selectionModel().selectedRows()]
        
        _files, _auto_ids = self.gui.library_view.model().get_preferred_formats_from_ids(ids, ['mobi', 'azw', 'prc'], exclude_auto=do_auto_convert)
        if do_auto_convert:
            ok_ids = list(set(ids).difference(_auto_ids))
            ids = [i for i in ids if i in ok_ids]
        else:
            _auto_ids = []
            
        metadata = self.gui.library_view.model().metadata_for(ids)
        ids = iter(ids)
        imetadata = iter(metadata)

        bad, good = [], []
        for f in _files:
            mi = imetadata.next()
            id = ids.next()
            if f is None:
                bad.append(mi.title)
            else:
                good.append((f, mi))

        template = config().parse().template
        if not isinstance(template, unicode):
            template = template.decode('utf-8')

        for f, mi in good:
            components = get_components(template, mi, f)
            if not components:
                components = [sanitize_file_name(mi.title)]

            def remove_trailing_periods(x):
                ans = x
                while ans.endswith('.'):
                    ans = ans[:-1].strip()
                if not ans:
                    ans = 'x'
                return ans
            
            #components = list(map(remove_trailing_periods, components))
            #components = shorten_components_to(250, components)
            #components = list(map(sanitize_file_name, components))
            #filepath = os.path.join(xraydir, *components)

            #xrayname = os.path.splitext(filepath)[0] + '.XRay'
            self.xray_mixin.generate_xray(f, xraydir)

        if bad:
            bad = '\n'.join('%s'%(i,) for i in bad)
            d = warning_dialog(self.gui, _('No suitable formats'),
                    _('Could not generate an XRay for the following books, '
                'as no suitable formats were found. Convert the book(s) to '
                'MOBI first.'
                ), bad)
            d.exec_()
    
    def generate_file(self):
        self.xray_mixin.genesis()
        
        filename = unicode(QFileDialog.getOpenFileName(self.gui, _('MOBI file for generating XRay'), self.gui.library_path, 'MOBI files (*.mobi *.azw *.prc)'))
        if not filename:
            return
        xraydir = unicode(QFileDialog.getExistingDirectory(self.gui, _('Directory to save XRay file'), self.gui.library_path, QFileDialog.ShowDirsOnly | QFileDialog.DontResolveSymlinks))
        if not xraydir:
            return
        #xrayname = os.path.join(xraydir, os.path.splitext(os.path.basename(filename))[0] + '.XRay')
        
        self.xray_mixin.generate_xray(filename, xraydir)


class XRayJob(BaseJob):
    
    def __init__(self, callback, description, job_manager, filename, xraydir, asin, database, uniqid, shelfariUrl, wikiUrl):
        BaseJob.__init__(self, description)
        self.exception = None
        self.job_manager = job_manager
        self.args = (filename, xraydir, asin, database, uniqid, shelfariUrl, wikiUrl)
        self.callback = callback
        self.log_path = None
        self._log_file = cStringIO.StringIO()
        self._log_file.write(self.description.encode('utf-8') + '\n')

    @property
    def log_file(self):
        if self.log_path is not None:
            return open(self.log_path, 'rb')
        return cStringIO.StringIO(self._log_file.getvalue())

    def start_work(self):
        self.start_time = time.time()
        self.job_manager.changed_queue.put(self)

    def job_done(self):
        self.duration = time.time() - self.start_time
        self.percent = 1
        # Dump log onto disk
        lf = PersistentTemporaryFile('xray_generate_log')
        lf.write(self._log_file.getvalue())
        lf.close()
        self.log_path = lf.name
        self._log_file.close()
        self._log_file = None

        self.job_manager.changed_queue.put(self)

    def log_write(self, what):
        self._log_file.write(what)
        
        
class XRayGenerator(Thread):
    
    def __init__(self, job_manager):
        Thread.__init__(self)
        self.daemon = True
        self.jobs = Queue()
        self.job_manager = job_manager
        self._run = True
        self.xray_builder = XRayBuilder()
        
    def stop(self):
        self._run = False
        self.jobs.put(None)
        
    def run(self):
        while self._run:
            try:
                job = self.jobs.get()
            except:
                break
            if job is None or not self._run:
                break
            
            failed, exc = False, None
            job.start_work()
            if job.kill_on_start:
                self._abort_job(job)
                continue
            
            try:
                self._generate_xray(job)
            except Exception as e:
                if not self._run:
                    return
                import traceback
                failed = True
                exc = e
                job.log_write('\nXRAY generation failed...\n')
                job.log_write(traceback.format_exc())

            if not self._run:
                break

            job.failed = failed
            job.exception = exc
            job.job_done()
            try:
                job.callback(job)
            except:
                import traceback
                traceback.print_exc()

    def _abort_job(self, job):
        job.log_write('Aborted\n')
        job.failed = False
        job.killed = True
        job.job_done()

    def _generate_xray(self, job):

        filename, xraydir, asin, database, uniqid, shelfariUrl, wikiUrl = job.args
        if not filename or not xraydir or not asin or not database or not uniqid or not shelfariUrl:
            raise Exception(_('Nothing to do.'))
        if not os.path.exists(xraydir):
            os.makedirs(xraydir)

        data = XRayData()
        data.shelfari (shelfariUrl)
        data.wikipedia (wikiUrl)

        #QMessageBox.about(self.gui, 'mode', str(shelfari.mode))
        #QMessageBox.about(self.gui, 'chars.len', str(len(shelfari.chars)))
        #for c in shelfari.chars:
        #    QMessageBox.about(self.gui, 'char', str(c))

        self.xray_builder.write_xray(filename, xraydir, asin, database, uniqid, data)
        
    def generate_xray(self, callback, filename, xraydir, asin, database, uniqid, shelfariUrl, wikiUrl):

        description = _('Generating XRay for %s') % os.path.splitext(os.path.basename(xraydir))[0]
        job = XRayJob(callback, description, self.job_manager, filename, xraydir, asin, database, uniqid, shelfariUrl, wikiUrl)
        self.job_manager.add_job(job)
        self.jobs.put(job)


class XRayMixin(object):

    def __init__(self, gui):
        self.gui = gui
    
    def genesis(self):
        '''
        Genesis must always be called before using an XRayMixin object.
        Plugins are initalized before the GUI initalizes the job_manager.
        We cannot create the XRayGenerator during __init__. Instead call
        genesis before using generate_xray to ensure the XRayGenerator
        has been properly created with the job_manager.
        '''
        if not hasattr(self.gui, 'xray_generator'):
            self.gui.xray_generator = XRayGenerator(self.gui.job_manager)

    def generate_xray(self, filename, xraydir):
        from calibre.ebooks.metadata.meta import get_metadata
        from calibre.ebooks.metadata.mobi import MetadataUpdater
        from struct import unpack

        if not self.gui.xray_generator.is_alive():
            self.gui.xray_generator.start()

        file = open(filename, 'r+b')
        ext  = os.path.splitext(filename)[-1][1:].lower()
        mi = get_metadata(file, ext)

        mu = MetadataUpdater (file)
        db = mu.data[0:32]
        db = re.sub ( r'\u0000', '', db)
        uniqid, = unpack ('>I', mu.record0[32:36])
        try:
            asin = mu.original_exth_records[113]
        except KeyError:
            d = warning_dialog(self.gui, _('No ASIN'),
                    _('Could not find ASIN in book'), mi.title.decode('utf-8'))
            d.exec_()
            raise

        #QMessageBox.about(self.gui, 'database name', db)
        #QMessageBox.about(self.gui, 'unique id', str(uniqid))
        #QMessageBox.about(self.gui, 'ASIN', asin)

        shelfariUrl, ok = QInputDialog.getText(self.gui, 'Input Dialog', 'Enter Shelfari URL for ' + mi.title.decode('utf-8'))
        if ok:
            wikiUrl, wikiOk = QInputDialog.getText(self.gui, 'Input Dialog', 'Enter Wikipedia URL for ' + mi.title.decode('utf-8') + ' (optional)' )
            self.gui.xray_generator.generate_xray(Dispatcher(self.xray_generated), filename, xraydir, asin, db, uniqid, shelfariUrl, wikiUrl)
            self.gui.status_bar.show_message(_('Generating XRay for %s') % os.path.splitext(os.path.basename(xraydir))[0], 3000)
    
    def xray_generated(self, job):
        if job.failed:
            self.gui.job_exception(job, dialog_title=_('Failed to generate XRay'))
            return
        self.gui.status_bar.show_message(job.description + ' ' + _('finished'), 5000)

class XRayBuilder(object):
    def wikireplace(self, str):
        result = re.sub (r'\s?\([^)]+\)', '', str)
        result = re.sub (r'\s?["\'(]\\w+["\')]', '', result)
        result = re.sub ('[.,\u201c\u201d()\/\"?\':\[\]]', '', result)
        result = re.sub (r'\s', '_', result)
        return result

    def charreplace(self, str):
        result = re.sub (r'"', '', str)
        # Replace all non-ASCII characters. Can we be smarter and just escape them?
        result = re.sub (r'[^ -~]', '', result)
        return result

    def write_xray(self, mobi_file_path, xraydir, asin, database, uniqid, data):
        xray = '{"asin":"'+asin+'","guid":"' + database +':'+ re.sub ('L$', '', str(hex(uniqid)[2:]).upper()) + '","version":"1","terms":['

        if data.wikiUrl != '' :
            xray += '{"type":"topic","term":"Wikipedia Info","desc":"%s","descSrc":"wiki","descUrl":"%s","locs":[[100,100,100,5]]},' \
            % ( data.wikiText, data.wikiUrl ); 

        data.chars.sort()
        for c in data.chars:
            try:
                key, value = c.split (':', 1)
                xray += '{"type":"character","term":"%s","desc":"%s","descSrc":"wiki","descUrl":"http://m.wikipedia.org/wiki/%s","locs":[[100,100,100,6]]},' \
                % ( self.charreplace(key.strip()), self.charreplace(value.strip()), self.wikireplace (key.strip()) )
            except ValueError:
                key = c
                value = c

        data.topics.sort()
        for t in data.topics:
            try:
                key, value = t.split (':', 1)
            except ValueError:
                key = t
                value = t

            xray += '{"type":"topic","term":"%s","desc":"%s","descSrc":"wiki","descUrl":"http://m.wikipedia.org/wiki/%s","locs":[[100,100,100,6]]},' \
            % ( key.strip(), self.charreplace(value.strip()), self.wikireplace (key.strip()) )

        xray = re.sub ( ',$', '', xray) # remove last comma

        xray += '],"chapters":[{"name":null,"start":1,"end":9999999}]}'
        xrayfile = os.path.join (xraydir, "XRAY.entities." + asin + ".asc")
        with open(xrayfile, 'wb') as xrayf:
            xrayf.write(xray)

class XRayData(object):
    def __init__(self):
        self.chars = []
        self.topics = []
        self.wikiText = ''
        self.wikiUrl = ''

    def shelfari (self, shelfariUrl):
        import urllib2

        response = urllib2.urlopen(str(shelfariUrl))
        shelHtml = response.read()
        shelDoc = html5lib.parse(shelHtml, treebuilder='lxml', namespaceHTMLElements=False)

        characters = CSSSelector("#WikiModule_Characters ul.li_6 li")
        for character in characters (shelDoc):
            self.chars.append ( "".join (character.itertext()) )

        glossary = CSSSelector("#WikiModule_Glossary ul.li_6 li")
        for gloss in glossary (shelDoc):
            self.topics.append ( "".join (gloss.itertext()) )

        orgs = CSSSelector("#WikiModule_Organizations ul.li_6 li")
        for org in orgs (shelDoc):
            self.topics.append ( "".join (org.itertext()) )

        settings = CSSSelector("#WikiModule_Settings ul.li_6 li")
        for setting in settings (shelDoc):
            self.topics.append ( "".join (setting.itertext()) )

        themes = CSSSelector("#WikiModule_Themes ul.li_6 li")
        for theme in themes (shelDoc):
            self.topics.append ( "".join (theme.itertext()) )

    def wikipedia (self, wikiUrl):
        import urllib2
        self.wikiUrl = wikiUrl
        if wikiUrl != '':
            req = urllib2.Request(str(wikiUrl), headers={'User-Agent' : "Calibre"}) 
            con = urllib2.urlopen(req)
            wikiHtml = con.read()
            wikiDoc = html5lib.parse(wikiHtml, treebuilder='lxml', namespaceHTMLElements=False)
            selector = CSSSelector(".mw-content-ltr p")
            self.wikiText = "".join(selector(wikiDoc)[0].itertext())

