__license__ = 'GPL 3'
__copyright__ = '2012 Saulius P.'
__docformat__ = 'restructuredtext en'

from calibre_plugins.docx_input_plugin.structures.docx_types import IndividualElement,\
    ContainerElement, ParagraphFormatting, RichFormatting, ParagraphNumber
from calibre_plugins.docx_input_plugin.structures.text_properties import ParagraphProperties,\
    FontProperties

class StyleBasedOn(IndividualElement):
    TAG = 'w:basedOn'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.basedOn = None
        
    def preProcessTag(self, attrs):
        self.basedOn = self._getAttributeValue(attrs, "w:val")

class AbstractStyle(ContainerElement, IndividualElement):
    TAG = 'w:style'
    TYPE = 'A'
    def __init__(self, sId, context):
        ContainerElement.__init__(self, context)
        self.id = sId
        self.default = False
        self.used = False
        self.basedOn = None
        
    def preProcessTag(self, attrs):
        d = self._getAttributeValue(attrs, "w:default")
        if d is not None and int(d) == 1:
            self.default = True
    
    def mergeStyle(self, style):
        return
        
    def getCssId(self):
        cid = self.id.lower().replace(' ', '-').replace('+', '')
        if cid[0].isdigit():
            return 'docx'+cid;
        else:
            return cid
    
    def getFontCss(self):
        return ''
    
    def appendElement(self, element):
        if element.TAG == StyleBasedOn.TAG:
            self.basedOn = element.basedOn
        
    def debugStyle(self):
        print("AbstractStyle: %s" % self.id)

class TableStyle(AbstractStyle):
    TYPE = 'T'
    def __init__(self, sId, context):
        AbstractStyle.__init__(self, sId, context)
        
    def debugStyle(self):
        print("TableStyle: %s" % self.id)
        
class NumberingStyle(AbstractStyle):
    TYPE = 'N'
    def __init__(self, sId, context):
        AbstractStyle.__init__(self, sId, context)
        self.level = 0
        self.numId = 0
        
    def debugStyle(self):
        print("NumberingStyle: %s" % self.id)
    
    def appendElement(self, element):
        print 'NumberingStyle:', self.id, ' append element ', element.TAG
        AbstractStyle.appendElement(self, element)
        if element.TAG == ParagraphFormatting.TAG:
            if element.number is not None:
                print 'NumberingStyle: append element ', element.number.level, element.number.numId
                self.level = element.number.level
                self.numId = element.number.numId

class CharacterStyle(AbstractStyle):
    TYPE = 'C'
    def __init__(self, sId, context):
        AbstractStyle.__init__(self, sId, context)
        self.used = False
        self.fontProperties = FontProperties(self.context)
        
    def appendElement(self, element):
        AbstractStyle.appendElement(self, element)
        if element.TAG == RichFormatting.TAG:
            self.fontProperties = element.fontProperties
    
    def mergeStyle(self, style):
        self.fontProperties.merge(style.fontProperties)
    
    def isMergeble(self):
        if self.fontProperties.caps is not None or self.fontProperties.size is not None or self.fontProperties.color is not None:
            return False
        if self.fontProperties.bold is not None or self.fontProperties.italic is not None:
            return True
        else:
            return False
    
    def getFontCss(self):
        return self.fontProperties.getCss(self)
 
    def debugStyle(self):
        print("CharacterStyle: %s" % self.id)
        print("  is mergeable: %d" % self.isMergeble())
        self.fontProperties.debug()
    
    
class ParagraphStyle(CharacterStyle):
    TYPE = 'P'
    def __init__(self, sId, context):
        CharacterStyle.__init__(self, sId, context)
        self.paragraphProperties = ParagraphProperties(self.context)
    
    def appendElement(self, element):
        CharacterStyle.appendElement(self, element)
        if element.TAG == ParagraphFormatting.TAG:
            self.paragraphProperties = element.paragraphProperties
    
    def mergeStyle(self, style):
        CharacterStyle.mergeStyle(self, style)
        self.paragraphProperties.merge(style.paragraphProperties)
            
    def postProcessTag(self):
        if self.default and self.context.opts.substitute_normal_serif:
            self.fontProperties.fontFamily = 'Serif'
            
    def getFontCss(self):
        if self.fontProperties.fontFamily is None and self.basedOn is not None and self.basedOn in self.context.styles:
            baseStyle = self.context.styles[self.basedOn]
            self.fontProperties.fontFamily = baseStyle.fontProperties.fontFamily
        return self.fontProperties.getCss(self)
        
    def debugStyle(self):
        print("ParagraphStyle: %s" % self.id)
        self.fontProperties.debug()
        self.paragraphProperties.debug()
