__license__ = 'GPL 3'
__copyright__ = '2012 Saulius P.'
__docformat__ = 'restructuredtext en'

from calibre_plugins.docx_input_plugin.structures.docx_types import ContainerElement,\
    ParagraphFormatting, RichFormatting, IndividualElement
from calibre_plugins.docx_input_plugin.structures.text_properties import FontProperties,\
    ParagraphProperties

class AbstractNum(ContainerElement):
    TAG = 'w:abstractNum'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        self.aNumId = None
        self.style = None
        self.levels = {}
    
    def preProcessTag(self, attrs):
        self.aNumId = int(self._getAttributeValue(attrs, 'w:abstractNumId'))
    
    def postProcessTag(self):
        self.context.abstractNums[self.aNumId] = self
        
    def appendElement(self, element):
        #print 'AbstractNum::appendElement:', element.TAG
        if element.TAG == ListLevel.TAG:
            element.parent = self
            self.levels[element.level] = element
        elif element.TAG == NumStyleLink.TAG:
            self.style = element.style
    
    def getText(self, numId, level):
        #print 'AbstractNum::getText: self.getLevels:', self.getLevels()
        levels = self.getLevels()
        if levels is not None:
            return levels[level].getText(numId)
        else:
            return ''
    
    def getLevels(self):
        if len(self.levels) > 0:
            return self.levels
        elif self.style is not None:
            return self.context.abstractNums[self.context.listNums[self.context.styles[self.style].numId].aNumId].getLevels()
        else:
            return None

    
class ListLevel(ContainerElement):
    TAG = 'w:lvl'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        self.parent = None
        self.level = 0
        self.start = 1
        self.numberFormat = None
        self.textFormat = None
        self.fontProperties = FontProperties(context)
        self.paragraphProperties = ParagraphProperties(context)
        self.typeMap = {
                        "bullet": self._bullet,
                        "decimal": self._decimal,
                        "decimalZero": self._decimalZero,
                        "upperRoman": self._upperRoman,
                        "lowerRoman": self._lowerRoman,
                        "lowerLetter": self._lowerLetter,
                        "upperLetter": self._upperLetter
                        }
    
    def preProcessTag(self, attrs):
        self.level = int(self._getAttributeValue(attrs, 'w:ilvl'))
    
    def appendElement(self, element):
        if element.TAG == ParagraphFormatting.TAG:
            self.paragraphProperties = element.paragraphProperties
        elif element.TAG == RichFormatting.TAG:
            self.fontProperties = element.fontProperties
        elif element.TAG == ListStartValue.TAG:
            self.start = element.start
        elif element.TAG == ListNumberFormatType.TAG:
            self.numberFormat = element.numberFormat
        elif element.TAG == ListTextFormat.TAG:
            self.textFormat = element.textFormat
    
    def getText(self, numId):
        self._addUsedFonts()
        t = ''
        if self.numberFormat == 'bullet':
            b = self.getNumber(numId)
            if len(b) == 1 and ord(b) == 61623:
                self.fontProperties.fontFamily = None
                t += unichr(8226) + '&nbsp;'
            else:
                t += b + '&nbsp;'
        else:
            tFormat = self.textFormat
            #print 'ListLevel::getText: format:', tFormat
            for i in reversed(range(0, self.level+1)):
                #print 'ListLevel::getText: iteration:', i
                tFormat = tFormat.replace('%'+str(i+1), self.parent.levels[i].getNumber(numId, i==self.level))
            t += tFormat+'&nbsp;'
        return '<span style="'+self.fontProperties.getCss()+'">'+t+'</span>'
    
    def getNumber(self, numId, increase=True):
        #print 'ListLevel::getNumber: ', self.level, numId
        if not numId in self.context.numCounters:
            #print 'ListLevel::getNumber: add numId to map'
            self.context.numCounters[numId] = {}
            self.context.numCounters[numId][self.level] = 2
        else:
            if self.level in self.context.numCounters[numId]:
                #print 'ListLevel::getNumber: increase level counter'
                if increase:
                    self.context.numCounters[numId][self.level] += 1
            else:
                #print 'ListLevel::getNumber: add level to map'
                self.context.numCounters[numId][self.level] = 2
        
        if self.numberFormat in self.typeMap:
            #print 'ListLevel::getNumber: get type', self.numberFormat
            return self.typeMap[self.numberFormat](self.context.numCounters[numId][self.level]-1)
        else:
            #print 'ListLevel::getNumber: default to decimal ', self.numberFormat
            return self._decimal(self.context.numCounters[numId][self.level]-1)
        
    def _bullet(self, number):
        return self.textFormat
    
    def _decimal(self, number):
        return str(number)
    
    def _decimalZero(self, number):
        if number<10:
            return '0'+self._decimal(number)
        else:
            return self._decimal(number)
    
    def _upperRoman(self, number):
        numerals = { 1 : "I", 4 : "IV", 5 : "V", 9 : "IX", 10 : "X", 40 : "XL", 
            50 : "L", 90 : "XC", 100 : "C", 400 : "CD", 500 : "D", 900 : "CM", 1000 : "M" }
        result = ""
        for value, numeral in sorted(numerals.items(), reverse=True):
            while number >= value:
                result += numeral
                number -= value
        return result
    
    def _lowerRoman(self, number):
        return self._upperRoman(number).lower()
    
    def _lowerLetter(self, number):
        return chr(96+number)
    
    def _upperLetter(self, number):
        return chr(64+number)
    
    def _addUsedFonts(self):
        if self.fontProperties.fontFamily is not None:
            family = self.fontProperties.fontFamily
            bold = self.fontProperties.bold
            italic = self.fontProperties.italic
            if not family in self.context.fontStyles:
                self.context.fontStyles[family] = []
            
            if bold is not True and italic is not True:
                if 'Regular' not in self.context.fontStyles[family]:
                    self.context.fontStyles[family].append('Regular')
            elif bold is True and italic is True:
                if 'BoldItalic' not in self.context.fontStyles[family]:
                    self.context.fontStyles[family].append('BoldItalic')
            elif bold is True:
                if 'Bold' not in self.context.fontStyles[family]:
                    self.context.fontStyles[family].append('Bold')
            elif italic is True:
                if 'Italic' not in self.context.fontStyles[family]:
                    self.context.fontStyles[family].append('Italic')

class ListStartValue(IndividualElement):
    TAG = 'w:start'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.start = 1
    
    def preProcessTag(self, attrs):
        self.start = int(self._getAttributeValue(attrs, 'w:val'))

class ListNumberFormatType(IndividualElement):
    TAG = 'w:numFmt'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.numberFormat = None
    
    def preProcessTag(self, attrs):
        self.numberFormat = self._getAttributeValue(attrs, 'w:val')
    
class ListTextFormat(IndividualElement):
    TAG = 'w:lvlText'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.textFormat = None
    
    def preProcessTag(self, attrs):
        self.textFormat = self._getAttributeValue(attrs, 'w:val')

class ListNum(ContainerElement):
    TAG = 'w:num'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        self.numId = 0
        self.aNumId = 0
    
    def preProcessTag(self, attrs):
        self.numId = int(self._getAttributeValue(attrs, 'w:numId'))
    
    def postProcessTag(self):
        self.context.listNums[self.numId] = self
        
    def appendElement(self, element):
        if element.TAG == AbstractNumId.TAG:
            self.aNumId = element.aNumId

class AbstractNumId(IndividualElement):
    TAG = 'w:abstractNumId'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.aNumId = 0
    
    def preProcessTag(self, attrs):
        self.aNumId = int(self._getAttributeValue(attrs, 'w:val'))

class NumStyleLink(IndividualElement):
    TAG = 'w:numStyleLink'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.style = None
    
    def preProcessTag(self, attrs):
        self.style = self._getAttributeValue(attrs, 'w:val')
