__license__ = 'GPL 3'
__copyright__ = '2012 Saulius P.'
__docformat__ = 'restructuredtext en'

'''
Abstract type of any DOCX element, providing the methods
to be overridden.
'''
from calibre_plugins.docx_input_plugin.structures.text_properties import FontProperties
from calibre_plugins.docx_input_plugin.structures.text_properties import ParagraphProperties

class Element(object):
    '''
    The constant to map an element to the matching tag
    '''
    TAG = ''
    '''
    The constant to map tags, that don't represent some individual elements,
    but set particular properties of this element.
    '''
    SUBTAGS = []
    '''
    Mark if element can process characters
    '''
    CHARACTERS = False
    
    def __init__(self, context):
        self.context = context
        self.parent = None
    '''
    Process this particular element's tag and its attributes when it is open
    '''
    def preProcessTag(self, attrs):
        return
    '''
    Postprocess this element's tag when it is closed.
    '''
    def postProcessTag(self):
        return
    '''
    Process matching this particular element's subtag and its attributes
    '''
    def processSubtag(self, attrs):
        return

    '''
    Extract particular tag's attribute
    '''
    def _getAttributeValue(self, attrs, attrName):
        for (name, value) in attrs.items():
            if name == attrName:
                return value
        return None
    
    def debug(self):
        print('Debugging Element')

'''
A class, which represents some individual element,
i.e. having no more sub-elements, like word text tag or
footnote reference.
'''
class IndividualElement(Element):
    def __init__(self, context):
        Element.__init__(self, context)

'''
TextElement is an abstract class representing some text in the document.
'''
class TextElement(Element):
    '''
    Supposed to return HTML-formatted text of the element
    '''
    def getText(self):
        return ''
    '''
    Supposed to return plain text (might be required in metadata tags)
    '''
    def getPlainText(self):
        return ''


'''
An abstract class, which has several sub-elements
'''
class ContainerElement(Element):
    def __init__(self, context):
        Element.__init__(self, context)
        self._elements = []
    '''
    Append some other processed element
    '''
    def appendElement(self, element):
        self._elements.append(element)
        
'''
A class, which represents an actual text in Word document.
The only one, which can process characters.
'''
class WordTextElement(TextElement):
    TAG = 'w:t'
    CHARACTERS = True
    
    def __init__(self, context):
        TextElement.__init__(self, context)
        self._text = ''
    
    def processCharacters(self, content):
        self._text += content
    
    def postProcessTag(self):
        self._text = self._text.replace(unichr(160), '&nbsp;')
        self._text = self._text.replace(unichr(8212), '&mdash;')
        self._text = self._text.replace(unichr(8211), '&ndash;')
        self._text = self._text.replace(unichr(8222), '&bdquo;')
        self._text = self._text.replace(unichr(8220), '&ldquo;')
        
    def getText(self):
        return self._text

    def getPlainText(self):
        return self._text
    
    def debug(self):
        print('Debugging WordTextElement:')
        print('    text: %s' % self._text)

class FootnoteReference(TextElement, IndividualElement):
    TAG = 'w:footnoteReference'
    
    def __init__(self, context):
        TextElement.__init__(self, context)
        self.refId = None
        self.count = None
        
    def preProcessTag(self, attrs):
        self.refId = self._getAttributeValue(attrs, 'w:id')
        self.context.footnoteCount += 1
        self.context.footnotes[self.refId] = self.context.footnoteCount 
        self.count = self.context.footnoteCount
        
    def getText(self):
        self.context.footnoteAnchors[self.refId] = '%s.html#%s' % (('%d' % self.context.htmlCount).zfill(3), self.refId)
        return '<sup><a id="'+self.refId+'" href="ft_'+('%d' % self.count).zfill(3)+'.html">{'+('%d' % self.count)+'}</a></sup>'
    
    def getPlainText(self):
        return '[%d]' % self.count
    
    def debug(self):
        print('Debugging FootnoteReference:')
        print '    refId: ', self.refId
        print '    count: ', self.count

class WordParagraph(ContainerElement, TextElement):
    TAG = 'w:p'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        self._tag = 'p'
        self._class = ''
        self._cssStyle = ''
        self._style = None
        self._number = None
        self._paragraphProperties = ParagraphProperties(self.context)
        self._pageBreak = False
        self.dropCap = False
        #self._rFormat = None
        
    def getText(self):
        self._transformStyle()
        (tb, ta) = self._transformBeforeAfter()
        textStart = tb+"<" + self._tag + self._class+ self._cssStyle+">"
        textEnd = "</" + self._tag + ">\n"+ta
        
        if self._number is not None:
            text = self._number.getText()
        else:
            text = ''
        if self.context.dropCapParagraph is not None:
            #print 'Radom dropcapa. Tekstas sito paragrafo: ', self.getPlainText()
            dcPara = self.context.dropCapParagraph
            self.context.dropCapParagraph = None
            text += dcPara.getText()
            #print 'Radom dropcapa. Tekstas sito paragrafo: ', self.getPlainText()
        for t in self._elements:
            text += t.getText()
        text = textStart + text + textEnd
        if text == textStart + textEnd:
            text = textStart + '&nbsp;' + textEnd
        
        return text 

    def getPlainText(self):
        text = ''
        for t in self._elements:
            text += t.getPlainText()
        return text
    
    def getStyle(self):
        return self._style
    
    def getFontProperties(self):
        fontFamily = None
        bold = None
        italic = None
        
        if self._style is not None:
            sstyle = self._style
        else:
            sstyle = self.context.styles["DefaultParagraphStyle"]
        
        if sstyle.basedOn is not None and sstyle.basedOn in self.context.styles:
            baseStyle = self.context.styles[sstyle.basedOn]
            if baseStyle.fontProperties.fontFamily is not None:
                fontFamily = baseStyle.fontProperties.fontFamily
            if baseStyle.fontProperties.bold is not None:
                bold = baseStyle.fontProperties.bold
            if baseStyle.fontProperties.italic is not None:
                italic = baseStyle.fontProperties.italic
        if sstyle.TYPE == 'P':
            if sstyle.fontProperties.fontFamily is not None:
                fontFamily = sstyle.fontProperties.fontFamily
            if sstyle.fontProperties.bold is not None:
                bold = sstyle.fontProperties.bold
            if sstyle.fontProperties.italic is not None:
                italic = sstyle.fontProperties.italic
        
        return (fontFamily, bold, italic)
    
    def appendElement(self, element):
        if type(element) is RichContainer:
            ContainerElement.appendElement(self, element)
            element.parent = self
        elif type(element) is ParagraphFormatting:
            self._paragraphProperties = element.paragraphProperties
            self._number = element.number
            if element.styleId is not None:
                self._style = self.context.styles[element.styleId]
                
    def postProcessTag(self):
        if self._paragraphProperties.dropCap is True:
            self.dropCap = True
    
    def _transformStyle(self):
        if self._style is not None:
            if self._style.paragraphProperties.outline is not None:
                self._tag = "h%d" % (self._style.paragraphProperties.outline+1)
                self._class = " class=\""+self._style.getCssId()+"\""
                self._style.used = True
            else:
                if not self._style.default:
                    self._class = " class=\""+self._style.getCssId()+"\""
                    self._style.used = True
        styleParaCss = self._paragraphProperties.getCss(defaults=False)
        if self._paragraphProperties.dropCap is True:
            self._tag = 'span'
            styleParaCss += 'float: left; margin-bottom: -0.3em; margin-top: -0.3em; line-height: 3em;'
        if styleParaCss != '':
            #print("========== STYLE PARA CSS: %s" % styleParaCss)
            self._cssStyle = ' style="'+styleParaCss+'"'
    
    def _transformBeforeAfter(self):
        if self.context.opts.replace_bap is False:
            return ('', '')
        
        tb, ta = '', ''
        if self._style is not None:
            sstyle = self._style
        else:
            sstyle = self.context.styles["DefaultParagraphStyle"]
        
        if sstyle.basedOn is not None and sstyle.basedOn in self.context.styles:
            baseStyle = self.context.styles[sstyle.basedOn]
            if baseStyle.paragraphProperties.before is not None and int(baseStyle.paragraphProperties.before) > 0:
                tb = "<p>&nbsp;</p>\n"
            if baseStyle.paragraphProperties.after is not None and int(baseStyle.paragraphProperties.after) > 0:
                ta = "<p>&nbsp;</p>\n"

        if sstyle.paragraphProperties.before is not None:
            if int(sstyle.paragraphProperties.before) > 0:
                tb = "<p>&nbsp;</p>\n"
            else:
                tb = ''
        if sstyle.paragraphProperties.after is not None:
            if int(sstyle.paragraphProperties.after) > 0:
                ta = "<p>&nbsp;</p>\n"
            else:
                ta = ''
        
        if self._paragraphProperties.before is not None:
            if int(self._paragraphProperties.before) > 0:
                tb = "<p>&nbsp;</p>\n"
            else:
                tb = ''
                
        if self._paragraphProperties.after is not None:
            if int(self._paragraphProperties.after) > 0:
                ta = "<p>&nbsp;</p>\n"
            else:
                ta = ''
        
        return (tb, ta)
            
    def debug(self):
        print('Debugging WordParagraph:')
        print '    _tag: ', self._tag
        print '    _class: ', self._class
        print '    _cssStyle: ', self._cssStyle
        print '    _style: ', self._style
        print '    _pageBreak: ', self._pageBreak
        print '  _paragraphProperties:'
        self._paragraphProperties.debug()
        for e in self._elements:
            e.debug()
        print('Debugging WordParagraph END.')
        

class RichContainer(ContainerElement, TextElement):
    TAG = 'w:r'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        self._style = None
        self._fontProperties = FontProperties(self.context)
        self._origRichFormat = None
    
    def getText(self):
        # Add used fonts for embedding dictionary
        fst = self.getFontProperties()
        self.addUsedFonts(fst)
        
        self._styleMerge()
        text = self._startTransformation()
        for t in self._elements:
            text += t.getText()
        text += self._endTransformation()
        return text 

    def getPlainText(self):
        text = ''
        for t in self._elements:
            text += t.getPlainText()
        return text
    
    def appendElement(self, element):
        if type(element) is WordTextElement or \
            type(element) is FootnoteReference or \
            type(element) is Break or \
            type(element) is Drawing or \
            type(element) is Picture:
            ContainerElement.appendElement(self, element)
        elif type(element) is RichFormatting:
            self._origRichFormat = element
            if element.styleId is not None:
                self._style = self.context.styles[element.styleId]
            self._fontProperties = element.fontProperties
        elif type(element) is LastRenderedPageBreak:
            self.context.renderedPageBreakFound = True
    
    def _startTransformation(self):
        t = ''
        
        if self._fontProperties.bold is True:
            t += '<b>'
        elif self._fontProperties.bold is False:
            t += '<span style="font-weight: normal;">'
        
        if self._fontProperties.italic is True:
            t += '<i>'
        elif self._fontProperties.italic is False:
            t += '<span style="font-style:normal;">'
        
        if self._fontProperties.vertAlign is not None:
            if self._fontProperties.vertAlign == 'superscript':
                t += '<sup>'
            elif self._fontProperties.vertAlign == 'subscript':
                t += '<sub>'
            
        style = self._fontProperties.getCss(False, False, False)
        tClass = ''
        if style != '':
            style = ' style="'+style+'"'
        if self._style is not None:
            tClass = ' class="'+self._style.getCssId()+'"'
            self._style.used = True
        
        if style !='' or tClass != '':
            t += '<span'+tClass+style+'>'
        
        return t
    
    def _endTransformation(self):
        t = ''
        if self._fontProperties.getCss(False, False, False) != '' or self._style is not None:
            t += '</span>'
        
        if self._fontProperties.vertAlign is not None:
            if self._fontProperties.vertAlign == 'superscript':
                t += '</sup>'
            elif self._fontProperties.vertAlign == 'subscript':
                t += '</sub>'
        
        if self._fontProperties.italic is True:
            t += '</i>'
        elif self._fontProperties.italic is False:
            t += '</span>'
        
        if self._fontProperties.bold is True:
            t += '</b>'
        elif self._fontProperties.bold is False:
            t += '</span>'
        
        return t
    
    def _styleMerge(self):
        if self._style is None:
            return
        
        if self._fontProperties.vertAlign is None:
            self._fontProperties.vertAlign = self._style.fontProperties.vertAlign
        
        if not self._style.isMergeble():
            return
        if self._fontProperties.bold is None:
            self._fontProperties.bold = self._style.fontProperties.bold
        if self._fontProperties.italic is None:
            self._fontProperties.italic = self._style.fontProperties.italic
        self._style = None
        
    def getFontProperties(self):
        family, bold, italic = self.parent.getFontProperties()
        sid = None
        bid = None
        if self._style is not None:
            sid = self._style.id
            if self._style.basedOn is not None:
                bid = self._style.basedOn
                baseStyle = self.context.styles[self._style.basedOn]
                if baseStyle.fontProperties.fontFamily is not None:
                    family = baseStyle.fontProperties.fontFamily
                if baseStyle.fontProperties.bold is not None:
                    bold = baseStyle.fontProperties.bold
                if baseStyle.fontProperties.italic is not None:
                    italic = baseStyle.fontProperties.italic
            if self._style.fontProperties.fontFamily is not None:
                family = self._style.fontProperties.fontFamily
            if self._style.fontProperties.bold is not None:
                bold = self._style.fontProperties.bold
            if self._style.fontProperties.italic is not None:
                italic = self._style.fontProperties.italic
                
        if self._fontProperties.fontFamily is not None:
            family = self._fontProperties.fontFamily
        if self._fontProperties.bold is not None:
            bold = self._fontProperties.bold
        if self._fontProperties.italic is not None:
            italic = self._fontProperties.italic
        #print "RichContainer::getFontProperties: styleId, basedOn, family, bold, italic: ", sid, bid, family, bold, italic
        return (family, bold, italic)
    
    def addUsedFonts(self, used):
        family, bold, italic = used
        if not family in self.context.fontStyles:
            self.context.fontStyles[family] = []
        
        if bold is not True and italic is not True:
            if 'Regular' not in self.context.fontStyles[family]:
                self.context.fontStyles[family].append('Regular')
            if 'Book' not in self.context.fontStyles[family]:
                self.context.fontStyles[family].append('Book')
        elif bold is True and italic is True:
            if 'BoldItalic' not in self.context.fontStyles[family]:
                self.context.fontStyles[family].append('BoldItalic')
        elif bold is True:
            if 'Bold' not in self.context.fontStyles[family]:
                self.context.fontStyles[family].append('Bold')
        elif italic is True:
            if 'Italic' not in self.context.fontStyles[family]:
                self.context.fontStyles[family].append('Italic')
            
        
    def debug(self):
        print('Debugging RichContainer:')
        print '    _style: ', self._style
        print '    _fontProperties:'
        self._fontProperties.debug()
        for e in self._elements:
            e.debug()
        if self._origRichFormat is not None:
            self._origRichFormat.debug()
        print('Debugging RichContainer END.')

class RichFormatting(ContainerElement):
    TAG = 'w:rPr'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        self.styleId = None
        self.fontProperties = FontProperties(self.context)
        self._origBold = None
        self._origItalic = None
        
    def appendElement(self, element):
        if type(element) is RichStyle:
            self.styleId = element.styleId
        elif type(element) is Bold:
            self._origBold = element
            self.fontProperties.bold = element.bold
        elif type(element) is Italic:
            self._origItalic = element
            self.fontProperties.italic = element.italic
        elif type(element) is FontSize:
            self.fontProperties.size = element.size
        elif type(element) is FontCaps:
            self.fontProperties.caps = element.caps
        elif type(element) is FontColor:
            self.fontProperties.color = element.color
        elif type(element) is StrikeThrough or type(element) is DoubleStrikeThrough:
            self.fontProperties.strike = element.strike
        elif type(element) is VerticalAlign:
            self.fontProperties.vertAlign = element.vertAlign
        elif type(element) is Underline:
            self.fontProperties.underline = element.underline
        elif type(element) is RichFont:
            self.fontProperties.fontFamily = element.fontFamily
        elif type(element) is Highlight:
            self.fontProperties.highlightColor = element.highlightColor
    '''
    def postProcessTag(self):
        if self.fontProperties.fontFamily is None:
            return
        if not self.fontProperties.fontFamily in self.context.fontStyles:
            self.context.fontStyles[self.fontProperties.fontFamily] = []
        
        if not self.fontProperties.bold and not self.fontProperties.italic:
            if "Regular" not in self.context.fontStyles[self.fontProperties.fontFamily]:
                self.context.fontStyles[self.fontProperties.fontFamily].append("Regular")
        elif self.fontProperties.bold is True and self.fontProperties.italic is True:
            if "BoldItalic" not in self.context.fontStyles[self.fontProperties.fontFamily]:
                self.context.fontStyles[self.fontProperties.fontFamily].append("BoldItalic")
        elif self.fontProperties.bold is True:
            if "Bold" not in self.context.fontStyles[self.fontProperties.fontFamily]:
                self.context.fontStyles[self.fontProperties.fontFamily].append("Bold")
        elif self.fontProperties.italic is True:
            if "Italic" not in self.context.fontStyles[self.fontProperties.fontFamily]:
                self.context.fontStyles[self.fontProperties.fontFamily].append("Italic")
    '''
    
    def debug(self):
        print('Debugging RichFormatting:')
        print('    fontProperties:')
        self.fontProperties.debug()
        if self._origBold is not None:
            self._origBold.debug()
        if self._origItalic is not None:
            self._origItalic.debug()


class RichFont(IndividualElement):
    TAG = 'w:rFonts'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.fontFamily = None
    
    def preProcessTag(self, attrs):
        self.fontFamily = self._getAttributeValue(attrs, 'w:ascii')


class RichStyle(IndividualElement):
    TAG = 'w:rStyle'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.styleId = None
        
    def preProcessTag(self, attrs):
        self.styleId = self._getAttributeValue(attrs, 'w:val')
        
class PStyle(IndividualElement):
    TAG = 'w:pStyle'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.styleId = None
        
    def preProcessTag(self, attrs):
        self.styleId = self._getAttributeValue(attrs, 'w:val')
        
class Bold(IndividualElement):
    TAG = 'w:b'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.bold = None
        self._debugInfo = None
    
    def preProcessTag(self, attrs):
        self.bold = True
        self._debugInfo = ('Bold attribute: ', self._getAttributeValue(attrs, 'w:val'))
        value = self._getAttributeValue(attrs, 'w:val')
        if value is not None and (value == '0' or value == u'0'):
            self.bold = False
         
    def debug(self):
        print('Debugging Bold:')
        print '    bold: ', self.bold
        print '    debugInfo: ', self._debugInfo
        
class Italic(IndividualElement):
    TAG = 'w:i'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.italic = None
    
    def preProcessTag(self, attrs):
        self.italic = True
        self._debugInfo = ('Italic attribute: ', self._getAttributeValue(attrs, 'w:val'))
        value = self._getAttributeValue(attrs, 'w:val')
        if value is not None and (value == '0' or value == u'0'):
            self.italic = False
            
    def debug(self):
        print('Debugging Italic:')
        print '    bold: ', self.italic
        print '    debugInfo: ', self._debugInfo

class Underline(IndividualElement):
    TAG = 'w:u'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.underline = None
    
    def preProcessTag(self, attrs):
        self.underline = True
        value = self._getAttributeValue(attrs, 'w:val')
        if value is not None and (value == '0' or value == 'none'):
            self.underline = False

class FontSize(IndividualElement):
    TAG = 'w:sz'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.size = None
    
    def preProcessTag(self, attrs):
        self.size = self._getAttributeValue(attrs, 'w:val')
            
class FontColor(IndividualElement):
    TAG = 'w:color'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.color = None
    
    def preProcessTag(self, attrs):
        self.color = self._getAttributeValue(attrs, 'w:val')

class FontCaps(IndividualElement):
    TAG = 'w:caps'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.caps = None
    
    def preProcessTag(self, attrs):
        self.caps = True
        val = self._getAttributeValue(attrs, 'w:val')
        if val is not None and val == '0':
            self.caps = False

class StrikeThrough(IndividualElement):
    TAG = 'w:strike'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.strike = None
    
    def preProcessTag(self, attrs):
        self.strike = True
        val = self._getAttributeValue(attrs, 'w:val')
        if val is not None and val == '0':
            self.strike = False

class DoubleStrikeThrough(IndividualElement):
    TAG = 'w:dstrike'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.strike = None
    
    def preProcessTag(self, attrs):
        self.strike = True
        val = self._getAttributeValue(attrs, 'w:val')
        if val is not None and val == '0':
            self.strike = False
            
class VerticalAlign(IndividualElement):
    TAG = 'w:vertAlign'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.vertAlign = None
    
    def preProcessTag(self, attrs):
        self.vertAlign = self._getAttributeValue(attrs, 'w:val')

class Highlight(IndividualElement):
    TAG = 'w:highlight'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.highlightColor = None
    
    def preProcessTag(self, attrs):
        self.highlightColor = self._getAttributeValue(attrs, 'w:val')
            
class ParagraphFormatting(ContainerElement):
    TAG = 'w:pPr'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        self.styleId = None
        self.number = None
        self.paragraphProperties = ParagraphProperties(self.context)
        
    def appendElement(self, element):
        if type(element) is PageBreak:
            self.paragraphProperties.pageBreak = True
        elif type(element) is PStyle:
            self.styleId = element.styleId
        elif type(element) is TextAlign:
            self.paragraphProperties.align = element.align
        elif type(element) is TextIndent:
            self.paragraphProperties.firstLine = element.firstLine
            self.paragraphProperties.indLeft = element.indLeft
            self.paragraphProperties.indRight = element.indRight
        elif type(element) is ParagraphSpacing:
            self.paragraphProperties.before = element.before
            self.paragraphProperties.after = element.after
        elif type(element) is OutlineLevel:
            self.paragraphProperties.outline = element.outline
        elif type(element) is ParagraphFrame:
            if element.dropCap is True:
                self.paragraphProperties.dropCap = True
        elif type(element) is ParagraphNumber:
            self.paragraphProperties.list = True
            self.number = element
        elif type(element) is Shading:
            self.paragraphProperties.shadeColor = element.shadeColor

class TextAlign(IndividualElement):
    TAG = 'w:jc'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.align = None
    
    def preProcessTag(self, attrs):
        self.align = self._getAttributeValue(attrs, 'w:val')

class TextIndent(IndividualElement):
    TAG = 'w:ind'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.firstLine = None
        self.indLeft = None
        self.indRight = None
        
    def preProcessTag(self, attrs):
        for (name, value) in attrs.items():
            if name == 'w:firstLine':
                self.firstLine = value
            elif name == 'w:hanging':
                self.firstLine = '-'+value
            elif name == 'w:left':
                self.indLeft = value
            elif name == 'w:right':
                self.indRight = value

class ParagraphSpacing(IndividualElement):
    TAG = 'w:spacing'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.before = None
        self.after = None
    
    def preProcessTag(self, attrs):
        for (name, value) in attrs.items():
            if name == 'w:before':
                self.before = value
            elif name == 'w:after':
                self.after = value

class OutlineLevel(IndividualElement):
    TAG = 'w:outlineLvl'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.outline = None
    
    def preProcessTag(self, attrs):
        self.outline = int(self._getAttributeValue(attrs, 'w:val'))

class Shading(IndividualElement):
    TAG = 'w:shd'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.shadeColor = None
    
    def preProcessTag(self, attrs):
        self.shadeColor = self._getAttributeValue(attrs, 'w:color')

class Break(TextElement, IndividualElement):
    TAG = 'w:br'
    TYPE_PAGE = 'page'
    TYPE_LINE = 'line'
    
    def __init__(self, context):
        TextElement.__init__(self, context)
        self.type = None
        
    def preProcessTag(self, attrs):
        if len(attrs) == 0:
            self.type = self.TYPE_LINE
        else:
            self.type = self._getAttributeValue(attrs, 'w:type')
        
    def getText(self):
        if self.type == self.TYPE_PAGE:
            self.context.pageBreakFound = True
            return ''
        elif self.type == self.TYPE_LINE:
            return '<br />'
        else:
            return ''
    
    def getPlainText(self):
        return ''

class PageBreak(IndividualElement):
    TAG = 'w:pageBreakBefore'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        
class LastRenderedPageBreak(IndividualElement):
    TAG = 'w:lastRenderedPageBreak'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        
class ParagraphFrame(IndividualElement):
    TAG = 'w:framePr'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.dropCap = False
    
    def preProcessTag(self, attrs):
        dc = self._getAttributeValue(attrs, 'w:dropCap') 
        if dc is not None:
            self.dropCap = True

#########################################################
#############      IMAGE/MEDIA       ####################
#########################################################
class Drawing(ContainerElement, TextElement):
    TAG = 'w:drawing'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        self._wrapper = None
        self._image = None
        self._float = None
        
    def appendElement(self, element):
        if type(element) is DrawingAnchor or DrawingInline:
            self._wrapper = element
        if type(element) is DrawingAnchor:
            if element.offset < 2600000:
                self._float = 'left'
            else:
                self._float = 'right'
    
    def getText(self):
        if self._image is not None:
            if self._float is not None:
                style = ' style="float:'+self._float+';"'
            else:
                style = ''
            return '<img src="'+self._image+'" alt=""'+style+' />'
        else:
            return ''
    
    def getPlainText(self):
        return '['+self._image+']'
    
    def postProcessTag(self):
        if self._wrapper is not None and self._wrapper.blip is not None and self._wrapper.blip.refId is not None:
            self._image = self.context.relations[self._wrapper.blip.refId].target


class DrawingWrapper(ContainerElement):
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        self.blip = None
        self.offset = 0
    
    def appendElement(self, element):
        if type(element) is ABlip:
            self.blip = element
        elif type(element) is PositionH:
            self.offset = element.offset

class DrawingInline(DrawingWrapper):
    TAG = 'wp:inline'
    
class DrawingAnchor(DrawingWrapper):
    TAG = 'wp:anchor'
    
class PositionH(ContainerElement):
    TAG = 'wp:positionH'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        self.offset = 0
    
    def appendElement(self, element):
        if type(element) is PosOffset:
            self.offset = element.offset

class PosOffset(TextElement):
    TAG = 'wp:posOffset'
    CHARACTERS = True
    
    def __init__(self, context):
        TextElement.__init__(self, context)
        self._text = ''
        self.offset = 0
        
    def processCharacters(self, content):
        self._text += content
        
    def postProcessTag(self):
        try:
            #print 'PosOffset: ', self._text
            self.offset = int(self._text)
        except:
            self.offset = 0
    
    
class ABlip(IndividualElement):
    TAG = 'a:blip'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.refId = None
        
    def preProcessTag(self, attrs):
        self.refId = self._getAttributeValue(attrs, 'r:embed')
        self.context.relations[self.refId].used = True

class Picture(ContainerElement, TextElement):
    TAG = 'w:pict'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        self._image = None
    
    def appendElement(self, element):
        if element.TAG == ImageData.TAG:
            self._rid = element.rid
            self._image = self.context.relations[element.rid].target
    
    def getText(self):
        if self._image is not None:
            return '<img src="'+self._image+'" alt="" />'
        else:
            return ''
    
    def getPlainText(self):
        if self._image is not None:
            return '['+self._image+']'
        else:
            return ''

class ImageData(IndividualElement):
    TAG = 'v:imagedata'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.rid = None
    
    def preProcessTag(self, attrs):
        self.rid = self._getAttributeValue(attrs, 'r:id')

#########################################################
#############        TABLES          ####################
#########################################################
class Table(ContainerElement, TextElement):
    TAG = 'w:tbl'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        
    def getText(self):
        text = '<table style="border-collapse: collapse; border: 1px solid black; width: 100%">'
        for t in self._elements:
            text += t.getText()
        return text+'</table>\n'
    
    def getPlainText(self):
        text = ''
        for t in self._elements:
            text += t.getPlainText()
        return text
    
    def appendElement(self, element):
        if type(element) is TableRow:
            ContainerElement.appendElement(self, element)
    
class TableRow(ContainerElement, TextElement):
    TAG = 'w:tr'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        
    def getText(self):
        text = '<tr>'
        for t in self._elements:
            text += t.getText()
        return text+'</tr>\n'
    
    def getPlainText(self):
        text = ''
        for t in self._elements:
            text += t.getPlainText()
        return text
    
    def appendElement(self, element):
        if type(element) is TableCell:
            ContainerElement.appendElement(self, element)

class TableCell(ContainerElement, TextElement):
    TAG = 'w:tc'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        
    def getText(self):
        text = '<td style="border-collapse: collapse; border: 1px solid black;">'
        for t in self._elements:
            text += t.getText()
        return text+'</td>\n'
    
    def getPlainText(self):
        text = ''
        for t in self._elements:
            text += t.getPlainText()
        return text
    
    def appendElement(self, element):
        if type(element) is Table or type(element) is WordParagraph:
            ContainerElement.appendElement(self, element)
    
#########################################################
#############       FOOTNOTES        ####################
#########################################################
class Footnote(ContainerElement, TextElement):
    TAG = 'w:footnote'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        self.id = None
        self.seq = None
        
    def getText(self):
        text = '<p style="text-indent: 0;"><sup><a href="'+self.context.footnoteAnchors[self.id]+'">{'+self.id+'}</a></sup></p>\n'
        for t in self._elements:
            text += t.getText()
        return text
    
    def getPlainText(self):
        text = '{'+self.id+'}\n'
        for t in self._elements:
            text += t.getPlainText()
        return text
    
    def preProcessTag(self, attrs):
        self.id = self._getAttributeValue(attrs, 'w:id')
        if self.id is not None and self.id in self.context.footnotes:
            self.seq = self.context.footnotes[self.id]

    
#########################################################
#############         LISTS          ####################
#########################################################
class ParagraphNumber(ContainerElement, TextElement):
    TAG = 'w:numPr'
    
    def __init__(self, context):
        ContainerElement.__init__(self, context)
        self.level = 0
        self.numId = 0
    
    def appendElement(self, element):
        if element.TAG == NumberIlvl.TAG:
            self.level = element.level
        elif element.TAG == NumberId.TAG:
            self.numId = element.numId
    
    def getText(self):
        if self.numId > 0:
            #print 'ParagraphNumber::level, numId:', self.level, self.numId
            #print 'ParagraphNumber:: self.context.listNums[self.numId].aNumId:', self.context.listNums[self.numId].aNumId
            return self.context.abstractNums[self.context.listNums[self.numId].aNumId].getText(self.numId, self.level)
        else:
            return ''
    
class NumberIlvl(IndividualElement):
    TAG = 'w:ilvl'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.level = 0
    
    def preProcessTag(self, attrs):
        self.level = int(self._getAttributeValue(attrs, 'w:val'))

    
class NumberId(IndividualElement):
    TAG = 'w:numId'
    
    def __init__(self, context):
        IndividualElement.__init__(self, context)
        self.numId = 0
    
    def preProcessTag(self, attrs):
        self.numId = int(self._getAttributeValue(attrs, 'w:val'))
