__license__ = 'GPL 3'
__copyright__ = '2012 Saulius P.'
__docformat__ = 'restructuredtext en'

import os
import time
from xml.sax import make_parser
from calibre.customize.conversion import InputFormatPlugin, OptionRecommendation
from calibre_plugins.docx_input_plugin.content_handlers.docx2html import Docx2Html
from calibre_plugins.docx_input_plugin.content_handlers.docx2style import Docx2Style
from calibre_plugins.docx_input_plugin.structures.context import DocxContext

EMBED_OPTIONS = [
         _('All'),
         _('DejaVu Serif'),
         _('None')
     ]

class DocxInput(InputFormatPlugin):
    name        = 'DOCX Input'
    author      = 'Saulius P.'
    description = 'Convert DOCX files (.docx) to OPF'
    version = (0, 0, 22)
    minimum_calibre_version = (0, 8, 63)
    file_types = set(['docx'])
    
    options = set([
        OptionRecommendation(name='first_cover', recommended_value=True,
            help=_('Cover from document.')),
        OptionRecommendation(name='skip_first_pg', recommended_value=False,
            help=_('Skip first page as a cover.')),
        OptionRecommendation(name='replace_bap', recommended_value=False,
            help=_('Enhance display compatibility and replace paragraph spacing before or after with empty paragraph.')),
        OptionRecommendation(name='font_embed', recommended_value=_('All'),
            level=OptionRecommendation.LOW, choices=EMBED_OPTIONS,
            help=_('Font embedding options.')),
        OptionRecommendation(name='substitute_normal_serif', recommended_value=False,
            help=_('Instead of font family for default (usually "Normal") style use "Serif". Useful for AZW3 (KF8).'))
    ])
    
    def convert(self, stream, options, file_ext, log, accelerators):
        start = time.time()
        print('DocxInput::convert::path: '+os.getcwdu())
        self.opts = options
        context = DocxContext(self.opts)
        #print 'DocxInput::options.embed: ', options.embed
        from calibre_plugins.docx_input_plugin.writers.html_writer import HtmlWriter
        htmlWriter = HtmlWriter(context)
        ######### Extract DOCX file ########
        self._unzipDocx(stream)
        ######### Parse styles from styles XML ########
        #sContext = StyleContext()
        self._parseStyles(os.path.join('word', 'styles.xml'), context)
        if os.path.isfile(os.path.join('word', 'stylesWithEffects.xml')):
            self._parseStyles(os.path.join('word', 'stylesWithEffects.xml'), context)
        ######### Parse relations, like images, etc. #######
        context.relations = self._parseRelations(os.path.join('word', '_rels', 'document.xml.rels'))
        ######### Parse numbering types if exists ########
        if os.path.isfile(os.path.join('word', 'numbering.xml')):
            self._parseNumbering(os.path.join('word', 'numbering.xml'), context)
            print 'Context abstractNums:', context.abstractNums
            print 'Context listNums:', context.listNums
        ######### Parse DOCX and write it out to HTML files ########
        self._parseDocx(os.path.join('word', 'document.xml'), htmlWriter, context)
        ######### Parse Footnotes and write it out to HTML files ########
        if len(context.footnotes) > 0:
            self._parseFootnotes(os.path.join('word', 'footnotes.xml'), htmlWriter, context)
        ######### Write down fonts for embedding ########
        #fontList = self._writeDownVerdanaFonts()
        context.fontList = self._writeDownFonts(context)
        ######### Write parsed files to CSS files ########
        self._writeCss('style', context)
        #####################
        print "DOCX parsing resulted in the font style list:\n", context.fontStyles
        ######### Write down OPF file with all information merged in #######
        context.cssList = ['style']
        opf = self._writeOpfFile('content', htmlWriter.getFilesWritten(), context)

        #time.sleep(20)
        #print('DocxInput::convert::sleeping 20 seconds')
        print 'DOCX INPUT took: ', time.time() - start
        return opf 
        
    def _unzipDocx(self, stream):
        from calibre.utils.zipfile import ZipFile
        zf = ZipFile(stream)
        zf.extractall(os.getcwdu())
        
    def _parseStyles(self, stylePath, context):
        styleParser = make_parser()
        docx2style = Docx2Style(context)
        styleParser.setContentHandler(docx2style)
        wsPath = os.path.abspath(stylePath)
        print("Word styles: "+wsPath)
        styleParser.parse(wsPath)
        #return docx2style.getStyles()
    
    def _parseDocx(self, docxPath, htmlWriter, context):
        parser = make_parser()
        docx2html = Docx2Html(htmlWriter, context, self.opts)
        parser.setContentHandler(docx2html)
        wdPath = os.path.abspath(docxPath)
        print("Word document: "+wdPath)
        parser.parse(wdPath)
    
    def _parseFootnotes(self, footPath, htmlWriter, context):
        from calibre_plugins.docx_input_plugin.content_handlers.footnote2html import Footnote2Html
        parser = make_parser()
        foot2html = Footnote2Html(htmlWriter, context, self.opts)
        parser.setContentHandler(foot2html)
        wdPath = os.path.abspath(footPath)
        print("Footnote document: "+wdPath)
        parser.parse(wdPath)
        
    def _writeCss(self, cssName, context):
        from calibre_plugins.docx_input_plugin.writers.css_writer import CssWriter
        cssWriter = CssWriter(cssName, context)
        cssWriter.write()
        
    def _writeDownVerdanaFonts(self):
        from calibre_plugins.docx_input_plugin.writers.verdana_writer import VerdanaWriter
        verdanaWriter = VerdanaWriter()
        return verdanaWriter.write()
    
    def _writeDownFonts(self, context):
        if self.opts.font_embed == EMBED_OPTIONS[0]:
            from calibre_plugins.docx_input_plugin.writers.font_writer import FontWriter
            writer = FontWriter(context)
            return writer.write()
        elif self.opts.font_embed == EMBED_OPTIONS[1]:
            from calibre_plugins.docx_input_plugin.writers.dejavu_writer import DejaVuSerifWriter
            writer = DejaVuSerifWriter()
            return writer.write()
        else:
            return None
    
    def _writeOpfFile(self, opfName, htmlList, context):
        from calibre_plugins.docx_input_plugin.writers.opf_writer import OpfWriter
        opfWriter = OpfWriter('content', htmlList, context)
        return opfWriter.write()
    
    def _parseRelations(self, relPath):
        from calibre_plugins.docx_input_plugin.content_handlers.relations_parser import RelationsParser
        relParser = RelationsParser()
        parser = make_parser()
        parser.setContentHandler(relParser)
        parser.parse(os.path.abspath(relPath))
        return relParser.getRelations()
    
    def _parseNumbering(self, numPath, context):
        from calibre_plugins.docx_input_plugin.content_handlers.numbering_parser import NumberingParser
        numParser = NumberingParser(context)
        parser = make_parser()
        parser.setContentHandler(numParser)
        parser.parse(os.path.abspath(numPath))
    
    def gui_configuration_widget(self, parent, get_option_by_name,
            get_option_help, db, book_id=None):
        from calibre_plugins.docx_input_plugin.gui.docx_input_opt import PluginWidget
        return PluginWidget(parent, get_option_by_name, get_option_help, db, book_id)
