#!/usr/bin/env python
# -*- coding: utf-8 -*-

# This is a python script. You need a Python interpreter to run it.
# For example, ActiveState Python, which exists for windows.
# This script should work under Python 2.5.x, 2.6.x or 2.7.x
#
# This script replaces the existing cover image in a Kindle/Mobipocket ebook
# with a new image.

# Changelog
#  1.00 - initial version
#  1.01 - Unicode support under Windows
#  1.02 - Tkinter interface added

__version__ = "1.02"

# The EOF record content.
EOF_RECORD = '0xe90x8e\r\n'

# The content of the section  that divides Mobi/KF8 ebooks.
K8_BOUNDARY = 'BOUNDARY'

import sys, struct, os, getopt


# Wrap a stream so that output gets flushed immediately
# and also make sure that any unicode strings get
# encoded using "replace" before writing them.
class SafeUnbuffered:
    def __init__(self, stream):
        self.stream = stream
        self.encoding = stream.encoding
        if self.encoding == None:
            self.encoding = "utf-8"
    def write(self, data):
        if isinstance(data,unicode):
            data = data.encode(self.encoding,"replace")
        self.stream.write(data)
        self.stream.flush()
    def __getattr__(self, attr):
        return getattr(self.stream, attr)

class kindlecoverException(Exception):
    pass


def unicode_argv():
    if sys.platform.startswith("win"):
        # Uses shell32.GetCommandLineArgvW to get sys.argv as a list of Unicode
        # strings.

        # Versions 2.x of Python don't support Unicode in sys.argv on
        # Windows, with the underlying Windows API instead replacing multi-byte
        # characters with '?'.


        from ctypes import POINTER, byref, cdll, c_int, windll
        from ctypes.wintypes import LPCWSTR, LPWSTR

        GetCommandLineW = cdll.kernel32.GetCommandLineW
        GetCommandLineW.argtypes = []
        GetCommandLineW.restype = LPCWSTR

        CommandLineToArgvW = windll.shell32.CommandLineToArgvW
        CommandLineToArgvW.argtypes = [LPCWSTR, POINTER(c_int)]
        CommandLineToArgvW.restype = POINTER(LPWSTR)

        cmd = GetCommandLineW()
        argc = c_int(0)
        argv = CommandLineToArgvW(cmd, byref(argc))
        if argc.value > 0:
            # Remove Python executable and commands if present
            start = argc.value - len(sys.argv)
            return [argv[i] for i in
                    xrange(start, argc.value)]
        return []
    else:
        argvencoding = sys.stdin.encoding
        if argvencoding == None:
            argvencoding = "utf-8"
        return [unicode(sys.argv[i],argvencoding) for i in xrange(len(sys.argv))]

class Sectionizer:
    def __init__(self, filename):
        self.data = open(filename, "rb").read()
        self.palmheader = self.data[:78]
        self.palmname = self.data[:32]
        self.ident = self.palmheader[0x3C:0x3C+8]
        self.num_sections, = struct.unpack_from('>H', self.palmheader, 76)
        self.filelength = len(self.data)
        self.sectionsdata = list(struct.unpack_from('>%dL' % (self.num_sections*2), self.data, 78) + (self.filelength, 0))
        self.sectionoffsets = list(self.sectionsdata[::2])
        return

    def loadSection(self, section):
        before, after = self.sectionoffsets[section:section+2]
        return self.data[before:after]

    def replaceSection(self,section,newdata):
        before, after = self.sectionoffsets[section:section+2]
        for i in range(section+1,self.num_sections+1):
            self.sectionoffsets[i] += len(newdata)-(after-before)
        self.sectionsdata[::2] = self.sectionoffsets

        self.data = self.data[:78] + struct.pack('>%dL' % (self.num_sections*2), *self.sectionsdata[:-2]) + self.data[78+self.num_sections*2*4:before]+newdata+self.data[after:]

    def saveTo(self,filename):
        open(filename, "wb").write(self.data)

def sortedHeaderKeys(mheader):
    hdrkeys = sorted(mheader.keys(), key=lambda akey: mheader[akey][0])
    return hdrkeys


class MobiHeader:
    # all values are packed in big endian format
    palmdoc_header = {
            "compression_type"  : (0x00, '>H', 2),
            "fill0"             : (0x02, '>H', 2),
            "text_length"       : (0x04, '>L', 4),
            "text_records"      : (0x08, '>H', 2),
            "max_section_size"  : (0x0a, '>H', 2),
            "read_pos   "       : (0x0c, '>L', 4),
    }

    mobi6_header = {
            "compression_type"  : (0x00, '>H', 2),
            "fill0"             : (0x02, '>H', 2),
            "text_length"       : (0x04, '>L', 4),
            "text_records"      : (0x08, '>H', 2),
            "max_section_size"  : (0x0a, '>H', 2),
            "crypto_type"       : (0x0c, '>H', 2),
            "fill1"             : (0x0e, '>H', 2),
            "magic"             : (0x10, '4s', 4),
            "header_length (from MOBI)"     : (0x14, '>L', 4),
            "type"              : (0x18, '>L', 4),
            "codepage"          : (0x1c, '>L', 4),
            "unique_id"         : (0x20, '>L', 4),
            "version"           : (0x24, '>L', 4),
            "metaorthindex"     : (0x28, '>L', 4),
            "metainflindex"     : (0x2c, '>L', 4),
            "index_names"       : (0x30, '>L', 4),
            "index_keys"        : (0x34, '>L', 4),
            "extra_index0"      : (0x38, '>L', 4),
            "extra_index1"      : (0x3c, '>L', 4),
            "extra_index2"      : (0x40, '>L', 4),
            "extra_index3"      : (0x44, '>L', 4),
            "extra_index4"      : (0x48, '>L', 4),
            "extra_index5"      : (0x4c, '>L', 4),
            "first_nontext"     : (0x50, '>L', 4),
            "title_offset"      : (0x54, '>L', 4),
            "title_length"      : (0x58, '>L', 4),
            "language_code"     : (0x5c, '>L', 4),
            "dict_in_lang"      : (0x60, '>L', 4),
            "dict_out_lang"     : (0x64, '>L', 4),
            "min_version"       : (0x68, '>L', 4),
            "first_resc_offset" : (0x6c, '>L', 4),
            "huff_offset"       : (0x70, '>L', 4),
            "huff_num"          : (0x74, '>L', 4),
            "huff_tbl_offset"   : (0x78, '>L', 4),
            "huff_tbl_len"      : (0x7c, '>L', 4),
            "exth_flags"        : (0x80, '>L', 4),
            "fill3_a"           : (0x84, '>L', 4),
            "fill3_b"           : (0x88, '>L', 4),
            "fill3_c"           : (0x8c, '>L', 4),
            "fill3_d"           : (0x90, '>L', 4),
            "fill3_e"           : (0x94, '>L', 4),
            "fill3_f"           : (0x98, '>L', 4),
            "fill3_g"           : (0x9c, '>L', 4),
            "fill3_h"           : (0xa0, '>L', 4),
            "unknown0"          : (0xa4, '>L', 4),
            "drm_offset"        : (0xa8, '>L', 4),
            "drm_count"         : (0xac, '>L', 4),
            "drm_size"          : (0xb0, '>L', 4),
            "drm_flags"         : (0xb4, '>L', 4),
            "fill4_a"           : (0xb8, '>L', 4),
            "fill4_b"           : (0xbc, '>L', 4),
            "first_content"     : (0xc0, '>H', 2),
            "last_content"      : (0xc2, '>H', 2),
            "unknown0"          : (0xc4, '>L', 4),
            "fcis_offset"       : (0xc8, '>L', 4),
            "fcis_count"        : (0xcc, '>L', 4),
            "flis_offset"       : (0xd0, '>L', 4),
            "flis_count"        : (0xd4, '>L', 4),
            "unknown1"          : (0xd8, '>L', 4),
            "unknown2"          : (0xdc, '>L', 4),
            "srcs_offset"       : (0xe0, '>L', 4),
            "srcs_count"        : (0xe4, '>L', 4),
            "unknown3"          : (0xe8, '>L', 4),
            "unknown4"          : (0xec, '>L', 4),
            "fill5"             : (0xf0, '>H', 2),
            "traildata_flags"   : (0xf2, '>H', 2),
            "ncx_index"         : (0xf4, '>L', 4),
            "unknown5"          : (0xf8, '>L', 4),
            "unknown6"          : (0xfc, '>L', 4),
            "datp_offset"       : (0x100, '>L', 4),
            "unknown7"          : (0x104, '>L', 4),
            "Unknown    "       : (0x108, '>L', 4),
            "Unknown    "       : (0x10C, '>L', 4),
            "Unknown    "       : (0x110, '>L', 4),
            "Unknown    "       : (0x114, '>L', 4),
            "Unknown    "       : (0x118, '>L', 4),
            "Unknown    "       : (0x11C, '>L', 4),
            "Unknown    "       : (0x120, '>L', 4),
            "Unknown    "       : (0x124, '>L', 4),
            "Unknown    "       : (0x128, '>L', 4),
            "Unknown    "       : (0x12C, '>L', 4),
            "Unknown    "       : (0x130, '>L', 4),
            "Unknown    "       : (0x134, '>L', 4),
            "Unknown    "       : (0x138, '>L', 4),
            "Unknown    "       : (0x11C, '>L', 4),
            }

    mobi8_header = {
            "compression_type"  : (0x00, '>H', 2),
            "fill0"             : (0x02, '>H', 2),
            "text_length"       : (0x04, '>L', 4),
            "text_records"      : (0x08, '>H', 2),
            "max_section_size"  : (0x0a, '>H', 2),
            "crypto_type"       : (0x0c, '>H', 2),
            "fill1"             : (0x0e, '>H', 2),
            "magic"             : (0x10, '4s', 4),
            "header_length (from MOBI)"     : (0x14, '>L', 4),
            "type"              : (0x18, '>L', 4),
            "codepage"          : (0x1c, '>L', 4),
            "unique_id"         : (0x20, '>L', 4),
            "version"           : (0x24, '>L', 4),
            "metaorthindex"     : (0x28, '>L', 4),
            "metainflindex"     : (0x2c, '>L', 4),
            "index_names"       : (0x30, '>L', 4),
            "index_keys"        : (0x34, '>L', 4),
            "extra_index0"      : (0x38, '>L', 4),
            "extra_index1"      : (0x3c, '>L', 4),
            "extra_index2"      : (0x40, '>L', 4),
            "extra_index3"      : (0x44, '>L', 4),
            "extra_index4"      : (0x48, '>L', 4),
            "extra_index5"      : (0x4c, '>L', 4),
            "first_nontext"     : (0x50, '>L', 4),
            "title_offset"      : (0x54, '>L', 4),
            "title_length"      : (0x58, '>L', 4),
            "language_code"     : (0x5c, '>L', 4),
            "dict_in_lang"      : (0x60, '>L', 4),
            "dict_out_lang"     : (0x64, '>L', 4),
            "min_version"       : (0x68, '>L', 4),
            "first_resc_offset" : (0x6c, '>L', 4),
            "huff_offset"       : (0x70, '>L', 4),
            "huff_num"          : (0x74, '>L', 4),
            "huff_tbl_offset"   : (0x78, '>L', 4),
            "huff_tbl_len"      : (0x7c, '>L', 4),
            "exth_flags"        : (0x80, '>L', 4),
            "fill3_a"           : (0x84, '>L', 4),
            "fill3_b"           : (0x88, '>L', 4),
            "fill3_c"           : (0x8c, '>L', 4),
            "fill3_d"           : (0x90, '>L', 4),
            "fill3_e"           : (0x94, '>L', 4),
            "fill3_f"           : (0x98, '>L', 4),
            "fill3_g"           : (0x9c, '>L', 4),
            "fill3_h"           : (0xa0, '>L', 4),
            "unknown0"          : (0xa4, '>L', 4),
            "drm_offset"        : (0xa8, '>L', 4),
            "drm_count"         : (0xac, '>L', 4),
            "drm_size"          : (0xb0, '>L', 4),
            "drm_flags"         : (0xb4, '>L', 4),
            "fill4_a"           : (0xb8, '>L', 4),
            "fill4_b"           : (0xbc, '>L', 4),
            "fdst_offset"       : (0xc0, '>L', 4),
            "fdst_flow_count"   : (0xc4, '>L', 4),
            "fcis_offset"       : (0xc8, '>L', 4),
            "fcis_count"        : (0xcc, '>L', 4),
            "flis_offset"       : (0xd0, '>L', 4),
            "flis_count"        : (0xd4, '>L', 4),
            "unknown1"          : (0xd8, '>L', 4),
            "unknown2"          : (0xdc, '>L', 4),
            "srcs_offset"       : (0xe0, '>L', 4),
            "srcs_count"        : (0xe4, '>L', 4),
            "unknown3"          : (0xe8, '>L', 4),
            "unknown4"          : (0xec, '>L', 4),
            "fill5"             : (0xf0, '>H', 2),
            "traildata_flags"   : (0xf2, '>H', 2),
            "ncx_index"         : (0xf4, '>L', 4),
            "fragment_index"    : (0xf8, '>L', 4),
            "skeleton_index"    : (0xfc, '>L', 4),
            "datp_offset"       : (0x100, '>L', 4),
            "guide_index"       : (0x104, '>L', 4),
            "Unknown    "       : (0x108, '>L', 4),
            "Unknown    "       : (0x10C, '>L', 4),
            "Unknown    "       : (0x110, '>L', 4),
            "Unknown    "       : (0x114, '>L', 4),
            "Unknown    "       : (0x118, '>L', 4),
            "Unknown    "       : (0x11C, '>L', 4),
            "Unknown    "       : (0x120, '>L', 4),
            "Unknown    "       : (0x124, '>L', 4),
            "Unknown    "       : (0x128, '>L', 4),
            "Unknown    "       : (0x12C, '>L', 4),
            "Unknown    "       : (0x130, '>L', 4),
            "Unknown    "       : (0x134, '>L', 4),
            "Unknown    "       : (0x138, '>L', 4),
            "Unknown    "       : (0x11C, '>L', 4),
            }

    palmdoc_header_sorted_keys = sortedHeaderKeys(palmdoc_header)
    mobi6_header_sorted_keys = sortedHeaderKeys(mobi6_header)
    mobi8_header_sorted_keys = sortedHeaderKeys(mobi8_header)

    id_map_strings = {
        1 : "Drm Server Id",
        2 : "Drm Commerce Id",
        3 : "Drm Ebookbase Book Id",
        100 : "Creator",
        101 : "Publisher",
        102 : "Imprint",
        103 : "Description",
        104 : "ISBN",
        105 : "Subject",
        106 : "Published",
        107 : "Review",
        108 : "Contributor",
        109 : "Rights",
        110 : "SubjectCode",
        111 : "Type",
        112 : "Source",
        113 : "ASIN",
        114 : "versionNumber",
        117 : "Adult",
        118 : "Price",
        119 : "Currency",
        122 : "fixed-layout",
        123 : "book-type",
        124 : "orientation-lock",
        126 : "original-resolution",
        127 : "zero-gutter",
        128 : "zero-margin",
        129 : "K8(129)_Masthead/Cover_Image",
        132 : "RegionMagnification",
        200 : "DictShortName",
        208 : "Watermark",
        501 : "Document Type",
        502 : "last_update_time",
        503 : "Updated_Title",
        504 : "ASIN_(504)",
        524 : "Language_(524)",
        525 : "TextDirection",
        528 : "Unknown_Logical_Value_(528)",
        535 : "Kindlegen_BuildRev_Number",

    }
    id_map_values = {
        115 : "sample",
        116 : "StartOffset",
        121 : "K8(121)_Boundary_Section",
        125 : "K8(125)_Count_of_Resources_Fonts_Images",
        131 : "K8(131)_Unidentified_Count",
        201 : "CoverOffset",
        202 : "ThumbOffset",
        203 : "Has Fake Cover",
        204 : "Creator Software",
        205 : "Creator Major Version",
        206 : "Creator Minor Version",
        207 : "Creator Build Number",
        401 : "Clipping Limit",
        402 : "Publisher Limit",
        404 : "Text to Speech Disabled",
    }
    id_map_hexstrings = {
        209 : "Tamper Proof Keys (hex)",
        300 : "Font Signature (hex)",
        403 : "Unknown",
        405 : "Unknown",
        406 : "Unknown",
        403 : "Unknown",
        450 : "Unknown",
        451 : "Unknown",
        452 : "Unknown",
        453 : "Unknown",

    }

    def __init__(self, sect, sectNumber):
        self.sect = sect
        self.start = sectNumber
        self.header = self.sect.loadSection(self.start)
        if len(self.header)>20 and self.header[16:20] == 'MOBI':
            self.palm = False
        elif self.sect.ident == 'TEXtREAd':
            self.palm = True
        else:
            raise kindlecoverException("Unknown File Format")

        self.records, = struct.unpack_from('>H', self.header, 0x8)

        # set defaults in case this is a PalmDOC
        self.title = self.sect.palmname
        self.length = len(self.header)-16
        self.type = 3
        self.codepage = 1252
        self.codec = "windows-1252"
        self.unique_id = 0
        self.version = 0
        self.hasExth = False
        self.exth = ''
        self.exth_offset = self.length + 16
        self.exth_length = 0
        self.crypto_type = 0
        self.firstnontext = self.start+self.records + 1
        self.firstresource = self.start+self.records + 1
        self.ncxidx = 0xffffffff
        self.metaOrthIndex = 0xffffffff
        self.metaInflIndex = 0xffffffff
        self.skelidx = 0xffffffff
        self.dividx = 0xffffffff
        self.othidx = 0xfffffff
        self.fdst = 0xffffffff
        self.mlstart = self.sect.loadSection(self.start+1)[:4]


        if self.palm:
            return

        self.length, self.type, self.codepage, self.unique_id, self.version = struct.unpack('>LLLLL', self.header[20:40])
        codec_map = {
            1252 : "windows-1252",
            65001: "utf-8",
        }
        if self.codepage in codec_map.keys():
            self.codec = codec_map[self.codepage]

        exth_flag, = struct.unpack('>L', self.header[0x80:0x84])
        self.hasExth = exth_flag & 0x40
        self.exth = ''
        self.exth_offset = self.length + 16
        self.exth_length = 0
        if self.hasExth:
            self.exth_length, = struct.unpack_from('>L', self.header, self.exth_offset+4)
            self.exth_length = ((self.exth_length + 3)>>2)<<2 # round to next 4 byte boundary
            self.exth = self.header[self.exth_offset:self.exth_offset+self.exth_length]

        self.crypto_type, = struct.unpack_from('>H', self.header, 0xC)

        # Start sector for additional files such as images, fonts, resources, etc
        self.firstresource, = struct.unpack_from('>L', self.header, 0x6C)
        if self.firstresource != 0xffffffff:
            self.firstresource += self.start

    def isEncrypted(self):
        return self.crypto_type != 0

    def getMetaData(self):

        def addValue(name, value):
            if name not in self.metadata:
                self.metadata[name] = [value]
            else:
                self.metadata[name].append(value)

        self.metadata = {}
        codec=self.codec
        if self.hasExth:
            extheader=self.exth
            _length, num_items = struct.unpack('>LL', extheader[4:12])
            extheader = extheader[12:]
            pos = 0
            for _ in range(num_items):
                id, size = struct.unpack('>LL', extheader[pos:pos+8])
                content = extheader[pos + 8: pos + size]
                if id in MobiHeader.id_map_strings.keys():
                    name = MobiHeader.id_map_strings[id]
                    addValue(name, unicode(content, codec).encode("utf-8"))
                elif id in MobiHeader.id_map_values.keys():
                    name = MobiHeader.id_map_values[id]
                    if size == 9:
                        value, = struct.unpack('B',content)
                        addValue(name, str(value))
                    elif size == 10:
                        value, = struct.unpack('>H',content)
                        addValue(name, str(value))
                    elif size == 12:
                        value, = struct.unpack('>L',content)
                        addValue(name, str(value))
                    else:
                        addValue(name, content.encode("hex"))
                elif id in MobiHeader.id_map_hexstrings.keys():
                    name = MobiHeader.id_map_hexstrings[id]
                    addValue(name, content.encode("hex"))
                else:
                    name = "(0:d) (hex)".format(id)
                    addValue(name, content.encode("hex"))
                pos += size
        return self.metadata



def replacecoverimages(infile, coverfile, thumbfile, outfile, cli = True):
    # process the PalmDoc database header and verify it is a mobi
    sect = Sectionizer(infile)
    if sect.ident != 'BOOKMOBI' and sect.ident != 'TEXtREAd':
        raise kindlecoverException("Invalid file format")
    # we only need to consider the first header, even in a compound file
    mhlst = []
    mh = MobiHeader(sect,0)
    if mh.isEncrypted():
        raise kindlecoverException("Book is encrypted")

    metadata = mh.getMetaData()
    coversection = 0
    thumbsection = 0
    if coverfile != "":
        if "CoverOffset" not in metadata:
            raise kindlecoverException("Can't find existing cover image to replace.")
        if len(metadata["CoverOffset"]) > 1:
            raise kindlecoverException("More than one existing cover image.")
        coversection = mh.firstresource + int(metadata["CoverOffset"][0])
        coverdata = open(coverfile, "rb").read()
        if len(coverdata)<10:
            raise kindlecoverException("Replacement cover image too small")
        if coverdata[:4] != 'GIF8' and coverdata[6:10] != 'JFIF':
            raise kindlecoverException("Replacement cover image must be GIF or JPEG")
        if len(coverdata)>127*1024:
            raise kindlecoverException("Replacement cover image too big")
        if cli:
            print u"Replacing cover (section #{0}) with \"{1}\"".format(coversection, os.path.basename(coverfile))
        sect.replaceSection(coversection,coverdata)



    if thumbfile != "":
        if "ThumbOffset" not in metadata:
            raise kindlecoverException("Can't find existing thumbnail image to replace.")
        if len(metadata["ThumbOffset"]) > 1:
            raise kindlecoverException("More than one existing thumbnail image.")
        thumbsection = mh.firstresource + int(metadata["ThumbOffset"][0])
        thumbdata = open(thumbfile, "rb").read()
        if len(thumbdata)<10:
            raise kindlecoverException("Replacement thumbnail image too small")
        if thumbdata[:4] != 'GIF8' and thumbdata[6:10] != 'JFIF':
            raise kindlecoverException("Replacement thumbnail image must be GIF or JPEG")
        if len(thumbdata)>127*1024:
            raise kindlecoverException("Replacement thumbnail image too big")
        if cli:
            print u"Replacing thumbnail (section #{0}) with \"{1}\"".format(thumbsection, os.path.basename(thumbfile))
        sect.replaceSection(thumbsection,thumbdata)

    sect.saveTo(outfile)
    return


def usage(progname):
    print u""
    print u"Description:"
    print u"  Creates a new Kindle/Mobpocket ebook file, replaceing the existing"
    print u"  cover and thumbnail images in the original ebook."
    print u"Usage:"
    print u"  {0:s} [-c newcoverimage] [-t newthumbnailimage] infile outfile".format(progname)


def cli_main(argv=unicode_argv()):
    print u"kindlecover v{0:s}.".format(__version__)
    print u"   Copyright © 2012 Paul Durrant <paul@durrantco.uk>"
    print u"   Adapted in part from MobiUnpack."
    print u"   This program is free software: you can redistribute it and/or modify"
    print u"   it under the terms of the GNU General Public License as published by"
    print u"   the Free Software Foundation, version 3."

    progname = os.path.basename(argv[0])
    try:
        opts, args = getopt.getopt(argv[1:], "c:ht:")
    except getopt.GetoptError, err:
        print u"Error: {0}".format(err)
        usage(progname)
        sys.exit(2)

    if len(args)!=2:
        usage(progname)
        sys.exit(2)

    coverfile = u""
    thumbfile = u""
    for o, a in opts:
        if o == "-c":
            coverfile = a
        if o == "-t":
            thumbfile = a
        if o == "-h":
            usage(progname)
            sys.exit(0)

    infile, outfile = args

    infileext = os.path.splitext(infile)[1].upper()
    if infileext not in ['.MOBI', '.PRC', '.AZW', '.AZW3']:
        print u"Error: the input file must be a Kindle/Mobipocket ebook."
        return 1

    if coverfile=="" and thumbfile=="":
        print u"Error: at least one of cover image and thumbnail image must be specified"
        return 1

    if os.path.exists(outfile):
        print u"Error: the output file must not already exist."
        return 1


    try:
        replacecoverimages(infile, coverfile, thumbfile, outfile)
        print u"Completed."

    except Exception, e:
        print u"Error: {0}".format(e)
        return 1

    return 0

def gui_main():
    import Tkinter
    import Tkconstants
    import tkFileDialog

    class CoverChangeDialog(Tkinter.Frame):
        def __init__(self, root):
            Tkinter.Frame.__init__(self, root, border=5)
            self.status = Tkinter.Label(self, text="Select New Cover Files and Book File to Change")
            self.status.pack(fill=Tkconstants.X, expand=1)
            body = Tkinter.Frame(self)
            body.pack(fill=Tkconstants.X, expand=1)
            sticky = Tkconstants.E + Tkconstants.W
            body.grid_columnconfigure(1, weight=2)
            Tkinter.Label(body, text="Cover file").grid(row=0)
            self.coverpath = Tkinter.Entry(body, width=30)
            self.coverpath.grid(row=0, column=1, sticky=sticky)
            button = Tkinter.Button(body, text="...", command=self.get_coverpath)
            button.grid(row=0, column=2)
            Tkinter.Label(body, text="Thumbnail file").grid(row=1)
            self.thumbpath = Tkinter.Entry(body, width=30)
            self.thumbpath.grid(row=1, column=1, sticky=sticky)
            button = Tkinter.Button(body, text="...", command=self.get_thumbpath)
            button.grid(row=1, column=2)
            Tkinter.Label(body, text="Input file").grid(row=2)
            self.inpath = Tkinter.Entry(body, width=30)
            self.inpath.grid(row=2, column=1, sticky=sticky)
            button = Tkinter.Button(body, text="...", command=self.get_inpath)
            button.grid(row=2, column=2)
            Tkinter.Label(body, text="Output file").grid(row=3)
            self.outpath = Tkinter.Entry(body, width=30)
            self.outpath.grid(row=3, column=1, sticky=sticky)
            button = Tkinter.Button(body, text="...", command=self.get_outpath)
            button.grid(row=3, column=2)
            buttons = Tkinter.Frame(self)
            buttons.pack()
            botton = Tkinter.Button(
                buttons, text="Change Cover", width=20, command=self.changecover)
            botton.pack(side=Tkconstants.LEFT)
            Tkinter.Frame(buttons, width=10).pack(side=Tkconstants.LEFT)
            button = Tkinter.Button(
                buttons, text="Quit", width=10, command=self.quit)
            button.pack(side=Tkconstants.RIGHT)

        def get_coverpath(self):
            coverpath = tkFileDialog.askopenfilename(
                parent=None, title="Select a cover image file",
                defaultextension=".jpeg",
                filetypes=[("Image Files", (".jpeg", ".jpg", ".gif")),
                           ("Image Files", [],"JPEG"),
                           ("Image Files", [],"GIFf")])
            if coverpath:
                coverpath = os.path.normpath(coverpath)
                self.coverpath.delete(0, Tkconstants.END)
                self.coverpath.insert(0, coverpath)
            return

        def get_thumbpath(self):
            thumbpath = tkFileDialog.askopenfilename(
                parent=None, title="Select a thumbnail image file",
                defaultextension=".jpeg",
                filetypes=[("Image Files", (".jpeg", ".jpg", ".gif")),
                           ("Image Files", [],"JPEG"),
                           ("Image Files", [],"GIFf")])
            if thumbpath:
                thumbpath = os.path.normpath(thumbpath)
                self.thumbpath.delete(0, Tkconstants.END)
                self.thumbpath.insert(0, thumbpath)
            return

        def get_inpath(self):
            inpath = tkFileDialog.askopenfilename(
                parent=None, title="Select a Kindle/Mobipocket file",
                defaultextension=".mobi", filetypes=[("Kindle/Mobipocket files", (".mobi",".azw",".azw3",".prc"))])
            if inpath:
                inpath = os.path.normpath(inpath)
                self.inpath.delete(0, Tkconstants.END)
                self.inpath.insert(0, inpath)
            return

        def get_outpath(self):
            outpath = tkFileDialog.asksaveasfilename(
                parent=None, title="Save the new Kindle/Mobipocket file as",
                defaultextension=".mobi", filetypes=[("Kindle/Mobipocket files", (".mobi"))])
            if outpath:
                outpath = os.path.normpath(outpath)
                self.outpath.delete(0, Tkconstants.END)
                self.outpath.insert(0, outpath)
            return

        def changecover(self):
            coverpath = self.coverpath.get()
            thumbpath = self.thumbpath.get()
            inpath = self.inpath.get()
            outpath = self.outpath.get()
            if coverpath and not os.path.exists(coverpath):
                self.status["text"] = u"Specified cover file does not exist"
                return
            if thumbpath and not os.path.exists(thumbpath):
                self.status["text"] = u"Specified thumbnail file does not exist"
                return
            if not thumbpath and not coverpath:
                self.status["text"] = u"You must choose a cover and/or a thumbnail file"
                return
            if not inpath:
                self.status["text"] = u"Input file not specified"
                return
            if not os.path.exists(inpath):
                self.status["text"] = u"Specified input file does not exist"
                return
            if not outpath:
                self.status["text"] = u"Output file not specified"
                return
            if inpath == outpath:
                self.status["text"] = u"Must have different input and output files"
                return
            self.status["text"] = u"Changing Cover..."
            try:
                replacecoverimages(inpath, coverpath, thumbpath, outpath, False)
            except Exception, e:
                self.status["text"] = u"Error; {0}".format(e)
                return
            if thumbpath and coverpath:
                self.status["text"] = u"Cover and Thumbail successfully replaced"
            elif thumbpath:
                self.status["text"] = u"Thumbail successfully replaced"
            else:
                self.status["text"] = u"Cover successfully replaced"

    root = Tkinter.Tk()
    root.title("KindleCover {0:s}".format(__version__))
    root.resizable(True, False)
    root.minsize(300, 0)
    CoverChangeDialog(root).pack(fill=Tkconstants.X, expand=1)
    root.mainloop()
    return 0


if __name__ == "__main__":
    sys.stdout=SafeUnbuffered(sys.stdout)
    if len(sys.argv) > 1:
        sys.exit(cli_main())
    sys.exit(gui_main())
