#!/bin/bash

function transsize ()
{
  cat                   # well, transsize doesn't seem to do much
}

function hex2bin ()
{
  while read -n 2 X; do
    [ -z "$X" ] && continue
    echo -e "\\\\\\\x$X"
  done | \
  while read HEX; do
    echo -n -e $HEX
  done
}

function get_enc_key ()
{
  [ -z "$TEST" ] && echo "Reading encoding keys:"

  ENC_KEY_US1=$(hexdump -s $((0x10000)) -C -n 16 Info.us | head -1 | awk '{ print $2$3$4$5$6$7$8$9$10$11$12$13$14$15$16$17 }')
  ENC_KEY_EU1=$(hexdump -s $((0x10000)) -C -n 16 Info.eu | head -1 | awk '{ print $2$3$4$5$6$7$8$9$10$11$12$13$14$15$16$17 }')
  ENC_KEY_RU1=$(hexdump -s $((0x10000)) -C -n 16 Info.ru | head -1 | awk '{ print $2$3$4$5$6$7$8$9$10$11$12$13$14$15$16$17 }')
  ENC_KEY_US2=$(hexdump -s $((0x10010)) -C -n 16 Info.us | head -1 | awk '{ print $2$3$4$5$6$7$8$9$10$11$12$13$14$15$16$17 }')
  ENC_KEY_EU2=$(hexdump -s $((0x10010)) -C -n 16 Info.eu | head -1 | awk '{ print $2$3$4$5$6$7$8$9$10$11$12$13$14$15$16$17 }')
  ENC_KEY_RU2=$(hexdump -s $((0x10010)) -C -n 16 Info.ru | head -1 | awk '{ print $2$3$4$5$6$7$8$9$10$11$12$13$14$15$16$17 }')
  IV1=$(hexdump -s $((0x10020)) -C -n 16 Info.us | head -1 | awk '{ print $2$3$4$5$6$7$8$9$10$11$12$13$14$15$16$17 }')
  IV2=$(hexdump -s $((0x10030)) -C -n 16 Info.us | head -1 | awk '{ print $2$3$4$5$6$7$8$9$10$11$12$13$14$15$16$17 }')

  dd if=Info.us of=/dev/stdout bs=1K skip=1 count=1 2> /dev/null > /tmp/$ME.id_rsa
  dd if=Info.us of=/dev/stdout bs=1K skip=$((0x10400 / 1024)) count=1 2> /dev/null > /tmp/$ME.id_rsa.pub-us
  dd if=Info.eu of=/dev/stdout bs=1K skip=$((0x10400 / 1024)) count=1 2> /dev/null > /tmp/$ME.id_rsa.pub-eu
  dd if=Info.ru of=/dev/stdout bs=1K skip=$((0x10400 / 1024)) count=1 2> /dev/null > /tmp/$ME.id_rsa.pub-ru

  cat > /tmp/$ME.id_rsa.pub-p << EOF
-----BEGIN PUBLIC KEY-----
MIIBIzANBgkqhkiG9w0BAQEFAAOCARAAMIIBCwKCAQIApIfTEfDboajJWnaXXiIh
wei+bZ2iO55eXUeuEHRhrgjV2tiFT1Ca75oAMp8sXw4Trf1BVpPRI7w0wlREu1/g
dPfwtSCbaqyv2VJay8yTrIwG0uXNiiMKBevopv0MdO6sCdIAmYjzRfYBIr+EsveL
4QPqptm4cuSJP0nDuqIGbH51q0/y0mhwyXKDXiYsdeqavb/3MzsDuPUuHGRRGdq1
P4tG30tJslsUAk5WXAaYJ/ZVsGS3XPimwLGUCNkQ9eZEjwwjvr9e9Pg5rZX/yOJz
Y/pksSfx51hb7NS692g9mbPzNPsEq1ALxkkzysfewv4jy3Oae14fr5Pv84u4t/qR
GqkCAwEAAQ==
-----END PUBLIC KEY-----
EOF

  if [ -z "$TEST" ]; then
    echo "  key1 (US)             = $ENC_KEY_US1"
    echo "  key1 (EU)             = $ENC_KEY_EU1"
    echo "  key1 (RU)             = $ENC_KEY_RU1"
    echo "  iv1                   = $IV1"
    echo "  key2 (US)             = $ENC_KEY_US2"
    echo "  key2 (EU)             = $ENC_KEY_EU2"
    echo "  key2 (RU)             = $ENC_KEY_RU2"
    echo "  iv2                   = $IV2"
    echo "  private key           = (in /tmp/$ME.id_rsa)"
    echo "  public key (US)       = (in /tmp/$ME.id_rsa.pub-us)"
    echo "  public key (EU)       = (in /tmp/$ME.id_rsa.pub-eu)"
    echo "  public key (RU)       = (in /tmp/$ME.id_rsa.pub-ru)"
    echo "  public key (porkupan) = (in /tmp/$ME.id_rsa.pub-p)"
  fi
}

function unpack_package_info_sha1 ()
{
  PKG_FROM=Sony
  PKG_FOR=US
  ENC_KEY1=$ENC_KEY_US1
  ENC_KEY2=$ENC_KEY_US2
  ID_RSA_PUB=/tmp/$ME.id_rsa.pub-us

  tail -c 1024 "$1" | head -c 272 | openssl aes-128-cbc -d -K $ENC_KEY1 -iv $IV1 2> /dev/null | openssl rsautl -verify -pubin -inkey $ID_RSA_PUB > /tmp/$ME.sha1.verify 2> /dev/null

  if [ $? -ne 0 ]; then
    PKG_FROM=Porkupan
    tail -c 1024 "$1" | head -c 272 | openssl aes-128-cbc -d -K $ENC_KEY1 -iv $IV1 2> /dev/null | openssl rsautl -verify -pubin -inkey /tmp/$ME.id_rsa.pub-p > /tmp/$ME.sha1.verify 2> /dev/null
  else
    return 0
  fi

  if [ $? -ne 0 ]; then
    PKG_FROM=Sony
    PKG_FOR=EU
    ENC_KEY1=$ENC_KEY_EU1
    ENC_KEY2=$ENC_KEY_EU2
    ID_RSA_PUB=/tmp/$ME.id_rsa.pub-eu
    tail -c 1024 "$1" | head -c 272 | openssl aes-128-cbc -d -K $ENC_KEY1 -iv $IV1 2> /dev/null | openssl rsautl -verify -pubin -inkey $ID_RSA_PUB > /tmp/$ME.sha1.verify 2> /dev/null
  else
    return 0
  fi

  if [ $? -ne 0 ]; then
    PKG_FROM=Porkupan
    tail -c 1024 "$1" | head -c 272 | openssl aes-128-cbc -d -K $ENC_KEY1 -iv $IV1 2> /dev/null | openssl rsautl -verify -pubin -inkey /tmp/$ME.id_rsa.pub-p > /tmp/$ME.sha1.verify 2> /dev/null
  else
    return 0
  fi

  if [ $? -ne 0 ]; then
    PKG_FROM=Sony
    PKG_FOR=RU
    ENC_KEY1=$ENC_KEY_RU1
    ENC_KEY2=$ENC_KEY_RU2
    ID_RSA_PUB=/tmp/$ME.id_rsa.pub-ru
    tail -c 1024 "$1" | head -c 272 | openssl aes-128-cbc -d -K $ENC_KEY1 -iv $IV1 2> /dev/null | openssl rsautl -verify -pubin -inkey $ID_RSA_PUB > /tmp/$ME.sha1.verify 2> /dev/null
  else
    return 0
  fi

  if [ $? -ne 0 ]; then
    PKG_FROM=Porkupan
    tail -c 1024 "$1" | head -c 272 | openssl aes-128-cbc -d -K $ENC_KEY1 -iv $IV1 | openssl rsautl -verify -pubin -inkey /tmp/$ME.id_rsa.pub-p > /tmp/$ME.sha1.verify
  else
    return 0
  fi

  if [ $? -ne 0 ]; then
    PKG_FROM=nobody
    PKG_FOR=invalid
  fi
}

function unpack_package_info ()
{
  [ -z "$TEST" ] && echo "Reading $1 info:"

  unpack_package_info_sha1 "$1"
  ret=$?

  if [ "$TEST" ]; then
    return $ret
  fi

  if [ $ret -ne 0 ]; then
    echo "Package signature error!"
    return 1
  fi

  dd if="$1" of=/dev/stdout bs=1K count=$((($(stat -c "%s" "$1") - 1024) / 1024)) 2> /dev/null | openssl sha1 -binary > /tmp/$ME.sha1.created

  if ! cmp /tmp/$ME.sha1.created /tmp/$ME.sha1.verify; then
    echo "Package sha1 error!"
    return 1
  fi

  dd if="$1" of=/dev/stdout bs=1 count=48 2> /dev/null | openssl aes-128-cbc -d -K $ENC_KEY1 -iv $IV1 > /tmp/$ME.header

  if [ $? -ne 0 ]; then
    echo "Package header decryption error!"
    return 1
  fi

  SIZE=$(((($(hexdump -C -s 16 -n 4 /tmp/$ME.header | head -1 | awk '{ print "0x"$2$3$4$5 }') + 16) / 16) * 16))
  CHECKSUM=$(tail -c 12 /tmp/$ME.header | sed "s: ::g")

  echo "  size        = $SIZE"
  echo "  checksum    = $CHECKSUM"

  dd if="$1" of=/dev/stdout bs=1 skip=64 count=$SIZE 2> /dev/null | openssl aes-128-cbc -d -K $ENC_KEY2 -iv $IV2 > /tmp/$ME.info

  if [ $? -ne 0 ]; then
    echo "Package info decryption error!"
    return 1
  fi

  if [ $CHECKSUM != $(cksum /tmp/$ME.info | awk '{ print $1 }') ]; then
    echo "Package info checksum error!"
    return 1
  fi

  PKG_VERSION=$(sed -n "s:^# *VERSION \+\(.*\):\1:p" /tmp/$ME.info)

  HEADER_SIZE=$(((SIZE + 64 + 1023) / 1024))

  echo "  header size = $HEADER_SIZE kB"
}

function get_line_info ()
{
  OFFSET=$(echo "$1" | awk '{ print $2 }')
  BLOCK=$(echo "$1" | awk '{ print $3 }')
  CHECKSUM=$(echo "$1" | awk '{ print $4" "$5 }')
  SIZE=$(((($(echo "$1" | awk '{ print $5 }') + 16) / 16) * 16))

  if [ -z "$2" ]; then
    IV=$(echo "$1" | awk '{ print $6 }')
    SIGNATURE=$(echo "$1" | awk '{ print $7 }')
    CHECKSIG=$(echo "$1" | awk '{ print $8 }')
  else
    IV=$(echo "$1" | awk '{ print $7 }')
    SIGNATURE=$(echo "$1" | awk '{ print $8 }')
    CHECKSIG=$(echo "$1" | awk '{ print $9 }')
  fi

  if [ "${SIGNATURE:60}" ]; then
    SIG=${SIGNATURE:0:60}...
  else
    SIG=$SIGNATURE
  fi

  if [ "${CHECKSIG:60}" ]; then
    CHKSIG=${CHECKSIG:0:60}...
  else
    CHKSIG=$CHECKSIG
  fi

  echo "$2  offset    = $OFFSET"
  echo "$2  block     = $BLOCK"
  echo "$2  checksum  = $CHECKSUM"
  echo "$2  size      = $SIZE"
  echo "$2  iv        = $IV"
  echo "$2  signature = $SIG"
  echo "$2  checksig  = $CHKSIG"

  if [ -z "$OFFSET" -o -z "$BLOCK" -o -z "$CHECKSUM" -o -z "$IV" -o -z "$SIGNATURE" -o -z "$CHECKSIG" ]; then
    return 1
  fi
}

function unpack_package_updater ()
{
  echo "Reading $1 updater:"

  UPDATER=$(grep "^UPDATER" /tmp/$ME.info)
  get_line_info "$UPDATER"

  if [ $? -ne 0 ]; then
    echo "Package updater info error!"
    return 1
  fi

  OFFSET=$((OFFSET + HEADER_SIZE))

  dd if="$1" of=/dev/stdout bs=1K skip=$OFFSET count=$BLOCK 2> /dev/null | head -c $SIZE | openssl aes-128-cbc -d -K $ENC_KEY2 -iv $IV > /tmp/$ME.updater.tar

  if [ $? -ne 0 ]; then
    echo "Package updater decryption error!"
    return 1
  fi

  U_CHECKSUM=$(cksum /tmp/$ME.updater.tar | awk '{ print $1" "$2 }')

  if [ "$U_CHECKSUM" != "$CHECKSUM" ]; then
    echo "Package updater checksum error!"
    return 1
  fi

  echo -n $SIGNATURE | hex2bin | openssl rsautl -verify -pubin -inkey $ID_RSA_PUB > /dev/null 2>&1

  if [ $? -ne 0 ]; then
    echo -n $SIGNATURE | hex2bin | openssl rsautl -verify -pubin -inkey /tmp/$ME.id_rsa.pub-p > /dev/null
  fi

  if [ $? -ne 0 ]; then
    echo "Package updater signature error!"
    return 1
  fi

  U_CHECKSIG=$(cat /tmp/$ME.updater.tar | openssl sha1 -binary | openssl rsautl -sign -inkey /tmp/$ME.id_rsa | hexdump -C -n 512 -v | head -n 32 | awk '{ print $2$3$4$5$6$7$8$9$10$11$12$13$14$15$16$17 }')

  if [ "$(echo $U_CHECKSIG | sed "s: ::g")" != "$CHECKSIG" ]; then
    echo "Package updater data signature error!"
    return 1
  fi
}

function unpack_package_images ()
{
  echo "Reading $1 images:"

  IMAGES=$(grep -v "^#" /tmp/$ME.info | grep -v "^UPDATER" | grep -v "^EULA" | awk '{ print $1 }')

  if [ -z "$IMAGES" ]; then
    echo "  no images present"
    return 0
  fi

  echo -n "  images: "
  echo "$IMAGES" | sed "s:<sp>: :g" | while read i; do echo -n "'$i' "; done
  echo

  for IMAGE in $IMAGES; do

    IMG=$(echo $IMAGE | sed "s:<sp>: :g")

    echo "  Reading image '$IMG':"

    LINE=$(grep "^$IMAGE " /tmp/$ME.info)
    get_line_info "$LINE" "  "

    if [ $? -ne 0 ]; then
      echo "Package image error!"
      return 1
    fi

    OFFSET=$((OFFSET + HEADER_SIZE))

    dd if="$1" of=/dev/stdout bs=1K skip=$OFFSET count=$BLOCK 2> /dev/null | head -c $SIZE | openssl aes-128-cbc -d -K $ENC_KEY2 -iv $IV > "/tmp/$ME.$IMG.img"

    if [ $? -ne 0 ]; then
      echo "Package image decryption error!"
      return 1
    fi

    U_CHECKSUM=$(cat "/tmp/$ME.$IMG.img" | transsize | cksum | awk '{ print $1" "$2 }')

    if [ "$U_CHECKSUM" != "$CHECKSUM" ]; then
      echo "Package image checksum error!"
      return 1
    fi

    echo -n $SIGNATURE | hex2bin | openssl rsautl -verify -pubin -inkey $ID_RSA_PUB > /dev/null

    if [ $? -ne 0 ]; then
      echo "Package image signature error!"
      return 1
    fi

    U_CHECKSIG=$(cat "/tmp/$ME.$IMG.img" | transsize | openssl sha1 -binary | openssl rsautl -sign -inkey /tmp/$ME.id_rsa | hexdump -C -n 512 -v | head -n 32 | awk '{ print $2$3$4$5$6$7$8$9$10$11$12$13$14$15$16$17 }')

    if [ "$(echo $U_CHECKSIG | sed "s: ::g")" != "$CHECKSIG" ]; then
      echo "Package image data signature error!"
      return 1
    fi

    echo "Package image '$IMG' saved as /tmp/$ME.$IMG.img, ok."

  done
}

ME=$(basename "$0")

if [ -z "$1" -o \( "$1" = "-t" -a -z "$2" \) ]; then
  echo "Usage: $ME [-t] FILE"
  echo
  echo "  -t  test FILE, report and exit"
  echo
  echo "(FILE must be a PRS-T2 Updater.package)"
  exit 1
fi

if [ "$1" = "-t" ]; then
  TEST=yes
  shift
fi

get_enc_key
unpack_package_info "$1"
ret=$?

if [ -z "$TEST" ]; then

  if [ $ret -ne 0 ]; then
    exit 2
  fi

  unpack_package_updater "$1"

  if [ $? -ne 0 ]; then
    exit 3
  fi

  echo "Package updater saved as /tmp/$ME.updater.tar, ok."

  unpack_package_images "$1"

  if [ $? -ne 0 ]; then
    exit 4
  fi

  echo "Update $1 is from $PKG_FROM, $PKG_FOR version${PKG_VERSION:+ $PKG_VERSION} - successfully unpacked."

else

  echo "Update $1 is from $PKG_FROM, $PKG_FOR version."

fi
