# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011, meme'
__docformat__ = 'restructuredtext en'

#####################################################################
# Calibre to Kindle conversion routines
#####################################################################

import os, re, copy, datetime
from collections import defaultdict

import calibre_plugins.kindle_collections.messages as msg
import calibre_plugins.kindle_collections.config as cfg
import calibre_plugins.kindle_collections.kindle_device as kindle_device
import calibre_plugins.kindle_collections.calibre_info as calibre_info
import calibre_plugins.kindle_collections.kindle_collections as kindle_collections
import calibre_plugins.kindle_collections.kindle_books as kindle_books
import calibre_plugins.kindle_collections.save as save
import calibre_plugins.kindle_collections.reports as reports
from calibre_plugins.kindle_collections.__init__ import PLUGIN_NAME
from calibre_plugins.kindle_collections.utilities import debug_print, wording, check_text_in_pattern_list, check_name_in_list, array_to_csv

status = None

#####################################################################

# Create the Kindle collections file based on Calibre
def create(parent, preview):
    debug_print('BEGIN Create/Preview, preview = %s' % preview)
    gui = parent.gui

    if not cfg.init(parent):
        debug_print('END Create/Preview - initialization failed')
        return msg.message

    # If no columns were selected to use then just return so warning message is clear
    if not is_action_selected():
        msg.message.error('No Calibre columns have been selected in Customization.<P>Please set the Action to "Create" for the column(s) you want to use when automatically creating collections.')
        return msg.message

    calibre_info.ci.load_column_book_info()

    # Check that calibre returned books
    if len(calibre_info.ci.ids_on_device) < 1:
        msg.message.error('No books were found in Calibre that are on the Kindle.<P>The Calibre library has %d books, of which %d are found on the connected device.<P>If you know your books are on the device, try disconnecting/reconnecting your Kindle or restarting Calibre.' % (len(calibre_info.ci.all_ids), len(calibre_info.ci.ids_on_device)))
        return msg.message

    kindle_device.init(calibre_info.ci.device_path)
    # Completely discard the exported JSON DB if asked (Touch only)
    if cfg.config_settings['ignore_json_db']:
        kindle_collections.init(None, cfg.config_settings['reset_collection_times'])
    else:
        kindle_collections.init(kindle_device.kdevice.get_collections(), cfg.config_settings['reset_collection_times'])
    kindle_compare_collections = kindle_collections.KindleCollections(kindle_device.kdevice.get_compare_collections())
    kindle_current_collections = copy.deepcopy(kindle_collections.kc.collections)
    kindle_paths = kindle_device.kdevice.get_file_paths()

    # Check that books were found on the Kindle
    if len(kindle_paths) < 1:
        msg.message.error('No books were found on the Kindle.<P>If you know your books are on the device, try disconnecting/reconnecting your Kindle or restarting Calibre.')
        return msg.message

    kindle_books.init(kindle_paths)
    init_status()

    # Check if plugin has been customized or has current customization format, and error if not
    if cfg.store.is_current_store_format():
        # Check if any customizable values are invalid and warn user
        if not cfg.validate_configuration(cfg.config_table, array_to_csv(cfg.config_settings['ignore_all'])):
            msg.message.error('Please use "Customize" to correct the invalid patterns before running.')
        else:
            # For each book in Calibre's list add it to the collections, then save to file when done
            process_calibre_collections(preview)

            differences = compare_collections_different(kindle_current_collections, kindle_compare_collections.collections)
            if differences:
                msg.message.warning('The collections on the Kindle have been edited since the last time Create Collections was run.  If any Calibre-managed collections were edited the changes made previously (listed below) were ignored.  Any collections listed in the Never Delete field in Customize are not checked for changes.\n%s' % differences)
            if preview:
                msg.message.info('<P>Preview report generated %d %s.' %
                        (kindle_collections.kc.get_len(), wording('collection', 'collections', kindle_collections.kc.get_len())))
            else:
                # Save collections
                save.save_kindle_collections(compare=True)
    else:
        msg.message.info('Please run "Customize collections" and select OK to save your customizations before running Create or Preview.')

    debug_print('END Create/Preview, preview = %s' % preview)

    return msg.message

def compare_collections_different(collections, old_collections):
    debug_print('BEGIN Compare collections')
    differences = ''
    if len(collections.keys()) == 0 or len(old_collections.keys()) == 0:
        debug_print('One collection is empty, skipping comparison')
        return differences
    for c in collections.keys():
        if is_compare_collection(c):
            if c not in old_collections:
                differences += '    Collection "%s" was added\n' % c
            else:
                for item in collections[c]['items']:
                    if item not in old_collections[c]['items']:
                        title = kindle_books.kbooks.path_info[kindle_books.kbooks.code_paths[item][0]]['title'] if item in kindle_books.kbooks.code_paths else item
                        differences += '    Collection "%s" has a new item: %s\n' % (c, title)
        else:
            debug_print('Skipping collection %s' % c)
    for c in old_collections.keys():
        if is_compare_collection(c):
            if c not in collections:
                differences += '    Collection "%s" was deleted\n' % c
            else:
                for item in old_collections[c]['items']:
                    if item not in collections[c]['items']:
                        title = kindle_books.kbooks.path_info[kindle_books.kbooks.code_paths[item][0]]['title'] if item in kindle_books.kbooks.code_paths else item
                        differences += '    Collection "%s" had an item deleted: %s\n' % (c, title)
        else:
            debug_print('Skipping collection %s' % c)
    if not differences:
        debug_print('Collections have not changed since last Create Collections')
    debug_print('END Compare collections')
    return differences

def is_compare_collection(collection):
    return not check_text_in_pattern_list(collection, cfg.config_settings['ignore_all'], cfg.config_settings['ignore_case'])

def is_action_selected():
    found_action = False
    for label in calibre_info.ci.column_labels:
        if cfg.config_table[label]['action'] != cfg.OPT_NONE:
            found_action = True
            break
    return found_action

# Go through all Calibre columns/categories and create collections, then convert to Kindle style
def process_calibre_collections(preview):
    debug_print('BEGIN process calibre collections, preview = %s' % preview)

    collections = filter_collections(calibre_info.ci.active_collections)
    convert_calibre_collections_to_kindle(collections)
    delete_unpreserved_kindle_collections()
    reports.generate_report_all(preview, status)

    debug_print('END process calibre collections')

# Filter out any collections not selected by the customization settings
def filter_collections(old_collections):
    debug_print('BEGIN Filter Collections')
    new_collections = defaultdict()
    for label in calibre_info.ci.column_labels:
        action = cfg.config_table[label]['action']
        # Check if there is an action to do, and make sure the label exists (won't exist if no books use it)
        if action != cfg.OPT_NONE and label in old_collections:
            debug_print('    "%s"' % label)
            for collection in old_collections[label]:
                paths = old_collections[label][collection]
                cname = calibre_info.ci.column_labels[label]
                ignore_case = cfg.config_settings['ignore_case']
                ignore_list = cfg.config_table[label]['ignore']
                include_list = cfg.config_table[label]['include']
                rename_from = cfg.config_table[label]['rename_from']
                rename_to = cfg.config_table[label]['rename_to']
                split_char = cfg.config_table[label]['split_char']

                if split_char:
                    collections = collection.split(split_char)
                    debug_print('            Splitting "%s" into %s' % (collection, collections))
                else:
                    collections = [ collection ]

                for c in collections:
                    # Handle boolean columns, though they don't create useful collection names
                    if type(c) == bool:
                        if c:
                            c = calibre_info.ci.column_labels[label]
                    elif type(c) == int:
                        c = str(c)
                    elif type(c) == datetime.datetime:
                        c = "%d-%02d-%02d" % (c.year, c.month, c.day)
                    else:
                        if not c or c == '':
                            continue
                    c = c.strip()

                    # If column contains ";", convert to "," - hack to allow Author Sort names to be imported/exported
                    old_c = c
                    c = c.replace(';', ',')
                    if c != old_c:
                        debug_print('        Auto-Replacing "%s" with "%s"' % (old_c, c))

                    # Check ignore/include patterns before saving it to the list to process later
                    if check_text_in_pattern_list(c, ignore_list, ignore_case):
                        # Skip - in ignore list
                        status.set(c, 'None (due to ignore list for column "%s")' % cname)
                        msg.message.report('    Ignore collection:  %s (due to ignore list for column "%s")' % (c, cname))
                    elif (array_to_csv(include_list) != '') and not check_text_in_pattern_list(c, include_list, ignore_case):
                        # Skip - not in include list
                        status.set(c, 'None (due to not being on include list for column "%s")' % cname)
                        msg.message.report('    Not including:      %s (due to include list for column "%s")' % (c, cname))
                    else:
                        # Rename collection if requested
                        old_collection = c
                        new_collection = rename_collection_name(rename_from, rename_to, c)
                        if label not in new_collections:
                            new_collections[label] =  defaultdict()
                        if new_collection not in new_collections[label]:
                            new_collections[label][new_collection] = paths[:]
                        else:
                            new_collections[label][new_collection] += paths[:]
        else:
            debug_print('    "%s" ignored since action is blank' % label)
    debug_print('END filter collections')
    return new_collections

# Check if any collections are on the Kindle that aren't managed by Calibre that need to be deleted
def delete_unpreserved_kindle_collections():
    debug_print('Deleting collections on the Kindle if requested')
    # Delete any existing Kindle collections only if keep_kindle_only is False and they don't match the ignore all pattern
    for name in kindle_collections.kc.get_unsorted_names():
        debug_print('    "%s"' % name)
        if kindle_collections.kc.is_kindle_managed(name):
            # Its Kindle managed so we can look at it
            if cfg.config_settings['keep_kindle_only']:
                # Settings say we should keep it
                if not status.get(name):
                    status.set(name, 'None (preserve Kindle-only is True)')
            else:
                # Might be able to delete it
                if check_text_in_pattern_list(name, cfg.config_settings['ignore_all'], cfg.config_settings['ignore_case']):
                    # its on the ignore list so skip it
                    if not status.get(name):
                        status.set(name, 'None (matched never delete/modify list)')
                else:
                    # All okay to delete
                    kindle_collections.kc.delete(name)
                    status.set(name, 'Deleted (preserve kindle-only is False)')

# Transform a Calibre column/collection name using re.sub
def rename_collection_name(rename_from, rename_to, text):
    if len(rename_from) > 0 and len(rename_to) > 0:
        try:
            if cfg.config_settings['ignore_case']:
                oldtext = text
                text = re.sub(rename_from, rename_to, text, flags=re.IGNORECASE)
                # Force single character collections into uppercase if ignoring case
                if len(text) == 1:
                    text = text.upper()
                if oldtext != text:
                    debug_print('    Renamed:            Ignoring case "%s" to "%s" using "%s"->%s"' % (oldtext, text, rename_from, rename_to))
            else:
                oldtext = text
                text = re.sub(rename_from, rename_to, text)
                if oldtext != text:
                    debug_print('    Renamed:            Keeping case "%s" to "%s" using "%s"->%s"' % (oldtext, text, rename_from, rename_to))
        except:
            msg.message.warning('Unexpected error in mapping patterns - from "%s" to "%s" for collection "%s"' % (rename_from, rename_to, text))
    return text

def convert_calibre_collections_to_kindle(collections):
    debug_print('BEGIN Converting Calibre collections to Kindle collections')
    # For every Authors, Series, etc. which have lists of collection names to process
    book_codes = {}
    for column in collections.keys():
        action = cfg.config_table[column]['action']
        prefix = cfg.config_table[column]['prefix']
        suffix = cfg.config_table[column]['suffix']
        ignore_all = cfg.config_settings['ignore_all']
        ignore_prefix_suffix = cfg.config_settings['ignore_prefix_suffix']
        ignore_case = cfg.config_settings['ignore_case']
        fast_reboot = cfg.config_settings['fast_reboot']
        cname = calibre_info.ci.column_labels[column]
        mintext = cfg.config_table[column]['minimum']
        if mintext == '':
            minimum = 1
        else:
            minimum = int(mintext)

        debug_print('Convert column: "%s", action "%s", prefix "%s", suffix "%s"' % (column, action, prefix, suffix))

        # Check for conflicting names
        if ignore_prefix_suffix:
            strip_chars = '- _,;:!?/.`^~\'"()[]{}@$*&#%+<=>|'
            for collection in collections[column]:
                stripped_collection = collection.strip(strip_chars)
                if stripped_collection != collection and stripped_collection in collections[column]:
                    msg.message.warning('Calibre column "%s" contains collection entry "%s" which conflicts with "%s" - 1 of them will be ignored' % (column, collection, stripped_collection))

        # For every collection name in the column...
        for collection in collections[column]:
            pcollection = prefix + collection + suffix
            xcollection = collection if ignore_prefix_suffix else pcollection
            if xcollection not in book_codes:
                book_codes[xcollection] = set([])
            prev_column = len(book_codes[xcollection]) > 0
            collection_len = len(collections[column][collection])
            paths = collections[column][collection]
            for path in paths:
                # Collect codes for any books on the Kindle
                if path in kindle_books.kbooks.path_info:
                    code = kindle_books.kbooks.path_info[path]['code']
                    book_codes[xcollection].add(code)

            debug_print('    Collection: "%s", prefix: "%s", collection length: %d, book_codes: %d' % (collection, prefix, collection_len, len(book_codes[xcollection])))

            # Check if there are any matching collections already on the Kindle to deal with
            if ignore_prefix_suffix:
                matches = check_name_in_list(collection, kindle_collections.kc.get_unsorted_names(), ignore_case, ignore_prefix_suffix)
            else:
                matches = check_name_in_list(pcollection, kindle_collections.kc.get_unsorted_names(), ignore_case, ignore_prefix_suffix)

            if matches:
                for match in matches:
                    # Check if on the never delete/modify list
                    if check_text_in_pattern_list(match, ignore_all, ignore_case):
                        status.set(pcollection, 'None (due to never delete/modify list)')
                        if pcollection != match:
                            status.set(match, 'None (due to never delete/modify list)')
                    else:
                        if action == cfg.OPT_CREATE:
                            if collection_len >= minimum:
                                if pcollection != match:
                                    kindle_collections.kc.delete_quiet(match)
                                kindle_collections.kc.add(pcollection, list(book_codes[xcollection]))
                                column_names = '"' + cname + '" + others' if prev_column else cname
                                if len(matches) == 1 and pcollection == matches[0]:
                                    status.set(pcollection, 'Replaced (from column %s)' % column_names)
                                else:
                                    status.set(pcollection, 'Replaced "%s" (from column %s)' % (array_to_csv(matches), column_names))
                            else:
                                kindle_collections.kc.delete(match, kindle_managed=False)
                                status.set(match, 'Deleted "%s" (did not meet minimum of %d %s for column "%s")' % (array_to_csv(matches), minimum, wording('book', 'books', minimum), cname))

                        elif action == cfg.OPT_DELETE:
                            if pcollection != match:
                                kindle_collections.kc.delete_quiet(match)
                            kindle_collections.kc.delete(pcollection, kindle_managed=False)
                            status.set(match, 'Deleted (matched "%s" from column %s)' % (pcollection, cname))
            # no matches on Kindle
            elif action == cfg.OPT_CREATE:
                if collection_len >= minimum:
                    kindle_collections.kc.add(pcollection, list(book_codes[xcollection]))
                    column_names = '"' + cname + '" + others' if prev_column else cname
                    status.set(pcollection, 'Created (from column %s)' % column_names)
    debug_print('END Converting Calibre collections to Kindle collections')


##########################################################################

class Status():
    def __init__(self):
        self.clear()

    def clear(self):
        self.status = defaultdict()

    def get(self, item):
        value = ''
        if item in self.status:
            value = self.status[item]
        return value

    # Track add/update/delete status of collections
    def set(self, item, value):
        if item != '':
            self.status[item] = value
            debug_print('        %s' % self.status[item])

def init_status():
    global status
    status = Status()
