# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011, meme'
__docformat__ = 'restructuredtext en'

#####################################################################
# Configuration data read/write/validation routines
#####################################################################

import os, re
from functools import partial

from PyQt4 import QtGui
from PyQt4.Qt import Qt, QWidget, QVBoxLayout, QHBoxLayout, QLabel,  \
                     QLineEdit, QFormLayout, QTableWidget, QTableWidgetItem, \
                     QAbstractItemView, QComboBox, QMenu, QToolButton, QIcon, \
                     QCheckBox

import calibre_plugins.kindle_collections.messages as msg
import calibre_plugins.kindle_collections.calibre_info as calibre_info
from calibre_plugins.kindle_collections.utilities import debug_print, csv_to_array
from calibre_plugins.kindle_collections.__init__ import PLUGIN_NAME

PLUGIN_STORE_DIR = 'plugins'                # Location of plugins in the Calibre config dir
PLUGIN_STORE_FILE = '%s.json' % PLUGIN_NAME # Name of the file holding the customization data
STORE_VERSION = '1'                         # Update if changing what is stored in the customization json file
STORE_KEY_ROWS = 'Rows'                     # Specific key to hold the configuration for the table of columns/collections
STORE_KEY_SETTINGS = 'Settings'             # Specific key to hold additional settings

MENU_CLICK_STYLE = 'Kindle Collections: Menu Style'
MENU_DEFAULT = 'Show menu (default)'
MENU_CREATE = 'Create Collections'
MENU_PREVIEW = 'Preview Collections'
MENU_CUSTOMIZE = 'Customize plugin'
MENU_VIEW = 'View report'
MENU_EDIT = 'Edit collections'
MENU_IMPORT = 'Import collections'
MENU_MODIFY = 'Modify settings'
MENU_OPTIONS = [ MENU_DEFAULT, MENU_CREATE, MENU_PREVIEW, MENU_CUSTOMIZE, MENU_VIEW, MENU_EDIT, MENU_IMPORT, MENU_MODIFY ]

OPT_NONE   = ''                             # Options for customization
OPT_CREATE = 'Create'
OPT_DELETE = 'Delete'

ACTION_OPTIONS = [ OPT_NONE, OPT_CREATE, OPT_DELETE ]

CUSTOMIZE_DEFAULTS = {             # Default customization values for the table
#    'tags': {
#        'action':      OPT_CREATE,
#        'column':      '',
#        'prefix':      '_',
#        'suffix':      '',
#        'minimum':     '1',
#        'ignore':      [],
#        'include':     [],
#        'rename_from': '',
#        'rename_to':   '',
#        'split_char':  ''
#    }
}

SETTINGS_DEFAULTS = {                # Default customization values for additional settings
    'ignore_all':               [],
    'keep_kindle_only':         True,
    'ignore_prefix_suffix':     True,
    'ignore_case':              True,
    'reset_collection_times':   True,
    'fast_reboot':              True,
    'ignore_json_db':           False,
    'force_touch_model':        False,
    'store_version':            'Not Yet Customized'
}

OLD_STORE_NAME = 'Customization'     # Old key name for customization - new version uses library/device id
STORE_DEFAULTS = {                   # Format used by customization json file to store settings
    STORE_KEY_ROWS: CUSTOMIZE_DEFAULTS,
    STORE_KEY_SETTINGS: SETTINGS_DEFAULTS
}

CUSTOMIZE_COLUMNS = {
    'action':      { 'pos': 0, 'title': 'Action', 'blank': OPT_NONE},
    'column':      { 'pos': 1, 'title': 'Calibre Source', 'blank': '' },
    'prefix':      { 'pos': 2, 'title': 'Prefix', 'blank': '' },
    'suffix':      { 'pos': 3, 'title': 'Suffix', 'blank': '' },
    'minimum':     { 'pos': 4, 'title': 'Minimum', 'blank': '' },
    'ignore':      { 'pos': 5, 'title': 'Ignore names matching these patterns', 'blank': '' },
    'include':     { 'pos': 6, 'title': 'Include names matching these patterns', 'blank': '' },
    'rename_from': { 'pos': 7, 'title': 'Rename these patterns...', 'blank': '' },
    'rename_to':   { 'pos': 8, 'title': '...to these patterns', 'blank': '' },
    'split_char':  { 'pos': 9, 'title': 'Split on character', 'blank': '' }
}

config_table = None                 # Configuration info for table
config_settings = {}                # Configuration info for other choices
store = None

#####################################################################

def init(parent):
    global store
    debug_print('BEGIN Configuration loading config store')

    msg.init(parent)
    calibre_info.init(parent)

    ok = True
    if calibre_info.ci.library_uuid != '' and calibre_info.ci.device_uuid != '':
        if calibre_info.device_metadata_available:
            store = ConfigStore(calibre_info.ci.library_uuid, calibre_info.ci.device_uuid)
            store_data = store.read_store_data()
            load_config_settings(store_data.get(STORE_KEY_SETTINGS))
            load_config_table(store_data.get(STORE_KEY_ROWS))
        else:
            msg.message.error('Calibre has not finished loading data from the Kindle.<P>Please wait until the Jobs indicator shows 0 jobs running.')
            ok = False
    else:
        msg.message.error('Kindle not detected.<P>This plugin requires a Kindle to be connected so that the plugin can read existing collection information and the data Calibre stored on the Kindle.')
        ok = False

    debug_print('END Configuration loading config store')
    return ok

def load_config_settings(settings=STORE_DEFAULTS[STORE_KEY_SETTINGS]):
    global config_settings

    config_settings = settings

    for i in STORE_DEFAULTS[STORE_KEY_SETTINGS].keys():
        if i not in config_settings:
            config_settings[i] = STORE_DEFAULTS[STORE_KEY_SETTINGS][i]
            debug_print('Setting "%s" not found in json file, loading default value "%s"' % (i,config_settings[i]))

def load_config_table(table=STORE_DEFAULTS[STORE_KEY_ROWS]):
    global config_table

    config_table = table

    # Ensure config_table has every Calibre column - fill new ones with blank values
    for label in calibre_info.ci.column_labels.keys():
        if label not in config_table:
            config_table[label] = create_blank_row_data()
    # Remove any saved columns that don't exist in Calibre
    for row in config_table.keys():
        if row not in calibre_info.ci.column_labels:
            del config_table[row]

    # And make sure every value for a row exists (if extra, doesn't matter as they won't be saved/used)
    for row in config_table.keys():
        for entry in CUSTOMIZE_COLUMNS.keys():
            if entry == 'column':
                # Save a name for the title column in the table
                config_table[row]['column'] = calibre_info.ci.column_labels[row]
            elif entry not in config_table[row]:
                config_table[row][entry] = CUSTOMIZE_COLUMNS[entry]['blank']
                debug_print('Adding "%s" to row "%s"' % (entry, row))

def create_blank_row_data():
    data = {}
    for k in CUSTOMIZE_COLUMNS.keys():
        data[k] = CUSTOMIZE_COLUMNS[k]['blank']
    return data

##########################################################################

def validate_pattern(pattern):
    message_text = ''
    valid = True
    if len(pattern) > 0:
        try:
            r = re.match(pattern,'')
            if re.match('\|', pattern):
                valid=False
                message_text = '.  Cannot start with "|", try "\|"'
        except:
            valid = False
    return (valid, message_text)

# Check entered patterns to see if they are valid - to issue just 1 error message
def validate_configuration(row_data, ignore_all_pattern):
    debug_print('BEGIN Validating customization values')
    valid = True
    if len(row_data) == 0:
        valid = False
        msg.message.error('No table items - no Calibre data found')

    # Check the ignore/include fields for valid patterns
    for row in row_data.keys():
        fields = row_data[row]
        for field in [ 'ignore', 'include' ]:
            for pattern in fields[field]:
                (avalid, message_text) = validate_pattern(pattern)
                if not avalid:
                    valid = False
                    msg.message.error('Invalid pattern "%s" - in row "%s", column "%s"%s.' % (pattern, cc.column_labels[row], field, message_text))
        # Check the substitute columns for valid patterns
        rename_from = fields['rename_from']
        rename_to = fields['rename_to']
        rename_from_empty = (not rename_from) or rename_from.isspace()
        rename_to_empty = (not rename_to) or rename_to.isspace()
        avalid = True
        if len(rename_from) > 0 or len(rename_to) > 0:
            if rename_from_empty or rename_to_empty:
                avalid = False
            else:
                try:
                    r = re.sub(rename_from, rename_to, '')
                except:
                    avalid = False
        if avalid == False:
            valid = False
            msg.message.error('Invalid substitute patterns - from "%s", to "%s" - in row "%s".' % (rename_from, rename_to, row))

        minimum = fields['minimum']
        if minimum != '':
            if not minimum.isdigit():
                m = -1
            else:
                m = int(minimum)
            if m < 0:
                valid = False
                msg.message.error('Invalid number "%s" in customized row "%s", column "%s".' % (minimum, row, 'minimum'))

    # Check the overall ignore field
    for pattern in csv_to_array(ignore_all_pattern):
        (avalid, message_text) = validate_pattern(pattern)
        if not avalid:
            valid = False
            msg.message.error('Invalid pattern "%s" in ignore always field "%s"%s."' % (pattern, row, message_text))

    debug_print('END Validating customization values, returning %s' % valid)
    return valid

#####################################################################

class ConfigStore():

    def __init__(self, library_uuid, device_uuid):
        debug_print('BEGIN Initializing ConfigStore')
        self.library_uuid = library_uuid
        self.device_uuid = device_uuid
        self.plugin_config_store = ''
        debug_print('END Initializing ConfigStore')

    # 1.3.X had a bug that saved config settings in the wrong directory, and 1.4.0 renamed the file
    def fix_legacy_store_path(self):
        from calibre.utils.config import config_dir
        debug_print('BEGIN Fixing legacy store path')

        OLDNAME = 'Create Kindle Collections.json'

        new_store_absolute_path = self.get_calibre_plugin_store_absolute_path()
        if not os.path.isfile(new_store_absolute_path):
            # If pre 1.3.X version, just rename the file
            old_name_absolute_path = os.path.join(config_dir, PLUGIN_STORE_DIR, OLDNAME)
            if os.path.isfile(old_name_absolute_path):
                os.rename(old_name_absolute_path, new_store_absolute_path)
                debug_print('Fixed legacy store file location %s -> %s' % (old_name_absolute_path, new_store_absolute_path))

            incorrect_absolute_path = os.path.join(config_dir, re.sub('^.','',config_dir), PLUGIN_STORE_DIR, OLDNAME)
            # If 1.3.X version, move file from subdirectory (and overwrite any older version)
            if os.path.isfile(incorrect_absolute_path):
                # If pre 1.3.X version was there, just delete it and replace with newer version
                if os.path.isfile(new_store_absolute_path):
                    os.remove(new_store_absolute_path)
                os.rename(incorrect_absolute_path, new_store_absolute_path)
                debug_print('Fixed legacy incorrect store file location %s -> %s' % (incorrect_absolute_path, new_store_absolute_path))
        debug_print('END Fixing legacy store path')

    # 1.3.X and before did not use the actual field label as name, and did not have action column, different settings
    def fix_legacy_config_values(self, data):
        debug_print('BEGIN Fixing legacy config values')
        if data:
            table = data.get(STORE_KEY_ROWS)
            for row_index in table.keys():
                if 'collect' in table[row_index]:
                    if table[row_index]['collect']:
                        debug_print('Fixing legacy value collect for "%s" to use new action option' % row_index)
                        table[row_index]['action'] = OPT_CREATE
                    else:
                        table[row_index]['action'] = OPT_NONE
                    del table[row_index]['collect']
                if row_index not in calibre_info.ci.column_labels:
                    custom = '#' + row_index
                    if custom in calibre_info.ci.column_labels:
                        debug_print('Fixing legacy label - renaming %s to %s' % (custom, row_index))
                        table[custom] = table[row_index]
                        del table[row_index]
            settings = data.get(STORE_KEY_SETTINGS)
            # Version 1.2 had no settings, Version 1.3.X had different names, missing settings
            if settings:
                if 'ignore_prefix' in settings:
                    settings['ignore_prefix_suffix'] = settings['ignore_prefix']
                    debug_print('Fixing legacy settings for ignore_prefix to use ignore_prefix_suffix')
                if 'fast_reboot' not in settings:
                    settings['fast_reboot'] = STORE_DEFAULTS[STORE_KEY_SETTINGS]['fast_reboot']
                    debug_print('Fixing legacy settings to add fast_reboot option')
            else:
                    settings = STORE_DEFAULTS[STORE_KEY_SETTINGS]
            data[STORE_KEY_ROWS] =  table
            data[STORE_KEY_SETTINGS] =  settings
        debug_print('BEGIN Fixing legacy config values')

        return data

    def get_calibre_plugin_store_path(self):
        return os.path.join(PLUGIN_STORE_DIR, PLUGIN_STORE_FILE)

    def get_calibre_plugin_store_absolute_path(self):
        from calibre.utils.config import config_dir
        return os.path.join(config_dir, self.get_calibre_plugin_store_path())

    def is_current_store_format(self):
        debug_print('BEGIN Is Current Store Format')
        ok = True
        if 'store_version' in config_settings:
            store_version = config_settings['store_version']
        else:
            store_version = 'Unexpected error'
        if store_version != STORE_VERSION:
            ok = False
            debug_print('Invalid store version - found %s instead of %s' % (store_version, STORE_VERSION))

        debug_print('END Is Current Store Format=%s, store_version=%s' % (ok, store_version))
        return ok

    def fix_legacy_stuff(self):
        debug_print('Fixing legacy stuff')
        self.fix_legacy_store_path()
        # Legacy plugin used to store data under just one name
        c = self.config_store()
        if c:
            old_store_data = c.get(OLD_STORE_NAME, None)
            if old_store_data:
                debug_print('Fixing old store key to use library/Kindle uuid instead of "%s"' % OLD_STORE_NAME)
                new_store_data = self.fix_legacy_config_values(old_store_data)
                if OLD_STORE_NAME in c:
                    del c[OLD_STORE_NAME]
                self.write_store_data(new_store_data)

    def read_store_data(self):
        debug_print('Reading store information')

        # Plugin used to store json file under a different name/path
        self.fix_legacy_stuff()

        device_data = {}
        c = self.config_store()
        if c:
            # Read the data: library { device: data }
            library_data = c.get(self.library_uuid)
            if library_data and self.device_uuid in library_data:
                device_data = library_data[self.device_uuid]
        if not device_data:
            device_data = STORE_DEFAULTS
            debug_print('Loaded default configuration information')
        else:
            debug_print('Loaded data from library "%s", device "%s"' % (self.library_uuid, self.device_uuid))
        return device_data

    def write_store_data(self, device_data):
        debug_print('BEGIN Writing store information')
        c = self.config_store()
        library_data = c.get(self.library_uuid)
        if library_data:
            debug_print('Using existing store for this library')
            library_data[self.device_uuid] = device_data
        else:
            debug_print('No existing store for this library so creating it')
            library_data = { self.device_uuid: device_data }
        c.set(self.library_uuid, library_data)
        debug_print('END Writing store information')

    def config_store(self):
        from calibre.utils.config import JSONConfig

        if not self.plugin_config_store:
            filename = self.get_calibre_plugin_store_path()
            debug_print('Setting customization config_store to "%s"' % os.path.join(filename))
            try:
                # Instantiate configuration data store from JSON file
                self.plugin_config_store = JSONConfig(filename)
            except:
                debug_print('Unexpected error with JSONConfig')
            if not self.plugin_config_store:
                debug_print('No customization file found, will use defaults for all values')

        return self.plugin_config_store

