import os
import struct

from calibre.devices.kindle.driver import KINDLE2
from calibre.ebooks.metadata.mobi import get_metadata


class UpdateMBP(KINDLE2):
    name = '%s MBP Update Mod' % (KINDLE2.name.replace('/', ', '))
    author = '%s, cryzed' % (KINDLE2.author)
    description = '%s If *.mbp files are available they will now be updated to match possibly changed *.mobi files.' % (KINDLE2.description)

    def upload_books(self, files, names, on_card=None, end_session=True, metadata=None):
        path = self._sanity_check(on_card, files)
        original = []

        # Collect original UIDs for *.mobi files with an associated *.mbp file
        for name, metadatum in zip(names, metadata):
            path = self.normalize_path(self.create_upload_path(path, metadatum, name))
            mbp_path = '%s.mbp' % os.path.splitext(path)[0]
            if not os.path.exists(mbp_path):
                continue
            with open(path, 'rb') as file:
                original.append((path, mbp_path, get_metadata(file).uid))

        # Replace old *.mobi files with new *.mobi files. The UIDs might have
        # changed at this ppint.
        result = super(KINDLE2, self).upload_books(files, names, on_card, end_session, metadata)

        # Update the UIDs within the *.mbp files to match the new *.mobi files'
        for path, mbp_path, uid in original:
            with open(path, 'rb') as file:
                # Retrieve the metadata from the updated *.mobi file
                metadata = get_metadata(file)

            with open(mbp_path, 'rb') as file:
                content = file.read()

            # Replace the old UID with the new one
            uid = struct.pack('>I', uid)
            new_uid = struct.pack('>I', metadata.uid)
            content = content.replace(uid, new_uid)
            with open(mbp_path, 'wb') as file:
                file.write(content)

        return result
