/* Touch screen keyboard functions
 */

/* Copyright (C) 2012 JoppyFurr
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy 
 * of this software and associated documentation files (the "Software"), to 
 * deal in the Software without restriction, including without limitation the 
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or 
 * sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 * DEALINGS IN THE SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <pthread.h>

#include "constants.h"
#include "drawing.h"
#include "text.h"

/* Include the font */
#include "font.h"

/* Include the keyboard layout */
#include "keyboard.h"

void draw_keyboard(unsigned char *frame_buffer, int modifier) {
    int x = 0;
    int y = 0;
    unsigned char c = '\0';
    int text_x = 0;
    int text_y = 0;
    
    int offset = (WIDTH - KEYBOARD_COLUMNS * KEY_WIDTH) / 2;
 
    /* Keyboard grid */
    draw_grid(frame_buffer, offset, HEIGHT - (KEYBOARD_ROWS * KEY_WIDTH  + offset), KEYBOARD_COLUMNS * KEY_WIDTH + 1, KEYBOARD_ROWS * KEY_WIDTH + 1, KEY_WIDTH);

    /* Keyboard text */    
    for(x = 0; x < KEYBOARD_COLUMNS; x++) {
        for(y = 0; y < KEYBOARD_ROWS; y++) {
            
            /* Calculate the position to start writing text if this key has a string on it */
            text_x = offset + x * KEY_WIDTH + 3;
            text_y = HEIGHT - offset - (KEYBOARD_ROWS * KEY_WIDTH + 1) + y * KEY_WIDTH + 7;
            
            /* Find the key to draw */
            if(modifier == NONE) {
                c = keyboard_lower[x + KEYBOARD_COLUMNS * y];
            }
            else if(modifier == SHIFT) {
                c = keyboard_higher[x + KEYBOARD_COLUMNS * y];
            }
           
            /* Draw it */
            if(c > 0x20 && c <= 0x7E) { /* Printable character */
                blit_char(frame_buffer, c,                                                                           
                    offset + x * KEY_WIDTH + 12,                                                                      
                    HEIGHT - offset - (KEYBOARD_ROWS * KEY_WIDTH + 1) + y * KEY_WIDTH + 8, 3); 
            }  
            else if(c == ESCAPE_KEY) {
                blit_string(frame_buffer, "Esc", text_x, text_y, 1);
            }
            else if(c == BACKSPACE_KEY) {
                blit_string(frame_buffer, "Back", text_x, text_y, 1);
            }
            else if(c == DELETE_KEY) {
                blit_string(frame_buffer, "Del", text_x, text_y, 1);
            }
            else if(c == TAB_KEY) {
                blit_string(frame_buffer, "Tab", text_x, text_y, 1);
            }
            else if(c == SHIFT_KEY) {
                blit_string(frame_buffer, "Shift", text_x, text_y, 1);
            }
            else if(c == CTRL_KEY) {
                blit_string(frame_buffer, "Ctrl", text_x, text_y, 1);
            }
            else if(c == ALT_KEY) {
                blit_string(frame_buffer, "Alt", text_x, text_y, 1);
            }
            else if(c == UP_ARROW) {
                blit_string(frame_buffer, "Up", text_x, text_y, 1);
            }
            else if(c == DOWN_ARROW) {
                blit_string(frame_buffer, "Down", text_x, text_y, 1);
            }
            else if(c == LEFT_ARROW) {
                blit_string(frame_buffer, "Left", text_x, text_y, 1);
            }
            else if(c == RIGHT_ARROW) {
                blit_string(frame_buffer, "Right", text_x, text_y, 1);
            }
            else if(c == '\n') {
                blit_string(frame_buffer, "Enter", text_x, text_y, 1);
            }
            else if(c == ' ') {
                blit_string(frame_buffer, "Space", text_x, text_y, 1);
            } 
        }
    }
}

/* Look up a keypress given an x and y coordinate. */
/* Returns a null character if press was outside of range */
unsigned char read_keyboard(int x, int y, int modifier) {
    int offset = (WIDTH - KEYBOARD_COLUMNS * KEY_WIDTH) / 2;    
   
    /* Make x and y relitive to the keyboard */
    x -= offset;
    y -= HEIGHT - (KEYBOARD_ROWS * KEY_WIDTH + 1 + offset);

    if(x < 0 || x > KEYBOARD_COLUMNS * KEY_WIDTH ||
        y < 0 || y > KEYBOARD_ROWS * KEY_WIDTH) {
       
        return '\0';
    }
   
    if(modifier == NONE) {
        return keyboard_lower[x / KEY_WIDTH + (y / KEY_WIDTH) * KEYBOARD_COLUMNS];
    }
    else if(modifier == SHIFT) {
        return keyboard_higher[x / KEY_WIDTH + (y / KEY_WIDTH) * KEYBOARD_COLUMNS];
    }
    
    /* Default */
    return '\0';
}

