/*====================================================
 * 3D - Projects a 3D object to a 2D screen
 * plot, main, Copyright (c) 2012 PoP under MIT License:
 * gmlib 1.5a, newtrix 2.0 routines, Copyright (c) 2012 geekmaster under MIT License: 
 * http://www.opensource.org/licenses/mit-license.php
 * ---------------------------------------------------
 * Revision History:
 * v5.0  2012-06-13 PoP            ported to Kindle3 C http://www.mobileread.com/forums/showthread.php?p=2113970#post2113970
 * v4.0  2012-04-05 PoP            ported to Kindle3 bin/sh http://www.mobileread.com/forums/showthread.php?t=172182                
 * v3.0  1986-12-20 PoP            ported to MIB PC BAS
 * v2.0  1983-09-14 PoP            ported to C=64 BAS
 * v1.0  1982-11-11 PoP            ported to Apple][ BAS
 * v0.0  1981-01-01 A. Pickholtz   see pp 474-505 BYTE magazine: http://malus.exotica.org.uk/~buzz/byte/pdf/BYTE%20Vol%2007-11%201982-11%20Graphics.pdf
 * ---------------------------------------------------
 * This was tested on K3 and not tested on DX,DXG,K4(Mini),K5(Touch).
 * ----------------------------------------------------
 */

#include <stdio.h>      // printf
#include <stdlib.h>    // malloc, free
#include <string.h>   // memset, memcpy
#include <unistd.h>  // usleep
#include <fcntl.h>  // open, close, write
#include <time.h>  // time
#include <sys/mman.h>   // mmap, munmap
#include <sys/ioctl.h> // ioctl
#include <linux/fb.h> // screeninfo
typedef unsigned char u8;
#include <linux/einkfb.h>
#include <linux/mxcfb.h>
#include <math.h> // compile with tccmake 3D -lm

//#define FBIO_EINK_UPDATE_DISPLAY_AREA 0x46dd
enum GMLIB_op { GMLIB_INIT,GMLIB_CLOSE,GMLIB_UPDATE };
//typedef unsigned int u32;

// function prototypes
void circle(int,int,int);
void line(int,int,int,int,int);
int gmlib(int);
inline void setpx(int,int,int);
void dblit(void);
int getmsec(void);
void plot(float []);

// 3D global variables
float	B=0;	// bank -PI..PI
float	CB;	// cos(bank)
float	CH;	// cos(heading)  
float	CP;	// cos(pitch)
static float	BLOCK[]={
	8,
	 5.25,-2, 3.25,
	-5.25,-2, 3.25,
	-5.25,-2,-3.25,
	 5.25,-2,-3.25,
	 5.25, 2, 3.25,
	-5.25, 2, 3.25,
	-5.25, 2,-3.25,
	 5.25, 2,-3.25,
	17,
	-1,2,3,4,1,5,6,7,8,5,
	-2,6,3,7,2,
	-4,8 
	};	// vectors of a block
float	D=13.5;	// distance from object
static float	DODECA[]={
	20,
	  0,-2.1408,5.6052,
	  3.4644,-3.4644,3.4644,
	  5.6052,0,2.1408,
	  3.4644,3.4644,3.4644,
	  0,2.1408,5.6052,
	 -3.4644,3.4644,3.4644,
	 -5.6052,0,2.1408,
	 -3.4644,-3.4644,3.4644,
	 -2.1408,-5.6052,0,
	  2.1408,-5.6052,0,
	  2.1408,5.6052,0,
	 -2.1408,5.6052,0,
	 -3.4644,-3.4644,-3.4644,
	  0,-2.1408,-5.6052,
	  3.4644,-3.4644,-3.4644,
	  5.6052,0,-2.1408,
	  3.4644,3.4644,-3.4644,
	  0,2.1408,-5.6052,
	 -3.4644,3.4644,-3.4644,
	 -5.6052,0,-2.1408,
	40,
	 -1,2,3,4,5,6,7,8,1,5, -14,15,16,17,18,19,20,13,14,18, -8,9,10,2,
	 -13,9, -15,10, -4,11,12,6, -17,11, -19,12, -16,3, -7,20 
	 };	// vectors of a dodecahedron
float	EDGi,EDG;	// stores ith edge temporarily
float	EDGE[50];	// i'th edge 1..NE of object
int	fdKB=0;      // event0 device (keypad) file descriptor
float	H=0;	// heading -PI..PI 
int	h1;	// temp coordinate 
int	h2;	// temp coordinate
float	HX=300;	// half horiz
float	HY=400;	// half vert
float	MA,MB,MC,MD,ME,MF,MG,MH,MI;	// 3D matrix coefficients
int	NE;	// number of edges of object
int	NV;	// number of vector points of object
float	OBJ[50];	// current object 
static float	OCTA[]={
	6,
	   0,7.5,0,
	   0,0,-2.5,
	   -2.5,0,0,
	   0,0,2.5,
	   2.5,0,0,
	   0,-7.5,0,
	14, -1,2,6,4,1,3,6,5,1, -2,3,4,5,2
	};	// vectors of an octahedron
float	P=0;	// pitch -PI..PI
float	PI=3.141592;	//PI 
float	SB;	// sin(bank)
float	SH;	// sin(heading)
float	SP;	// sin(pitch)
static float	TETRA[]={
	4, -4.5,-4.5,-4.5,  4.5,-4.5,-4.5, 0,4.5,0, 0,4.5,-4.5,
	8, -1, 2, 3, 1, 4, 2, 3, 4
	};	// vectors of a tetrahedron
float	U;	// computing 2D projection 
float	V;	// computing 2D projection
int	v1;	// temp coordinate 
int	v2;	// temp coordinate 
float	VECT[50][3];	// coordinate X, Y, Z of i'th vector i=1..NV of object
float	XV;	// computing 2D projection
float	XX;	// computing 2D projection
float	YV;	// computing 2D projection  
float	YY;	// computing 2D projection
float	ZV;	// computing 2D projection  
float	ZZ;	// computing 2D projection
float	X3;	// computing 2D projection 
float	Y3;	// computing 2D projection 
float	Z3;	// computing 2D projection
unsigned char buffer[50];	// print buffer

// gmlib global vars
static __u8 dt[64]={
    3,129,34,160,10,136,42,168,192,66,223,97,200,73,231,105,50,
    176,18,144,58,184,26,152,239,113,207,81,247,121,215,89,14,
    140,46,180,7,133,38,164,203,77,235,109,196,70,227,101,62,188,
    30,156,54,180,22,148,251,125,219,93,243,117,211,85}; // 0-255 dither table
__u8 *psave=NULL;     // screensave pointer
__u8 *fb0=NULL;      // framebuffer pointer
__u8 *wb0=NULL;     // workbuffer pointer
__u32 mpu=100;     // msec/update
int fdFB=0;       // fb0 file descriptor
int fdWB=0;      // wb0 file descriptor
__u32 fs=0;     // fb0 stride
__u32 fb=0;    // fb0 bytes
__u32 MX=0;   // xres (visible)
__u32 MY=0;  // yres (visible)
__u8 ppb=0; // pixels per byte
int bs=340; // border size
__u8 gs=0; // 8-bit grayscale
__u32 teu=0; // eink update time
__u32 tpu=0; // physics update time
__u32 __invalid_size_argument_for_IOC; // ioctl.h bug fix for tcc

//====================================
// plot - draw projection of the 3D object 
// usage plot(object)
//------------------------------------
void plot(float OBJ[]) {
  int i;
  NV=OBJ[0]; 
  for (i=0;i<NV;i++) { VECT[i][0]=OBJ[i*3+1]; VECT[i][1]=OBJ[i*3+2]; VECT[i][2]=OBJ[i*3+3]; }
  NE=OBJ[NV*3+1];
  for (i=0;i<NE;i++) { EDGE[i]=OBJ[NV*3+i+2]; }
  while (1) {
    for (i=0;i<600*800;i++) wb0[i]=255;
    CH=cos(H); SH=sin(H);
    CP=cos(P); SP=sin(P);
    CB=cos(B); SB=sin(B);
    MA= CB*CH-SP*SB*SH;
    MB=-CB*SH-SP*SB*CH;
    MC= SB*CP;
    MD= SH*CP;
    ME= CP*CH;
    MF= SP;	
    MG=-CH*SB-SH*SP*CB;
    MH= SH*SB-SP*CH*CB;
    MI= CP*CB;
    XV=-D*CP*SH; YV=-D*CP*CH; ZV=-D*SP;
    for (i=0;i<NE;i++) {
      EDGi=EDGE[i];
      EDG=EDGi; if (EDG<0) {EDG=-EDG;}; EDG=EDG-1;
      XX=VECT[(int)EDG][0]; YY=VECT[(int)EDG][1]; ZZ=VECT[(int)EDG][2];
      XX=XX-XV; YY=YY-YV; ZZ=ZZ-ZV;
      X3=MA*XX+MB*YY+MC*ZZ; 
      Y3=MD*XX+ME*YY+MF*ZZ; 
      Z3=MG*XX+MH*YY+MI*ZZ;
      U=HX+30*D*X3/Y3; V=HY+30*D*Z3/Y3;
      if (EDGi>0) { h2=(int)(U+.5); v2=(int)(V+.5); line(h1,v1,h2,v2,1); }
      h1=(int)(U+.5); v1=(int)(V+.5);
    }
    gmlib(GMLIB_UPDATE); 
      // keyboard input new Pitch, Heading, Bank, or Distance
      read(fdKB, buffer, 16);read(fdKB, buffer, 16);
      switch ((int)buffer[10]) {
        case 25: P=P+PI/18; break; // p +10dg
        case 35: H=H+PI/18; break; // h +10dg
        case 48: B=B+PI/18; break; // b +10dg
        case 44: D=D-2; break; // z zoom in
        case 22: D=D+2; break; // u zoom out
        case 19: P=0; H=0; B=0; D=13.5; break; // r home
        case 102: buffer[10]=255; return; // HOME quit program
      }
    
   
  }
}

//====================================
// gmlib - geekmaster function library
// op (init, update, close)
//------------------------------------
int gmlib(int op) {
    static struct update_area_t ua={0,0,600,800,fx_invert,NULL};
    static struct mxcfb_update_data ur={
        {0,0,600,800},WAVEFORM_MODE_AUTO,0,1,TEMP_USE_AMBIENT,0};
    static int eupcode; static void *eupdata=NULL;
    struct fb_var_screeninfo screeninfo;
    if (GMLIB_INIT==op) { teu=getmsec();
        // save original screen
        fdFB=open("/dev/fb0",O_RDWR); // open eink framebuffer fb0
        ioctl(fdFB,FBIOGET_VSCREENINFO,&screeninfo);
        ppb=8/screeninfo.bits_per_pixel;  // pixels per byte
        fs=screeninfo.xres_virtual/ppb;  // fb0 stride
        fb=screeninfo.yres_virtual/ppb; // fb0 bytes
        MX=screeninfo.xres;            // max X+1
        MY=screeninfo.yres;           // max Y+1
        ua.x2=MX; ua.y2=MY;
        ur.update_region.width=MX; ur.update_region.height=MY;
        fdWB=open("/tmp/wb0",O_RDWR|O_CREAT); // open work framebuffer wb0
        lseek(fdWB,MY*MX-1,SEEK_SET); write(fdWB,"",1); // create work buffer file
        fb0=(__u8 *)mmap(0,MY*fs,PROT_READ|PROT_WRITE,MAP_SHARED,fdFB,0); // map fb0
        wb0=(__u8 *)mmap(0,MY*MX,PROT_READ|PROT_WRITE,MAP_SHARED,fdWB,0); // map wb0
        if (fb>MY) { eupcode=MXCFB_SEND_UPDATE; eupdata=&ur; ur.update_mode=0; }
        else { eupcode=FBIO_EINK_UPDATE_DISPLAY_AREA; eupdata=&ua; }
        psave=malloc(fs*MY); memcpy(psave,fb0,fs*MY);
        memset(wb0,0,MY*MX); gmlib(GMLIB_UPDATE); teu+=300;
        memset(wb0,255,MY*MX); gmlib(GMLIB_UPDATE); teu+=80;
    } else if (GMLIB_UPDATE==op) {
        while (teu>getmsec()); // async update needs time
        dblit(); ioctl(fdFB,eupcode,eupdata);
    } else if (GMLIB_CLOSE==op) {
        gmlib(GMLIB_UPDATE); sleep(3);
        memcpy(fb0,psave,fs*MY); // restore display
        free(psave);
        gmlib(GMLIB_UPDATE);   // update display
        munmap(wb0,MY*MX);    // unmap wb0
        close(fdWB);         // close wb0
        remove("/tmp/wb0"); // remove wb0
        munmap(fb0,MY*fs); // unmap fb0
        close(fdFB);      // close fb0
    } else { return -1; }
    return 0;
}

//========================================
// setpx - draw pixel to 8-bit work buffer
// x,y:screen coordinates, c:color(0-255).
//----------------------------------------
inline void setpx(int x,int y,int c) {
    wb0[y*MX+x]=c;
}

//==========================
// dblit - dither wb0 to fb0
//--------------------------
void dblit(void) {
    int x,y,ys; 
    __u8 p,d,*pi,*po,*pd;
    pi=wb0-1; po=fb0-1;
    if (gs) { // 16-color dither
      if (ppb/2) { // 2 px/byte
        for (y=0;y<MY;y++) { pd=dt+(y&7)*8+7;
            for (x=0;x<MX;x+=8) { pd-=8;
                p=*pi++; d=*pd++; 
                *po++=((p&240)+(d-(*pi&15)*255/15>>4&16)-16)&240|
                    ((*pi++&240)+(*pd++-(*pi&15)*255/15>>4&16)-16)/16&15;
                p=*pi++; d=*pd++; 
                *po++=((p&240)+(d-(*pi&15)*255/15>>4&16)-16)&240|
                    ((*pi++&240)+(*pd++-(*pi&15)*255/15>>4&16)-16)/16&15;
                p=*pi++; d=*pd++; 
                *po++=((p&240)+(d-(*pi&15)*255/15>>4&16)-16)&240|
                    ((*pi++&240)+(*pd++-(*pi&15)*255/15>>4&16)-16)/16&15;
                p=*pi++; d=*pd++; 
                *po++=((p&240)+(d-(*pi&15)*255/15>>4&16)-16)&240|
                    ((*pi++&240)+(*pd++-(*pi&15)*255/15>>4&16)-16)/16&15;
            }
        }
      } else { // 1 px/byte
        for (y=0;y<MY;y++) { pd=dt+(y&7)*8+7;
            for (x=0;x<MX;x+=8) { pd-=8;
//                *po++=(*pi++&240)+(*pd++-(*pi&15)*255/15>>4&16)-16;
                *po++=(*pi++&240)+(((*pi&15)*4080>>8)+*pd++>>4&16)-16;
                *po++=(*pi++&240)+(((*pi&15)*4080>>8)+*pd++>>4&16)-16;
                *po++=(*pi++&240)+(((*pi&15)*4080>>8)+*pd++>>4&16)-16;
                *po++=(*pi++&240)+(((*pi&15)*4080>>8)+*pd++>>4&16)-16;
                *po++=(*pi++&240)+(((*pi&15)*4080>>8)+*pd++>>4&16)-16;
                *po++=(*pi++&240)+(((*pi&15)*4080>>8)+*pd++>>4&16)-16;
                *po++=(*pi++&240)+(((*pi&15)*4080>>8)+*pd++>>4&16)-16;
                *po++=(*pi++&240)+(((*pi&15)*4080>>8)+*pd++>>4&16)-16;
            } po+=(fs-MX);
        }
      }
    } else { // 2-color dither
      if (ppb/2) { // 2 px/byte *** broken (high contrast) ***
        for (y=0;y<MY;y++) { pd=dt+(y&7)*8+7; ys=(y&7)*8;
            for (x=0;x<MX;x+=8) { pd-=8;
                p=*pi++; d=*pd++; *po++=d-p>>4&240|*pd++-*pi++>>8&15;
                p=*pi++; d=*pd++; *po++=d-p>>4&240|*pd++-*pi++>>8&15;
                p=*pi++; d=*pd++; *po++=d-p>>4&240|*pd++-*pi++>>8&15;
                p=*pi++; d=*pd++; *po++=d-p>>4&240|*pd++-*pi++>>8&15;
            }
        }

      } else { // 1 px/byte
        for (y=0;y<MY;y++) { pd=dt+(y&7)*8+7;
            for (x=0;x<MX;x+=8) { pd-=8;
                *po++=*pd++-*pi++>>8; *po++=*pd++-*pi++>>8;
                *po++=*pd++-*pi++>>8; *po++=*pd++-*pi++>>8;
                *po++=*pd++-*pi++>>8; *po++=*pd++-*pi++>>8;
                *po++=*pd++-*pi++>>8; *po++=*pd++-*pi++>>8;
            } po+=(fs-MX);
        }
      }
    }
}

//==============================================
// circle - optimized midpoint circle algorithm
//----------------------------------------------
void circle(int cx,int cy,int r) {
    int e=-r,x=r,y=0;
    while (x>y) {
        setpx(cx+y,cy-x,255); setpx(cx+x,cy-y,159);
        setpx(cx+x,cy+y,95); setpx(cx+y,cy+x,31);
        setpx(cx-y,cy+x,0); setpx(cx-x,cy+y,63);
        setpx(cx-x,cy-y,127); setpx(cx-y,cy-x,191);
        e+=y; y++; e+=y;
        if (e>0) { e-=x; x-=1; e-=x; }
    }
}

//==================================
// line - Bresenham's line algorithm
//----------------------------------
void line(int x0,int y0,int x1,int y1,int c) {
    int dx,ny,sx,sy,e,e2;
    if (x1>x0) { dx=x1-x0; sx=1; } else { dx=x0-x1; sx=-1; }
    if (y1>y0) { ny=y0-y1; sy=1; } else { ny=y1-y0; sy=-1; }
    e=dx+ny;
    for (;;) { circle(x0,y0,c);
        e2=e+e;
        if (x0==x1 && y0==y1) { break; }
        if (e2>ny) { e+=ny; x0+=sx; }
        if (e2<dx) { e+=dx; y0+=sy; }
    }
}


//====================================
// getmsec - get msec since first call
// (tick counter wraps every 12 days)
//------------------------------------
int getmsec(void) {
    int tc;
    static int ts=0;
    struct timeval tv;
    gettimeofday(&tv,NULL); tc=tv.tv_usec/1000+1000*(0xFFFFF&tv.tv_sec);
    if (0==tc) { ts=tc; }
    return tc-ts;
}

//=============================
// main 
//-----------------------------                  
int main(void) {
  gmlib(GMLIB_INIT);
  system("lipc-set-prop com.lab126.powerd preventScreenSaver 1; killall -stop cvm");
  fdKB = open("/dev/input/event0", O_RDONLY); // keyboard device
  while (1) {
      system("eips -c"); 
      system("eips 15  4 \"    _/_/_/    3D\"");
      system("eips 15  5 \"         _/  5.0 _/_/_/\"");
      system("eips 15  6 \"    _/_/    by  _/    _/\"");
      system("eips 15  7 \"       _/  PoP _/    _/\"");
      system("eips 15  8 \"_/_/_/        _/    _/\"");
      system("eips 15  9 \"             _/_/_/\"");
      system("eips 15 15 \"Select object to plot:\"");
      system("eips 15 16 \"Q    - BLOCK\"");
      system("eips 15 17 \"W    - OCTAHEDRON\"");
      system("eips 15 18 \"E    - TETRAHEDRON\"");
      system("eips 15 19 \"R    - DODECAHEDRON\"");
      system("eips 15 20 \"Home - Quit\"");
      system("eips 15 28 \"When object has been plot:\"");
      system("eips 15 29 \"R      reset view\"");
      system("eips 15 30 \"P      + pitch\"");
      system("eips 15 31 \"H      + heading\"");
      system("eips 15 32 \"B      + bank\"");
      system("eips 15 33 \"Z      zoom\"");
      system("eips 15 34 \"U      un zoom\"");
      system("eips 15 35 \"Home - Select new object\"");
      read(fdKB, buffer, 16);read(fdKB, buffer, 16);
      if (16==(int) buffer[10]) plot(BLOCK);
      if (17==(int) buffer[10]) plot(OCTA);
      if (18==(int) buffer[10]) plot(TETRA);
      if (19==(int) buffer[10]) plot(DODECA);
      if (102==(int) buffer[10]) {system("killall -cont cvm; lipc-set-prop com.lab126.powerd preventScreenSaver 0"); gmlib(GMLIB_CLOSE); return 0; } 
  } 
}