#!/usr/bin/perl

use strict;
use warnings;
use Image::Magick;

# 1 inch = 25.4 mm
my $INCH = 25.4;

# dots per inch, standard value
my $DPI = 72;

# this allows to crank up the resolution.
# Set to at least 1, and be conservative with
# the maximum (10 is probably too much).
my $ZOOM = 2;

# pixels per millimeter, assuming one pixel per dot
# Adjust for higher resolutions using $ZOOM
my $PPM = ($DPI / $INCH) * $ZOOM;

# line width, in pixels for convenience. Use an even number!
my $LW = 20 * $ZOOM;

# box size, again in pixels.
my $BS = 100 * $ZOOM;

my %FORMATS = (
	'A4' => { width => 210, height => 297 },
	'A5' => { width => 148, height => 210 },
	'letter' => { width => 216, height => 279 },
);

if ($#ARGV != 0) {
	die "Usage: $0 <format>\n";
}

my $format = $ARGV[0];
die ("unknown format \"$format\"; supported formats are: ". join(" ", sort keys %FORMATS)) unless defined $FORMATS{$format};

my $w = int($FORMATS{$format}->{width} * $PPM);
my $h = int($FORMATS{$format}->{height} * $PPM);

print "$format: ${w}x${h}\n";

my $i = Image::Magick->new;
$i->Set(size => "${w}x${h}");
$i->ReadImage("xc:white");

# top-left, bottom-right
my $tlx = $LW / 2;
my $tly = $LW / 2;
my $brx = $w - $tlx;
my $bry = $h - $tly;

&drawRectangle($i, $tlx, $tly, $brx, $bry);

# center
my $cx = $w / 2;
my $cy = $h / 2;

$tlx = $cx - $BS / 2;
$tly = $cy - $BS / 2;
$brx = $cx + $BS / 2;
$bry = $cy + $BS / 2;

&drawRectangle($i, $tlx, $tly, $brx, $bry);

my $error = $i->Write("$format.png");
die $error if $error;

# Don't ask. Directly converting to pdf will
# produce files that the Kindle can't open
# (out of memory). The alternative works nicely,
# though.
#system("convert $format.png $format.pdf");
system("convert $format.png $format.eps");
system("epstopdf $format.eps");
system("rm $format.eps");



sub drawRectangle {
	my ($i, $tx,$ty,$bx,$by) = @_;
	$i->Draw (
	  primitive   => "rectangle",
	  stroke      => "black",
	  strokewidth => $LW,
	  fill        => "white",
	  points      => "$tx,$ty, $bx,$by"
	);
}

