/* A Keyboard demonstation.
 */

/* Copyright (C) 2012 JoppyFurr
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy 
 * of this software and associated documentation files (the "Software"), to 
 * deal in the Software without restriction, including without limitation the 
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or 
 * sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 * DEALINGS IN THE SOFTWARE.
 */

/* Thanks Geekmaster for helping out with some code suggestions :3 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <pthread.h>


#include "constants.h"
#include "drawing.h"

/* Include the font */
#include "font.h"
#include "text.h"

/* Include the keyboard */
#include "keyboard.h"

/* Include terminal output */
#include "term.h"

/* Finger states */
#define FRESH_UP 0
#define IDLE_UP 1
#define FRESH_DOWN 2
#define IDLE_DOWN 3

/* Parameters passed to the terminal output function */
typedef struct Output_parameters_struct { 
    unsigned char *frame_buffer;
    int file_fd;
    unsigned char *character_buffer;
    int *state;
} Output_parameters;

void draw_interface(unsigned char *frame_buffer, int keyboard_modifier);

/* Update the eInk when there has been a change */
void *update_thread(void *state) {
    while(*(int *)state != EXIT) {
        if(*(int *)state == UPDATE) {
            
            if(*(int *)state != EXIT) {
                *(int *)state = IDLE; /* Reset state */
            }
            update(0);
        }
        else if(*(int *)state == REFRESH) {
            if(*(int *)state != EXIT) {
                *(int *)state = IDLE; /* Reset state */
            }
            update(1);
        }
        usleep(1000000 / 60);
    }
}

/* A thread for the terminal output window */
void *output_thread(void *params) {
    Output_parameters *parameters = (Output_parameters *)params;
    terminal_output(parameters->frame_buffer, parameters->file_fd, parameters->character_buffer, parameters->state);
}

int main(int argc, char **argv) {
    int frame_buffer_fd = 0;
    int touch_screen_fd = 0;
    unsigned char *frame_buffer = NULL;
    unsigned char event_buffer[16];

    /* Terminal things */
    pthread_t outputer;
    unsigned char character_buffer[1921]; /* 80x24 characters plus one byte for scroll data */
    Output_parameters output_params;
    int pty_fd = 0;
    pid_t pty_pid = 0;

    /* A text entry demo */
    int typed_index = 0;
    unsigned char c = '\0';
    int keyboard_modifier = NONE;
    int past_keyboard_modifier = NONE;

    /* First finger variables */
    int finger = IDLE_UP;
    int x = 300; 
    int y = 400;
    
    /* Second finger variables */
    int finger2 = IDLE_UP;
    int x2 = 0;
    int y2 = 0;
    int f2_mode = 0;
    
    int i = 0;
    int j = 0;
    
    /* Threads */
    pthread_t updater = 0;
    int state = 0;
    
    int sync = 0;

    /* Pause the Kindle framework */
    system(PAUSE_FRAMEWORK);

    /* Open files related to the screen */
    frame_buffer_fd = open(FRAME_BUFFER, O_RDWR);
    if(frame_buffer_fd == -1) {
        fprintf(stderr, "Error: Could not open frame buffer.\n");
        exit(EXIT_FAILURE);
    }
    frame_buffer = mmap(0, STRIDE * HEIGHT, PROT_READ | PROT_WRITE, MAP_SHARED, frame_buffer_fd, 0);
    touch_screen_fd = open(TOUCH_SCREEN, O_RDONLY);   
    if(touch_screen_fd == -1) {
        fprintf(stderr, "Error: Could not open touch screen.\n");
        exit(EXIT_FAILURE);
    }

    /* Draw a background */
    draw_interface(frame_buffer, keyboard_modifier);
    state = UPDATE;


    /* A thread to update the eInk */
    pthread_create(&updater, NULL, update_thread, (void *)&state);


    /* A process to run bash */
    pty_pid = forkpty(&pty_fd, NULL, NULL, NULL);
    if(!pty_pid) {
        printf("Welcome to JoppyTerm Alpha 0.0\n\n");
        execlp("/bin/sh", "/bin/sh", NULL);
        exit(EXIT_SUCCESS);
    }

    /* A thread to show output */                                                                                
    output_params.frame_buffer = frame_buffer;                                                                   
    output_params.file_fd = pty_fd;                                                                            
    output_params.character_buffer = character_buffer;                                                           
    output_params.state = &state;                                                                                
    pthread_create(&outputer, NULL, output_thread, (void *)&output_params); 
 
    /* Loop through touch screen input, writing pretty things to the screen */    
    sync = 1; /* An inital fake sync to render the first frame before any input is given */
    while(state != EXIT) {
        /* === Process input === */
        while(!sync) {
            read(touch_screen_fd, event_buffer, 16);
            if(event_buffer[0x08] == 0x03) {
                switch(event_buffer[0x0A]) {
                    case 0x2F:
                        /* When f2_mode is nonzero, data is treated as being for
                            the second finger */
                        f2_mode = event_buffer[0x0C];
                        break;
                    case 0x35:
                        /* X Coordinate change */
                        if(!f2_mode) {
                            x = (WIDTH * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                        }
                        else {
                            x2 = (WIDTH * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                        }
                        break;
                    case 0x36:
                        /* Y Coordinate change */
                        if(!f2_mode) {
                            y = (HEIGHT * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                        }
                        else {
                            y2 = (HEIGHT * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                        }
                        break;
                    case 0x39:
                        /* Finger up / down */
                        if(event_buffer[0x0C] == 0x00) {
                            finger = FRESH_DOWN;
                        }
                        else if(event_buffer[0x0C] == 0x01) {
                            finger2 = FRESH_DOWN;
                        }
                        else {
                            if(finger2 == IDLE_DOWN || finger2 == FRESH_DOWN) {
                                finger2 = FRESH_UP;
                            }
                            else {
                                finger = FRESH_UP;
                            }
                            
                        }
                        break;
                    default:
                        break;
                }
            }
            if(event_buffer[0x08] == 0x00) {
                sync = 1; 
            }
        }
        
        /* Logic */
        if(finger == FRESH_DOWN) { /* See if we tried to type */
            finger = IDLE_DOWN;
            past_keyboard_modifier = keyboard_modifier;
            c = read_keyboard(x, y, keyboard_modifier);
            if(c >= 0x20 && c <= 0x7E || c == 0x0A) { /* Printable characters */
                write(pty_fd, &c, 1); /* Write this character to the pty */
                
                keyboard_modifier = NONE;
            }
            else if(c == SHIFT_KEY) {
                if(keyboard_modifier == NONE) {
                    keyboard_modifier = SHIFT;
                }
                else if(keyboard_modifier == SHIFT) {
                    keyboard_modifier = NONE;
                }
            }
            else if(c == BACKSPACE_KEY) {
                c = 0x08;
                write(pty_fd, &c, 1); /* Write a backspace */
            }
            if(keyboard_modifier != past_keyboard_modifier) {
                draw_keyboard(frame_buffer, keyboard_modifier);
                if(state == IDLE) {
                    state = UPDATE;
               }
            }
        } 
         
        if(finger == FRESH_UP) { /* A finger has just now been released */
            finger = IDLE_UP;
            if(x > 500 && y < 100) { /* Release in the top right to exit */
                printf("Exiting.\n");
                break;
            }
            if(x < 100 && y < 100) { /* Release in the top lift to refresh the eInk */
                state = REFRESH;
                while(state == REFRESH) {
                    usleep(1000000/60);
                }
                
                /* Draw a new terminal box and keyboard */
                draw_interface(frame_buffer, keyboard_modifier);
                 
                /* Re-render text */
                render(character_buffer, frame_buffer);
                state = UPDATE;
                x = 300; /* Reset the finger position to screen centre */
                y = 400;
            }
        }

        sync = 0;
    }

    state = EXIT;
    pthread_join(outputer, NULL);
    pthread_join(updater, NULL);

    munmap(frame_buffer, STRIDE * HEIGHT);
    close(frame_buffer_fd);

    /* Resume the Kindle framework */
    system(RESUME_FRAMEWORK);

    return EXIT_SUCCESS;
}

void draw_interface(unsigned char *frame_buffer, int keyboard_modifier) {
                int offset = 0;
                
                clear(frame_buffer);                                                                                             
                offset = (WIDTH - (CHARACTER_WIDTH * TERMINAL_WIDTH)) / 2;                                                       
                draw_keyboard(frame_buffer, keyboard_modifier);                                                                               
                draw_box(frame_buffer, offset - 2, offset - 2, CHARACTER_WIDTH * TERMINAL_WIDTH + 4, CHARACTER_HEIGHT * TERMINAL_HEIGHT + 4);
                draw_box(frame_buffer, offset - 4, offset - 4, CHARACTER_WIDTH * TERMINAL_WIDTH + 8, CHARACTER_HEIGHT * TERMINAL_HEIGHT + 8);
}
