/* Text printing functions.
 */

/* Copyright (C) 2012 JoppyFurr
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy 
 * of this software and associated documentation files (the "Software"), to 
 * deal in the Software without restriction, including without limitation the 
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or 
 * sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 * DEALINGS IN THE SOFTWARE.
 */

/* Thanks Geekmaster for helping out with some code suggestions :3 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <pthread.h>

#include "constants.h"
#include "drawing.h"
#include "font.h"
#include "text.h"

/* Render the current text array to the frame buffer */
void render(unsigned char *character_buffer, unsigned char *frame_buffer) {
    int offset = (WIDTH - CHARACTER_WIDTH * TERMINAL_WIDTH) / 2; /* Centre the terminal */
    int character_x = 0;
    int character_y = 0;
    int pixel_x = 0;
    int pixel_y = 0;
    unsigned char c;
    
    for(character_y = 0; character_y < TERMINAL_HEIGHT; character_y++) {
        for(character_x = 0; character_x < TERMINAL_WIDTH; character_x++) {
            
            /* Find which character we are going to blit */
            c = character_buffer[character_x + character_y * TERMINAL_WIDTH];
            if(c < 0x20 || c > 0x7E) { /* Characters covered by the font */
                c = 0x20; /* Space character for unknown characters */
            }
            
            /* Copy this character from the font to the frame buffer */
            for(pixel_y = 0; pixel_y < CHARACTER_HEIGHT; pixel_y++) {
                for(pixel_x = 0; pixel_x < CHARACTER_WIDTH; pixel_x++) {
                    /* Pixel in frame buffer */
                    frame_buffer[offset + character_x * CHARACTER_WIDTH + pixel_x 
                     + STRIDE * (offset + character_y * CHARACTER_HEIGHT + pixel_y)] = 
                     /* Pixel in font */
                     255 * font[((c - 0x20) % 0x10) * CHARACTER_WIDTH + pixel_x 
                     + CHARACTER_WIDTH * 0x10 * (((c - 0x20) / 0x10) * CHARACTER_HEIGHT+ pixel_y)];
                }
            }
        }
    }
}

/* Blit a character to the frame buffer */
void blit_char(unsigned char *frame_buffer, unsigned char c, int x, int y, int scale) {
    int pixel_x = 0;
    int pixel_y = 0;
    
    for(pixel_y = 0; pixel_y < CHARACTER_HEIGHT * scale; pixel_y++) {
        for(pixel_x = 0; pixel_x < CHARACTER_WIDTH * scale; pixel_x++) {
            
            /* Pixel in frame buffer */
            frame_buffer[x + pixel_x + STRIDE * (y + pixel_y)] =
            
            /* Pixel in font */
            255 * font[((c - 0x20) % 0x10) * CHARACTER_WIDTH + pixel_x / scale
            + CHARACTER_WIDTH * 0x10 * (((c - 0x20) / 0x10) * CHARACTER_HEIGHT + pixel_y / scale)];
        }
    }
}

/* Blit a null terminated string to the frame buffer */
/* Assumes this will not try to go off the side of the screen */
void blit_string(unsigned char *frame_buffer, unsigned char *s, int x, int y, int scale) {
    int offset = 0;
    while(*s) {
        blit_char(frame_buffer, *s, x + offset, y, scale);
        s++;
        offset += scale * CHARACTER_WIDTH;
    }
}
