/* A Keyboard demonstation.
 */

/* Copyright (C) 2012 JoppyFurr
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy 
 * of this software and associated documentation files (the "Software"), to 
 * deal in the Software without restriction, including without limitation the 
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or 
 * sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 * DEALINGS IN THE SOFTWARE.
 */

/* Thanks Geekmaster for helping out with some code suggestions :3 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <pthread.h>


#include "constants.h"
#include "drawing.h"

/* Include the font */
#include "font.h"
#include "text.h"

/* Include the keyboard */
#include "keyboard.h"

/* Finger states */
#define FRESH_UP 0
#define IDLE_UP 1
#define FRESH_DOWN 2
#define IDLE_DOWN 3

void draw_interface(unsigned char *frame_buffer, int keyboard_modifier);

/* Update the eInk when there has been a change */
void *update_thread(void *state) {
    while(*(int *)state != EXIT) {
        if(*(int *)state == UPDATE) {
            update(0);
        }
        else if(*(int *)state == REFRESH) {
            update(1);
        }
        if(*(int *)state != EXIT) {
            *(int *)state = IDLE; /* Reset state */
        }
        usleep(1000000 / 60);
    }
}

int main(int argc, char **argv) {
    int frame_buffer_fd = 0;
    int touch_screen_fd = 0;
    unsigned char *frame_buffer = NULL;
    unsigned char event_buffer[16];
    unsigned char character_buffer[1920]; /* 80x24 characters */
    
    /* A text entry demo */
    unsigned char typed[51]; /* A small text entry buffer for a demonstration */
    int typed_index = 0;
    unsigned char c = '\0';
    int keyboard_modifier = NONE;
    int past_keyboard_modifier = NONE;

    /* First finger variables */
    int finger = IDLE_UP;
    int x = 300; 
    int y = 400;
    
    /* Second finger variables */
    int finger2 = IDLE_UP;
    int x2 = 0;
    int y2 = 0;
    int f2_mode = 0;
    
    int i = 0;
    int j = 0;
    
    /* Threads */
    pthread_t updater = 0;
    int display_state = 0;
    
    int sync = 0;

    /* Count the touches so that we can show the user */
    int touches = 0;

    /* Pause the Kindle framework */
    system(PAUSE_FRAMEWORK);

    /* Open files related to the screen */
    frame_buffer_fd = open(FRAME_BUFFER, O_RDWR);
    if(frame_buffer_fd == -1) {
        fprintf(stderr, "Error: Could not open frame buffer.\n");
        exit(EXIT_FAILURE);
    }
    frame_buffer = mmap(0, STRIDE * HEIGHT, PROT_READ | PROT_WRITE, MAP_SHARED, frame_buffer_fd, 0);
    touch_screen_fd = open(TOUCH_SCREEN, O_RDONLY);   
    if(touch_screen_fd == -1) {
        fprintf(stderr, "Error: Could not open touch screen.\n");
        exit(EXIT_FAILURE);
    }

    /* Fill the text entry buffer with nulls */
    for(i = 0; i < 51; i++) {
        typed[i] = '\0';
    }

    /* Draw a background */
    draw_interface(frame_buffer, keyboard_modifier);
    display_state = UPDATE;

    pthread_create(&updater, NULL, update_thread, (void *)&display_state);
    
    /* Loop through touch screen input, writing pretty things to the screen */    
    sync = 1; /* An inital fake sync to render the first frame before any input is given */
    while(1) {
        /* === Process input === */
        while(!sync) {
            read(touch_screen_fd, event_buffer, 16);
            if(event_buffer[0x08] == 0x03) {
                switch(event_buffer[0x0A]) {
                    case 0x2F:
                        /* When f2_mode is nonzero, data is treated as being for
                            the second finger */
                        f2_mode = event_buffer[0x0C];
                        break;
                    case 0x35:
                        /* X Coordinate change */
                        if(!f2_mode) {
                            x = (WIDTH * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                        }
                        else {
                            x2 = (WIDTH * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                        }
                        break;
                    case 0x36:
                        /* Y Coordinate change */
                        if(!f2_mode) {
                            y = (HEIGHT * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                        }
                        else {
                            y2 = (HEIGHT * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                        }
                        break;
                    case 0x39:
                        /* Finger up / down */
                        if(event_buffer[0x0C] == 0x00) {
                            printf("Finger 1 down.\n");
                            touches++;
                            finger = FRESH_DOWN;
                        }
                        else if(event_buffer[0x0C] == 0x01) {
                            printf("Finger 2 down.\n");
                            touches++;
                            finger2 = FRESH_DOWN;
                        }
                        else {
                            if(finger2 == IDLE_DOWN || finger2 == FRESH_DOWN) {
                                finger2 = FRESH_UP;
                                printf("Finger 2 up.\n");
                            }
                            else {
                                finger = FRESH_UP;
                                printf("Finger 1 up.\n");
                            }
                            
                        }
                        break;
                    default:
                        break;
                }
            }
            if(event_buffer[0x08] == 0x00) {
                sync = 1; 
            }
        }
        
        /* Logic */
        if(finger == FRESH_DOWN) { /* See if we tried to type */
            finger = IDLE_DOWN;
            past_keyboard_modifier = keyboard_modifier;
            c = read_keyboard(x, y, keyboard_modifier);
            if(c >= 0x20 && c <= 0x7E) {
                typed[typed_index++ % 50] = c; /* Re-write over the 50-character area if we fill it up */
                keyboard_modifier = NONE;
            }
            else if(c == SHIFT_KEY) {
                if(keyboard_modifier == NONE) {
                    keyboard_modifier = SHIFT;
                }
                else if(keyboard_modifier == SHIFT) {
                    keyboard_modifier = NONE;
                }
            }
            else if(c == BACKSPACE_KEY) {
                typed_index += 50;
                typed[--typed_index % 50] = ' ';
                typed_index %= 50;
            }
            if(keyboard_modifier != past_keyboard_modifier) {
                draw_keyboard(frame_buffer, keyboard_modifier);
                if(display_state == IDLE) {
                    display_state == UPDATE;
               }
            }
        } 
         
        if(finger == FRESH_UP) { /* A finger has just now been released */
            finger == IDLE_UP;
            if(x > 500 && y < 100) { /* Release in the top right to exit */
                printf("Exiting.\n");
                break;
            }
            if(x < 100 && y < 100) { /* Release in the top lift to refresh the eInk */
                display_state = REFRESH;
                while(display_state == REFRESH) {
                    usleep(1000000/60);
                }
                
                /* Draw a new terminal box and keyboard */
                draw_interface(frame_buffer, keyboard_modifier);
                 
                /* Re-render text */
                render(character_buffer, frame_buffer);
                display_state = UPDATE;
                x = 300; /* Reset the finger position to screen centre */
                y = 400;
            }
        }

        /* Render console text */
        for(i = 0; i < TERMINAL_WIDTH * TERMINAL_HEIGHT; i++) {
            character_buffer[i] = ' '; /* Clear the terminal */
        }
        
        sprintf(character_buffer + 80 * 0, "Hello, world!");
        
        sprintf(character_buffer + 80 * 2, "This is a test of my keyboard for the Kindle Touch.");
        sprintf(character_buffer + 80 * 3, "Currently working:");
        
        sprintf(character_buffer + 80 * 5, "  - Letters, numbers, symbols, and space.");
        sprintf(character_buffer + 80 * 6, "  - Shift keys.");
        sprintf(character_buffer + 80 * 7, "  - Backspace.");
        
        sprintf(character_buffer + 80 * 9, "Not yet assigned:");
        sprintf(character_buffer + 80 * 10, "  - Other modifiers.");
        sprintf(character_buffer + 80 * 11, "  - Arrows.");
        sprintf(character_buffer + 80 * 12, "  - Tab, delete, enter.");
        
        sprintf(character_buffer + 80 * 14, "Release a touch in the top left corner to refresh the screen.");
        sprintf(character_buffer + 80 * 15, "Release a touch in the top right corner to quit the program.");
        
        sprintf(character_buffer + 80 * 20, "                    You have touched the screen %i times :D", touches);
       
        sprintf(character_buffer + 80 * 22, "Try typing something:");
        sprintf(character_buffer + 80 * 22 + 22, "%s", typed);
        
        render(character_buffer, frame_buffer);
        
        blit_char(frame_buffer, 'A', 50, 400, 1);
        blit_char(frame_buffer, 'A', 100, 400, 2);
        blit_char(frame_buffer, 'A', 200, 400, 3);
        blit_char(frame_buffer, 'A', 300, 400, 4);
        blit_char(frame_buffer, 'A', 400, 400, 5);
        blit_char(frame_buffer, 'A', 500, 400, 6);
        
        if(display_state == IDLE) {
            display_state = UPDATE;
        }
        
        sync = 0;
    }

    display_state = EXIT;
    pthread_join(updater, NULL);

    munmap(frame_buffer, STRIDE * HEIGHT);
    close(frame_buffer_fd);

    /* Resume the Kindle framework */
    system(RESUME_FRAMEWORK);

    return EXIT_SUCCESS;
}

void draw_interface(unsigned char *frame_buffer, int keyboard_modifier) {
                int offset = 0;
                
                clear(frame_buffer);                                                                                             
                offset = (WIDTH - (CHARACTER_WIDTH * TERMINAL_WIDTH)) / 2;                                                       
                draw_keyboard(frame_buffer, keyboard_modifier);                                                                               
                draw_box(frame_buffer, offset - 2, offset - 2, CHARACTER_WIDTH * TERMINAL_WIDTH + 4, CHARACTER_HEIGHT * TERMINAL_HEIGHT + 4);
                draw_box(frame_buffer, offset - 4, offset - 4, CHARACTER_WIDTH * TERMINAL_WIDTH + 8, CHARACTER_HEIGHT * TERMINAL_HEIGHT + 8);
}
