/* A Text-output demonstation.
 */

/* Copyright (C) 2012 JoppyFurr
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy 
 * of this software and associated documentation files (the "Software"), to 
 * deal in the Software without restriction, including without limitation the 
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or 
 * sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 * DEALINGS IN THE SOFTWARE.
 */

/* Thanks Geekmaster for helping out with some code suggestions :3 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <pthread.h>

/* Include the font */
#include "font.h"

/* Frame buffer dimensions */
#define WIDTH 600
#define HEIGHT 800
#define STRIDE 608

/* Terminal dimensions, in characters */
#define TERMINAL_WIDTH 80
#define TERMINAL_HEIGHT 24

/* Expected locations of devices */
/* Note: The touch screen may not be guaranteed to be at event3 */
#define FRAME_BUFFER "/dev/fb0"
#define TOUCH_SCREEN "/dev/input/event3"

#define PAUSE_FRAMEWORK "killall -STOP Xorg cvm"
#define RESUME_FRAMEWORK "killall -CONT cvm Xorg"

/* Finger states */
#define UP 0
#define DOWN 1

/* Display states */
#define IDLE 0
#define UPDATE 1
#define REFRESH 2
#define EXIT 3


void update(int refresh);
void draw_grid(unsigned char *frame_buffer);

/* Update the eInk when there has been a change */
void *update_thread(void *state) {
    while(*(int *)state != EXIT) {
        if(*(int *)state == UPDATE) {
            update(0);
        }
        else if(*(int *)state == REFRESH) {
            update(1);
        }
        if(*(int *)state != EXIT) {
            *(int *)state = IDLE; /* Reset state */
        }
        usleep(1000000 / 60);
    }
}

/* Render the current text array to the frame buffer */
void render(unsigned char *character_buffer, unsigned char *frame_buffer) {
    int offset = (WIDTH - CHARACTER_WIDTH * TERMINAL_WIDTH) / 2; /* Centre the terminal */
    int character_x = 0;
    int character_y = 0;
    int pixel_x = 0;
    int pixel_y = 0;
    unsigned char c;
    
    for(character_y = 0; character_y < TERMINAL_HEIGHT; character_y++) {
        for(character_x = 0; character_x < TERMINAL_WIDTH; character_x++) {
            
            /* Find which character we are going to blit */
            c = character_buffer[character_x + character_y * TERMINAL_WIDTH];
            if(c < 0x20 || c > 0x7E) { /* Characters covered by the font */
                c = 0x20; /* Space character for unknown characters */
            }
            
            /* Copy this character from the font to the frame buffer */
            for(pixel_y = 0; pixel_y < CHARACTER_HEIGHT; pixel_y++) {
                for(pixel_x = 0; pixel_x < CHARACTER_WIDTH; pixel_x++) {
                    /* Pixel in frame buffer */
                    frame_buffer[offset + character_x * CHARACTER_WIDTH + pixel_x 
                     + STRIDE * (offset + character_y * CHARACTER_HEIGHT + pixel_y)] = 
                     /* Pixel in font */
                     255 * font[((c - 0x20) % 0x10) * CHARACTER_WIDTH + pixel_x 
                     + CHARACTER_WIDTH * 0x10 * (((c - 0x20) / 16) * CHARACTER_HEIGHT+ pixel_y)];
                }
            }
        }
    }
}

int main(int argc, char **argv) {
    int frame_buffer_fd = 0;
    int touch_screen_fd = 0;
    unsigned char *frame_buffer = NULL;
    unsigned char event_buffer[16];
    unsigned char character_buffer[1920]; /* 80x24 characters */
    
    /* First finger variables */
    int finger = UP;
    int x = 300; 
    int y = 400;
    
    /* Second finger variables */
    int finger2 = UP;
    int x2 = 0;
    int y2 = 0;
    int f2_mode = 0;
    
    int i = 0;
    int j = 0;
    
    /* Threads */
    pthread_t updater = 0;
    int display_state = 0;
    
    int sync = 0;

    /* Count the touches so that we can show the user */
    int touches = 0;

    /* Pause the Kindle framework */
    system(PAUSE_FRAMEWORK);

    /* Open files related to the screen */
    frame_buffer_fd = open(FRAME_BUFFER, O_RDWR);
    if(frame_buffer_fd == -1) {
        fprintf(stderr, "Error: Could not open frame buffer.\n");
        exit(EXIT_FAILURE);
    }
    frame_buffer = mmap(0, STRIDE * HEIGHT, PROT_READ | PROT_WRITE, MAP_SHARED, frame_buffer_fd, 0);
    touch_screen_fd = open(TOUCH_SCREEN, O_RDONLY);   
    if(touch_screen_fd == -1) {
        fprintf(stderr, "Error: Could not open touch screen.\n");
        exit(EXIT_FAILURE);
    }

    /* Draw a background */
    draw_grid(frame_buffer);
    display_state = UPDATE;

    pthread_create(&updater, NULL, update_thread, (void *)&display_state);
    
    /* Loop through touch screen input, writing pretty things to the screen */    
    sync = 1; /* An inital fake sync to render the first frame before any input is given */
    while(1) {
        /* === Process input === */
        while(!sync) {
            read(touch_screen_fd, event_buffer, 16);
            if(event_buffer[0x08] == 0x03) {
                switch(event_buffer[0x0A]) {
                    case 0x2F:
                        /* When f2_mode is nonzero, data is treated as being for
                            the second finger */
                        f2_mode = event_buffer[0x0C];
                        break;
                    case 0x35:
                        /* X Coordinate change */
                        if(!f2_mode) {
                            x = (WIDTH * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                        }
                        else {
                            x2 = (WIDTH * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                        }
                        break;
                    case 0x36:
                        /* Y Coordinate change */
                        if(!f2_mode) {
                            y = (HEIGHT * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                        }
                        else {
                            y2 = (HEIGHT * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                        }
                        break;
                    case 0x39:
                        /* Finger up / down */
                        if(event_buffer[0x0C] == 0x00) {
                            printf("Finger 1 down.\n");
                            touches++;
                            finger = DOWN;
                        }
                        else if(event_buffer[0x0C] == 0x01) {
                            printf("Finger 2 down.\n");
                            touches++;
                            finger2 = DOWN;
                        }
                        else {
                            if(finger2 == DOWN) {
                                finger2 = UP;
                                printf("Finger 2 up.\n");
                            }
                            else {
                                finger = UP;
                                printf("Finger 1 up.\n");
                            }
                            
                        }
                        break;
                    default:
                        break;
                }
            }
            if(event_buffer[0x08] == 0x00) {
                sync = 1; 
            }
        }
        
        /* Logic */
        /* Fingers are up. Did we just push exit / clear? */
        if(finger == UP) { /* Release your finger in the top right to exit */
            if(x > 500 && y < 100) {
                printf("Exiting.\n");
                break;
            }
            if(x < 100 && y < 100) { /* Release your finger in the top left 
                                            to refresh the screen */
                display_state = REFRESH;
                while(display_state == REFRESH) {
                    usleep(1000000/60);
                }
                draw_grid(frame_buffer);               
                render(character_buffer, frame_buffer);
                display_state = UPDATE;
                x = 300; /* Reset the finger position to screen centre */
                y = 400;
            }
        }

        /* Render */
        for(i = 0; i < TERMINAL_WIDTH * TERMINAL_HEIGHT; i++) {
            character_buffer[i] = ' '; /* Clear the terminal */
        }
        
        sprintf(character_buffer + 80 * 0, "Hello, world!");
        
        sprintf(character_buffer + 80 * 2, "This is a test of my font and text output system for the Kindle Touch.");
        sprintf(character_buffer + 80 * 3, "UTF-8 characters 0x20 - 0x7E are covered:");
        
        sprintf(character_buffer + 80 * 5, "      ! \" # $ % & ' ( ) * + , - . /");
        sprintf(character_buffer + 80 * 6, "    0 1 2 3 4 5 6 7 8 9 : ; < = > ?");
        sprintf(character_buffer + 80 * 7, "    @ A B C D E F G H I J K L M N O");
        sprintf(character_buffer + 80 * 8, "    P Q R S T U V W X Y Z [ \\ ] ^ _");
        sprintf(character_buffer + 80 * 9, "    ` a b c d e f g h i j k l m n o");
        sprintf(character_buffer + 80 * 10,"    p q r s t u v w x y z { | } ~  ");
        
        sprintf(character_buffer + 80 * 12,"Eventually I hope to develop this into a terminal emulator.");
       
        sprintf(character_buffer + 80 * 14, "Release a touch in the top left corner to refresh the screen.");
        sprintf(character_buffer + 80 * 15, "Release a touch in the top right corner to quit the program.");
        
        sprintf(character_buffer + 80 * 20, "                    You have touched the screen %i times :D", touches);
        
        render(character_buffer, frame_buffer);
        if(display_state == IDLE) {
            display_state = UPDATE;
        }
        
        sync = 0;
    }

    display_state = EXIT;
    pthread_join(updater, NULL);

    munmap(frame_buffer, STRIDE * HEIGHT);
    close(frame_buffer_fd);

    /* Resume the Kindle framework */
    system(RESUME_FRAMEWORK);

    return EXIT_SUCCESS;
}

void update(int refresh) {
    if(refresh) {
        /* Note: This will eraise the frame buffer */
        system("eips -d l=0,w=600,h=800 -f");
    }
    else {
        system("eips ''");
    }
}

void draw_grid(unsigned char *frame_buffer) {
    int x = 0;
    int y = 0;
    
    for(y = 0; y < HEIGHT; y++) {                                             
        for(x = 0; x < WIDTH; x++) {                                          
            if((y % 32) == 0 || (x % 32) == 0) {                              
                frame_buffer[x + STRIDE * y] = 255;                           
            }                                                                 
            else {                                                            
                frame_buffer[x + STRIDE * y] = 0;                             
            }                                                                 
        }                                                                     
    }
}
