/* A simple drawing program developed while learning to use the Kindle Touch
 */

/* Copyright (C) 2012 JoppyFurr
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy 
 * of this software and associated documentation files (the "Software"), to 
 * deal in the Software without restriction, including without limitation the 
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or 
 * sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 * DEALINGS IN THE SOFTWARE.
 */

/* Thanks Geekmaster for helping out with some code suggestions :3 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <pthread.h>

#define WIDTH 600
#define HEIGHT 800
#define STRIDE 608

#define FRAME_BUFFER "/dev/fb0"
#define TOUCH_SCREEN "/dev/input/event3"

#define PAUSE_FRAMEWORK "killall -STOP Xorg cvm"
#define RESUME_FRAMEWORK "killall -CONT cvm Xorg"

#define UP 0
#define DOWN 1

void update(int refresh);
void draw_grid(unsigned char *frame_buffer);

void *update_thread(void *running) {
    while(*(int *)running) {
        update(0);
        usleep(1000000 / 60);
    }
}

int main(int argc, char **argv) {
    int frame_buffer_fd = 0;
    int touch_screen_fd = 0;
    unsigned char *frame_buffer = NULL;
    unsigned char event_buffer[16];
    
    /* First finger variables */
    int finger = UP;
    int x = 300; 
    int y = 400;
    
    /* Second finger variables */
    int finger2 = UP;
    int x2 = 0;
    int y2 = 0;
    int f2_mode = 0;
    
    int i = 0;
    int j = 0;
    
    /* Threads */
    pthread_t updater = 0;
    int running = 1;
    
    int sync = 0;


    /* Pause the Kindle framework */
    system(PAUSE_FRAMEWORK);

    /* Open files related to the screen */
    frame_buffer_fd = open(FRAME_BUFFER, O_RDWR);
    if(frame_buffer_fd == -1) {
        fprintf(stderr, "Error: Could not open frame buffer.\n");
        exit(EXIT_FAILURE);
    }
    frame_buffer = mmap(0, STRIDE * HEIGHT, PROT_READ | PROT_WRITE, MAP_SHARED, frame_buffer_fd, 0);
    touch_screen_fd = open(TOUCH_SCREEN, O_RDONLY);   
    if(touch_screen_fd == -1) {
        fprintf(stderr, "Error: Could not open touch screen.\n");
        exit(EXIT_FAILURE);
    }

    /* Draw a background */
    draw_grid(frame_buffer);
    
    pthread_create(&updater, NULL, update_thread, (void *)&running);
    
    /* Loop through touch screen input, writing pretty things to the screen */    
    while(1) {
        read(touch_screen_fd, event_buffer, 16);
        if(event_buffer[0x08] == 0x03) {
            switch(event_buffer[0x0A]) {
                case 0x2F:
                    /* When f2_mode is nonzero, data is treated as being for
                        the second finger */
                    f2_mode = event_buffer[0x0C];
                    break;
                case 0x35:
                    /* X Coordinate change */
                    if(!f2_mode) {
                        x = (WIDTH * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                    }
                    else {
                        x2 = (WIDTH * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                    }
                    break;
                case 0x36:
                    /* Y Coordinate change */
                    if(!f2_mode) {
                        y = (HEIGHT * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                    }
                    else {
                        y2 = (HEIGHT * (event_buffer[0x0C] + (event_buffer[0x0D] << 8))) / 0x1000;
                    }
                    break;
                case 0x39:
                    /* Finger up / down */
                    if(event_buffer[0x0C] == 0x00) {
                        printf("Finger 1 down.\n");
                        finger = DOWN;
                    }
                    else if(event_buffer[0x0C] == 0x01) {
                        printf("Finger 2 down.\n");
                        finger2 = DOWN;
                    }
                    else {
                        if(finger2 == DOWN) {
                            finger2 = UP;
                            printf("Finger 2 up.\n");
                        }
                        else {
                            finger = UP;
                            printf("Finger 1 up.\n");
                        }
                            
                    }
                    break;
                default:
                    break;
            }
        }
        if(event_buffer[0x08] == 0x00) {
            sync = 1; 
        }
        
        if(sync) { /* We have recieved a sync */
            
            /* If finger 1 is down, draw for it */
            if(finger == DOWN) {
                for(i = -2; i < 3; i++) {
                    for(j = -2; j < 3; j++) {
                        if(x + i >= 0 && x + i < 600 && y + j >= 0 && y + j < 800) {
                            frame_buffer[(x + i) + STRIDE * (y + j)] = 255;
                        }
                    }
                }
            }
            
            /* If finger 2 is down, draw for it */
            if(finger2 == DOWN) {
                for(i = -2; i < 3; i++) {                                           
                    for(j = -2; j < 3; j++) {                                       
                        if(x2 + i >= 0 && x2 + i < 600 && y2 + j >= 0 && y2 + j < 800) {
                            frame_buffer[(x2 + i) + STRIDE * (y2 + j)] = 255;         
                        }
                    }                                                           
                } 
            }
        
            /* Fingers are up. Did we just push exit / clear? */
            if(finger == UP) { /* Release your finger in the top right to exit */
                if(x > 500 && y < 100) {
                    printf("Exiting.\n");
                    break;
                }
                if(x < 100 && y < 100) { /* Release your finger in the top left 
                                            to clear the screen */
                    draw_grid(frame_buffer);
                    x = 300; 
                    y = 400;
                }
            }
        }
        sync = 0;
    }

    running = 0;
    pthread_join(updater, NULL);

    munmap(frame_buffer, STRIDE * HEIGHT);
    close(frame_buffer_fd);

    /* Resume the Kindle framework */
    system(RESUME_FRAMEWORK);

    return EXIT_SUCCESS;
}

void update(int refresh) {
    if(refresh) {
        system("eips -f ''");
    }
    else {
        system("eips ''");
    }
}

void draw_grid(unsigned char *frame_buffer) {
    int x = 0;
    int y = 0;
    
    for(y = 0; y < HEIGHT; y++) {                                             
        for(x = 0; x < WIDTH; x++) {                                          
            if((y % 32) == 0 || (x % 32) == 0) {                              
                frame_buffer[x + STRIDE * y] = 255;                           
            }                                                                 
            else {                                                            
                frame_buffer[x + STRIDE * y] = 0;                             
            }                                                                 
        }                                                                     
    }
}
