//====================================================
// kTimepiece 1.0 - Application that draws a clock directly to framebuffer and updates it every second/minute
// Copyright (C) 2012 by geekmaster, with MIT license:
// Copyright (C) 2012 by kiri, with MIT license:
// http://www.opensource.org/licenses/mit-license.php
//----------------------------------------------------
//  This was tested on K3 only.
//  build with:
//    tcc -I/mnt/us/tcc/include -lm-2.5 -o kTimepiece kTimepiece.c
//----------------------------------------------------

#include <stdio.h>      // printf
#include <stdlib.h>    // malloc, free, atoi
#include <string.h>   // memset, memcpy
#include <unistd.h>  // usleep, sleep
#include <fcntl.h>  // open, close, write
#include <time.h>  // time
#include <math.h> // sin, cos
#include <sys/ioctl.h>   // ioctl
#include <sys/time.h>   // gettime
#include <sys/mman.h>  // mmap, munmap
#include <linux/fb.h> // screeninfo

#define M_PI 3.14159265358979323846

enum eupd_op { EUPD_OPEN,EUPD_CLOSE,EUPD_UPDATE };
typedef unsigned char u8;
typedef unsigned int u32;

// function prototypes
inline void setpx(int,int,int);
void line(int,int,int,int,int, int);
void circle(int, int, int, int);
void displayTime();
int eupdate(int);
int getmsec(void);

// global var7
u32 mpu=200;      // msec/update
u8 *fb0=NULL;   // framebuffer pointer
int fdFB=0;    // fb0 file descriptor
u32 fs=0;     // fb0 stride
u32 MX=0;    // xres (visible)
u32 MY=0;   // yres (visible)
u8 blk=0;  // black
u8 wht=0; // white
u8 pb=0; // pixel bits

void startClock(void) {
	int x,y,tn;
	struct fb_var_screeninfo screeninfo;
	fdFB=open("/dev/fb0",O_RDWR); // eink framebuffer

	// calculate model-specific vars
	ioctl(fdFB,FBIOGET_VSCREENINFO,&screeninfo);
	MX=screeninfo.xres;  // max X+1
	MY=screeninfo.yres; // max Y+1
	pb=screeninfo.bits_per_pixel;     // pixel bits
	fs=screeninfo.xres_virtual*pb/8; // fb0 stride
	blk=pb/8-1; // black
	wht=~blk;  // white
	fb0=(u8 *)mmap(0,MY*fs,PROT_READ|PROT_WRITE,MAP_SHARED,fdFB,0); // map fb0
	eupdate(EUPD_OPEN); // open fb0 update proc

	system("eips -c"); //clear screen

	int cu, i, j, k;

	double q;
	double centerX = MX/2, centerY = MY/2;
	//double centerX = 298, centerY = 312, kSecondRadius = 48.0, kMinuteRadius = 48.0, kHourRadius = 40.0, kOffCenter = 5.0; //smaller clock, not central
	//double kSecondRadius = 290.0, kMinuteRadius = 250.0, kHourRadius = 200.0, kOffCenter = 30.0; //no image frame
	double kSecondRadius = 175.0, kMinuteRadius = 175.0, kHourRadius = 150.0, kOffCenter = 20.0; //clockFrame3 or clockFrame2
	double secondPhi, minutePhi, hourPhi, x, y, x2, y2, x3, y3, dx, dy, dist;
	
	/*
	//this code draws a basic clock frame, although it's a lot better to load an external file containing a frame, with eips -g
	circle((int)centerX, (int)centerY, 295, 0);
	circle((int)centerX, (int)centerY, 294, 0);
	circle((int)centerX, (int)centerY, 296, 0);
	
	for (i=1; i<=60;i++) {
		double fi=(i*2*M_PI)/60;
		double sfi = sin(fi);
		double cfi = cos(fi);
		x = centerX + 295*sfi;
		y = centerY - 295*cfi;
		x2 = centerX + 285*sfi;
		y2 = centerY - 285*cfi;
		if (i%5==0) {
			x2 = centerX + 265*sfi;
			y2 = centerY - 265*cfi;
			
			x3 = centerX + 270*sfi;
			y3 = centerY - 270*cfi;
			
			int ix = (int)x3/12;
			int iy = (int)y3/20;
			int k = i/5;
			
			char buffer [20];
			//sprintf (buffer, "eips %d %d '%d'", ix, iy, k); //display numbers. it's constrained to a grid
			system(buffer);
		}
		
		line((int)x2, (int)y2, (int)x, (int)y, 0, 1);
	} */

	system("eips -g clockFrame2.png");
	
	double dq = kMinuteRadius*kMinuteRadius;
	
	int startY =-(int)kMinuteRadius+centerY;
	int endY =(int)kMinuteRadius+centerY;
	int startX =-(int)kMinuteRadius+centerX;
	int endX =(int)kMinuteRadius+centerX;
	
	//for (cu=0; cu<60; cu++) {
	for (;;) {
		tn=mpu+getmsec();
		
		struct tm *now = NULL;
		int hour = 0;
		time_t time_value = 0;

		time_value = time(NULL);
		now = localtime(&time_value);
		hour = now->tm_hour;
		if(hour>12)
			hour -= 12;

		//clear center
		//TODO: precalc the pixels that need to be cleared
		for (y=startY; y<=endY; y++)
			for (x=startX; x<=endX; x++) {
				dx=centerX-x; dy=centerY-y;
				dist = (dx*dx+dy*dy);
				if (dist<=dq+2)
					//setpx(x,y,64*(1+((double)(now->tm_sec)/60.0)-dist/dq)); //radial gradient fill, based on #seconds
					//setpx(x,y,64*(1.25-dist/dq)); //radial gradient fill
					setpx(x,y,64);
					
					/*if (dist<=dq/2) //3 levels fill
						if (dist<=dq/3)
							setpx(x,y,32);
						else setpx(x,y,48);
					else setpx(x,y,64);*/
			}
			
		secondPhi = ((double)(now->tm_sec)*2*M_PI)/60.0;
		minutePhi = ((((double)(now->tm_sec)*2*M_PI)/60.0)+(double)(now->tm_min)*2*M_PI)/60.0;
		hourPhi = ((((double)(now->tm_min)*2*M_PI)/60.0)+(double)hour*2*M_PI)/12.0;
		
		x = centerX + kSecondRadius*sin(secondPhi);
		y = centerY - kSecondRadius*cos(secondPhi);
		x2 = centerX + kOffCenter*sin(secondPhi + M_PI);
		y2 = centerY - kOffCenter*cos(secondPhi + M_PI);
		line((int)x2, (int)y2, (int)x, (int)y, 0, 0); //this is cool, but I don't think the battery may have the same opinion
		
		x = centerX + kMinuteRadius*sin(minutePhi);
		y = centerY - kMinuteRadius*cos(minutePhi);
		x2 = centerX + kOffCenter*sin(minutePhi + M_PI);
		y2 = centerY - kOffCenter*cos(minutePhi + M_PI);
		line((int)x2, (int)y2, (int)x, (int)y, 0, 1);

		x = centerX + kHourRadius*sin(hourPhi);
		y = centerY - kHourRadius*cos(hourPhi);
		x2 = centerX + kOffCenter*sin(hourPhi + M_PI);
		y2 = centerY - kOffCenter*cos(hourPhi + M_PI);
		line((int)x2, (int)y2, (int)x, (int)y, 0, 2);

		eupdate(EUPD_UPDATE); // update display
		while (tn>getmsec()); // wait until next update time
		
		//sleep(60);
		sleep(1);
	}

	// cleanup - close and free resources
	eupdate(EUPD_UPDATE);    // update display
	eupdate(EUPD_CLOSE);    // close fb0 update proc port
	munmap(fb0,fs*(MY+1)); // unmap fb0
	close(fdFB);          // close fb0
}

//===============================
// eupdate - eink update display
// op (open, close, update)
//-------------------------------
int eupdate(int op) {
	static int fdUpdate=-1;
	if (EUPD_OPEN==op) {
		fdUpdate=open("/proc/eink_fb/update_display",O_WRONLY);
	} else if (EUPD_CLOSE==op) {
		close(fdUpdate);
	} else if (EUPD_UPDATE==op) {
		if (-1==fdUpdate) {
			system("eips ''");
		} else {
			write(fdUpdate,"1\n",2);
		}
	} else {
		return -1;    // bad op code
	}
	return fdUpdate;
}

//========================================
// setpx - draw pixel using ordered dither
// x,y: screen coordinates, c: color(0-64).
// (This works on all eink kindle models.)
//----------------------------------------
inline void setpx(int x,int y,int c) {
	static int dt[64] = {
		1, 33,  9, 41,  3, 35, 11, 43,
		49, 17, 57, 25, 51, 19, 59, 27,
		13, 45,  5, 37, 15, 47,  7, 39,
		61, 29, 53, 21, 63, 31, 55, 23,
		4, 36, 12, 44,  2, 34, 10, 42,
		52, 20, 60, 28, 50, 18, 58, 26,
		16, 48,  8, 40, 14, 46,  6, 38,
		64, 32, 56, 24, 62, 30, 54, 22
	}; // dither table

	fb0[pb*x/8+fs*y] = ((128&(c-dt[(7&x)+8*(7&y)]))/128*(blk&(240*(1&~x)|
		15*(1&x)|fb0[pb*x/8+fs*y])))|((128&(dt[(7&x)+8*(7&y)]-c))/128*wht|
		(blk&((240*(1&x)|15*(1&~x))&fb0[pb*x/8+fs*y]))); // geekmaster formula 42
}

void line(int x0, int y0, int x1, int y1, int c, int thick) {
	int dx = abs(x1-x0), sx = x0<x1 ? 1 : -1;
	int dy = abs(y1-y0), sy = y0<y1 ? 1 : -1; 
	int err = (dx>dy ? dx : -dy)/2, e2;
	int i, j;
	for(;;){
		for (i=-thick;i<=thick;i++) 
			for (j=-thick;j<=thick;j++) 
				setpx(x0+i,y0+j,c);
		if (x0==x1 && y0==y1) break;
		e2 = err;
		if (e2 >-dx) { err -= dy; x0 += sx; }
		if (e2 < dy) { err += dx; y0 += sy; }
	}
}

void circle(int x0, int y0, int radius, int c) {
	int f = 1 - radius;
	int ddF_x = 0;
	int ddF_y = -2 * radius;
	int x = 0;
	int y = radius;

	setpx(x0, y0 + radius, c);
	setpx(x0, y0 - radius, c);
	setpx(x0 + radius, y0, c);
	setpx(x0 - radius, y0, c);

	while(x < y) {
		if(f >= 0) {
			y--;
			ddF_y += 2;
			f += ddF_y;
		}
		x++;
		ddF_x += 2;
		f += ddF_x + 1;    
		setpx(x0 + x, y0 + y, c);
		setpx(x0 - x, y0 + y, c);
		setpx(x0 + x, y0 - y, c);
		setpx(x0 - x, y0 - y, c);
		setpx(x0 + y, y0 + x, c);
		setpx(x0 - y, y0 + x, c);
		setpx(x0 + y, y0 - x, c);
		setpx(x0 - y, y0 - x, c);
	}
}

//====================================
// getmsec - get msec since first call
// (tick counter wraps every 12 days)
//------------------------------------
int getmsec(void) {
	int tc;
	static int ts=0;
	struct timeval tv;
	gettimeofday(&tv,NULL);
	tc=tv.tv_usec/1000+1000*(0xFFFFF&tv.tv_sec);
	if (0==tc) {
		ts=tc;
	}
	return tc-ts;
}

//==================
// main - start here
//------------------
int main(int argc,char **argv) {
	if (argc>1) {
		mpu=atoi(argv[1]);
	}
	startClock();
	return 0;
}
