//====================================================
// hoser - eink demo showing all-new dither design
// (yet another "dithermation" demo by geekmaster)
// Copyright (C) 2012 by geekmaster, with MIT license:
// http://www.opensource.org/licenses/mit-license.php
//----------------------------------------------------
//  The speed is limited by the eink device drivers.
//  Newer kindle models are faster, but need delays.
//  This was tested on DX,DXG,K3,K4(Mini),K5(Touch).
//----------------------------------------------------

#include <stdio.h>      // printf
#include <stdlib.h>    // malloc, free
#include <string.h>   // memset, memcpy
#include <unistd.h>  // usleep
#include <fcntl.h>  // open, close, write
#include <time.h>  // time
#include <sys/mman.h>   // mmap, munmap
#include <sys/ioctl.h> // ioctl
#include <linux/fb.h> // screeninfo

#define ED6 570000      // k4 eink delay best quality
#define ED5 500000     // k4 eink delay good quality
#define ED4 230000    // k4 eink delay K3 speed
#define ED3 100000   // k4 eink delay okay
#define ED2 80000   // k4 eink delay fast
//#define ED1 40000   // k4 eink delay fast
#define ED1 0      // k4 eink delay none, bad
#define K4DLY ED1 // k4 eink delay

enum GMLIB_op { GMLIB_INIT,GMLIB_CLOSE,GMLIB_UPDATE };
typedef unsigned char u8;
typedef unsigned int u32;

// function prototypes
//inline void setpxd(int,int,int);
void d4w(void);
void d8w(void);
void d8b(void);
inline void setpx(int,int,int);
void box(int,int,int,int);
void line(int,int,int,int,int);
void circle(int,int,int);
int gmlib(int);
int getmsec(void);

// global vars
static u8 dt[64] = {
    3,129,34,160,10,136,42,168,192,66,223,97,200,73,231,105,50,
    176,18,144,58,184,26,152,239,113,207,81,247,121,215,89,14,
    140,46,180,7,133,38,164,203,77,235,109,196,70,227,101,62,188,
    30,156,54,180,22,148,251,125,219,93,243,117,211,85 }; // 0-255 dither table
//static u8 dt[64] = { 0,32,8,40,2,34,10,42,48,16,56,24,50,18,58,26,12,44,4,
//    36,14,46,6,38,60,28,52,20,62,30,54,22,3,35,11,43,1,33,9,41,51,19,59,27,49,
//    17,57,25,15,47,7,39,13,45,5,37,63,31,55,23,61,29,53,21 }; // dither table
u8 *fb0=NULL;       // framebuffer pointer
u8 *wb=NULL;       // workbuffer pointer
u32 mpu=100;      // msec/update
int fdFB=0;      // fb0 file descriptor
int fdWB=0;     // wb file descriptor
u32 fs=0;      // fb0 stride
u32 MX=0;     // xres (visible)
u32 MY=0;    // yres (visible)
u8 blk=0;   // black
u8 msk=0;  // black mask
u8 ppb=0; // pixels per byte


//===============================================
// hoser - eink demo showing all-new dither design
// This works on all kindle eink models.   Enjoy!
//-----------------------------------------------
void hoser(void) {
    u32 x,y,c,px1,py1,vx1,vy1,px2,py2,vx2,vy2,dx,dy,cc,cu,cl,tn;
    gmlib(GMLIB_INIT);      // init eink library

// do dithered gray demo
    c=0,px1=MX/2,py1=MY/2,vx1=1,vy1=5,cc=31,cl=16;
    for (cu=0;cu<20000;cu++) {
        if (0==cu%3000) { // periodic background display
          for (y=0; y<=MY/2; y++) {
            for (x=0; x<=MX/2; x++) {
                dx=MX/2-x; dy=MY/2-y; c=255-(dx*dx+dy*dy)*255/(MX*220);
                setpx(x,y,c); setpx(MX-x,y,c);
                setpx(x,MY-y,c); setpx(MX-x,MY-y,c);
            }
          }
        }
        circle(px1,py1,30); circle(px1,py1,31); circle(px1,py1,32);
        circle(px1,py1,29); circle(px1,py1,28); circle(px1,py1,27);
        circle(px1,py1,26); circle(px1,py1,25); circle(px1,py1,24);
        circle(px1,py1,23); circle(px1,py1,22); circle(px1,py1,21);
        circle(px1,py1,20); circle(px1,py1,19); circle(px1,py1,18);
        circle(px1,py1,17); circle(px1,py1,16); circle(px1,py1,15);
        circle(px1,py1,14); circle(px1,py1,13); circle(px1,py1,12);
        px1+=vx1; py1+=vy1;
        if (px1>MX-40 || px1<40) vx1=-vx1;
        if (py1<40) { py1=40; vy1=-vy1; }
        if (py1>MY-40) { py1=MY-40; vy1=-vy1; }
        if (0==cu%cl) {
            gmlib(GMLIB_UPDATE);
            vy1++;
        } // update display
        cc=(cc+4)%256; // cycle big box color
    }

// cleanup - close and free resources
    gmlib(GMLIB_UPDATE); // update display
    gmlib(GMLIB_CLOSE); // close fb0 update proc port
    munmap(fb0,MY*fs);  // unmap fb0
    munmap(wb,MY*MX);  // unmap wb
    close(fdFB);      // close fb0
    close(fdWB);     // close wb
}

//======================================
// gmlib - geekmaster's function library
// op (init, close, update)
//--------------------------------------
int gmlib(int op) {
    int x,y;
    struct fb_var_screeninfo screeninfo;
    static int fdUpdate=-1;
    if (GMLIB_INIT==op) {
        fdWB=open("/tmp/wb0",O_RDWR|O_CREAT); // work framebuffer
        fdFB=open("/dev/fb0",O_RDWR);        // eink framebuffer
        ioctl(fdFB,FBIOGET_VSCREENINFO,&screeninfo);
        ppb=8/screeninfo.bits_per_pixel; // pixels per byte
        fs=screeninfo.xres_virtual/ppb; // fb0 stride
        blk=screeninfo.rotate-1;       // black
        MX=screeninfo.xres;           // max X+1
        MY=screeninfo.yres;          // max Y+1
        msk=1/ppb-1;                // black mask (4-bit=255,8-bit=0)
        fb0=(u8 *)mmap(0,MY*fs,PROT_READ|PROT_WRITE,MAP_SHARED,fdFB,0); // map fb0
        lseek(fdWB,MY*MX-1,SEEK_SET); write(fdWB,"",1); // create work buffer file
        wb=(u8 *)mmap(0,MY*MX,PROT_READ|PROT_WRITE,MAP_SHARED,fdWB,0); // map wb
        fdUpdate=open("/proc/eink_fb/update_display",O_WRONLY);
    } else if (GMLIB_CLOSE==op) { close(fdUpdate);
    } else if (GMLIB_UPDATE==op) {
        if (ppb/2) { d4w(); write(fdUpdate,"1\n",2); }
        else if (blk) { d8w(); write(fdUpdate,"1\n",2); }
        else { d8b(); system("eips ''");  }
    } else { return -1; }
    return fdUpdate;
}

//========================================
// setpx - draw pixel to 8-bit work buffer
// x,y:screen coordinates, c:color(0-255).
//----------------------------------------
inline void setpx(int x,int y,int c) {
    wb[y*MX+x]=c;
}

//==================================
// d8b - dither 8-bit black 0 buffer
//----------------------------------
void d8b(void) {
    u8 *pi,*po;
    int x,y;
    pi=wb; po=fb0;
    for (y=0;y<MY;y++) {
        for (x=0;x<MX;x++) { *po++=dt[(y&7)*8|x&7]-*pi++>>8; }
        po+=(fs-MX);
    }
}

//==================================
// d8w - dither 8-bit white 0 buffer
//----------------------------------
void d8w(void) {
    u8 *pi,*po;
    int x,y;
    pi=wb; po=fb0;
    for (y=0;y<MY;y++) {
        for (x=0;x<MX;x++) { *po++=~(dt[(y&7)*8|x&7]-*pi++>>8); }
        po+=(fs-MX);
    }
}

//========================================
// d4w - dither 4-bit white 0 buffer
//----------------------------------------
void d4w(void) {
    u8 *pi,*po;
    int x,y,ys;
    pi=wb; po=fb0;
    for (y=0;y<MY;y++) {
        ys=(y&7)*8;
        for (x=0;x<MX;x+=8) {
             *po++=(~(dt[ys]-*pi++>>8)|15)&(~(dt[ys+1]-*pi++>>8)|240);
             *po++=(~(dt[ys+2]-*pi++>>8)|15)&(~(dt[ys+3]-*pi++>>8)|240);
             *po++=(~(dt[ys+4]-*pi++>>8)|15)&(~(dt[ys+5]-*pi++>>8)|240);
             *po++=(~(dt[ys+6]-*pi++>>8)|15)&(~(dt[ys+7]-*pi++>>8)|240);
        }
    }
}

//======================
// box - simple box draw
//----------------------
void box(int x,int y,int d,int c) {
    int i;
    for (i=0;i<d;++i) {
        setpx(x+i,y+d,c); setpx(x+i,y-d,c); setpx(x-i,y+d,c); setpx(x-i,y-d,c);
        setpx(x+d,y+i,c); setpx(x+d,y-i,c); setpx(x-d,y+i,c); setpx(x-d,y-i,c);
    }
}

//==============================================
// circle - optimized midpoint circle algorithm
//----------------------------------------------
void circle(int cx,int cy,int r) {
    int e=-r,x=r,y=0;
    while (x>y) {
        setpx(cx+y,cy-x,255); setpx(cx+x,cy-y,159);
        setpx(cx+x,cy+y,95); setpx(cx+y,cy+x,31);
        setpx(cx-y,cy+x,0); setpx(cx-x,cy+y,63);
        setpx(cx-x,cy-y,127); setpx(cx-y,cy-x,191);
        e+=y; y++; e+=y;
        if (e>0) { e-=x; x-=1; e-=x; }
    }
}

//====================================
// getmsec - get msec since first call
// (tick counter wraps every 12 days)
//------------------------------------
int getmsec(void) {
    int tc;
    static int ts=0;
    struct timeval tv;
    gettimeofday(&tv,NULL); tc=tv.tv_usec/1000+1000*(0xFFFFF&tv.tv_sec);
    if (0==tc) { ts=tc; }
    return tc-ts;
}

//==================
// main - start here
//------------------
int main(int argc,char **argv) {
    if (argc>1) { mpu=atoi(argv[1]); }
    hoser(); // do the hoser demo :D
    return 0;
}
