'''
Created on Dec 6, 2009

@author: napier

Modified on January, 2012
@modifer: jchisholm
    added ability to open a url (http://...)

Modified on July, 2011
@modifer: jchisholm
    added read64; corrected parse_atom to use it.
    added class to handle MP4 epoch datetime values.
    added ATOM_WITH_VERSION to identify those known to have the extra verion_flags DWORD
    changed Atom.__init__ to also take a versionflags
    brought up to MPEG-4 Part 14 (1998 draft)
    brought up to Adobe Video File Format Specification Version 9 (2008)
    defined many special tags so I could see what their structure is
    added ATOM_WITH_ONE_CHILD to identify those defined with exactly one child
    added ATOM_WITH_COUNT_CHILD to identify those with a DWORD count and children
    added ability for create_atom to create the right kind of generic atom on the fly
    added method to pre-create all those redundant atoms
        to avoid later exceptions
'''
import os, sys, re
import struct
import time, calendar, datetime

from atomsearch import find_path, findall_path

import logging
log = logging.getLogger("mp4file")
# log_h = logging.FileHandler("C:/Users/Jesse/calibre_hold/dev/AudioBookReader/_readme.log")
# log_f = logging.Formatter("%(asctime)s %(levelname)s %(message)s")
# log_g.setFormatter(log_f)
# log.addHandler(log_h)
# log.setLevel(logger.DEBUG)

VERBOSE = 0

# log.debug(".mp4.atom: __file__   : %s" % (__file__))
# log.debug(".mp4.atom: __package__: %s" % (__package__))

class EndOFFile(Exception):
    def __init_(self):
        Exception.__init__(self)

# 2011.07.22 - patch parse_atom
def read64(file):
    data = file.read(8)
    if (data is None or len(data) <> 8):
        raise EndOFFile()
    return struct.unpack(">Q", data)[0]

def read32(file):
    data = file.read(4)
    if (data is None or len(data) <> 4):
        raise EndOFFile()
    return struct.unpack(">I", data)[0]

def read16(file):
    data = file.read(2)
    if (data is None or len(data) <> 2):
        raise EndOFFile()
    return struct.unpack(">H", data)[0]

def read8(file):
    data = file.read(1)
    if (data is None or len(data) <> 1):
        raise EndOFFile()
    return struct.unpack(">B", data)[0]

# returns tuple ( length, string )
def readstr8(file, encoding="utf-8"):
    howMuch = read8(file)
    val = unicode(file.read(howMuch), encoding)
    return (howMuch, val)

def type_to_str(data):
    a = (data >> 0) & 0xff
    b = (data >> 8) & 0xff
    c = (data >> 16) & 0xff
    d = (data >> 24) & 0xff
    return '%c%c%c%c' % (d, c, b, a)

# massaged from: http://snippets.dzone.com/posts/show/1694
import time,calendar
class mp4DateTime(object):
    """MP4 files store date/time in either a 32 or 64 bit field,
    as number of seconds since the epoch beginning.
    mp4 time      epoch is 1904-01-01 00:00:00 (UTC)
    time.gmtime() epoch is 1970-01-01 00:00:00 (UTC)
    There are 2082758400 seconds between 1904 and 1970.
    """
    epoch_shift = 2082758400 # seconds between 1904 and 1970
    #
    _hr_names = ['year', 'month', 'day',
                 'hour', 'minute', 'second',
                 'weekday', 'julianday', 'isdst']
    _tm_names = ['tm_year', 'tm_mon', 'tm_mday',
                 'tm_hour', 'tm_min', 'tm_sec',
                 'tm_wday', 'tm_yday', 'isdst']
    #
    def __init__(self, *argv):
        """mp4DateTime() represents now
        mp4DateTime(seconds) represents seconds since mp4 epoch of 1904 in UTC.
        mp4DateTime(y,m,d,h,m,s) specifies the date precisely.
        """
        if not argv:
            # no argument means 'now' - get as UTC time tuple
            self.tm = time.gmtime()
            self.t = [self.tm.tm_year,self.tm.tm_mon,self.tm.tm_mday,
                      self.tm.tm_hour,self.tm.tm_min,self.tm.tm_sec,
                      self.tm.tm_wday,self.tm.tm_yday,self.tm.tm_isdst]
        elif len(argv) == 1:
            if isinstance(argv[0],datetime.datetime):
                self.tm = argv[0].utctimetuple()
                self.t = [self.tm.tm_year,self.tm.tm_mon,self.tm.tm_mday,
                          self.tm.tm_hour,self.tm.tm_min,self.tm.tm_sec,
                          self.tm.tm_wday,self.tm.tm_yday,self.tm.tm_isdst]
            else:
                # one argument is seconds in MP4 epoch - convert to UTC time tuple
                log.debug("mp4DateTime(%s): shifted = %s" % (
                    argv[0],argv[0] - self.epoch_shift))
                try:
                    self.tm = time.gmtime(argv[0] - self.epoch_shift)
                except:
                    log.error("mp4DateTime(%s): shifted = %s : gmtime @ %s" % (
                        argv[0],argv[0] - self.epoch_shift, _sofar()))
                    print "error: mp4DateTime(%s): shifted = %s : gmtime barfed!" % (
                        argv[0],argv[0] - self.epoch_shift)
                    self.tm = time.gmtime()
                self.t = [self.tm.tm_year,self.tm.tm_mon,self.tm.tm_mday,
                          self.tm.tm_hour,self.tm.tm_min,self.tm.tm_sec,
                          self.tm.tm_wday,self.tm.tm_yday,self.tm.tm_isdst]
        else:
            # leading parts of the tuple given:
            #    (year,month,day,hour,minute,second,weekday,julian,isdst)
            t = argv+(0,)*(9-len(argv))    # append to length 9 
            log.debug("mp4DateTime(%s):" % (t))
            self.t = [t[0],t[1],t[2],t[3],t[4],t[5],t[6],t[7],t[8]]
            self.tm = time.struct_time(self.t)

    def __getattr__(self, name):
        if name in self._hr_names:
            i = self._hr_names.index(name)
            return self.t[i]
        if name in self._tm_names:
            i = self._tm_names.index(name)
            return self.tm[i]
        print("error: AttributeError: %r object has no attribute %r" %
                                      (type(self).__name__, name))
        log.error("AttributeError: %r object has no attribute %r" %
                                   (type(self).__name__, name))
        raise AttributeError("%r object has no attribute %r" %
                             (type(self).__name__, name))
    def __len__(self): return len(self.t)
    def __getitem__(self, key): return self.t[key]
    def __repr__(self): return repr(self.t)
    def mp4(self):
        """Get the MP4 representaiton of the time.
        """
        # mp4 time      epoch is 1904-01-01 00:00:00 (UTC)
        # time.gmtime() epoch is 1970-01-01 00:00:00 (UTC)
        temp = self.t
        temp[0] -= self.epoch_shift
        log.debug("mp4DateTime.mp4: %s shifted = %s" % (self.t, temp))
        try:
            temp = calendar.timegm(temp)
        except:
            print "error: mp4DateTime.mp4: timegm barfed on %s" % (temp)
            log.error("mp4DateTime.mp4: timegm barfed on %s" % (temp))
            temp = 0
        return temp
    def now(self=None):
        return mp4DateTime()
    now = staticmethod(now)
    def strftime(self, fmt="%Y-%m-%d %H:%M:%S"):
        return time.strftime(fmt, self.tm)

__sofar = []
def _sofar_push(type):
    global __sofar
    __sofar.append(type)
def _sofar_pop():
    global __sofar
    __sofar = __sofar[:-1]
def _sofar_peek():
    global __sofar
    if __sofar:
        return __sofar[:-1]
    return None
def _sofar():
    global __sofar
    return "%s" % (__sofar)

# 2011.07.22 - verison 0.2 calls read32 by mistake; added version_flag read
def parse_atom(file):
    type = "~"    # not a None but not a 4 character tag either
    try:
        offset = file.tell()
        size = read32(file)
        type = type_to_str(read32(file))
        _sofar_push(type)
        if (size == 1):
            size = read64(file)
            log.debug("parse_atom: '%s' large size: %s" % (type, size))
        if size < 8:
            log.error("parse_atom: '%s' s=%s o=%s" % (type, size, offset))
            _sofar_pop()
            return None
        versionflags = 0
        if type in ATOM_WITH_VERSION:
            versionflags = read32(file)
            log.debug("parse_atom: '%s' version,flags: %s,%s" %
                        (type, (versionflags >> 24), (versionflags & 0xFFFFFF)))
        atom = create_atom(size, type, versionflags, offset, file)
        if atom:
            atom._report_slop(file)
        _sofar_pop()
        return atom
    except EndOFFile:
        # in case we got past the push before the EOF
        if _sofar_peek() == type:
            _sofar_pop()
        return None

# Mapping of ATOM type names that are NOT valid python class names.
#    to a more acceptable synonym that is a valid python class name.
# Also, mappings from annoyingly cryptic ATOM type names
#    to more humanly meaningful names.
#
# calls to {find, findall} take paths of type names or paths of humanly readable names.
#
# see: http://atomicparsley.sourceforge.net/mpeg-4files.html
#
ATOM_TYPE_MAP = { '\xa9too': 'encoder',
                  '\xa9nam': 'title',
                  '\xa9alb': 'album',
                  '\xa9art': 'artist',
                  '\xa9ART': 'artist',
                  'aART': 'artist',
                  '\xa9cmt': 'comment',
                  '\xa9gen': 'genre',
                  'gnre': 'genre',
                  '\xa9day': 'year',
                  'trkn': 'tracknum',
                  'disk': 'disknum',
                  '\xa9wrt': 'composer',
                  'tmpo': 'bpm',
                  'cprt': 'copyright',
                  'cpil': 'compilation',
                  'covr': 'coverart',
                  'rtng': 'rating',
                  '\xa9grp': 'grouping',
                  'pcst': 'podcast',
                  'catg': 'category',
                  'keyw': 'keyword',
                  'purl': 'podcasturl',
                  'egid': 'episodeguid',
                  'desc': 'description',
                  '\xa9lyr': 'lyrics',
                  'tvnn': 'tvnetwork',
                  'tvsh': 'tvshow',
                  'tven': 'tvepisodenum',
                  'tvsn': 'tvseason',
                  'tves': 'tvepisode',
                  'purd': 'purcahsedate',
                  'pgap': 'gapless',
                  #
                  '----': 'four_dashes'
                }

# 2011.07.22 - many atoms have a version_and_flags field
ATOM_WITH_VERSION = ['mvhd','iods','tkhd','mdhd','hdlr',
                     'vmhd','smhd','hmhd','nmhd','url ',
                     'urn ','dref','stts','dtts','esds',
                     'stsd','stsz','stsc','stco','co64',
                     'stss','stsh','stdp','elst','cprt',
                     # above in MPEG-4 Part 14, Dec 1998
                     'titl','dscp','auth',
                     # or in Flash Video File Formats Spec v9
                     'data',
                     # or in http://atomicparsley.sourceforge.net/mpeg-4files.html
                     'ID32',
                    ]

# There are a lot of atom's with children, and no other attributes.
# No need to create special classes for all of them
# 2011.07.24 - see: declare_RedundantAtoms
#
ATOM_WITH_CHILDREN = ['moov',   'trak',   'tref',   'mdia',
                      'minf',   'dinf',   'stbl',   'edts',
                      'udta',
                      # above in MPEG-4 Part 14, Dec 1998
                      'covr',   'meta',   'stik',
                      # or in Flash Video File Formats Spec v9
                      # or in http://atomicparsley.sourceforge.net/mpeg-4files.html
                     ]
# There a few atoms with exactly one child.
# No need to create special classes for all of them
# 2011.07.24 - see: declare_RedundantAtoms
#
ATOM_WITH_ONE_CHILD = ['dinf',
                      # above in MPEG-4 Part 14, Dec 1998
                      # below in http://atomicparsley.sourceforge.net/mpeg-4files.html
                      '\xa9alb','\xa9art','\xa9cmt','\xa9day',
                      '\xa9gen','\xa9grp','\xa9lyr','\xa9nam',
                      '\xa9too','\xa9wrt',
                      '\xa9ART','aART', # aliases for '\xa9art'
                      'catg',   'cpil',   'cptr',   'disk',
                      'egid',   'gmhd',   'gnre',   'keyw',
                      'pcst',   'pgap',   'purd',   'purl',
                      'rtng',   'tmpo',   'trkn',   'tven',
                      'tves',   'tvnn',   'tvsh',   'tvsn',
                      ]
# There a few atoms with a count attribute followed by exactly N children.
# No need to create special classes for all of them
# 2011.07.24 - see: declare_RedundantAtoms
#
ATOM_WITH_COUNT_CHILDREN = ['dref','stsd',
                      ]    # removed: 'ilst',

def _type_valid(type):
    if not re.match("^[a-zA-Z_]\w*$", type):
        return False
    return True
def _make_valid(type):
    if not re.match("^[a-zA-Z_]\w*$", type):
        type = "atom_%s" % (re.sub("[^\w]",'_',type))
    return type

# 2011.07.22 - extended for flags and version
def create_atom(size, type, versionflags, offset, file):
    log.debug("create_atom(%s, '%s', %s, %s, ...)" %(size,type,versionflags,offset))
    clz = type or "Atom"
    # Possibly remap atom types that aren't valid
    # python variable names
    if (ATOM_TYPE_MAP.has_key(type)):
        clz = ATOM_TYPE_MAP[type]
        log.debug("create_atom: mapped '%s' => '%s'" %(type,clz))
    # remove any trailing spaces from clz, regardless of mapped or not.
    clz = clz.strip()
    # last ditch safety net!
    if not _type_valid(clz):
        clz = _make_valid(clz)
        log.debug("create_atom: mapped '%s' => '%s'" %(type,clz))
    try:
        # if it isn't declared yet,
        #   try to declare it on the fly
        if not clz in globals():
            declare_RedundantAtoms(type)

        # if it was already declared, or it is NOW declared, then use it
        #
        # Try and eval the class into existance
        if clz in globals():
            log.debug("create_atom: eval: %s(size, type, clz, versionflags, offset, file)" % (clz))
            return eval("%s(size, type, clz, versionflags, offset, file)" % clz)
    except (NameError, TypeError) as err:
        log.error("create_atom: eval: %s(size, type, clz, versionflags, offset, file)" % (clz))
        log.error("create_atom: %s" % (_sofar()))
        log.error("create_atom: [%s] = %s" % (clz,globals()[clz]))
        print err
        print err.args
        log.error(err.args)
    # Not defined, use generic Atom
    log.warning("create_atom: using generic atom for: '%s'" % (clz))
    atom = None
    if type in ATOM_WITH_ONE_CHILD:
        atom = AtomWithOneChild(size, type, clz, versionflags, offset, file)
    elif type in ATOM_WITH_COUNT_CHILDREN:
        atom = AtomWithCountChildren(size, type, clz, versionflags, offset, file)
    elif type in ATOM_WITH_CHILDREN:
        atom = AtomWithChildren(size, type, clz, versionflags, offset, file)
    else:
        atom = Atom(size, type, clz, versionflags, offset, file)
    return atom

def parse_atoms(file, maxFileOffset):
    atoms = []
    while (file.tell()+8) <= maxFileOffset:
        atom = parse_atom(file)
        if atom:
            atoms.append(atom)

    return atoms

def parse_N_atoms(N, file, maxFileOffset):
    atoms = []
    for i in range(0,N):
        if (file.tell()+8) <= maxFileOffset:
            atom = parse_atom(file)
            if atom:
                atoms.append(atom)

    return atoms

class Atom(object):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("Atom: t='%s' n='%s' s=%s o=%s" % (type,name,size,offset))
        self.parent = None    # set later if added to someone's children
        self.size = size
        self.atomsize = file.tell() - offset    # size sans children; derived may update
        self.type = type
        self.name = name
        self.hasversion = type in ATOM_WITH_VERSION
        self.flags = self.hasversion and (versionflags & 0xFFFFFF) or 0
        self.version = self.hasversion and (versionflags >> 24) or 0
        self.offset = offset
        self.file = file        # hope it is still open if/when you use it
        self.children = []
        self.attrs = {}
        self.attr_order = []    # the order the attrs were added, for those who care

    def _set_attr(self, key, value):
        self.attrs[key] = value
        if not key in self.attr_order:
            self.attr_order.append(key)

    def _set_children(self, children):
        # Tell the children who their parent is
        for child in children:
            child.parent = self
        self.children = children

    # sanity check, one at a time, or all at once breadth first
    def verify(self,child=None):
        err = 0
        if child:
            if child.parent <> self:
                log.warning("child doesn't know parent: %s" % (child.fulltype()))
                err += 1
            if not child in self.children:
                log.warning("parent doesn't know child: %s" % (child.fulltype()))
                err += 1
            if child.offset < (self.offset + self.atomsize):
                log.warning("child starts too early: %s" % (child.fulltype()))
                err += 1
            if (child.offset + child.size) > (self.offset + self.size):
                log.warning("child ends too late: %s" % (child.fulltype()))
                err += 1
            # When this is true, then one of the others is also true
            #
            # if child.size > (self.size - self.atomsize):
            #     log.warning("child larger than parent: %s" % (child.fulltype()))
            #     err += 1
        else:
            for child in self.children:
                err += self.verify(child)
            for child in self.children:
                err += child.verify(None)
            if self.children:
                child = self.children[-1]
                if (child.offset + child.size) <> (self.offset + self.size):
                    log.warning("last child doesn't end parent: %s" % (child.fulltype()))
                    err += 1
        return err

    def get_attribute(self, key):
        if key in self.attrs:
            return self.attrs[key]
        return None

    def get_atoms(self):
        return self.children

    def fulltype(self):
        if not self.name and not self.parent: return "."
        full_pop = self.parent and self.parent.fulltype() or "."
        return "%s/%s" % (full_pop, self.type)

    def fullname(self):
        if not self.name and not self.parent: return "."
        full_pop = self.parent and self.parent.fullname() or "."
        return "%s/%s" % (full_pop, self.name)

    def find(self, path):
        return find_path(self, path)

    def findall(self, path):
        return findall_path(self, path)

    # put your own "if DEBUG:" or "if VERBOSE:" on your call to dump. ;-)
    def dump(self, lead="", verbose=0):
        if self:
            if self.hasversion:
                print ("%s '%s' [v=%s, f=%s]" % (lead, self.name, self.version, self.flags))
            else:
                print ("%s '%s'" % (lead, self.name))
            print lead," . fullname   = ",self.fullname()
            if verbose:
                print lead," . fulltype   = ",self.fulltype()
                print lead," . atomsize   = ",self.atomsize
                print lead," . offset     = ",self.offset
                print lead," . size       = ",self.size
                print lead," . endpos     = ",(self.offset + self.size)
            if self.attrs:
                for key in self.attr_order:
                    print lead," . [",key,"] = ",
                    val = self.attrs[key]
                    if type(val) == atom_vector:
                        print len(val.data),"#[...]"
                    else:
                        print val
            if self.children:
                for child in self.children:
                    child.dump(lead + "+-", verbose)

    # same as dump, but to the log file, instead of to the console.
    def log(self, lead="", verbose=0):
        if self:
            if self.hasversion:
                log.debug("%s '%s' [v=%s, f=%s]" % (lead, self.name, self.version, self.flags))
            else:
                log.debug("%s '%s'" % (lead, self.name))
            log.debug("%s . fullname   = %s" % (lead,self.fullname()))
            if verbose:
                log.debug("%s . fulltype   = %s" % (lead,self.fulltype()))
                log.debug("%s . atomsize   = %s" % (lead,self.atomsize))
                log.debug("%s . offset     = %s" % (lead,self.offset))
                log.debug("%s . size       = %s" % (lead,self.size))
                log.debug("%s = endpos     = %s" % (lead,(self.offset + self.size)))
            if self.attrs:
                for key in self.attr_order:
                    val = self.attrs[key]
                    val = (type(val) == atom_vector) and ("%s #[...]" % (len(val.data))) or val
                    log.debug("%s . [%s] = %s" % (lead,key,val))
            if self.children:
                for child in self.children:
                    child.log(lead + "+-", verbose)

    def _report_slop(self,file):
        """Only call atom._report_slop(file) immediately after parsing the atom from the file."""
        slop = (self.offset + self.size) - file.tell()
        if slop <> 0:
            self._set_attr("_slop_size", slop)
            if slop < 0:
                log.error("read %s too much: '%s'" % (-slop, _sofar()))
                print "error: read %s too much: '%s'" % (-slop, _sofar())
                # self.dump()
                # self.log()
            if slop > 0:
                if slop <= 32:
                    slop = file.read(slop)
                    self._set_attr("_slop", slop)
                else:
                    slop = file.read(32)
                    slop = "%s ..." % (slop)
                    self._set_attr("_slop[:32]", slop)
                # self._set_attr("_slop", atom_vector(file.read(slop)))

            # Seek to the defiend end of the atom to correct for any slop
            file.seek(self.offset + self.size, os.SEEK_SET)

class AtomWithChildren(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("AtomWithChildren: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self._set_children(parse_atoms(file, offset + size))

class AtomWithOneChild(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("AtomWithOneChild: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self._set_children(parse_N_atoms(1, file, offset + size))

class AtomWithCountChildren(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("AtomWithCountChildren: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 4
        count = read32(file)
        self._set_attr('count', count)
        log.debug("%s: count=%s children expected" % (name, count))
        self._set_children(parse_N_atoms(count, file, offset + size))

def _init_Atom(atom, size, type, name, versionflags, offset, file):
    log.debug(".atom...: t='%s' n='%s'" % (type,name))
    Atom.__init__(atom, size, type, name, versionflags, offset, file)
def _init_AtomWithChildren(atom, size, type, name, versionflags, offset, file):
    log.debug(".atomWC.: t='%s' n='%s'" % (type,name))
    AtomWithChildren.__init__(atom, size, type, name, versionflags, offset, file)
def _init_AtomWithOneChild(atom, size, type, name, versionflags, offset, file):
    log.debug(".atom1C.: t='%s' n='%s'" % (type,name))
    AtomWithOneChild.__init__(atom, size, type, name, versionflags, offset, file)
def _init_AtomWithCountChildren(atom, size, type, name, versionflags, offset, file):
    log.debug(".atomNC.: t='%s' n='%s'" % (type,name))
    AtomWithCountChildren.__init__(atom, size, type, name, versionflags, offset, file)
def _add_method(instance, method, name=None):
    if name is None:
        name = method.func_name
    setattr(instance.__class__, name, method)
def _declare_Class(classname,baseClass):
    if classname in globals():
        log.error("_declare_Class: '%s' already exists in %s" % (classname,__package__))
        return None
    log.debug("_declare_Class: '%s' as '%s'" % (classname,baseClass))
    classobj = type(classname, (baseClass,), {})
    globals()[classname] = classobj
    return classobj
def _set_Init(classobj,method):
    if classobj is None: return
    if "__init__" in classobj.__dict__:
        log.error("_set_Init: '%s' already has an '__init__'" % (classobj))
    else:
        log.debug("_set_Init: '%s' as '%s'" % (classobj,method))
        setattr(classobj,"__init__",method)
  
def _declare_Atom(classname):
    global Atom, _init_Atom
    if not classname in globals():
        log.debug("_declare_Atom: '%s'" % (classname))
        _set_Init(_declare_Class(classname,Atom),_init_Atom)
def _declare_AtomWithChildren(classname):
    global AtomWithChildren, _init_AtomWithChildren
    if not classname in globals():
        log.debug("_declare_AtomWithChildren: '%s'" % (classname))
        _set_Init(_declare_Class(classname,AtomWithChildren),_init_AtomWithChildren)
def _declare_AtomWithOneChild(classname):
    global AtomWithOneChild, _init_AtomWithOneChild
    if not classname in globals():
        log.debug("_declare_AtomWithOneChild: '%s'" % (classname))
        _set_Init(_declare_Class(classname,AtomWithOneChild),_init_AtomWithOneChild)
def _declare_AtomWithCountChildren(classname):
    global AtomWithCountChildren, _init_AtomWithCountChildren
    if not classname in globals():
        log.debug("_declare_AtomWithCountChildren: '%s'" % (classname))
        _set_Init(_declare_Class(classname,AtomWithCountChildren),_init_AtomWithCountChildren)

# actually declare all atoms we know the basic structure of:
#    those atoms with children but with no other attributes.
#    those atoms defined to have exactly 1 child, or N children.
# SOME really odd atoms have both fields and children.
#
# But if it is not one of the odd ones, then it is a simple atom
#   with unknown fields, but no children.
#
def declare_RedundantAtoms(name=None):
    # if called with a name, then do right by that one atom name.
    # if called with no arguments, then recursively do all known atoms.
    #
    if name:
        clz = name
        if (ATOM_TYPE_MAP.has_key(name)):
            clz = ATOM_TYPE_MAP[name]
        clz = clz.strip()
        clz2 = ((clz<>name) and _type_valid(name)) and name or None
        if name in ATOM_WITH_ONE_CHILD:
            _declare_AtomWithOneChild(clz)
            if clz2: _declare_AtomWithOneChild(clz2)
        elif name in ATOM_WITH_COUNT_CHILDREN:
            _declare_AtomWithCountChildren(clz)
            if clz2: _declare_AtomWithCountChildren(clz2)
        elif name in ATOM_WITH_CHILDREN:
            _declare_AtomWithChildren(clz)
            if clz2: _declare_AtomWithChildren(clz2)
        else:
            _declare_Atom(clz)
            if clz2: _declare_Atom(clz2)
    else:
        for name in ATOM_WITH_ONE_CHILD:
            declare_RedundantAtoms(name)
        for name in ATOM_WITH_COUNT_CHILDREN:
            declare_RedundantAtoms(name)
        for name in ATOM_WITH_CHILDREN:
            # handle cases like 'dinf' that are in both lists for different reasons
            if not name in ATOM_WITH_ONE_CHILD:
                declare_RedundantAtoms(name)

# wrapper so I can test for this type, otherwise still a vector
class atom_vector(object):
    def __init__(self,data): self.data = data
    def __len__(self): return len(self.data)
    def __getitem__(self, key): return self.data[key]
    def __repr__(self): return repr(self.data)

class ftyp(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("ftyp: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self._set_attr('major_version', type_to_str(read32(file)))
        self._set_attr('minor_version', read32(file))
        brands = []
        limit = offset + size
        while (file.tell()+4) <= limit:
            brands.append(type_to_str(read32(file)))
        self._set_attr('compatible_brands', brands)
        self.atomsize += 4 + 4 + (4 * len(brands))

# 2011.07.24 - docs say that 'meta' does NOT have the extra DWORD before its children.
#    so just make it an ATOM_WITH_CHILDREN
#
# EXPERIENCE shows that Napier was correct, this unknown DWORD _MUST_ be consumed.
#
class meta(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("meta: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        log.debug("meta: s='%s' h='%s'" % (self.size,self.atomsize))
        # meta has an extra null after the atom header.  consume it here
        self._set_attr('unknown', read32(file))
        self.atomsize += 4
        log.debug("meta: u='%s'" % (self.get_attribute('unknown')))
        self._set_children(parse_atoms(file, offset + size))

class mvhd(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("mvhd: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        extra_header = (self.version == 1) and (8+8+4+8+4+2+2+8+(9*4)+(6*4)+4) or (4+4+4+4+4+2+2+8+(9*4)+(6*4)+4)
        self.atomsize += extra_header
        self._set_attr('creation_time', mp4DateTime( (self.version == 1) and read64(file) or read32(file)))
        self._set_attr('modification_time', mp4DateTime( (self.version == 1) and read64(file) or read32(file)))
        time_scale = read32(file)
        self._set_attr('time_scale', time_scale)
        duration = (self.version == 1) and read64(file) or read32(file)
        self._set_attr('duration', duration)
        fd = float(duration)/float(time_scale)
        td = datetime.timedelta(0,fd)
        self._set_attr('#duration', fd)
        self._set_attr('rate', read32(file))
        self._set_attr('volume', read16(file))
        self._set_attr('reserved1', read16(file))
        self._set_attr('reserved2', [read32(file),read32(file)])
        matrix = [[0,0,0],[0,0,0],[0,0,0]]
        for i in range(0,3):
            for j in range(0,3):
                matrix[i][j] = read32(file)
        self._set_attr('matrix', matrix)
        self._set_attr('reserved3', [read32(file),read32(file),read32(file),
                                     read32(file),read32(file),read32(file)])
        self._set_attr('next_track_id', read32(file))
        
        # ??? structure AND children ???
        slop = size - (file.tell() - offset)
        if slop >= 8:
            self._set_attr('_slop_as_children', slop)
            self._set_children(parse_atoms(file, offset + size))

class tkhd(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("tkhd: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        extra_header = (self.version == 1) and (8+8+8+8+8+2+2+2+2+(9*4)+4+4) or (4+4+8+4+8+2+2+2+2+(9*4)+4+4)
        self.atomsize += extra_header
        self._set_attr('creation_time', mp4DateTime( (self.version == 1) and read64(file) or read32(file)))
        self._set_attr('modification_time', mp4DateTime( (self.version == 1) and read64(file) or read32(file)))
        self._set_attr('track_id', read32(file))
        self._set_attr('reserved1', read32(file))
        duration = (self.version == 1) and read64(file) or read32(file)
        # fd = float(duration)/float(time_scale)    # oops! time_scale not in this atom. see mvhd
        # td = datetime.timedelta(0,fd,0,0,0,0,0)
        # self._set_attr('#duration', fd)
        self._set_attr('duration', duration)
        self._set_attr('reserved2', [read32(file),read32(file)])
        self._set_attr('layer', read16(file))
        self._set_attr('alternate_group', read16(file))
        self._set_attr('volume', read16(file))
        self._set_attr('reserved3', read16(file))
        transform = [[0,0,0],[0,0,0],[0,0,0]]
        for i in range(0,3):
            for j in range(0,3):
                transform[i][j] = read32(file)
        self._set_attr('transform_matrix', transform)
        self._set_attr('width', [read16(file),read16(file)]) # read32(file)
        self._set_attr('height', [read16(file),read16(file)]) # read32(file)

class mdhd(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("mdhd: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        extra_header = (self.version == 1) and (8+8+4+8+2+2) or (4+4+4+4+2+2)
        self.atomsize += extra_header
        self._set_attr('creation_time', mp4DateTime( (self.version == 1) and read64(file) or read32(file)))
        self._set_attr('modification_time', mp4DateTime( (self.version == 1) and read64(file) or read32(file)))
        time_scale = read32(file)
        self._set_attr('time_scale', time_scale)
        duration = (self.version == 1) and read64(file) or read32(file)
        self._set_attr('duration', duration)
        fd = float(duration)/float(time_scale)
        td = datetime.timedelta(0,fd,0,0,0,0,0)
        self._set_attr('#duration', fd)
        lang = read16(file)    # high bit is pad; 5 bits each chars
        lang = [0x60 + (lang & 0x1F), 0x60 + ((lang>>5) & 0x1F), 0x60 + ((lang>>10) & 0x1F)]
        lang = '%c%c%c' % (lang[0],lang[1],lang[2])
        self._set_attr('language', lang)
        self._set_attr('reserved', read16(file))

class stsc_record(object):
    def __init__(self, file):
        self.first_chunk = read32(file)
        self.samples_per_chunk = read32(file)
        self.sample_desc_index = read32(file)
class stsc(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("stsc: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 4
        count = read32(file)
        self._set_attr('count', count)

        # records = []
        # for i in range(0,count):
            # records.append(stsc_record(file))
        # self._set_attr('entries', atom_vector(records))
        # self._set_attr('_entry_size', 12)

        # Seek to the end of the expected data
        file.seek(self.offset + self.atomsize + (count*12), os.SEEK_SET)

    def entry(self, n):
        limit = self.get_attribute('count')
        if (n < 0) or (limit <= n):
            return None
        file.seek(self.offset + self.atomsize + (n*12), os.SEEK_SET)
        return stsc_record(self.file)

class stts_record(object):
    def __init__(self, file):
        self.sample_count = read32(file)
        self.sample_delta = read32(file)
class stts(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("tts: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 4
        count = read32(file)
        self._set_attr('count', count)

        # records = []
        # for i in range(0,count):
            # records.append(stts_record(file))
        # self._set_attr('entries', atom_vector(records))
        # self._set_attr('_entry_size', 8)

        # Seek to the end of the expected data
        file.seek(self.offset + self.atomsize + (count*8), os.SEEK_SET)

    def entry(self, n):
        limit = self.get_attribute('count')
        if (n < 0) or (limit <= n):
            return None
        file.seek(self.offset + self.atomsize + (n*8), os.SEEK_SET)
        return stts_record(self.file)

class ctts_record(object):
    def __init__(self, file):
        self.sample_count = read32(file)
        self.sample_offset = read32(file)
class ctts(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("ctts: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 4
        count = read32(file)
        self._set_attr('count', count)

        # records = []
        # for i in range(0,count):
            # records.append(ctts_record(file))
        # self._set_attr('entries', atom_vector(records))
        # self._set_attr('_entry_size', 8)

        # Seek to the end of the expected data
        file.seek(self.offset + self.atomsize + (count*8), os.SEEK_SET)

    def entry(self, n):
        limit = self.get_attribute('count')
        if (n < 0) or (limit <= n):
            return None
        file.seek(self.offset + self.atomsize + (n*8), os.SEEK_SET)
        return ctts_record(self.file)

class stco(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("stco: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 4
        count = read32(file)
        self._set_attr('offset_count', count)

        # records = []
        # for i in range(0,count):
            # records.append(read32(file))
        # self._set_attr('offsets', atom_vector(records))
        # self._set_attr('_offset_size', 4)
        # seek past payload

        # Seek to the end of the expected data
        file.seek(self.offset + self.atomsize + (count*4), os.SEEK_SET)

    def entry(self, n):
        limit = self.get_attribute('offset_count')
        if (n < 0) or (limit <= n):
            return None
        file.seek(self.offset + self.atomsize + (n*4), os.SEEK_SET)
        return read32(self.file)

class co64(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("co64: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 4
        count = read32(file)
        self._set_attr('offset_count', count)

        # records = []
        # for i in range(0,count):
            # records.append(read64(file))
        # self._set_attr('offsets', atom_vector(records))
        # self._set_attr('_offset_size', 8)

        # Seek to the end of the expected data
        file.seek(self.offset + self.atomsize + (count*4), os.SEEK_SET)

    def entry(self, n):
        limit = self.get_attribute('offset_count')
        if (n < 0) or (limit <= n):
            return None
        file.seek(self.offset + self.atomsize + (n*8), os.SEEK_SET)
        return read64(self.file)

class stss(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("stss: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 4
        count = read32(file)
        self._set_attr('sync_count', count)

        # records = []
        # for i in range(0,count):
            # records.append(read32(file))
        # self._set_attr('sync_table', atom_vector(records))
        # self._set_attr('_sync_size', 4)

        # Seek to the end of the expected data
        file.seek(self.offset + self.atomsize + (count*4), os.SEEK_SET)

    def entry(self, n):
        limit = self.get_attribute('sync_count')
        if (n < 0) or (limit <= n):
            return None
        file.seek(self.offset + self.atomsize + (n*4), os.SEEK_SET)
        return read32(self.file)

class stsz(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("stsz: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 8
        size = read32(file)
        self._set_attr('constant_size', size)
        count = read32(file)
        self._set_attr('size_count', count)

        # records = []
        # if size == 0:
            # for i in range(0,count):
                # records.append(read32(file))
        # self._set_attr('size_table', atom_vector(records))
        # self._set_attr('_size_size', 4)

        # Seek to the end of the expected data
        file.seek(self.offset + self.atomsize + ((size==0) and (count*4) or 0), os.SEEK_SET)

    def entry(self, n):
        constant_size = self.get_attribute('constant_size')
        size_count = self.get_attribute('size_count')
        if constant_size:
            return constant_size
        if (n < 0) or (size_count <= n):
            return None
        file.seek(self.offset + self.atomsize + (n*4), os.SEEK_SET)
        return read32(self.file)

class chpl_record(object):
    def __init__(self, file):
        self.timestamp = read64(file)
        self.title_size = read8(file)
        self.title = str(read(self.title_size))
class chpl(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("chpl: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 4
        count = read32(file)
        self._set_attr('count', count)

        # records = []
        # for i in range(0,count):
            # records.append(chpl_record(file))
        # self._set_attr('chapters', atom_vector(records))
        # self._set_attr('_chapter_size', 12)

        # Seek to the end of the expected data
        file.seek(self.offset + self.atomsize + (count*12), os.SEEK_SET)

    def entry(self, n):
        limit = self.get_attribute('count')
        if (n < 0) or (limit <= n):
            return None
        file.seek(self.offset + self.atomsize + (n*12), os.SEEK_SET)
        return chpl_record(self.file)

class pdin_record(object):
    def __init__(self, file):
        self.bit_rate = read32(file)
        self.initial_delay = read32(file)
class pdin(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("pdin: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        # this isn't children, it is an array until end of box of:
        #    {bit_rate=read32(), initial_delay=read32()}
        # self._set_children(parse_atoms(file, offset + size))

        # limit = offset + size
        # records = []
        # while file.tell() <= limit:
            # records.append(pdin_record(file))
        # self._set_attr('rate_delay', atom_vector(records))
        # self._set_attr('_entry_size', 8)

        # Seek to the end of the expected data
        file.seek(self.offset + self.atomsize + limit, os.SEEK_SET)

    def entry(self, n):
        limit = (self.size - self.atomsize) / 8
        if (n < 0) or (limit <= n):
            return None
        file.seek(self.offset + self.atomsize + (n*8), os.SEEK_SET)
        return pdin_record(self.file)

class elst_record(object):
    def __init__(self, file, version=0):
        self.version = version
        self.segment_duration = version==1 and read64(file) or read32(file)
        self.media_time = version==1 and read64(file) or read32(file)
        self.media_rate = read32(file)
class elst(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("elst: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 4
        count = read32(file)
        self._set_attr('count', count)

        # records = []
        # for i in range(0,count):
            # records.append(elst_record(file, self.version))
        # self._set_attr('rate_delay', atom_vector(records))
        # self._set_attr('_entry_size', self.version==1 and 20 or 12)

        # Seek to the end of the expected data
        file.seek(self.offset + self.atomsize + (self.version==1 and 20 or 12), os.SEEK_SET)

    def entry(self, n):
        entry_size = (self.version==1 and 20 or 12)
        limit = (self.size - self.atomsize) / entry_size
        if (n < 0) or (limit <= n):
            return None
        file.seek(self.offset + self.atomsize + (n*entry_size), os.SEEK_SET)
        return pdin_record(self.file, self.version)

class auth(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("auth: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 2
        lang = read16(file)    # high bit is pad; 5 bits each chars
        lang = [0x60 + (lang & 0x1F), 0x60 + ((lang>>5) & 0x1F), 0x60 + ((lang>>10) & 0x1F)]
        lang = '%c%c%c' % (lang[0],lang[1],lang[2])
        self._set_attr('language_code', lang)
        howMuch = size - (file.tell() - offset)
        self._set_attr('tag_string', unicode(file.read(howMuch), "utf-8"))

class titl(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("titl: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 2
        lang = read16(file)    # high bit is pad; 5 bits each chars
        lang = [0x60 + (lang & 0x1F), 0x60 + ((lang>>5) & 0x1F), 0x60 + ((lang>>10) & 0x1F)]
        lang = '%c%c%c' % (lang[0],lang[1],lang[2])
        self._set_attr('language_code', lang)
        howMuch = size - (file.tell() - offset)
        self._set_attr('tag_string', unicode(file.read(howMuch), "utf-8"))

class desc(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("desc: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 2
        lang = read16(file)    # high bit is pad; 5 bits each chars
        lang = [0x60 + (lang & 0x1F), 0x60 + ((lang>>5) & 0x1F), 0x60 + ((lang>>10) & 0x1F)]
        lang = '%c%c%c' % (lang[0],lang[1],lang[2])
        self._set_attr('language_code', lang)
        howMuch = size - (file.tell() - offset)
        self._set_attr('tag_string', unicode(file.read(howMuch), "utf-8"))

class cprt(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("cprt: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 2
        lang = read16(file)    # high bit is pad; 5 bits each chars
        lang = [0x60 + (lang & 0x1F), 0x60 + ((lang>>5) & 0x1F), 0x60 + ((lang>>10) & 0x1F)]
        lang = '%c%c%c' % (lang[0],lang[1],lang[2])
        self._set_attr('language_code', lang)
        howMuch = size - (file.tell() - offset)
        self._set_attr('tag_string', unicode(file.read(howMuch), "utf-8"))

class data(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("data: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 4    # don't count payload
        self._set_attr("reserved", read32(file))
        payload_size = size - (file.tell() - offset)
        self._set_attr("_payload_size", payload_size)
        payload = None
        if self.flags == 1:    # text data
            if payload_size>255: log.warning("payload_size %s greater than 255" % (payload_size))
            payload = str(file.read(payload_size))
        elif self.flags == 21 or self.flags == 0: # 0=custom; 21=generic
            if payload_size  == 0: payload = 0
            elif payload_size  == 1: payload = read8(file)
            elif payload_size  == 2: payload = read16(file)
            elif payload_size  == 3: payload = [read8(file),read8(file),read8(file)]
            elif payload_size  == 4: payload = read32(file)
            elif payload_size  == 8: payload = [read32(file),read32(file)]
            else:
                payload = [read8(file),read8(file),read8(file),read8(file)]
                # the rest as slop
        elif self.flags == 13 or self.flags == 14: #binary
            payload = atom_vector(file.read(payload_size))
        else:
            log.warning("unknown self.flags in '%s': %s @ %s" % (self.type,self.flags, _sofar()))
        self._set_attr("payload", payload)

class avc1(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("avc1: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self._set_attr("unknownA", read32(file))
        self._set_attr("unknownB", read16(file))
        self._set_attr("dref_index", read16(file))
        self._set_attr("application", type_to_str(read32(file)))
        self._set_attr("unknownC", [read32(file),read32(file)])
        self._set_attr("reservedC", [read16(file),read16(file)]) # read32(file)
        self._set_attr("reservedD", [read16(file),read16(file)]) # read32(file)
        self._set_attr("reservedE", [read16(file),read16(file)]) # read32(file)
        self._set_attr("reservedF", [read16(file),read16(file)]) # read32(file)
        self._set_attr("reservedG", read16(file))
        name_len = read8(file)
        name_buf = file.read(31)
        self._set_attr("name_len", name_len)
        name, slop = name_len and _parse_nul_string(name_buf) or ("","")
        self._set_attr("name", str(name))
        if slop: self._set_attr("name_slop", atom_vector(slop))
        self._set_attr("reservedH", read16(file))
        self._set_attr("unknownD", read16(file))

        # # ??? structure AND a child ???
        # self._set_children(parse_atoms(file, offset + size))

        self._set_attr("unknownE", [read16(file),read16(file)]) # read32(file)
        self._set_children(parse_N_atoms(1, file, offset + size))

class mp4v(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("mp4v: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += ((6+1)+2+(4*4)+4+4+4+4+2+1+31+2+2)
        self._set_attr("reservedA", [read8(file),read8(file),read8(file),read8(file),read8(file),read8(file)])
        self._set_attr("dref_index", read16(file))
        self._set_attr("reservedB", [read32(file),read32(file),read32(file),read32(file)])
        self._set_attr("reservedC", [read16(file),read16(file)]) # read32(file)
        self._set_attr("reservedD", [read16(file),read16(file)]) # read32(file)
        self._set_attr("reservedE", [read16(file),read16(file)]) # read32(file)
        self._set_attr("reservedF", [read16(file),read16(file)]) # read32(file)
        self._set_attr("reservedG", read16(file))
        name_len = read8(file)
        name_buf = file.read(31)
        self._set_attr("name_len", name_len)
        name, slop = name_len and _parse_nul_string(name_buf) or ("","")
        self._set_attr("name", str(name))
        if slop: self._set_attr("name_slop", atom_vector(slop))
        self._set_attr("reservedH", read16(file))
        self._set_attr("reservedI", read16(file))

        # ??? structure AND children ???
        slop = size - (file.tell() - offset)
        if slop >= 8:
            self._set_attr('_slop_as_children', slop)
            self._set_children(parse_atoms(file, offset + size))

class vmhd(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("vmhd: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += (2+(3*2))
        self._set_attr("graphics_mode", read16(file))
        self._set_attr("op_color", [read16(file),read16(file),read16(file)])

class smhd(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("smhd: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += (2+2)
        self._set_attr("balance", read16(file))
        self._set_attr("reserved", read16(file))

class hmhd(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("hmhd: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += (2+2+2+2+4)
        self._set_attr("max_PDU_size", read16(file))
        self._set_attr("avg_PDU_size", read16(file))
        self._set_attr("max_bit_rate", read16(file))
        self._set_attr("avg_bit_rate", read16(file))
        self._set_attr("reserved", read32(file))

class dlay(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("dlay: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 4
        self._set_attr("scroll_delay", read32(file))

        # ??? structure AND children ???
        slop = size - (file.tell() - offset)
        if slop >= 8:
            self._set_attr('_slop_as_child', slop)
            self._set_children(parse_N_atoms(1, file, offset + size))

# returns a tuple with [0]=all up to the first NUL byte, [1]=all after the NUL byte
def _parse_nul_string(data):
    if not data:
        return ("", "")
    first_nul_index = -1
    for i in range(0,len(data)):
        if 0 == data[i]:
            first_nul_index = i
            break;
    if first_nul_index < 0:
        return (data, "")
    return ( data[:first_nul_index], data[first_nul_index+1:] )

class urn(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("urn : t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        slop = size - (file.tell() - offset)
        slop = slop and file.read(slop) or None
        # TODO: parse name = slop up to first NUL as string; remove name and NUL from slop
        name, slop = slop and _parse_nul_string(slop) or ("","")
        # TODO: parse location = slop up to first NUL as string; remove location and NUL from slop
        location, slop = slop and _parse_nul_string(slop) or ("","")
        self._set_attr("name", unicode(name, "utf-8"))
        self._set_attr("location", unicode(location, "utf-8"))

class url(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("url : t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        slop = size - (file.tell() - offset)
        slop = slop and file.read(slop) or None
        # TODO: parse location = slop up to first NUL as string; remove location and NUL from slop
        location, slop = slop and _parse_nul_string(slop) or ("","")
        self._set_attr("location", location)

class mp4a(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("mp4a: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += (6+2+8+2+2+4+2+2)
        self._set_attr("reservedA", [read8(file),read8(file),read8(file),read8(file),read8(file),read8(file)])
        self._set_attr("dref_index", read16(file))
        self._set_attr("reservedB", [read32(file),read32(file)])
        self._set_attr("channel_count", read16(file))
        self._set_attr("sample_size", read16(file))
        self._set_attr("reservedC", read32(file))
        self._set_attr("time_scale", read16(file))
        self._set_attr("reservedD", read16(file))
        esda_size = size - (file.tell() - offset)

        # ??? really one 'esda' atom ??? leave esda_size for normal read ???
        if esda_size >= 8:
            self._set_children(parse_N_atoms(1, file, offset + size))
        else:
            self._set_attr("ESDAtom", atom_vector(file.read(esda_size)))

class uuid(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("uuid: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += (16*1)
        bytes = []
        for i in range(0,16): bytes.append(read8(file))
        self._set_attr("uuid", atom_vector(bytes))

class hdlr(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("hdlr: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += (4+4+12) # don't count 'name', we don't know its length yet
        self._set_attr("reserved1", read32(file))
        self._set_attr("handler_type", type_to_str(read32(file)))
        bytes = []
        for i in range(0,12): bytes.append(read8(file))
        self._set_attr("reserved2", atom_vector(bytes))
        name_len = self.size - self.atomsize
        name_buf = file.read(name_len)
        name, slop = name_len and _parse_nul_string(name_buf) or ("","")
        self._set_attr("handler_name", str(name))
        if slop: self._set_attr("name_slop", atom_vector(slop))
        self.atomsize += len(name) # now we know the name's size

# ilst is an AtomWithCountChildren, except that it is different!
#    it can't call parse_atom, because it needs to declare a side effect.
#
class ilst(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("ilst: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)

        # according to Flash documentation, this is a counted list.
        #     but an abnormal list, is that the contents are versioned
        #     although they are likely not in our known versioned list.
        #
        # self.atomsize += 4
        # count = read32(file)
        # self._set_attr("count", count)
        # tags = []
        # for i in range(0,count):
        #     try:
        #         tag_offset = file.tell()
        #         tag_size = read32(file)
        #         tag_type = type_to_str(read32(file))
        #         _sofar_push(tag_type)
        #         print "debug: ilst: sofar:", _sofar()
        #         if (tag_size == 1):
        #             tag_size = read64(file)
        #             # if VERBOSE: print "debug: atom '%s' large size: %s" % (type, size)
        #         # HERE IS THE DIFFERENCE!
        #         #    ilst.tag is ALWAYS versioned, but it is not in ATOM_WITH_VERSION
        #         #        since it is an arbitrary tag we may not know about.
        #         tag_versionflags = read32(file)
        #         tag = create_atom(tag_size, tag_type, tag_versionflags, tag_offset, file)
        #         _sofar_pop()
        #         if tag:
        #             tags.append(tag)
        #     except EndOFFile:
        #         pass
        # self.set_children(tags)

        # according to Napier's original code, (and according to experience)
        #     ilst is a normal AtomWithChildren, not a counted Atom with children
        #
        self._set_children(parse_atoms(file, offset + size))

# atom actual type '----'
#   to avoid python class name issues, it is mapped to 'four_dashes'
#   appears to be a special Apple iTunes atom having to do with reverse dns
#       children:
#           'mean'
#           'name'
#           'data'
class four_dashes(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("four_dashes: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self._set_children(parse_atoms(file, offset + size))
class mean(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("mean: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 4
        self._set_attr('unknown',read32(file))
        payload_size = size - (file.tell() - offset)
        self._set_attr("_payload_size", payload_size)
        payload = str(file.read(payload_size))
        self._set_attr("payload", payload)
class name(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("name: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        self.atomsize += 4
        self._set_attr('unknown',read32(file))
        payload_size = size - (file.tell() - offset)
        self._set_attr("_payload_size", payload_size)
        payload = str(file.read(payload_size))
        self._set_attr("payload", payload)

# according to: http://www.mp4ra.org/specs.html#id3v2
# TODO: massage eyed3 to be able to aprse this data for us
class ID32(Atom):
    def __init__(self, size, type, name, versionflags, offset, file):
        log.debug("ID32: t='%s' n='%s'" % (type,name))
        Atom.__init__(self, size, type, name, versionflags, offset, file)
        lang = read16(file)    # high bit is pad; 5 bits each chars
        lang = [0x60 + (lang & 0x1F), 0x60 + ((lang>>5) & 0x1F), 0x60 + ((lang>>10) & 0x1F)]
        lang = '%c%c%c' % (lang[0],lang[1],lang[2])
        self._set_attr('language', lang)
        id32_size = size - (file.tell() - offset)
        self._set_attr("id32_size", id32_size)
        id32 = atom_vector(file.read(id32_size))
        self._set_attr("id32_tags", id32)



class AtomFileException(Exception):
    def __init_(self):
        Exception.__init__(self)

###############################################################################
# convenience to read in an entire file.
#   without having to care about the details of calling the deeper classes
#
class atomFile(AtomWithChildren):
    """
    A convenience class to read in an entire file, following the MPEG-4 Part 14 structure.
        (and related documents and web sites)
    
    -param: filename : May be the path to a file, or may be an open file stream like object.
                       It must be seekable.

    -param: file_offset=0 : Optional starting offset within filename for this "AtomFile"

    -param: file_size=0   : Optional total size within filename for this "AtomFile"
    
    If both offset and filesize are zero (not specified) then the entire file is parsed.
    """
    def __init__(self, filename, file_offset=0, file_size=0):
        log.debug("atomFile: %s" % ('-'*32))
        log.debug("atomFile: %s" % (filename))
        if isinstance(filename, basestring):
            # 2012.01.05 - Jesse Chisholm
            if -1 != filename.find('://'):
                import urllib2
                file = urllib2.urlopen(filename)
            else:
                file = open(filename,"rb")
            if not file_size:
                file.seek(0, os.SEEK_END)
                file_size = file.tell() - file_offset
            file.seek(file_offset, os.SEEK_SET)
            AtomWithChildren.__init__(self, file_size, '', '', 0, file_offset, file)
            self._report_slop(file)
        elif isinstance(filename, file):
            if not file_size:
                filename.seek(0, os.SEEK_END)
                file_size = filename.tell() - file_offset
            filename.seek(file_offset, os.SEEK_SET)
            AtomWithChildren.__init__(self, file_size, '', '', 0, file_offset, filename)
            self._report_slop(filename)
        else:
            raise AtomFileException("'filename' is neither a file nor the path to a file")
