# -*- coding: utf-8 -*-
# -*- coding: utf-8 -*-

__license__ = 'GPL 3'
__copyright__ = '2011, Jesse Chisholm <jesse.chisholm@gmail.com>'
__docformat__ = 'restructuredtext en'

import os, re

from common import DEBUG
from common import VERBOSE
from common import log
from common import AUDIOBOOK_KNOWN_GENRE
from common import AudioBookException
from baseWrapper import baseWrapper

###############################################################################
#
#   a convenience class to use mp4file to fetch the information we need
#   and allow easy requests for information fields, whether they exist or not.
#
#   usage:
#      m4 = mp4Wrapper(afile)
#      title = m4.get('title')                  # None returned if not available
#      author = m4.get('author', _('Unknown'))  # specify a default value
#
class mp4Wrapper(baseWrapper):
    # @param:afile - the open file object, or the path to a file
    # @throws: IOError if afile is not one of the above
    #
    def __init__(self, afile):
        baseWrapper.__init__(self, afile)

    def _parse_file(self, afile):
        from mp4file import atom as mp4Atom
        # mp4Atom.declare_RedundantAtoms()
        try:
            mp4File = mp4Atom.atomFile(self.filename)   # set in baseWrapper.__init__
        except:
            print("error: unknown argument type: %s : %s" % (type(afile), str(afile)))
            log.error("unknown argument type: %s : %s" % (type(afile), str(afile)))
            raise AudioBookException("mp4Wrapper can't handle argument of type: %s" % (type(afile)))
        #
        if VERBOSE: print("debug: mp4Wrapper: filename : %s" % (self.filename))
        if DEBUG: mp4File.verify()
        #
        ID32 = _mp4_item_value(mp4File, "ID32")
        if ID32:
            # special 'ID32' tags contains ID3v2 tags
            #   save for caller! not as an atom_vector, but as a byte[]
            self.set('ID32', ID32.data)
        #
        title = _mp4_item_value(mp4File, "title")
        album = _mp4_item_value(mp4File, "album")
        track = _mp4_item_value(mp4File, "tracknum")    # returns tuple (track#, numTracks)
        genre = _mp4_item_value(mp4File, "genre")
        author = _mp4_item_value(mp4File, "artist")
        pubdate = _mp4_earliest_date(mp4File)
        duration = _mp4_duration(mp4File)
        copyright = _mp4_item_value(mp4File, "copyright")
        # copyright date takes precedence over mere file creation date
        if copyright and pubdate:
            # assume format: "Copyright \xa9 YYYY, company." or similar.
            temp = re.search("(\d+)", copyright)
            if temp:
                log.debug("pubdate: replacing '%s' with '%s'" % (pubdate, temp.group(1)))
                pubdate = temp.group(1)
        cover, cover_data = None, None
        (dir, name) = os.path.split(self.filename);
        (sname, ext) = os.path.splitext(name)
        (cover, cover_data) = _get_mp4_art(mp4File)
        #
        # track, if defined, is tuple (track#, numTracks)
        chapter = track and ("%d" % (track[0])) or None
        title_sort = title
        if chapter and title:
            title_sort = "%#03.f %s" % (float(chapter), title)

        #
        genre_list = []
        if genre:
            genre_list.append(genre)
            ok = False;
            for entry in AUDIOBOOK_KNOWN_GENRE:
                ok = (genre.strip().lower() == entry.lower()) or ok;
            if not ok:
                print ("warning: Genre unexpected: '%s'" % (genre));
                log.warning("Genre unexpected: '%s'" % (genre));
                genre_list.append(AUDIOBOOK_KNOWN_GENRE[0])
        else:
            # just to have something for the tags field.
            genre_list.append(AUDIOBOOK_KNOWN_GENRE[0])
        #
        # now set it all for the convenience of our owner
        #
        if title: self.set('title',title)
        if title_sort: self.set('title_sort',title_sort)
        if album: self.set('series',album)
        if chapter: self.set('series_index', float(chapter))
        if author:
            if "&" in author:
                aus = author.split("&")
                for a in range(0,len(aus)): aus[a] = aus[a].strip()
                self.set('authors',aus)
            elif " and " in author:
                aus = author.split(" and ")
                for a in range(0,len(aus)): aus[a] = aus[a].strip()
                self.set('authors',aus)
            else:
                self.set('authors',[author])
        self.set('tags',genre_list)
        # self.set('publisher',publisher)
        if pubdate: self.set('pubdate',pubdate)
        if duration: self.set('#duration', duration)    # a custom column value
        #
        # TODO: does tags have anything for stream mime?
        #
        self.set('mime',"audio/mpeg")
        mp4File.log()

###############################################################################

# fetch the value of a 'User Data Item'
# @param:atom - the atom that is the mp4 file
# @param:name - the name, or type, of the item requested
# @return: the string that represents the value; or None
#
def _mp4_item_value(atom, name):
    obj = atom and atom.find("./moov/udta/meta/ilst/"+name+"/data") or None
    val = obj and obj.get_attribute("payload") or None
    # from calibre_plugins.audiobookmetadatareader import mp4file as mp4
    # from . import mp4file as mp4
    import mp4file as mp4
    if type(val) == mp4.atom.atom_vector:
        val = val.data
    return val

###############################################################################

# fetch the entire file's duration
# @param:atom - the atom that is the mp4 file
# @return: the floating point number of seconds in the file; or None
#
def _mp4_duration(atom):
    mvhd = atom and atom.find("./moov/mvhd/") or None
    val = mvhd and mvhd.get_attribute("#duration") or None
    return val

# fetch an image from the mp4 style file
# @param:atom - the atom that is the mp4 file
# @return:
#     on failure: tuple (None, None)
#     on success: tuple (filepath, (mimetype, imagedata))
#
def _get_mp4_art(atom):
    # from calibre_plugins.audiobookmetadatareader import mp4file as mp4
    # from . import mp4file as mp4
    import mp4file as mp4
    cover, cover_data = None, None;
    #
    coverObj = atom.find(".*/coverart/data")
    coverLoad = coverObj and coverObj.get_attribute("payload")
    if coverLoad and type(coverLoad) == mp4.atom.atom_vector:
        # TODO: determine mime type from data
        cover_data = ( _get_mime_from_data(coverLoad.data), coverLoad.data)
    #
    return (cover, cover_data);

# look in all tags that have a creation datetime or modification datetime
# and choose the farthest into the past.
# @param:atom - the atom that is the mp4 file
# @return: the string that represents the datetime; or None
#
def _mp4_earliest_date(atom):
    choices = atom.findall("/moov/mvhd")
    choices.extend(atom.findall("/moov/trak/tkhd"))
    choices.extend(atom.findall("/moov/trak/mdia/mdhd"))
    ours = None
    for item in choices:
        temp = item.get_attribute("creation_time")
        if not ours or temp and temp.strftime() < ours.strftime(): ours = temp
        temp = item.get_attribute("modification_time")
        if not ours or temp and temp.strftime() < ours.strftime(): ours = temp
    return ours and ours.strftime() or None

###############################################################################
