#!/usr/bin/env python
from __future__ import (unicode_literals, division, absolute_import,print_function)

__license__   = 'GPL v3'
__copyright__ = '2012, Sebastiaan van Schaik <bas@tuxes.nl>'
__docformat__ = 'restructuredtext en'


SONY_VENDOR_ID = [0x054c]
SONY_PRST1_PRODUCT_ID = [0x05c2]

from PyQt4.Qt import QDialog, QVBoxLayout, QPushButton, QMessageBox, QLabel, QTableWidget, QButtonGroup, QHBoxLayout
import os
import calibre.devices.prst1.driver as prst1_driver
import sqlite3 as sqlite
from contextlib import closing
from calibre_plugins.prst1_notes.PRST1NotesBooksTable import *
from calibre_plugins.prst1_notes.PRST1NotesMergeInterface import *
from calibre import prints

class PRST1NotesDialog(QDialog):
	def __init__(self, gui, icon, do_user_config):
		QDialog.__init__(self, gui)
		
		self.gui = gui
		self.icon = icon

		self.l = QVBoxLayout()
		self.setLayout(self.l)

		self.setWindowTitle('Sony PRS-T1 notes plugin')
		self.setWindowIcon(icon)

		# Find out if its a Sony PRS-T1 we're dealing with here
		self.sony_dbpath = None
		self.sony_path = None
		if gui.device_manager.connected_device is not None:
			self.connected_device_info = self.gui.device_manager.get_current_device_information().get('info', None)
			self.connected_device = self.gui.device_manager.connected_device
			vendor_id = self.connected_device.VENDOR_ID
			product_id = self.connected_device.PRODUCT_ID

			
			if (vendor_id == SONY_VENDOR_ID and product_id == SONY_PRST1_PRODUCT_ID):
				# Sony PRS-T1 connected
				# TODO: this might not work in case the PRS-T1 has a SD-card inserted to it...
				self.sony_dbpath = self.connected_device.normalize_path(self.connected_device._main_prefix + prst1_driver.DBPATH)
				self.sony_path = self.connected_device.normalize_path(self.connected_device._main_prefix)
			# end if
		# end if

		self.initInterface()
	# end def __init__

	def initInterface(self):
		if (self.sony_dbpath is None):
			# Unsupported or no device
			if gui.device_manager.connected_device is not None:
				connected_device_label = QLabel('<b>Connected device</b>: ' + str(self.connected_device_info[1]) + ' (UNSUPPORTED)')
			else:
				connected_device_label = QLabel('<b>Connected device</b>: none')
			# end if
		else:
			connected_device_label = QLabel('<b>Connected device</b>: Sony PRS-T1 (supported)')
		# end if

		self.l.addWidget(connected_device_label)


		books_table_label = QLabel('<b>PDF books with notes on device</b>:')
		self.l.addWidget(books_table_label)
		self.books = None

		if self.sony_dbpath is None:
			self.l.addWidget(QLabel('No device connected, or the connected device is unsupported. Sorry.'))
			self.resize(self.sizeHint())
		else:
			# Display table with books with notes
			self.books = self.getBooksWithNotes();
			self.books_table = PRST1NotesBooksTable(self, self.books)
			self.books_table.clicked.connect(self.books_table_clicked)
			self.l.addWidget(self.books_table)
		
			self.resize(600,300)
		# end if


		# Create PDF button (disabled by default)
		buttonFrame = QFrame(self)
		buttonFrameLayout = QHBoxLayout()
		buttonFrameLayout.setContentsMargins(0,0,0,0)
		buttonFrame.setLayout(buttonFrameLayout)
	
		self.createpdf_button = QPushButton('Create PDF: merge book and handwriting', self)
		self.createpdf_button.clicked.connect(self.createPdf)
		self.createpdf_button.setEnabled(False)
		buttonFrameLayout.addWidget(self.createpdf_button)		

		about_button = QPushButton('About', self)
		about_button.clicked.connect(self.about)
		buttonFrameLayout.addWidget(about_button)

		self.l.addWidget(buttonFrame)


	def getBooksWithNotes(self):
		books = []
		
		prints("Querying device for PDF books with freehand annotations...")

		with closing(sqlite.connect(self.sony_dbpath)) as connection:
			cursor = connection.cursor()

			query = 'SELECT books._id, books.title, books.author, books.file_path, count(freehand.page) FROM freehand,books WHERE content_id=books._id and books.file_path LIKE \'%pdf%\' GROUP BY freehand.content_id ORDER BY books.title;'
			cursor.execute(query)

			for i, row in enumerate(cursor):
				book = {}
				book['id'] = row[0]
				book['title'] = row[1]
				book['author'] = row[2]
				book['file_path'] = row[3]
				book['numpageswithnotes'] = row[4]

				books.append(book)

				prints("Found book with freehand annotations: ")
				prints(book)
			# end for

			cursor.close()
		# end with
	
		prints("Books with freehand annotations:")
		prints(books)


		return books
	# end def getBooks
	
	def about(self):
		# Get the about text from a file inside the plugin zip file
		text = get_resources('about.txt')
		QMessageBox.about(self, 'About the Sony PRS-T1 notes plugin', text.decode('utf-8'))

	def books_table_clicked(self):
		selectedRows = self.books_table.selectionModel().selectedRows()
		if (len(selectedRows) == 1):
			self.createpdf_button.setEnabled(True)
		else:
			self.createpdf_button.setEnabled(False)
		# end if

	# end def

	

	def createPdf(self):
		# Create a PDF by merging handwriting notes with original book
		selectedRows = self.books_table.selectionModel().selectedRows()
		if (len(selectedRows) != 1): return

		selectedIndices = self.books_table.selectedIndexes()
		rowIndex = selectedIndices[0].row()
		book = self.books[rowIndex]
	
		mergeDialog = PRST1NotesMergeInterface(self.gui, self, self.icon, book)
		mergeDialog.show()	
	# end def
# end class

