__license__   = 'GPL v3'
__copyright__ = '2012, Sebastiaan van Schaik <bas@tuxes.nl>'
__docformat__ = 'restructuredtext en'


from PyQt4.QtGui import *
from PyQt4.QtCore import *
from PyQt4.QtSvg import *
from pyPdf import PdfFileWriter, PdfFileReader
import sys
import sqlite3 as sqlite
import tempfile
import os
import shutil
import re
import calibre.devices.prst1.driver as prst1_driver
from contextlib import closing
from calibre import prints


''' This class implements the actual merging of notes with books, i.e. all the PDF operations'''

class PRST1NotesMerger:
	def __init__(self, mainInterface, mergeInterface, book):
		self.mergeInterface = mergeInterface
		self.mainInterface = mainInterface
		self.sony_path = mainInterface.sony_path
		self.sony_dbpath = mainInterface.sony_dbpath
		self.book = book
		self.bookid = book['id']
	# end def __init__

	def getPyPDFBook(self, overrideFilename = False):
		if overrideFilename is not False:
			pdfpath = overrideFilename
		else:
			pdfpath = os.path.join(self.sony_path, self.book['file_path'])
		# end if

		prints("Opening PDF file: " + pdfpath)

		if os.path.exists(pdfpath):
			reader = PdfFileReader(open(pdfpath, "rb"))
			prints("PDF file successfully opened")
			return reader
		else:
			# File does not exist?
			prints("Error: PDF file does not exist?")
			return None
		# end if
	# end def

	def doMerge(self, outputPdf, onlyPagesWithNotes):
		tmpdir = tempfile.mkdtemp()
		prints("Using temporary directory: " + tmpdir)

		# Copy book to tmpdir
		book_tmpfile = os.path.join(tmpdir, "book.pdf")
		book_sonypath = os.path.join(self.sony_path, self.book["file_path"])
		prints("Copying book PDF to tmpdir: " + book_sonypath + " => " + book_tmpfile)
		shutil.copy(book_sonypath, book_tmpfile)	

		# Open book using PyPDF
		pypdf_original_book = self.getPyPDFBook(book_tmpfile)
		if pypdf_original_book is None:
			return false
		# end if

		# Output file
		pypdf_output_book = PdfFileWriter()

		# Range of points in SVG file:
		#  - from 0-600 in x-direction
		#  - from 0-770 in y-direction
		EREADER_WIDTH = 600.0
		EREADER_HEIGHT = 770.0

		# Range over each and every single page
		svgfiles = self.getSVGFiles()
		for pageno in range(1, self.getBookNumPages() + 1):
			if pageno in svgfiles:
				# We need to do something with an SVG file
				svgfile = svgfiles[pageno]
				svg_tmpfile = os.path.join(tmpdir, "p" + str(pageno) + "-notes.svg")
				svg_tmpfile_unpatched = svg_tmpfile + "-sony"
				pdfnotes_tmpfile = os.path.join(tmpdir, "p" + str(pageno) + "-notes.pdf")
				merged_tmpfile = os.path.join(tmpdir, "p" + str(pageno) + "-merged.pdf")
				svg_sonypath = os.path.join(self.sony_path, svgfile)

				prints("Copying notes on page " + str(pageno) + " to tmpdir: " + svg_sonypath + " => " + svg_tmpfile_unpatched)
				shutil.copy(svg_sonypath, svg_tmpfile_unpatched)

				# Convert Sony SVG format to regular format
				prints("Converting Sony SVG format to regular SVG: " + svg_tmpfile_unpatched + " => " + svg_tmpfile)
				infile = open(svg_tmpfile_unpatched, "r")
				outfile = open(svg_tmpfile, "w")
				svg_contents = infile.read()
				infile.close()
				
				# Remove "n1:"-prefixes from tags, e.g.  <n1:polyline (...)>
				svg_contents = re.sub("n1:","", svg_contents)

				# Remove a lot of tags that will just confuse the SVG renderer
				svg_contents = re.sub("<n0:notepad.*<svg", "<svg", svg_contents)
				svg_contents = re.sub("<\/n0:page.*notepad>", "", svg_contents)

				outfile.write(svg_contents)
				outfile.close()

				prints("Converting SVG file into PDF file: " + svg_tmpfile + " => " + pdfnotes_tmpfile)

				# Convert the SVG file into a PDF file (without actually rendering the SVG into a bitmap!). The
				# resulting PDF file will later be merged with the pages from the book. This means the SVG should
				# be rescaled and repositioned!
				svg_renderer = QSvgRenderer(svg_tmpfile)
				svg_viewbox = svg_renderer.viewBoxF()
				
				# Determine the size of the book page
				pypdf_page = pypdf_original_book.getPage(pageno - 1)
				pypdf_page_width = float(pypdf_page.mediaBox[2])
				pypdf_page_height = float(pypdf_page.mediaBox[3])
				#prints("Page " + str(pageno) + " has size " + str(pypdf_page_width) + "x" + str(pypdf_page_height))

				# The SVG will be "printed" on a PDF printer
				pdf_printer = QPrinter(QPrinter.HighResolution)
				pdf_printer.setPaperSize(QSizeF(pypdf_page_width,pypdf_page_height), QPrinter.Point)
				pdf_printer.setOutputFormat(QPrinter.PdfFormat)
				pdf_printer.setOutputFileName(pdfnotes_tmpfile)

				painter = QPainter(pdf_printer)


				# We know the size of the PDF page, we know the size of the e-reader screen. Assuming
				# the ereader rescaled the PDF to fit to the screen, which offset should we use for the
				# coordinates of the handwriting?
				reader_xscale = (EREADER_WIDTH / pypdf_page_width)
				reader_yscale = (EREADER_HEIGHT / pypdf_page_height)
				reader_scale = min(reader_xscale, reader_yscale)

				reader_offset_x = 30 + (EREADER_WIDTH - (pypdf_page_width * reader_scale)) / 2.0
				reader_offset_y = 30 + (EREADER_HEIGHT - (pypdf_page_height * reader_scale)) / 2.0

				# The renderer viewbox tells us the width of the drawing, but this doesn't include its x_offset from 0. So a rectangle
				# of size 100 and a height of 50 in the middle of the screen will yield a viewbox of 100x50.
				# We call the drawing "the block", which has a width, height and x,y offset
				svg_blockwidth = svg_viewbox.width()
				svg_blockheight = svg_viewbox.height()
				svg_blockoffset_x = svg_viewbox.x()
				svg_blockoffset_y = svg_viewbox.y()

				# We know that the reader centered the PDF in either horizontal or vertical direction, correct for this
				svg_blockoffset_x = svg_blockoffset_x - reader_offset_x
				svg_blockoffset_y = svg_blockoffset_y - reader_offset_y

				# svg_pagewidth and svg_blockheight are the size of the block plus their x,y offset
				svg_pagewidth = svg_blockwidth + svg_blockoffset_x
				svg_pageheight = svg_blockheight + svg_blockoffset_y

				# available space on painter (output PDF)
				pdf_painter_width = painter.viewport().width()
				pdf_painter_height = painter.viewport().height()

				# Compute ratio: how much would we have to scale up the SVG in order to perfectly fit the target PDF page size?
				xscale = (pdf_painter_width) / (EREADER_WIDTH - 2 * reader_offset_x)
				yscale = (pdf_painter_height) / (EREADER_HEIGHT - 2 * reader_offset_y)
				scale = min(xscale,yscale)

				svg_rescaled_blockwidth = svg_blockwidth * scale
				svg_rescaled_blockheight = svg_blockheight * scale

				svg_target_x = svg_blockoffset_x * scale
				svg_target_y = svg_blockoffset_y * scale
				svg_target_rect = QRectF(svg_target_x,svg_target_y,svg_rescaled_blockwidth,svg_rescaled_blockheight)

				svg_renderer.render(painter, svg_target_rect)
				painter.end()

				# The PDF file has been created and is stored as
				# pdfnotes_tmpfile
				# Now combine the book page with the handwriting!
				input_handwriting = PdfFileReader(open(pdfnotes_tmpfile, "rb"))

				prints("Merging handwriting and book page into output PDF: page " + str(pageno) + "...")
				output_page = pypdf_original_book.getPage(pageno - 1)
				output_page.mergePage(input_handwriting.getPage(0))
				pypdf_output_book.addPage(output_page)

				input_handwriting.stream.close()
			else:
				# This page number doesn't have any handwriting
				if not onlyPagesWithNotes:
					# include this page anyway
					prints("Including page without handwriting in output PDF: page " + str(pageno) + "...")
					pypdf_output_book.addPage(pypdf_original_book.getPage(pageno - 1))
				# else: don't include page	
			# end if: handwriting on this page or not?

			self.mergeInterface.progressbar.setValue(pageno)
			QCoreApplication.processEvents()
		# end for each page

		outputStream = file(outputPdf, "wb")
		pypdf_output_book.write(outputStream)
		outputStream.close()
		pypdf_original_book.stream.close()

		# Delete temporary directory (recursively)
		shutil.rmtree(tmpdir)

		self.mergeInterface.progressbar.setValue(self.mergeInterface.progressbar.maximum())
		prints("Merged book and handwriting into PDF: " + outputPdf)
		return True
	# end def doMerge

	def getBookNumPages(self):
		# Open PDF and return number of pages
		pypdf_book = self.getPyPDFBook()
		if (pypdf_book is None):
			return -1
		else:
			num_pages = pypdf_book.getNumPages()
			prints("Number of pages in PDF: " + str(num_pages))

			# Make sure the file is closed
			pypdf_book.stream.close()

			return num_pages
		# end if
	# end def getBookNumPages


	''' Will return an associative array (dict) with the page numbers as keys, and the relative path to the SVG file as a value'''
	def getSVGFiles(self):
		pages = {}
		with closing(sqlite.connect(self.sony_dbpath)) as connection:
			cursor = connection.cursor()

			query = 'SELECT page,svg_file FROM freehand WHERE content_id=' + str(self.bookid) + ' ORDER BY freehand.page;'
			cursor.execute(query)

			for i, row in enumerate(cursor):
				pageno = int(round(row[0]))
				svg_file = row[1]
				pages[pageno] = svg_file
			# end for	
		# end with

		return pages
	# end def extractBookFromDevice

# end class	
