/** 
 * Instrumented by phoeagon 
 * on March 31, 2012
 * this file comes with absolutely no warranty
 * -----------------------------------------------------------
 * edited to allow "javascript:" locations and "file://" protocol to 
 * 		access other than "/mnt/us/"
 * */
/**
 * Returns an object that handles navigation of an iframe element and manages
 * navigation history
 *
 * @module navigator
 * @param {Object} spec Object containing init parameters
 * @param {String} spec.id Component id
 * @param {Function} spec.log Method for logging messages
 * @param {Function} spec.store Method for storing objects
 * @param {Function} spec.retrieve Method for retrieving objects from storage
 * @param {Function} spec.clear Method for clearing objects from storage
 * @param {Element} spec.window Window object to control navigation in
 * @param {Number} spec.maxHistory Max number of history items to store
 * @param {Function} spec.onBadProtocol Triggered when user tries to use an
 *                                      invalid protocol
 * @param {Function} spec.onGo Triggered in navigator's go method
 */
wafjs.navigator = function (spec) {

    //////////////////////////////////////////
    // Private variables and methods
    //////////////////////////////////////////

    /**
     * navigator object to return
     * @private
     */
    var that,

    /**
     * Array of history objects to be saved in persistent storage
     * @private
     */
    storedHist,

    /**
     * Array of history objects that represents the session navigation history
     * @private
     */
    navHist = [],

    /**
     * Current index within the navHist array.  Start at -1 so opening
     * navigation starts at 0
     * @private
     */
    index = -1,

    /**
     * This is a flag set to designate a given navigation started from the back
     * or forward buttons
     * @private
     */
    isHistoryNav = false;

    /**
     * Verify user entered a url with a valid protocol by creating an a element
     * @private
     * @param {Object} url
     * @returns Boolean result
     */
    var checkProtocol = function (url) {
        // initialize result
        var isValid = false,
        // create a element
        a = document.createElement('a');

        // add url
        a.href = url;

        // get and validate protocol
        if (a.protocol === 'http:' ||
            a.protocol === 'https:' ||
             a.href.search(/javascript:/i) === 0 ||
            // or if url begins with file:///mnt/us
            a.href.search(/file:\/\/\/mnt\/us/i) === 0) {

            isValid = true;
        }

        // return result
        return isValid;
    }

    /**
     * Add history item to stored history array
     * @param {Object} hist History item
     * @param {String} hist.name Name of history item
     * @param {String} hist.url URL of history item
     * @param {String} hist.date Date history item was created
     */
    var addToStoredHist = function (hist) {
        // loop through stored history and look for same url
        for (var i=0; i<storedHist.length; i++) {
            if (storedHist[i].url === hist.url) {
                storedHist.splice(i, 1);
                break;
            }
        }

        // add to stored history
        storedHist.push(hist);

        // remove first item if reached max
        if (storedHist.length > spec.maxHistory) {
            storedHist.shift();
        }

        // save history
        spec.store( { name: 'history', value: storedHist } );

        // log
        spec.log({
            event: spec.id,
            msg: 'added to history',
            level: 'info'
        });
    }

    //////////////////////////////////////////
    // Public methods
    //////////////////////////////////////////

    /**
     * Add public methods and properties
     */
    that = {
        /**
         * Initialize navigator object by checking for and retrieving stored
         * history.  Should be called be navigator is used
         * @public
         */
        init: function () {
            // get from storage
            storedHist = spec.retrieve( { name: 'history' } );

            // if no history, then create empty array
            if (!storedHist) {
                storedHist = [];
            }
        },

        /**
         * Navigate window backwards in history
         * @public
         */
        back: function () {
            // get last history
            var hist = (index > 0)? navHist[index-1] : false;

            // log
            spec.log({
                event: spec.id,
                msg: 'back',
                level: 'info'
            });

            // if not at 0
            if (hist) {
                // set flag so won't be added to history
                isHistoryNav = true;

                // navigate to history url
                that.go(hist.url);

                // decrement index
                index--;

                // enable forward button
                spec.toggleForwardState(true);
            }
            else {
                // if kindle object then go to last app
                if (typeof kindle !== 'undefined') {
                    kindle.appmgr.back();
                }
            }
        },

        /**
         * Navigate window forwards in history
         * @public
         */
        forw: function () {
            var hist = (index < navHist.length - 1)? navHist[index+1] : false;

            // if there is a forward
            if (hist) {
                // set flag so won't be added to history
                isHistoryNav = true;

                // navigate to history url
                that.go(hist.url);

                // increment index
                index++;

                // determine if now on last item
                if (index === navHist.length - 1) {
                    // disable forward button
                    spec.toggleForwardState(false);
                }

                // log
                spec.log({
                    event: spec.id,
                    msg: 'forward',
                    level: 'info'
                });
            }
        },

        /**
         * Navigate window to given url
         * @public
         * @param {String} url URL to navigate window to
         */
        go: function (url) {
            if (url !== 'about:blank' && url.indexOf("javascript:") !== 0) {
                // insert default protocol into url if none
                if (url.indexOf("://") < 0) {
                    url = "http://" + url;
                }
            }

            // log go event
            spec.log({
                event: spec.id,
                msg: 'go',
                level: 'info'
            });

            // check if url has valid protocol
            if (url === 'about:blank' || url.indexOf("javascript:")===0 || checkProtocol(url)) {
                // go
                spec.window.location.href = url;

                // trigger ongo event
                spec.onGo(url);
            }
            else {
                // show error
                spec.onBadProtocol();
            }
        },

        /**
         * Refresh content window to current page
         * @public
         */
        refresh: function () {
            spec.window.location.reload(true);

            // log
            spec.log({
                event: spec.id,
                msg: 'refresh',
                level: 'info'
            });
        },

        /**
         * Get the current location's name and url
         * @public
         * @returns an object containing current location's name and url
         */
        getCurrent: function () {
            var name = spec.window.document.title;
            var url = spec.window.location.href;

            return {
                name: name,
                url: url || 'about:blank'
            };
        },

        /**
         * Get the current history item
         * @Public
         * @returns a history object
         */
        getCurrentHistory: function () {
            return navHist[index];
        },

        /**
         * Get the stored history array
         * @public
         * @returns Array of history items
         */
        getHistory: function () {
            // return stored history
            return storedHist;
        },

        /**
         * Clear stored history and save empty array to persistent storage and
         * clear navigation history
         * @public
         */
        clearHistory: function () {
            // clear stored history
            storedHist = [];
            spec.store( {
                name: 'history',
                value: storedHist
            } );

            // clear session history
            navHist = [];

            // reset index
            index = 0;
        },

        /**
         * Signal navigator to add current page to history
         * @public
         * @param {Object} hist History object
         * @param {String} hist.name Name of history, usually page title
         * @param {String} hist.url URL of history item
         */
        addToHistory: function(hist) {
            // history obj
            hist = hist || that.getCurrent();

            // check if this is a history navigation (back or forward)
            if (isHistoryNav) {
                isHistoryNav = false;

                // log
                spec.log({
                    event: spec.id,
                    msg: 'history nav - not added to history',
                    level: 'debug'
                });

                return;
            }
            else {
                // check whether this is a refresh of the current page
                // or a blank page
                if (navHist[index] && navHist[index].url === hist.url
                    || hist.url === 'about:blank') {
                    // if so, dont add to history
                    // log
                    spec.log({
                        event: spec.id,
                        msg: 'url is refresh or blank - not added to history',
                        level: 'debug'
                    });

                    return;
                }

                // get current date and convert to utc string
                hist.date = new Date().getTime();

                // if index isn't in the last position
                if (index < navHist.length - 1) {
                    // slice at index and add history to end
                    navHist = navHist.slice(0, index + 1);
                    navHist.push(hist);
                }
                else {
                    // if max items has been hit
                    if (navHist.length > spec.maxHistory) {
                        // remove first item
                        navHist.shift();
                    }

                    // push onto end
                    navHist.push(hist);
                }

                // add to stored history
                addToStoredHist(hist);

                // increment index
                index++;

                // disable forward button
                spec.toggleForwardState(false);
            }
        },

        /**
         * Cancel current load request
         * @public
         */
        cancel: function () {
            spec.window.stop();
        },

        /**
         * Application has been paused
         * @public
         */
        onAppPause: function () {
            // nothing to do currently
        },

        /**
         * Retrieve the source of the site icon
         * @public
         * @return Site icon source
         */
        getFaviconSrc: function () {
            // default source to favicon.ico
            var src = '/favicon.ico';

            // check for link with shortcut icon
            var linkElement = spec.window.document.querySelector('link[rel~=icon]');
            if (linkElement) {
                src = linkElement.href;
            }

            // get full path using anchor element
            var a = spec.window.document.createElement('a');
            a.href = src;

            return  a.href;
        }
    };

    return that;
};
