import csv

class Sheet(object):
    def __init__(self, name):
        # create empty dict for rows
        self.rows = {}
        self.name = name
        self.default_properties = Properties()
        self.default_properties.add_prop('name', 'default')
        # default width and height of the cells
        self.default_properties.add_prop('width', 100)
        self.default_properties.add_prop('height', 20)
        # default width and height of headers
        self.default_properties.add_prop('header-width', 40)
        self.default_properties.add_prop('header-height', 20)
        self.cycle_id = 0 # used to detect calc cycles

    def _get_cell(self, col_nr, row_nr):
        if row_nr in self.rows:
            #print 'get_cell(%d,%d) = %s' % (col_nr, row_nr, self.rows[row_nr].get_cell(col_nr))
            return self.rows[row_nr].get_cell(col_nr)
        else:
            #print 'get_cell(%d,%d) = None'% (col_nr, row_nr)
            return None

    def get_cell_content(self, col_nr, row_nr):
        cell = self._get_cell(col_nr, row_nr)
        if cell != None:
            # return text as is
            return cell.get_content()
        else:
            return None

    # public interface
    def get_cell_value(self, col_nr, row_nr):
        cell = self._get_cell(col_nr, row_nr)
        if cell != None:
            # create new cycle_id for each call to public get_cel_value
            self.cycle_id += 1
            # return text as is
            return cell.get_value(self, col_nr, row_nr, self.cycle_id)
        else:
            return None

    # internal interface
    def _get_cell_value(self, col_nr, row_nr, cycle_id):
        cell = self._get_cell(col_nr, row_nr)
        if cell != None:
            self.cycle_id += 1
            # return text as is
            return cell.get_value(self, col_nr, row_nr, cycle_id)
        else:
            return None

    def set_cell_content(self, col_nr, row_nr, value):
        if value != None:
            self.add_cell(col_nr, row_nr, value)
        else:
            self.del_cell(col_nr, row_nr)
        
    def add_cell(self, col_nr, row_nr, value):
        # determine type of the value and create correct content and cell
        if (len(value) > 1) and (value[0] == '='):
            cell = FormulaContent(value)
        else:
            # todo other types
            cell = TextContent(value)
        self._add_cell(col_nr, row_nr, cell)

    def _add_cell(self, col_nr, row_nr, cell):
        if not row_nr in self.rows:
            # first create row
            self.rows[row_nr] = RowHeader()
        self.rows[row_nr].add_cell(col_nr, cell)

    def del_cell(self, col_nr, row_nr):
        if row_nr in self.rows:
            self.rows[row_nr].del_cell(col_nr)

    def clear(self):
        '''remove all cells'''
        for row in self.rows.values():
            row.cells.clear()
        self.rows.clear()

    def get_row_properties(self):
        # only default properties allowed
        return self.default_properties
        
    def get_row_heights(self, start_y, h):
        # for now fixed size
        row_height = self.default_properties.find_property('height')
        current_y = start_y
        row_height_list = []
        while current_y < h:
            row_height_list.append(row_height)
            current_y += row_height
        # add one extra
        row_height_list.append(row_height)
        current_y += row_height
        return row_height_list

    def get_col_properties(self):
        # only default properties allowed
        return self.default_properties

    def get_col_widths(self, start_x, w):
        # for now fixed size
        col_width = self.default_properties.find_property('width')

        current_x = start_x
        col_width_list = []
        while current_x < w:
            col_width_list.append(col_width)
            current_x += col_width
        # add one extra
        col_width_list.append(col_width)
        current_x += col_width
        
        return col_width_list

    def load_from_csv(self, filename):
        self.clear()
        f = open(filename, 'rb')
        reader = csv.reader(f)
        row_num = 0
        for row in reader:
            col_num = 0
            for col in row:
                if col == "":
                    #print "[-]",
                    pass
                else:
                    #print "[%d,%d][%s]" % (row_num, col_num,col),
                    self.add_cell(col_num, row_num, col)
                col_num += 1
            #print ""
            row_num += 1

        f.close()

    def save_to_csv(self, filename):
        f = open(filename, 'wb')
        writer = csv.writer(f)
        # rows in sheet
        keys_row = self.rows.keys()
        keys_row.sort()
        # get last value, which is max used row
        max_row = keys_row[len(keys_row)-1]
        # need to handle all row upto max_row
        for row in range(max_row+1):
            if row in self.rows:
                # construct list
                row_lst = []
                cells = self.rows[row].cells
                keys_col = cells.keys()
                keys_col.sort()
                # get last value
                max_col = keys_col[len(keys_col)-1]
                for col in range(max_col+1):
                    if col in cells:
                        row_lst.append(cells[col].get_content())
                    else:
                        row_lst.append('')
            else:
                row_lst = ['']
            print 'row', row, row_lst
            writer.writerow(row_lst)
        f.close()
        
    def invalidate(self):
        """ make all pre-calculated values invalid """
        for row in self.rows.values():
            for col in row.cells.values():
                col.invalidate()
                
        
class RowHeader(object):
    def __init__(self):
        # create empty dict for the cells in this row, the key for the cell
        # will be the col_nr
        self.cells = {}

    def get_cell(self, col_nr):
        if col_nr in self.cells:
            return self.cells[col_nr]
        else:
            return None

    def add_cell(self, col_nr, cell):
        if col_nr in self.cells:
            del self.cells[col_nr]
        self.cells[col_nr] = cell

    def del_cell(self, col_nr):
        del self.cells[col_nr]

class Cell(object):
    def __init__(self, props, content):
        self.props = props
        self.content = content

    def get_content(self):
        return self.content

    def get_props(self):
        return self.props

    def find_property(self, key):
        if self.props != None:
            return self.props.find_property(key)

class Properties(object):
    def __init__(self, base=None, left=-1, top=-1, right=-1, bottom=-1):
        # list is a dictionary with prop-value
        self.list = {}
        self.base = base
        self.left = left
        self.top = top
        self.right = right
        self.bottom = bottom

    def add_prop(self, key, value):
        # todo: check if property is in base and has same value?
        self.list[key] = value

    def find_property(self, key):
        if key in self.list:
            return self.list[key]
        elif self.base != None:
            return self.base.find_property(key)

class Content(object):
    def __init__(self):
        self.type = None

    def get_value(self, sheet=None, col=0, row=0, cycle_id=0):
        pass

    def get_content(self):
        pass

    def invalidate(self):
        pass
        
class TextContent(Content):
    def __init__(self, text):
        self.text = text

    def get_value(self, sheet=None, col=0, row=0, cycle_id=0):
        return self.text

    def get_content(self):
        return self.text

    def invalidate(self):
        pass
        
class ValueContent(Content):
    def __init__(self, value):
        self.value = value
        
    def get_value(self, sheet=None, col=0, row=0, cycle_id=0):
        return self.value

    def get_content(self):
        return self.value

    def invalidate(self):
        pass
        
class FormulaContent(Content):
    def __init__(self, formula):
        self.formula = formula
        self.ok = True # formula valid
        self.value = None
        self.value_valid = False # value valid
        self.cycle_id = 0
        self.error = None
        # tokenise and parse the formula
        ret = self.tokenise()
        if ret < 0:
            self.ok = False
            self.error = '#ERR at %d#' % -ret
            print 'Tokeniser failed on:', self.formula
        else:
            if self.parse() == None:
                self.ok = False
                print 'Parser failed on:', self.formula
        
    def get_value(self, sheet, col, row, cycle_id):
        #evaluate formula
        if self.ok:
            value = self.evaluate(self.tree, 0, sheet, col, row, cycle_id)
            if isinstance(value, float):
                return '%.2f' % value
            return str(value)
        else:
            if self.error != None:
                return self.error
            else:
                return '#ERROR#'

    def get_content(self):
        return self.formula

    def invalidate(self):
        self.value_valid = False
        
    def tokenise(self):
        #print 'Tokenising:', self.formula
        self.tokens = []
        index = 0
        while index < len(self.formula):
            if self.formula[index] == '=':
                #print '='
                token = Token('=', None)
                self.tokens.append(token)
            elif self.formula[index] == '(':
                #print '('
                token = Token('(', None)
                self.tokens.append(token)
            elif self.formula[index] == ')':
                #print ')'
                token = Token(')', None)
                self.tokens.append(token)
            elif self.formula[index] == '+':
                #print '+'
                token = Token('+', None)
                self.tokens.append(token)
            elif self.formula[index] == '-':
                #print '-'
                token = Token('-', None)
                self.tokens.append(token)
            elif self.formula[index] == '*':
                #print '*'
                token = Token('*', None)
                self.tokens.append(token)
            elif self.formula[index] == '/':
                #print '/'
                token = Token('/', None)
                self.tokens.append(token)
            elif self.formula[index] == ' ':
                #print 'space'
                pass # ignore white space
            elif self.formula[index] == '[':
                value = ''
                while ((index+1) < len(self.formula)) and (self.formula[index+1] != ']'):
                    index += 1
                    value += self.formula[index]
                if ((index+1) < len(self.formula)) and (self.formula[index+1] != ']'):
                    # not expected end
                    return 0 - index - 1
                index += 1 # skip ']'
                #print 'ref(%s)' % value
                token = Token('ref', value)
                self.tokens.append(token)
            elif self.formula[index].isdigit():
                value = self.formula[index]
                while ((index+1) < len(self.formula)) and (self.formula[index+1].isdigit()):
                    index += 1
                    value += self.formula[index]
                #print 'number(%s)' % value
                token = Token('num', value)
                self.tokens.append(token)
            elif self.formula[index].isalpha():
                value = self.formula[index]
                while ((index+1) < len(self.formula)) and (self.formula[index+1].isalnum()):
                    index += 1
                    value += self.formula[index]
                #print 'id(%s)' % value
                token = Token('id', value)
                self.tokens.append(token)
            else:
                # not recognised
                print '*************************** NOT RECOGNISED', self.formula[index]
                return 0 - index - 1
            # next
            index += 1
        token = Token('EOF', None)
        self.tokens.append(token)

        #print 'Tokens:', len(self.tokens)
        return index

    def getsym(self):
        self.index += 1
        self.sym = self.tokens[self.index].type
        #print 'getsym[%d]:'%self.index, self.sym
        
    def accept(self, token):
        if self.sym == token:
            self.id = self.tokens[self.index].id
            self.getsym()
            return True
        else:
            return False

    def expect(self, token):
        if self.accept(token):
            return True
        else:
            print "Expect: unexpected symbol, got %s expected %s" % (self.sym, token)
            self.error = '#ERR%s#' % token
            return False

    def factor(self):
        if self.accept('ref'):
            fac = Expr(self.id, None, 'ref')
        elif self.accept('num'):
            fac = Expr(self.id, None, 'num')
        elif self.accept('id'):
            fac = Expr(self.id, None, 'id')
        elif self.accept('('):
            fac = self.expression()
            self.expect(')')
        else:
            print "Factor: Syntax error, got %s" % self.sym
            self.getsym()
            self.error = '#ERR-Fac#'
            fac = None
        return fac

    def term(self):
        fac1 = self.factor()
        if fac1 == None:
            self.error = '#ERRTerm1#'
            return None # error
        while (self.sym == '*') or (self.sym == '/'):
            op = self.sym
            self.getsym()
            fac2 = self.factor()
            if fac2 == None:
                self.error = '#ERRTerm2#'
                return None # error
            fac1 = Expr(fac1, fac2, op)

        return fac1

    def expression(self):
        self.accept('=')
        term1 = self.term()
        if term1 == None:
            self.error = '#ERRExpr1#'
            return None # error
        while (self.sym == '+') or (self.sym == '-'):
            op = self.sym
            self.getsym()
            term2 = self.term()
            if term2 == None:
                self.error = '#ERRExpr2#'
                return None # error
            term1 = Expr(term1, term2, op)

        return term1

    def parse(self):
        # based on: http://en.wikipedia.org/wiki/Recursive_descent_parser
        # reset parser
        self.index = 0
        self.sym = self.tokens[self.index].type
        self.id = None # last accepted id

        self.tree = self.expression()
        return self.tree

    def print_tree(self, term):
        if (term.operator == '+') or (term.operator == '-') or (term.operator == '*') or (term.operator == '/'):
            print '(',
            self.print_tree(term.left)
            print term.operator,
            self.print_tree(term.right)
            print ')',
        elif term.operator == 'ref':
            print '[%s]' % term.left,
        elif term.operator == 'id':
            print '[%s]' % term.left,
        elif term.operator == 'num':
            print term.left,
            
    def evaluate(self, term, value, sheet, col, row, cycle_id):
        if self.value_valid:
            # return pre-calculated value
            return self.value
        else:
            self.value = self._evaluate(term, value, sheet, col, row, cycle_id, True)
            if self.value != None:
                self.value_valid = True
            else:
                if self.error != None:
                    self.value = self.error
                else:
                    self.value = '#ERROR#'
                self.value_valid = False
            return self.value
    
    def _evaluate(self, term, value, sheet, col, row, cycle_id, top):
        #print '_evaluate', col, row, cycle_id
        # todo: check if formula OK
        letters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ'
        digits = '0123456789'
        if (term.operator == '+') or (term.operator == '-') or (term.operator == '*') or (term.operator == '/'):
            val1 = self._evaluate(term.left, value, sheet, col, row, cycle_id, False)
            if val1 == None:
                self.error = '#ERRVal1#'
                return None
            val2 = self._evaluate(term.right, value, sheet, col, row, cycle_id, False)
            if val2 == None:
                self.error = '#ERRVal2#'
                return None
            if term.operator == '+':
                val = val1 + val2
            elif term.operator == '-':
                val = val1 - val2
            elif term.operator == '*':
                val = val1 * val2
            elif term.operator == '/':
                val = val1 / val2
            else:
                print 'ERROR: eval unknown operator', term.operator
                self.error = '#ERROp#'
                val = None
            #print 'eval:', val, '=', val1, term.operator, val2
        elif (term.operator == 'ref') or (term.operator == 'id'):
            i = 0
            cell_col = 0
            cell_row = 0
            if term.left.find(',') == -1:
                # 'A1' (or '[A1]' with [] removed)
                if not term.left[i].isalpha():
                    print 'ERROR:', term.left[i], 'is not a letter'
                    self.error = '#ERRAlpha#'
                    return None # error
                while term.left[i].isalpha():
                    cell_col *= 26
                    cell_col += letters.find(term.left[i].upper())
                    i += 1
                if not term.left[i].isdigit():
                    print 'ERROR:', term.left[i], 'is not a digit'
                    self.error = '#ERRDigit#'
                    return None # error
                while (i<len(term.left)) and (term.left[i].isdigit()):
                    cell_row *= 10
                    cell_row += digits.find(term.left[i])
                    i += 1
                if cell_row > 0:
                    cell_row -= 1 # A1 -> 0,0
            else:
                # relative
                col_offset, row_offset = term.left.split(',')
                cell_col = col + int(col_offset)
                cell_row = row + int(row_offset)

            # protect against cycles
            cycles = False
            if self.cycle_id == cycle_id:
                cycles = True
            self.cycle_id = cycle_id

            # need access to sheet!!
            if (top == True) and (cycles == True):
                print 'ERROR: detected CYCLE in formula:', col, row
                self.error = '#CYCLE#'
                val = None
            elif (cell_col >= 0) and (cell_row >= 0):
                try:
                    val = float(sheet._get_cell_value(cell_col, cell_row, cycle_id))
                except:
                    # cell does not exsist
                    val = None
                    self.error = '#ERRExists#'
                    print 'Cell does not exist:', cell_col, cell_row
            else:
                print 'ERROR: incorrect reference'
                self.error = '#ERRRange#'
                val = None
            #print 'eval-ref:', val

            
        elif term.operator == 'num':
            val = float(term.left)
            #print 'eval-num:', val

        else:
            print 'Unexpected operator:', term.operator
            self.error = '#ERRUnOp#'
            val = None
        
        return val       
        
class Expr(object):
    def __init__(self, left, right, operator):
        #print 'Expr:', operator
        self.operator = operator
        self.left = left
        self.right = right

class Token(object):
    def __init__(self, type, id):
        self.type = type
        self.id = id

#------------------- Unit test --------------------
   
if __name__ == "__main__":
    # We create a dictionary of rows for our test
    rows = {}
    rows[0] = RowHeader()
    rows[1] = RowHeader()
    rows[2] = RowHeader()

    # First we need some properties
    base_row_props = Properties(None)
    base_row_props.add_prop('name', 'row')
    base_row_props.add_prop('width', 100)

    base_col_props = Properties(base_row_props)
    base_col_props.add_prop('name', 'column')
    base_col_props.add_prop('height', 20)

    base_props = Properties(base_col_props)
    base_props.add_prop('name', 'default')
    base_props.add_prop('font_size', 10)
    base_props.add_prop('fg_color', 0xffffff)
    base_props.add_prop('bg_color', 0x000000)

    props = Properties(base_props)
    props.add_prop('name', 'cell')
    props.add_prop('fg_color', 0x808080)

    # Now create some cells and add them to row
    sheet = Sheet('Test Sheet')
    #cell1 = Cell(base_props, TextContent('Test1'))
    #sheet._add_cell(0, 1, cell1) # A2
    #cell2 = Cell(base_props, FormulaContent('=5+3'))
    #sheet._add_cell(1, 0, cell1) # B1
    #cell3 = Cell(props, TextContent('Test3'))
    #sheet._add_cell(2, 2, cell1) # C3
    #cell4 = Cell(props, ValueContent(3.14))
    #sheet._add_cell(0, 0, cell1) # A1

    def print_cell(cell, name):
        print 'Name: %s' % name
        width = cell.find_property('width')
        height = cell.find_property('height')
        col = cell.find_property('fg_color')
        print 'Width:%d Height:%d Color:%x' %(width, height, col)
        print 'Value: ',
        print cell.get_content().get_value(sheet, 4, 5)
        
    #print_cell(cell1, 'cell1')
    #print_cell(cell2, 'cell2')
    #print_cell(cell3, 'cell3')
    #print_cell(cell4, 'cell4')

    def print_formula(f):
        print f.tokenise()
        tree = f.parse()
        print '------------------- Tree -----------'
        print f.get_content()
        f.print_tree(tree)
        print f.get_value( sheet, 5, 4)
        print ''
        print '------------------------------------'
    
    f = FormulaContent('=1+5') # simple numbers
    print_formula(f)
    
    f = FormulaContent('=[a1] + 4') # cell + space
    print_formula(f)
    f = FormulaContent('=[a1]+4') # cell no space
    print_formula(f)
    f = FormulaContent('=[-2,-1]+4') # rel cell no space
    print_formula(f)
    f = FormulaContent('=([-2,-1]+4)/3') # rel cell no space
    print_formula(f)
    f = FormulaContent('=[-2,-1]+4/3') # rel cell no space
    print_formula(f)
    
    
