# -*- coding: utf-8 -*-

from __future__ import (division, absolute_import, print_function)

__license__ = 'GPL 3'
__copyright__ = '2012, Saulius P. <a@a.com>'
__docformat__ = 'restructuredtext en'

import os, time
import cStringIO
import struct
from threading import Thread
from Queue import Queue

from PyQt4.Qt import Qt, QMenu, QFileDialog, QIcon, QPixmap

from calibre import sanitize_file_name
from calibre.gui2 import Dispatcher, warning_dialog
from calibre.gui2.actions import InterfaceAction
from calibre.library.save_to_disk import get_components
from calibre.library.save_to_disk import config
from calibre.ptempfile import PersistentTemporaryFile
from calibre.utils.filenames import shorten_components_to
from calibre.utils.ipc.job import BaseJob
from calibre.ebooks.mobi.reader import MobiReader
from calibre.ebooks.pdb.header import PdbHeaderReader
from calibre.utils.logging import default_log

class APNXAction(InterfaceAction):

    name = 'APNX'
    action_spec = (_('APNX'), None, None, None)
    
    def genesis(self):
        self.apnx_mixin = APNXMixin(self.gui)
        # Read the icons and assign to our global for potential sharing with the configuration dialog
        # Assign our menu to this action and an icon
        self.qaction.setIcon(get_icons('images/plugin_apnx_apnx.png'))
        self.qaction.triggered.connect(self.generate_selected)
        self.apnx_menu = QMenu()
        self.load_menu()
        
    def load_menu(self):
        self.apnx_menu.clear()
        self.apnx_menu.addAction(_('Generate from selected books...'), self.generate_selected)
        self.apnx_menu.addAction(_('Generate from file...'), self.generate_file)
        self.qaction.setMenu(self.apnx_menu)

    def generate_selected(self):
        self.apnx_mixin.genesis()
        
        apnxdir = unicode(QFileDialog.getExistingDirectory(self.gui, _('Directory to save APNX file'), self.gui.library_path, QFileDialog.ShowDirsOnly | QFileDialog.DontResolveSymlinks))
        if not apnxdir:
            return
        
        self._generate_selected(apnxdir)
        
    def _generate_selected(self, apnxdir, ids=None, do_auto_convert=False):
        if not ids:
            ids = [self.gui.library_view.model().id(r) for r in self.gui.library_view.selectionModel().selectedRows()]
        
        _files, _auto_ids = self.gui.library_view.model().get_preferred_formats_from_ids(ids, ['mobi', 'azw', 'prc'], exclude_auto=do_auto_convert)
        if do_auto_convert:
            ok_ids = list(set(ids).difference(_auto_ids))
            ids = [i for i in ids if i in ok_ids]
        else:
            _auto_ids = []
            
        metadata = self.gui.library_view.model().metadata_for(ids)
        ids = iter(ids)
        imetadata = iter(metadata)

        bad, good = [], []
        for f in _files:
            mi = imetadata.next()
            id = ids.next()
            if f is None:
                bad.append(mi.title)
            else:
                good.append((f, mi))

        template = config().parse().template
        if not isinstance(template, unicode):
            template = template.decode('utf-8')

        for f, mi in good:
            components = get_components(template, mi, f)
            if not components:
                components = [sanitize_file_name(mi.title)]

            def remove_trailing_periods(x):
                ans = x
                while ans.endswith('.'):
                    ans = ans[:-1].strip()
                if not ans:
                    ans = 'x'
                return ans
            
            components = list(map(remove_trailing_periods, components))
            components = shorten_components_to(250, components)
            components = list(map(sanitize_file_name, components))
            filepath = os.path.join(apnxdir, *components)

            apnxname = os.path.splitext(filepath)[0] + '.apnx'
            self.apnx_mixin.generate_apnx(f, apnxname)

        if bad:
            bad = '\n'.join('%s'%(i,) for i in bad)
            d = warning_dialog(self.gui, _('No suitable formats'),
                    _('Could not generate an APNX for the following books, '
                'as no suitable formats were found. Convert the book(s) to '
                'MOBI first.'
                ), bad)
            d.exec_()
    
    def generate_file(self):
        self.apnx_mixin.genesis()
        
        filename = unicode(QFileDialog.getOpenFileName(self.gui, _('MOBI file for generating APNX'), self.gui.library_path, 'MOBI files (*.mobi *.azw *.prc)'))
        if not filename:
            return
        apnxdir = unicode(QFileDialog.getExistingDirectory(self.gui, _('Directory to save APNX file'), self.gui.library_path, QFileDialog.ShowDirsOnly | QFileDialog.DontResolveSymlinks))
        if not apnxdir:
            return
        apnxname = os.path.join(apnxdir, os.path.splitext(os.path.basename(filename))[0] + '.apnx')
        
        self.apnx_mixin.generate_apnx(filename, apnxname)


class APNXJob(BaseJob):
    
    def __init__(self, callback, description, job_manager, filename, apnxname):
        BaseJob.__init__(self, description)
        self.exception = None
        self.job_manager = job_manager
        self.args = (filename, apnxname)
        self.callback = callback
        self.log_path = None
        self._log_file = cStringIO.StringIO()
        self._log_file.write(self.description.encode('utf-8') + '\n')

    @property
    def log_file(self):
        if self.log_path is not None:
            return open(self.log_path, 'rb')
        return cStringIO.StringIO(self._log_file.getvalue())

    def start_work(self):
        self.start_time = time.time()
        self.job_manager.changed_queue.put(self)

    def job_done(self):
        self.duration = time.time() - self.start_time
        self.percent = 1
        # Dump log onto disk
        lf = PersistentTemporaryFile('apnx_generate_log')
        lf.write(self._log_file.getvalue())
        lf.close()
        self.log_path = lf.name
        self._log_file.close()
        self._log_file = None

        self.job_manager.changed_queue.put(self)

    def log_write(self, what):
        self._log_file.write(what)
        
        
class APNXGenerator(Thread):
    
    def __init__(self, job_manager):
        Thread.__init__(self)
        self.daemon = True
        self.jobs = Queue()
        self.job_manager = job_manager
        self._run = True
        self.apnx_builder = APNXBuilder()
        
    def stop(self):
        self._run = False
        self.jobs.put(None)
        
    def run(self):
        while self._run:
            try:
                job = self.jobs.get()
            except:
                break
            if job is None or not self._run:
                break
            
            failed, exc = False, None
            job.start_work()
            if job.kill_on_start:
                self._abort_job(job)
                continue
            
            try:
                self._generate_apnx(job)
            except Exception as e:
                if not self._run:
                    return
                import traceback
                failed = True
                exc = e
                job.log_write('\nAPNX generation failed...\n')
                job.log_write(traceback.format_exc())

            if not self._run:
                break

            job.failed = failed
            job.exception = exc
            job.job_done()
            try:
                job.callback(job)
            except:
                import traceback
                traceback.print_exc()

    def _abort_job(self, job):
        job.log_write('Aborted\n')
        job.failed = False
        job.killed = True
        job.job_done()

    def _generate_apnx(self, job):
        filename, apnxname = job.args
        if not filename or not apnxname:
            raise Exception(_('Nothing to do.'))
        dirs = os.path.dirname(apnxname)
        if not os.path.exists(dirs):
            os.makedirs(dirs)
        self.apnx_builder.write_apnx(filename, apnxname)
        
    def generate_apnx(self, callback, filename, apnxname):
        description = _('Generating APNX for %s') % os.path.splitext(os.path.basename(apnxname))[0]
        job = APNXJob(callback, description, self.job_manager, filename, apnxname)
        self.job_manager.add_job(job)
        self.jobs.put(job)


class APNXMixin(object):

    def __init__(self, gui):
        self.gui = gui
    
    def genesis(self):
        '''
        Genesis must always be called before using an APNXMixin object.
        Plugins are initalized before the GUI initalizes the job_manager.
        We cannot create the APNXGenerator during __init__. Instead call
        genesis before using generate_apnx to ensure the APNXGenerator
        has been properly created with the job_manager.
        '''
        if not hasattr(self.gui, 'apnx_generator'):
            self.gui.apnx_generator = APNXGenerator(self.gui.job_manager)

    def generate_apnx(self, filename, apnxname):
        if not self.gui.apnx_generator.is_alive():
            self.gui.apnx_generator.start()
        self.gui.apnx_generator.generate_apnx(Dispatcher(self.apnx_generated), filename, apnxname)
        self.gui.status_bar.show_message(_('Generating APNX for %s') % os.path.splitext(os.path.basename(apnxname))[0], 3000)
    
    def apnx_generated(self, job):
        if job.failed:
            self.gui.job_exception(job, dialog_title=_('Failed to generate APNX'))
            return
        self.gui.status_bar.show_message(job.description + ' ' + _('finished'), 5000)

class APNXBuilder(object):
    '''
    Create an APNX file using a pseudo page mapping.
    '''

    def write_apnx(self, mobi_file_path, apnx_path, accurate=False):
        # Check that this is really a MOBI file.
        with open(mobi_file_path, 'rb') as mf:
            ident = PdbHeaderReader(mf).identity()
        if ident != 'BOOKMOBI':
            raise Exception(_('Not a valid MOBI file. Reports identity of %s') % ident)

        # Get the pages depending on the chosen parser
        pages = []
        if accurate:
            try:
                pages = self.get_pages_accurate(mobi_file_path)
            except:
                # Fall back to the fast parser if we can't
                # use the accurate one. Typically this is
                # due to the file having DRM.
                pages = self.get_pages_exact(mobi_file_path, 349)
        else:
            pages = self.get_pages_exact(mobi_file_path, 349)

        if not pages:
            raise Exception(_('Could not generate page mapping.'))

        # Generate the APNX file from the page mapping.
        apnx = self.generate_apnx(pages)

        # Write the APNX.
        with open(apnx_path, 'wb') as apnxf:
            apnxf.write(apnx)

    def generate_apnx(self, pages):
        import uuid
        apnx = ''

        content_vals = {
            'guid': str(uuid.uuid4()).replace('-', '')[:8],
            'isbn': '',
        }

        content_header = '{"contentGuid":"%(guid)s","asin":"%(isbn)s","cdeType":"EBOK","fileRevisionId":"1"}' % content_vals
        page_header = '{"asin":"%(isbn)s","pageMap":"(1,a,1)"}' % content_vals

        apnx += struct.pack('>I', 65537)
        apnx += struct.pack('>I', 12 + len(content_header))
        apnx += struct.pack('>I', len(content_header))
        apnx += content_header
        apnx += struct.pack('>H', 1)
        apnx += struct.pack('>H', len(page_header))
        apnx += struct.pack('>H', len(pages))
        apnx += struct.pack('>H', 32)
        apnx += page_header

        # Write page values to APNX.
        for page in pages:
            print("Write page: ", page)
            apnx += struct.pack('>I', page)

        return apnx

    def get_pages_exact(self, mobi_file_path, pages_cnt):
        text_length = 0
        pages = []
        count = 0
        print("get_pages_exact: ", pages_cnt)

        with open(mobi_file_path, 'rb') as mf:
            phead = PdbHeaderReader(mf)
            r0 = phead.section_data(0)
            print("header-section-data: ", r0[4:8])
            text_length = struct.unpack('>I', r0[4:8])[0]
            print("text length: ", text_length)

        while count < text_length:
            pages.append(round(count))
            count += text_length/pages_cnt

        return pages

    def get_pages_fast(self, mobi_file_path):
        '''
        2300 characters of uncompressed text per page. This is
        not meant to map 1 to 1 to a print book but to be a
        close enough measure.

        A test book was chosen and the characters were counted
        on one page. This number was round to 2240 then 60
        characters of markup were added to the total giving
        2300.

        Uncompressed text length is used because it's easily
        accessible in MOBI files (part of the header). Also,
        It's faster to work off of the length then to
        decompress and parse the actual text.
        '''
        text_length = 0
        pages = []
        count = 0

        with open(mobi_file_path, 'rb') as mf:
            phead = PdbHeaderReader(mf)
            r0 = phead.section_data(0)
            text_length = struct.unpack('>I', r0[4:8])[0]

        while count < text_length:
            pages.append(count)
            count += 2300

        return pages

    def get_pages_accurate(self, mobi_file_path):
        '''
        A more accurate but much more resource intensive and slower
        method to calculate the page length.

        Parses the uncompressed text. In an average paper back book
        There are 32 lines per page and a maximum of 70 characters
        per line.

        Each paragraph starts a new line and every 70 characters
        (minus markup) in a paragraph starts a new line. The
        position after every 30 lines will be marked as a new
        page.

        This can be make more accurate by accounting for
        <div class="mbp_pagebreak" /> as a new page marker.
        And <br> elements as an empty line.
        '''
        print("Get pages accurate: ", mobi_file_path)
        pages = []

        # Get the MOBI html.
        mr = MobiReader(mobi_file_path, default_log)
        if mr.book_header.encryption_type != 0:
            # DRMed book
            return self.get_pages_fast(mobi_file_path)
        mr.extract_text()

        # States
        in_tag = False
        in_p = False
        check_p = False
        closing = False
        p_char_count = 0

        # Get positions of every line
        # A line is either a paragraph starting
        # or every 70 characters in a paragraph.
        lines = []
        pos = -1
        # We want this to be as fast as possible so we
        # are going to do one pass across the text. re
        # and string functions will parse the text each
        # time they are called.
        #
        # We can can use .lower() here because we are
        # not modifying the text. In this case the case
        # doesn't matter just the absolute character and
        # the position within the stream.
        for c in mr.mobi_html.lower():
            pos += 1

            # Check if we are starting or stopping a p tag.
            if check_p:
                if c == '/':
                    closing = True
                    continue
                elif c == 'p':
                    if closing:
                        in_p = False
                    else:
                        in_p = True
                        lines.append(pos - 2)
                check_p = False
                closing = False
                continue

            if c == '<':
                in_tag = True
                check_p = True
                continue
            elif c == '>':
                in_tag = False
                check_p = False
                continue

            if in_p and not in_tag:
                p_char_count += 1
                if p_char_count == 70:
                    lines.append(pos)
                    p_char_count = 0

        # Every 30 lines is a new page
        for i in xrange(0, len(lines), 32):
            pages.append(lines[i])

        return pages
