#!/bin/sh
## See http://www.mplayerhq.hu/DOCS/tech/slave.txt for docs

## /mnt/us is the root directory when mounting the Kindle via USB
INSTALLDIR=/mnt/us/mplayer
#INSTALLDIR=/mnt/data/mplayer-pokus
LOGDIR="$INSTALLDIR/audio-logs"
AUDIODIR=/mnt/us/audible
#AUDIODIR=/mnt/data/hovorene_slovo
PLAYLIST="$LOGDIR/playlist"

## Value between -20 and 19, decrease in case of lags
NICENESS="-10"

FIFO=/tmp/mplayer.fifo
LOG="$LOGDIR/mplayer.log"
HISTORY="$LOGDIR/history"
MPLAYER="nice -n$NICENESS $INSTALLDIR/mplayer -ao alsa -slave -quiet -input file=$FIFO"
SHUF="$INSTALLDIR/shuf"

BOOK=
BOOKMARK=
filename=
time=0

if [ ! -e $FIFO ]; then
  mkfifo $FIFO
fi
if [ ! -e $LOGDIR ]; then
  mkdir $LOGDIR
fi
if [ ! -e $HISTORY ]; then
  touch $HISTORY
fi

listmusic() {
    ## We can't allow non-valid file in the playlist because it would make prev behave weirdly
    find "$1" -maxdepth 1 -type f  -regex '.*\.\(3gp\|aac\|flac\|ogg\|m3u\|m4a\|mp3\|pls\|wav\|wma\)' | sort
}

cmd() {
    if [ "x$(pidof mplayer)" = "x" ]; then
        return 1;
    fi
    echo "$@" > $FIFO
    return 0;
}

loadplaylist() {
    sed 1d $PLAYLIST > ${PLAYLIST}_temp
    first=`head -1 $PLAYLIST`
    $MPLAYER "$first" -ss $time -playlist ${PLAYLIST}_temp >$LOG 2>&1 &
}

makeplaylist(){
    if [ "a`cat $PLAYLIST | head -1 | grep "$filename"`" == "a" ]; then                                     
      cat $PLAYLIST | grep -A1000 "$filename" > ${PLAYLIST}_temp                                            
      cat $PLAYLIST | grep -B1000 "$filename" | grep -v "$filename" >> ${PLAYLIST}_temp                     
      mv ${PLAYLIST}_temp $PLAYLIST                                                                         
    fi
}

playfrom() {
    makeplaylist
    loadplaylist
}

loadplay() {
    if [ -e "$BOOKMARK" ]; then
        loadbookmark
        echo "playing from bookmark $BOOKMARK: file $filename, time $time"
        playfrom
    else
        echo 'playing from scratch'
        time=0
        loadplaylist
    fi
}

stopplayback() {
    if [ "x$(pidof mplayer)" != "x" ]; then
      savebookmark
      makeplaylist
      killall mplayer
      usleep 100000
    fi
}

book2bookmark() {
    BOOKMARK="$LOGDIR/$BOOK"
    if [ "a$BOOK" = "a" ]; then
      BOOKMARK="$LOGDIR/main"
    fi
}

bookmark2book() {
    BOOK=`basename "$BOOKMARK"`
    if [ "a$BOOK" = "amain" ]; then
      BOOK=
    fi
}

checkbook() {
    if [ ! -e "$AUDIODIR/$BOOK" -o "a$BOOK" == "a" ]; then
      findfirstbook
    fi
    addbookmarktohistory
}

findfirstbook() {
    BOOK=`find $AUDIODIR -type d | sort | head -2 | tail -1`
    BOOK=`basename "$BOOK"`
    book2bookmark
    echo "Found first book: $BOOK"
}

findnextprevbook() {
    if [ $1 = 'prev' ]; then
      AB='B'
    else
      AB='A'
    fi
    # find the next/previous book
    if [ "a$BOOK" != "a" ]; then
      BOOK=`find $AUDIODIR -type d | sort | grep -${AB}1 "$BOOK" | grep -v "$BOOK"`
      BOOK=`basename "$BOOK"`
    fi
    book2bookmark
    checkbook
    listmusic "$AUDIODIR/$BOOK" > $PLAYLIST
    echo "Found $1 book: $BOOK"
}

findprevbookinhistory() {
    BOOKMARK=`tail -1 $HISTORY`
    bookmark2book
    checkbook
    if [ "a$1" != 'akeepplaylist' ]; then
      listmusic "$AUDIODIR/$BOOK" > $PLAYLIST
    fi
    echo "Found previous book in history: $BOOK with bookmark $BOOKMARK"
}


addbookmarktohistory() {
    if [ "a`tail -1 $HISTORY`" != "a$BOOKMARK" ]; then
      echo "$BOOKMARK" >> $HISTORY
    fi
}

# save bookmark and add it to history
savebookmark() {
    cmd "pause"
    cmd "get_time_pos"
    usleep 100000
    time=`tail -1 $LOG | grep -oe "[0-9]\+"`
    time=`echo $time | awk '{print $1;}'`
    cmd "get_file_name"
    usleep 100000
    filename=`tail -1 $LOG | grep -oe "'.\+'"`
    filename=`echo "$filename" | sed "s/'//g"`
    if [ "$BOOKMARK" ]; then
      addbookmarktohistory
    else
      findprevbookinhistory 'keepplaylist'
    fi
    echo $time > "$BOOKMARK"
    echo "$filename" >> "$BOOKMARK"
    echo "bookmarked as $BOOKMARK: time $time, file $filename"
}

loadbookmark() {
    time=`head -1 "$BOOKMARK"`
    filename=`head -2 "$BOOKMARK" | tail -1`
    if [ "a$filename" = "a" -o "a$filename" = "a$BOOK" ]; then
      listmusic "$AUDIODIR/$BOOK" > $PLAYLIST
      filename=`head -1 $PLAYLIST`
      filename=`basename "$filename"`
    fi
    filename="$BOOK/$filename"
    if [ "a$time" = "a" ]; then
      time=0
    fi
}

seek() {
    stopplayback
    time=$(($time+$1))
    if [ $time -lt 0 ]; then
      time=0
    fi
    playfrom
}

# just to make sure that mplayer is certainly killed and not eating the battery
killthemall() {
    if [ "x$(pidof mplayer)" != "x" ]; then
      killall -9 mplayer
    fi
}



case "$1" in
    playtoggle)
        if [ "x$(pidof mplayer)" != "x" ]; then
          stopplayback
          killthemall
        else
          findprevbookinhistory
          loadplay
        fi
        ;;
    playbook)
        findprevbookinhistory
        stopplayback
        loadplay
        ;;
    playlist)
        loadplaylist
        ;;
    pause)
        cmd "pause"
        ;;
    stop)
        stopplayback
        killthemall
        ;;
    prev)
        stopplayback
        time=0
        filename=`cat $PLAYLIST | tail -1`
        playfrom
        ;;
    next)
        stopplayback
        time=0
        filename=`cat $PLAYLIST | head -2 | tail -1`
        playfrom
        ;;
    prevbook)
        stopplayback
        findnextprevbook 'prev'
        loadplay
        ;;
    nextbook)
        stopplayback
        findnextprevbook 'next'
        loadplay
        ;;
    back)
        seek -11
        ;;
    rewind)
        seek -100
        ;;
    forward)
        seek 10
        ;;
    fastforward)
        seek 100
        ;;
    beginning)
        seek -100000000
        ;;
    beginningbook)
        stopplayback
        findprevbookinhistory
        time=0
        filename=
        playfrom
        ;;
    *)
        echo "Usage: $0 {playtoggle|playbook|playlist|pause|stop|prev|next|back|forward|rewind|fastforward|beginning|beginningbook|prevbook|nextbook}"
        exit 1
        ;;
esac

exit 0
