import os
import re
import zipfile
import tempfile
import shutil
from calibre.customize import FileTypePlugin

class MarkdownNotes(FileTypePlugin):

    name                = 'Markdown Notes Plugin'
    description         = 'Footnotes and endnotes for markdown formatted text documents.'
    supported_platforms = ['windows', 'osx', 'linux']
    author              = 'draganHR'
    version             = (1, 0, 1)
    file_types          = set(['txt', 'txtz'])
    on_preprocess = True
    minimum_calibre_version = (0, 8, 32)

    def run(self, path_to_ebook):

        # Check if file iz txt or txtz
        if zipfile.is_zipfile(path_to_ebook):
            self.is_zip = True
        elif not is_binary(path_to_ebook):
            self.is_zip = False
        else:
            print "MarkdownNotes: file format not recognized!"
        
        print "MarkdownNotes: processing %s..." % path_to_ebook
        
        try:
            text = self.read(path_to_ebook)
            text = self.do_notes(text)
            path_to_ebook = self.write(path_to_ebook, text)
        except MarkdownNotesError, e:
            print "MarkdownNotes: Error - %s" % (e,)
        except Exception, e:
            import traceback
            print "MarkdownNotes: Exception - %s" % (e,)
            traceback.print_exc()
            return path_to_ebook

        print "MarkdownNotes: done!"
        
        return path_to_ebook
    
    def read(self, path_to_ebook):
        
        if self.is_zip:
            f = zipfile.ZipFile(path_to_ebook, "r")
            text = f.read("index.txt")
            f.close()
        else:
            text = open(path_to_ebook, 'r').read()
            
        text = text.replace('\r\n', '\n')
        text = text.replace('\r', '\n')
        return text

    def write(self, path_to_ebook, text):
        
        if self.is_zip:
            remove_from_zip(path_to_ebook, "index.txt")
            f = zipfile.ZipFile(path_to_ebook, "a")
            f.writestr("index.txt", text, zipfile.ZIP_DEFLATED)
            f.close()
        else:
            #path_to_ebook = "%s.markdown_notes.txt" % path_to_ebook
            file = open(path_to_ebook, 'wb')
            file.write(text)
            file.close()
            
        return path_to_ebook
    
    def do_notes(self, text):
        """
        Syntax for endnotes:
            _† Lorem ipsum_
        Syntax for inline footnotes:
            >_† Lorem ipsum_
        """
        FOOTNOTE = 'f'
        ENDNOTE = 'e'
        
        notes = {}
        marks = []

        try:
            text.decode('utf-8')
        except UnicodeDecodeError:
            dagger = "\x86"
        else:
            dagger = "\xe2\x80\xa0"

        lines = text.split("\n")
        
        # Collect note and mark positions
        for i, line in enumerate(lines):
            if not line: continue
            
            np = re.match(r' ?\s*(>?)\s*[_\*]\s*'+dagger+'\s*(.+)\s*[_\*]\s*', line)
            if np:
                line = '%s_%s %s_' %(np.group(1), dagger, np.group(2))
                if line.startswith(">"): notes[i] = FOOTNOTE
                else: notes[i] = ENDNOTE
                lines[i] = line
            else:
                for _ in range(line.count(dagger)):
                    marks.append(i)
        
        if len(notes) != len(marks):
            raise MarkdownNotesError(
                "warning: number of notes does not match number of note marks!"+
                "(found: %s notes, %s note marks)" % (len(notes), len(marks))
            )
        
        #notes_index contains list of line numbers containing notes
        notes_index = sorted(notes.keys())
        
        i = len(marks)-1
        FC = fc = len([k for k, v in notes.items() if v == FOOTNOTE])
        EC = ec = len([k for k, v in notes.items() if v == ENDNOTE])

        # Process marks
        for j in sorted(list(set(marks)), reverse=True): #reversed({}.fromkeys(marks).keys()):
            line = lines[j]
            matches = re.finditer('('+dagger+')', line)
            for m in reversed(list(matches)):

                note_index = notes_index[i]
                if notes[note_index]==FOOTNOTE:
                    replacement = '<span class="footnote-mark"><a href="#fn%(c)s" name="fm%(c)s">%(dagger)s</a></span>' % {'c': fc, 'dagger': dagger}
                    fc -= 1
                else:
                    replacement = '<sup class="endnote-mark"><a href="#en%(c)s" name="em%(c)s">%(c)s</a></sup>' % {'c': ec}
                    ec -= 1
                    
                line = line[0:m.start()] + replacement + line[m.end():]

                i -= 1
            
            lines[j] = line
            
        # Check count
        if i != -1: raise MarkdownNotesError("Invalid `i` value after mark iterration")
        
        i = len(marks)-1
        fc = FC
        ec = EC

        endnotes = []
        delete_token = """<!--{delete}-->"""
        
        # Process notes
        for j in sorted(notes.keys(), reverse=True):
            line = lines[j]
            line = re.match(r'>?_'+dagger+' (.+)_', line).group(1)
            
            note_index = notes_index[i]
            if notes[note_index]==FOOTNOTE:
                line = '<p class="footnote"><a href="#fm%(c)s" name="fn%(c)s">%(dagger)s</a> %(line)s</p>' % {'line': line, 'c': fc, 'dagger': dagger}
                fc -= 1
            else:
                endnotes.insert(0, line)
                line = delete_token
                ec -= 1
                
            lines[j] = line
            i -= 1
        
        # Check count
        if i != -1: raise MarkdownNotesError("Invalid `i` value after note iterration")

        # Append endnotes
        if endnotes:

            if not re.search(r"^\s*$", lines[-1]): lines.append("")
            lines.append("## Notes")
            lines.append("")
            
            for ec, endnote in enumerate(endnotes, start=1):
                lines.append('<p class="endnote"><a href="#em%(c)s" name="en%(c)s">&#91;%(c)s&#93;</a> %(endnote)s</p>' % {'endnote': endnote,'c': ec})
        
        # Postprocessing
        text = "\n".join(lines)
        
        n = 1
        while n: text, n = re.subn(re.compile("\n+%s\n+" % delete_token), "\n\n", text, 1)
              
        print "MarkdownNotes: found %s footnotes and %s endnotes" % (FC, EC)
        
        return text


class MarkdownNotesError(Exception):
    pass


def remove_from_zip(zipfname, *filenames):
    tempdir = tempfile.mkdtemp()
    try:
        tempname = os.path.join(tempdir, 'new.zip')
        with zipfile.ZipFile(zipfname, 'r') as zipread:
            with zipfile.ZipFile(tempname, 'w') as zipwrite:
                for item in zipread.infolist():
                    if item.filename not in filenames:
                        data = zipread.read(item.filename)
                        zipwrite.writestr(item, data)
        shutil.move(tempname, zipfname)
    finally:
        shutil.rmtree(tempdir)


def is_binary(filename):
    """Return true if the given filename is binary.
    @raise EnvironmentError: if the file does not exist or cannot be accessed.
    @attention: found @ http://bytes.com/topic/python/answers/21222-determine-file-type-binary-text on 6/08/2010
    @author: Trent Mick <TrentM@ActiveState.com>
    @author: Jorge Orpinel <jorge@orpinel.com>"""
    f = open(filename, 'rb')
    try:
        CHUNKSIZE = 1024
        while 1:
            chunk = f.read(CHUNKSIZE)
            if '\0' in chunk: # found null byte
                return True
            if len(chunk) < CHUNKSIZE:
                break # done

    finally:
        f.close()

    return False
