#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2012, Dennis Cranston <dennis_cranston@yahoo.com>'
__docformat__ = 'restructuredtext en'

import os, traceback

from calibre.utils.ipc.server import Server
from calibre.utils.ipc.job import ParallelJob

from calibre_plugins.webos_kindle_import.statistics import estimate_locations, generate_asin

def do_get_info(books_to_scan, cpus, notification=lambda x,y:x):
    '''
    Master job, to launch child jobs to estimate locations and generate ASIN 
    for this list of books
    '''
    server = Server(pool_size=cpus)

    # Queue all the jobs
    for book_id, title, format, book_path in books_to_scan:
        args = ['calibre_plugins.webos_kindle_import.jobs', 'do_get_info_for_book',
                (format, book_path)]
        job = ParallelJob('arbitrary', str(book_id), done=None, args=args)
        job._book_id = book_id
        job._title = title
        job._format = format
        server.add_job(job)

    # This server is an arbitrary_n job, so there is a notifier available.
    # Set the % complete to a small number to avoid the 'unavailable' indicator
    notification(0.01, 'Getting Kindle-Import Info...')

    # Dequeue the job results as they arrive, saving the results
    total = len(books_to_scan)
    count = 0
    book_locations_map = dict()
    book_asin_map = dict()
    while True:
        job = server.changed_jobs_queue.get()
        # A job can 'change' when it is not finished, for example if it
        # produces a notification. Ignore these.
        job.update()
        if not job.is_finished:
            continue
        # A job really finished. Get the information.
        locations, asin = job.result
        book_id = job._book_id
        count = count + 1
        notification(float(count)/total, 'Getting WebOS Kindle-Import Info')
        # Add this job's output to the current log
        print('-------------------------------')
        print('Logfile for book ID %d (%s - %s)'%(book_id, job._title, job._format))
        if locations:
            book_locations_map[book_id] = locations
            print('\tEstimated Locations: %d'%locations)
        else:
            print('\tFAILED TO ESTIMATE LOCATIONS')
        if asin:
            book_asin_map[book_id] = asin
            print('\tGenerated ASIN: %s'%asin)
        else:
            print('\tFAILED TO GENERATE ASIN')
        print(job.details)
        if count >= total:
            # All done!
            break

    server.close()
    # return the map as the job result
    return book_locations_map, book_asin_map


def do_get_info_for_book(format, book_path):
    '''
    Child job, to get statistics in this specific book
    '''
    try:
        locations = estimate_locations(format, book_path)
        asin = generate_asin(format, book_path)
        if os.path.exists(book_path):
            os.remove(book_path)
        return locations, asin
    except:
        traceback.print_exc()
        return None, None

