#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2012, Dennis Cranston <dennis_cranston@yahoo.com>'
__docformat__ = 'restructuredtext en'

from PyQt4.Qt import (QWidget, QGridLayout, QLabel, QLineEdit, QPushButton,
                      QGroupBox, QComboBox, QRadioButton, QVBoxLayout)
from calibre.utils.config import JSONConfig

from calibre_plugins.webos_kindle_import.common_utils import get_library_uuid, CustomColumnComboBox
from calibre_plugins.webos_kindle_import.common_utils import KeyboardConfigDialog

STORE_COLUMNS = 'columns'
KEY_LOCATIONS = 'Locations'
KEY_ASIN = 'ASIN'

DEFAULT_COLUMNS = { KEY_LOCATIONS: '',
                    KEY_ASIN: '' }

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/WebOS Kindle-Import')

# Set defaults
plugin_prefs.defaults[STORE_COLUMNS] = {}

def get_custom_columns_for_library(db):
    library_id = get_library_uuid(db)
    libraries = plugin_prefs[STORE_COLUMNS]
    library = libraries.get(library_id, DEFAULT_COLUMNS)
    locations_custom_column = library.get(KEY_LOCATIONS, DEFAULT_COLUMNS[KEY_LOCATIONS])
    asin_custom_column = library.get(KEY_ASIN, DEFAULT_COLUMNS[KEY_ASIN])
    return locations_custom_column, asin_custom_column


class ConfigWidget(QWidget):

    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        avail_location_columns = self.get_location_custom_columns()
        avail_asin_columns = self.get_asin_custom_columns()
        location_col, asin_col = get_custom_columns_for_library(self.plugin_action.gui.current_db)

        # --- LOCATIONS ---
        location_group_box = QGroupBox('Estimated Locations', self)
        layout.addWidget(location_group_box)
        location_group_box_layout = QGridLayout()
        location_group_box.setLayout(location_group_box_layout)

        location_column_label = QLabel('&Custom column:', self)
        location_column_label.setToolTip('Select an integer custom column to store the estimated locations')
        self.location_column_combo = CustomColumnComboBox(self, avail_location_columns, location_col)
        location_column_label.setBuddy(self.location_column_combo)
        location_group_box_layout.addWidget(location_column_label, 0, 0, 1, 1)
        location_group_box_layout.addWidget(self.location_column_combo, 0, 1, 1, 2)

        # --- ASIN ---
        layout.addSpacing(5)
        asin_group_box = QGroupBox('Generated ASIN', self)
        layout.addWidget(asin_group_box)
        asin_group_box_layout = QGridLayout()
        asin_group_box.setLayout(asin_group_box_layout)

        asin_column_label = QLabel('C&ustom column:', self)
        asin_column_label.setToolTip('Select a text custom column to store the generate ASIN')
        self.asin_column_combo = CustomColumnComboBox(self, avail_asin_columns, asin_col)
        asin_column_label.setBuddy(self.asin_column_combo)
        asin_group_box_layout.addWidget(asin_column_label, 0, 0, 1, 1)
        asin_group_box_layout.addWidget(self.asin_column_combo, 0, 1, 1, 2)

        keyboard_shortcuts_button = QPushButton('Keyboard Shortcuts...', self)
        keyboard_shortcuts_button.setToolTip(_(
                    'Edit the keyboard shortcuts associated with this plugin'))
        keyboard_shortcuts_button.clicked.connect(self.edit_shortcuts)
        layout.addWidget(keyboard_shortcuts_button)

    def save_settings(self):

        libraries = plugin_prefs[STORE_COLUMNS]
        library_id = get_library_uuid(self.plugin_action.gui.current_db)
        library = libraries.get(library_id, DEFAULT_COLUMNS)
        library[KEY_LOCATIONS] = self.location_column_combo.get_selected_column()
        library[KEY_ASIN] = self.asin_column_combo.get_selected_column()
        libraries[library_id] = library
        plugin_prefs[STORE_COLUMNS] = libraries

    def get_location_custom_columns(self):
        column_types = ['int']
        custom_columns = self.plugin_action.gui.library_view.model().custom_columns
        available_columns = {}
        for key, column in custom_columns.iteritems():
            typ = column['datatype']
            if typ in column_types:
                available_columns[key] = column
        return available_columns

    def get_asin_custom_columns(self):
        column_types = ['text']
        custom_columns = self.plugin_action.gui.library_view.model().custom_columns
        available_columns = {}
        for key, column in custom_columns.iteritems():
            typ = column['datatype']
            if typ in column_types:
                available_columns[key] = column
        return available_columns

    def edit_shortcuts(self):
        d = KeyboardConfigDialog(self.plugin_action.gui, self.plugin_action.action_spec[0])
        if d.exec_() == d.Accepted:
            self.plugin_action.gui.keyboard.finalize()
