#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2012, Dennis Cranston <dennis_cranston@yahoo.com>'
__docformat__ = 'restructuredtext en'

import os
from calibre.ebooks.metadata.book.base import Metadata
from calibre.gui2 import question_dialog, Dispatcher
from calibre.gui2.actions import InterfaceAction
from calibre.gui2.dialogs.message_box import ProceedNotification, ErrorNotification
from calibre.ptempfile import PersistentTemporaryDirectory, remove_dir

import calibre_plugins.webos_kindle_import.config as cfg
from calibre_plugins.webos_kindle_import.common_utils import set_plugin_icon_resources, get_icon
from calibre_plugins.webos_kindle_import.dialogs import QueueProgressDialog

PLUGIN_ICONS = ['images/webos_kindle_import.png']

class GetInfoAction(InterfaceAction):

    name = 'WebOS Kindle-Import Plugin'
    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = ('Get WebOS Kindle-Import Info', None, 'Estimate total locations and generate '
                   'ASIN from MOBI files to store in custom columns.', ())
    action_type = 'current'

    def genesis(self):
        # Read the plugin icons and store for potential sharing with the config widget
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)

        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.count_locations)

    def count_locations(self):
        # Verify we have a custom column configured to store the locations and asin
        db = self.gui.library_view.model().db
        locations_col, asin_col = cfg.get_custom_columns_for_library(db)
        all_cols = db.field_metadata.custom_field_metadata()
        is_valid_location_col = len(locations_col) > 0 and locations_col in all_cols
        is_valid_asin_col = len(asin_col) > 0 and asin_col in all_cols
        if not is_valid_location_col or not is_valid_asin_col:
            if not question_dialog(self.gui, 'Configure WebOS Kindle-Import Plugin', '<p>'+
                'Custom columns have not been setup to store the <b>WebOS Kindle-Import</b> '
                'plugin information.  Do you want to configure them now?',
                show_copy_button=False):
                return
            self.show_configuration()
            return

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return
        book_ids = self.gui.library_view.get_selected_ids()

        # Create a temporary directory to copy all the Mobi files to while scanning
        tdir = PersistentTemporaryDirectory('_kindle_import', prefix='')

        # Queue all the books and kick off the job
        QueueProgressDialog(self.gui, book_ids, tdir, locations_col,
                            asin_col, self._queue_job, db)

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)

    def _queue_job(self, tdir, books_to_scan, locations_custom_column,
                   asin_custom_column):
        if not books_to_scan:
            # All failed so cleanup our temp directory
            remove_dir(tdir)
            return

        func = 'arbitrary_n'
        cpus = self.gui.job_manager.server.pool_size
        args = ['calibre_plugins.webos_kindle_import.jobs', 'do_get_info',
                (books_to_scan, cpus)]
        desc = 'Getting WebOS Kindle-Import info'
        job = self.gui.job_manager.run_job(
                self.Dispatcher(self._get_info_completed), func, args=args,
                    description=desc)
        job.tdir = tdir
        job.locations_custom_column = locations_custom_column
        job.asin_custom_column = asin_custom_column
        self.gui.status_bar.show_message('Getting WebOS Kindle-Import Info For %d Book(s)'%len(books_to_scan))

    def _get_info_completed(self, job):
        remove_dir(job.tdir)
        if job.failed:
            return self.gui.job_exception(job, dialog_title='WebOS Kindle-Import Failed')
        self.gui.status_bar.show_message('Getting Kindle-Import Info Completed', 3000)
        book_locations_map, book_asin_map = job.result

        if len(book_locations_map) + len(book_asin_map) == 0:
            # Must have been some sort of error in processing this book
            msg = '<b>WebOS Kindle-Import</b> plugin failed to get information. <b>View Log</b> for details.'
            p = ErrorNotification(job.details, 'WebOS Kindle-Import Plugin Log', 'Get Locations Failed', msg,
                    show_copy_button=False, parent=self.gui)
        else:
            payload = (book_locations_map, job.locations_custom_column, book_asin_map, job.asin_custom_column)
            all_ids = set(book_locations_map.keys()) | set(book_asin_map.keys())
            msg = '<p>WebOS Kindle-Import plugin found <b>%d statistic(s)</b>. ' % len(all_ids) + \
                  'Proceed with updating your library?'
            p = ProceedNotification(self._update_database_columns,
                    payload, job.details,
                    'WebOS Kindle-Import Plugin Log', 'Get Kindle-Import Info Complete', msg,
                    show_copy_button=False, parent=self.gui)
        p.show()

    def _update_database_columns(self, payload):
        (book_locations_map, locations_custom_column, book_asin_map, asin_custom_column) = payload

        modified = set()
        db = self.gui.current_db
        custom_cols = db.field_metadata.custom_field_metadata()
        if locations_custom_column:
            locations_col = custom_cols[locations_custom_column]
        if asin_custom_column:
            asin_col = custom_cols[asin_custom_column]

        # At this point we want to re-use code in edit_metadata to go ahead and
        # apply the changes. So we will create empty Metadata objects so only
        # the custom column field gets updated
        id_map = {}
        all_ids = set(book_locations_map.keys()) | set(book_asin_map.keys())
        for id in all_ids:
            mi = Metadata(_('Unknown'))
            if id in book_locations_map:
                locations_col['#value#'] = book_locations_map[id]
                mi.set_user_metadata(locations_custom_column, locations_col)
            if id in book_asin_map:
                asin_col['#value#'] = book_asin_map[id]
                mi.set_user_metadata(asin_custom_column, asin_col)
            id_map[id] = mi
        edit_metadata_action = self.gui.iactions['Edit Metadata']
        edit_metadata_action.apply_metadata_changes(id_map)
