#!/bin/env python

"""
Python wrapper for liberipc
"""

import ctypes as C

try:
    # Load the Library
    eripc = C.cdll.LoadLibrary( 'liberipc.so.0' )
except:
    # Could not load
    eripc = None
    print "ERROR: could not load liberipc"
    
    

def CFunc(fcn, argin, argout=None):
    """Convenience function for DLL-function-declaration"""
    ret = fcn
    ret.argtypes = argin
    if argout != None:
        ret.restype = argout
    return ret

#------------------- Defines and typedefs -----------------
class ERIPC_ERROR(object):
    """
    SUCCESS
    INVALID
    OOM
    ERROR
    TIMEOUT
    use print ERIPC_ERROR.SUCCESS to get a value
    """
    values = ['SUCCESS','INVALID','OOM','ERROR','TIMEOUT']

    class __metaclass__(type):
        def __getattr__(self, name):
            return self.values.index(name)

        def nameOf(self, i):
            return self.values[i]


class ERIPC_DATA(object):
    """
    INVALID
    BOOL
    BYTE
    INT
    UINT
    DOUBLE
    STRING
    DATA
    """
    values = ['INVALID','BOOL','BYTE','INT','UINT','DOUBLE','STRING','DATA']

    class __metaclass__(type):
        def __getattr__(self, name):
            return self.values.index(name)

        def nameOf(self, i):
            return self.values[i]

class ERIPC_BUS(object):
    """
    IRRELEVANT
    SYSTEM
    SESSION
    BOTH
    """
    values = ['IRRELEVANT','SYSTEM','SESSION','BOTH']

    class __metaclass__(type):
        def __getattr__(self, name):
            return self.values.index(name)

        def nameOf(self, i):
            return self.values[i]

class ERIPC_EVENT(object):
    """
    NONE
    MESSAGE
    SIGNAL
    MESSAGE_OR_SIGNAL
    REPLY
    ERROR
    REPLY_OR_ERROR
    """
    values = ['NONE','MESSAGE','SIGNAL','MESSAGE_OR_SIGNAL','REPLY','ERROR','REPLY_OR_ERROR']

    class __metaclass__(type):
        def __getattr__(self, name):
            return self.values.index(name)

        def nameOf(self, i):
            return self.values[i]

#------------------- Structures -----------------
class CstructPrint(C.Structure):
    """Print all fields of a structure"""
    def __str__(self):
        r = ""
        for name,type in self._fields_:
            item =self.__getattribute__(name) 
            try:
                s = str( item[:] )
            except:
                s = str( item )
            r = r + ("\t%s = %s\n" % (name,s))
        return r

class ArgValue(C.Union):
    _fields_ = [
        ('b', C.c_int32),
        ('by', C.c_uint8),
        ('i', C.c_int32),
        ('u', C.c_uint32),
        ('d', C.c_double),
        ('s', C.c_char_p),
        ('data', C.c_void_p)
        ]
    

class TEripcArg(CstructPrint):
    _fields_ = (
        ('type', C.c_uint32 ),
        ('data_len', C.c_uint32 ),
        ('value', ArgValue ),
        )

class TEripcInfo(CstructPrint):
    _fields_ = (
        ('service', C.c_char_p ),
        ('path', C.c_char_p ),
        ('interface', C.c_char_p ),
        ('name', C.c_char_p ),
        ('event_type', C.c_uint32 ),
        ('bus_type', C.c_uint32 ),
        ('message_id', C.c_char_p ),
        ('error', C.c_char_p ),
        ('args', C.POINTER(TEripcArg )),
        ('dbus_message', C.c_char_p ),
        )

class Context(object):
    def __init__(self, name, version):
        self.context = eripc.eripc_init(name, version, 0x0)
        if self.context == 0:
            print "ERROR: Could not create ipc environment!"
            self.context = None
        else:
            print 'Init Context OK'

    def set_signal_handler(self, handler, user_data, bus_type, source, signal_name):
        handler_id = C.c_int32()
        error = eripc.eripc_set_signal_handler(self.context,
                                               handler,
                                               None,
                                               bus_type,
                                               source,
                                               signal_name,
                                               C.byref(handler_id))
        return error, handler_id

    def set_message_handler(self, handler, user_data, bus_type, source, message_name):
        handler_id = C.c_int32()
        error = eripc.eripc_set_message_handler(self.context,
                                               handler,
                                               None,
                                               bus_type,
                                               source,
                                               message_name,
                                               C.byref(handler_id))
        return error, handler_id

    def unset_handler(self, handler_id):
        pass

    def send_varargs(self, destination, message_name, *in_args):
        x = ERIPC_ERROR.ERROR
        if not self.context == None:
            out_args = [self.context, 0, 0, ERIPC_BUS.SESSION, destination, message_name]
            for arg in in_args:
                # TODO: error checking
                out_args.append(arg)
            out_args.append(ERIPC_DATA.INVALID)
            #print out_args
            x = eripc.eripc_send_varargs(*out_args)
        else:
            print 'ERROR send_varargs called too early, signal[%s]' % signal_name
        return x

    def send_varargs_and_wait(self, destination, message_name, *in_args):
        if not self.context == None:
            info = TEripcInfo()
            pinfo = C.pointer(info)
            out_args = [self.context, C.byref(pinfo), ERIPC_BUS.SESSION, destination, message_name]
            for arg in in_args:
                # TODO: error checking
                out_args.append(arg)
            out_args.append(ERIPC_DATA.INVALID)
            #print out_args
            x = eripc.eripc_send_varargs_and_wait(*out_args)
            return x, pinfo
        else:
            return ERIPC_ERROR.ERROR, None

    def send_signal_varargs(self, signal_path, signal_interface, signal_name, *in_args):
        x = ERIPC_ERROR.ERROR
        if not self.context == None:
            out_args = [self.context, ERIPC_BUS.SESSION, signal_path, signal_interface, signal_name]
            for arg in in_args:
                # TODO: error checking
                out_args.append(arg)
            out_args.append(ERIPC_DATA.INVALID)
            #print out_args
            x = eripc.eripc_send_signal_varargs(*out_args)
        else:
            print 'ERROR send_signal_varargs called too early, signal[%s]' % signal_name
        return x

    def reply_bool(self, message_id, val):
        if not self.context == None:
            eripc.eripc_reply_bool(self.context, message_id, val)

    def reply_varargs(self, message_id, *in_args):
        if not self.context == None:
            out_args = [self.context, message_id]
            for arg in in_args:
                # TODO: error checking
                out_args.append(arg)
            out_args.append(ERIPC_DATA.INVALID)
            print out_args
            eripc.eripc_reply_varargs(*out_args)
        

