#!/bin/env python
# -*- coding: cp1252 -*-

"""
Application interface for DR applications
Based on vala implementation of Inigo
"""

import eripc
import os
import ctypes as C
from pdb import set_trace

# Global data, needed for callback functions
g_ipc = None

class DrApplication(object):
    """
    Application interface for DR applications
    Based on vala implementation of Iigo
    """
    def getMainWindow(self):
        """
        Get the application's main window
        """
        raise NotImplementedError
        
    def onMenuClick(self, item, group, menu, state):
        """
        Called when the a menu item is clicked in the app's menu
        """
        raise NotImplementedError

    def onFileOpen(self, filename):
        """
        Called when the app should open a file
        return a window id, xid, of the window which displays the opened file
        or a error string, and xid=-1
        """
        raise NotImplementedError

    def onFileClose(self, filename):
        """
        Called when the app should close a file
        """
        raise NotImplementedError

    def onWindowChange(self, xid, activated):
        """
        Called when the app's window is activated or deactivated.
        On activation, the app should .show() its menu.
        """
        raise NotImplementedError

    def onPrepareUnmount(self, device):
        """
        Called when the app should close any open files on device
        """
        raise NotImplementedError

    def onUnmounted(self, device):
        """
        Called when the system unmounts a device
        """
        raise NotImplementedError

    def onMounted(self, device):
        """
        Called when the system mounts a device
        """
        raise NotImplementedError

    def onPrepareHibernate(self):
        """
        Called when the system is about to hibernate
        """
        raise NotImplementedError

    def onChangeLocale(self, locale):
        """
        Called when the system locale has changed
        """
        raise NotImplementedError

    def onChangedOrientation(self, orientation):
        """
        Called when the device orientation changes
        """
        raise NotImplementedError

# Interface definition for the handler functions
HANDLER = C.CFUNCTYPE(None, C.POINTER(C.c_int32), C.POINTER(eripc.TEripcInfo), C.POINTER(C.c_int32))

# Helper functions for the handler functions
# TODO: can class function be directly used with lambda functions?
def my_on_menu(ctx, pinfo, user_data):
    item  = pinfo[0].args[0].value.s
    group = pinfo[0].args[1].value.s
    menu  = pinfo[0].args[2].value.s
    state = pinfo[0].args[3].value.s
    print "MENU: [%s][%s][%s][%s]" % (menu, group, item, state)
    g_ipc.on_menu_item(ctx, pinfo)
    
def my_on_window_activated(ctx, pinfo, user_data):
    g_ipc.on_window_activated(ctx, pinfo)
    
def my_on_window_deactivated(ctx, pinfo, user_data):
    g_ipc.on_window_deactivated(ctx, pinfo)
    
def my_on_file_open(ctx, pinfo, user_data):
    g_ipc.on_file_open(ctx, pinfo)
    
def my_on_file_close(ctx, pinfo, user_data):
    g_ipc.on_file_close(ctx, pinfo)
    
class DrIpc(object):
    def __init__(self, name, version, application):
        global g_ipc
        self.appname = name
        self.app = application
        self.handler_ids = []
        self.context = eripc.Context(name, version)
        if self.context == None:
            print "Failed to initialise eripc context"
        self.num_handlers = 0
        self.set_all_handlers()
        # init global var for callbacks (note that DrIpc will now never be freed)
        # TODO: extra check that g_ipc is still None!! (Singleton)
        g_ipc = self
        
    def __del__(self):
        self.unset_all_handlers()

    def get_appname(self):
        return self.appname

    def get_dbus_interface(self):
        return "com.irexnet." + self.appname

    def get_dbus_path(self):
        return "/com/irexnet/" + self.appname

    def add_signal_handler(self, handler, interface_name, message_name, user_data):
        error, id = self.context.set_signal_handler(handler,
                                                    userdata,
                                                    eripc.ERIPC_BUS.SESSION,
                                                    interface_name,
                                                    message_name)
        self.handler_ids.append(id)
        return id
    
    def add_message_handler(self, handler, message_name, user_data):
        error, id = self.context.set_message_handler(handler,
                                                     user_data,
                                                     eripc.ERIPC_BUS.SESSION,
                                                     self.get_dbus_interface(),
                                                     message_name)
        self.handler_ids.append(id)
        return id

    def unset_all_handlers(self):
        for id in self.handler_ids:
            self.context.unset_handler(id)

    def send_startup_complete(self, window):
        my_pid = os.getpid()
        my_xid = window
        is_multidoc = True
        show_taskbar = True

        self.context.send_signal_varargs(self.get_dbus_path(),
                                         "com.irexnet.sysd",
                                         "startupComplete",
                                         eripc.ERIPC_DATA.STRING, self.get_appname(),
                                         eripc.ERIPC_DATA.INT, my_pid,
                                         eripc.ERIPC_DATA.BOOL, is_multidoc,
                                         eripc.ERIPC_DATA.STRING, self.get_dbus_interface(),
                                         eripc.ERIPC_DATA.INT, my_xid,
                                         eripc.ERIPC_DATA.BOOL, show_taskbar)

    def get_battery_state(self):
        pinfo = self.context.send_varargs_and_wait("com.irexnet.sysd",
                                                   "sysGetBatteryState")
        print "Level[%s]: %d"    % (eripc.ERIPC_DATA.nameOf(pinfo[0].args[0].type), pinfo[0].args[0].value.i)
        print "State[%s]: %s"    % (eripc.ERIPC_DATA.nameOf(pinfo[0].args[1].type), pinfo[0].args[1].value.s)
        print "TimeLeft[%s]: %d" % (eripc.ERIPC_DATA.nameOf(pinfo[0].args[2].type), pinfo[0].args[2].value.i)

        # TODO: check types
        level = pinfo[0].args[0].value.i
        state = pinfo[0].args[1].value.s

        # TODO: move to Context
        eripc.eripc_event_info_free(self.context.context, pinfo)

        return level, state

    def start_task(self, command_line, working_dir, label, file_icon):
        pinfo = self.context.send_varargs_and_wait("com.irexnet.sysd",
                                                   "startTask",
                                                   eripc.ERIPC_DATA.STRING, command_line,
                                                   eripc.ERIPC_DATA.STRING, working_dir,
                                                   eripc.ERIPC_DATA.STRING, label,
                                                   eripc.ERIPC_DATA.STRING, file_icon)

        # TODO: check types
        ok = pinfo[0].args[0].value.i
        
        # TODO: move to Context
        eripc.eripc_event_info_free(self.context.context, pinfo)

    def on_menu_item(self, context, pinfo):
        print 'on_menu_item-Called'
        item  = pinfo[0].args[0].value.s
        group = pinfo[0].args[1].value.s
        menu  = pinfo[0].args[2].value.s
        state = pinfo[0].args[3].value.s
        self.app.onMenuClick(item, group, menu, state)

    def on_window_activated(self, context, pinfo):
        print 'on_window_activated-Called'
        xid = pinfo[0].args[0].value.i
        self.app.onWindowChange(xid, True)
        self.context.reply_bool(pinfo[0].message_id, True)

    def on_window_deactivated(self, context, pinfo):
        print 'on_window_deactivated-Called'
        xid = pinfo[0].args[0].value.i
        self.app.onWindowChange(xid, False)
        self.context.reply_bool(pinfo[0].message_id, True)

    def on_file_open(self, context, pinfo):
        print 'on_file_open-Called'
        #pass

    def on_file_close(self, context, pinfo):
        print 'on_file_close-Called'
        #pass

    def set_all_handlers(self):
        self.func_on_menu_item = HANDLER(my_on_menu)
        self.add_message_handler(self.func_on_menu_item, "menuItemActivated", self)

        self.func_on_window_activated = HANDLER(my_on_window_activated)
        self.add_message_handler(self.func_on_window_activated, "activatedWindow", self)

        self.func_on_window_deactivated = HANDLER(my_on_window_deactivated)
        self.add_message_handler(self.func_on_window_deactivated, "deactivatedWindow", self)

        self.func_on_file_open = HANDLER(my_on_file_open)
        self.add_message_handler(self.func_on_file_open, "openFile", self)

        self.func_on_file_close = HANDLER(my_on_file_close)
        self.add_message_handler(self.func_on_file_close, "closeFile", self)

class MenuItem(object):
    def __init__(self, mm, group, name, label, image):
        self.manager = mm
        self.name = name
        self.label = label
        self.image = image
        self.group = group

    def realise(self):
        self.manager.add_item(self.name, self.group.getName(), self.image)
        self.manager.set_item_label(self.name, self.group.getName(), self.label)

class MenuGroup(object):
    def __init__(self, mm, name, label, parent=""):
        self.manager = mm
        # construct unique name with appname and group-name
        self.name = mm.getAppName() + "_" + name
        self.label = label
        self.parent = parent
        self.items = []
        self.subgroups = []

    def addItem(self, name, label, image):
        item = MenuItem(self.manager, self, name, label, image)
        self.items.append(item)
        return item

    def addSubGroup(self, name, label):
        subgroup = MenuGroup(self.manager, name, label, self.name)
        self.subgroups.append(subgroup)
        return subgroup

    def realise(self):
        self.manager.add_group(self.name, self.parent, "folder")
        self.manager.set_group_label(self.name, self.label)
        for item in self.items:
            item.realise()
        for subgroup in self.subgroups:
            subgroup.realise()

    def getName(self):
        return self.name

class Menu(object):
    def __init__(self, mm, name, label):
        self.manager = mm
        # construct unique name with appname and menu-name
        self.name = mm.getAppName() + "_" + name
        self.label = label
        self.groups = []

    def __del__(self):
        set_trace()
        self.manager.remove_menu(self.name)

    def addGroup(self, name, label):
        group = MenuGroup(self.manager, name, label)
        self.groups.append(group)
        return group

    def realise(self):
        for group in self.groups:
            group.realise()

        group0 = None
        group1 = None
        group2 = None
        if len(self.groups) > 0:
            group0 = self.groups[0].getName()
        if len(self.groups) > 1:
            group0 = self.groups[1].getName()
        if len(self.groups) > 2:
            group0 = self.groups[2].getName()
        self.manager.add_menu(self.name,
                              group0,
                              group1,
                              group2)
        self.manager.set_menu_label(self.name, self.label)

    def show(self):
        self.manager.show_menu(self.name)

class MenuManager(object):
    def __init__(self, ipc):
        self.ipc = ipc

    def getAppName(self):
        return self.ipc.get_appname()

    def add_menu(self, name, group1, group2, group3):
        self.ipc.context.send_varargs("com.irexnet.popupmenu",
                                      "addMenu",
                                      eripc.ERIPC_DATA.STRING, name,
                                      eripc.ERIPC_DATA.STRING, "",
                                      eripc.ERIPC_DATA.STRING, self.ipc.get_dbus_interface(),
                                      eripc.ERIPC_DATA.STRING, group1,
                                      eripc.ERIPC_DATA.STRING, group2,
                                      eripc.ERIPC_DATA.STRING, group3,
                                      eripc.ERIPC_DATA.STRING, "")
        return True

    def add_group(self, name, parent, image):
        self.ipc.context.send_varargs("com.irexnet.popupmenu",
                                      "addGroup",
                                      eripc.ERIPC_DATA.STRING, name,
                                      eripc.ERIPC_DATA.STRING, parent,
                                      eripc.ERIPC_DATA.STRING, "",
                                      eripc.ERIPC_DATA.STRING, image)
        return True

    def add_item(self, name, parent, image):
        self.ipc.context.send_varargs("com.irexnet.popupmenu",
                                      "addItem",
                                      eripc.ERIPC_DATA.STRING, name,
                                      eripc.ERIPC_DATA.STRING, parent,
                                      eripc.ERIPC_DATA.STRING, "",
                                      eripc.ERIPC_DATA.STRING, image)
        return True
        
    def set_menu_label(self, name, label):
        self.ipc.context.send_varargs("com.irexnet.popupmenu",
                                      "setMenuLabel",
                                      eripc.ERIPC_DATA.STRING, name,
                                      eripc.ERIPC_DATA.STRING, label)
        return True
        
    def set_group_label(self, name, label):
        self.ipc.context.send_varargs("com.irexnet.popupmenu",
                                      "setGroupLabel",
                                      eripc.ERIPC_DATA.STRING, name,
                                      eripc.ERIPC_DATA.STRING, label)
        return True
        
    def set_item_label(self, name, parent, label):
        self.ipc.context.send_varargs("com.irexnet.popupmenu",
                                      "setItemLabel",
                                      eripc.ERIPC_DATA.STRING, name,
                                      eripc.ERIPC_DATA.STRING, parent,
                                      eripc.ERIPC_DATA.STRING, label)
        return True
        
    def set_item_state(self, name, parent, state):
        self.ipc.context.send_varargs("com.irexnet.popupmenu",
                                      "setItemState",
                                      eripc.ERIPC_DATA.STRING, name,
                                      eripc.ERIPC_DATA.STRING, parent,
                                      eripc.ERIPC_DATA.STRING, state)
        return True
        
    def set_status_item_state(self, name, state):
        self.ipc.context.send_varargs("com.irexnet.popupmenu",
                                      "setStatusItemState",
                                      eripc.ERIPC_DATA.STRING, name,
                                      eripc.ERIPC_DATA.STRING, state)
        return True
        
    def set_status_item_show(self, name, mode):
        self.ipc.context.send_varargs("com.irexnet.popupmenu",
                                      "setStatusItemShow",
                                      eripc.ERIPC_DATA.STRING, name,
                                      eripc.ERIPC_DATA.STRING, mode)
        return True
        
    def show_menu(self, name):
        self.ipc.context.send_varargs("com.irexnet.popupmenu",
                                      "showMenu",
                                      eripc.ERIPC_DATA.STRING, name)
        return True
        
    def remove_menu(self, name):
        self.ipc.context.send_varargs("com.irexnet.popupmenu",
                                      "removeMenu",
                                      eripc.ERIPC_DATA.STRING, name)
        return True
        
    def add_toolbar_item(self, menu, group, item):
        self.ipc.context.send_varargs("com.irexnet.popupmenu",
                                      "addToolbarItem",
                                      eripc.ERIPC_DATA.STRING, menu,
                                      eripc.ERIPC_DATA.STRING, group,
                                      eripc.ERIPC_DATA.STRING, item)
        return True
        
    def clear_toolbar_item(self, arg):
        self.ipc.context.send_varargs("com.irexnet.popupmenu",
                                      "addToolbarItems",
                                      eripc.ERIPC_DATA.STRING, arg)
        return True
        

        
    
