#!/usr/bin/perl
use strict;
use warnings;

if (3 != scalar @ARGV) {
	print "Usage:   js2template.pl <source file> <template file> <properties file>\n";
	print "         Produces template and properties file from source file.\n\n";
	print "Example: js2template.pl search_bar_strings.js search_bar_strings.tjs search_bar_strings.properties\n\n";
}

my ($source, $template, $properties) = @ARGV;

open SOURCE, $source or die $!;
open TEMPLATE, ">$template" or die $!;
open PROPERTIES, ">$properties" or die $!;

# "fully qualified" context used as identifier.
my @context;

while (<SOURCE>) {
#	print;
	my $line = $_;
	
	if ($line =~ /^(\s*var\s+)?\s*(\S+)\s*=/) {
		# Usually first line, or new variable definition
		push @context, $2;
	}
	elsif ($line =~ /^(\s*)(\S+)(\s*:\s*)(.*?)([\r\n]+)$/) {
		# any other line matching a single identifier, a colon, and some rest
		my ($pre, $id, $colon, $rest, $nl) = ($1, $2, $3, $4, $5);
		$context[-1] = $id;
		$rest = &extractAndReplace($rest);
		$line = "$pre$id$colon$rest$nl";
	}
	if ($line =~ /([\[\{])\s*$/) {
		# a line starting a new context (ending with [ or {)
		if ($1 eq '[') {
			# array context:
			push @context, 1;
		} else {
			# hash context
			push @context, '';
		}
	} elsif ($line =~ /([\]\}])\s*,?\s*$/) {
		# a line ending the current context
		pop @context;

		# if the topmost context is an array, increment it.
		if ($context[-1] =~ /^\d+$/) {
			$context[-1]++;
		}
	}
	print TEMPLATE $line;
}
close SOURCE;
close TEMPLATE;
close PROPERTIES;

exit;

# takes a "right-of-colon" expression.
# If the expression contains a string, it is replaced by
# "%context%", and a corresponding property is written to
# the properties file. Otherwise, it is just returned unchanged.
sub extractAndReplace {
	my ($expr) = @_;

	my $ctx = join('.',@context);
	my $string = undef;

	# String enclosed in double quotes (simple)
	if ($expr =~ /^(\s*)"(.*)"([^"]*)$/) {
		$string = $2;
	}
	# String enclosed in single quotes (simple)
	elsif ($expr =~ /^(\s*)'(.*)'([^']*)$/) {
		$string = $2;
	}
	# String enclosed in double quotes (complex)
	elsif ($expr =~ /^(.*?)"(.*)"(.*)$/) {
		$string = $2;
	}
	# String enclosed in single quotes (complex)
	elsif ($expr =~ /^(.*?)'(.*)'(.*)$/) {
		$string = $2;
	}
	if (defined $string) {
		$expr = $1.'%'.$ctx.'%'.$3;
		print PROPERTIES "$ctx=$string\n";
	}
	return $expr;
}

