#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>, Pavel Kvasnicka'
__docformat__ = 'restructuredtext en'

import time, re, dateutil
from datetime import datetime
from urllib import quote, unquote, urlencode
from Queue import Queue, Empty

from lxml import etree

from calibre import as_unicode
from calibre.ebooks.metadata.sources.base import Source
from calibre.ebooks.metadata.book.base import Metadata
from calibre.ebooks.metadata import string_to_authors
from calibre.utils.icu import lower
from calibre.utils.cleantext import clean_ascii_chars
from calibre.utils.localization import get_udc


class OnlineKnihovna(Source):

    name                    = 'OnlineKnihovna.cz'
    description             = _('Stahuje metadata a obaly z OnlineKnihovna.cz')
    author                  = 'Pavel Kvasnicka'
    version                 = (1, 0, 1)
    minimum_calibre_version = (0, 8, 0)

    capabilities = frozenset(['identify', 'cover'])
    touched_fields = frozenset(['title', 'authors', 'identifier:isbn', 'identifier:onlineknihovna',
                                'comments', 'publisher', 'pubdate', 'rating', 'tags', 'series'])
    has_html_comments = True
    supports_gzip_transfer_encoding = False

    BASE_URL = 'http://www.onlineknihovna.cz/calibre_tasks.php?'

    def get_book_url(self, identifiers):
        ff_id = identifiers.get('onlineknihovna', None)
        if ff_id:
            url = self.BASE_URL
            url=url+urlencode({
                'version':OnlineKnihovna.version,
                'task': 'book_url',
                'book_url':ff_id,
                })
            return ('OnlineKnihovna', ff_id, url)

    def get_cached_cover_url(self, identifiers):
        url = None
        ff_id = identifiers.get('onlineknihovna', None)
        if ff_id is not None:
            url = self.cached_identifier_to_cover_url(ff_id)
        return url

    def identify(self, log, result_queue, abort, title=None, authors=None,
            identifiers={}, timeout=30):
        '''
        Note this method will retry without identifiers automatically if no
        match is found with identifiers.
        '''
        url = self.BASE_URL
        url=url+urlencode({
            'version':OnlineKnihovna.version,
            })+'&'

        # If we have a OnlineKnihovna id then we do not need to fire a "search".
        # Instead we will go straight to the URL for that book.
        onlineknihovna_id = identifiers.get('onlineknihovna', None)
        isbn = identifiers.get('isbn', None)
        br = self.browser
        if onlineknihovna_id:
            url = url+urlencode({
                'task':'identify',
                'onlineknihovna_id':onlineknihovna_id,
                })+'&'
        if isbn:
            url=url+urlencode({
                'isbn':isbn,
                })+'&'
        if title:
            url=url+urlencode({
                'title':title,
                })+'&'
        if authors:
            for x in authors:
                url=url+urlencode({
                    'authors[]':x,
                    })+'&'
        # Download all metadata
        response_xml = br.open_novisit(url, timeout=timeout).read()
        
        
        try:
            root = etree.fromstring(response_xml)
            err = root.attrib["stat"]
            if err is not None and err == 'fail':
                return None
            items=root.findall('.//item')
            for item in items:
                mi = Metadata(title, authors)
                try:
                    mi.isbn = item.findtext('.//isbn')
                    #mi.set_identifier('isbn', mi.isbn)
                except:
                    log.exception('Error parsing isbn')
                try:
                    mi.title_sort = mi.title = item.findtext('.//title')
                except:
                    log.exception('Error parsing title')
                try:
                    auths = [x.text for x in item.findall('.//author')]
                    if auths:
                        mi.authors = []
                        for x in auths:
                            mi.authors.extend(string_to_authors(x))
                except:
                    log.exception('Error parsing authors')
                try:
                    mi.author_sort = item.findtext('.//author_sort')
                except:
                    log.exception('Error parsing authors sort')
                try:
                    mi.rating = item.findtext('.//rating')
                except:
                    log.exception('Error parsing rating')
                try:
                    mi.publisher = item.findtext('.//publisher')
                except:
                    log.exception('Error parsing publisher')
                try:
                    mi.tags = [x.text for x in item.findall('.//tag')]
                except:
                    log.exception('Error parsing tags')
                try:
                    mi.series = item.findtext('.//series')
                    # series_index
                except:
                    log.exception('Error parsing series')
                default = datetime.utcnow()
                default = datetime(default.year, default.month, 15)
                try:
                    mi.timestamp = dateutil.parser.parse(item.findtext('.//timestamp'), default=default)
                except:
                    log.exception('Error parsing timestamp')
                try:
                    mi.pubdate = dateutil.parser.parse(item.findtext('.//pubdate'), default=default)
                except:
                    log.exception('Error parsing published date')
                try:
                    mi.language = item.findtext('.//language')
                except:
                    pass
                try:
                    mi.comments = item.findtext('.//comments')
                except:
                    log.exception('Error parsing comments')
                try:
                    mi.cover_url = item.findtext('.//cover_url')
                except:
                    log.exception('Error parsing cover')
                mi.has_cover = bool(mi.cover_url)
                try:
                    onlineknihovna_reply_id = item.findtext('.//id')
                except:
                    log.exception('Error parsing OnlineKnihovna id')
                    onlineknihovna_reply_id = None
                
                if onlineknihovna_reply_id:
                    mi.set_identifier('onlineknihovna', onlineknihovna_reply_id)
                    if mi.cover_url:
                        self.cache_identifier_to_cover_url(onlineknihovna_reply_id, mi.cover_url)
                
                self.clean_downloaded_metadata(mi)
                result_queue.put(mi)
        except Exception as e:
            err = 'Failed to get metadata - received invalid response from www.onlineknihovna.cz'
            log.exception(err)
            return as_unicode(e)
        return None


    def download_cover(self, log, result_queue, abort,
            title=None, authors=None, identifiers={}, timeout=30):
        cached_url = self.get_cached_cover_url(identifiers)
        if cached_url is None:
            log.info('No cached cover found, running identify')
            rq = Queue()
            self.identify(log, rq, abort, title=title, authors=authors,
                    identifiers=identifiers)
            if abort.is_set():
                return
            results = []
            while True:
                try:
                    results.append(rq.get_nowait())
                except Empty:
                    break
            results.sort(key=self.identify_results_keygen(
                title=title, authors=authors, identifiers=identifiers))
            for mi in results:
                cached_url = self.get_cached_cover_url(mi.identifiers)
                if cached_url is not None:
                    break
        if cached_url is None:
            log.info('No cover found')
            return

        if abort.is_set():
            return
        br = self.browser
        log('Downloading cover from:', cached_url)
        try:
            cdata = br.open_novisit(cached_url, timeout=timeout).read()
            result_queue.put((self, cdata))
        except:
            log.exception('Failed to download cover from:', cached_url)


if __name__ == '__main__': # tests
    # To run these test use:
    # calibre-debug -e __init__.py
    from calibre.ebooks.metadata.sources.test import (test_identify_plugin,
            title_test, authors_test, series_test)
    test_identify_plugin(OnlineKnihovna.name,
        [

            ( # A book with no ISBN specified
                {'title':"Harry Potter and the Sorcerer's Stone", 'authors':['J.K. Rowling']},
                [title_test("Harry Potter and the Sorcerer's Stone",
                    exact=True), authors_test(['J. K. Rowling']),
                    series_test('Harry Potter', 1.0)]

            ),

            ( # A book with an ISBN
                {'identifiers':{'isbn': '9780439064866'},
                    'title':'Chamber of Secrets', 'authors':['J.K. Rowling']},
                [title_test('Harry Potter and the Chamber of Secrets',
                    exact=True), authors_test(['J. K. Rowling']),
                    series_test('Harry Potter', 2.0)]

            ),

            ( # A book with a OnlineKnihovna id
                {'identifiers':{'ff': '61-Hours/Lee-Child/e/9780440243694'},
                    'title':'61 Hours', 'authors':['Lee Child']},
                [title_test('61 Hours',
                    exact=True), authors_test(['Lee Child']),
                    series_test('Jack Reacher', 14.0)]

            ),

        ])

