# -*- coding: utf-8 -*-

__license__   = 'GPL v3'
__copyright__ = '2009, John Schember <john at nachtimwald.com>'
__docformat__ = 'restructuredtext en'

'''
Device driver for Barns and Nobel's Nook
'''

import os

import cStringIO

from calibre.devices.usbms.driver import USBMS

import zipfile

class NOOK(USBMS):

	name           = 'Nook Device Interface (NEW)'
	gui_name       = _('Nook')
	description    = _('Communicate with the Nook eBook reader.')
	author         = 'John Schember with enhancements by Joel Ricker'
	icon           = I('devices/nook.jpg')
	version        = (1, 1, 0)
	supported_platforms = ['windows', 'linux', 'osx']

	# Ordered list of supported formats
	FORMATS     = ['epub', 'pdb', 'pdf']

	VENDOR_ID   = [0x2080, 0x18d1] # 0x18d1 is for softrooted nook
	PRODUCT_ID  = [0x001]
	BCD         = [0x322]

	VENDOR_NAME = 'B&N'
	WINDOWS_MAIN_MEM = 'NOOK'
	WINDOWS_CARD_A_MEM = 'NOOK'

	OSX_MAIN_MEM = 'B&N nook Media'
	OSX_CARD_A_MEM = OSX_MAIN_MEM

	MAIN_MEMORY_VOLUME_LABEL  = 'Nook Main Memory'
	STORAGE_CARD_VOLUME_LABEL = 'Nook Storage Card'

	EBOOK_DIR_MAIN = 'my documents'
	THUMBNAIL_HEIGHT = 144
	DELETE_EXTS = ['.jpg']
	SUPPORTS_SUB_DIRS = True
	
	def upload_cover(self, path, filename, metadata, filepath):
	
		try:
			from PIL import Image, ImageDraw
			Image, ImageDraw
		except ImportError:
			import Image, ImageDraw


		coverdata = getattr(metadata, 'thumbnail', None)
		if coverdata and coverdata[2]:
			cover = Image.open(cStringIO.StringIO(coverdata[2]))
		else:
			coverdata = open(I('library.png'), 'rb').read()

			cover = Image.new('RGB', (96, 144), 'black')
			im = Image.open(cStringIO.StringIO(coverdata))
			im.thumbnail((96, 144), Image.ANTIALIAS)

			x, y = im.size
			cover.paste(im, ((96-x)/2, (144-y)/2))

			draw = ImageDraw.Draw(cover)
			draw.text((1, 15), metadata.get('title', _('Unknown')).encode('ascii', 'ignore'))
			draw.text((1, 115), metadata.get('authors', _('Unknown')).encode('ascii', 'ignore'))

		data = cStringIO.StringIO()
		cover.save(data, 'JPEG')
		coverdata = data.getvalue()

		with open('%s.jpg' % os.path.join(path, filename), 'wb') as coverfile:
			coverfile.write(coverdata)

	def sanitize_path_components(self, components):
		return [x.replace('#', '_') for x in components]

class NOOK_COLOR(NOOK):
	description    = _('Communicate with the Nook Color, Nook Simple Touch and TSR eBook readers.')

	PRODUCT_ID  = [0x002, 0x003]
	BCD         = [0x216]

	WINDOWS_MAIN_MEM = WINDOWS_CARD_A_MEM = 'EBOOK_DISK'
	EBOOK_DIR_MAIN = 'My Files'
	NEWS_IN_FOLDER = False

	def upload_cover(self, path, filename, metadata, filepath):
		
		if not filepath.endswith(".epub"): 
			# print filepath + " is not an .epub. skipping cover upload."
			return
		
		# path to cover image that calibre has for this epub
		coverpath = getattr(metadata, 'cover', None)			
		
		zf = zipfile.ZipFile(filepath, mode="a")
		
		try:
			opf = [info for info in zf.infolist() if info.filename.endswith(".opf")][0]
		except Exception, e:
			# print "failed to find opf file: " + e
			return
		
		import xml.dom.minidom
		dom = xml.dom.minidom.parseString(zf.read(opf))
		
		
		if not self._has_valid_cover(zf, opf, dom):
			# most ebooks will have a proper namespace defined in the
			# opf file. I had 1 book out of 256 that didn't so it will
			# throw an exception and the next _fix_meta statement will
			# take care of it.
			try:
				self._fix_meta(dom, 'http://www.idpf.org/2007/opf')
			except:
				self._fix_meta(dom)
			self._add_cover(zf, coverpath, opf, dom)
			print "Fixed " + filename
		else:
			print filename + " has a valid cover."
		zf.close()
	
	def _fix_meta(self, dom, NS=None):
		# Check that that there is a meta tag that looks like this:
		#       <meta name="cover" content="cover"/>
		# and an item tag that looks like this:
		#       <item id="cover" href="images/cover.png" media-type="image/png"/>
		# where the content value == id value (cover)
		# if (content/id) != cover, change it.

		for meta_element in dom.getElementsByTagNameNS(NS, 'meta'):
			if meta_element.getAttribute('name') == 'cover' and \
				meta_element.getAttribute('content') == 'cover':
				break
			else: meta_element = None
		if not meta_element:
			meta_element = dom.createElement('meta')
			meta_element.setAttribute('name', 'cover')
			meta_element.setAttribute('content', 'cover')
			
			first_meta_element = dom.getElementsByTagNameNS(NS,'metadata')[0]
			first_meta_element.insertBefore(meta_element, first_meta_element.firstChild)
			
		
		for item_element in dom.getElementsByTagNameNS(NS, 'item'):
			if item_element.getAttribute('id') == meta_element.getAttribute('content'):
				meta_element.setAttribute('content', 'cover')
				item_element.setAttribute('id', 'cover')
			else: 
				item_element = None
		if not item_element:
			# these will get set in the next step when we verify
			# the location of the cover image
			item_element = dom.createElement('item')
			item_element.setAttribute('id', 'cover')
			item_element.setAttribute('href', '')
			item_element.setAttribute('media-type', '')
		
			first_item_element = dom.getElementsByTagNameNS(NS, 'manifest')[0]
			first_item_element.insertBefore(item_element, first_item_element.firstChild)
		
		return dom
		
	def _has_valid_cover(self, zf, opf, dom):
		# Check check that
		#       <item id="cover" href="images/cover.png" media-type="image/png"/>
		# points to a valid image. If not, copy over a new image from the same directory
		# as the original epub and update href accordingly.
		has_cover = False
		
		for item_element in dom.getElementsByTagNameNS('http://www.idpf.org/2007/opf', 'item'):
			if item_element.getAttribute('id') == 'cover':
			
				has_cover = True			
			
				if not (item_element.getAttribute('href').endswith('jpg') or \
					item_element.getAttribute('href').endswith('png')):
					# probably points to a cover page.
					return False

				# make sure path exists and points to a valid looking image file
				try:
					# if an exception is thrown by zf.read, then the file doesn't 
					# exist in the zip archive.
					
					opf_basename = os.path.split(opf.filename)[0]
						
					if opf_basename:
						data = zf.read(opf_basename + "/" + item_element.getAttribute('href'))
						#print "Looking for image at: " + opf_basename + "/" + item_element.getAttribute('href')
					else:
						data = zf.read(item_element.getAttribute('href'))
						#print "Look for image at: " + item_element.getAttribute('href')
				except KeyError:
					#print "Couldn't find a valid image in opf " + opf.filename
					return False
		return has_cover
		
	def _add_cover(self, zf, coverpath, opf, dom):
	
		for item_element in dom.getElementsByTagName('item'):
			if item_element.getAttribute('id') == 'cover':
			
				covername = os.path.split(coverpath)[1]
				opf_path = opf.filename
				opf_basename = os.path.split(opf_path)[0]
				item_element.setAttribute('href', covername)
				
				if covername.endswith('jpg'):
					item_element.setAttribute('media-type', 'image/jpeg')
				elif covername.endswith('png'):
					item_element.setAttribute('media-type', 'image/png')
				
				zf = self._remove_opf(zipfile.ZipFile(zf.fp.name, "a"), opf)
				
				import codecs
				zf.writestr(opf_path, dom.toxml("utf-8"), zipfile.ZIP_STORED)
				zf.write(coverpath, opf_basename + "/" + covername)
	
				break
				
	
	def _remove_opf(self, zin, opf):
		
		#removes the old content.opf file to make way for a new one.
		#zipfile does not have a delete command. the best i can do 
		#is create a temporary archive, copy in everything except 
		#the opf file, the copy the temporary archive over the 
		#original archive
		#
		#returns a new zipfile object pointing to the original epub.
		original_epub = zin.fp.name
		temporary_epub = self.temporary_file('.epub')
		
		zout = zipfile.ZipFile (temporary_epub, 'w')
		for item in zin.infolist():
			buffer = zin.read(item.filename)
			if (item.filename != opf.filename):
				zout.writestr(item.filename, buffer)
		zin.close()
		zout.close()
		
		# copy temporary archive to permanent
		with open(original_epub, 'wb') as fout:
			with open(temporary_epub.name, 'rb') as fin:
				fout.write(fin.read())
				fout.close()
				fin.close()
		
		return zipfile.ZipFile(original_epub, 'a') 
          
	def get_carda_ebook_dir(self, for_upload=False):
		if for_upload:
			return self.EBOOK_DIR_MAIN
		return ''

	def create_upload_path(self, path, mdata, fname, create_dirs=True):
		is_news = mdata.tags and _('News') in mdata.tags
		subdir = 'Magazines' if is_news else 'Books'
		path = os.path.join(path, subdir)
		return USBMS.create_upload_path(self, path, mdata, fname,
				create_dirs=create_dirs)		
