
# This script contains the sequence of steps required to complete
# the software update process.

# update defaults
UPDATE_TYPE=normal
UPDATE_ORIGIN=user

# code/image - to - device/volume mapping
# ntim
NTIM_FILENAME=ntim.bin.crc
NTIM_DEV=/dev/mtd8
NTIM_OFFSET=0x00000000
NTIM_ERASE_BLKS=2

# bootloader
LOADER_FILENAME=bootloader.bin.crc
LOADER_DEV=/dev/mtd8
LOADER_OFFSET=0x00020000
LOADER_ERASE_BLKS=2

# u-boot
BOOT_FILENAME=u-boot.bin.crc
BOOT_DEV=/dev/mtd9
BOOT_OFFSET=0x00000000
BOOT_ERASE_BLKS=8

# primary kernel
KERNEL_FILENAME=zImage.crc
KERNEL_DEV=/dev/mtd11
KERNEL_OFFSET=0x00000000
KERNEL_ERASE_BLKS=48

# android ramdisk
RD_ANDROID_FILENAME=ramdisk.android.gz.crc
RD_ANDROID_DEV=/dev/mtd5
RD_ANDROID_OFFSET=0x00000000
RD_ANDROID_ERASE_BLKS=2
RD_ANDROID_SIZE=0x00100000

# maint ramdisk
RD_MAINT_FILENAME=ramdisk.maint.gz.crc
RD_MAINT_DEV=/dev/mtd5
RD_MAINT_OFFSET=0x00100000
RD_MAINT_ERASE_BLKS=48
RD_MAINT_SIZE=0x01800000

# android - system (see edge_fs/android_ubinize.cfg)
SYSTEM_DEV_NUM=6
SYSTEM_MTD_DEVICE=/dev/mtd$SYSTEM_DEV_NUM
SYSTEM_VOL_NAME=system
USERDATA_VOL_NAME=userdata
CACHE_VOL_NAME=cache
VFAT_VOL_NAME=storage
CONTENT_VOL_NAME=userdata
VFAT_IMG_FILENAME=vfat.img
VFAT_NUM_BLOCKS=2617152

# flash device specific
SPI_NOR_BLK_SIZE=64K

UPDATE_SH=update.sh
UPDATE_INFO=update.info

TMP_LOG=/tmp/update.log
LOG_FILE_PATH=recovery
LOG_FILE_NAME=update.log
LOG_FILE_VOL_NAME=$CACHE_VOL_NAME
CMD_FILE_NAME=recovery/command

# zip names
BOOT_ZIP=boot.zip
ROOT_ZIP=root.zip
SYSTEM1_ZIP=system1.zip
SYSTEM2_ZIP=system2.zip
DATA_ZIP=data.zip
SDCARD_ZIP=sdcard.zip
CONTENT_ZIP=content.zip
TOOLS_ZIP=tools.zip

# script/executable names
SHOW_PROGRESS=show_progress
PRE_UPDATE=pre_update.sh
POST_UPDATE=post_update.sh

WORK_DIR=/tmp
SRC_MOUNT_POINT=/srcmnt
DST_MOUNT_POINT=/mnt
VFAT_MOUNT_POINT=/vfatmnt

#TMP_LOG=$SRC_MOUNT_POINT/update.log


ANDROID_ROOT=$WORK_DIR/android
ANDROID_SYSTEM_ROOT=$ANDROID_ROOT/system
ANDROID_DATA_ROOT=$ANDROID_ROOT/data
CONTENT_DST_DIR=$DST_MOUNT_POINT
SDCARD_DST_DIR=$DST_MOUNT_POINT

# for crc'ing images
CRC_OK=false
UBI_CRC_OK=false
EXIST_AND_CRC_OK=false
EXIST_OK=false
TMP_FILE1=tmp1.img
TMP_FILE2=tmp2.img

# from boot command, figure out where update file is...
BOOT_COMMAND=$1
for i in $BOOT_COMMAND
do
	case $i in
		"--update_package=CACHE:update.zip"*)
			UPDATE_FILE=update.zip
			UPDATE_SRC_DEV=ubi:cache
			UPDATE_SRC_FS=ubifs
	   		;;

		"--update_package=SDESI:update.zip"*)
			UPDATE_FILE=update.golden.zip
			UPDATE_SRC_DEV=/dev/mmcblk0p1
			UPDATE_SRC_FS=auto
			UPDATE_ORIGIN=factory
	   		;;

		"--update_package=SDCARD:update.zip"*)
			UPDATE_FILE=update.zip
			UPDATE_SRC_DEV=/dev/mmcblk0p1
			UPDATE_SRC_FS=vfat
	   		;;

		"--update_package=USB:update.zip"*)
			UPDATE_FILE=update.zip
			UPDATE_SRC_DEV=/dev/sda
			UPDATE_SRC_FS=vfat
	   		;;

		"user"*)
			UPDATE_FILE=update.zip
			UPDATE_SRC_DEV=$2
			UPDATE_SRC_FS=auto
	   		;;

		*)
			echo "unknown command '$i' ..."
			reboot
			sleep 5
			exit 1
	   		;;
	esac
done

echo "fn=$UPDATE_FILE, dev=$UPDATE_SRC_DEV, fs=$UPDATE_SRC_FS" >> $TMP_LOG
echo "fn=$UPDATE_FILE, dev=$UPDATE_SRC_DEV, fs=$UPDATE_SRC_FS"

#   ---------------------------------------------
#   checks execution status of unzip.
#
#   arg1 : Execution status
#   arg2 : reboot (0), ignore (1)
#   arg3 : string for error reporting
#   usage: status $? 1
#   ---------------------------------------------
status ()
{
	if [ $1 -ne 0 ] ;
	then
		# an error occurred, handle it
		# "11" (no matching files found) is OK in some cases
		if [ $1 -eq 11 ];
		then
			if [ $2 -eq 1 ];
			then
				# ignore this error
				echo "file $3 not found ($1), but that's OK..."
				return
			fi
		fi

		# else we've got a problem
		echo "error extracting $3 ($1), rebooting..."
		umount $SRC_MOUNT_POINT
		umount $DST_MOUNT_POINT
		umount $VFAT_MOUNT_POINT
		reboot
		sleep 5
		exit 1
	fi
}

#   --------------------------------------------------
#   Raw read from SPI NOR flash
#
#   arg1 : mtd device
#   arg2 : offset within mtd device
#   arg3 : filename to read image in to
#   arg4 : size (in bytes) to read
#   usage: raw_nor_read /dev/mtd8 0x40000 tmp.img 12345
#   --------------------------------------------------
raw_nor_read ()
{
	# try mtd_debug instead
	#dd if=$1 of=$3 bs=$4 count=1 >> $TMP_LOG 2>&1

	# mtd_debu way
	mtd_debug read $1 $2 $4 $3
}

#   --------------------------------------------------
#   Raw write to SPI NOR flash
#
#   arg1 : mtd device
#   arg2 : offset within mtd device
#   arg3 : filename of image to burn
#   usage: raw_nor_write /dev/mtd8 0x40000 u-boot.bin
#   --------------------------------------------------
raw_nor_write ()
{
	# try mtd_debug instead
	#dd if=$3 of=$1 bs=64K >> $TMP_LOG 2>&1

	# mtd_debug way
	WR_LEN=`stat -c %s $3`
	mtd_debug write $1 $2 $WR_LEN $3
}

#   --------------------------------------------------
#   crc32 check on image
#   - requires that image have appended crc32
#
#   arg1 : filename of image to verify
#   usage: crc_check tmp.img
#   --------------------------------------------------
crc_check ()
{
	CRC=`ubicrc32 $1`
	if [ "$CRC" == "0x00000000" ];
	then
		CRC_OK=true
		echo "crc OK ($1)"
		echo "crc OK ($1)" >> $TMP_LOG
	else
		CRC_OK=false
		echo "crc FAILED ($1, $CRC)"
		echo "crc FAILED ($1, $CRC)" >> $TMP_LOG
	fi
}

#   --------------------------------------------------
#   check for existence of file and crc it
#   - requires that image have appended crc32
#
#   arg1 : filename of image to verify
#   usage: existence_and_crc_check tmp.img
#   --------------------------------------------------
existence_and_crc_check ()
{
	if [ -e $1 ];
	then
		EXIST_OK=true
		crc_check $1
		if [ "$CRC_OK" == "true" ];
		then
			EXIST_AND_CRC_OK=true
		else
			EXIST_AND_CRC_OK=false
		fi
	else
		EXIST_OK=false
	fi
}

#   --------------------------------------------------
#   Burns code to SPI NOR flash and verifies CRC32
#   - requires that image have appended crc32
#
#   arg1 : mtd device
#   arg2 : offset within mtd device
#   arg3 : number of blocks to erase
#   arg4 : filename of image to burn
#   usage: nor_burn /dev/mtd8 0x40000 4 u-boot.bin
#   --------------------------------------------------
nor_burn ()
{
	echo "nor_burn: entry: $1 $2 $3 $4"
	echo "nor_burn: entry: $1 $2 $3 $4" >> $TMP_LOG

	if [[ -e $1 && -e $4 ]];
	then
		SIZE=`stat -c %s $4`
		# start of erase/burn/verify loop
		for i in {1..3}
		do
			echo "NOR erasing ($i)..."
			echo "NOR erasing ($i)..." >> $TMP_LOG
			flash_erase $1 $2 $3
			echo "NOR burning ($i)..."
			echo "NOR burning ($i)..." >> $TMP_LOG
			raw_nor_write $1 $2 $4
			echo "NOR verifying ($i)..."
			echo "NOR verifying ($i)..." >> $TMP_LOG
			raw_nor_read $1 $2 $TMP_FILE1 $SIZE
			crc_check $TMP_FILE1
			rm $TMP_FILE1
			if [ "$CRC_OK" == "true" ];
			then
				break
			fi
		done
		rm $4
	else
		echo "nor_burn: $1 or $4 does not exist, skipping"
		echo "nor_burn: $1 or $4 does not exist, skipping" >> $TMP_LOG
	fi
}

#   --------------------------------------------------
#   Burns code to NAND flash
#
#   arg1 : mtd device
#   arg2 : offset within mtd device
#   arg3 : number of blocks to erase
#   arg4 : filename of image to burn
#   usage: nand_burn /dev/mtd5 0x0000 2 ramdisk.android.gz
#   --------------------------------------------------
nand_burn ()
{
	echo "nand_burn: entry: $1 $2 $3 $4"
	echo "nand_burn: entry: $1 $2 $3 $4" >> $TMP_LOG

	if [[ -e $1 && -e $4 ]];
	then
		SIZE=`stat -c %s $4`
		# start of erase/burn/verify loop
		for i in {1..3}
		do
			echo "NAND erasing ($i)..."
			echo "NAND erasing ($i)..." >> $TMP_LOG
			flash_erase $1 $2 $3
			echo "NAND burning ($i)..."
			echo "NAND burning ($i)..." >> $TMP_LOG
			nandwrite -s $2 -p $1 $4 >> $TMP_LOG 2>&1
			echo "NAND verifying ($i)..."
			echo "NAND verifying ($i)..." >> $TMP_LOG
			nanddump -o -b -l $SIZE -s $2 -f $TMP_FILE1 $1 >> $TMP_LOG 2>&1
			# truncate file to actual size, not 4k multiple of nanddump
			dd if=$TMP_FILE1 of=$TMP_FILE2 bs=$SIZE count=1
			crc_check $TMP_FILE2
			rm $TMP_FILE1
			rm $TMP_FILE2
			if [ "$CRC_OK" == "true" ];
			then
				break
			fi
		done
		rm $4
	else
		echo "nand_burn: $1 or $4 does not exist, skipping"
		echo "nand_burn: $1 or $4 does not exist, skipping" >> $TMP_LOG
	fi
}

#   --------------------------------------------------
#   Format NAND/mtd for ubi/ubifs use
#
#   arg1 : 
#   arg2 : 
#   usage: ubi_ubifs_prep arg1 arg2
#   --------------------------------------------------
ubi_ubifs_prep ()
{
	# must be detached first
	if [ -e /dev/ubi0 ];
	then
		ubidetach /dev/ubi_ctrl -m $SYSTEM_DEV_NUM
		sleep 1
	fi

	echo "UBI erasing $NUM_BLOCKS..."
	echo "UBI erasing $NUM_BLOCKS..." >> $TMP_LOG
	flash_erase $SYSTEM_MTD_DEVICE 0 $NUM_BLOCKS

	# ubiformat
	echo "UBI formatting..."
	echo "UBI formatting..." >> $TMP_LOG
	ubinize -o ubi.img -p $ERASE_SIZE -m $WRITE_SIZE android_ubinize.cfg
	sleep 1
	ubiformat $SYSTEM_MTD_DEVICE -s $WRITE_SIZE -f ubi.img
	sleep 1
	rm ubi.img

	# ubiattach
	sleep 1
	ubiattach /dev/ubi_ctrl -m $SYSTEM_DEV_NUM
	sleep 1

	# mkfs.ubifs
	###mkfs.ubifs -v -m $WRITE_SIZE -e $LEB_SIZE -c 712 /dev/ubi0_0 >> $TMP_LOG
	###mkfs.ubifs -v -m $WRITE_SIZE -e $LEB_SIZE -c 712 /dev/ubi0_1 >> $TMP_LOG
	###mkfs.ubifs -v -m $WRITE_SIZE -e $LEB_SIZE -c 712 /dev/ubi0_2 >> $TMP_LOG
	###mkfs.ubifs -v -m $WRITE_SIZE -e $LEB_SIZE -c 5645 /dev/ubi0_3 >> $TMP_LOG

}

#   --------------------------------------------------
#   Strips 4 bytes (crc) from a file
#
#   arg1 : filename to truncate
#   arg2 : num bytes to strip
#   usage: truncate android_ubi.img
#   --------------------------------------------------
truncate ()
{
	SIZE=`stat -c %s $1`
	FINAL_SIZE=$(($SIZE - $2))
	echo "size=$SIZE, final_size=$FINAL_SIZE"
	dd if=/dev/null of=$1 seek=1 bs=$FINAL_SIZE
}

#   --------------------------------------------------
#   Gets mtd info required for ubi/ubifs
#
#   arg1 : mtd device
#   usage: get_mtd_info /dev/mtd6
#   --------------------------------------------------
get_mtd_info ()
{
	mtd_debug info $1 > mtd.info
	SIZE_PREFIX="mtd.size = "
	ERASE_SIZE_PREFIX="mtd.erasesize = "
	WRITE_SIZE_PREFIX="mtd.writesize = "
	SUFFIX="(*"
	SIZE=0
	ERASE_SIZE=0
	WRITE_SIZE=0

	while read line
	do
		case $line in
			"$SIZE_PREFIX"*)
				SIZE=${line##$SIZE_PREFIX} 
				SIZE=${SIZE%%$SUFFIX} 
				;;

			"$ERASE_SIZE_PREFIX"*)
				ERASE_SIZE=${line##$ERASE_SIZE_PREFIX} 
				ERASE_SIZE=${ERASE_SIZE%%$SUFFIX} 
				;;

			"$WRITE_SIZE_PREFIX"*)
				WRITE_SIZE=${line##$WRITE_SIZE_PREFIX} 
				WRITE_SIZE=${WRITE_SIZE%%$SUFFIX} 
				;;
		esac
	done < mtd.info

	echo SIZE=$SIZE
	echo ERASE_SIZE=$ERASE_SIZE
	echo WRITE_SIZE=$WRITE_SIZE
}

#   --------------------------------------------------
#   config_android_fs:
#	enforces file/dir ownership and permissions based on
#	system/core/include/private/android_filesystem_config.h
#
#   arg1 : system or data zip file
#   usage: config_android_fs system1.zip
#   --------------------------------------------------
config_android_fs ()
{
	echo "performing android FS config on '$1'"
	echo "performing android FS config on '$1'" >> $TMP_LOG
	cd $ANDROID_ROOT
	unzip -Z1 ../$1 | ../fs_set_stats >> $TMP_LOG
	cd -
}


# initrd/.../edge_update already did this, but, just for good measure...
# lets put some fire behind that LCD to let the man know we're here (0-255)...
LCD_BRIGHTNESS=/sys/class/backlight/pwm-backlight.0/brightness
echo 225 > $LCD_BRIGHTNESS

# ensure mount points exist
mkdir -p $SRC_MOUNT_POINT
mkdir -p $DST_MOUNT_POINT
mkdir -p $VFAT_MOUNT_POINT
mkdir -p $ANDROID_SYSTEM_ROOT
mkdir -p $ANDROID_DATA_ROOT

# timestamp the start
BUILD_DATE_STR=`date`
echo "start:  $BUILD_DATE_STR ($UPDATE_ORIGIN)" >> $TMP_LOG
echo "start:  $BUILD_DATE_STR ($UPDATE_ORIGIN)"

# begin
echo "executing update script..."

# find and access update.zip file
#################################
mount -t $UPDATE_SRC_FS $UPDATE_SRC_DEV $SRC_MOUNT_POINT
if [ ! -e $SRC_MOUNT_POINT/$UPDATE_FILE ];
then
		echo "$UPDATE_FILE does not exist on $UPDATE_SRC_DEV"
		echo "$UPDATE_FILE does not exist on $UPDATE_SRC_DEV" >> $TMP_LOG
		rm -rf $SRC_MOUNT_POINT/$CMD_FILE_NAME
		umount $SRC_MOUNT_POINT
		reboot
		sleep 5
		exit 1
else
		echo "$UPDATE_FILE exists on $UPDATE_SRC_DEV"
		echo "$UPDATE_FILE exists on $UPDATE_SRC_DEV" >> $TMP_LOG
		# verification already done in edge_update
fi

# extract tools
###############
echo "unpacking tools in "`pwd`"..."
unzip -o $SRC_MOUNT_POINT/$UPDATE_FILE $TOOLS_ZIP >> $TMP_LOG 2>&1
status $? 0 $TOOLS_ZIP
unzip -o $TOOLS_ZIP >> $TMP_LOG 2>&1
status $? 0 $TOOLS_ZIP
rm $TOOLS_ZIP
sync
# allow any binaries in TOOLS_ZIP to override existing binaries
export PATH=$WORK_DIR:$PATH

# show progess if present (run in background)
#############################################
# first get 'flipped' switch from cmdline
FLIP_SWITCH=
FLIP_COMMAND="`cat /proc/cmdline`"
for i in $FLIP_COMMAND
do
	case $i in
		# handle "--flipped" (legacy) and "flipped" (current)
		"flipped=1")
			FLIP_SWITCH=-f
			;;
	esac
done

if [ -e $SHOW_PROGRESS ];
then
	echo "show progress..."
	chmod +x $SHOW_PROGRESS
	./$SHOW_PROGRESS -p img-update $FLIP_SWITCH &
fi

# check device type of target vs. update.zip
############################################
# build.prop is part of tools.zip, which is unzipped into /tmp
# check it against build.prop in ubi:system
# ALSO, use this as an opportunity to get other pertinent info from build.prop
UPDATE_DEVICE=unknown
NEW_VER=unknown
DEVICE_NAME="ro.product.device="
DESCRIPTION_NAME="ro.build.description="
BUILD_TYPE_NAME="ro.build.type="
BUILD_TYPE=none
while read line
do
	case $line in
		"$DEVICE_NAME"*)
			UPDATE_DEVICE=${line#*$DEVICE_NAME}
	   		;;

		"$DESCRIPTION_NAME"*)
			NEW_VER=${line#*$DESCRIPTION_NAME}
	   		;;

		"$BUILD_TYPE_NAME"*)
			BUILD_TYPE=${line#*$BUILD_TYPE_NAME}
	   		;;
	esac
done < /tmp/build.prop

# now get the target the update is for by either
# a) looking in usb:system/build.prop or
# b) reading boot version from /dev/mtd8

TARGET_DEVICE=unknown
OLD_VER=unknown

# make sure ubi device exists first before looking for build.prop
if [ -e /dev/ubi0 ];
then
	mount -t ubifs ubi:$SYSTEM_VOL_NAME $DST_MOUNT_POINT
	if [ -e $DST_MOUNT_POINT/build.prop ];
	then
		# use ubi:system/build.prop to ID hardware
		while read line
		do
			case $line in
				"$DEVICE_NAME"*)
					TARGET_DEVICE=${line#*$DEVICE_NAME}
	   				;;
	
				"$DESCRIPTION_NAME"*)
					OLD_VER=${line#*$DESCRIPTION_NAME}
	   				;;
			esac
		done < $DST_MOUNT_POINT/build.prop
	fi
	umount $DST_MOUNT_POINT
fi

if [ "$TARGET_DEVICE" == "unknown" ];
then
	# use boot code to ID hardware
	BOOT_ID=unknown
	raw_nor_read $BOOT_DEV 0x20 boot.id 4
	BOOT_ID=$(< boot.id)
	echo "boot_id = $BOOT_ID"
	if [ "$BOOT_ID" == "ES01" ];
	then
		TARGET_DEVICE=edge
	fi
	if [ "$BOOT_ID" == "ES02" ];
	then
		TARGET_DEVICE=edgejr
	fi
	echo "BOOT_ID = $BOOT_ID"
	echo "BOOT_ID = $BOOT_ID" >> $TMP_LOG
fi

echo "UPDATE_DEVICE = $UPDATE_DEVICE"
echo "UPDATE_DEVICE = $UPDATE_DEVICE" >> $TMP_LOG
echo "TARGET_DEVICE = $TARGET_DEVICE"
echo "TARGET_DEVICE = $TARGET_DEVICE" >> $TMP_LOG
echo "BUILD_TYPE =    $BUILD_TYPE"
echo "BUILD_TYPE =    $BUILD_TYPE" >> $TMP_LOG

echo "*********************************************************************"
echo " current: $OLD_VER"
echo " new:     $NEW_VER"

echo "*********************************************************************" >> $TMP_LOG
echo " current:  $OLD_VER" >> $TMP_LOG
echo " new:      $NEW_VER" >> $TMP_LOG

if [ "$UPDATE_DEVICE" != "$TARGET_DEVICE" ];
then
	echo " MISMATCH: not aborting because we expected that."
	echo " MISMATCH: not aborting because we expected that." >> $TMP_LOG
#	echo " MISMATCH: aborting update"
#	echo " MISMATCH: aborting update" >> $TMP_LOG
#	echo "*********************************************************************"
#	rm $SRC_MOUNT_POINT/$CMD_FILE_NAME
#	rm $SRC_MOUNT_POINT/$UPDATE_FILE
#	sync
#	umount $SRC_MOUNT_POINT
#	reboot
#	sleep 5
#	###exit 1
fi

echo "*********************************************************************"
echo "*********************************************************************" >> $TMP_LOG

# debug
#exit

# perform special pre-processing if applicable
##############################################
if [ -e $PRE_UPDATE ];
then
	echo "executing pre-processing..."
	echo "executing pre-processing..." >> $TMP_LOG
	chmod +x $PRE_UPDATE
	./$PRE_UPDATE
	echo "pre-processing done..."
	echo "pre-processing done..." >> $TMP_LOG
fi

# start unpacking and updating...

# first get NAND flash info
###########################
SIZE=0
ERASE_SIZE=0
WRITE_SIZE=0
NUM_BLOCKS=0
get_mtd_info /dev/mtd6
if [[ $SIZE == 0 || $ERASE_SIZE == 0 || $WRITE_SIZE == 0 ]];
then
	echo "*** We've got problems ***"
fi

# now use mtd info to calculate a few ubi-specific parameters
NUM_BLOCKS=$(($SIZE/$ERASE_SIZE))         
LEB_SIZE=$(($ERASE_SIZE - 2 * $WRITE_SIZE))         
RD_ANDROID_ERASE_BLKS=$(($RD_ANDROID_SIZE/$ERASE_SIZE))
RD_MAINT_ERASE_BLKS=$(($RD_MAINT_SIZE/$ERASE_SIZE))
echo "NUM_BLOCKS=$NUM_BLOCKS"
echo "LEB_SIZE=$LEB_SIZE"
echo "RD_ANDROID_ERASE_BLKS=$RD_ANDROID_ERASE_BLKS"
echo "RD_MAINT_ERASE_BLKS=$RD_MAINT_ERASE_BLKS"

# boot files
############
echo "extracting boot files..."
echo "extracting boot files..." >> $TMP_LOG
# get only desired zip from update.zip
unzip -o $SRC_MOUNT_POINT/$UPDATE_FILE $BOOT_ZIP >> $TMP_LOG 2>&1
status $? 1 $BOOT_ZIP
if [ -e $BOOT_ZIP ];
then
	echo "updating boot files..."
	echo "updating boot files..." >> $TMP_LOG
	unzip -o $BOOT_ZIP >> $TMP_LOG 2>&1
	status $? 0 $BOOT_ZIP

	# ntim
	for i in {1..3}
	do
		existence_and_crc_check $NTIM_FILENAME
		if [ "$EXIST_AND_CRC_OK" == "true" ];
		then
			nor_burn $NTIM_DEV $NTIM_OFFSET $NTIM_ERASE_BLKS $NTIM_FILENAME
			break
		fi
		if [ "$EXIST_OK" == "false" ];
		then
			echo "$NTIM_FILENAME not included in package"
			echo "$NTIM_FILENAME not included in package" >> $TMP_LOG
			break
		fi
		echo "crc check ($i) FAILED on file $NTIM_FILENAME"
		# unzip again and start over
		unzip -o $BOOT_ZIP >> $TMP_LOG 2>&1
		status $? 0 $BOOT_ZIP
	done
	###

	# bootloader
	for i in {1..3}
	do
		existence_and_crc_check $LOADER_FILENAME
		if [ "$EXIST_AND_CRC_OK" == "true" ];
		then
			nor_burn $LOADER_DEV $LOADER_OFFSET $LOADER_ERASE_BLKS $LOADER_FILENAME
			break
		fi
		if [ "$EXIST_OK" == "false" ];
		then
			echo "$LOADER_FILENAME not included in package"
			echo "$LOADER_FILENAME not included in package" >> $TMP_LOG
			break
		fi
		echo "crc check ($i) FAILED on file $LOADER_FILENAME"
		# unzip again and start over
		unzip -o $BOOT_ZIP >> $TMP_LOG 2>&1
		status $? 0 $BOOT_ZIP
	done
	###

	# u-boot
	for i in {1..3}
	do
		existence_and_crc_check $BOOT_FILENAME
		if [ "$EXIST_AND_CRC_OK" == "true" ];
		then
			nor_burn $BOOT_DEV $BOOT_OFFSET $BOOT_ERASE_BLKS $BOOT_FILENAME
			break
		fi
		if [ "$EXIST_OK" == "false" ];
		then
			echo "$BOOT_FILENAME not included in package"
			echo "$BOOT_FILENAME not included in package" >> $TMP_LOG
			break
		fi
		echo "crc check ($i) FAILED on file $BOOT_FILENAME"
		# unzip again and start over
		unzip -o $BOOT_ZIP >> $TMP_LOG 2>&1
		status $? 0 $BOOT_ZIP
	done
	###

	# kernel
	for i in {1..3}
	do
		existence_and_crc_check $KERNEL_FILENAME
		if [ "$EXIST_AND_CRC_OK" == "true" ];
		then
			nor_burn $KERNEL_DEV $KERNEL_OFFSET $KERNEL_ERASE_BLKS $KERNEL_FILENAME
			break
		fi
		if [ "$EXIST_OK" == "false" ];
		then
			echo "$KERNEL_FILENAME not included in package"
			echo "$KERNEL_FILENAME not included in package" >> $TMP_LOG
			break
		fi
		echo "crc check ($i) FAILED on file $KERNEL_FILENAME"
		# unzip again and start over
		unzip -o $BOOT_ZIP >> $TMP_LOG 2>&1
		status $? 0 $BOOT_ZIP
	done
	###

	rm $BOOT_ZIP
	sync
	sleep 1
else
	echo "$BOOT_ZIP not included in package."
	echo "$BOOT_ZIP not included in package." >> $TMP_LOG
fi

# root files
############
echo "extracting root files..."
echo "extracting root files..." >> $TMP_LOG
# get only desired zip from update.zip
unzip -o $SRC_MOUNT_POINT/$UPDATE_FILE $ROOT_ZIP >> $TMP_LOG 2>&1
status $? 1 $ROOT_ZIP
if [ -e $ROOT_ZIP ];
then
	echo "updating root files..."
	echo "updating root files..." >> $TMP_LOG
	unzip -o $ROOT_ZIP >> $TMP_LOG 2>&1
	status $? 0 $ROOT_ZIP
	# ramdisk.android
	for i in {1..3}
	do
		existence_and_crc_check $RD_ANDROID_FILENAME
		if [ "$EXIST_AND_CRC_OK" == "true" ];
		then
			nand_burn $RD_ANDROID_DEV $RD_ANDROID_OFFSET $RD_ANDROID_ERASE_BLKS $RD_ANDROID_FILENAME
			break
		fi
		if [ "$EXIST_OK" == "false" ];
		then
			echo "$RD_ANDROID_FILENAME not included in package"
			echo "$RD_ANDROID_FILENAME not included in package" >> $TMP_LOG
			break
		fi
		echo "crc check ($i) FAILED on file $RD_ANDROID_FILENAME"
		# unzip again and start over
		unzip -o $ROOT_ZIP >> $TMP_LOG 2>&1
		status $? 0 $ROOT_ZIP
	done
	###
	
	# ramdisk.maint
	for i in {1..3}
	do
		existence_and_crc_check $RD_MAINT_FILENAME
		if [ "$EXIST_AND_CRC_OK" == "true" ];
		then
			nand_burn $RD_MAINT_DEV $RD_MAINT_OFFSET $RD_MAINT_ERASE_BLKS $RD_MAINT_FILENAME
			break
		fi
		if [ "$EXIST_OK" == "false" ];
		then
			echo "$RD_MAINT_FILENAME not included in package"
			echo "$RD_MAINT_FILENAME not included in package" >> $TMP_LOG
			break
		fi
		echo "crc check ($i) FAILED on file $RD_MAINT_FILENAME"
		# unzip again and start over
		unzip -o $ROOT_ZIP >> $TMP_LOG 2>&1
		status $? 0 $ROOT_ZIP
	done
	###

	rm $ROOT_ZIP
	sync
	sleep 1
else
	echo "$ROOT_ZIP not included in package."
	echo "$ROOT_ZIP not included in package." >> $TMP_LOG
fi

# from here on, everything is on UBI device
###########################################

# Determine if this is a normal update, or a golden (destructive) update
# via the new update info file (optionally part of tools.zip)
# Sourcing this file allows local variables to be overridden.
########################################################################
echo "checking update info file..."
echo "checking update info file..." >> $TMP_LOG
if [ -e $UPDATE_INFO ];
then
	echo "using overrides from $UPDATE_INFO..."
	echo "using overrides from $UPDATE_INFO..." >> $TMP_LOG
	echo "******** $UPDATE_INFO contents: start ************"
	cat $UPDATE_INFO
	echo "******** $UPDATE_INFO contents: end   ************"
	# source them so they take affect locally
	. ./$UPDATE_INFO
fi

# GOLDEN (destructive) update
#############################
# This will completely erase and reformat the primary mtd device that holds
# all system volumes: android (/system, /data, /cache) and ESI (/storage).

if [ $UPDATE_TYPE == "golden" ];
then
	echo "*** GOLDEN update ***"
	echo "*** GOLDEN update ***" >> $TMP_LOG

	echo "reformatting and updating system image..."
	echo "reformatting and updating system image..." >> $TMP_LOG

	ubi_ubifs_prep
	###exit

	# now mount storage and create vfat image file
	mount -t ubifs ubi:$VFAT_VOL_NAME $DST_MOUNT_POINT >> $TMP_LOG 2>&1
	echo "creating vfat image..."
	echo "creating vfat image..." >> $TMP_LOG
	mkfs.vfat -F 32 -n eDGe -C $DST_MOUNT_POINT/$VFAT_IMG_FILENAME $VFAT_NUM_BLOCKS >> $TMP_LOG 2>&1
	sync
	umount $DST_MOUNT_POINT
	sleep 1
fi

# NORMAL (non-destructive) update
#################################

# FIRST, whack everything in system/xbin for release, i.e. "user" updates
# ...this is in an effort to harden against misuse and abuse
if [ "$BUILD_TYPE" == "user" ];
then
	echo "leaving xbin alone..."
	echo "leaving xbin alone..." >> $TMP_LOG
#	echo "removing eng binaries..."
#	echo "removing eng binaries..." >> $TMP_LOG
#	mount -t ubifs ubi:$SYSTEM_VOL_NAME $DST_MOUNT_POINT >> $TMP_LOG 2>&1
#	rm -rf $DST_MOUNT_POINT/xbin
#	sync
#	umount $DST_MOUNT_POINT
fi

# system partition
# MULTIPLE zips - be careful!!!
###############################
echo "extracting system files..."
echo "extracting system files..." >> $TMP_LOG
# get only desired zip from update.zip
unzip -o $SRC_MOUNT_POINT/$UPDATE_FILE $SYSTEM1_ZIP >> $TMP_LOG 2>&1
status $? 1 $SYSTEM1_ZIP
if [ -e $SYSTEM1_ZIP ];
then
	echo "updating system files..."
	echo "updating system files..." >> $TMP_LOG
	mount -t ubifs ubi:$SYSTEM_VOL_NAME $ANDROID_SYSTEM_ROOT >> $TMP_LOG 2>&1
	echo "removing old system files..."
	echo "removing old system files..."  >> $TMP_LOG
	rm -rf $ANDROID_SYSTEM_ROOT/*
	unzip -o $SYSTEM1_ZIP -d $ANDROID_ROOT >> $TMP_LOG 2>&1
	status $? 0 $SYSTEM1_ZIP
	### enforce android FS ownership/permissions here ###
	config_android_fs $SYSTEM1_ZIP
	rm $SYSTEM1_ZIP
	sync
	df -h $ANDROID_SYSTEM_ROOT >> $TMP_LOG
	umount $ANDROID_SYSTEM_ROOT
	sleep 1
else
	echo "$SYSTEM1_ZIP not included in package"
	echo "$SYSTEM1_ZIP not included in package" >> $TMP_LOG
fi

echo "extracting more system files..."
echo "extracting more system files..." >> $TMP_LOG
unzip -o $SRC_MOUNT_POINT/$UPDATE_FILE $SYSTEM2_ZIP >> $TMP_LOG 2>&1
status $? 1 $SYSTEM2_ZIP
if [ -e $SYSTEM2_ZIP ];
then
	echo "updating more system files..."
	echo "updating more system files..." >> $TMP_LOG
	mount -t ubifs ubi:$SYSTEM_VOL_NAME $ANDROID_SYSTEM_ROOT >> $TMP_LOG 2>&1
	unzip -o $SYSTEM2_ZIP -d $ANDROID_ROOT >> $TMP_LOG 2>&1
	status $? 0 $SYSTEM2_ZIP
	### enforce android FS ownership/permissions here ###
	config_android_fs $SYSTEM2_ZIP
	rm $SYSTEM2_ZIP
	sync
	df -h $ANDROID_SYSTEM_ROOT >> $TMP_LOG
	umount $ANDROID_SYSTEM_ROOT
	sleep 1
else
	echo "$SYSTEM2_ZIP not included in package"
	echo "$SYSTEM2_ZIP not included in package" >> $TMP_LOG
	# mount/unmount to handle any "unrecovered" condition
	mount -t ubifs ubi:$SYSTEM_VOL_NAME $DST_MOUNT_POINT >> $TMP_LOG 2>&1
	sleep 2
	umount $DST_MOUNT_POINT
fi

# data partition
################
echo "extracting data files..."
echo "extracting data files..." >> $TMP_LOG
# get only desired zip from update.zip
unzip -o $SRC_MOUNT_POINT/$UPDATE_FILE $DATA_ZIP >> $TMP_LOG 2>&1
status $? 1 $DATA_ZIP
if [ -e $DATA_ZIP ];
then
	echo "updating data files..."
	echo "updating data files..." >> $TMP_LOG
	mount -t ubifs ubi:$USERDATA_VOL_NAME $ANDROID_DATA_ROOT >> $TMP_LOG 2>&1
	unzip -o $DATA_ZIP -d $ANDROID_ROOT >> $TMP_LOG 2>&1
	status $? 0 $DATA_ZIP
	### enforce android FS ownership/permissions here ###
	config_android_fs $DATA_ZIP
	rm $DATA_ZIP
	sync
	df -h $ANDROID_DATA_ROOT >> $TMP_LOG
	umount $ANDROID_DATA_ROOT
	sleep 1
else
	echo "$DATA_ZIP not included in package"
	echo "$DATA_ZIP not included in package" >> $TMP_LOG
	# mount/unmount to handle any "unrecovered" condition
	mount -t ubifs ubi:$USERDATA_VOL_NAME $DST_MOUNT_POINT >> $TMP_LOG 2>&1
	sleep 2
	umount $DST_MOUNT_POINT
fi

# sdcard 
########
# At this time, "sdcard" is destined for wherever the library is located.
# This content is not put into the library at build time so it's not permanently
# in the OTA update.zip, but selectively added via add_content.

echo "extracting sdcard files..."
echo "extracting sdcard files..." >> $TMP_LOG
# get only desired zip from update.zip
unzip -o $SRC_MOUNT_POINT/$UPDATE_FILE $SDCARD_ZIP >> $TMP_LOG 2>&1
status $? 1 $SDCARD_ZIP
if [ -e $SDCARD_ZIP ];
then
	echo "determine library location..."
	echo "determine library location..." >> $TMP_LOG
	mount -t ubifs ubi:$USERDATA_VOL_NAME $DST_MOUNT_POINT >> $TMP_LOG 2>&1
	if [ -e $DST_MOUNT_POINT/property/persist.sys.esi_is_external ];
	then
		IS_EXTERNAL=`cat $DST_MOUNT_POINT/property/persist.sys.esi_is_external`
	else
		IS_EXTERNAL=0
	fi
	umount $DST_MOUNT_POINT
	if [ "$IS_EXTERNAL" == "1" ];
	then
		echo "library is external..."
		echo "library is external..." >> $TMP_LOG
		DST_DEV=/dev/mmcblk0p1 
	else
		echo "library is internal..."
		echo "library is internal..." >> $TMP_LOG
		mount -t ubifs ubi:$VFAT_VOL_NAME $VFAT_MOUNT_POINT >> $TMP_LOG 2>&1
		DST_DEV=$VFAT_MOUNT_POINT/$VFAT_IMG_FILENAME
	fi
	if [ -e $DST_DEV ];
	then
		echo "updating sdcard files..."
		echo "updating sdcard files..." >> $TMP_LOG
		mount -t vfat $DST_DEV $DST_MOUNT_POINT >> $TMP_LOG 2>&1
		unzip -o $SDCARD_ZIP -d $SDCARD_DST_DIR >> $TMP_LOG 2>&1
		status $? 0 $SDCARD_ZIP
		rm $SDCARD_ZIP
		sync
		df -h $DST_MOUNT_POINT >> $TMP_LOG
		umount $DST_MOUNT_POINT
	else
		echo "$DST_DEV does not exist..."
		echo "$DST_DEV does not exist..." >> $TMP_LOG
	fi
	if [ "$IS_EXTERNAL" != "1" ];
	then
		umount $VFAT_MOUNT_POINT >> $TMP_LOG 2>&1
	fi
	sleep 1
fi

# content partition
###################
# At this time, "content" is destined for /data
# This content is not put into /data at build time so it's not permanently
# in the OTA update.zip, but selectively added via add_content.
# If and when we have content destined for /fridge, duplicate this code

echo "extracting content files..."
echo "extracting content files..." >> $TMP_LOG
# get only desired zip from update.zip
unzip -o $SRC_MOUNT_POINT/$UPDATE_FILE $CONTENT_ZIP >> $TMP_LOG 2>&1
status $? 1 $CONTENT_ZIP
if [ -e $CONTENT_ZIP ];
then
	echo "updating content files..."
	echo "updating content files..." >> $TMP_LOG
	mount -t ubifs ubi:$CONTENT_VOL_NAME $DST_MOUNT_POINT >> $TMP_LOG 2>&1
	unzip -o $CONTENT_ZIP -d $CONTENT_DST_DIR >> $TMP_LOG 2>&1
	status $? 0 $CONTENT_ZIP
	rm $CONTENT_ZIP
	sync
	df -h $DST_MOUNT_POINT >> $TMP_LOG
	umount $DST_MOUNT_POINT
	sleep 1
fi


### special for messing with perms of files on /intstorage
mount -t ubifs ubi:$VFAT_VOL_NAME $DST_MOUNT_POINT >> $TMP_LOG 2>&1
echo "adjusting owner/group/permissions of /intstorage files..."
echo "adjusting owner/group/permissions of /intstorage files..." >> $TMP_LOG
### set permissons of the reader db file so everyone can read/write
chmod 666 $DST_MOUNT_POINT/.edge/.esi-data/.databases/library.db
sync
umount $DST_MOUNT_POINT

ALLMINE_ZIP=allmine.zip

echo "extracting allmine files..."
echo "extracting allmine files..." >> $TMP_LOG
unzip -o $SRC_MOUNT_POINT/$UPDATE_FILE $ALLMINE_ZIP >> $TMP_LOG 2>&1
status $? 1 $ALLMINE_ZIP
if [ -e $ALLMINE_ZIP ];
then
	echo "updating allmine files..."
	echo "updating allmine files..." >> $TMP_LOG
	mount -t ubifs ubi:$SYSTEM_VOL_NAME $ANDROID_SYSTEM_ROOT >> $TMP_LOG 2>&1
	unzip -o $ALLMINE_ZIP -d $ANDROID_ROOT >> $TMP_LOG 2>&1
	status $? 0 $ALLMINE_ZIP
	### enforce android FS ownership/permissions here ###
#	config_android_fs $SYSTEM2_ZIP
	rm $ALLMINE_ZIP
	sync
	df -h $ANDROID_SYSTEM_ROOT >> $TMP_LOG
	umount $ANDROID_SYSTEM_ROOT
	sleep 1
else
	echo "$ALLMINE_ZIP not included in package (wait, what?)"
	echo "$ALLMINE_ZIP not included in package (wait, what?)" >> $TMP_LOG
	# mount/unmount to handle any "unrecovered" condition
	mount -t ubifs ubi:$SYSTEM_VOL_NAME $DST_MOUNT_POINT >> $TMP_LOG 2>&1
	sleep 2
	umount $DST_MOUNT_POINT
fi

echo "creating serial number file"
echo "creating serial number file" >> $TMP_LOG
mount -t ubifs ubi:$SYSTEM_VOL_NAME $ANDROID_SYSTEM_ROOT >> $TMP_LOG 2>&1

#First we have to create the fb2 pipe to pump commands into
insmod fb_sys_fops.ko
insmod sysimgblt.ko
insmod sysfillrect.ko
insmod syscopyarea.ko
insmod ep3522fb.ko
#get the serial number from hardware
echo "SERIAL_NUMBER" >> /sys/class/graphics/fb2/properties
read -r SERIAL_NO < /sys/class/graphics/fb2/properties
echo "hardware reports serial number is: $SERIAL_NO"
echo "hardware reports serial number is: $SERIAL_NO" >> $TMP_LOG
#fix it to have the correct number of digits
SERIAL_NO=`expr substr $SERIAL_NO 1 8`0`expr substr $SERIAL_NO 9 5`
echo "           fixed serial number is: $SERIAL_NO"
echo "           fixed serial number is: $SERIAL_NO" >> $TMP_LOG
#save it to our cheaty file
echo $SERIAL_NO > $ANDROID_SYSTEM_ROOT/serial_number
sync
df -h $ANDROID_SYSTEM_ROOT >> $TMP_LOG
umount $ANDROID_SYSTEM_ROOT
sleep 1

#Changing the serial number breaks the library, so fix it.
echo "Fixing the Library"
echo "Fixing the Library" >> $TMP_LOG
mount -t ubifs ubi:$SYSTEM_VOL_NAME $ANDROID_SYSTEM_ROOT >> $TMP_LOG 2>&1
mkdir $ANDROID_ROOT/intstorage
mount -t ubifs ubi:$VFAT_VOL_NAME $ANDROID_ROOT/intstorage >> $TMP_LOG 2>&1
chroot /tmp/android /system/xbin/sqlite3 /intstorage/.edge/.esi-data/.databases/library.db "update esimeta set device='${SERIAL_NO}' where 1;" >>  $TMP_LOG 2>&1
sync
umount $ANDROID_ROOT/intstorage
sleep 1
umount $ANDROID_SYSTEM_ROOT
sleep 1
echo "extracting gapps files..."
echo "extracting gapps files..." >> $TMP_LOG
#find the google apps zip
for gappsfile in $SRC_MOUNT_POINT/gapps*
do
	if [ -f "$gappsfile" ];
	then
		GAPPS_ZIP=$gappsfile
		echo "found gapps file named $GAPPS_ZIP"
		echo "found gapps file named $GAPPS_ZIP" >> $TMP_LOG
		break
	fi

done
status $? 1 $GAPPS_ZIP
if [ -e $GAPPS_ZIP ];
then
	echo "updating gapps files..."
	echo "updating gapps files..." >> $TMP_LOG
	mount -t ubifs ubi:$SYSTEM_VOL_NAME $ANDROID_SYSTEM_ROOT >> $TMP_LOG 2>&1
	unzip -o $GAPPS_ZIP -d $ANDROID_ROOT >> $TMP_LOG 2>&1
	status $? 0 $GAPPS_ZIP
	sync
	df -h $ANDROID_SYSTEM_ROOT >> $TMP_LOG
	umount $ANDROID_SYSTEM_ROOT
	sleep 1
else
	echo "$GAPPS_ZIP not included in package"
	echo "$GAPPS_ZIP not included in package" >> $TMP_LOG
	# mount/unmount to handle any "unrecovered" condition
	mount -t ubifs ubi:$SYSTEM_VOL_NAME $DST_MOUNT_POINT >> $TMP_LOG 2>&1
	sleep 2
	umount $DST_MOUNT_POINT
fi




# all done
sync

if [ $UPDATE_TYPE != "golden" ];
then
	# removing update.zip and getting outta here
	############################################
	echo "removing command file, leaving update..."
	echo "removing command file, leaving update..." >> $TMP_LOG
	if [ -e $SRC_MOUNT_POINT/$CMD_FILE_NAME ];
	then
		rm $SRC_MOUNT_POINT/$CMD_FILE_NAME
	fi

#	rm $SRC_MOUNT_POINT/$UPDATE_FILE
	sync
fi

sleep 1

# perform special post-processing if applicable
###############################################
if [ -e $POST_UPDATE ];
then
	echo "executing post-processing..."
	echo "executing post-processing..." >> $TMP_LOG
	chmod +x $POST_UPDATE
	./$POST_UPDATE
	echo "post-processing done..."
	echo "post-processing done..." >> $TMP_LOG
fi

# timestamp the finish
BUILD_DATE_STR=`date`
echo "finish: $BUILD_DATE_STR" >> $TMP_LOG
echo "finish: $BUILD_DATE_STR"
echo "" >> $TMP_LOG

# saving partitioning log
#########################
mount -t ubifs ubi:$LOG_FILE_VOL_NAME $DST_MOUNT_POINT >> $TMP_LOG 2>&1
mkdir -p $DST_MOUNT_POINT/$LOG_FILE_PATH
cp $TMP_LOG $DST_MOUNT_POINT/$LOG_FILE_PATH/$LOG_FILE_NAME
sync
umount $DST_MOUNT_POINT
sleep 2

#also save it to the source mount point for safe-keeping
cp $TMP_LOG $SRC_MOUNT_POINT
umount $SRC_MOUNT_POINT
sleep 2

# detach UBI device
ubidetach /dev/ubi_ctrl -m $SYSTEM_DEV_NUM

sleep 1

if [ $UPDATE_ORIGIN == "factory" ];
then
	echo "updating $UPDATE_DEVICE EPD waveform file..."
	echo "updating $UPDATE_DEVICE EPD waveform file..." >> $TMP_LOG
	# automatically burn embedded EPD waveform to eeprom
	# NOTE: this burns the waveform contained in maint FS on golden SD card,
	#       NOT what was just burned as ramdisk.maint.gz
	#       (they should be identical)
	if [ "$UPDATE_DEVICE" == "edge" ];
	then
		cd /usr/esi/edge2
		./burn_edge2_epd_flash.sh ./*.epdflash >> $TMP_LOG 2>&1
	else
		cd /usr/esi/pocket
		./burn_pocket_epd_flash.sh ./*.epdflash >> $TMP_LOG 2>&1
	fi
fi

# done
echo "update done."

if [ $UPDATE_ORIGIN == "factory" ];
then
	# for factory, power down (otherwise there is no indicator we're done)
	echo "factory update: powering down..."
	poweroff
	sleep 20
else
	# for user, just reboot
	echo "user update: rebooting..."
	echo ""
	reboot
fi

