# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011, meme'
__docformat__ = 'restructuredtext en'

#####################################################################
# General utility functions
#####################################################################

import re, time, os, sys

from PyQt4 import QtGui
from PyQt4.Qt import Qt, QComboBox, QCheckBox, QTableWidgetItem, QDialog, QPixmap, QIcon

from calibre.gui2 import gprefs
from calibre.utils.config import config_dir
from calibre.constants import iswindows

plugin_name = None
plugin_icon_resources = {}

#####################################################################

# Debug print statements - not in Messages so it can be used at any time
def debug_print(text):
    from calibre.constants import DEBUG
    if DEBUG:
        time_string = time.strftime('%Y/%m/%d %H:%M:%S  ', time.gmtime())
        text = re.sub('<P>|[\n\r]','\n' + time_string , text)
        print time_string + text

# Convert an array of strings to a comma separated string
def array_to_csv(array_data):
    return ', '.join(array_data)

# Convert a string of comma separated text to an array keeping leading/trailing space
def csv_to_array(data):
    return [ text.strip() for text in data.split(',') ]

# Grammar - choose singular or plural form of word/text based on number
def wording(singular, plural, number):
    return singular if number == 1 else plural

# Return the maximum len of all strings in a list
def max_string_len(stringlist):
    max = 0
    for i in stringlist:
        l = len(i)
        if l > max:
            max = l
    return max

def quote(text):
    if text:
        if text != text.strip():
            text = '"' + text + '"'
    else:
        text = '""'
    return text

# Check if the text matches any pattern in a list of patterns
def check_text_in_pattern_list(text, pattern_list, ignore_case):
    found = ''
    if len(pattern_list) > 0:
        for pattern in pattern_list:
            newpattern = pattern + '$'
            if len(pattern):
                try:
                    if ignore_case:
                        if re.match(newpattern, text, re.IGNORECASE):
                            found = pattern
                            break
                    else:
                        if re.match(newpattern, text):
                            found = pattern
                            break
                except:
                    debug_print('Unexpected error in pattern matching, matching "%s" against "%s"' % (pattern,text))
    return found

# Check if exact match or no-prefix match, ignoring case as appropriate
def check_name_in_list(name, text_list, ignore_case, ignore_prefix_suffix):
    found = []
    name_lower = name.lower()
    if len(text_list) > 0:
        for text in text_list:
            if ignore_case:
                cmptext = text.lower()
                cmpname = name_lower
            else:
                cmptext = text
                cmpname = name
            if ignore_prefix_suffix:
                try:
                    newtext = re.sub('^[^0-9a-zA-Z]*(' + re.escape(cmpname) + ')[^0-9a-zA-Z]*$',  r'\1', cmptext)
                    cmptext = newtext
                except:
                    debug_print('Unexpected error in searching for name, matching "%s" against "%s"' % (name, text))
            if cmptext == cmpname:
                found.append(text)
    return found

#####################################################################

def get_pref(unique_pref_name):
    return gprefs.get(unique_pref_name, None)

def set_pref(unique_pref_name, value):
    gprefs[unique_pref_name] = value

def set_plugin_icon_resources(name, resources):
    '''
    Set our global store of plugin name and icon resources for sharing between
    the InterfaceAction class which reads them and the ConfigWidget
    if needed for use on the customization dialog for this plugin.
    '''
    global plugin_icon_resources, plugin_name
    plugin_name = name
    plugin_icon_resources = resources

def get_icon(icon_name):
    '''
    Retrieve a QIcon for the named image from the zip file if it exists,
    or if not then from Calibre's image cache.
    '''
    if icon_name:
        pixmap = get_pixmap(icon_name)
        if pixmap is None:
            # Look in Calibre's cache for the icon
            return QIcon(I(icon_name))
        else:
            return QIcon(pixmap)
    return QIcon()

def get_pixmap(icon_name):
    '''
    Retrieve a QPixmap for the named image
    Any icons belonging to the plugin must be prefixed with 'images/'
    '''
    if not icon_name.startswith('images/'):
        # We know this is definitely not an icon belonging to this plugin
        pixmap = QPixmap()
        pixmap.load(I(icon_name))
        return pixmap

    # Check to see whether the icon exists as a Calibre resource
    # This will enable skinning if the user stores icons within a folder like:
    # ...\AppData\Roaming\calibre\resources\images\Plugin Name\
    if plugin_name:
        local_images_dir = get_local_images_dir(plugin_name)
        local_image_path = os.path.join(local_images_dir, icon_name.replace('images/', ''))
        if os.path.exists(local_image_path):
            pixmap = QPixmap()
            pixmap.load(local_image_path)
            return pixmap

    # As we did not find an icon elsewhere, look within our zip resources
    if icon_name in plugin_icon_resources:
        pixmap = QPixmap()
        pixmap.loadFromData(plugin_icon_resources[icon_name])
        return pixmap
    return None

def get_local_images_dir(subfolder=None):
    '''
    Returns a path to the user's local resources/images folder
    If a subfolder name parameter is specified, appends this to the path
    '''
    images_dir = os.path.join(config_dir, 'resources/images')
    if subfolder:
        images_dir = os.path.join(images_dir, subfolder)
    if iswindows:
        images_dir = os.path.normpath(images_dir)
    return images_dir

#####################################################################
# Dialog widgets shared by Customization and Edit tables
#####################################################################

class SizePersistedDialog(QDialog):
    '''
    This dialog is a base class for any dialogs that want their size/position
    restored when they are next opened.
    '''

    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)

    def resize_dialog(self):
        if self.geom is None:
            self.resize(self.sizeHint())
        else:
            self.restoreGeometry(self.geom)

    def dialog_closing(self, result):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom

class NoWheelComboBox(QComboBox):
    def wheelEvent (self, event):
        # Disable the mouse wheel on top of the combo box changing selection as plays havoc in a grid
        event.ignore()

# Define a combo field in the table
class ComboTableWidgetItem(NoWheelComboBox):

    def __init__(self, parent, selected_text, options=''):
        NoWheelComboBox.__init__(self, parent)
        self.populate_combo(selected_text, options)

    def populate_combo(self, selected_text, options):
        if options != '':
            self.addItems(options)
        idx = self.findText(selected_text)
        self.setCurrentIndex(idx)

# Define a checkbox field 
class CheckableBoxWidgetItem(QCheckBox):

    def __init__(self, checked, text, tooltip=''):
        QCheckBox.__init__(self, text)
        self.setToolTip(tooltip)
        if checked:
            self.setCheckState(Qt.Checked)
        else:
            self.setCheckState(Qt.Unchecked)

# Define a checkbox field in the table
class CheckableTableWidgetItem(QTableWidgetItem):

    def __init__(self, checked):
        QTableWidgetItem.__init__(self, '')
        self.setFlags(Qt.ItemFlags(Qt.ItemIsSelectable | Qt.ItemIsUserCheckable | Qt.ItemIsEnabled ))
        if checked:
            self.setCheckState(Qt.Checked)
        else:
            self.setCheckState(Qt.Unchecked)

class ReadOnlyTableWidgetItem(QTableWidgetItem):
    def __init__(self, text):
        QTableWidgetItem.__init__(self, text, QtGui.QTableWidgetItem.UserType)
        self.setFlags(Qt.ItemIsSelectable|Qt.ItemIsEnabled)

