# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011, meme'
__docformat__ = 'restructuredtext en'

#####################################################################
# Kindle Device related functions 
#####################################################################

import os, json, re
from calibre.constants import iswindows
from collections import defaultdict
from shutil import copyfile 

from calibre_plugins.kindle_collections.utilities import debug_print

PATH_CACHE_FILE = 'kindle_collections_path_cache.calibre'
COLLECTION_DIR = 'system'
COLLECTION_FILE = 'collections.json'
IGNORE_EXTENSIONS = ('apnx', 'mbp', 'mp3', 'tan', 'phl', 'ea', 'eal', 'json')
BOOK_DIRECTORIES = [ 'documents', 'audible' ]
READER_PREF_FILE = 'system/com.amazon.ebook.booklet.reader/reader.pref'
FONT_DIR = 'fonts'

kdevice = None

#####################################################################

def init(device_path):
    global kdevice
    kdevice = KindleDevice(device_path)
    debug_print('Set kdevice')

# Class for reading from/writing to the Kindle
class KindleDevice():

    def __init__(self, root):
        debug_print('BEGIN Initializing KindleDevice with %s' % root)

        self.collections = {}

        # Check for systems directory to be sure its mounted ok, and appears to be a Kindle
        self.root = ''
        if root:
            try:
                system_dir = os.path.join(root, COLLECTION_DIR)
                if os.path.isdir(system_dir):
                    self.root = root
                    debug_print('Kindle device found')
                else:
                    debug_print('Device does not seem to be a Kindle - no "%s" directory found' % system_dir)
            except:
                debug_print('Unexpected error in checking Kindle system directory: %s' % system_dir)
                pass
            self.collections = self.get_collections()
        debug_print('END Initializing KindleDevice')

    def get_fullpath(self, path):
         return os.path.normpath(os.path.join(self.root, path))

    def get_collections(self):
        debug_print('BEGIN getting collections from file')
        if not self.collections:
            self.collections = self.read_json_file(self.get_collections_filename())
        debug_print('END getting collections from file')
        return self.collections

    def get_compare_collections(self):
        collections = self.read_json_file(self.get_compare_filename())
        return collections

    def get_collections_filename(self):
        return os.path.join(self.root, COLLECTION_DIR, COLLECTION_FILE)

    def get_backup_filename(self):
        return self.get_collections_filename() + '.backup'

    def get_compare_filename(self):
        return self.get_collections_filename() + '.compare'

    def get_original_backup_filename(self):
        return self.get_collections_filename() + '.original'

    def get_tmp_backup_filename(self):
        return self.get_collections_filename() + '.tmpfile'

    # Return contents of a json formatted file - return None if no file/invalid data 
    def read_json_file(self, pathname):
        json_list = None
        try:
            debug_print('Reading file: %s' % pathname)
            if os.path.isfile(pathname):
                with open(pathname, 'rb') as f:
                    json_list = json.load(f)
            else:
                debug_print('No such file: %s' % pathname)
        except:
            debug_print('Unexpected error, unable to read file: %s' % pathname)
            pass
        return json_list

    # Return relative path of an absolute path on the Kindle
    def absolute_path(self, lpath):
        return os.path.join(self.root, lpath)

    # Return absolute path of a relative path on the Kindle
    def relative_path(self, path):
        return os.path.relpath(path, self.root)

    def restore_collections(self):
        json_file = self.get_collections_filename()
        backup_json_file = self.get_backup_filename()
        original_backup_json_file = self.get_original_backup_filename()
        tmp_json_file = self.get_tmp_backup_filename()
        debug_print('BEGIN Restore, trying to restore Kindle collections file: %s' % json_file)

        if not os.path.isfile(backup_json_file):
            if os.path.isfile(original_backup_json_file):
                debug_print('Using original backup collections file %s' % original_backup_json_file)
                try:
                   copyfile(original_backup_json_file, backup_json_file)
                except:
                    raise ValueError('Unable to copy original backup file.')
            else:
                raise ValueError('No backup collections file to restore.')
        else:
            debug_print('Using backup collections file %s' % backup_json_file)
        if os.path.isfile(backup_json_file):
            try:
                # Swap backup file and current file
                if os.path.isfile(json_file):
                    if os.path.isfile(tmp_json_file):
                        os.remove(tmp_json_file)
                    os.rename(json_file, tmp_json_file)
                os.rename(backup_json_file, json_file)
                if os.path.isfile(tmp_json_file):
                    os.rename(tmp_json_file, backup_json_file)
            except any as e:
                raise ValueError('Unable to restore backup file. %s' % str(e))
        else:
            # No file to restore
            raise ValueError('No backup collections file exists.')
        debug_print('END Restore')

    # Save collections list to Kindle file after making a backup of the existing file
    def save_collections(self, collections, compare=False):
        json_file = self.get_collections_filename()
        backup_json_file = self.get_backup_filename()
        original_backup_json_file = self.get_original_backup_filename()
        compare_json_file = self.get_compare_filename()
        debug_print('Trying to save collections to Kindle file: %s' % json_file)
        try:
            # Save and always keep the first json file that existed before this plugin ran
            if os.path.isfile(original_backup_json_file):
                # Rename might not work if destination exists, and we don't want to just delete backup
                if os.path.isfile(backup_json_file):
                    os.remove(backup_json_file)
                if os.path.isfile(json_file):
                    os.rename(json_file, backup_json_file)
                debug_print('Backup Kindle collections file created: %s' % backup_json_file)
            else:
                if os.path.isfile(json_file):
                    os.rename(json_file, original_backup_json_file)
                    debug_print('Original backup Kindle collections file created: %s' % original_backup_json_file)
        except:
                raise ValueError('No collections modified - unable to backup existing collection file: %s' % json_file)
        else:
            if not collections or len(collections) < 1:
                # No collections - delete the file
                try:
                    if os.path.isfile(json_file):
                        os.remove(json_file)
                except:
                    raise ValueError('Collections were empty - but unable to remove existing collection file')
            else:
                try:
                    cf = open(json_file,'wb')
                except:
                    raise ValueError('No collections modified - unable to open collections file for writing: %s' % json_file)
                else:
                    try:
                        json.dump(collections, cf)
                        cf.close
                        debug_print('Saved collections file')
                    except:
                        # Restore the backup of the file since unable to save new file
                        debug_print('\nRestoring backup Kindle collections file since new file could not be modified: %s' % backup_json_file)
                        try:
                            if os.path.isfile(json_file):
                                os.remove(json_file)
                            if os.path.isfile(backup_json_file):
                                os.rename(backup_json_file, json_file)
                            raise ValueError('No collections modified.  Unable to save data to collections file.  Restored original file from backup: %s' % json_file)
                        except:
                                raise ValueError('Problem saving collections file.  Unable to restore backup file: %s' % backup_json_file)

        # Copy the current file if this is Create Collections for later comparison
        if compare and os.path.isfile(json_file):
            try:
                f = open(compare_json_file,'wb')
                json.dump(collections, f)
                f.close
                debug_print('Saved copy of collections for later comparison')
            except:
                raise ValueError('Unable to copy json file for later comparison')

    def get_file_paths(self):
        debug_print('BEGIN getting file paths on Kindle')
        self.paths = []
        prefix_length = len(self.root) 
        for ebook_dir in BOOK_DIRECTORIES:
            ebook_dir = unicode(os.path.join(self.root, ebook_dir))
            debug_print('Checking Kindle directory for books: %s' % ebook_dir)
            if not os.path.exists(ebook_dir): 
                debug_print('Directory does not exist on Kindle: %s' % ebook_dir)
                continue
            for path, dirs, files in os.walk(ebook_dir):
                file_count = len(files) if files else 0
                dir_count = len(dirs) if dirs else 0
                debug_print('Checking path: %s, %d files, %d directories' % (path, file_count, dir_count))
                for filename in files:
                    if not filename.endswith(IGNORE_EXTENSIONS):
                        fullpath = os.path.normpath(os.path.join(path, filename))
                        if iswindows:
                            fullpath = fullpath.replace('\\', '/')
                        self.paths.append(fullpath[prefix_length:])
        debug_print('%d paths found' % len(self.paths))
        debug_print('END getting file paths on Kindle')
        return self.paths

    # Return the path to the reboot trigger file (SS/Fonts hack)
    def get_kindle_hack_reboot_trigger_file(self):
        fonts_hack_dir = os.path.join(self.root, 'linkfonts')
        ss_hack_dir = os.path.join(self.root, 'linkss')
        # If both hacks are installed, the Fonts hack watchdog will be the one used
        # (we're checking the pidfile, so this shouldn't really matter anymore, but let's continue doing it that way)
        reboot_file = ''
        if os.path.isfile(os.path.join(fonts_hack_dir, 'run', 'usb-watchdog.pid')):
            reboot_file = os.path.join(fonts_hack_dir, 'reboot')
        elif os.path.isfile(os.path.join(ss_hack_dir, 'run', 'usb-watchdog.pid')):
            reboot_file = os.path.join(ss_hack_dir, 'reboot')
        return reboot_file

    # Create the reboot trigger file for the SS/Fonts Hack - to force reboot when USB unplugged
    def create_kindle_hack_reboot_trigger_file(self):
        reboot_file = self.get_kindle_hack_reboot_trigger_file()
        if reboot_file:
            debug_print('Creating Kindle hack reboot trigger file: %s\n' % reboot_file)
            try:
                open(reboot_file, 'wb').close
            except:
                raise ValueError('Unable to create the Kindle hack reboot trigger file "%s"' % reboot_file)

    def get_path_cache_filename(self):
        return os.path.join(self.root, PATH_CACHE_FILE)

    def read_path_info_cache(self):
        return self.read_json_file(self.get_path_cache_filename())

    def get_cache_time(self):
        cache_time = 0
        try:
            cache_time = os.stat(PATH_CACHE_FILE).st_mtime
        except:
            pass
        return cache_time

    def save_path_info_cache(self, data):
        json_file = self.get_path_cache_filename()
        debug_print('Trying to save path cache file: %s' % json_file)
        try:
            cf = open(json_file,'wb')
        except:
            debug_print('Unable to open path cache file for writing')
        else:
            try:
                json.dump(data, cf)
                cf.close
                debug_print('Path cache file saved')
            except:
                debug_print('Unable to save path cache file: %s' % json_file)

    def get_reader_pref_filename(self):
        return os.path.join(self.root, READER_PREF_FILE)

    def read_reader_prefs(self):
        pref_file = self.get_reader_pref_filename()
        prefs = defaultdict()
        prefs['comments'] = ''
        try:
            for line in open(pref_file, 'rb'):
                if line[0] == '#':
                    prefs['comments'] += line
                else:
                    (variable, value) =  line.split('=', 1)
                    if variable:
                        prefs[variable] = unicode(value).strip()
                        debug_print('Preference %s = %s' % (variable, prefs[variable]))
                    else:
                        debug_print('Unexpected line format: %s' % line)
        except:
            debug_print('Unable to read pref file')
        return prefs
        
    def write_reader_prefs(self, prefs):
        pref_file = self.get_reader_pref_filename()
        write_okay = True
        try:
            pf = open(pref_file,'wb')
        except:
            debug_print('Unable to open reader pref file for writing')
            write_okay = False
        else:
            try:
                if 'comments' in prefs:
                    pf.write(prefs['comments'])
                for pref in sorted(prefs.keys()):
                    if pref != 'comments':
                        pf.write(pref + '=' + unicode(prefs[pref]) + '\n')
                pf.close() #            except:
            except:
                write_okay = False
                debug_print('Unable to write data to reader pref file')
        return write_okay


    def get_font_dirname(self):
        return os.path.join(self.root, FONT_DIR)

    def get_font_filenames(self):
        names = []
        font_dir = self.get_font_dirname()
        if os.path.isdir(font_dir):
            try:
                names = os.listdir(font_dir)
            except:
                debug_print('Unable to list font directory')
        else:
            debug_print('No custom fonts directory')
        file_names = []
        for name in names:
            if os.path.isfile(os.path.join(font_dir, name)):
                file_names.append(name)
        return file_names

    def update_font_files(self, fontname):
        debug_print('Updating alt font files from "%s"' % fontname)
        font_dir = self.get_font_dirname()
        update_okay = True
        try:
            os.chdir(font_dir)
        except:
            update_okay = False
            debug_print('Unable to change to font directory %s' % font_dir)
        else:
            for t in [ 'Regular', 'Bold', 'Italic', 'BoldItalic' ]:
                try:
                    copyfile(fontname + '-' + t + '.ttf', 'alt-' + t + '.ttf')
                except:
                    update_okay = False
                    debug_print('Unable to copy font file "%s"' % t)
        return update_okay
        

        









