#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011, Karl Weckworth <kweckwor@gmail.com>'
__docformat__ = 'restructuredtext en'

# History:
#  Version 1.0.0 Initial release
#  Version 1.0.1 Implement proxy wrapper to ensure modules only get loaded when required
#  Version 1.0.2 Fix regression introduced in 1.0.1 which resulted in JSON error when no prior
#                plugin configuration exists.
#  Version 1.0.3 Restructure code; add timestamp check on read/write of cache to make sure no
#                background updates have happened; change order of processing so that read/write
#                operation is not interrupted by user interaction (try to avoid stale data scenario);
#                fix date sort bug
#  Version 1.0.4 Use locale neutral version of strptime
#  Version 1.0.5 Account for GST when setting tz value for timezone offset
#  Version 1.0.6 Use locale neutral version of strftime
#  Version 1.1.0 Refactor code to use new Calibre plugin API; change configuration management to
#                use recommended method (include migration of old config as well)
#  Version 1.1.1 Use latest book timestamp/timezone as reference for timestamp update rather than 
#                computer timestamp/timezone to avoid problems due to device/computer discrepancies
#  Version 1.2.0 Enable management of unread collection and allow selection criteria based on library
#                query
#  Version 1.2.1 Change default query expression to something more reasonable and always valid
#  Version 1.2.2 Handle non-English user input (i.e. accented characters)
#  Version 1.2.3 Code cleanup. Normalize cache file path on read/write. Update toolbar restriction
#                to include Cover Browser context menu.
#  Version 1.2.4 Preserve "author sort" in Ubuntu. Add warning message when setting Unread List.
#  Version 1.2.5 Set Books List book timestamps to a date one year in the future to make them more
#                permanent
#  Version 1.2.6 Handle scenario where books are deleted (either through OS or Calibre), leading to
#                metadata cache records without any backing book files
#  Version 1.2.7 Handle case where user adds books through file system before Calibre connects and
#                before reader processes the book. Make library queries insensitive to library view
#                restrictions.

PLUGIN_NAME = _('Manage Sony x50 Reader Book List')

from calibre.customize import InterfaceActionBase

class ManageSonyX50BookList(InterfaceActionBase):
    name = PLUGIN_NAME
    author = 'Karl Weckworth'
    description = _('Manage the Sony x50 Reader book lists explicitly or based on metadata criteria.')
    version = (1, 2, 6)
    supported_platforms = ['windows', 'osx', 'linux']
    minimum_calibre_version = (0, 7, 53)
    can_be_disabled = True
    
    actual_plugin = 'calibre_plugins.ManageSonyX50BookList.action:ManageSonyX50BookListAction'

    def is_customizable(self):
        return True

    def config_widget(self):
        import json
        from calibre_plugins.ManageSonyX50BookList.action import getPluginCustomization
        from calibre_plugins.ManageSonyX50BookList.action import Config
        from calibre_plugins.ManageSonyX50BookList.view import ManageBookListConfigView
        from calibre_plugins.ManageSonyX50BookList.view import ManageBookListConfigController
        
        config = Config(getPluginCustomization(self))
        configWidget = ManageBookListConfigView()
        controller = ManageBookListConfigController(config.config, configWidget, config)
        configWidget.setController(controller) # Hang on to controller reference since widget is externally controlled
        
        return configWidget

    def save_settings(self, configWidget):
        from calibre_plugins.ManageSonyX50BookList.action import prefs
        from calibre_plugins.ManageSonyX50BookList.action import CONFIG_BOOKS_LIST_KEY
        from calibre_plugins.ManageSonyX50BookList.action import CONFIG_UNREAD_LIST_KEY

        prefs[CONFIG_BOOKS_LIST_KEY] = configWidget.controller.getBooksListConfig()
        prefs[CONFIG_UNREAD_LIST_KEY] = configWidget.controller.getUnreadListConfig()
    