/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License. 
 * To view a copy of this license, visit http://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter 
 * to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

/**
 * @package WT
 * @brief Package containing code from developerAntony Jordan.
 */
/**
 * @package WT.Kindle
 * @brief Package containing all code relating to the kindle.
 */
/**
 * @package WT.Kindle.Clippings
 * @brief Package containing the code for the ClippingsParser library.
 */
package WT.Kindle.Clippings;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;

/**
 * @file Utility.java
 * @brief This file holds the source code for the Utility class.
 * @date Created: 26/08/2011
 * @date Modified: 27/08/2011
 * @author Antony Jordan
 */
/**
 * @class Utility
 * @brief The class contains useful methods for the ClippingsParser library.
 * Methods are provided to parse the "My Clippings.txt" file found on kindle devices into a useful format using the Clipping and Book classes.
 * @date Created: 26/08/2011
 * @date Modified: 27/08/2011
 * @author Antony Jordan
 * @todo Add code to handle errors such as incorrect file layout.
 */
public class Utility {   
	/**
	 * @brief Converts a Calendar object into a date formatted string. 
	 * @param cal The calendar to convert.
	 * @return The String representation of the date.
	 */
	public static String calendarToDateString(Calendar cal){
    	int intDay =cal.get(Calendar.DAY_OF_MONTH);
    	int intMonth =cal.get(Calendar.MONTH);
    	int intYear = cal.get(Calendar.YEAR);
    	
    	intMonth++;
    	
    	String strDay;
    	String strMonth;

    	if(intDay < 10){
    		strDay = "0" + intDay;    		
    	}
    	else{
    		strDay = "" + intDay;	
    	}
    	if(intMonth < 10){
    		strMonth = "0" + intMonth;    		
    	}
    	else{
    		strMonth = "" + intMonth;	
    	}
    	
    	String retVal = strDay + "/" + strMonth + "/" + intYear;
    	
    	return retVal;
    }
	/**
	 * @brief Converts a Calendar object into a time formatted string. 
	 * @param cal The calendar to convert.
	 * @return The String representation of the time.
	 */
	public static String calendarToTimeString(Calendar cal){
    	int intHour = cal.get(Calendar.HOUR_OF_DAY);
    	int intMin = cal.get(Calendar.MINUTE);
    	
    	String strHour;
    	String strMin;
    	
    	if(intHour < 10){
    		strHour = "0" + intHour;
    	}
    	else{
    		strHour = "" + intHour;
    	}
    	if(intMin < 10){
    		strMin = "0" + intMin;
    	}
    	else{
    		strMin = "" + intMin;
    	}
    	
    	String retVal = strHour + ":" + strMin;
    	
    	return retVal;
    }
    
	/**
	 * @brief Parses a number of clippings contained in one string into an Array of CLipping. 
	 * @param input The string containing the multiple clippings.
	 * @return An Array of Clipping containing all of the clippings.
	 */
	public static Clipping[] parseClippings(String input) {
        ArrayList<Clipping> retVal = new ArrayList<Clipping>();

        String delimiter = "==========";

        input = input.substring(0,input.lastIndexOf(delimiter));
        String[] clippings = input.split(delimiter);

        for (String clipping : clippings) {
            Clipping clip = new Clipping(clipping.trim());
            retVal.add(clip);
        }

        return retVal.toArray(new Clipping[]{});
    }
	/**
	 * @brief Generates an ArrayList of Book from a string containing clippings.
	 * @param input The string containing the clippings.
	 * @return An ArrayList of Book containign the clippings.
	 */
	public static ArrayList<Book> generateBooksList(String input) {
        Clipping[] clippings = Utility.parseClippings(input);
        ArrayList<Book> books = new ArrayList<Book>();
                
        for (Clipping clipping : clippings) {
        	String clippingTitle = clipping.getTitle();
        	
            boolean added = false;
            
            for (Book book : books) {
                if (clippingTitle.equals(book.getTitle()))
                {
                    book.getClippings().add(clipping);
                    added = true;
                }
            }
            if (!added) {
                Book book = new Book(clippingTitle);
                book.getClippings().add(clipping);
                books.add(book);
            }
        }
        return books;
    }
    
	/**
	 * @brief Generates an XML representation of the clippings.
	 * @param input The string containing the clippings.
	 * @return A string of XML code containing the clippings
	 */
    public static String generateBooksXML(String input) {
        String xml = generateBooksXML(generateBooksList(input));

        return xml;
    }
	/**
	 * @brief Generates an XML representation of the clippings.
	 * @param input An ArrayList of Book.
	 * @return A string of XML code containing the clippings
	 */
    public static String generateBooksXML(ArrayList<Book> input) {
        ArrayList<Book> books = input;
        
        String xml;

        xml = "<Clippings>";

        for (Book book : books) {
           xml += book.toXML(1);
        }
        
        xml += "\n";
		xml += "</Clippings>";

        return xml;
    }

    /// @brief A boolean value to turn the debug method on and off.
	private static boolean debugON = false;
	/**
	 * @brief A method to output debug information in a uniform way.
	 * @param classs The class name that the message originates in.
	 * @param location The location within the class e.g. method name.
	 * @param variableName The name of the variable to display.
	 * @param variable The actual variable to show.
	 */
	public static void debug(String classs,String location,String variableName,String variable){
      	if(isDebugOn()){System.out.println("Debug("+classs+":"+location+"): "+variableName+" = "+variable);}
    }
	/**
	 * @brief A method to output debug information in a uniform way.
	 * @param classs The class name that the message originates in.
	 * @param location The location within the class e.g. method name.
	 * @param message The message to display.
	 */
    public static void debug(String classs,String location,String message){
       	if(isDebugOn()){System.out.println("Debug("+classs+":"+location+"): "+message);}
    }
    /**
     * @brief A method to check if the debug methods are turned on at runtime.
     * @return True for yes, false for no.
     */
	public static boolean isDebugOn(){return debugON;}
	
	/**
	 * @brief Loads a file from disk into a string.
	 * @param filePath The path to the file.
	 * @return The string holding the files contents.
	 */
	public static String fileToString(String filePath){
		byte[] buffer = new byte[(int) new File(filePath).length()];
		
		FileInputStream f;
		
		try {
			f = new FileInputStream(filePath);
			
			f.read(buffer);
			f.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
		
		Utility.debug("Utility", "fileToString", "buffer", new String(buffer));
		
		return new String(buffer);	
	}
}