/*
 * This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License. 
 * To view a copy of this license, visit http://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter 
 * to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 */

package WT.Kindle.Clippings;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;

/**
 * @file Clipping.java
 * @brief This file holds the source code for the Clipping class.
 * @date Created: 25/08/2011
 * @date Modified: 27/08/2011
 * @author Antony Jordan
 */
/**
 * @class Clipping
 * @brief The class stores the data for a single clipping from the "My Clippings.txt" of the kindle.
 * Methods are provided to parse each item in the clipping into individual objects.
 * @date Created: 25/08/2011
 * @date Modified: 27/08/2011
 * @author Antony Jordan
 * @todo Add code to handle errors such as incorrect file layout.
 */
public class Clipping
{
	/// @brief String to store the author of the clipping.
    private String author;
    /// @brief String to store the title of the clipping.
    private String title;
    /// @brief Integer array to hold the start and end location of the clipping.
    private int[] startEnd;
    /// @brief String to hold the type of the clipping.
    private String type;
    /// @brief Calendar to store the date and time of the clipping.
    private Calendar date;
    /// @brief String to hold the text of the clipping. Either the highlighted text or the user's note.
    private String text;

    /// @brief Constructor that initialises class to null values.
    public Clipping() {
    	author = null;
    	title = null;
    	startEnd = null;
    	type = null;
    	date = Calendar.getInstance();
    	text = null;
    }
    /**
     * @brief Constructor that takes a clipping as a string and parses each individual element into the correct fields. 
     * @param input The clipping as a string.
     */
    public Clipping(String input) {
        author = parseAuthor(input);
        title = parseTitle(input);
        startEnd = parseStartEnd(input);
        type = parseType(input);
        date = Calendar.getInstance();
        date = parseDateTime(input);
        text = parseText(input);
    }
    
    /**
     * @brief Parses the author of the clipping from the input string.
     * @param input The clipping as a string.
     * @return A string holding just the author of the clipping.
     */
    private String parseAuthor(String input) {
    	String temp = input.substring(0, input.indexOf("\n"));
            
        Utility.debug("Clipping", "parseAuthor", "temp",temp);
            
        Integer authorStartIndex = temp.lastIndexOf("(");
            
        Utility.debug("Clipping", "parseAuthor", "authorStartIndex",authorStartIndex.toString());
            
        authorStartIndex++;

        Utility.debug("Clipping", "parseAuthor", "authorStartIndex",authorStartIndex.toString());
           
        Integer authorEndIndex = (temp.lastIndexOf(")"));

        Utility.debug("Clipping", "parseAuthor", "authorEndIndex",authorEndIndex.toString());
            
        String author = temp.substring(authorStartIndex, authorEndIndex);

        author = author.trim();

        return author;
    }
    /**
     * @brief Parses the title of the clipping from the input string.
     * @param input The clipping as a string.
     * @return A string holding just the title of the clipping.
     */
    private String parseTitle(String input) {
        String temp = input.substring(0, input.indexOf("\n"));
        int authorStartIndex = temp.lastIndexOf("(");

        String title = temp.substring(0, authorStartIndex);

        title = title.trim();

        return title;
    }
    /**
     * @brief Parses the start and end locations of the clipping from the input string.
     * @param input The clipping as a string.
     * @return An array containing two values the first of which is the start location the last is the end location.
     */
    private int[] parseStartEnd(String input) {
        int[] retVal = new int[2];
        String startEnd = input.substring(input.indexOf("Loc.") + 5,input.indexOf("|"));
        startEnd = startEnd.trim();
        if (startEnd.contains("-")) {
            String[] temp = startEnd.split("-");
                
            temp[0] = temp[0].trim();
            temp[1] = temp[1].trim();
                
            retVal[0] = Integer.parseInt(temp[0]);
            retVal[1] = Integer.parseInt(temp[1]);
        }
        else
        {
            retVal[0] = retVal[1] = Integer.parseInt(startEnd);
        }
        return retVal;
    }
    /**
     * @brief Parses the type of the clipping from the input string.
     * @param input The clipping as a string.
     * @return A string holding just the type of the clipping.
     */
    private String parseType(String input) {
        String type = input.substring(input.indexOf("\n") + 3,input.indexOf("Loc."));

        type = type.trim();

        return type;
    }
    /**
     * @brief Parses the date and time of the clipping from the input string.
     * @param input The clipping as a string.
     * @return A calendar representation of the date and time.
     */
    private Calendar parseDateTime(String input) {
        String date = input.substring(input.indexOf("|"));
            
        Utility.debug("Clipping", "parseDateTime", "date", date.replace("\n", "#"));
           
        if(date.contains("\n")){
        	date = date.substring(0,date.indexOf("\n"));
        }
        date = date.replace("| Added on ", "");
         
        DateFormat formatter = new SimpleDateFormat("EEEE, MMMM dd, yyyy, hh:mm a");
          
        Calendar calendar = Calendar.getInstance();
        try {
        	calendar.setTime(formatter.parse(date));
		} 
		catch (ParseException e) {
			e.printStackTrace();
		}
            
        return calendar;
    }
    /**
     * @brief Parses the text of the clipping from the input string. That is the highlighted text or user inputed note.
     * @param input The clipping as a string.
     * @return A string holding just the text of the clipping.
     */
    private String parseText(String input) {
        String text = "";
        if(text.contains("\n")){text = input.substring((input.indexOf("\n", input.indexOf("\n") + 1)));}
        text = text.trim();

        return text;
    }

    /**
     * @brief Gets the author of the CLipping.
     * @return A string holding the author of the Clipping.
     */
	public String getAuthor(){return author;}
	/**
     * @brief Gets the title of the CLipping.
     * @return A string holding the title of the Clipping.
     */
	public String getTitle(){return title;}
	/**
     * @brief Gets the start and end locations of the CLipping.
     * @return An array containing two values the first of which is the start location the last is the end location.
     */
	public int[] getStartEnd(){return startEnd;}
	/**
     * @brief Gets the type of the CLipping.
     * @return A string holding the type of the Clipping.
     */
	public String getType(){return type;}
	/**
     * @brief Gets the date and time of the CLipping.
     * @return A calendar reprisenting the date and time of the Clipping.
     */
	public Calendar getDate(){return date;}
	/**
     * @brief Gets the text of the CLipping.
     * @return A string holding the text of the Clipping.
     */
	public String getText(){return text;}

	/**
	 * @brief Sets the author of the clipping.
	 * @param author The value to set.
	 */
	public void setAuthor(String author){this.author = author;}
	/**
	 * @brief Sets the title of the clipping.
	 * @param title The value to set.
	 */
	public void setTitle(String title){this.title = title;}
	/**
	 * @brief Sets the start and end location of the clipping.
	 * @param author The values to set as an Array[2].
	 */
	public void setStartEnd(int[] startEnd){this.startEnd = startEnd;}
	/**
	 * @brief Sets the type of the clipping.
	 * @param type The value to set.
	 */
	public void setType(String type){this.type = type;}
	/**
	 * @brief Sets the date of the clipping.
	 * @param date The value to set.
	 */
	public void setDate(Calendar date){this.date = date;}
	/**
	 * @brief Sets the text of the clipping.
	 * @param text The value to set.
	 */
	public void setText(String text){this.text = text;}
	
	/**
	 * @brief Outputs the current clipping in XML format.
	 * @param indentation The number of tabs to indent by.
	 * @return A string containing the XML code.
	 */
	public String toXML(int indentation){
		String indent = "";
		for(int i = 0;i<indentation;i++){indent += "\t";}
		
		String xml = "";
		
		xml += "\n" 	+ indent + "<" + type + ">";
		
		xml += "\n\t"	+ indent + "<Author>" 	+ author 								+ "</Author>";
        xml += "\n\t"	+ indent + "<Start>" 	+ startEnd[0] 							+ "</Start>";
        xml += "\n\t"	+ indent + "<End>" 		+ startEnd[1] 							+ "</End>";
        xml += "\n\t"	+ indent + "<Date>" 	+ Utility.calendarToDateString(date) 	+ "</Date>";
        xml += "\n\t"	+ indent + "<Time>" 	+ Utility.calendarToTimeString(date) 	+ "</Time>";
        xml += "\n\t"	+ indent + "<Text>" 	+ text 									+ "</Text>";
		
        xml += "\n"		+ indent + "</" + type + ">";
        return xml;
	}
	
	/**
	 * @brief Gives a string representation of the Clipping.
	 * @return A string containing all the data held in the class.
	 */
    public String toString() {
        String retval;

        retval = "Clipping[" + title            + ", " 
                             + author           + ", " 
                             + startEnd[0]      + ", " 
                             + startEnd[1]      + ", "
                             + type             + ", " 
                             + date.toString()  + ", " 
                             + text 
               + "]";

        return retval;
    }
}
