
use strict;
use warnings;

# If run outside the normal test suite on an ASCII platform, you can
# just create a latin1_to_native() function that just returns its
# inputs, because that's the only function used from test.pl
require "test.pl";

# Test qr/\X/ and the \p{} regular expression constructs.  This file is
# constructed by mktables from the tables it generates, so if mktables is
# buggy, this won't necessarily catch those bugs.  Tests are generated for all
# feasible properties; a few aren't currently feasible; see
# is_code_point_usable() in mktables for details.

# Standard test packages are not used because this manipulates SIG_WARN.  It
# exits 0 if every non-skipped test succeeded; -1 if any failed.

my $Tests = 0;
my $Fails = 0;

sub Expect($$$$) {
    my $expected = shift;
    my $ord = shift;
    my $regex  = shift;
    my $warning_type = shift;   # Type of warning message, like 'deprecated'
                                # or empty if none
    my $line   = (caller)[2];
    $ord = ord(latin1_to_native(chr($ord)));

    # Convert the code point to hex form
    my $string = sprintf "\"\\x{%04X}\"", $ord;

    my @tests = "";

    # The first time through, use all warnings.  If the input should generate
    # a warning, add another time through with them turned off
    push @tests, "no warnings '$warning_type';" if $warning_type;

    foreach my $no_warnings (@tests) {

        # Store any warning messages instead of outputting them
        local $SIG{__WARN__} = $SIG{__WARN__};
        my $warning_message;
        $SIG{__WARN__} = sub { $warning_message = $_[0] };

        $Tests++;

        # A string eval is needed because of the 'no warnings'.
        # Assumes no parens in the regular expression
        my $result = eval "$no_warnings
                            my \$RegObj = qr($regex);
                            $string =~ \$RegObj ? 1 : 0";
        if (not defined $result) {
            print "not ok $Tests - couldn't compile /$regex/; line $line: $@\n";
            $Fails++;
        }
        elsif ($result ^ $expected) {
            print "not ok $Tests - expected $expected but got $result for $string =~ qr/$regex/; line $line\n";
            $Fails++;
        }
        elsif ($warning_message) {
            if (! $warning_type || ($warning_type && $no_warnings)) {
                print "not ok $Tests - for qr/$regex/ did not expect warning message '$warning_message'; line $line\n";
                $Fails++;
            }
            else {
                print "ok $Tests - expected and got a warning message for qr/$regex/; line $line\n";
            }
        }
        elsif ($warning_type && ! $no_warnings) {
            print "not ok $Tests - for qr/$regex/ expected a $warning_type warning message, but got none; line $line\n";
            $Fails++;
        }
        else {
            print "ok $Tests - got $result for $string =~ qr/$regex/; line $line\n";
        }
    }
    return;
}

sub Error($) {
    my $regex  = shift;
    $Tests++;
    if (eval { 'x' =~ qr/$regex/; 1 }) {
        $Fails++;
        my $line = (caller)[2];
        print "not ok $Tests - re compiled ok, but expected error for qr/$regex/; line $line: $@\n";
    }
    else {
        my $line = (caller)[2];
        print "ok $Tests - got and expected error for qr/$regex/; line $line\n";
    }
    return;
}

# GCBTest.txt character that separates grapheme clusters
my $breakable_utf8 = my $breakable = chr(0xF7);
utf8::upgrade($breakable_utf8);

# GCBTest.txt character that indicates that the adjoining code points are part
# of the same grapheme cluster
my $nobreak_utf8 = my $nobreak = chr(0xD7);
utf8::upgrade($nobreak_utf8);

sub Test_X($) {
    # Test qr/\X/ matches.  The input is a line from auxiliary/GCBTest.txt
    # Each such line is a sequence of code points given by their hex numbers,
    # separated by the two characters defined just before this subroutine that
    # indicate that either there can or cannot be a break between the adjacent
    # code points.  If there isn't a break, that means the sequence forms an
    # extended grapheme cluster, which means that \X should match the whole
    # thing.  If there is a break, \X should stop there.  This is all
    # converted by this routine into a match:
    #   $string =~ /(\X)/,
    # Each \X should match the next cluster; and that is what is checked.

    my $template = shift;

    my $line   = (caller)[2];

    # The line contains characters above the ASCII range, but in Latin1.  It
    # may or may not be in utf8, and if it is, it may or may not know it.  So,
    # convert these characters to 8 bits.  If knows is in utf8, simply
    # downgrade.
    if (utf8::is_utf8($template)) {
        utf8::downgrade($template);
    } else {

        # Otherwise, if it is in utf8, but doesn't know it, the next lines
        # convert the two problematic characters to their 8-bit equivalents.
        # If it isn't in utf8, they don't harm anything.
        use bytes;
        $template =~ s/$nobreak_utf8/$nobreak/g;
        $template =~ s/$breakable_utf8/$breakable/g;
    }

    # Get rid of the leading and trailing breakables
    $template =~ s/^ \s* $breakable \s* //x;
    $template =~ s/ \s* $breakable \s* $ //x;

    # And no-breaks become just a space.
    $template =~ s/ \s* $nobreak \s* / /xg;

    # Split the input into segments that are breakable between them.
    my @segments = split /\s*$breakable\s*/, $template;

    my $string = "";
    my $display_string = "";
    my @should_match;
    my @should_display;

    # Convert the code point sequence in each segment into a Perl string of
    # characters
    foreach my $segment (@segments) {
        my @code_points = split /\s+/, $segment;
        my $this_string = "";
        my $this_display = "";
        foreach my $code_point (@code_points) {
            $this_string .= latin1_to_native(chr(hex $code_point));
            $this_display .= "\\x{$code_point}";
        }

        # The next cluster should match the string in this segment.
        push @should_match, $this_string;
        push @should_display, $this_display;
        $string .= $this_string;
        $display_string .= $this_display;
    }

    # If a string can be represented in both non-ut8 and utf8, test both cases
    UPGRADE:
    for my $to_upgrade (0 .. 1) {

        if ($to_upgrade) {

            # If already in utf8, would just be a repeat
            next UPGRADE if utf8::is_utf8($string);

            utf8::upgrade($string);
        }

        # Finally, do the \X match.
        my @matches = $string =~ /(\X)/g;

        # Look through each matched cluster to verify that it matches what we
        # expect.
        my $min = (@matches < @should_match) ? @matches : @should_match;
        for my $i (0 .. $min - 1) {
            $Tests++;
            if ($matches[$i] eq $should_match[$i]) {
                print "ok $Tests - ";
                if ($i == 0) {
                    print "In \"$display_string\" =~ /(\\X)/g, \\X #1";
                } else {
                    print "And \\X #", $i + 1,
                }
                print " correctly matched $should_display[$i]; line $line\n";
            } else {
                $matches[$i] = join("", map { sprintf "\\x{%04X}", $_ }
                                                    unpack("U*", $matches[$i]));
                print "not ok $Tests - In \"$display_string\" =~ /(\\X)/g, \\X #",
                    $i + 1,
                    " should have matched $should_display[$i]",
                    " but instead matched $matches[$i]",
                    ".  Abandoning rest of line $line\n";
                next UPGRADE;
            }
        }

        # And the number of matches should equal the number of expected matches.
        $Tests++;
        if (@matches == @should_match) {
            print "ok $Tests - Nothing was left over; line $line\n";
        } else {
            print "not ok $Tests - There were ", scalar @should_match, " \\X matches expected, but got ", scalar @matches, " instead; line $line\n";
        }
    }

    return;
}

sub Finished() {
    print "1..$Tests\n";
    exit($Fails ? -1 : 0);
}

Error('\p{Script=InGreek}');    # Bug #69018
Test_X("1100 $nobreak 1161");  # Bug #70940
Expect(0, 0x2028, '\p{Print}', ""); # Bug # 71722
Expect(0, 0x2029, '\p{Print}', ""); # Bug # 71722
Expect(1, 0xFF10, '\p{XDigit}', ""); # Bug # 71726
Error('\p{Other_Lowercase=:=	Yes}');
Error('\P{Other_Lowercase=:=	Yes}');
Expect(1, 696, '\p{Other_Lowercase=yes}', 'deprecated');
Expect(0, 696, '\p{^Other_Lowercase=yes}', 'deprecated');
Expect(0, 696, '\P{Other_Lowercase=yes}', 'deprecated');
Expect(1, 696, '\P{^Other_Lowercase=yes}', 'deprecated');
Expect(0, 687, '\p{Other_Lowercase=yes}', 'deprecated');
Expect(1, 687, '\p{^Other_Lowercase=yes}', 'deprecated');
Expect(1, 687, '\P{Other_Lowercase=yes}', 'deprecated');
Expect(0, 687, '\P{^Other_Lowercase=yes}', 'deprecated');
Expect(1, 696, '\p{Other_Lowercase=--Yes}', 'deprecated');
Expect(0, 696, '\p{^Other_Lowercase=--Yes}', 'deprecated');
Expect(0, 696, '\P{Other_Lowercase=--Yes}', 'deprecated');
Expect(1, 696, '\P{^Other_Lowercase=--Yes}', 'deprecated');
Expect(0, 687, '\p{Other_Lowercase=--Yes}', 'deprecated');
Expect(1, 687, '\p{^Other_Lowercase=--Yes}', 'deprecated');
Expect(1, 687, '\P{Other_Lowercase=--Yes}', 'deprecated');
Expect(0, 687, '\P{^Other_Lowercase=--Yes}', 'deprecated');
Error('\p{OLower=		Y:=}');
Error('\P{OLower=		Y:=}');
Expect(1, 696, '\p{OLower=y}', 'deprecated');
Expect(0, 696, '\p{^OLower=y}', 'deprecated');
Expect(0, 696, '\P{OLower=y}', 'deprecated');
Expect(1, 696, '\P{^OLower=y}', 'deprecated');
Expect(0, 687, '\p{OLower=y}', 'deprecated');
Expect(1, 687, '\p{^OLower=y}', 'deprecated');
Expect(1, 687, '\P{OLower=y}', 'deprecated');
Expect(0, 687, '\P{^OLower=y}', 'deprecated');
Expect(1, 696, '\p{OLower:	Y}', 'deprecated');
Expect(0, 696, '\p{^OLower:	Y}', 'deprecated');
Expect(0, 696, '\P{OLower:	Y}', 'deprecated');
Expect(1, 696, '\P{^OLower:	Y}', 'deprecated');
Expect(0, 687, '\p{OLower:	Y}', 'deprecated');
Expect(1, 687, '\p{^OLower:	Y}', 'deprecated');
Expect(1, 687, '\P{OLower:	Y}', 'deprecated');
Expect(0, 687, '\P{^OLower:	Y}', 'deprecated');
Error('\p{Is_Other_Lowercase=/a/		T}');
Error('\P{Is_Other_Lowercase=/a/		T}');
Expect(1, 696, '\p{Is_Other_Lowercase=t}', 'deprecated');
Expect(0, 696, '\p{^Is_Other_Lowercase=t}', 'deprecated');
Expect(0, 696, '\P{Is_Other_Lowercase=t}', 'deprecated');
Expect(1, 696, '\P{^Is_Other_Lowercase=t}', 'deprecated');
Expect(0, 687, '\p{Is_Other_Lowercase=t}', 'deprecated');
Expect(1, 687, '\p{^Is_Other_Lowercase=t}', 'deprecated');
Expect(1, 687, '\P{Is_Other_Lowercase=t}', 'deprecated');
Expect(0, 687, '\P{^Is_Other_Lowercase=t}', 'deprecated');
Expect(1, 696, '\p{Is_Other_Lowercase=-T}', 'deprecated');
Expect(0, 696, '\p{^Is_Other_Lowercase=-T}', 'deprecated');
Expect(0, 696, '\P{Is_Other_Lowercase=-T}', 'deprecated');
Expect(1, 696, '\P{^Is_Other_Lowercase=-T}', 'deprecated');
Expect(0, 687, '\p{Is_Other_Lowercase=-T}', 'deprecated');
Expect(1, 687, '\p{^Is_Other_Lowercase=-T}', 'deprecated');
Expect(1, 687, '\P{Is_Other_Lowercase=-T}', 'deprecated');
Expect(0, 687, '\P{^Is_Other_Lowercase=-T}', 'deprecated');
Error('\p{Is_OLower=		True/a/}');
Error('\P{Is_OLower=		True/a/}');
Expect(1, 696, '\p{Is_OLower=true}', 'deprecated');
Expect(0, 696, '\p{^Is_OLower=true}', 'deprecated');
Expect(0, 696, '\P{Is_OLower=true}', 'deprecated');
Expect(1, 696, '\P{^Is_OLower=true}', 'deprecated');
Expect(0, 687, '\p{Is_OLower=true}', 'deprecated');
Expect(1, 687, '\p{^Is_OLower=true}', 'deprecated');
Expect(1, 687, '\P{Is_OLower=true}', 'deprecated');
Expect(0, 687, '\P{^Is_OLower=true}', 'deprecated');
Expect(1, 696, '\p{Is_OLower=_-true}', 'deprecated');
Expect(0, 696, '\p{^Is_OLower=_-true}', 'deprecated');
Expect(0, 696, '\P{Is_OLower=_-true}', 'deprecated');
Expect(1, 696, '\P{^Is_OLower=_-true}', 'deprecated');
Expect(0, 687, '\p{Is_OLower=_-true}', 'deprecated');
Expect(1, 687, '\p{^Is_OLower=_-true}', 'deprecated');
Expect(1, 687, '\P{Is_OLower=_-true}', 'deprecated');
Expect(0, 687, '\P{^Is_OLower=_-true}', 'deprecated');
Error('\p{Other_Lowercase=_ NO:=}');
Error('\P{Other_Lowercase=_ NO:=}');
Expect(1, 687, '\p{Other_Lowercase=no}', 'deprecated');
Expect(0, 687, '\p{^Other_Lowercase=no}', 'deprecated');
Expect(0, 687, '\P{Other_Lowercase=no}', 'deprecated');
Expect(1, 687, '\P{^Other_Lowercase=no}', 'deprecated');
Expect(0, 696, '\p{Other_Lowercase=no}', 'deprecated');
Expect(1, 696, '\p{^Other_Lowercase=no}', 'deprecated');
Expect(1, 696, '\P{Other_Lowercase=no}', 'deprecated');
Expect(0, 696, '\P{^Other_Lowercase=no}', 'deprecated');
Expect(1, 687, '\p{Other_Lowercase=_	no}', 'deprecated');
Expect(0, 687, '\p{^Other_Lowercase=_	no}', 'deprecated');
Expect(0, 687, '\P{Other_Lowercase=_	no}', 'deprecated');
Expect(1, 687, '\P{^Other_Lowercase=_	no}', 'deprecated');
Expect(0, 696, '\p{Other_Lowercase=_	no}', 'deprecated');
Expect(1, 696, '\p{^Other_Lowercase=_	no}', 'deprecated');
Expect(1, 696, '\P{Other_Lowercase=_	no}', 'deprecated');
Expect(0, 696, '\P{^Other_Lowercase=_	no}', 'deprecated');
Error('\p{OLower:   /a/	N}');
Error('\P{OLower:   /a/	N}');
Expect(1, 687, '\p{OLower=n}', 'deprecated');
Expect(0, 687, '\p{^OLower=n}', 'deprecated');
Expect(0, 687, '\P{OLower=n}', 'deprecated');
Expect(1, 687, '\P{^OLower=n}', 'deprecated');
Expect(0, 696, '\p{OLower=n}', 'deprecated');
Expect(1, 696, '\p{^OLower=n}', 'deprecated');
Expect(1, 696, '\P{OLower=n}', 'deprecated');
Expect(0, 696, '\P{^OLower=n}', 'deprecated');
Expect(1, 687, '\p{OLower= n}', 'deprecated');
Expect(0, 687, '\p{^OLower= n}', 'deprecated');
Expect(0, 687, '\P{OLower= n}', 'deprecated');
Expect(1, 687, '\P{^OLower= n}', 'deprecated');
Expect(0, 696, '\p{OLower= n}', 'deprecated');
Expect(1, 696, '\p{^OLower= n}', 'deprecated');
Expect(1, 696, '\P{OLower= n}', 'deprecated');
Expect(0, 696, '\P{^OLower= n}', 'deprecated');
Error('\p{Is_Other_Lowercase::=-	F}');
Error('\P{Is_Other_Lowercase::=-	F}');
Expect(1, 687, '\p{Is_Other_Lowercase=f}', 'deprecated');
Expect(0, 687, '\p{^Is_Other_Lowercase=f}', 'deprecated');
Expect(0, 687, '\P{Is_Other_Lowercase=f}', 'deprecated');
Expect(1, 687, '\P{^Is_Other_Lowercase=f}', 'deprecated');
Expect(0, 696, '\p{Is_Other_Lowercase=f}', 'deprecated');
Expect(1, 696, '\p{^Is_Other_Lowercase=f}', 'deprecated');
Expect(1, 696, '\P{Is_Other_Lowercase=f}', 'deprecated');
Expect(0, 696, '\P{^Is_Other_Lowercase=f}', 'deprecated');
Expect(1, 687, '\p{Is_Other_Lowercase:  F}', 'deprecated');
Expect(0, 687, '\p{^Is_Other_Lowercase:  F}', 'deprecated');
Expect(0, 687, '\P{Is_Other_Lowercase:  F}', 'deprecated');
Expect(1, 687, '\P{^Is_Other_Lowercase:  F}', 'deprecated');
Expect(0, 696, '\p{Is_Other_Lowercase:  F}', 'deprecated');
Expect(1, 696, '\p{^Is_Other_Lowercase:  F}', 'deprecated');
Expect(1, 696, '\P{Is_Other_Lowercase:  F}', 'deprecated');
Expect(0, 696, '\P{^Is_Other_Lowercase:  F}', 'deprecated');
Error('\p{Is_OLower=	:=False}');
Error('\P{Is_OLower=	:=False}');
Expect(1, 687, '\p{Is_OLower=false}', 'deprecated');
Expect(0, 687, '\p{^Is_OLower=false}', 'deprecated');
Expect(0, 687, '\P{Is_OLower=false}', 'deprecated');
Expect(1, 687, '\P{^Is_OLower=false}', 'deprecated');
Expect(0, 696, '\p{Is_OLower=false}', 'deprecated');
Expect(1, 696, '\p{^Is_OLower=false}', 'deprecated');
Expect(1, 696, '\P{Is_OLower=false}', 'deprecated');
Expect(0, 696, '\P{^Is_OLower=false}', 'deprecated');
Expect(1, 687, '\p{Is_OLower=_	False}', 'deprecated');
Expect(0, 687, '\p{^Is_OLower=_	False}', 'deprecated');
Expect(0, 687, '\P{Is_OLower=_	False}', 'deprecated');
Expect(1, 687, '\P{^Is_OLower=_	False}', 'deprecated');
Expect(0, 696, '\p{Is_OLower=_	False}', 'deprecated');
Expect(1, 696, '\p{^Is_OLower=_	False}', 'deprecated');
Expect(1, 696, '\P{Is_OLower=_	False}', 'deprecated');
Expect(0, 696, '\P{^Is_OLower=_	False}', 'deprecated');
Error('\p{Variation_Selector= Yes:=}');
Error('\P{Variation_Selector= Yes:=}');
Expect(1, 6157, '\p{Variation_Selector=yes}', "");
Expect(0, 6157, '\p{^Variation_Selector=yes}', "");
Expect(0, 6157, '\P{Variation_Selector=yes}', "");
Expect(1, 6157, '\P{^Variation_Selector=yes}', "");
Expect(0, 6154, '\p{Variation_Selector=yes}', "");
Expect(1, 6154, '\p{^Variation_Selector=yes}', "");
Expect(1, 6154, '\P{Variation_Selector=yes}', "");
Expect(0, 6154, '\P{^Variation_Selector=yes}', "");
Expect(1, 6157, '\p{Variation_Selector:   -_Yes}', "");
Expect(0, 6157, '\p{^Variation_Selector:   -_Yes}', "");
Expect(0, 6157, '\P{Variation_Selector:   -_Yes}', "");
Expect(1, 6157, '\P{^Variation_Selector:   -_Yes}', "");
Expect(0, 6154, '\p{Variation_Selector:   -_Yes}', "");
Expect(1, 6154, '\p{^Variation_Selector:   -_Yes}', "");
Expect(1, 6154, '\P{Variation_Selector:   -_Yes}', "");
Expect(0, 6154, '\P{^Variation_Selector:   -_Yes}', "");
Error('\p{VS=-_y/a/}');
Error('\P{VS=-_y/a/}');
Expect(1, 6157, '\p{VS=y}', "");
Expect(0, 6157, '\p{^VS=y}', "");
Expect(0, 6157, '\P{VS=y}', "");
Expect(1, 6157, '\P{^VS=y}', "");
Expect(0, 6154, '\p{VS=y}', "");
Expect(1, 6154, '\p{^VS=y}', "");
Expect(1, 6154, '\P{VS=y}', "");
Expect(0, 6154, '\P{^VS=y}', "");
Expect(1, 6157, '\p{VS=	 Y}', "");
Expect(0, 6157, '\p{^VS=	 Y}', "");
Expect(0, 6157, '\P{VS=	 Y}', "");
Expect(1, 6157, '\P{^VS=	 Y}', "");
Expect(0, 6154, '\p{VS=	 Y}', "");
Expect(1, 6154, '\p{^VS=	 Y}', "");
Expect(1, 6154, '\P{VS=	 Y}', "");
Expect(0, 6154, '\P{^VS=	 Y}', "");
Error('\p{Is_Variation_Selector:	:=_T}');
Error('\P{Is_Variation_Selector:	:=_T}');
Expect(1, 6157, '\p{Is_Variation_Selector=t}', "");
Expect(0, 6157, '\p{^Is_Variation_Selector=t}', "");
Expect(0, 6157, '\P{Is_Variation_Selector=t}', "");
Expect(1, 6157, '\P{^Is_Variation_Selector=t}', "");
Expect(0, 6154, '\p{Is_Variation_Selector=t}', "");
Expect(1, 6154, '\p{^Is_Variation_Selector=t}', "");
Expect(1, 6154, '\P{Is_Variation_Selector=t}', "");
Expect(0, 6154, '\P{^Is_Variation_Selector=t}', "");
Expect(1, 6157, '\p{Is_Variation_Selector= T}', "");
Expect(0, 6157, '\p{^Is_Variation_Selector= T}', "");
Expect(0, 6157, '\P{Is_Variation_Selector= T}', "");
Expect(1, 6157, '\P{^Is_Variation_Selector= T}', "");
Expect(0, 6154, '\p{Is_Variation_Selector= T}', "");
Expect(1, 6154, '\p{^Is_Variation_Selector= T}', "");
Expect(1, 6154, '\P{Is_Variation_Selector= T}', "");
Expect(0, 6154, '\P{^Is_Variation_Selector= T}', "");
Error('\p{Is_VS=/a/--true}');
Error('\P{Is_VS=/a/--true}');
Expect(1, 6157, '\p{Is_VS=true}', "");
Expect(0, 6157, '\p{^Is_VS=true}', "");
Expect(0, 6157, '\P{Is_VS=true}', "");
Expect(1, 6157, '\P{^Is_VS=true}', "");
Expect(0, 6154, '\p{Is_VS=true}', "");
Expect(1, 6154, '\p{^Is_VS=true}', "");
Expect(1, 6154, '\P{Is_VS=true}', "");
Expect(0, 6154, '\P{^Is_VS=true}', "");
Expect(1, 6157, '\p{Is_VS: true}', "");
Expect(0, 6157, '\p{^Is_VS: true}', "");
Expect(0, 6157, '\P{Is_VS: true}', "");
Expect(1, 6157, '\P{^Is_VS: true}', "");
Expect(0, 6154, '\p{Is_VS: true}', "");
Expect(1, 6154, '\p{^Is_VS: true}', "");
Expect(1, 6154, '\P{Is_VS: true}', "");
Expect(0, 6154, '\P{^Is_VS: true}', "");
Error('\p{Variation_Selector= No/a/}');
Error('\P{Variation_Selector= No/a/}');
Expect(1, 6154, '\p{Variation_Selector=no}', "");
Expect(0, 6154, '\p{^Variation_Selector=no}', "");
Expect(0, 6154, '\P{Variation_Selector=no}', "");
Expect(1, 6154, '\P{^Variation_Selector=no}', "");
Expect(0, 6157, '\p{Variation_Selector=no}', "");
Expect(1, 6157, '\p{^Variation_Selector=no}', "");
Expect(1, 6157, '\P{Variation_Selector=no}', "");
Expect(0, 6157, '\P{^Variation_Selector=no}', "");
Expect(1, 6154, '\p{Variation_Selector=-	no}', "");
Expect(0, 6154, '\p{^Variation_Selector=-	no}', "");
Expect(0, 6154, '\P{Variation_Selector=-	no}', "");
Expect(1, 6154, '\P{^Variation_Selector=-	no}', "");
Expect(0, 6157, '\p{Variation_Selector=-	no}', "");
Expect(1, 6157, '\p{^Variation_Selector=-	no}', "");
Expect(1, 6157, '\P{Variation_Selector=-	no}', "");
Expect(0, 6157, '\P{^Variation_Selector=-	no}', "");
Error('\p{VS=:=N}');
Error('\P{VS=:=N}');
Expect(1, 6154, '\p{VS=n}', "");
Expect(0, 6154, '\p{^VS=n}', "");
Expect(0, 6154, '\P{VS=n}', "");
Expect(1, 6154, '\P{^VS=n}', "");
Expect(0, 6157, '\p{VS=n}', "");
Expect(1, 6157, '\p{^VS=n}', "");
Expect(1, 6157, '\P{VS=n}', "");
Expect(0, 6157, '\P{^VS=n}', "");
Expect(1, 6154, '\p{VS=__N}', "");
Expect(0, 6154, '\p{^VS=__N}', "");
Expect(0, 6154, '\P{VS=__N}', "");
Expect(1, 6154, '\P{^VS=__N}', "");
Expect(0, 6157, '\p{VS=__N}', "");
Expect(1, 6157, '\p{^VS=__N}', "");
Expect(1, 6157, '\P{VS=__N}', "");
Expect(0, 6157, '\P{^VS=__N}', "");
Error('\p{Is_Variation_Selector=	F/a/}');
Error('\P{Is_Variation_Selector=	F/a/}');
Expect(1, 6154, '\p{Is_Variation_Selector:f}', "");
Expect(0, 6154, '\p{^Is_Variation_Selector:f}', "");
Expect(0, 6154, '\P{Is_Variation_Selector:f}', "");
Expect(1, 6154, '\P{^Is_Variation_Selector:f}', "");
Expect(0, 6157, '\p{Is_Variation_Selector:f}', "");
Expect(1, 6157, '\p{^Is_Variation_Selector:f}', "");
Expect(1, 6157, '\P{Is_Variation_Selector:f}', "");
Expect(0, 6157, '\P{^Is_Variation_Selector:f}', "");
Expect(1, 6154, '\p{Is_Variation_Selector=_-F}', "");
Expect(0, 6154, '\p{^Is_Variation_Selector=_-F}', "");
Expect(0, 6154, '\P{Is_Variation_Selector=_-F}', "");
Expect(1, 6154, '\P{^Is_Variation_Selector=_-F}', "");
Expect(0, 6157, '\p{Is_Variation_Selector=_-F}', "");
Expect(1, 6157, '\p{^Is_Variation_Selector=_-F}', "");
Expect(1, 6157, '\P{Is_Variation_Selector=_-F}', "");
Expect(0, 6157, '\P{^Is_Variation_Selector=_-F}', "");
Error('\p{Is_VS=/a/		False}');
Error('\P{Is_VS=/a/		False}');
Expect(1, 6154, '\p{Is_VS=false}', "");
Expect(0, 6154, '\p{^Is_VS=false}', "");
Expect(0, 6154, '\P{Is_VS=false}', "");
Expect(1, 6154, '\P{^Is_VS=false}', "");
Expect(0, 6157, '\p{Is_VS=false}', "");
Expect(1, 6157, '\p{^Is_VS=false}', "");
Expect(1, 6157, '\P{Is_VS=false}', "");
Expect(0, 6157, '\P{^Is_VS=false}', "");
Expect(1, 6154, '\p{Is_VS=	-FALSE}', "");
Expect(0, 6154, '\p{^Is_VS=	-FALSE}', "");
Expect(0, 6154, '\P{Is_VS=	-FALSE}', "");
Expect(1, 6154, '\P{^Is_VS=	-FALSE}', "");
Expect(0, 6157, '\p{Is_VS=	-FALSE}', "");
Expect(1, 6157, '\p{^Is_VS=	-FALSE}', "");
Expect(1, 6157, '\P{Is_VS=	-FALSE}', "");
Expect(0, 6157, '\P{^Is_VS=	-FALSE}', "");
Error('\p{Full_Composition_Exclusion=-yes:=}');
Error('\P{Full_Composition_Exclusion=-yes:=}');
Expect(1, 833, '\p{Full_Composition_Exclusion=yes}', "");
Expect(0, 833, '\p{^Full_Composition_Exclusion=yes}', "");
Expect(0, 833, '\P{Full_Composition_Exclusion=yes}', "");
Expect(1, 833, '\P{^Full_Composition_Exclusion=yes}', "");
Expect(0, 831, '\p{Full_Composition_Exclusion=yes}', "");
Expect(1, 831, '\p{^Full_Composition_Exclusion=yes}', "");
Expect(1, 831, '\P{Full_Composition_Exclusion=yes}', "");
Expect(0, 831, '\P{^Full_Composition_Exclusion=yes}', "");
Expect(1, 833, '\p{Full_Composition_Exclusion= YES}', "");
Expect(0, 833, '\p{^Full_Composition_Exclusion= YES}', "");
Expect(0, 833, '\P{Full_Composition_Exclusion= YES}', "");
Expect(1, 833, '\P{^Full_Composition_Exclusion= YES}', "");
Expect(0, 831, '\p{Full_Composition_Exclusion= YES}', "");
Expect(1, 831, '\p{^Full_Composition_Exclusion= YES}', "");
Expect(1, 831, '\P{Full_Composition_Exclusion= YES}', "");
Expect(0, 831, '\P{^Full_Composition_Exclusion= YES}', "");
Error('\p{Comp_Ex=/a/	-Y}');
Error('\P{Comp_Ex=/a/	-Y}');
Expect(1, 833, '\p{Comp_Ex=y}', "");
Expect(0, 833, '\p{^Comp_Ex=y}', "");
Expect(0, 833, '\P{Comp_Ex=y}', "");
Expect(1, 833, '\P{^Comp_Ex=y}', "");
Expect(0, 831, '\p{Comp_Ex=y}', "");
Expect(1, 831, '\p{^Comp_Ex=y}', "");
Expect(1, 831, '\P{Comp_Ex=y}', "");
Expect(0, 831, '\P{^Comp_Ex=y}', "");
Expect(1, 833, '\p{Comp_Ex=-y}', "");
Expect(0, 833, '\p{^Comp_Ex=-y}', "");
Expect(0, 833, '\P{Comp_Ex=-y}', "");
Expect(1, 833, '\P{^Comp_Ex=-y}', "");
Expect(0, 831, '\p{Comp_Ex=-y}', "");
Expect(1, 831, '\p{^Comp_Ex=-y}', "");
Expect(1, 831, '\P{Comp_Ex=-y}', "");
Expect(0, 831, '\P{^Comp_Ex=-y}', "");
Error('\p{Is_Full_Composition_Exclusion=	 T:=}');
Error('\P{Is_Full_Composition_Exclusion=	 T:=}');
Expect(1, 833, '\p{Is_Full_Composition_Exclusion=t}', "");
Expect(0, 833, '\p{^Is_Full_Composition_Exclusion=t}', "");
Expect(0, 833, '\P{Is_Full_Composition_Exclusion=t}', "");
Expect(1, 833, '\P{^Is_Full_Composition_Exclusion=t}', "");
Expect(0, 831, '\p{Is_Full_Composition_Exclusion=t}', "");
Expect(1, 831, '\p{^Is_Full_Composition_Exclusion=t}', "");
Expect(1, 831, '\P{Is_Full_Composition_Exclusion=t}', "");
Expect(0, 831, '\P{^Is_Full_Composition_Exclusion=t}', "");
Expect(1, 833, '\p{Is_Full_Composition_Exclusion=  T}', "");
Expect(0, 833, '\p{^Is_Full_Composition_Exclusion=  T}', "");
Expect(0, 833, '\P{Is_Full_Composition_Exclusion=  T}', "");
Expect(1, 833, '\P{^Is_Full_Composition_Exclusion=  T}', "");
Expect(0, 831, '\p{Is_Full_Composition_Exclusion=  T}', "");
Expect(1, 831, '\p{^Is_Full_Composition_Exclusion=  T}', "");
Expect(1, 831, '\P{Is_Full_Composition_Exclusion=  T}', "");
Expect(0, 831, '\P{^Is_Full_Composition_Exclusion=  T}', "");
Error('\p{Is_Comp_Ex=-True/a/}');
Error('\P{Is_Comp_Ex=-True/a/}');
Expect(1, 833, '\p{Is_Comp_Ex=true}', "");
Expect(0, 833, '\p{^Is_Comp_Ex=true}', "");
Expect(0, 833, '\P{Is_Comp_Ex=true}', "");
Expect(1, 833, '\P{^Is_Comp_Ex=true}', "");
Expect(0, 831, '\p{Is_Comp_Ex=true}', "");
Expect(1, 831, '\p{^Is_Comp_Ex=true}', "");
Expect(1, 831, '\P{Is_Comp_Ex=true}', "");
Expect(0, 831, '\P{^Is_Comp_Ex=true}', "");
Expect(1, 833, '\p{Is_Comp_Ex=TRUE}', "");
Expect(0, 833, '\p{^Is_Comp_Ex=TRUE}', "");
Expect(0, 833, '\P{Is_Comp_Ex=TRUE}', "");
Expect(1, 833, '\P{^Is_Comp_Ex=TRUE}', "");
Expect(0, 831, '\p{Is_Comp_Ex=TRUE}', "");
Expect(1, 831, '\p{^Is_Comp_Ex=TRUE}', "");
Expect(1, 831, '\P{Is_Comp_Ex=TRUE}', "");
Expect(0, 831, '\P{^Is_Comp_Ex=TRUE}', "");
Error('\p{Full_Composition_Exclusion=--No/a/}');
Error('\P{Full_Composition_Exclusion=--No/a/}');
Expect(1, 831, '\p{Full_Composition_Exclusion=no}', "");
Expect(0, 831, '\p{^Full_Composition_Exclusion=no}', "");
Expect(0, 831, '\P{Full_Composition_Exclusion=no}', "");
Expect(1, 831, '\P{^Full_Composition_Exclusion=no}', "");
Expect(0, 833, '\p{Full_Composition_Exclusion=no}', "");
Expect(1, 833, '\p{^Full_Composition_Exclusion=no}', "");
Expect(1, 833, '\P{Full_Composition_Exclusion=no}', "");
Expect(0, 833, '\P{^Full_Composition_Exclusion=no}', "");
Expect(1, 831, '\p{Full_Composition_Exclusion=	_No}', "");
Expect(0, 831, '\p{^Full_Composition_Exclusion=	_No}', "");
Expect(0, 831, '\P{Full_Composition_Exclusion=	_No}', "");
Expect(1, 831, '\P{^Full_Composition_Exclusion=	_No}', "");
Expect(0, 833, '\p{Full_Composition_Exclusion=	_No}', "");
Expect(1, 833, '\p{^Full_Composition_Exclusion=	_No}', "");
Expect(1, 833, '\P{Full_Composition_Exclusion=	_No}', "");
Expect(0, 833, '\P{^Full_Composition_Exclusion=	_No}', "");
Error('\p{Comp_Ex=:= _N}');
Error('\P{Comp_Ex=:= _N}');
Expect(1, 831, '\p{Comp_Ex=n}', "");
Expect(0, 831, '\p{^Comp_Ex=n}', "");
Expect(0, 831, '\P{Comp_Ex=n}', "");
Expect(1, 831, '\P{^Comp_Ex=n}', "");
Expect(0, 833, '\p{Comp_Ex=n}', "");
Expect(1, 833, '\p{^Comp_Ex=n}', "");
Expect(1, 833, '\P{Comp_Ex=n}', "");
Expect(0, 833, '\P{^Comp_Ex=n}', "");
Expect(1, 831, '\p{Comp_Ex=_N}', "");
Expect(0, 831, '\p{^Comp_Ex=_N}', "");
Expect(0, 831, '\P{Comp_Ex=_N}', "");
Expect(1, 831, '\P{^Comp_Ex=_N}', "");
Expect(0, 833, '\p{Comp_Ex=_N}', "");
Expect(1, 833, '\p{^Comp_Ex=_N}', "");
Expect(1, 833, '\P{Comp_Ex=_N}', "");
Expect(0, 833, '\P{^Comp_Ex=_N}', "");
Error('\p{Is_Full_Composition_Exclusion= -F/a/}');
Error('\P{Is_Full_Composition_Exclusion= -F/a/}');
Expect(1, 831, '\p{Is_Full_Composition_Exclusion=f}', "");
Expect(0, 831, '\p{^Is_Full_Composition_Exclusion=f}', "");
Expect(0, 831, '\P{Is_Full_Composition_Exclusion=f}', "");
Expect(1, 831, '\P{^Is_Full_Composition_Exclusion=f}', "");
Expect(0, 833, '\p{Is_Full_Composition_Exclusion=f}', "");
Expect(1, 833, '\p{^Is_Full_Composition_Exclusion=f}', "");
Expect(1, 833, '\P{Is_Full_Composition_Exclusion=f}', "");
Expect(0, 833, '\P{^Is_Full_Composition_Exclusion=f}', "");
Expect(1, 831, '\p{Is_Full_Composition_Exclusion= -F}', "");
Expect(0, 831, '\p{^Is_Full_Composition_Exclusion= -F}', "");
Expect(0, 831, '\P{Is_Full_Composition_Exclusion= -F}', "");
Expect(1, 831, '\P{^Is_Full_Composition_Exclusion= -F}', "");
Expect(0, 833, '\p{Is_Full_Composition_Exclusion= -F}', "");
Expect(1, 833, '\p{^Is_Full_Composition_Exclusion= -F}', "");
Expect(1, 833, '\P{Is_Full_Composition_Exclusion= -F}', "");
Expect(0, 833, '\P{^Is_Full_Composition_Exclusion= -F}', "");
Error('\p{Is_Comp_Ex=		FALSE/a/}');
Error('\P{Is_Comp_Ex=		FALSE/a/}');
Expect(1, 831, '\p{Is_Comp_Ex=false}', "");
Expect(0, 831, '\p{^Is_Comp_Ex=false}', "");
Expect(0, 831, '\P{Is_Comp_Ex=false}', "");
Expect(1, 831, '\P{^Is_Comp_Ex=false}', "");
Expect(0, 833, '\p{Is_Comp_Ex=false}', "");
Expect(1, 833, '\p{^Is_Comp_Ex=false}', "");
Expect(1, 833, '\P{Is_Comp_Ex=false}', "");
Expect(0, 833, '\P{^Is_Comp_Ex=false}', "");
Expect(1, 831, '\p{Is_Comp_Ex=  false}', "");
Expect(0, 831, '\p{^Is_Comp_Ex=  false}', "");
Expect(0, 831, '\P{Is_Comp_Ex=  false}', "");
Expect(1, 831, '\P{^Is_Comp_Ex=  false}', "");
Expect(0, 833, '\p{Is_Comp_Ex=  false}', "");
Expect(1, 833, '\p{^Is_Comp_Ex=  false}', "");
Expect(1, 833, '\P{Is_Comp_Ex=  false}', "");
Expect(0, 833, '\P{^Is_Comp_Ex=  false}', "");
Error('\p{Changes_When_NFKC_Casefolded=		Yes:=}');
Error('\P{Changes_When_NFKC_Casefolded=		Yes:=}');
Expect(1, 90, '\p{Changes_When_NFKC_Casefolded=yes}', "");
Expect(0, 90, '\p{^Changes_When_NFKC_Casefolded=yes}', "");
Expect(0, 90, '\P{Changes_When_NFKC_Casefolded=yes}', "");
Expect(1, 90, '\P{^Changes_When_NFKC_Casefolded=yes}', "");
Expect(0, 64, '\p{Changes_When_NFKC_Casefolded=yes}', "");
Expect(1, 64, '\p{^Changes_When_NFKC_Casefolded=yes}', "");
Expect(1, 64, '\P{Changes_When_NFKC_Casefolded=yes}', "");
Expect(0, 64, '\P{^Changes_When_NFKC_Casefolded=yes}', "");
Expect(1, 90, '\p{Changes_When_NFKC_Casefolded=Yes}', "");
Expect(0, 90, '\p{^Changes_When_NFKC_Casefolded=Yes}', "");
Expect(0, 90, '\P{Changes_When_NFKC_Casefolded=Yes}', "");
Expect(1, 90, '\P{^Changes_When_NFKC_Casefolded=Yes}', "");
Expect(0, 64, '\p{Changes_When_NFKC_Casefolded=Yes}', "");
Expect(1, 64, '\p{^Changes_When_NFKC_Casefolded=Yes}', "");
Expect(1, 64, '\P{Changes_When_NFKC_Casefolded=Yes}', "");
Expect(0, 64, '\P{^Changes_When_NFKC_Casefolded=Yes}', "");
Error('\p{CWKCF= :=Y}');
Error('\P{CWKCF= :=Y}');
Expect(1, 90, '\p{CWKCF=y}', "");
Expect(0, 90, '\p{^CWKCF=y}', "");
Expect(0, 90, '\P{CWKCF=y}', "");
Expect(1, 90, '\P{^CWKCF=y}', "");
Expect(0, 64, '\p{CWKCF=y}', "");
Expect(1, 64, '\p{^CWKCF=y}', "");
Expect(1, 64, '\P{CWKCF=y}', "");
Expect(0, 64, '\P{^CWKCF=y}', "");
Expect(1, 90, '\p{CWKCF=  Y}', "");
Expect(0, 90, '\p{^CWKCF=  Y}', "");
Expect(0, 90, '\P{CWKCF=  Y}', "");
Expect(1, 90, '\P{^CWKCF=  Y}', "");
Expect(0, 64, '\p{CWKCF=  Y}', "");
Expect(1, 64, '\p{^CWKCF=  Y}', "");
Expect(1, 64, '\P{CWKCF=  Y}', "");
Expect(0, 64, '\P{^CWKCF=  Y}', "");
Error('\p{Is_Changes_When_NFKC_Casefolded=/a/-T}');
Error('\P{Is_Changes_When_NFKC_Casefolded=/a/-T}');
Expect(1, 90, '\p{Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(0, 90, '\p{^Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(0, 90, '\P{Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(1, 90, '\P{^Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(0, 64, '\p{Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(1, 64, '\p{^Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(1, 64, '\P{Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(0, 64, '\P{^Is_Changes_When_NFKC_Casefolded=t}', "");
Expect(1, 90, '\p{Is_Changes_When_NFKC_Casefolded=	 T}', "");
Expect(0, 90, '\p{^Is_Changes_When_NFKC_Casefolded=	 T}', "");
Expect(0, 90, '\P{Is_Changes_When_NFKC_Casefolded=	 T}', "");
Expect(1, 90, '\P{^Is_Changes_When_NFKC_Casefolded=	 T}', "");
Expect(0, 64, '\p{Is_Changes_When_NFKC_Casefolded=	 T}', "");
Expect(1, 64, '\p{^Is_Changes_When_NFKC_Casefolded=	 T}', "");
Expect(1, 64, '\P{Is_Changes_When_NFKC_Casefolded=	 T}', "");
Expect(0, 64, '\P{^Is_Changes_When_NFKC_Casefolded=	 T}', "");
Error('\p{Is_CWKCF: _-true/a/}');
Error('\P{Is_CWKCF: _-true/a/}');
Expect(1, 90, '\p{Is_CWKCF=true}', "");
Expect(0, 90, '\p{^Is_CWKCF=true}', "");
Expect(0, 90, '\P{Is_CWKCF=true}', "");
Expect(1, 90, '\P{^Is_CWKCF=true}', "");
Expect(0, 64, '\p{Is_CWKCF=true}', "");
Expect(1, 64, '\p{^Is_CWKCF=true}', "");
Expect(1, 64, '\P{Is_CWKCF=true}', "");
Expect(0, 64, '\P{^Is_CWKCF=true}', "");
Expect(1, 90, '\p{Is_CWKCF= 	True}', "");
Expect(0, 90, '\p{^Is_CWKCF= 	True}', "");
Expect(0, 90, '\P{Is_CWKCF= 	True}', "");
Expect(1, 90, '\P{^Is_CWKCF= 	True}', "");
Expect(0, 64, '\p{Is_CWKCF= 	True}', "");
Expect(1, 64, '\p{^Is_CWKCF= 	True}', "");
Expect(1, 64, '\P{Is_CWKCF= 	True}', "");
Expect(0, 64, '\P{^Is_CWKCF= 	True}', "");
Error('\p{Changes_When_NFKC_Casefolded:	/a/	-NO}');
Error('\P{Changes_When_NFKC_Casefolded:	/a/	-NO}');
Expect(1, 64, '\p{Changes_When_NFKC_Casefolded=no}', "");
Expect(0, 64, '\p{^Changes_When_NFKC_Casefolded=no}', "");
Expect(0, 64, '\P{Changes_When_NFKC_Casefolded=no}', "");
Expect(1, 64, '\P{^Changes_When_NFKC_Casefolded=no}', "");
Expect(0, 90, '\p{Changes_When_NFKC_Casefolded=no}', "");
Expect(1, 90, '\p{^Changes_When_NFKC_Casefolded=no}', "");
Expect(1, 90, '\P{Changes_When_NFKC_Casefolded=no}', "");
Expect(0, 90, '\P{^Changes_When_NFKC_Casefolded=no}', "");
Expect(1, 64, '\p{Changes_When_NFKC_Casefolded=  No}', "");
Expect(0, 64, '\p{^Changes_When_NFKC_Casefolded=  No}', "");
Expect(0, 64, '\P{Changes_When_NFKC_Casefolded=  No}', "");
Expect(1, 64, '\P{^Changes_When_NFKC_Casefolded=  No}', "");
Expect(0, 90, '\p{Changes_When_NFKC_Casefolded=  No}', "");
Expect(1, 90, '\p{^Changes_When_NFKC_Casefolded=  No}', "");
Expect(1, 90, '\P{Changes_When_NFKC_Casefolded=  No}', "");
Expect(0, 90, '\P{^Changes_When_NFKC_Casefolded=  No}', "");
Error('\p{CWKCF=-	n:=}');
Error('\P{CWKCF=-	n:=}');
Expect(1, 64, '\p{CWKCF=n}', "");
Expect(0, 64, '\p{^CWKCF=n}', "");
Expect(0, 64, '\P{CWKCF=n}', "");
Expect(1, 64, '\P{^CWKCF=n}', "");
Expect(0, 90, '\p{CWKCF=n}', "");
Expect(1, 90, '\p{^CWKCF=n}', "");
Expect(1, 90, '\P{CWKCF=n}', "");
Expect(0, 90, '\P{^CWKCF=n}', "");
Expect(1, 64, '\p{CWKCF=	_n}', "");
Expect(0, 64, '\p{^CWKCF=	_n}', "");
Expect(0, 64, '\P{CWKCF=	_n}', "");
Expect(1, 64, '\P{^CWKCF=	_n}', "");
Expect(0, 90, '\p{CWKCF=	_n}', "");
Expect(1, 90, '\p{^CWKCF=	_n}', "");
Expect(1, 90, '\P{CWKCF=	_n}', "");
Expect(0, 90, '\P{^CWKCF=	_n}', "");
Error('\p{Is_Changes_When_NFKC_Casefolded=_	F/a/}');
Error('\P{Is_Changes_When_NFKC_Casefolded=_	F/a/}');
Expect(1, 64, '\p{Is_Changes_When_NFKC_Casefolded=f}', "");
Expect(0, 64, '\p{^Is_Changes_When_NFKC_Casefolded=f}', "");
Expect(0, 64, '\P{Is_Changes_When_NFKC_Casefolded=f}', "");
Expect(1, 64, '\P{^Is_Changes_When_NFKC_Casefolded=f}', "");
Expect(0, 90, '\p{Is_Changes_When_NFKC_Casefolded=f}', "");
Expect(1, 90, '\p{^Is_Changes_When_NFKC_Casefolded=f}', "");
Expect(1, 90, '\P{Is_Changes_When_NFKC_Casefolded=f}', "");
Expect(0, 90, '\P{^Is_Changes_When_NFKC_Casefolded=f}', "");
Expect(1, 64, '\p{Is_Changes_When_NFKC_Casefolded=-f}', "");
Expect(0, 64, '\p{^Is_Changes_When_NFKC_Casefolded=-f}', "");
Expect(0, 64, '\P{Is_Changes_When_NFKC_Casefolded=-f}', "");
Expect(1, 64, '\P{^Is_Changes_When_NFKC_Casefolded=-f}', "");
Expect(0, 90, '\p{Is_Changes_When_NFKC_Casefolded=-f}', "");
Expect(1, 90, '\p{^Is_Changes_When_NFKC_Casefolded=-f}', "");
Expect(1, 90, '\P{Is_Changes_When_NFKC_Casefolded=-f}', "");
Expect(0, 90, '\P{^Is_Changes_When_NFKC_Casefolded=-f}', "");
Error('\p{Is_CWKCF=	/a/FALSE}');
Error('\P{Is_CWKCF=	/a/FALSE}');
Expect(1, 64, '\p{Is_CWKCF=false}', "");
Expect(0, 64, '\p{^Is_CWKCF=false}', "");
Expect(0, 64, '\P{Is_CWKCF=false}', "");
Expect(1, 64, '\P{^Is_CWKCF=false}', "");
Expect(0, 90, '\p{Is_CWKCF=false}', "");
Expect(1, 90, '\p{^Is_CWKCF=false}', "");
Expect(1, 90, '\P{Is_CWKCF=false}', "");
Expect(0, 90, '\P{^Is_CWKCF=false}', "");
Expect(1, 64, '\p{Is_CWKCF:	-_False}', "");
Expect(0, 64, '\p{^Is_CWKCF:	-_False}', "");
Expect(0, 64, '\P{Is_CWKCF:	-_False}', "");
Expect(1, 64, '\P{^Is_CWKCF:	-_False}', "");
Expect(0, 90, '\p{Is_CWKCF:	-_False}', "");
Expect(1, 90, '\p{^Is_CWKCF:	-_False}', "");
Expect(1, 90, '\P{Is_CWKCF:	-_False}', "");
Expect(0, 90, '\P{^Is_CWKCF:	-_False}', "");
Error('\p{NFKC_Quick_Check=:=-_YES}');
Error('\P{NFKC_Quick_Check=:=-_YES}');
Expect(1, 159, '\p{NFKC_Quick_Check:yes}', "");
Expect(0, 159, '\p{^NFKC_Quick_Check:yes}', "");
Expect(0, 159, '\P{NFKC_Quick_Check:yes}', "");
Expect(1, 159, '\P{^NFKC_Quick_Check:yes}', "");
Expect(0, 160, '\p{NFKC_Quick_Check:yes}', "");
Expect(1, 160, '\p{^NFKC_Quick_Check:yes}', "");
Expect(1, 160, '\P{NFKC_Quick_Check:yes}', "");
Expect(0, 160, '\P{^NFKC_Quick_Check:yes}', "");
Expect(1, 159, '\p{NFKC_Quick_Check=_	YES}', "");
Expect(0, 159, '\p{^NFKC_Quick_Check=_	YES}', "");
Expect(0, 159, '\P{NFKC_Quick_Check=_	YES}', "");
Expect(1, 159, '\P{^NFKC_Quick_Check=_	YES}', "");
Expect(0, 160, '\p{NFKC_Quick_Check=_	YES}', "");
Expect(1, 160, '\p{^NFKC_Quick_Check=_	YES}', "");
Expect(1, 160, '\P{NFKC_Quick_Check=_	YES}', "");
Expect(0, 160, '\P{^NFKC_Quick_Check=_	YES}', "");
Error('\p{NFKC_QC=_/a/Y}');
Error('\P{NFKC_QC=_/a/Y}');
Expect(1, 159, '\p{NFKC_QC=y}', "");
Expect(0, 159, '\p{^NFKC_QC=y}', "");
Expect(0, 159, '\P{NFKC_QC=y}', "");
Expect(1, 159, '\P{^NFKC_QC=y}', "");
Expect(0, 160, '\p{NFKC_QC=y}', "");
Expect(1, 160, '\p{^NFKC_QC=y}', "");
Expect(1, 160, '\P{NFKC_QC=y}', "");
Expect(0, 160, '\P{^NFKC_QC=y}', "");
Expect(1, 159, '\p{NFKC_QC=Y}', "");
Expect(0, 159, '\p{^NFKC_QC=Y}', "");
Expect(0, 159, '\P{NFKC_QC=Y}', "");
Expect(1, 159, '\P{^NFKC_QC=Y}', "");
Expect(0, 160, '\p{NFKC_QC=Y}', "");
Expect(1, 160, '\p{^NFKC_QC=Y}', "");
Expect(1, 160, '\P{NFKC_QC=Y}', "");
Expect(0, 160, '\P{^NFKC_QC=Y}', "");
Error('\p{Is_NFKC_Quick_Check=__YES:=}');
Error('\P{Is_NFKC_Quick_Check=__YES:=}');
Expect(1, 159, '\p{Is_NFKC_Quick_Check=yes}', "");
Expect(0, 159, '\p{^Is_NFKC_Quick_Check=yes}', "");
Expect(0, 159, '\P{Is_NFKC_Quick_Check=yes}', "");
Expect(1, 159, '\P{^Is_NFKC_Quick_Check=yes}', "");
Expect(0, 160, '\p{Is_NFKC_Quick_Check=yes}', "");
Expect(1, 160, '\p{^Is_NFKC_Quick_Check=yes}', "");
Expect(1, 160, '\P{Is_NFKC_Quick_Check=yes}', "");
Expect(0, 160, '\P{^Is_NFKC_Quick_Check=yes}', "");
Expect(1, 159, '\p{Is_NFKC_Quick_Check=_-Yes}', "");
Expect(0, 159, '\p{^Is_NFKC_Quick_Check=_-Yes}', "");
Expect(0, 159, '\P{Is_NFKC_Quick_Check=_-Yes}', "");
Expect(1, 159, '\P{^Is_NFKC_Quick_Check=_-Yes}', "");
Expect(0, 160, '\p{Is_NFKC_Quick_Check=_-Yes}', "");
Expect(1, 160, '\p{^Is_NFKC_Quick_Check=_-Yes}', "");
Expect(1, 160, '\P{Is_NFKC_Quick_Check=_-Yes}', "");
Expect(0, 160, '\P{^Is_NFKC_Quick_Check=_-Yes}', "");
Error('\p{Is_NFKC_QC=--y/a/}');
Error('\P{Is_NFKC_QC=--y/a/}');
Expect(1, 159, '\p{Is_NFKC_QC=y}', "");
Expect(0, 159, '\p{^Is_NFKC_QC=y}', "");
Expect(0, 159, '\P{Is_NFKC_QC=y}', "");
Expect(1, 159, '\P{^Is_NFKC_QC=y}', "");
Expect(0, 160, '\p{Is_NFKC_QC=y}', "");
Expect(1, 160, '\p{^Is_NFKC_QC=y}', "");
Expect(1, 160, '\P{Is_NFKC_QC=y}', "");
Expect(0, 160, '\P{^Is_NFKC_QC=y}', "");
Expect(1, 159, '\p{Is_NFKC_QC:    y}', "");
Expect(0, 159, '\p{^Is_NFKC_QC:    y}', "");
Expect(0, 159, '\P{Is_NFKC_QC:    y}', "");
Expect(1, 159, '\P{^Is_NFKC_QC:    y}', "");
Expect(0, 160, '\p{Is_NFKC_QC:    y}', "");
Expect(1, 160, '\p{^Is_NFKC_QC:    y}', "");
Expect(1, 160, '\P{Is_NFKC_QC:    y}', "");
Expect(0, 160, '\P{^Is_NFKC_QC:    y}', "");
Error('\p{NFKC_Quick_Check=:=_no}');
Error('\P{NFKC_Quick_Check=:=_no}');
Expect(1, 160, '\p{NFKC_Quick_Check:no}', "");
Expect(0, 160, '\p{^NFKC_Quick_Check:no}', "");
Expect(0, 160, '\P{NFKC_Quick_Check:no}', "");
Expect(1, 160, '\P{^NFKC_Quick_Check:no}', "");
Expect(0, 159, '\p{NFKC_Quick_Check:no}', "");
Expect(1, 159, '\p{^NFKC_Quick_Check:no}', "");
Expect(1, 159, '\P{NFKC_Quick_Check:no}', "");
Expect(0, 159, '\P{^NFKC_Quick_Check:no}', "");
Expect(1, 160, '\p{NFKC_Quick_Check=_	No}', "");
Expect(0, 160, '\p{^NFKC_Quick_Check=_	No}', "");
Expect(0, 160, '\P{NFKC_Quick_Check=_	No}', "");
Expect(1, 160, '\P{^NFKC_Quick_Check=_	No}', "");
Expect(0, 159, '\p{NFKC_Quick_Check=_	No}', "");
Expect(1, 159, '\p{^NFKC_Quick_Check=_	No}', "");
Expect(1, 159, '\P{NFKC_Quick_Check=_	No}', "");
Expect(0, 159, '\P{^NFKC_Quick_Check=_	No}', "");
Error('\p{NFKC_QC=--n:=}');
Error('\P{NFKC_QC=--n:=}');
Expect(1, 160, '\p{NFKC_QC=n}', "");
Expect(0, 160, '\p{^NFKC_QC=n}', "");
Expect(0, 160, '\P{NFKC_QC=n}', "");
Expect(1, 160, '\P{^NFKC_QC=n}', "");
Expect(0, 159, '\p{NFKC_QC=n}', "");
Expect(1, 159, '\p{^NFKC_QC=n}', "");
Expect(1, 159, '\P{NFKC_QC=n}', "");
Expect(0, 159, '\P{^NFKC_QC=n}', "");
Expect(1, 160, '\p{NFKC_QC=__n}', "");
Expect(0, 160, '\p{^NFKC_QC=__n}', "");
Expect(0, 160, '\P{NFKC_QC=__n}', "");
Expect(1, 160, '\P{^NFKC_QC=__n}', "");
Expect(0, 159, '\p{NFKC_QC=__n}', "");
Expect(1, 159, '\p{^NFKC_QC=__n}', "");
Expect(1, 159, '\P{NFKC_QC=__n}', "");
Expect(0, 159, '\P{^NFKC_QC=__n}', "");
Error('\p{Is_NFKC_Quick_Check=:=		No}');
Error('\P{Is_NFKC_Quick_Check=:=		No}');
Expect(1, 160, '\p{Is_NFKC_Quick_Check=no}', "");
Expect(0, 160, '\p{^Is_NFKC_Quick_Check=no}', "");
Expect(0, 160, '\P{Is_NFKC_Quick_Check=no}', "");
Expect(1, 160, '\P{^Is_NFKC_Quick_Check=no}', "");
Expect(0, 159, '\p{Is_NFKC_Quick_Check=no}', "");
Expect(1, 159, '\p{^Is_NFKC_Quick_Check=no}', "");
Expect(1, 159, '\P{Is_NFKC_Quick_Check=no}', "");
Expect(0, 159, '\P{^Is_NFKC_Quick_Check=no}', "");
Expect(1, 160, '\p{Is_NFKC_Quick_Check=__no}', "");
Expect(0, 160, '\p{^Is_NFKC_Quick_Check=__no}', "");
Expect(0, 160, '\P{Is_NFKC_Quick_Check=__no}', "");
Expect(1, 160, '\P{^Is_NFKC_Quick_Check=__no}', "");
Expect(0, 159, '\p{Is_NFKC_Quick_Check=__no}', "");
Expect(1, 159, '\p{^Is_NFKC_Quick_Check=__no}', "");
Expect(1, 159, '\P{Is_NFKC_Quick_Check=__no}', "");
Expect(0, 159, '\P{^Is_NFKC_Quick_Check=__no}', "");
Error('\p{Is_NFKC_QC=_:=n}');
Error('\P{Is_NFKC_QC=_:=n}');
Expect(1, 160, '\p{Is_NFKC_QC=n}', "");
Expect(0, 160, '\p{^Is_NFKC_QC=n}', "");
Expect(0, 160, '\P{Is_NFKC_QC=n}', "");
Expect(1, 160, '\P{^Is_NFKC_QC=n}', "");
Expect(0, 159, '\p{Is_NFKC_QC=n}', "");
Expect(1, 159, '\p{^Is_NFKC_QC=n}', "");
Expect(1, 159, '\P{Is_NFKC_QC=n}', "");
Expect(0, 159, '\P{^Is_NFKC_QC=n}', "");
Expect(1, 160, '\p{Is_NFKC_QC=_n}', "");
Expect(0, 160, '\p{^Is_NFKC_QC=_n}', "");
Expect(0, 160, '\P{Is_NFKC_QC=_n}', "");
Expect(1, 160, '\P{^Is_NFKC_QC=_n}', "");
Expect(0, 159, '\p{Is_NFKC_QC=_n}', "");
Expect(1, 159, '\p{^Is_NFKC_QC=_n}', "");
Expect(1, 159, '\P{Is_NFKC_QC=_n}', "");
Expect(0, 159, '\P{^Is_NFKC_QC=_n}', "");
Error('\p{NFKC_Quick_Check=/a/maybe}');
Error('\P{NFKC_Quick_Check=/a/maybe}');
Expect(1, 772, '\p{NFKC_Quick_Check=maybe}', "");
Expect(0, 772, '\p{^NFKC_Quick_Check=maybe}', "");
Expect(0, 772, '\P{NFKC_Quick_Check=maybe}', "");
Expect(1, 772, '\P{^NFKC_Quick_Check=maybe}', "");
Expect(0, 767, '\p{NFKC_Quick_Check=maybe}', "");
Expect(1, 767, '\p{^NFKC_Quick_Check=maybe}', "");
Expect(1, 767, '\P{NFKC_Quick_Check=maybe}', "");
Expect(0, 767, '\P{^NFKC_Quick_Check=maybe}', "");
Expect(1, 772, '\p{NFKC_Quick_Check=__Maybe}', "");
Expect(0, 772, '\p{^NFKC_Quick_Check=__Maybe}', "");
Expect(0, 772, '\P{NFKC_Quick_Check=__Maybe}', "");
Expect(1, 772, '\P{^NFKC_Quick_Check=__Maybe}', "");
Expect(0, 767, '\p{NFKC_Quick_Check=__Maybe}', "");
Expect(1, 767, '\p{^NFKC_Quick_Check=__Maybe}', "");
Expect(1, 767, '\P{NFKC_Quick_Check=__Maybe}', "");
Expect(0, 767, '\P{^NFKC_Quick_Check=__Maybe}', "");
Error('\p{NFKC_QC:/a/		M}');
Error('\P{NFKC_QC:/a/		M}');
Expect(1, 772, '\p{NFKC_QC=m}', "");
Expect(0, 772, '\p{^NFKC_QC=m}', "");
Expect(0, 772, '\P{NFKC_QC=m}', "");
Expect(1, 772, '\P{^NFKC_QC=m}', "");
Expect(0, 767, '\p{NFKC_QC=m}', "");
Expect(1, 767, '\p{^NFKC_QC=m}', "");
Expect(1, 767, '\P{NFKC_QC=m}', "");
Expect(0, 767, '\P{^NFKC_QC=m}', "");
Expect(1, 772, '\p{NFKC_QC:   _M}', "");
Expect(0, 772, '\p{^NFKC_QC:   _M}', "");
Expect(0, 772, '\P{NFKC_QC:   _M}', "");
Expect(1, 772, '\P{^NFKC_QC:   _M}', "");
Expect(0, 767, '\p{NFKC_QC:   _M}', "");
Expect(1, 767, '\p{^NFKC_QC:   _M}', "");
Expect(1, 767, '\P{NFKC_QC:   _M}', "");
Expect(0, 767, '\P{^NFKC_QC:   _M}', "");
Error('\p{Is_NFKC_Quick_Check=-/a/MAYBE}');
Error('\P{Is_NFKC_Quick_Check=-/a/MAYBE}');
Expect(1, 772, '\p{Is_NFKC_Quick_Check=maybe}', "");
Expect(0, 772, '\p{^Is_NFKC_Quick_Check=maybe}', "");
Expect(0, 772, '\P{Is_NFKC_Quick_Check=maybe}', "");
Expect(1, 772, '\P{^Is_NFKC_Quick_Check=maybe}', "");
Expect(0, 767, '\p{Is_NFKC_Quick_Check=maybe}', "");
Expect(1, 767, '\p{^Is_NFKC_Quick_Check=maybe}', "");
Expect(1, 767, '\P{Is_NFKC_Quick_Check=maybe}', "");
Expect(0, 767, '\P{^Is_NFKC_Quick_Check=maybe}', "");
Expect(1, 772, '\p{Is_NFKC_Quick_Check= -Maybe}', "");
Expect(0, 772, '\p{^Is_NFKC_Quick_Check= -Maybe}', "");
Expect(0, 772, '\P{Is_NFKC_Quick_Check= -Maybe}', "");
Expect(1, 772, '\P{^Is_NFKC_Quick_Check= -Maybe}', "");
Expect(0, 767, '\p{Is_NFKC_Quick_Check= -Maybe}', "");
Expect(1, 767, '\p{^Is_NFKC_Quick_Check= -Maybe}', "");
Expect(1, 767, '\P{Is_NFKC_Quick_Check= -Maybe}', "");
Expect(0, 767, '\P{^Is_NFKC_Quick_Check= -Maybe}', "");
Error('\p{Is_NFKC_QC=	 M/a/}');
Error('\P{Is_NFKC_QC=	 M/a/}');
Expect(1, 772, '\p{Is_NFKC_QC=m}', "");
Expect(0, 772, '\p{^Is_NFKC_QC=m}', "");
Expect(0, 772, '\P{Is_NFKC_QC=m}', "");
Expect(1, 772, '\P{^Is_NFKC_QC=m}', "");
Expect(0, 767, '\p{Is_NFKC_QC=m}', "");
Expect(1, 767, '\p{^Is_NFKC_QC=m}', "");
Expect(1, 767, '\P{Is_NFKC_QC=m}', "");
Expect(0, 767, '\P{^Is_NFKC_QC=m}', "");
Expect(1, 772, '\p{Is_NFKC_QC=- M}', "");
Expect(0, 772, '\p{^Is_NFKC_QC=- M}', "");
Expect(0, 772, '\P{Is_NFKC_QC=- M}', "");
Expect(1, 772, '\P{^Is_NFKC_QC=- M}', "");
Expect(0, 767, '\p{Is_NFKC_QC=- M}', "");
Expect(1, 767, '\p{^Is_NFKC_QC=- M}', "");
Expect(1, 767, '\P{Is_NFKC_QC=- M}', "");
Expect(0, 767, '\P{^Is_NFKC_QC=- M}', "");
Error('\p{Jamo_Short_Name=}');
Error('\P{Jamo_Short_Name=}');
Error('\p{JSN=}');
Error('\P{JSN=}');
Error('\p{Is_Jamo_Short_Name=}');
Error('\P{Is_Jamo_Short_Name=}');
Error('\p{Is_JSN:}');
Error('\P{Is_JSN:}');
Error('\p{Jamo_Short_Name=S}');
Error('\P{Jamo_Short_Name=S}');
Error('\p{JSN:	S}');
Error('\P{JSN:	S}');
Error('\p{Is_Jamo_Short_Name: S}');
Error('\P{Is_Jamo_Short_Name: S}');
Error('\p{Is_JSN=S}');
Error('\P{Is_JSN=S}');
Error('\p{Jamo_Short_Name=GG}');
Error('\P{Jamo_Short_Name=GG}');
Error('\p{JSN=GG}');
Error('\P{JSN=GG}');
Error('\p{Is_Jamo_Short_Name=GG}');
Error('\P{Is_Jamo_Short_Name=GG}');
Error('\p{Is_JSN=GG}');
Error('\P{Is_JSN=GG}');
Error('\p{Jamo_Short_Name=YU}');
Error('\P{Jamo_Short_Name=YU}');
Error('\p{JSN=YU}');
Error('\P{JSN=YU}');
Error('\p{Is_Jamo_Short_Name=YU}');
Error('\P{Is_Jamo_Short_Name=YU}');
Error('\p{Is_JSN=YU}');
Error('\P{Is_JSN=YU}');
Error('\p{Jamo_Short_Name=WE}');
Error('\P{Jamo_Short_Name=WE}');
Error('\p{JSN=WE}');
Error('\P{JSN=WE}');
Error('\p{Is_Jamo_Short_Name=WE}');
Error('\P{Is_Jamo_Short_Name=WE}');
Error('\p{Is_JSN=WE}');
Error('\P{Is_JSN=WE}');
Error('\p{Jamo_Short_Name=LM}');
Error('\P{Jamo_Short_Name=LM}');
Error('\p{JSN=LM}');
Error('\P{JSN=LM}');
Error('\p{Is_Jamo_Short_Name=LM}');
Error('\P{Is_Jamo_Short_Name=LM}');
Error('\p{Is_JSN=LM}');
Error('\P{Is_JSN=LM}');
Error('\p{Jamo_Short_Name=K}');
Error('\P{Jamo_Short_Name=K}');
Error('\p{JSN=K}');
Error('\P{JSN=K}');
Error('\p{Is_Jamo_Short_Name=K}');
Error('\P{Is_Jamo_Short_Name=K}');
Error('\p{Is_JSN=K}');
Error('\P{Is_JSN=K}');
Error('\p{Jamo_Short_Name:   BB}');
Error('\P{Jamo_Short_Name:   BB}');
Error('\p{JSN=BB}');
Error('\P{JSN=BB}');
Error('\p{Is_Jamo_Short_Name=BB}');
Error('\P{Is_Jamo_Short_Name=BB}');
Error('\p{Is_JSN=BB}');
Error('\P{Is_JSN=BB}');
Error('\p{Jamo_Short_Name=D}');
Error('\P{Jamo_Short_Name=D}');
Error('\p{JSN: D}');
Error('\P{JSN: D}');
Error('\p{Is_Jamo_Short_Name=D}');
Error('\P{Is_Jamo_Short_Name=D}');
Error('\p{Is_JSN=D}');
Error('\P{Is_JSN=D}');
Error('\p{Jamo_Short_Name:   E}');
Error('\P{Jamo_Short_Name:   E}');
Error('\p{JSN=E}');
Error('\P{JSN=E}');
Error('\p{Is_Jamo_Short_Name=E}');
Error('\P{Is_Jamo_Short_Name=E}');
Error('\p{Is_JSN=E}');
Error('\P{Is_JSN=E}');
Error('\p{Jamo_Short_Name=JJ}');
Error('\P{Jamo_Short_Name=JJ}');
Error('\p{JSN=JJ}');
Error('\P{JSN=JJ}');
Error('\p{Is_Jamo_Short_Name=JJ}');
Error('\P{Is_Jamo_Short_Name=JJ}');
Error('\p{Is_JSN=JJ}');
Error('\P{Is_JSN=JJ}');
Error('\p{Jamo_Short_Name=G}');
Error('\P{Jamo_Short_Name=G}');
Error('\p{JSN:	G}');
Error('\P{JSN:	G}');
Error('\p{Is_Jamo_Short_Name=G}');
Error('\P{Is_Jamo_Short_Name=G}');
Error('\p{Is_JSN=G}');
Error('\P{Is_JSN=G}');
Error('\p{Jamo_Short_Name=YA}');
Error('\P{Jamo_Short_Name=YA}');
Error('\p{JSN=YA}');
Error('\P{JSN=YA}');
Error('\p{Is_Jamo_Short_Name=YA}');
Error('\P{Is_Jamo_Short_Name=YA}');
Error('\p{Is_JSN=YA}');
Error('\P{Is_JSN=YA}');
Error('\p{Jamo_Short_Name=SS}');
Error('\P{Jamo_Short_Name=SS}');
Error('\p{JSN=SS}');
Error('\P{JSN=SS}');
Error('\p{Is_Jamo_Short_Name=SS}');
Error('\P{Is_Jamo_Short_Name=SS}');
Error('\p{Is_JSN=SS}');
Error('\P{Is_JSN=SS}');
Error('\p{Jamo_Short_Name=LB}');
Error('\P{Jamo_Short_Name=LB}');
Error('\p{JSN=LB}');
Error('\P{JSN=LB}');
Error('\p{Is_Jamo_Short_Name:LB}');
Error('\P{Is_Jamo_Short_Name:LB}');
Error('\p{Is_JSN:   LB}');
Error('\P{Is_JSN:   LB}');
Error('\p{Jamo_Short_Name=J}');
Error('\P{Jamo_Short_Name=J}');
Error('\p{JSN=J}');
Error('\P{JSN=J}');
Error('\p{Is_Jamo_Short_Name=J}');
Error('\P{Is_Jamo_Short_Name=J}');
Error('\p{Is_JSN:	J}');
Error('\P{Is_JSN:	J}');
Error('\p{Jamo_Short_Name=NH}');
Error('\P{Jamo_Short_Name=NH}');
Error('\p{JSN=NH}');
Error('\P{JSN=NH}');
Error('\p{Is_Jamo_Short_Name=NH}');
Error('\P{Is_Jamo_Short_Name=NH}');
Error('\p{Is_JSN=NH}');
Error('\P{Is_JSN=NH}');
Error('\p{Jamo_Short_Name=YEO}');
Error('\P{Jamo_Short_Name=YEO}');
Error('\p{JSN=YEO}');
Error('\P{JSN=YEO}');
Error('\p{Is_Jamo_Short_Name=YEO}');
Error('\P{Is_Jamo_Short_Name=YEO}');
Error('\p{Is_JSN=YEO}');
Error('\P{Is_JSN=YEO}');
Error('\p{Jamo_Short_Name=LP}');
Error('\P{Jamo_Short_Name=LP}');
Error('\p{JSN=LP}');
Error('\P{JSN=LP}');
Error('\p{Is_Jamo_Short_Name=LP}');
Error('\P{Is_Jamo_Short_Name=LP}');
Error('\p{Is_JSN=LP}');
Error('\P{Is_JSN=LP}');
Error('\p{Jamo_Short_Name:   NJ}');
Error('\P{Jamo_Short_Name:   NJ}');
Error('\p{JSN: NJ}');
Error('\P{JSN: NJ}');
Error('\p{Is_Jamo_Short_Name=NJ}');
Error('\P{Is_Jamo_Short_Name=NJ}');
Error('\p{Is_JSN=NJ}');
Error('\P{Is_JSN=NJ}');
Error('\p{Jamo_Short_Name=B}');
Error('\P{Jamo_Short_Name=B}');
Error('\p{JSN=B}');
Error('\P{JSN=B}');
Error('\p{Is_Jamo_Short_Name:B}');
Error('\P{Is_Jamo_Short_Name:B}');
Error('\p{Is_JSN=B}');
Error('\P{Is_JSN=B}');
Error('\p{Jamo_Short_Name=WA}');
Error('\P{Jamo_Short_Name=WA}');
Error('\p{JSN: WA}');
Error('\P{JSN: WA}');
Error('\p{Is_Jamo_Short_Name=WA}');
Error('\P{Is_Jamo_Short_Name=WA}');
Error('\p{Is_JSN=WA}');
Error('\P{Is_JSN=WA}');
Error('\p{Jamo_Short_Name=LH}');
Error('\P{Jamo_Short_Name=LH}');
Error('\p{JSN=LH}');
Error('\P{JSN=LH}');
Error('\p{Is_Jamo_Short_Name=LH}');
Error('\P{Is_Jamo_Short_Name=LH}');
Error('\p{Is_JSN=LH}');
Error('\P{Is_JSN=LH}');
Error('\p{Jamo_Short_Name=YI}');
Error('\P{Jamo_Short_Name=YI}');
Error('\p{JSN=YI}');
Error('\P{JSN=YI}');
Error('\p{Is_Jamo_Short_Name=YI}');
Error('\P{Is_Jamo_Short_Name=YI}');
Error('\p{Is_JSN=YI}');
Error('\P{Is_JSN=YI}');
Error('\p{Jamo_Short_Name=WEO}');
Error('\P{Jamo_Short_Name=WEO}');
Error('\p{JSN=WEO}');
Error('\P{JSN=WEO}');
Error('\p{Is_Jamo_Short_Name=WEO}');
Error('\P{Is_Jamo_Short_Name=WEO}');
Error('\p{Is_JSN=WEO}');
Error('\P{Is_JSN=WEO}');
Error('\p{Jamo_Short_Name=LS}');
Error('\P{Jamo_Short_Name=LS}');
Error('\p{JSN=LS}');
Error('\P{JSN=LS}');
Error('\p{Is_Jamo_Short_Name=LS}');
Error('\P{Is_Jamo_Short_Name=LS}');
Error('\p{Is_JSN=LS}');
Error('\P{Is_JSN=LS}');
Error('\p{Jamo_Short_Name: BS}');
Error('\P{Jamo_Short_Name: BS}');
Error('\p{JSN=BS}');
Error('\P{JSN=BS}');
Error('\p{Is_Jamo_Short_Name=BS}');
Error('\P{Is_Jamo_Short_Name=BS}');
Error('\p{Is_JSN=BS}');
Error('\P{Is_JSN=BS}');
Error('\p{Jamo_Short_Name=YO}');
Error('\P{Jamo_Short_Name=YO}');
Error('\p{JSN=YO}');
Error('\P{JSN=YO}');
Error('\p{Is_Jamo_Short_Name=YO}');
Error('\P{Is_Jamo_Short_Name=YO}');
Error('\p{Is_JSN=YO}');
Error('\P{Is_JSN=YO}');
Error('\p{Jamo_Short_Name: OE}');
Error('\P{Jamo_Short_Name: OE}');
Error('\p{JSN=OE}');
Error('\P{JSN=OE}');
Error('\p{Is_Jamo_Short_Name=OE}');
Error('\P{Is_Jamo_Short_Name=OE}');
Error('\p{Is_JSN=OE}');
Error('\P{Is_JSN=OE}');
Error('\p{Jamo_Short_Name=M}');
Error('\P{Jamo_Short_Name=M}');
Error('\p{JSN=M}');
Error('\P{JSN=M}');
Error('\p{Is_Jamo_Short_Name=M}');
Error('\P{Is_Jamo_Short_Name=M}');
Error('\p{Is_JSN=M}');
Error('\P{Is_JSN=M}');
Error('\p{Jamo_Short_Name=EO}');
Error('\P{Jamo_Short_Name=EO}');
Error('\p{JSN:EO}');
Error('\P{JSN:EO}');
Error('\p{Is_Jamo_Short_Name=EO}');
Error('\P{Is_Jamo_Short_Name=EO}');
Error('\p{Is_JSN=EO}');
Error('\P{Is_JSN=EO}');
Error('\p{Jamo_Short_Name=C}');
Error('\P{Jamo_Short_Name=C}');
Error('\p{JSN=C}');
Error('\P{JSN=C}');
Error('\p{Is_Jamo_Short_Name=C}');
Error('\P{Is_Jamo_Short_Name=C}');
Error('\p{Is_JSN=C}');
Error('\P{Is_JSN=C}');
Error('\p{Jamo_Short_Name=WI}');
Error('\P{Jamo_Short_Name=WI}');
Error('\p{JSN=WI}');
Error('\P{JSN=WI}');
Error('\p{Is_Jamo_Short_Name=WI}');
Error('\P{Is_Jamo_Short_Name=WI}');
Error('\p{Is_JSN=WI}');
Error('\P{Is_JSN=WI}');
Error('\p{Jamo_Short_Name=EU}');
Error('\P{Jamo_Short_Name=EU}');
Error('\p{JSN=EU}');
Error('\P{JSN=EU}');
Error('\p{Is_Jamo_Short_Name=EU}');
Error('\P{Is_Jamo_Short_Name=EU}');
Error('\p{Is_JSN=EU}');
Error('\P{Is_JSN=EU}');
Error('\p{Jamo_Short_Name=LG}');
Error('\P{Jamo_Short_Name=LG}');
Error('\p{JSN=LG}');
Error('\P{JSN=LG}');
Error('\p{Is_Jamo_Short_Name=LG}');
Error('\P{Is_Jamo_Short_Name=LG}');
Error('\p{Is_JSN=LG}');
Error('\P{Is_JSN=LG}');
Error('\p{Jamo_Short_Name=L}');
Error('\P{Jamo_Short_Name=L}');
Error('\p{JSN=L}');
Error('\P{JSN=L}');
Error('\p{Is_Jamo_Short_Name=L}');
Error('\P{Is_Jamo_Short_Name=L}');
Error('\p{Is_JSN:	L}');
Error('\P{Is_JSN:	L}');
Error('\p{Jamo_Short_Name=P}');
Error('\P{Jamo_Short_Name=P}');
Error('\p{JSN=P}');
Error('\P{JSN=P}');
Error('\p{Is_Jamo_Short_Name:	P}');
Error('\P{Is_Jamo_Short_Name:	P}');
Error('\p{Is_JSN=P}');
Error('\P{Is_JSN=P}');
Error('\p{Jamo_Short_Name=DD}');
Error('\P{Jamo_Short_Name=DD}');
Error('\p{JSN=DD}');
Error('\P{JSN=DD}');
Error('\p{Is_Jamo_Short_Name: DD}');
Error('\P{Is_Jamo_Short_Name: DD}');
Error('\p{Is_JSN=DD}');
Error('\P{Is_JSN=DD}');
Error('\p{Jamo_Short_Name:   NG}');
Error('\P{Jamo_Short_Name:   NG}');
Error('\p{JSN=NG}');
Error('\P{JSN=NG}');
Error('\p{Is_Jamo_Short_Name=NG}');
Error('\P{Is_Jamo_Short_Name=NG}');
Error('\p{Is_JSN=NG}');
Error('\P{Is_JSN=NG}');
Error('\p{Jamo_Short_Name=AE}');
Error('\P{Jamo_Short_Name=AE}');
Error('\p{JSN=AE}');
Error('\P{JSN=AE}');
Error('\p{Is_Jamo_Short_Name=AE}');
Error('\P{Is_Jamo_Short_Name=AE}');
Error('\p{Is_JSN=AE}');
Error('\P{Is_JSN=AE}');
Error('\p{Jamo_Short_Name=T}');
Error('\P{Jamo_Short_Name=T}');
Error('\p{JSN=T}');
Error('\P{JSN=T}');
Error('\p{Is_Jamo_Short_Name=T}');
Error('\P{Is_Jamo_Short_Name=T}');
Error('\p{Is_JSN=T}');
Error('\P{Is_JSN=T}');
Error('\p{Jamo_Short_Name=A}');
Error('\P{Jamo_Short_Name=A}');
Error('\p{JSN=A}');
Error('\P{JSN=A}');
Error('\p{Is_Jamo_Short_Name=A}');
Error('\P{Is_Jamo_Short_Name=A}');
Error('\p{Is_JSN=A}');
Error('\P{Is_JSN=A}');
Error('\p{Jamo_Short_Name=N}');
Error('\P{Jamo_Short_Name=N}');
Error('\p{JSN=N}');
Error('\P{JSN=N}');
Error('\p{Is_Jamo_Short_Name=N}');
Error('\P{Is_Jamo_Short_Name=N}');
Error('\p{Is_JSN:	N}');
Error('\P{Is_JSN:	N}');
Error('\p{Jamo_Short_Name=YAE}');
Error('\P{Jamo_Short_Name=YAE}');
Error('\p{JSN=YAE}');
Error('\P{JSN=YAE}');
Error('\p{Is_Jamo_Short_Name=YAE}');
Error('\P{Is_Jamo_Short_Name=YAE}');
Error('\p{Is_JSN=YAE}');
Error('\P{Is_JSN=YAE}');
Error('\p{Jamo_Short_Name=U}');
Error('\P{Jamo_Short_Name=U}');
Error('\p{JSN=U}');
Error('\P{JSN=U}');
Error('\p{Is_Jamo_Short_Name=U}');
Error('\P{Is_Jamo_Short_Name=U}');
Error('\p{Is_JSN=U}');
Error('\P{Is_JSN=U}');
Error('\p{Jamo_Short_Name=WAE}');
Error('\P{Jamo_Short_Name=WAE}');
Error('\p{JSN=WAE}');
Error('\P{JSN=WAE}');
Error('\p{Is_Jamo_Short_Name=WAE}');
Error('\P{Is_Jamo_Short_Name=WAE}');
Error('\p{Is_JSN:	WAE}');
Error('\P{Is_JSN:	WAE}');
Error('\p{Jamo_Short_Name=GS}');
Error('\P{Jamo_Short_Name=GS}');
Error('\p{JSN=GS}');
Error('\P{JSN=GS}');
Error('\p{Is_Jamo_Short_Name=GS}');
Error('\P{Is_Jamo_Short_Name=GS}');
Error('\p{Is_JSN=GS}');
Error('\P{Is_JSN=GS}');
Error('\p{Jamo_Short_Name=H}');
Error('\P{Jamo_Short_Name=H}');
Error('\p{JSN=H}');
Error('\P{JSN=H}');
Error('\p{Is_Jamo_Short_Name=H}');
Error('\P{Is_Jamo_Short_Name=H}');
Error('\p{Is_JSN=H}');
Error('\P{Is_JSN=H}');
Error('\p{Jamo_Short_Name=LT}');
Error('\P{Jamo_Short_Name=LT}');
Error('\p{JSN=LT}');
Error('\P{JSN=LT}');
Error('\p{Is_Jamo_Short_Name=LT}');
Error('\P{Is_Jamo_Short_Name=LT}');
Error('\p{Is_JSN=LT}');
Error('\P{Is_JSN=LT}');
Error('\p{Jamo_Short_Name=YE}');
Error('\P{Jamo_Short_Name=YE}');
Error('\p{JSN=YE}');
Error('\P{JSN=YE}');
Error('\p{Is_Jamo_Short_Name=YE}');
Error('\P{Is_Jamo_Short_Name=YE}');
Error('\p{Is_JSN=YE}');
Error('\P{Is_JSN=YE}');
Error('\p{Jamo_Short_Name=I}');
Error('\P{Jamo_Short_Name=I}');
Error('\p{JSN=I}');
Error('\P{JSN=I}');
Error('\p{Is_Jamo_Short_Name=I}');
Error('\P{Is_Jamo_Short_Name=I}');
Error('\p{Is_JSN=I}');
Error('\P{Is_JSN=I}');
Error('\p{Jamo_Short_Name:   R}');
Error('\P{Jamo_Short_Name:   R}');
Error('\p{JSN=R}');
Error('\P{JSN=R}');
Error('\p{Is_Jamo_Short_Name=R}');
Error('\P{Is_Jamo_Short_Name=R}');
Error('\p{Is_JSN=R}');
Error('\P{Is_JSN=R}');
Error('\p{Jamo_Short_Name=O}');
Error('\P{Jamo_Short_Name=O}');
Error('\p{JSN=O}');
Error('\P{JSN=O}');
Error('\p{Is_Jamo_Short_Name=O}');
Error('\P{Is_Jamo_Short_Name=O}');
Error('\p{Is_JSN=O}');
Error('\P{Is_JSN=O}');
Error('\p{Age=	:=+00003.2}');
Error('\P{Age=	:=+00003.2}');
Expect(1, 544, '\p{Age:0_3.2}', "");
Expect(0, 544, '\p{^Age:0_3.2}', "");
Expect(0, 544, '\P{Age:0_3.2}', "");
Expect(1, 544, '\P{^Age:0_3.2}', "");
Expect(0, 543, '\p{Age:0_3.2}', "");
Expect(1, 543, '\p{^Age:0_3.2}', "");
Expect(1, 543, '\P{Age:0_3.2}', "");
Expect(0, 543, '\P{^Age:0_3.2}', "");
Error('\p{Is_Age=	/a/0_0 03.2}');
Error('\P{Is_Age=	/a/0_0 03.2}');
Expect(1, 544, '\p{Is_Age=+00003.2}', "");
Expect(0, 544, '\p{^Is_Age=+00003.2}', "");
Expect(0, 544, '\P{Is_Age=+00003.2}', "");
Expect(1, 544, '\P{^Is_Age=+00003.2}', "");
Expect(0, 543, '\p{Is_Age=+00003.2}', "");
Expect(1, 543, '\p{^Is_Age=+00003.2}', "");
Expect(1, 543, '\P{Is_Age=+00003.2}', "");
Expect(0, 543, '\P{^Is_Age=+00003.2}', "");
Error('\p{Age=0002.0/a/}');
Error('\P{Age=0002.0/a/}');
Expect(1, 1441, '\p{Age=+00_00_02.0}', "");
Expect(0, 1441, '\p{^Age=+00_00_02.0}', "");
Expect(0, 1441, '\P{Age=+00_00_02.0}', "");
Expect(1, 1441, '\P{^Age=+00_00_02.0}', "");
Expect(0, 1424, '\p{Age=+00_00_02.0}', "");
Expect(1, 1424, '\p{^Age=+00_00_02.0}', "");
Expect(1, 1424, '\P{Age=+00_00_02.0}', "");
Expect(0, 1424, '\P{^Age=+00_00_02.0}', "");
Error('\p{Is_Age=:=  0_0 0-0	0-00002.0}');
Error('\P{Is_Age=:=  0_0 0-0	0-00002.0}');
Expect(1, 1441, '\p{Is_Age=+0_0_0_0_0_002.0}', "");
Expect(0, 1441, '\p{^Is_Age=+0_0_0_0_0_002.0}', "");
Expect(0, 1441, '\P{Is_Age=+0_0_0_0_0_002.0}', "");
Expect(1, 1441, '\P{^Is_Age=+0_0_0_0_0_002.0}', "");
Expect(0, 1424, '\p{Is_Age=+0_0_0_0_0_002.0}', "");
Expect(1, 1424, '\p{^Is_Age=+0_0_0_0_0_002.0}', "");
Expect(1, 1424, '\P{Is_Age=+0_0_0_0_0_002.0}', "");
Expect(0, 1424, '\P{^Is_Age=+0_0_0_0_0_002.0}', "");
Error('\p{Age=_:=000_0001.1}');
Error('\P{Age=_:=000_0001.1}');
Expect(1, 501, '\p{Age=+000_1.1}', "");
Expect(0, 501, '\p{^Age=+000_1.1}', "");
Expect(0, 501, '\P{Age=+000_1.1}', "");
Expect(1, 501, '\P{^Age=+000_1.1}', "");
Expect(0, 505, '\p{Age=+000_1.1}', "");
Expect(1, 505, '\p{^Age=+000_1.1}', "");
Expect(1, 505, '\P{Age=+000_1.1}', "");
Expect(0, 505, '\P{^Age=+000_1.1}', "");
Error('\p{Is_Age= _00000001.1/a/}');
Error('\P{Is_Age= _00000001.1/a/}');
Expect(1, 501, '\p{Is_Age=0000000_1.1}', "");
Expect(0, 501, '\p{^Is_Age=0000000_1.1}', "");
Expect(0, 501, '\P{Is_Age=0000000_1.1}', "");
Expect(1, 501, '\P{^Is_Age=0000000_1.1}', "");
Expect(0, 505, '\p{Is_Age=0000000_1.1}', "");
Expect(1, 505, '\p{^Is_Age=0000000_1.1}', "");
Expect(1, 505, '\P{Is_Age=0000000_1.1}', "");
Expect(0, 505, '\P{^Is_Age=0000000_1.1}', "");
Error('\p{Age=	_+0_5.0:=}');
Error('\P{Age=	_+0_5.0:=}');
Expect(1, 591, '\p{Age=+00000000_5.0}', "");
Expect(0, 591, '\p{^Age=+00000000_5.0}', "");
Expect(0, 591, '\P{Age=+00000000_5.0}', "");
Expect(1, 591, '\P{^Age=+00000000_5.0}', "");
Expect(0, 577, '\p{Age=+00000000_5.0}', "");
Expect(1, 577, '\p{^Age=+00000000_5.0}', "");
Expect(1, 577, '\P{Age=+00000000_5.0}', "");
Expect(0, 577, '\P{^Age=+00000000_5.0}', "");
Error('\p{Is_Age=-005.0:=}');
Error('\P{Is_Age=-005.0:=}');
Expect(1, 591, '\p{Is_Age:	00000005.0}', "");
Expect(0, 591, '\p{^Is_Age:	00000005.0}', "");
Expect(0, 591, '\P{Is_Age:	00000005.0}', "");
Expect(1, 591, '\P{^Is_Age:	00000005.0}', "");
Expect(0, 577, '\p{Is_Age:	00000005.0}', "");
Expect(1, 577, '\p{^Is_Age:	00000005.0}', "");
Expect(1, 577, '\P{Is_Age:	00000005.0}', "");
Expect(0, 577, '\P{^Is_Age:	00000005.0}', "");
Error('\p{Age= -3.0/a/}');
Error('\P{Age= -3.0/a/}');
Expect(1, 505, '\p{Age=+0003.0}', "");
Expect(0, 505, '\p{^Age=+0003.0}', "");
Expect(0, 505, '\P{Age=+0003.0}', "");
Expect(1, 505, '\P{^Age=+0003.0}', "");
Expect(0, 501, '\p{Age=+0003.0}', "");
Expect(1, 501, '\p{^Age=+0003.0}', "");
Expect(1, 501, '\P{Age=+0003.0}', "");
Expect(0, 501, '\P{^Age=+0003.0}', "");
Error('\p{Is_Age=	_003.0/a/}');
Error('\P{Is_Age=	_003.0/a/}');
Expect(1, 505, '\p{Is_Age:   +00003.0}', "");
Expect(0, 505, '\p{^Is_Age:   +00003.0}', "");
Expect(0, 505, '\P{Is_Age:   +00003.0}', "");
Expect(1, 505, '\P{^Is_Age:   +00003.0}', "");
Expect(0, 501, '\p{Is_Age:   +00003.0}', "");
Expect(1, 501, '\p{^Is_Age:   +00003.0}', "");
Expect(1, 501, '\P{Is_Age:   +00003.0}', "");
Expect(0, 501, '\P{^Is_Age:   +00003.0}', "");
Error('\p{Age::= -00000006.0}');
Error('\P{Age::= -00000006.0}');
Expect(1, 1319, '\p{Age=6.0}', "");
Expect(0, 1319, '\p{^Age=6.0}', "");
Expect(0, 1319, '\P{Age=6.0}', "");
Expect(1, 1319, '\P{^Age=6.0}', "");
Expect(0, 1317, '\p{Age=6.0}', "");
Expect(1, 1317, '\p{^Age=6.0}', "");
Expect(1, 1317, '\P{Age=6.0}', "");
Expect(0, 1317, '\P{^Age=6.0}', "");
Error('\p{Is_Age=:=_ 0000006.0}');
Error('\P{Is_Age=:=_ 0000006.0}');
Expect(1, 1319, '\p{Is_Age:	+0_0_0_0_0006.0}', "");
Expect(0, 1319, '\p{^Is_Age:	+0_0_0_0_0006.0}', "");
Expect(0, 1319, '\P{Is_Age:	+0_0_0_0_0006.0}', "");
Expect(1, 1319, '\P{^Is_Age:	+0_0_0_0_0006.0}', "");
Expect(0, 1317, '\p{Is_Age:	+0_0_0_0_0006.0}', "");
Expect(1, 1317, '\p{^Is_Age:	+0_0_0_0_0006.0}', "");
Expect(1, 1317, '\P{Is_Age:	+0_0_0_0_0006.0}', "");
Expect(0, 1317, '\P{^Is_Age:	+0_0_0_0_0006.0}', "");
Error('\p{Age=/a/ 	Unassigned}');
Error('\P{Age=/a/ 	Unassigned}');
Expect(1, 889, '\p{Age:unassigned}', "");
Expect(0, 889, '\p{^Age:unassigned}', "");
Expect(0, 889, '\P{Age:unassigned}', "");
Expect(1, 889, '\P{^Age:unassigned}', "");
Expect(0, 887, '\p{Age:unassigned}', "");
Expect(1, 887, '\p{^Age:unassigned}', "");
Expect(1, 887, '\P{Age:unassigned}', "");
Expect(0, 887, '\P{^Age:unassigned}', "");
Expect(1, 889, '\p{Age=-	UNASSIGNED}', "");
Expect(0, 889, '\p{^Age=-	UNASSIGNED}', "");
Expect(0, 889, '\P{Age=-	UNASSIGNED}', "");
Expect(1, 889, '\P{^Age=-	UNASSIGNED}', "");
Expect(0, 887, '\p{Age=-	UNASSIGNED}', "");
Expect(1, 887, '\p{^Age=-	UNASSIGNED}', "");
Expect(1, 887, '\P{Age=-	UNASSIGNED}', "");
Expect(0, 887, '\P{^Age=-	UNASSIGNED}', "");
Error('\p{Is_Age= /a/Unassigned}');
Error('\P{Is_Age= /a/Unassigned}');
Expect(1, 889, '\p{Is_Age=unassigned}', "");
Expect(0, 889, '\p{^Is_Age=unassigned}', "");
Expect(0, 889, '\P{Is_Age=unassigned}', "");
Expect(1, 889, '\P{^Is_Age=unassigned}', "");
Expect(0, 887, '\p{Is_Age=unassigned}', "");
Expect(1, 887, '\p{^Is_Age=unassigned}', "");
Expect(1, 887, '\P{Is_Age=unassigned}', "");
Expect(0, 887, '\P{^Is_Age=unassigned}', "");
Expect(1, 889, '\p{Is_Age= 	Unassigned}', "");
Expect(0, 889, '\p{^Is_Age= 	Unassigned}', "");
Expect(0, 889, '\P{Is_Age= 	Unassigned}', "");
Expect(1, 889, '\P{^Is_Age= 	Unassigned}', "");
Expect(0, 887, '\p{Is_Age= 	Unassigned}', "");
Expect(1, 887, '\p{^Is_Age= 	Unassigned}', "");
Expect(1, 887, '\P{Is_Age= 	Unassigned}', "");
Expect(0, 887, '\P{^Is_Age= 	Unassigned}', "");
Error('\p{Age=	_005.1/a/}');
Error('\P{Age=	_005.1/a/}');
Expect(1, 883, '\p{Age=+000_5.1}', "");
Expect(0, 883, '\p{^Age=+000_5.1}', "");
Expect(0, 883, '\P{Age=+000_5.1}', "");
Expect(1, 883, '\P{^Age=+000_5.1}', "");
Expect(0, 879, '\p{Age=+000_5.1}', "");
Expect(1, 879, '\p{^Age=+000_5.1}', "");
Expect(1, 879, '\P{Age=+000_5.1}', "");
Expect(0, 879, '\P{^Age=+000_5.1}', "");
Error('\p{Is_Age=		0-00/a/00_00005.1}');
Error('\P{Is_Age=		0-00/a/00_00005.1}');
Expect(1, 883, '\p{Is_Age=+0_0_0_0_0_0_0_0_05.1}', "");
Expect(0, 883, '\p{^Is_Age=+0_0_0_0_0_0_0_0_05.1}', "");
Expect(0, 883, '\P{Is_Age=+0_0_0_0_0_0_0_0_05.1}', "");
Expect(1, 883, '\P{^Is_Age=+0_0_0_0_0_0_0_0_05.1}', "");
Expect(0, 879, '\p{Is_Age=+0_0_0_0_0_0_0_0_05.1}', "");
Expect(1, 879, '\p{^Is_Age=+0_0_0_0_0_0_0_0_05.1}', "");
Expect(1, 879, '\P{Is_Age=+0_0_0_0_0_0_0_0_05.1}', "");
Expect(0, 879, '\P{^Is_Age=+0_0_0_0_0_0_0_0_05.1}', "");
Error('\p{Age:_	+004.0/a/}');
Error('\P{Age:_	+004.0/a/}');
Expect(1, 545, '\p{Age=000000004.0}', "");
Expect(0, 545, '\p{^Age=000000004.0}', "");
Expect(0, 545, '\P{Age=000000004.0}', "");
Expect(1, 545, '\P{^Age=000000004.0}', "");
Expect(0, 544, '\p{Age=000000004.0}', "");
Expect(1, 544, '\p{^Age=000000004.0}', "");
Expect(1, 544, '\P{Age=000000004.0}', "");
Expect(0, 544, '\P{^Age=000000004.0}', "");
Error('\p{Is_Age=/a/  004.0}');
Error('\P{Is_Age=/a/  004.0}');
Expect(1, 545, '\p{Is_Age=0000004.0}', "");
Expect(0, 545, '\p{^Is_Age=0000004.0}', "");
Expect(0, 545, '\P{Is_Age=0000004.0}', "");
Expect(1, 545, '\P{^Is_Age=0000004.0}', "");
Expect(0, 544, '\p{Is_Age=0000004.0}', "");
Expect(1, 544, '\p{^Is_Age=0000004.0}', "");
Expect(1, 544, '\P{Is_Age=0000004.0}', "");
Expect(0, 544, '\P{^Is_Age=0000004.0}', "");
Error('\p{Age=:=_ +0 5.2}');
Error('\P{Age=:=_ +0 5.2}');
Expect(1, 1317, '\p{Age=+0000000005.2}', "");
Expect(0, 1317, '\p{^Age=+0000000005.2}', "");
Expect(0, 1317, '\P{Age=+0000000005.2}', "");
Expect(1, 1317, '\P{^Age=+0000000005.2}', "");
Expect(0, 1315, '\p{Age=+0000000005.2}', "");
Expect(1, 1315, '\p{^Age=+0000000005.2}', "");
Expect(1, 1315, '\P{Age=+0000000005.2}', "");
Expect(0, 1315, '\P{^Age=+0000000005.2}', "");
Error('\p{Is_Age:   _/a/05.2}');
Error('\P{Is_Age:   _/a/05.2}');
Expect(1, 1317, '\p{Is_Age=00000005.2}', "");
Expect(0, 1317, '\p{^Is_Age=00000005.2}', "");
Expect(0, 1317, '\P{Is_Age=00000005.2}', "");
Expect(1, 1317, '\P{^Is_Age=00000005.2}', "");
Expect(0, 1315, '\p{Is_Age=00000005.2}', "");
Expect(1, 1315, '\p{^Is_Age=00000005.2}', "");
Expect(1, 1315, '\P{Is_Age=00000005.2}', "");
Expect(0, 1315, '\P{^Is_Age=00000005.2}', "");
Error('\p{Age=	000003.1:=}');
Error('\P{Age=	000003.1:=}');
Expect(1, 1013, '\p{Age:00_00_00_00_3.1}', "");
Expect(0, 1013, '\p{^Age:00_00_00_00_3.1}', "");
Expect(0, 1013, '\P{Age:00_00_00_00_3.1}', "");
Expect(1, 1013, '\P{^Age:00_00_00_00_3.1}', "");
Expect(0, 1011, '\p{Age:00_00_00_00_3.1}', "");
Expect(1, 1011, '\p{^Age:00_00_00_00_3.1}', "");
Expect(1, 1011, '\P{Age:00_00_00_00_3.1}', "");
Expect(0, 1011, '\P{^Age:00_00_00_00_3.1}', "");
Error('\p{Is_Age=	_+0	00:=03.1}');
Error('\P{Is_Age=	_+0	00:=03.1}');
Expect(1, 1013, '\p{Is_Age=+000003.1}', "");
Expect(0, 1013, '\p{^Is_Age=+000003.1}', "");
Expect(0, 1013, '\P{Is_Age=+000003.1}', "");
Expect(1, 1013, '\P{^Is_Age=+000003.1}', "");
Expect(0, 1011, '\p{Is_Age=+000003.1}', "");
Expect(1, 1011, '\p{^Is_Age=+000003.1}', "");
Expect(1, 1011, '\P{Is_Age=+000003.1}', "");
Expect(0, 1011, '\P{^Is_Age=+000003.1}', "");
Error('\p{Age:    _00000002.1:=}');
Error('\P{Age:    _00000002.1:=}');
Expect(1, 8364, '\p{Age=+0_2.1}', "");
Expect(0, 8364, '\p{^Age=+0_2.1}', "");
Expect(0, 8364, '\P{Age=+0_2.1}', "");
Expect(1, 8364, '\P{^Age=+0_2.1}', "");
Expect(0, 8363, '\p{Age=+0_2.1}', "");
Expect(1, 8363, '\p{^Age=+0_2.1}', "");
Expect(1, 8363, '\P{Age=+0_2.1}', "");
Expect(0, 8363, '\P{^Age=+0_2.1}', "");
Error('\p{Is_Age=_ 0	0:=0_00_02.1}');
Error('\P{Is_Age=_ 0	0:=0_00_02.1}');
Expect(1, 8364, '\p{Is_Age=00_00_00_00_2.1}', "");
Expect(0, 8364, '\p{^Is_Age=00_00_00_00_2.1}', "");
Expect(0, 8364, '\P{Is_Age=00_00_00_00_2.1}', "");
Expect(1, 8364, '\P{^Is_Age=00_00_00_00_2.1}', "");
Expect(0, 8363, '\p{Is_Age=00_00_00_00_2.1}', "");
Expect(1, 8363, '\p{^Is_Age=00_00_00_00_2.1}', "");
Expect(1, 8363, '\P{Is_Age=00_00_00_00_2.1}', "");
Expect(0, 8363, '\P{^Is_Age=00_00_00_00_2.1}', "");
Error('\p{Age=-	04.1:=}');
Error('\P{Age=-	04.1:=}');
Expect(1, 577, '\p{Age=04.1}', "");
Expect(0, 577, '\p{^Age=04.1}', "");
Expect(0, 577, '\P{Age=04.1}', "");
Expect(1, 577, '\P{^Age=04.1}', "");
Expect(0, 566, '\p{Age=04.1}', "");
Expect(1, 566, '\p{^Age=04.1}', "");
Expect(1, 566, '\P{Age=04.1}', "");
Expect(0, 566, '\P{^Age=04.1}', "");
Error('\p{Is_Age= _00/a/00_00_00_04.1}');
Error('\P{Is_Age= _00/a/00_00_00_04.1}');
Expect(1, 577, '\p{Is_Age: 000_4.1}', "");
Expect(0, 577, '\p{^Is_Age: 000_4.1}', "");
Expect(0, 577, '\P{Is_Age: 000_4.1}', "");
Expect(1, 577, '\P{^Is_Age: 000_4.1}', "");
Expect(0, 566, '\p{Is_Age: 000_4.1}', "");
Expect(1, 566, '\p{^Is_Age: 000_4.1}', "");
Expect(1, 566, '\P{Is_Age: 000_4.1}', "");
Expect(0, 566, '\P{^Is_Age: 000_4.1}', "");
Error('\p{Expands_On_NFKD=Yes}');
Error('\P{Expands_On_NFKD=Yes}');
Error('\p{XO_NFKD=Y}');
Error('\P{XO_NFKD=Y}');
Error('\p{Is_Expands_On_NFKD=T}');
Error('\P{Is_Expands_On_NFKD=T}');
Error('\p{Is_XO_NFKD: True}');
Error('\P{Is_XO_NFKD: True}');
Error('\p{Expands_On_NFKD:No}');
Error('\P{Expands_On_NFKD:No}');
Error('\p{XO_NFKD=N}');
Error('\P{XO_NFKD=N}');
Error('\p{Is_Expands_On_NFKD=F}');
Error('\P{Is_Expands_On_NFKD=F}');
Error('\p{Is_XO_NFKD=False}');
Error('\P{Is_XO_NFKD=False}');
Error('\p{Grapheme_Cluster_Break=_/a/spacingmark}');
Error('\P{Grapheme_Cluster_Break=_/a/spacingmark}');
Expect(1, 2307, '\p{Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 2307, '\p{^Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 2307, '\P{Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 2307, '\P{^Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 2306, '\p{Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 2306, '\p{^Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 2306, '\P{Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 2306, '\P{^Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 2307, '\p{Grapheme_Cluster_Break=- SpacingMark}', "");
Expect(0, 2307, '\p{^Grapheme_Cluster_Break=- SpacingMark}', "");
Expect(0, 2307, '\P{Grapheme_Cluster_Break=- SpacingMark}', "");
Expect(1, 2307, '\P{^Grapheme_Cluster_Break=- SpacingMark}', "");
Expect(0, 2306, '\p{Grapheme_Cluster_Break=- SpacingMark}', "");
Expect(1, 2306, '\p{^Grapheme_Cluster_Break=- SpacingMark}', "");
Expect(1, 2306, '\P{Grapheme_Cluster_Break=- SpacingMark}', "");
Expect(0, 2306, '\P{^Grapheme_Cluster_Break=- SpacingMark}', "");
Error('\p{GCB=/a/- SM}');
Error('\P{GCB=/a/- SM}');
Expect(1, 2307, '\p{GCB: sm}', "");
Expect(0, 2307, '\p{^GCB: sm}', "");
Expect(0, 2307, '\P{GCB: sm}', "");
Expect(1, 2307, '\P{^GCB: sm}', "");
Expect(0, 2306, '\p{GCB: sm}', "");
Expect(1, 2306, '\p{^GCB: sm}', "");
Expect(1, 2306, '\P{GCB: sm}', "");
Expect(0, 2306, '\P{^GCB: sm}', "");
Expect(1, 2307, '\p{GCB=	_SM}', "");
Expect(0, 2307, '\p{^GCB=	_SM}', "");
Expect(0, 2307, '\P{GCB=	_SM}', "");
Expect(1, 2307, '\P{^GCB=	_SM}', "");
Expect(0, 2306, '\p{GCB=	_SM}', "");
Expect(1, 2306, '\p{^GCB=	_SM}', "");
Expect(1, 2306, '\P{GCB=	_SM}', "");
Expect(0, 2306, '\P{^GCB=	_SM}', "");
Error('\p{Is_Grapheme_Cluster_Break=	-SpacingMark:=}');
Error('\P{Is_Grapheme_Cluster_Break=	-SpacingMark:=}');
Expect(1, 2307, '\p{Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 2307, '\p{^Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 2307, '\P{Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 2307, '\P{^Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 2306, '\p{Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 2306, '\p{^Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 2306, '\P{Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(0, 2306, '\P{^Is_Grapheme_Cluster_Break=spacingmark}', "");
Expect(1, 2307, '\p{Is_Grapheme_Cluster_Break=- SpacingMark}', "");
Expect(0, 2307, '\p{^Is_Grapheme_Cluster_Break=- SpacingMark}', "");
Expect(0, 2307, '\P{Is_Grapheme_Cluster_Break=- SpacingMark}', "");
Expect(1, 2307, '\P{^Is_Grapheme_Cluster_Break=- SpacingMark}', "");
Expect(0, 2306, '\p{Is_Grapheme_Cluster_Break=- SpacingMark}', "");
Expect(1, 2306, '\p{^Is_Grapheme_Cluster_Break=- SpacingMark}', "");
Expect(1, 2306, '\P{Is_Grapheme_Cluster_Break=- SpacingMark}', "");
Expect(0, 2306, '\P{^Is_Grapheme_Cluster_Break=- SpacingMark}', "");
Error('\p{Is_GCB: /a/-	sm}');
Error('\P{Is_GCB: /a/-	sm}');
Expect(1, 2307, '\p{Is_GCB=sm}', "");
Expect(0, 2307, '\p{^Is_GCB=sm}', "");
Expect(0, 2307, '\P{Is_GCB=sm}', "");
Expect(1, 2307, '\P{^Is_GCB=sm}', "");
Expect(0, 2306, '\p{Is_GCB=sm}', "");
Expect(1, 2306, '\p{^Is_GCB=sm}', "");
Expect(1, 2306, '\P{Is_GCB=sm}', "");
Expect(0, 2306, '\P{^Is_GCB=sm}', "");
Expect(1, 2307, '\p{Is_GCB= 	SM}', "");
Expect(0, 2307, '\p{^Is_GCB= 	SM}', "");
Expect(0, 2307, '\P{Is_GCB= 	SM}', "");
Expect(1, 2307, '\P{^Is_GCB= 	SM}', "");
Expect(0, 2306, '\p{Is_GCB= 	SM}', "");
Expect(1, 2306, '\p{^Is_GCB= 	SM}', "");
Expect(1, 2306, '\P{Is_GCB= 	SM}', "");
Expect(0, 2306, '\P{^Is_GCB= 	SM}', "");
Error('\p{Grapheme_Cluster_Break=/a/- T}');
Error('\P{Grapheme_Cluster_Break=/a/- T}');
Expect(1, 4607, '\p{Grapheme_Cluster_Break:t}', "");
Expect(0, 4607, '\p{^Grapheme_Cluster_Break:t}', "");
Expect(0, 4607, '\P{Grapheme_Cluster_Break:t}', "");
Expect(1, 4607, '\P{^Grapheme_Cluster_Break:t}', "");
Expect(0, 4519, '\p{Grapheme_Cluster_Break:t}', "");
Expect(1, 4519, '\p{^Grapheme_Cluster_Break:t}', "");
Expect(1, 4519, '\P{Grapheme_Cluster_Break:t}', "");
Expect(0, 4519, '\P{^Grapheme_Cluster_Break:t}', "");
Expect(1, 4607, '\p{Grapheme_Cluster_Break=		T}', "");
Expect(0, 4607, '\p{^Grapheme_Cluster_Break=		T}', "");
Expect(0, 4607, '\P{Grapheme_Cluster_Break=		T}', "");
Expect(1, 4607, '\P{^Grapheme_Cluster_Break=		T}', "");
Expect(0, 4519, '\p{Grapheme_Cluster_Break=		T}', "");
Expect(1, 4519, '\p{^Grapheme_Cluster_Break=		T}', "");
Expect(1, 4519, '\P{Grapheme_Cluster_Break=		T}', "");
Expect(0, 4519, '\P{^Grapheme_Cluster_Break=		T}', "");
Error('\p{GCB=-:=T}');
Error('\P{GCB=-:=T}');
Expect(1, 4607, '\p{GCB=t}', "");
Expect(0, 4607, '\p{^GCB=t}', "");
Expect(0, 4607, '\P{GCB=t}', "");
Expect(1, 4607, '\P{^GCB=t}', "");
Expect(0, 4519, '\p{GCB=t}', "");
Expect(1, 4519, '\p{^GCB=t}', "");
Expect(1, 4519, '\P{GCB=t}', "");
Expect(0, 4519, '\P{^GCB=t}', "");
Expect(1, 4607, '\p{GCB= t}', "");
Expect(0, 4607, '\p{^GCB= t}', "");
Expect(0, 4607, '\P{GCB= t}', "");
Expect(1, 4607, '\P{^GCB= t}', "");
Expect(0, 4519, '\p{GCB= t}', "");
Expect(1, 4519, '\p{^GCB= t}', "");
Expect(1, 4519, '\P{GCB= t}', "");
Expect(0, 4519, '\P{^GCB= t}', "");
Error('\p{Is_Grapheme_Cluster_Break=_	T/a/}');
Error('\P{Is_Grapheme_Cluster_Break=_	T/a/}');
Expect(1, 4607, '\p{Is_Grapheme_Cluster_Break=t}', "");
Expect(0, 4607, '\p{^Is_Grapheme_Cluster_Break=t}', "");
Expect(0, 4607, '\P{Is_Grapheme_Cluster_Break=t}', "");
Expect(1, 4607, '\P{^Is_Grapheme_Cluster_Break=t}', "");
Expect(0, 4519, '\p{Is_Grapheme_Cluster_Break=t}', "");
Expect(1, 4519, '\p{^Is_Grapheme_Cluster_Break=t}', "");
Expect(1, 4519, '\P{Is_Grapheme_Cluster_Break=t}', "");
Expect(0, 4519, '\P{^Is_Grapheme_Cluster_Break=t}', "");
Expect(1, 4607, '\p{Is_Grapheme_Cluster_Break=	T}', "");
Expect(0, 4607, '\p{^Is_Grapheme_Cluster_Break=	T}', "");
Expect(0, 4607, '\P{Is_Grapheme_Cluster_Break=	T}', "");
Expect(1, 4607, '\P{^Is_Grapheme_Cluster_Break=	T}', "");
Expect(0, 4519, '\p{Is_Grapheme_Cluster_Break=	T}', "");
Expect(1, 4519, '\p{^Is_Grapheme_Cluster_Break=	T}', "");
Expect(1, 4519, '\P{Is_Grapheme_Cluster_Break=	T}', "");
Expect(0, 4519, '\P{^Is_Grapheme_Cluster_Break=	T}', "");
Error('\p{Is_GCB:--T:=}');
Error('\P{Is_GCB:--T:=}');
Expect(1, 4607, '\p{Is_GCB=t}', "");
Expect(0, 4607, '\p{^Is_GCB=t}', "");
Expect(0, 4607, '\P{Is_GCB=t}', "");
Expect(1, 4607, '\P{^Is_GCB=t}', "");
Expect(0, 4519, '\p{Is_GCB=t}', "");
Expect(1, 4519, '\p{^Is_GCB=t}', "");
Expect(1, 4519, '\P{Is_GCB=t}', "");
Expect(0, 4519, '\P{^Is_GCB=t}', "");
Expect(1, 4607, '\p{Is_GCB= T}', "");
Expect(0, 4607, '\p{^Is_GCB= T}', "");
Expect(0, 4607, '\P{Is_GCB= T}', "");
Expect(1, 4607, '\P{^Is_GCB= T}', "");
Expect(0, 4519, '\p{Is_GCB= T}', "");
Expect(1, 4519, '\p{^Is_GCB= T}', "");
Expect(1, 4519, '\P{Is_GCB= T}', "");
Expect(0, 4519, '\P{^Is_GCB= T}', "");
Error('\p{Grapheme_Cluster_Break=/a/LVT}');
Error('\P{Grapheme_Cluster_Break=/a/LVT}');
Expect(1, 44059, '\p{Grapheme_Cluster_Break:lvt}', "");
Expect(0, 44059, '\p{^Grapheme_Cluster_Break:lvt}', "");
Expect(0, 44059, '\P{Grapheme_Cluster_Break:lvt}', "");
Expect(1, 44059, '\P{^Grapheme_Cluster_Break:lvt}', "");
Expect(0, 44032, '\p{Grapheme_Cluster_Break:lvt}', "");
Expect(1, 44032, '\p{^Grapheme_Cluster_Break:lvt}', "");
Expect(1, 44032, '\P{Grapheme_Cluster_Break:lvt}', "");
Expect(0, 44032, '\P{^Grapheme_Cluster_Break:lvt}', "");
Expect(1, 44059, '\p{Grapheme_Cluster_Break=  lvt}', "");
Expect(0, 44059, '\p{^Grapheme_Cluster_Break=  lvt}', "");
Expect(0, 44059, '\P{Grapheme_Cluster_Break=  lvt}', "");
Expect(1, 44059, '\P{^Grapheme_Cluster_Break=  lvt}', "");
Expect(0, 44032, '\p{Grapheme_Cluster_Break=  lvt}', "");
Expect(1, 44032, '\p{^Grapheme_Cluster_Break=  lvt}', "");
Expect(1, 44032, '\P{Grapheme_Cluster_Break=  lvt}', "");
Expect(0, 44032, '\P{^Grapheme_Cluster_Break=  lvt}', "");
Error('\p{GCB=/a/LVT}');
Error('\P{GCB=/a/LVT}');
Expect(1, 44059, '\p{GCB=lvt}', "");
Expect(0, 44059, '\p{^GCB=lvt}', "");
Expect(0, 44059, '\P{GCB=lvt}', "");
Expect(1, 44059, '\P{^GCB=lvt}', "");
Expect(0, 44032, '\p{GCB=lvt}', "");
Expect(1, 44032, '\p{^GCB=lvt}', "");
Expect(1, 44032, '\P{GCB=lvt}', "");
Expect(0, 44032, '\P{^GCB=lvt}', "");
Expect(1, 44059, '\p{GCB=_	lvt}', "");
Expect(0, 44059, '\p{^GCB=_	lvt}', "");
Expect(0, 44059, '\P{GCB=_	lvt}', "");
Expect(1, 44059, '\P{^GCB=_	lvt}', "");
Expect(0, 44032, '\p{GCB=_	lvt}', "");
Expect(1, 44032, '\p{^GCB=_	lvt}', "");
Expect(1, 44032, '\P{GCB=_	lvt}', "");
Expect(0, 44032, '\P{^GCB=_	lvt}', "");
Error('\p{Is_Grapheme_Cluster_Break: 	LVT/a/}');
Error('\P{Is_Grapheme_Cluster_Break: 	LVT/a/}');
Expect(1, 44059, '\p{Is_Grapheme_Cluster_Break=lvt}', "");
Expect(0, 44059, '\p{^Is_Grapheme_Cluster_Break=lvt}', "");
Expect(0, 44059, '\P{Is_Grapheme_Cluster_Break=lvt}', "");
Expect(1, 44059, '\P{^Is_Grapheme_Cluster_Break=lvt}', "");
Expect(0, 44032, '\p{Is_Grapheme_Cluster_Break=lvt}', "");
Expect(1, 44032, '\p{^Is_Grapheme_Cluster_Break=lvt}', "");
Expect(1, 44032, '\P{Is_Grapheme_Cluster_Break=lvt}', "");
Expect(0, 44032, '\P{^Is_Grapheme_Cluster_Break=lvt}', "");
Expect(1, 44059, '\p{Is_Grapheme_Cluster_Break=_LVT}', "");
Expect(0, 44059, '\p{^Is_Grapheme_Cluster_Break=_LVT}', "");
Expect(0, 44059, '\P{Is_Grapheme_Cluster_Break=_LVT}', "");
Expect(1, 44059, '\P{^Is_Grapheme_Cluster_Break=_LVT}', "");
Expect(0, 44032, '\p{Is_Grapheme_Cluster_Break=_LVT}', "");
Expect(1, 44032, '\p{^Is_Grapheme_Cluster_Break=_LVT}', "");
Expect(1, 44032, '\P{Is_Grapheme_Cluster_Break=_LVT}', "");
Expect(0, 44032, '\P{^Is_Grapheme_Cluster_Break=_LVT}', "");
Error('\p{Is_GCB=/a/_LVT}');
Error('\P{Is_GCB=/a/_LVT}');
Expect(1, 44059, '\p{Is_GCB=lvt}', "");
Expect(0, 44059, '\p{^Is_GCB=lvt}', "");
Expect(0, 44059, '\P{Is_GCB=lvt}', "");
Expect(1, 44059, '\P{^Is_GCB=lvt}', "");
Expect(0, 44032, '\p{Is_GCB=lvt}', "");
Expect(1, 44032, '\p{^Is_GCB=lvt}', "");
Expect(1, 44032, '\P{Is_GCB=lvt}', "");
Expect(0, 44032, '\P{^Is_GCB=lvt}', "");
Expect(1, 44059, '\p{Is_GCB= lvt}', "");
Expect(0, 44059, '\p{^Is_GCB= lvt}', "");
Expect(0, 44059, '\P{Is_GCB= lvt}', "");
Expect(1, 44059, '\P{^Is_GCB= lvt}', "");
Expect(0, 44032, '\p{Is_GCB= lvt}', "");
Expect(1, 44032, '\p{^Is_GCB= lvt}', "");
Expect(1, 44032, '\P{Is_GCB= lvt}', "");
Expect(0, 44032, '\P{^Is_GCB= lvt}', "");
Error('\p{Grapheme_Cluster_Break:   	/a/control}');
Error('\P{Grapheme_Cluster_Break:   	/a/control}');
Expect(1, 9, '\p{Grapheme_Cluster_Break=control}', "");
Expect(0, 9, '\p{^Grapheme_Cluster_Break=control}', "");
Expect(0, 9, '\P{Grapheme_Cluster_Break=control}', "");
Expect(1, 9, '\P{^Grapheme_Cluster_Break=control}', "");
Expect(0, 10, '\p{Grapheme_Cluster_Break=control}', "");
Expect(1, 10, '\p{^Grapheme_Cluster_Break=control}', "");
Expect(1, 10, '\P{Grapheme_Cluster_Break=control}', "");
Expect(0, 10, '\P{^Grapheme_Cluster_Break=control}', "");
Expect(1, 9, '\p{Grapheme_Cluster_Break=--Control}', "");
Expect(0, 9, '\p{^Grapheme_Cluster_Break=--Control}', "");
Expect(0, 9, '\P{Grapheme_Cluster_Break=--Control}', "");
Expect(1, 9, '\P{^Grapheme_Cluster_Break=--Control}', "");
Expect(0, 10, '\p{Grapheme_Cluster_Break=--Control}', "");
Expect(1, 10, '\p{^Grapheme_Cluster_Break=--Control}', "");
Expect(1, 10, '\P{Grapheme_Cluster_Break=--Control}', "");
Expect(0, 10, '\P{^Grapheme_Cluster_Break=--Control}', "");
Error('\p{GCB=CN/a/}');
Error('\P{GCB=CN/a/}');
Expect(1, 9, '\p{GCB=cn}', "");
Expect(0, 9, '\p{^GCB=cn}', "");
Expect(0, 9, '\P{GCB=cn}', "");
Expect(1, 9, '\P{^GCB=cn}', "");
Expect(0, 10, '\p{GCB=cn}', "");
Expect(1, 10, '\p{^GCB=cn}', "");
Expect(1, 10, '\P{GCB=cn}', "");
Expect(0, 10, '\P{^GCB=cn}', "");
Expect(1, 9, '\p{GCB=--CN}', "");
Expect(0, 9, '\p{^GCB=--CN}', "");
Expect(0, 9, '\P{GCB=--CN}', "");
Expect(1, 9, '\P{^GCB=--CN}', "");
Expect(0, 10, '\p{GCB=--CN}', "");
Expect(1, 10, '\p{^GCB=--CN}', "");
Expect(1, 10, '\P{GCB=--CN}', "");
Expect(0, 10, '\P{^GCB=--CN}', "");
Error('\p{Is_Grapheme_Cluster_Break:		/a/CONTROL}');
Error('\P{Is_Grapheme_Cluster_Break:		/a/CONTROL}');
Expect(1, 9, '\p{Is_Grapheme_Cluster_Break=control}', "");
Expect(0, 9, '\p{^Is_Grapheme_Cluster_Break=control}', "");
Expect(0, 9, '\P{Is_Grapheme_Cluster_Break=control}', "");
Expect(1, 9, '\P{^Is_Grapheme_Cluster_Break=control}', "");
Expect(0, 10, '\p{Is_Grapheme_Cluster_Break=control}', "");
Expect(1, 10, '\p{^Is_Grapheme_Cluster_Break=control}', "");
Expect(1, 10, '\P{Is_Grapheme_Cluster_Break=control}', "");
Expect(0, 10, '\P{^Is_Grapheme_Cluster_Break=control}', "");
Expect(1, 9, '\p{Is_Grapheme_Cluster_Break=- CONTROL}', "");
Expect(0, 9, '\p{^Is_Grapheme_Cluster_Break=- CONTROL}', "");
Expect(0, 9, '\P{Is_Grapheme_Cluster_Break=- CONTROL}', "");
Expect(1, 9, '\P{^Is_Grapheme_Cluster_Break=- CONTROL}', "");
Expect(0, 10, '\p{Is_Grapheme_Cluster_Break=- CONTROL}', "");
Expect(1, 10, '\p{^Is_Grapheme_Cluster_Break=- CONTROL}', "");
Expect(1, 10, '\P{Is_Grapheme_Cluster_Break=- CONTROL}', "");
Expect(0, 10, '\P{^Is_Grapheme_Cluster_Break=- CONTROL}', "");
Error('\p{Is_GCB=/a/  CN}');
Error('\P{Is_GCB=/a/  CN}');
Expect(1, 9, '\p{Is_GCB=cn}', "");
Expect(0, 9, '\p{^Is_GCB=cn}', "");
Expect(0, 9, '\P{Is_GCB=cn}', "");
Expect(1, 9, '\P{^Is_GCB=cn}', "");
Expect(0, 10, '\p{Is_GCB=cn}', "");
Expect(1, 10, '\p{^Is_GCB=cn}', "");
Expect(1, 10, '\P{Is_GCB=cn}', "");
Expect(0, 10, '\P{^Is_GCB=cn}', "");
Expect(1, 9, '\p{Is_GCB=-_CN}', "");
Expect(0, 9, '\p{^Is_GCB=-_CN}', "");
Expect(0, 9, '\P{Is_GCB=-_CN}', "");
Expect(1, 9, '\P{^Is_GCB=-_CN}', "");
Expect(0, 10, '\p{Is_GCB=-_CN}', "");
Expect(1, 10, '\p{^Is_GCB=-_CN}', "");
Expect(1, 10, '\P{Is_GCB=-_CN}', "");
Expect(0, 10, '\P{^Is_GCB=-_CN}', "");
Error('\p{Grapheme_Cluster_Break=Prepend:=}');
Error('\P{Grapheme_Cluster_Break=Prepend:=}');
Expect(1, 3652, '\p{Grapheme_Cluster_Break=prepend}', "");
Expect(0, 3652, '\p{^Grapheme_Cluster_Break=prepend}', "");
Expect(0, 3652, '\P{Grapheme_Cluster_Break=prepend}', "");
Expect(1, 3652, '\P{^Grapheme_Cluster_Break=prepend}', "");
Expect(0, 3647, '\p{Grapheme_Cluster_Break=prepend}', "");
Expect(1, 3647, '\p{^Grapheme_Cluster_Break=prepend}', "");
Expect(1, 3647, '\P{Grapheme_Cluster_Break=prepend}', "");
Expect(0, 3647, '\P{^Grapheme_Cluster_Break=prepend}', "");
Expect(1, 3652, '\p{Grapheme_Cluster_Break= 	Prepend}', "");
Expect(0, 3652, '\p{^Grapheme_Cluster_Break= 	Prepend}', "");
Expect(0, 3652, '\P{Grapheme_Cluster_Break= 	Prepend}', "");
Expect(1, 3652, '\P{^Grapheme_Cluster_Break= 	Prepend}', "");
Expect(0, 3647, '\p{Grapheme_Cluster_Break= 	Prepend}', "");
Expect(1, 3647, '\p{^Grapheme_Cluster_Break= 	Prepend}', "");
Expect(1, 3647, '\P{Grapheme_Cluster_Break= 	Prepend}', "");
Expect(0, 3647, '\P{^Grapheme_Cluster_Break= 	Prepend}', "");
Error('\p{GCB= -PP:=}');
Error('\P{GCB= -PP:=}');
Expect(1, 3652, '\p{GCB=pp}', "");
Expect(0, 3652, '\p{^GCB=pp}', "");
Expect(0, 3652, '\P{GCB=pp}', "");
Expect(1, 3652, '\P{^GCB=pp}', "");
Expect(0, 3647, '\p{GCB=pp}', "");
Expect(1, 3647, '\p{^GCB=pp}', "");
Expect(1, 3647, '\P{GCB=pp}', "");
Expect(0, 3647, '\P{^GCB=pp}', "");
Expect(1, 3652, '\p{GCB=- PP}', "");
Expect(0, 3652, '\p{^GCB=- PP}', "");
Expect(0, 3652, '\P{GCB=- PP}', "");
Expect(1, 3652, '\P{^GCB=- PP}', "");
Expect(0, 3647, '\p{GCB=- PP}', "");
Expect(1, 3647, '\p{^GCB=- PP}', "");
Expect(1, 3647, '\P{GCB=- PP}', "");
Expect(0, 3647, '\P{^GCB=- PP}', "");
Error('\p{Is_Grapheme_Cluster_Break=-:=Prepend}');
Error('\P{Is_Grapheme_Cluster_Break=-:=Prepend}');
Expect(1, 3652, '\p{Is_Grapheme_Cluster_Break:	prepend}', "");
Expect(0, 3652, '\p{^Is_Grapheme_Cluster_Break:	prepend}', "");
Expect(0, 3652, '\P{Is_Grapheme_Cluster_Break:	prepend}', "");
Expect(1, 3652, '\P{^Is_Grapheme_Cluster_Break:	prepend}', "");
Expect(0, 3647, '\p{Is_Grapheme_Cluster_Break:	prepend}', "");
Expect(1, 3647, '\p{^Is_Grapheme_Cluster_Break:	prepend}', "");
Expect(1, 3647, '\P{Is_Grapheme_Cluster_Break:	prepend}', "");
Expect(0, 3647, '\P{^Is_Grapheme_Cluster_Break:	prepend}', "");
Expect(1, 3652, '\p{Is_Grapheme_Cluster_Break= 	prepend}', "");
Expect(0, 3652, '\p{^Is_Grapheme_Cluster_Break= 	prepend}', "");
Expect(0, 3652, '\P{Is_Grapheme_Cluster_Break= 	prepend}', "");
Expect(1, 3652, '\P{^Is_Grapheme_Cluster_Break= 	prepend}', "");
Expect(0, 3647, '\p{Is_Grapheme_Cluster_Break= 	prepend}', "");
Expect(1, 3647, '\p{^Is_Grapheme_Cluster_Break= 	prepend}', "");
Expect(1, 3647, '\P{Is_Grapheme_Cluster_Break= 	prepend}', "");
Expect(0, 3647, '\P{^Is_Grapheme_Cluster_Break= 	prepend}', "");
Error('\p{Is_GCB=_:=PP}');
Error('\P{Is_GCB=_:=PP}');
Expect(1, 3652, '\p{Is_GCB=pp}', "");
Expect(0, 3652, '\p{^Is_GCB=pp}', "");
Expect(0, 3652, '\P{Is_GCB=pp}', "");
Expect(1, 3652, '\P{^Is_GCB=pp}', "");
Expect(0, 3647, '\p{Is_GCB=pp}', "");
Expect(1, 3647, '\p{^Is_GCB=pp}', "");
Expect(1, 3647, '\P{Is_GCB=pp}', "");
Expect(0, 3647, '\P{^Is_GCB=pp}', "");
Expect(1, 3652, '\p{Is_GCB=	PP}', "");
Expect(0, 3652, '\p{^Is_GCB=	PP}', "");
Expect(0, 3652, '\P{Is_GCB=	PP}', "");
Expect(1, 3652, '\P{^Is_GCB=	PP}', "");
Expect(0, 3647, '\p{Is_GCB=	PP}', "");
Expect(1, 3647, '\p{^Is_GCB=	PP}', "");
Expect(1, 3647, '\P{Is_GCB=	PP}', "");
Expect(0, 3647, '\P{^Is_GCB=	PP}', "");
Error('\p{Grapheme_Cluster_Break:/a/_LV}');
Error('\P{Grapheme_Cluster_Break:/a/_LV}');
Expect(1, 44032, '\p{Grapheme_Cluster_Break=lv}', "");
Expect(0, 44032, '\p{^Grapheme_Cluster_Break=lv}', "");
Expect(0, 44032, '\P{Grapheme_Cluster_Break=lv}', "");
Expect(1, 44032, '\P{^Grapheme_Cluster_Break=lv}', "");
Expect(0, 44031, '\p{Grapheme_Cluster_Break=lv}', "");
Expect(1, 44031, '\p{^Grapheme_Cluster_Break=lv}', "");
Expect(1, 44031, '\P{Grapheme_Cluster_Break=lv}', "");
Expect(0, 44031, '\P{^Grapheme_Cluster_Break=lv}', "");
Expect(1, 44032, '\p{Grapheme_Cluster_Break=_	LV}', "");
Expect(0, 44032, '\p{^Grapheme_Cluster_Break=_	LV}', "");
Expect(0, 44032, '\P{Grapheme_Cluster_Break=_	LV}', "");
Expect(1, 44032, '\P{^Grapheme_Cluster_Break=_	LV}', "");
Expect(0, 44031, '\p{Grapheme_Cluster_Break=_	LV}', "");
Expect(1, 44031, '\p{^Grapheme_Cluster_Break=_	LV}', "");
Expect(1, 44031, '\P{Grapheme_Cluster_Break=_	LV}', "");
Expect(0, 44031, '\P{^Grapheme_Cluster_Break=_	LV}', "");
Error('\p{GCB=:=_LV}');
Error('\P{GCB=:=_LV}');
Expect(1, 44032, '\p{GCB:   lv}', "");
Expect(0, 44032, '\p{^GCB:   lv}', "");
Expect(0, 44032, '\P{GCB:   lv}', "");
Expect(1, 44032, '\P{^GCB:   lv}', "");
Expect(0, 44031, '\p{GCB:   lv}', "");
Expect(1, 44031, '\p{^GCB:   lv}', "");
Expect(1, 44031, '\P{GCB:   lv}', "");
Expect(0, 44031, '\P{^GCB:   lv}', "");
Expect(1, 44032, '\p{GCB:	-LV}', "");
Expect(0, 44032, '\p{^GCB:	-LV}', "");
Expect(0, 44032, '\P{GCB:	-LV}', "");
Expect(1, 44032, '\P{^GCB:	-LV}', "");
Expect(0, 44031, '\p{GCB:	-LV}', "");
Expect(1, 44031, '\p{^GCB:	-LV}', "");
Expect(1, 44031, '\P{GCB:	-LV}', "");
Expect(0, 44031, '\P{^GCB:	-LV}', "");
Error('\p{Is_Grapheme_Cluster_Break=:= 	lv}');
Error('\P{Is_Grapheme_Cluster_Break=:= 	lv}');
Expect(1, 44032, '\p{Is_Grapheme_Cluster_Break=lv}', "");
Expect(0, 44032, '\p{^Is_Grapheme_Cluster_Break=lv}', "");
Expect(0, 44032, '\P{Is_Grapheme_Cluster_Break=lv}', "");
Expect(1, 44032, '\P{^Is_Grapheme_Cluster_Break=lv}', "");
Expect(0, 44031, '\p{Is_Grapheme_Cluster_Break=lv}', "");
Expect(1, 44031, '\p{^Is_Grapheme_Cluster_Break=lv}', "");
Expect(1, 44031, '\P{Is_Grapheme_Cluster_Break=lv}', "");
Expect(0, 44031, '\P{^Is_Grapheme_Cluster_Break=lv}', "");
Expect(1, 44032, '\p{Is_Grapheme_Cluster_Break= lv}', "");
Expect(0, 44032, '\p{^Is_Grapheme_Cluster_Break= lv}', "");
Expect(0, 44032, '\P{Is_Grapheme_Cluster_Break= lv}', "");
Expect(1, 44032, '\P{^Is_Grapheme_Cluster_Break= lv}', "");
Expect(0, 44031, '\p{Is_Grapheme_Cluster_Break= lv}', "");
Expect(1, 44031, '\p{^Is_Grapheme_Cluster_Break= lv}', "");
Expect(1, 44031, '\P{Is_Grapheme_Cluster_Break= lv}', "");
Expect(0, 44031, '\P{^Is_Grapheme_Cluster_Break= lv}', "");
Error('\p{Is_GCB= /a/LV}');
Error('\P{Is_GCB= /a/LV}');
Expect(1, 44032, '\p{Is_GCB=lv}', "");
Expect(0, 44032, '\p{^Is_GCB=lv}', "");
Expect(0, 44032, '\P{Is_GCB=lv}', "");
Expect(1, 44032, '\P{^Is_GCB=lv}', "");
Expect(0, 44031, '\p{Is_GCB=lv}', "");
Expect(1, 44031, '\p{^Is_GCB=lv}', "");
Expect(1, 44031, '\P{Is_GCB=lv}', "");
Expect(0, 44031, '\P{^Is_GCB=lv}', "");
Expect(1, 44032, '\p{Is_GCB= lv}', "");
Expect(0, 44032, '\p{^Is_GCB= lv}', "");
Expect(0, 44032, '\P{Is_GCB= lv}', "");
Expect(1, 44032, '\P{^Is_GCB= lv}', "");
Expect(0, 44031, '\p{Is_GCB= lv}', "");
Expect(1, 44031, '\p{^Is_GCB= lv}', "");
Expect(1, 44031, '\P{Is_GCB= lv}', "");
Expect(0, 44031, '\P{^Is_GCB= lv}', "");
Error('\p{Grapheme_Cluster_Break=:= _CR}');
Error('\P{Grapheme_Cluster_Break=:= _CR}');
Expect(1, 13, '\p{Grapheme_Cluster_Break=cr}', "");
Expect(0, 13, '\p{^Grapheme_Cluster_Break=cr}', "");
Expect(0, 13, '\P{Grapheme_Cluster_Break=cr}', "");
Expect(1, 13, '\P{^Grapheme_Cluster_Break=cr}', "");
Expect(0, 12, '\p{Grapheme_Cluster_Break=cr}', "");
Expect(1, 12, '\p{^Grapheme_Cluster_Break=cr}', "");
Expect(1, 12, '\P{Grapheme_Cluster_Break=cr}', "");
Expect(0, 12, '\P{^Grapheme_Cluster_Break=cr}', "");
Expect(1, 13, '\p{Grapheme_Cluster_Break=-CR}', "");
Expect(0, 13, '\p{^Grapheme_Cluster_Break=-CR}', "");
Expect(0, 13, '\P{Grapheme_Cluster_Break=-CR}', "");
Expect(1, 13, '\P{^Grapheme_Cluster_Break=-CR}', "");
Expect(0, 12, '\p{Grapheme_Cluster_Break=-CR}', "");
Expect(1, 12, '\p{^Grapheme_Cluster_Break=-CR}', "");
Expect(1, 12, '\P{Grapheme_Cluster_Break=-CR}', "");
Expect(0, 12, '\P{^Grapheme_Cluster_Break=-CR}', "");
Error('\p{GCB=/a/-CR}');
Error('\P{GCB=/a/-CR}');
Expect(1, 13, '\p{GCB=cr}', "");
Expect(0, 13, '\p{^GCB=cr}', "");
Expect(0, 13, '\P{GCB=cr}', "");
Expect(1, 13, '\P{^GCB=cr}', "");
Expect(0, 12, '\p{GCB=cr}', "");
Expect(1, 12, '\p{^GCB=cr}', "");
Expect(1, 12, '\P{GCB=cr}', "");
Expect(0, 12, '\P{^GCB=cr}', "");
Expect(1, 13, '\p{GCB=		CR}', "");
Expect(0, 13, '\p{^GCB=		CR}', "");
Expect(0, 13, '\P{GCB=		CR}', "");
Expect(1, 13, '\P{^GCB=		CR}', "");
Expect(0, 12, '\p{GCB=		CR}', "");
Expect(1, 12, '\p{^GCB=		CR}', "");
Expect(1, 12, '\P{GCB=		CR}', "");
Expect(0, 12, '\P{^GCB=		CR}', "");
Error('\p{Is_Grapheme_Cluster_Break=	 CR:=}');
Error('\P{Is_Grapheme_Cluster_Break=	 CR:=}');
Expect(1, 13, '\p{Is_Grapheme_Cluster_Break=cr}', "");
Expect(0, 13, '\p{^Is_Grapheme_Cluster_Break=cr}', "");
Expect(0, 13, '\P{Is_Grapheme_Cluster_Break=cr}', "");
Expect(1, 13, '\P{^Is_Grapheme_Cluster_Break=cr}', "");
Expect(0, 12, '\p{Is_Grapheme_Cluster_Break=cr}', "");
Expect(1, 12, '\p{^Is_Grapheme_Cluster_Break=cr}', "");
Expect(1, 12, '\P{Is_Grapheme_Cluster_Break=cr}', "");
Expect(0, 12, '\P{^Is_Grapheme_Cluster_Break=cr}', "");
Expect(1, 13, '\p{Is_Grapheme_Cluster_Break= cr}', "");
Expect(0, 13, '\p{^Is_Grapheme_Cluster_Break= cr}', "");
Expect(0, 13, '\P{Is_Grapheme_Cluster_Break= cr}', "");
Expect(1, 13, '\P{^Is_Grapheme_Cluster_Break= cr}', "");
Expect(0, 12, '\p{Is_Grapheme_Cluster_Break= cr}', "");
Expect(1, 12, '\p{^Is_Grapheme_Cluster_Break= cr}', "");
Expect(1, 12, '\P{Is_Grapheme_Cluster_Break= cr}', "");
Expect(0, 12, '\P{^Is_Grapheme_Cluster_Break= cr}', "");
Error('\p{Is_GCB= _CR:=}');
Error('\P{Is_GCB= _CR:=}');
Expect(1, 13, '\p{Is_GCB:   cr}', "");
Expect(0, 13, '\p{^Is_GCB:   cr}', "");
Expect(0, 13, '\P{Is_GCB:   cr}', "");
Expect(1, 13, '\P{^Is_GCB:   cr}', "");
Expect(0, 12, '\p{Is_GCB:   cr}', "");
Expect(1, 12, '\p{^Is_GCB:   cr}', "");
Expect(1, 12, '\P{Is_GCB:   cr}', "");
Expect(0, 12, '\P{^Is_GCB:   cr}', "");
Expect(1, 13, '\p{Is_GCB=	_cr}', "");
Expect(0, 13, '\p{^Is_GCB=	_cr}', "");
Expect(0, 13, '\P{Is_GCB=	_cr}', "");
Expect(1, 13, '\P{^Is_GCB=	_cr}', "");
Expect(0, 12, '\p{Is_GCB=	_cr}', "");
Expect(1, 12, '\p{^Is_GCB=	_cr}', "");
Expect(1, 12, '\P{Is_GCB=	_cr}', "");
Expect(0, 12, '\P{^Is_GCB=	_cr}', "");
Error('\p{Grapheme_Cluster_Break=	:=OTHER}');
Error('\P{Grapheme_Cluster_Break=	:=OTHER}');
Expect(1, 126, '\p{Grapheme_Cluster_Break=other}', "");
Expect(0, 126, '\p{^Grapheme_Cluster_Break=other}', "");
Expect(0, 126, '\P{Grapheme_Cluster_Break=other}', "");
Expect(1, 126, '\P{^Grapheme_Cluster_Break=other}', "");
Expect(0, 31, '\p{Grapheme_Cluster_Break=other}', "");
Expect(1, 31, '\p{^Grapheme_Cluster_Break=other}', "");
Expect(1, 31, '\P{Grapheme_Cluster_Break=other}', "");
Expect(0, 31, '\P{^Grapheme_Cluster_Break=other}', "");
Expect(1, 126, '\p{Grapheme_Cluster_Break= Other}', "");
Expect(0, 126, '\p{^Grapheme_Cluster_Break= Other}', "");
Expect(0, 126, '\P{Grapheme_Cluster_Break= Other}', "");
Expect(1, 126, '\P{^Grapheme_Cluster_Break= Other}', "");
Expect(0, 31, '\p{Grapheme_Cluster_Break= Other}', "");
Expect(1, 31, '\p{^Grapheme_Cluster_Break= Other}', "");
Expect(1, 31, '\P{Grapheme_Cluster_Break= Other}', "");
Expect(0, 31, '\P{^Grapheme_Cluster_Break= Other}', "");
Error('\p{GCB=/a/		XX}');
Error('\P{GCB=/a/		XX}');
Expect(1, 126, '\p{GCB:	xx}', "");
Expect(0, 126, '\p{^GCB:	xx}', "");
Expect(0, 126, '\P{GCB:	xx}', "");
Expect(1, 126, '\P{^GCB:	xx}', "");
Expect(0, 31, '\p{GCB:	xx}', "");
Expect(1, 31, '\p{^GCB:	xx}', "");
Expect(1, 31, '\P{GCB:	xx}', "");
Expect(0, 31, '\P{^GCB:	xx}', "");
Expect(1, 126, '\p{GCB=		xx}', "");
Expect(0, 126, '\p{^GCB=		xx}', "");
Expect(0, 126, '\P{GCB=		xx}', "");
Expect(1, 126, '\P{^GCB=		xx}', "");
Expect(0, 31, '\p{GCB=		xx}', "");
Expect(1, 31, '\p{^GCB=		xx}', "");
Expect(1, 31, '\P{GCB=		xx}', "");
Expect(0, 31, '\P{^GCB=		xx}', "");
Error('\p{Is_Grapheme_Cluster_Break=/a/	-Other}');
Error('\P{Is_Grapheme_Cluster_Break=/a/	-Other}');
Expect(1, 126, '\p{Is_Grapheme_Cluster_Break=other}', "");
Expect(0, 126, '\p{^Is_Grapheme_Cluster_Break=other}', "");
Expect(0, 126, '\P{Is_Grapheme_Cluster_Break=other}', "");
Expect(1, 126, '\P{^Is_Grapheme_Cluster_Break=other}', "");
Expect(0, 31, '\p{Is_Grapheme_Cluster_Break=other}', "");
Expect(1, 31, '\p{^Is_Grapheme_Cluster_Break=other}', "");
Expect(1, 31, '\P{Is_Grapheme_Cluster_Break=other}', "");
Expect(0, 31, '\P{^Is_Grapheme_Cluster_Break=other}', "");
Expect(1, 126, '\p{Is_Grapheme_Cluster_Break=_-OTHER}', "");
Expect(0, 126, '\p{^Is_Grapheme_Cluster_Break=_-OTHER}', "");
Expect(0, 126, '\P{Is_Grapheme_Cluster_Break=_-OTHER}', "");
Expect(1, 126, '\P{^Is_Grapheme_Cluster_Break=_-OTHER}', "");
Expect(0, 31, '\p{Is_Grapheme_Cluster_Break=_-OTHER}', "");
Expect(1, 31, '\p{^Is_Grapheme_Cluster_Break=_-OTHER}', "");
Expect(1, 31, '\P{Is_Grapheme_Cluster_Break=_-OTHER}', "");
Expect(0, 31, '\P{^Is_Grapheme_Cluster_Break=_-OTHER}', "");
Error('\p{Is_GCB=_/a/XX}');
Error('\P{Is_GCB=_/a/XX}');
Expect(1, 126, '\p{Is_GCB:xx}', "");
Expect(0, 126, '\p{^Is_GCB:xx}', "");
Expect(0, 126, '\P{Is_GCB:xx}', "");
Expect(1, 126, '\P{^Is_GCB:xx}', "");
Expect(0, 31, '\p{Is_GCB:xx}', "");
Expect(1, 31, '\p{^Is_GCB:xx}', "");
Expect(1, 31, '\P{Is_GCB:xx}', "");
Expect(0, 31, '\P{^Is_GCB:xx}', "");
Expect(1, 126, '\p{Is_GCB=		XX}', "");
Expect(0, 126, '\p{^Is_GCB=		XX}', "");
Expect(0, 126, '\P{Is_GCB=		XX}', "");
Expect(1, 126, '\P{^Is_GCB=		XX}', "");
Expect(0, 31, '\p{Is_GCB=		XX}', "");
Expect(1, 31, '\p{^Is_GCB=		XX}', "");
Expect(1, 31, '\P{Is_GCB=		XX}', "");
Expect(0, 31, '\P{^Is_GCB=		XX}', "");
Error('\p{Grapheme_Cluster_Break=-/a/Extend}');
Error('\P{Grapheme_Cluster_Break=-/a/Extend}');
Expect(1, 879, '\p{Grapheme_Cluster_Break=extend}', "");
Expect(0, 879, '\p{^Grapheme_Cluster_Break=extend}', "");
Expect(0, 879, '\P{Grapheme_Cluster_Break=extend}', "");
Expect(1, 879, '\P{^Grapheme_Cluster_Break=extend}', "");
Expect(0, 767, '\p{Grapheme_Cluster_Break=extend}', "");
Expect(1, 767, '\p{^Grapheme_Cluster_Break=extend}', "");
Expect(1, 767, '\P{Grapheme_Cluster_Break=extend}', "");
Expect(0, 767, '\P{^Grapheme_Cluster_Break=extend}', "");
Expect(1, 879, '\p{Grapheme_Cluster_Break=	extend}', "");
Expect(0, 879, '\p{^Grapheme_Cluster_Break=	extend}', "");
Expect(0, 879, '\P{Grapheme_Cluster_Break=	extend}', "");
Expect(1, 879, '\P{^Grapheme_Cluster_Break=	extend}', "");
Expect(0, 767, '\p{Grapheme_Cluster_Break=	extend}', "");
Expect(1, 767, '\p{^Grapheme_Cluster_Break=	extend}', "");
Expect(1, 767, '\P{Grapheme_Cluster_Break=	extend}', "");
Expect(0, 767, '\P{^Grapheme_Cluster_Break=	extend}', "");
Error('\p{GCB:	- EX/a/}');
Error('\P{GCB:	- EX/a/}');
Expect(1, 879, '\p{GCB=ex}', "");
Expect(0, 879, '\p{^GCB=ex}', "");
Expect(0, 879, '\P{GCB=ex}', "");
Expect(1, 879, '\P{^GCB=ex}', "");
Expect(0, 767, '\p{GCB=ex}', "");
Expect(1, 767, '\p{^GCB=ex}', "");
Expect(1, 767, '\P{GCB=ex}', "");
Expect(0, 767, '\P{^GCB=ex}', "");
Expect(1, 879, '\p{GCB=	ex}', "");
Expect(0, 879, '\p{^GCB=	ex}', "");
Expect(0, 879, '\P{GCB=	ex}', "");
Expect(1, 879, '\P{^GCB=	ex}', "");
Expect(0, 767, '\p{GCB=	ex}', "");
Expect(1, 767, '\p{^GCB=	ex}', "");
Expect(1, 767, '\P{GCB=	ex}', "");
Expect(0, 767, '\P{^GCB=	ex}', "");
Error('\p{Is_Grapheme_Cluster_Break=/a/ _EXTEND}');
Error('\P{Is_Grapheme_Cluster_Break=/a/ _EXTEND}');
Expect(1, 879, '\p{Is_Grapheme_Cluster_Break=extend}', "");
Expect(0, 879, '\p{^Is_Grapheme_Cluster_Break=extend}', "");
Expect(0, 879, '\P{Is_Grapheme_Cluster_Break=extend}', "");
Expect(1, 879, '\P{^Is_Grapheme_Cluster_Break=extend}', "");
Expect(0, 767, '\p{Is_Grapheme_Cluster_Break=extend}', "");
Expect(1, 767, '\p{^Is_Grapheme_Cluster_Break=extend}', "");
Expect(1, 767, '\P{Is_Grapheme_Cluster_Break=extend}', "");
Expect(0, 767, '\P{^Is_Grapheme_Cluster_Break=extend}', "");
Expect(1, 879, '\p{Is_Grapheme_Cluster_Break= EXTEND}', "");
Expect(0, 879, '\p{^Is_Grapheme_Cluster_Break= EXTEND}', "");
Expect(0, 879, '\P{Is_Grapheme_Cluster_Break= EXTEND}', "");
Expect(1, 879, '\P{^Is_Grapheme_Cluster_Break= EXTEND}', "");
Expect(0, 767, '\p{Is_Grapheme_Cluster_Break= EXTEND}', "");
Expect(1, 767, '\p{^Is_Grapheme_Cluster_Break= EXTEND}', "");
Expect(1, 767, '\P{Is_Grapheme_Cluster_Break= EXTEND}', "");
Expect(0, 767, '\P{^Is_Grapheme_Cluster_Break= EXTEND}', "");
Error('\p{Is_GCB=:=	 EX}');
Error('\P{Is_GCB=:=	 EX}');
Expect(1, 879, '\p{Is_GCB=ex}', "");
Expect(0, 879, '\p{^Is_GCB=ex}', "");
Expect(0, 879, '\P{Is_GCB=ex}', "");
Expect(1, 879, '\P{^Is_GCB=ex}', "");
Expect(0, 767, '\p{Is_GCB=ex}', "");
Expect(1, 767, '\p{^Is_GCB=ex}', "");
Expect(1, 767, '\P{Is_GCB=ex}', "");
Expect(0, 767, '\P{^Is_GCB=ex}', "");
Expect(1, 879, '\p{Is_GCB= 	EX}', "");
Expect(0, 879, '\p{^Is_GCB= 	EX}', "");
Expect(0, 879, '\P{Is_GCB= 	EX}', "");
Expect(1, 879, '\P{^Is_GCB= 	EX}', "");
Expect(0, 767, '\p{Is_GCB= 	EX}', "");
Expect(1, 767, '\p{^Is_GCB= 	EX}', "");
Expect(1, 767, '\P{Is_GCB= 	EX}', "");
Expect(0, 767, '\P{^Is_GCB= 	EX}', "");
Error('\p{Grapheme_Cluster_Break=	-v/a/}');
Error('\P{Grapheme_Cluster_Break=	-v/a/}');
Expect(1, 4519, '\p{Grapheme_Cluster_Break=v}', "");
Expect(0, 4519, '\p{^Grapheme_Cluster_Break=v}', "");
Expect(0, 4519, '\P{Grapheme_Cluster_Break=v}', "");
Expect(1, 4519, '\P{^Grapheme_Cluster_Break=v}', "");
Expect(0, 4447, '\p{Grapheme_Cluster_Break=v}', "");
Expect(1, 4447, '\p{^Grapheme_Cluster_Break=v}', "");
Expect(1, 4447, '\P{Grapheme_Cluster_Break=v}', "");
Expect(0, 4447, '\P{^Grapheme_Cluster_Break=v}', "");
Expect(1, 4519, '\p{Grapheme_Cluster_Break=-V}', "");
Expect(0, 4519, '\p{^Grapheme_Cluster_Break=-V}', "");
Expect(0, 4519, '\P{Grapheme_Cluster_Break=-V}', "");
Expect(1, 4519, '\P{^Grapheme_Cluster_Break=-V}', "");
Expect(0, 4447, '\p{Grapheme_Cluster_Break=-V}', "");
Expect(1, 4447, '\p{^Grapheme_Cluster_Break=-V}', "");
Expect(1, 4447, '\P{Grapheme_Cluster_Break=-V}', "");
Expect(0, 4447, '\P{^Grapheme_Cluster_Break=-V}', "");
Error('\p{GCB= -v/a/}');
Error('\P{GCB= -v/a/}');
Expect(1, 4519, '\p{GCB=v}', "");
Expect(0, 4519, '\p{^GCB=v}', "");
Expect(0, 4519, '\P{GCB=v}', "");
Expect(1, 4519, '\P{^GCB=v}', "");
Expect(0, 4447, '\p{GCB=v}', "");
Expect(1, 4447, '\p{^GCB=v}', "");
Expect(1, 4447, '\P{GCB=v}', "");
Expect(0, 4447, '\P{^GCB=v}', "");
Expect(1, 4519, '\p{GCB= 	v}', "");
Expect(0, 4519, '\p{^GCB= 	v}', "");
Expect(0, 4519, '\P{GCB= 	v}', "");
Expect(1, 4519, '\P{^GCB= 	v}', "");
Expect(0, 4447, '\p{GCB= 	v}', "");
Expect(1, 4447, '\p{^GCB= 	v}', "");
Expect(1, 4447, '\P{GCB= 	v}', "");
Expect(0, 4447, '\P{^GCB= 	v}', "");
Error('\p{Is_Grapheme_Cluster_Break=_v:=}');
Error('\P{Is_Grapheme_Cluster_Break=_v:=}');
Expect(1, 4519, '\p{Is_Grapheme_Cluster_Break:	v}', "");
Expect(0, 4519, '\p{^Is_Grapheme_Cluster_Break:	v}', "");
Expect(0, 4519, '\P{Is_Grapheme_Cluster_Break:	v}', "");
Expect(1, 4519, '\P{^Is_Grapheme_Cluster_Break:	v}', "");
Expect(0, 4447, '\p{Is_Grapheme_Cluster_Break:	v}', "");
Expect(1, 4447, '\p{^Is_Grapheme_Cluster_Break:	v}', "");
Expect(1, 4447, '\P{Is_Grapheme_Cluster_Break:	v}', "");
Expect(0, 4447, '\P{^Is_Grapheme_Cluster_Break:	v}', "");
Expect(1, 4519, '\p{Is_Grapheme_Cluster_Break=  V}', "");
Expect(0, 4519, '\p{^Is_Grapheme_Cluster_Break=  V}', "");
Expect(0, 4519, '\P{Is_Grapheme_Cluster_Break=  V}', "");
Expect(1, 4519, '\P{^Is_Grapheme_Cluster_Break=  V}', "");
Expect(0, 4447, '\p{Is_Grapheme_Cluster_Break=  V}', "");
Expect(1, 4447, '\p{^Is_Grapheme_Cluster_Break=  V}', "");
Expect(1, 4447, '\P{Is_Grapheme_Cluster_Break=  V}', "");
Expect(0, 4447, '\P{^Is_Grapheme_Cluster_Break=  V}', "");
Error('\p{Is_GCB=:= V}');
Error('\P{Is_GCB=:= V}');
Expect(1, 4519, '\p{Is_GCB=v}', "");
Expect(0, 4519, '\p{^Is_GCB=v}', "");
Expect(0, 4519, '\P{Is_GCB=v}', "");
Expect(1, 4519, '\P{^Is_GCB=v}', "");
Expect(0, 4447, '\p{Is_GCB=v}', "");
Expect(1, 4447, '\p{^Is_GCB=v}', "");
Expect(1, 4447, '\P{Is_GCB=v}', "");
Expect(0, 4447, '\P{^Is_GCB=v}', "");
Expect(1, 4519, '\p{Is_GCB:	 V}', "");
Expect(0, 4519, '\p{^Is_GCB:	 V}', "");
Expect(0, 4519, '\P{Is_GCB:	 V}', "");
Expect(1, 4519, '\P{^Is_GCB:	 V}', "");
Expect(0, 4447, '\p{Is_GCB:	 V}', "");
Expect(1, 4447, '\p{^Is_GCB:	 V}', "");
Expect(1, 4447, '\P{Is_GCB:	 V}', "");
Expect(0, 4447, '\P{^Is_GCB:	 V}', "");
Error('\p{Grapheme_Cluster_Break=:=	_lf}');
Error('\P{Grapheme_Cluster_Break=:=	_lf}');
Expect(1, 10, '\p{Grapheme_Cluster_Break:	lf}', "");
Expect(0, 10, '\p{^Grapheme_Cluster_Break:	lf}', "");
Expect(0, 10, '\P{Grapheme_Cluster_Break:	lf}', "");
Expect(1, 10, '\P{^Grapheme_Cluster_Break:	lf}', "");
Expect(0, 9, '\p{Grapheme_Cluster_Break:	lf}', "");
Expect(1, 9, '\p{^Grapheme_Cluster_Break:	lf}', "");
Expect(1, 9, '\P{Grapheme_Cluster_Break:	lf}', "");
Expect(0, 9, '\P{^Grapheme_Cluster_Break:	lf}', "");
Expect(1, 10, '\p{Grapheme_Cluster_Break=	 LF}', "");
Expect(0, 10, '\p{^Grapheme_Cluster_Break=	 LF}', "");
Expect(0, 10, '\P{Grapheme_Cluster_Break=	 LF}', "");
Expect(1, 10, '\P{^Grapheme_Cluster_Break=	 LF}', "");
Expect(0, 9, '\p{Grapheme_Cluster_Break=	 LF}', "");
Expect(1, 9, '\p{^Grapheme_Cluster_Break=	 LF}', "");
Expect(1, 9, '\P{Grapheme_Cluster_Break=	 LF}', "");
Expect(0, 9, '\P{^Grapheme_Cluster_Break=	 LF}', "");
Error('\p{GCB=:= 	LF}');
Error('\P{GCB=:= 	LF}');
Expect(1, 10, '\p{GCB=lf}', "");
Expect(0, 10, '\p{^GCB=lf}', "");
Expect(0, 10, '\P{GCB=lf}', "");
Expect(1, 10, '\P{^GCB=lf}', "");
Expect(0, 9, '\p{GCB=lf}', "");
Expect(1, 9, '\p{^GCB=lf}', "");
Expect(1, 9, '\P{GCB=lf}', "");
Expect(0, 9, '\P{^GCB=lf}', "");
Expect(1, 10, '\p{GCB=- LF}', "");
Expect(0, 10, '\p{^GCB=- LF}', "");
Expect(0, 10, '\P{GCB=- LF}', "");
Expect(1, 10, '\P{^GCB=- LF}', "");
Expect(0, 9, '\p{GCB=- LF}', "");
Expect(1, 9, '\p{^GCB=- LF}', "");
Expect(1, 9, '\P{GCB=- LF}', "");
Expect(0, 9, '\P{^GCB=- LF}', "");
Error('\p{Is_Grapheme_Cluster_Break=	lf/a/}');
Error('\P{Is_Grapheme_Cluster_Break=	lf/a/}');
Expect(1, 10, '\p{Is_Grapheme_Cluster_Break=lf}', "");
Expect(0, 10, '\p{^Is_Grapheme_Cluster_Break=lf}', "");
Expect(0, 10, '\P{Is_Grapheme_Cluster_Break=lf}', "");
Expect(1, 10, '\P{^Is_Grapheme_Cluster_Break=lf}', "");
Expect(0, 9, '\p{Is_Grapheme_Cluster_Break=lf}', "");
Expect(1, 9, '\p{^Is_Grapheme_Cluster_Break=lf}', "");
Expect(1, 9, '\P{Is_Grapheme_Cluster_Break=lf}', "");
Expect(0, 9, '\P{^Is_Grapheme_Cluster_Break=lf}', "");
Expect(1, 10, '\p{Is_Grapheme_Cluster_Break=__LF}', "");
Expect(0, 10, '\p{^Is_Grapheme_Cluster_Break=__LF}', "");
Expect(0, 10, '\P{Is_Grapheme_Cluster_Break=__LF}', "");
Expect(1, 10, '\P{^Is_Grapheme_Cluster_Break=__LF}', "");
Expect(0, 9, '\p{Is_Grapheme_Cluster_Break=__LF}', "");
Expect(1, 9, '\p{^Is_Grapheme_Cluster_Break=__LF}', "");
Expect(1, 9, '\P{Is_Grapheme_Cluster_Break=__LF}', "");
Expect(0, 9, '\P{^Is_Grapheme_Cluster_Break=__LF}', "");
Error('\p{Is_GCB=  LF:=}');
Error('\P{Is_GCB=  LF:=}');
Expect(1, 10, '\p{Is_GCB=lf}', "");
Expect(0, 10, '\p{^Is_GCB=lf}', "");
Expect(0, 10, '\P{Is_GCB=lf}', "");
Expect(1, 10, '\P{^Is_GCB=lf}', "");
Expect(0, 9, '\p{Is_GCB=lf}', "");
Expect(1, 9, '\p{^Is_GCB=lf}', "");
Expect(1, 9, '\P{Is_GCB=lf}', "");
Expect(0, 9, '\P{^Is_GCB=lf}', "");
Expect(1, 10, '\p{Is_GCB= -LF}', "");
Expect(0, 10, '\p{^Is_GCB= -LF}', "");
Expect(0, 10, '\P{Is_GCB= -LF}', "");
Expect(1, 10, '\P{^Is_GCB= -LF}', "");
Expect(0, 9, '\p{Is_GCB= -LF}', "");
Expect(1, 9, '\p{^Is_GCB= -LF}', "");
Expect(1, 9, '\P{Is_GCB= -LF}', "");
Expect(0, 9, '\P{^Is_GCB= -LF}', "");
Error('\p{Grapheme_Cluster_Break:	  l:=}');
Error('\P{Grapheme_Cluster_Break:	  l:=}');
Expect(1, 4447, '\p{Grapheme_Cluster_Break=l}', "");
Expect(0, 4447, '\p{^Grapheme_Cluster_Break=l}', "");
Expect(0, 4447, '\P{Grapheme_Cluster_Break=l}', "");
Expect(1, 4447, '\P{^Grapheme_Cluster_Break=l}', "");
Expect(0, 4351, '\p{Grapheme_Cluster_Break=l}', "");
Expect(1, 4351, '\p{^Grapheme_Cluster_Break=l}', "");
Expect(1, 4351, '\P{Grapheme_Cluster_Break=l}', "");
Expect(0, 4351, '\P{^Grapheme_Cluster_Break=l}', "");
Expect(1, 4447, '\p{Grapheme_Cluster_Break=L}', "");
Expect(0, 4447, '\p{^Grapheme_Cluster_Break=L}', "");
Expect(0, 4447, '\P{Grapheme_Cluster_Break=L}', "");
Expect(1, 4447, '\P{^Grapheme_Cluster_Break=L}', "");
Expect(0, 4351, '\p{Grapheme_Cluster_Break=L}', "");
Expect(1, 4351, '\p{^Grapheme_Cluster_Break=L}', "");
Expect(1, 4351, '\P{Grapheme_Cluster_Break=L}', "");
Expect(0, 4351, '\P{^Grapheme_Cluster_Break=L}', "");
Error('\p{GCB=	_L:=}');
Error('\P{GCB=	_L:=}');
Expect(1, 4447, '\p{GCB=l}', "");
Expect(0, 4447, '\p{^GCB=l}', "");
Expect(0, 4447, '\P{GCB=l}', "");
Expect(1, 4447, '\P{^GCB=l}', "");
Expect(0, 4351, '\p{GCB=l}', "");
Expect(1, 4351, '\p{^GCB=l}', "");
Expect(1, 4351, '\P{GCB=l}', "");
Expect(0, 4351, '\P{^GCB=l}', "");
Expect(1, 4447, '\p{GCB=-_L}', "");
Expect(0, 4447, '\p{^GCB=-_L}', "");
Expect(0, 4447, '\P{GCB=-_L}', "");
Expect(1, 4447, '\P{^GCB=-_L}', "");
Expect(0, 4351, '\p{GCB=-_L}', "");
Expect(1, 4351, '\p{^GCB=-_L}', "");
Expect(1, 4351, '\P{GCB=-_L}', "");
Expect(0, 4351, '\P{^GCB=-_L}', "");
Error('\p{Is_Grapheme_Cluster_Break:  L:=}');
Error('\P{Is_Grapheme_Cluster_Break:  L:=}');
Expect(1, 4447, '\p{Is_Grapheme_Cluster_Break=l}', "");
Expect(0, 4447, '\p{^Is_Grapheme_Cluster_Break=l}', "");
Expect(0, 4447, '\P{Is_Grapheme_Cluster_Break=l}', "");
Expect(1, 4447, '\P{^Is_Grapheme_Cluster_Break=l}', "");
Expect(0, 4351, '\p{Is_Grapheme_Cluster_Break=l}', "");
Expect(1, 4351, '\p{^Is_Grapheme_Cluster_Break=l}', "");
Expect(1, 4351, '\P{Is_Grapheme_Cluster_Break=l}', "");
Expect(0, 4351, '\P{^Is_Grapheme_Cluster_Break=l}', "");
Expect(1, 4447, '\p{Is_Grapheme_Cluster_Break=__L}', "");
Expect(0, 4447, '\p{^Is_Grapheme_Cluster_Break=__L}', "");
Expect(0, 4447, '\P{Is_Grapheme_Cluster_Break=__L}', "");
Expect(1, 4447, '\P{^Is_Grapheme_Cluster_Break=__L}', "");
Expect(0, 4351, '\p{Is_Grapheme_Cluster_Break=__L}', "");
Expect(1, 4351, '\p{^Is_Grapheme_Cluster_Break=__L}', "");
Expect(1, 4351, '\P{Is_Grapheme_Cluster_Break=__L}', "");
Expect(0, 4351, '\P{^Is_Grapheme_Cluster_Break=__L}', "");
Error('\p{Is_GCB= :=l}');
Error('\P{Is_GCB= :=l}');
Expect(1, 4447, '\p{Is_GCB:   l}', "");
Expect(0, 4447, '\p{^Is_GCB:   l}', "");
Expect(0, 4447, '\P{Is_GCB:   l}', "");
Expect(1, 4447, '\P{^Is_GCB:   l}', "");
Expect(0, 4351, '\p{Is_GCB:   l}', "");
Expect(1, 4351, '\p{^Is_GCB:   l}', "");
Expect(1, 4351, '\P{Is_GCB:   l}', "");
Expect(0, 4351, '\P{^Is_GCB:   l}', "");
Expect(1, 4447, '\p{Is_GCB=	_L}', "");
Expect(0, 4447, '\p{^Is_GCB=	_L}', "");
Expect(0, 4447, '\P{Is_GCB=	_L}', "");
Expect(1, 4447, '\P{^Is_GCB=	_L}', "");
Expect(0, 4351, '\p{Is_GCB=	_L}', "");
Expect(1, 4351, '\p{^Is_GCB=	_L}', "");
Expect(1, 4351, '\P{Is_GCB=	_L}', "");
Expect(0, 4351, '\P{^Is_GCB=	_L}', "");
Error('\p{Logical_Order_Exception=	-Yes/a/}');
Error('\P{Logical_Order_Exception=	-Yes/a/}');
Expect(1, 3652, '\p{Logical_Order_Exception=yes}', "");
Expect(0, 3652, '\p{^Logical_Order_Exception=yes}', "");
Expect(0, 3652, '\P{Logical_Order_Exception=yes}', "");
Expect(1, 3652, '\P{^Logical_Order_Exception=yes}', "");
Expect(0, 3647, '\p{Logical_Order_Exception=yes}', "");
Expect(1, 3647, '\p{^Logical_Order_Exception=yes}', "");
Expect(1, 3647, '\P{Logical_Order_Exception=yes}', "");
Expect(0, 3647, '\P{^Logical_Order_Exception=yes}', "");
Expect(1, 3652, '\p{Logical_Order_Exception=	_Yes}', "");
Expect(0, 3652, '\p{^Logical_Order_Exception=	_Yes}', "");
Expect(0, 3652, '\P{Logical_Order_Exception=	_Yes}', "");
Expect(1, 3652, '\P{^Logical_Order_Exception=	_Yes}', "");
Expect(0, 3647, '\p{Logical_Order_Exception=	_Yes}', "");
Expect(1, 3647, '\p{^Logical_Order_Exception=	_Yes}', "");
Expect(1, 3647, '\P{Logical_Order_Exception=	_Yes}', "");
Expect(0, 3647, '\P{^Logical_Order_Exception=	_Yes}', "");
Error('\p{LOE::=	y}');
Error('\P{LOE::=	y}');
Expect(1, 3652, '\p{LOE: y}', "");
Expect(0, 3652, '\p{^LOE: y}', "");
Expect(0, 3652, '\P{LOE: y}', "");
Expect(1, 3652, '\P{^LOE: y}', "");
Expect(0, 3647, '\p{LOE: y}', "");
Expect(1, 3647, '\p{^LOE: y}', "");
Expect(1, 3647, '\P{LOE: y}', "");
Expect(0, 3647, '\P{^LOE: y}', "");
Expect(1, 3652, '\p{LOE= Y}', "");
Expect(0, 3652, '\p{^LOE= Y}', "");
Expect(0, 3652, '\P{LOE= Y}', "");
Expect(1, 3652, '\P{^LOE= Y}', "");
Expect(0, 3647, '\p{LOE= Y}', "");
Expect(1, 3647, '\p{^LOE= Y}', "");
Expect(1, 3647, '\P{LOE= Y}', "");
Expect(0, 3647, '\P{^LOE= Y}', "");
Error('\p{Is_Logical_Order_Exception=  t/a/}');
Error('\P{Is_Logical_Order_Exception=  t/a/}');
Expect(1, 3652, '\p{Is_Logical_Order_Exception=t}', "");
Expect(0, 3652, '\p{^Is_Logical_Order_Exception=t}', "");
Expect(0, 3652, '\P{Is_Logical_Order_Exception=t}', "");
Expect(1, 3652, '\P{^Is_Logical_Order_Exception=t}', "");
Expect(0, 3647, '\p{Is_Logical_Order_Exception=t}', "");
Expect(1, 3647, '\p{^Is_Logical_Order_Exception=t}', "");
Expect(1, 3647, '\P{Is_Logical_Order_Exception=t}', "");
Expect(0, 3647, '\P{^Is_Logical_Order_Exception=t}', "");
Expect(1, 3652, '\p{Is_Logical_Order_Exception:	-T}', "");
Expect(0, 3652, '\p{^Is_Logical_Order_Exception:	-T}', "");
Expect(0, 3652, '\P{Is_Logical_Order_Exception:	-T}', "");
Expect(1, 3652, '\P{^Is_Logical_Order_Exception:	-T}', "");
Expect(0, 3647, '\p{Is_Logical_Order_Exception:	-T}', "");
Expect(1, 3647, '\p{^Is_Logical_Order_Exception:	-T}', "");
Expect(1, 3647, '\P{Is_Logical_Order_Exception:	-T}', "");
Expect(0, 3647, '\P{^Is_Logical_Order_Exception:	-T}', "");
Error('\p{Is_LOE=	True:=}');
Error('\P{Is_LOE=	True:=}');
Expect(1, 3652, '\p{Is_LOE: true}', "");
Expect(0, 3652, '\p{^Is_LOE: true}', "");
Expect(0, 3652, '\P{Is_LOE: true}', "");
Expect(1, 3652, '\P{^Is_LOE: true}', "");
Expect(0, 3647, '\p{Is_LOE: true}', "");
Expect(1, 3647, '\p{^Is_LOE: true}', "");
Expect(1, 3647, '\P{Is_LOE: true}', "");
Expect(0, 3647, '\P{^Is_LOE: true}', "");
Expect(1, 3652, '\p{Is_LOE=_ true}', "");
Expect(0, 3652, '\p{^Is_LOE=_ true}', "");
Expect(0, 3652, '\P{Is_LOE=_ true}', "");
Expect(1, 3652, '\P{^Is_LOE=_ true}', "");
Expect(0, 3647, '\p{Is_LOE=_ true}', "");
Expect(1, 3647, '\p{^Is_LOE=_ true}', "");
Expect(1, 3647, '\P{Is_LOE=_ true}', "");
Expect(0, 3647, '\P{^Is_LOE=_ true}', "");
Error('\p{Logical_Order_Exception=:=- No}');
Error('\P{Logical_Order_Exception=:=- No}');
Expect(1, 3647, '\p{Logical_Order_Exception=no}', "");
Expect(0, 3647, '\p{^Logical_Order_Exception=no}', "");
Expect(0, 3647, '\P{Logical_Order_Exception=no}', "");
Expect(1, 3647, '\P{^Logical_Order_Exception=no}', "");
Expect(0, 3652, '\p{Logical_Order_Exception=no}', "");
Expect(1, 3652, '\p{^Logical_Order_Exception=no}', "");
Expect(1, 3652, '\P{Logical_Order_Exception=no}', "");
Expect(0, 3652, '\P{^Logical_Order_Exception=no}', "");
Expect(1, 3647, '\p{Logical_Order_Exception=-No}', "");
Expect(0, 3647, '\p{^Logical_Order_Exception=-No}', "");
Expect(0, 3647, '\P{Logical_Order_Exception=-No}', "");
Expect(1, 3647, '\P{^Logical_Order_Exception=-No}', "");
Expect(0, 3652, '\p{Logical_Order_Exception=-No}', "");
Expect(1, 3652, '\p{^Logical_Order_Exception=-No}', "");
Expect(1, 3652, '\P{Logical_Order_Exception=-No}', "");
Expect(0, 3652, '\P{^Logical_Order_Exception=-No}', "");
Error('\p{LOE=_	N:=}');
Error('\P{LOE=_	N:=}');
Expect(1, 3647, '\p{LOE=n}', "");
Expect(0, 3647, '\p{^LOE=n}', "");
Expect(0, 3647, '\P{LOE=n}', "");
Expect(1, 3647, '\P{^LOE=n}', "");
Expect(0, 3652, '\p{LOE=n}', "");
Expect(1, 3652, '\p{^LOE=n}', "");
Expect(1, 3652, '\P{LOE=n}', "");
Expect(0, 3652, '\P{^LOE=n}', "");
Expect(1, 3647, '\p{LOE=	_n}', "");
Expect(0, 3647, '\p{^LOE=	_n}', "");
Expect(0, 3647, '\P{LOE=	_n}', "");
Expect(1, 3647, '\P{^LOE=	_n}', "");
Expect(0, 3652, '\p{LOE=	_n}', "");
Expect(1, 3652, '\p{^LOE=	_n}', "");
Expect(1, 3652, '\P{LOE=	_n}', "");
Expect(0, 3652, '\P{^LOE=	_n}', "");
Error('\p{Is_Logical_Order_Exception=__f/a/}');
Error('\P{Is_Logical_Order_Exception=__f/a/}');
Expect(1, 3647, '\p{Is_Logical_Order_Exception=f}', "");
Expect(0, 3647, '\p{^Is_Logical_Order_Exception=f}', "");
Expect(0, 3647, '\P{Is_Logical_Order_Exception=f}', "");
Expect(1, 3647, '\P{^Is_Logical_Order_Exception=f}', "");
Expect(0, 3652, '\p{Is_Logical_Order_Exception=f}', "");
Expect(1, 3652, '\p{^Is_Logical_Order_Exception=f}', "");
Expect(1, 3652, '\P{Is_Logical_Order_Exception=f}', "");
Expect(0, 3652, '\P{^Is_Logical_Order_Exception=f}', "");
Expect(1, 3647, '\p{Is_Logical_Order_Exception=	 f}', "");
Expect(0, 3647, '\p{^Is_Logical_Order_Exception=	 f}', "");
Expect(0, 3647, '\P{Is_Logical_Order_Exception=	 f}', "");
Expect(1, 3647, '\P{^Is_Logical_Order_Exception=	 f}', "");
Expect(0, 3652, '\p{Is_Logical_Order_Exception=	 f}', "");
Expect(1, 3652, '\p{^Is_Logical_Order_Exception=	 f}', "");
Expect(1, 3652, '\P{Is_Logical_Order_Exception=	 f}', "");
Expect(0, 3652, '\P{^Is_Logical_Order_Exception=	 f}', "");
Error('\p{Is_LOE= False/a/}');
Error('\P{Is_LOE= False/a/}');
Expect(1, 3647, '\p{Is_LOE=false}', "");
Expect(0, 3647, '\p{^Is_LOE=false}', "");
Expect(0, 3647, '\P{Is_LOE=false}', "");
Expect(1, 3647, '\P{^Is_LOE=false}', "");
Expect(0, 3652, '\p{Is_LOE=false}', "");
Expect(1, 3652, '\p{^Is_LOE=false}', "");
Expect(1, 3652, '\P{Is_LOE=false}', "");
Expect(0, 3652, '\P{^Is_LOE=false}', "");
Expect(1, 3647, '\p{Is_LOE=_ FALSE}', "");
Expect(0, 3647, '\p{^Is_LOE=_ FALSE}', "");
Expect(0, 3647, '\P{Is_LOE=_ FALSE}', "");
Expect(1, 3647, '\P{^Is_LOE=_ FALSE}', "");
Expect(0, 3652, '\p{Is_LOE=_ FALSE}', "");
Expect(1, 3652, '\p{^Is_LOE=_ FALSE}', "");
Expect(1, 3652, '\P{Is_LOE=_ FALSE}', "");
Expect(0, 3652, '\P{^Is_LOE=_ FALSE}', "");
Error('\p{Bidi_Mirrored=--YES/a/}');
Error('\P{Bidi_Mirrored=--YES/a/}');
Expect(1, 41, '\p{Bidi_Mirrored: yes}', "");
Expect(0, 41, '\p{^Bidi_Mirrored: yes}', "");
Expect(0, 41, '\P{Bidi_Mirrored: yes}', "");
Expect(1, 41, '\P{^Bidi_Mirrored: yes}', "");
Expect(0, 39, '\p{Bidi_Mirrored: yes}', "");
Expect(1, 39, '\p{^Bidi_Mirrored: yes}', "");
Expect(1, 39, '\P{Bidi_Mirrored: yes}', "");
Expect(0, 39, '\P{^Bidi_Mirrored: yes}', "");
Expect(1, 41, '\p{Bidi_Mirrored:   - Yes}', "");
Expect(0, 41, '\p{^Bidi_Mirrored:   - Yes}', "");
Expect(0, 41, '\P{Bidi_Mirrored:   - Yes}', "");
Expect(1, 41, '\P{^Bidi_Mirrored:   - Yes}', "");
Expect(0, 39, '\p{Bidi_Mirrored:   - Yes}', "");
Expect(1, 39, '\p{^Bidi_Mirrored:   - Yes}', "");
Expect(1, 39, '\P{Bidi_Mirrored:   - Yes}', "");
Expect(0, 39, '\P{^Bidi_Mirrored:   - Yes}', "");
Error('\p{Bidi_M=-:=Y}');
Error('\P{Bidi_M=-:=Y}');
Expect(1, 41, '\p{Bidi_M=y}', "");
Expect(0, 41, '\p{^Bidi_M=y}', "");
Expect(0, 41, '\P{Bidi_M=y}', "");
Expect(1, 41, '\P{^Bidi_M=y}', "");
Expect(0, 39, '\p{Bidi_M=y}', "");
Expect(1, 39, '\p{^Bidi_M=y}', "");
Expect(1, 39, '\P{Bidi_M=y}', "");
Expect(0, 39, '\P{^Bidi_M=y}', "");
Expect(1, 41, '\p{Bidi_M=-_y}', "");
Expect(0, 41, '\p{^Bidi_M=-_y}', "");
Expect(0, 41, '\P{Bidi_M=-_y}', "");
Expect(1, 41, '\P{^Bidi_M=-_y}', "");
Expect(0, 39, '\p{Bidi_M=-_y}', "");
Expect(1, 39, '\p{^Bidi_M=-_y}', "");
Expect(1, 39, '\P{Bidi_M=-_y}', "");
Expect(0, 39, '\P{^Bidi_M=-_y}', "");
Error('\p{Is_Bidi_Mirrored=:=_ T}');
Error('\P{Is_Bidi_Mirrored=:=_ T}');
Expect(1, 41, '\p{Is_Bidi_Mirrored=t}', "");
Expect(0, 41, '\p{^Is_Bidi_Mirrored=t}', "");
Expect(0, 41, '\P{Is_Bidi_Mirrored=t}', "");
Expect(1, 41, '\P{^Is_Bidi_Mirrored=t}', "");
Expect(0, 39, '\p{Is_Bidi_Mirrored=t}', "");
Expect(1, 39, '\p{^Is_Bidi_Mirrored=t}', "");
Expect(1, 39, '\P{Is_Bidi_Mirrored=t}', "");
Expect(0, 39, '\P{^Is_Bidi_Mirrored=t}', "");
Expect(1, 41, '\p{Is_Bidi_Mirrored= 	T}', "");
Expect(0, 41, '\p{^Is_Bidi_Mirrored= 	T}', "");
Expect(0, 41, '\P{Is_Bidi_Mirrored= 	T}', "");
Expect(1, 41, '\P{^Is_Bidi_Mirrored= 	T}', "");
Expect(0, 39, '\p{Is_Bidi_Mirrored= 	T}', "");
Expect(1, 39, '\p{^Is_Bidi_Mirrored= 	T}', "");
Expect(1, 39, '\P{Is_Bidi_Mirrored= 	T}', "");
Expect(0, 39, '\P{^Is_Bidi_Mirrored= 	T}', "");
Error('\p{Is_Bidi_M:- TRUE/a/}');
Error('\P{Is_Bidi_M:- TRUE/a/}');
Expect(1, 41, '\p{Is_Bidi_M=true}', "");
Expect(0, 41, '\p{^Is_Bidi_M=true}', "");
Expect(0, 41, '\P{Is_Bidi_M=true}', "");
Expect(1, 41, '\P{^Is_Bidi_M=true}', "");
Expect(0, 39, '\p{Is_Bidi_M=true}', "");
Expect(1, 39, '\p{^Is_Bidi_M=true}', "");
Expect(1, 39, '\P{Is_Bidi_M=true}', "");
Expect(0, 39, '\P{^Is_Bidi_M=true}', "");
Expect(1, 41, '\p{Is_Bidi_M=_True}', "");
Expect(0, 41, '\p{^Is_Bidi_M=_True}', "");
Expect(0, 41, '\P{Is_Bidi_M=_True}', "");
Expect(1, 41, '\P{^Is_Bidi_M=_True}', "");
Expect(0, 39, '\p{Is_Bidi_M=_True}', "");
Expect(1, 39, '\p{^Is_Bidi_M=_True}', "");
Expect(1, 39, '\P{Is_Bidi_M=_True}', "");
Expect(0, 39, '\P{^Is_Bidi_M=_True}', "");
Error('\p{Bidi_Mirrored=:=-_no}');
Error('\P{Bidi_Mirrored=:=-_no}');
Expect(1, 39, '\p{Bidi_Mirrored=no}', "");
Expect(0, 39, '\p{^Bidi_Mirrored=no}', "");
Expect(0, 39, '\P{Bidi_Mirrored=no}', "");
Expect(1, 39, '\P{^Bidi_Mirrored=no}', "");
Expect(0, 41, '\p{Bidi_Mirrored=no}', "");
Expect(1, 41, '\p{^Bidi_Mirrored=no}', "");
Expect(1, 41, '\P{Bidi_Mirrored=no}', "");
Expect(0, 41, '\P{^Bidi_Mirrored=no}', "");
Expect(1, 39, '\p{Bidi_Mirrored=-_No}', "");
Expect(0, 39, '\p{^Bidi_Mirrored=-_No}', "");
Expect(0, 39, '\P{Bidi_Mirrored=-_No}', "");
Expect(1, 39, '\P{^Bidi_Mirrored=-_No}', "");
Expect(0, 41, '\p{Bidi_Mirrored=-_No}', "");
Expect(1, 41, '\p{^Bidi_Mirrored=-_No}', "");
Expect(1, 41, '\P{Bidi_Mirrored=-_No}', "");
Expect(0, 41, '\P{^Bidi_Mirrored=-_No}', "");
Error('\p{Bidi_M=/a/__N}');
Error('\P{Bidi_M=/a/__N}');
Expect(1, 39, '\p{Bidi_M=n}', "");
Expect(0, 39, '\p{^Bidi_M=n}', "");
Expect(0, 39, '\P{Bidi_M=n}', "");
Expect(1, 39, '\P{^Bidi_M=n}', "");
Expect(0, 41, '\p{Bidi_M=n}', "");
Expect(1, 41, '\p{^Bidi_M=n}', "");
Expect(1, 41, '\P{Bidi_M=n}', "");
Expect(0, 41, '\P{^Bidi_M=n}', "");
Expect(1, 39, '\p{Bidi_M=_N}', "");
Expect(0, 39, '\p{^Bidi_M=_N}', "");
Expect(0, 39, '\P{Bidi_M=_N}', "");
Expect(1, 39, '\P{^Bidi_M=_N}', "");
Expect(0, 41, '\p{Bidi_M=_N}', "");
Expect(1, 41, '\p{^Bidi_M=_N}', "");
Expect(1, 41, '\P{Bidi_M=_N}', "");
Expect(0, 41, '\P{^Bidi_M=_N}', "");
Error('\p{Is_Bidi_Mirrored=/a/ f}');
Error('\P{Is_Bidi_Mirrored=/a/ f}');
Expect(1, 39, '\p{Is_Bidi_Mirrored:	f}', "");
Expect(0, 39, '\p{^Is_Bidi_Mirrored:	f}', "");
Expect(0, 39, '\P{Is_Bidi_Mirrored:	f}', "");
Expect(1, 39, '\P{^Is_Bidi_Mirrored:	f}', "");
Expect(0, 41, '\p{Is_Bidi_Mirrored:	f}', "");
Expect(1, 41, '\p{^Is_Bidi_Mirrored:	f}', "");
Expect(1, 41, '\P{Is_Bidi_Mirrored:	f}', "");
Expect(0, 41, '\P{^Is_Bidi_Mirrored:	f}', "");
Expect(1, 39, '\p{Is_Bidi_Mirrored=-f}', "");
Expect(0, 39, '\p{^Is_Bidi_Mirrored=-f}', "");
Expect(0, 39, '\P{Is_Bidi_Mirrored=-f}', "");
Expect(1, 39, '\P{^Is_Bidi_Mirrored=-f}', "");
Expect(0, 41, '\p{Is_Bidi_Mirrored=-f}', "");
Expect(1, 41, '\p{^Is_Bidi_Mirrored=-f}', "");
Expect(1, 41, '\P{Is_Bidi_Mirrored=-f}', "");
Expect(0, 41, '\P{^Is_Bidi_Mirrored=-f}', "");
Error('\p{Is_Bidi_M=_false:=}');
Error('\P{Is_Bidi_M=_false:=}');
Expect(1, 39, '\p{Is_Bidi_M=false}', "");
Expect(0, 39, '\p{^Is_Bidi_M=false}', "");
Expect(0, 39, '\P{Is_Bidi_M=false}', "");
Expect(1, 39, '\P{^Is_Bidi_M=false}', "");
Expect(0, 41, '\p{Is_Bidi_M=false}', "");
Expect(1, 41, '\p{^Is_Bidi_M=false}', "");
Expect(1, 41, '\P{Is_Bidi_M=false}', "");
Expect(0, 41, '\P{^Is_Bidi_M=false}', "");
Expect(1, 39, '\p{Is_Bidi_M= false}', "");
Expect(0, 39, '\p{^Is_Bidi_M= false}', "");
Expect(0, 39, '\P{Is_Bidi_M= false}', "");
Expect(1, 39, '\P{^Is_Bidi_M= false}', "");
Expect(0, 41, '\p{Is_Bidi_M= false}', "");
Expect(1, 41, '\p{^Is_Bidi_M= false}', "");
Expect(1, 41, '\P{Is_Bidi_M= false}', "");
Expect(0, 41, '\P{^Is_Bidi_M= false}', "");
Error('\p{Lowercase:/a/Yes}');
Error('\P{Lowercase:/a/Yes}');
Expect(1, 122, '\p{Lowercase=yes}', "");
Expect(0, 122, '\p{^Lowercase=yes}', "");
Expect(0, 122, '\P{Lowercase=yes}', "");
Expect(1, 122, '\P{^Lowercase=yes}', "");
Expect(0, 96, '\p{Lowercase=yes}', "");
Expect(1, 96, '\p{^Lowercase=yes}', "");
Expect(1, 96, '\P{Lowercase=yes}', "");
Expect(0, 96, '\P{^Lowercase=yes}', "");
Expect(1, 122, '\p{Lowercase=		YES}', "");
Expect(0, 122, '\p{^Lowercase=		YES}', "");
Expect(0, 122, '\P{Lowercase=		YES}', "");
Expect(1, 122, '\P{^Lowercase=		YES}', "");
Expect(0, 96, '\p{Lowercase=		YES}', "");
Expect(1, 96, '\p{^Lowercase=		YES}', "");
Expect(1, 96, '\P{Lowercase=		YES}', "");
Expect(0, 96, '\P{^Lowercase=		YES}', "");
Error('\p{Lower=:=_-Y}');
Error('\P{Lower=:=_-Y}');
Expect(1, 122, '\p{Lower=y}', "");
Expect(0, 122, '\p{^Lower=y}', "");
Expect(0, 122, '\P{Lower=y}', "");
Expect(1, 122, '\P{^Lower=y}', "");
Expect(0, 96, '\p{Lower=y}', "");
Expect(1, 96, '\p{^Lower=y}', "");
Expect(1, 96, '\P{Lower=y}', "");
Expect(0, 96, '\P{^Lower=y}', "");
Expect(1, 122, '\p{Lower:	 -Y}', "");
Expect(0, 122, '\p{^Lower:	 -Y}', "");
Expect(0, 122, '\P{Lower:	 -Y}', "");
Expect(1, 122, '\P{^Lower:	 -Y}', "");
Expect(0, 96, '\p{Lower:	 -Y}', "");
Expect(1, 96, '\p{^Lower:	 -Y}', "");
Expect(1, 96, '\P{Lower:	 -Y}', "");
Expect(0, 96, '\P{^Lower:	 -Y}', "");
Error('\p{Is_Lowercase= 	T/a/}');
Error('\P{Is_Lowercase= 	T/a/}');
Expect(1, 122, '\p{Is_Lowercase=t}', "");
Expect(0, 122, '\p{^Is_Lowercase=t}', "");
Expect(0, 122, '\P{Is_Lowercase=t}', "");
Expect(1, 122, '\P{^Is_Lowercase=t}', "");
Expect(0, 96, '\p{Is_Lowercase=t}', "");
Expect(1, 96, '\p{^Is_Lowercase=t}', "");
Expect(1, 96, '\P{Is_Lowercase=t}', "");
Expect(0, 96, '\P{^Is_Lowercase=t}', "");
Expect(1, 122, '\p{Is_Lowercase=--t}', "");
Expect(0, 122, '\p{^Is_Lowercase=--t}', "");
Expect(0, 122, '\P{Is_Lowercase=--t}', "");
Expect(1, 122, '\P{^Is_Lowercase=--t}', "");
Expect(0, 96, '\p{Is_Lowercase=--t}', "");
Expect(1, 96, '\p{^Is_Lowercase=--t}', "");
Expect(1, 96, '\P{Is_Lowercase=--t}', "");
Expect(0, 96, '\P{^Is_Lowercase=--t}', "");
Error('\p{Is_Lower=TRUE:=}');
Error('\P{Is_Lower=TRUE:=}');
Expect(1, 122, '\p{Is_Lower=true}', "");
Expect(0, 122, '\p{^Is_Lower=true}', "");
Expect(0, 122, '\P{Is_Lower=true}', "");
Expect(1, 122, '\P{^Is_Lower=true}', "");
Expect(0, 96, '\p{Is_Lower=true}', "");
Expect(1, 96, '\p{^Is_Lower=true}', "");
Expect(1, 96, '\P{Is_Lower=true}', "");
Expect(0, 96, '\P{^Is_Lower=true}', "");
Expect(1, 122, '\p{Is_Lower=-TRUE}', "");
Expect(0, 122, '\p{^Is_Lower=-TRUE}', "");
Expect(0, 122, '\P{Is_Lower=-TRUE}', "");
Expect(1, 122, '\P{^Is_Lower=-TRUE}', "");
Expect(0, 96, '\p{Is_Lower=-TRUE}', "");
Expect(1, 96, '\p{^Is_Lower=-TRUE}', "");
Expect(1, 96, '\P{Is_Lower=-TRUE}', "");
Expect(0, 96, '\P{^Is_Lower=-TRUE}', "");
Error('\p{Lowercase=_	NO:=}');
Error('\P{Lowercase=_	NO:=}');
Expect(1, 96, '\p{Lowercase=no}', "");
Expect(0, 96, '\p{^Lowercase=no}', "");
Expect(0, 96, '\P{Lowercase=no}', "");
Expect(1, 96, '\P{^Lowercase=no}', "");
Expect(0, 122, '\p{Lowercase=no}', "");
Expect(1, 122, '\p{^Lowercase=no}', "");
Expect(1, 122, '\P{Lowercase=no}', "");
Expect(0, 122, '\P{^Lowercase=no}', "");
Expect(1, 96, '\p{Lowercase=-NO}', "");
Expect(0, 96, '\p{^Lowercase=-NO}', "");
Expect(0, 96, '\P{Lowercase=-NO}', "");
Expect(1, 96, '\P{^Lowercase=-NO}', "");
Expect(0, 122, '\p{Lowercase=-NO}', "");
Expect(1, 122, '\p{^Lowercase=-NO}', "");
Expect(1, 122, '\P{Lowercase=-NO}', "");
Expect(0, 122, '\P{^Lowercase=-NO}', "");
Error('\p{Lower=_	N:=}');
Error('\P{Lower=_	N:=}');
Expect(1, 96, '\p{Lower=n}', "");
Expect(0, 96, '\p{^Lower=n}', "");
Expect(0, 96, '\P{Lower=n}', "");
Expect(1, 96, '\P{^Lower=n}', "");
Expect(0, 122, '\p{Lower=n}', "");
Expect(1, 122, '\p{^Lower=n}', "");
Expect(1, 122, '\P{Lower=n}', "");
Expect(0, 122, '\P{^Lower=n}', "");
Expect(1, 96, '\p{Lower= N}', "");
Expect(0, 96, '\p{^Lower= N}', "");
Expect(0, 96, '\P{Lower= N}', "");
Expect(1, 96, '\P{^Lower= N}', "");
Expect(0, 122, '\p{Lower= N}', "");
Expect(1, 122, '\p{^Lower= N}', "");
Expect(1, 122, '\P{Lower= N}', "");
Expect(0, 122, '\P{^Lower= N}', "");
Error('\p{Is_Lowercase=-/a/F}');
Error('\P{Is_Lowercase=-/a/F}');
Expect(1, 96, '\p{Is_Lowercase:	f}', "");
Expect(0, 96, '\p{^Is_Lowercase:	f}', "");
Expect(0, 96, '\P{Is_Lowercase:	f}', "");
Expect(1, 96, '\P{^Is_Lowercase:	f}', "");
Expect(0, 122, '\p{Is_Lowercase:	f}', "");
Expect(1, 122, '\p{^Is_Lowercase:	f}', "");
Expect(1, 122, '\P{Is_Lowercase:	f}', "");
Expect(0, 122, '\P{^Is_Lowercase:	f}', "");
Expect(1, 96, '\p{Is_Lowercase=-_F}', "");
Expect(0, 96, '\p{^Is_Lowercase=-_F}', "");
Expect(0, 96, '\P{Is_Lowercase=-_F}', "");
Expect(1, 96, '\P{^Is_Lowercase=-_F}', "");
Expect(0, 122, '\p{Is_Lowercase=-_F}', "");
Expect(1, 122, '\p{^Is_Lowercase=-_F}', "");
Expect(1, 122, '\P{Is_Lowercase=-_F}', "");
Expect(0, 122, '\P{^Is_Lowercase=-_F}', "");
Error('\p{Is_Lower: _:=false}');
Error('\P{Is_Lower: _:=false}');
Expect(1, 96, '\p{Is_Lower=false}', "");
Expect(0, 96, '\p{^Is_Lower=false}', "");
Expect(0, 96, '\P{Is_Lower=false}', "");
Expect(1, 96, '\P{^Is_Lower=false}', "");
Expect(0, 122, '\p{Is_Lower=false}', "");
Expect(1, 122, '\p{^Is_Lower=false}', "");
Expect(1, 122, '\P{Is_Lower=false}', "");
Expect(0, 122, '\P{^Is_Lower=false}', "");
Expect(1, 96, '\p{Is_Lower=_false}', "");
Expect(0, 96, '\p{^Is_Lower=_false}', "");
Expect(0, 96, '\P{Is_Lower=_false}', "");
Expect(1, 96, '\P{^Is_Lower=_false}', "");
Expect(0, 122, '\p{Is_Lower=_false}', "");
Expect(1, 122, '\p{^Is_Lower=_false}', "");
Expect(1, 122, '\P{Is_Lower=_false}', "");
Expect(0, 122, '\P{^Is_Lower=_false}', "");
Error('\p{Uppercase=_-YES/a/}');
Error('\P{Uppercase=_-YES/a/}');
Expect(1, 90, '\p{Uppercase=yes}', "");
Expect(0, 90, '\p{^Uppercase=yes}', "");
Expect(0, 90, '\P{Uppercase=yes}', "");
Expect(1, 90, '\P{^Uppercase=yes}', "");
Expect(0, 64, '\p{Uppercase=yes}', "");
Expect(1, 64, '\p{^Uppercase=yes}', "");
Expect(1, 64, '\P{Uppercase=yes}', "");
Expect(0, 64, '\P{^Uppercase=yes}', "");
Expect(1, 90, '\p{Uppercase= 	YES}', "");
Expect(0, 90, '\p{^Uppercase= 	YES}', "");
Expect(0, 90, '\P{Uppercase= 	YES}', "");
Expect(1, 90, '\P{^Uppercase= 	YES}', "");
Expect(0, 64, '\p{Uppercase= 	YES}', "");
Expect(1, 64, '\p{^Uppercase= 	YES}', "");
Expect(1, 64, '\P{Uppercase= 	YES}', "");
Expect(0, 64, '\P{^Uppercase= 	YES}', "");
Error('\p{Upper=:=_Y}');
Error('\P{Upper=:=_Y}');
Expect(1, 90, '\p{Upper: y}', "");
Expect(0, 90, '\p{^Upper: y}', "");
Expect(0, 90, '\P{Upper: y}', "");
Expect(1, 90, '\P{^Upper: y}', "");
Expect(0, 64, '\p{Upper: y}', "");
Expect(1, 64, '\p{^Upper: y}', "");
Expect(1, 64, '\P{Upper: y}', "");
Expect(0, 64, '\P{^Upper: y}', "");
Expect(1, 90, '\p{Upper=	Y}', "");
Expect(0, 90, '\p{^Upper=	Y}', "");
Expect(0, 90, '\P{Upper=	Y}', "");
Expect(1, 90, '\P{^Upper=	Y}', "");
Expect(0, 64, '\p{Upper=	Y}', "");
Expect(1, 64, '\p{^Upper=	Y}', "");
Expect(1, 64, '\P{Upper=	Y}', "");
Expect(0, 64, '\P{^Upper=	Y}', "");
Error('\p{Is_Uppercase=-t:=}');
Error('\P{Is_Uppercase=-t:=}');
Expect(1, 90, '\p{Is_Uppercase=t}', "");
Expect(0, 90, '\p{^Is_Uppercase=t}', "");
Expect(0, 90, '\P{Is_Uppercase=t}', "");
Expect(1, 90, '\P{^Is_Uppercase=t}', "");
Expect(0, 64, '\p{Is_Uppercase=t}', "");
Expect(1, 64, '\p{^Is_Uppercase=t}', "");
Expect(1, 64, '\P{Is_Uppercase=t}', "");
Expect(0, 64, '\P{^Is_Uppercase=t}', "");
Expect(1, 90, '\p{Is_Uppercase=-T}', "");
Expect(0, 90, '\p{^Is_Uppercase=-T}', "");
Expect(0, 90, '\P{Is_Uppercase=-T}', "");
Expect(1, 90, '\P{^Is_Uppercase=-T}', "");
Expect(0, 64, '\p{Is_Uppercase=-T}', "");
Expect(1, 64, '\p{^Is_Uppercase=-T}', "");
Expect(1, 64, '\P{Is_Uppercase=-T}', "");
Expect(0, 64, '\P{^Is_Uppercase=-T}', "");
Error('\p{Is_Upper=  True:=}');
Error('\P{Is_Upper=  True:=}');
Expect(1, 90, '\p{Is_Upper=true}', "");
Expect(0, 90, '\p{^Is_Upper=true}', "");
Expect(0, 90, '\P{Is_Upper=true}', "");
Expect(1, 90, '\P{^Is_Upper=true}', "");
Expect(0, 64, '\p{Is_Upper=true}', "");
Expect(1, 64, '\p{^Is_Upper=true}', "");
Expect(1, 64, '\P{Is_Upper=true}', "");
Expect(0, 64, '\P{^Is_Upper=true}', "");
Expect(1, 90, '\p{Is_Upper=-TRUE}', "");
Expect(0, 90, '\p{^Is_Upper=-TRUE}', "");
Expect(0, 90, '\P{Is_Upper=-TRUE}', "");
Expect(1, 90, '\P{^Is_Upper=-TRUE}', "");
Expect(0, 64, '\p{Is_Upper=-TRUE}', "");
Expect(1, 64, '\p{^Is_Upper=-TRUE}', "");
Expect(1, 64, '\P{Is_Upper=-TRUE}', "");
Expect(0, 64, '\P{^Is_Upper=-TRUE}', "");
Error('\p{Uppercase=/a/ NO}');
Error('\P{Uppercase=/a/ NO}');
Expect(1, 64, '\p{Uppercase=no}', "");
Expect(0, 64, '\p{^Uppercase=no}', "");
Expect(0, 64, '\P{Uppercase=no}', "");
Expect(1, 64, '\P{^Uppercase=no}', "");
Expect(0, 90, '\p{Uppercase=no}', "");
Expect(1, 90, '\p{^Uppercase=no}', "");
Expect(1, 90, '\P{Uppercase=no}', "");
Expect(0, 90, '\P{^Uppercase=no}', "");
Expect(1, 64, '\p{Uppercase:No}', "");
Expect(0, 64, '\p{^Uppercase:No}', "");
Expect(0, 64, '\P{Uppercase:No}', "");
Expect(1, 64, '\P{^Uppercase:No}', "");
Expect(0, 90, '\p{Uppercase:No}', "");
Expect(1, 90, '\p{^Uppercase:No}', "");
Expect(1, 90, '\P{Uppercase:No}', "");
Expect(0, 90, '\P{^Uppercase:No}', "");
Error('\p{Upper=/a/_N}');
Error('\P{Upper=/a/_N}');
Expect(1, 64, '\p{Upper=n}', "");
Expect(0, 64, '\p{^Upper=n}', "");
Expect(0, 64, '\P{Upper=n}', "");
Expect(1, 64, '\P{^Upper=n}', "");
Expect(0, 90, '\p{Upper=n}', "");
Expect(1, 90, '\p{^Upper=n}', "");
Expect(1, 90, '\P{Upper=n}', "");
Expect(0, 90, '\P{^Upper=n}', "");
Expect(1, 64, '\p{Upper= N}', "");
Expect(0, 64, '\p{^Upper= N}', "");
Expect(0, 64, '\P{Upper= N}', "");
Expect(1, 64, '\P{^Upper= N}', "");
Expect(0, 90, '\p{Upper= N}', "");
Expect(1, 90, '\p{^Upper= N}', "");
Expect(1, 90, '\P{Upper= N}', "");
Expect(0, 90, '\P{^Upper= N}', "");
Error('\p{Is_Uppercase=-F:=}');
Error('\P{Is_Uppercase=-F:=}');
Expect(1, 64, '\p{Is_Uppercase=f}', "");
Expect(0, 64, '\p{^Is_Uppercase=f}', "");
Expect(0, 64, '\P{Is_Uppercase=f}', "");
Expect(1, 64, '\P{^Is_Uppercase=f}', "");
Expect(0, 90, '\p{Is_Uppercase=f}', "");
Expect(1, 90, '\p{^Is_Uppercase=f}', "");
Expect(1, 90, '\P{Is_Uppercase=f}', "");
Expect(0, 90, '\P{^Is_Uppercase=f}', "");
Expect(1, 64, '\p{Is_Uppercase=	F}', "");
Expect(0, 64, '\p{^Is_Uppercase=	F}', "");
Expect(0, 64, '\P{Is_Uppercase=	F}', "");
Expect(1, 64, '\P{^Is_Uppercase=	F}', "");
Expect(0, 90, '\p{Is_Uppercase=	F}', "");
Expect(1, 90, '\p{^Is_Uppercase=	F}', "");
Expect(1, 90, '\P{Is_Uppercase=	F}', "");
Expect(0, 90, '\P{^Is_Uppercase=	F}', "");
Error('\p{Is_Upper=	_False/a/}');
Error('\P{Is_Upper=	_False/a/}');
Expect(1, 64, '\p{Is_Upper=false}', "");
Expect(0, 64, '\p{^Is_Upper=false}', "");
Expect(0, 64, '\P{Is_Upper=false}', "");
Expect(1, 64, '\P{^Is_Upper=false}', "");
Expect(0, 90, '\p{Is_Upper=false}', "");
Expect(1, 90, '\p{^Is_Upper=false}', "");
Expect(1, 90, '\P{Is_Upper=false}', "");
Expect(0, 90, '\P{^Is_Upper=false}', "");
Expect(1, 64, '\p{Is_Upper=  FALSE}', "");
Expect(0, 64, '\p{^Is_Upper=  FALSE}', "");
Expect(0, 64, '\P{Is_Upper=  FALSE}', "");
Expect(1, 64, '\P{^Is_Upper=  FALSE}', "");
Expect(0, 90, '\p{Is_Upper=  FALSE}', "");
Expect(1, 90, '\p{^Is_Upper=  FALSE}', "");
Expect(1, 90, '\P{Is_Upper=  FALSE}', "");
Expect(0, 90, '\P{^Is_Upper=  FALSE}', "");
Error('\p{Expands_On_NFKC:	Yes}');
Error('\P{Expands_On_NFKC:	Yes}');
Error('\p{XO_NFKC=Y}');
Error('\P{XO_NFKC=Y}');
Error('\p{Is_Expands_On_NFKC:	T}');
Error('\P{Is_Expands_On_NFKC:	T}');
Error('\p{Is_XO_NFKC=True}');
Error('\P{Is_XO_NFKC=True}');
Error('\p{Expands_On_NFKC=No}');
Error('\P{Expands_On_NFKC=No}');
Error('\p{XO_NFKC=N}');
Error('\P{XO_NFKC=N}');
Error('\p{Is_Expands_On_NFKC=F}');
Error('\P{Is_Expands_On_NFKC=F}');
Error('\p{Is_XO_NFKC=False}');
Error('\P{Is_XO_NFKC=False}');
Error('\p{NFD_Quick_Check=:=		yes}');
Error('\P{NFD_Quick_Check=:=		yes}');
Expect(1, 191, '\p{NFD_Quick_Check=yes}', "");
Expect(0, 191, '\p{^NFD_Quick_Check=yes}', "");
Expect(0, 191, '\P{NFD_Quick_Check=yes}', "");
Expect(1, 191, '\P{^NFD_Quick_Check=yes}', "");
Expect(0, 197, '\p{NFD_Quick_Check=yes}', "");
Expect(1, 197, '\p{^NFD_Quick_Check=yes}', "");
Expect(1, 197, '\P{NFD_Quick_Check=yes}', "");
Expect(0, 197, '\P{^NFD_Quick_Check=yes}', "");
Expect(1, 191, '\p{NFD_Quick_Check=_	Yes}', "");
Expect(0, 191, '\p{^NFD_Quick_Check=_	Yes}', "");
Expect(0, 191, '\P{NFD_Quick_Check=_	Yes}', "");
Expect(1, 191, '\P{^NFD_Quick_Check=_	Yes}', "");
Expect(0, 197, '\p{NFD_Quick_Check=_	Yes}', "");
Expect(1, 197, '\p{^NFD_Quick_Check=_	Yes}', "");
Expect(1, 197, '\P{NFD_Quick_Check=_	Yes}', "");
Expect(0, 197, '\P{^NFD_Quick_Check=_	Yes}', "");
Error('\p{NFD_QC:/a/		y}');
Error('\P{NFD_QC:/a/		y}');
Expect(1, 191, '\p{NFD_QC:y}', "");
Expect(0, 191, '\p{^NFD_QC:y}', "");
Expect(0, 191, '\P{NFD_QC:y}', "");
Expect(1, 191, '\P{^NFD_QC:y}', "");
Expect(0, 197, '\p{NFD_QC:y}', "");
Expect(1, 197, '\p{^NFD_QC:y}', "");
Expect(1, 197, '\P{NFD_QC:y}', "");
Expect(0, 197, '\P{^NFD_QC:y}', "");
Expect(1, 191, '\p{NFD_QC=__Y}', "");
Expect(0, 191, '\p{^NFD_QC=__Y}', "");
Expect(0, 191, '\P{NFD_QC=__Y}', "");
Expect(1, 191, '\P{^NFD_QC=__Y}', "");
Expect(0, 197, '\p{NFD_QC=__Y}', "");
Expect(1, 197, '\p{^NFD_QC=__Y}', "");
Expect(1, 197, '\P{NFD_QC=__Y}', "");
Expect(0, 197, '\P{^NFD_QC=__Y}', "");
Error('\p{Is_NFD_Quick_Check=:=_ yes}');
Error('\P{Is_NFD_Quick_Check=:=_ yes}');
Expect(1, 191, '\p{Is_NFD_Quick_Check=yes}', "");
Expect(0, 191, '\p{^Is_NFD_Quick_Check=yes}', "");
Expect(0, 191, '\P{Is_NFD_Quick_Check=yes}', "");
Expect(1, 191, '\P{^Is_NFD_Quick_Check=yes}', "");
Expect(0, 197, '\p{Is_NFD_Quick_Check=yes}', "");
Expect(1, 197, '\p{^Is_NFD_Quick_Check=yes}', "");
Expect(1, 197, '\P{Is_NFD_Quick_Check=yes}', "");
Expect(0, 197, '\P{^Is_NFD_Quick_Check=yes}', "");
Expect(1, 191, '\p{Is_NFD_Quick_Check=	_YES}', "");
Expect(0, 191, '\p{^Is_NFD_Quick_Check=	_YES}', "");
Expect(0, 191, '\P{Is_NFD_Quick_Check=	_YES}', "");
Expect(1, 191, '\P{^Is_NFD_Quick_Check=	_YES}', "");
Expect(0, 197, '\p{Is_NFD_Quick_Check=	_YES}', "");
Expect(1, 197, '\p{^Is_NFD_Quick_Check=	_YES}', "");
Expect(1, 197, '\P{Is_NFD_Quick_Check=	_YES}', "");
Expect(0, 197, '\P{^Is_NFD_Quick_Check=	_YES}', "");
Error('\p{Is_NFD_QC:   -	y/a/}');
Error('\P{Is_NFD_QC:   -	y/a/}');
Expect(1, 191, '\p{Is_NFD_QC=y}', "");
Expect(0, 191, '\p{^Is_NFD_QC=y}', "");
Expect(0, 191, '\P{Is_NFD_QC=y}', "");
Expect(1, 191, '\P{^Is_NFD_QC=y}', "");
Expect(0, 197, '\p{Is_NFD_QC=y}', "");
Expect(1, 197, '\p{^Is_NFD_QC=y}', "");
Expect(1, 197, '\P{Is_NFD_QC=y}', "");
Expect(0, 197, '\P{^Is_NFD_QC=y}', "");
Expect(1, 191, '\p{Is_NFD_QC=	y}', "");
Expect(0, 191, '\p{^Is_NFD_QC=	y}', "");
Expect(0, 191, '\P{Is_NFD_QC=	y}', "");
Expect(1, 191, '\P{^Is_NFD_QC=	y}', "");
Expect(0, 197, '\p{Is_NFD_QC=	y}', "");
Expect(1, 197, '\p{^Is_NFD_QC=	y}', "");
Expect(1, 197, '\P{Is_NFD_QC=	y}', "");
Expect(0, 197, '\P{^Is_NFD_QC=	y}', "");
Error('\p{NFD_Quick_Check= _No:=}');
Error('\P{NFD_Quick_Check= _No:=}');
Expect(1, 197, '\p{NFD_Quick_Check=no}', "");
Expect(0, 197, '\p{^NFD_Quick_Check=no}', "");
Expect(0, 197, '\P{NFD_Quick_Check=no}', "");
Expect(1, 197, '\P{^NFD_Quick_Check=no}', "");
Expect(0, 191, '\p{NFD_Quick_Check=no}', "");
Expect(1, 191, '\p{^NFD_Quick_Check=no}', "");
Expect(1, 191, '\P{NFD_Quick_Check=no}', "");
Expect(0, 191, '\P{^NFD_Quick_Check=no}', "");
Expect(1, 197, '\p{NFD_Quick_Check=_-No}', "");
Expect(0, 197, '\p{^NFD_Quick_Check=_-No}', "");
Expect(0, 197, '\P{NFD_Quick_Check=_-No}', "");
Expect(1, 197, '\P{^NFD_Quick_Check=_-No}', "");
Expect(0, 191, '\p{NFD_Quick_Check=_-No}', "");
Expect(1, 191, '\p{^NFD_Quick_Check=_-No}', "");
Expect(1, 191, '\P{NFD_Quick_Check=_-No}', "");
Expect(0, 191, '\P{^NFD_Quick_Check=_-No}', "");
Error('\p{NFD_QC=- N/a/}');
Error('\P{NFD_QC=- N/a/}');
Expect(1, 197, '\p{NFD_QC=n}', "");
Expect(0, 197, '\p{^NFD_QC=n}', "");
Expect(0, 197, '\P{NFD_QC=n}', "");
Expect(1, 197, '\P{^NFD_QC=n}', "");
Expect(0, 191, '\p{NFD_QC=n}', "");
Expect(1, 191, '\p{^NFD_QC=n}', "");
Expect(1, 191, '\P{NFD_QC=n}', "");
Expect(0, 191, '\P{^NFD_QC=n}', "");
Expect(1, 197, '\p{NFD_QC=-N}', "");
Expect(0, 197, '\p{^NFD_QC=-N}', "");
Expect(0, 197, '\P{NFD_QC=-N}', "");
Expect(1, 197, '\P{^NFD_QC=-N}', "");
Expect(0, 191, '\p{NFD_QC=-N}', "");
Expect(1, 191, '\p{^NFD_QC=-N}', "");
Expect(1, 191, '\P{NFD_QC=-N}', "");
Expect(0, 191, '\P{^NFD_QC=-N}', "");
Error('\p{Is_NFD_Quick_Check=-/a/no}');
Error('\P{Is_NFD_Quick_Check=-/a/no}');
Expect(1, 197, '\p{Is_NFD_Quick_Check=no}', "");
Expect(0, 197, '\p{^Is_NFD_Quick_Check=no}', "");
Expect(0, 197, '\P{Is_NFD_Quick_Check=no}', "");
Expect(1, 197, '\P{^Is_NFD_Quick_Check=no}', "");
Expect(0, 191, '\p{Is_NFD_Quick_Check=no}', "");
Expect(1, 191, '\p{^Is_NFD_Quick_Check=no}', "");
Expect(1, 191, '\P{Is_NFD_Quick_Check=no}', "");
Expect(0, 191, '\P{^Is_NFD_Quick_Check=no}', "");
Expect(1, 197, '\p{Is_NFD_Quick_Check= _No}', "");
Expect(0, 197, '\p{^Is_NFD_Quick_Check= _No}', "");
Expect(0, 197, '\P{Is_NFD_Quick_Check= _No}', "");
Expect(1, 197, '\P{^Is_NFD_Quick_Check= _No}', "");
Expect(0, 191, '\p{Is_NFD_Quick_Check= _No}', "");
Expect(1, 191, '\p{^Is_NFD_Quick_Check= _No}', "");
Expect(1, 191, '\P{Is_NFD_Quick_Check= _No}', "");
Expect(0, 191, '\P{^Is_NFD_Quick_Check= _No}', "");
Error('\p{Is_NFD_QC=:=N}');
Error('\P{Is_NFD_QC=:=N}');
Expect(1, 197, '\p{Is_NFD_QC=n}', "");
Expect(0, 197, '\p{^Is_NFD_QC=n}', "");
Expect(0, 197, '\P{Is_NFD_QC=n}', "");
Expect(1, 197, '\P{^Is_NFD_QC=n}', "");
Expect(0, 191, '\p{Is_NFD_QC=n}', "");
Expect(1, 191, '\p{^Is_NFD_QC=n}', "");
Expect(1, 191, '\P{Is_NFD_QC=n}', "");
Expect(0, 191, '\P{^Is_NFD_QC=n}', "");
Expect(1, 197, '\p{Is_NFD_QC:   	_N}', "");
Expect(0, 197, '\p{^Is_NFD_QC:   	_N}', "");
Expect(0, 197, '\P{Is_NFD_QC:   	_N}', "");
Expect(1, 197, '\P{^Is_NFD_QC:   	_N}', "");
Expect(0, 191, '\p{Is_NFD_QC:   	_N}', "");
Expect(1, 191, '\p{^Is_NFD_QC:   	_N}', "");
Expect(1, 191, '\P{Is_NFD_QC:   	_N}', "");
Expect(0, 191, '\P{^Is_NFD_QC:   	_N}', "");
Error('\p{Line_Break=-:=SPACE}');
Error('\P{Line_Break=-:=SPACE}');
Expect(1, 32, '\p{Line_Break=space}', "");
Expect(0, 32, '\p{^Line_Break=space}', "");
Expect(0, 32, '\P{Line_Break=space}', "");
Expect(1, 32, '\P{^Line_Break=space}', "");
Expect(0, 31, '\p{Line_Break=space}', "");
Expect(1, 31, '\p{^Line_Break=space}', "");
Expect(1, 31, '\P{Line_Break=space}', "");
Expect(0, 31, '\P{^Line_Break=space}', "");
Expect(1, 32, '\p{Line_Break=_SPACE}', "");
Expect(0, 32, '\p{^Line_Break=_SPACE}', "");
Expect(0, 32, '\P{Line_Break=_SPACE}', "");
Expect(1, 32, '\P{^Line_Break=_SPACE}', "");
Expect(0, 31, '\p{Line_Break=_SPACE}', "");
Expect(1, 31, '\p{^Line_Break=_SPACE}', "");
Expect(1, 31, '\P{Line_Break=_SPACE}', "");
Expect(0, 31, '\P{^Line_Break=_SPACE}', "");
Error('\p{Lb=-_SP/a/}');
Error('\P{Lb=-_SP/a/}');
Expect(1, 32, '\p{Lb=sp}', "");
Expect(0, 32, '\p{^Lb=sp}', "");
Expect(0, 32, '\P{Lb=sp}', "");
Expect(1, 32, '\P{^Lb=sp}', "");
Expect(0, 31, '\p{Lb=sp}', "");
Expect(1, 31, '\p{^Lb=sp}', "");
Expect(1, 31, '\P{Lb=sp}', "");
Expect(0, 31, '\P{^Lb=sp}', "");
Expect(1, 32, '\p{Lb=	 SP}', "");
Expect(0, 32, '\p{^Lb=	 SP}', "");
Expect(0, 32, '\P{Lb=	 SP}', "");
Expect(1, 32, '\P{^Lb=	 SP}', "");
Expect(0, 31, '\p{Lb=	 SP}', "");
Expect(1, 31, '\p{^Lb=	 SP}', "");
Expect(1, 31, '\P{Lb=	 SP}', "");
Expect(0, 31, '\P{^Lb=	 SP}', "");
Error('\p{Is_Line_Break=_/a/SPACE}');
Error('\P{Is_Line_Break=_/a/SPACE}');
Expect(1, 32, '\p{Is_Line_Break=space}', "");
Expect(0, 32, '\p{^Is_Line_Break=space}', "");
Expect(0, 32, '\P{Is_Line_Break=space}', "");
Expect(1, 32, '\P{^Is_Line_Break=space}', "");
Expect(0, 31, '\p{Is_Line_Break=space}', "");
Expect(1, 31, '\p{^Is_Line_Break=space}', "");
Expect(1, 31, '\P{Is_Line_Break=space}', "");
Expect(0, 31, '\P{^Is_Line_Break=space}', "");
Expect(1, 32, '\p{Is_Line_Break=_-Space}', "");
Expect(0, 32, '\p{^Is_Line_Break=_-Space}', "");
Expect(0, 32, '\P{Is_Line_Break=_-Space}', "");
Expect(1, 32, '\P{^Is_Line_Break=_-Space}', "");
Expect(0, 31, '\p{Is_Line_Break=_-Space}', "");
Expect(1, 31, '\p{^Is_Line_Break=_-Space}', "");
Expect(1, 31, '\P{Is_Line_Break=_-Space}', "");
Expect(0, 31, '\P{^Is_Line_Break=_-Space}', "");
Error('\p{Is_Lb=-:=SP}');
Error('\P{Is_Lb=-:=SP}');
Expect(1, 32, '\p{Is_Lb=sp}', "");
Expect(0, 32, '\p{^Is_Lb=sp}', "");
Expect(0, 32, '\P{Is_Lb=sp}', "");
Expect(1, 32, '\P{^Is_Lb=sp}', "");
Expect(0, 31, '\p{Is_Lb=sp}', "");
Expect(1, 31, '\p{^Is_Lb=sp}', "");
Expect(1, 31, '\P{Is_Lb=sp}', "");
Expect(0, 31, '\P{^Is_Lb=sp}', "");
Expect(1, 32, '\p{Is_Lb=-sp}', "");
Expect(0, 32, '\p{^Is_Lb=-sp}', "");
Expect(0, 32, '\P{Is_Lb=-sp}', "");
Expect(1, 32, '\P{^Is_Lb=-sp}', "");
Expect(0, 31, '\p{Is_Lb=-sp}', "");
Expect(1, 31, '\p{^Is_Lb=-sp}', "");
Expect(1, 31, '\P{Is_Lb=-sp}', "");
Expect(0, 31, '\P{^Is_Lb=-sp}', "");
Error('\p{Line_Break= Glue/a/}');
Error('\P{Line_Break= Glue/a/}');
Expect(1, 160, '\p{Line_Break=glue}', "");
Expect(0, 160, '\p{^Line_Break=glue}', "");
Expect(0, 160, '\P{Line_Break=glue}', "");
Expect(1, 160, '\P{^Line_Break=glue}', "");
Expect(0, 159, '\p{Line_Break=glue}', "");
Expect(1, 159, '\p{^Line_Break=glue}', "");
Expect(1, 159, '\P{Line_Break=glue}', "");
Expect(0, 159, '\P{^Line_Break=glue}', "");
Expect(1, 160, '\p{Line_Break=	 Glue}', "");
Expect(0, 160, '\p{^Line_Break=	 Glue}', "");
Expect(0, 160, '\P{Line_Break=	 Glue}', "");
Expect(1, 160, '\P{^Line_Break=	 Glue}', "");
Expect(0, 159, '\p{Line_Break=	 Glue}', "");
Expect(1, 159, '\p{^Line_Break=	 Glue}', "");
Expect(1, 159, '\P{Line_Break=	 Glue}', "");
Expect(0, 159, '\P{^Line_Break=	 Glue}', "");
Error('\p{Lb= gl/a/}');
Error('\P{Lb= gl/a/}');
Expect(1, 160, '\p{Lb=gl}', "");
Expect(0, 160, '\p{^Lb=gl}', "");
Expect(0, 160, '\P{Lb=gl}', "");
Expect(1, 160, '\P{^Lb=gl}', "");
Expect(0, 159, '\p{Lb=gl}', "");
Expect(1, 159, '\p{^Lb=gl}', "");
Expect(1, 159, '\P{Lb=gl}', "");
Expect(0, 159, '\P{^Lb=gl}', "");
Expect(1, 160, '\p{Lb=--GL}', "");
Expect(0, 160, '\p{^Lb=--GL}', "");
Expect(0, 160, '\P{Lb=--GL}', "");
Expect(1, 160, '\P{^Lb=--GL}', "");
Expect(0, 159, '\p{Lb=--GL}', "");
Expect(1, 159, '\p{^Lb=--GL}', "");
Expect(1, 159, '\P{Lb=--GL}', "");
Expect(0, 159, '\P{^Lb=--GL}', "");
Error('\p{Is_Line_Break=	:=glue}');
Error('\P{Is_Line_Break=	:=glue}');
Expect(1, 160, '\p{Is_Line_Break=glue}', "");
Expect(0, 160, '\p{^Is_Line_Break=glue}', "");
Expect(0, 160, '\P{Is_Line_Break=glue}', "");
Expect(1, 160, '\P{^Is_Line_Break=glue}', "");
Expect(0, 159, '\p{Is_Line_Break=glue}', "");
Expect(1, 159, '\p{^Is_Line_Break=glue}', "");
Expect(1, 159, '\P{Is_Line_Break=glue}', "");
Expect(0, 159, '\P{^Is_Line_Break=glue}', "");
Expect(1, 160, '\p{Is_Line_Break= 	Glue}', "");
Expect(0, 160, '\p{^Is_Line_Break= 	Glue}', "");
Expect(0, 160, '\P{Is_Line_Break= 	Glue}', "");
Expect(1, 160, '\P{^Is_Line_Break= 	Glue}', "");
Expect(0, 159, '\p{Is_Line_Break= 	Glue}', "");
Expect(1, 159, '\p{^Is_Line_Break= 	Glue}', "");
Expect(1, 159, '\P{Is_Line_Break= 	Glue}', "");
Expect(0, 159, '\P{^Is_Line_Break= 	Glue}', "");
Error('\p{Is_Lb: - GL:=}');
Error('\P{Is_Lb: - GL:=}');
Expect(1, 160, '\p{Is_Lb=gl}', "");
Expect(0, 160, '\p{^Is_Lb=gl}', "");
Expect(0, 160, '\P{Is_Lb=gl}', "");
Expect(1, 160, '\P{^Is_Lb=gl}', "");
Expect(0, 159, '\p{Is_Lb=gl}', "");
Expect(1, 159, '\p{^Is_Lb=gl}', "");
Expect(1, 159, '\P{Is_Lb=gl}', "");
Expect(0, 159, '\P{^Is_Lb=gl}', "");
Expect(1, 160, '\p{Is_Lb=--gl}', "");
Expect(0, 160, '\p{^Is_Lb=--gl}', "");
Expect(0, 160, '\P{Is_Lb=--gl}', "");
Expect(1, 160, '\P{^Is_Lb=--gl}', "");
Expect(0, 159, '\p{Is_Lb=--gl}', "");
Expect(1, 159, '\p{^Is_Lb=--gl}', "");
Expect(1, 159, '\P{Is_Lb=--gl}', "");
Expect(0, 159, '\P{^Is_Lb=--gl}', "");
Error('\p{Line_Break=/a/ -SURROGATE}');
Error('\P{Line_Break=/a/ -SURROGATE}');
Expect(1, 57343, '\p{Line_Break=surrogate}', 'deprecated');
Expect(0, 57343, '\p{^Line_Break=surrogate}', 'deprecated');
Expect(0, 57343, '\P{Line_Break=surrogate}', 'deprecated');
Expect(1, 57343, '\P{^Line_Break=surrogate}', 'deprecated');
Expect(0, 55295, '\p{Line_Break=surrogate}', 'deprecated');
Expect(1, 55295, '\p{^Line_Break=surrogate}', 'deprecated');
Expect(1, 55295, '\P{Line_Break=surrogate}', 'deprecated');
Expect(0, 55295, '\P{^Line_Break=surrogate}', 'deprecated');
Expect(1, 57343, '\p{Line_Break=_surrogate}', 'deprecated');
Expect(0, 57343, '\p{^Line_Break=_surrogate}', 'deprecated');
Expect(0, 57343, '\P{Line_Break=_surrogate}', 'deprecated');
Expect(1, 57343, '\P{^Line_Break=_surrogate}', 'deprecated');
Expect(0, 55295, '\p{Line_Break=_surrogate}', 'deprecated');
Expect(1, 55295, '\p{^Line_Break=_surrogate}', 'deprecated');
Expect(1, 55295, '\P{Line_Break=_surrogate}', 'deprecated');
Expect(0, 55295, '\P{^Line_Break=_surrogate}', 'deprecated');
Error('\p{Lb=:=--SG}');
Error('\P{Lb=:=--SG}');
Expect(1, 57343, '\p{Lb=sg}', 'deprecated');
Expect(0, 57343, '\p{^Lb=sg}', 'deprecated');
Expect(0, 57343, '\P{Lb=sg}', 'deprecated');
Expect(1, 57343, '\P{^Lb=sg}', 'deprecated');
Expect(0, 55295, '\p{Lb=sg}', 'deprecated');
Expect(1, 55295, '\p{^Lb=sg}', 'deprecated');
Expect(1, 55295, '\P{Lb=sg}', 'deprecated');
Expect(0, 55295, '\P{^Lb=sg}', 'deprecated');
Expect(1, 57343, '\p{Lb=	_SG}', 'deprecated');
Expect(0, 57343, '\p{^Lb=	_SG}', 'deprecated');
Expect(0, 57343, '\P{Lb=	_SG}', 'deprecated');
Expect(1, 57343, '\P{^Lb=	_SG}', 'deprecated');
Expect(0, 55295, '\p{Lb=	_SG}', 'deprecated');
Expect(1, 55295, '\p{^Lb=	_SG}', 'deprecated');
Expect(1, 55295, '\P{Lb=	_SG}', 'deprecated');
Expect(0, 55295, '\P{^Lb=	_SG}', 'deprecated');
Error('\p{Is_Line_Break=- Surrogate:=}');
Error('\P{Is_Line_Break=- Surrogate:=}');
Expect(1, 57343, '\p{Is_Line_Break:surrogate}', 'deprecated');
Expect(0, 57343, '\p{^Is_Line_Break:surrogate}', 'deprecated');
Expect(0, 57343, '\P{Is_Line_Break:surrogate}', 'deprecated');
Expect(1, 57343, '\P{^Is_Line_Break:surrogate}', 'deprecated');
Expect(0, 55295, '\p{Is_Line_Break:surrogate}', 'deprecated');
Expect(1, 55295, '\p{^Is_Line_Break:surrogate}', 'deprecated');
Expect(1, 55295, '\P{Is_Line_Break:surrogate}', 'deprecated');
Expect(0, 55295, '\P{^Is_Line_Break:surrogate}', 'deprecated');
Expect(1, 57343, '\p{Is_Line_Break=-Surrogate}', 'deprecated');
Expect(0, 57343, '\p{^Is_Line_Break=-Surrogate}', 'deprecated');
Expect(0, 57343, '\P{Is_Line_Break=-Surrogate}', 'deprecated');
Expect(1, 57343, '\P{^Is_Line_Break=-Surrogate}', 'deprecated');
Expect(0, 55295, '\p{Is_Line_Break=-Surrogate}', 'deprecated');
Expect(1, 55295, '\p{^Is_Line_Break=-Surrogate}', 'deprecated');
Expect(1, 55295, '\P{Is_Line_Break=-Surrogate}', 'deprecated');
Expect(0, 55295, '\P{^Is_Line_Break=-Surrogate}', 'deprecated');
Error('\p{Is_Lb= _SG/a/}');
Error('\P{Is_Lb= _SG/a/}');
Expect(1, 57343, '\p{Is_Lb=sg}', 'deprecated');
Expect(0, 57343, '\p{^Is_Lb=sg}', 'deprecated');
Expect(0, 57343, '\P{Is_Lb=sg}', 'deprecated');
Expect(1, 57343, '\P{^Is_Lb=sg}', 'deprecated');
Expect(0, 55295, '\p{Is_Lb=sg}', 'deprecated');
Expect(1, 55295, '\p{^Is_Lb=sg}', 'deprecated');
Expect(1, 55295, '\P{Is_Lb=sg}', 'deprecated');
Expect(0, 55295, '\P{^Is_Lb=sg}', 'deprecated');
Expect(1, 57343, '\p{Is_Lb=		SG}', 'deprecated');
Expect(0, 57343, '\p{^Is_Lb=		SG}', 'deprecated');
Expect(0, 57343, '\P{Is_Lb=		SG}', 'deprecated');
Expect(1, 57343, '\P{^Is_Lb=		SG}', 'deprecated');
Expect(0, 55295, '\p{Is_Lb=		SG}', 'deprecated');
Expect(1, 55295, '\p{^Is_Lb=		SG}', 'deprecated');
Expect(1, 55295, '\P{Is_Lb=		SG}', 'deprecated');
Expect(0, 55295, '\P{^Is_Lb=		SG}', 'deprecated');
Error('\p{Line_Break=_/a/Unknown}');
Error('\P{Line_Break=_/a/Unknown}');
Expect(1, 889, '\p{Line_Break=unknown}', "");
Expect(0, 889, '\p{^Line_Break=unknown}', "");
Expect(0, 889, '\P{Line_Break=unknown}', "");
Expect(1, 889, '\P{^Line_Break=unknown}', "");
Expect(0, 887, '\p{Line_Break=unknown}', "");
Expect(1, 887, '\p{^Line_Break=unknown}', "");
Expect(1, 887, '\P{Line_Break=unknown}', "");
Expect(0, 887, '\P{^Line_Break=unknown}', "");
Expect(1, 889, '\p{Line_Break=	-Unknown}', "");
Expect(0, 889, '\p{^Line_Break=	-Unknown}', "");
Expect(0, 889, '\P{Line_Break=	-Unknown}', "");
Expect(1, 889, '\P{^Line_Break=	-Unknown}', "");
Expect(0, 887, '\p{Line_Break=	-Unknown}', "");
Expect(1, 887, '\p{^Line_Break=	-Unknown}', "");
Expect(1, 887, '\P{Line_Break=	-Unknown}', "");
Expect(0, 887, '\P{^Line_Break=	-Unknown}', "");
Error('\p{Lb=  XX:=}');
Error('\P{Lb=  XX:=}');
Expect(1, 889, '\p{Lb=xx}', "");
Expect(0, 889, '\p{^Lb=xx}', "");
Expect(0, 889, '\P{Lb=xx}', "");
Expect(1, 889, '\P{^Lb=xx}', "");
Expect(0, 887, '\p{Lb=xx}', "");
Expect(1, 887, '\p{^Lb=xx}', "");
Expect(1, 887, '\P{Lb=xx}', "");
Expect(0, 887, '\P{^Lb=xx}', "");
Expect(1, 889, '\p{Lb=	 XX}', "");
Expect(0, 889, '\p{^Lb=	 XX}', "");
Expect(0, 889, '\P{Lb=	 XX}', "");
Expect(1, 889, '\P{^Lb=	 XX}', "");
Expect(0, 887, '\p{Lb=	 XX}', "");
Expect(1, 887, '\p{^Lb=	 XX}', "");
Expect(1, 887, '\P{Lb=	 XX}', "");
Expect(0, 887, '\P{^Lb=	 XX}', "");
Error('\p{Is_Line_Break=/a/-	Unknown}');
Error('\P{Is_Line_Break=/a/-	Unknown}');
Expect(1, 889, '\p{Is_Line_Break=unknown}', "");
Expect(0, 889, '\p{^Is_Line_Break=unknown}', "");
Expect(0, 889, '\P{Is_Line_Break=unknown}', "");
Expect(1, 889, '\P{^Is_Line_Break=unknown}', "");
Expect(0, 887, '\p{Is_Line_Break=unknown}', "");
Expect(1, 887, '\p{^Is_Line_Break=unknown}', "");
Expect(1, 887, '\P{Is_Line_Break=unknown}', "");
Expect(0, 887, '\P{^Is_Line_Break=unknown}', "");
Expect(1, 889, '\p{Is_Line_Break=_UNKNOWN}', "");
Expect(0, 889, '\p{^Is_Line_Break=_UNKNOWN}', "");
Expect(0, 889, '\P{Is_Line_Break=_UNKNOWN}', "");
Expect(1, 889, '\P{^Is_Line_Break=_UNKNOWN}', "");
Expect(0, 887, '\p{Is_Line_Break=_UNKNOWN}', "");
Expect(1, 887, '\p{^Is_Line_Break=_UNKNOWN}', "");
Expect(1, 887, '\P{Is_Line_Break=_UNKNOWN}', "");
Expect(0, 887, '\P{^Is_Line_Break=_UNKNOWN}', "");
Error('\p{Is_Lb=:=xx}');
Error('\P{Is_Lb=:=xx}');
Expect(1, 889, '\p{Is_Lb=xx}', "");
Expect(0, 889, '\p{^Is_Lb=xx}', "");
Expect(0, 889, '\P{Is_Lb=xx}', "");
Expect(1, 889, '\P{^Is_Lb=xx}', "");
Expect(0, 887, '\p{Is_Lb=xx}', "");
Expect(1, 887, '\p{^Is_Lb=xx}', "");
Expect(1, 887, '\P{Is_Lb=xx}', "");
Expect(0, 887, '\P{^Is_Lb=xx}', "");
Expect(1, 889, '\p{Is_Lb=__XX}', "");
Expect(0, 889, '\p{^Is_Lb=__XX}', "");
Expect(0, 889, '\P{Is_Lb=__XX}', "");
Expect(1, 889, '\P{^Is_Lb=__XX}', "");
Expect(0, 887, '\p{Is_Lb=__XX}', "");
Expect(1, 887, '\p{^Is_Lb=__XX}', "");
Expect(1, 887, '\P{Is_Lb=__XX}', "");
Expect(0, 887, '\P{^Is_Lb=__XX}', "");
Error('\p{Line_Break=	:=numeric}');
Error('\P{Line_Break=	:=numeric}');
Expect(1, 57, '\p{Line_Break=numeric}', "");
Expect(0, 57, '\p{^Line_Break=numeric}', "");
Expect(0, 57, '\P{Line_Break=numeric}', "");
Expect(1, 57, '\P{^Line_Break=numeric}', "");
Expect(0, 47, '\p{Line_Break=numeric}', "");
Expect(1, 47, '\p{^Line_Break=numeric}', "");
Expect(1, 47, '\P{Line_Break=numeric}', "");
Expect(0, 47, '\P{^Line_Break=numeric}', "");
Expect(1, 57, '\p{Line_Break=	-numeric}', "");
Expect(0, 57, '\p{^Line_Break=	-numeric}', "");
Expect(0, 57, '\P{Line_Break=	-numeric}', "");
Expect(1, 57, '\P{^Line_Break=	-numeric}', "");
Expect(0, 47, '\p{Line_Break=	-numeric}', "");
Expect(1, 47, '\p{^Line_Break=	-numeric}', "");
Expect(1, 47, '\P{Line_Break=	-numeric}', "");
Expect(0, 47, '\P{^Line_Break=	-numeric}', "");
Error('\p{Lb=		nu:=}');
Error('\P{Lb=		nu:=}');
Expect(1, 57, '\p{Lb=nu}', "");
Expect(0, 57, '\p{^Lb=nu}', "");
Expect(0, 57, '\P{Lb=nu}', "");
Expect(1, 57, '\P{^Lb=nu}', "");
Expect(0, 47, '\p{Lb=nu}', "");
Expect(1, 47, '\p{^Lb=nu}', "");
Expect(1, 47, '\P{Lb=nu}', "");
Expect(0, 47, '\P{^Lb=nu}', "");
Expect(1, 57, '\p{Lb= NU}', "");
Expect(0, 57, '\p{^Lb= NU}', "");
Expect(0, 57, '\P{Lb= NU}', "");
Expect(1, 57, '\P{^Lb= NU}', "");
Expect(0, 47, '\p{Lb= NU}', "");
Expect(1, 47, '\p{^Lb= NU}', "");
Expect(1, 47, '\P{Lb= NU}', "");
Expect(0, 47, '\P{^Lb= NU}', "");
Error('\p{Is_Line_Break=	:=Numeric}');
Error('\P{Is_Line_Break=	:=Numeric}');
Expect(1, 57, '\p{Is_Line_Break=numeric}', "");
Expect(0, 57, '\p{^Is_Line_Break=numeric}', "");
Expect(0, 57, '\P{Is_Line_Break=numeric}', "");
Expect(1, 57, '\P{^Is_Line_Break=numeric}', "");
Expect(0, 47, '\p{Is_Line_Break=numeric}', "");
Expect(1, 47, '\p{^Is_Line_Break=numeric}', "");
Expect(1, 47, '\P{Is_Line_Break=numeric}', "");
Expect(0, 47, '\P{^Is_Line_Break=numeric}', "");
Expect(1, 57, '\p{Is_Line_Break= -NUMERIC}', "");
Expect(0, 57, '\p{^Is_Line_Break= -NUMERIC}', "");
Expect(0, 57, '\P{Is_Line_Break= -NUMERIC}', "");
Expect(1, 57, '\P{^Is_Line_Break= -NUMERIC}', "");
Expect(0, 47, '\p{Is_Line_Break= -NUMERIC}', "");
Expect(1, 47, '\p{^Is_Line_Break= -NUMERIC}', "");
Expect(1, 47, '\P{Is_Line_Break= -NUMERIC}', "");
Expect(0, 47, '\P{^Is_Line_Break= -NUMERIC}', "");
Error('\p{Is_Lb=-:=nu}');
Error('\P{Is_Lb=-:=nu}');
Expect(1, 57, '\p{Is_Lb:	nu}', "");
Expect(0, 57, '\p{^Is_Lb:	nu}', "");
Expect(0, 57, '\P{Is_Lb:	nu}', "");
Expect(1, 57, '\P{^Is_Lb:	nu}', "");
Expect(0, 47, '\p{Is_Lb:	nu}', "");
Expect(1, 47, '\p{^Is_Lb:	nu}', "");
Expect(1, 47, '\P{Is_Lb:	nu}', "");
Expect(0, 47, '\P{^Is_Lb:	nu}', "");
Expect(1, 57, '\p{Is_Lb=-NU}', "");
Expect(0, 57, '\p{^Is_Lb=-NU}', "");
Expect(0, 57, '\P{Is_Lb=-NU}', "");
Expect(1, 57, '\P{^Is_Lb=-NU}', "");
Expect(0, 47, '\p{Is_Lb=-NU}', "");
Expect(1, 47, '\p{^Is_Lb=-NU}', "");
Expect(1, 47, '\P{Is_Lb=-NU}', "");
Expect(0, 47, '\P{^Is_Lb=-NU}', "");
Error('\p{Line_Break=-Break Before/a/}');
Error('\P{Line_Break=-Break Before/a/}');
Expect(1, 180, '\p{Line_Break=breakbefore}', "");
Expect(0, 180, '\p{^Line_Break=breakbefore}', "");
Expect(0, 180, '\P{Line_Break=breakbefore}', "");
Expect(1, 180, '\P{^Line_Break=breakbefore}', "");
Expect(0, 179, '\p{Line_Break=breakbefore}', "");
Expect(1, 179, '\p{^Line_Break=breakbefore}', "");
Expect(1, 179, '\P{Line_Break=breakbefore}', "");
Expect(0, 179, '\P{^Line_Break=breakbefore}', "");
Expect(1, 180, '\p{Line_Break=_-breakBefore}', "");
Expect(0, 180, '\p{^Line_Break=_-breakBefore}', "");
Expect(0, 180, '\P{Line_Break=_-breakBefore}', "");
Expect(1, 180, '\P{^Line_Break=_-breakBefore}', "");
Expect(0, 179, '\p{Line_Break=_-breakBefore}', "");
Expect(1, 179, '\p{^Line_Break=_-breakBefore}', "");
Expect(1, 179, '\P{Line_Break=_-breakBefore}', "");
Expect(0, 179, '\P{^Line_Break=_-breakBefore}', "");
Error('\p{Lb=/a/_BB}');
Error('\P{Lb=/a/_BB}');
Expect(1, 180, '\p{Lb=bb}', "");
Expect(0, 180, '\p{^Lb=bb}', "");
Expect(0, 180, '\P{Lb=bb}', "");
Expect(1, 180, '\P{^Lb=bb}', "");
Expect(0, 179, '\p{Lb=bb}', "");
Expect(1, 179, '\p{^Lb=bb}', "");
Expect(1, 179, '\P{Lb=bb}', "");
Expect(0, 179, '\P{^Lb=bb}', "");
Expect(1, 180, '\p{Lb=-_BB}', "");
Expect(0, 180, '\p{^Lb=-_BB}', "");
Expect(0, 180, '\P{Lb=-_BB}', "");
Expect(1, 180, '\P{^Lb=-_BB}', "");
Expect(0, 179, '\p{Lb=-_BB}', "");
Expect(1, 179, '\p{^Lb=-_BB}', "");
Expect(1, 179, '\P{Lb=-_BB}', "");
Expect(0, 179, '\P{^Lb=-_BB}', "");
Error('\p{Is_Line_Break=/a/	 Break BEFORE}');
Error('\P{Is_Line_Break=/a/	 Break BEFORE}');
Expect(1, 180, '\p{Is_Line_Break=breakbefore}', "");
Expect(0, 180, '\p{^Is_Line_Break=breakbefore}', "");
Expect(0, 180, '\P{Is_Line_Break=breakbefore}', "");
Expect(1, 180, '\P{^Is_Line_Break=breakbefore}', "");
Expect(0, 179, '\p{Is_Line_Break=breakbefore}', "");
Expect(1, 179, '\p{^Is_Line_Break=breakbefore}', "");
Expect(1, 179, '\P{Is_Line_Break=breakbefore}', "");
Expect(0, 179, '\P{^Is_Line_Break=breakbefore}', "");
Expect(1, 180, '\p{Is_Line_Break=- break_Before}', "");
Expect(0, 180, '\p{^Is_Line_Break=- break_Before}', "");
Expect(0, 180, '\P{Is_Line_Break=- break_Before}', "");
Expect(1, 180, '\P{^Is_Line_Break=- break_Before}', "");
Expect(0, 179, '\p{Is_Line_Break=- break_Before}', "");
Expect(1, 179, '\p{^Is_Line_Break=- break_Before}', "");
Expect(1, 179, '\P{Is_Line_Break=- break_Before}', "");
Expect(0, 179, '\P{^Is_Line_Break=- break_Before}', "");
Error('\p{Is_Lb=	BB/a/}');
Error('\P{Is_Lb=	BB/a/}');
Expect(1, 180, '\p{Is_Lb=bb}', "");
Expect(0, 180, '\p{^Is_Lb=bb}', "");
Expect(0, 180, '\P{Is_Lb=bb}', "");
Expect(1, 180, '\P{^Is_Lb=bb}', "");
Expect(0, 179, '\p{Is_Lb=bb}', "");
Expect(1, 179, '\p{^Is_Lb=bb}', "");
Expect(1, 179, '\P{Is_Lb=bb}', "");
Expect(0, 179, '\P{^Is_Lb=bb}', "");
Expect(1, 180, '\p{Is_Lb=-BB}', "");
Expect(0, 180, '\p{^Is_Lb=-BB}', "");
Expect(0, 180, '\P{Is_Lb=-BB}', "");
Expect(1, 180, '\P{^Is_Lb=-BB}', "");
Expect(0, 179, '\p{Is_Lb=-BB}', "");
Expect(1, 179, '\p{^Is_Lb=-BB}', "");
Expect(1, 179, '\P{Is_Lb=-BB}', "");
Expect(0, 179, '\P{^Is_Lb=-BB}', "");
Error('\p{Line_Break=/a/--zwspace}');
Error('\P{Line_Break=/a/--zwspace}');
Expect(1, 8203, '\p{Line_Break=zwspace}', "");
Expect(0, 8203, '\p{^Line_Break=zwspace}', "");
Expect(0, 8203, '\P{Line_Break=zwspace}', "");
Expect(1, 8203, '\P{^Line_Break=zwspace}', "");
Expect(0, 8202, '\p{Line_Break=zwspace}', "");
Expect(1, 8202, '\p{^Line_Break=zwspace}', "");
Expect(1, 8202, '\P{Line_Break=zwspace}', "");
Expect(0, 8202, '\P{^Line_Break=zwspace}', "");
Expect(1, 8203, '\p{Line_Break=__ZWSPACE}', "");
Expect(0, 8203, '\p{^Line_Break=__ZWSPACE}', "");
Expect(0, 8203, '\P{Line_Break=__ZWSPACE}', "");
Expect(1, 8203, '\P{^Line_Break=__ZWSPACE}', "");
Expect(0, 8202, '\p{Line_Break=__ZWSPACE}', "");
Expect(1, 8202, '\p{^Line_Break=__ZWSPACE}', "");
Expect(1, 8202, '\P{Line_Break=__ZWSPACE}', "");
Expect(0, 8202, '\P{^Line_Break=__ZWSPACE}', "");
Error('\p{Lb=	ZW/a/}');
Error('\P{Lb=	ZW/a/}');
Expect(1, 8203, '\p{Lb=zw}', "");
Expect(0, 8203, '\p{^Lb=zw}', "");
Expect(0, 8203, '\P{Lb=zw}', "");
Expect(1, 8203, '\P{^Lb=zw}', "");
Expect(0, 8202, '\p{Lb=zw}', "");
Expect(1, 8202, '\p{^Lb=zw}', "");
Expect(1, 8202, '\P{Lb=zw}', "");
Expect(0, 8202, '\P{^Lb=zw}', "");
Expect(1, 8203, '\p{Lb=	zw}', "");
Expect(0, 8203, '\p{^Lb=	zw}', "");
Expect(0, 8203, '\P{Lb=	zw}', "");
Expect(1, 8203, '\P{^Lb=	zw}', "");
Expect(0, 8202, '\p{Lb=	zw}', "");
Expect(1, 8202, '\p{^Lb=	zw}', "");
Expect(1, 8202, '\P{Lb=	zw}', "");
Expect(0, 8202, '\P{^Lb=	zw}', "");
Error('\p{Is_Line_Break=_/a/zwspace}');
Error('\P{Is_Line_Break=_/a/zwspace}');
Expect(1, 8203, '\p{Is_Line_Break=zwspace}', "");
Expect(0, 8203, '\p{^Is_Line_Break=zwspace}', "");
Expect(0, 8203, '\P{Is_Line_Break=zwspace}', "");
Expect(1, 8203, '\P{^Is_Line_Break=zwspace}', "");
Expect(0, 8202, '\p{Is_Line_Break=zwspace}', "");
Expect(1, 8202, '\p{^Is_Line_Break=zwspace}', "");
Expect(1, 8202, '\P{Is_Line_Break=zwspace}', "");
Expect(0, 8202, '\P{^Is_Line_Break=zwspace}', "");
Expect(1, 8203, '\p{Is_Line_Break=-	ZWSPACE}', "");
Expect(0, 8203, '\p{^Is_Line_Break=-	ZWSPACE}', "");
Expect(0, 8203, '\P{Is_Line_Break=-	ZWSPACE}', "");
Expect(1, 8203, '\P{^Is_Line_Break=-	ZWSPACE}', "");
Expect(0, 8202, '\p{Is_Line_Break=-	ZWSPACE}', "");
Expect(1, 8202, '\p{^Is_Line_Break=-	ZWSPACE}', "");
Expect(1, 8202, '\P{Is_Line_Break=-	ZWSPACE}', "");
Expect(0, 8202, '\P{^Is_Line_Break=-	ZWSPACE}', "");
Error('\p{Is_Lb=-ZW/a/}');
Error('\P{Is_Lb=-ZW/a/}');
Expect(1, 8203, '\p{Is_Lb=zw}', "");
Expect(0, 8203, '\p{^Is_Lb=zw}', "");
Expect(0, 8203, '\P{Is_Lb=zw}', "");
Expect(1, 8203, '\P{^Is_Lb=zw}', "");
Expect(0, 8202, '\p{Is_Lb=zw}', "");
Expect(1, 8202, '\p{^Is_Lb=zw}', "");
Expect(1, 8202, '\P{Is_Lb=zw}', "");
Expect(0, 8202, '\P{^Is_Lb=zw}', "");
Expect(1, 8203, '\p{Is_Lb=	_ZW}', "");
Expect(0, 8203, '\p{^Is_Lb=	_ZW}', "");
Expect(0, 8203, '\P{Is_Lb=	_ZW}', "");
Expect(1, 8203, '\P{^Is_Lb=	_ZW}', "");
Expect(0, 8202, '\p{Is_Lb=	_ZW}', "");
Expect(1, 8202, '\p{^Is_Lb=	_ZW}', "");
Expect(1, 8202, '\P{Is_Lb=	_ZW}', "");
Expect(0, 8202, '\P{^Is_Lb=	_ZW}', "");
Error('\p{Line_Break=/a/-jl}');
Error('\P{Line_Break=/a/-jl}');
Expect(1, 4447, '\p{Line_Break=jl}', "");
Expect(0, 4447, '\p{^Line_Break=jl}', "");
Expect(0, 4447, '\P{Line_Break=jl}', "");
Expect(1, 4447, '\P{^Line_Break=jl}', "");
Expect(0, 4351, '\p{Line_Break=jl}', "");
Expect(1, 4351, '\p{^Line_Break=jl}', "");
Expect(1, 4351, '\P{Line_Break=jl}', "");
Expect(0, 4351, '\P{^Line_Break=jl}', "");
Expect(1, 4447, '\p{Line_Break= -JL}', "");
Expect(0, 4447, '\p{^Line_Break= -JL}', "");
Expect(0, 4447, '\P{Line_Break= -JL}', "");
Expect(1, 4447, '\P{^Line_Break= -JL}', "");
Expect(0, 4351, '\p{Line_Break= -JL}', "");
Expect(1, 4351, '\p{^Line_Break= -JL}', "");
Expect(1, 4351, '\P{Line_Break= -JL}', "");
Expect(0, 4351, '\P{^Line_Break= -JL}', "");
Error('\p{Lb=_:=JL}');
Error('\P{Lb=_:=JL}');
Expect(1, 4447, '\p{Lb=jl}', "");
Expect(0, 4447, '\p{^Lb=jl}', "");
Expect(0, 4447, '\P{Lb=jl}', "");
Expect(1, 4447, '\P{^Lb=jl}', "");
Expect(0, 4351, '\p{Lb=jl}', "");
Expect(1, 4351, '\p{^Lb=jl}', "");
Expect(1, 4351, '\P{Lb=jl}', "");
Expect(0, 4351, '\P{^Lb=jl}', "");
Expect(1, 4447, '\p{Lb=_	JL}', "");
Expect(0, 4447, '\p{^Lb=_	JL}', "");
Expect(0, 4447, '\P{Lb=_	JL}', "");
Expect(1, 4447, '\P{^Lb=_	JL}', "");
Expect(0, 4351, '\p{Lb=_	JL}', "");
Expect(1, 4351, '\p{^Lb=_	JL}', "");
Expect(1, 4351, '\P{Lb=_	JL}', "");
Expect(0, 4351, '\P{^Lb=_	JL}', "");
Error('\p{Is_Line_Break: 	-jl:=}');
Error('\P{Is_Line_Break: 	-jl:=}');
Expect(1, 4447, '\p{Is_Line_Break=jl}', "");
Expect(0, 4447, '\p{^Is_Line_Break=jl}', "");
Expect(0, 4447, '\P{Is_Line_Break=jl}', "");
Expect(1, 4447, '\P{^Is_Line_Break=jl}', "");
Expect(0, 4351, '\p{Is_Line_Break=jl}', "");
Expect(1, 4351, '\p{^Is_Line_Break=jl}', "");
Expect(1, 4351, '\P{Is_Line_Break=jl}', "");
Expect(0, 4351, '\P{^Is_Line_Break=jl}', "");
Expect(1, 4447, '\p{Is_Line_Break=__jl}', "");
Expect(0, 4447, '\p{^Is_Line_Break=__jl}', "");
Expect(0, 4447, '\P{Is_Line_Break=__jl}', "");
Expect(1, 4447, '\P{^Is_Line_Break=__jl}', "");
Expect(0, 4351, '\p{Is_Line_Break=__jl}', "");
Expect(1, 4351, '\p{^Is_Line_Break=__jl}', "");
Expect(1, 4351, '\P{Is_Line_Break=__jl}', "");
Expect(0, 4351, '\P{^Is_Line_Break=__jl}', "");
Error('\p{Is_Lb=:=-_JL}');
Error('\P{Is_Lb=:=-_JL}');
Expect(1, 4447, '\p{Is_Lb=jl}', "");
Expect(0, 4447, '\p{^Is_Lb=jl}', "");
Expect(0, 4447, '\P{Is_Lb=jl}', "");
Expect(1, 4447, '\P{^Is_Lb=jl}', "");
Expect(0, 4351, '\p{Is_Lb=jl}', "");
Expect(1, 4351, '\p{^Is_Lb=jl}', "");
Expect(1, 4351, '\P{Is_Lb=jl}', "");
Expect(0, 4351, '\P{^Is_Lb=jl}', "");
Expect(1, 4447, '\p{Is_Lb=	JL}', "");
Expect(0, 4447, '\p{^Is_Lb=	JL}', "");
Expect(0, 4447, '\P{Is_Lb=	JL}', "");
Expect(1, 4447, '\P{^Is_Lb=	JL}', "");
Expect(0, 4351, '\p{Is_Lb=	JL}', "");
Expect(1, 4351, '\p{^Is_Lb=	JL}', "");
Expect(1, 4351, '\P{Is_Lb=	JL}', "");
Expect(0, 4351, '\P{^Is_Lb=	JL}', "");
Error('\p{Line_Break=_/a/break	Symbols}');
Error('\P{Line_Break=_/a/break	Symbols}');
Expect(1, 47, '\p{Line_Break=breaksymbols}', "");
Expect(0, 47, '\p{^Line_Break=breaksymbols}', "");
Expect(0, 47, '\P{Line_Break=breaksymbols}', "");
Expect(1, 47, '\P{^Line_Break=breaksymbols}', "");
Expect(0, 46, '\p{Line_Break=breaksymbols}', "");
Expect(1, 46, '\p{^Line_Break=breaksymbols}', "");
Expect(1, 46, '\P{Line_Break=breaksymbols}', "");
Expect(0, 46, '\P{^Line_Break=breaksymbols}', "");
Expect(1, 47, '\p{Line_Break=		break-SYMBOLS}', "");
Expect(0, 47, '\p{^Line_Break=		break-SYMBOLS}', "");
Expect(0, 47, '\P{Line_Break=		break-SYMBOLS}', "");
Expect(1, 47, '\P{^Line_Break=		break-SYMBOLS}', "");
Expect(0, 46, '\p{Line_Break=		break-SYMBOLS}', "");
Expect(1, 46, '\p{^Line_Break=		break-SYMBOLS}', "");
Expect(1, 46, '\P{Line_Break=		break-SYMBOLS}', "");
Expect(0, 46, '\P{^Line_Break=		break-SYMBOLS}', "");
Error('\p{Lb=__SY/a/}');
Error('\P{Lb=__SY/a/}');
Expect(1, 47, '\p{Lb=sy}', "");
Expect(0, 47, '\p{^Lb=sy}', "");
Expect(0, 47, '\P{Lb=sy}', "");
Expect(1, 47, '\P{^Lb=sy}', "");
Expect(0, 46, '\p{Lb=sy}', "");
Expect(1, 46, '\p{^Lb=sy}', "");
Expect(1, 46, '\P{Lb=sy}', "");
Expect(0, 46, '\P{^Lb=sy}', "");
Expect(1, 47, '\p{Lb=_SY}', "");
Expect(0, 47, '\p{^Lb=_SY}', "");
Expect(0, 47, '\P{Lb=_SY}', "");
Expect(1, 47, '\P{^Lb=_SY}', "");
Expect(0, 46, '\p{Lb=_SY}', "");
Expect(1, 46, '\p{^Lb=_SY}', "");
Expect(1, 46, '\P{Lb=_SY}', "");
Expect(0, 46, '\P{^Lb=_SY}', "");
Error('\p{Is_Line_Break:   -	Break:=Symbols}');
Error('\P{Is_Line_Break:   -	Break:=Symbols}');
Expect(1, 47, '\p{Is_Line_Break=breaksymbols}', "");
Expect(0, 47, '\p{^Is_Line_Break=breaksymbols}', "");
Expect(0, 47, '\P{Is_Line_Break=breaksymbols}', "");
Expect(1, 47, '\P{^Is_Line_Break=breaksymbols}', "");
Expect(0, 46, '\p{Is_Line_Break=breaksymbols}', "");
Expect(1, 46, '\p{^Is_Line_Break=breaksymbols}', "");
Expect(1, 46, '\P{Is_Line_Break=breaksymbols}', "");
Expect(0, 46, '\P{^Is_Line_Break=breaksymbols}', "");
Expect(1, 47, '\p{Is_Line_Break=_-Break	SYMBOLS}', "");
Expect(0, 47, '\p{^Is_Line_Break=_-Break	SYMBOLS}', "");
Expect(0, 47, '\P{Is_Line_Break=_-Break	SYMBOLS}', "");
Expect(1, 47, '\P{^Is_Line_Break=_-Break	SYMBOLS}', "");
Expect(0, 46, '\p{Is_Line_Break=_-Break	SYMBOLS}', "");
Expect(1, 46, '\p{^Is_Line_Break=_-Break	SYMBOLS}', "");
Expect(1, 46, '\P{Is_Line_Break=_-Break	SYMBOLS}', "");
Expect(0, 46, '\P{^Is_Line_Break=_-Break	SYMBOLS}', "");
Error('\p{Is_Lb=SY/a/}');
Error('\P{Is_Lb=SY/a/}');
Expect(1, 47, '\p{Is_Lb: sy}', "");
Expect(0, 47, '\p{^Is_Lb: sy}', "");
Expect(0, 47, '\P{Is_Lb: sy}', "");
Expect(1, 47, '\P{^Is_Lb: sy}', "");
Expect(0, 46, '\p{Is_Lb: sy}', "");
Expect(1, 46, '\p{^Is_Lb: sy}', "");
Expect(1, 46, '\P{Is_Lb: sy}', "");
Expect(0, 46, '\P{^Is_Lb: sy}', "");
Expect(1, 47, '\p{Is_Lb= _SY}', "");
Expect(0, 47, '\p{^Is_Lb= _SY}', "");
Expect(0, 47, '\P{Is_Lb= _SY}', "");
Expect(1, 47, '\P{^Is_Lb= _SY}', "");
Expect(0, 46, '\p{Is_Lb= _SY}', "");
Expect(1, 46, '\p{^Is_Lb= _SY}', "");
Expect(1, 46, '\P{Is_Lb= _SY}', "");
Expect(0, 46, '\P{^Is_Lb= _SY}', "");
Error('\p{Line_Break=- prefix/a/Numeric}');
Error('\P{Line_Break=- prefix/a/Numeric}');
Expect(1, 36, '\p{Line_Break=prefixnumeric}', "");
Expect(0, 36, '\p{^Line_Break=prefixnumeric}', "");
Expect(0, 36, '\P{Line_Break=prefixnumeric}', "");
Expect(1, 36, '\P{^Line_Break=prefixnumeric}', "");
Expect(0, 35, '\p{Line_Break=prefixnumeric}', "");
Expect(1, 35, '\p{^Line_Break=prefixnumeric}', "");
Expect(1, 35, '\P{Line_Break=prefixnumeric}', "");
Expect(0, 35, '\P{^Line_Break=prefixnumeric}', "");
Expect(1, 36, '\p{Line_Break= PREFIX	NUMERIC}', "");
Expect(0, 36, '\p{^Line_Break= PREFIX	NUMERIC}', "");
Expect(0, 36, '\P{Line_Break= PREFIX	NUMERIC}', "");
Expect(1, 36, '\P{^Line_Break= PREFIX	NUMERIC}', "");
Expect(0, 35, '\p{Line_Break= PREFIX	NUMERIC}', "");
Expect(1, 35, '\p{^Line_Break= PREFIX	NUMERIC}', "");
Expect(1, 35, '\P{Line_Break= PREFIX	NUMERIC}', "");
Expect(0, 35, '\P{^Line_Break= PREFIX	NUMERIC}', "");
Error('\p{Lb= :=PR}');
Error('\P{Lb= :=PR}');
Expect(1, 36, '\p{Lb=pr}', "");
Expect(0, 36, '\p{^Lb=pr}', "");
Expect(0, 36, '\P{Lb=pr}', "");
Expect(1, 36, '\P{^Lb=pr}', "");
Expect(0, 35, '\p{Lb=pr}', "");
Expect(1, 35, '\p{^Lb=pr}', "");
Expect(1, 35, '\P{Lb=pr}', "");
Expect(0, 35, '\P{^Lb=pr}', "");
Expect(1, 36, '\p{Lb=_PR}', "");
Expect(0, 36, '\p{^Lb=_PR}', "");
Expect(0, 36, '\P{Lb=_PR}', "");
Expect(1, 36, '\P{^Lb=_PR}', "");
Expect(0, 35, '\p{Lb=_PR}', "");
Expect(1, 35, '\p{^Lb=_PR}', "");
Expect(1, 35, '\P{Lb=_PR}', "");
Expect(0, 35, '\P{^Lb=_PR}', "");
Error('\p{Is_Line_Break=	-Prefix/a/Numeric}');
Error('\P{Is_Line_Break=	-Prefix/a/Numeric}');
Expect(1, 36, '\p{Is_Line_Break=prefixnumeric}', "");
Expect(0, 36, '\p{^Is_Line_Break=prefixnumeric}', "");
Expect(0, 36, '\P{Is_Line_Break=prefixnumeric}', "");
Expect(1, 36, '\P{^Is_Line_Break=prefixnumeric}', "");
Expect(0, 35, '\p{Is_Line_Break=prefixnumeric}', "");
Expect(1, 35, '\p{^Is_Line_Break=prefixnumeric}', "");
Expect(1, 35, '\P{Is_Line_Break=prefixnumeric}', "");
Expect(0, 35, '\P{^Is_Line_Break=prefixnumeric}', "");
Expect(1, 36, '\p{Is_Line_Break=- Prefix-numeric}', "");
Expect(0, 36, '\p{^Is_Line_Break=- Prefix-numeric}', "");
Expect(0, 36, '\P{Is_Line_Break=- Prefix-numeric}', "");
Expect(1, 36, '\P{^Is_Line_Break=- Prefix-numeric}', "");
Expect(0, 35, '\p{Is_Line_Break=- Prefix-numeric}', "");
Expect(1, 35, '\p{^Is_Line_Break=- Prefix-numeric}', "");
Expect(1, 35, '\P{Is_Line_Break=- Prefix-numeric}', "");
Expect(0, 35, '\P{^Is_Line_Break=- Prefix-numeric}', "");
Error('\p{Is_Lb= /a/pr}');
Error('\P{Is_Lb= /a/pr}');
Expect(1, 36, '\p{Is_Lb=pr}', "");
Expect(0, 36, '\p{^Is_Lb=pr}', "");
Expect(0, 36, '\P{Is_Lb=pr}', "");
Expect(1, 36, '\P{^Is_Lb=pr}', "");
Expect(0, 35, '\p{Is_Lb=pr}', "");
Expect(1, 35, '\p{^Is_Lb=pr}', "");
Expect(1, 35, '\P{Is_Lb=pr}', "");
Expect(0, 35, '\P{^Is_Lb=pr}', "");
Expect(1, 36, '\p{Is_Lb=__PR}', "");
Expect(0, 36, '\p{^Is_Lb=__PR}', "");
Expect(0, 36, '\P{Is_Lb=__PR}', "");
Expect(1, 36, '\P{^Is_Lb=__PR}', "");
Expect(0, 35, '\p{Is_Lb=__PR}', "");
Expect(1, 35, '\p{^Is_Lb=__PR}', "");
Expect(1, 35, '\P{Is_Lb=__PR}', "");
Expect(0, 35, '\P{^Is_Lb=__PR}', "");
Error('\p{Line_Break=Close-Parenthesis/a/}');
Error('\P{Line_Break=Close-Parenthesis/a/}');
Expect(1, 41, '\p{Line_Break=closeparenthesis}', "");
Expect(0, 41, '\p{^Line_Break=closeparenthesis}', "");
Expect(0, 41, '\P{Line_Break=closeparenthesis}', "");
Expect(1, 41, '\P{^Line_Break=closeparenthesis}', "");
Expect(0, 40, '\p{Line_Break=closeparenthesis}', "");
Expect(1, 40, '\p{^Line_Break=closeparenthesis}', "");
Expect(1, 40, '\P{Line_Break=closeparenthesis}', "");
Expect(0, 40, '\P{^Line_Break=closeparenthesis}', "");
Expect(1, 41, '\p{Line_Break=--ClosePARENTHESIS}', "");
Expect(0, 41, '\p{^Line_Break=--ClosePARENTHESIS}', "");
Expect(0, 41, '\P{Line_Break=--ClosePARENTHESIS}', "");
Expect(1, 41, '\P{^Line_Break=--ClosePARENTHESIS}', "");
Expect(0, 40, '\p{Line_Break=--ClosePARENTHESIS}', "");
Expect(1, 40, '\p{^Line_Break=--ClosePARENTHESIS}', "");
Expect(1, 40, '\P{Line_Break=--ClosePARENTHESIS}', "");
Expect(0, 40, '\P{^Line_Break=--ClosePARENTHESIS}', "");
Error('\p{Lb:   --CP/a/}');
Error('\P{Lb:   --CP/a/}');
Expect(1, 41, '\p{Lb:cp}', "");
Expect(0, 41, '\p{^Lb:cp}', "");
Expect(0, 41, '\P{Lb:cp}', "");
Expect(1, 41, '\P{^Lb:cp}', "");
Expect(0, 40, '\p{Lb:cp}', "");
Expect(1, 40, '\p{^Lb:cp}', "");
Expect(1, 40, '\P{Lb:cp}', "");
Expect(0, 40, '\P{^Lb:cp}', "");
Expect(1, 41, '\p{Lb=	-cp}', "");
Expect(0, 41, '\p{^Lb=	-cp}', "");
Expect(0, 41, '\P{Lb=	-cp}', "");
Expect(1, 41, '\P{^Lb=	-cp}', "");
Expect(0, 40, '\p{Lb=	-cp}', "");
Expect(1, 40, '\p{^Lb=	-cp}', "");
Expect(1, 40, '\P{Lb=	-cp}', "");
Expect(0, 40, '\P{^Lb=	-cp}', "");
Error('\p{Is_Line_Break=/a/_-CLOSE_Parenthesis}');
Error('\P{Is_Line_Break=/a/_-CLOSE_Parenthesis}');
Expect(1, 41, '\p{Is_Line_Break=closeparenthesis}', "");
Expect(0, 41, '\p{^Is_Line_Break=closeparenthesis}', "");
Expect(0, 41, '\P{Is_Line_Break=closeparenthesis}', "");
Expect(1, 41, '\P{^Is_Line_Break=closeparenthesis}', "");
Expect(0, 40, '\p{Is_Line_Break=closeparenthesis}', "");
Expect(1, 40, '\p{^Is_Line_Break=closeparenthesis}', "");
Expect(1, 40, '\P{Is_Line_Break=closeparenthesis}', "");
Expect(0, 40, '\P{^Is_Line_Break=closeparenthesis}', "");
Expect(1, 41, '\p{Is_Line_Break=-	Close	Parenthesis}', "");
Expect(0, 41, '\p{^Is_Line_Break=-	Close	Parenthesis}', "");
Expect(0, 41, '\P{Is_Line_Break=-	Close	Parenthesis}', "");
Expect(1, 41, '\P{^Is_Line_Break=-	Close	Parenthesis}', "");
Expect(0, 40, '\p{Is_Line_Break=-	Close	Parenthesis}', "");
Expect(1, 40, '\p{^Is_Line_Break=-	Close	Parenthesis}', "");
Expect(1, 40, '\P{Is_Line_Break=-	Close	Parenthesis}', "");
Expect(0, 40, '\P{^Is_Line_Break=-	Close	Parenthesis}', "");
Error('\p{Is_Lb=-CP/a/}');
Error('\P{Is_Lb=-CP/a/}');
Expect(1, 41, '\p{Is_Lb=cp}', "");
Expect(0, 41, '\p{^Is_Lb=cp}', "");
Expect(0, 41, '\P{Is_Lb=cp}', "");
Expect(1, 41, '\P{^Is_Lb=cp}', "");
Expect(0, 40, '\p{Is_Lb=cp}', "");
Expect(1, 40, '\p{^Is_Lb=cp}', "");
Expect(1, 40, '\P{Is_Lb=cp}', "");
Expect(0, 40, '\P{^Is_Lb=cp}', "");
Expect(1, 41, '\p{Is_Lb= 	CP}', "");
Expect(0, 41, '\p{^Is_Lb= 	CP}', "");
Expect(0, 41, '\P{Is_Lb= 	CP}', "");
Expect(1, 41, '\P{^Is_Lb= 	CP}', "");
Expect(0, 40, '\p{Is_Lb= 	CP}', "");
Expect(1, 40, '\p{^Is_Lb= 	CP}', "");
Expect(1, 40, '\P{Is_Lb= 	CP}', "");
Expect(0, 40, '\P{^Is_Lb= 	CP}', "");
Error('\p{Line_Break=:=-H3}');
Error('\P{Line_Break=:=-H3}');
Expect(1, 44059, '\p{Line_Break=h3}', "");
Expect(0, 44059, '\p{^Line_Break=h3}', "");
Expect(0, 44059, '\P{Line_Break=h3}', "");
Expect(1, 44059, '\P{^Line_Break=h3}', "");
Expect(0, 44032, '\p{Line_Break=h3}', "");
Expect(1, 44032, '\p{^Line_Break=h3}', "");
Expect(1, 44032, '\P{Line_Break=h3}', "");
Expect(0, 44032, '\P{^Line_Break=h3}', "");
Expect(1, 44059, '\p{Line_Break:   -_H3}', "");
Expect(0, 44059, '\p{^Line_Break:   -_H3}', "");
Expect(0, 44059, '\P{Line_Break:   -_H3}', "");
Expect(1, 44059, '\P{^Line_Break:   -_H3}', "");
Expect(0, 44032, '\p{Line_Break:   -_H3}', "");
Expect(1, 44032, '\p{^Line_Break:   -_H3}', "");
Expect(1, 44032, '\P{Line_Break:   -_H3}', "");
Expect(0, 44032, '\P{^Line_Break:   -_H3}', "");
Error('\p{Lb=	h3:=}');
Error('\P{Lb=	h3:=}');
Expect(1, 44059, '\p{Lb=h3}', "");
Expect(0, 44059, '\p{^Lb=h3}', "");
Expect(0, 44059, '\P{Lb=h3}', "");
Expect(1, 44059, '\P{^Lb=h3}', "");
Expect(0, 44032, '\p{Lb=h3}', "");
Expect(1, 44032, '\p{^Lb=h3}', "");
Expect(1, 44032, '\P{Lb=h3}', "");
Expect(0, 44032, '\P{^Lb=h3}', "");
Expect(1, 44059, '\p{Lb= -H3}', "");
Expect(0, 44059, '\p{^Lb= -H3}', "");
Expect(0, 44059, '\P{Lb= -H3}', "");
Expect(1, 44059, '\P{^Lb= -H3}', "");
Expect(0, 44032, '\p{Lb= -H3}', "");
Expect(1, 44032, '\p{^Lb= -H3}', "");
Expect(1, 44032, '\P{Lb= -H3}', "");
Expect(0, 44032, '\P{^Lb= -H3}', "");
Error('\p{Is_Line_Break=-H3:=}');
Error('\P{Is_Line_Break=-H3:=}');
Expect(1, 44059, '\p{Is_Line_Break=h3}', "");
Expect(0, 44059, '\p{^Is_Line_Break=h3}', "");
Expect(0, 44059, '\P{Is_Line_Break=h3}', "");
Expect(1, 44059, '\P{^Is_Line_Break=h3}', "");
Expect(0, 44032, '\p{Is_Line_Break=h3}', "");
Expect(1, 44032, '\p{^Is_Line_Break=h3}', "");
Expect(1, 44032, '\P{Is_Line_Break=h3}', "");
Expect(0, 44032, '\P{^Is_Line_Break=h3}', "");
Expect(1, 44059, '\p{Is_Line_Break= H3}', "");
Expect(0, 44059, '\p{^Is_Line_Break= H3}', "");
Expect(0, 44059, '\P{Is_Line_Break= H3}', "");
Expect(1, 44059, '\P{^Is_Line_Break= H3}', "");
Expect(0, 44032, '\p{Is_Line_Break= H3}', "");
Expect(1, 44032, '\p{^Is_Line_Break= H3}', "");
Expect(1, 44032, '\P{Is_Line_Break= H3}', "");
Expect(0, 44032, '\P{^Is_Line_Break= H3}', "");
Error('\p{Is_Lb=-:=h3}');
Error('\P{Is_Lb=-:=h3}');
Expect(1, 44059, '\p{Is_Lb=h3}', "");
Expect(0, 44059, '\p{^Is_Lb=h3}', "");
Expect(0, 44059, '\P{Is_Lb=h3}', "");
Expect(1, 44059, '\P{^Is_Lb=h3}', "");
Expect(0, 44032, '\p{Is_Lb=h3}', "");
Expect(1, 44032, '\p{^Is_Lb=h3}', "");
Expect(1, 44032, '\P{Is_Lb=h3}', "");
Expect(0, 44032, '\P{^Is_Lb=h3}', "");
Expect(1, 44059, '\p{Is_Lb=	-H3}', "");
Expect(0, 44059, '\p{^Is_Lb=	-H3}', "");
Expect(0, 44059, '\P{Is_Lb=	-H3}', "");
Expect(1, 44059, '\P{^Is_Lb=	-H3}', "");
Expect(0, 44032, '\p{Is_Lb=	-H3}', "");
Expect(1, 44032, '\p{^Is_Lb=	-H3}', "");
Expect(1, 44032, '\P{Is_Lb=	-H3}', "");
Expect(0, 44032, '\P{^Is_Lb=	-H3}', "");
Error('\p{Line_Break= _carriage Return/a/}');
Error('\P{Line_Break= _carriage Return/a/}');
Expect(1, 13, '\p{Line_Break=carriagereturn}', "");
Expect(0, 13, '\p{^Line_Break=carriagereturn}', "");
Expect(0, 13, '\P{Line_Break=carriagereturn}', "");
Expect(1, 13, '\P{^Line_Break=carriagereturn}', "");
Expect(0, 12, '\p{Line_Break=carriagereturn}', "");
Expect(1, 12, '\p{^Line_Break=carriagereturn}', "");
Expect(1, 12, '\P{Line_Break=carriagereturn}', "");
Expect(0, 12, '\P{^Line_Break=carriagereturn}', "");
Expect(1, 13, '\p{Line_Break= _carriage	Return}', "");
Expect(0, 13, '\p{^Line_Break= _carriage	Return}', "");
Expect(0, 13, '\P{Line_Break= _carriage	Return}', "");
Expect(1, 13, '\P{^Line_Break= _carriage	Return}', "");
Expect(0, 12, '\p{Line_Break= _carriage	Return}', "");
Expect(1, 12, '\p{^Line_Break= _carriage	Return}', "");
Expect(1, 12, '\P{Line_Break= _carriage	Return}', "");
Expect(0, 12, '\P{^Line_Break= _carriage	Return}', "");
Error('\p{Lb=-:=CR}');
Error('\P{Lb=-:=CR}');
Expect(1, 13, '\p{Lb=cr}', "");
Expect(0, 13, '\p{^Lb=cr}', "");
Expect(0, 13, '\P{Lb=cr}', "");
Expect(1, 13, '\P{^Lb=cr}', "");
Expect(0, 12, '\p{Lb=cr}', "");
Expect(1, 12, '\p{^Lb=cr}', "");
Expect(1, 12, '\P{Lb=cr}', "");
Expect(0, 12, '\P{^Lb=cr}', "");
Expect(1, 13, '\p{Lb=_CR}', "");
Expect(0, 13, '\p{^Lb=_CR}', "");
Expect(0, 13, '\P{Lb=_CR}', "");
Expect(1, 13, '\P{^Lb=_CR}', "");
Expect(0, 12, '\p{Lb=_CR}', "");
Expect(1, 12, '\p{^Lb=_CR}', "");
Expect(1, 12, '\P{Lb=_CR}', "");
Expect(0, 12, '\P{^Lb=_CR}', "");
Error('\p{Is_Line_Break=-:=Carriage	return}');
Error('\P{Is_Line_Break=-:=Carriage	return}');
Expect(1, 13, '\p{Is_Line_Break:carriagereturn}', "");
Expect(0, 13, '\p{^Is_Line_Break:carriagereturn}', "");
Expect(0, 13, '\P{Is_Line_Break:carriagereturn}', "");
Expect(1, 13, '\P{^Is_Line_Break:carriagereturn}', "");
Expect(0, 12, '\p{Is_Line_Break:carriagereturn}', "");
Expect(1, 12, '\p{^Is_Line_Break:carriagereturn}', "");
Expect(1, 12, '\P{Is_Line_Break:carriagereturn}', "");
Expect(0, 12, '\P{^Is_Line_Break:carriagereturn}', "");
Expect(1, 13, '\p{Is_Line_Break=_Carriage Return}', "");
Expect(0, 13, '\p{^Is_Line_Break=_Carriage Return}', "");
Expect(0, 13, '\P{Is_Line_Break=_Carriage Return}', "");
Expect(1, 13, '\P{^Is_Line_Break=_Carriage Return}', "");
Expect(0, 12, '\p{Is_Line_Break=_Carriage Return}', "");
Expect(1, 12, '\p{^Is_Line_Break=_Carriage Return}', "");
Expect(1, 12, '\P{Is_Line_Break=_Carriage Return}', "");
Expect(0, 12, '\P{^Is_Line_Break=_Carriage Return}', "");
Error('\p{Is_Lb=-:=cr}');
Error('\P{Is_Lb=-:=cr}');
Expect(1, 13, '\p{Is_Lb=cr}', "");
Expect(0, 13, '\p{^Is_Lb=cr}', "");
Expect(0, 13, '\P{Is_Lb=cr}', "");
Expect(1, 13, '\P{^Is_Lb=cr}', "");
Expect(0, 12, '\p{Is_Lb=cr}', "");
Expect(1, 12, '\p{^Is_Lb=cr}', "");
Expect(1, 12, '\P{Is_Lb=cr}', "");
Expect(0, 12, '\P{^Is_Lb=cr}', "");
Expect(1, 13, '\p{Is_Lb=-_CR}', "");
Expect(0, 13, '\p{^Is_Lb=-_CR}', "");
Expect(0, 13, '\P{Is_Lb=-_CR}', "");
Expect(1, 13, '\P{^Is_Lb=-_CR}', "");
Expect(0, 12, '\p{Is_Lb=-_CR}', "");
Expect(1, 12, '\p{^Is_Lb=-_CR}', "");
Expect(1, 12, '\P{Is_Lb=-_CR}', "");
Expect(0, 12, '\P{^Is_Lb=-_CR}', "");
Error('\p{Line_Break= _CLOSE:=punctuation}');
Error('\P{Line_Break= _CLOSE:=punctuation}');
Expect(1, 125, '\p{Line_Break=closepunctuation}', "");
Expect(0, 125, '\p{^Line_Break=closepunctuation}', "");
Expect(0, 125, '\P{Line_Break=closepunctuation}', "");
Expect(1, 125, '\P{^Line_Break=closepunctuation}', "");
Expect(0, 124, '\p{Line_Break=closepunctuation}', "");
Expect(1, 124, '\p{^Line_Break=closepunctuation}', "");
Expect(1, 124, '\P{Line_Break=closepunctuation}', "");
Expect(0, 124, '\P{^Line_Break=closepunctuation}', "");
Expect(1, 125, '\p{Line_Break: 	CLOSE Punctuation}', "");
Expect(0, 125, '\p{^Line_Break: 	CLOSE Punctuation}', "");
Expect(0, 125, '\P{Line_Break: 	CLOSE Punctuation}', "");
Expect(1, 125, '\P{^Line_Break: 	CLOSE Punctuation}', "");
Expect(0, 124, '\p{Line_Break: 	CLOSE Punctuation}', "");
Expect(1, 124, '\p{^Line_Break: 	CLOSE Punctuation}', "");
Expect(1, 124, '\P{Line_Break: 	CLOSE Punctuation}', "");
Expect(0, 124, '\P{^Line_Break: 	CLOSE Punctuation}', "");
Error('\p{Lb=/a/	 CL}');
Error('\P{Lb=/a/	 CL}');
Expect(1, 125, '\p{Lb=cl}', "");
Expect(0, 125, '\p{^Lb=cl}', "");
Expect(0, 125, '\P{Lb=cl}', "");
Expect(1, 125, '\P{^Lb=cl}', "");
Expect(0, 124, '\p{Lb=cl}', "");
Expect(1, 124, '\p{^Lb=cl}', "");
Expect(1, 124, '\P{Lb=cl}', "");
Expect(0, 124, '\P{^Lb=cl}', "");
Expect(1, 125, '\p{Lb=-CL}', "");
Expect(0, 125, '\p{^Lb=-CL}', "");
Expect(0, 125, '\P{Lb=-CL}', "");
Expect(1, 125, '\P{^Lb=-CL}', "");
Expect(0, 124, '\p{Lb=-CL}', "");
Expect(1, 124, '\p{^Lb=-CL}', "");
Expect(1, 124, '\P{Lb=-CL}', "");
Expect(0, 124, '\P{^Lb=-CL}', "");
Error('\p{Is_Line_Break=-close:=PUNCTUATION}');
Error('\P{Is_Line_Break=-close:=PUNCTUATION}');
Expect(1, 125, '\p{Is_Line_Break:   closepunctuation}', "");
Expect(0, 125, '\p{^Is_Line_Break:   closepunctuation}', "");
Expect(0, 125, '\P{Is_Line_Break:   closepunctuation}', "");
Expect(1, 125, '\P{^Is_Line_Break:   closepunctuation}', "");
Expect(0, 124, '\p{Is_Line_Break:   closepunctuation}', "");
Expect(1, 124, '\p{^Is_Line_Break:   closepunctuation}', "");
Expect(1, 124, '\P{Is_Line_Break:   closepunctuation}', "");
Expect(0, 124, '\P{^Is_Line_Break:   closepunctuation}', "");
Expect(1, 125, '\p{Is_Line_Break=--Close	punctuation}', "");
Expect(0, 125, '\p{^Is_Line_Break=--Close	punctuation}', "");
Expect(0, 125, '\P{Is_Line_Break=--Close	punctuation}', "");
Expect(1, 125, '\P{^Is_Line_Break=--Close	punctuation}', "");
Expect(0, 124, '\p{Is_Line_Break=--Close	punctuation}', "");
Expect(1, 124, '\p{^Is_Line_Break=--Close	punctuation}', "");
Expect(1, 124, '\P{Is_Line_Break=--Close	punctuation}', "");
Expect(0, 124, '\P{^Is_Line_Break=--Close	punctuation}', "");
Error('\p{Is_Lb=	:=cl}');
Error('\P{Is_Lb=	:=cl}');
Expect(1, 125, '\p{Is_Lb=cl}', "");
Expect(0, 125, '\p{^Is_Lb=cl}', "");
Expect(0, 125, '\P{Is_Lb=cl}', "");
Expect(1, 125, '\P{^Is_Lb=cl}', "");
Expect(0, 124, '\p{Is_Lb=cl}', "");
Expect(1, 124, '\p{^Is_Lb=cl}', "");
Expect(1, 124, '\P{Is_Lb=cl}', "");
Expect(0, 124, '\P{^Is_Lb=cl}', "");
Expect(1, 125, '\p{Is_Lb=		cl}', "");
Expect(0, 125, '\p{^Is_Lb=		cl}', "");
Expect(0, 125, '\P{Is_Lb=		cl}', "");
Expect(1, 125, '\P{^Is_Lb=		cl}', "");
Expect(0, 124, '\p{Is_Lb=		cl}', "");
Expect(1, 124, '\p{^Is_Lb=		cl}', "");
Expect(1, 124, '\P{Is_Lb=		cl}', "");
Expect(0, 124, '\P{^Is_Lb=		cl}', "");
Error('\p{Line_Break=/a/-CombiningMARK}');
Error('\P{Line_Break=/a/-CombiningMARK}');
Expect(1, 8, '\p{Line_Break=combiningmark}', "");
Expect(0, 8, '\p{^Line_Break=combiningmark}', "");
Expect(0, 8, '\P{Line_Break=combiningmark}', "");
Expect(1, 8, '\P{^Line_Break=combiningmark}', "");
Expect(0, 13, '\p{Line_Break=combiningmark}', "");
Expect(1, 13, '\p{^Line_Break=combiningmark}', "");
Expect(1, 13, '\P{Line_Break=combiningmark}', "");
Expect(0, 13, '\P{^Line_Break=combiningmark}', "");
Expect(1, 8, '\p{Line_Break=-Combining	MARK}', "");
Expect(0, 8, '\p{^Line_Break=-Combining	MARK}', "");
Expect(0, 8, '\P{Line_Break=-Combining	MARK}', "");
Expect(1, 8, '\P{^Line_Break=-Combining	MARK}', "");
Expect(0, 13, '\p{Line_Break=-Combining	MARK}', "");
Expect(1, 13, '\p{^Line_Break=-Combining	MARK}', "");
Expect(1, 13, '\P{Line_Break=-Combining	MARK}', "");
Expect(0, 13, '\P{^Line_Break=-Combining	MARK}', "");
Error('\p{Lb=:=CM}');
Error('\P{Lb=:=CM}');
Expect(1, 8, '\p{Lb=cm}', "");
Expect(0, 8, '\p{^Lb=cm}', "");
Expect(0, 8, '\P{Lb=cm}', "");
Expect(1, 8, '\P{^Lb=cm}', "");
Expect(0, 13, '\p{Lb=cm}', "");
Expect(1, 13, '\p{^Lb=cm}', "");
Expect(1, 13, '\P{Lb=cm}', "");
Expect(0, 13, '\P{^Lb=cm}', "");
Expect(1, 8, '\p{Lb=	CM}', "");
Expect(0, 8, '\p{^Lb=	CM}', "");
Expect(0, 8, '\P{Lb=	CM}', "");
Expect(1, 8, '\P{^Lb=	CM}', "");
Expect(0, 13, '\p{Lb=	CM}', "");
Expect(1, 13, '\p{^Lb=	CM}', "");
Expect(1, 13, '\P{Lb=	CM}', "");
Expect(0, 13, '\P{^Lb=	CM}', "");
Error('\p{Is_Line_Break=/a/ -Combiningmark}');
Error('\P{Is_Line_Break=/a/ -Combiningmark}');
Expect(1, 8, '\p{Is_Line_Break=combiningmark}', "");
Expect(0, 8, '\p{^Is_Line_Break=combiningmark}', "");
Expect(0, 8, '\P{Is_Line_Break=combiningmark}', "");
Expect(1, 8, '\P{^Is_Line_Break=combiningmark}', "");
Expect(0, 13, '\p{Is_Line_Break=combiningmark}', "");
Expect(1, 13, '\p{^Is_Line_Break=combiningmark}', "");
Expect(1, 13, '\P{Is_Line_Break=combiningmark}', "");
Expect(0, 13, '\P{^Is_Line_Break=combiningmark}', "");
Expect(1, 8, '\p{Is_Line_Break=	_Combining_MARK}', "");
Expect(0, 8, '\p{^Is_Line_Break=	_Combining_MARK}', "");
Expect(0, 8, '\P{Is_Line_Break=	_Combining_MARK}', "");
Expect(1, 8, '\P{^Is_Line_Break=	_Combining_MARK}', "");
Expect(0, 13, '\p{Is_Line_Break=	_Combining_MARK}', "");
Expect(1, 13, '\p{^Is_Line_Break=	_Combining_MARK}', "");
Expect(1, 13, '\P{Is_Line_Break=	_Combining_MARK}', "");
Expect(0, 13, '\P{^Is_Line_Break=	_Combining_MARK}', "");
Error('\p{Is_Lb::=_CM}');
Error('\P{Is_Lb::=_CM}');
Expect(1, 8, '\p{Is_Lb=cm}', "");
Expect(0, 8, '\p{^Is_Lb=cm}', "");
Expect(0, 8, '\P{Is_Lb=cm}', "");
Expect(1, 8, '\P{^Is_Lb=cm}', "");
Expect(0, 13, '\p{Is_Lb=cm}', "");
Expect(1, 13, '\p{^Is_Lb=cm}', "");
Expect(1, 13, '\P{Is_Lb=cm}', "");
Expect(0, 13, '\P{^Is_Lb=cm}', "");
Expect(1, 8, '\p{Is_Lb= _CM}', "");
Expect(0, 8, '\p{^Is_Lb= _CM}', "");
Expect(0, 8, '\P{Is_Lb= _CM}', "");
Expect(1, 8, '\P{^Is_Lb= _CM}', "");
Expect(0, 13, '\p{Is_Lb= _CM}', "");
Expect(1, 13, '\p{^Is_Lb= _CM}', "");
Expect(1, 13, '\P{Is_Lb= _CM}', "");
Expect(0, 13, '\P{^Is_Lb= _CM}', "");
Error('\p{Line_Break=-/a/mandatory	Break}');
Error('\P{Line_Break=-/a/mandatory	Break}');
Expect(1, 12, '\p{Line_Break=mandatorybreak}', "");
Expect(0, 12, '\p{^Line_Break=mandatorybreak}', "");
Expect(0, 12, '\P{Line_Break=mandatorybreak}', "");
Expect(1, 12, '\P{^Line_Break=mandatorybreak}', "");
Expect(0, 10, '\p{Line_Break=mandatorybreak}', "");
Expect(1, 10, '\p{^Line_Break=mandatorybreak}', "");
Expect(1, 10, '\P{Line_Break=mandatorybreak}', "");
Expect(0, 10, '\P{^Line_Break=mandatorybreak}', "");
Expect(1, 12, '\p{Line_Break=_	mandatory_BREAK}', "");
Expect(0, 12, '\p{^Line_Break=_	mandatory_BREAK}', "");
Expect(0, 12, '\P{Line_Break=_	mandatory_BREAK}', "");
Expect(1, 12, '\P{^Line_Break=_	mandatory_BREAK}', "");
Expect(0, 10, '\p{Line_Break=_	mandatory_BREAK}', "");
Expect(1, 10, '\p{^Line_Break=_	mandatory_BREAK}', "");
Expect(1, 10, '\P{Line_Break=_	mandatory_BREAK}', "");
Expect(0, 10, '\P{^Line_Break=_	mandatory_BREAK}', "");
Error('\p{Lb=	:=BK}');
Error('\P{Lb=	:=BK}');
Expect(1, 12, '\p{Lb=bk}', "");
Expect(0, 12, '\p{^Lb=bk}', "");
Expect(0, 12, '\P{Lb=bk}', "");
Expect(1, 12, '\P{^Lb=bk}', "");
Expect(0, 10, '\p{Lb=bk}', "");
Expect(1, 10, '\p{^Lb=bk}', "");
Expect(1, 10, '\P{Lb=bk}', "");
Expect(0, 10, '\P{^Lb=bk}', "");
Expect(1, 12, '\p{Lb=_bk}', "");
Expect(0, 12, '\p{^Lb=_bk}', "");
Expect(0, 12, '\P{Lb=_bk}', "");
Expect(1, 12, '\P{^Lb=_bk}', "");
Expect(0, 10, '\p{Lb=_bk}', "");
Expect(1, 10, '\p{^Lb=_bk}', "");
Expect(1, 10, '\P{Lb=_bk}', "");
Expect(0, 10, '\P{^Lb=_bk}', "");
Error('\p{Is_Line_Break=_mandatory:=break}');
Error('\P{Is_Line_Break=_mandatory:=break}');
Expect(1, 12, '\p{Is_Line_Break=mandatorybreak}', "");
Expect(0, 12, '\p{^Is_Line_Break=mandatorybreak}', "");
Expect(0, 12, '\P{Is_Line_Break=mandatorybreak}', "");
Expect(1, 12, '\P{^Is_Line_Break=mandatorybreak}', "");
Expect(0, 10, '\p{Is_Line_Break=mandatorybreak}', "");
Expect(1, 10, '\p{^Is_Line_Break=mandatorybreak}', "");
Expect(1, 10, '\P{Is_Line_Break=mandatorybreak}', "");
Expect(0, 10, '\P{^Is_Line_Break=mandatorybreak}', "");
Expect(1, 12, '\p{Is_Line_Break:   __MandatoryBreak}', "");
Expect(0, 12, '\p{^Is_Line_Break:   __MandatoryBreak}', "");
Expect(0, 12, '\P{Is_Line_Break:   __MandatoryBreak}', "");
Expect(1, 12, '\P{^Is_Line_Break:   __MandatoryBreak}', "");
Expect(0, 10, '\p{Is_Line_Break:   __MandatoryBreak}', "");
Expect(1, 10, '\p{^Is_Line_Break:   __MandatoryBreak}', "");
Expect(1, 10, '\P{Is_Line_Break:   __MandatoryBreak}', "");
Expect(0, 10, '\P{^Is_Line_Break:   __MandatoryBreak}', "");
Error('\p{Is_Lb=--BK:=}');
Error('\P{Is_Lb=--BK:=}');
Expect(1, 12, '\p{Is_Lb=bk}', "");
Expect(0, 12, '\p{^Is_Lb=bk}', "");
Expect(0, 12, '\P{Is_Lb=bk}', "");
Expect(1, 12, '\P{^Is_Lb=bk}', "");
Expect(0, 10, '\p{Is_Lb=bk}', "");
Expect(1, 10, '\p{^Is_Lb=bk}', "");
Expect(1, 10, '\P{Is_Lb=bk}', "");
Expect(0, 10, '\P{^Is_Lb=bk}', "");
Expect(1, 12, '\p{Is_Lb=	BK}', "");
Expect(0, 12, '\p{^Is_Lb=	BK}', "");
Expect(0, 12, '\P{Is_Lb=	BK}', "");
Expect(1, 12, '\P{^Is_Lb=	BK}', "");
Expect(0, 10, '\p{Is_Lb=	BK}', "");
Expect(1, 10, '\p{^Is_Lb=	BK}', "");
Expect(1, 10, '\P{Is_Lb=	BK}', "");
Expect(0, 10, '\P{^Is_Lb=	BK}', "");
Error('\p{Line_Break= /a/NONSTARTER}');
Error('\P{Line_Break= /a/NONSTARTER}');
Expect(1, 6102, '\p{Line_Break=nonstarter}', "");
Expect(0, 6102, '\p{^Line_Break=nonstarter}', "");
Expect(0, 6102, '\P{Line_Break=nonstarter}', "");
Expect(1, 6102, '\P{^Line_Break=nonstarter}', "");
Expect(0, 6101, '\p{Line_Break=nonstarter}', "");
Expect(1, 6101, '\p{^Line_Break=nonstarter}', "");
Expect(1, 6101, '\P{Line_Break=nonstarter}', "");
Expect(0, 6101, '\P{^Line_Break=nonstarter}', "");
Expect(1, 6102, '\p{Line_Break= 	NONSTARTER}', "");
Expect(0, 6102, '\p{^Line_Break= 	NONSTARTER}', "");
Expect(0, 6102, '\P{Line_Break= 	NONSTARTER}', "");
Expect(1, 6102, '\P{^Line_Break= 	NONSTARTER}', "");
Expect(0, 6101, '\p{Line_Break= 	NONSTARTER}', "");
Expect(1, 6101, '\p{^Line_Break= 	NONSTARTER}', "");
Expect(1, 6101, '\P{Line_Break= 	NONSTARTER}', "");
Expect(0, 6101, '\P{^Line_Break= 	NONSTARTER}', "");
Error('\p{Lb: :=NS}');
Error('\P{Lb: :=NS}');
Expect(1, 6102, '\p{Lb=ns}', "");
Expect(0, 6102, '\p{^Lb=ns}', "");
Expect(0, 6102, '\P{Lb=ns}', "");
Expect(1, 6102, '\P{^Lb=ns}', "");
Expect(0, 6101, '\p{Lb=ns}', "");
Expect(1, 6101, '\p{^Lb=ns}', "");
Expect(1, 6101, '\P{Lb=ns}', "");
Expect(0, 6101, '\P{^Lb=ns}', "");
Expect(1, 6102, '\p{Lb=_	NS}', "");
Expect(0, 6102, '\p{^Lb=_	NS}', "");
Expect(0, 6102, '\P{Lb=_	NS}', "");
Expect(1, 6102, '\P{^Lb=_	NS}', "");
Expect(0, 6101, '\p{Lb=_	NS}', "");
Expect(1, 6101, '\p{^Lb=_	NS}', "");
Expect(1, 6101, '\P{Lb=_	NS}', "");
Expect(0, 6101, '\P{^Lb=_	NS}', "");
Error('\p{Is_Line_Break=-:=NONSTARTER}');
Error('\P{Is_Line_Break=-:=NONSTARTER}');
Expect(1, 6102, '\p{Is_Line_Break=nonstarter}', "");
Expect(0, 6102, '\p{^Is_Line_Break=nonstarter}', "");
Expect(0, 6102, '\P{Is_Line_Break=nonstarter}', "");
Expect(1, 6102, '\P{^Is_Line_Break=nonstarter}', "");
Expect(0, 6101, '\p{Is_Line_Break=nonstarter}', "");
Expect(1, 6101, '\p{^Is_Line_Break=nonstarter}', "");
Expect(1, 6101, '\P{Is_Line_Break=nonstarter}', "");
Expect(0, 6101, '\P{^Is_Line_Break=nonstarter}', "");
Expect(1, 6102, '\p{Is_Line_Break=_	Nonstarter}', "");
Expect(0, 6102, '\p{^Is_Line_Break=_	Nonstarter}', "");
Expect(0, 6102, '\P{Is_Line_Break=_	Nonstarter}', "");
Expect(1, 6102, '\P{^Is_Line_Break=_	Nonstarter}', "");
Expect(0, 6101, '\p{Is_Line_Break=_	Nonstarter}', "");
Expect(1, 6101, '\p{^Is_Line_Break=_	Nonstarter}', "");
Expect(1, 6101, '\P{Is_Line_Break=_	Nonstarter}', "");
Expect(0, 6101, '\P{^Is_Line_Break=_	Nonstarter}', "");
Error('\p{Is_Lb=:=-	NS}');
Error('\P{Is_Lb=:=-	NS}');
Expect(1, 6102, '\p{Is_Lb=ns}', "");
Expect(0, 6102, '\p{^Is_Lb=ns}', "");
Expect(0, 6102, '\P{Is_Lb=ns}', "");
Expect(1, 6102, '\P{^Is_Lb=ns}', "");
Expect(0, 6101, '\p{Is_Lb=ns}', "");
Expect(1, 6101, '\p{^Is_Lb=ns}', "");
Expect(1, 6101, '\P{Is_Lb=ns}', "");
Expect(0, 6101, '\P{^Is_Lb=ns}', "");
Expect(1, 6102, '\p{Is_Lb=NS}', "");
Expect(0, 6102, '\p{^Is_Lb=NS}', "");
Expect(0, 6102, '\P{Is_Lb=NS}', "");
Expect(1, 6102, '\P{^Is_Lb=NS}', "");
Expect(0, 6101, '\p{Is_Lb=NS}', "");
Expect(1, 6101, '\p{^Is_Lb=NS}', "");
Expect(1, 6101, '\P{Is_Lb=NS}', "");
Expect(0, 6101, '\P{^Is_Lb=NS}', "");
Error('\p{Line_Break=	/a/JV}');
Error('\P{Line_Break=	/a/JV}');
Expect(1, 4519, '\p{Line_Break=jv}', "");
Expect(0, 4519, '\p{^Line_Break=jv}', "");
Expect(0, 4519, '\P{Line_Break=jv}', "");
Expect(1, 4519, '\P{^Line_Break=jv}', "");
Expect(0, 4447, '\p{Line_Break=jv}', "");
Expect(1, 4447, '\p{^Line_Break=jv}', "");
Expect(1, 4447, '\P{Line_Break=jv}', "");
Expect(0, 4447, '\P{^Line_Break=jv}', "");
Expect(1, 4519, '\p{Line_Break=  JV}', "");
Expect(0, 4519, '\p{^Line_Break=  JV}', "");
Expect(0, 4519, '\P{Line_Break=  JV}', "");
Expect(1, 4519, '\P{^Line_Break=  JV}', "");
Expect(0, 4447, '\p{Line_Break=  JV}', "");
Expect(1, 4447, '\p{^Line_Break=  JV}', "");
Expect(1, 4447, '\P{Line_Break=  JV}', "");
Expect(0, 4447, '\P{^Line_Break=  JV}', "");
Error('\p{Lb:	 JV/a/}');
Error('\P{Lb:	 JV/a/}');
Expect(1, 4519, '\p{Lb=jv}', "");
Expect(0, 4519, '\p{^Lb=jv}', "");
Expect(0, 4519, '\P{Lb=jv}', "");
Expect(1, 4519, '\P{^Lb=jv}', "");
Expect(0, 4447, '\p{Lb=jv}', "");
Expect(1, 4447, '\p{^Lb=jv}', "");
Expect(1, 4447, '\P{Lb=jv}', "");
Expect(0, 4447, '\P{^Lb=jv}', "");
Expect(1, 4519, '\p{Lb=-JV}', "");
Expect(0, 4519, '\p{^Lb=-JV}', "");
Expect(0, 4519, '\P{Lb=-JV}', "");
Expect(1, 4519, '\P{^Lb=-JV}', "");
Expect(0, 4447, '\p{Lb=-JV}', "");
Expect(1, 4447, '\p{^Lb=-JV}', "");
Expect(1, 4447, '\P{Lb=-JV}', "");
Expect(0, 4447, '\P{^Lb=-JV}', "");
Error('\p{Is_Line_Break=_/a/JV}');
Error('\P{Is_Line_Break=_/a/JV}');
Expect(1, 4519, '\p{Is_Line_Break=jv}', "");
Expect(0, 4519, '\p{^Is_Line_Break=jv}', "");
Expect(0, 4519, '\P{Is_Line_Break=jv}', "");
Expect(1, 4519, '\P{^Is_Line_Break=jv}', "");
Expect(0, 4447, '\p{Is_Line_Break=jv}', "");
Expect(1, 4447, '\p{^Is_Line_Break=jv}', "");
Expect(1, 4447, '\P{Is_Line_Break=jv}', "");
Expect(0, 4447, '\P{^Is_Line_Break=jv}', "");
Expect(1, 4519, '\p{Is_Line_Break=-JV}', "");
Expect(0, 4519, '\p{^Is_Line_Break=-JV}', "");
Expect(0, 4519, '\P{Is_Line_Break=-JV}', "");
Expect(1, 4519, '\P{^Is_Line_Break=-JV}', "");
Expect(0, 4447, '\p{Is_Line_Break=-JV}', "");
Expect(1, 4447, '\p{^Is_Line_Break=-JV}', "");
Expect(1, 4447, '\P{Is_Line_Break=-JV}', "");
Expect(0, 4447, '\P{^Is_Line_Break=-JV}', "");
Error('\p{Is_Lb=:=JV}');
Error('\P{Is_Lb=:=JV}');
Expect(1, 4519, '\p{Is_Lb=jv}', "");
Expect(0, 4519, '\p{^Is_Lb=jv}', "");
Expect(0, 4519, '\P{Is_Lb=jv}', "");
Expect(1, 4519, '\P{^Is_Lb=jv}', "");
Expect(0, 4447, '\p{Is_Lb=jv}', "");
Expect(1, 4447, '\p{^Is_Lb=jv}', "");
Expect(1, 4447, '\P{Is_Lb=jv}', "");
Expect(0, 4447, '\P{^Is_Lb=jv}', "");
Expect(1, 4519, '\p{Is_Lb=	JV}', "");
Expect(0, 4519, '\p{^Is_Lb=	JV}', "");
Expect(0, 4519, '\P{Is_Lb=	JV}', "");
Expect(1, 4519, '\P{^Is_Lb=	JV}', "");
Expect(0, 4447, '\p{Is_Lb=	JV}', "");
Expect(1, 4447, '\p{^Is_Lb=	JV}', "");
Expect(1, 4447, '\P{Is_Lb=	JV}', "");
Expect(0, 4447, '\P{^Is_Lb=	JV}', "");
Error('\p{Line_Break: _	QUOTATION/a/}');
Error('\P{Line_Break: _	QUOTATION/a/}');
Expect(1, 34, '\p{Line_Break=quotation}', "");
Expect(0, 34, '\p{^Line_Break=quotation}', "");
Expect(0, 34, '\P{Line_Break=quotation}', "");
Expect(1, 34, '\P{^Line_Break=quotation}', "");
Expect(0, 33, '\p{Line_Break=quotation}', "");
Expect(1, 33, '\p{^Line_Break=quotation}', "");
Expect(1, 33, '\P{Line_Break=quotation}', "");
Expect(0, 33, '\P{^Line_Break=quotation}', "");
Expect(1, 34, '\p{Line_Break= 	quotation}', "");
Expect(0, 34, '\p{^Line_Break= 	quotation}', "");
Expect(0, 34, '\P{Line_Break= 	quotation}', "");
Expect(1, 34, '\P{^Line_Break= 	quotation}', "");
Expect(0, 33, '\p{Line_Break= 	quotation}', "");
Expect(1, 33, '\p{^Line_Break= 	quotation}', "");
Expect(1, 33, '\P{Line_Break= 	quotation}', "");
Expect(0, 33, '\P{^Line_Break= 	quotation}', "");
Error('\p{Lb=	/a/qu}');
Error('\P{Lb=	/a/qu}');
Expect(1, 34, '\p{Lb=qu}', "");
Expect(0, 34, '\p{^Lb=qu}', "");
Expect(0, 34, '\P{Lb=qu}', "");
Expect(1, 34, '\P{^Lb=qu}', "");
Expect(0, 33, '\p{Lb=qu}', "");
Expect(1, 33, '\p{^Lb=qu}', "");
Expect(1, 33, '\P{Lb=qu}', "");
Expect(0, 33, '\P{^Lb=qu}', "");
Expect(1, 34, '\p{Lb=-	qu}', "");
Expect(0, 34, '\p{^Lb=-	qu}', "");
Expect(0, 34, '\P{Lb=-	qu}', "");
Expect(1, 34, '\P{^Lb=-	qu}', "");
Expect(0, 33, '\p{Lb=-	qu}', "");
Expect(1, 33, '\p{^Lb=-	qu}', "");
Expect(1, 33, '\P{Lb=-	qu}', "");
Expect(0, 33, '\P{^Lb=-	qu}', "");
Error('\p{Is_Line_Break: _quotation/a/}');
Error('\P{Is_Line_Break: _quotation/a/}');
Expect(1, 34, '\p{Is_Line_Break=quotation}', "");
Expect(0, 34, '\p{^Is_Line_Break=quotation}', "");
Expect(0, 34, '\P{Is_Line_Break=quotation}', "");
Expect(1, 34, '\P{^Is_Line_Break=quotation}', "");
Expect(0, 33, '\p{Is_Line_Break=quotation}', "");
Expect(1, 33, '\p{^Is_Line_Break=quotation}', "");
Expect(1, 33, '\P{Is_Line_Break=quotation}', "");
Expect(0, 33, '\P{^Is_Line_Break=quotation}', "");
Expect(1, 34, '\p{Is_Line_Break=QUOTATION}', "");
Expect(0, 34, '\p{^Is_Line_Break=QUOTATION}', "");
Expect(0, 34, '\P{Is_Line_Break=QUOTATION}', "");
Expect(1, 34, '\P{^Is_Line_Break=QUOTATION}', "");
Expect(0, 33, '\p{Is_Line_Break=QUOTATION}', "");
Expect(1, 33, '\p{^Is_Line_Break=QUOTATION}', "");
Expect(1, 33, '\P{Is_Line_Break=QUOTATION}', "");
Expect(0, 33, '\P{^Is_Line_Break=QUOTATION}', "");
Error('\p{Is_Lb=_-QU:=}');
Error('\P{Is_Lb=_-QU:=}');
Expect(1, 34, '\p{Is_Lb=qu}', "");
Expect(0, 34, '\p{^Is_Lb=qu}', "");
Expect(0, 34, '\P{Is_Lb=qu}', "");
Expect(1, 34, '\P{^Is_Lb=qu}', "");
Expect(0, 33, '\p{Is_Lb=qu}', "");
Expect(1, 33, '\p{^Is_Lb=qu}', "");
Expect(1, 33, '\P{Is_Lb=qu}', "");
Expect(0, 33, '\P{^Is_Lb=qu}', "");
Expect(1, 34, '\p{Is_Lb= _QU}', "");
Expect(0, 34, '\p{^Is_Lb= _QU}', "");
Expect(0, 34, '\P{Is_Lb= _QU}', "");
Expect(1, 34, '\P{^Is_Lb= _QU}', "");
Expect(0, 33, '\p{Is_Lb= _QU}', "");
Expect(1, 33, '\p{^Is_Lb= _QU}', "");
Expect(1, 33, '\P{Is_Lb= _QU}', "");
Expect(0, 33, '\P{^Is_Lb= _QU}', "");
Error('\p{Line_Break=:= -infix Numeric}');
Error('\P{Line_Break=:= -infix Numeric}');
Expect(1, 44, '\p{Line_Break=infixnumeric}', "");
Expect(0, 44, '\p{^Line_Break=infixnumeric}', "");
Expect(0, 44, '\P{Line_Break=infixnumeric}', "");
Expect(1, 44, '\P{^Line_Break=infixnumeric}', "");
Expect(0, 43, '\p{Line_Break=infixnumeric}', "");
Expect(1, 43, '\p{^Line_Break=infixnumeric}', "");
Expect(1, 43, '\P{Line_Break=infixnumeric}', "");
Expect(0, 43, '\P{^Line_Break=infixnumeric}', "");
Expect(1, 44, '\p{Line_Break=_-Infix_Numeric}', "");
Expect(0, 44, '\p{^Line_Break=_-Infix_Numeric}', "");
Expect(0, 44, '\P{Line_Break=_-Infix_Numeric}', "");
Expect(1, 44, '\P{^Line_Break=_-Infix_Numeric}', "");
Expect(0, 43, '\p{Line_Break=_-Infix_Numeric}', "");
Expect(1, 43, '\p{^Line_Break=_-Infix_Numeric}', "");
Expect(1, 43, '\P{Line_Break=_-Infix_Numeric}', "");
Expect(0, 43, '\P{^Line_Break=_-Infix_Numeric}', "");
Error('\p{Lb=_/a/IS}');
Error('\P{Lb=_/a/IS}');
Expect(1, 44, '\p{Lb=is}', "");
Expect(0, 44, '\p{^Lb=is}', "");
Expect(0, 44, '\P{Lb=is}', "");
Expect(1, 44, '\P{^Lb=is}', "");
Expect(0, 43, '\p{Lb=is}', "");
Expect(1, 43, '\p{^Lb=is}', "");
Expect(1, 43, '\P{Lb=is}', "");
Expect(0, 43, '\P{^Lb=is}', "");
Expect(1, 44, '\p{Lb=	_IS}', "");
Expect(0, 44, '\p{^Lb=	_IS}', "");
Expect(0, 44, '\P{Lb=	_IS}', "");
Expect(1, 44, '\P{^Lb=	_IS}', "");
Expect(0, 43, '\p{Lb=	_IS}', "");
Expect(1, 43, '\p{^Lb=	_IS}', "");
Expect(1, 43, '\P{Lb=	_IS}', "");
Expect(0, 43, '\P{^Lb=	_IS}', "");
Error('\p{Is_Line_Break=- infix/a/numeric}');
Error('\P{Is_Line_Break=- infix/a/numeric}');
Expect(1, 44, '\p{Is_Line_Break: infixnumeric}', "");
Expect(0, 44, '\p{^Is_Line_Break: infixnumeric}', "");
Expect(0, 44, '\P{Is_Line_Break: infixnumeric}', "");
Expect(1, 44, '\P{^Is_Line_Break: infixnumeric}', "");
Expect(0, 43, '\p{Is_Line_Break: infixnumeric}', "");
Expect(1, 43, '\p{^Is_Line_Break: infixnumeric}', "");
Expect(1, 43, '\P{Is_Line_Break: infixnumeric}', "");
Expect(0, 43, '\P{^Is_Line_Break: infixnumeric}', "");
Expect(1, 44, '\p{Is_Line_Break=_	INFIX_Numeric}', "");
Expect(0, 44, '\p{^Is_Line_Break=_	INFIX_Numeric}', "");
Expect(0, 44, '\P{Is_Line_Break=_	INFIX_Numeric}', "");
Expect(1, 44, '\P{^Is_Line_Break=_	INFIX_Numeric}', "");
Expect(0, 43, '\p{Is_Line_Break=_	INFIX_Numeric}', "");
Expect(1, 43, '\p{^Is_Line_Break=_	INFIX_Numeric}', "");
Expect(1, 43, '\P{Is_Line_Break=_	INFIX_Numeric}', "");
Expect(0, 43, '\P{^Is_Line_Break=_	INFIX_Numeric}', "");
Error('\p{Is_Lb=_ is:=}');
Error('\P{Is_Lb=_ is:=}');
Expect(1, 44, '\p{Is_Lb:is}', "");
Expect(0, 44, '\p{^Is_Lb:is}', "");
Expect(0, 44, '\P{Is_Lb:is}', "");
Expect(1, 44, '\P{^Is_Lb:is}', "");
Expect(0, 43, '\p{Is_Lb:is}', "");
Expect(1, 43, '\p{^Is_Lb:is}', "");
Expect(1, 43, '\P{Is_Lb:is}', "");
Expect(0, 43, '\P{^Is_Lb:is}', "");
Expect(1, 44, '\p{Is_Lb=	_IS}', "");
Expect(0, 44, '\p{^Is_Lb=	_IS}', "");
Expect(0, 44, '\P{Is_Lb=	_IS}', "");
Expect(1, 44, '\P{^Is_Lb=	_IS}', "");
Expect(0, 43, '\p{Is_Lb=	_IS}', "");
Expect(1, 43, '\p{^Is_Lb=	_IS}', "");
Expect(1, 43, '\P{Is_Lb=	_IS}', "");
Expect(0, 43, '\P{^Is_Lb=	_IS}', "");
Error('\p{Line_Break= _Inseparable:=}');
Error('\P{Line_Break= _Inseparable:=}');
Expect(1, 8230, '\p{Line_Break:   inseparable}', "");
Expect(0, 8230, '\p{^Line_Break:   inseparable}', "");
Expect(0, 8230, '\P{Line_Break:   inseparable}', "");
Expect(1, 8230, '\P{^Line_Break:   inseparable}', "");
Expect(0, 8227, '\p{Line_Break:   inseparable}', "");
Expect(1, 8227, '\p{^Line_Break:   inseparable}', "");
Expect(1, 8227, '\P{Line_Break:   inseparable}', "");
Expect(0, 8227, '\P{^Line_Break:   inseparable}', "");
Expect(1, 8230, '\p{Line_Break=- inseparable}', "");
Expect(0, 8230, '\p{^Line_Break=- inseparable}', "");
Expect(0, 8230, '\P{Line_Break=- inseparable}', "");
Expect(1, 8230, '\P{^Line_Break=- inseparable}', "");
Expect(0, 8227, '\p{Line_Break=- inseparable}', "");
Expect(1, 8227, '\p{^Line_Break=- inseparable}', "");
Expect(1, 8227, '\P{Line_Break=- inseparable}', "");
Expect(0, 8227, '\P{^Line_Break=- inseparable}', "");
Error('\p{Lb=:= IN}');
Error('\P{Lb=:= IN}');
Expect(1, 8230, '\p{Lb=in}', "");
Expect(0, 8230, '\p{^Lb=in}', "");
Expect(0, 8230, '\P{Lb=in}', "");
Expect(1, 8230, '\P{^Lb=in}', "");
Expect(0, 8227, '\p{Lb=in}', "");
Expect(1, 8227, '\p{^Lb=in}', "");
Expect(1, 8227, '\P{Lb=in}', "");
Expect(0, 8227, '\P{^Lb=in}', "");
Expect(1, 8230, '\p{Lb:   -IN}', "");
Expect(0, 8230, '\p{^Lb:   -IN}', "");
Expect(0, 8230, '\P{Lb:   -IN}', "");
Expect(1, 8230, '\P{^Lb:   -IN}', "");
Expect(0, 8227, '\p{Lb:   -IN}', "");
Expect(1, 8227, '\p{^Lb:   -IN}', "");
Expect(1, 8227, '\P{Lb:   -IN}', "");
Expect(0, 8227, '\P{^Lb:   -IN}', "");
Error('\p{Is_Line_Break=:=_Inseperable}');
Error('\P{Is_Line_Break=:=_Inseperable}');
Expect(1, 8230, '\p{Is_Line_Break=inseperable}', "");
Expect(0, 8230, '\p{^Is_Line_Break=inseperable}', "");
Expect(0, 8230, '\P{Is_Line_Break=inseperable}', "");
Expect(1, 8230, '\P{^Is_Line_Break=inseperable}', "");
Expect(0, 8227, '\p{Is_Line_Break=inseperable}', "");
Expect(1, 8227, '\p{^Is_Line_Break=inseperable}', "");
Expect(1, 8227, '\P{Is_Line_Break=inseperable}', "");
Expect(0, 8227, '\P{^Is_Line_Break=inseperable}', "");
Expect(1, 8230, '\p{Is_Line_Break=  Inseperable}', "");
Expect(0, 8230, '\p{^Is_Line_Break=  Inseperable}', "");
Expect(0, 8230, '\P{Is_Line_Break=  Inseperable}', "");
Expect(1, 8230, '\P{^Is_Line_Break=  Inseperable}', "");
Expect(0, 8227, '\p{Is_Line_Break=  Inseperable}', "");
Expect(1, 8227, '\p{^Is_Line_Break=  Inseperable}', "");
Expect(1, 8227, '\P{Is_Line_Break=  Inseperable}', "");
Expect(0, 8227, '\P{^Is_Line_Break=  Inseperable}', "");
Error('\p{Is_Lb= Inseparable/a/}');
Error('\P{Is_Lb= Inseparable/a/}');
Expect(1, 8230, '\p{Is_Lb:   inseparable}', "");
Expect(0, 8230, '\p{^Is_Lb:   inseparable}', "");
Expect(0, 8230, '\P{Is_Lb:   inseparable}', "");
Expect(1, 8230, '\P{^Is_Lb:   inseparable}', "");
Expect(0, 8227, '\p{Is_Lb:   inseparable}', "");
Expect(1, 8227, '\p{^Is_Lb:   inseparable}', "");
Expect(1, 8227, '\P{Is_Lb:   inseparable}', "");
Expect(0, 8227, '\P{^Is_Lb:   inseparable}', "");
Expect(1, 8230, '\p{Is_Lb=	 INSEPARABLE}', "");
Expect(0, 8230, '\p{^Is_Lb=	 INSEPARABLE}', "");
Expect(0, 8230, '\P{Is_Lb=	 INSEPARABLE}', "");
Expect(1, 8230, '\P{^Is_Lb=	 INSEPARABLE}', "");
Expect(0, 8227, '\p{Is_Lb=	 INSEPARABLE}', "");
Expect(1, 8227, '\p{^Is_Lb=	 INSEPARABLE}', "");
Expect(1, 8227, '\P{Is_Lb=	 INSEPARABLE}', "");
Expect(0, 8227, '\P{^Is_Lb=	 INSEPARABLE}', "");
Error('\p{Line_Break=-:=complex_CONTEXT}');
Error('\P{Line_Break=-:=complex_CONTEXT}');
Expect(1, 3642, '\p{Line_Break=complexcontext}', "");
Expect(0, 3642, '\p{^Line_Break=complexcontext}', "");
Expect(0, 3642, '\P{Line_Break=complexcontext}', "");
Expect(1, 3642, '\P{^Line_Break=complexcontext}', "");
Expect(0, 3584, '\p{Line_Break=complexcontext}', "");
Expect(1, 3584, '\p{^Line_Break=complexcontext}', "");
Expect(1, 3584, '\P{Line_Break=complexcontext}', "");
Expect(0, 3584, '\P{^Line_Break=complexcontext}', "");
Expect(1, 3642, '\p{Line_Break=-	ComplexCONTEXT}', "");
Expect(0, 3642, '\p{^Line_Break=-	ComplexCONTEXT}', "");
Expect(0, 3642, '\P{Line_Break=-	ComplexCONTEXT}', "");
Expect(1, 3642, '\P{^Line_Break=-	ComplexCONTEXT}', "");
Expect(0, 3584, '\p{Line_Break=-	ComplexCONTEXT}', "");
Expect(1, 3584, '\p{^Line_Break=-	ComplexCONTEXT}', "");
Expect(1, 3584, '\P{Line_Break=-	ComplexCONTEXT}', "");
Expect(0, 3584, '\P{^Line_Break=-	ComplexCONTEXT}', "");
Error('\p{Lb=	:=SA}');
Error('\P{Lb=	:=SA}');
Expect(1, 3642, '\p{Lb=sa}', "");
Expect(0, 3642, '\p{^Lb=sa}', "");
Expect(0, 3642, '\P{Lb=sa}', "");
Expect(1, 3642, '\P{^Lb=sa}', "");
Expect(0, 3584, '\p{Lb=sa}', "");
Expect(1, 3584, '\p{^Lb=sa}', "");
Expect(1, 3584, '\P{Lb=sa}', "");
Expect(0, 3584, '\P{^Lb=sa}', "");
Expect(1, 3642, '\p{Lb=-SA}', "");
Expect(0, 3642, '\p{^Lb=-SA}', "");
Expect(0, 3642, '\P{Lb=-SA}', "");
Expect(1, 3642, '\P{^Lb=-SA}', "");
Expect(0, 3584, '\p{Lb=-SA}', "");
Expect(1, 3584, '\p{^Lb=-SA}', "");
Expect(1, 3584, '\P{Lb=-SA}', "");
Expect(0, 3584, '\P{^Lb=-SA}', "");
Error('\p{Is_Line_Break=__ComplexContext/a/}');
Error('\P{Is_Line_Break=__ComplexContext/a/}');
Expect(1, 3642, '\p{Is_Line_Break=complexcontext}', "");
Expect(0, 3642, '\p{^Is_Line_Break=complexcontext}', "");
Expect(0, 3642, '\P{Is_Line_Break=complexcontext}', "");
Expect(1, 3642, '\P{^Is_Line_Break=complexcontext}', "");
Expect(0, 3584, '\p{Is_Line_Break=complexcontext}', "");
Expect(1, 3584, '\p{^Is_Line_Break=complexcontext}', "");
Expect(1, 3584, '\P{Is_Line_Break=complexcontext}', "");
Expect(0, 3584, '\P{^Is_Line_Break=complexcontext}', "");
Expect(1, 3642, '\p{Is_Line_Break=-_ComplexCONTEXT}', "");
Expect(0, 3642, '\p{^Is_Line_Break=-_ComplexCONTEXT}', "");
Expect(0, 3642, '\P{Is_Line_Break=-_ComplexCONTEXT}', "");
Expect(1, 3642, '\P{^Is_Line_Break=-_ComplexCONTEXT}', "");
Expect(0, 3584, '\p{Is_Line_Break=-_ComplexCONTEXT}', "");
Expect(1, 3584, '\p{^Is_Line_Break=-_ComplexCONTEXT}', "");
Expect(1, 3584, '\P{Is_Line_Break=-_ComplexCONTEXT}', "");
Expect(0, 3584, '\P{^Is_Line_Break=-_ComplexCONTEXT}', "");
Error('\p{Is_Lb=- SA:=}');
Error('\P{Is_Lb=- SA:=}');
Expect(1, 3642, '\p{Is_Lb=sa}', "");
Expect(0, 3642, '\p{^Is_Lb=sa}', "");
Expect(0, 3642, '\P{Is_Lb=sa}', "");
Expect(1, 3642, '\P{^Is_Lb=sa}', "");
Expect(0, 3584, '\p{Is_Lb=sa}', "");
Expect(1, 3584, '\p{^Is_Lb=sa}', "");
Expect(1, 3584, '\P{Is_Lb=sa}', "");
Expect(0, 3584, '\P{^Is_Lb=sa}', "");
Expect(1, 3642, '\p{Is_Lb=__SA}', "");
Expect(0, 3642, '\p{^Is_Lb=__SA}', "");
Expect(0, 3642, '\P{Is_Lb=__SA}', "");
Expect(1, 3642, '\P{^Is_Lb=__SA}', "");
Expect(0, 3584, '\p{Is_Lb=__SA}', "");
Expect(1, 3584, '\p{^Is_Lb=__SA}', "");
Expect(1, 3584, '\P{Is_Lb=__SA}', "");
Expect(0, 3584, '\P{^Is_Lb=__SA}', "");
Error('\p{Line_Break=-/a/hyphen}');
Error('\P{Line_Break=-/a/hyphen}');
Expect(1, 45, '\p{Line_Break:   hyphen}', "");
Expect(0, 45, '\p{^Line_Break:   hyphen}', "");
Expect(0, 45, '\P{Line_Break:   hyphen}', "");
Expect(1, 45, '\P{^Line_Break:   hyphen}', "");
Expect(0, 44, '\p{Line_Break:   hyphen}', "");
Expect(1, 44, '\p{^Line_Break:   hyphen}', "");
Expect(1, 44, '\P{Line_Break:   hyphen}', "");
Expect(0, 44, '\P{^Line_Break:   hyphen}', "");
Expect(1, 45, '\p{Line_Break:   	 HYPHEN}', "");
Expect(0, 45, '\p{^Line_Break:   	 HYPHEN}', "");
Expect(0, 45, '\P{Line_Break:   	 HYPHEN}', "");
Expect(1, 45, '\P{^Line_Break:   	 HYPHEN}', "");
Expect(0, 44, '\p{Line_Break:   	 HYPHEN}', "");
Expect(1, 44, '\p{^Line_Break:   	 HYPHEN}', "");
Expect(1, 44, '\P{Line_Break:   	 HYPHEN}', "");
Expect(0, 44, '\P{^Line_Break:   	 HYPHEN}', "");
Error('\p{Lb=:= 	HY}');
Error('\P{Lb=:= 	HY}');
Expect(1, 45, '\p{Lb=hy}', "");
Expect(0, 45, '\p{^Lb=hy}', "");
Expect(0, 45, '\P{Lb=hy}', "");
Expect(1, 45, '\P{^Lb=hy}', "");
Expect(0, 44, '\p{Lb=hy}', "");
Expect(1, 44, '\p{^Lb=hy}', "");
Expect(1, 44, '\P{Lb=hy}', "");
Expect(0, 44, '\P{^Lb=hy}', "");
Expect(1, 45, '\p{Lb:	_HY}', "");
Expect(0, 45, '\p{^Lb:	_HY}', "");
Expect(0, 45, '\P{Lb:	_HY}', "");
Expect(1, 45, '\P{^Lb:	_HY}', "");
Expect(0, 44, '\p{Lb:	_HY}', "");
Expect(1, 44, '\p{^Lb:	_HY}', "");
Expect(1, 44, '\P{Lb:	_HY}', "");
Expect(0, 44, '\P{^Lb:	_HY}', "");
Error('\p{Is_Line_Break=__HYPHEN:=}');
Error('\P{Is_Line_Break=__HYPHEN:=}');
Expect(1, 45, '\p{Is_Line_Break=hyphen}', "");
Expect(0, 45, '\p{^Is_Line_Break=hyphen}', "");
Expect(0, 45, '\P{Is_Line_Break=hyphen}', "");
Expect(1, 45, '\P{^Is_Line_Break=hyphen}', "");
Expect(0, 44, '\p{Is_Line_Break=hyphen}', "");
Expect(1, 44, '\p{^Is_Line_Break=hyphen}', "");
Expect(1, 44, '\P{Is_Line_Break=hyphen}', "");
Expect(0, 44, '\P{^Is_Line_Break=hyphen}', "");
Expect(1, 45, '\p{Is_Line_Break=_-Hyphen}', "");
Expect(0, 45, '\p{^Is_Line_Break=_-Hyphen}', "");
Expect(0, 45, '\P{Is_Line_Break=_-Hyphen}', "");
Expect(1, 45, '\P{^Is_Line_Break=_-Hyphen}', "");
Expect(0, 44, '\p{Is_Line_Break=_-Hyphen}', "");
Expect(1, 44, '\p{^Is_Line_Break=_-Hyphen}', "");
Expect(1, 44, '\P{Is_Line_Break=_-Hyphen}', "");
Expect(0, 44, '\P{^Is_Line_Break=_-Hyphen}', "");
Error('\p{Is_Lb=-/a/HY}');
Error('\P{Is_Lb=-/a/HY}');
Expect(1, 45, '\p{Is_Lb:hy}', "");
Expect(0, 45, '\p{^Is_Lb:hy}', "");
Expect(0, 45, '\P{Is_Lb:hy}', "");
Expect(1, 45, '\P{^Is_Lb:hy}', "");
Expect(0, 44, '\p{Is_Lb:hy}', "");
Expect(1, 44, '\p{^Is_Lb:hy}', "");
Expect(1, 44, '\P{Is_Lb:hy}', "");
Expect(0, 44, '\P{^Is_Lb:hy}', "");
Expect(1, 45, '\p{Is_Lb=__HY}', "");
Expect(0, 45, '\p{^Is_Lb=__HY}', "");
Expect(0, 45, '\P{Is_Lb=__HY}', "");
Expect(1, 45, '\P{^Is_Lb=__HY}', "");
Expect(0, 44, '\p{Is_Lb=__HY}', "");
Expect(1, 44, '\p{^Is_Lb=__HY}', "");
Expect(1, 44, '\P{Is_Lb=__HY}', "");
Expect(0, 44, '\P{^Is_Lb=__HY}', "");
Error('\p{Line_Break=_-Break/a/after}');
Error('\P{Line_Break=_-Break/a/after}');
Expect(1, 9, '\p{Line_Break:   breakafter}', "");
Expect(0, 9, '\p{^Line_Break:   breakafter}', "");
Expect(0, 9, '\P{Line_Break:   breakafter}', "");
Expect(1, 9, '\P{^Line_Break:   breakafter}', "");
Expect(0, 8, '\p{Line_Break:   breakafter}', "");
Expect(1, 8, '\p{^Line_Break:   breakafter}', "");
Expect(1, 8, '\P{Line_Break:   breakafter}', "");
Expect(0, 8, '\P{^Line_Break:   breakafter}', "");
Expect(1, 9, '\p{Line_Break: -_BREAK After}', "");
Expect(0, 9, '\p{^Line_Break: -_BREAK After}', "");
Expect(0, 9, '\P{Line_Break: -_BREAK After}', "");
Expect(1, 9, '\P{^Line_Break: -_BREAK After}', "");
Expect(0, 8, '\p{Line_Break: -_BREAK After}', "");
Expect(1, 8, '\p{^Line_Break: -_BREAK After}', "");
Expect(1, 8, '\P{Line_Break: -_BREAK After}', "");
Expect(0, 8, '\P{^Line_Break: -_BREAK After}', "");
Error('\p{Lb=	/a/BA}');
Error('\P{Lb=	/a/BA}');
Expect(1, 9, '\p{Lb:   ba}', "");
Expect(0, 9, '\p{^Lb:   ba}', "");
Expect(0, 9, '\P{Lb:   ba}', "");
Expect(1, 9, '\P{^Lb:   ba}', "");
Expect(0, 8, '\p{Lb:   ba}', "");
Expect(1, 8, '\p{^Lb:   ba}', "");
Expect(1, 8, '\P{Lb:   ba}', "");
Expect(0, 8, '\P{^Lb:   ba}', "");
Expect(1, 9, '\p{Lb:    	BA}', "");
Expect(0, 9, '\p{^Lb:    	BA}', "");
Expect(0, 9, '\P{Lb:    	BA}', "");
Expect(1, 9, '\P{^Lb:    	BA}', "");
Expect(0, 8, '\p{Lb:    	BA}', "");
Expect(1, 8, '\p{^Lb:    	BA}', "");
Expect(1, 8, '\P{Lb:    	BA}', "");
Expect(0, 8, '\P{^Lb:    	BA}', "");
Error('\p{Is_Line_Break:	-_BreakAFTER:=}');
Error('\P{Is_Line_Break:	-_BreakAFTER:=}');
Expect(1, 9, '\p{Is_Line_Break=breakafter}', "");
Expect(0, 9, '\p{^Is_Line_Break=breakafter}', "");
Expect(0, 9, '\P{Is_Line_Break=breakafter}', "");
Expect(1, 9, '\P{^Is_Line_Break=breakafter}', "");
Expect(0, 8, '\p{Is_Line_Break=breakafter}', "");
Expect(1, 8, '\p{^Is_Line_Break=breakafter}', "");
Expect(1, 8, '\P{Is_Line_Break=breakafter}', "");
Expect(0, 8, '\P{^Is_Line_Break=breakafter}', "");
Expect(1, 9, '\p{Is_Line_Break=__Break-After}', "");
Expect(0, 9, '\p{^Is_Line_Break=__Break-After}', "");
Expect(0, 9, '\P{Is_Line_Break=__Break-After}', "");
Expect(1, 9, '\P{^Is_Line_Break=__Break-After}', "");
Expect(0, 8, '\p{Is_Line_Break=__Break-After}', "");
Expect(1, 8, '\p{^Is_Line_Break=__Break-After}', "");
Expect(1, 8, '\P{Is_Line_Break=__Break-After}', "");
Expect(0, 8, '\P{^Is_Line_Break=__Break-After}', "");
Error('\p{Is_Lb=:= -ba}');
Error('\P{Is_Lb=:= -ba}');
Expect(1, 9, '\p{Is_Lb=ba}', "");
Expect(0, 9, '\p{^Is_Lb=ba}', "");
Expect(0, 9, '\P{Is_Lb=ba}', "");
Expect(1, 9, '\P{^Is_Lb=ba}', "");
Expect(0, 8, '\p{Is_Lb=ba}', "");
Expect(1, 8, '\p{^Is_Lb=ba}', "");
Expect(1, 8, '\P{Is_Lb=ba}', "");
Expect(0, 8, '\P{^Is_Lb=ba}', "");
Expect(1, 9, '\p{Is_Lb=-_BA}', "");
Expect(0, 9, '\p{^Is_Lb=-_BA}', "");
Expect(0, 9, '\P{Is_Lb=-_BA}', "");
Expect(1, 9, '\P{^Is_Lb=-_BA}', "");
Expect(0, 8, '\p{Is_Lb=-_BA}', "");
Expect(1, 8, '\p{^Is_Lb=-_BA}', "");
Expect(1, 8, '\P{Is_Lb=-_BA}', "");
Expect(0, 8, '\P{^Is_Lb=-_BA}', "");
Error('\p{Line_Break=-LINE_feed:=}');
Error('\P{Line_Break=-LINE_feed:=}');
Expect(1, 10, '\p{Line_Break=linefeed}', "");
Expect(0, 10, '\p{^Line_Break=linefeed}', "");
Expect(0, 10, '\P{Line_Break=linefeed}', "");
Expect(1, 10, '\P{^Line_Break=linefeed}', "");
Expect(0, 9, '\p{Line_Break=linefeed}', "");
Expect(1, 9, '\p{^Line_Break=linefeed}', "");
Expect(1, 9, '\P{Line_Break=linefeed}', "");
Expect(0, 9, '\P{^Line_Break=linefeed}', "");
Expect(1, 10, '\p{Line_Break:	_-lineFeed}', "");
Expect(0, 10, '\p{^Line_Break:	_-lineFeed}', "");
Expect(0, 10, '\P{Line_Break:	_-lineFeed}', "");
Expect(1, 10, '\P{^Line_Break:	_-lineFeed}', "");
Expect(0, 9, '\p{Line_Break:	_-lineFeed}', "");
Expect(1, 9, '\p{^Line_Break:	_-lineFeed}', "");
Expect(1, 9, '\P{Line_Break:	_-lineFeed}', "");
Expect(0, 9, '\P{^Line_Break:	_-lineFeed}', "");
Error('\p{Lb=-LF/a/}');
Error('\P{Lb=-LF/a/}');
Expect(1, 10, '\p{Lb=lf}', "");
Expect(0, 10, '\p{^Lb=lf}', "");
Expect(0, 10, '\P{Lb=lf}', "");
Expect(1, 10, '\P{^Lb=lf}', "");
Expect(0, 9, '\p{Lb=lf}', "");
Expect(1, 9, '\p{^Lb=lf}', "");
Expect(1, 9, '\P{Lb=lf}', "");
Expect(0, 9, '\P{^Lb=lf}', "");
Expect(1, 10, '\p{Lb= -LF}', "");
Expect(0, 10, '\p{^Lb= -LF}', "");
Expect(0, 10, '\P{Lb= -LF}', "");
Expect(1, 10, '\P{^Lb= -LF}', "");
Expect(0, 9, '\p{Lb= -LF}', "");
Expect(1, 9, '\p{^Lb= -LF}', "");
Expect(1, 9, '\P{Lb= -LF}', "");
Expect(0, 9, '\P{^Lb= -LF}', "");
Error('\p{Is_Line_Break=/a/	-Line	Feed}');
Error('\P{Is_Line_Break=/a/	-Line	Feed}');
Expect(1, 10, '\p{Is_Line_Break=linefeed}', "");
Expect(0, 10, '\p{^Is_Line_Break=linefeed}', "");
Expect(0, 10, '\P{Is_Line_Break=linefeed}', "");
Expect(1, 10, '\P{^Is_Line_Break=linefeed}', "");
Expect(0, 9, '\p{Is_Line_Break=linefeed}', "");
Expect(1, 9, '\p{^Is_Line_Break=linefeed}', "");
Expect(1, 9, '\P{Is_Line_Break=linefeed}', "");
Expect(0, 9, '\P{^Is_Line_Break=linefeed}', "");
Expect(1, 10, '\p{Is_Line_Break= lineFeed}', "");
Expect(0, 10, '\p{^Is_Line_Break= lineFeed}', "");
Expect(0, 10, '\P{Is_Line_Break= lineFeed}', "");
Expect(1, 10, '\P{^Is_Line_Break= lineFeed}', "");
Expect(0, 9, '\p{Is_Line_Break= lineFeed}', "");
Expect(1, 9, '\p{^Is_Line_Break= lineFeed}', "");
Expect(1, 9, '\P{Is_Line_Break= lineFeed}', "");
Expect(0, 9, '\P{^Is_Line_Break= lineFeed}', "");
Error('\p{Is_Lb= :=LF}');
Error('\P{Is_Lb= :=LF}');
Expect(1, 10, '\p{Is_Lb=lf}', "");
Expect(0, 10, '\p{^Is_Lb=lf}', "");
Expect(0, 10, '\P{Is_Lb=lf}', "");
Expect(1, 10, '\P{^Is_Lb=lf}', "");
Expect(0, 9, '\p{Is_Lb=lf}', "");
Expect(1, 9, '\p{^Is_Lb=lf}', "");
Expect(1, 9, '\P{Is_Lb=lf}', "");
Expect(0, 9, '\P{^Is_Lb=lf}', "");
Expect(1, 10, '\p{Is_Lb= _LF}', "");
Expect(0, 10, '\p{^Is_Lb= _LF}', "");
Expect(0, 10, '\P{Is_Lb= _LF}', "");
Expect(1, 10, '\P{^Is_Lb= _LF}', "");
Expect(0, 9, '\p{Is_Lb= _LF}', "");
Expect(1, 9, '\p{^Is_Lb= _LF}', "");
Expect(1, 9, '\P{Is_Lb= _LF}', "");
Expect(0, 9, '\P{^Is_Lb= _LF}', "");
Error('\p{Line_Break:	 AMBIGUOUS:=}');
Error('\P{Line_Break:	 AMBIGUOUS:=}');
Expect(1, 168, '\p{Line_Break=ambiguous}', "");
Expect(0, 168, '\p{^Line_Break=ambiguous}', "");
Expect(0, 168, '\P{Line_Break=ambiguous}', "");
Expect(1, 168, '\P{^Line_Break=ambiguous}', "");
Expect(0, 166, '\p{Line_Break=ambiguous}', "");
Expect(1, 166, '\p{^Line_Break=ambiguous}', "");
Expect(1, 166, '\P{Line_Break=ambiguous}', "");
Expect(0, 166, '\P{^Line_Break=ambiguous}', "");
Expect(1, 168, '\p{Line_Break=--Ambiguous}', "");
Expect(0, 168, '\p{^Line_Break=--Ambiguous}', "");
Expect(0, 168, '\P{Line_Break=--Ambiguous}', "");
Expect(1, 168, '\P{^Line_Break=--Ambiguous}', "");
Expect(0, 166, '\p{Line_Break=--Ambiguous}', "");
Expect(1, 166, '\p{^Line_Break=--Ambiguous}', "");
Expect(1, 166, '\P{Line_Break=--Ambiguous}', "");
Expect(0, 166, '\P{^Line_Break=--Ambiguous}', "");
Error('\p{Lb=/a/_AI}');
Error('\P{Lb=/a/_AI}');
Expect(1, 168, '\p{Lb=ai}', "");
Expect(0, 168, '\p{^Lb=ai}', "");
Expect(0, 168, '\P{Lb=ai}', "");
Expect(1, 168, '\P{^Lb=ai}', "");
Expect(0, 166, '\p{Lb=ai}', "");
Expect(1, 166, '\p{^Lb=ai}', "");
Expect(1, 166, '\P{Lb=ai}', "");
Expect(0, 166, '\P{^Lb=ai}', "");
Expect(1, 168, '\p{Lb=		AI}', "");
Expect(0, 168, '\p{^Lb=		AI}', "");
Expect(0, 168, '\P{Lb=		AI}', "");
Expect(1, 168, '\P{^Lb=		AI}', "");
Expect(0, 166, '\p{Lb=		AI}', "");
Expect(1, 166, '\p{^Lb=		AI}', "");
Expect(1, 166, '\P{Lb=		AI}', "");
Expect(0, 166, '\P{^Lb=		AI}', "");
Error('\p{Is_Line_Break=_:=Ambiguous}');
Error('\P{Is_Line_Break=_:=Ambiguous}');
Expect(1, 168, '\p{Is_Line_Break=ambiguous}', "");
Expect(0, 168, '\p{^Is_Line_Break=ambiguous}', "");
Expect(0, 168, '\P{Is_Line_Break=ambiguous}', "");
Expect(1, 168, '\P{^Is_Line_Break=ambiguous}', "");
Expect(0, 166, '\p{Is_Line_Break=ambiguous}', "");
Expect(1, 166, '\p{^Is_Line_Break=ambiguous}', "");
Expect(1, 166, '\P{Is_Line_Break=ambiguous}', "");
Expect(0, 166, '\P{^Is_Line_Break=ambiguous}', "");
Expect(1, 168, '\p{Is_Line_Break=__AMBIGUOUS}', "");
Expect(0, 168, '\p{^Is_Line_Break=__AMBIGUOUS}', "");
Expect(0, 168, '\P{Is_Line_Break=__AMBIGUOUS}', "");
Expect(1, 168, '\P{^Is_Line_Break=__AMBIGUOUS}', "");
Expect(0, 166, '\p{Is_Line_Break=__AMBIGUOUS}', "");
Expect(1, 166, '\p{^Is_Line_Break=__AMBIGUOUS}', "");
Expect(1, 166, '\P{Is_Line_Break=__AMBIGUOUS}', "");
Expect(0, 166, '\P{^Is_Line_Break=__AMBIGUOUS}', "");
Error('\p{Is_Lb: _ ai/a/}');
Error('\P{Is_Lb: _ ai/a/}');
Expect(1, 168, '\p{Is_Lb=ai}', "");
Expect(0, 168, '\p{^Is_Lb=ai}', "");
Expect(0, 168, '\P{Is_Lb=ai}', "");
Expect(1, 168, '\P{^Is_Lb=ai}', "");
Expect(0, 166, '\p{Is_Lb=ai}', "");
Expect(1, 166, '\p{^Is_Lb=ai}', "");
Expect(1, 166, '\P{Is_Lb=ai}', "");
Expect(0, 166, '\P{^Is_Lb=ai}', "");
Expect(1, 168, '\p{Is_Lb=  ai}', "");
Expect(0, 168, '\p{^Is_Lb=  ai}', "");
Expect(0, 168, '\P{Is_Lb=  ai}', "");
Expect(1, 168, '\P{^Is_Lb=  ai}', "");
Expect(0, 166, '\p{Is_Lb=  ai}', "");
Expect(1, 166, '\p{^Is_Lb=  ai}', "");
Expect(1, 166, '\P{Is_Lb=  ai}', "");
Expect(0, 166, '\P{^Is_Lb=  ai}', "");
Error('\p{Line_Break=/a/contingent_Break}');
Error('\P{Line_Break=/a/contingent_Break}');
Expect(1, 65532, '\p{Line_Break=contingentbreak}', "");
Expect(0, 65532, '\p{^Line_Break=contingentbreak}', "");
Expect(0, 65532, '\P{Line_Break=contingentbreak}', "");
Expect(1, 65532, '\P{^Line_Break=contingentbreak}', "");
Expect(0, 65531, '\p{Line_Break=contingentbreak}', "");
Expect(1, 65531, '\p{^Line_Break=contingentbreak}', "");
Expect(1, 65531, '\P{Line_Break=contingentbreak}', "");
Expect(0, 65531, '\P{^Line_Break=contingentbreak}', "");
Expect(1, 65532, '\p{Line_Break=-Contingent	Break}', "");
Expect(0, 65532, '\p{^Line_Break=-Contingent	Break}', "");
Expect(0, 65532, '\P{Line_Break=-Contingent	Break}', "");
Expect(1, 65532, '\P{^Line_Break=-Contingent	Break}', "");
Expect(0, 65531, '\p{Line_Break=-Contingent	Break}', "");
Expect(1, 65531, '\p{^Line_Break=-Contingent	Break}', "");
Expect(1, 65531, '\P{Line_Break=-Contingent	Break}', "");
Expect(0, 65531, '\P{^Line_Break=-Contingent	Break}', "");
Error('\p{Lb= 	CB/a/}');
Error('\P{Lb= 	CB/a/}');
Expect(1, 65532, '\p{Lb=cb}', "");
Expect(0, 65532, '\p{^Lb=cb}', "");
Expect(0, 65532, '\P{Lb=cb}', "");
Expect(1, 65532, '\P{^Lb=cb}', "");
Expect(0, 65531, '\p{Lb=cb}', "");
Expect(1, 65531, '\p{^Lb=cb}', "");
Expect(1, 65531, '\P{Lb=cb}', "");
Expect(0, 65531, '\P{^Lb=cb}', "");
Expect(1, 65532, '\p{Lb=	 CB}', "");
Expect(0, 65532, '\p{^Lb=	 CB}', "");
Expect(0, 65532, '\P{Lb=	 CB}', "");
Expect(1, 65532, '\P{^Lb=	 CB}', "");
Expect(0, 65531, '\p{Lb=	 CB}', "");
Expect(1, 65531, '\p{^Lb=	 CB}', "");
Expect(1, 65531, '\P{Lb=	 CB}', "");
Expect(0, 65531, '\P{^Lb=	 CB}', "");
Error('\p{Is_Line_Break=- Contingent:=Break}');
Error('\P{Is_Line_Break=- Contingent:=Break}');
Expect(1, 65532, '\p{Is_Line_Break=contingentbreak}', "");
Expect(0, 65532, '\p{^Is_Line_Break=contingentbreak}', "");
Expect(0, 65532, '\P{Is_Line_Break=contingentbreak}', "");
Expect(1, 65532, '\P{^Is_Line_Break=contingentbreak}', "");
Expect(0, 65531, '\p{Is_Line_Break=contingentbreak}', "");
Expect(1, 65531, '\p{^Is_Line_Break=contingentbreak}', "");
Expect(1, 65531, '\P{Is_Line_Break=contingentbreak}', "");
Expect(0, 65531, '\P{^Is_Line_Break=contingentbreak}', "");
Expect(1, 65532, '\p{Is_Line_Break=	_Contingent	break}', "");
Expect(0, 65532, '\p{^Is_Line_Break=	_Contingent	break}', "");
Expect(0, 65532, '\P{Is_Line_Break=	_Contingent	break}', "");
Expect(1, 65532, '\P{^Is_Line_Break=	_Contingent	break}', "");
Expect(0, 65531, '\p{Is_Line_Break=	_Contingent	break}', "");
Expect(1, 65531, '\p{^Is_Line_Break=	_Contingent	break}', "");
Expect(1, 65531, '\P{Is_Line_Break=	_Contingent	break}', "");
Expect(0, 65531, '\P{^Is_Line_Break=	_Contingent	break}', "");
Error('\p{Is_Lb=:=	-cb}');
Error('\P{Is_Lb=:=	-cb}');
Expect(1, 65532, '\p{Is_Lb=cb}', "");
Expect(0, 65532, '\p{^Is_Lb=cb}', "");
Expect(0, 65532, '\P{Is_Lb=cb}', "");
Expect(1, 65532, '\P{^Is_Lb=cb}', "");
Expect(0, 65531, '\p{Is_Lb=cb}', "");
Expect(1, 65531, '\p{^Is_Lb=cb}', "");
Expect(1, 65531, '\P{Is_Lb=cb}', "");
Expect(0, 65531, '\P{^Is_Lb=cb}', "");
Expect(1, 65532, '\p{Is_Lb=-	CB}', "");
Expect(0, 65532, '\p{^Is_Lb=-	CB}', "");
Expect(0, 65532, '\P{Is_Lb=-	CB}', "");
Expect(1, 65532, '\P{^Is_Lb=-	CB}', "");
Expect(0, 65531, '\p{Is_Lb=-	CB}', "");
Expect(1, 65531, '\p{^Is_Lb=-	CB}', "");
Expect(1, 65531, '\P{Is_Lb=-	CB}', "");
Expect(0, 65531, '\P{^Is_Lb=-	CB}', "");
Error('\p{Line_Break:   -/a/Exclamation}');
Error('\P{Line_Break:   -/a/Exclamation}');
Expect(1, 33, '\p{Line_Break=exclamation}', "");
Expect(0, 33, '\p{^Line_Break=exclamation}', "");
Expect(0, 33, '\P{Line_Break=exclamation}', "");
Expect(1, 33, '\P{^Line_Break=exclamation}', "");
Expect(0, 32, '\p{Line_Break=exclamation}', "");
Expect(1, 32, '\p{^Line_Break=exclamation}', "");
Expect(1, 32, '\P{Line_Break=exclamation}', "");
Expect(0, 32, '\P{^Line_Break=exclamation}', "");
Expect(1, 33, '\p{Line_Break= -EXCLAMATION}', "");
Expect(0, 33, '\p{^Line_Break= -EXCLAMATION}', "");
Expect(0, 33, '\P{Line_Break= -EXCLAMATION}', "");
Expect(1, 33, '\P{^Line_Break= -EXCLAMATION}', "");
Expect(0, 32, '\p{Line_Break= -EXCLAMATION}', "");
Expect(1, 32, '\p{^Line_Break= -EXCLAMATION}', "");
Expect(1, 32, '\P{Line_Break= -EXCLAMATION}', "");
Expect(0, 32, '\P{^Line_Break= -EXCLAMATION}', "");
Error('\p{Lb=/a/	EX}');
Error('\P{Lb=/a/	EX}');
Expect(1, 33, '\p{Lb=ex}', "");
Expect(0, 33, '\p{^Lb=ex}', "");
Expect(0, 33, '\P{Lb=ex}', "");
Expect(1, 33, '\P{^Lb=ex}', "");
Expect(0, 32, '\p{Lb=ex}', "");
Expect(1, 32, '\p{^Lb=ex}', "");
Expect(1, 32, '\P{Lb=ex}', "");
Expect(0, 32, '\P{^Lb=ex}', "");
Expect(1, 33, '\p{Lb=__EX}', "");
Expect(0, 33, '\p{^Lb=__EX}', "");
Expect(0, 33, '\P{Lb=__EX}', "");
Expect(1, 33, '\P{^Lb=__EX}', "");
Expect(0, 32, '\p{Lb=__EX}', "");
Expect(1, 32, '\p{^Lb=__EX}', "");
Expect(1, 32, '\P{Lb=__EX}', "");
Expect(0, 32, '\P{^Lb=__EX}', "");
Error('\p{Is_Line_Break= /a/Exclamation}');
Error('\P{Is_Line_Break= /a/Exclamation}');
Expect(1, 33, '\p{Is_Line_Break:exclamation}', "");
Expect(0, 33, '\p{^Is_Line_Break:exclamation}', "");
Expect(0, 33, '\P{Is_Line_Break:exclamation}', "");
Expect(1, 33, '\P{^Is_Line_Break:exclamation}', "");
Expect(0, 32, '\p{Is_Line_Break:exclamation}', "");
Expect(1, 32, '\p{^Is_Line_Break:exclamation}', "");
Expect(1, 32, '\P{Is_Line_Break:exclamation}', "");
Expect(0, 32, '\P{^Is_Line_Break:exclamation}', "");
Expect(1, 33, '\p{Is_Line_Break=-EXCLAMATION}', "");
Expect(0, 33, '\p{^Is_Line_Break=-EXCLAMATION}', "");
Expect(0, 33, '\P{Is_Line_Break=-EXCLAMATION}', "");
Expect(1, 33, '\P{^Is_Line_Break=-EXCLAMATION}', "");
Expect(0, 32, '\p{Is_Line_Break=-EXCLAMATION}', "");
Expect(1, 32, '\p{^Is_Line_Break=-EXCLAMATION}', "");
Expect(1, 32, '\P{Is_Line_Break=-EXCLAMATION}', "");
Expect(0, 32, '\P{^Is_Line_Break=-EXCLAMATION}', "");
Error('\p{Is_Lb=		ex/a/}');
Error('\P{Is_Lb=		ex/a/}');
Expect(1, 33, '\p{Is_Lb=ex}', "");
Expect(0, 33, '\p{^Is_Lb=ex}', "");
Expect(0, 33, '\P{Is_Lb=ex}', "");
Expect(1, 33, '\P{^Is_Lb=ex}', "");
Expect(0, 32, '\p{Is_Lb=ex}', "");
Expect(1, 32, '\p{^Is_Lb=ex}', "");
Expect(1, 32, '\P{Is_Lb=ex}', "");
Expect(0, 32, '\P{^Is_Lb=ex}', "");
Expect(1, 33, '\p{Is_Lb=_EX}', "");
Expect(0, 33, '\p{^Is_Lb=_EX}', "");
Expect(0, 33, '\P{Is_Lb=_EX}', "");
Expect(1, 33, '\P{^Is_Lb=_EX}', "");
Expect(0, 32, '\p{Is_Lb=_EX}', "");
Expect(1, 32, '\p{^Is_Lb=_EX}', "");
Expect(1, 32, '\P{Is_Lb=_EX}', "");
Expect(0, 32, '\P{^Is_Lb=_EX}', "");
Error('\p{Line_Break=/a/--Ideographic}');
Error('\P{Line_Break=/a/--Ideographic}');
Expect(1, 11929, '\p{Line_Break=ideographic}', "");
Expect(0, 11929, '\p{^Line_Break=ideographic}', "");
Expect(0, 11929, '\P{Line_Break=ideographic}', "");
Expect(1, 11929, '\P{^Line_Break=ideographic}', "");
Expect(0, 11903, '\p{Line_Break=ideographic}', "");
Expect(1, 11903, '\p{^Line_Break=ideographic}', "");
Expect(1, 11903, '\P{Line_Break=ideographic}', "");
Expect(0, 11903, '\P{^Line_Break=ideographic}', "");
Expect(1, 11929, '\p{Line_Break=	Ideographic}', "");
Expect(0, 11929, '\p{^Line_Break=	Ideographic}', "");
Expect(0, 11929, '\P{Line_Break=	Ideographic}', "");
Expect(1, 11929, '\P{^Line_Break=	Ideographic}', "");
Expect(0, 11903, '\p{Line_Break=	Ideographic}', "");
Expect(1, 11903, '\p{^Line_Break=	Ideographic}', "");
Expect(1, 11903, '\P{Line_Break=	Ideographic}', "");
Expect(0, 11903, '\P{^Line_Break=	Ideographic}', "");
Error('\p{Lb= _id/a/}');
Error('\P{Lb= _id/a/}');
Expect(1, 11929, '\p{Lb:	id}', "");
Expect(0, 11929, '\p{^Lb:	id}', "");
Expect(0, 11929, '\P{Lb:	id}', "");
Expect(1, 11929, '\P{^Lb:	id}', "");
Expect(0, 11903, '\p{Lb:	id}', "");
Expect(1, 11903, '\p{^Lb:	id}', "");
Expect(1, 11903, '\P{Lb:	id}', "");
Expect(0, 11903, '\P{^Lb:	id}', "");
Expect(1, 11929, '\p{Lb= 	ID}', "");
Expect(0, 11929, '\p{^Lb= 	ID}', "");
Expect(0, 11929, '\P{Lb= 	ID}', "");
Expect(1, 11929, '\P{^Lb= 	ID}', "");
Expect(0, 11903, '\p{Lb= 	ID}', "");
Expect(1, 11903, '\p{^Lb= 	ID}', "");
Expect(1, 11903, '\P{Lb= 	ID}', "");
Expect(0, 11903, '\P{^Lb= 	ID}', "");
Error('\p{Is_Line_Break=-Ideographic:=}');
Error('\P{Is_Line_Break=-Ideographic:=}');
Expect(1, 11929, '\p{Is_Line_Break=ideographic}', "");
Expect(0, 11929, '\p{^Is_Line_Break=ideographic}', "");
Expect(0, 11929, '\P{Is_Line_Break=ideographic}', "");
Expect(1, 11929, '\P{^Is_Line_Break=ideographic}', "");
Expect(0, 11903, '\p{Is_Line_Break=ideographic}', "");
Expect(1, 11903, '\p{^Is_Line_Break=ideographic}', "");
Expect(1, 11903, '\P{Is_Line_Break=ideographic}', "");
Expect(0, 11903, '\P{^Is_Line_Break=ideographic}', "");
Expect(1, 11929, '\p{Is_Line_Break=_Ideographic}', "");
Expect(0, 11929, '\p{^Is_Line_Break=_Ideographic}', "");
Expect(0, 11929, '\P{Is_Line_Break=_Ideographic}', "");
Expect(1, 11929, '\P{^Is_Line_Break=_Ideographic}', "");
Expect(0, 11903, '\p{Is_Line_Break=_Ideographic}', "");
Expect(1, 11903, '\p{^Is_Line_Break=_Ideographic}', "");
Expect(1, 11903, '\P{Is_Line_Break=_Ideographic}', "");
Expect(0, 11903, '\P{^Is_Line_Break=_Ideographic}', "");
Error('\p{Is_Lb=-:=ID}');
Error('\P{Is_Lb=-:=ID}');
Expect(1, 11929, '\p{Is_Lb=id}', "");
Expect(0, 11929, '\p{^Is_Lb=id}', "");
Expect(0, 11929, '\P{Is_Lb=id}', "");
Expect(1, 11929, '\P{^Is_Lb=id}', "");
Expect(0, 11903, '\p{Is_Lb=id}', "");
Expect(1, 11903, '\p{^Is_Lb=id}', "");
Expect(1, 11903, '\P{Is_Lb=id}', "");
Expect(0, 11903, '\P{^Is_Lb=id}', "");
Expect(1, 11929, '\p{Is_Lb=-ID}', "");
Expect(0, 11929, '\p{^Is_Lb=-ID}', "");
Expect(0, 11929, '\P{Is_Lb=-ID}', "");
Expect(1, 11929, '\P{^Is_Lb=-ID}', "");
Expect(0, 11903, '\p{Is_Lb=-ID}', "");
Expect(1, 11903, '\p{^Is_Lb=-ID}', "");
Expect(1, 11903, '\P{Is_Lb=-ID}', "");
Expect(0, 11903, '\P{^Is_Lb=-ID}', "");
Error('\p{Line_Break=	-WORD:=Joiner}');
Error('\P{Line_Break=	-WORD:=Joiner}');
Expect(1, 8288, '\p{Line_Break=wordjoiner}', "");
Expect(0, 8288, '\p{^Line_Break=wordjoiner}', "");
Expect(0, 8288, '\P{Line_Break=wordjoiner}', "");
Expect(1, 8288, '\P{^Line_Break=wordjoiner}', "");
Expect(0, 8287, '\p{Line_Break=wordjoiner}', "");
Expect(1, 8287, '\p{^Line_Break=wordjoiner}', "");
Expect(1, 8287, '\P{Line_Break=wordjoiner}', "");
Expect(0, 8287, '\P{^Line_Break=wordjoiner}', "");
Expect(1, 8288, '\p{Line_Break:	 Word Joiner}', "");
Expect(0, 8288, '\p{^Line_Break:	 Word Joiner}', "");
Expect(0, 8288, '\P{Line_Break:	 Word Joiner}', "");
Expect(1, 8288, '\P{^Line_Break:	 Word Joiner}', "");
Expect(0, 8287, '\p{Line_Break:	 Word Joiner}', "");
Expect(1, 8287, '\p{^Line_Break:	 Word Joiner}', "");
Expect(1, 8287, '\P{Line_Break:	 Word Joiner}', "");
Expect(0, 8287, '\P{^Line_Break:	 Word Joiner}', "");
Error('\p{Lb=_WJ:=}');
Error('\P{Lb=_WJ:=}');
Expect(1, 8288, '\p{Lb=wj}', "");
Expect(0, 8288, '\p{^Lb=wj}', "");
Expect(0, 8288, '\P{Lb=wj}', "");
Expect(1, 8288, '\P{^Lb=wj}', "");
Expect(0, 8287, '\p{Lb=wj}', "");
Expect(1, 8287, '\p{^Lb=wj}', "");
Expect(1, 8287, '\P{Lb=wj}', "");
Expect(0, 8287, '\P{^Lb=wj}', "");
Expect(1, 8288, '\p{Lb=-wj}', "");
Expect(0, 8288, '\p{^Lb=-wj}', "");
Expect(0, 8288, '\P{Lb=-wj}', "");
Expect(1, 8288, '\P{^Lb=-wj}', "");
Expect(0, 8287, '\p{Lb=-wj}', "");
Expect(1, 8287, '\p{^Lb=-wj}', "");
Expect(1, 8287, '\P{Lb=-wj}', "");
Expect(0, 8287, '\P{^Lb=-wj}', "");
Error('\p{Is_Line_Break=-/a/Word	Joiner}');
Error('\P{Is_Line_Break=-/a/Word	Joiner}');
Expect(1, 8288, '\p{Is_Line_Break=wordjoiner}', "");
Expect(0, 8288, '\p{^Is_Line_Break=wordjoiner}', "");
Expect(0, 8288, '\P{Is_Line_Break=wordjoiner}', "");
Expect(1, 8288, '\P{^Is_Line_Break=wordjoiner}', "");
Expect(0, 8287, '\p{Is_Line_Break=wordjoiner}', "");
Expect(1, 8287, '\p{^Is_Line_Break=wordjoiner}', "");
Expect(1, 8287, '\P{Is_Line_Break=wordjoiner}', "");
Expect(0, 8287, '\P{^Is_Line_Break=wordjoiner}', "");
Expect(1, 8288, '\p{Is_Line_Break=--word_Joiner}', "");
Expect(0, 8288, '\p{^Is_Line_Break=--word_Joiner}', "");
Expect(0, 8288, '\P{Is_Line_Break=--word_Joiner}', "");
Expect(1, 8288, '\P{^Is_Line_Break=--word_Joiner}', "");
Expect(0, 8287, '\p{Is_Line_Break=--word_Joiner}', "");
Expect(1, 8287, '\p{^Is_Line_Break=--word_Joiner}', "");
Expect(1, 8287, '\P{Is_Line_Break=--word_Joiner}', "");
Expect(0, 8287, '\P{^Is_Line_Break=--word_Joiner}', "");
Error('\p{Is_Lb=:=_	WJ}');
Error('\P{Is_Lb=:=_	WJ}');
Expect(1, 8288, '\p{Is_Lb=wj}', "");
Expect(0, 8288, '\p{^Is_Lb=wj}', "");
Expect(0, 8288, '\P{Is_Lb=wj}', "");
Expect(1, 8288, '\P{^Is_Lb=wj}', "");
Expect(0, 8287, '\p{Is_Lb=wj}', "");
Expect(1, 8287, '\p{^Is_Lb=wj}', "");
Expect(1, 8287, '\P{Is_Lb=wj}', "");
Expect(0, 8287, '\P{^Is_Lb=wj}', "");
Expect(1, 8288, '\p{Is_Lb:	 _WJ}', "");
Expect(0, 8288, '\p{^Is_Lb:	 _WJ}', "");
Expect(0, 8288, '\P{Is_Lb:	 _WJ}', "");
Expect(1, 8288, '\P{^Is_Lb:	 _WJ}', "");
Expect(0, 8287, '\p{Is_Lb:	 _WJ}', "");
Expect(1, 8287, '\p{^Is_Lb:	 _WJ}', "");
Expect(1, 8287, '\P{Is_Lb:	 _WJ}', "");
Expect(0, 8287, '\P{^Is_Lb:	 _WJ}', "");
Error('\p{Line_Break=/a/_h2}');
Error('\P{Line_Break=/a/_h2}');
Expect(1, 44032, '\p{Line_Break:	h2}', "");
Expect(0, 44032, '\p{^Line_Break:	h2}', "");
Expect(0, 44032, '\P{Line_Break:	h2}', "");
Expect(1, 44032, '\P{^Line_Break:	h2}', "");
Expect(0, 44031, '\p{Line_Break:	h2}', "");
Expect(1, 44031, '\p{^Line_Break:	h2}', "");
Expect(1, 44031, '\P{Line_Break:	h2}', "");
Expect(0, 44031, '\P{^Line_Break:	h2}', "");
Expect(1, 44032, '\p{Line_Break= -H2}', "");
Expect(0, 44032, '\p{^Line_Break= -H2}', "");
Expect(0, 44032, '\P{Line_Break= -H2}', "");
Expect(1, 44032, '\P{^Line_Break= -H2}', "");
Expect(0, 44031, '\p{Line_Break= -H2}', "");
Expect(1, 44031, '\p{^Line_Break= -H2}', "");
Expect(1, 44031, '\P{Line_Break= -H2}', "");
Expect(0, 44031, '\P{^Line_Break= -H2}', "");
Error('\p{Lb=/a/	-H2}');
Error('\P{Lb=/a/	-H2}');
Expect(1, 44032, '\p{Lb=h2}', "");
Expect(0, 44032, '\p{^Lb=h2}', "");
Expect(0, 44032, '\P{Lb=h2}', "");
Expect(1, 44032, '\P{^Lb=h2}', "");
Expect(0, 44031, '\p{Lb=h2}', "");
Expect(1, 44031, '\p{^Lb=h2}', "");
Expect(1, 44031, '\P{Lb=h2}', "");
Expect(0, 44031, '\P{^Lb=h2}', "");
Expect(1, 44032, '\p{Lb= 	H2}', "");
Expect(0, 44032, '\p{^Lb= 	H2}', "");
Expect(0, 44032, '\P{Lb= 	H2}', "");
Expect(1, 44032, '\P{^Lb= 	H2}', "");
Expect(0, 44031, '\p{Lb= 	H2}', "");
Expect(1, 44031, '\p{^Lb= 	H2}', "");
Expect(1, 44031, '\P{Lb= 	H2}', "");
Expect(0, 44031, '\P{^Lb= 	H2}', "");
Error('\p{Is_Line_Break=-h2/a/}');
Error('\P{Is_Line_Break=-h2/a/}');
Expect(1, 44032, '\p{Is_Line_Break=h2}', "");
Expect(0, 44032, '\p{^Is_Line_Break=h2}', "");
Expect(0, 44032, '\P{Is_Line_Break=h2}', "");
Expect(1, 44032, '\P{^Is_Line_Break=h2}', "");
Expect(0, 44031, '\p{Is_Line_Break=h2}', "");
Expect(1, 44031, '\p{^Is_Line_Break=h2}', "");
Expect(1, 44031, '\P{Is_Line_Break=h2}', "");
Expect(0, 44031, '\P{^Is_Line_Break=h2}', "");
Expect(1, 44032, '\p{Is_Line_Break=--H2}', "");
Expect(0, 44032, '\p{^Is_Line_Break=--H2}', "");
Expect(0, 44032, '\P{Is_Line_Break=--H2}', "");
Expect(1, 44032, '\P{^Is_Line_Break=--H2}', "");
Expect(0, 44031, '\p{Is_Line_Break=--H2}', "");
Expect(1, 44031, '\p{^Is_Line_Break=--H2}', "");
Expect(1, 44031, '\P{Is_Line_Break=--H2}', "");
Expect(0, 44031, '\P{^Is_Line_Break=--H2}', "");
Error('\p{Is_Lb=-:=H2}');
Error('\P{Is_Lb=-:=H2}');
Expect(1, 44032, '\p{Is_Lb=h2}', "");
Expect(0, 44032, '\p{^Is_Lb=h2}', "");
Expect(0, 44032, '\P{Is_Lb=h2}', "");
Expect(1, 44032, '\P{^Is_Lb=h2}', "");
Expect(0, 44031, '\p{Is_Lb=h2}', "");
Expect(1, 44031, '\p{^Is_Lb=h2}', "");
Expect(1, 44031, '\P{Is_Lb=h2}', "");
Expect(0, 44031, '\P{^Is_Lb=h2}', "");
Expect(1, 44032, '\p{Is_Lb=	-H2}', "");
Expect(0, 44032, '\p{^Is_Lb=	-H2}', "");
Expect(0, 44032, '\P{Is_Lb=	-H2}', "");
Expect(1, 44032, '\P{^Is_Lb=	-H2}', "");
Expect(0, 44031, '\p{Is_Lb=	-H2}', "");
Expect(1, 44031, '\p{^Is_Lb=	-H2}', "");
Expect(1, 44031, '\P{Is_Lb=	-H2}', "");
Expect(0, 44031, '\P{^Is_Lb=	-H2}', "");
Error('\p{Line_Break=-:=Break_BOTH}');
Error('\P{Line_Break=-:=Break_BOTH}');
Expect(1, 8212, '\p{Line_Break=breakboth}', "");
Expect(0, 8212, '\p{^Line_Break=breakboth}', "");
Expect(0, 8212, '\P{Line_Break=breakboth}', "");
Expect(1, 8212, '\P{^Line_Break=breakboth}', "");
Expect(0, 8211, '\p{Line_Break=breakboth}', "");
Expect(1, 8211, '\p{^Line_Break=breakboth}', "");
Expect(1, 8211, '\P{Line_Break=breakboth}', "");
Expect(0, 8211, '\P{^Line_Break=breakboth}', "");
Expect(1, 8212, '\p{Line_Break=_ Break BOTH}', "");
Expect(0, 8212, '\p{^Line_Break=_ Break BOTH}', "");
Expect(0, 8212, '\P{Line_Break=_ Break BOTH}', "");
Expect(1, 8212, '\P{^Line_Break=_ Break BOTH}', "");
Expect(0, 8211, '\p{Line_Break=_ Break BOTH}', "");
Expect(1, 8211, '\p{^Line_Break=_ Break BOTH}', "");
Expect(1, 8211, '\P{Line_Break=_ Break BOTH}', "");
Expect(0, 8211, '\P{^Line_Break=_ Break BOTH}', "");
Error('\p{Lb=:=-B2}');
Error('\P{Lb=:=-B2}');
Expect(1, 8212, '\p{Lb=b2}', "");
Expect(0, 8212, '\p{^Lb=b2}', "");
Expect(0, 8212, '\P{Lb=b2}', "");
Expect(1, 8212, '\P{^Lb=b2}', "");
Expect(0, 8211, '\p{Lb=b2}', "");
Expect(1, 8211, '\p{^Lb=b2}', "");
Expect(1, 8211, '\P{Lb=b2}', "");
Expect(0, 8211, '\P{^Lb=b2}', "");
Expect(1, 8212, '\p{Lb:	 B2}', "");
Expect(0, 8212, '\p{^Lb:	 B2}', "");
Expect(0, 8212, '\P{Lb:	 B2}', "");
Expect(1, 8212, '\P{^Lb:	 B2}', "");
Expect(0, 8211, '\p{Lb:	 B2}', "");
Expect(1, 8211, '\p{^Lb:	 B2}', "");
Expect(1, 8211, '\P{Lb:	 B2}', "");
Expect(0, 8211, '\P{^Lb:	 B2}', "");
Error('\p{Is_Line_Break=--break/a/both}');
Error('\P{Is_Line_Break=--break/a/both}');
Expect(1, 8212, '\p{Is_Line_Break:	breakboth}', "");
Expect(0, 8212, '\p{^Is_Line_Break:	breakboth}', "");
Expect(0, 8212, '\P{Is_Line_Break:	breakboth}', "");
Expect(1, 8212, '\P{^Is_Line_Break:	breakboth}', "");
Expect(0, 8211, '\p{Is_Line_Break:	breakboth}', "");
Expect(1, 8211, '\p{^Is_Line_Break:	breakboth}', "");
Expect(1, 8211, '\P{Is_Line_Break:	breakboth}', "");
Expect(0, 8211, '\P{^Is_Line_Break:	breakboth}', "");
Expect(1, 8212, '\p{Is_Line_Break=_	Break-BOTH}', "");
Expect(0, 8212, '\p{^Is_Line_Break=_	Break-BOTH}', "");
Expect(0, 8212, '\P{Is_Line_Break=_	Break-BOTH}', "");
Expect(1, 8212, '\P{^Is_Line_Break=_	Break-BOTH}', "");
Expect(0, 8211, '\p{Is_Line_Break=_	Break-BOTH}', "");
Expect(1, 8211, '\p{^Is_Line_Break=_	Break-BOTH}', "");
Expect(1, 8211, '\P{Is_Line_Break=_	Break-BOTH}', "");
Expect(0, 8211, '\P{^Is_Line_Break=_	Break-BOTH}', "");
Error('\p{Is_Lb=_/a/B2}');
Error('\P{Is_Lb=_/a/B2}');
Expect(1, 8212, '\p{Is_Lb=b2}', "");
Expect(0, 8212, '\p{^Is_Lb=b2}', "");
Expect(0, 8212, '\P{Is_Lb=b2}', "");
Expect(1, 8212, '\P{^Is_Lb=b2}', "");
Expect(0, 8211, '\p{Is_Lb=b2}', "");
Expect(1, 8211, '\p{^Is_Lb=b2}', "");
Expect(1, 8211, '\P{Is_Lb=b2}', "");
Expect(0, 8211, '\P{^Is_Lb=b2}', "");
Expect(1, 8212, '\p{Is_Lb=	_B2}', "");
Expect(0, 8212, '\p{^Is_Lb=	_B2}', "");
Expect(0, 8212, '\P{Is_Lb=	_B2}', "");
Expect(1, 8212, '\P{^Is_Lb=	_B2}', "");
Expect(0, 8211, '\p{Is_Lb=	_B2}', "");
Expect(1, 8211, '\p{^Is_Lb=	_B2}', "");
Expect(1, 8211, '\P{Is_Lb=	_B2}', "");
Expect(0, 8211, '\P{^Is_Lb=	_B2}', "");
Error('\p{Line_Break=/a/-Alphabetic}');
Error('\P{Line_Break=/a/-Alphabetic}');
Expect(1, 35, '\p{Line_Break=alphabetic}', "");
Expect(0, 35, '\p{^Line_Break=alphabetic}', "");
Expect(0, 35, '\P{Line_Break=alphabetic}', "");
Expect(1, 35, '\P{^Line_Break=alphabetic}', "");
Expect(0, 34, '\p{Line_Break=alphabetic}', "");
Expect(1, 34, '\p{^Line_Break=alphabetic}', "");
Expect(1, 34, '\P{Line_Break=alphabetic}', "");
Expect(0, 34, '\P{^Line_Break=alphabetic}', "");
Expect(1, 35, '\p{Line_Break=	alphabetic}', "");
Expect(0, 35, '\p{^Line_Break=	alphabetic}', "");
Expect(0, 35, '\P{Line_Break=	alphabetic}', "");
Expect(1, 35, '\P{^Line_Break=	alphabetic}', "");
Expect(0, 34, '\p{Line_Break=	alphabetic}', "");
Expect(1, 34, '\p{^Line_Break=	alphabetic}', "");
Expect(1, 34, '\P{Line_Break=	alphabetic}', "");
Expect(0, 34, '\P{^Line_Break=	alphabetic}', "");
Error('\p{Lb=_:=AL}');
Error('\P{Lb=_:=AL}');
Expect(1, 35, '\p{Lb=al}', "");
Expect(0, 35, '\p{^Lb=al}', "");
Expect(0, 35, '\P{Lb=al}', "");
Expect(1, 35, '\P{^Lb=al}', "");
Expect(0, 34, '\p{Lb=al}', "");
Expect(1, 34, '\p{^Lb=al}', "");
Expect(1, 34, '\P{Lb=al}', "");
Expect(0, 34, '\P{^Lb=al}', "");
Expect(1, 35, '\p{Lb: _ AL}', "");
Expect(0, 35, '\p{^Lb: _ AL}', "");
Expect(0, 35, '\P{Lb: _ AL}', "");
Expect(1, 35, '\P{^Lb: _ AL}', "");
Expect(0, 34, '\p{Lb: _ AL}', "");
Expect(1, 34, '\p{^Lb: _ AL}', "");
Expect(1, 34, '\P{Lb: _ AL}', "");
Expect(0, 34, '\P{^Lb: _ AL}', "");
Error('\p{Is_Line_Break=:=_-Alphabetic}');
Error('\P{Is_Line_Break=:=_-Alphabetic}');
Expect(1, 35, '\p{Is_Line_Break=alphabetic}', "");
Expect(0, 35, '\p{^Is_Line_Break=alphabetic}', "");
Expect(0, 35, '\P{Is_Line_Break=alphabetic}', "");
Expect(1, 35, '\P{^Is_Line_Break=alphabetic}', "");
Expect(0, 34, '\p{Is_Line_Break=alphabetic}', "");
Expect(1, 34, '\p{^Is_Line_Break=alphabetic}', "");
Expect(1, 34, '\P{Is_Line_Break=alphabetic}', "");
Expect(0, 34, '\P{^Is_Line_Break=alphabetic}', "");
Expect(1, 35, '\p{Is_Line_Break=- Alphabetic}', "");
Expect(0, 35, '\p{^Is_Line_Break=- Alphabetic}', "");
Expect(0, 35, '\P{Is_Line_Break=- Alphabetic}', "");
Expect(1, 35, '\P{^Is_Line_Break=- Alphabetic}', "");
Expect(0, 34, '\p{Is_Line_Break=- Alphabetic}', "");
Expect(1, 34, '\p{^Is_Line_Break=- Alphabetic}', "");
Expect(1, 34, '\P{Is_Line_Break=- Alphabetic}', "");
Expect(0, 34, '\P{^Is_Line_Break=- Alphabetic}', "");
Error('\p{Is_Lb: _ al/a/}');
Error('\P{Is_Lb: _ al/a/}');
Expect(1, 35, '\p{Is_Lb=al}', "");
Expect(0, 35, '\p{^Is_Lb=al}', "");
Expect(0, 35, '\P{Is_Lb=al}', "");
Expect(1, 35, '\P{^Is_Lb=al}', "");
Expect(0, 34, '\p{Is_Lb=al}', "");
Expect(1, 34, '\p{^Is_Lb=al}', "");
Expect(1, 34, '\P{Is_Lb=al}', "");
Expect(0, 34, '\P{^Is_Lb=al}', "");
Expect(1, 35, '\p{Is_Lb=-	al}', "");
Expect(0, 35, '\p{^Is_Lb=-	al}', "");
Expect(0, 35, '\P{Is_Lb=-	al}', "");
Expect(1, 35, '\P{^Is_Lb=-	al}', "");
Expect(0, 34, '\p{Is_Lb=-	al}', "");
Expect(1, 34, '\p{^Is_Lb=-	al}', "");
Expect(1, 34, '\P{Is_Lb=-	al}', "");
Expect(0, 34, '\P{^Is_Lb=-	al}', "");
Error('\p{Line_Break=	 postfix numeric/a/}');
Error('\P{Line_Break=	 postfix numeric/a/}');
Expect(1, 37, '\p{Line_Break=postfixnumeric}', "");
Expect(0, 37, '\p{^Line_Break=postfixnumeric}', "");
Expect(0, 37, '\P{Line_Break=postfixnumeric}', "");
Expect(1, 37, '\P{^Line_Break=postfixnumeric}', "");
Expect(0, 36, '\p{Line_Break=postfixnumeric}', "");
Expect(1, 36, '\p{^Line_Break=postfixnumeric}', "");
Expect(1, 36, '\P{Line_Break=postfixnumeric}', "");
Expect(0, 36, '\P{^Line_Break=postfixnumeric}', "");
Expect(1, 37, '\p{Line_Break= _POSTFIX	NUMERIC}', "");
Expect(0, 37, '\p{^Line_Break= _POSTFIX	NUMERIC}', "");
Expect(0, 37, '\P{Line_Break= _POSTFIX	NUMERIC}', "");
Expect(1, 37, '\P{^Line_Break= _POSTFIX	NUMERIC}', "");
Expect(0, 36, '\p{Line_Break= _POSTFIX	NUMERIC}', "");
Expect(1, 36, '\p{^Line_Break= _POSTFIX	NUMERIC}', "");
Expect(1, 36, '\P{Line_Break= _POSTFIX	NUMERIC}', "");
Expect(0, 36, '\P{^Line_Break= _POSTFIX	NUMERIC}', "");
Error('\p{Lb=_/a/po}');
Error('\P{Lb=_/a/po}');
Expect(1, 37, '\p{Lb=po}', "");
Expect(0, 37, '\p{^Lb=po}', "");
Expect(0, 37, '\P{Lb=po}', "");
Expect(1, 37, '\P{^Lb=po}', "");
Expect(0, 36, '\p{Lb=po}', "");
Expect(1, 36, '\p{^Lb=po}', "");
Expect(1, 36, '\P{Lb=po}', "");
Expect(0, 36, '\P{^Lb=po}', "");
Expect(1, 37, '\p{Lb:	_PO}', "");
Expect(0, 37, '\p{^Lb:	_PO}', "");
Expect(0, 37, '\P{Lb:	_PO}', "");
Expect(1, 37, '\P{^Lb:	_PO}', "");
Expect(0, 36, '\p{Lb:	_PO}', "");
Expect(1, 36, '\p{^Lb:	_PO}', "");
Expect(1, 36, '\P{Lb:	_PO}', "");
Expect(0, 36, '\P{^Lb:	_PO}', "");
Error('\p{Is_Line_Break=-postfix Numeric/a/}');
Error('\P{Is_Line_Break=-postfix Numeric/a/}');
Expect(1, 37, '\p{Is_Line_Break=postfixnumeric}', "");
Expect(0, 37, '\p{^Is_Line_Break=postfixnumeric}', "");
Expect(0, 37, '\P{Is_Line_Break=postfixnumeric}', "");
Expect(1, 37, '\P{^Is_Line_Break=postfixnumeric}', "");
Expect(0, 36, '\p{Is_Line_Break=postfixnumeric}', "");
Expect(1, 36, '\p{^Is_Line_Break=postfixnumeric}', "");
Expect(1, 36, '\P{Is_Line_Break=postfixnumeric}', "");
Expect(0, 36, '\P{^Is_Line_Break=postfixnumeric}', "");
Expect(1, 37, '\p{Is_Line_Break=_ Postfix Numeric}', "");
Expect(0, 37, '\p{^Is_Line_Break=_ Postfix Numeric}', "");
Expect(0, 37, '\P{Is_Line_Break=_ Postfix Numeric}', "");
Expect(1, 37, '\P{^Is_Line_Break=_ Postfix Numeric}', "");
Expect(0, 36, '\p{Is_Line_Break=_ Postfix Numeric}', "");
Expect(1, 36, '\p{^Is_Line_Break=_ Postfix Numeric}', "");
Expect(1, 36, '\P{Is_Line_Break=_ Postfix Numeric}', "");
Expect(0, 36, '\P{^Is_Line_Break=_ Postfix Numeric}', "");
Error('\p{Is_Lb=	:=PO}');
Error('\P{Is_Lb=	:=PO}');
Expect(1, 37, '\p{Is_Lb=po}', "");
Expect(0, 37, '\p{^Is_Lb=po}', "");
Expect(0, 37, '\P{Is_Lb=po}', "");
Expect(1, 37, '\P{^Is_Lb=po}', "");
Expect(0, 36, '\p{Is_Lb=po}', "");
Expect(1, 36, '\p{^Is_Lb=po}', "");
Expect(1, 36, '\P{Is_Lb=po}', "");
Expect(0, 36, '\P{^Is_Lb=po}', "");
Expect(1, 37, '\p{Is_Lb=-_PO}', "");
Expect(0, 37, '\p{^Is_Lb=-_PO}', "");
Expect(0, 37, '\P{Is_Lb=-_PO}', "");
Expect(1, 37, '\P{^Is_Lb=-_PO}', "");
Expect(0, 36, '\p{Is_Lb=-_PO}', "");
Expect(1, 36, '\p{^Is_Lb=-_PO}', "");
Expect(1, 36, '\P{Is_Lb=-_PO}', "");
Expect(0, 36, '\P{^Is_Lb=-_PO}', "");
Error('\p{Line_Break=/a/-Open	Punctuation}');
Error('\P{Line_Break=/a/-Open	Punctuation}');
Expect(1, 40, '\p{Line_Break=openpunctuation}', "");
Expect(0, 40, '\p{^Line_Break=openpunctuation}', "");
Expect(0, 40, '\P{Line_Break=openpunctuation}', "");
Expect(1, 40, '\P{^Line_Break=openpunctuation}', "");
Expect(0, 39, '\p{Line_Break=openpunctuation}', "");
Expect(1, 39, '\p{^Line_Break=openpunctuation}', "");
Expect(1, 39, '\P{Line_Break=openpunctuation}', "");
Expect(0, 39, '\P{^Line_Break=openpunctuation}', "");
Expect(1, 40, '\p{Line_Break=-open_Punctuation}', "");
Expect(0, 40, '\p{^Line_Break=-open_Punctuation}', "");
Expect(0, 40, '\P{Line_Break=-open_Punctuation}', "");
Expect(1, 40, '\P{^Line_Break=-open_Punctuation}', "");
Expect(0, 39, '\p{Line_Break=-open_Punctuation}', "");
Expect(1, 39, '\p{^Line_Break=-open_Punctuation}', "");
Expect(1, 39, '\P{Line_Break=-open_Punctuation}', "");
Expect(0, 39, '\P{^Line_Break=-open_Punctuation}', "");
Error('\p{Lb=--OP/a/}');
Error('\P{Lb=--OP/a/}');
Expect(1, 40, '\p{Lb=op}', "");
Expect(0, 40, '\p{^Lb=op}', "");
Expect(0, 40, '\P{Lb=op}', "");
Expect(1, 40, '\P{^Lb=op}', "");
Expect(0, 39, '\p{Lb=op}', "");
Expect(1, 39, '\p{^Lb=op}', "");
Expect(1, 39, '\P{Lb=op}', "");
Expect(0, 39, '\P{^Lb=op}', "");
Expect(1, 40, '\p{Lb=_op}', "");
Expect(0, 40, '\p{^Lb=_op}', "");
Expect(0, 40, '\P{Lb=_op}', "");
Expect(1, 40, '\P{^Lb=_op}', "");
Expect(0, 39, '\p{Lb=_op}', "");
Expect(1, 39, '\p{^Lb=_op}', "");
Expect(1, 39, '\P{Lb=_op}', "");
Expect(0, 39, '\P{^Lb=_op}', "");
Error('\p{Is_Line_Break= :=Open-Punctuation}');
Error('\P{Is_Line_Break= :=Open-Punctuation}');
Expect(1, 40, '\p{Is_Line_Break=openpunctuation}', "");
Expect(0, 40, '\p{^Is_Line_Break=openpunctuation}', "");
Expect(0, 40, '\P{Is_Line_Break=openpunctuation}', "");
Expect(1, 40, '\P{^Is_Line_Break=openpunctuation}', "");
Expect(0, 39, '\p{Is_Line_Break=openpunctuation}', "");
Expect(1, 39, '\p{^Is_Line_Break=openpunctuation}', "");
Expect(1, 39, '\P{Is_Line_Break=openpunctuation}', "");
Expect(0, 39, '\P{^Is_Line_Break=openpunctuation}', "");
Expect(1, 40, '\p{Is_Line_Break=		OPEN	Punctuation}', "");
Expect(0, 40, '\p{^Is_Line_Break=		OPEN	Punctuation}', "");
Expect(0, 40, '\P{Is_Line_Break=		OPEN	Punctuation}', "");
Expect(1, 40, '\P{^Is_Line_Break=		OPEN	Punctuation}', "");
Expect(0, 39, '\p{Is_Line_Break=		OPEN	Punctuation}', "");
Expect(1, 39, '\p{^Is_Line_Break=		OPEN	Punctuation}', "");
Expect(1, 39, '\P{Is_Line_Break=		OPEN	Punctuation}', "");
Expect(0, 39, '\P{^Is_Line_Break=		OPEN	Punctuation}', "");
Error('\p{Is_Lb=/a/op}');
Error('\P{Is_Lb=/a/op}');
Expect(1, 40, '\p{Is_Lb: op}', "");
Expect(0, 40, '\p{^Is_Lb: op}', "");
Expect(0, 40, '\P{Is_Lb: op}', "");
Expect(1, 40, '\P{^Is_Lb: op}', "");
Expect(0, 39, '\p{Is_Lb: op}', "");
Expect(1, 39, '\p{^Is_Lb: op}', "");
Expect(1, 39, '\P{Is_Lb: op}', "");
Expect(0, 39, '\P{^Is_Lb: op}', "");
Expect(1, 40, '\p{Is_Lb= -OP}', "");
Expect(0, 40, '\p{^Is_Lb= -OP}', "");
Expect(0, 40, '\P{Is_Lb= -OP}', "");
Expect(1, 40, '\P{^Is_Lb= -OP}', "");
Expect(0, 39, '\p{Is_Lb= -OP}', "");
Expect(1, 39, '\p{^Is_Lb= -OP}', "");
Expect(1, 39, '\P{Is_Lb= -OP}', "");
Expect(0, 39, '\P{^Is_Lb= -OP}', "");
Error('\p{Line_Break=:=-Next Line}');
Error('\P{Line_Break=:=-Next Line}');
Expect(1, 133, '\p{Line_Break=nextline}', "");
Expect(0, 133, '\p{^Line_Break=nextline}', "");
Expect(0, 133, '\P{Line_Break=nextline}', "");
Expect(1, 133, '\P{^Line_Break=nextline}', "");
Expect(0, 132, '\p{Line_Break=nextline}', "");
Expect(1, 132, '\p{^Line_Break=nextline}', "");
Expect(1, 132, '\P{Line_Break=nextline}', "");
Expect(0, 132, '\P{^Line_Break=nextline}', "");
Expect(1, 133, '\p{Line_Break= _Next-LINE}', "");
Expect(0, 133, '\p{^Line_Break= _Next-LINE}', "");
Expect(0, 133, '\P{Line_Break= _Next-LINE}', "");
Expect(1, 133, '\P{^Line_Break= _Next-LINE}', "");
Expect(0, 132, '\p{Line_Break= _Next-LINE}', "");
Expect(1, 132, '\p{^Line_Break= _Next-LINE}', "");
Expect(1, 132, '\P{Line_Break= _Next-LINE}', "");
Expect(0, 132, '\P{^Line_Break= _Next-LINE}', "");
Error('\p{Lb= NL/a/}');
Error('\P{Lb= NL/a/}');
Expect(1, 133, '\p{Lb=nl}', "");
Expect(0, 133, '\p{^Lb=nl}', "");
Expect(0, 133, '\P{Lb=nl}', "");
Expect(1, 133, '\P{^Lb=nl}', "");
Expect(0, 132, '\p{Lb=nl}', "");
Expect(1, 132, '\p{^Lb=nl}', "");
Expect(1, 132, '\P{Lb=nl}', "");
Expect(0, 132, '\P{^Lb=nl}', "");
Expect(1, 133, '\p{Lb:	 NL}', "");
Expect(0, 133, '\p{^Lb:	 NL}', "");
Expect(0, 133, '\P{Lb:	 NL}', "");
Expect(1, 133, '\P{^Lb:	 NL}', "");
Expect(0, 132, '\p{Lb:	 NL}', "");
Expect(1, 132, '\p{^Lb:	 NL}', "");
Expect(1, 132, '\P{Lb:	 NL}', "");
Expect(0, 132, '\P{^Lb:	 NL}', "");
Error('\p{Is_Line_Break=- Next Line/a/}');
Error('\P{Is_Line_Break=- Next Line/a/}');
Expect(1, 133, '\p{Is_Line_Break:nextline}', "");
Expect(0, 133, '\p{^Is_Line_Break:nextline}', "");
Expect(0, 133, '\P{Is_Line_Break:nextline}', "");
Expect(1, 133, '\P{^Is_Line_Break:nextline}', "");
Expect(0, 132, '\p{Is_Line_Break:nextline}', "");
Expect(1, 132, '\p{^Is_Line_Break:nextline}', "");
Expect(1, 132, '\P{Is_Line_Break:nextline}', "");
Expect(0, 132, '\P{^Is_Line_Break:nextline}', "");
Expect(1, 133, '\p{Is_Line_Break=_	NextLine}', "");
Expect(0, 133, '\p{^Is_Line_Break=_	NextLine}', "");
Expect(0, 133, '\P{Is_Line_Break=_	NextLine}', "");
Expect(1, 133, '\P{^Is_Line_Break=_	NextLine}', "");
Expect(0, 132, '\p{Is_Line_Break=_	NextLine}', "");
Expect(1, 132, '\p{^Is_Line_Break=_	NextLine}', "");
Expect(1, 132, '\P{Is_Line_Break=_	NextLine}', "");
Expect(0, 132, '\P{^Is_Line_Break=_	NextLine}', "");
Error('\p{Is_Lb=-nl/a/}');
Error('\P{Is_Lb=-nl/a/}');
Expect(1, 133, '\p{Is_Lb=nl}', "");
Expect(0, 133, '\p{^Is_Lb=nl}', "");
Expect(0, 133, '\P{Is_Lb=nl}', "");
Expect(1, 133, '\P{^Is_Lb=nl}', "");
Expect(0, 132, '\p{Is_Lb=nl}', "");
Expect(1, 132, '\p{^Is_Lb=nl}', "");
Expect(1, 132, '\P{Is_Lb=nl}', "");
Expect(0, 132, '\P{^Is_Lb=nl}', "");
Expect(1, 133, '\p{Is_Lb=	-nl}', "");
Expect(0, 133, '\p{^Is_Lb=	-nl}', "");
Expect(0, 133, '\P{Is_Lb=	-nl}', "");
Expect(1, 133, '\P{^Is_Lb=	-nl}', "");
Expect(0, 132, '\p{Is_Lb=	-nl}', "");
Expect(1, 132, '\p{^Is_Lb=	-nl}', "");
Expect(1, 132, '\P{Is_Lb=	-nl}', "");
Expect(0, 132, '\P{^Is_Lb=	-nl}', "");
Error('\p{Line_Break:	:=-	JT}');
Error('\P{Line_Break:	:=-	JT}');
Expect(1, 4607, '\p{Line_Break=jt}', "");
Expect(0, 4607, '\p{^Line_Break=jt}', "");
Expect(0, 4607, '\P{Line_Break=jt}', "");
Expect(1, 4607, '\P{^Line_Break=jt}', "");
Expect(0, 4519, '\p{Line_Break=jt}', "");
Expect(1, 4519, '\p{^Line_Break=jt}', "");
Expect(1, 4519, '\P{Line_Break=jt}', "");
Expect(0, 4519, '\P{^Line_Break=jt}', "");
Expect(1, 4607, '\p{Line_Break=	-JT}', "");
Expect(0, 4607, '\p{^Line_Break=	-JT}', "");
Expect(0, 4607, '\P{Line_Break=	-JT}', "");
Expect(1, 4607, '\P{^Line_Break=	-JT}', "");
Expect(0, 4519, '\p{Line_Break=	-JT}', "");
Expect(1, 4519, '\p{^Line_Break=	-JT}', "");
Expect(1, 4519, '\P{Line_Break=	-JT}', "");
Expect(0, 4519, '\P{^Line_Break=	-JT}', "");
Error('\p{Lb=/a/		JT}');
Error('\P{Lb=/a/		JT}');
Expect(1, 4607, '\p{Lb=jt}', "");
Expect(0, 4607, '\p{^Lb=jt}', "");
Expect(0, 4607, '\P{Lb=jt}', "");
Expect(1, 4607, '\P{^Lb=jt}', "");
Expect(0, 4519, '\p{Lb=jt}', "");
Expect(1, 4519, '\p{^Lb=jt}', "");
Expect(1, 4519, '\P{Lb=jt}', "");
Expect(0, 4519, '\P{^Lb=jt}', "");
Expect(1, 4607, '\p{Lb= JT}', "");
Expect(0, 4607, '\p{^Lb= JT}', "");
Expect(0, 4607, '\P{Lb= JT}', "");
Expect(1, 4607, '\P{^Lb= JT}', "");
Expect(0, 4519, '\p{Lb= JT}', "");
Expect(1, 4519, '\p{^Lb= JT}', "");
Expect(1, 4519, '\P{Lb= JT}', "");
Expect(0, 4519, '\P{^Lb= JT}', "");
Error('\p{Is_Line_Break=_jt:=}');
Error('\P{Is_Line_Break=_jt:=}');
Expect(1, 4607, '\p{Is_Line_Break=jt}', "");
Expect(0, 4607, '\p{^Is_Line_Break=jt}', "");
Expect(0, 4607, '\P{Is_Line_Break=jt}', "");
Expect(1, 4607, '\P{^Is_Line_Break=jt}', "");
Expect(0, 4519, '\p{Is_Line_Break=jt}', "");
Expect(1, 4519, '\p{^Is_Line_Break=jt}', "");
Expect(1, 4519, '\P{Is_Line_Break=jt}', "");
Expect(0, 4519, '\P{^Is_Line_Break=jt}', "");
Expect(1, 4607, '\p{Is_Line_Break=JT}', "");
Expect(0, 4607, '\p{^Is_Line_Break=JT}', "");
Expect(0, 4607, '\P{Is_Line_Break=JT}', "");
Expect(1, 4607, '\P{^Is_Line_Break=JT}', "");
Expect(0, 4519, '\p{Is_Line_Break=JT}', "");
Expect(1, 4519, '\p{^Is_Line_Break=JT}', "");
Expect(1, 4519, '\P{Is_Line_Break=JT}', "");
Expect(0, 4519, '\P{^Is_Line_Break=JT}', "");
Error('\p{Is_Lb=/a/ -JT}');
Error('\P{Is_Lb=/a/ -JT}');
Expect(1, 4607, '\p{Is_Lb=jt}', "");
Expect(0, 4607, '\p{^Is_Lb=jt}', "");
Expect(0, 4607, '\P{Is_Lb=jt}', "");
Expect(1, 4607, '\P{^Is_Lb=jt}', "");
Expect(0, 4519, '\p{Is_Lb=jt}', "");
Expect(1, 4519, '\p{^Is_Lb=jt}', "");
Expect(1, 4519, '\P{Is_Lb=jt}', "");
Expect(0, 4519, '\P{^Is_Lb=jt}', "");
Expect(1, 4607, '\p{Is_Lb=  JT}', "");
Expect(0, 4607, '\p{^Is_Lb=  JT}', "");
Expect(0, 4607, '\P{Is_Lb=  JT}', "");
Expect(1, 4607, '\P{^Is_Lb=  JT}', "");
Expect(0, 4519, '\p{Is_Lb=  JT}', "");
Expect(1, 4519, '\p{^Is_Lb=  JT}', "");
Expect(1, 4519, '\P{Is_Lb=  JT}', "");
Expect(0, 4519, '\P{^Is_Lb=  JT}', "");
Error('\p{Block=/a/_ New_Tai_Lue}');
Error('\P{Block=/a/_ New_Tai_Lue}');
Expect(1, 6623, '\p{Block=newtailue}', "");
Expect(0, 6623, '\p{^Block=newtailue}', "");
Expect(0, 6623, '\P{Block=newtailue}', "");
Expect(1, 6623, '\P{^Block=newtailue}', "");
Expect(0, 6527, '\p{Block=newtailue}', "");
Expect(1, 6527, '\p{^Block=newtailue}', "");
Expect(1, 6527, '\P{Block=newtailue}', "");
Expect(0, 6527, '\P{^Block=newtailue}', "");
Expect(1, 6623, '\p{Block:-New Tai_Lue}', "");
Expect(0, 6623, '\p{^Block:-New Tai_Lue}', "");
Expect(0, 6623, '\P{Block:-New Tai_Lue}', "");
Expect(1, 6623, '\P{^Block:-New Tai_Lue}', "");
Expect(0, 6527, '\p{Block:-New Tai_Lue}', "");
Expect(1, 6527, '\p{^Block:-New Tai_Lue}', "");
Expect(1, 6527, '\P{Block:-New Tai_Lue}', "");
Expect(0, 6527, '\P{^Block:-New Tai_Lue}', "");
Error('\p{Blk=_:=NEW-Tai Lue}');
Error('\P{Blk=_:=NEW-Tai Lue}');
Expect(1, 6623, '\p{Blk=newtailue}', "");
Expect(0, 6623, '\p{^Blk=newtailue}', "");
Expect(0, 6623, '\P{Blk=newtailue}', "");
Expect(1, 6623, '\P{^Blk=newtailue}', "");
Expect(0, 6527, '\p{Blk=newtailue}', "");
Expect(1, 6527, '\p{^Blk=newtailue}', "");
Expect(1, 6527, '\P{Blk=newtailue}', "");
Expect(0, 6527, '\P{^Blk=newtailue}', "");
Expect(1, 6623, '\p{Blk=	-new Tai Lue}', "");
Expect(0, 6623, '\p{^Blk=	-new Tai Lue}', "");
Expect(0, 6623, '\P{Blk=	-new Tai Lue}', "");
Expect(1, 6623, '\P{^Blk=	-new Tai Lue}', "");
Expect(0, 6527, '\p{Blk=	-new Tai Lue}', "");
Expect(1, 6527, '\p{^Blk=	-new Tai Lue}', "");
Expect(1, 6527, '\P{Blk=	-new Tai Lue}', "");
Expect(0, 6527, '\P{^Blk=	-new Tai Lue}', "");
Error('\p{Is_Block: -:=NewTAI	lue}');
Error('\P{Is_Block: -:=NewTAI	lue}');
Expect(1, 6623, '\p{Is_Block=newtailue}', "");
Expect(0, 6623, '\p{^Is_Block=newtailue}', "");
Expect(0, 6623, '\P{Is_Block=newtailue}', "");
Expect(1, 6623, '\P{^Is_Block=newtailue}', "");
Expect(0, 6527, '\p{Is_Block=newtailue}', "");
Expect(1, 6527, '\p{^Is_Block=newtailue}', "");
Expect(1, 6527, '\P{Is_Block=newtailue}', "");
Expect(0, 6527, '\P{^Is_Block=newtailue}', "");
Expect(1, 6623, '\p{Is_Block=_	New	Tai	Lue}', "");
Expect(0, 6623, '\p{^Is_Block=_	New	Tai	Lue}', "");
Expect(0, 6623, '\P{Is_Block=_	New	Tai	Lue}', "");
Expect(1, 6623, '\P{^Is_Block=_	New	Tai	Lue}', "");
Expect(0, 6527, '\p{Is_Block=_	New	Tai	Lue}', "");
Expect(1, 6527, '\p{^Is_Block=_	New	Tai	Lue}', "");
Expect(1, 6527, '\P{Is_Block=_	New	Tai	Lue}', "");
Expect(0, 6527, '\P{^Is_Block=_	New	Tai	Lue}', "");
Error('\p{Is_Blk:	 :=New-TAI-Lue}');
Error('\P{Is_Blk:	 :=New-TAI-Lue}');
Expect(1, 6623, '\p{Is_Blk=newtailue}', "");
Expect(0, 6623, '\p{^Is_Blk=newtailue}', "");
Expect(0, 6623, '\P{Is_Blk=newtailue}', "");
Expect(1, 6623, '\P{^Is_Blk=newtailue}', "");
Expect(0, 6527, '\p{Is_Blk=newtailue}', "");
Expect(1, 6527, '\p{^Is_Blk=newtailue}', "");
Expect(1, 6527, '\P{Is_Blk=newtailue}', "");
Expect(0, 6527, '\P{^Is_Blk=newtailue}', "");
Expect(1, 6623, '\p{Is_Blk=-new_tai_LUE}', "");
Expect(0, 6623, '\p{^Is_Blk=-new_tai_LUE}', "");
Expect(0, 6623, '\P{Is_Blk=-new_tai_LUE}', "");
Expect(1, 6623, '\P{^Is_Blk=-new_tai_LUE}', "");
Expect(0, 6527, '\p{Is_Blk=-new_tai_LUE}', "");
Expect(1, 6527, '\p{^Is_Blk=-new_tai_LUE}', "");
Expect(1, 6527, '\P{Is_Blk=-new_tai_LUE}', "");
Expect(0, 6527, '\P{^Is_Blk=-new_tai_LUE}', "");
Error('\p{Block= :=CJKUnified Ideographs EXTENSION-A}');
Error('\P{Block= :=CJKUnified Ideographs EXTENSION-A}');
Expect(1, 19903, '\p{Block=cjkunifiedideographsextensiona}', "");
Expect(0, 19903, '\p{^Block=cjkunifiedideographsextensiona}', "");
Expect(0, 19903, '\P{Block=cjkunifiedideographsextensiona}', "");
Expect(1, 19903, '\P{^Block=cjkunifiedideographsextensiona}', "");
Expect(0, 13311, '\p{Block=cjkunifiedideographsextensiona}', "");
Expect(1, 13311, '\p{^Block=cjkunifiedideographsextensiona}', "");
Expect(1, 13311, '\P{Block=cjkunifiedideographsextensiona}', "");
Expect(0, 13311, '\P{^Block=cjkunifiedideographsextensiona}', "");
Expect(1, 19903, '\p{Block=-_cjk-Unified	IdeographsEXTENSION a}', "");
Expect(0, 19903, '\p{^Block=-_cjk-Unified	IdeographsEXTENSION a}', "");
Expect(0, 19903, '\P{Block=-_cjk-Unified	IdeographsEXTENSION a}', "");
Expect(1, 19903, '\P{^Block=-_cjk-Unified	IdeographsEXTENSION a}', "");
Expect(0, 13311, '\p{Block=-_cjk-Unified	IdeographsEXTENSION a}', "");
Expect(1, 13311, '\p{^Block=-_cjk-Unified	IdeographsEXTENSION a}', "");
Expect(1, 13311, '\P{Block=-_cjk-Unified	IdeographsEXTENSION a}', "");
Expect(0, 13311, '\P{^Block=-_cjk-Unified	IdeographsEXTENSION a}', "");
Error('\p{Blk=-CJK_UNIFIEDIDEOGRAPHS:=EXTENSION A}');
Error('\P{Blk=-CJK_UNIFIEDIDEOGRAPHS:=EXTENSION A}');
Expect(1, 19903, '\p{Blk=cjkunifiedideographsextensiona}', "");
Expect(0, 19903, '\p{^Blk=cjkunifiedideographsextensiona}', "");
Expect(0, 19903, '\P{Blk=cjkunifiedideographsextensiona}', "");
Expect(1, 19903, '\P{^Blk=cjkunifiedideographsextensiona}', "");
Expect(0, 13311, '\p{Blk=cjkunifiedideographsextensiona}', "");
Expect(1, 13311, '\p{^Blk=cjkunifiedideographsextensiona}', "");
Expect(1, 13311, '\P{Blk=cjkunifiedideographsextensiona}', "");
Expect(0, 13311, '\P{^Blk=cjkunifiedideographsextensiona}', "");
Expect(1, 19903, '\p{Blk=-	CJK-unified IDEOGRAPHS	EXTENSION	A}', "");
Expect(0, 19903, '\p{^Blk=-	CJK-unified IDEOGRAPHS	EXTENSION	A}', "");
Expect(0, 19903, '\P{Blk=-	CJK-unified IDEOGRAPHS	EXTENSION	A}', "");
Expect(1, 19903, '\P{^Blk=-	CJK-unified IDEOGRAPHS	EXTENSION	A}', "");
Expect(0, 13311, '\p{Blk=-	CJK-unified IDEOGRAPHS	EXTENSION	A}', "");
Expect(1, 13311, '\p{^Blk=-	CJK-unified IDEOGRAPHS	EXTENSION	A}', "");
Expect(1, 13311, '\P{Blk=-	CJK-unified IDEOGRAPHS	EXTENSION	A}', "");
Expect(0, 13311, '\P{^Blk=-	CJK-unified IDEOGRAPHS	EXTENSION	A}', "");
Error('\p{Is_Block=_	CJK-unified:=ideographs_Extension A}');
Error('\P{Is_Block=_	CJK-unified:=ideographs_Extension A}');
Expect(1, 19903, '\p{Is_Block=cjkunifiedideographsextensiona}', "");
Expect(0, 19903, '\p{^Is_Block=cjkunifiedideographsextensiona}', "");
Expect(0, 19903, '\P{Is_Block=cjkunifiedideographsextensiona}', "");
Expect(1, 19903, '\P{^Is_Block=cjkunifiedideographsextensiona}', "");
Expect(0, 13311, '\p{Is_Block=cjkunifiedideographsextensiona}', "");
Expect(1, 13311, '\p{^Is_Block=cjkunifiedideographsextensiona}', "");
Expect(1, 13311, '\P{Is_Block=cjkunifiedideographsextensiona}', "");
Expect(0, 13311, '\P{^Is_Block=cjkunifiedideographsextensiona}', "");
Expect(1, 19903, '\p{Is_Block=- CJK-unifiedIdeographs_Extension A}', "");
Expect(0, 19903, '\p{^Is_Block=- CJK-unifiedIdeographs_Extension A}', "");
Expect(0, 19903, '\P{Is_Block=- CJK-unifiedIdeographs_Extension A}', "");
Expect(1, 19903, '\P{^Is_Block=- CJK-unifiedIdeographs_Extension A}', "");
Expect(0, 13311, '\p{Is_Block=- CJK-unifiedIdeographs_Extension A}', "");
Expect(1, 13311, '\p{^Is_Block=- CJK-unifiedIdeographs_Extension A}', "");
Expect(1, 13311, '\P{Is_Block=- CJK-unifiedIdeographs_Extension A}', "");
Expect(0, 13311, '\P{^Is_Block=- CJK-unifiedIdeographs_Extension A}', "");
Error('\p{Is_Blk=	:=CJK-unified-Ideographs-EXTENSION_a}');
Error('\P{Is_Blk=	:=CJK-unified-Ideographs-EXTENSION_a}');
Expect(1, 19903, '\p{Is_Blk=cjkunifiedideographsextensiona}', "");
Expect(0, 19903, '\p{^Is_Blk=cjkunifiedideographsextensiona}', "");
Expect(0, 19903, '\P{Is_Blk=cjkunifiedideographsextensiona}', "");
Expect(1, 19903, '\P{^Is_Blk=cjkunifiedideographsextensiona}', "");
Expect(0, 13311, '\p{Is_Blk=cjkunifiedideographsextensiona}', "");
Expect(1, 13311, '\p{^Is_Blk=cjkunifiedideographsextensiona}', "");
Expect(1, 13311, '\P{Is_Blk=cjkunifiedideographsextensiona}', "");
Expect(0, 13311, '\P{^Is_Blk=cjkunifiedideographsextensiona}', "");
Expect(1, 19903, '\p{Is_Blk= cjkUNIFIED Ideographs Extension-a}', "");
Expect(0, 19903, '\p{^Is_Blk= cjkUNIFIED Ideographs Extension-a}', "");
Expect(0, 19903, '\P{Is_Blk= cjkUNIFIED Ideographs Extension-a}', "");
Expect(1, 19903, '\P{^Is_Blk= cjkUNIFIED Ideographs Extension-a}', "");
Expect(0, 13311, '\p{Is_Blk= cjkUNIFIED Ideographs Extension-a}', "");
Expect(1, 13311, '\p{^Is_Blk= cjkUNIFIED Ideographs Extension-a}', "");
Expect(1, 13311, '\P{Is_Blk= cjkUNIFIED Ideographs Extension-a}', "");
Expect(0, 13311, '\P{^Is_Blk= cjkUNIFIED Ideographs Extension-a}', "");
Error('\p{Block:   :=		Hangul jamo}');
Error('\P{Block:   :=		Hangul jamo}');
Expect(1, 4607, '\p{Block=hanguljamo}', "");
Expect(0, 4607, '\p{^Block=hanguljamo}', "");
Expect(0, 4607, '\P{Block=hanguljamo}', "");
Expect(1, 4607, '\P{^Block=hanguljamo}', "");
Expect(0, 4351, '\p{Block=hanguljamo}', "");
Expect(1, 4351, '\p{^Block=hanguljamo}', "");
Expect(1, 4351, '\P{Block=hanguljamo}', "");
Expect(0, 4351, '\P{^Block=hanguljamo}', "");
Expect(1, 4607, '\p{Block=_Hangul_Jamo}', "");
Expect(0, 4607, '\p{^Block=_Hangul_Jamo}', "");
Expect(0, 4607, '\P{Block=_Hangul_Jamo}', "");
Expect(1, 4607, '\P{^Block=_Hangul_Jamo}', "");
Expect(0, 4351, '\p{Block=_Hangul_Jamo}', "");
Expect(1, 4351, '\p{^Block=_Hangul_Jamo}', "");
Expect(1, 4351, '\P{Block=_Hangul_Jamo}', "");
Expect(0, 4351, '\P{^Block=_Hangul_Jamo}', "");
Error('\p{Blk=-hangul/a/jamo}');
Error('\P{Blk=-hangul/a/jamo}');
Expect(1, 4607, '\p{Blk=hanguljamo}', "");
Expect(0, 4607, '\p{^Blk=hanguljamo}', "");
Expect(0, 4607, '\P{Blk=hanguljamo}', "");
Expect(1, 4607, '\P{^Blk=hanguljamo}', "");
Expect(0, 4351, '\p{Blk=hanguljamo}', "");
Expect(1, 4351, '\p{^Blk=hanguljamo}', "");
Expect(1, 4351, '\P{Blk=hanguljamo}', "");
Expect(0, 4351, '\P{^Blk=hanguljamo}', "");
Expect(1, 4607, '\p{Blk=_hangul_JAMO}', "");
Expect(0, 4607, '\p{^Blk=_hangul_JAMO}', "");
Expect(0, 4607, '\P{Blk=_hangul_JAMO}', "");
Expect(1, 4607, '\P{^Blk=_hangul_JAMO}', "");
Expect(0, 4351, '\p{Blk=_hangul_JAMO}', "");
Expect(1, 4351, '\p{^Blk=_hangul_JAMO}', "");
Expect(1, 4351, '\P{Blk=_hangul_JAMO}', "");
Expect(0, 4351, '\P{^Blk=_hangul_JAMO}', "");
Error('\p{Is_Block=-/a/Hangul-Jamo}');
Error('\P{Is_Block=-/a/Hangul-Jamo}');
Expect(1, 4607, '\p{Is_Block=hanguljamo}', "");
Expect(0, 4607, '\p{^Is_Block=hanguljamo}', "");
Expect(0, 4607, '\P{Is_Block=hanguljamo}', "");
Expect(1, 4607, '\P{^Is_Block=hanguljamo}', "");
Expect(0, 4351, '\p{Is_Block=hanguljamo}', "");
Expect(1, 4351, '\p{^Is_Block=hanguljamo}', "");
Expect(1, 4351, '\P{Is_Block=hanguljamo}', "");
Expect(0, 4351, '\P{^Is_Block=hanguljamo}', "");
Expect(1, 4607, '\p{Is_Block=- HANGUL_Jamo}', "");
Expect(0, 4607, '\p{^Is_Block=- HANGUL_Jamo}', "");
Expect(0, 4607, '\P{Is_Block=- HANGUL_Jamo}', "");
Expect(1, 4607, '\P{^Is_Block=- HANGUL_Jamo}', "");
Expect(0, 4351, '\p{Is_Block=- HANGUL_Jamo}', "");
Expect(1, 4351, '\p{^Is_Block=- HANGUL_Jamo}', "");
Expect(1, 4351, '\P{Is_Block=- HANGUL_Jamo}', "");
Expect(0, 4351, '\P{^Is_Block=- HANGUL_Jamo}', "");
Error('\p{Is_Blk=-/a/Hangul	jamo}');
Error('\P{Is_Blk=-/a/Hangul	jamo}');
Expect(1, 4607, '\p{Is_Blk=hanguljamo}', "");
Expect(0, 4607, '\p{^Is_Blk=hanguljamo}', "");
Expect(0, 4607, '\P{Is_Blk=hanguljamo}', "");
Expect(1, 4607, '\P{^Is_Blk=hanguljamo}', "");
Expect(0, 4351, '\p{Is_Blk=hanguljamo}', "");
Expect(1, 4351, '\p{^Is_Blk=hanguljamo}', "");
Expect(1, 4351, '\P{Is_Blk=hanguljamo}', "");
Expect(0, 4351, '\P{^Is_Blk=hanguljamo}', "");
Expect(1, 4607, '\p{Is_Blk=-_hangul_Jamo}', "");
Expect(0, 4607, '\p{^Is_Blk=-_hangul_Jamo}', "");
Expect(0, 4607, '\P{Is_Blk=-_hangul_Jamo}', "");
Expect(1, 4607, '\P{^Is_Blk=-_hangul_Jamo}', "");
Expect(0, 4351, '\p{Is_Blk=-_hangul_Jamo}', "");
Expect(1, 4351, '\p{^Is_Blk=-_hangul_Jamo}', "");
Expect(1, 4351, '\P{Is_Blk=-_hangul_Jamo}', "");
Expect(0, 4351, '\P{^Is_Blk=-_hangul_Jamo}', "");
Error('\p{Block= _dingbats/a/}');
Error('\P{Block= _dingbats/a/}');
Expect(1, 10175, '\p{Block=dingbats}', "");
Expect(0, 10175, '\p{^Block=dingbats}', "");
Expect(0, 10175, '\P{Block=dingbats}', "");
Expect(1, 10175, '\P{^Block=dingbats}', "");
Expect(0, 9983, '\p{Block=dingbats}', "");
Expect(1, 9983, '\p{^Block=dingbats}', "");
Expect(1, 9983, '\P{Block=dingbats}', "");
Expect(0, 9983, '\P{^Block=dingbats}', "");
Expect(1, 10175, '\p{Block=_Dingbats}', "");
Expect(0, 10175, '\p{^Block=_Dingbats}', "");
Expect(0, 10175, '\P{Block=_Dingbats}', "");
Expect(1, 10175, '\P{^Block=_Dingbats}', "");
Expect(0, 9983, '\p{Block=_Dingbats}', "");
Expect(1, 9983, '\p{^Block=_Dingbats}', "");
Expect(1, 9983, '\P{Block=_Dingbats}', "");
Expect(0, 9983, '\P{^Block=_Dingbats}', "");
Error('\p{Blk=  dingbats/a/}');
Error('\P{Blk=  dingbats/a/}');
Expect(1, 10175, '\p{Blk=dingbats}', "");
Expect(0, 10175, '\p{^Blk=dingbats}', "");
Expect(0, 10175, '\P{Blk=dingbats}', "");
Expect(1, 10175, '\P{^Blk=dingbats}', "");
Expect(0, 9983, '\p{Blk=dingbats}', "");
Expect(1, 9983, '\p{^Blk=dingbats}', "");
Expect(1, 9983, '\P{Blk=dingbats}', "");
Expect(0, 9983, '\P{^Blk=dingbats}', "");
Expect(1, 10175, '\p{Blk=Dingbats}', "");
Expect(0, 10175, '\p{^Blk=Dingbats}', "");
Expect(0, 10175, '\P{Blk=Dingbats}', "");
Expect(1, 10175, '\P{^Blk=Dingbats}', "");
Expect(0, 9983, '\p{Blk=Dingbats}', "");
Expect(1, 9983, '\p{^Blk=Dingbats}', "");
Expect(1, 9983, '\P{Blk=Dingbats}', "");
Expect(0, 9983, '\P{^Blk=Dingbats}', "");
Error('\p{Is_Block=/a/_	Dingbats}');
Error('\P{Is_Block=/a/_	Dingbats}');
Expect(1, 10175, '\p{Is_Block:dingbats}', "");
Expect(0, 10175, '\p{^Is_Block:dingbats}', "");
Expect(0, 10175, '\P{Is_Block:dingbats}', "");
Expect(1, 10175, '\P{^Is_Block:dingbats}', "");
Expect(0, 9983, '\p{Is_Block:dingbats}', "");
Expect(1, 9983, '\p{^Is_Block:dingbats}', "");
Expect(1, 9983, '\P{Is_Block:dingbats}', "");
Expect(0, 9983, '\P{^Is_Block:dingbats}', "");
Expect(1, 10175, '\p{Is_Block=	-Dingbats}', "");
Expect(0, 10175, '\p{^Is_Block=	-Dingbats}', "");
Expect(0, 10175, '\P{Is_Block=	-Dingbats}', "");
Expect(1, 10175, '\P{^Is_Block=	-Dingbats}', "");
Expect(0, 9983, '\p{Is_Block=	-Dingbats}', "");
Expect(1, 9983, '\p{^Is_Block=	-Dingbats}', "");
Expect(1, 9983, '\P{Is_Block=	-Dingbats}', "");
Expect(0, 9983, '\P{^Is_Block=	-Dingbats}', "");
Error('\p{Is_Blk=:=_-DINGBATS}');
Error('\P{Is_Blk=:=_-DINGBATS}');
Expect(1, 10175, '\p{Is_Blk=dingbats}', "");
Expect(0, 10175, '\p{^Is_Blk=dingbats}', "");
Expect(0, 10175, '\P{Is_Blk=dingbats}', "");
Expect(1, 10175, '\P{^Is_Blk=dingbats}', "");
Expect(0, 9983, '\p{Is_Blk=dingbats}', "");
Expect(1, 9983, '\p{^Is_Blk=dingbats}', "");
Expect(1, 9983, '\P{Is_Blk=dingbats}', "");
Expect(0, 9983, '\P{^Is_Blk=dingbats}', "");
Expect(1, 10175, '\p{Is_Blk=-DINGBATS}', "");
Expect(0, 10175, '\p{^Is_Blk=-DINGBATS}', "");
Expect(0, 10175, '\P{Is_Blk=-DINGBATS}', "");
Expect(1, 10175, '\P{^Is_Blk=-DINGBATS}', "");
Expect(0, 9983, '\p{Is_Blk=-DINGBATS}', "");
Expect(1, 9983, '\p{^Is_Blk=-DINGBATS}', "");
Expect(1, 9983, '\P{Is_Blk=-DINGBATS}', "");
Expect(0, 9983, '\P{^Is_Blk=-DINGBATS}', "");
Error('\p{Block=- Bamum:=}');
Error('\P{Block=- Bamum:=}');
Expect(1, 42751, '\p{Block=bamum}', "");
Expect(0, 42751, '\p{^Block=bamum}', "");
Expect(0, 42751, '\P{Block=bamum}', "");
Expect(1, 42751, '\P{^Block=bamum}', "");
Expect(0, 42655, '\p{Block=bamum}', "");
Expect(1, 42655, '\p{^Block=bamum}', "");
Expect(1, 42655, '\P{Block=bamum}', "");
Expect(0, 42655, '\P{^Block=bamum}', "");
Expect(1, 42751, '\p{Block:   _-Bamum}', "");
Expect(0, 42751, '\p{^Block:   _-Bamum}', "");
Expect(0, 42751, '\P{Block:   _-Bamum}', "");
Expect(1, 42751, '\P{^Block:   _-Bamum}', "");
Expect(0, 42655, '\p{Block:   _-Bamum}', "");
Expect(1, 42655, '\p{^Block:   _-Bamum}', "");
Expect(1, 42655, '\P{Block:   _-Bamum}', "");
Expect(0, 42655, '\P{^Block:   _-Bamum}', "");
Error('\p{Blk=	:=BAMUM}');
Error('\P{Blk=	:=BAMUM}');
Expect(1, 42751, '\p{Blk=bamum}', "");
Expect(0, 42751, '\p{^Blk=bamum}', "");
Expect(0, 42751, '\P{Blk=bamum}', "");
Expect(1, 42751, '\P{^Blk=bamum}', "");
Expect(0, 42655, '\p{Blk=bamum}', "");
Expect(1, 42655, '\p{^Blk=bamum}', "");
Expect(1, 42655, '\P{Blk=bamum}', "");
Expect(0, 42655, '\P{^Blk=bamum}', "");
Expect(1, 42751, '\p{Blk=-_bamum}', "");
Expect(0, 42751, '\p{^Blk=-_bamum}', "");
Expect(0, 42751, '\P{Blk=-_bamum}', "");
Expect(1, 42751, '\P{^Blk=-_bamum}', "");
Expect(0, 42655, '\p{Blk=-_bamum}', "");
Expect(1, 42655, '\p{^Blk=-_bamum}', "");
Expect(1, 42655, '\P{Blk=-_bamum}', "");
Expect(0, 42655, '\P{^Blk=-_bamum}', "");
Error('\p{Is_Block=	/a/Bamum}');
Error('\P{Is_Block=	/a/Bamum}');
Expect(1, 42751, '\p{Is_Block=bamum}', "");
Expect(0, 42751, '\p{^Is_Block=bamum}', "");
Expect(0, 42751, '\P{Is_Block=bamum}', "");
Expect(1, 42751, '\P{^Is_Block=bamum}', "");
Expect(0, 42655, '\p{Is_Block=bamum}', "");
Expect(1, 42655, '\p{^Is_Block=bamum}', "");
Expect(1, 42655, '\P{Is_Block=bamum}', "");
Expect(0, 42655, '\P{^Is_Block=bamum}', "");
Expect(1, 42751, '\p{Is_Block:	-Bamum}', "");
Expect(0, 42751, '\p{^Is_Block:	-Bamum}', "");
Expect(0, 42751, '\P{Is_Block:	-Bamum}', "");
Expect(1, 42751, '\P{^Is_Block:	-Bamum}', "");
Expect(0, 42655, '\p{Is_Block:	-Bamum}', "");
Expect(1, 42655, '\p{^Is_Block:	-Bamum}', "");
Expect(1, 42655, '\P{Is_Block:	-Bamum}', "");
Expect(0, 42655, '\P{^Is_Block:	-Bamum}', "");
Error('\p{Is_Blk=:=		BAMUM}');
Error('\P{Is_Blk=:=		BAMUM}');
Expect(1, 42751, '\p{Is_Blk=bamum}', "");
Expect(0, 42751, '\p{^Is_Blk=bamum}', "");
Expect(0, 42751, '\P{Is_Blk=bamum}', "");
Expect(1, 42751, '\P{^Is_Blk=bamum}', "");
Expect(0, 42655, '\p{Is_Blk=bamum}', "");
Expect(1, 42655, '\p{^Is_Blk=bamum}', "");
Expect(1, 42655, '\P{Is_Blk=bamum}', "");
Expect(0, 42655, '\P{^Is_Blk=bamum}', "");
Expect(1, 42751, '\p{Is_Blk:	__bamum}', "");
Expect(0, 42751, '\p{^Is_Blk:	__bamum}', "");
Expect(0, 42751, '\P{Is_Blk:	__bamum}', "");
Expect(1, 42751, '\P{^Is_Blk:	__bamum}', "");
Expect(0, 42655, '\p{Is_Blk:	__bamum}', "");
Expect(1, 42655, '\p{^Is_Blk:	__bamum}', "");
Expect(1, 42655, '\P{Is_Blk:	__bamum}', "");
Expect(0, 42655, '\P{^Is_Blk:	__bamum}', "");
Error('\p{Block=	 Inscriptional	parthian:=}');
Error('\P{Block=	 Inscriptional	parthian:=}');
Expect(1, 68447, '\p{Block: inscriptionalparthian}', "");
Expect(0, 68447, '\p{^Block: inscriptionalparthian}', "");
Expect(0, 68447, '\P{Block: inscriptionalparthian}', "");
Expect(1, 68447, '\P{^Block: inscriptionalparthian}', "");
Expect(0, 68415, '\p{Block: inscriptionalparthian}', "");
Expect(1, 68415, '\p{^Block: inscriptionalparthian}', "");
Expect(1, 68415, '\P{Block: inscriptionalparthian}', "");
Expect(0, 68415, '\P{^Block: inscriptionalparthian}', "");
Expect(1, 68447, '\p{Block=		INSCRIPTIONAL	PARTHIAN}', "");
Expect(0, 68447, '\p{^Block=		INSCRIPTIONAL	PARTHIAN}', "");
Expect(0, 68447, '\P{Block=		INSCRIPTIONAL	PARTHIAN}', "");
Expect(1, 68447, '\P{^Block=		INSCRIPTIONAL	PARTHIAN}', "");
Expect(0, 68415, '\p{Block=		INSCRIPTIONAL	PARTHIAN}', "");
Expect(1, 68415, '\p{^Block=		INSCRIPTIONAL	PARTHIAN}', "");
Expect(1, 68415, '\P{Block=		INSCRIPTIONAL	PARTHIAN}', "");
Expect(0, 68415, '\P{^Block=		INSCRIPTIONAL	PARTHIAN}', "");
Error('\p{Blk=	_inscriptional/a/Parthian}');
Error('\P{Blk=	_inscriptional/a/Parthian}');
Expect(1, 68447, '\p{Blk=inscriptionalparthian}', "");
Expect(0, 68447, '\p{^Blk=inscriptionalparthian}', "");
Expect(0, 68447, '\P{Blk=inscriptionalparthian}', "");
Expect(1, 68447, '\P{^Blk=inscriptionalparthian}', "");
Expect(0, 68415, '\p{Blk=inscriptionalparthian}', "");
Expect(1, 68415, '\p{^Blk=inscriptionalparthian}', "");
Expect(1, 68415, '\P{Blk=inscriptionalparthian}', "");
Expect(0, 68415, '\P{^Blk=inscriptionalparthian}', "");
Expect(1, 68447, '\p{Blk=--inscriptional-Parthian}', "");
Expect(0, 68447, '\p{^Blk=--inscriptional-Parthian}', "");
Expect(0, 68447, '\P{Blk=--inscriptional-Parthian}', "");
Expect(1, 68447, '\P{^Blk=--inscriptional-Parthian}', "");
Expect(0, 68415, '\p{Blk=--inscriptional-Parthian}', "");
Expect(1, 68415, '\p{^Blk=--inscriptional-Parthian}', "");
Expect(1, 68415, '\P{Blk=--inscriptional-Parthian}', "");
Expect(0, 68415, '\P{^Blk=--inscriptional-Parthian}', "");
Error('\p{Is_Block=/a/inscriptional-PARTHIAN}');
Error('\P{Is_Block=/a/inscriptional-PARTHIAN}');
Expect(1, 68447, '\p{Is_Block=inscriptionalparthian}', "");
Expect(0, 68447, '\p{^Is_Block=inscriptionalparthian}', "");
Expect(0, 68447, '\P{Is_Block=inscriptionalparthian}', "");
Expect(1, 68447, '\P{^Is_Block=inscriptionalparthian}', "");
Expect(0, 68415, '\p{Is_Block=inscriptionalparthian}', "");
Expect(1, 68415, '\p{^Is_Block=inscriptionalparthian}', "");
Expect(1, 68415, '\P{Is_Block=inscriptionalparthian}', "");
Expect(0, 68415, '\P{^Is_Block=inscriptionalparthian}', "");
Expect(1, 68447, '\p{Is_Block=_	Inscriptional	Parthian}', "");
Expect(0, 68447, '\p{^Is_Block=_	Inscriptional	Parthian}', "");
Expect(0, 68447, '\P{Is_Block=_	Inscriptional	Parthian}', "");
Expect(1, 68447, '\P{^Is_Block=_	Inscriptional	Parthian}', "");
Expect(0, 68415, '\p{Is_Block=_	Inscriptional	Parthian}', "");
Expect(1, 68415, '\p{^Is_Block=_	Inscriptional	Parthian}', "");
Expect(1, 68415, '\P{Is_Block=_	Inscriptional	Parthian}', "");
Expect(0, 68415, '\P{^Is_Block=_	Inscriptional	Parthian}', "");
Error('\p{Is_Blk=_Inscriptional	parthian:=}');
Error('\P{Is_Blk=_Inscriptional	parthian:=}');
Expect(1, 68447, '\p{Is_Blk=inscriptionalparthian}', "");
Expect(0, 68447, '\p{^Is_Blk=inscriptionalparthian}', "");
Expect(0, 68447, '\P{Is_Blk=inscriptionalparthian}', "");
Expect(1, 68447, '\P{^Is_Blk=inscriptionalparthian}', "");
Expect(0, 68415, '\p{Is_Blk=inscriptionalparthian}', "");
Expect(1, 68415, '\p{^Is_Blk=inscriptionalparthian}', "");
Expect(1, 68415, '\P{Is_Blk=inscriptionalparthian}', "");
Expect(0, 68415, '\P{^Is_Blk=inscriptionalparthian}', "");
Expect(1, 68447, '\p{Is_Blk=-Inscriptionalparthian}', "");
Expect(0, 68447, '\p{^Is_Blk=-Inscriptionalparthian}', "");
Expect(0, 68447, '\P{Is_Blk=-Inscriptionalparthian}', "");
Expect(1, 68447, '\P{^Is_Blk=-Inscriptionalparthian}', "");
Expect(0, 68415, '\p{Is_Blk=-Inscriptionalparthian}', "");
Expect(1, 68415, '\p{^Is_Blk=-Inscriptionalparthian}', "");
Expect(1, 68415, '\P{Is_Blk=-Inscriptionalparthian}', "");
Expect(0, 68415, '\P{^Is_Blk=-Inscriptionalparthian}', "");
Error('\p{Block=:=UnifiedCANADIAN Aboriginal	syllabics}');
Error('\P{Block=:=UnifiedCANADIAN Aboriginal	syllabics}');
Expect(1, 5759, '\p{Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(0, 5759, '\p{^Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(0, 5759, '\P{Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(1, 5759, '\P{^Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(0, 5119, '\p{Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(1, 5119, '\p{^Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(1, 5119, '\P{Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(0, 5119, '\P{^Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(1, 5759, '\p{Block:  Unified	canadian-Aboriginal_SYLLABICS}', "");
Expect(0, 5759, '\p{^Block:  Unified	canadian-Aboriginal_SYLLABICS}', "");
Expect(0, 5759, '\P{Block:  Unified	canadian-Aboriginal_SYLLABICS}', "");
Expect(1, 5759, '\P{^Block:  Unified	canadian-Aboriginal_SYLLABICS}', "");
Expect(0, 5119, '\p{Block:  Unified	canadian-Aboriginal_SYLLABICS}', "");
Expect(1, 5119, '\p{^Block:  Unified	canadian-Aboriginal_SYLLABICS}', "");
Expect(1, 5119, '\P{Block:  Unified	canadian-Aboriginal_SYLLABICS}', "");
Expect(0, 5119, '\P{^Block:  Unified	canadian-Aboriginal_SYLLABICS}', "");
Error('\p{Blk::= _canadian-Syllabics}');
Error('\P{Blk::= _canadian-Syllabics}');
Expect(1, 5759, '\p{Blk=canadiansyllabics}', "");
Expect(0, 5759, '\p{^Blk=canadiansyllabics}', "");
Expect(0, 5759, '\P{Blk=canadiansyllabics}', "");
Expect(1, 5759, '\P{^Blk=canadiansyllabics}', "");
Expect(0, 5119, '\p{Blk=canadiansyllabics}', "");
Expect(1, 5119, '\p{^Blk=canadiansyllabics}', "");
Expect(1, 5119, '\P{Blk=canadiansyllabics}', "");
Expect(0, 5119, '\P{^Blk=canadiansyllabics}', "");
Expect(1, 5759, '\p{Blk:--CANADIAN_Syllabics}', "");
Expect(0, 5759, '\p{^Blk:--CANADIAN_Syllabics}', "");
Expect(0, 5759, '\P{Blk:--CANADIAN_Syllabics}', "");
Expect(1, 5759, '\P{^Blk:--CANADIAN_Syllabics}', "");
Expect(0, 5119, '\p{Blk:--CANADIAN_Syllabics}', "");
Expect(1, 5119, '\p{^Blk:--CANADIAN_Syllabics}', "");
Expect(1, 5119, '\P{Blk:--CANADIAN_Syllabics}', "");
Expect(0, 5119, '\P{^Blk:--CANADIAN_Syllabics}', "");
Error('\p{Is_Block=/a/UNIFIED_Canadian_ABORIGINAL syllabics}');
Error('\P{Is_Block=/a/UNIFIED_Canadian_ABORIGINAL syllabics}');
Expect(1, 5759, '\p{Is_Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(0, 5759, '\p{^Is_Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(0, 5759, '\P{Is_Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(1, 5759, '\P{^Is_Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(0, 5119, '\p{Is_Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(1, 5119, '\p{^Is_Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(1, 5119, '\P{Is_Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(0, 5119, '\P{^Is_Block=unifiedcanadianaboriginalsyllabics}', "");
Expect(1, 5759, '\p{Is_Block= -Unified CANADIAN	ABORIGINAL_syllabics}', "");
Expect(0, 5759, '\p{^Is_Block= -Unified CANADIAN	ABORIGINAL_syllabics}', "");
Expect(0, 5759, '\P{Is_Block= -Unified CANADIAN	ABORIGINAL_syllabics}', "");
Expect(1, 5759, '\P{^Is_Block= -Unified CANADIAN	ABORIGINAL_syllabics}', "");
Expect(0, 5119, '\p{Is_Block= -Unified CANADIAN	ABORIGINAL_syllabics}', "");
Expect(1, 5119, '\p{^Is_Block= -Unified CANADIAN	ABORIGINAL_syllabics}', "");
Expect(1, 5119, '\P{Is_Block= -Unified CANADIAN	ABORIGINAL_syllabics}', "");
Expect(0, 5119, '\P{^Is_Block= -Unified CANADIAN	ABORIGINAL_syllabics}', "");
Error('\p{Is_Blk=_:=Canadian-Syllabics}');
Error('\P{Is_Blk=_:=Canadian-Syllabics}');
Expect(1, 5759, '\p{Is_Blk=canadiansyllabics}', "");
Expect(0, 5759, '\p{^Is_Blk=canadiansyllabics}', "");
Expect(0, 5759, '\P{Is_Blk=canadiansyllabics}', "");
Expect(1, 5759, '\P{^Is_Blk=canadiansyllabics}', "");
Expect(0, 5119, '\p{Is_Blk=canadiansyllabics}', "");
Expect(1, 5119, '\p{^Is_Blk=canadiansyllabics}', "");
Expect(1, 5119, '\P{Is_Blk=canadiansyllabics}', "");
Expect(0, 5119, '\P{^Is_Blk=canadiansyllabics}', "");
Expect(1, 5759, '\p{Is_Blk:   	-canadiansyllabics}', "");
Expect(0, 5759, '\p{^Is_Blk:   	-canadiansyllabics}', "");
Expect(0, 5759, '\P{Is_Blk:   	-canadiansyllabics}', "");
Expect(1, 5759, '\P{^Is_Blk:   	-canadiansyllabics}', "");
Expect(0, 5119, '\p{Is_Blk:   	-canadiansyllabics}', "");
Expect(1, 5119, '\p{^Is_Blk:   	-canadiansyllabics}', "");
Expect(1, 5119, '\P{Is_Blk:   	-canadiansyllabics}', "");
Expect(0, 5119, '\P{^Is_Blk:   	-canadiansyllabics}', "");
Error('\p{Block=	_Variation:=Selectors}');
Error('\P{Block=	_Variation:=Selectors}');
Expect(1, 65039, '\p{Block:variationselectors}', "");
Expect(0, 65039, '\p{^Block:variationselectors}', "");
Expect(0, 65039, '\P{Block:variationselectors}', "");
Expect(1, 65039, '\P{^Block:variationselectors}', "");
Expect(0, 65023, '\p{Block:variationselectors}', "");
Expect(1, 65023, '\p{^Block:variationselectors}', "");
Expect(1, 65023, '\P{Block:variationselectors}', "");
Expect(0, 65023, '\P{^Block:variationselectors}', "");
Expect(1, 65039, '\p{Block= _variation SELECTORS}', "");
Expect(0, 65039, '\p{^Block= _variation SELECTORS}', "");
Expect(0, 65039, '\P{Block= _variation SELECTORS}', "");
Expect(1, 65039, '\P{^Block= _variation SELECTORS}', "");
Expect(0, 65023, '\p{Block= _variation SELECTORS}', "");
Expect(1, 65023, '\p{^Block= _variation SELECTORS}', "");
Expect(1, 65023, '\P{Block= _variation SELECTORS}', "");
Expect(0, 65023, '\P{^Block= _variation SELECTORS}', "");
Error('\p{Blk= 	Variation/a/SELECTORS}');
Error('\P{Blk= 	Variation/a/SELECTORS}');
Expect(1, 65039, '\p{Blk:variationselectors}', "");
Expect(0, 65039, '\p{^Blk:variationselectors}', "");
Expect(0, 65039, '\P{Blk:variationselectors}', "");
Expect(1, 65039, '\P{^Blk:variationselectors}', "");
Expect(0, 65023, '\p{Blk:variationselectors}', "");
Expect(1, 65023, '\p{^Blk:variationselectors}', "");
Expect(1, 65023, '\P{Blk:variationselectors}', "");
Expect(0, 65023, '\P{^Blk:variationselectors}', "");
Expect(1, 65039, '\p{Blk= _variation_Selectors}', "");
Expect(0, 65039, '\p{^Blk= _variation_Selectors}', "");
Expect(0, 65039, '\P{Blk= _variation_Selectors}', "");
Expect(1, 65039, '\P{^Blk= _variation_Selectors}', "");
Expect(0, 65023, '\p{Blk= _variation_Selectors}', "");
Expect(1, 65023, '\p{^Blk= _variation_Selectors}', "");
Expect(1, 65023, '\P{Blk= _variation_Selectors}', "");
Expect(0, 65023, '\P{^Blk= _variation_Selectors}', "");
Error('\p{Is_Block=:=_	VARIATION-Selectors}');
Error('\P{Is_Block=:=_	VARIATION-Selectors}');
Expect(1, 65039, '\p{Is_Block=variationselectors}', "");
Expect(0, 65039, '\p{^Is_Block=variationselectors}', "");
Expect(0, 65039, '\P{Is_Block=variationselectors}', "");
Expect(1, 65039, '\P{^Is_Block=variationselectors}', "");
Expect(0, 65023, '\p{Is_Block=variationselectors}', "");
Expect(1, 65023, '\p{^Is_Block=variationselectors}', "");
Expect(1, 65023, '\P{Is_Block=variationselectors}', "");
Expect(0, 65023, '\P{^Is_Block=variationselectors}', "");
Expect(1, 65039, '\p{Is_Block=	VARIATIONSelectors}', "");
Expect(0, 65039, '\p{^Is_Block=	VARIATIONSelectors}', "");
Expect(0, 65039, '\P{Is_Block=	VARIATIONSelectors}', "");
Expect(1, 65039, '\P{^Is_Block=	VARIATIONSelectors}', "");
Expect(0, 65023, '\p{Is_Block=	VARIATIONSelectors}', "");
Expect(1, 65023, '\p{^Is_Block=	VARIATIONSelectors}', "");
Expect(1, 65023, '\P{Is_Block=	VARIATIONSelectors}', "");
Expect(0, 65023, '\P{^Is_Block=	VARIATIONSelectors}', "");
Error('\p{Is_Blk=/a/-	VARIATION	SELECTORS}');
Error('\P{Is_Blk=/a/-	VARIATION	SELECTORS}');
Expect(1, 65039, '\p{Is_Blk=variationselectors}', "");
Expect(0, 65039, '\p{^Is_Blk=variationselectors}', "");
Expect(0, 65039, '\P{Is_Blk=variationselectors}', "");
Expect(1, 65039, '\P{^Is_Blk=variationselectors}', "");
Expect(0, 65023, '\p{Is_Blk=variationselectors}', "");
Expect(1, 65023, '\p{^Is_Blk=variationselectors}', "");
Expect(1, 65023, '\P{Is_Blk=variationselectors}', "");
Expect(0, 65023, '\P{^Is_Blk=variationselectors}', "");
Expect(1, 65039, '\p{Is_Blk= Variation-Selectors}', "");
Expect(0, 65039, '\p{^Is_Blk= Variation-Selectors}', "");
Expect(0, 65039, '\P{Is_Blk= Variation-Selectors}', "");
Expect(1, 65039, '\P{^Is_Blk= Variation-Selectors}', "");
Expect(0, 65023, '\p{Is_Blk= Variation-Selectors}', "");
Expect(1, 65023, '\p{^Is_Blk= Variation-Selectors}', "");
Expect(1, 65023, '\P{Is_Blk= Variation-Selectors}', "");
Expect(0, 65023, '\P{^Is_Blk= Variation-Selectors}', "");
Error('\p{Block=	 number:=forms}');
Error('\P{Block=	 number:=forms}');
Expect(1, 8591, '\p{Block=numberforms}', "");
Expect(0, 8591, '\p{^Block=numberforms}', "");
Expect(0, 8591, '\P{Block=numberforms}', "");
Expect(1, 8591, '\P{^Block=numberforms}', "");
Expect(0, 8527, '\p{Block=numberforms}', "");
Expect(1, 8527, '\p{^Block=numberforms}', "");
Expect(1, 8527, '\P{Block=numberforms}', "");
Expect(0, 8527, '\P{^Block=numberforms}', "");
Expect(1, 8591, '\p{Block=- number	Forms}', "");
Expect(0, 8591, '\p{^Block=- number	Forms}', "");
Expect(0, 8591, '\P{Block=- number	Forms}', "");
Expect(1, 8591, '\P{^Block=- number	Forms}', "");
Expect(0, 8527, '\p{Block=- number	Forms}', "");
Expect(1, 8527, '\p{^Block=- number	Forms}', "");
Expect(1, 8527, '\P{Block=- number	Forms}', "");
Expect(0, 8527, '\P{^Block=- number	Forms}', "");
Error('\p{Blk=/a/	-NUMBER	Forms}');
Error('\P{Blk=/a/	-NUMBER	Forms}');
Expect(1, 8591, '\p{Blk=numberforms}', "");
Expect(0, 8591, '\p{^Blk=numberforms}', "");
Expect(0, 8591, '\P{Blk=numberforms}', "");
Expect(1, 8591, '\P{^Blk=numberforms}', "");
Expect(0, 8527, '\p{Blk=numberforms}', "");
Expect(1, 8527, '\p{^Blk=numberforms}', "");
Expect(1, 8527, '\P{Blk=numberforms}', "");
Expect(0, 8527, '\P{^Blk=numberforms}', "");
Expect(1, 8591, '\p{Blk= 	NUMBERForms}', "");
Expect(0, 8591, '\p{^Blk= 	NUMBERForms}', "");
Expect(0, 8591, '\P{Blk= 	NUMBERForms}', "");
Expect(1, 8591, '\P{^Blk= 	NUMBERForms}', "");
Expect(0, 8527, '\p{Blk= 	NUMBERForms}', "");
Expect(1, 8527, '\p{^Blk= 	NUMBERForms}', "");
Expect(1, 8527, '\P{Blk= 	NUMBERForms}', "");
Expect(0, 8527, '\P{^Blk= 	NUMBERForms}', "");
Error('\p{Is_Block=-NUMBER/a/Forms}');
Error('\P{Is_Block=-NUMBER/a/Forms}');
Expect(1, 8591, '\p{Is_Block=numberforms}', "");
Expect(0, 8591, '\p{^Is_Block=numberforms}', "");
Expect(0, 8591, '\P{Is_Block=numberforms}', "");
Expect(1, 8591, '\P{^Is_Block=numberforms}', "");
Expect(0, 8527, '\p{Is_Block=numberforms}', "");
Expect(1, 8527, '\p{^Is_Block=numberforms}', "");
Expect(1, 8527, '\P{Is_Block=numberforms}', "");
Expect(0, 8527, '\P{^Is_Block=numberforms}', "");
Expect(1, 8591, '\p{Is_Block=	NUMBER_Forms}', "");
Expect(0, 8591, '\p{^Is_Block=	NUMBER_Forms}', "");
Expect(0, 8591, '\P{Is_Block=	NUMBER_Forms}', "");
Expect(1, 8591, '\P{^Is_Block=	NUMBER_Forms}', "");
Expect(0, 8527, '\p{Is_Block=	NUMBER_Forms}', "");
Expect(1, 8527, '\p{^Is_Block=	NUMBER_Forms}', "");
Expect(1, 8527, '\P{Is_Block=	NUMBER_Forms}', "");
Expect(0, 8527, '\P{^Is_Block=	NUMBER_Forms}', "");
Error('\p{Is_Blk=_/a/Number	FORMS}');
Error('\P{Is_Blk=_/a/Number	FORMS}');
Expect(1, 8591, '\p{Is_Blk=numberforms}', "");
Expect(0, 8591, '\p{^Is_Blk=numberforms}', "");
Expect(0, 8591, '\P{Is_Blk=numberforms}', "");
Expect(1, 8591, '\P{^Is_Blk=numberforms}', "");
Expect(0, 8527, '\p{Is_Blk=numberforms}', "");
Expect(1, 8527, '\p{^Is_Blk=numberforms}', "");
Expect(1, 8527, '\P{Is_Blk=numberforms}', "");
Expect(0, 8527, '\P{^Is_Blk=numberforms}', "");
Expect(1, 8591, '\p{Is_Blk=-_NUMBERForms}', "");
Expect(0, 8591, '\p{^Is_Blk=-_NUMBERForms}', "");
Expect(0, 8591, '\P{Is_Blk=-_NUMBERForms}', "");
Expect(1, 8591, '\P{^Is_Blk=-_NUMBERForms}', "");
Expect(0, 8527, '\p{Is_Blk=-_NUMBERForms}', "");
Expect(1, 8527, '\p{^Is_Blk=-_NUMBERForms}', "");
Expect(1, 8527, '\P{Is_Blk=-_NUMBERForms}', "");
Expect(0, 8527, '\P{^Is_Blk=-_NUMBERForms}', "");
Error('\p{Block=_Control_PICTURES:=}');
Error('\P{Block=_Control_PICTURES:=}');
Expect(1, 9279, '\p{Block=controlpictures}', "");
Expect(0, 9279, '\p{^Block=controlpictures}', "");
Expect(0, 9279, '\P{Block=controlpictures}', "");
Expect(1, 9279, '\P{^Block=controlpictures}', "");
Expect(0, 9215, '\p{Block=controlpictures}', "");
Expect(1, 9215, '\p{^Block=controlpictures}', "");
Expect(1, 9215, '\P{Block=controlpictures}', "");
Expect(0, 9215, '\P{^Block=controlpictures}', "");
Expect(1, 9279, '\p{Block:   	CONTROL-Pictures}', "");
Expect(0, 9279, '\p{^Block:   	CONTROL-Pictures}', "");
Expect(0, 9279, '\P{Block:   	CONTROL-Pictures}', "");
Expect(1, 9279, '\P{^Block:   	CONTROL-Pictures}', "");
Expect(0, 9215, '\p{Block:   	CONTROL-Pictures}', "");
Expect(1, 9215, '\p{^Block:   	CONTROL-Pictures}', "");
Expect(1, 9215, '\P{Block:   	CONTROL-Pictures}', "");
Expect(0, 9215, '\P{^Block:   	CONTROL-Pictures}', "");
Error('\p{Blk=	-control:=pictures}');
Error('\P{Blk=	-control:=pictures}');
Expect(1, 9279, '\p{Blk=controlpictures}', "");
Expect(0, 9279, '\p{^Blk=controlpictures}', "");
Expect(0, 9279, '\P{Blk=controlpictures}', "");
Expect(1, 9279, '\P{^Blk=controlpictures}', "");
Expect(0, 9215, '\p{Blk=controlpictures}', "");
Expect(1, 9215, '\p{^Blk=controlpictures}', "");
Expect(1, 9215, '\P{Blk=controlpictures}', "");
Expect(0, 9215, '\P{^Blk=controlpictures}', "");
Expect(1, 9279, '\p{Blk:-Control	Pictures}', "");
Expect(0, 9279, '\p{^Blk:-Control	Pictures}', "");
Expect(0, 9279, '\P{Blk:-Control	Pictures}', "");
Expect(1, 9279, '\P{^Blk:-Control	Pictures}', "");
Expect(0, 9215, '\p{Blk:-Control	Pictures}', "");
Expect(1, 9215, '\p{^Blk:-Control	Pictures}', "");
Expect(1, 9215, '\P{Blk:-Control	Pictures}', "");
Expect(0, 9215, '\P{^Blk:-Control	Pictures}', "");
Error('\p{Is_Block= -ControlPictures:=}');
Error('\P{Is_Block= -ControlPictures:=}');
Expect(1, 9279, '\p{Is_Block=controlpictures}', "");
Expect(0, 9279, '\p{^Is_Block=controlpictures}', "");
Expect(0, 9279, '\P{Is_Block=controlpictures}', "");
Expect(1, 9279, '\P{^Is_Block=controlpictures}', "");
Expect(0, 9215, '\p{Is_Block=controlpictures}', "");
Expect(1, 9215, '\p{^Is_Block=controlpictures}', "");
Expect(1, 9215, '\P{Is_Block=controlpictures}', "");
Expect(0, 9215, '\P{^Is_Block=controlpictures}', "");
Expect(1, 9279, '\p{Is_Block=-	CONTROL	PICTURES}', "");
Expect(0, 9279, '\p{^Is_Block=-	CONTROL	PICTURES}', "");
Expect(0, 9279, '\P{Is_Block=-	CONTROL	PICTURES}', "");
Expect(1, 9279, '\P{^Is_Block=-	CONTROL	PICTURES}', "");
Expect(0, 9215, '\p{Is_Block=-	CONTROL	PICTURES}', "");
Expect(1, 9215, '\p{^Is_Block=-	CONTROL	PICTURES}', "");
Expect(1, 9215, '\P{Is_Block=-	CONTROL	PICTURES}', "");
Expect(0, 9215, '\P{^Is_Block=-	CONTROL	PICTURES}', "");
Error('\p{Is_Blk=	_CONTROL-Pictures:=}');
Error('\P{Is_Blk=	_CONTROL-Pictures:=}');
Expect(1, 9279, '\p{Is_Blk=controlpictures}', "");
Expect(0, 9279, '\p{^Is_Blk=controlpictures}', "");
Expect(0, 9279, '\P{Is_Blk=controlpictures}', "");
Expect(1, 9279, '\P{^Is_Blk=controlpictures}', "");
Expect(0, 9215, '\p{Is_Blk=controlpictures}', "");
Expect(1, 9215, '\p{^Is_Blk=controlpictures}', "");
Expect(1, 9215, '\P{Is_Blk=controlpictures}', "");
Expect(0, 9215, '\P{^Is_Blk=controlpictures}', "");
Expect(1, 9279, '\p{Is_Blk=  Control	PICTURES}', "");
Expect(0, 9279, '\p{^Is_Blk=  Control	PICTURES}', "");
Expect(0, 9279, '\P{Is_Blk=  Control	PICTURES}', "");
Expect(1, 9279, '\P{^Is_Blk=  Control	PICTURES}', "");
Expect(0, 9215, '\p{Is_Blk=  Control	PICTURES}', "");
Expect(1, 9215, '\p{^Is_Blk=  Control	PICTURES}', "");
Expect(1, 9215, '\P{Is_Blk=  Control	PICTURES}', "");
Expect(0, 9215, '\P{^Is_Blk=  Control	PICTURES}', "");
Error('\p{Block=		High-Private Use-surrogates/a/}');
Error('\P{Block=		High-Private Use-surrogates/a/}');
Expect(1, 56319, '\p{Block=highprivateusesurrogates}', "");
Expect(0, 56319, '\p{^Block=highprivateusesurrogates}', "");
Expect(0, 56319, '\P{Block=highprivateusesurrogates}', "");
Expect(1, 56319, '\P{^Block=highprivateusesurrogates}', "");
Expect(0, 1, '\p{Block=highprivateusesurrogates}', "");
Expect(1, 1, '\p{^Block=highprivateusesurrogates}', "");
Expect(1, 1, '\P{Block=highprivateusesurrogates}', "");
Expect(0, 1, '\P{^Block=highprivateusesurrogates}', "");
Expect(1, 56319, '\p{Block=  High-private-Use	Surrogates}', "");
Expect(0, 56319, '\p{^Block=  High-private-Use	Surrogates}', "");
Expect(0, 56319, '\P{Block=  High-private-Use	Surrogates}', "");
Expect(1, 56319, '\P{^Block=  High-private-Use	Surrogates}', "");
Expect(0, 1, '\p{Block=  High-private-Use	Surrogates}', "");
Expect(1, 1, '\p{^Block=  High-private-Use	Surrogates}', "");
Expect(1, 1, '\P{Block=  High-private-Use	Surrogates}', "");
Expect(0, 1, '\P{^Block=  High-private-Use	Surrogates}', "");
Error('\p{Blk=	/a/High_privateUse-Surrogates}');
Error('\P{Blk=	/a/High_privateUse-Surrogates}');
Expect(1, 56319, '\p{Blk=highprivateusesurrogates}', "");
Expect(0, 56319, '\p{^Blk=highprivateusesurrogates}', "");
Expect(0, 56319, '\P{Blk=highprivateusesurrogates}', "");
Expect(1, 56319, '\P{^Blk=highprivateusesurrogates}', "");
Expect(0, 1, '\p{Blk=highprivateusesurrogates}', "");
Expect(1, 1, '\p{^Blk=highprivateusesurrogates}', "");
Expect(1, 1, '\P{Blk=highprivateusesurrogates}', "");
Expect(0, 1, '\P{^Blk=highprivateusesurrogates}', "");
Expect(1, 56319, '\p{Blk=-high_PrivateUse_SURROGATES}', "");
Expect(0, 56319, '\p{^Blk=-high_PrivateUse_SURROGATES}', "");
Expect(0, 56319, '\P{Blk=-high_PrivateUse_SURROGATES}', "");
Expect(1, 56319, '\P{^Blk=-high_PrivateUse_SURROGATES}', "");
Expect(0, 1, '\p{Blk=-high_PrivateUse_SURROGATES}', "");
Expect(1, 1, '\p{^Blk=-high_PrivateUse_SURROGATES}', "");
Expect(1, 1, '\P{Blk=-high_PrivateUse_SURROGATES}', "");
Expect(0, 1, '\P{^Blk=-high_PrivateUse_SURROGATES}', "");
Error('\p{Is_Block=_:=HIGH-PRIVATE-USE surrogates}');
Error('\P{Is_Block=_:=HIGH-PRIVATE-USE surrogates}');
Expect(1, 56319, '\p{Is_Block=highprivateusesurrogates}', "");
Expect(0, 56319, '\p{^Is_Block=highprivateusesurrogates}', "");
Expect(0, 56319, '\P{Is_Block=highprivateusesurrogates}', "");
Expect(1, 56319, '\P{^Is_Block=highprivateusesurrogates}', "");
Expect(0, 1, '\p{Is_Block=highprivateusesurrogates}', "");
Expect(1, 1, '\p{^Is_Block=highprivateusesurrogates}', "");
Expect(1, 1, '\P{Is_Block=highprivateusesurrogates}', "");
Expect(0, 1, '\P{^Is_Block=highprivateusesurrogates}', "");
Expect(1, 56319, '\p{Is_Block=- High-Private	Use SURROGATES}', "");
Expect(0, 56319, '\p{^Is_Block=- High-Private	Use SURROGATES}', "");
Expect(0, 56319, '\P{Is_Block=- High-Private	Use SURROGATES}', "");
Expect(1, 56319, '\P{^Is_Block=- High-Private	Use SURROGATES}', "");
Expect(0, 1, '\p{Is_Block=- High-Private	Use SURROGATES}', "");
Expect(1, 1, '\p{^Is_Block=- High-Private	Use SURROGATES}', "");
Expect(1, 1, '\P{Is_Block=- High-Private	Use SURROGATES}', "");
Expect(0, 1, '\P{^Is_Block=- High-Private	Use SURROGATES}', "");
Error('\p{Is_Blk=_-HIGH/a/Private Use	surrogates}');
Error('\P{Is_Blk=_-HIGH/a/Private Use	surrogates}');
Expect(1, 56319, '\p{Is_Blk=highprivateusesurrogates}', "");
Expect(0, 56319, '\p{^Is_Blk=highprivateusesurrogates}', "");
Expect(0, 56319, '\P{Is_Blk=highprivateusesurrogates}', "");
Expect(1, 56319, '\P{^Is_Blk=highprivateusesurrogates}', "");
Expect(0, 1, '\p{Is_Blk=highprivateusesurrogates}', "");
Expect(1, 1, '\p{^Is_Blk=highprivateusesurrogates}', "");
Expect(1, 1, '\P{Is_Blk=highprivateusesurrogates}', "");
Expect(0, 1, '\P{^Is_Blk=highprivateusesurrogates}', "");
Expect(1, 56319, '\p{Is_Blk= high-PRIVATE-use surrogates}', "");
Expect(0, 56319, '\p{^Is_Blk= high-PRIVATE-use surrogates}', "");
Expect(0, 56319, '\P{Is_Blk= high-PRIVATE-use surrogates}', "");
Expect(1, 56319, '\P{^Is_Blk= high-PRIVATE-use surrogates}', "");
Expect(0, 1, '\p{Is_Blk= high-PRIVATE-use surrogates}', "");
Expect(1, 1, '\p{^Is_Blk= high-PRIVATE-use surrogates}', "");
Expect(1, 1, '\P{Is_Blk= high-PRIVATE-use surrogates}', "");
Expect(0, 1, '\P{^Is_Blk= high-PRIVATE-use surrogates}', "");
Error('\p{Block:   :=ugaritic}');
Error('\P{Block:   :=ugaritic}');
Expect(1, 66463, '\p{Block=ugaritic}', "");
Expect(0, 66463, '\p{^Block=ugaritic}', "");
Expect(0, 66463, '\P{Block=ugaritic}', "");
Expect(1, 66463, '\P{^Block=ugaritic}', "");
Expect(0, 66431, '\p{Block=ugaritic}', "");
Expect(1, 66431, '\p{^Block=ugaritic}', "");
Expect(1, 66431, '\P{Block=ugaritic}', "");
Expect(0, 66431, '\P{^Block=ugaritic}', "");
Expect(1, 66463, '\p{Block=-	UGARITIC}', "");
Expect(0, 66463, '\p{^Block=-	UGARITIC}', "");
Expect(0, 66463, '\P{Block=-	UGARITIC}', "");
Expect(1, 66463, '\P{^Block=-	UGARITIC}', "");
Expect(0, 66431, '\p{Block=-	UGARITIC}', "");
Expect(1, 66431, '\p{^Block=-	UGARITIC}', "");
Expect(1, 66431, '\P{Block=-	UGARITIC}', "");
Expect(0, 66431, '\P{^Block=-	UGARITIC}', "");
Error('\p{Blk=  UGARITIC/a/}');
Error('\P{Blk=  UGARITIC/a/}');
Expect(1, 66463, '\p{Blk=ugaritic}', "");
Expect(0, 66463, '\p{^Blk=ugaritic}', "");
Expect(0, 66463, '\P{Blk=ugaritic}', "");
Expect(1, 66463, '\P{^Blk=ugaritic}', "");
Expect(0, 66431, '\p{Blk=ugaritic}', "");
Expect(1, 66431, '\p{^Blk=ugaritic}', "");
Expect(1, 66431, '\P{Blk=ugaritic}', "");
Expect(0, 66431, '\P{^Blk=ugaritic}', "");
Expect(1, 66463, '\p{Blk=  ugaritic}', "");
Expect(0, 66463, '\p{^Blk=  ugaritic}', "");
Expect(0, 66463, '\P{Blk=  ugaritic}', "");
Expect(1, 66463, '\P{^Blk=  ugaritic}', "");
Expect(0, 66431, '\p{Blk=  ugaritic}', "");
Expect(1, 66431, '\p{^Blk=  ugaritic}', "");
Expect(1, 66431, '\P{Blk=  ugaritic}', "");
Expect(0, 66431, '\P{^Blk=  ugaritic}', "");
Error('\p{Is_Block= :=UGARITIC}');
Error('\P{Is_Block= :=UGARITIC}');
Expect(1, 66463, '\p{Is_Block=ugaritic}', "");
Expect(0, 66463, '\p{^Is_Block=ugaritic}', "");
Expect(0, 66463, '\P{Is_Block=ugaritic}', "");
Expect(1, 66463, '\P{^Is_Block=ugaritic}', "");
Expect(0, 66431, '\p{Is_Block=ugaritic}', "");
Expect(1, 66431, '\p{^Is_Block=ugaritic}', "");
Expect(1, 66431, '\P{Is_Block=ugaritic}', "");
Expect(0, 66431, '\P{^Is_Block=ugaritic}', "");
Expect(1, 66463, '\p{Is_Block=_-UGARITIC}', "");
Expect(0, 66463, '\p{^Is_Block=_-UGARITIC}', "");
Expect(0, 66463, '\P{Is_Block=_-UGARITIC}', "");
Expect(1, 66463, '\P{^Is_Block=_-UGARITIC}', "");
Expect(0, 66431, '\p{Is_Block=_-UGARITIC}', "");
Expect(1, 66431, '\p{^Is_Block=_-UGARITIC}', "");
Expect(1, 66431, '\P{Is_Block=_-UGARITIC}', "");
Expect(0, 66431, '\P{^Is_Block=_-UGARITIC}', "");
Error('\p{Is_Blk=/a/	_Ugaritic}');
Error('\P{Is_Blk=/a/	_Ugaritic}');
Expect(1, 66463, '\p{Is_Blk=ugaritic}', "");
Expect(0, 66463, '\p{^Is_Blk=ugaritic}', "");
Expect(0, 66463, '\P{Is_Blk=ugaritic}', "");
Expect(1, 66463, '\P{^Is_Blk=ugaritic}', "");
Expect(0, 66431, '\p{Is_Blk=ugaritic}', "");
Expect(1, 66431, '\p{^Is_Blk=ugaritic}', "");
Expect(1, 66431, '\P{Is_Blk=ugaritic}', "");
Expect(0, 66431, '\P{^Is_Blk=ugaritic}', "");
Expect(1, 66463, '\p{Is_Blk=_ Ugaritic}', "");
Expect(0, 66463, '\p{^Is_Blk=_ Ugaritic}', "");
Expect(0, 66463, '\P{Is_Blk=_ Ugaritic}', "");
Expect(1, 66463, '\P{^Is_Blk=_ Ugaritic}', "");
Expect(0, 66431, '\p{Is_Blk=_ Ugaritic}', "");
Expect(1, 66431, '\p{^Is_Blk=_ Ugaritic}', "");
Expect(1, 66431, '\P{Is_Blk=_ Ugaritic}', "");
Expect(0, 66431, '\P{^Is_Blk=_ Ugaritic}', "");
Error('\p{Block=:=  ARMENIAN}');
Error('\P{Block=:=  ARMENIAN}');
Expect(1, 1423, '\p{Block=armenian}', "");
Expect(0, 1423, '\p{^Block=armenian}', "");
Expect(0, 1423, '\P{Block=armenian}', "");
Expect(1, 1423, '\P{^Block=armenian}', "");
Expect(0, 1327, '\p{Block=armenian}', "");
Expect(1, 1327, '\p{^Block=armenian}', "");
Expect(1, 1327, '\P{Block=armenian}', "");
Expect(0, 1327, '\P{^Block=armenian}', "");
Expect(1, 1423, '\p{Block=  Armenian}', "");
Expect(0, 1423, '\p{^Block=  Armenian}', "");
Expect(0, 1423, '\P{Block=  Armenian}', "");
Expect(1, 1423, '\P{^Block=  Armenian}', "");
Expect(0, 1327, '\p{Block=  Armenian}', "");
Expect(1, 1327, '\p{^Block=  Armenian}', "");
Expect(1, 1327, '\P{Block=  Armenian}', "");
Expect(0, 1327, '\P{^Block=  Armenian}', "");
Error('\p{Blk=/a/_-ARMENIAN}');
Error('\P{Blk=/a/_-ARMENIAN}');
Expect(1, 1423, '\p{Blk=armenian}', "");
Expect(0, 1423, '\p{^Blk=armenian}', "");
Expect(0, 1423, '\P{Blk=armenian}', "");
Expect(1, 1423, '\P{^Blk=armenian}', "");
Expect(0, 1327, '\p{Blk=armenian}', "");
Expect(1, 1327, '\p{^Blk=armenian}', "");
Expect(1, 1327, '\P{Blk=armenian}', "");
Expect(0, 1327, '\P{^Blk=armenian}', "");
Expect(1, 1423, '\p{Blk:  ARMENIAN}', "");
Expect(0, 1423, '\p{^Blk:  ARMENIAN}', "");
Expect(0, 1423, '\P{Blk:  ARMENIAN}', "");
Expect(1, 1423, '\P{^Blk:  ARMENIAN}', "");
Expect(0, 1327, '\p{Blk:  ARMENIAN}', "");
Expect(1, 1327, '\p{^Blk:  ARMENIAN}', "");
Expect(1, 1327, '\P{Blk:  ARMENIAN}', "");
Expect(0, 1327, '\P{^Blk:  ARMENIAN}', "");
Error('\p{Is_Block=/a/ ARMENIAN}');
Error('\P{Is_Block=/a/ ARMENIAN}');
Expect(1, 1423, '\p{Is_Block=armenian}', "");
Expect(0, 1423, '\p{^Is_Block=armenian}', "");
Expect(0, 1423, '\P{Is_Block=armenian}', "");
Expect(1, 1423, '\P{^Is_Block=armenian}', "");
Expect(0, 1327, '\p{Is_Block=armenian}', "");
Expect(1, 1327, '\p{^Is_Block=armenian}', "");
Expect(1, 1327, '\P{Is_Block=armenian}', "");
Expect(0, 1327, '\P{^Is_Block=armenian}', "");
Expect(1, 1423, '\p{Is_Block=	-Armenian}', "");
Expect(0, 1423, '\p{^Is_Block=	-Armenian}', "");
Expect(0, 1423, '\P{Is_Block=	-Armenian}', "");
Expect(1, 1423, '\P{^Is_Block=	-Armenian}', "");
Expect(0, 1327, '\p{Is_Block=	-Armenian}', "");
Expect(1, 1327, '\p{^Is_Block=	-Armenian}', "");
Expect(1, 1327, '\P{Is_Block=	-Armenian}', "");
Expect(0, 1327, '\P{^Is_Block=	-Armenian}', "");
Error('\p{Is_Blk=/a/	 Armenian}');
Error('\P{Is_Blk=/a/	 Armenian}');
Expect(1, 1423, '\p{Is_Blk=armenian}', "");
Expect(0, 1423, '\p{^Is_Blk=armenian}', "");
Expect(0, 1423, '\P{Is_Blk=armenian}', "");
Expect(1, 1423, '\P{^Is_Blk=armenian}', "");
Expect(0, 1327, '\p{Is_Blk=armenian}', "");
Expect(1, 1327, '\p{^Is_Blk=armenian}', "");
Expect(1, 1327, '\P{Is_Blk=armenian}', "");
Expect(0, 1327, '\P{^Is_Blk=armenian}', "");
Expect(1, 1423, '\p{Is_Blk=-ARMENIAN}', "");
Expect(0, 1423, '\p{^Is_Blk=-ARMENIAN}', "");
Expect(0, 1423, '\P{Is_Blk=-ARMENIAN}', "");
Expect(1, 1423, '\P{^Is_Blk=-ARMENIAN}', "");
Expect(0, 1327, '\p{Is_Blk=-ARMENIAN}', "");
Expect(1, 1327, '\p{^Is_Blk=-ARMENIAN}', "");
Expect(1, 1327, '\P{Is_Blk=-ARMENIAN}', "");
Expect(0, 1327, '\P{^Is_Blk=-ARMENIAN}', "");
Error('\p{Block= 	private	use	Area/a/}');
Error('\P{Block= 	private	use	Area/a/}');
Expect(1, 63743, '\p{Block=privateusearea}', "");
Expect(0, 63743, '\p{^Block=privateusearea}', "");
Expect(0, 63743, '\P{Block=privateusearea}', "");
Expect(1, 63743, '\P{^Block=privateusearea}', "");
Expect(0, 1, '\p{Block=privateusearea}', "");
Expect(1, 1, '\p{^Block=privateusearea}', "");
Expect(1, 1, '\P{Block=privateusearea}', "");
Expect(0, 1, '\P{^Block=privateusearea}', "");
Expect(1, 63743, '\p{Block:	-Private Use-AREA}', "");
Expect(0, 63743, '\p{^Block:	-Private Use-AREA}', "");
Expect(0, 63743, '\P{Block:	-Private Use-AREA}', "");
Expect(1, 63743, '\P{^Block:	-Private Use-AREA}', "");
Expect(0, 1, '\p{Block:	-Private Use-AREA}', "");
Expect(1, 1, '\p{^Block:	-Private Use-AREA}', "");
Expect(1, 1, '\P{Block:	-Private Use-AREA}', "");
Expect(0, 1, '\P{^Block:	-Private Use-AREA}', "");
Error('\p{Blk=	private:=Use}');
Error('\P{Blk=	private:=Use}');
Expect(1, 63743, '\p{Blk=privateuse}', "");
Expect(0, 63743, '\p{^Blk=privateuse}', "");
Expect(0, 63743, '\P{Blk=privateuse}', "");
Expect(1, 63743, '\P{^Blk=privateuse}', "");
Expect(0, 1, '\p{Blk=privateuse}', "");
Expect(1, 1, '\p{^Blk=privateuse}', "");
Expect(1, 1, '\P{Blk=privateuse}', "");
Expect(0, 1, '\P{^Blk=privateuse}', "");
Expect(1, 63743, '\p{Blk=-_Private-Use}', "");
Expect(0, 63743, '\p{^Blk=-_Private-Use}', "");
Expect(0, 63743, '\P{Blk=-_Private-Use}', "");
Expect(1, 63743, '\P{^Blk=-_Private-Use}', "");
Expect(0, 1, '\p{Blk=-_Private-Use}', "");
Expect(1, 1, '\p{^Blk=-_Private-Use}', "");
Expect(1, 1, '\P{Blk=-_Private-Use}', "");
Expect(0, 1, '\P{^Blk=-_Private-Use}', "");
Error('\p{Is_Block=_:=PRIVATE-Use area}');
Error('\P{Is_Block=_:=PRIVATE-Use area}');
Expect(1, 63743, '\p{Is_Block=privateusearea}', "");
Expect(0, 63743, '\p{^Is_Block=privateusearea}', "");
Expect(0, 63743, '\P{Is_Block=privateusearea}', "");
Expect(1, 63743, '\P{^Is_Block=privateusearea}', "");
Expect(0, 1, '\p{Is_Block=privateusearea}', "");
Expect(1, 1, '\p{^Is_Block=privateusearea}', "");
Expect(1, 1, '\P{Is_Block=privateusearea}', "");
Expect(0, 1, '\P{^Is_Block=privateusearea}', "");
Expect(1, 63743, '\p{Is_Block=	-private-useArea}', "");
Expect(0, 63743, '\p{^Is_Block=	-private-useArea}', "");
Expect(0, 63743, '\P{Is_Block=	-private-useArea}', "");
Expect(1, 63743, '\P{^Is_Block=	-private-useArea}', "");
Expect(0, 1, '\p{Is_Block=	-private-useArea}', "");
Expect(1, 1, '\p{^Is_Block=	-private-useArea}', "");
Expect(1, 1, '\P{Is_Block=	-private-useArea}', "");
Expect(0, 1, '\P{^Is_Block=	-private-useArea}', "");
Error('\p{Is_Blk=/a/  Private	Use}');
Error('\P{Is_Blk=/a/  Private	Use}');
Expect(1, 63743, '\p{Is_Blk=privateuse}', "");
Expect(0, 63743, '\p{^Is_Blk=privateuse}', "");
Expect(0, 63743, '\P{Is_Blk=privateuse}', "");
Expect(1, 63743, '\P{^Is_Blk=privateuse}', "");
Expect(0, 1, '\p{Is_Blk=privateuse}', "");
Expect(1, 1, '\p{^Is_Blk=privateuse}', "");
Expect(1, 1, '\P{Is_Blk=privateuse}', "");
Expect(0, 1, '\P{^Is_Blk=privateuse}', "");
Expect(1, 63743, '\p{Is_Blk=__privateuse}', "");
Expect(0, 63743, '\p{^Is_Blk=__privateuse}', "");
Expect(0, 63743, '\P{Is_Blk=__privateuse}', "");
Expect(1, 63743, '\P{^Is_Blk=__privateuse}', "");
Expect(0, 1, '\p{Is_Blk=__privateuse}', "");
Expect(1, 1, '\p{^Is_Blk=__privateuse}', "");
Expect(1, 1, '\P{Is_Blk=__privateuse}', "");
Expect(0, 1, '\P{^Is_Blk=__privateuse}', "");
Error('\p{Block=/a/_ Syriac}');
Error('\P{Block=/a/_ Syriac}');
Expect(1, 1871, '\p{Block=syriac}', "");
Expect(0, 1871, '\p{^Block=syriac}', "");
Expect(0, 1871, '\P{Block=syriac}', "");
Expect(1, 1871, '\P{^Block=syriac}', "");
Expect(0, 1791, '\p{Block=syriac}', "");
Expect(1, 1791, '\p{^Block=syriac}', "");
Expect(1, 1791, '\P{Block=syriac}', "");
Expect(0, 1791, '\P{^Block=syriac}', "");
Expect(1, 1871, '\p{Block= 	Syriac}', "");
Expect(0, 1871, '\p{^Block= 	Syriac}', "");
Expect(0, 1871, '\P{Block= 	Syriac}', "");
Expect(1, 1871, '\P{^Block= 	Syriac}', "");
Expect(0, 1791, '\p{Block= 	Syriac}', "");
Expect(1, 1791, '\p{^Block= 	Syriac}', "");
Expect(1, 1791, '\P{Block= 	Syriac}', "");
Expect(0, 1791, '\P{^Block= 	Syriac}', "");
Error('\p{Blk=:=_ Syriac}');
Error('\P{Blk=:=_ Syriac}');
Expect(1, 1871, '\p{Blk=syriac}', "");
Expect(0, 1871, '\p{^Blk=syriac}', "");
Expect(0, 1871, '\P{Blk=syriac}', "");
Expect(1, 1871, '\P{^Blk=syriac}', "");
Expect(0, 1791, '\p{Blk=syriac}', "");
Expect(1, 1791, '\p{^Blk=syriac}', "");
Expect(1, 1791, '\P{Blk=syriac}', "");
Expect(0, 1791, '\P{^Blk=syriac}', "");
Expect(1, 1871, '\p{Blk:	Syriac}', "");
Expect(0, 1871, '\p{^Blk:	Syriac}', "");
Expect(0, 1871, '\P{Blk:	Syriac}', "");
Expect(1, 1871, '\P{^Blk:	Syriac}', "");
Expect(0, 1791, '\p{Blk:	Syriac}', "");
Expect(1, 1791, '\p{^Blk:	Syriac}', "");
Expect(1, 1791, '\P{Blk:	Syriac}', "");
Expect(0, 1791, '\P{^Blk:	Syriac}', "");
Error('\p{Is_Block=/a/ 	Syriac}');
Error('\P{Is_Block=/a/ 	Syriac}');
Expect(1, 1871, '\p{Is_Block=syriac}', "");
Expect(0, 1871, '\p{^Is_Block=syriac}', "");
Expect(0, 1871, '\P{Is_Block=syriac}', "");
Expect(1, 1871, '\P{^Is_Block=syriac}', "");
Expect(0, 1791, '\p{Is_Block=syriac}', "");
Expect(1, 1791, '\p{^Is_Block=syriac}', "");
Expect(1, 1791, '\P{Is_Block=syriac}', "");
Expect(0, 1791, '\P{^Is_Block=syriac}', "");
Expect(1, 1871, '\p{Is_Block=	 Syriac}', "");
Expect(0, 1871, '\p{^Is_Block=	 Syriac}', "");
Expect(0, 1871, '\P{Is_Block=	 Syriac}', "");
Expect(1, 1871, '\P{^Is_Block=	 Syriac}', "");
Expect(0, 1791, '\p{Is_Block=	 Syriac}', "");
Expect(1, 1791, '\p{^Is_Block=	 Syriac}', "");
Expect(1, 1791, '\P{Is_Block=	 Syriac}', "");
Expect(0, 1791, '\P{^Is_Block=	 Syriac}', "");
Error('\p{Is_Blk: _syriac:=}');
Error('\P{Is_Blk: _syriac:=}');
Expect(1, 1871, '\p{Is_Blk=syriac}', "");
Expect(0, 1871, '\p{^Is_Blk=syriac}', "");
Expect(0, 1871, '\P{Is_Blk=syriac}', "");
Expect(1, 1871, '\P{^Is_Blk=syriac}', "");
Expect(0, 1791, '\p{Is_Blk=syriac}', "");
Expect(1, 1791, '\p{^Is_Blk=syriac}', "");
Expect(1, 1791, '\P{Is_Blk=syriac}', "");
Expect(0, 1791, '\P{^Is_Blk=syriac}', "");
Expect(1, 1871, '\p{Is_Blk=	_SYRIAC}', "");
Expect(0, 1871, '\p{^Is_Blk=	_SYRIAC}', "");
Expect(0, 1871, '\P{Is_Blk=	_SYRIAC}', "");
Expect(1, 1871, '\P{^Is_Blk=	_SYRIAC}', "");
Expect(0, 1791, '\p{Is_Blk=	_SYRIAC}', "");
Expect(1, 1791, '\p{^Is_Blk=	_SYRIAC}', "");
Expect(1, 1791, '\P{Is_Blk=	_SYRIAC}', "");
Expect(0, 1791, '\P{^Is_Blk=	_SYRIAC}', "");
Error('\p{Block= -Ancient-Greek_Musical:=NOTATION}');
Error('\P{Block= -Ancient-Greek_Musical:=NOTATION}');
Expect(1, 119375, '\p{Block=ancientgreekmusicalnotation}', "");
Expect(0, 119375, '\p{^Block=ancientgreekmusicalnotation}', "");
Expect(0, 119375, '\P{Block=ancientgreekmusicalnotation}', "");
Expect(1, 119375, '\P{^Block=ancientgreekmusicalnotation}', "");
Expect(0, 119295, '\p{Block=ancientgreekmusicalnotation}', "");
Expect(1, 119295, '\p{^Block=ancientgreekmusicalnotation}', "");
Expect(1, 119295, '\P{Block=ancientgreekmusicalnotation}', "");
Expect(0, 119295, '\P{^Block=ancientgreekmusicalnotation}', "");
Expect(1, 119375, '\p{Block=  ANCIENT	greek	MUSICAL	notation}', "");
Expect(0, 119375, '\p{^Block=  ANCIENT	greek	MUSICAL	notation}', "");
Expect(0, 119375, '\P{Block=  ANCIENT	greek	MUSICAL	notation}', "");
Expect(1, 119375, '\P{^Block=  ANCIENT	greek	MUSICAL	notation}', "");
Expect(0, 119295, '\p{Block=  ANCIENT	greek	MUSICAL	notation}', "");
Expect(1, 119295, '\p{^Block=  ANCIENT	greek	MUSICAL	notation}', "");
Expect(1, 119295, '\P{Block=  ANCIENT	greek	MUSICAL	notation}', "");
Expect(0, 119295, '\P{^Block=  ANCIENT	greek	MUSICAL	notation}', "");
Error('\p{Blk=	ANCIENTGREEK-Musical	notation/a/}');
Error('\P{Blk=	ANCIENTGREEK-Musical	notation/a/}');
Expect(1, 119375, '\p{Blk:ancientgreekmusicalnotation}', "");
Expect(0, 119375, '\p{^Blk:ancientgreekmusicalnotation}', "");
Expect(0, 119375, '\P{Blk:ancientgreekmusicalnotation}', "");
Expect(1, 119375, '\P{^Blk:ancientgreekmusicalnotation}', "");
Expect(0, 119295, '\p{Blk:ancientgreekmusicalnotation}', "");
Expect(1, 119295, '\p{^Blk:ancientgreekmusicalnotation}', "");
Expect(1, 119295, '\P{Blk:ancientgreekmusicalnotation}', "");
Expect(0, 119295, '\P{^Blk:ancientgreekmusicalnotation}', "");
Expect(1, 119375, '\p{Blk=--Ancient_GREEK-Musical Notation}', "");
Expect(0, 119375, '\p{^Blk=--Ancient_GREEK-Musical Notation}', "");
Expect(0, 119375, '\P{Blk=--Ancient_GREEK-Musical Notation}', "");
Expect(1, 119375, '\P{^Blk=--Ancient_GREEK-Musical Notation}', "");
Expect(0, 119295, '\p{Blk=--Ancient_GREEK-Musical Notation}', "");
Expect(1, 119295, '\p{^Blk=--Ancient_GREEK-Musical Notation}', "");
Expect(1, 119295, '\P{Blk=--Ancient_GREEK-Musical Notation}', "");
Expect(0, 119295, '\P{^Blk=--Ancient_GREEK-Musical Notation}', "");
Error('\p{Is_Block=ANCIENT Greek Musical:=NOTATION}');
Error('\P{Is_Block=ANCIENT Greek Musical:=NOTATION}');
Expect(1, 119375, '\p{Is_Block=ancientgreekmusicalnotation}', "");
Expect(0, 119375, '\p{^Is_Block=ancientgreekmusicalnotation}', "");
Expect(0, 119375, '\P{Is_Block=ancientgreekmusicalnotation}', "");
Expect(1, 119375, '\P{^Is_Block=ancientgreekmusicalnotation}', "");
Expect(0, 119295, '\p{Is_Block=ancientgreekmusicalnotation}', "");
Expect(1, 119295, '\p{^Is_Block=ancientgreekmusicalnotation}', "");
Expect(1, 119295, '\P{Is_Block=ancientgreekmusicalnotation}', "");
Expect(0, 119295, '\P{^Is_Block=ancientgreekmusicalnotation}', "");
Expect(1, 119375, '\p{Is_Block=  Ancient-GREEK_musical-notation}', "");
Expect(0, 119375, '\p{^Is_Block=  Ancient-GREEK_musical-notation}', "");
Expect(0, 119375, '\P{Is_Block=  Ancient-GREEK_musical-notation}', "");
Expect(1, 119375, '\P{^Is_Block=  Ancient-GREEK_musical-notation}', "");
Expect(0, 119295, '\p{Is_Block=  Ancient-GREEK_musical-notation}', "");
Expect(1, 119295, '\p{^Is_Block=  Ancient-GREEK_musical-notation}', "");
Expect(1, 119295, '\P{Is_Block=  Ancient-GREEK_musical-notation}', "");
Expect(0, 119295, '\P{^Is_Block=  Ancient-GREEK_musical-notation}', "");
Error('\p{Is_Blk:	 _ancientGreek/a/musical-Notation}');
Error('\P{Is_Blk:	 _ancientGreek/a/musical-Notation}');
Expect(1, 119375, '\p{Is_Blk=ancientgreekmusicalnotation}', "");
Expect(0, 119375, '\p{^Is_Blk=ancientgreekmusicalnotation}', "");
Expect(0, 119375, '\P{Is_Blk=ancientgreekmusicalnotation}', "");
Expect(1, 119375, '\P{^Is_Blk=ancientgreekmusicalnotation}', "");
Expect(0, 119295, '\p{Is_Blk=ancientgreekmusicalnotation}', "");
Expect(1, 119295, '\p{^Is_Blk=ancientgreekmusicalnotation}', "");
Expect(1, 119295, '\P{Is_Blk=ancientgreekmusicalnotation}', "");
Expect(0, 119295, '\P{^Is_Blk=ancientgreekmusicalnotation}', "");
Expect(1, 119375, '\p{Is_Blk:    	Ancient Greek Musical_Notation}', "");
Expect(0, 119375, '\p{^Is_Blk:    	Ancient Greek Musical_Notation}', "");
Expect(0, 119375, '\P{Is_Blk:    	Ancient Greek Musical_Notation}', "");
Expect(1, 119375, '\P{^Is_Blk:    	Ancient Greek Musical_Notation}', "");
Expect(0, 119295, '\p{Is_Blk:    	Ancient Greek Musical_Notation}', "");
Expect(1, 119295, '\p{^Is_Blk:    	Ancient Greek Musical_Notation}', "");
Expect(1, 119295, '\P{Is_Blk:    	Ancient Greek Musical_Notation}', "");
Expect(0, 119295, '\P{^Is_Blk:    	Ancient Greek Musical_Notation}', "");
Error('\p{Block=/a/_ phags	pa}');
Error('\P{Block=/a/_ phags	pa}');
Expect(1, 43135, '\p{Block=phagspa}', "");
Expect(0, 43135, '\p{^Block=phagspa}', "");
Expect(0, 43135, '\P{Block=phagspa}', "");
Expect(1, 43135, '\P{^Block=phagspa}', "");
Expect(0, 43071, '\p{Block=phagspa}', "");
Expect(1, 43071, '\p{^Block=phagspa}', "");
Expect(1, 43071, '\P{Block=phagspa}', "");
Expect(0, 43071, '\P{^Block=phagspa}', "");
Expect(1, 43135, '\p{Block=-phags_Pa}', "");
Expect(0, 43135, '\p{^Block=-phags_Pa}', "");
Expect(0, 43135, '\P{Block=-phags_Pa}', "");
Expect(1, 43135, '\P{^Block=-phags_Pa}', "");
Expect(0, 43071, '\p{Block=-phags_Pa}', "");
Expect(1, 43071, '\p{^Block=-phags_Pa}', "");
Expect(1, 43071, '\P{Block=-phags_Pa}', "");
Expect(0, 43071, '\P{^Block=-phags_Pa}', "");
Error('\p{Blk=	:=PHAGS_Pa}');
Error('\P{Blk=	:=PHAGS_Pa}');
Expect(1, 43135, '\p{Blk:phagspa}', "");
Expect(0, 43135, '\p{^Blk:phagspa}', "");
Expect(0, 43135, '\P{Blk:phagspa}', "");
Expect(1, 43135, '\P{^Blk:phagspa}', "");
Expect(0, 43071, '\p{Blk:phagspa}', "");
Expect(1, 43071, '\p{^Blk:phagspa}', "");
Expect(1, 43071, '\P{Blk:phagspa}', "");
Expect(0, 43071, '\P{^Blk:phagspa}', "");
Expect(1, 43135, '\p{Blk=_ PHAGS-Pa}', "");
Expect(0, 43135, '\p{^Blk=_ PHAGS-Pa}', "");
Expect(0, 43135, '\P{Blk=_ PHAGS-Pa}', "");
Expect(1, 43135, '\P{^Blk=_ PHAGS-Pa}', "");
Expect(0, 43071, '\p{Blk=_ PHAGS-Pa}', "");
Expect(1, 43071, '\p{^Blk=_ PHAGS-Pa}', "");
Expect(1, 43071, '\P{Blk=_ PHAGS-Pa}', "");
Expect(0, 43071, '\P{^Blk=_ PHAGS-Pa}', "");
Error('\p{Is_Block=/a/_Phags Pa}');
Error('\P{Is_Block=/a/_Phags Pa}');
Expect(1, 43135, '\p{Is_Block=phagspa}', "");
Expect(0, 43135, '\p{^Is_Block=phagspa}', "");
Expect(0, 43135, '\P{Is_Block=phagspa}', "");
Expect(1, 43135, '\P{^Is_Block=phagspa}', "");
Expect(0, 43071, '\p{Is_Block=phagspa}', "");
Expect(1, 43071, '\p{^Is_Block=phagspa}', "");
Expect(1, 43071, '\P{Is_Block=phagspa}', "");
Expect(0, 43071, '\P{^Is_Block=phagspa}', "");
Expect(1, 43135, '\p{Is_Block=		PHAGSPA}', "");
Expect(0, 43135, '\p{^Is_Block=		PHAGSPA}', "");
Expect(0, 43135, '\P{Is_Block=		PHAGSPA}', "");
Expect(1, 43135, '\P{^Is_Block=		PHAGSPA}', "");
Expect(0, 43071, '\p{Is_Block=		PHAGSPA}', "");
Expect(1, 43071, '\p{^Is_Block=		PHAGSPA}', "");
Expect(1, 43071, '\P{Is_Block=		PHAGSPA}', "");
Expect(0, 43071, '\P{^Is_Block=		PHAGSPA}', "");
Error('\p{Is_Blk=:=__Phags	pa}');
Error('\P{Is_Blk=:=__Phags	pa}');
Expect(1, 43135, '\p{Is_Blk=phagspa}', "");
Expect(0, 43135, '\p{^Is_Blk=phagspa}', "");
Expect(0, 43135, '\P{Is_Blk=phagspa}', "");
Expect(1, 43135, '\P{^Is_Blk=phagspa}', "");
Expect(0, 43071, '\p{Is_Blk=phagspa}', "");
Expect(1, 43071, '\p{^Is_Blk=phagspa}', "");
Expect(1, 43071, '\P{Is_Blk=phagspa}', "");
Expect(0, 43071, '\P{^Is_Blk=phagspa}', "");
Expect(1, 43135, '\p{Is_Blk=		PHAGSPa}', "");
Expect(0, 43135, '\p{^Is_Blk=		PHAGSPa}', "");
Expect(0, 43135, '\P{Is_Blk=		PHAGSPa}', "");
Expect(1, 43135, '\P{^Is_Blk=		PHAGSPa}', "");
Expect(0, 43071, '\p{Is_Blk=		PHAGSPa}', "");
Expect(1, 43071, '\p{^Is_Blk=		PHAGSPa}', "");
Expect(1, 43071, '\P{Is_Blk=		PHAGSPa}', "");
Expect(0, 43071, '\P{^Is_Blk=		PHAGSPa}', "");
Error('\p{Block=	/a/CJK compatibility Ideographs}');
Error('\P{Block=	/a/CJK compatibility Ideographs}');
Expect(1, 64255, '\p{Block=cjkcompatibilityideographs}', "");
Expect(0, 64255, '\p{^Block=cjkcompatibilityideographs}', "");
Expect(0, 64255, '\P{Block=cjkcompatibilityideographs}', "");
Expect(1, 64255, '\P{^Block=cjkcompatibilityideographs}', "");
Expect(0, 63743, '\p{Block=cjkcompatibilityideographs}', "");
Expect(1, 63743, '\p{^Block=cjkcompatibilityideographs}', "");
Expect(1, 63743, '\P{Block=cjkcompatibilityideographs}', "");
Expect(0, 63743, '\P{^Block=cjkcompatibilityideographs}', "");
Expect(1, 64255, '\p{Block:   _ cjk_CompatibilityIdeographs}', "");
Expect(0, 64255, '\p{^Block:   _ cjk_CompatibilityIdeographs}', "");
Expect(0, 64255, '\P{Block:   _ cjk_CompatibilityIdeographs}', "");
Expect(1, 64255, '\P{^Block:   _ cjk_CompatibilityIdeographs}', "");
Expect(0, 63743, '\p{Block:   _ cjk_CompatibilityIdeographs}', "");
Expect(1, 63743, '\p{^Block:   _ cjk_CompatibilityIdeographs}', "");
Expect(1, 63743, '\P{Block:   _ cjk_CompatibilityIdeographs}', "");
Expect(0, 63743, '\P{^Block:   _ cjk_CompatibilityIdeographs}', "");
Error('\p{Blk=/a/ _cjk_CompatibilityIdeographs}');
Error('\P{Blk=/a/ _cjk_CompatibilityIdeographs}');
Expect(1, 64255, '\p{Blk=cjkcompatibilityideographs}', "");
Expect(0, 64255, '\p{^Blk=cjkcompatibilityideographs}', "");
Expect(0, 64255, '\P{Blk=cjkcompatibilityideographs}', "");
Expect(1, 64255, '\P{^Blk=cjkcompatibilityideographs}', "");
Expect(0, 63743, '\p{Blk=cjkcompatibilityideographs}', "");
Expect(1, 63743, '\p{^Blk=cjkcompatibilityideographs}', "");
Expect(1, 63743, '\P{Blk=cjkcompatibilityideographs}', "");
Expect(0, 63743, '\P{^Blk=cjkcompatibilityideographs}', "");
Expect(1, 64255, '\p{Blk=_	CJK_compatibilityIDEOGRAPHS}', "");
Expect(0, 64255, '\p{^Blk=_	CJK_compatibilityIDEOGRAPHS}', "");
Expect(0, 64255, '\P{Blk=_	CJK_compatibilityIDEOGRAPHS}', "");
Expect(1, 64255, '\P{^Blk=_	CJK_compatibilityIDEOGRAPHS}', "");
Expect(0, 63743, '\p{Blk=_	CJK_compatibilityIDEOGRAPHS}', "");
Expect(1, 63743, '\p{^Blk=_	CJK_compatibilityIDEOGRAPHS}', "");
Expect(1, 63743, '\P{Blk=_	CJK_compatibilityIDEOGRAPHS}', "");
Expect(0, 63743, '\P{^Blk=_	CJK_compatibilityIDEOGRAPHS}', "");
Error('\p{Is_Block=  CJK COMPATIBILITY/a/ideographs}');
Error('\P{Is_Block=  CJK COMPATIBILITY/a/ideographs}');
Expect(1, 64255, '\p{Is_Block=cjkcompatibilityideographs}', "");
Expect(0, 64255, '\p{^Is_Block=cjkcompatibilityideographs}', "");
Expect(0, 64255, '\P{Is_Block=cjkcompatibilityideographs}', "");
Expect(1, 64255, '\P{^Is_Block=cjkcompatibilityideographs}', "");
Expect(0, 63743, '\p{Is_Block=cjkcompatibilityideographs}', "");
Expect(1, 63743, '\p{^Is_Block=cjkcompatibilityideographs}', "");
Expect(1, 63743, '\P{Is_Block=cjkcompatibilityideographs}', "");
Expect(0, 63743, '\P{^Is_Block=cjkcompatibilityideographs}', "");
Expect(1, 64255, '\p{Is_Block=cjkCompatibilityIDEOGRAPHS}', "");
Expect(0, 64255, '\p{^Is_Block=cjkCompatibilityIDEOGRAPHS}', "");
Expect(0, 64255, '\P{Is_Block=cjkCompatibilityIDEOGRAPHS}', "");
Expect(1, 64255, '\P{^Is_Block=cjkCompatibilityIDEOGRAPHS}', "");
Expect(0, 63743, '\p{Is_Block=cjkCompatibilityIDEOGRAPHS}', "");
Expect(1, 63743, '\p{^Is_Block=cjkCompatibilityIDEOGRAPHS}', "");
Expect(1, 63743, '\P{Is_Block=cjkCompatibilityIDEOGRAPHS}', "");
Expect(0, 63743, '\P{^Is_Block=cjkCompatibilityIDEOGRAPHS}', "");
Error('\p{Is_Blk=/a/-	CJK-Compatibility ideographs}');
Error('\P{Is_Blk=/a/-	CJK-Compatibility ideographs}');
Expect(1, 64255, '\p{Is_Blk=cjkcompatibilityideographs}', "");
Expect(0, 64255, '\p{^Is_Blk=cjkcompatibilityideographs}', "");
Expect(0, 64255, '\P{Is_Blk=cjkcompatibilityideographs}', "");
Expect(1, 64255, '\P{^Is_Blk=cjkcompatibilityideographs}', "");
Expect(0, 63743, '\p{Is_Blk=cjkcompatibilityideographs}', "");
Expect(1, 63743, '\p{^Is_Blk=cjkcompatibilityideographs}', "");
Expect(1, 63743, '\P{Is_Blk=cjkcompatibilityideographs}', "");
Expect(0, 63743, '\P{^Is_Blk=cjkcompatibilityideographs}', "");
Expect(1, 64255, '\p{Is_Blk=--CJKcompatibility Ideographs}', "");
Expect(0, 64255, '\p{^Is_Blk=--CJKcompatibility Ideographs}', "");
Expect(0, 64255, '\P{Is_Blk=--CJKcompatibility Ideographs}', "");
Expect(1, 64255, '\P{^Is_Blk=--CJKcompatibility Ideographs}', "");
Expect(0, 63743, '\p{Is_Blk=--CJKcompatibility Ideographs}', "");
Expect(1, 63743, '\p{^Is_Blk=--CJKcompatibility Ideographs}', "");
Expect(1, 63743, '\P{Is_Blk=--CJKcompatibility Ideographs}', "");
Expect(0, 63743, '\P{^Is_Blk=--CJKcompatibility Ideographs}', "");
Error('\p{Block=:=_	Thaana}');
Error('\P{Block=:=_	Thaana}');
Expect(1, 1983, '\p{Block=thaana}', "");
Expect(0, 1983, '\p{^Block=thaana}', "");
Expect(0, 1983, '\P{Block=thaana}', "");
Expect(1, 1983, '\P{^Block=thaana}', "");
Expect(0, 1919, '\p{Block=thaana}', "");
Expect(1, 1919, '\p{^Block=thaana}', "");
Expect(1, 1919, '\P{Block=thaana}', "");
Expect(0, 1919, '\P{^Block=thaana}', "");
Expect(1, 1983, '\p{Block=	-Thaana}', "");
Expect(0, 1983, '\p{^Block=	-Thaana}', "");
Expect(0, 1983, '\P{Block=	-Thaana}', "");
Expect(1, 1983, '\P{^Block=	-Thaana}', "");
Expect(0, 1919, '\p{Block=	-Thaana}', "");
Expect(1, 1919, '\p{^Block=	-Thaana}', "");
Expect(1, 1919, '\P{Block=	-Thaana}', "");
Expect(0, 1919, '\P{^Block=	-Thaana}', "");
Error('\p{Blk=-	THAANA:=}');
Error('\P{Blk=-	THAANA:=}');
Expect(1, 1983, '\p{Blk=thaana}', "");
Expect(0, 1983, '\p{^Blk=thaana}', "");
Expect(0, 1983, '\P{Blk=thaana}', "");
Expect(1, 1983, '\P{^Blk=thaana}', "");
Expect(0, 1919, '\p{Blk=thaana}', "");
Expect(1, 1919, '\p{^Blk=thaana}', "");
Expect(1, 1919, '\P{Blk=thaana}', "");
Expect(0, 1919, '\P{^Blk=thaana}', "");
Expect(1, 1983, '\p{Blk=- thaana}', "");
Expect(0, 1983, '\p{^Blk=- thaana}', "");
Expect(0, 1983, '\P{Blk=- thaana}', "");
Expect(1, 1983, '\P{^Blk=- thaana}', "");
Expect(0, 1919, '\p{Blk=- thaana}', "");
Expect(1, 1919, '\p{^Blk=- thaana}', "");
Expect(1, 1919, '\P{Blk=- thaana}', "");
Expect(0, 1919, '\P{^Blk=- thaana}', "");
Error('\p{Is_Block: :=	thaana}');
Error('\P{Is_Block: :=	thaana}');
Expect(1, 1983, '\p{Is_Block:   thaana}', "");
Expect(0, 1983, '\p{^Is_Block:   thaana}', "");
Expect(0, 1983, '\P{Is_Block:   thaana}', "");
Expect(1, 1983, '\P{^Is_Block:   thaana}', "");
Expect(0, 1919, '\p{Is_Block:   thaana}', "");
Expect(1, 1919, '\p{^Is_Block:   thaana}', "");
Expect(1, 1919, '\P{Is_Block:   thaana}', "");
Expect(0, 1919, '\P{^Is_Block:   thaana}', "");
Expect(1, 1983, '\p{Is_Block=-THAANA}', "");
Expect(0, 1983, '\p{^Is_Block=-THAANA}', "");
Expect(0, 1983, '\P{Is_Block=-THAANA}', "");
Expect(1, 1983, '\P{^Is_Block=-THAANA}', "");
Expect(0, 1919, '\p{Is_Block=-THAANA}', "");
Expect(1, 1919, '\p{^Is_Block=-THAANA}', "");
Expect(1, 1919, '\P{Is_Block=-THAANA}', "");
Expect(0, 1919, '\P{^Is_Block=-THAANA}', "");
Error('\p{Is_Blk=_ thaana:=}');
Error('\P{Is_Blk=_ thaana:=}');
Expect(1, 1983, '\p{Is_Blk=thaana}', "");
Expect(0, 1983, '\p{^Is_Blk=thaana}', "");
Expect(0, 1983, '\P{Is_Blk=thaana}', "");
Expect(1, 1983, '\P{^Is_Blk=thaana}', "");
Expect(0, 1919, '\p{Is_Blk=thaana}', "");
Expect(1, 1919, '\p{^Is_Blk=thaana}', "");
Expect(1, 1919, '\P{Is_Blk=thaana}', "");
Expect(0, 1919, '\P{^Is_Blk=thaana}', "");
Expect(1, 1983, '\p{Is_Blk=	thaana}', "");
Expect(0, 1983, '\p{^Is_Blk=	thaana}', "");
Expect(0, 1983, '\P{Is_Blk=	thaana}', "");
Expect(1, 1983, '\P{^Is_Blk=	thaana}', "");
Expect(0, 1919, '\p{Is_Blk=	thaana}', "");
Expect(1, 1919, '\p{^Is_Blk=	thaana}', "");
Expect(1, 1919, '\P{Is_Blk=	thaana}', "");
Expect(0, 1919, '\P{^Is_Blk=	thaana}', "");
Error('\p{Block=_-Enclosed Alphanumeric:=Supplement}');
Error('\P{Block=_-Enclosed Alphanumeric:=Supplement}');
Expect(1, 127487, '\p{Block:enclosedalphanumericsupplement}', "");
Expect(0, 127487, '\p{^Block:enclosedalphanumericsupplement}', "");
Expect(0, 127487, '\P{Block:enclosedalphanumericsupplement}', "");
Expect(1, 127487, '\P{^Block:enclosedalphanumericsupplement}', "");
Expect(0, 127231, '\p{Block:enclosedalphanumericsupplement}', "");
Expect(1, 127231, '\p{^Block:enclosedalphanumericsupplement}', "");
Expect(1, 127231, '\P{Block:enclosedalphanumericsupplement}', "");
Expect(0, 127231, '\P{^Block:enclosedalphanumericsupplement}', "");
Expect(1, 127487, '\p{Block=_enclosed_alphanumeric	supplement}', "");
Expect(0, 127487, '\p{^Block=_enclosed_alphanumeric	supplement}', "");
Expect(0, 127487, '\P{Block=_enclosed_alphanumeric	supplement}', "");
Expect(1, 127487, '\P{^Block=_enclosed_alphanumeric	supplement}', "");
Expect(0, 127231, '\p{Block=_enclosed_alphanumeric	supplement}', "");
Expect(1, 127231, '\p{^Block=_enclosed_alphanumeric	supplement}', "");
Expect(1, 127231, '\P{Block=_enclosed_alphanumeric	supplement}', "");
Expect(0, 127231, '\P{^Block=_enclosed_alphanumeric	supplement}', "");
Error('\p{Blk=:=-enclosed Alphanumeric-Supplement}');
Error('\P{Blk=:=-enclosed Alphanumeric-Supplement}');
Expect(1, 127487, '\p{Blk=enclosedalphanumericsupplement}', "");
Expect(0, 127487, '\p{^Blk=enclosedalphanumericsupplement}', "");
Expect(0, 127487, '\P{Blk=enclosedalphanumericsupplement}', "");
Expect(1, 127487, '\P{^Blk=enclosedalphanumericsupplement}', "");
Expect(0, 127231, '\p{Blk=enclosedalphanumericsupplement}', "");
Expect(1, 127231, '\p{^Blk=enclosedalphanumericsupplement}', "");
Expect(1, 127231, '\P{Blk=enclosedalphanumericsupplement}', "");
Expect(0, 127231, '\P{^Blk=enclosedalphanumericsupplement}', "");
Expect(1, 127487, '\p{Blk=--Enclosed	Alphanumeric supplement}', "");
Expect(0, 127487, '\p{^Blk=--Enclosed	Alphanumeric supplement}', "");
Expect(0, 127487, '\P{Blk=--Enclosed	Alphanumeric supplement}', "");
Expect(1, 127487, '\P{^Blk=--Enclosed	Alphanumeric supplement}', "");
Expect(0, 127231, '\p{Blk=--Enclosed	Alphanumeric supplement}', "");
Expect(1, 127231, '\p{^Blk=--Enclosed	Alphanumeric supplement}', "");
Expect(1, 127231, '\P{Blk=--Enclosed	Alphanumeric supplement}', "");
Expect(0, 127231, '\P{^Blk=--Enclosed	Alphanumeric supplement}', "");
Error('\p{Is_Block=--Enclosed-Alphanumeric_Supplement/a/}');
Error('\P{Is_Block=--Enclosed-Alphanumeric_Supplement/a/}');
Expect(1, 127487, '\p{Is_Block=enclosedalphanumericsupplement}', "");
Expect(0, 127487, '\p{^Is_Block=enclosedalphanumericsupplement}', "");
Expect(0, 127487, '\P{Is_Block=enclosedalphanumericsupplement}', "");
Expect(1, 127487, '\P{^Is_Block=enclosedalphanumericsupplement}', "");
Expect(0, 127231, '\p{Is_Block=enclosedalphanumericsupplement}', "");
Expect(1, 127231, '\p{^Is_Block=enclosedalphanumericsupplement}', "");
Expect(1, 127231, '\P{Is_Block=enclosedalphanumericsupplement}', "");
Expect(0, 127231, '\P{^Is_Block=enclosedalphanumericsupplement}', "");
Expect(1, 127487, '\p{Is_Block:  EnclosedAlphanumeric	Supplement}', "");
Expect(0, 127487, '\p{^Is_Block:  EnclosedAlphanumeric	Supplement}', "");
Expect(0, 127487, '\P{Is_Block:  EnclosedAlphanumeric	Supplement}', "");
Expect(1, 127487, '\P{^Is_Block:  EnclosedAlphanumeric	Supplement}', "");
Expect(0, 127231, '\p{Is_Block:  EnclosedAlphanumeric	Supplement}', "");
Expect(1, 127231, '\p{^Is_Block:  EnclosedAlphanumeric	Supplement}', "");
Expect(1, 127231, '\P{Is_Block:  EnclosedAlphanumeric	Supplement}', "");
Expect(0, 127231, '\P{^Is_Block:  EnclosedAlphanumeric	Supplement}', "");
Error('\p{Is_Blk=	Enclosed/a/alphanumericSUPPLEMENT}');
Error('\P{Is_Blk=	Enclosed/a/alphanumericSUPPLEMENT}');
Expect(1, 127487, '\p{Is_Blk=enclosedalphanumericsupplement}', "");
Expect(0, 127487, '\p{^Is_Blk=enclosedalphanumericsupplement}', "");
Expect(0, 127487, '\P{Is_Blk=enclosedalphanumericsupplement}', "");
Expect(1, 127487, '\P{^Is_Blk=enclosedalphanumericsupplement}', "");
Expect(0, 127231, '\p{Is_Blk=enclosedalphanumericsupplement}', "");
Expect(1, 127231, '\p{^Is_Blk=enclosedalphanumericsupplement}', "");
Expect(1, 127231, '\P{Is_Blk=enclosedalphanumericsupplement}', "");
Expect(0, 127231, '\P{^Is_Blk=enclosedalphanumericsupplement}', "");
Expect(1, 127487, '\p{Is_Blk=_-EnclosedAlphanumeric	SUPPLEMENT}', "");
Expect(0, 127487, '\p{^Is_Blk=_-EnclosedAlphanumeric	SUPPLEMENT}', "");
Expect(0, 127487, '\P{Is_Blk=_-EnclosedAlphanumeric	SUPPLEMENT}', "");
Expect(1, 127487, '\P{^Is_Blk=_-EnclosedAlphanumeric	SUPPLEMENT}', "");
Expect(0, 127231, '\p{Is_Blk=_-EnclosedAlphanumeric	SUPPLEMENT}', "");
Expect(1, 127231, '\p{^Is_Blk=_-EnclosedAlphanumeric	SUPPLEMENT}', "");
Expect(1, 127231, '\P{Is_Blk=_-EnclosedAlphanumeric	SUPPLEMENT}', "");
Expect(0, 127231, '\P{^Is_Blk=_-EnclosedAlphanumeric	SUPPLEMENT}', "");
Error('\p{Block=	:=ethiopic	EXTENDED}');
Error('\P{Block=	:=ethiopic	EXTENDED}');
Expect(1, 11743, '\p{Block=ethiopicextended}', "");
Expect(0, 11743, '\p{^Block=ethiopicextended}', "");
Expect(0, 11743, '\P{Block=ethiopicextended}', "");
Expect(1, 11743, '\P{^Block=ethiopicextended}', "");
Expect(0, 11647, '\p{Block=ethiopicextended}', "");
Expect(1, 11647, '\p{^Block=ethiopicextended}', "");
Expect(1, 11647, '\P{Block=ethiopicextended}', "");
Expect(0, 11647, '\P{^Block=ethiopicextended}', "");
Expect(1, 11743, '\p{Block=	-ethiopic extended}', "");
Expect(0, 11743, '\p{^Block=	-ethiopic extended}', "");
Expect(0, 11743, '\P{Block=	-ethiopic extended}', "");
Expect(1, 11743, '\P{^Block=	-ethiopic extended}', "");
Expect(0, 11647, '\p{Block=	-ethiopic extended}', "");
Expect(1, 11647, '\p{^Block=	-ethiopic extended}', "");
Expect(1, 11647, '\P{Block=	-ethiopic extended}', "");
Expect(0, 11647, '\P{^Block=	-ethiopic extended}', "");
Error('\p{Blk=	Ethiopic/a/EXTENDED}');
Error('\P{Blk=	Ethiopic/a/EXTENDED}');
Expect(1, 11743, '\p{Blk:	ethiopicextended}', "");
Expect(0, 11743, '\p{^Blk:	ethiopicextended}', "");
Expect(0, 11743, '\P{Blk:	ethiopicextended}', "");
Expect(1, 11743, '\P{^Blk:	ethiopicextended}', "");
Expect(0, 11647, '\p{Blk:	ethiopicextended}', "");
Expect(1, 11647, '\p{^Blk:	ethiopicextended}', "");
Expect(1, 11647, '\P{Blk:	ethiopicextended}', "");
Expect(0, 11647, '\P{^Blk:	ethiopicextended}', "");
Expect(1, 11743, '\p{Blk:   - Ethiopic-EXTENDED}', "");
Expect(0, 11743, '\p{^Blk:   - Ethiopic-EXTENDED}', "");
Expect(0, 11743, '\P{Blk:   - Ethiopic-EXTENDED}', "");
Expect(1, 11743, '\P{^Blk:   - Ethiopic-EXTENDED}', "");
Expect(0, 11647, '\p{Blk:   - Ethiopic-EXTENDED}', "");
Expect(1, 11647, '\p{^Blk:   - Ethiopic-EXTENDED}', "");
Expect(1, 11647, '\P{Blk:   - Ethiopic-EXTENDED}', "");
Expect(0, 11647, '\P{^Blk:   - Ethiopic-EXTENDED}', "");
Error('\p{Is_Block=/a/_ethiopic_extended}');
Error('\P{Is_Block=/a/_ethiopic_extended}');
Expect(1, 11743, '\p{Is_Block=ethiopicextended}', "");
Expect(0, 11743, '\p{^Is_Block=ethiopicextended}', "");
Expect(0, 11743, '\P{Is_Block=ethiopicextended}', "");
Expect(1, 11743, '\P{^Is_Block=ethiopicextended}', "");
Expect(0, 11647, '\p{Is_Block=ethiopicextended}', "");
Expect(1, 11647, '\p{^Is_Block=ethiopicextended}', "");
Expect(1, 11647, '\P{Is_Block=ethiopicextended}', "");
Expect(0, 11647, '\P{^Is_Block=ethiopicextended}', "");
Expect(1, 11743, '\p{Is_Block= EthiopicExtended}', "");
Expect(0, 11743, '\p{^Is_Block= EthiopicExtended}', "");
Expect(0, 11743, '\P{Is_Block= EthiopicExtended}', "");
Expect(1, 11743, '\P{^Is_Block= EthiopicExtended}', "");
Expect(0, 11647, '\p{Is_Block= EthiopicExtended}', "");
Expect(1, 11647, '\p{^Is_Block= EthiopicExtended}', "");
Expect(1, 11647, '\P{Is_Block= EthiopicExtended}', "");
Expect(0, 11647, '\P{^Is_Block= EthiopicExtended}', "");
Error('\p{Is_Blk=-:=ethiopicExtended}');
Error('\P{Is_Blk=-:=ethiopicExtended}');
Expect(1, 11743, '\p{Is_Blk=ethiopicextended}', "");
Expect(0, 11743, '\p{^Is_Blk=ethiopicextended}', "");
Expect(0, 11743, '\P{Is_Blk=ethiopicextended}', "");
Expect(1, 11743, '\P{^Is_Blk=ethiopicextended}', "");
Expect(0, 11647, '\p{Is_Blk=ethiopicextended}', "");
Expect(1, 11647, '\p{^Is_Blk=ethiopicextended}', "");
Expect(1, 11647, '\P{Is_Blk=ethiopicextended}', "");
Expect(0, 11647, '\P{^Is_Blk=ethiopicextended}', "");
Expect(1, 11743, '\p{Is_Blk=_EthiopicExtended}', "");
Expect(0, 11743, '\p{^Is_Blk=_EthiopicExtended}', "");
Expect(0, 11743, '\P{Is_Blk=_EthiopicExtended}', "");
Expect(1, 11743, '\P{^Is_Blk=_EthiopicExtended}', "");
Expect(0, 11647, '\p{Is_Blk=_EthiopicExtended}', "");
Expect(1, 11647, '\p{^Is_Blk=_EthiopicExtended}', "");
Expect(1, 11647, '\P{Is_Blk=_EthiopicExtended}', "");
Expect(0, 11647, '\P{^Is_Blk=_EthiopicExtended}', "");
Error('\p{Block= 	phoenician/a/}');
Error('\P{Block= 	phoenician/a/}');
Expect(1, 67871, '\p{Block=phoenician}', "");
Expect(0, 67871, '\p{^Block=phoenician}', "");
Expect(0, 67871, '\P{Block=phoenician}', "");
Expect(1, 67871, '\P{^Block=phoenician}', "");
Expect(0, 67839, '\p{Block=phoenician}', "");
Expect(1, 67839, '\p{^Block=phoenician}', "");
Expect(1, 67839, '\P{Block=phoenician}', "");
Expect(0, 67839, '\P{^Block=phoenician}', "");
Expect(1, 67871, '\p{Block= 	PHOENICIAN}', "");
Expect(0, 67871, '\p{^Block= 	PHOENICIAN}', "");
Expect(0, 67871, '\P{Block= 	PHOENICIAN}', "");
Expect(1, 67871, '\P{^Block= 	PHOENICIAN}', "");
Expect(0, 67839, '\p{Block= 	PHOENICIAN}', "");
Expect(1, 67839, '\p{^Block= 	PHOENICIAN}', "");
Expect(1, 67839, '\P{Block= 	PHOENICIAN}', "");
Expect(0, 67839, '\P{^Block= 	PHOENICIAN}', "");
Error('\p{Blk=-Phoenician/a/}');
Error('\P{Blk=-Phoenician/a/}');
Expect(1, 67871, '\p{Blk=phoenician}', "");
Expect(0, 67871, '\p{^Blk=phoenician}', "");
Expect(0, 67871, '\P{Blk=phoenician}', "");
Expect(1, 67871, '\P{^Blk=phoenician}', "");
Expect(0, 67839, '\p{Blk=phoenician}', "");
Expect(1, 67839, '\p{^Blk=phoenician}', "");
Expect(1, 67839, '\P{Blk=phoenician}', "");
Expect(0, 67839, '\P{^Blk=phoenician}', "");
Expect(1, 67871, '\p{Blk= 	PHOENICIAN}', "");
Expect(0, 67871, '\p{^Blk= 	PHOENICIAN}', "");
Expect(0, 67871, '\P{Blk= 	PHOENICIAN}', "");
Expect(1, 67871, '\P{^Blk= 	PHOENICIAN}', "");
Expect(0, 67839, '\p{Blk= 	PHOENICIAN}', "");
Expect(1, 67839, '\p{^Blk= 	PHOENICIAN}', "");
Expect(1, 67839, '\P{Blk= 	PHOENICIAN}', "");
Expect(0, 67839, '\P{^Blk= 	PHOENICIAN}', "");
Error('\p{Is_Block=/a/	_Phoenician}');
Error('\P{Is_Block=/a/	_Phoenician}');
Expect(1, 67871, '\p{Is_Block=phoenician}', "");
Expect(0, 67871, '\p{^Is_Block=phoenician}', "");
Expect(0, 67871, '\P{Is_Block=phoenician}', "");
Expect(1, 67871, '\P{^Is_Block=phoenician}', "");
Expect(0, 67839, '\p{Is_Block=phoenician}', "");
Expect(1, 67839, '\p{^Is_Block=phoenician}', "");
Expect(1, 67839, '\P{Is_Block=phoenician}', "");
Expect(0, 67839, '\P{^Is_Block=phoenician}', "");
Expect(1, 67871, '\p{Is_Block=	_Phoenician}', "");
Expect(0, 67871, '\p{^Is_Block=	_Phoenician}', "");
Expect(0, 67871, '\P{Is_Block=	_Phoenician}', "");
Expect(1, 67871, '\P{^Is_Block=	_Phoenician}', "");
Expect(0, 67839, '\p{Is_Block=	_Phoenician}', "");
Expect(1, 67839, '\p{^Is_Block=	_Phoenician}', "");
Expect(1, 67839, '\P{Is_Block=	_Phoenician}', "");
Expect(0, 67839, '\P{^Is_Block=	_Phoenician}', "");
Error('\p{Is_Blk=:= Phoenician}');
Error('\P{Is_Blk=:= Phoenician}');
Expect(1, 67871, '\p{Is_Blk=phoenician}', "");
Expect(0, 67871, '\p{^Is_Blk=phoenician}', "");
Expect(0, 67871, '\P{Is_Blk=phoenician}', "");
Expect(1, 67871, '\P{^Is_Blk=phoenician}', "");
Expect(0, 67839, '\p{Is_Blk=phoenician}', "");
Expect(1, 67839, '\p{^Is_Blk=phoenician}', "");
Expect(1, 67839, '\P{Is_Blk=phoenician}', "");
Expect(0, 67839, '\P{^Is_Blk=phoenician}', "");
Expect(1, 67871, '\p{Is_Blk=_Phoenician}', "");
Expect(0, 67871, '\p{^Is_Blk=_Phoenician}', "");
Expect(0, 67871, '\P{Is_Blk=_Phoenician}', "");
Expect(1, 67871, '\P{^Is_Blk=_Phoenician}', "");
Expect(0, 67839, '\p{Is_Blk=_Phoenician}', "");
Expect(1, 67839, '\p{^Is_Blk=_Phoenician}', "");
Expect(1, 67839, '\P{Is_Blk=_Phoenician}', "");
Expect(0, 67839, '\P{^Is_Blk=_Phoenician}', "");
Error('\p{Block=-/a/Latin	1-SUPPLEMENT}');
Error('\P{Block=-/a/Latin	1-SUPPLEMENT}');
Expect(1, 255, '\p{Block=latin1supplement}', "");
Expect(0, 255, '\p{^Block=latin1supplement}', "");
Expect(0, 255, '\P{Block=latin1supplement}', "");
Expect(1, 255, '\P{^Block=latin1supplement}', "");
Expect(0, 127, '\p{Block=latin1supplement}', "");
Expect(1, 127, '\p{^Block=latin1supplement}', "");
Expect(1, 127, '\P{Block=latin1supplement}', "");
Expect(0, 127, '\P{^Block=latin1supplement}', "");
Expect(1, 255, '\p{Block=-	Latin 1-supplement}', "");
Expect(0, 255, '\p{^Block=-	Latin 1-supplement}', "");
Expect(0, 255, '\P{Block=-	Latin 1-supplement}', "");
Expect(1, 255, '\P{^Block=-	Latin 1-supplement}', "");
Expect(0, 127, '\p{Block=-	Latin 1-supplement}', "");
Expect(1, 127, '\p{^Block=-	Latin 1-supplement}', "");
Expect(1, 127, '\P{Block=-	Latin 1-supplement}', "");
Expect(0, 127, '\P{^Block=-	Latin 1-supplement}', "");
Error('\p{Blk= latin/a/1}');
Error('\P{Blk= latin/a/1}');
Expect(1, 255, '\p{Blk=latin1}', "");
Expect(0, 255, '\p{^Blk=latin1}', "");
Expect(0, 255, '\P{Blk=latin1}', "");
Expect(1, 255, '\P{^Blk=latin1}', "");
Expect(0, 127, '\p{Blk=latin1}', "");
Expect(1, 127, '\p{^Blk=latin1}', "");
Expect(1, 127, '\P{Blk=latin1}', "");
Expect(0, 127, '\P{^Blk=latin1}', "");
Expect(1, 255, '\p{Blk=LATIN-1}', "");
Expect(0, 255, '\p{^Blk=LATIN-1}', "");
Expect(0, 255, '\P{Blk=LATIN-1}', "");
Expect(1, 255, '\P{^Blk=LATIN-1}', "");
Expect(0, 127, '\p{Blk=LATIN-1}', "");
Expect(1, 127, '\p{^Blk=LATIN-1}', "");
Expect(1, 127, '\P{Blk=LATIN-1}', "");
Expect(0, 127, '\P{^Blk=LATIN-1}', "");
Error('\p{Is_Block=_-Latin-1:=SUPPLEMENT}');
Error('\P{Is_Block=_-Latin-1:=SUPPLEMENT}');
Expect(1, 255, '\p{Is_Block=latin1supplement}', "");
Expect(0, 255, '\p{^Is_Block=latin1supplement}', "");
Expect(0, 255, '\P{Is_Block=latin1supplement}', "");
Expect(1, 255, '\P{^Is_Block=latin1supplement}', "");
Expect(0, 127, '\p{Is_Block=latin1supplement}', "");
Expect(1, 127, '\p{^Is_Block=latin1supplement}', "");
Expect(1, 127, '\P{Is_Block=latin1supplement}', "");
Expect(0, 127, '\P{^Is_Block=latin1supplement}', "");
Expect(1, 255, '\p{Is_Block=	 Latin-1-supplement}', "");
Expect(0, 255, '\p{^Is_Block=	 Latin-1-supplement}', "");
Expect(0, 255, '\P{Is_Block=	 Latin-1-supplement}', "");
Expect(1, 255, '\P{^Is_Block=	 Latin-1-supplement}', "");
Expect(0, 127, '\p{Is_Block=	 Latin-1-supplement}', "");
Expect(1, 127, '\p{^Is_Block=	 Latin-1-supplement}', "");
Expect(1, 127, '\P{Is_Block=	 Latin-1-supplement}', "");
Expect(0, 127, '\P{^Is_Block=	 Latin-1-supplement}', "");
Error('\p{Is_Blk=_	Latin:=1}');
Error('\P{Is_Blk=_	Latin:=1}');
Expect(1, 255, '\p{Is_Blk=latin1}', "");
Expect(0, 255, '\p{^Is_Blk=latin1}', "");
Expect(0, 255, '\P{Is_Blk=latin1}', "");
Expect(1, 255, '\P{^Is_Blk=latin1}', "");
Expect(0, 127, '\p{Is_Blk=latin1}', "");
Expect(1, 127, '\p{^Is_Blk=latin1}', "");
Expect(1, 127, '\P{Is_Blk=latin1}', "");
Expect(0, 127, '\P{^Is_Blk=latin1}', "");
Expect(1, 255, '\p{Is_Blk=_Latin	1}', "");
Expect(0, 255, '\p{^Is_Blk=_Latin	1}', "");
Expect(0, 255, '\P{Is_Blk=_Latin	1}', "");
Expect(1, 255, '\P{^Is_Blk=_Latin	1}', "");
Expect(0, 127, '\p{Is_Blk=_Latin	1}', "");
Expect(1, 127, '\p{^Is_Blk=_Latin	1}', "");
Expect(1, 127, '\P{Is_Blk=_Latin	1}', "");
Expect(0, 127, '\P{^Is_Blk=_Latin	1}', "");
Error('\p{Block= _devanagari/a/}');
Error('\P{Block= _devanagari/a/}');
Expect(1, 2431, '\p{Block=devanagari}', "");
Expect(0, 2431, '\p{^Block=devanagari}', "");
Expect(0, 2431, '\P{Block=devanagari}', "");
Expect(1, 2431, '\P{^Block=devanagari}', "");
Expect(0, 2303, '\p{Block=devanagari}', "");
Expect(1, 2303, '\p{^Block=devanagari}', "");
Expect(1, 2303, '\P{Block=devanagari}', "");
Expect(0, 2303, '\P{^Block=devanagari}', "");
Expect(1, 2431, '\p{Block=_-Devanagari}', "");
Expect(0, 2431, '\p{^Block=_-Devanagari}', "");
Expect(0, 2431, '\P{Block=_-Devanagari}', "");
Expect(1, 2431, '\P{^Block=_-Devanagari}', "");
Expect(0, 2303, '\p{Block=_-Devanagari}', "");
Expect(1, 2303, '\p{^Block=_-Devanagari}', "");
Expect(1, 2303, '\P{Block=_-Devanagari}', "");
Expect(0, 2303, '\P{^Block=_-Devanagari}', "");
Error('\p{Blk:	/a/_	DEVANAGARI}');
Error('\P{Blk:	/a/_	DEVANAGARI}');
Expect(1, 2431, '\p{Blk=devanagari}', "");
Expect(0, 2431, '\p{^Blk=devanagari}', "");
Expect(0, 2431, '\P{Blk=devanagari}', "");
Expect(1, 2431, '\P{^Blk=devanagari}', "");
Expect(0, 2303, '\p{Blk=devanagari}', "");
Expect(1, 2303, '\p{^Blk=devanagari}', "");
Expect(1, 2303, '\P{Blk=devanagari}', "");
Expect(0, 2303, '\P{^Blk=devanagari}', "");
Expect(1, 2431, '\p{Blk=		devanagari}', "");
Expect(0, 2431, '\p{^Blk=		devanagari}', "");
Expect(0, 2431, '\P{Blk=		devanagari}', "");
Expect(1, 2431, '\P{^Blk=		devanagari}', "");
Expect(0, 2303, '\p{Blk=		devanagari}', "");
Expect(1, 2303, '\p{^Blk=		devanagari}', "");
Expect(1, 2303, '\P{Blk=		devanagari}', "");
Expect(0, 2303, '\P{^Blk=		devanagari}', "");
Error('\p{Is_Block=__Devanagari:=}');
Error('\P{Is_Block=__Devanagari:=}');
Expect(1, 2431, '\p{Is_Block=devanagari}', "");
Expect(0, 2431, '\p{^Is_Block=devanagari}', "");
Expect(0, 2431, '\P{Is_Block=devanagari}', "");
Expect(1, 2431, '\P{^Is_Block=devanagari}', "");
Expect(0, 2303, '\p{Is_Block=devanagari}', "");
Expect(1, 2303, '\p{^Is_Block=devanagari}', "");
Expect(1, 2303, '\P{Is_Block=devanagari}', "");
Expect(0, 2303, '\P{^Is_Block=devanagari}', "");
Expect(1, 2431, '\p{Is_Block=__Devanagari}', "");
Expect(0, 2431, '\p{^Is_Block=__Devanagari}', "");
Expect(0, 2431, '\P{Is_Block=__Devanagari}', "");
Expect(1, 2431, '\P{^Is_Block=__Devanagari}', "");
Expect(0, 2303, '\p{Is_Block=__Devanagari}', "");
Expect(1, 2303, '\p{^Is_Block=__Devanagari}', "");
Expect(1, 2303, '\P{Is_Block=__Devanagari}', "");
Expect(0, 2303, '\P{^Is_Block=__Devanagari}', "");
Error('\p{Is_Blk=-_devanagari:=}');
Error('\P{Is_Blk=-_devanagari:=}');
Expect(1, 2431, '\p{Is_Blk=devanagari}', "");
Expect(0, 2431, '\p{^Is_Blk=devanagari}', "");
Expect(0, 2431, '\P{Is_Blk=devanagari}', "");
Expect(1, 2431, '\P{^Is_Blk=devanagari}', "");
Expect(0, 2303, '\p{Is_Blk=devanagari}', "");
Expect(1, 2303, '\p{^Is_Blk=devanagari}', "");
Expect(1, 2303, '\P{Is_Blk=devanagari}', "");
Expect(0, 2303, '\P{^Is_Blk=devanagari}', "");
Expect(1, 2431, '\p{Is_Blk= Devanagari}', "");
Expect(0, 2431, '\p{^Is_Blk= Devanagari}', "");
Expect(0, 2431, '\P{Is_Blk= Devanagari}', "");
Expect(1, 2431, '\P{^Is_Blk= Devanagari}', "");
Expect(0, 2303, '\p{Is_Blk= Devanagari}', "");
Expect(1, 2303, '\p{^Is_Blk= Devanagari}', "");
Expect(1, 2303, '\P{Is_Blk= Devanagari}', "");
Expect(0, 2303, '\P{^Is_Blk= Devanagari}', "");
Error('\p{Block=_-hanguljamo_Extended-A/a/}');
Error('\P{Block=_-hanguljamo_Extended-A/a/}');
Expect(1, 43391, '\p{Block=hanguljamoextendeda}', "");
Expect(0, 43391, '\p{^Block=hanguljamoextendeda}', "");
Expect(0, 43391, '\P{Block=hanguljamoextendeda}', "");
Expect(1, 43391, '\P{^Block=hanguljamoextendeda}', "");
Expect(0, 43359, '\p{Block=hanguljamoextendeda}', "");
Expect(1, 43359, '\p{^Block=hanguljamoextendeda}', "");
Expect(1, 43359, '\P{Block=hanguljamoextendeda}', "");
Expect(0, 43359, '\P{^Block=hanguljamoextendeda}', "");
Expect(1, 43391, '\p{Block=--Hangul Jamo Extended A}', "");
Expect(0, 43391, '\p{^Block=--Hangul Jamo Extended A}', "");
Expect(0, 43391, '\P{Block=--Hangul Jamo Extended A}', "");
Expect(1, 43391, '\P{^Block=--Hangul Jamo Extended A}', "");
Expect(0, 43359, '\p{Block=--Hangul Jamo Extended A}', "");
Expect(1, 43359, '\p{^Block=--Hangul Jamo Extended A}', "");
Expect(1, 43359, '\P{Block=--Hangul Jamo Extended A}', "");
Expect(0, 43359, '\P{^Block=--Hangul Jamo Extended A}', "");
Error('\p{Blk=Hangul:=Jamo_Extended	A}');
Error('\P{Blk=Hangul:=Jamo_Extended	A}');
Expect(1, 43391, '\p{Blk=hanguljamoextendeda}', "");
Expect(0, 43391, '\p{^Blk=hanguljamoextendeda}', "");
Expect(0, 43391, '\P{Blk=hanguljamoextendeda}', "");
Expect(1, 43391, '\P{^Blk=hanguljamoextendeda}', "");
Expect(0, 43359, '\p{Blk=hanguljamoextendeda}', "");
Expect(1, 43359, '\p{^Blk=hanguljamoextendeda}', "");
Expect(1, 43359, '\P{Blk=hanguljamoextendeda}', "");
Expect(0, 43359, '\P{^Blk=hanguljamoextendeda}', "");
Expect(1, 43391, '\p{Blk:     Hangul-JAMO	ExtendedA}', "");
Expect(0, 43391, '\p{^Blk:     Hangul-JAMO	ExtendedA}', "");
Expect(0, 43391, '\P{Blk:     Hangul-JAMO	ExtendedA}', "");
Expect(1, 43391, '\P{^Blk:     Hangul-JAMO	ExtendedA}', "");
Expect(0, 43359, '\p{Blk:     Hangul-JAMO	ExtendedA}', "");
Expect(1, 43359, '\p{^Blk:     Hangul-JAMO	ExtendedA}', "");
Expect(1, 43359, '\P{Blk:     Hangul-JAMO	ExtendedA}', "");
Expect(0, 43359, '\P{^Blk:     Hangul-JAMO	ExtendedA}', "");
Error('\p{Is_Block=_	Hangul	JAMO_Extended:=a}');
Error('\P{Is_Block=_	Hangul	JAMO_Extended:=a}');
Expect(1, 43391, '\p{Is_Block:   hanguljamoextendeda}', "");
Expect(0, 43391, '\p{^Is_Block:   hanguljamoextendeda}', "");
Expect(0, 43391, '\P{Is_Block:   hanguljamoextendeda}', "");
Expect(1, 43391, '\P{^Is_Block:   hanguljamoextendeda}', "");
Expect(0, 43359, '\p{Is_Block:   hanguljamoextendeda}', "");
Expect(1, 43359, '\p{^Is_Block:   hanguljamoextendeda}', "");
Expect(1, 43359, '\P{Is_Block:   hanguljamoextendeda}', "");
Expect(0, 43359, '\P{^Is_Block:   hanguljamoextendeda}', "");
Expect(1, 43391, '\p{Is_Block: - Hangul	JAMO	extended_a}', "");
Expect(0, 43391, '\p{^Is_Block: - Hangul	JAMO	extended_a}', "");
Expect(0, 43391, '\P{Is_Block: - Hangul	JAMO	extended_a}', "");
Expect(1, 43391, '\P{^Is_Block: - Hangul	JAMO	extended_a}', "");
Expect(0, 43359, '\p{Is_Block: - Hangul	JAMO	extended_a}', "");
Expect(1, 43359, '\p{^Is_Block: - Hangul	JAMO	extended_a}', "");
Expect(1, 43359, '\P{Is_Block: - Hangul	JAMO	extended_a}', "");
Expect(0, 43359, '\P{^Is_Block: - Hangul	JAMO	extended_a}', "");
Error('\p{Is_Blk=-:=hangul_jamo_Extended_A}');
Error('\P{Is_Blk=-:=hangul_jamo_Extended_A}');
Expect(1, 43391, '\p{Is_Blk=hanguljamoextendeda}', "");
Expect(0, 43391, '\p{^Is_Blk=hanguljamoextendeda}', "");
Expect(0, 43391, '\P{Is_Blk=hanguljamoextendeda}', "");
Expect(1, 43391, '\P{^Is_Blk=hanguljamoextendeda}', "");
Expect(0, 43359, '\p{Is_Blk=hanguljamoextendeda}', "");
Expect(1, 43359, '\p{^Is_Blk=hanguljamoextendeda}', "");
Expect(1, 43359, '\P{Is_Blk=hanguljamoextendeda}', "");
Expect(0, 43359, '\P{^Is_Blk=hanguljamoextendeda}', "");
Expect(1, 43391, '\p{Is_Blk= Hangul-jamo Extended A}', "");
Expect(0, 43391, '\p{^Is_Blk= Hangul-jamo Extended A}', "");
Expect(0, 43391, '\P{Is_Blk= Hangul-jamo Extended A}', "");
Expect(1, 43391, '\P{^Is_Blk= Hangul-jamo Extended A}', "");
Expect(0, 43359, '\p{Is_Blk= Hangul-jamo Extended A}', "");
Expect(1, 43359, '\p{^Is_Blk= Hangul-jamo Extended A}', "");
Expect(1, 43359, '\P{Is_Blk= Hangul-jamo Extended A}', "");
Expect(0, 43359, '\P{^Is_Blk= Hangul-jamo Extended A}', "");
Error('\p{Block=/a/	Carian}');
Error('\P{Block=/a/	Carian}');
Expect(1, 66271, '\p{Block=carian}', "");
Expect(0, 66271, '\p{^Block=carian}', "");
Expect(0, 66271, '\P{Block=carian}', "");
Expect(1, 66271, '\P{^Block=carian}', "");
Expect(0, 66207, '\p{Block=carian}', "");
Expect(1, 66207, '\p{^Block=carian}', "");
Expect(1, 66207, '\P{Block=carian}', "");
Expect(0, 66207, '\P{^Block=carian}', "");
Expect(1, 66271, '\p{Block= _carian}', "");
Expect(0, 66271, '\p{^Block= _carian}', "");
Expect(0, 66271, '\P{Block= _carian}', "");
Expect(1, 66271, '\P{^Block= _carian}', "");
Expect(0, 66207, '\p{Block= _carian}', "");
Expect(1, 66207, '\p{^Block= _carian}', "");
Expect(1, 66207, '\P{Block= _carian}', "");
Expect(0, 66207, '\P{^Block= _carian}', "");
Error('\p{Blk=_ Carian:=}');
Error('\P{Blk=_ Carian:=}');
Expect(1, 66271, '\p{Blk=carian}', "");
Expect(0, 66271, '\p{^Blk=carian}', "");
Expect(0, 66271, '\P{Blk=carian}', "");
Expect(1, 66271, '\P{^Blk=carian}', "");
Expect(0, 66207, '\p{Blk=carian}', "");
Expect(1, 66207, '\p{^Blk=carian}', "");
Expect(1, 66207, '\P{Blk=carian}', "");
Expect(0, 66207, '\P{^Blk=carian}', "");
Expect(1, 66271, '\p{Blk=- Carian}', "");
Expect(0, 66271, '\p{^Blk=- Carian}', "");
Expect(0, 66271, '\P{Blk=- Carian}', "");
Expect(1, 66271, '\P{^Blk=- Carian}', "");
Expect(0, 66207, '\p{Blk=- Carian}', "");
Expect(1, 66207, '\p{^Blk=- Carian}', "");
Expect(1, 66207, '\P{Blk=- Carian}', "");
Expect(0, 66207, '\P{^Blk=- Carian}', "");
Error('\p{Is_Block=/a/Carian}');
Error('\P{Is_Block=/a/Carian}');
Expect(1, 66271, '\p{Is_Block=carian}', "");
Expect(0, 66271, '\p{^Is_Block=carian}', "");
Expect(0, 66271, '\P{Is_Block=carian}', "");
Expect(1, 66271, '\P{^Is_Block=carian}', "");
Expect(0, 66207, '\p{Is_Block=carian}', "");
Expect(1, 66207, '\p{^Is_Block=carian}', "");
Expect(1, 66207, '\P{Is_Block=carian}', "");
Expect(0, 66207, '\P{^Is_Block=carian}', "");
Expect(1, 66271, '\p{Is_Block= 	carian}', "");
Expect(0, 66271, '\p{^Is_Block= 	carian}', "");
Expect(0, 66271, '\P{Is_Block= 	carian}', "");
Expect(1, 66271, '\P{^Is_Block= 	carian}', "");
Expect(0, 66207, '\p{Is_Block= 	carian}', "");
Expect(1, 66207, '\p{^Is_Block= 	carian}', "");
Expect(1, 66207, '\P{Is_Block= 	carian}', "");
Expect(0, 66207, '\P{^Is_Block= 	carian}', "");
Error('\p{Is_Blk=-:=CARIAN}');
Error('\P{Is_Blk=-:=CARIAN}');
Expect(1, 66271, '\p{Is_Blk=carian}', "");
Expect(0, 66271, '\p{^Is_Blk=carian}', "");
Expect(0, 66271, '\P{Is_Blk=carian}', "");
Expect(1, 66271, '\P{^Is_Blk=carian}', "");
Expect(0, 66207, '\p{Is_Blk=carian}', "");
Expect(1, 66207, '\p{^Is_Blk=carian}', "");
Expect(1, 66207, '\P{Is_Blk=carian}', "");
Expect(0, 66207, '\P{^Is_Blk=carian}', "");
Expect(1, 66271, '\p{Is_Blk=	Carian}', "");
Expect(0, 66271, '\p{^Is_Blk=	Carian}', "");
Expect(0, 66271, '\P{Is_Blk=	Carian}', "");
Expect(1, 66271, '\P{^Is_Blk=	Carian}', "");
Expect(0, 66207, '\p{Is_Blk=	Carian}', "");
Expect(1, 66207, '\p{^Is_Blk=	Carian}', "");
Expect(1, 66207, '\P{Is_Blk=	Carian}', "");
Expect(0, 66207, '\P{^Is_Blk=	Carian}', "");
Error('\p{Block=-/a/buhid}');
Error('\P{Block=-/a/buhid}');
Expect(1, 5983, '\p{Block=buhid}', "");
Expect(0, 5983, '\p{^Block=buhid}', "");
Expect(0, 5983, '\P{Block=buhid}', "");
Expect(1, 5983, '\P{^Block=buhid}', "");
Expect(0, 5951, '\p{Block=buhid}', "");
Expect(1, 5951, '\p{^Block=buhid}', "");
Expect(1, 5951, '\P{Block=buhid}', "");
Expect(0, 5951, '\P{^Block=buhid}', "");
Expect(1, 5983, '\p{Block=-buhid}', "");
Expect(0, 5983, '\p{^Block=-buhid}', "");
Expect(0, 5983, '\P{Block=-buhid}', "");
Expect(1, 5983, '\P{^Block=-buhid}', "");
Expect(0, 5951, '\p{Block=-buhid}', "");
Expect(1, 5951, '\p{^Block=-buhid}', "");
Expect(1, 5951, '\P{Block=-buhid}', "");
Expect(0, 5951, '\P{^Block=-buhid}', "");
Error('\p{Blk=	buhid:=}');
Error('\P{Blk=	buhid:=}');
Expect(1, 5983, '\p{Blk=buhid}', "");
Expect(0, 5983, '\p{^Blk=buhid}', "");
Expect(0, 5983, '\P{Blk=buhid}', "");
Expect(1, 5983, '\P{^Blk=buhid}', "");
Expect(0, 5951, '\p{Blk=buhid}', "");
Expect(1, 5951, '\p{^Blk=buhid}', "");
Expect(1, 5951, '\P{Blk=buhid}', "");
Expect(0, 5951, '\P{^Blk=buhid}', "");
Expect(1, 5983, '\p{Blk=	-BUHID}', "");
Expect(0, 5983, '\p{^Blk=	-BUHID}', "");
Expect(0, 5983, '\P{Blk=	-BUHID}', "");
Expect(1, 5983, '\P{^Blk=	-BUHID}', "");
Expect(0, 5951, '\p{Blk=	-BUHID}', "");
Expect(1, 5951, '\p{^Blk=	-BUHID}', "");
Expect(1, 5951, '\P{Blk=	-BUHID}', "");
Expect(0, 5951, '\P{^Blk=	-BUHID}', "");
Error('\p{Is_Block:	/a/buhid}');
Error('\P{Is_Block:	/a/buhid}');
Expect(1, 5983, '\p{Is_Block=buhid}', "");
Expect(0, 5983, '\p{^Is_Block=buhid}', "");
Expect(0, 5983, '\P{Is_Block=buhid}', "");
Expect(1, 5983, '\P{^Is_Block=buhid}', "");
Expect(0, 5951, '\p{Is_Block=buhid}', "");
Expect(1, 5951, '\p{^Is_Block=buhid}', "");
Expect(1, 5951, '\P{Is_Block=buhid}', "");
Expect(0, 5951, '\P{^Is_Block=buhid}', "");
Expect(1, 5983, '\p{Is_Block: _ Buhid}', "");
Expect(0, 5983, '\p{^Is_Block: _ Buhid}', "");
Expect(0, 5983, '\P{Is_Block: _ Buhid}', "");
Expect(1, 5983, '\P{^Is_Block: _ Buhid}', "");
Expect(0, 5951, '\p{Is_Block: _ Buhid}', "");
Expect(1, 5951, '\p{^Is_Block: _ Buhid}', "");
Expect(1, 5951, '\P{Is_Block: _ Buhid}', "");
Expect(0, 5951, '\P{^Is_Block: _ Buhid}', "");
Error('\p{Is_Blk=/a/ 	buhid}');
Error('\P{Is_Blk=/a/ 	buhid}');
Expect(1, 5983, '\p{Is_Blk=buhid}', "");
Expect(0, 5983, '\p{^Is_Blk=buhid}', "");
Expect(0, 5983, '\P{Is_Blk=buhid}', "");
Expect(1, 5983, '\P{^Is_Blk=buhid}', "");
Expect(0, 5951, '\p{Is_Blk=buhid}', "");
Expect(1, 5951, '\p{^Is_Blk=buhid}', "");
Expect(1, 5951, '\P{Is_Blk=buhid}', "");
Expect(0, 5951, '\P{^Is_Blk=buhid}', "");
Expect(1, 5983, '\p{Is_Blk:   _Buhid}', "");
Expect(0, 5983, '\p{^Is_Blk:   _Buhid}', "");
Expect(0, 5983, '\P{Is_Blk:   _Buhid}', "");
Expect(1, 5983, '\P{^Is_Blk:   _Buhid}', "");
Expect(0, 5951, '\p{Is_Blk:   _Buhid}', "");
Expect(1, 5951, '\p{^Is_Blk:   _Buhid}', "");
Expect(1, 5951, '\P{Is_Blk:   _Buhid}', "");
Expect(0, 5951, '\P{^Is_Blk:   _Buhid}', "");
Error('\p{Block=	 Ideographic/a/DESCRIPTION_characters}');
Error('\P{Block=	 Ideographic/a/DESCRIPTION_characters}');
Expect(1, 12287, '\p{Block=ideographicdescriptioncharacters}', "");
Expect(0, 12287, '\p{^Block=ideographicdescriptioncharacters}', "");
Expect(0, 12287, '\P{Block=ideographicdescriptioncharacters}', "");
Expect(1, 12287, '\P{^Block=ideographicdescriptioncharacters}', "");
Expect(0, 12271, '\p{Block=ideographicdescriptioncharacters}', "");
Expect(1, 12271, '\p{^Block=ideographicdescriptioncharacters}', "");
Expect(1, 12271, '\P{Block=ideographicdescriptioncharacters}', "");
Expect(0, 12271, '\P{^Block=ideographicdescriptioncharacters}', "");
Expect(1, 12287, '\p{Block= 	Ideographic	DESCRIPTION Characters}', "");
Expect(0, 12287, '\p{^Block= 	Ideographic	DESCRIPTION Characters}', "");
Expect(0, 12287, '\P{Block= 	Ideographic	DESCRIPTION Characters}', "");
Expect(1, 12287, '\P{^Block= 	Ideographic	DESCRIPTION Characters}', "");
Expect(0, 12271, '\p{Block= 	Ideographic	DESCRIPTION Characters}', "");
Expect(1, 12271, '\p{^Block= 	Ideographic	DESCRIPTION Characters}', "");
Expect(1, 12271, '\P{Block= 	Ideographic	DESCRIPTION Characters}', "");
Expect(0, 12271, '\P{^Block= 	Ideographic	DESCRIPTION Characters}', "");
Error('\p{Blk=-	ideographic DESCRIPTION	Characters/a/}');
Error('\P{Blk=-	ideographic DESCRIPTION	Characters/a/}');
Expect(1, 12287, '\p{Blk=ideographicdescriptioncharacters}', "");
Expect(0, 12287, '\p{^Blk=ideographicdescriptioncharacters}', "");
Expect(0, 12287, '\P{Blk=ideographicdescriptioncharacters}', "");
Expect(1, 12287, '\P{^Blk=ideographicdescriptioncharacters}', "");
Expect(0, 12271, '\p{Blk=ideographicdescriptioncharacters}', "");
Expect(1, 12271, '\p{^Blk=ideographicdescriptioncharacters}', "");
Expect(1, 12271, '\P{Blk=ideographicdescriptioncharacters}', "");
Expect(0, 12271, '\P{^Blk=ideographicdescriptioncharacters}', "");
Expect(1, 12287, '\p{Blk=-Ideographic	Description_Characters}', "");
Expect(0, 12287, '\p{^Blk=-Ideographic	Description_Characters}', "");
Expect(0, 12287, '\P{Blk=-Ideographic	Description_Characters}', "");
Expect(1, 12287, '\P{^Blk=-Ideographic	Description_Characters}', "");
Expect(0, 12271, '\p{Blk=-Ideographic	Description_Characters}', "");
Expect(1, 12271, '\p{^Blk=-Ideographic	Description_Characters}', "");
Expect(1, 12271, '\P{Blk=-Ideographic	Description_Characters}', "");
Expect(0, 12271, '\P{^Blk=-Ideographic	Description_Characters}', "");
Error('\p{Is_Block= :=IDEOGRAPHIC-Description	Characters}');
Error('\P{Is_Block= :=IDEOGRAPHIC-Description	Characters}');
Expect(1, 12287, '\p{Is_Block=ideographicdescriptioncharacters}', "");
Expect(0, 12287, '\p{^Is_Block=ideographicdescriptioncharacters}', "");
Expect(0, 12287, '\P{Is_Block=ideographicdescriptioncharacters}', "");
Expect(1, 12287, '\P{^Is_Block=ideographicdescriptioncharacters}', "");
Expect(0, 12271, '\p{Is_Block=ideographicdescriptioncharacters}', "");
Expect(1, 12271, '\p{^Is_Block=ideographicdescriptioncharacters}', "");
Expect(1, 12271, '\P{Is_Block=ideographicdescriptioncharacters}', "");
Expect(0, 12271, '\P{^Is_Block=ideographicdescriptioncharacters}', "");
Expect(1, 12287, '\p{Is_Block=-ideographicdescription_characters}', "");
Expect(0, 12287, '\p{^Is_Block=-ideographicdescription_characters}', "");
Expect(0, 12287, '\P{Is_Block=-ideographicdescription_characters}', "");
Expect(1, 12287, '\P{^Is_Block=-ideographicdescription_characters}', "");
Expect(0, 12271, '\p{Is_Block=-ideographicdescription_characters}', "");
Expect(1, 12271, '\p{^Is_Block=-ideographicdescription_characters}', "");
Expect(1, 12271, '\P{Is_Block=-ideographicdescription_characters}', "");
Expect(0, 12271, '\P{^Is_Block=-ideographicdescription_characters}', "");
Error('\p{Is_Blk=/a/- ideographic-Description	Characters}');
Error('\P{Is_Blk=/a/- ideographic-Description	Characters}');
Expect(1, 12287, '\p{Is_Blk=ideographicdescriptioncharacters}', "");
Expect(0, 12287, '\p{^Is_Blk=ideographicdescriptioncharacters}', "");
Expect(0, 12287, '\P{Is_Blk=ideographicdescriptioncharacters}', "");
Expect(1, 12287, '\P{^Is_Blk=ideographicdescriptioncharacters}', "");
Expect(0, 12271, '\p{Is_Blk=ideographicdescriptioncharacters}', "");
Expect(1, 12271, '\p{^Is_Blk=ideographicdescriptioncharacters}', "");
Expect(1, 12271, '\P{Is_Blk=ideographicdescriptioncharacters}', "");
Expect(0, 12271, '\P{^Is_Blk=ideographicdescriptioncharacters}', "");
Expect(1, 12287, '\p{Is_Blk=--ideographic-description_characters}', "");
Expect(0, 12287, '\p{^Is_Blk=--ideographic-description_characters}', "");
Expect(0, 12287, '\P{Is_Blk=--ideographic-description_characters}', "");
Expect(1, 12287, '\P{^Is_Blk=--ideographic-description_characters}', "");
Expect(0, 12271, '\p{Is_Blk=--ideographic-description_characters}', "");
Expect(1, 12271, '\p{^Is_Blk=--ideographic-description_characters}', "");
Expect(1, 12271, '\P{Is_Blk=--ideographic-description_characters}', "");
Expect(0, 12271, '\P{^Is_Blk=--ideographic-description_characters}', "");
Error('\p{Block=__Miscellaneous/a/Symbols And-Pictographs}');
Error('\P{Block=__Miscellaneous/a/Symbols And-Pictographs}');
Expect(1, 128511, '\p{Block=miscellaneoussymbolsandpictographs}', "");
Expect(0, 128511, '\p{^Block=miscellaneoussymbolsandpictographs}', "");
Expect(0, 128511, '\P{Block=miscellaneoussymbolsandpictographs}', "");
Expect(1, 128511, '\P{^Block=miscellaneoussymbolsandpictographs}', "");
Expect(0, 127743, '\p{Block=miscellaneoussymbolsandpictographs}', "");
Expect(1, 127743, '\p{^Block=miscellaneoussymbolsandpictographs}', "");
Expect(1, 127743, '\P{Block=miscellaneoussymbolsandpictographs}', "");
Expect(0, 127743, '\P{^Block=miscellaneoussymbolsandpictographs}', "");
Expect(1, 128511, '\p{Block=_miscellaneous symbols_AND-Pictographs}', "");
Expect(0, 128511, '\p{^Block=_miscellaneous symbols_AND-Pictographs}', "");
Expect(0, 128511, '\P{Block=_miscellaneous symbols_AND-Pictographs}', "");
Expect(1, 128511, '\P{^Block=_miscellaneous symbols_AND-Pictographs}', "");
Expect(0, 127743, '\p{Block=_miscellaneous symbols_AND-Pictographs}', "");
Expect(1, 127743, '\p{^Block=_miscellaneous symbols_AND-Pictographs}', "");
Expect(1, 127743, '\P{Block=_miscellaneous symbols_AND-Pictographs}', "");
Expect(0, 127743, '\P{^Block=_miscellaneous symbols_AND-Pictographs}', "");
Error('\p{Blk=__Miscellaneous/a/symbols-And	PICTOGRAPHS}');
Error('\P{Blk=__Miscellaneous/a/symbols-And	PICTOGRAPHS}');
Expect(1, 128511, '\p{Blk=miscellaneoussymbolsandpictographs}', "");
Expect(0, 128511, '\p{^Blk=miscellaneoussymbolsandpictographs}', "");
Expect(0, 128511, '\P{Blk=miscellaneoussymbolsandpictographs}', "");
Expect(1, 128511, '\P{^Blk=miscellaneoussymbolsandpictographs}', "");
Expect(0, 127743, '\p{Blk=miscellaneoussymbolsandpictographs}', "");
Expect(1, 127743, '\p{^Blk=miscellaneoussymbolsandpictographs}', "");
Expect(1, 127743, '\P{Blk=miscellaneoussymbolsandpictographs}', "");
Expect(0, 127743, '\P{^Blk=miscellaneoussymbolsandpictographs}', "");
Expect(1, 128511, '\p{Blk: 	 Miscellaneous SYMBOLS And PICTOGRAPHS}', "");
Expect(0, 128511, '\p{^Blk: 	 Miscellaneous SYMBOLS And PICTOGRAPHS}', "");
Expect(0, 128511, '\P{Blk: 	 Miscellaneous SYMBOLS And PICTOGRAPHS}', "");
Expect(1, 128511, '\P{^Blk: 	 Miscellaneous SYMBOLS And PICTOGRAPHS}', "");
Expect(0, 127743, '\p{Blk: 	 Miscellaneous SYMBOLS And PICTOGRAPHS}', "");
Expect(1, 127743, '\p{^Blk: 	 Miscellaneous SYMBOLS And PICTOGRAPHS}', "");
Expect(1, 127743, '\P{Blk: 	 Miscellaneous SYMBOLS And PICTOGRAPHS}', "");
Expect(0, 127743, '\P{^Blk: 	 Miscellaneous SYMBOLS And PICTOGRAPHS}', "");
Error('\p{Is_Block=	 MISCELLANEOUSsymbols_AND/a/Pictographs}');
Error('\P{Is_Block=	 MISCELLANEOUSsymbols_AND/a/Pictographs}');
Expect(1, 128511, '\p{Is_Block=miscellaneoussymbolsandpictographs}', "");
Expect(0, 128511, '\p{^Is_Block=miscellaneoussymbolsandpictographs}', "");
Expect(0, 128511, '\P{Is_Block=miscellaneoussymbolsandpictographs}', "");
Expect(1, 128511, '\P{^Is_Block=miscellaneoussymbolsandpictographs}', "");
Expect(0, 127743, '\p{Is_Block=miscellaneoussymbolsandpictographs}', "");
Expect(1, 127743, '\p{^Is_Block=miscellaneoussymbolsandpictographs}', "");
Expect(1, 127743, '\P{Is_Block=miscellaneoussymbolsandpictographs}', "");
Expect(0, 127743, '\P{^Is_Block=miscellaneoussymbolsandpictographs}', "");
Expect(1, 128511, '\p{Is_Block=-_Miscellaneous	SYMBOLS-AndPictographs}', "");
Expect(0, 128511, '\p{^Is_Block=-_Miscellaneous	SYMBOLS-AndPictographs}', "");
Expect(0, 128511, '\P{Is_Block=-_Miscellaneous	SYMBOLS-AndPictographs}', "");
Expect(1, 128511, '\P{^Is_Block=-_Miscellaneous	SYMBOLS-AndPictographs}', "");
Expect(0, 127743, '\p{Is_Block=-_Miscellaneous	SYMBOLS-AndPictographs}', "");
Expect(1, 127743, '\p{^Is_Block=-_Miscellaneous	SYMBOLS-AndPictographs}', "");
Expect(1, 127743, '\P{Is_Block=-_Miscellaneous	SYMBOLS-AndPictographs}', "");
Expect(0, 127743, '\P{^Is_Block=-_Miscellaneous	SYMBOLS-AndPictographs}', "");
Error('\p{Is_Blk= :=Miscellaneous	SYMBOLSAnd	pictographs}');
Error('\P{Is_Blk= :=Miscellaneous	SYMBOLSAnd	pictographs}');
Expect(1, 128511, '\p{Is_Blk=miscellaneoussymbolsandpictographs}', "");
Expect(0, 128511, '\p{^Is_Blk=miscellaneoussymbolsandpictographs}', "");
Expect(0, 128511, '\P{Is_Blk=miscellaneoussymbolsandpictographs}', "");
Expect(1, 128511, '\P{^Is_Blk=miscellaneoussymbolsandpictographs}', "");
Expect(0, 127743, '\p{Is_Blk=miscellaneoussymbolsandpictographs}', "");
Expect(1, 127743, '\p{^Is_Blk=miscellaneoussymbolsandpictographs}', "");
Expect(1, 127743, '\P{Is_Blk=miscellaneoussymbolsandpictographs}', "");
Expect(0, 127743, '\P{^Is_Blk=miscellaneoussymbolsandpictographs}', "");
Expect(1, 128511, '\p{Is_Blk=--Miscellaneous-symbols-ANDPICTOGRAPHS}', "");
Expect(0, 128511, '\p{^Is_Blk=--Miscellaneous-symbols-ANDPICTOGRAPHS}', "");
Expect(0, 128511, '\P{Is_Blk=--Miscellaneous-symbols-ANDPICTOGRAPHS}', "");
Expect(1, 128511, '\P{^Is_Blk=--Miscellaneous-symbols-ANDPICTOGRAPHS}', "");
Expect(0, 127743, '\p{Is_Blk=--Miscellaneous-symbols-ANDPICTOGRAPHS}', "");
Expect(1, 127743, '\p{^Is_Blk=--Miscellaneous-symbols-ANDPICTOGRAPHS}', "");
Expect(1, 127743, '\P{Is_Blk=--Miscellaneous-symbols-ANDPICTOGRAPHS}', "");
Expect(0, 127743, '\P{^Is_Blk=--Miscellaneous-symbols-ANDPICTOGRAPHS}', "");
Error('\p{Block=	/a/cjk_unified-Ideographs extension b}');
Error('\P{Block=	/a/cjk_unified-Ideographs extension b}');
Expect(1, 173791, '\p{Block=cjkunifiedideographsextensionb}', "");
Expect(0, 173791, '\p{^Block=cjkunifiedideographsextensionb}', "");
Expect(0, 173791, '\P{Block=cjkunifiedideographsextensionb}', "");
Expect(1, 173791, '\P{^Block=cjkunifiedideographsextensionb}', "");
Expect(0, 1, '\p{Block=cjkunifiedideographsextensionb}', "");
Expect(1, 1, '\p{^Block=cjkunifiedideographsextensionb}', "");
Expect(1, 1, '\P{Block=cjkunifiedideographsextensionb}', "");
Expect(0, 1, '\P{^Block=cjkunifiedideographsextensionb}', "");
Expect(1, 173791, '\p{Block=	 CJK unified_ideographs extension_B}', "");
Expect(0, 173791, '\p{^Block=	 CJK unified_ideographs extension_B}', "");
Expect(0, 173791, '\P{Block=	 CJK unified_ideographs extension_B}', "");
Expect(1, 173791, '\P{^Block=	 CJK unified_ideographs extension_B}', "");
Expect(0, 1, '\p{Block=	 CJK unified_ideographs extension_B}', "");
Expect(1, 1, '\p{^Block=	 CJK unified_ideographs extension_B}', "");
Expect(1, 1, '\P{Block=	 CJK unified_ideographs extension_B}', "");
Expect(0, 1, '\P{^Block=	 CJK unified_ideographs extension_B}', "");
Error('\p{Blk= :=cjk UNIFIED	Ideographs extensionB}');
Error('\P{Blk= :=cjk UNIFIED	Ideographs extensionB}');
Expect(1, 173791, '\p{Blk=cjkunifiedideographsextensionb}', "");
Expect(0, 173791, '\p{^Blk=cjkunifiedideographsextensionb}', "");
Expect(0, 173791, '\P{Blk=cjkunifiedideographsextensionb}', "");
Expect(1, 173791, '\P{^Blk=cjkunifiedideographsextensionb}', "");
Expect(0, 1, '\p{Blk=cjkunifiedideographsextensionb}', "");
Expect(1, 1, '\p{^Blk=cjkunifiedideographsextensionb}', "");
Expect(1, 1, '\P{Blk=cjkunifiedideographsextensionb}', "");
Expect(0, 1, '\P{^Blk=cjkunifiedideographsextensionb}', "");
Expect(1, 173791, '\p{Blk=_-CJK unifiedIdeographs_ExtensionB}', "");
Expect(0, 173791, '\p{^Blk=_-CJK unifiedIdeographs_ExtensionB}', "");
Expect(0, 173791, '\P{Blk=_-CJK unifiedIdeographs_ExtensionB}', "");
Expect(1, 173791, '\P{^Blk=_-CJK unifiedIdeographs_ExtensionB}', "");
Expect(0, 1, '\p{Blk=_-CJK unifiedIdeographs_ExtensionB}', "");
Expect(1, 1, '\p{^Blk=_-CJK unifiedIdeographs_ExtensionB}', "");
Expect(1, 1, '\P{Blk=_-CJK unifiedIdeographs_ExtensionB}', "");
Expect(0, 1, '\P{^Blk=_-CJK unifiedIdeographs_ExtensionB}', "");
Error('\p{Is_Block=-/a/CJK_Unified Ideographs_Extension_b}');
Error('\P{Is_Block=-/a/CJK_Unified Ideographs_Extension_b}');
Expect(1, 173791, '\p{Is_Block=cjkunifiedideographsextensionb}', "");
Expect(0, 173791, '\p{^Is_Block=cjkunifiedideographsextensionb}', "");
Expect(0, 173791, '\P{Is_Block=cjkunifiedideographsextensionb}', "");
Expect(1, 173791, '\P{^Is_Block=cjkunifiedideographsextensionb}', "");
Expect(0, 1, '\p{Is_Block=cjkunifiedideographsextensionb}', "");
Expect(1, 1, '\p{^Is_Block=cjkunifiedideographsextensionb}', "");
Expect(1, 1, '\P{Is_Block=cjkunifiedideographsextensionb}', "");
Expect(0, 1, '\P{^Is_Block=cjkunifiedideographsextensionb}', "");
Expect(1, 173791, '\p{Is_Block=_ CJK-Unified-Ideographs-extension	b}', "");
Expect(0, 173791, '\p{^Is_Block=_ CJK-Unified-Ideographs-extension	b}', "");
Expect(0, 173791, '\P{Is_Block=_ CJK-Unified-Ideographs-extension	b}', "");
Expect(1, 173791, '\P{^Is_Block=_ CJK-Unified-Ideographs-extension	b}', "");
Expect(0, 1, '\p{Is_Block=_ CJK-Unified-Ideographs-extension	b}', "");
Expect(1, 1, '\p{^Is_Block=_ CJK-Unified-Ideographs-extension	b}', "");
Expect(1, 1, '\P{Is_Block=_ CJK-Unified-Ideographs-extension	b}', "");
Expect(0, 1, '\P{^Is_Block=_ CJK-Unified-Ideographs-extension	b}', "");
Error('\p{Is_Blk=:=CJK_UNIFIED	IDEOGRAPHS_extension B}');
Error('\P{Is_Blk=:=CJK_UNIFIED	IDEOGRAPHS_extension B}');
Expect(1, 173791, '\p{Is_Blk:   cjkunifiedideographsextensionb}', "");
Expect(0, 173791, '\p{^Is_Blk:   cjkunifiedideographsextensionb}', "");
Expect(0, 173791, '\P{Is_Blk:   cjkunifiedideographsextensionb}', "");
Expect(1, 173791, '\P{^Is_Blk:   cjkunifiedideographsextensionb}', "");
Expect(0, 1, '\p{Is_Blk:   cjkunifiedideographsextensionb}', "");
Expect(1, 1, '\p{^Is_Blk:   cjkunifiedideographsextensionb}', "");
Expect(1, 1, '\P{Is_Blk:   cjkunifiedideographsextensionb}', "");
Expect(0, 1, '\P{^Is_Blk:   cjkunifiedideographsextensionb}', "");
Expect(1, 173791, '\p{Is_Blk:  cjkUNIFIEDIdeographs_Extension	B}', "");
Expect(0, 173791, '\p{^Is_Blk:  cjkUNIFIEDIdeographs_Extension	B}', "");
Expect(0, 173791, '\P{Is_Blk:  cjkUNIFIEDIdeographs_Extension	B}', "");
Expect(1, 173791, '\P{^Is_Blk:  cjkUNIFIEDIdeographs_Extension	B}', "");
Expect(0, 1, '\p{Is_Blk:  cjkUNIFIEDIdeographs_Extension	B}', "");
Expect(1, 1, '\p{^Is_Blk:  cjkUNIFIEDIdeographs_Extension	B}', "");
Expect(1, 1, '\P{Is_Blk:  cjkUNIFIEDIdeographs_Extension	B}', "");
Expect(0, 1, '\P{^Is_Blk:  cjkUNIFIEDIdeographs_Extension	B}', "");
Error('\p{Block=/a/		Old_Italic}');
Error('\P{Block=/a/		Old_Italic}');
Expect(1, 66351, '\p{Block=olditalic}', "");
Expect(0, 66351, '\p{^Block=olditalic}', "");
Expect(0, 66351, '\P{Block=olditalic}', "");
Expect(1, 66351, '\P{^Block=olditalic}', "");
Expect(0, 66303, '\p{Block=olditalic}', "");
Expect(1, 66303, '\p{^Block=olditalic}', "");
Expect(1, 66303, '\P{Block=olditalic}', "");
Expect(0, 66303, '\P{^Block=olditalic}', "");
Expect(1, 66351, '\p{Block=- OLD	Italic}', "");
Expect(0, 66351, '\p{^Block=- OLD	Italic}', "");
Expect(0, 66351, '\P{Block=- OLD	Italic}', "");
Expect(1, 66351, '\P{^Block=- OLD	Italic}', "");
Expect(0, 66303, '\p{Block=- OLD	Italic}', "");
Expect(1, 66303, '\p{^Block=- OLD	Italic}', "");
Expect(1, 66303, '\P{Block=- OLD	Italic}', "");
Expect(0, 66303, '\P{^Block=- OLD	Italic}', "");
Error('\p{Blk=:= 	old Italic}');
Error('\P{Blk=:= 	old Italic}');
Expect(1, 66351, '\p{Blk=olditalic}', "");
Expect(0, 66351, '\p{^Blk=olditalic}', "");
Expect(0, 66351, '\P{Blk=olditalic}', "");
Expect(1, 66351, '\P{^Blk=olditalic}', "");
Expect(0, 66303, '\p{Blk=olditalic}', "");
Expect(1, 66303, '\p{^Blk=olditalic}', "");
Expect(1, 66303, '\P{Blk=olditalic}', "");
Expect(0, 66303, '\P{^Blk=olditalic}', "");
Expect(1, 66351, '\p{Blk=	 old Italic}', "");
Expect(0, 66351, '\p{^Blk=	 old Italic}', "");
Expect(0, 66351, '\P{Blk=	 old Italic}', "");
Expect(1, 66351, '\P{^Blk=	 old Italic}', "");
Expect(0, 66303, '\p{Blk=	 old Italic}', "");
Expect(1, 66303, '\p{^Blk=	 old Italic}', "");
Expect(1, 66303, '\P{Blk=	 old Italic}', "");
Expect(0, 66303, '\P{^Blk=	 old Italic}', "");
Error('\p{Is_Block=:= OldITALIC}');
Error('\P{Is_Block=:= OldITALIC}');
Expect(1, 66351, '\p{Is_Block=olditalic}', "");
Expect(0, 66351, '\p{^Is_Block=olditalic}', "");
Expect(0, 66351, '\P{Is_Block=olditalic}', "");
Expect(1, 66351, '\P{^Is_Block=olditalic}', "");
Expect(0, 66303, '\p{Is_Block=olditalic}', "");
Expect(1, 66303, '\p{^Is_Block=olditalic}', "");
Expect(1, 66303, '\P{Is_Block=olditalic}', "");
Expect(0, 66303, '\P{^Is_Block=olditalic}', "");
Expect(1, 66351, '\p{Is_Block=	_OLD Italic}', "");
Expect(0, 66351, '\p{^Is_Block=	_OLD Italic}', "");
Expect(0, 66351, '\P{Is_Block=	_OLD Italic}', "");
Expect(1, 66351, '\P{^Is_Block=	_OLD Italic}', "");
Expect(0, 66303, '\p{Is_Block=	_OLD Italic}', "");
Expect(1, 66303, '\p{^Is_Block=	_OLD Italic}', "");
Expect(1, 66303, '\P{Is_Block=	_OLD Italic}', "");
Expect(0, 66303, '\P{^Is_Block=	_OLD Italic}', "");
Error('\p{Is_Blk= OLD	ITALIC/a/}');
Error('\P{Is_Blk= OLD	ITALIC/a/}');
Expect(1, 66351, '\p{Is_Blk=olditalic}', "");
Expect(0, 66351, '\p{^Is_Blk=olditalic}', "");
Expect(0, 66351, '\P{Is_Blk=olditalic}', "");
Expect(1, 66351, '\P{^Is_Blk=olditalic}', "");
Expect(0, 66303, '\p{Is_Blk=olditalic}', "");
Expect(1, 66303, '\p{^Is_Blk=olditalic}', "");
Expect(1, 66303, '\P{Is_Blk=olditalic}', "");
Expect(0, 66303, '\P{^Is_Blk=olditalic}', "");
Expect(1, 66351, '\p{Is_Blk= 	old	italic}', "");
Expect(0, 66351, '\p{^Is_Blk= 	old	italic}', "");
Expect(0, 66351, '\P{Is_Blk= 	old	italic}', "");
Expect(1, 66351, '\P{^Is_Blk= 	old	italic}', "");
Expect(0, 66303, '\p{Is_Blk= 	old	italic}', "");
Expect(1, 66303, '\p{^Is_Blk= 	old	italic}', "");
Expect(1, 66303, '\P{Is_Blk= 	old	italic}', "");
Expect(0, 66303, '\P{^Is_Blk= 	old	italic}', "");
Error('\p{Block=-hangul/a/syllables}');
Error('\P{Block=-hangul/a/syllables}');
Expect(1, 55215, '\p{Block=hangulsyllables}', "");
Expect(0, 55215, '\p{^Block=hangulsyllables}', "");
Expect(0, 55215, '\P{Block=hangulsyllables}', "");
Expect(1, 55215, '\P{^Block=hangulsyllables}', "");
Expect(0, 44031, '\p{Block=hangulsyllables}', "");
Expect(1, 44031, '\p{^Block=hangulsyllables}', "");
Expect(1, 44031, '\P{Block=hangulsyllables}', "");
Expect(0, 44031, '\P{^Block=hangulsyllables}', "");
Expect(1, 55215, '\p{Block= Hangul-Syllables}', "");
Expect(0, 55215, '\p{^Block= Hangul-Syllables}', "");
Expect(0, 55215, '\P{Block= Hangul-Syllables}', "");
Expect(1, 55215, '\P{^Block= Hangul-Syllables}', "");
Expect(0, 44031, '\p{Block= Hangul-Syllables}', "");
Expect(1, 44031, '\p{^Block= Hangul-Syllables}', "");
Expect(1, 44031, '\P{Block= Hangul-Syllables}', "");
Expect(0, 44031, '\P{^Block= Hangul-Syllables}', "");
Error('\p{Blk=	:=hangulSyllables}');
Error('\P{Blk=	:=hangulSyllables}');
Expect(1, 55215, '\p{Blk=hangulsyllables}', "");
Expect(0, 55215, '\p{^Blk=hangulsyllables}', "");
Expect(0, 55215, '\P{Blk=hangulsyllables}', "");
Expect(1, 55215, '\P{^Blk=hangulsyllables}', "");
Expect(0, 44031, '\p{Blk=hangulsyllables}', "");
Expect(1, 44031, '\p{^Blk=hangulsyllables}', "");
Expect(1, 44031, '\P{Blk=hangulsyllables}', "");
Expect(0, 44031, '\P{^Blk=hangulsyllables}', "");
Expect(1, 55215, '\p{Blk=	_HANGULSYLLABLES}', "");
Expect(0, 55215, '\p{^Blk=	_HANGULSYLLABLES}', "");
Expect(0, 55215, '\P{Blk=	_HANGULSYLLABLES}', "");
Expect(1, 55215, '\P{^Blk=	_HANGULSYLLABLES}', "");
Expect(0, 44031, '\p{Blk=	_HANGULSYLLABLES}', "");
Expect(1, 44031, '\p{^Blk=	_HANGULSYLLABLES}', "");
Expect(1, 44031, '\P{Blk=	_HANGULSYLLABLES}', "");
Expect(0, 44031, '\P{^Blk=	_HANGULSYLLABLES}', "");
Error('\p{Is_Block=/a/-Hangul_Syllables}');
Error('\P{Is_Block=/a/-Hangul_Syllables}');
Expect(1, 55215, '\p{Is_Block: hangulsyllables}', "");
Expect(0, 55215, '\p{^Is_Block: hangulsyllables}', "");
Expect(0, 55215, '\P{Is_Block: hangulsyllables}', "");
Expect(1, 55215, '\P{^Is_Block: hangulsyllables}', "");
Expect(0, 44031, '\p{Is_Block: hangulsyllables}', "");
Expect(1, 44031, '\p{^Is_Block: hangulsyllables}', "");
Expect(1, 44031, '\P{Is_Block: hangulsyllables}', "");
Expect(0, 44031, '\P{^Is_Block: hangulsyllables}', "");
Expect(1, 55215, '\p{Is_Block=	-Hangul-Syllables}', "");
Expect(0, 55215, '\p{^Is_Block=	-Hangul-Syllables}', "");
Expect(0, 55215, '\P{Is_Block=	-Hangul-Syllables}', "");
Expect(1, 55215, '\P{^Is_Block=	-Hangul-Syllables}', "");
Expect(0, 44031, '\p{Is_Block=	-Hangul-Syllables}', "");
Expect(1, 44031, '\p{^Is_Block=	-Hangul-Syllables}', "");
Expect(1, 44031, '\P{Is_Block=	-Hangul-Syllables}', "");
Expect(0, 44031, '\P{^Is_Block=	-Hangul-Syllables}', "");
Error('\p{Is_Blk=:=	HANGUL	syllables}');
Error('\P{Is_Blk=:=	HANGUL	syllables}');
Expect(1, 55215, '\p{Is_Blk=hangulsyllables}', "");
Expect(0, 55215, '\p{^Is_Blk=hangulsyllables}', "");
Expect(0, 55215, '\P{Is_Blk=hangulsyllables}', "");
Expect(1, 55215, '\P{^Is_Blk=hangulsyllables}', "");
Expect(0, 44031, '\p{Is_Blk=hangulsyllables}', "");
Expect(1, 44031, '\p{^Is_Blk=hangulsyllables}', "");
Expect(1, 44031, '\P{Is_Blk=hangulsyllables}', "");
Expect(0, 44031, '\P{^Is_Blk=hangulsyllables}', "");
Expect(1, 55215, '\p{Is_Blk= HANGUL Syllables}', "");
Expect(0, 55215, '\p{^Is_Blk= HANGUL Syllables}', "");
Expect(0, 55215, '\P{Is_Blk= HANGUL Syllables}', "");
Expect(1, 55215, '\P{^Is_Blk= HANGUL Syllables}', "");
Expect(0, 44031, '\p{Is_Blk= HANGUL Syllables}', "");
Expect(1, 44031, '\p{^Is_Blk= HANGUL Syllables}', "");
Expect(1, 44031, '\P{Is_Blk= HANGUL Syllables}', "");
Expect(0, 44031, '\P{^Is_Blk= HANGUL Syllables}', "");
Error('\p{Block=		Kana	Supplement:=}');
Error('\P{Block=		Kana	Supplement:=}');
Expect(1, 110847, '\p{Block=kanasupplement}', "");
Expect(0, 110847, '\p{^Block=kanasupplement}', "");
Expect(0, 110847, '\P{Block=kanasupplement}', "");
Expect(1, 110847, '\P{^Block=kanasupplement}', "");
Expect(0, 110591, '\p{Block=kanasupplement}', "");
Expect(1, 110591, '\p{^Block=kanasupplement}', "");
Expect(1, 110591, '\P{Block=kanasupplement}', "");
Expect(0, 110591, '\P{^Block=kanasupplement}', "");
Expect(1, 110847, '\p{Block= _KANASupplement}', "");
Expect(0, 110847, '\p{^Block= _KANASupplement}', "");
Expect(0, 110847, '\P{Block= _KANASupplement}', "");
Expect(1, 110847, '\P{^Block= _KANASupplement}', "");
Expect(0, 110591, '\p{Block= _KANASupplement}', "");
Expect(1, 110591, '\p{^Block= _KANASupplement}', "");
Expect(1, 110591, '\P{Block= _KANASupplement}', "");
Expect(0, 110591, '\P{^Block= _KANASupplement}', "");
Error('\p{Blk=:=kana-SUPPLEMENT}');
Error('\P{Blk=:=kana-SUPPLEMENT}');
Expect(1, 110847, '\p{Blk=kanasupplement}', "");
Expect(0, 110847, '\p{^Blk=kanasupplement}', "");
Expect(0, 110847, '\P{Blk=kanasupplement}', "");
Expect(1, 110847, '\P{^Blk=kanasupplement}', "");
Expect(0, 110591, '\p{Blk=kanasupplement}', "");
Expect(1, 110591, '\p{^Blk=kanasupplement}', "");
Expect(1, 110591, '\P{Blk=kanasupplement}', "");
Expect(0, 110591, '\P{^Blk=kanasupplement}', "");
Expect(1, 110847, '\p{Blk= 	KANA	supplement}', "");
Expect(0, 110847, '\p{^Blk= 	KANA	supplement}', "");
Expect(0, 110847, '\P{Blk= 	KANA	supplement}', "");
Expect(1, 110847, '\P{^Blk= 	KANA	supplement}', "");
Expect(0, 110591, '\p{Blk= 	KANA	supplement}', "");
Expect(1, 110591, '\p{^Blk= 	KANA	supplement}', "");
Expect(1, 110591, '\P{Blk= 	KANA	supplement}', "");
Expect(0, 110591, '\P{^Blk= 	KANA	supplement}', "");
Error('\p{Is_Block=  kana	SUPPLEMENT/a/}');
Error('\P{Is_Block=  kana	SUPPLEMENT/a/}');
Expect(1, 110847, '\p{Is_Block:	kanasupplement}', "");
Expect(0, 110847, '\p{^Is_Block:	kanasupplement}', "");
Expect(0, 110847, '\P{Is_Block:	kanasupplement}', "");
Expect(1, 110847, '\P{^Is_Block:	kanasupplement}', "");
Expect(0, 110591, '\p{Is_Block:	kanasupplement}', "");
Expect(1, 110591, '\p{^Is_Block:	kanasupplement}', "");
Expect(1, 110591, '\P{Is_Block:	kanasupplement}', "");
Expect(0, 110591, '\P{^Is_Block:	kanasupplement}', "");
Expect(1, 110847, '\p{Is_Block=- KANA-Supplement}', "");
Expect(0, 110847, '\p{^Is_Block=- KANA-Supplement}', "");
Expect(0, 110847, '\P{Is_Block=- KANA-Supplement}', "");
Expect(1, 110847, '\P{^Is_Block=- KANA-Supplement}', "");
Expect(0, 110591, '\p{Is_Block=- KANA-Supplement}', "");
Expect(1, 110591, '\p{^Is_Block=- KANA-Supplement}', "");
Expect(1, 110591, '\P{Is_Block=- KANA-Supplement}', "");
Expect(0, 110591, '\P{^Is_Block=- KANA-Supplement}', "");
Error('\p{Is_Blk:-_kana/a/Supplement}');
Error('\P{Is_Blk:-_kana/a/Supplement}');
Expect(1, 110847, '\p{Is_Blk=kanasupplement}', "");
Expect(0, 110847, '\p{^Is_Blk=kanasupplement}', "");
Expect(0, 110847, '\P{Is_Blk=kanasupplement}', "");
Expect(1, 110847, '\P{^Is_Blk=kanasupplement}', "");
Expect(0, 110591, '\p{Is_Blk=kanasupplement}', "");
Expect(1, 110591, '\p{^Is_Blk=kanasupplement}', "");
Expect(1, 110591, '\P{Is_Blk=kanasupplement}', "");
Expect(0, 110591, '\P{^Is_Blk=kanasupplement}', "");
Expect(1, 110847, '\p{Is_Blk=	Kana-supplement}', "");
Expect(0, 110847, '\p{^Is_Blk=	Kana-supplement}', "");
Expect(0, 110847, '\P{Is_Blk=	Kana-supplement}', "");
Expect(1, 110847, '\P{^Is_Blk=	Kana-supplement}', "");
Expect(0, 110591, '\p{Is_Blk=	Kana-supplement}', "");
Expect(1, 110591, '\p{^Is_Blk=	Kana-supplement}', "");
Expect(1, 110591, '\P{Is_Blk=	Kana-supplement}', "");
Expect(0, 110591, '\P{^Is_Blk=	Kana-supplement}', "");
Error('\p{Block=:=Yijing-HEXAGRAM	symbols}');
Error('\P{Block=:=Yijing-HEXAGRAM	symbols}');
Expect(1, 19967, '\p{Block=yijinghexagramsymbols}', "");
Expect(0, 19967, '\p{^Block=yijinghexagramsymbols}', "");
Expect(0, 19967, '\P{Block=yijinghexagramsymbols}', "");
Expect(1, 19967, '\P{^Block=yijinghexagramsymbols}', "");
Expect(0, 19903, '\p{Block=yijinghexagramsymbols}', "");
Expect(1, 19903, '\p{^Block=yijinghexagramsymbols}', "");
Expect(1, 19903, '\P{Block=yijinghexagramsymbols}', "");
Expect(0, 19903, '\P{^Block=yijinghexagramsymbols}', "");
Expect(1, 19967, '\p{Block=-_Yijing_hexagram	symbols}', "");
Expect(0, 19967, '\p{^Block=-_Yijing_hexagram	symbols}', "");
Expect(0, 19967, '\P{Block=-_Yijing_hexagram	symbols}', "");
Expect(1, 19967, '\P{^Block=-_Yijing_hexagram	symbols}', "");
Expect(0, 19903, '\p{Block=-_Yijing_hexagram	symbols}', "");
Expect(1, 19903, '\p{^Block=-_Yijing_hexagram	symbols}', "");
Expect(1, 19903, '\P{Block=-_Yijing_hexagram	symbols}', "");
Expect(0, 19903, '\P{^Block=-_Yijing_hexagram	symbols}', "");
Error('\p{Blk=	 yijing:=Hexagram_SYMBOLS}');
Error('\P{Blk=	 yijing:=Hexagram_SYMBOLS}');
Expect(1, 19967, '\p{Blk=yijinghexagramsymbols}', "");
Expect(0, 19967, '\p{^Blk=yijinghexagramsymbols}', "");
Expect(0, 19967, '\P{Blk=yijinghexagramsymbols}', "");
Expect(1, 19967, '\P{^Blk=yijinghexagramsymbols}', "");
Expect(0, 19903, '\p{Blk=yijinghexagramsymbols}', "");
Expect(1, 19903, '\p{^Blk=yijinghexagramsymbols}', "");
Expect(1, 19903, '\P{Blk=yijinghexagramsymbols}', "");
Expect(0, 19903, '\P{^Blk=yijinghexagramsymbols}', "");
Expect(1, 19967, '\p{Blk=-	yijing Hexagram SYMBOLS}', "");
Expect(0, 19967, '\p{^Blk=-	yijing Hexagram SYMBOLS}', "");
Expect(0, 19967, '\P{Blk=-	yijing Hexagram SYMBOLS}', "");
Expect(1, 19967, '\P{^Blk=-	yijing Hexagram SYMBOLS}', "");
Expect(0, 19903, '\p{Blk=-	yijing Hexagram SYMBOLS}', "");
Expect(1, 19903, '\p{^Blk=-	yijing Hexagram SYMBOLS}', "");
Expect(1, 19903, '\P{Blk=-	yijing Hexagram SYMBOLS}', "");
Expect(0, 19903, '\P{^Blk=-	yijing Hexagram SYMBOLS}', "");
Error('\p{Is_Block=_yijing	hexagram-SYMBOLS:=}');
Error('\P{Is_Block=_yijing	hexagram-SYMBOLS:=}');
Expect(1, 19967, '\p{Is_Block=yijinghexagramsymbols}', "");
Expect(0, 19967, '\p{^Is_Block=yijinghexagramsymbols}', "");
Expect(0, 19967, '\P{Is_Block=yijinghexagramsymbols}', "");
Expect(1, 19967, '\P{^Is_Block=yijinghexagramsymbols}', "");
Expect(0, 19903, '\p{Is_Block=yijinghexagramsymbols}', "");
Expect(1, 19903, '\p{^Is_Block=yijinghexagramsymbols}', "");
Expect(1, 19903, '\P{Is_Block=yijinghexagramsymbols}', "");
Expect(0, 19903, '\P{^Is_Block=yijinghexagramsymbols}', "");
Expect(1, 19967, '\p{Is_Block=- YIJINGhexagram-SYMBOLS}', "");
Expect(0, 19967, '\p{^Is_Block=- YIJINGhexagram-SYMBOLS}', "");
Expect(0, 19967, '\P{Is_Block=- YIJINGhexagram-SYMBOLS}', "");
Expect(1, 19967, '\P{^Is_Block=- YIJINGhexagram-SYMBOLS}', "");
Expect(0, 19903, '\p{Is_Block=- YIJINGhexagram-SYMBOLS}', "");
Expect(1, 19903, '\p{^Is_Block=- YIJINGhexagram-SYMBOLS}', "");
Expect(1, 19903, '\P{Is_Block=- YIJINGhexagram-SYMBOLS}', "");
Expect(0, 19903, '\P{^Is_Block=- YIJINGhexagram-SYMBOLS}', "");
Error('\p{Is_Blk= _Yijing hexagram:=Symbols}');
Error('\P{Is_Blk= _Yijing hexagram:=Symbols}');
Expect(1, 19967, '\p{Is_Blk:   yijinghexagramsymbols}', "");
Expect(0, 19967, '\p{^Is_Blk:   yijinghexagramsymbols}', "");
Expect(0, 19967, '\P{Is_Blk:   yijinghexagramsymbols}', "");
Expect(1, 19967, '\P{^Is_Blk:   yijinghexagramsymbols}', "");
Expect(0, 19903, '\p{Is_Blk:   yijinghexagramsymbols}', "");
Expect(1, 19903, '\p{^Is_Blk:   yijinghexagramsymbols}', "");
Expect(1, 19903, '\P{Is_Blk:   yijinghexagramsymbols}', "");
Expect(0, 19903, '\P{^Is_Blk:   yijinghexagramsymbols}', "");
Expect(1, 19967, '\p{Is_Blk=		yijing hexagram-symbols}', "");
Expect(0, 19967, '\p{^Is_Blk=		yijing hexagram-symbols}', "");
Expect(0, 19967, '\P{Is_Blk=		yijing hexagram-symbols}', "");
Expect(1, 19967, '\P{^Is_Blk=		yijing hexagram-symbols}', "");
Expect(0, 19903, '\p{Is_Blk=		yijing hexagram-symbols}', "");
Expect(1, 19903, '\p{^Is_Blk=		yijing hexagram-symbols}', "");
Expect(1, 19903, '\P{Is_Blk=		yijing hexagram-symbols}', "");
Expect(0, 19903, '\P{^Is_Blk=		yijing hexagram-symbols}', "");
Error('\p{Block=_:=PHAISTOSDISC}');
Error('\P{Block=_:=PHAISTOSDISC}');
Expect(1, 66047, '\p{Block=phaistosdisc}', "");
Expect(0, 66047, '\p{^Block=phaistosdisc}', "");
Expect(0, 66047, '\P{Block=phaistosdisc}', "");
Expect(1, 66047, '\P{^Block=phaistosdisc}', "");
Expect(0, 65999, '\p{Block=phaistosdisc}', "");
Expect(1, 65999, '\p{^Block=phaistosdisc}', "");
Expect(1, 65999, '\P{Block=phaistosdisc}', "");
Expect(0, 65999, '\P{^Block=phaistosdisc}', "");
Expect(1, 66047, '\p{Block=-PHAISTOS_DISC}', "");
Expect(0, 66047, '\p{^Block=-PHAISTOS_DISC}', "");
Expect(0, 66047, '\P{Block=-PHAISTOS_DISC}', "");
Expect(1, 66047, '\P{^Block=-PHAISTOS_DISC}', "");
Expect(0, 65999, '\p{Block=-PHAISTOS_DISC}', "");
Expect(1, 65999, '\p{^Block=-PHAISTOS_DISC}', "");
Expect(1, 65999, '\P{Block=-PHAISTOS_DISC}', "");
Expect(0, 65999, '\P{^Block=-PHAISTOS_DISC}', "");
Error('\p{Blk:    /a/Phaistos Disc}');
Error('\P{Blk:    /a/Phaistos Disc}');
Expect(1, 66047, '\p{Blk=phaistosdisc}', "");
Expect(0, 66047, '\p{^Blk=phaistosdisc}', "");
Expect(0, 66047, '\P{Blk=phaistosdisc}', "");
Expect(1, 66047, '\P{^Blk=phaistosdisc}', "");
Expect(0, 65999, '\p{Blk=phaistosdisc}', "");
Expect(1, 65999, '\p{^Blk=phaistosdisc}', "");
Expect(1, 65999, '\P{Blk=phaistosdisc}', "");
Expect(0, 65999, '\P{^Blk=phaistosdisc}', "");
Expect(1, 66047, '\p{Blk:-_Phaistos	Disc}', "");
Expect(0, 66047, '\p{^Blk:-_Phaistos	Disc}', "");
Expect(0, 66047, '\P{Blk:-_Phaistos	Disc}', "");
Expect(1, 66047, '\P{^Blk:-_Phaistos	Disc}', "");
Expect(0, 65999, '\p{Blk:-_Phaistos	Disc}', "");
Expect(1, 65999, '\p{^Blk:-_Phaistos	Disc}', "");
Expect(1, 65999, '\P{Blk:-_Phaistos	Disc}', "");
Expect(0, 65999, '\P{^Blk:-_Phaistos	Disc}', "");
Error('\p{Is_Block=_/a/phaistos-Disc}');
Error('\P{Is_Block=_/a/phaistos-Disc}');
Expect(1, 66047, '\p{Is_Block=phaistosdisc}', "");
Expect(0, 66047, '\p{^Is_Block=phaistosdisc}', "");
Expect(0, 66047, '\P{Is_Block=phaistosdisc}', "");
Expect(1, 66047, '\P{^Is_Block=phaistosdisc}', "");
Expect(0, 65999, '\p{Is_Block=phaistosdisc}', "");
Expect(1, 65999, '\p{^Is_Block=phaistosdisc}', "");
Expect(1, 65999, '\P{Is_Block=phaistosdisc}', "");
Expect(0, 65999, '\P{^Is_Block=phaistosdisc}', "");
Expect(1, 66047, '\p{Is_Block= PHAISTOS_Disc}', "");
Expect(0, 66047, '\p{^Is_Block= PHAISTOS_Disc}', "");
Expect(0, 66047, '\P{Is_Block= PHAISTOS_Disc}', "");
Expect(1, 66047, '\P{^Is_Block= PHAISTOS_Disc}', "");
Expect(0, 65999, '\p{Is_Block= PHAISTOS_Disc}', "");
Expect(1, 65999, '\p{^Is_Block= PHAISTOS_Disc}', "");
Expect(1, 65999, '\P{Is_Block= PHAISTOS_Disc}', "");
Expect(0, 65999, '\P{^Is_Block= PHAISTOS_Disc}', "");
Error('\p{Is_Blk=/a/phaistos_Disc}');
Error('\P{Is_Blk=/a/phaistos_Disc}');
Expect(1, 66047, '\p{Is_Blk=phaistosdisc}', "");
Expect(0, 66047, '\p{^Is_Blk=phaistosdisc}', "");
Expect(0, 66047, '\P{Is_Blk=phaistosdisc}', "");
Expect(1, 66047, '\P{^Is_Blk=phaistosdisc}', "");
Expect(0, 65999, '\p{Is_Blk=phaistosdisc}', "");
Expect(1, 65999, '\p{^Is_Blk=phaistosdisc}', "");
Expect(1, 65999, '\P{Is_Blk=phaistosdisc}', "");
Expect(0, 65999, '\P{^Is_Blk=phaistosdisc}', "");
Expect(1, 66047, '\p{Is_Blk=__PhaistosDisc}', "");
Expect(0, 66047, '\p{^Is_Blk=__PhaistosDisc}', "");
Expect(0, 66047, '\P{Is_Blk=__PhaistosDisc}', "");
Expect(1, 66047, '\P{^Is_Blk=__PhaistosDisc}', "");
Expect(0, 65999, '\p{Is_Blk=__PhaistosDisc}', "");
Expect(1, 65999, '\p{^Is_Blk=__PhaistosDisc}', "");
Expect(1, 65999, '\P{Is_Blk=__PhaistosDisc}', "");
Expect(0, 65999, '\P{^Is_Blk=__PhaistosDisc}', "");
Error('\p{Block=/a/-_SMALL	Form	variants}');
Error('\P{Block=/a/-_SMALL	Form	variants}');
Expect(1, 65135, '\p{Block=smallformvariants}', "");
Expect(0, 65135, '\p{^Block=smallformvariants}', "");
Expect(0, 65135, '\P{Block=smallformvariants}', "");
Expect(1, 65135, '\P{^Block=smallformvariants}', "");
Expect(0, 65103, '\p{Block=smallformvariants}', "");
Expect(1, 65103, '\p{^Block=smallformvariants}', "");
Expect(1, 65103, '\P{Block=smallformvariants}', "");
Expect(0, 65103, '\P{^Block=smallformvariants}', "");
Expect(1, 65135, '\p{Block:	 SMALL-formVariants}', "");
Expect(0, 65135, '\p{^Block:	 SMALL-formVariants}', "");
Expect(0, 65135, '\P{Block:	 SMALL-formVariants}', "");
Expect(1, 65135, '\P{^Block:	 SMALL-formVariants}', "");
Expect(0, 65103, '\p{Block:	 SMALL-formVariants}', "");
Expect(1, 65103, '\p{^Block:	 SMALL-formVariants}', "");
Expect(1, 65103, '\P{Block:	 SMALL-formVariants}', "");
Expect(0, 65103, '\P{^Block:	 SMALL-formVariants}', "");
Error('\p{Blk=_/a/small_Form VARIANTS}');
Error('\P{Blk=_/a/small_Form VARIANTS}');
Expect(1, 65135, '\p{Blk=smallformvariants}', "");
Expect(0, 65135, '\p{^Blk=smallformvariants}', "");
Expect(0, 65135, '\P{Blk=smallformvariants}', "");
Expect(1, 65135, '\P{^Blk=smallformvariants}', "");
Expect(0, 65103, '\p{Blk=smallformvariants}', "");
Expect(1, 65103, '\p{^Blk=smallformvariants}', "");
Expect(1, 65103, '\P{Blk=smallformvariants}', "");
Expect(0, 65103, '\P{^Blk=smallformvariants}', "");
Expect(1, 65135, '\p{Blk=--Small	FORM-Variants}', "");
Expect(0, 65135, '\p{^Blk=--Small	FORM-Variants}', "");
Expect(0, 65135, '\P{Blk=--Small	FORM-Variants}', "");
Expect(1, 65135, '\P{^Blk=--Small	FORM-Variants}', "");
Expect(0, 65103, '\p{Blk=--Small	FORM-Variants}', "");
Expect(1, 65103, '\p{^Blk=--Small	FORM-Variants}', "");
Expect(1, 65103, '\P{Blk=--Small	FORM-Variants}', "");
Expect(0, 65103, '\P{^Blk=--Small	FORM-Variants}', "");
Error('\p{Is_Block=_Small-form:=variants}');
Error('\P{Is_Block=_Small-form:=variants}');
Expect(1, 65135, '\p{Is_Block=smallformvariants}', "");
Expect(0, 65135, '\p{^Is_Block=smallformvariants}', "");
Expect(0, 65135, '\P{Is_Block=smallformvariants}', "");
Expect(1, 65135, '\P{^Is_Block=smallformvariants}', "");
Expect(0, 65103, '\p{Is_Block=smallformvariants}', "");
Expect(1, 65103, '\p{^Is_Block=smallformvariants}', "");
Expect(1, 65103, '\P{Is_Block=smallformvariants}', "");
Expect(0, 65103, '\P{^Is_Block=smallformvariants}', "");
Expect(1, 65135, '\p{Is_Block=_Small_Formvariants}', "");
Expect(0, 65135, '\p{^Is_Block=_Small_Formvariants}', "");
Expect(0, 65135, '\P{Is_Block=_Small_Formvariants}', "");
Expect(1, 65135, '\P{^Is_Block=_Small_Formvariants}', "");
Expect(0, 65103, '\p{Is_Block=_Small_Formvariants}', "");
Expect(1, 65103, '\p{^Is_Block=_Small_Formvariants}', "");
Expect(1, 65103, '\P{Is_Block=_Small_Formvariants}', "");
Expect(0, 65103, '\P{^Is_Block=_Small_Formvariants}', "");
Error('\p{Is_Blk= -SMALL Form-Variants/a/}');
Error('\P{Is_Blk= -SMALL Form-Variants/a/}');
Expect(1, 65135, '\p{Is_Blk=smallformvariants}', "");
Expect(0, 65135, '\p{^Is_Blk=smallformvariants}', "");
Expect(0, 65135, '\P{Is_Blk=smallformvariants}', "");
Expect(1, 65135, '\P{^Is_Blk=smallformvariants}', "");
Expect(0, 65103, '\p{Is_Blk=smallformvariants}', "");
Expect(1, 65103, '\p{^Is_Blk=smallformvariants}', "");
Expect(1, 65103, '\P{Is_Blk=smallformvariants}', "");
Expect(0, 65103, '\P{^Is_Blk=smallformvariants}', "");
Expect(1, 65135, '\p{Is_Blk=	SMALL_form Variants}', "");
Expect(0, 65135, '\p{^Is_Blk=	SMALL_form Variants}', "");
Expect(0, 65135, '\P{Is_Blk=	SMALL_form Variants}', "");
Expect(1, 65135, '\P{^Is_Blk=	SMALL_form Variants}', "");
Expect(0, 65103, '\p{Is_Blk=	SMALL_form Variants}', "");
Expect(1, 65103, '\p{^Is_Blk=	SMALL_form Variants}', "");
Expect(1, 65103, '\P{Is_Blk=	SMALL_form Variants}', "");
Expect(0, 65103, '\P{^Is_Blk=	SMALL_form Variants}', "");
Error('\p{Block=	:=egyptian-HIEROGLYPHS}');
Error('\P{Block=	:=egyptian-HIEROGLYPHS}');
Expect(1, 78895, '\p{Block=egyptianhieroglyphs}', "");
Expect(0, 78895, '\p{^Block=egyptianhieroglyphs}', "");
Expect(0, 78895, '\P{Block=egyptianhieroglyphs}', "");
Expect(1, 78895, '\P{^Block=egyptianhieroglyphs}', "");
Expect(0, 77823, '\p{Block=egyptianhieroglyphs}', "");
Expect(1, 77823, '\p{^Block=egyptianhieroglyphs}', "");
Expect(1, 77823, '\P{Block=egyptianhieroglyphs}', "");
Expect(0, 77823, '\P{^Block=egyptianhieroglyphs}', "");
Expect(1, 78895, '\p{Block=-egyptian Hieroglyphs}', "");
Expect(0, 78895, '\p{^Block=-egyptian Hieroglyphs}', "");
Expect(0, 78895, '\P{Block=-egyptian Hieroglyphs}', "");
Expect(1, 78895, '\P{^Block=-egyptian Hieroglyphs}', "");
Expect(0, 77823, '\p{Block=-egyptian Hieroglyphs}', "");
Expect(1, 77823, '\p{^Block=-egyptian Hieroglyphs}', "");
Expect(1, 77823, '\P{Block=-egyptian Hieroglyphs}', "");
Expect(0, 77823, '\P{^Block=-egyptian Hieroglyphs}', "");
Error('\p{Blk=/a/_	Egyptian-Hieroglyphs}');
Error('\P{Blk=/a/_	Egyptian-Hieroglyphs}');
Expect(1, 78895, '\p{Blk=egyptianhieroglyphs}', "");
Expect(0, 78895, '\p{^Blk=egyptianhieroglyphs}', "");
Expect(0, 78895, '\P{Blk=egyptianhieroglyphs}', "");
Expect(1, 78895, '\P{^Blk=egyptianhieroglyphs}', "");
Expect(0, 77823, '\p{Blk=egyptianhieroglyphs}', "");
Expect(1, 77823, '\p{^Blk=egyptianhieroglyphs}', "");
Expect(1, 77823, '\P{Blk=egyptianhieroglyphs}', "");
Expect(0, 77823, '\P{^Blk=egyptianhieroglyphs}', "");
Expect(1, 78895, '\p{Blk:	_	EGYPTIAN	HIEROGLYPHS}', "");
Expect(0, 78895, '\p{^Blk:	_	EGYPTIAN	HIEROGLYPHS}', "");
Expect(0, 78895, '\P{Blk:	_	EGYPTIAN	HIEROGLYPHS}', "");
Expect(1, 78895, '\P{^Blk:	_	EGYPTIAN	HIEROGLYPHS}', "");
Expect(0, 77823, '\p{Blk:	_	EGYPTIAN	HIEROGLYPHS}', "");
Expect(1, 77823, '\p{^Blk:	_	EGYPTIAN	HIEROGLYPHS}', "");
Expect(1, 77823, '\P{Blk:	_	EGYPTIAN	HIEROGLYPHS}', "");
Expect(0, 77823, '\P{^Blk:	_	EGYPTIAN	HIEROGLYPHS}', "");
Error('\p{Is_Block=	:=Egyptian_HIEROGLYPHS}');
Error('\P{Is_Block=	:=Egyptian_HIEROGLYPHS}');
Expect(1, 78895, '\p{Is_Block=egyptianhieroglyphs}', "");
Expect(0, 78895, '\p{^Is_Block=egyptianhieroglyphs}', "");
Expect(0, 78895, '\P{Is_Block=egyptianhieroglyphs}', "");
Expect(1, 78895, '\P{^Is_Block=egyptianhieroglyphs}', "");
Expect(0, 77823, '\p{Is_Block=egyptianhieroglyphs}', "");
Expect(1, 77823, '\p{^Is_Block=egyptianhieroglyphs}', "");
Expect(1, 77823, '\P{Is_Block=egyptianhieroglyphs}', "");
Expect(0, 77823, '\P{^Is_Block=egyptianhieroglyphs}', "");
Expect(1, 78895, '\p{Is_Block=-	EGYPTIAN HIEROGLYPHS}', "");
Expect(0, 78895, '\p{^Is_Block=-	EGYPTIAN HIEROGLYPHS}', "");
Expect(0, 78895, '\P{Is_Block=-	EGYPTIAN HIEROGLYPHS}', "");
Expect(1, 78895, '\P{^Is_Block=-	EGYPTIAN HIEROGLYPHS}', "");
Expect(0, 77823, '\p{Is_Block=-	EGYPTIAN HIEROGLYPHS}', "");
Expect(1, 77823, '\p{^Is_Block=-	EGYPTIAN HIEROGLYPHS}', "");
Expect(1, 77823, '\P{Is_Block=-	EGYPTIAN HIEROGLYPHS}', "");
Expect(0, 77823, '\P{^Is_Block=-	EGYPTIAN HIEROGLYPHS}', "");
Error('\p{Is_Blk=/a/	 Egyptianhieroglyphs}');
Error('\P{Is_Blk=/a/	 Egyptianhieroglyphs}');
Expect(1, 78895, '\p{Is_Blk=egyptianhieroglyphs}', "");
Expect(0, 78895, '\p{^Is_Blk=egyptianhieroglyphs}', "");
Expect(0, 78895, '\P{Is_Blk=egyptianhieroglyphs}', "");
Expect(1, 78895, '\P{^Is_Blk=egyptianhieroglyphs}', "");
Expect(0, 77823, '\p{Is_Blk=egyptianhieroglyphs}', "");
Expect(1, 77823, '\p{^Is_Blk=egyptianhieroglyphs}', "");
Expect(1, 77823, '\P{Is_Blk=egyptianhieroglyphs}', "");
Expect(0, 77823, '\P{^Is_Blk=egyptianhieroglyphs}', "");
Expect(1, 78895, '\p{Is_Blk:_-EGYPTIAN-hieroglyphs}', "");
Expect(0, 78895, '\p{^Is_Blk:_-EGYPTIAN-hieroglyphs}', "");
Expect(0, 78895, '\P{Is_Blk:_-EGYPTIAN-hieroglyphs}', "");
Expect(1, 78895, '\P{^Is_Blk:_-EGYPTIAN-hieroglyphs}', "");
Expect(0, 77823, '\p{Is_Blk:_-EGYPTIAN-hieroglyphs}', "");
Expect(1, 77823, '\p{^Is_Blk:_-EGYPTIAN-hieroglyphs}', "");
Expect(1, 77823, '\P{Is_Blk:_-EGYPTIAN-hieroglyphs}', "");
Expect(0, 77823, '\P{^Is_Blk:_-EGYPTIAN-hieroglyphs}', "");
Error('\p{Block=_/a/Latin_EXTENDED-D}');
Error('\P{Block=_/a/Latin_EXTENDED-D}');
Expect(1, 43007, '\p{Block=latinextendedd}', "");
Expect(0, 43007, '\p{^Block=latinextendedd}', "");
Expect(0, 43007, '\P{Block=latinextendedd}', "");
Expect(1, 43007, '\P{^Block=latinextendedd}', "");
Expect(0, 42783, '\p{Block=latinextendedd}', "");
Expect(1, 42783, '\p{^Block=latinextendedd}', "");
Expect(1, 42783, '\P{Block=latinextendedd}', "");
Expect(0, 42783, '\P{^Block=latinextendedd}', "");
Expect(1, 43007, '\p{Block=		latin-EXTENDEDD}', "");
Expect(0, 43007, '\p{^Block=		latin-EXTENDEDD}', "");
Expect(0, 43007, '\P{Block=		latin-EXTENDEDD}', "");
Expect(1, 43007, '\P{^Block=		latin-EXTENDEDD}', "");
Expect(0, 42783, '\p{Block=		latin-EXTENDEDD}', "");
Expect(1, 42783, '\p{^Block=		latin-EXTENDEDD}', "");
Expect(1, 42783, '\P{Block=		latin-EXTENDEDD}', "");
Expect(0, 42783, '\P{^Block=		latin-EXTENDEDD}', "");
Error('\p{Blk=- Latin:=EXTENDED-D}');
Error('\P{Blk=- Latin:=EXTENDED-D}');
Expect(1, 43007, '\p{Blk=latinextendedd}', "");
Expect(0, 43007, '\p{^Blk=latinextendedd}', "");
Expect(0, 43007, '\P{Blk=latinextendedd}', "");
Expect(1, 43007, '\P{^Blk=latinextendedd}', "");
Expect(0, 42783, '\p{Blk=latinextendedd}', "");
Expect(1, 42783, '\p{^Blk=latinextendedd}', "");
Expect(1, 42783, '\P{Blk=latinextendedd}', "");
Expect(0, 42783, '\P{^Blk=latinextendedd}', "");
Expect(1, 43007, '\p{Blk=	latin-Extended	d}', "");
Expect(0, 43007, '\p{^Blk=	latin-Extended	d}', "");
Expect(0, 43007, '\P{Blk=	latin-Extended	d}', "");
Expect(1, 43007, '\P{^Blk=	latin-Extended	d}', "");
Expect(0, 42783, '\p{Blk=	latin-Extended	d}', "");
Expect(1, 42783, '\p{^Blk=	latin-Extended	d}', "");
Expect(1, 42783, '\P{Blk=	latin-Extended	d}', "");
Expect(0, 42783, '\P{^Blk=	latin-Extended	d}', "");
Error('\p{Is_Block= :=LATIN	ExtendedD}');
Error('\P{Is_Block= :=LATIN	ExtendedD}');
Expect(1, 43007, '\p{Is_Block=latinextendedd}', "");
Expect(0, 43007, '\p{^Is_Block=latinextendedd}', "");
Expect(0, 43007, '\P{Is_Block=latinextendedd}', "");
Expect(1, 43007, '\P{^Is_Block=latinextendedd}', "");
Expect(0, 42783, '\p{Is_Block=latinextendedd}', "");
Expect(1, 42783, '\p{^Is_Block=latinextendedd}', "");
Expect(1, 42783, '\P{Is_Block=latinextendedd}', "");
Expect(0, 42783, '\P{^Is_Block=latinextendedd}', "");
Expect(1, 43007, '\p{Is_Block=	-LATIN Extended_D}', "");
Expect(0, 43007, '\p{^Is_Block=	-LATIN Extended_D}', "");
Expect(0, 43007, '\P{Is_Block=	-LATIN Extended_D}', "");
Expect(1, 43007, '\P{^Is_Block=	-LATIN Extended_D}', "");
Expect(0, 42783, '\p{Is_Block=	-LATIN Extended_D}', "");
Expect(1, 42783, '\p{^Is_Block=	-LATIN Extended_D}', "");
Expect(1, 42783, '\P{Is_Block=	-LATIN Extended_D}', "");
Expect(0, 42783, '\P{^Is_Block=	-LATIN Extended_D}', "");
Error('\p{Is_Blk=_/a/Latin EXTENDED D}');
Error('\P{Is_Blk=_/a/Latin EXTENDED D}');
Expect(1, 43007, '\p{Is_Blk=latinextendedd}', "");
Expect(0, 43007, '\p{^Is_Blk=latinextendedd}', "");
Expect(0, 43007, '\P{Is_Blk=latinextendedd}', "");
Expect(1, 43007, '\P{^Is_Blk=latinextendedd}', "");
Expect(0, 42783, '\p{Is_Blk=latinextendedd}', "");
Expect(1, 42783, '\p{^Is_Blk=latinextendedd}', "");
Expect(1, 42783, '\P{Is_Blk=latinextendedd}', "");
Expect(0, 42783, '\P{^Is_Blk=latinextendedd}', "");
Expect(1, 43007, '\p{Is_Blk=-	LATINExtendedD}', "");
Expect(0, 43007, '\p{^Is_Blk=-	LATINExtendedD}', "");
Expect(0, 43007, '\P{Is_Blk=-	LATINExtendedD}', "");
Expect(1, 43007, '\P{^Is_Blk=-	LATINExtendedD}', "");
Expect(0, 42783, '\p{Is_Blk=-	LATINExtendedD}', "");
Expect(1, 42783, '\p{^Is_Blk=-	LATINExtendedD}', "");
Expect(1, 42783, '\P{Is_Blk=-	LATINExtendedD}', "");
Expect(0, 42783, '\P{^Is_Blk=-	LATINExtendedD}', "");
Error('\p{Block=	Alphabetic-Presentation_FORMS:=}');
Error('\P{Block=	Alphabetic-Presentation_FORMS:=}');
Expect(1, 64335, '\p{Block:alphabeticpresentationforms}', "");
Expect(0, 64335, '\p{^Block:alphabeticpresentationforms}', "");
Expect(0, 64335, '\P{Block:alphabeticpresentationforms}', "");
Expect(1, 64335, '\P{^Block:alphabeticpresentationforms}', "");
Expect(0, 64255, '\p{Block:alphabeticpresentationforms}', "");
Expect(1, 64255, '\p{^Block:alphabeticpresentationforms}', "");
Expect(1, 64255, '\P{Block:alphabeticpresentationforms}', "");
Expect(0, 64255, '\P{^Block:alphabeticpresentationforms}', "");
Expect(1, 64335, '\p{Block:	-Alphabetic PRESENTATION-forms}', "");
Expect(0, 64335, '\p{^Block:	-Alphabetic PRESENTATION-forms}', "");
Expect(0, 64335, '\P{Block:	-Alphabetic PRESENTATION-forms}', "");
Expect(1, 64335, '\P{^Block:	-Alphabetic PRESENTATION-forms}', "");
Expect(0, 64255, '\p{Block:	-Alphabetic PRESENTATION-forms}', "");
Expect(1, 64255, '\p{^Block:	-Alphabetic PRESENTATION-forms}', "");
Expect(1, 64255, '\P{Block:	-Alphabetic PRESENTATION-forms}', "");
Expect(0, 64255, '\P{^Block:	-Alphabetic PRESENTATION-forms}', "");
Error('\p{Blk=__alphabetic:=PRESENTATIONFORMS}');
Error('\P{Blk=__alphabetic:=PRESENTATIONFORMS}');
Expect(1, 64335, '\p{Blk=alphabeticpresentationforms}', "");
Expect(0, 64335, '\p{^Blk=alphabeticpresentationforms}', "");
Expect(0, 64335, '\P{Blk=alphabeticpresentationforms}', "");
Expect(1, 64335, '\P{^Blk=alphabeticpresentationforms}', "");
Expect(0, 64255, '\p{Blk=alphabeticpresentationforms}', "");
Expect(1, 64255, '\p{^Blk=alphabeticpresentationforms}', "");
Expect(1, 64255, '\P{Blk=alphabeticpresentationforms}', "");
Expect(0, 64255, '\P{^Blk=alphabeticpresentationforms}', "");
Expect(1, 64335, '\p{Blk= _alphabetic	PRESENTATION	Forms}', "");
Expect(0, 64335, '\p{^Blk= _alphabetic	PRESENTATION	Forms}', "");
Expect(0, 64335, '\P{Blk= _alphabetic	PRESENTATION	Forms}', "");
Expect(1, 64335, '\P{^Blk= _alphabetic	PRESENTATION	Forms}', "");
Expect(0, 64255, '\p{Blk= _alphabetic	PRESENTATION	Forms}', "");
Expect(1, 64255, '\p{^Blk= _alphabetic	PRESENTATION	Forms}', "");
Expect(1, 64255, '\P{Blk= _alphabetic	PRESENTATION	Forms}', "");
Expect(0, 64255, '\P{^Blk= _alphabetic	PRESENTATION	Forms}', "");
Error('\p{Is_Block= :=Alphabetic	presentation Forms}');
Error('\P{Is_Block= :=Alphabetic	presentation Forms}');
Expect(1, 64335, '\p{Is_Block=alphabeticpresentationforms}', "");
Expect(0, 64335, '\p{^Is_Block=alphabeticpresentationforms}', "");
Expect(0, 64335, '\P{Is_Block=alphabeticpresentationforms}', "");
Expect(1, 64335, '\P{^Is_Block=alphabeticpresentationforms}', "");
Expect(0, 64255, '\p{Is_Block=alphabeticpresentationforms}', "");
Expect(1, 64255, '\p{^Is_Block=alphabeticpresentationforms}', "");
Expect(1, 64255, '\P{Is_Block=alphabeticpresentationforms}', "");
Expect(0, 64255, '\P{^Is_Block=alphabeticpresentationforms}', "");
Expect(1, 64335, '\p{Is_Block=_	Alphabetic-PRESENTATION_Forms}', "");
Expect(0, 64335, '\p{^Is_Block=_	Alphabetic-PRESENTATION_Forms}', "");
Expect(0, 64335, '\P{Is_Block=_	Alphabetic-PRESENTATION_Forms}', "");
Expect(1, 64335, '\P{^Is_Block=_	Alphabetic-PRESENTATION_Forms}', "");
Expect(0, 64255, '\p{Is_Block=_	Alphabetic-PRESENTATION_Forms}', "");
Expect(1, 64255, '\p{^Is_Block=_	Alphabetic-PRESENTATION_Forms}', "");
Expect(1, 64255, '\P{Is_Block=_	Alphabetic-PRESENTATION_Forms}', "");
Expect(0, 64255, '\P{^Is_Block=_	Alphabetic-PRESENTATION_Forms}', "");
Error('\p{Is_Blk:    /a/AlphabeticPRESENTATION-FORMS}');
Error('\P{Is_Blk:    /a/AlphabeticPRESENTATION-FORMS}');
Expect(1, 64335, '\p{Is_Blk=alphabeticpresentationforms}', "");
Expect(0, 64335, '\p{^Is_Blk=alphabeticpresentationforms}', "");
Expect(0, 64335, '\P{Is_Blk=alphabeticpresentationforms}', "");
Expect(1, 64335, '\P{^Is_Blk=alphabeticpresentationforms}', "");
Expect(0, 64255, '\p{Is_Blk=alphabeticpresentationforms}', "");
Expect(1, 64255, '\p{^Is_Blk=alphabeticpresentationforms}', "");
Expect(1, 64255, '\P{Is_Blk=alphabeticpresentationforms}', "");
Expect(0, 64255, '\P{^Is_Blk=alphabeticpresentationforms}', "");
Expect(1, 64335, '\p{Is_Blk:   _alphabetic	Presentation Forms}', "");
Expect(0, 64335, '\p{^Is_Blk:   _alphabetic	Presentation Forms}', "");
Expect(0, 64335, '\P{Is_Blk:   _alphabetic	Presentation Forms}', "");
Expect(1, 64335, '\P{^Is_Blk:   _alphabetic	Presentation Forms}', "");
Expect(0, 64255, '\p{Is_Blk:   _alphabetic	Presentation Forms}', "");
Expect(1, 64255, '\p{^Is_Blk:   _alphabetic	Presentation Forms}', "");
Expect(1, 64255, '\P{Is_Blk:   _alphabetic	Presentation Forms}', "");
Expect(0, 64255, '\P{^Is_Blk:   _alphabetic	Presentation Forms}', "");
Error('\p{Block=-:=optical	CHARACTER recognition}');
Error('\P{Block=-:=optical	CHARACTER recognition}');
Expect(1, 9311, '\p{Block=opticalcharacterrecognition}', "");
Expect(0, 9311, '\p{^Block=opticalcharacterrecognition}', "");
Expect(0, 9311, '\P{Block=opticalcharacterrecognition}', "");
Expect(1, 9311, '\P{^Block=opticalcharacterrecognition}', "");
Expect(0, 9279, '\p{Block=opticalcharacterrecognition}', "");
Expect(1, 9279, '\p{^Block=opticalcharacterrecognition}', "");
Expect(1, 9279, '\P{Block=opticalcharacterrecognition}', "");
Expect(0, 9279, '\P{^Block=opticalcharacterrecognition}', "");
Expect(1, 9311, '\p{Block=-opticalCharacterRecognition}', "");
Expect(0, 9311, '\p{^Block=-opticalCharacterRecognition}', "");
Expect(0, 9311, '\P{Block=-opticalCharacterRecognition}', "");
Expect(1, 9311, '\P{^Block=-opticalCharacterRecognition}', "");
Expect(0, 9279, '\p{Block=-opticalCharacterRecognition}', "");
Expect(1, 9279, '\p{^Block=-opticalCharacterRecognition}', "");
Expect(1, 9279, '\P{Block=-opticalCharacterRecognition}', "");
Expect(0, 9279, '\P{^Block=-opticalCharacterRecognition}', "");
Error('\p{Blk=/a/ _Optical-CHARACTERRECOGNITION}');
Error('\P{Blk=/a/ _Optical-CHARACTERRECOGNITION}');
Expect(1, 9311, '\p{Blk=opticalcharacterrecognition}', "");
Expect(0, 9311, '\p{^Blk=opticalcharacterrecognition}', "");
Expect(0, 9311, '\P{Blk=opticalcharacterrecognition}', "");
Expect(1, 9311, '\P{^Blk=opticalcharacterrecognition}', "");
Expect(0, 9279, '\p{Blk=opticalcharacterrecognition}', "");
Expect(1, 9279, '\p{^Blk=opticalcharacterrecognition}', "");
Expect(1, 9279, '\P{Blk=opticalcharacterrecognition}', "");
Expect(0, 9279, '\P{^Blk=opticalcharacterrecognition}', "");
Expect(1, 9311, '\p{Blk=-OPTICAL character	RECOGNITION}', "");
Expect(0, 9311, '\p{^Blk=-OPTICAL character	RECOGNITION}', "");
Expect(0, 9311, '\P{Blk=-OPTICAL character	RECOGNITION}', "");
Expect(1, 9311, '\P{^Blk=-OPTICAL character	RECOGNITION}', "");
Expect(0, 9279, '\p{Blk=-OPTICAL character	RECOGNITION}', "");
Expect(1, 9279, '\p{^Blk=-OPTICAL character	RECOGNITION}', "");
Expect(1, 9279, '\P{Blk=-OPTICAL character	RECOGNITION}', "");
Expect(0, 9279, '\P{^Blk=-OPTICAL character	RECOGNITION}', "");
Error('\p{Is_Block=:=	Opticalcharacter_Recognition}');
Error('\P{Is_Block=:=	Opticalcharacter_Recognition}');
Expect(1, 9311, '\p{Is_Block=opticalcharacterrecognition}', "");
Expect(0, 9311, '\p{^Is_Block=opticalcharacterrecognition}', "");
Expect(0, 9311, '\P{Is_Block=opticalcharacterrecognition}', "");
Expect(1, 9311, '\P{^Is_Block=opticalcharacterrecognition}', "");
Expect(0, 9279, '\p{Is_Block=opticalcharacterrecognition}', "");
Expect(1, 9279, '\p{^Is_Block=opticalcharacterrecognition}', "");
Expect(1, 9279, '\P{Is_Block=opticalcharacterrecognition}', "");
Expect(0, 9279, '\P{^Is_Block=opticalcharacterrecognition}', "");
Expect(1, 9311, '\p{Is_Block=- OpticalCHARACTER	RECOGNITION}', "");
Expect(0, 9311, '\p{^Is_Block=- OpticalCHARACTER	RECOGNITION}', "");
Expect(0, 9311, '\P{Is_Block=- OpticalCHARACTER	RECOGNITION}', "");
Expect(1, 9311, '\P{^Is_Block=- OpticalCHARACTER	RECOGNITION}', "");
Expect(0, 9279, '\p{Is_Block=- OpticalCHARACTER	RECOGNITION}', "");
Expect(1, 9279, '\p{^Is_Block=- OpticalCHARACTER	RECOGNITION}', "");
Expect(1, 9279, '\P{Is_Block=- OpticalCHARACTER	RECOGNITION}', "");
Expect(0, 9279, '\P{^Is_Block=- OpticalCHARACTER	RECOGNITION}', "");
Error('\p{Is_Blk:    _Optical:=CharacterRecognition}');
Error('\P{Is_Blk:    _Optical:=CharacterRecognition}');
Expect(1, 9311, '\p{Is_Blk=opticalcharacterrecognition}', "");
Expect(0, 9311, '\p{^Is_Blk=opticalcharacterrecognition}', "");
Expect(0, 9311, '\P{Is_Blk=opticalcharacterrecognition}', "");
Expect(1, 9311, '\P{^Is_Blk=opticalcharacterrecognition}', "");
Expect(0, 9279, '\p{Is_Blk=opticalcharacterrecognition}', "");
Expect(1, 9279, '\p{^Is_Blk=opticalcharacterrecognition}', "");
Expect(1, 9279, '\P{Is_Blk=opticalcharacterrecognition}', "");
Expect(0, 9279, '\P{^Is_Blk=opticalcharacterrecognition}', "");
Expect(1, 9311, '\p{Is_Blk:   	optical	character Recognition}', "");
Expect(0, 9311, '\p{^Is_Blk:   	optical	character Recognition}', "");
Expect(0, 9311, '\P{Is_Blk:   	optical	character Recognition}', "");
Expect(1, 9311, '\P{^Is_Blk:   	optical	character Recognition}', "");
Expect(0, 9279, '\p{Is_Blk:   	optical	character Recognition}', "");
Expect(1, 9279, '\p{^Is_Blk:   	optical	character Recognition}', "");
Expect(1, 9279, '\P{Is_Blk:   	optical	character Recognition}', "");
Expect(0, 9279, '\P{^Is_Blk:   	optical	character Recognition}', "");
Error('\p{Block=CJK:=Unified_Ideographs}');
Error('\P{Block=CJK:=Unified_Ideographs}');
Expect(1, 40959, '\p{Block=cjkunifiedideographs}', "");
Expect(0, 40959, '\p{^Block=cjkunifiedideographs}', "");
Expect(0, 40959, '\P{Block=cjkunifiedideographs}', "");
Expect(1, 40959, '\P{^Block=cjkunifiedideographs}', "");
Expect(0, 19967, '\p{Block=cjkunifiedideographs}', "");
Expect(1, 19967, '\p{^Block=cjkunifiedideographs}', "");
Expect(1, 19967, '\P{Block=cjkunifiedideographs}', "");
Expect(0, 19967, '\P{^Block=cjkunifiedideographs}', "");
Expect(1, 40959, '\p{Block=--CJK_unified Ideographs}', "");
Expect(0, 40959, '\p{^Block=--CJK_unified Ideographs}', "");
Expect(0, 40959, '\P{Block=--CJK_unified Ideographs}', "");
Expect(1, 40959, '\P{^Block=--CJK_unified Ideographs}', "");
Expect(0, 19967, '\p{Block=--CJK_unified Ideographs}', "");
Expect(1, 19967, '\p{^Block=--CJK_unified Ideographs}', "");
Expect(1, 19967, '\P{Block=--CJK_unified Ideographs}', "");
Expect(0, 19967, '\P{^Block=--CJK_unified Ideographs}', "");
Error('\p{Blk:	 	CJK_Unified/a/IDEOGRAPHS}');
Error('\P{Blk:	 	CJK_Unified/a/IDEOGRAPHS}');
Expect(1, 40959, '\p{Blk=cjkunifiedideographs}', "");
Expect(0, 40959, '\p{^Blk=cjkunifiedideographs}', "");
Expect(0, 40959, '\P{Blk=cjkunifiedideographs}', "");
Expect(1, 40959, '\P{^Blk=cjkunifiedideographs}', "");
Expect(0, 19967, '\p{Blk=cjkunifiedideographs}', "");
Expect(1, 19967, '\p{^Blk=cjkunifiedideographs}', "");
Expect(1, 19967, '\P{Blk=cjkunifiedideographs}', "");
Expect(0, 19967, '\P{^Blk=cjkunifiedideographs}', "");
Expect(1, 40959, '\p{Blk= 	CJK_UNIFIED-ideographs}', "");
Expect(0, 40959, '\p{^Blk= 	CJK_UNIFIED-ideographs}', "");
Expect(0, 40959, '\P{Blk= 	CJK_UNIFIED-ideographs}', "");
Expect(1, 40959, '\P{^Blk= 	CJK_UNIFIED-ideographs}', "");
Expect(0, 19967, '\p{Blk= 	CJK_UNIFIED-ideographs}', "");
Expect(1, 19967, '\p{^Blk= 	CJK_UNIFIED-ideographs}', "");
Expect(1, 19967, '\P{Blk= 	CJK_UNIFIED-ideographs}', "");
Expect(0, 19967, '\P{^Blk= 	CJK_UNIFIED-ideographs}', "");
Error('\p{Is_Block: _	cjk/a/unified-IDEOGRAPHS}');
Error('\P{Is_Block: _	cjk/a/unified-IDEOGRAPHS}');
Expect(1, 40959, '\p{Is_Block=cjkunifiedideographs}', "");
Expect(0, 40959, '\p{^Is_Block=cjkunifiedideographs}', "");
Expect(0, 40959, '\P{Is_Block=cjkunifiedideographs}', "");
Expect(1, 40959, '\P{^Is_Block=cjkunifiedideographs}', "");
Expect(0, 19967, '\p{Is_Block=cjkunifiedideographs}', "");
Expect(1, 19967, '\p{^Is_Block=cjkunifiedideographs}', "");
Expect(1, 19967, '\P{Is_Block=cjkunifiedideographs}', "");
Expect(0, 19967, '\P{^Is_Block=cjkunifiedideographs}', "");
Expect(1, 40959, '\p{Is_Block=	cjk-Unified ideographs}', "");
Expect(0, 40959, '\p{^Is_Block=	cjk-Unified ideographs}', "");
Expect(0, 40959, '\P{Is_Block=	cjk-Unified ideographs}', "");
Expect(1, 40959, '\P{^Is_Block=	cjk-Unified ideographs}', "");
Expect(0, 19967, '\p{Is_Block=	cjk-Unified ideographs}', "");
Expect(1, 19967, '\p{^Is_Block=	cjk-Unified ideographs}', "");
Expect(1, 19967, '\P{Is_Block=	cjk-Unified ideographs}', "");
Expect(0, 19967, '\P{^Is_Block=	cjk-Unified ideographs}', "");
Error('\p{Is_Blk:   -CJK/a/UnifiedIdeographs}');
Error('\P{Is_Blk:   -CJK/a/UnifiedIdeographs}');
Expect(1, 40959, '\p{Is_Blk=cjkunifiedideographs}', "");
Expect(0, 40959, '\p{^Is_Blk=cjkunifiedideographs}', "");
Expect(0, 40959, '\P{Is_Blk=cjkunifiedideographs}', "");
Expect(1, 40959, '\P{^Is_Blk=cjkunifiedideographs}', "");
Expect(0, 19967, '\p{Is_Blk=cjkunifiedideographs}', "");
Expect(1, 19967, '\p{^Is_Blk=cjkunifiedideographs}', "");
Expect(1, 19967, '\P{Is_Blk=cjkunifiedideographs}', "");
Expect(0, 19967, '\P{^Is_Blk=cjkunifiedideographs}', "");
Expect(1, 40959, '\p{Is_Blk=		cjk UNIFIED ideographs}', "");
Expect(0, 40959, '\p{^Is_Blk=		cjk UNIFIED ideographs}', "");
Expect(0, 40959, '\P{Is_Blk=		cjk UNIFIED ideographs}', "");
Expect(1, 40959, '\P{^Is_Blk=		cjk UNIFIED ideographs}', "");
Expect(0, 19967, '\p{Is_Blk=		cjk UNIFIED ideographs}', "");
Expect(1, 19967, '\p{^Is_Blk=		cjk UNIFIED ideographs}', "");
Expect(1, 19967, '\P{Is_Blk=		cjk UNIFIED ideographs}', "");
Expect(0, 19967, '\P{^Is_Blk=		cjk UNIFIED ideographs}', "");
Error('\p{Block=_/a/Ancient Symbols}');
Error('\P{Block=_/a/Ancient Symbols}');
Expect(1, 65999, '\p{Block=ancientsymbols}', "");
Expect(0, 65999, '\p{^Block=ancientsymbols}', "");
Expect(0, 65999, '\P{Block=ancientsymbols}', "");
Expect(1, 65999, '\P{^Block=ancientsymbols}', "");
Expect(0, 65935, '\p{Block=ancientsymbols}', "");
Expect(1, 65935, '\p{^Block=ancientsymbols}', "");
Expect(1, 65935, '\P{Block=ancientsymbols}', "");
Expect(0, 65935, '\P{^Block=ancientsymbols}', "");
Expect(1, 65999, '\p{Block=  ancient_symbols}', "");
Expect(0, 65999, '\p{^Block=  ancient_symbols}', "");
Expect(0, 65999, '\P{Block=  ancient_symbols}', "");
Expect(1, 65999, '\P{^Block=  ancient_symbols}', "");
Expect(0, 65935, '\p{Block=  ancient_symbols}', "");
Expect(1, 65935, '\p{^Block=  ancient_symbols}', "");
Expect(1, 65935, '\P{Block=  ancient_symbols}', "");
Expect(0, 65935, '\P{^Block=  ancient_symbols}', "");
Error('\p{Blk=/a/-ancient_Symbols}');
Error('\P{Blk=/a/-ancient_Symbols}');
Expect(1, 65999, '\p{Blk: ancientsymbols}', "");
Expect(0, 65999, '\p{^Blk: ancientsymbols}', "");
Expect(0, 65999, '\P{Blk: ancientsymbols}', "");
Expect(1, 65999, '\P{^Blk: ancientsymbols}', "");
Expect(0, 65935, '\p{Blk: ancientsymbols}', "");
Expect(1, 65935, '\p{^Blk: ancientsymbols}', "");
Expect(1, 65935, '\P{Blk: ancientsymbols}', "");
Expect(0, 65935, '\P{^Blk: ancientsymbols}', "");
Expect(1, 65999, '\p{Blk=Ancient_Symbols}', "");
Expect(0, 65999, '\p{^Blk=Ancient_Symbols}', "");
Expect(0, 65999, '\P{Blk=Ancient_Symbols}', "");
Expect(1, 65999, '\P{^Blk=Ancient_Symbols}', "");
Expect(0, 65935, '\p{Blk=Ancient_Symbols}', "");
Expect(1, 65935, '\p{^Blk=Ancient_Symbols}', "");
Expect(1, 65935, '\P{Blk=Ancient_Symbols}', "");
Expect(0, 65935, '\P{^Blk=Ancient_Symbols}', "");
Error('\p{Is_Block=:=_ANCIENT-Symbols}');
Error('\P{Is_Block=:=_ANCIENT-Symbols}');
Expect(1, 65999, '\p{Is_Block: ancientsymbols}', "");
Expect(0, 65999, '\p{^Is_Block: ancientsymbols}', "");
Expect(0, 65999, '\P{Is_Block: ancientsymbols}', "");
Expect(1, 65999, '\P{^Is_Block: ancientsymbols}', "");
Expect(0, 65935, '\p{Is_Block: ancientsymbols}', "");
Expect(1, 65935, '\p{^Is_Block: ancientsymbols}', "");
Expect(1, 65935, '\P{Is_Block: ancientsymbols}', "");
Expect(0, 65935, '\P{^Is_Block: ancientsymbols}', "");
Expect(1, 65999, '\p{Is_Block= _ANCIENTSYMBOLS}', "");
Expect(0, 65999, '\p{^Is_Block= _ANCIENTSYMBOLS}', "");
Expect(0, 65999, '\P{Is_Block= _ANCIENTSYMBOLS}', "");
Expect(1, 65999, '\P{^Is_Block= _ANCIENTSYMBOLS}', "");
Expect(0, 65935, '\p{Is_Block= _ANCIENTSYMBOLS}', "");
Expect(1, 65935, '\p{^Is_Block= _ANCIENTSYMBOLS}', "");
Expect(1, 65935, '\P{Is_Block= _ANCIENTSYMBOLS}', "");
Expect(0, 65935, '\P{^Is_Block= _ANCIENTSYMBOLS}', "");
Error('\p{Is_Blk=/a/ Ancientsymbols}');
Error('\P{Is_Blk=/a/ Ancientsymbols}');
Expect(1, 65999, '\p{Is_Blk=ancientsymbols}', "");
Expect(0, 65999, '\p{^Is_Blk=ancientsymbols}', "");
Expect(0, 65999, '\P{Is_Blk=ancientsymbols}', "");
Expect(1, 65999, '\P{^Is_Blk=ancientsymbols}', "");
Expect(0, 65935, '\p{Is_Blk=ancientsymbols}', "");
Expect(1, 65935, '\p{^Is_Blk=ancientsymbols}', "");
Expect(1, 65935, '\P{Is_Blk=ancientsymbols}', "");
Expect(0, 65935, '\P{^Is_Blk=ancientsymbols}', "");
Expect(1, 65999, '\p{Is_Blk=	_ancient-SYMBOLS}', "");
Expect(0, 65999, '\p{^Is_Blk=	_ancient-SYMBOLS}', "");
Expect(0, 65999, '\P{Is_Blk=	_ancient-SYMBOLS}', "");
Expect(1, 65999, '\P{^Is_Blk=	_ancient-SYMBOLS}', "");
Expect(0, 65935, '\p{Is_Blk=	_ancient-SYMBOLS}', "");
Expect(1, 65935, '\p{^Is_Blk=	_ancient-SYMBOLS}', "");
Expect(1, 65935, '\P{Is_Blk=	_ancient-SYMBOLS}', "");
Expect(0, 65935, '\P{^Is_Blk=	_ancient-SYMBOLS}', "");
Error('\p{Block=_-Gothic/a/}');
Error('\P{Block=_-Gothic/a/}');
Expect(1, 66383, '\p{Block=gothic}', "");
Expect(0, 66383, '\p{^Block=gothic}', "");
Expect(0, 66383, '\P{Block=gothic}', "");
Expect(1, 66383, '\P{^Block=gothic}', "");
Expect(0, 66351, '\p{Block=gothic}', "");
Expect(1, 66351, '\p{^Block=gothic}', "");
Expect(1, 66351, '\P{Block=gothic}', "");
Expect(0, 66351, '\P{^Block=gothic}', "");
Expect(1, 66383, '\p{Block=-gothic}', "");
Expect(0, 66383, '\p{^Block=-gothic}', "");
Expect(0, 66383, '\P{Block=-gothic}', "");
Expect(1, 66383, '\P{^Block=-gothic}', "");
Expect(0, 66351, '\p{Block=-gothic}', "");
Expect(1, 66351, '\p{^Block=-gothic}', "");
Expect(1, 66351, '\P{Block=-gothic}', "");
Expect(0, 66351, '\P{^Block=-gothic}', "");
Error('\p{Blk=:=--GOTHIC}');
Error('\P{Blk=:=--GOTHIC}');
Expect(1, 66383, '\p{Blk=gothic}', "");
Expect(0, 66383, '\p{^Blk=gothic}', "");
Expect(0, 66383, '\P{Blk=gothic}', "");
Expect(1, 66383, '\P{^Blk=gothic}', "");
Expect(0, 66351, '\p{Blk=gothic}', "");
Expect(1, 66351, '\p{^Blk=gothic}', "");
Expect(1, 66351, '\P{Blk=gothic}', "");
Expect(0, 66351, '\P{^Blk=gothic}', "");
Expect(1, 66383, '\p{Blk: _Gothic}', "");
Expect(0, 66383, '\p{^Blk: _Gothic}', "");
Expect(0, 66383, '\P{Blk: _Gothic}', "");
Expect(1, 66383, '\P{^Blk: _Gothic}', "");
Expect(0, 66351, '\p{Blk: _Gothic}', "");
Expect(1, 66351, '\p{^Blk: _Gothic}', "");
Expect(1, 66351, '\P{Blk: _Gothic}', "");
Expect(0, 66351, '\P{^Blk: _Gothic}', "");
Error('\p{Is_Block=-/a/gothic}');
Error('\P{Is_Block=-/a/gothic}');
Expect(1, 66383, '\p{Is_Block=gothic}', "");
Expect(0, 66383, '\p{^Is_Block=gothic}', "");
Expect(0, 66383, '\P{Is_Block=gothic}', "");
Expect(1, 66383, '\P{^Is_Block=gothic}', "");
Expect(0, 66351, '\p{Is_Block=gothic}', "");
Expect(1, 66351, '\p{^Is_Block=gothic}', "");
Expect(1, 66351, '\P{Is_Block=gothic}', "");
Expect(0, 66351, '\P{^Is_Block=gothic}', "");
Expect(1, 66383, '\p{Is_Block=-_Gothic}', "");
Expect(0, 66383, '\p{^Is_Block=-_Gothic}', "");
Expect(0, 66383, '\P{Is_Block=-_Gothic}', "");
Expect(1, 66383, '\P{^Is_Block=-_Gothic}', "");
Expect(0, 66351, '\p{Is_Block=-_Gothic}', "");
Expect(1, 66351, '\p{^Is_Block=-_Gothic}', "");
Expect(1, 66351, '\P{Is_Block=-_Gothic}', "");
Expect(0, 66351, '\P{^Is_Block=-_Gothic}', "");
Error('\p{Is_Blk=	/a/GOTHIC}');
Error('\P{Is_Blk=	/a/GOTHIC}');
Expect(1, 66383, '\p{Is_Blk=gothic}', "");
Expect(0, 66383, '\p{^Is_Blk=gothic}', "");
Expect(0, 66383, '\P{Is_Blk=gothic}', "");
Expect(1, 66383, '\P{^Is_Blk=gothic}', "");
Expect(0, 66351, '\p{Is_Blk=gothic}', "");
Expect(1, 66351, '\p{^Is_Blk=gothic}', "");
Expect(1, 66351, '\P{Is_Blk=gothic}', "");
Expect(0, 66351, '\P{^Is_Blk=gothic}', "");
Expect(1, 66383, '\p{Is_Blk=-Gothic}', "");
Expect(0, 66383, '\p{^Is_Blk=-Gothic}', "");
Expect(0, 66383, '\P{Is_Blk=-Gothic}', "");
Expect(1, 66383, '\P{^Is_Blk=-Gothic}', "");
Expect(0, 66351, '\p{Is_Blk=-Gothic}', "");
Expect(1, 66351, '\p{^Is_Blk=-Gothic}', "");
Expect(1, 66351, '\P{Is_Blk=-Gothic}', "");
Expect(0, 66351, '\P{^Is_Blk=-Gothic}', "");
Error('\p{Block:	/a/ETHIOPIC}');
Error('\P{Block:	/a/ETHIOPIC}');
Expect(1, 4991, '\p{Block=ethiopic}', "");
Expect(0, 4991, '\p{^Block=ethiopic}', "");
Expect(0, 4991, '\P{Block=ethiopic}', "");
Expect(1, 4991, '\P{^Block=ethiopic}', "");
Expect(0, 4607, '\p{Block=ethiopic}', "");
Expect(1, 4607, '\p{^Block=ethiopic}', "");
Expect(1, 4607, '\P{Block=ethiopic}', "");
Expect(0, 4607, '\P{^Block=ethiopic}', "");
Expect(1, 4991, '\p{Block= Ethiopic}', "");
Expect(0, 4991, '\p{^Block= Ethiopic}', "");
Expect(0, 4991, '\P{Block= Ethiopic}', "");
Expect(1, 4991, '\P{^Block= Ethiopic}', "");
Expect(0, 4607, '\p{Block= Ethiopic}', "");
Expect(1, 4607, '\p{^Block= Ethiopic}', "");
Expect(1, 4607, '\P{Block= Ethiopic}', "");
Expect(0, 4607, '\P{^Block= Ethiopic}', "");
Error('\p{Blk=/a/ -Ethiopic}');
Error('\P{Blk=/a/ -Ethiopic}');
Expect(1, 4991, '\p{Blk=ethiopic}', "");
Expect(0, 4991, '\p{^Blk=ethiopic}', "");
Expect(0, 4991, '\P{Blk=ethiopic}', "");
Expect(1, 4991, '\P{^Blk=ethiopic}', "");
Expect(0, 4607, '\p{Blk=ethiopic}', "");
Expect(1, 4607, '\p{^Blk=ethiopic}', "");
Expect(1, 4607, '\P{Blk=ethiopic}', "");
Expect(0, 4607, '\P{^Blk=ethiopic}', "");
Expect(1, 4991, '\p{Blk=_ethiopic}', "");
Expect(0, 4991, '\p{^Blk=_ethiopic}', "");
Expect(0, 4991, '\P{Blk=_ethiopic}', "");
Expect(1, 4991, '\P{^Blk=_ethiopic}', "");
Expect(0, 4607, '\p{Blk=_ethiopic}', "");
Expect(1, 4607, '\p{^Blk=_ethiopic}', "");
Expect(1, 4607, '\P{Blk=_ethiopic}', "");
Expect(0, 4607, '\P{^Blk=_ethiopic}', "");
Error('\p{Is_Block=	:=ethiopic}');
Error('\P{Is_Block=	:=ethiopic}');
Expect(1, 4991, '\p{Is_Block=ethiopic}', "");
Expect(0, 4991, '\p{^Is_Block=ethiopic}', "");
Expect(0, 4991, '\P{Is_Block=ethiopic}', "");
Expect(1, 4991, '\P{^Is_Block=ethiopic}', "");
Expect(0, 4607, '\p{Is_Block=ethiopic}', "");
Expect(1, 4607, '\p{^Is_Block=ethiopic}', "");
Expect(1, 4607, '\P{Is_Block=ethiopic}', "");
Expect(0, 4607, '\P{^Is_Block=ethiopic}', "");
Expect(1, 4991, '\p{Is_Block: 	 ETHIOPIC}', "");
Expect(0, 4991, '\p{^Is_Block: 	 ETHIOPIC}', "");
Expect(0, 4991, '\P{Is_Block: 	 ETHIOPIC}', "");
Expect(1, 4991, '\P{^Is_Block: 	 ETHIOPIC}', "");
Expect(0, 4607, '\p{Is_Block: 	 ETHIOPIC}', "");
Expect(1, 4607, '\p{^Is_Block: 	 ETHIOPIC}', "");
Expect(1, 4607, '\P{Is_Block: 	 ETHIOPIC}', "");
Expect(0, 4607, '\P{^Is_Block: 	 ETHIOPIC}', "");
Error('\p{Is_Blk= 	ethiopic:=}');
Error('\P{Is_Blk= 	ethiopic:=}');
Expect(1, 4991, '\p{Is_Blk=ethiopic}', "");
Expect(0, 4991, '\p{^Is_Blk=ethiopic}', "");
Expect(0, 4991, '\P{Is_Blk=ethiopic}', "");
Expect(1, 4991, '\P{^Is_Blk=ethiopic}', "");
Expect(0, 4607, '\p{Is_Blk=ethiopic}', "");
Expect(1, 4607, '\p{^Is_Blk=ethiopic}', "");
Expect(1, 4607, '\P{Is_Blk=ethiopic}', "");
Expect(0, 4607, '\P{^Is_Blk=ethiopic}', "");
Expect(1, 4991, '\p{Is_Blk=	-ethiopic}', "");
Expect(0, 4991, '\p{^Is_Blk=	-ethiopic}', "");
Expect(0, 4991, '\P{Is_Blk=	-ethiopic}', "");
Expect(1, 4991, '\P{^Is_Blk=	-ethiopic}', "");
Expect(0, 4607, '\p{Is_Blk=	-ethiopic}', "");
Expect(1, 4607, '\p{^Is_Blk=	-ethiopic}', "");
Expect(1, 4607, '\P{Is_Blk=	-ethiopic}', "");
Expect(0, 4607, '\P{^Is_Blk=	-ethiopic}', "");
Error('\p{Block=_:=Buginese}');
Error('\P{Block=_:=Buginese}');
Expect(1, 6687, '\p{Block=buginese}', "");
Expect(0, 6687, '\p{^Block=buginese}', "");
Expect(0, 6687, '\P{Block=buginese}', "");
Expect(1, 6687, '\P{^Block=buginese}', "");
Expect(0, 6655, '\p{Block=buginese}', "");
Expect(1, 6655, '\p{^Block=buginese}', "");
Expect(1, 6655, '\P{Block=buginese}', "");
Expect(0, 6655, '\P{^Block=buginese}', "");
Expect(1, 6687, '\p{Block= _Buginese}', "");
Expect(0, 6687, '\p{^Block= _Buginese}', "");
Expect(0, 6687, '\P{Block= _Buginese}', "");
Expect(1, 6687, '\P{^Block= _Buginese}', "");
Expect(0, 6655, '\p{Block= _Buginese}', "");
Expect(1, 6655, '\p{^Block= _Buginese}', "");
Expect(1, 6655, '\P{Block= _Buginese}', "");
Expect(0, 6655, '\P{^Block= _Buginese}', "");
Error('\p{Blk:		_Buginese/a/}');
Error('\P{Blk:		_Buginese/a/}');
Expect(1, 6687, '\p{Blk=buginese}', "");
Expect(0, 6687, '\p{^Blk=buginese}', "");
Expect(0, 6687, '\P{Blk=buginese}', "");
Expect(1, 6687, '\P{^Blk=buginese}', "");
Expect(0, 6655, '\p{Blk=buginese}', "");
Expect(1, 6655, '\p{^Blk=buginese}', "");
Expect(1, 6655, '\P{Blk=buginese}', "");
Expect(0, 6655, '\P{^Blk=buginese}', "");
Expect(1, 6687, '\p{Blk=	_Buginese}', "");
Expect(0, 6687, '\p{^Blk=	_Buginese}', "");
Expect(0, 6687, '\P{Blk=	_Buginese}', "");
Expect(1, 6687, '\P{^Blk=	_Buginese}', "");
Expect(0, 6655, '\p{Blk=	_Buginese}', "");
Expect(1, 6655, '\p{^Blk=	_Buginese}', "");
Expect(1, 6655, '\P{Blk=	_Buginese}', "");
Expect(0, 6655, '\P{^Blk=	_Buginese}', "");
Error('\p{Is_Block=:=	_BUGINESE}');
Error('\P{Is_Block=:=	_BUGINESE}');
Expect(1, 6687, '\p{Is_Block=buginese}', "");
Expect(0, 6687, '\p{^Is_Block=buginese}', "");
Expect(0, 6687, '\P{Is_Block=buginese}', "");
Expect(1, 6687, '\P{^Is_Block=buginese}', "");
Expect(0, 6655, '\p{Is_Block=buginese}', "");
Expect(1, 6655, '\p{^Is_Block=buginese}', "");
Expect(1, 6655, '\P{Is_Block=buginese}', "");
Expect(0, 6655, '\P{^Is_Block=buginese}', "");
Expect(1, 6687, '\p{Is_Block=	_BUGINESE}', "");
Expect(0, 6687, '\p{^Is_Block=	_BUGINESE}', "");
Expect(0, 6687, '\P{Is_Block=	_BUGINESE}', "");
Expect(1, 6687, '\P{^Is_Block=	_BUGINESE}', "");
Expect(0, 6655, '\p{Is_Block=	_BUGINESE}', "");
Expect(1, 6655, '\p{^Is_Block=	_BUGINESE}', "");
Expect(1, 6655, '\P{Is_Block=	_BUGINESE}', "");
Expect(0, 6655, '\P{^Is_Block=	_BUGINESE}', "");
Error('\p{Is_Blk=__buginese:=}');
Error('\P{Is_Blk=__buginese:=}');
Expect(1, 6687, '\p{Is_Blk=buginese}', "");
Expect(0, 6687, '\p{^Is_Blk=buginese}', "");
Expect(0, 6687, '\P{Is_Blk=buginese}', "");
Expect(1, 6687, '\P{^Is_Blk=buginese}', "");
Expect(0, 6655, '\p{Is_Blk=buginese}', "");
Expect(1, 6655, '\p{^Is_Blk=buginese}', "");
Expect(1, 6655, '\P{Is_Blk=buginese}', "");
Expect(0, 6655, '\P{^Is_Blk=buginese}', "");
Expect(1, 6687, '\p{Is_Blk=  Buginese}', "");
Expect(0, 6687, '\p{^Is_Blk=  Buginese}', "");
Expect(0, 6687, '\P{Is_Blk=  Buginese}', "");
Expect(1, 6687, '\P{^Is_Blk=  Buginese}', "");
Expect(0, 6655, '\p{Is_Blk=  Buginese}', "");
Expect(1, 6655, '\p{^Is_Blk=  Buginese}', "");
Expect(1, 6655, '\P{Is_Blk=  Buginese}', "");
Expect(0, 6655, '\P{^Is_Blk=  Buginese}', "");
Error('\p{Block=:= Mathematical	Alphanumeric-symbols}');
Error('\P{Block=:= Mathematical	Alphanumeric-symbols}');
Expect(1, 120831, '\p{Block: mathematicalalphanumericsymbols}', "");
Expect(0, 120831, '\p{^Block: mathematicalalphanumericsymbols}', "");
Expect(0, 120831, '\P{Block: mathematicalalphanumericsymbols}', "");
Expect(1, 120831, '\P{^Block: mathematicalalphanumericsymbols}', "");
Expect(0, 119807, '\p{Block: mathematicalalphanumericsymbols}', "");
Expect(1, 119807, '\p{^Block: mathematicalalphanumericsymbols}', "");
Expect(1, 119807, '\P{Block: mathematicalalphanumericsymbols}', "");
Expect(0, 119807, '\P{^Block: mathematicalalphanumericsymbols}', "");
Expect(1, 120831, '\p{Block=- Mathematical	Alphanumeric-Symbols}', "");
Expect(0, 120831, '\p{^Block=- Mathematical	Alphanumeric-Symbols}', "");
Expect(0, 120831, '\P{Block=- Mathematical	Alphanumeric-Symbols}', "");
Expect(1, 120831, '\P{^Block=- Mathematical	Alphanumeric-Symbols}', "");
Expect(0, 119807, '\p{Block=- Mathematical	Alphanumeric-Symbols}', "");
Expect(1, 119807, '\p{^Block=- Mathematical	Alphanumeric-Symbols}', "");
Expect(1, 119807, '\P{Block=- Mathematical	Alphanumeric-Symbols}', "");
Expect(0, 119807, '\P{^Block=- Mathematical	Alphanumeric-Symbols}', "");
Error('\p{Blk=	-mathematicalALPHANUMERICSymbols:=}');
Error('\P{Blk=	-mathematicalALPHANUMERICSymbols:=}');
Expect(1, 120831, '\p{Blk=mathematicalalphanumericsymbols}', "");
Expect(0, 120831, '\p{^Blk=mathematicalalphanumericsymbols}', "");
Expect(0, 120831, '\P{Blk=mathematicalalphanumericsymbols}', "");
Expect(1, 120831, '\P{^Blk=mathematicalalphanumericsymbols}', "");
Expect(0, 119807, '\p{Blk=mathematicalalphanumericsymbols}', "");
Expect(1, 119807, '\p{^Blk=mathematicalalphanumericsymbols}', "");
Expect(1, 119807, '\P{Blk=mathematicalalphanumericsymbols}', "");
Expect(0, 119807, '\P{^Blk=mathematicalalphanumericsymbols}', "");
Expect(1, 120831, '\p{Blk= -Mathematical	AlphanumericSymbols}', "");
Expect(0, 120831, '\p{^Blk= -Mathematical	AlphanumericSymbols}', "");
Expect(0, 120831, '\P{Blk= -Mathematical	AlphanumericSymbols}', "");
Expect(1, 120831, '\P{^Blk= -Mathematical	AlphanumericSymbols}', "");
Expect(0, 119807, '\p{Blk= -Mathematical	AlphanumericSymbols}', "");
Expect(1, 119807, '\p{^Blk= -Mathematical	AlphanumericSymbols}', "");
Expect(1, 119807, '\P{Blk= -Mathematical	AlphanumericSymbols}', "");
Expect(0, 119807, '\P{^Blk= -Mathematical	AlphanumericSymbols}', "");
Error('\p{Is_Block=_	mathematical alphanumeric:=Symbols}');
Error('\P{Is_Block=_	mathematical alphanumeric:=Symbols}');
Expect(1, 120831, '\p{Is_Block=mathematicalalphanumericsymbols}', "");
Expect(0, 120831, '\p{^Is_Block=mathematicalalphanumericsymbols}', "");
Expect(0, 120831, '\P{Is_Block=mathematicalalphanumericsymbols}', "");
Expect(1, 120831, '\P{^Is_Block=mathematicalalphanumericsymbols}', "");
Expect(0, 119807, '\p{Is_Block=mathematicalalphanumericsymbols}', "");
Expect(1, 119807, '\p{^Is_Block=mathematicalalphanumericsymbols}', "");
Expect(1, 119807, '\P{Is_Block=mathematicalalphanumericsymbols}', "");
Expect(0, 119807, '\P{^Is_Block=mathematicalalphanumericsymbols}', "");
Expect(1, 120831, '\p{Is_Block=-Mathematical	alphanumeric-symbols}', "");
Expect(0, 120831, '\p{^Is_Block=-Mathematical	alphanumeric-symbols}', "");
Expect(0, 120831, '\P{Is_Block=-Mathematical	alphanumeric-symbols}', "");
Expect(1, 120831, '\P{^Is_Block=-Mathematical	alphanumeric-symbols}', "");
Expect(0, 119807, '\p{Is_Block=-Mathematical	alphanumeric-symbols}', "");
Expect(1, 119807, '\p{^Is_Block=-Mathematical	alphanumeric-symbols}', "");
Expect(1, 119807, '\P{Is_Block=-Mathematical	alphanumeric-symbols}', "");
Expect(0, 119807, '\P{^Is_Block=-Mathematical	alphanumeric-symbols}', "");
Error('\p{Is_Blk= :=Mathematical alphanumericSYMBOLS}');
Error('\P{Is_Blk= :=Mathematical alphanumericSYMBOLS}');
Expect(1, 120831, '\p{Is_Blk: mathematicalalphanumericsymbols}', "");
Expect(0, 120831, '\p{^Is_Blk: mathematicalalphanumericsymbols}', "");
Expect(0, 120831, '\P{Is_Blk: mathematicalalphanumericsymbols}', "");
Expect(1, 120831, '\P{^Is_Blk: mathematicalalphanumericsymbols}', "");
Expect(0, 119807, '\p{Is_Blk: mathematicalalphanumericsymbols}', "");
Expect(1, 119807, '\p{^Is_Blk: mathematicalalphanumericsymbols}', "");
Expect(1, 119807, '\P{Is_Blk: mathematicalalphanumericsymbols}', "");
Expect(0, 119807, '\P{^Is_Blk: mathematicalalphanumericsymbols}', "");
Expect(1, 120831, '\p{Is_Blk:		mathematical	AlphanumericSymbols}', "");
Expect(0, 120831, '\p{^Is_Blk:		mathematical	AlphanumericSymbols}', "");
Expect(0, 120831, '\P{Is_Blk:		mathematical	AlphanumericSymbols}', "");
Expect(1, 120831, '\P{^Is_Blk:		mathematical	AlphanumericSymbols}', "");
Expect(0, 119807, '\p{Is_Blk:		mathematical	AlphanumericSymbols}', "");
Expect(1, 119807, '\p{^Is_Blk:		mathematical	AlphanumericSymbols}', "");
Expect(1, 119807, '\P{Is_Blk:		mathematical	AlphanumericSymbols}', "");
Expect(0, 119807, '\P{^Is_Blk:		mathematical	AlphanumericSymbols}', "");
Error('\p{Block=-	Inscriptional-Pahlavi:=}');
Error('\P{Block=-	Inscriptional-Pahlavi:=}');
Expect(1, 68479, '\p{Block=inscriptionalpahlavi}', "");
Expect(0, 68479, '\p{^Block=inscriptionalpahlavi}', "");
Expect(0, 68479, '\P{Block=inscriptionalpahlavi}', "");
Expect(1, 68479, '\P{^Block=inscriptionalpahlavi}', "");
Expect(0, 68447, '\p{Block=inscriptionalpahlavi}', "");
Expect(1, 68447, '\p{^Block=inscriptionalpahlavi}', "");
Expect(1, 68447, '\P{Block=inscriptionalpahlavi}', "");
Expect(0, 68447, '\P{^Block=inscriptionalpahlavi}', "");
Expect(1, 68479, '\p{Block=__INSCRIPTIONAL_Pahlavi}', "");
Expect(0, 68479, '\p{^Block=__INSCRIPTIONAL_Pahlavi}', "");
Expect(0, 68479, '\P{Block=__INSCRIPTIONAL_Pahlavi}', "");
Expect(1, 68479, '\P{^Block=__INSCRIPTIONAL_Pahlavi}', "");
Expect(0, 68447, '\p{Block=__INSCRIPTIONAL_Pahlavi}', "");
Expect(1, 68447, '\p{^Block=__INSCRIPTIONAL_Pahlavi}', "");
Expect(1, 68447, '\P{Block=__INSCRIPTIONAL_Pahlavi}', "");
Expect(0, 68447, '\P{^Block=__INSCRIPTIONAL_Pahlavi}', "");
Error('\p{Blk=INSCRIPTIONAL/a/PAHLAVI}');
Error('\P{Blk=INSCRIPTIONAL/a/PAHLAVI}');
Expect(1, 68479, '\p{Blk=inscriptionalpahlavi}', "");
Expect(0, 68479, '\p{^Blk=inscriptionalpahlavi}', "");
Expect(0, 68479, '\P{Blk=inscriptionalpahlavi}', "");
Expect(1, 68479, '\P{^Blk=inscriptionalpahlavi}', "");
Expect(0, 68447, '\p{Blk=inscriptionalpahlavi}', "");
Expect(1, 68447, '\p{^Blk=inscriptionalpahlavi}', "");
Expect(1, 68447, '\P{Blk=inscriptionalpahlavi}', "");
Expect(0, 68447, '\P{^Blk=inscriptionalpahlavi}', "");
Expect(1, 68479, '\p{Blk= _Inscriptional PAHLAVI}', "");
Expect(0, 68479, '\p{^Blk= _Inscriptional PAHLAVI}', "");
Expect(0, 68479, '\P{Blk= _Inscriptional PAHLAVI}', "");
Expect(1, 68479, '\P{^Blk= _Inscriptional PAHLAVI}', "");
Expect(0, 68447, '\p{Blk= _Inscriptional PAHLAVI}', "");
Expect(1, 68447, '\p{^Blk= _Inscriptional PAHLAVI}', "");
Expect(1, 68447, '\P{Blk= _Inscriptional PAHLAVI}', "");
Expect(0, 68447, '\P{^Blk= _Inscriptional PAHLAVI}', "");
Error('\p{Is_Block=:= INSCRIPTIONAL_Pahlavi}');
Error('\P{Is_Block=:= INSCRIPTIONAL_Pahlavi}');
Expect(1, 68479, '\p{Is_Block=inscriptionalpahlavi}', "");
Expect(0, 68479, '\p{^Is_Block=inscriptionalpahlavi}', "");
Expect(0, 68479, '\P{Is_Block=inscriptionalpahlavi}', "");
Expect(1, 68479, '\P{^Is_Block=inscriptionalpahlavi}', "");
Expect(0, 68447, '\p{Is_Block=inscriptionalpahlavi}', "");
Expect(1, 68447, '\p{^Is_Block=inscriptionalpahlavi}', "");
Expect(1, 68447, '\P{Is_Block=inscriptionalpahlavi}', "");
Expect(0, 68447, '\P{^Is_Block=inscriptionalpahlavi}', "");
Expect(1, 68479, '\p{Is_Block=-inscriptional	PAHLAVI}', "");
Expect(0, 68479, '\p{^Is_Block=-inscriptional	PAHLAVI}', "");
Expect(0, 68479, '\P{Is_Block=-inscriptional	PAHLAVI}', "");
Expect(1, 68479, '\P{^Is_Block=-inscriptional	PAHLAVI}', "");
Expect(0, 68447, '\p{Is_Block=-inscriptional	PAHLAVI}', "");
Expect(1, 68447, '\p{^Is_Block=-inscriptional	PAHLAVI}', "");
Expect(1, 68447, '\P{Is_Block=-inscriptional	PAHLAVI}', "");
Expect(0, 68447, '\P{^Is_Block=-inscriptional	PAHLAVI}', "");
Error('\p{Is_Blk:-Inscriptional-Pahlavi:=}');
Error('\P{Is_Blk:-Inscriptional-Pahlavi:=}');
Expect(1, 68479, '\p{Is_Blk=inscriptionalpahlavi}', "");
Expect(0, 68479, '\p{^Is_Blk=inscriptionalpahlavi}', "");
Expect(0, 68479, '\P{Is_Blk=inscriptionalpahlavi}', "");
Expect(1, 68479, '\P{^Is_Blk=inscriptionalpahlavi}', "");
Expect(0, 68447, '\p{Is_Blk=inscriptionalpahlavi}', "");
Expect(1, 68447, '\p{^Is_Blk=inscriptionalpahlavi}', "");
Expect(1, 68447, '\P{Is_Blk=inscriptionalpahlavi}', "");
Expect(0, 68447, '\P{^Is_Blk=inscriptionalpahlavi}', "");
Expect(1, 68479, '\p{Is_Blk=-inscriptional PAHLAVI}', "");
Expect(0, 68479, '\p{^Is_Blk=-inscriptional PAHLAVI}', "");
Expect(0, 68479, '\P{Is_Blk=-inscriptional PAHLAVI}', "");
Expect(1, 68479, '\P{^Is_Blk=-inscriptional PAHLAVI}', "");
Expect(0, 68447, '\p{Is_Blk=-inscriptional PAHLAVI}', "");
Expect(1, 68447, '\p{^Is_Blk=-inscriptional PAHLAVI}', "");
Expect(1, 68447, '\P{Is_Blk=-inscriptional PAHLAVI}', "");
Expect(0, 68447, '\P{^Is_Blk=-inscriptional PAHLAVI}', "");
Error('\p{Block:  Unified/a/Canadian	aboriginalSyllabics EXTENDED}');
Error('\P{Block:  Unified/a/Canadian	aboriginalSyllabics EXTENDED}');
Expect(1, 6399, '\p{Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6399, '\p{^Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6399, '\P{Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6399, '\P{^Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6319, '\p{Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6319, '\p{^Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6319, '\P{Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6319, '\P{^Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6399, '\p{Block= -Unified-canadian-aboriginalsyllabics-Extended}', "");
Expect(0, 6399, '\p{^Block= -Unified-canadian-aboriginalsyllabics-Extended}', "");
Expect(0, 6399, '\P{Block= -Unified-canadian-aboriginalsyllabics-Extended}', "");
Expect(1, 6399, '\P{^Block= -Unified-canadian-aboriginalsyllabics-Extended}', "");
Expect(0, 6319, '\p{Block= -Unified-canadian-aboriginalsyllabics-Extended}', "");
Expect(1, 6319, '\p{^Block= -Unified-canadian-aboriginalsyllabics-Extended}', "");
Expect(1, 6319, '\P{Block= -Unified-canadian-aboriginalsyllabics-Extended}', "");
Expect(0, 6319, '\P{^Block= -Unified-canadian-aboriginalsyllabics-Extended}', "");
Error('\p{Blk=/a/UNIFIEDCanadian-Aboriginal_Syllabics-extended}');
Error('\P{Blk=/a/UNIFIEDCanadian-Aboriginal_Syllabics-extended}');
Expect(1, 6399, '\p{Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6399, '\p{^Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6399, '\P{Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6399, '\P{^Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6319, '\p{Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6319, '\p{^Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6319, '\P{Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6319, '\P{^Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6399, '\p{Blk=_-Unified-Canadian	Aboriginal Syllabics-EXTENDED}', "");
Expect(0, 6399, '\p{^Blk=_-Unified-Canadian	Aboriginal Syllabics-EXTENDED}', "");
Expect(0, 6399, '\P{Blk=_-Unified-Canadian	Aboriginal Syllabics-EXTENDED}', "");
Expect(1, 6399, '\P{^Blk=_-Unified-Canadian	Aboriginal Syllabics-EXTENDED}', "");
Expect(0, 6319, '\p{Blk=_-Unified-Canadian	Aboriginal Syllabics-EXTENDED}', "");
Expect(1, 6319, '\p{^Blk=_-Unified-Canadian	Aboriginal Syllabics-EXTENDED}', "");
Expect(1, 6319, '\P{Blk=_-Unified-Canadian	Aboriginal Syllabics-EXTENDED}', "");
Expect(0, 6319, '\P{^Blk=_-Unified-Canadian	Aboriginal Syllabics-EXTENDED}', "");
Error('\p{Is_Block=	 unified-CANADIAN	Aboriginal_syllabics:=Extended}');
Error('\P{Is_Block=	 unified-CANADIAN	Aboriginal_syllabics:=Extended}');
Expect(1, 6399, '\p{Is_Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6399, '\p{^Is_Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6399, '\P{Is_Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6399, '\P{^Is_Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6319, '\p{Is_Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6319, '\p{^Is_Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6319, '\P{Is_Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6319, '\P{^Is_Block=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6399, '\p{Is_Block=	 unified	CANADIAN	AboriginalSyllabics	Extended}', "");
Expect(0, 6399, '\p{^Is_Block=	 unified	CANADIAN	AboriginalSyllabics	Extended}', "");
Expect(0, 6399, '\P{Is_Block=	 unified	CANADIAN	AboriginalSyllabics	Extended}', "");
Expect(1, 6399, '\P{^Is_Block=	 unified	CANADIAN	AboriginalSyllabics	Extended}', "");
Expect(0, 6319, '\p{Is_Block=	 unified	CANADIAN	AboriginalSyllabics	Extended}', "");
Expect(1, 6319, '\p{^Is_Block=	 unified	CANADIAN	AboriginalSyllabics	Extended}', "");
Expect(1, 6319, '\P{Is_Block=	 unified	CANADIAN	AboriginalSyllabics	Extended}', "");
Expect(0, 6319, '\P{^Is_Block=	 unified	CANADIAN	AboriginalSyllabics	Extended}', "");
Error('\p{Is_Blk= Unified	Canadian/a/ABORIGINAL SyllabicsExtended}');
Error('\P{Is_Blk= Unified	Canadian/a/ABORIGINAL SyllabicsExtended}');
Expect(1, 6399, '\p{Is_Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6399, '\p{^Is_Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6399, '\P{Is_Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6399, '\P{^Is_Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6319, '\p{Is_Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6319, '\p{^Is_Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6319, '\P{Is_Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(0, 6319, '\P{^Is_Blk=unifiedcanadianaboriginalsyllabicsextended}', "");
Expect(1, 6399, '\p{Is_Blk= _Unified-canadian-Aboriginal_syllabics EXTENDED}', "");
Expect(0, 6399, '\p{^Is_Blk= _Unified-canadian-Aboriginal_syllabics EXTENDED}', "");
Expect(0, 6399, '\P{Is_Blk= _Unified-canadian-Aboriginal_syllabics EXTENDED}', "");
Expect(1, 6399, '\P{^Is_Blk= _Unified-canadian-Aboriginal_syllabics EXTENDED}', "");
Expect(0, 6319, '\p{Is_Blk= _Unified-canadian-Aboriginal_syllabics EXTENDED}', "");
Expect(1, 6319, '\p{^Is_Blk= _Unified-canadian-Aboriginal_syllabics EXTENDED}', "");
Expect(1, 6319, '\P{Is_Blk= _Unified-canadian-Aboriginal_syllabics EXTENDED}', "");
Expect(0, 6319, '\P{^Is_Blk= _Unified-canadian-Aboriginal_syllabics EXTENDED}', "");
Error('\p{Block=		MANDAIC/a/}');
Error('\P{Block=		MANDAIC/a/}');
Expect(1, 2143, '\p{Block=mandaic}', "");
Expect(0, 2143, '\p{^Block=mandaic}', "");
Expect(0, 2143, '\P{Block=mandaic}', "");
Expect(1, 2143, '\P{^Block=mandaic}', "");
Expect(0, 2111, '\p{Block=mandaic}', "");
Expect(1, 2111, '\p{^Block=mandaic}', "");
Expect(1, 2111, '\P{Block=mandaic}', "");
Expect(0, 2111, '\P{^Block=mandaic}', "");
Expect(1, 2143, '\p{Block=-_Mandaic}', "");
Expect(0, 2143, '\p{^Block=-_Mandaic}', "");
Expect(0, 2143, '\P{Block=-_Mandaic}', "");
Expect(1, 2143, '\P{^Block=-_Mandaic}', "");
Expect(0, 2111, '\p{Block=-_Mandaic}', "");
Expect(1, 2111, '\p{^Block=-_Mandaic}', "");
Expect(1, 2111, '\P{Block=-_Mandaic}', "");
Expect(0, 2111, '\P{^Block=-_Mandaic}', "");
Error('\p{Blk=/a/	-mandaic}');
Error('\P{Blk=/a/	-mandaic}');
Expect(1, 2143, '\p{Blk=mandaic}', "");
Expect(0, 2143, '\p{^Blk=mandaic}', "");
Expect(0, 2143, '\P{Blk=mandaic}', "");
Expect(1, 2143, '\P{^Blk=mandaic}', "");
Expect(0, 2111, '\p{Blk=mandaic}', "");
Expect(1, 2111, '\p{^Blk=mandaic}', "");
Expect(1, 2111, '\P{Blk=mandaic}', "");
Expect(0, 2111, '\P{^Blk=mandaic}', "");
Expect(1, 2143, '\p{Blk=	 MANDAIC}', "");
Expect(0, 2143, '\p{^Blk=	 MANDAIC}', "");
Expect(0, 2143, '\P{Blk=	 MANDAIC}', "");
Expect(1, 2143, '\P{^Blk=	 MANDAIC}', "");
Expect(0, 2111, '\p{Blk=	 MANDAIC}', "");
Expect(1, 2111, '\p{^Blk=	 MANDAIC}', "");
Expect(1, 2111, '\P{Blk=	 MANDAIC}', "");
Expect(0, 2111, '\P{^Blk=	 MANDAIC}', "");
Error('\p{Is_Block=:=-_Mandaic}');
Error('\P{Is_Block=:=-_Mandaic}');
Expect(1, 2143, '\p{Is_Block=mandaic}', "");
Expect(0, 2143, '\p{^Is_Block=mandaic}', "");
Expect(0, 2143, '\P{Is_Block=mandaic}', "");
Expect(1, 2143, '\P{^Is_Block=mandaic}', "");
Expect(0, 2111, '\p{Is_Block=mandaic}', "");
Expect(1, 2111, '\p{^Is_Block=mandaic}', "");
Expect(1, 2111, '\P{Is_Block=mandaic}', "");
Expect(0, 2111, '\P{^Is_Block=mandaic}', "");
Expect(1, 2143, '\p{Is_Block=		Mandaic}', "");
Expect(0, 2143, '\p{^Is_Block=		Mandaic}', "");
Expect(0, 2143, '\P{Is_Block=		Mandaic}', "");
Expect(1, 2143, '\P{^Is_Block=		Mandaic}', "");
Expect(0, 2111, '\p{Is_Block=		Mandaic}', "");
Expect(1, 2111, '\p{^Is_Block=		Mandaic}', "");
Expect(1, 2111, '\P{Is_Block=		Mandaic}', "");
Expect(0, 2111, '\P{^Is_Block=		Mandaic}', "");
Error('\p{Is_Blk: -/a/Mandaic}');
Error('\P{Is_Blk: -/a/Mandaic}');
Expect(1, 2143, '\p{Is_Blk=mandaic}', "");
Expect(0, 2143, '\p{^Is_Blk=mandaic}', "");
Expect(0, 2143, '\P{Is_Blk=mandaic}', "");
Expect(1, 2143, '\P{^Is_Blk=mandaic}', "");
Expect(0, 2111, '\p{Is_Blk=mandaic}', "");
Expect(1, 2111, '\p{^Is_Blk=mandaic}', "");
Expect(1, 2111, '\P{Is_Blk=mandaic}', "");
Expect(0, 2111, '\P{^Is_Blk=mandaic}', "");
Expect(1, 2143, '\p{Is_Blk: -	mandaic}', "");
Expect(0, 2143, '\p{^Is_Blk: -	mandaic}', "");
Expect(0, 2143, '\P{Is_Blk: -	mandaic}', "");
Expect(1, 2143, '\P{^Is_Blk: -	mandaic}', "");
Expect(0, 2111, '\p{Is_Blk: -	mandaic}', "");
Expect(1, 2111, '\p{^Is_Blk: -	mandaic}', "");
Expect(1, 2111, '\P{Is_Blk: -	mandaic}', "");
Expect(0, 2111, '\P{^Is_Blk: -	mandaic}', "");
Error('\p{Block=	:=Kangxi radicals}');
Error('\P{Block=	:=Kangxi radicals}');
Expect(1, 12255, '\p{Block:   kangxiradicals}', "");
Expect(0, 12255, '\p{^Block:   kangxiradicals}', "");
Expect(0, 12255, '\P{Block:   kangxiradicals}', "");
Expect(1, 12255, '\P{^Block:   kangxiradicals}', "");
Expect(0, 12031, '\p{Block:   kangxiradicals}', "");
Expect(1, 12031, '\p{^Block:   kangxiradicals}', "");
Expect(1, 12031, '\P{Block:   kangxiradicals}', "");
Expect(0, 12031, '\P{^Block:   kangxiradicals}', "");
Expect(1, 12255, '\p{Block= KANGXI Radicals}', "");
Expect(0, 12255, '\p{^Block= KANGXI Radicals}', "");
Expect(0, 12255, '\P{Block= KANGXI Radicals}', "");
Expect(1, 12255, '\P{^Block= KANGXI Radicals}', "");
Expect(0, 12031, '\p{Block= KANGXI Radicals}', "");
Expect(1, 12031, '\p{^Block= KANGXI Radicals}', "");
Expect(1, 12031, '\P{Block= KANGXI Radicals}', "");
Expect(0, 12031, '\P{^Block= KANGXI Radicals}', "");
Error('\p{Blk=Kangxi/a/radicals}');
Error('\P{Blk=Kangxi/a/radicals}');
Expect(1, 12255, '\p{Blk:	kangxiradicals}', "");
Expect(0, 12255, '\p{^Blk:	kangxiradicals}', "");
Expect(0, 12255, '\P{Blk:	kangxiradicals}', "");
Expect(1, 12255, '\P{^Blk:	kangxiradicals}', "");
Expect(0, 12031, '\p{Blk:	kangxiradicals}', "");
Expect(1, 12031, '\p{^Blk:	kangxiradicals}', "");
Expect(1, 12031, '\P{Blk:	kangxiradicals}', "");
Expect(0, 12031, '\P{^Blk:	kangxiradicals}', "");
Expect(1, 12255, '\p{Blk=	-kangxi RADICALS}', "");
Expect(0, 12255, '\p{^Blk=	-kangxi RADICALS}', "");
Expect(0, 12255, '\P{Blk=	-kangxi RADICALS}', "");
Expect(1, 12255, '\P{^Blk=	-kangxi RADICALS}', "");
Expect(0, 12031, '\p{Blk=	-kangxi RADICALS}', "");
Expect(1, 12031, '\p{^Blk=	-kangxi RADICALS}', "");
Expect(1, 12031, '\P{Blk=	-kangxi RADICALS}', "");
Expect(0, 12031, '\P{^Blk=	-kangxi RADICALS}', "");
Error('\p{Is_Block=	:=Kangxi_Radicals}');
Error('\P{Is_Block=	:=Kangxi_Radicals}');
Expect(1, 12255, '\p{Is_Block=kangxiradicals}', "");
Expect(0, 12255, '\p{^Is_Block=kangxiradicals}', "");
Expect(0, 12255, '\P{Is_Block=kangxiradicals}', "");
Expect(1, 12255, '\P{^Is_Block=kangxiradicals}', "");
Expect(0, 12031, '\p{Is_Block=kangxiradicals}', "");
Expect(1, 12031, '\p{^Is_Block=kangxiradicals}', "");
Expect(1, 12031, '\P{Is_Block=kangxiradicals}', "");
Expect(0, 12031, '\P{^Is_Block=kangxiradicals}', "");
Expect(1, 12255, '\p{Is_Block=KangxiRadicals}', "");
Expect(0, 12255, '\p{^Is_Block=KangxiRadicals}', "");
Expect(0, 12255, '\P{Is_Block=KangxiRadicals}', "");
Expect(1, 12255, '\P{^Is_Block=KangxiRadicals}', "");
Expect(0, 12031, '\p{Is_Block=KangxiRadicals}', "");
Expect(1, 12031, '\p{^Is_Block=KangxiRadicals}', "");
Expect(1, 12031, '\P{Is_Block=KangxiRadicals}', "");
Expect(0, 12031, '\P{^Is_Block=KangxiRadicals}', "");
Error('\p{Is_Blk=:= 	Kangxi-radicals}');
Error('\P{Is_Blk=:= 	Kangxi-radicals}');
Expect(1, 12255, '\p{Is_Blk=kangxiradicals}', "");
Expect(0, 12255, '\p{^Is_Blk=kangxiradicals}', "");
Expect(0, 12255, '\P{Is_Blk=kangxiradicals}', "");
Expect(1, 12255, '\P{^Is_Blk=kangxiradicals}', "");
Expect(0, 12031, '\p{Is_Blk=kangxiradicals}', "");
Expect(1, 12031, '\p{^Is_Blk=kangxiradicals}', "");
Expect(1, 12031, '\P{Is_Blk=kangxiradicals}', "");
Expect(0, 12031, '\P{^Is_Blk=kangxiradicals}', "");
Expect(1, 12255, '\p{Is_Blk=__Kangxi_RADICALS}', "");
Expect(0, 12255, '\p{^Is_Blk=__Kangxi_RADICALS}', "");
Expect(0, 12255, '\P{Is_Blk=__Kangxi_RADICALS}', "");
Expect(1, 12255, '\P{^Is_Blk=__Kangxi_RADICALS}', "");
Expect(0, 12031, '\p{Is_Blk=__Kangxi_RADICALS}', "");
Expect(1, 12031, '\p{^Is_Blk=__Kangxi_RADICALS}', "");
Expect(1, 12031, '\P{Is_Blk=__Kangxi_RADICALS}', "");
Expect(0, 12031, '\P{^Is_Blk=__Kangxi_RADICALS}', "");
Error('\p{Block: -ARABICpresentation	Forms	B/a/}');
Error('\P{Block: -ARABICpresentation	Forms	B/a/}');
Expect(1, 65279, '\p{Block=arabicpresentationformsb}', "");
Expect(0, 65279, '\p{^Block=arabicpresentationformsb}', "");
Expect(0, 65279, '\P{Block=arabicpresentationformsb}', "");
Expect(1, 65279, '\P{^Block=arabicpresentationformsb}', "");
Expect(0, 65135, '\p{Block=arabicpresentationformsb}', "");
Expect(1, 65135, '\p{^Block=arabicpresentationformsb}', "");
Expect(1, 65135, '\P{Block=arabicpresentationformsb}', "");
Expect(0, 65135, '\P{^Block=arabicpresentationformsb}', "");
Expect(1, 65279, '\p{Block= arabicpresentation_forms	B}', "");
Expect(0, 65279, '\p{^Block= arabicpresentation_forms	B}', "");
Expect(0, 65279, '\P{Block= arabicpresentation_forms	B}', "");
Expect(1, 65279, '\P{^Block= arabicpresentation_forms	B}', "");
Expect(0, 65135, '\p{Block= arabicpresentation_forms	B}', "");
Expect(1, 65135, '\p{^Block= arabicpresentation_forms	B}', "");
Expect(1, 65135, '\P{Block= arabicpresentation_forms	B}', "");
Expect(0, 65135, '\P{^Block= arabicpresentation_forms	B}', "");
Error('\p{Blk= /a/ARABICpresentation forms_B}');
Error('\P{Blk= /a/ARABICpresentation forms_B}');
Expect(1, 65279, '\p{Blk=arabicpresentationformsb}', "");
Expect(0, 65279, '\p{^Blk=arabicpresentationformsb}', "");
Expect(0, 65279, '\P{Blk=arabicpresentationformsb}', "");
Expect(1, 65279, '\P{^Blk=arabicpresentationformsb}', "");
Expect(0, 65135, '\p{Blk=arabicpresentationformsb}', "");
Expect(1, 65135, '\p{^Blk=arabicpresentationformsb}', "");
Expect(1, 65135, '\P{Blk=arabicpresentationformsb}', "");
Expect(0, 65135, '\P{^Blk=arabicpresentationformsb}', "");
Expect(1, 65279, '\p{Blk=	-Arabic	Presentation-FORMS B}', "");
Expect(0, 65279, '\p{^Blk=	-Arabic	Presentation-FORMS B}', "");
Expect(0, 65279, '\P{Blk=	-Arabic	Presentation-FORMS B}', "");
Expect(1, 65279, '\P{^Blk=	-Arabic	Presentation-FORMS B}', "");
Expect(0, 65135, '\p{Blk=	-Arabic	Presentation-FORMS B}', "");
Expect(1, 65135, '\p{^Blk=	-Arabic	Presentation-FORMS B}', "");
Expect(1, 65135, '\P{Blk=	-Arabic	Presentation-FORMS B}', "");
Expect(0, 65135, '\P{^Blk=	-Arabic	Presentation-FORMS B}', "");
Error('\p{Is_Block=	 Arabic_PRESENTATIONforms_B/a/}');
Error('\P{Is_Block=	 Arabic_PRESENTATIONforms_B/a/}');
Expect(1, 65279, '\p{Is_Block:arabicpresentationformsb}', "");
Expect(0, 65279, '\p{^Is_Block:arabicpresentationformsb}', "");
Expect(0, 65279, '\P{Is_Block:arabicpresentationformsb}', "");
Expect(1, 65279, '\P{^Is_Block:arabicpresentationformsb}', "");
Expect(0, 65135, '\p{Is_Block:arabicpresentationformsb}', "");
Expect(1, 65135, '\p{^Is_Block:arabicpresentationformsb}', "");
Expect(1, 65135, '\P{Is_Block:arabicpresentationformsb}', "");
Expect(0, 65135, '\P{^Is_Block:arabicpresentationformsb}', "");
Expect(1, 65279, '\p{Is_Block= ARABIC presentation	Forms	B}', "");
Expect(0, 65279, '\p{^Is_Block= ARABIC presentation	Forms	B}', "");
Expect(0, 65279, '\P{Is_Block= ARABIC presentation	Forms	B}', "");
Expect(1, 65279, '\P{^Is_Block= ARABIC presentation	Forms	B}', "");
Expect(0, 65135, '\p{Is_Block= ARABIC presentation	Forms	B}', "");
Expect(1, 65135, '\p{^Is_Block= ARABIC presentation	Forms	B}', "");
Expect(1, 65135, '\P{Is_Block= ARABIC presentation	Forms	B}', "");
Expect(0, 65135, '\P{^Is_Block= ARABIC presentation	Forms	B}', "");
Error('\p{Is_Blk=_-Arabic:=Presentation FORMSB}');
Error('\P{Is_Blk=_-Arabic:=Presentation FORMSB}');
Expect(1, 65279, '\p{Is_Blk=arabicpresentationformsb}', "");
Expect(0, 65279, '\p{^Is_Blk=arabicpresentationformsb}', "");
Expect(0, 65279, '\P{Is_Blk=arabicpresentationformsb}', "");
Expect(1, 65279, '\P{^Is_Blk=arabicpresentationformsb}', "");
Expect(0, 65135, '\p{Is_Blk=arabicpresentationformsb}', "");
Expect(1, 65135, '\p{^Is_Blk=arabicpresentationformsb}', "");
Expect(1, 65135, '\P{Is_Blk=arabicpresentationformsb}', "");
Expect(0, 65135, '\P{^Is_Blk=arabicpresentationformsb}', "");
Expect(1, 65279, '\p{Is_Blk=- arabic_PRESENTATIONForms B}', "");
Expect(0, 65279, '\p{^Is_Blk=- arabic_PRESENTATIONForms B}', "");
Expect(0, 65279, '\P{Is_Blk=- arabic_PRESENTATIONForms B}', "");
Expect(1, 65279, '\P{^Is_Blk=- arabic_PRESENTATIONForms B}', "");
Expect(0, 65135, '\p{Is_Blk=- arabic_PRESENTATIONForms B}', "");
Expect(1, 65135, '\p{^Is_Blk=- arabic_PRESENTATIONForms B}', "");
Expect(1, 65135, '\P{Is_Blk=- arabic_PRESENTATIONForms B}', "");
Expect(0, 65135, '\P{^Is_Blk=- arabic_PRESENTATIONForms B}', "");
Error('\p{Block=/a/GURMUKHI}');
Error('\P{Block=/a/GURMUKHI}');
Expect(1, 2687, '\p{Block=gurmukhi}', "");
Expect(0, 2687, '\p{^Block=gurmukhi}', "");
Expect(0, 2687, '\P{Block=gurmukhi}', "");
Expect(1, 2687, '\P{^Block=gurmukhi}', "");
Expect(0, 2559, '\p{Block=gurmukhi}', "");
Expect(1, 2559, '\p{^Block=gurmukhi}', "");
Expect(1, 2559, '\P{Block=gurmukhi}', "");
Expect(0, 2559, '\P{^Block=gurmukhi}', "");
Expect(1, 2687, '\p{Block=_gurmukhi}', "");
Expect(0, 2687, '\p{^Block=_gurmukhi}', "");
Expect(0, 2687, '\P{Block=_gurmukhi}', "");
Expect(1, 2687, '\P{^Block=_gurmukhi}', "");
Expect(0, 2559, '\p{Block=_gurmukhi}', "");
Expect(1, 2559, '\p{^Block=_gurmukhi}', "");
Expect(1, 2559, '\P{Block=_gurmukhi}', "");
Expect(0, 2559, '\P{^Block=_gurmukhi}', "");
Error('\p{Blk=-:=Gurmukhi}');
Error('\P{Blk=-:=Gurmukhi}');
Expect(1, 2687, '\p{Blk=gurmukhi}', "");
Expect(0, 2687, '\p{^Blk=gurmukhi}', "");
Expect(0, 2687, '\P{Blk=gurmukhi}', "");
Expect(1, 2687, '\P{^Blk=gurmukhi}', "");
Expect(0, 2559, '\p{Blk=gurmukhi}', "");
Expect(1, 2559, '\p{^Blk=gurmukhi}', "");
Expect(1, 2559, '\P{Blk=gurmukhi}', "");
Expect(0, 2559, '\P{^Blk=gurmukhi}', "");
Expect(1, 2687, '\p{Blk:   	Gurmukhi}', "");
Expect(0, 2687, '\p{^Blk:   	Gurmukhi}', "");
Expect(0, 2687, '\P{Blk:   	Gurmukhi}', "");
Expect(1, 2687, '\P{^Blk:   	Gurmukhi}', "");
Expect(0, 2559, '\p{Blk:   	Gurmukhi}', "");
Expect(1, 2559, '\p{^Blk:   	Gurmukhi}', "");
Expect(1, 2559, '\P{Blk:   	Gurmukhi}', "");
Expect(0, 2559, '\P{^Blk:   	Gurmukhi}', "");
Error('\p{Is_Block=	_Gurmukhi:=}');
Error('\P{Is_Block=	_Gurmukhi:=}');
Expect(1, 2687, '\p{Is_Block=gurmukhi}', "");
Expect(0, 2687, '\p{^Is_Block=gurmukhi}', "");
Expect(0, 2687, '\P{Is_Block=gurmukhi}', "");
Expect(1, 2687, '\P{^Is_Block=gurmukhi}', "");
Expect(0, 2559, '\p{Is_Block=gurmukhi}', "");
Expect(1, 2559, '\p{^Is_Block=gurmukhi}', "");
Expect(1, 2559, '\P{Is_Block=gurmukhi}', "");
Expect(0, 2559, '\P{^Is_Block=gurmukhi}', "");
Expect(1, 2687, '\p{Is_Block=		gurmukhi}', "");
Expect(0, 2687, '\p{^Is_Block=		gurmukhi}', "");
Expect(0, 2687, '\P{Is_Block=		gurmukhi}', "");
Expect(1, 2687, '\P{^Is_Block=		gurmukhi}', "");
Expect(0, 2559, '\p{Is_Block=		gurmukhi}', "");
Expect(1, 2559, '\p{^Is_Block=		gurmukhi}', "");
Expect(1, 2559, '\P{Is_Block=		gurmukhi}', "");
Expect(0, 2559, '\P{^Is_Block=		gurmukhi}', "");
Error('\p{Is_Blk=	:=Gurmukhi}');
Error('\P{Is_Blk=	:=Gurmukhi}');
Expect(1, 2687, '\p{Is_Blk=gurmukhi}', "");
Expect(0, 2687, '\p{^Is_Blk=gurmukhi}', "");
Expect(0, 2687, '\P{Is_Blk=gurmukhi}', "");
Expect(1, 2687, '\P{^Is_Blk=gurmukhi}', "");
Expect(0, 2559, '\p{Is_Blk=gurmukhi}', "");
Expect(1, 2559, '\p{^Is_Blk=gurmukhi}', "");
Expect(1, 2559, '\P{Is_Blk=gurmukhi}', "");
Expect(0, 2559, '\P{^Is_Blk=gurmukhi}', "");
Expect(1, 2687, '\p{Is_Blk=-gurmukhi}', "");
Expect(0, 2687, '\p{^Is_Blk=-gurmukhi}', "");
Expect(0, 2687, '\P{Is_Blk=-gurmukhi}', "");
Expect(1, 2687, '\P{^Is_Blk=-gurmukhi}', "");
Expect(0, 2559, '\p{Is_Blk=-gurmukhi}', "");
Expect(1, 2559, '\p{^Is_Blk=-gurmukhi}', "");
Expect(1, 2559, '\P{Is_Blk=-gurmukhi}', "");
Expect(0, 2559, '\P{^Is_Blk=-gurmukhi}', "");
Error('\p{Block=  Enclosed:=IDEOGRAPHICSUPPLEMENT}');
Error('\P{Block=  Enclosed:=IDEOGRAPHICSUPPLEMENT}');
Expect(1, 127743, '\p{Block=enclosedideographicsupplement}', "");
Expect(0, 127743, '\p{^Block=enclosedideographicsupplement}', "");
Expect(0, 127743, '\P{Block=enclosedideographicsupplement}', "");
Expect(1, 127743, '\P{^Block=enclosedideographicsupplement}', "");
Expect(0, 127487, '\p{Block=enclosedideographicsupplement}', "");
Expect(1, 127487, '\p{^Block=enclosedideographicsupplement}', "");
Expect(1, 127487, '\P{Block=enclosedideographicsupplement}', "");
Expect(0, 127487, '\P{^Block=enclosedideographicsupplement}', "");
Expect(1, 127743, '\p{Block=-	enclosed-ideographic	SUPPLEMENT}', "");
Expect(0, 127743, '\p{^Block=-	enclosed-ideographic	SUPPLEMENT}', "");
Expect(0, 127743, '\P{Block=-	enclosed-ideographic	SUPPLEMENT}', "");
Expect(1, 127743, '\P{^Block=-	enclosed-ideographic	SUPPLEMENT}', "");
Expect(0, 127487, '\p{Block=-	enclosed-ideographic	SUPPLEMENT}', "");
Expect(1, 127487, '\p{^Block=-	enclosed-ideographic	SUPPLEMENT}', "");
Expect(1, 127487, '\P{Block=-	enclosed-ideographic	SUPPLEMENT}', "");
Expect(0, 127487, '\P{^Block=-	enclosed-ideographic	SUPPLEMENT}', "");
Error('\p{Blk=/a/--enclosed	Ideographic-Supplement}');
Error('\P{Blk=/a/--enclosed	Ideographic-Supplement}');
Expect(1, 127743, '\p{Blk=enclosedideographicsupplement}', "");
Expect(0, 127743, '\p{^Blk=enclosedideographicsupplement}', "");
Expect(0, 127743, '\P{Blk=enclosedideographicsupplement}', "");
Expect(1, 127743, '\P{^Blk=enclosedideographicsupplement}', "");
Expect(0, 127487, '\p{Blk=enclosedideographicsupplement}', "");
Expect(1, 127487, '\p{^Blk=enclosedideographicsupplement}', "");
Expect(1, 127487, '\P{Blk=enclosedideographicsupplement}', "");
Expect(0, 127487, '\P{^Blk=enclosedideographicsupplement}', "");
Expect(1, 127743, '\p{Blk= -Enclosed_Ideographic	SUPPLEMENT}', "");
Expect(0, 127743, '\p{^Blk= -Enclosed_Ideographic	SUPPLEMENT}', "");
Expect(0, 127743, '\P{Blk= -Enclosed_Ideographic	SUPPLEMENT}', "");
Expect(1, 127743, '\P{^Blk= -Enclosed_Ideographic	SUPPLEMENT}', "");
Expect(0, 127487, '\p{Blk= -Enclosed_Ideographic	SUPPLEMENT}', "");
Expect(1, 127487, '\p{^Blk= -Enclosed_Ideographic	SUPPLEMENT}', "");
Expect(1, 127487, '\P{Blk= -Enclosed_Ideographic	SUPPLEMENT}', "");
Expect(0, 127487, '\P{^Blk= -Enclosed_Ideographic	SUPPLEMENT}', "");
Error('\p{Is_Block=-	enclosed:=Ideographic_SUPPLEMENT}');
Error('\P{Is_Block=-	enclosed:=Ideographic_SUPPLEMENT}');
Expect(1, 127743, '\p{Is_Block=enclosedideographicsupplement}', "");
Expect(0, 127743, '\p{^Is_Block=enclosedideographicsupplement}', "");
Expect(0, 127743, '\P{Is_Block=enclosedideographicsupplement}', "");
Expect(1, 127743, '\P{^Is_Block=enclosedideographicsupplement}', "");
Expect(0, 127487, '\p{Is_Block=enclosedideographicsupplement}', "");
Expect(1, 127487, '\p{^Is_Block=enclosedideographicsupplement}', "");
Expect(1, 127487, '\P{Is_Block=enclosedideographicsupplement}', "");
Expect(0, 127487, '\P{^Is_Block=enclosedideographicsupplement}', "");
Expect(1, 127743, '\p{Is_Block=_	Enclosed_Ideographic_SUPPLEMENT}', "");
Expect(0, 127743, '\p{^Is_Block=_	Enclosed_Ideographic_SUPPLEMENT}', "");
Expect(0, 127743, '\P{Is_Block=_	Enclosed_Ideographic_SUPPLEMENT}', "");
Expect(1, 127743, '\P{^Is_Block=_	Enclosed_Ideographic_SUPPLEMENT}', "");
Expect(0, 127487, '\p{Is_Block=_	Enclosed_Ideographic_SUPPLEMENT}', "");
Expect(1, 127487, '\p{^Is_Block=_	Enclosed_Ideographic_SUPPLEMENT}', "");
Expect(1, 127487, '\P{Is_Block=_	Enclosed_Ideographic_SUPPLEMENT}', "");
Expect(0, 127487, '\P{^Is_Block=_	Enclosed_Ideographic_SUPPLEMENT}', "");
Error('\p{Is_Blk= 	Enclosed:=ideographic	SUPPLEMENT}');
Error('\P{Is_Blk= 	Enclosed:=ideographic	SUPPLEMENT}');
Expect(1, 127743, '\p{Is_Blk=enclosedideographicsupplement}', "");
Expect(0, 127743, '\p{^Is_Blk=enclosedideographicsupplement}', "");
Expect(0, 127743, '\P{Is_Blk=enclosedideographicsupplement}', "");
Expect(1, 127743, '\P{^Is_Blk=enclosedideographicsupplement}', "");
Expect(0, 127487, '\p{Is_Blk=enclosedideographicsupplement}', "");
Expect(1, 127487, '\p{^Is_Blk=enclosedideographicsupplement}', "");
Expect(1, 127487, '\P{Is_Blk=enclosedideographicsupplement}', "");
Expect(0, 127487, '\P{^Is_Blk=enclosedideographicsupplement}', "");
Expect(1, 127743, '\p{Is_Blk:	 	enclosed-IDEOGRAPHIC-Supplement}', "");
Expect(0, 127743, '\p{^Is_Blk:	 	enclosed-IDEOGRAPHIC-Supplement}', "");
Expect(0, 127743, '\P{Is_Blk:	 	enclosed-IDEOGRAPHIC-Supplement}', "");
Expect(1, 127743, '\P{^Is_Blk:	 	enclosed-IDEOGRAPHIC-Supplement}', "");
Expect(0, 127487, '\p{Is_Blk:	 	enclosed-IDEOGRAPHIC-Supplement}', "");
Expect(1, 127487, '\p{^Is_Blk:	 	enclosed-IDEOGRAPHIC-Supplement}', "");
Expect(1, 127487, '\P{Is_Blk:	 	enclosed-IDEOGRAPHIC-Supplement}', "");
Expect(0, 127487, '\P{^Is_Blk:	 	enclosed-IDEOGRAPHIC-Supplement}', "");
Error('\p{Block=	_Phonetic:=EXTENSIONS-supplement}');
Error('\P{Block=	_Phonetic:=EXTENSIONS-supplement}');
Expect(1, 7615, '\p{Block=phoneticextensionssupplement}', "");
Expect(0, 7615, '\p{^Block=phoneticextensionssupplement}', "");
Expect(0, 7615, '\P{Block=phoneticextensionssupplement}', "");
Expect(1, 7615, '\P{^Block=phoneticextensionssupplement}', "");
Expect(0, 7551, '\p{Block=phoneticextensionssupplement}', "");
Expect(1, 7551, '\p{^Block=phoneticextensionssupplement}', "");
Expect(1, 7551, '\P{Block=phoneticextensionssupplement}', "");
Expect(0, 7551, '\P{^Block=phoneticextensionssupplement}', "");
Expect(1, 7615, '\p{Block=- Phonetic_extensions_Supplement}', "");
Expect(0, 7615, '\p{^Block=- Phonetic_extensions_Supplement}', "");
Expect(0, 7615, '\P{Block=- Phonetic_extensions_Supplement}', "");
Expect(1, 7615, '\P{^Block=- Phonetic_extensions_Supplement}', "");
Expect(0, 7551, '\p{Block=- Phonetic_extensions_Supplement}', "");
Expect(1, 7551, '\p{^Block=- Phonetic_extensions_Supplement}', "");
Expect(1, 7551, '\P{Block=- Phonetic_extensions_Supplement}', "");
Expect(0, 7551, '\P{^Block=- Phonetic_extensions_Supplement}', "");
Error('\p{Blk= :=Phoneticextensions	Supplement}');
Error('\P{Blk= :=Phoneticextensions	Supplement}');
Expect(1, 7615, '\p{Blk=phoneticextensionssupplement}', "");
Expect(0, 7615, '\p{^Blk=phoneticextensionssupplement}', "");
Expect(0, 7615, '\P{Blk=phoneticextensionssupplement}', "");
Expect(1, 7615, '\P{^Blk=phoneticextensionssupplement}', "");
Expect(0, 7551, '\p{Blk=phoneticextensionssupplement}', "");
Expect(1, 7551, '\p{^Blk=phoneticextensionssupplement}', "");
Expect(1, 7551, '\P{Blk=phoneticextensionssupplement}', "");
Expect(0, 7551, '\P{^Blk=phoneticextensionssupplement}', "");
Expect(1, 7615, '\p{Blk=-_PHONETIC-extensionsSUPPLEMENT}', "");
Expect(0, 7615, '\p{^Blk=-_PHONETIC-extensionsSUPPLEMENT}', "");
Expect(0, 7615, '\P{Blk=-_PHONETIC-extensionsSUPPLEMENT}', "");
Expect(1, 7615, '\P{^Blk=-_PHONETIC-extensionsSUPPLEMENT}', "");
Expect(0, 7551, '\p{Blk=-_PHONETIC-extensionsSUPPLEMENT}', "");
Expect(1, 7551, '\p{^Blk=-_PHONETIC-extensionsSUPPLEMENT}', "");
Expect(1, 7551, '\P{Blk=-_PHONETIC-extensionsSUPPLEMENT}', "");
Expect(0, 7551, '\P{^Blk=-_PHONETIC-extensionsSUPPLEMENT}', "");
Error('\p{Is_Block:   _ phonetic:=Extensions Supplement}');
Error('\P{Is_Block:   _ phonetic:=Extensions Supplement}');
Expect(1, 7615, '\p{Is_Block=phoneticextensionssupplement}', "");
Expect(0, 7615, '\p{^Is_Block=phoneticextensionssupplement}', "");
Expect(0, 7615, '\P{Is_Block=phoneticextensionssupplement}', "");
Expect(1, 7615, '\P{^Is_Block=phoneticextensionssupplement}', "");
Expect(0, 7551, '\p{Is_Block=phoneticextensionssupplement}', "");
Expect(1, 7551, '\p{^Is_Block=phoneticextensionssupplement}', "");
Expect(1, 7551, '\P{Is_Block=phoneticextensionssupplement}', "");
Expect(0, 7551, '\P{^Is_Block=phoneticextensionssupplement}', "");
Expect(1, 7615, '\p{Is_Block=_ PHONETIC-Extensions_SUPPLEMENT}', "");
Expect(0, 7615, '\p{^Is_Block=_ PHONETIC-Extensions_SUPPLEMENT}', "");
Expect(0, 7615, '\P{Is_Block=_ PHONETIC-Extensions_SUPPLEMENT}', "");
Expect(1, 7615, '\P{^Is_Block=_ PHONETIC-Extensions_SUPPLEMENT}', "");
Expect(0, 7551, '\p{Is_Block=_ PHONETIC-Extensions_SUPPLEMENT}', "");
Expect(1, 7551, '\p{^Is_Block=_ PHONETIC-Extensions_SUPPLEMENT}', "");
Expect(1, 7551, '\P{Is_Block=_ PHONETIC-Extensions_SUPPLEMENT}', "");
Expect(0, 7551, '\P{^Is_Block=_ PHONETIC-Extensions_SUPPLEMENT}', "");
Error('\p{Is_Blk=_	Phonetic/a/EXTENSIONS	supplement}');
Error('\P{Is_Blk=_	Phonetic/a/EXTENSIONS	supplement}');
Expect(1, 7615, '\p{Is_Blk=phoneticextensionssupplement}', "");
Expect(0, 7615, '\p{^Is_Blk=phoneticextensionssupplement}', "");
Expect(0, 7615, '\P{Is_Blk=phoneticextensionssupplement}', "");
Expect(1, 7615, '\P{^Is_Blk=phoneticextensionssupplement}', "");
Expect(0, 7551, '\p{Is_Blk=phoneticextensionssupplement}', "");
Expect(1, 7551, '\p{^Is_Blk=phoneticextensionssupplement}', "");
Expect(1, 7551, '\P{Is_Blk=phoneticextensionssupplement}', "");
Expect(0, 7551, '\P{^Is_Blk=phoneticextensionssupplement}', "");
Expect(1, 7615, '\p{Is_Blk=_ Phonetic_EXTENSIONSSupplement}', "");
Expect(0, 7615, '\p{^Is_Blk=_ Phonetic_EXTENSIONSSupplement}', "");
Expect(0, 7615, '\P{Is_Blk=_ Phonetic_EXTENSIONSSupplement}', "");
Expect(1, 7615, '\P{^Is_Blk=_ Phonetic_EXTENSIONSSupplement}', "");
Expect(0, 7551, '\p{Is_Blk=_ Phonetic_EXTENSIONSSupplement}', "");
Expect(1, 7551, '\p{^Is_Blk=_ Phonetic_EXTENSIONSSupplement}', "");
Expect(1, 7551, '\P{Is_Blk=_ Phonetic_EXTENSIONSSupplement}', "");
Expect(0, 7551, '\P{^Is_Blk=_ Phonetic_EXTENSIONSSupplement}', "");
Error('\p{Block=	-BATAK/a/}');
Error('\P{Block=	-BATAK/a/}');
Expect(1, 7167, '\p{Block=batak}', "");
Expect(0, 7167, '\p{^Block=batak}', "");
Expect(0, 7167, '\P{Block=batak}', "");
Expect(1, 7167, '\P{^Block=batak}', "");
Expect(0, 7103, '\p{Block=batak}', "");
Expect(1, 7103, '\p{^Block=batak}', "");
Expect(1, 7103, '\P{Block=batak}', "");
Expect(0, 7103, '\P{^Block=batak}', "");
Expect(1, 7167, '\p{Block= BATAK}', "");
Expect(0, 7167, '\p{^Block= BATAK}', "");
Expect(0, 7167, '\P{Block= BATAK}', "");
Expect(1, 7167, '\P{^Block= BATAK}', "");
Expect(0, 7103, '\p{Block= BATAK}', "");
Expect(1, 7103, '\p{^Block= BATAK}', "");
Expect(1, 7103, '\P{Block= BATAK}', "");
Expect(0, 7103, '\P{^Block= BATAK}', "");
Error('\p{Blk=:=__batak}');
Error('\P{Blk=:=__batak}');
Expect(1, 7167, '\p{Blk=batak}', "");
Expect(0, 7167, '\p{^Blk=batak}', "");
Expect(0, 7167, '\P{Blk=batak}', "");
Expect(1, 7167, '\P{^Blk=batak}', "");
Expect(0, 7103, '\p{Blk=batak}', "");
Expect(1, 7103, '\p{^Blk=batak}', "");
Expect(1, 7103, '\P{Blk=batak}', "");
Expect(0, 7103, '\P{^Blk=batak}', "");
Expect(1, 7167, '\p{Blk=- Batak}', "");
Expect(0, 7167, '\p{^Blk=- Batak}', "");
Expect(0, 7167, '\P{Blk=- Batak}', "");
Expect(1, 7167, '\P{^Blk=- Batak}', "");
Expect(0, 7103, '\p{Blk=- Batak}', "");
Expect(1, 7103, '\p{^Blk=- Batak}', "");
Expect(1, 7103, '\P{Blk=- Batak}', "");
Expect(0, 7103, '\P{^Blk=- Batak}', "");
Error('\p{Is_Block=_Batak/a/}');
Error('\P{Is_Block=_Batak/a/}');
Expect(1, 7167, '\p{Is_Block=batak}', "");
Expect(0, 7167, '\p{^Is_Block=batak}', "");
Expect(0, 7167, '\P{Is_Block=batak}', "");
Expect(1, 7167, '\P{^Is_Block=batak}', "");
Expect(0, 7103, '\p{Is_Block=batak}', "");
Expect(1, 7103, '\p{^Is_Block=batak}', "");
Expect(1, 7103, '\P{Is_Block=batak}', "");
Expect(0, 7103, '\P{^Is_Block=batak}', "");
Expect(1, 7167, '\p{Is_Block= Batak}', "");
Expect(0, 7167, '\p{^Is_Block= Batak}', "");
Expect(0, 7167, '\P{Is_Block= Batak}', "");
Expect(1, 7167, '\P{^Is_Block= Batak}', "");
Expect(0, 7103, '\p{Is_Block= Batak}', "");
Expect(1, 7103, '\p{^Is_Block= Batak}', "");
Expect(1, 7103, '\P{Is_Block= Batak}', "");
Expect(0, 7103, '\P{^Is_Block= Batak}', "");
Error('\p{Is_Blk=	Batak:=}');
Error('\P{Is_Blk=	Batak:=}');
Expect(1, 7167, '\p{Is_Blk=batak}', "");
Expect(0, 7167, '\p{^Is_Blk=batak}', "");
Expect(0, 7167, '\P{Is_Blk=batak}', "");
Expect(1, 7167, '\P{^Is_Blk=batak}', "");
Expect(0, 7103, '\p{Is_Blk=batak}', "");
Expect(1, 7103, '\p{^Is_Blk=batak}', "");
Expect(1, 7103, '\P{Is_Blk=batak}', "");
Expect(0, 7103, '\P{^Is_Blk=batak}', "");
Expect(1, 7167, '\p{Is_Blk=	_Batak}', "");
Expect(0, 7167, '\p{^Is_Blk=	_Batak}', "");
Expect(0, 7167, '\P{Is_Blk=	_Batak}', "");
Expect(1, 7167, '\P{^Is_Blk=	_Batak}', "");
Expect(0, 7103, '\p{Is_Blk=	_Batak}', "");
Expect(1, 7103, '\p{^Is_Blk=	_Batak}', "");
Expect(1, 7103, '\P{Is_Blk=	_Batak}', "");
Expect(0, 7103, '\P{^Is_Blk=	_Batak}', "");
Error('\p{Block=- greek/a/extended}');
Error('\P{Block=- greek/a/extended}');
Expect(1, 8191, '\p{Block=greekextended}', "");
Expect(0, 8191, '\p{^Block=greekextended}', "");
Expect(0, 8191, '\P{Block=greekextended}', "");
Expect(1, 8191, '\P{^Block=greekextended}', "");
Expect(0, 7935, '\p{Block=greekextended}', "");
Expect(1, 7935, '\p{^Block=greekextended}', "");
Expect(1, 7935, '\P{Block=greekextended}', "");
Expect(0, 7935, '\P{^Block=greekextended}', "");
Expect(1, 8191, '\p{Block=-	GREEK-Extended}', "");
Expect(0, 8191, '\p{^Block=-	GREEK-Extended}', "");
Expect(0, 8191, '\P{Block=-	GREEK-Extended}', "");
Expect(1, 8191, '\P{^Block=-	GREEK-Extended}', "");
Expect(0, 7935, '\p{Block=-	GREEK-Extended}', "");
Expect(1, 7935, '\p{^Block=-	GREEK-Extended}', "");
Expect(1, 7935, '\P{Block=-	GREEK-Extended}', "");
Expect(0, 7935, '\P{^Block=-	GREEK-Extended}', "");
Error('\p{Blk= GREEKExtended/a/}');
Error('\P{Blk= GREEKExtended/a/}');
Expect(1, 8191, '\p{Blk:   greekextended}', "");
Expect(0, 8191, '\p{^Blk:   greekextended}', "");
Expect(0, 8191, '\P{Blk:   greekextended}', "");
Expect(1, 8191, '\P{^Blk:   greekextended}', "");
Expect(0, 7935, '\p{Blk:   greekextended}', "");
Expect(1, 7935, '\p{^Blk:   greekextended}', "");
Expect(1, 7935, '\P{Blk:   greekextended}', "");
Expect(0, 7935, '\P{^Blk:   greekextended}', "");
Expect(1, 8191, '\p{Blk= _GreekExtended}', "");
Expect(0, 8191, '\p{^Blk= _GreekExtended}', "");
Expect(0, 8191, '\P{Blk= _GreekExtended}', "");
Expect(1, 8191, '\P{^Blk= _GreekExtended}', "");
Expect(0, 7935, '\p{Blk= _GreekExtended}', "");
Expect(1, 7935, '\p{^Blk= _GreekExtended}', "");
Expect(1, 7935, '\P{Blk= _GreekExtended}', "");
Expect(0, 7935, '\P{^Blk= _GreekExtended}', "");
Error('\p{Is_Block=_/a/greek-EXTENDED}');
Error('\P{Is_Block=_/a/greek-EXTENDED}');
Expect(1, 8191, '\p{Is_Block=greekextended}', "");
Expect(0, 8191, '\p{^Is_Block=greekextended}', "");
Expect(0, 8191, '\P{Is_Block=greekextended}', "");
Expect(1, 8191, '\P{^Is_Block=greekextended}', "");
Expect(0, 7935, '\p{Is_Block=greekextended}', "");
Expect(1, 7935, '\p{^Is_Block=greekextended}', "");
Expect(1, 7935, '\P{Is_Block=greekextended}', "");
Expect(0, 7935, '\P{^Is_Block=greekextended}', "");
Expect(1, 8191, '\p{Is_Block= _Greek-EXTENDED}', "");
Expect(0, 8191, '\p{^Is_Block= _Greek-EXTENDED}', "");
Expect(0, 8191, '\P{Is_Block= _Greek-EXTENDED}', "");
Expect(1, 8191, '\P{^Is_Block= _Greek-EXTENDED}', "");
Expect(0, 7935, '\p{Is_Block= _Greek-EXTENDED}', "");
Expect(1, 7935, '\p{^Is_Block= _Greek-EXTENDED}', "");
Expect(1, 7935, '\P{Is_Block= _Greek-EXTENDED}', "");
Expect(0, 7935, '\P{^Is_Block= _Greek-EXTENDED}', "");
Error('\p{Is_Blk=:=_	greek-Extended}');
Error('\P{Is_Blk=:=_	greek-Extended}');
Expect(1, 8191, '\p{Is_Blk:greekextended}', "");
Expect(0, 8191, '\p{^Is_Blk:greekextended}', "");
Expect(0, 8191, '\P{Is_Blk:greekextended}', "");
Expect(1, 8191, '\P{^Is_Blk:greekextended}', "");
Expect(0, 7935, '\p{Is_Blk:greekextended}', "");
Expect(1, 7935, '\p{^Is_Blk:greekextended}', "");
Expect(1, 7935, '\P{Is_Blk:greekextended}', "");
Expect(0, 7935, '\P{^Is_Blk:greekextended}', "");
Expect(1, 8191, '\p{Is_Blk:	_	GREEK EXTENDED}', "");
Expect(0, 8191, '\p{^Is_Blk:	_	GREEK EXTENDED}', "");
Expect(0, 8191, '\P{Is_Blk:	_	GREEK EXTENDED}', "");
Expect(1, 8191, '\P{^Is_Blk:	_	GREEK EXTENDED}', "");
Expect(0, 7935, '\p{Is_Blk:	_	GREEK EXTENDED}', "");
Expect(1, 7935, '\p{^Is_Blk:	_	GREEK EXTENDED}', "");
Expect(1, 7935, '\P{Is_Blk:	_	GREEK EXTENDED}', "");
Expect(0, 7935, '\P{^Is_Blk:	_	GREEK EXTENDED}', "");
Error('\p{Block=	/a/DESERET}');
Error('\P{Block=	/a/DESERET}');
Expect(1, 66639, '\p{Block=deseret}', "");
Expect(0, 66639, '\p{^Block=deseret}', "");
Expect(0, 66639, '\P{Block=deseret}', "");
Expect(1, 66639, '\P{^Block=deseret}', "");
Expect(0, 66559, '\p{Block=deseret}', "");
Expect(1, 66559, '\p{^Block=deseret}', "");
Expect(1, 66559, '\P{Block=deseret}', "");
Expect(0, 66559, '\P{^Block=deseret}', "");
Expect(1, 66639, '\p{Block=-Deseret}', "");
Expect(0, 66639, '\p{^Block=-Deseret}', "");
Expect(0, 66639, '\P{Block=-Deseret}', "");
Expect(1, 66639, '\P{^Block=-Deseret}', "");
Expect(0, 66559, '\p{Block=-Deseret}', "");
Expect(1, 66559, '\p{^Block=-Deseret}', "");
Expect(1, 66559, '\P{Block=-Deseret}', "");
Expect(0, 66559, '\P{^Block=-Deseret}', "");
Error('\p{Blk=_ DESERET/a/}');
Error('\P{Blk=_ DESERET/a/}');
Expect(1, 66639, '\p{Blk=deseret}', "");
Expect(0, 66639, '\p{^Blk=deseret}', "");
Expect(0, 66639, '\P{Blk=deseret}', "");
Expect(1, 66639, '\P{^Blk=deseret}', "");
Expect(0, 66559, '\p{Blk=deseret}', "");
Expect(1, 66559, '\p{^Blk=deseret}', "");
Expect(1, 66559, '\P{Blk=deseret}', "");
Expect(0, 66559, '\P{^Blk=deseret}', "");
Expect(1, 66639, '\p{Blk= Deseret}', "");
Expect(0, 66639, '\p{^Blk= Deseret}', "");
Expect(0, 66639, '\P{Blk= Deseret}', "");
Expect(1, 66639, '\P{^Blk= Deseret}', "");
Expect(0, 66559, '\p{Blk= Deseret}', "");
Expect(1, 66559, '\p{^Blk= Deseret}', "");
Expect(1, 66559, '\P{Blk= Deseret}', "");
Expect(0, 66559, '\P{^Blk= Deseret}', "");
Error('\p{Is_Block= -deseret:=}');
Error('\P{Is_Block= -deseret:=}');
Expect(1, 66639, '\p{Is_Block=deseret}', "");
Expect(0, 66639, '\p{^Is_Block=deseret}', "");
Expect(0, 66639, '\P{Is_Block=deseret}', "");
Expect(1, 66639, '\P{^Is_Block=deseret}', "");
Expect(0, 66559, '\p{Is_Block=deseret}', "");
Expect(1, 66559, '\p{^Is_Block=deseret}', "");
Expect(1, 66559, '\P{Is_Block=deseret}', "");
Expect(0, 66559, '\P{^Is_Block=deseret}', "");
Expect(1, 66639, '\p{Is_Block= -Deseret}', "");
Expect(0, 66639, '\p{^Is_Block= -Deseret}', "");
Expect(0, 66639, '\P{Is_Block= -Deseret}', "");
Expect(1, 66639, '\P{^Is_Block= -Deseret}', "");
Expect(0, 66559, '\p{Is_Block= -Deseret}', "");
Expect(1, 66559, '\p{^Is_Block= -Deseret}', "");
Expect(1, 66559, '\P{Is_Block= -Deseret}', "");
Expect(0, 66559, '\P{^Is_Block= -Deseret}', "");
Error('\p{Is_Blk=/a/_-DESERET}');
Error('\P{Is_Blk=/a/_-DESERET}');
Expect(1, 66639, '\p{Is_Blk=deseret}', "");
Expect(0, 66639, '\p{^Is_Blk=deseret}', "");
Expect(0, 66639, '\P{Is_Blk=deseret}', "");
Expect(1, 66639, '\P{^Is_Blk=deseret}', "");
Expect(0, 66559, '\p{Is_Blk=deseret}', "");
Expect(1, 66559, '\p{^Is_Blk=deseret}', "");
Expect(1, 66559, '\P{Is_Blk=deseret}', "");
Expect(0, 66559, '\P{^Is_Blk=deseret}', "");
Expect(1, 66639, '\p{Is_Blk=	_Deseret}', "");
Expect(0, 66639, '\p{^Is_Blk=	_Deseret}', "");
Expect(0, 66639, '\P{Is_Blk=	_Deseret}', "");
Expect(1, 66639, '\P{^Is_Blk=	_Deseret}', "");
Expect(0, 66559, '\p{Is_Blk=	_Deseret}', "");
Expect(1, 66559, '\p{^Is_Blk=	_Deseret}', "");
Expect(1, 66559, '\P{Is_Blk=	_Deseret}', "");
Expect(0, 66559, '\P{^Is_Blk=	_Deseret}', "");
Error('\p{Block= _Tai:=Tham}');
Error('\P{Block= _Tai:=Tham}');
Expect(1, 6831, '\p{Block=taitham}', "");
Expect(0, 6831, '\p{^Block=taitham}', "");
Expect(0, 6831, '\P{Block=taitham}', "");
Expect(1, 6831, '\P{^Block=taitham}', "");
Expect(0, 6687, '\p{Block=taitham}', "");
Expect(1, 6687, '\p{^Block=taitham}', "");
Expect(1, 6687, '\P{Block=taitham}', "");
Expect(0, 6687, '\P{^Block=taitham}', "");
Expect(1, 6831, '\p{Block=__Taitham}', "");
Expect(0, 6831, '\p{^Block=__Taitham}', "");
Expect(0, 6831, '\P{Block=__Taitham}', "");
Expect(1, 6831, '\P{^Block=__Taitham}', "");
Expect(0, 6687, '\p{Block=__Taitham}', "");
Expect(1, 6687, '\p{^Block=__Taitham}', "");
Expect(1, 6687, '\P{Block=__Taitham}', "");
Expect(0, 6687, '\P{^Block=__Taitham}', "");
Error('\p{Blk=_TAI:=tham}');
Error('\P{Blk=_TAI:=tham}');
Expect(1, 6831, '\p{Blk=taitham}', "");
Expect(0, 6831, '\p{^Blk=taitham}', "");
Expect(0, 6831, '\P{Blk=taitham}', "");
Expect(1, 6831, '\P{^Blk=taitham}', "");
Expect(0, 6687, '\p{Blk=taitham}', "");
Expect(1, 6687, '\p{^Blk=taitham}', "");
Expect(1, 6687, '\P{Blk=taitham}', "");
Expect(0, 6687, '\P{^Blk=taitham}', "");
Expect(1, 6831, '\p{Blk:   		TAItham}', "");
Expect(0, 6831, '\p{^Blk:   		TAItham}', "");
Expect(0, 6831, '\P{Blk:   		TAItham}', "");
Expect(1, 6831, '\P{^Blk:   		TAItham}', "");
Expect(0, 6687, '\p{Blk:   		TAItham}', "");
Expect(1, 6687, '\p{^Blk:   		TAItham}', "");
Expect(1, 6687, '\P{Blk:   		TAItham}', "");
Expect(0, 6687, '\P{^Blk:   		TAItham}', "");
Error('\p{Is_Block=_-Tai/a/Tham}');
Error('\P{Is_Block=_-Tai/a/Tham}');
Expect(1, 6831, '\p{Is_Block=taitham}', "");
Expect(0, 6831, '\p{^Is_Block=taitham}', "");
Expect(0, 6831, '\P{Is_Block=taitham}', "");
Expect(1, 6831, '\P{^Is_Block=taitham}', "");
Expect(0, 6687, '\p{Is_Block=taitham}', "");
Expect(1, 6687, '\p{^Is_Block=taitham}', "");
Expect(1, 6687, '\P{Is_Block=taitham}', "");
Expect(0, 6687, '\P{^Is_Block=taitham}', "");
Expect(1, 6831, '\p{Is_Block=TAI Tham}', "");
Expect(0, 6831, '\p{^Is_Block=TAI Tham}', "");
Expect(0, 6831, '\P{Is_Block=TAI Tham}', "");
Expect(1, 6831, '\P{^Is_Block=TAI Tham}', "");
Expect(0, 6687, '\p{Is_Block=TAI Tham}', "");
Expect(1, 6687, '\p{^Is_Block=TAI Tham}', "");
Expect(1, 6687, '\P{Is_Block=TAI Tham}', "");
Expect(0, 6687, '\P{^Is_Block=TAI Tham}', "");
Error('\p{Is_Blk=	TAI	Tham:=}');
Error('\P{Is_Blk=	TAI	Tham:=}');
Expect(1, 6831, '\p{Is_Blk=taitham}', "");
Expect(0, 6831, '\p{^Is_Blk=taitham}', "");
Expect(0, 6831, '\P{Is_Blk=taitham}', "");
Expect(1, 6831, '\P{^Is_Blk=taitham}', "");
Expect(0, 6687, '\p{Is_Blk=taitham}', "");
Expect(1, 6687, '\p{^Is_Blk=taitham}', "");
Expect(1, 6687, '\P{Is_Blk=taitham}', "");
Expect(0, 6687, '\P{^Is_Blk=taitham}', "");
Expect(1, 6831, '\p{Is_Blk=-	tai_Tham}', "");
Expect(0, 6831, '\p{^Is_Blk=-	tai_Tham}', "");
Expect(0, 6831, '\P{Is_Blk=-	tai_Tham}', "");
Expect(1, 6831, '\P{^Is_Blk=-	tai_Tham}', "");
Expect(0, 6687, '\p{Is_Blk=-	tai_Tham}', "");
Expect(1, 6687, '\p{^Is_Blk=-	tai_Tham}', "");
Expect(1, 6687, '\P{Is_Blk=-	tai_Tham}', "");
Expect(0, 6687, '\P{^Is_Blk=-	tai_Tham}', "");
Error('\p{Block=	-tagbanwa:=}');
Error('\P{Block=	-tagbanwa:=}');
Expect(1, 6015, '\p{Block=tagbanwa}', "");
Expect(0, 6015, '\p{^Block=tagbanwa}', "");
Expect(0, 6015, '\P{Block=tagbanwa}', "");
Expect(1, 6015, '\P{^Block=tagbanwa}', "");
Expect(0, 5983, '\p{Block=tagbanwa}', "");
Expect(1, 5983, '\p{^Block=tagbanwa}', "");
Expect(1, 5983, '\P{Block=tagbanwa}', "");
Expect(0, 5983, '\P{^Block=tagbanwa}', "");
Expect(1, 6015, '\p{Block=_ Tagbanwa}', "");
Expect(0, 6015, '\p{^Block=_ Tagbanwa}', "");
Expect(0, 6015, '\P{Block=_ Tagbanwa}', "");
Expect(1, 6015, '\P{^Block=_ Tagbanwa}', "");
Expect(0, 5983, '\p{Block=_ Tagbanwa}', "");
Expect(1, 5983, '\p{^Block=_ Tagbanwa}', "");
Expect(1, 5983, '\P{Block=_ Tagbanwa}', "");
Expect(0, 5983, '\P{^Block=_ Tagbanwa}', "");
Error('\p{Blk=	-Tagbanwa:=}');
Error('\P{Blk=	-Tagbanwa:=}');
Expect(1, 6015, '\p{Blk=tagbanwa}', "");
Expect(0, 6015, '\p{^Blk=tagbanwa}', "");
Expect(0, 6015, '\P{Blk=tagbanwa}', "");
Expect(1, 6015, '\P{^Blk=tagbanwa}', "");
Expect(0, 5983, '\p{Blk=tagbanwa}', "");
Expect(1, 5983, '\p{^Blk=tagbanwa}', "");
Expect(1, 5983, '\P{Blk=tagbanwa}', "");
Expect(0, 5983, '\P{^Blk=tagbanwa}', "");
Expect(1, 6015, '\p{Blk=		Tagbanwa}', "");
Expect(0, 6015, '\p{^Blk=		Tagbanwa}', "");
Expect(0, 6015, '\P{Blk=		Tagbanwa}', "");
Expect(1, 6015, '\P{^Blk=		Tagbanwa}', "");
Expect(0, 5983, '\p{Blk=		Tagbanwa}', "");
Expect(1, 5983, '\p{^Blk=		Tagbanwa}', "");
Expect(1, 5983, '\P{Blk=		Tagbanwa}', "");
Expect(0, 5983, '\P{^Blk=		Tagbanwa}', "");
Error('\p{Is_Block=	/a/Tagbanwa}');
Error('\P{Is_Block=	/a/Tagbanwa}');
Expect(1, 6015, '\p{Is_Block=tagbanwa}', "");
Expect(0, 6015, '\p{^Is_Block=tagbanwa}', "");
Expect(0, 6015, '\P{Is_Block=tagbanwa}', "");
Expect(1, 6015, '\P{^Is_Block=tagbanwa}', "");
Expect(0, 5983, '\p{Is_Block=tagbanwa}', "");
Expect(1, 5983, '\p{^Is_Block=tagbanwa}', "");
Expect(1, 5983, '\P{Is_Block=tagbanwa}', "");
Expect(0, 5983, '\P{^Is_Block=tagbanwa}', "");
Expect(1, 6015, '\p{Is_Block=__Tagbanwa}', "");
Expect(0, 6015, '\p{^Is_Block=__Tagbanwa}', "");
Expect(0, 6015, '\P{Is_Block=__Tagbanwa}', "");
Expect(1, 6015, '\P{^Is_Block=__Tagbanwa}', "");
Expect(0, 5983, '\p{Is_Block=__Tagbanwa}', "");
Expect(1, 5983, '\p{^Is_Block=__Tagbanwa}', "");
Expect(1, 5983, '\P{Is_Block=__Tagbanwa}', "");
Expect(0, 5983, '\P{^Is_Block=__Tagbanwa}', "");
Error('\p{Is_Blk= tagbanwa:=}');
Error('\P{Is_Blk= tagbanwa:=}');
Expect(1, 6015, '\p{Is_Blk=tagbanwa}', "");
Expect(0, 6015, '\p{^Is_Blk=tagbanwa}', "");
Expect(0, 6015, '\P{Is_Blk=tagbanwa}', "");
Expect(1, 6015, '\P{^Is_Blk=tagbanwa}', "");
Expect(0, 5983, '\p{Is_Blk=tagbanwa}', "");
Expect(1, 5983, '\p{^Is_Blk=tagbanwa}', "");
Expect(1, 5983, '\P{Is_Blk=tagbanwa}', "");
Expect(0, 5983, '\P{^Is_Blk=tagbanwa}', "");
Expect(1, 6015, '\p{Is_Blk=_TAGBANWA}', "");
Expect(0, 6015, '\p{^Is_Blk=_TAGBANWA}', "");
Expect(0, 6015, '\P{Is_Blk=_TAGBANWA}', "");
Expect(1, 6015, '\P{^Is_Blk=_TAGBANWA}', "");
Expect(0, 5983, '\p{Is_Blk=_TAGBANWA}', "");
Expect(1, 5983, '\p{^Is_Blk=_TAGBANWA}', "");
Expect(1, 5983, '\P{Is_Blk=_TAGBANWA}', "");
Expect(0, 5983, '\P{^Is_Blk=_TAGBANWA}', "");
Error('\p{Block=:=	Arabic}');
Error('\P{Block=:=	Arabic}');
Expect(1, 1791, '\p{Block=arabic}', "");
Expect(0, 1791, '\p{^Block=arabic}', "");
Expect(0, 1791, '\P{Block=arabic}', "");
Expect(1, 1791, '\P{^Block=arabic}', "");
Expect(0, 1535, '\p{Block=arabic}', "");
Expect(1, 1535, '\p{^Block=arabic}', "");
Expect(1, 1535, '\P{Block=arabic}', "");
Expect(0, 1535, '\P{^Block=arabic}', "");
Expect(1, 1791, '\p{Block=-_ARABIC}', "");
Expect(0, 1791, '\p{^Block=-_ARABIC}', "");
Expect(0, 1791, '\P{Block=-_ARABIC}', "");
Expect(1, 1791, '\P{^Block=-_ARABIC}', "");
Expect(0, 1535, '\p{Block=-_ARABIC}', "");
Expect(1, 1535, '\p{^Block=-_ARABIC}', "");
Expect(1, 1535, '\P{Block=-_ARABIC}', "");
Expect(0, 1535, '\P{^Block=-_ARABIC}', "");
Error('\p{Blk= _ARABIC:=}');
Error('\P{Blk= _ARABIC:=}');
Expect(1, 1791, '\p{Blk:   arabic}', "");
Expect(0, 1791, '\p{^Blk:   arabic}', "");
Expect(0, 1791, '\P{Blk:   arabic}', "");
Expect(1, 1791, '\P{^Blk:   arabic}', "");
Expect(0, 1535, '\p{Blk:   arabic}', "");
Expect(1, 1535, '\p{^Blk:   arabic}', "");
Expect(1, 1535, '\P{Blk:   arabic}', "");
Expect(0, 1535, '\P{^Blk:   arabic}', "");
Expect(1, 1791, '\p{Blk=_	ARABIC}', "");
Expect(0, 1791, '\p{^Blk=_	ARABIC}', "");
Expect(0, 1791, '\P{Blk=_	ARABIC}', "");
Expect(1, 1791, '\P{^Blk=_	ARABIC}', "");
Expect(0, 1535, '\p{Blk=_	ARABIC}', "");
Expect(1, 1535, '\p{^Blk=_	ARABIC}', "");
Expect(1, 1535, '\P{Blk=_	ARABIC}', "");
Expect(0, 1535, '\P{^Blk=_	ARABIC}', "");
Error('\p{Is_Block:   /a/-Arabic}');
Error('\P{Is_Block:   /a/-Arabic}');
Expect(1, 1791, '\p{Is_Block=arabic}', "");
Expect(0, 1791, '\p{^Is_Block=arabic}', "");
Expect(0, 1791, '\P{Is_Block=arabic}', "");
Expect(1, 1791, '\P{^Is_Block=arabic}', "");
Expect(0, 1535, '\p{Is_Block=arabic}', "");
Expect(1, 1535, '\p{^Is_Block=arabic}', "");
Expect(1, 1535, '\P{Is_Block=arabic}', "");
Expect(0, 1535, '\P{^Is_Block=arabic}', "");
Expect(1, 1791, '\p{Is_Block= 	Arabic}', "");
Expect(0, 1791, '\p{^Is_Block= 	Arabic}', "");
Expect(0, 1791, '\P{Is_Block= 	Arabic}', "");
Expect(1, 1791, '\P{^Is_Block= 	Arabic}', "");
Expect(0, 1535, '\p{Is_Block= 	Arabic}', "");
Expect(1, 1535, '\p{^Is_Block= 	Arabic}', "");
Expect(1, 1535, '\P{Is_Block= 	Arabic}', "");
Expect(0, 1535, '\P{^Is_Block= 	Arabic}', "");
Error('\p{Is_Blk=/a/		arabic}');
Error('\P{Is_Blk=/a/		arabic}');
Expect(1, 1791, '\p{Is_Blk=arabic}', "");
Expect(0, 1791, '\p{^Is_Blk=arabic}', "");
Expect(0, 1791, '\P{Is_Blk=arabic}', "");
Expect(1, 1791, '\P{^Is_Blk=arabic}', "");
Expect(0, 1535, '\p{Is_Blk=arabic}', "");
Expect(1, 1535, '\p{^Is_Blk=arabic}', "");
Expect(1, 1535, '\P{Is_Blk=arabic}', "");
Expect(0, 1535, '\P{^Is_Blk=arabic}', "");
Expect(1, 1791, '\p{Is_Blk=_-arabic}', "");
Expect(0, 1791, '\p{^Is_Blk=_-arabic}', "");
Expect(0, 1791, '\P{Is_Blk=_-arabic}', "");
Expect(1, 1791, '\P{^Is_Blk=_-arabic}', "");
Expect(0, 1535, '\p{Is_Blk=_-arabic}', "");
Expect(1, 1535, '\p{^Is_Blk=_-arabic}', "");
Expect(1, 1535, '\P{Is_Blk=_-arabic}', "");
Expect(0, 1535, '\P{^Is_Blk=_-arabic}', "");
Error('\p{Block=-_KATAKANA-Phonetic Extensions:=}');
Error('\P{Block=-_KATAKANA-Phonetic Extensions:=}');
Expect(1, 12799, '\p{Block=katakanaphoneticextensions}', "");
Expect(0, 12799, '\p{^Block=katakanaphoneticextensions}', "");
Expect(0, 12799, '\P{Block=katakanaphoneticextensions}', "");
Expect(1, 12799, '\P{^Block=katakanaphoneticextensions}', "");
Expect(0, 12783, '\p{Block=katakanaphoneticextensions}', "");
Expect(1, 12783, '\p{^Block=katakanaphoneticextensions}', "");
Expect(1, 12783, '\P{Block=katakanaphoneticextensions}', "");
Expect(0, 12783, '\P{^Block=katakanaphoneticextensions}', "");
Expect(1, 12799, '\p{Block=	_Katakana_PhoneticExtensions}', "");
Expect(0, 12799, '\p{^Block=	_Katakana_PhoneticExtensions}', "");
Expect(0, 12799, '\P{Block=	_Katakana_PhoneticExtensions}', "");
Expect(1, 12799, '\P{^Block=	_Katakana_PhoneticExtensions}', "");
Expect(0, 12783, '\p{Block=	_Katakana_PhoneticExtensions}', "");
Expect(1, 12783, '\p{^Block=	_Katakana_PhoneticExtensions}', "");
Expect(1, 12783, '\P{Block=	_Katakana_PhoneticExtensions}', "");
Expect(0, 12783, '\P{^Block=	_Katakana_PhoneticExtensions}', "");
Error('\p{Blk=-katakana	phonetic/a/extensions}');
Error('\P{Blk=-katakana	phonetic/a/extensions}');
Expect(1, 12799, '\p{Blk:	katakanaphoneticextensions}', "");
Expect(0, 12799, '\p{^Blk:	katakanaphoneticextensions}', "");
Expect(0, 12799, '\P{Blk:	katakanaphoneticextensions}', "");
Expect(1, 12799, '\P{^Blk:	katakanaphoneticextensions}', "");
Expect(0, 12783, '\p{Blk:	katakanaphoneticextensions}', "");
Expect(1, 12783, '\p{^Blk:	katakanaphoneticextensions}', "");
Expect(1, 12783, '\P{Blk:	katakanaphoneticextensions}', "");
Expect(0, 12783, '\P{^Blk:	katakanaphoneticextensions}', "");
Expect(1, 12799, '\p{Blk=	Katakana_PhoneticEXTENSIONS}', "");
Expect(0, 12799, '\p{^Blk=	Katakana_PhoneticEXTENSIONS}', "");
Expect(0, 12799, '\P{Blk=	Katakana_PhoneticEXTENSIONS}', "");
Expect(1, 12799, '\P{^Blk=	Katakana_PhoneticEXTENSIONS}', "");
Expect(0, 12783, '\p{Blk=	Katakana_PhoneticEXTENSIONS}', "");
Expect(1, 12783, '\p{^Blk=	Katakana_PhoneticEXTENSIONS}', "");
Expect(1, 12783, '\P{Blk=	Katakana_PhoneticEXTENSIONS}', "");
Expect(0, 12783, '\P{^Blk=	Katakana_PhoneticEXTENSIONS}', "");
Error('\p{Is_Block=-:=KatakanaPHONETIC_EXTENSIONS}');
Error('\P{Is_Block=-:=KatakanaPHONETIC_EXTENSIONS}');
Expect(1, 12799, '\p{Is_Block=katakanaphoneticextensions}', "");
Expect(0, 12799, '\p{^Is_Block=katakanaphoneticextensions}', "");
Expect(0, 12799, '\P{Is_Block=katakanaphoneticextensions}', "");
Expect(1, 12799, '\P{^Is_Block=katakanaphoneticextensions}', "");
Expect(0, 12783, '\p{Is_Block=katakanaphoneticextensions}', "");
Expect(1, 12783, '\p{^Is_Block=katakanaphoneticextensions}', "");
Expect(1, 12783, '\P{Is_Block=katakanaphoneticextensions}', "");
Expect(0, 12783, '\P{^Is_Block=katakanaphoneticextensions}', "");
Expect(1, 12799, '\p{Is_Block=_ Katakana phonetic-Extensions}', "");
Expect(0, 12799, '\p{^Is_Block=_ Katakana phonetic-Extensions}', "");
Expect(0, 12799, '\P{Is_Block=_ Katakana phonetic-Extensions}', "");
Expect(1, 12799, '\P{^Is_Block=_ Katakana phonetic-Extensions}', "");
Expect(0, 12783, '\p{Is_Block=_ Katakana phonetic-Extensions}', "");
Expect(1, 12783, '\p{^Is_Block=_ Katakana phonetic-Extensions}', "");
Expect(1, 12783, '\P{Is_Block=_ Katakana phonetic-Extensions}', "");
Expect(0, 12783, '\P{^Is_Block=_ Katakana phonetic-Extensions}', "");
Error('\p{Is_Blk=:=-	KatakanaPhoneticExtensions}');
Error('\P{Is_Blk=:=-	KatakanaPhoneticExtensions}');
Expect(1, 12799, '\p{Is_Blk=katakanaphoneticextensions}', "");
Expect(0, 12799, '\p{^Is_Blk=katakanaphoneticextensions}', "");
Expect(0, 12799, '\P{Is_Blk=katakanaphoneticextensions}', "");
Expect(1, 12799, '\P{^Is_Blk=katakanaphoneticextensions}', "");
Expect(0, 12783, '\p{Is_Blk=katakanaphoneticextensions}', "");
Expect(1, 12783, '\p{^Is_Blk=katakanaphoneticextensions}', "");
Expect(1, 12783, '\P{Is_Blk=katakanaphoneticextensions}', "");
Expect(0, 12783, '\P{^Is_Blk=katakanaphoneticextensions}', "");
Expect(1, 12799, '\p{Is_Blk= 	KATAKANA_Phonetic Extensions}', "");
Expect(0, 12799, '\p{^Is_Blk= 	KATAKANA_Phonetic Extensions}', "");
Expect(0, 12799, '\P{Is_Blk= 	KATAKANA_Phonetic Extensions}', "");
Expect(1, 12799, '\P{^Is_Blk= 	KATAKANA_Phonetic Extensions}', "");
Expect(0, 12783, '\p{Is_Blk= 	KATAKANA_Phonetic Extensions}', "");
Expect(1, 12783, '\p{^Is_Blk= 	KATAKANA_Phonetic Extensions}', "");
Expect(1, 12783, '\P{Is_Blk= 	KATAKANA_Phonetic Extensions}', "");
Expect(0, 12783, '\P{^Is_Blk= 	KATAKANA_Phonetic Extensions}', "");
Error('\p{Block=/a/ -Letterlike_SYMBOLS}');
Error('\P{Block=/a/ -Letterlike_SYMBOLS}');
Expect(1, 8527, '\p{Block=letterlikesymbols}', "");
Expect(0, 8527, '\p{^Block=letterlikesymbols}', "");
Expect(0, 8527, '\P{Block=letterlikesymbols}', "");
Expect(1, 8527, '\P{^Block=letterlikesymbols}', "");
Expect(0, 8447, '\p{Block=letterlikesymbols}', "");
Expect(1, 8447, '\p{^Block=letterlikesymbols}', "");
Expect(1, 8447, '\P{Block=letterlikesymbols}', "");
Expect(0, 8447, '\P{^Block=letterlikesymbols}', "");
Expect(1, 8527, '\p{Block=--LetterlikeSYMBOLS}', "");
Expect(0, 8527, '\p{^Block=--LetterlikeSYMBOLS}', "");
Expect(0, 8527, '\P{Block=--LetterlikeSYMBOLS}', "");
Expect(1, 8527, '\P{^Block=--LetterlikeSYMBOLS}', "");
Expect(0, 8447, '\p{Block=--LetterlikeSYMBOLS}', "");
Expect(1, 8447, '\p{^Block=--LetterlikeSYMBOLS}', "");
Expect(1, 8447, '\P{Block=--LetterlikeSYMBOLS}', "");
Expect(0, 8447, '\P{^Block=--LetterlikeSYMBOLS}', "");
Error('\p{Blk=	letterlike	Symbols/a/}');
Error('\P{Blk=	letterlike	Symbols/a/}');
Expect(1, 8527, '\p{Blk: letterlikesymbols}', "");
Expect(0, 8527, '\p{^Blk: letterlikesymbols}', "");
Expect(0, 8527, '\P{Blk: letterlikesymbols}', "");
Expect(1, 8527, '\P{^Blk: letterlikesymbols}', "");
Expect(0, 8447, '\p{Blk: letterlikesymbols}', "");
Expect(1, 8447, '\p{^Blk: letterlikesymbols}', "");
Expect(1, 8447, '\P{Blk: letterlikesymbols}', "");
Expect(0, 8447, '\P{^Blk: letterlikesymbols}', "");
Expect(1, 8527, '\p{Blk= Letterlike-SYMBOLS}', "");
Expect(0, 8527, '\p{^Blk= Letterlike-SYMBOLS}', "");
Expect(0, 8527, '\P{Blk= Letterlike-SYMBOLS}', "");
Expect(1, 8527, '\P{^Blk= Letterlike-SYMBOLS}', "");
Expect(0, 8447, '\p{Blk= Letterlike-SYMBOLS}', "");
Expect(1, 8447, '\p{^Blk= Letterlike-SYMBOLS}', "");
Expect(1, 8447, '\P{Blk= Letterlike-SYMBOLS}', "");
Expect(0, 8447, '\P{^Blk= Letterlike-SYMBOLS}', "");
Error('\p{Is_Block=_ LETTERLIKE/a/Symbols}');
Error('\P{Is_Block=_ LETTERLIKE/a/Symbols}');
Expect(1, 8527, '\p{Is_Block=letterlikesymbols}', "");
Expect(0, 8527, '\p{^Is_Block=letterlikesymbols}', "");
Expect(0, 8527, '\P{Is_Block=letterlikesymbols}', "");
Expect(1, 8527, '\P{^Is_Block=letterlikesymbols}', "");
Expect(0, 8447, '\p{Is_Block=letterlikesymbols}', "");
Expect(1, 8447, '\p{^Is_Block=letterlikesymbols}', "");
Expect(1, 8447, '\P{Is_Block=letterlikesymbols}', "");
Expect(0, 8447, '\P{^Is_Block=letterlikesymbols}', "");
Expect(1, 8527, '\p{Is_Block=_ Letterlike Symbols}', "");
Expect(0, 8527, '\p{^Is_Block=_ Letterlike Symbols}', "");
Expect(0, 8527, '\P{Is_Block=_ Letterlike Symbols}', "");
Expect(1, 8527, '\P{^Is_Block=_ Letterlike Symbols}', "");
Expect(0, 8447, '\p{Is_Block=_ Letterlike Symbols}', "");
Expect(1, 8447, '\p{^Is_Block=_ Letterlike Symbols}', "");
Expect(1, 8447, '\P{Is_Block=_ Letterlike Symbols}', "");
Expect(0, 8447, '\P{^Is_Block=_ Letterlike Symbols}', "");
Error('\p{Is_Blk=-:=Letterlike-symbols}');
Error('\P{Is_Blk=-:=Letterlike-symbols}');
Expect(1, 8527, '\p{Is_Blk=letterlikesymbols}', "");
Expect(0, 8527, '\p{^Is_Blk=letterlikesymbols}', "");
Expect(0, 8527, '\P{Is_Blk=letterlikesymbols}', "");
Expect(1, 8527, '\P{^Is_Blk=letterlikesymbols}', "");
Expect(0, 8447, '\p{Is_Blk=letterlikesymbols}', "");
Expect(1, 8447, '\p{^Is_Blk=letterlikesymbols}', "");
Expect(1, 8447, '\P{Is_Blk=letterlikesymbols}', "");
Expect(0, 8447, '\P{^Is_Blk=letterlikesymbols}', "");
Expect(1, 8527, '\p{Is_Blk=__Letterlike Symbols}', "");
Expect(0, 8527, '\p{^Is_Blk=__Letterlike Symbols}', "");
Expect(0, 8527, '\P{Is_Blk=__Letterlike Symbols}', "");
Expect(1, 8527, '\P{^Is_Blk=__Letterlike Symbols}', "");
Expect(0, 8447, '\p{Is_Blk=__Letterlike Symbols}', "");
Expect(1, 8447, '\p{^Is_Blk=__Letterlike Symbols}', "");
Expect(1, 8447, '\P{Is_Blk=__Letterlike Symbols}', "");
Expect(0, 8447, '\P{^Is_Blk=__Letterlike Symbols}', "");
Error('\p{Block: /a/LATINextended_C}');
Error('\P{Block: /a/LATINextended_C}');
Expect(1, 11391, '\p{Block=latinextendedc}', "");
Expect(0, 11391, '\p{^Block=latinextendedc}', "");
Expect(0, 11391, '\P{Block=latinextendedc}', "");
Expect(1, 11391, '\P{^Block=latinextendedc}', "");
Expect(0, 11359, '\p{Block=latinextendedc}', "");
Expect(1, 11359, '\p{^Block=latinextendedc}', "");
Expect(1, 11359, '\P{Block=latinextendedc}', "");
Expect(0, 11359, '\P{^Block=latinextendedc}', "");
Expect(1, 11391, '\p{Block=_Latin-EXTENDED C}', "");
Expect(0, 11391, '\p{^Block=_Latin-EXTENDED C}', "");
Expect(0, 11391, '\P{Block=_Latin-EXTENDED C}', "");
Expect(1, 11391, '\P{^Block=_Latin-EXTENDED C}', "");
Expect(0, 11359, '\p{Block=_Latin-EXTENDED C}', "");
Expect(1, 11359, '\p{^Block=_Latin-EXTENDED C}', "");
Expect(1, 11359, '\P{Block=_Latin-EXTENDED C}', "");
Expect(0, 11359, '\P{^Block=_Latin-EXTENDED C}', "");
Error('\p{Blk=:=__Latin	EXTENDED_C}');
Error('\P{Blk=:=__Latin	EXTENDED_C}');
Expect(1, 11391, '\p{Blk=latinextendedc}', "");
Expect(0, 11391, '\p{^Blk=latinextendedc}', "");
Expect(0, 11391, '\P{Blk=latinextendedc}', "");
Expect(1, 11391, '\P{^Blk=latinextendedc}', "");
Expect(0, 11359, '\p{Blk=latinextendedc}', "");
Expect(1, 11359, '\p{^Blk=latinextendedc}', "");
Expect(1, 11359, '\P{Blk=latinextendedc}', "");
Expect(0, 11359, '\P{^Blk=latinextendedc}', "");
Expect(1, 11391, '\p{Blk= latin-EXTENDED_C}', "");
Expect(0, 11391, '\p{^Blk= latin-EXTENDED_C}', "");
Expect(0, 11391, '\P{Blk= latin-EXTENDED_C}', "");
Expect(1, 11391, '\P{^Blk= latin-EXTENDED_C}', "");
Expect(0, 11359, '\p{Blk= latin-EXTENDED_C}', "");
Expect(1, 11359, '\p{^Blk= latin-EXTENDED_C}', "");
Expect(1, 11359, '\P{Blk= latin-EXTENDED_C}', "");
Expect(0, 11359, '\P{^Blk= latin-EXTENDED_C}', "");
Error('\p{Is_Block=- latin-Extended:=c}');
Error('\P{Is_Block=- latin-Extended:=c}');
Expect(1, 11391, '\p{Is_Block=latinextendedc}', "");
Expect(0, 11391, '\p{^Is_Block=latinextendedc}', "");
Expect(0, 11391, '\P{Is_Block=latinextendedc}', "");
Expect(1, 11391, '\P{^Is_Block=latinextendedc}', "");
Expect(0, 11359, '\p{Is_Block=latinextendedc}', "");
Expect(1, 11359, '\p{^Is_Block=latinextendedc}', "");
Expect(1, 11359, '\P{Is_Block=latinextendedc}', "");
Expect(0, 11359, '\P{^Is_Block=latinextendedc}', "");
Expect(1, 11391, '\p{Is_Block=		LatinExtendedC}', "");
Expect(0, 11391, '\p{^Is_Block=		LatinExtendedC}', "");
Expect(0, 11391, '\P{Is_Block=		LatinExtendedC}', "");
Expect(1, 11391, '\P{^Is_Block=		LatinExtendedC}', "");
Expect(0, 11359, '\p{Is_Block=		LatinExtendedC}', "");
Expect(1, 11359, '\p{^Is_Block=		LatinExtendedC}', "");
Expect(1, 11359, '\P{Is_Block=		LatinExtendedC}', "");
Expect(0, 11359, '\P{^Is_Block=		LatinExtendedC}', "");
Error('\p{Is_Blk:		 LATIN-Extended	C:=}');
Error('\P{Is_Blk:		 LATIN-Extended	C:=}');
Expect(1, 11391, '\p{Is_Blk=latinextendedc}', "");
Expect(0, 11391, '\p{^Is_Blk=latinextendedc}', "");
Expect(0, 11391, '\P{Is_Blk=latinextendedc}', "");
Expect(1, 11391, '\P{^Is_Blk=latinextendedc}', "");
Expect(0, 11359, '\p{Is_Blk=latinextendedc}', "");
Expect(1, 11359, '\p{^Is_Blk=latinextendedc}', "");
Expect(1, 11359, '\P{Is_Blk=latinextendedc}', "");
Expect(0, 11359, '\P{^Is_Blk=latinextendedc}', "");
Expect(1, 11391, '\p{Is_Blk=- latin	extendedc}', "");
Expect(0, 11391, '\p{^Is_Blk=- latin	extendedc}', "");
Expect(0, 11391, '\P{Is_Blk=- latin	extendedc}', "");
Expect(1, 11391, '\P{^Is_Blk=- latin	extendedc}', "");
Expect(0, 11359, '\p{Is_Blk=- latin	extendedc}', "");
Expect(1, 11359, '\p{^Is_Blk=- latin	extendedc}', "");
Expect(1, 11359, '\P{Is_Blk=- latin	extendedc}', "");
Expect(0, 11359, '\P{^Is_Blk=- latin	extendedc}', "");
Error('\p{Block=/a/--Common_indic	NumberForms}');
Error('\P{Block=/a/--Common_indic	NumberForms}');
Expect(1, 43071, '\p{Block=commonindicnumberforms}', "");
Expect(0, 43071, '\p{^Block=commonindicnumberforms}', "");
Expect(0, 43071, '\P{Block=commonindicnumberforms}', "");
Expect(1, 43071, '\P{^Block=commonindicnumberforms}', "");
Expect(0, 43055, '\p{Block=commonindicnumberforms}', "");
Expect(1, 43055, '\p{^Block=commonindicnumberforms}', "");
Expect(1, 43055, '\P{Block=commonindicnumberforms}', "");
Expect(0, 43055, '\P{^Block=commonindicnumberforms}', "");
Expect(1, 43071, '\p{Block=_common Indicnumber forms}', "");
Expect(0, 43071, '\p{^Block=_common Indicnumber forms}', "");
Expect(0, 43071, '\P{Block=_common Indicnumber forms}', "");
Expect(1, 43071, '\P{^Block=_common Indicnumber forms}', "");
Expect(0, 43055, '\p{Block=_common Indicnumber forms}', "");
Expect(1, 43055, '\p{^Block=_common Indicnumber forms}', "");
Expect(1, 43055, '\P{Block=_common Indicnumber forms}', "");
Expect(0, 43055, '\P{^Block=_common Indicnumber forms}', "");
Error('\p{Blk= _Common	indic_Number/a/FORMS}');
Error('\P{Blk= _Common	indic_Number/a/FORMS}');
Expect(1, 43071, '\p{Blk=commonindicnumberforms}', "");
Expect(0, 43071, '\p{^Blk=commonindicnumberforms}', "");
Expect(0, 43071, '\P{Blk=commonindicnumberforms}', "");
Expect(1, 43071, '\P{^Blk=commonindicnumberforms}', "");
Expect(0, 43055, '\p{Blk=commonindicnumberforms}', "");
Expect(1, 43055, '\p{^Blk=commonindicnumberforms}', "");
Expect(1, 43055, '\P{Blk=commonindicnumberforms}', "");
Expect(0, 43055, '\P{^Blk=commonindicnumberforms}', "");
Expect(1, 43071, '\p{Blk: _-COMMON	Indic-number_FORMS}', "");
Expect(0, 43071, '\p{^Blk: _-COMMON	Indic-number_FORMS}', "");
Expect(0, 43071, '\P{Blk: _-COMMON	Indic-number_FORMS}', "");
Expect(1, 43071, '\P{^Blk: _-COMMON	Indic-number_FORMS}', "");
Expect(0, 43055, '\p{Blk: _-COMMON	Indic-number_FORMS}', "");
Expect(1, 43055, '\p{^Blk: _-COMMON	Indic-number_FORMS}', "");
Expect(1, 43055, '\P{Blk: _-COMMON	Indic-number_FORMS}', "");
Expect(0, 43055, '\P{^Blk: _-COMMON	Indic-number_FORMS}', "");
Error('\p{Is_Block=	_common INDIC/a/Number Forms}');
Error('\P{Is_Block=	_common INDIC/a/Number Forms}');
Expect(1, 43071, '\p{Is_Block=commonindicnumberforms}', "");
Expect(0, 43071, '\p{^Is_Block=commonindicnumberforms}', "");
Expect(0, 43071, '\P{Is_Block=commonindicnumberforms}', "");
Expect(1, 43071, '\P{^Is_Block=commonindicnumberforms}', "");
Expect(0, 43055, '\p{Is_Block=commonindicnumberforms}', "");
Expect(1, 43055, '\p{^Is_Block=commonindicnumberforms}', "");
Expect(1, 43055, '\P{Is_Block=commonindicnumberforms}', "");
Expect(0, 43055, '\P{^Is_Block=commonindicnumberforms}', "");
Expect(1, 43071, '\p{Is_Block:--Common-Indic Numberforms}', "");
Expect(0, 43071, '\p{^Is_Block:--Common-Indic Numberforms}', "");
Expect(0, 43071, '\P{Is_Block:--Common-Indic Numberforms}', "");
Expect(1, 43071, '\P{^Is_Block:--Common-Indic Numberforms}', "");
Expect(0, 43055, '\p{Is_Block:--Common-Indic Numberforms}', "");
Expect(1, 43055, '\p{^Is_Block:--Common-Indic Numberforms}', "");
Expect(1, 43055, '\P{Is_Block:--Common-Indic Numberforms}', "");
Expect(0, 43055, '\P{^Is_Block:--Common-Indic Numberforms}', "");
Error('\p{Is_Blk=	Common_Indic-Number forms/a/}');
Error('\P{Is_Blk=	Common_Indic-Number forms/a/}');
Expect(1, 43071, '\p{Is_Blk=commonindicnumberforms}', "");
Expect(0, 43071, '\p{^Is_Blk=commonindicnumberforms}', "");
Expect(0, 43071, '\P{Is_Blk=commonindicnumberforms}', "");
Expect(1, 43071, '\P{^Is_Blk=commonindicnumberforms}', "");
Expect(0, 43055, '\p{Is_Blk=commonindicnumberforms}', "");
Expect(1, 43055, '\p{^Is_Blk=commonindicnumberforms}', "");
Expect(1, 43055, '\P{Is_Blk=commonindicnumberforms}', "");
Expect(0, 43055, '\P{^Is_Blk=commonindicnumberforms}', "");
Expect(1, 43071, '\p{Is_Blk:  commonIndic Number	Forms}', "");
Expect(0, 43071, '\p{^Is_Blk:  commonIndic Number	Forms}', "");
Expect(0, 43071, '\P{Is_Blk:  commonIndic Number	Forms}', "");
Expect(1, 43071, '\P{^Is_Blk:  commonIndic Number	Forms}', "");
Expect(0, 43055, '\p{Is_Blk:  commonIndic Number	Forms}', "");
Expect(1, 43055, '\p{^Is_Blk:  commonIndic Number	Forms}', "");
Expect(1, 43055, '\P{Is_Blk:  commonIndic Number	Forms}', "");
Expect(0, 43055, '\P{^Is_Blk:  commonIndic Number	Forms}', "");
Error('\p{Block=:=_meetei	Mayek}');
Error('\P{Block=:=_meetei	Mayek}');
Expect(1, 44031, '\p{Block=meeteimayek}', "");
Expect(0, 44031, '\p{^Block=meeteimayek}', "");
Expect(0, 44031, '\P{Block=meeteimayek}', "");
Expect(1, 44031, '\P{^Block=meeteimayek}', "");
Expect(0, 43967, '\p{Block=meeteimayek}', "");
Expect(1, 43967, '\p{^Block=meeteimayek}', "");
Expect(1, 43967, '\P{Block=meeteimayek}', "");
Expect(0, 43967, '\P{^Block=meeteimayek}', "");
Expect(1, 44031, '\p{Block=_	meetei_Mayek}', "");
Expect(0, 44031, '\p{^Block=_	meetei_Mayek}', "");
Expect(0, 44031, '\P{Block=_	meetei_Mayek}', "");
Expect(1, 44031, '\P{^Block=_	meetei_Mayek}', "");
Expect(0, 43967, '\p{Block=_	meetei_Mayek}', "");
Expect(1, 43967, '\p{^Block=_	meetei_Mayek}', "");
Expect(1, 43967, '\P{Block=_	meetei_Mayek}', "");
Expect(0, 43967, '\P{^Block=_	meetei_Mayek}', "");
Error('\p{Blk=:=_meeteiMayek}');
Error('\P{Blk=:=_meeteiMayek}');
Expect(1, 44031, '\p{Blk: meeteimayek}', "");
Expect(0, 44031, '\p{^Blk: meeteimayek}', "");
Expect(0, 44031, '\P{Blk: meeteimayek}', "");
Expect(1, 44031, '\P{^Blk: meeteimayek}', "");
Expect(0, 43967, '\p{Blk: meeteimayek}', "");
Expect(1, 43967, '\p{^Blk: meeteimayek}', "");
Expect(1, 43967, '\P{Blk: meeteimayek}', "");
Expect(0, 43967, '\P{^Blk: meeteimayek}', "");
Expect(1, 44031, '\p{Blk=_Meetei MAYEK}', "");
Expect(0, 44031, '\p{^Blk=_Meetei MAYEK}', "");
Expect(0, 44031, '\P{Blk=_Meetei MAYEK}', "");
Expect(1, 44031, '\P{^Blk=_Meetei MAYEK}', "");
Expect(0, 43967, '\p{Blk=_Meetei MAYEK}', "");
Expect(1, 43967, '\p{^Blk=_Meetei MAYEK}', "");
Expect(1, 43967, '\P{Blk=_Meetei MAYEK}', "");
Expect(0, 43967, '\P{^Blk=_Meetei MAYEK}', "");
Error('\p{Is_Block=/a/	-Meetei	Mayek}');
Error('\P{Is_Block=/a/	-Meetei	Mayek}');
Expect(1, 44031, '\p{Is_Block=meeteimayek}', "");
Expect(0, 44031, '\p{^Is_Block=meeteimayek}', "");
Expect(0, 44031, '\P{Is_Block=meeteimayek}', "");
Expect(1, 44031, '\P{^Is_Block=meeteimayek}', "");
Expect(0, 43967, '\p{Is_Block=meeteimayek}', "");
Expect(1, 43967, '\p{^Is_Block=meeteimayek}', "");
Expect(1, 43967, '\P{Is_Block=meeteimayek}', "");
Expect(0, 43967, '\P{^Is_Block=meeteimayek}', "");
Expect(1, 44031, '\p{Is_Block:   --meetei-Mayek}', "");
Expect(0, 44031, '\p{^Is_Block:   --meetei-Mayek}', "");
Expect(0, 44031, '\P{Is_Block:   --meetei-Mayek}', "");
Expect(1, 44031, '\P{^Is_Block:   --meetei-Mayek}', "");
Expect(0, 43967, '\p{Is_Block:   --meetei-Mayek}', "");
Expect(1, 43967, '\p{^Is_Block:   --meetei-Mayek}', "");
Expect(1, 43967, '\P{Is_Block:   --meetei-Mayek}', "");
Expect(0, 43967, '\P{^Is_Block:   --meetei-Mayek}', "");
Error('\p{Is_Blk=Meetei/a/Mayek}');
Error('\P{Is_Blk=Meetei/a/Mayek}');
Expect(1, 44031, '\p{Is_Blk=meeteimayek}', "");
Expect(0, 44031, '\p{^Is_Blk=meeteimayek}', "");
Expect(0, 44031, '\P{Is_Blk=meeteimayek}', "");
Expect(1, 44031, '\P{^Is_Blk=meeteimayek}', "");
Expect(0, 43967, '\p{Is_Blk=meeteimayek}', "");
Expect(1, 43967, '\p{^Is_Blk=meeteimayek}', "");
Expect(1, 43967, '\P{Is_Blk=meeteimayek}', "");
Expect(0, 43967, '\P{^Is_Blk=meeteimayek}', "");
Expect(1, 44031, '\p{Is_Blk=	 MEETEIMayek}', "");
Expect(0, 44031, '\p{^Is_Blk=	 MEETEIMayek}', "");
Expect(0, 44031, '\P{Is_Blk=	 MEETEIMayek}', "");
Expect(1, 44031, '\P{^Is_Blk=	 MEETEIMayek}', "");
Expect(0, 43967, '\p{Is_Blk=	 MEETEIMayek}', "");
Expect(1, 43967, '\p{^Is_Blk=	 MEETEIMayek}', "");
Expect(1, 43967, '\P{Is_Blk=	 MEETEIMayek}', "");
Expect(0, 43967, '\P{^Is_Blk=	 MEETEIMayek}', "");
Error('\p{Block= /a/Supplemental	mathematical-operators}');
Error('\P{Block= /a/Supplemental	mathematical-operators}');
Expect(1, 11007, '\p{Block=supplementalmathematicaloperators}', "");
Expect(0, 11007, '\p{^Block=supplementalmathematicaloperators}', "");
Expect(0, 11007, '\P{Block=supplementalmathematicaloperators}', "");
Expect(1, 11007, '\P{^Block=supplementalmathematicaloperators}', "");
Expect(0, 10751, '\p{Block=supplementalmathematicaloperators}', "");
Expect(1, 10751, '\p{^Block=supplementalmathematicaloperators}', "");
Expect(1, 10751, '\P{Block=supplementalmathematicaloperators}', "");
Expect(0, 10751, '\P{^Block=supplementalmathematicaloperators}', "");
Expect(1, 11007, '\p{Block=	Supplemental MathematicalOperators}', "");
Expect(0, 11007, '\p{^Block=	Supplemental MathematicalOperators}', "");
Expect(0, 11007, '\P{Block=	Supplemental MathematicalOperators}', "");
Expect(1, 11007, '\P{^Block=	Supplemental MathematicalOperators}', "");
Expect(0, 10751, '\p{Block=	Supplemental MathematicalOperators}', "");
Expect(1, 10751, '\p{^Block=	Supplemental MathematicalOperators}', "");
Expect(1, 10751, '\P{Block=	Supplemental MathematicalOperators}', "");
Expect(0, 10751, '\P{^Block=	Supplemental MathematicalOperators}', "");
Error('\p{Blk=-_supplemental_Mathematical/a/Operators}');
Error('\P{Blk=-_supplemental_Mathematical/a/Operators}');
Expect(1, 11007, '\p{Blk=supplementalmathematicaloperators}', "");
Expect(0, 11007, '\p{^Blk=supplementalmathematicaloperators}', "");
Expect(0, 11007, '\P{Blk=supplementalmathematicaloperators}', "");
Expect(1, 11007, '\P{^Blk=supplementalmathematicaloperators}', "");
Expect(0, 10751, '\p{Blk=supplementalmathematicaloperators}', "");
Expect(1, 10751, '\p{^Blk=supplementalmathematicaloperators}', "");
Expect(1, 10751, '\P{Blk=supplementalmathematicaloperators}', "");
Expect(0, 10751, '\P{^Blk=supplementalmathematicaloperators}', "");
Expect(1, 11007, '\p{Blk=-	Supplemental-MathematicalOPERATORS}', "");
Expect(0, 11007, '\p{^Blk=-	Supplemental-MathematicalOPERATORS}', "");
Expect(0, 11007, '\P{Blk=-	Supplemental-MathematicalOPERATORS}', "");
Expect(1, 11007, '\P{^Blk=-	Supplemental-MathematicalOPERATORS}', "");
Expect(0, 10751, '\p{Blk=-	Supplemental-MathematicalOPERATORS}', "");
Expect(1, 10751, '\p{^Blk=-	Supplemental-MathematicalOPERATORS}', "");
Expect(1, 10751, '\P{Blk=-	Supplemental-MathematicalOPERATORS}', "");
Expect(0, 10751, '\P{^Blk=-	Supplemental-MathematicalOPERATORS}', "");
Error('\p{Is_Block= SupplementalMATHEMATICAL_OPERATORS/a/}');
Error('\P{Is_Block= SupplementalMATHEMATICAL_OPERATORS/a/}');
Expect(1, 11007, '\p{Is_Block=supplementalmathematicaloperators}', "");
Expect(0, 11007, '\p{^Is_Block=supplementalmathematicaloperators}', "");
Expect(0, 11007, '\P{Is_Block=supplementalmathematicaloperators}', "");
Expect(1, 11007, '\P{^Is_Block=supplementalmathematicaloperators}', "");
Expect(0, 10751, '\p{Is_Block=supplementalmathematicaloperators}', "");
Expect(1, 10751, '\p{^Is_Block=supplementalmathematicaloperators}', "");
Expect(1, 10751, '\P{Is_Block=supplementalmathematicaloperators}', "");
Expect(0, 10751, '\P{^Is_Block=supplementalmathematicaloperators}', "");
Expect(1, 11007, '\p{Is_Block=		supplemental mathematical	Operators}', "");
Expect(0, 11007, '\p{^Is_Block=		supplemental mathematical	Operators}', "");
Expect(0, 11007, '\P{Is_Block=		supplemental mathematical	Operators}', "");
Expect(1, 11007, '\P{^Is_Block=		supplemental mathematical	Operators}', "");
Expect(0, 10751, '\p{Is_Block=		supplemental mathematical	Operators}', "");
Expect(1, 10751, '\p{^Is_Block=		supplemental mathematical	Operators}', "");
Expect(1, 10751, '\P{Is_Block=		supplemental mathematical	Operators}', "");
Expect(0, 10751, '\P{^Is_Block=		supplemental mathematical	Operators}', "");
Error('\p{Is_Blk= /a/supplemental_Mathematical-Operators}');
Error('\P{Is_Blk= /a/supplemental_Mathematical-Operators}');
Expect(1, 11007, '\p{Is_Blk=supplementalmathematicaloperators}', "");
Expect(0, 11007, '\p{^Is_Blk=supplementalmathematicaloperators}', "");
Expect(0, 11007, '\P{Is_Blk=supplementalmathematicaloperators}', "");
Expect(1, 11007, '\P{^Is_Blk=supplementalmathematicaloperators}', "");
Expect(0, 10751, '\p{Is_Blk=supplementalmathematicaloperators}', "");
Expect(1, 10751, '\p{^Is_Blk=supplementalmathematicaloperators}', "");
Expect(1, 10751, '\P{Is_Blk=supplementalmathematicaloperators}', "");
Expect(0, 10751, '\P{^Is_Blk=supplementalmathematicaloperators}', "");
Expect(1, 11007, '\p{Is_Blk= -Supplemental Mathematical Operators}', "");
Expect(0, 11007, '\p{^Is_Blk= -Supplemental Mathematical Operators}', "");
Expect(0, 11007, '\P{Is_Blk= -Supplemental Mathematical Operators}', "");
Expect(1, 11007, '\P{^Is_Blk= -Supplemental Mathematical Operators}', "");
Expect(0, 10751, '\p{Is_Blk= -Supplemental Mathematical Operators}', "");
Expect(1, 10751, '\p{^Is_Blk= -Supplemental Mathematical Operators}', "");
Expect(1, 10751, '\P{Is_Blk= -Supplemental Mathematical Operators}', "");
Expect(0, 10751, '\P{^Is_Blk= -Supplemental Mathematical Operators}', "");
Error('\p{Block=_:=Shavian}');
Error('\P{Block=_:=Shavian}');
Expect(1, 66687, '\p{Block=shavian}', "");
Expect(0, 66687, '\p{^Block=shavian}', "");
Expect(0, 66687, '\P{Block=shavian}', "");
Expect(1, 66687, '\P{^Block=shavian}', "");
Expect(0, 66639, '\p{Block=shavian}', "");
Expect(1, 66639, '\p{^Block=shavian}', "");
Expect(1, 66639, '\P{Block=shavian}', "");
Expect(0, 66639, '\P{^Block=shavian}', "");
Expect(1, 66687, '\p{Block=_-Shavian}', "");
Expect(0, 66687, '\p{^Block=_-Shavian}', "");
Expect(0, 66687, '\P{Block=_-Shavian}', "");
Expect(1, 66687, '\P{^Block=_-Shavian}', "");
Expect(0, 66639, '\p{Block=_-Shavian}', "");
Expect(1, 66639, '\p{^Block=_-Shavian}', "");
Expect(1, 66639, '\P{Block=_-Shavian}', "");
Expect(0, 66639, '\P{^Block=_-Shavian}', "");
Error('\p{Blk= _Shavian:=}');
Error('\P{Blk= _Shavian:=}');
Expect(1, 66687, '\p{Blk: shavian}', "");
Expect(0, 66687, '\p{^Blk: shavian}', "");
Expect(0, 66687, '\P{Blk: shavian}', "");
Expect(1, 66687, '\P{^Blk: shavian}', "");
Expect(0, 66639, '\p{Blk: shavian}', "");
Expect(1, 66639, '\p{^Blk: shavian}', "");
Expect(1, 66639, '\P{Blk: shavian}', "");
Expect(0, 66639, '\P{^Blk: shavian}', "");
Expect(1, 66687, '\p{Blk:	- Shavian}', "");
Expect(0, 66687, '\p{^Blk:	- Shavian}', "");
Expect(0, 66687, '\P{Blk:	- Shavian}', "");
Expect(1, 66687, '\P{^Blk:	- Shavian}', "");
Expect(0, 66639, '\p{Blk:	- Shavian}', "");
Expect(1, 66639, '\p{^Blk:	- Shavian}', "");
Expect(1, 66639, '\P{Blk:	- Shavian}', "");
Expect(0, 66639, '\P{^Blk:	- Shavian}', "");
Error('\p{Is_Block=_-Shavian:=}');
Error('\P{Is_Block=_-Shavian:=}');
Expect(1, 66687, '\p{Is_Block=shavian}', "");
Expect(0, 66687, '\p{^Is_Block=shavian}', "");
Expect(0, 66687, '\P{Is_Block=shavian}', "");
Expect(1, 66687, '\P{^Is_Block=shavian}', "");
Expect(0, 66639, '\p{Is_Block=shavian}', "");
Expect(1, 66639, '\p{^Is_Block=shavian}', "");
Expect(1, 66639, '\P{Is_Block=shavian}', "");
Expect(0, 66639, '\P{^Is_Block=shavian}', "");
Expect(1, 66687, '\p{Is_Block= _Shavian}', "");
Expect(0, 66687, '\p{^Is_Block= _Shavian}', "");
Expect(0, 66687, '\P{Is_Block= _Shavian}', "");
Expect(1, 66687, '\P{^Is_Block= _Shavian}', "");
Expect(0, 66639, '\p{Is_Block= _Shavian}', "");
Expect(1, 66639, '\p{^Is_Block= _Shavian}', "");
Expect(1, 66639, '\P{Is_Block= _Shavian}', "");
Expect(0, 66639, '\P{^Is_Block= _Shavian}', "");
Error('\p{Is_Blk=/a/SHAVIAN}');
Error('\P{Is_Blk=/a/SHAVIAN}');
Expect(1, 66687, '\p{Is_Blk=shavian}', "");
Expect(0, 66687, '\p{^Is_Blk=shavian}', "");
Expect(0, 66687, '\P{Is_Blk=shavian}', "");
Expect(1, 66687, '\P{^Is_Blk=shavian}', "");
Expect(0, 66639, '\p{Is_Blk=shavian}', "");
Expect(1, 66639, '\p{^Is_Blk=shavian}', "");
Expect(1, 66639, '\P{Is_Blk=shavian}', "");
Expect(0, 66639, '\P{^Is_Blk=shavian}', "");
Expect(1, 66687, '\p{Is_Blk=_-Shavian}', "");
Expect(0, 66687, '\p{^Is_Blk=_-Shavian}', "");
Expect(0, 66687, '\P{Is_Blk=_-Shavian}', "");
Expect(1, 66687, '\P{^Is_Blk=_-Shavian}', "");
Expect(0, 66639, '\p{Is_Blk=_-Shavian}', "");
Expect(1, 66639, '\p{^Is_Blk=_-Shavian}', "");
Expect(1, 66639, '\P{Is_Blk=_-Shavian}', "");
Expect(0, 66639, '\P{^Is_Blk=_-Shavian}', "");
Error('\p{Block=:=Lydian}');
Error('\P{Block=:=Lydian}');
Expect(1, 67903, '\p{Block=lydian}', "");
Expect(0, 67903, '\p{^Block=lydian}', "");
Expect(0, 67903, '\P{Block=lydian}', "");
Expect(1, 67903, '\P{^Block=lydian}', "");
Expect(0, 67871, '\p{Block=lydian}', "");
Expect(1, 67871, '\p{^Block=lydian}', "");
Expect(1, 67871, '\P{Block=lydian}', "");
Expect(0, 67871, '\P{^Block=lydian}', "");
Expect(1, 67903, '\p{Block=-Lydian}', "");
Expect(0, 67903, '\p{^Block=-Lydian}', "");
Expect(0, 67903, '\P{Block=-Lydian}', "");
Expect(1, 67903, '\P{^Block=-Lydian}', "");
Expect(0, 67871, '\p{Block=-Lydian}', "");
Expect(1, 67871, '\p{^Block=-Lydian}', "");
Expect(1, 67871, '\P{Block=-Lydian}', "");
Expect(0, 67871, '\P{^Block=-Lydian}', "");
Error('\p{Blk=-/a/Lydian}');
Error('\P{Blk=-/a/Lydian}');
Expect(1, 67903, '\p{Blk=lydian}', "");
Expect(0, 67903, '\p{^Blk=lydian}', "");
Expect(0, 67903, '\P{Blk=lydian}', "");
Expect(1, 67903, '\P{^Blk=lydian}', "");
Expect(0, 67871, '\p{Blk=lydian}', "");
Expect(1, 67871, '\p{^Blk=lydian}', "");
Expect(1, 67871, '\P{Blk=lydian}', "");
Expect(0, 67871, '\P{^Blk=lydian}', "");
Expect(1, 67903, '\p{Blk= -LYDIAN}', "");
Expect(0, 67903, '\p{^Blk= -LYDIAN}', "");
Expect(0, 67903, '\P{Blk= -LYDIAN}', "");
Expect(1, 67903, '\P{^Blk= -LYDIAN}', "");
Expect(0, 67871, '\p{Blk= -LYDIAN}', "");
Expect(1, 67871, '\p{^Blk= -LYDIAN}', "");
Expect(1, 67871, '\P{Blk= -LYDIAN}', "");
Expect(0, 67871, '\P{^Blk= -LYDIAN}', "");
Error('\p{Is_Block=:=-	LYDIAN}');
Error('\P{Is_Block=:=-	LYDIAN}');
Expect(1, 67903, '\p{Is_Block=lydian}', "");
Expect(0, 67903, '\p{^Is_Block=lydian}', "");
Expect(0, 67903, '\P{Is_Block=lydian}', "");
Expect(1, 67903, '\P{^Is_Block=lydian}', "");
Expect(0, 67871, '\p{Is_Block=lydian}', "");
Expect(1, 67871, '\p{^Is_Block=lydian}', "");
Expect(1, 67871, '\P{Is_Block=lydian}', "");
Expect(0, 67871, '\P{^Is_Block=lydian}', "");
Expect(1, 67903, '\p{Is_Block=-LYDIAN}', "");
Expect(0, 67903, '\p{^Is_Block=-LYDIAN}', "");
Expect(0, 67903, '\P{Is_Block=-LYDIAN}', "");
Expect(1, 67903, '\P{^Is_Block=-LYDIAN}', "");
Expect(0, 67871, '\p{Is_Block=-LYDIAN}', "");
Expect(1, 67871, '\p{^Is_Block=-LYDIAN}', "");
Expect(1, 67871, '\P{Is_Block=-LYDIAN}', "");
Expect(0, 67871, '\P{^Is_Block=-LYDIAN}', "");
Error('\p{Is_Blk: :=__Lydian}');
Error('\P{Is_Blk: :=__Lydian}');
Expect(1, 67903, '\p{Is_Blk=lydian}', "");
Expect(0, 67903, '\p{^Is_Blk=lydian}', "");
Expect(0, 67903, '\P{Is_Blk=lydian}', "");
Expect(1, 67903, '\P{^Is_Blk=lydian}', "");
Expect(0, 67871, '\p{Is_Blk=lydian}', "");
Expect(1, 67871, '\p{^Is_Blk=lydian}', "");
Expect(1, 67871, '\P{Is_Blk=lydian}', "");
Expect(0, 67871, '\P{^Is_Blk=lydian}', "");
Expect(1, 67903, '\p{Is_Blk=LYDIAN}', "");
Expect(0, 67903, '\p{^Is_Blk=LYDIAN}', "");
Expect(0, 67903, '\P{Is_Blk=LYDIAN}', "");
Expect(1, 67903, '\P{^Is_Blk=LYDIAN}', "");
Expect(0, 67871, '\p{Is_Blk=LYDIAN}', "");
Expect(1, 67871, '\p{^Is_Blk=LYDIAN}', "");
Expect(1, 67871, '\P{Is_Blk=LYDIAN}', "");
Expect(0, 67871, '\P{^Is_Blk=LYDIAN}', "");
Error('\p{Block=/a/	CJK-Unified_IDEOGRAPHS	Extension-C}');
Error('\P{Block=/a/	CJK-Unified_IDEOGRAPHS	Extension-C}');
Expect(1, 177983, '\p{Block=cjkunifiedideographsextensionc}', "");
Expect(0, 177983, '\p{^Block=cjkunifiedideographsextensionc}', "");
Expect(0, 177983, '\P{Block=cjkunifiedideographsextensionc}', "");
Expect(1, 177983, '\P{^Block=cjkunifiedideographsextensionc}', "");
Expect(0, 173823, '\p{Block=cjkunifiedideographsextensionc}', "");
Expect(1, 173823, '\p{^Block=cjkunifiedideographsextensionc}', "");
Expect(1, 173823, '\P{Block=cjkunifiedideographsextensionc}', "");
Expect(0, 173823, '\P{^Block=cjkunifiedideographsextensionc}', "");
Expect(1, 177983, '\p{Block=__CJK	unified	IDEOGRAPHS extensionC}', "");
Expect(0, 177983, '\p{^Block=__CJK	unified	IDEOGRAPHS extensionC}', "");
Expect(0, 177983, '\P{Block=__CJK	unified	IDEOGRAPHS extensionC}', "");
Expect(1, 177983, '\P{^Block=__CJK	unified	IDEOGRAPHS extensionC}', "");
Expect(0, 173823, '\p{Block=__CJK	unified	IDEOGRAPHS extensionC}', "");
Expect(1, 173823, '\p{^Block=__CJK	unified	IDEOGRAPHS extensionC}', "");
Expect(1, 173823, '\P{Block=__CJK	unified	IDEOGRAPHS extensionC}', "");
Expect(0, 173823, '\P{^Block=__CJK	unified	IDEOGRAPHS extensionC}', "");
Error('\p{Blk=	_CJK-Unified:=IDEOGRAPHS	EXTENSIONC}');
Error('\P{Blk=	_CJK-Unified:=IDEOGRAPHS	EXTENSIONC}');
Expect(1, 177983, '\p{Blk:   cjkunifiedideographsextensionc}', "");
Expect(0, 177983, '\p{^Blk:   cjkunifiedideographsextensionc}', "");
Expect(0, 177983, '\P{Blk:   cjkunifiedideographsextensionc}', "");
Expect(1, 177983, '\P{^Blk:   cjkunifiedideographsextensionc}', "");
Expect(0, 173823, '\p{Blk:   cjkunifiedideographsextensionc}', "");
Expect(1, 173823, '\p{^Blk:   cjkunifiedideographsextensionc}', "");
Expect(1, 173823, '\P{Blk:   cjkunifiedideographsextensionc}', "");
Expect(0, 173823, '\P{^Blk:   cjkunifiedideographsextensionc}', "");
Expect(1, 177983, '\p{Blk:  CJK	UNIFIED_IdeographsExtension C}', "");
Expect(0, 177983, '\p{^Blk:  CJK	UNIFIED_IdeographsExtension C}', "");
Expect(0, 177983, '\P{Blk:  CJK	UNIFIED_IdeographsExtension C}', "");
Expect(1, 177983, '\P{^Blk:  CJK	UNIFIED_IdeographsExtension C}', "");
Expect(0, 173823, '\p{Blk:  CJK	UNIFIED_IdeographsExtension C}', "");
Expect(1, 173823, '\p{^Blk:  CJK	UNIFIED_IdeographsExtension C}', "");
Expect(1, 173823, '\P{Blk:  CJK	UNIFIED_IdeographsExtension C}', "");
Expect(0, 173823, '\P{^Blk:  CJK	UNIFIED_IdeographsExtension C}', "");
Error('\p{Is_Block=	:=CJK-UNIFIED-Ideographs EXTENSIONC}');
Error('\P{Is_Block=	:=CJK-UNIFIED-Ideographs EXTENSIONC}');
Expect(1, 177983, '\p{Is_Block=cjkunifiedideographsextensionc}', "");
Expect(0, 177983, '\p{^Is_Block=cjkunifiedideographsextensionc}', "");
Expect(0, 177983, '\P{Is_Block=cjkunifiedideographsextensionc}', "");
Expect(1, 177983, '\P{^Is_Block=cjkunifiedideographsextensionc}', "");
Expect(0, 173823, '\p{Is_Block=cjkunifiedideographsextensionc}', "");
Expect(1, 173823, '\p{^Is_Block=cjkunifiedideographsextensionc}', "");
Expect(1, 173823, '\P{Is_Block=cjkunifiedideographsextensionc}', "");
Expect(0, 173823, '\P{^Is_Block=cjkunifiedideographsextensionc}', "");
Expect(1, 177983, '\p{Is_Block=	_CJKUnified-IDEOGRAPHS_Extension-C}', "");
Expect(0, 177983, '\p{^Is_Block=	_CJKUnified-IDEOGRAPHS_Extension-C}', "");
Expect(0, 177983, '\P{Is_Block=	_CJKUnified-IDEOGRAPHS_Extension-C}', "");
Expect(1, 177983, '\P{^Is_Block=	_CJKUnified-IDEOGRAPHS_Extension-C}', "");
Expect(0, 173823, '\p{Is_Block=	_CJKUnified-IDEOGRAPHS_Extension-C}', "");
Expect(1, 173823, '\p{^Is_Block=	_CJKUnified-IDEOGRAPHS_Extension-C}', "");
Expect(1, 173823, '\P{Is_Block=	_CJKUnified-IDEOGRAPHS_Extension-C}', "");
Expect(0, 173823, '\P{^Is_Block=	_CJKUnified-IDEOGRAPHS_Extension-C}', "");
Error('\p{Is_Blk: /a/_cjk-unified-Ideographs	extension-C}');
Error('\P{Is_Blk: /a/_cjk-unified-Ideographs	extension-C}');
Expect(1, 177983, '\p{Is_Blk=cjkunifiedideographsextensionc}', "");
Expect(0, 177983, '\p{^Is_Blk=cjkunifiedideographsextensionc}', "");
Expect(0, 177983, '\P{Is_Blk=cjkunifiedideographsextensionc}', "");
Expect(1, 177983, '\P{^Is_Blk=cjkunifiedideographsextensionc}', "");
Expect(0, 173823, '\p{Is_Blk=cjkunifiedideographsextensionc}', "");
Expect(1, 173823, '\p{^Is_Blk=cjkunifiedideographsextensionc}', "");
Expect(1, 173823, '\P{Is_Blk=cjkunifiedideographsextensionc}', "");
Expect(0, 173823, '\P{^Is_Blk=cjkunifiedideographsextensionc}', "");
Expect(1, 177983, '\p{Is_Blk=	 cjk	UNIFIED ideographs-EXTENSION	C}', "");
Expect(0, 177983, '\p{^Is_Blk=	 cjk	UNIFIED ideographs-EXTENSION	C}', "");
Expect(0, 177983, '\P{Is_Blk=	 cjk	UNIFIED ideographs-EXTENSION	C}', "");
Expect(1, 177983, '\P{^Is_Blk=	 cjk	UNIFIED ideographs-EXTENSION	C}', "");
Expect(0, 173823, '\p{Is_Blk=	 cjk	UNIFIED ideographs-EXTENSION	C}', "");
Expect(1, 173823, '\p{^Is_Blk=	 cjk	UNIFIED ideographs-EXTENSION	C}', "");
Expect(1, 173823, '\P{Is_Blk=	 cjk	UNIFIED ideographs-EXTENSION	C}', "");
Expect(0, 173823, '\P{^Is_Blk=	 cjk	UNIFIED ideographs-EXTENSION	C}', "");
Error('\p{Block:_mongolian/a/}');
Error('\P{Block:_mongolian/a/}');
Expect(1, 6319, '\p{Block=mongolian}', "");
Expect(0, 6319, '\p{^Block=mongolian}', "");
Expect(0, 6319, '\P{Block=mongolian}', "");
Expect(1, 6319, '\P{^Block=mongolian}', "");
Expect(0, 6143, '\p{Block=mongolian}', "");
Expect(1, 6143, '\p{^Block=mongolian}', "");
Expect(1, 6143, '\P{Block=mongolian}', "");
Expect(0, 6143, '\P{^Block=mongolian}', "");
Expect(1, 6319, '\p{Block=_-mongolian}', "");
Expect(0, 6319, '\p{^Block=_-mongolian}', "");
Expect(0, 6319, '\P{Block=_-mongolian}', "");
Expect(1, 6319, '\P{^Block=_-mongolian}', "");
Expect(0, 6143, '\p{Block=_-mongolian}', "");
Expect(1, 6143, '\p{^Block=_-mongolian}', "");
Expect(1, 6143, '\P{Block=_-mongolian}', "");
Expect(0, 6143, '\P{^Block=_-mongolian}', "");
Error('\p{Blk=	Mongolian:=}');
Error('\P{Blk=	Mongolian:=}');
Expect(1, 6319, '\p{Blk=mongolian}', "");
Expect(0, 6319, '\p{^Blk=mongolian}', "");
Expect(0, 6319, '\P{Blk=mongolian}', "");
Expect(1, 6319, '\P{^Blk=mongolian}', "");
Expect(0, 6143, '\p{Blk=mongolian}', "");
Expect(1, 6143, '\p{^Blk=mongolian}', "");
Expect(1, 6143, '\P{Blk=mongolian}', "");
Expect(0, 6143, '\P{^Blk=mongolian}', "");
Expect(1, 6319, '\p{Blk=	MONGOLIAN}', "");
Expect(0, 6319, '\p{^Blk=	MONGOLIAN}', "");
Expect(0, 6319, '\P{Blk=	MONGOLIAN}', "");
Expect(1, 6319, '\P{^Blk=	MONGOLIAN}', "");
Expect(0, 6143, '\p{Blk=	MONGOLIAN}', "");
Expect(1, 6143, '\p{^Blk=	MONGOLIAN}', "");
Expect(1, 6143, '\P{Blk=	MONGOLIAN}', "");
Expect(0, 6143, '\P{^Blk=	MONGOLIAN}', "");
Error('\p{Is_Block= /a/Mongolian}');
Error('\P{Is_Block= /a/Mongolian}');
Expect(1, 6319, '\p{Is_Block:	mongolian}', "");
Expect(0, 6319, '\p{^Is_Block:	mongolian}', "");
Expect(0, 6319, '\P{Is_Block:	mongolian}', "");
Expect(1, 6319, '\P{^Is_Block:	mongolian}', "");
Expect(0, 6143, '\p{Is_Block:	mongolian}', "");
Expect(1, 6143, '\p{^Is_Block:	mongolian}', "");
Expect(1, 6143, '\P{Is_Block:	mongolian}', "");
Expect(0, 6143, '\P{^Is_Block:	mongolian}', "");
Expect(1, 6319, '\p{Is_Block= 	MONGOLIAN}', "");
Expect(0, 6319, '\p{^Is_Block= 	MONGOLIAN}', "");
Expect(0, 6319, '\P{Is_Block= 	MONGOLIAN}', "");
Expect(1, 6319, '\P{^Is_Block= 	MONGOLIAN}', "");
Expect(0, 6143, '\p{Is_Block= 	MONGOLIAN}', "");
Expect(1, 6143, '\p{^Is_Block= 	MONGOLIAN}', "");
Expect(1, 6143, '\P{Is_Block= 	MONGOLIAN}', "");
Expect(0, 6143, '\P{^Is_Block= 	MONGOLIAN}', "");
Error('\p{Is_Blk=--MONGOLIAN/a/}');
Error('\P{Is_Blk=--MONGOLIAN/a/}');
Expect(1, 6319, '\p{Is_Blk=mongolian}', "");
Expect(0, 6319, '\p{^Is_Blk=mongolian}', "");
Expect(0, 6319, '\P{Is_Blk=mongolian}', "");
Expect(1, 6319, '\P{^Is_Blk=mongolian}', "");
Expect(0, 6143, '\p{Is_Blk=mongolian}', "");
Expect(1, 6143, '\p{^Is_Blk=mongolian}', "");
Expect(1, 6143, '\P{Is_Blk=mongolian}', "");
Expect(0, 6143, '\P{^Is_Blk=mongolian}', "");
Expect(1, 6319, '\p{Is_Blk=-Mongolian}', "");
Expect(0, 6319, '\p{^Is_Blk=-Mongolian}', "");
Expect(0, 6319, '\P{Is_Blk=-Mongolian}', "");
Expect(1, 6319, '\P{^Is_Blk=-Mongolian}', "");
Expect(0, 6143, '\p{Is_Blk=-Mongolian}', "");
Expect(1, 6143, '\p{^Is_Blk=-Mongolian}', "");
Expect(1, 6143, '\P{Is_Blk=-Mongolian}', "");
Expect(0, 6143, '\P{^Is_Blk=-Mongolian}', "");
Error('\p{Block=:=_Lao}');
Error('\P{Block=:=_Lao}');
Expect(1, 3839, '\p{Block:lao}', "");
Expect(0, 3839, '\p{^Block:lao}', "");
Expect(0, 3839, '\P{Block:lao}', "");
Expect(1, 3839, '\P{^Block:lao}', "");
Expect(0, 3711, '\p{Block:lao}', "");
Expect(1, 3711, '\p{^Block:lao}', "");
Expect(1, 3711, '\P{Block:lao}', "");
Expect(0, 3711, '\P{^Block:lao}', "");
Expect(1, 3839, '\p{Block=_ Lao}', "");
Expect(0, 3839, '\p{^Block=_ Lao}', "");
Expect(0, 3839, '\P{Block=_ Lao}', "");
Expect(1, 3839, '\P{^Block=_ Lao}', "");
Expect(0, 3711, '\p{Block=_ Lao}', "");
Expect(1, 3711, '\p{^Block=_ Lao}', "");
Expect(1, 3711, '\P{Block=_ Lao}', "");
Expect(0, 3711, '\P{^Block=_ Lao}', "");
Error('\p{Blk=_/a/lao}');
Error('\P{Blk=_/a/lao}');
Expect(1, 3839, '\p{Blk=lao}', "");
Expect(0, 3839, '\p{^Blk=lao}', "");
Expect(0, 3839, '\P{Blk=lao}', "");
Expect(1, 3839, '\P{^Blk=lao}', "");
Expect(0, 3711, '\p{Blk=lao}', "");
Expect(1, 3711, '\p{^Blk=lao}', "");
Expect(1, 3711, '\P{Blk=lao}', "");
Expect(0, 3711, '\P{^Blk=lao}', "");
Expect(1, 3839, '\p{Blk=-_LAO}', "");
Expect(0, 3839, '\p{^Blk=-_LAO}', "");
Expect(0, 3839, '\P{Blk=-_LAO}', "");
Expect(1, 3839, '\P{^Blk=-_LAO}', "");
Expect(0, 3711, '\p{Blk=-_LAO}', "");
Expect(1, 3711, '\p{^Blk=-_LAO}', "");
Expect(1, 3711, '\P{Blk=-_LAO}', "");
Expect(0, 3711, '\P{^Blk=-_LAO}', "");
Error('\p{Is_Block=/a/-	Lao}');
Error('\P{Is_Block=/a/-	Lao}');
Expect(1, 3839, '\p{Is_Block=lao}', "");
Expect(0, 3839, '\p{^Is_Block=lao}', "");
Expect(0, 3839, '\P{Is_Block=lao}', "");
Expect(1, 3839, '\P{^Is_Block=lao}', "");
Expect(0, 3711, '\p{Is_Block=lao}', "");
Expect(1, 3711, '\p{^Is_Block=lao}', "");
Expect(1, 3711, '\P{Is_Block=lao}', "");
Expect(0, 3711, '\P{^Is_Block=lao}', "");
Expect(1, 3839, '\p{Is_Block=_Lao}', "");
Expect(0, 3839, '\p{^Is_Block=_Lao}', "");
Expect(0, 3839, '\P{Is_Block=_Lao}', "");
Expect(1, 3839, '\P{^Is_Block=_Lao}', "");
Expect(0, 3711, '\p{Is_Block=_Lao}', "");
Expect(1, 3711, '\p{^Is_Block=_Lao}', "");
Expect(1, 3711, '\P{Is_Block=_Lao}', "");
Expect(0, 3711, '\P{^Is_Block=_Lao}', "");
Error('\p{Is_Blk= _Lao/a/}');
Error('\P{Is_Blk= _Lao/a/}');
Expect(1, 3839, '\p{Is_Blk=lao}', "");
Expect(0, 3839, '\p{^Is_Blk=lao}', "");
Expect(0, 3839, '\P{Is_Blk=lao}', "");
Expect(1, 3839, '\P{^Is_Blk=lao}', "");
Expect(0, 3711, '\p{Is_Blk=lao}', "");
Expect(1, 3711, '\p{^Is_Blk=lao}', "");
Expect(1, 3711, '\P{Is_Blk=lao}', "");
Expect(0, 3711, '\P{^Is_Blk=lao}', "");
Expect(1, 3839, '\p{Is_Blk=	lao}', "");
Expect(0, 3839, '\p{^Is_Blk=	lao}', "");
Expect(0, 3839, '\P{Is_Blk=	lao}', "");
Expect(1, 3839, '\P{^Is_Blk=	lao}', "");
Expect(0, 3711, '\p{Is_Blk=	lao}', "");
Expect(1, 3711, '\p{^Is_Blk=	lao}', "");
Expect(1, 3711, '\P{Is_Blk=	lao}', "");
Expect(0, 3711, '\P{^Is_Blk=	lao}', "");
Error('\p{Block=:= -HIRAGANA}');
Error('\P{Block=:= -HIRAGANA}');
Expect(1, 12447, '\p{Block:	hiragana}', "");
Expect(0, 12447, '\p{^Block:	hiragana}', "");
Expect(0, 12447, '\P{Block:	hiragana}', "");
Expect(1, 12447, '\P{^Block:	hiragana}', "");
Expect(0, 12351, '\p{Block:	hiragana}', "");
Expect(1, 12351, '\p{^Block:	hiragana}', "");
Expect(1, 12351, '\P{Block:	hiragana}', "");
Expect(0, 12351, '\P{^Block:	hiragana}', "");
Expect(1, 12447, '\p{Block=- Hiragana}', "");
Expect(0, 12447, '\p{^Block=- Hiragana}', "");
Expect(0, 12447, '\P{Block=- Hiragana}', "");
Expect(1, 12447, '\P{^Block=- Hiragana}', "");
Expect(0, 12351, '\p{Block=- Hiragana}', "");
Expect(1, 12351, '\p{^Block=- Hiragana}', "");
Expect(1, 12351, '\P{Block=- Hiragana}', "");
Expect(0, 12351, '\P{^Block=- Hiragana}', "");
Error('\p{Blk=-:=Hiragana}');
Error('\P{Blk=-:=Hiragana}');
Expect(1, 12447, '\p{Blk=hiragana}', "");
Expect(0, 12447, '\p{^Blk=hiragana}', "");
Expect(0, 12447, '\P{Blk=hiragana}', "");
Expect(1, 12447, '\P{^Blk=hiragana}', "");
Expect(0, 12351, '\p{Blk=hiragana}', "");
Expect(1, 12351, '\p{^Blk=hiragana}', "");
Expect(1, 12351, '\P{Blk=hiragana}', "");
Expect(0, 12351, '\P{^Blk=hiragana}', "");
Expect(1, 12447, '\p{Blk=-Hiragana}', "");
Expect(0, 12447, '\p{^Blk=-Hiragana}', "");
Expect(0, 12447, '\P{Blk=-Hiragana}', "");
Expect(1, 12447, '\P{^Blk=-Hiragana}', "");
Expect(0, 12351, '\p{Blk=-Hiragana}', "");
Expect(1, 12351, '\p{^Blk=-Hiragana}', "");
Expect(1, 12351, '\P{Blk=-Hiragana}', "");
Expect(0, 12351, '\P{^Blk=-Hiragana}', "");
Error('\p{Is_Block:   /a/-	hiragana}');
Error('\P{Is_Block:   /a/-	hiragana}');
Expect(1, 12447, '\p{Is_Block=hiragana}', "");
Expect(0, 12447, '\p{^Is_Block=hiragana}', "");
Expect(0, 12447, '\P{Is_Block=hiragana}', "");
Expect(1, 12447, '\P{^Is_Block=hiragana}', "");
Expect(0, 12351, '\p{Is_Block=hiragana}', "");
Expect(1, 12351, '\p{^Is_Block=hiragana}', "");
Expect(1, 12351, '\P{Is_Block=hiragana}', "");
Expect(0, 12351, '\P{^Is_Block=hiragana}', "");
Expect(1, 12447, '\p{Is_Block=	 Hiragana}', "");
Expect(0, 12447, '\p{^Is_Block=	 Hiragana}', "");
Expect(0, 12447, '\P{Is_Block=	 Hiragana}', "");
Expect(1, 12447, '\P{^Is_Block=	 Hiragana}', "");
Expect(0, 12351, '\p{Is_Block=	 Hiragana}', "");
Expect(1, 12351, '\p{^Is_Block=	 Hiragana}', "");
Expect(1, 12351, '\P{Is_Block=	 Hiragana}', "");
Expect(0, 12351, '\P{^Is_Block=	 Hiragana}', "");
Error('\p{Is_Blk=__hiragana/a/}');
Error('\P{Is_Blk=__hiragana/a/}');
Expect(1, 12447, '\p{Is_Blk=hiragana}', "");
Expect(0, 12447, '\p{^Is_Blk=hiragana}', "");
Expect(0, 12447, '\P{Is_Blk=hiragana}', "");
Expect(1, 12447, '\P{^Is_Blk=hiragana}', "");
Expect(0, 12351, '\p{Is_Blk=hiragana}', "");
Expect(1, 12351, '\p{^Is_Blk=hiragana}', "");
Expect(1, 12351, '\P{Is_Blk=hiragana}', "");
Expect(0, 12351, '\P{^Is_Blk=hiragana}', "");
Expect(1, 12447, '\p{Is_Blk=--HIRAGANA}', "");
Expect(0, 12447, '\p{^Is_Blk=--HIRAGANA}', "");
Expect(0, 12447, '\P{Is_Blk=--HIRAGANA}', "");
Expect(1, 12447, '\P{^Is_Blk=--HIRAGANA}', "");
Expect(0, 12351, '\p{Is_Blk=--HIRAGANA}', "");
Expect(1, 12351, '\p{^Is_Blk=--HIRAGANA}', "");
Expect(1, 12351, '\P{Is_Blk=--HIRAGANA}', "");
Expect(0, 12351, '\P{^Is_Blk=--HIRAGANA}', "");
Error('\p{Block=	_lepcha/a/}');
Error('\P{Block=	_lepcha/a/}');
Expect(1, 7247, '\p{Block=lepcha}', "");
Expect(0, 7247, '\p{^Block=lepcha}', "");
Expect(0, 7247, '\P{Block=lepcha}', "");
Expect(1, 7247, '\P{^Block=lepcha}', "");
Expect(0, 7167, '\p{Block=lepcha}', "");
Expect(1, 7167, '\p{^Block=lepcha}', "");
Expect(1, 7167, '\P{Block=lepcha}', "");
Expect(0, 7167, '\P{^Block=lepcha}', "");
Expect(1, 7247, '\p{Block=	_Lepcha}', "");
Expect(0, 7247, '\p{^Block=	_Lepcha}', "");
Expect(0, 7247, '\P{Block=	_Lepcha}', "");
Expect(1, 7247, '\P{^Block=	_Lepcha}', "");
Expect(0, 7167, '\p{Block=	_Lepcha}', "");
Expect(1, 7167, '\p{^Block=	_Lepcha}', "");
Expect(1, 7167, '\P{Block=	_Lepcha}', "");
Expect(0, 7167, '\P{^Block=	_Lepcha}', "");
Error('\p{Blk=LEPCHA:=}');
Error('\P{Blk=LEPCHA:=}');
Expect(1, 7247, '\p{Blk=lepcha}', "");
Expect(0, 7247, '\p{^Blk=lepcha}', "");
Expect(0, 7247, '\P{Blk=lepcha}', "");
Expect(1, 7247, '\P{^Blk=lepcha}', "");
Expect(0, 7167, '\p{Blk=lepcha}', "");
Expect(1, 7167, '\p{^Blk=lepcha}', "");
Expect(1, 7167, '\P{Blk=lepcha}', "");
Expect(0, 7167, '\P{^Blk=lepcha}', "");
Expect(1, 7247, '\p{Blk=	_lepcha}', "");
Expect(0, 7247, '\p{^Blk=	_lepcha}', "");
Expect(0, 7247, '\P{Blk=	_lepcha}', "");
Expect(1, 7247, '\P{^Blk=	_lepcha}', "");
Expect(0, 7167, '\p{Blk=	_lepcha}', "");
Expect(1, 7167, '\p{^Blk=	_lepcha}', "");
Expect(1, 7167, '\P{Blk=	_lepcha}', "");
Expect(0, 7167, '\P{^Blk=	_lepcha}', "");
Error('\p{Is_Block=/a/-_Lepcha}');
Error('\P{Is_Block=/a/-_Lepcha}');
Expect(1, 7247, '\p{Is_Block:	lepcha}', "");
Expect(0, 7247, '\p{^Is_Block:	lepcha}', "");
Expect(0, 7247, '\P{Is_Block:	lepcha}', "");
Expect(1, 7247, '\P{^Is_Block:	lepcha}', "");
Expect(0, 7167, '\p{Is_Block:	lepcha}', "");
Expect(1, 7167, '\p{^Is_Block:	lepcha}', "");
Expect(1, 7167, '\P{Is_Block:	lepcha}', "");
Expect(0, 7167, '\P{^Is_Block:	lepcha}', "");
Expect(1, 7247, '\p{Is_Block=-	Lepcha}', "");
Expect(0, 7247, '\p{^Is_Block=-	Lepcha}', "");
Expect(0, 7247, '\P{Is_Block=-	Lepcha}', "");
Expect(1, 7247, '\P{^Is_Block=-	Lepcha}', "");
Expect(0, 7167, '\p{Is_Block=-	Lepcha}', "");
Expect(1, 7167, '\p{^Is_Block=-	Lepcha}', "");
Expect(1, 7167, '\P{Is_Block=-	Lepcha}', "");
Expect(0, 7167, '\P{^Is_Block=-	Lepcha}', "");
Error('\p{Is_Blk=_/a/LEPCHA}');
Error('\P{Is_Blk=_/a/LEPCHA}');
Expect(1, 7247, '\p{Is_Blk=lepcha}', "");
Expect(0, 7247, '\p{^Is_Blk=lepcha}', "");
Expect(0, 7247, '\P{Is_Blk=lepcha}', "");
Expect(1, 7247, '\P{^Is_Blk=lepcha}', "");
Expect(0, 7167, '\p{Is_Blk=lepcha}', "");
Expect(1, 7167, '\p{^Is_Blk=lepcha}', "");
Expect(1, 7167, '\P{Is_Blk=lepcha}', "");
Expect(0, 7167, '\P{^Is_Blk=lepcha}', "");
Expect(1, 7247, '\p{Is_Blk=_-Lepcha}', "");
Expect(0, 7247, '\p{^Is_Blk=_-Lepcha}', "");
Expect(0, 7247, '\P{Is_Blk=_-Lepcha}', "");
Expect(1, 7247, '\P{^Is_Blk=_-Lepcha}', "");
Expect(0, 7167, '\p{Is_Blk=_-Lepcha}', "");
Expect(1, 7167, '\p{^Is_Blk=_-Lepcha}', "");
Expect(1, 7167, '\P{Is_Blk=_-Lepcha}', "");
Expect(0, 7167, '\P{^Is_Blk=_-Lepcha}', "");
Error('\p{Block=/a/_syloti	Nagri}');
Error('\P{Block=/a/_syloti	Nagri}');
Expect(1, 43055, '\p{Block=sylotinagri}', "");
Expect(0, 43055, '\p{^Block=sylotinagri}', "");
Expect(0, 43055, '\P{Block=sylotinagri}', "");
Expect(1, 43055, '\P{^Block=sylotinagri}', "");
Expect(0, 43007, '\p{Block=sylotinagri}', "");
Expect(1, 43007, '\p{^Block=sylotinagri}', "");
Expect(1, 43007, '\P{Block=sylotinagri}', "");
Expect(0, 43007, '\P{^Block=sylotinagri}', "");
Expect(1, 43055, '\p{Block:   --SYLOTI NAGRI}', "");
Expect(0, 43055, '\p{^Block:   --SYLOTI NAGRI}', "");
Expect(0, 43055, '\P{Block:   --SYLOTI NAGRI}', "");
Expect(1, 43055, '\P{^Block:   --SYLOTI NAGRI}', "");
Expect(0, 43007, '\p{Block:   --SYLOTI NAGRI}', "");
Expect(1, 43007, '\p{^Block:   --SYLOTI NAGRI}', "");
Expect(1, 43007, '\P{Block:   --SYLOTI NAGRI}', "");
Expect(0, 43007, '\P{^Block:   --SYLOTI NAGRI}', "");
Error('\p{Blk= 	Syloti	nagri/a/}');
Error('\P{Blk= 	Syloti	nagri/a/}');
Expect(1, 43055, '\p{Blk=sylotinagri}', "");
Expect(0, 43055, '\p{^Blk=sylotinagri}', "");
Expect(0, 43055, '\P{Blk=sylotinagri}', "");
Expect(1, 43055, '\P{^Blk=sylotinagri}', "");
Expect(0, 43007, '\p{Blk=sylotinagri}', "");
Expect(1, 43007, '\p{^Blk=sylotinagri}', "");
Expect(1, 43007, '\P{Blk=sylotinagri}', "");
Expect(0, 43007, '\P{^Blk=sylotinagri}', "");
Expect(1, 43055, '\p{Blk=--Syloti NAGRI}', "");
Expect(0, 43055, '\p{^Blk=--Syloti NAGRI}', "");
Expect(0, 43055, '\P{Blk=--Syloti NAGRI}', "");
Expect(1, 43055, '\P{^Blk=--Syloti NAGRI}', "");
Expect(0, 43007, '\p{Blk=--Syloti NAGRI}', "");
Expect(1, 43007, '\p{^Blk=--Syloti NAGRI}', "");
Expect(1, 43007, '\P{Blk=--Syloti NAGRI}', "");
Expect(0, 43007, '\P{^Blk=--Syloti NAGRI}', "");
Error('\p{Is_Block= /a/Syloti NAGRI}');
Error('\P{Is_Block= /a/Syloti NAGRI}');
Expect(1, 43055, '\p{Is_Block=sylotinagri}', "");
Expect(0, 43055, '\p{^Is_Block=sylotinagri}', "");
Expect(0, 43055, '\P{Is_Block=sylotinagri}', "");
Expect(1, 43055, '\P{^Is_Block=sylotinagri}', "");
Expect(0, 43007, '\p{Is_Block=sylotinagri}', "");
Expect(1, 43007, '\p{^Is_Block=sylotinagri}', "");
Expect(1, 43007, '\P{Is_Block=sylotinagri}', "");
Expect(0, 43007, '\P{^Is_Block=sylotinagri}', "");
Expect(1, 43055, '\p{Is_Block=	syloti Nagri}', "");
Expect(0, 43055, '\p{^Is_Block=	syloti Nagri}', "");
Expect(0, 43055, '\P{Is_Block=	syloti Nagri}', "");
Expect(1, 43055, '\P{^Is_Block=	syloti Nagri}', "");
Expect(0, 43007, '\p{Is_Block=	syloti Nagri}', "");
Expect(1, 43007, '\p{^Is_Block=	syloti Nagri}', "");
Expect(1, 43007, '\P{Is_Block=	syloti Nagri}', "");
Expect(0, 43007, '\P{^Is_Block=	syloti Nagri}', "");
Error('\p{Is_Blk= :=Syloti Nagri}');
Error('\P{Is_Blk= :=Syloti Nagri}');
Expect(1, 43055, '\p{Is_Blk:   sylotinagri}', "");
Expect(0, 43055, '\p{^Is_Blk:   sylotinagri}', "");
Expect(0, 43055, '\P{Is_Blk:   sylotinagri}', "");
Expect(1, 43055, '\P{^Is_Blk:   sylotinagri}', "");
Expect(0, 43007, '\p{Is_Blk:   sylotinagri}', "");
Expect(1, 43007, '\p{^Is_Blk:   sylotinagri}', "");
Expect(1, 43007, '\P{Is_Blk:   sylotinagri}', "");
Expect(0, 43007, '\P{^Is_Blk:   sylotinagri}', "");
Expect(1, 43055, '\p{Is_Blk=	Syloti-Nagri}', "");
Expect(0, 43055, '\p{^Is_Blk=	Syloti-Nagri}', "");
Expect(0, 43055, '\P{Is_Blk=	Syloti-Nagri}', "");
Expect(1, 43055, '\P{^Is_Blk=	Syloti-Nagri}', "");
Expect(0, 43007, '\p{Is_Blk=	Syloti-Nagri}', "");
Expect(1, 43007, '\p{^Is_Blk=	Syloti-Nagri}', "");
Expect(1, 43007, '\P{Is_Blk=	Syloti-Nagri}', "");
Expect(0, 43007, '\P{^Is_Blk=	Syloti-Nagri}', "");
Error('\p{Block=_	Combining:=diacritical Marks ForSymbols}');
Error('\P{Block=_	Combining:=diacritical Marks ForSymbols}');
Expect(1, 8447, '\p{Block=combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8447, '\p{^Block=combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8447, '\P{Block=combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8447, '\P{^Block=combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8399, '\p{Block=combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8399, '\p{^Block=combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8399, '\P{Block=combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8399, '\P{^Block=combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8447, '\p{Block=__combining	Diacritical_marks	For	SYMBOLS}', "");
Expect(0, 8447, '\p{^Block=__combining	Diacritical_marks	For	SYMBOLS}', "");
Expect(0, 8447, '\P{Block=__combining	Diacritical_marks	For	SYMBOLS}', "");
Expect(1, 8447, '\P{^Block=__combining	Diacritical_marks	For	SYMBOLS}', "");
Expect(0, 8399, '\p{Block=__combining	Diacritical_marks	For	SYMBOLS}', "");
Expect(1, 8399, '\p{^Block=__combining	Diacritical_marks	For	SYMBOLS}', "");
Expect(1, 8399, '\P{Block=__combining	Diacritical_marks	For	SYMBOLS}', "");
Expect(0, 8399, '\P{^Block=__combining	Diacritical_marks	For	SYMBOLS}', "");
Error('\p{Blk=-:=Combining marks_For	Symbols}');
Error('\P{Blk=-:=Combining marks_For	Symbols}');
Expect(1, 8447, '\p{Blk=combiningmarksforsymbols}', "");
Expect(0, 8447, '\p{^Blk=combiningmarksforsymbols}', "");
Expect(0, 8447, '\P{Blk=combiningmarksforsymbols}', "");
Expect(1, 8447, '\P{^Blk=combiningmarksforsymbols}', "");
Expect(0, 8399, '\p{Blk=combiningmarksforsymbols}', "");
Expect(1, 8399, '\p{^Blk=combiningmarksforsymbols}', "");
Expect(1, 8399, '\P{Blk=combiningmarksforsymbols}', "");
Expect(0, 8399, '\P{^Blk=combiningmarksforsymbols}', "");
Expect(1, 8447, '\p{Blk:-	Combining_marks-For symbols}', "");
Expect(0, 8447, '\p{^Blk:-	Combining_marks-For symbols}', "");
Expect(0, 8447, '\P{Blk:-	Combining_marks-For symbols}', "");
Expect(1, 8447, '\P{^Blk:-	Combining_marks-For symbols}', "");
Expect(0, 8399, '\p{Blk:-	Combining_marks-For symbols}', "");
Expect(1, 8399, '\p{^Blk:-	Combining_marks-For symbols}', "");
Expect(1, 8399, '\P{Blk:-	Combining_marks-For symbols}', "");
Expect(0, 8399, '\P{^Blk:-	Combining_marks-For symbols}', "");
Error('\p{Is_Block=  Combining_Diacritical/a/Marksforsymbols}');
Error('\P{Is_Block=  Combining_Diacritical/a/Marksforsymbols}');
Expect(1, 8447, '\p{Is_Block=combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8447, '\p{^Is_Block=combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8447, '\P{Is_Block=combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8447, '\P{^Is_Block=combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8399, '\p{Is_Block=combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8399, '\p{^Is_Block=combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8399, '\P{Is_Block=combiningdiacriticalmarksforsymbols}', "");
Expect(0, 8399, '\P{^Is_Block=combiningdiacriticalmarksforsymbols}', "");
Expect(1, 8447, '\p{Is_Block=_combining DIACRITICAL	Marksforsymbols}', "");
Expect(0, 8447, '\p{^Is_Block=_combining DIACRITICAL	Marksforsymbols}', "");
Expect(0, 8447, '\P{Is_Block=_combining DIACRITICAL	Marksforsymbols}', "");
Expect(1, 8447, '\P{^Is_Block=_combining DIACRITICAL	Marksforsymbols}', "");
Expect(0, 8399, '\p{Is_Block=_combining DIACRITICAL	Marksforsymbols}', "");
Expect(1, 8399, '\p{^Is_Block=_combining DIACRITICAL	Marksforsymbols}', "");
Expect(1, 8399, '\P{Is_Block=_combining DIACRITICAL	Marksforsymbols}', "");
Expect(0, 8399, '\P{^Is_Block=_combining DIACRITICAL	Marksforsymbols}', "");
Error('\p{Is_Blk=__COMBINING Marks-for/a/SYMBOLS}');
Error('\P{Is_Blk=__COMBINING Marks-for/a/SYMBOLS}');
Expect(1, 8447, '\p{Is_Blk=combiningmarksforsymbols}', "");
Expect(0, 8447, '\p{^Is_Blk=combiningmarksforsymbols}', "");
Expect(0, 8447, '\P{Is_Blk=combiningmarksforsymbols}', "");
Expect(1, 8447, '\P{^Is_Blk=combiningmarksforsymbols}', "");
Expect(0, 8399, '\p{Is_Blk=combiningmarksforsymbols}', "");
Expect(1, 8399, '\p{^Is_Blk=combiningmarksforsymbols}', "");
Expect(1, 8399, '\P{Is_Blk=combiningmarksforsymbols}', "");
Expect(0, 8399, '\P{^Is_Blk=combiningmarksforsymbols}', "");
Expect(1, 8447, '\p{Is_Blk= _COMBINING MarksFor_symbols}', "");
Expect(0, 8447, '\p{^Is_Blk= _COMBINING MarksFor_symbols}', "");
Expect(0, 8447, '\P{Is_Blk= _COMBINING MarksFor_symbols}', "");
Expect(1, 8447, '\P{^Is_Blk= _COMBINING MarksFor_symbols}', "");
Expect(0, 8399, '\p{Is_Blk= _COMBINING MarksFor_symbols}', "");
Expect(1, 8399, '\p{^Is_Blk= _COMBINING MarksFor_symbols}', "");
Expect(1, 8399, '\P{Is_Blk= _COMBINING MarksFor_symbols}', "");
Expect(0, 8399, '\P{^Is_Blk= _COMBINING MarksFor_symbols}', "");
Error('\p{Block=  Basic/a/latin}');
Error('\P{Block=  Basic/a/latin}');
Expect(1, 127, '\p{Block=basiclatin}', "");
Expect(0, 127, '\p{^Block=basiclatin}', "");
Expect(0, 127, '\P{Block=basiclatin}', "");
Expect(1, 127, '\P{^Block=basiclatin}', "");
Expect(0, 128, '\p{Block=basiclatin}', "");
Expect(1, 128, '\p{^Block=basiclatin}', "");
Expect(1, 128, '\P{Block=basiclatin}', "");
Expect(0, 128, '\P{^Block=basiclatin}', "");
Expect(1, 127, '\p{Block=	basic Latin}', "");
Expect(0, 127, '\p{^Block=	basic Latin}', "");
Expect(0, 127, '\P{Block=	basic Latin}', "");
Expect(1, 127, '\P{^Block=	basic Latin}', "");
Expect(0, 128, '\p{Block=	basic Latin}', "");
Expect(1, 128, '\p{^Block=	basic Latin}', "");
Expect(1, 128, '\P{Block=	basic Latin}', "");
Expect(0, 128, '\P{^Block=	basic Latin}', "");
Error('\p{Blk=:=__ASCII}');
Error('\P{Blk=:=__ASCII}');
Expect(1, 127, '\p{Blk=ascii}', "");
Expect(0, 127, '\p{^Blk=ascii}', "");
Expect(0, 127, '\P{Blk=ascii}', "");
Expect(1, 127, '\P{^Blk=ascii}', "");
Expect(0, 128, '\p{Blk=ascii}', "");
Expect(1, 128, '\p{^Blk=ascii}', "");
Expect(1, 128, '\P{Blk=ascii}', "");
Expect(0, 128, '\P{^Blk=ascii}', "");
Expect(1, 127, '\p{Blk= ASCII}', "");
Expect(0, 127, '\p{^Blk= ASCII}', "");
Expect(0, 127, '\P{Blk= ASCII}', "");
Expect(1, 127, '\P{^Blk= ASCII}', "");
Expect(0, 128, '\p{Blk= ASCII}', "");
Expect(1, 128, '\p{^Blk= ASCII}', "");
Expect(1, 128, '\P{Blk= ASCII}', "");
Expect(0, 128, '\P{^Blk= ASCII}', "");
Error('\p{Is_Block= /a/Basiclatin}');
Error('\P{Is_Block= /a/Basiclatin}');
Expect(1, 127, '\p{Is_Block=basiclatin}', "");
Expect(0, 127, '\p{^Is_Block=basiclatin}', "");
Expect(0, 127, '\P{Is_Block=basiclatin}', "");
Expect(1, 127, '\P{^Is_Block=basiclatin}', "");
Expect(0, 128, '\p{Is_Block=basiclatin}', "");
Expect(1, 128, '\p{^Is_Block=basiclatin}', "");
Expect(1, 128, '\P{Is_Block=basiclatin}', "");
Expect(0, 128, '\P{^Is_Block=basiclatin}', "");
Expect(1, 127, '\p{Is_Block=- Basic	Latin}', "");
Expect(0, 127, '\p{^Is_Block=- Basic	Latin}', "");
Expect(0, 127, '\P{Is_Block=- Basic	Latin}', "");
Expect(1, 127, '\P{^Is_Block=- Basic	Latin}', "");
Expect(0, 128, '\p{Is_Block=- Basic	Latin}', "");
Expect(1, 128, '\p{^Is_Block=- Basic	Latin}', "");
Expect(1, 128, '\P{Is_Block=- Basic	Latin}', "");
Expect(0, 128, '\P{^Is_Block=- Basic	Latin}', "");
Error('\p{Is_Blk=/a/ASCII}');
Error('\P{Is_Blk=/a/ASCII}');
Expect(1, 127, '\p{Is_Blk=ascii}', "");
Expect(0, 127, '\p{^Is_Blk=ascii}', "");
Expect(0, 127, '\P{Is_Blk=ascii}', "");
Expect(1, 127, '\P{^Is_Blk=ascii}', "");
Expect(0, 128, '\p{Is_Blk=ascii}', "");
Expect(1, 128, '\p{^Is_Blk=ascii}', "");
Expect(1, 128, '\P{Is_Blk=ascii}', "");
Expect(0, 128, '\P{^Is_Blk=ascii}', "");
Expect(1, 127, '\p{Is_Blk:	 -ASCII}', "");
Expect(0, 127, '\p{^Is_Blk:	 -ASCII}', "");
Expect(0, 127, '\P{Is_Blk:	 -ASCII}', "");
Expect(1, 127, '\P{^Is_Blk:	 -ASCII}', "");
Expect(0, 128, '\p{Is_Blk:	 -ASCII}', "");
Expect(1, 128, '\p{^Is_Blk:	 -ASCII}', "");
Expect(1, 128, '\P{Is_Blk:	 -ASCII}', "");
Expect(0, 128, '\P{^Is_Blk:	 -ASCII}', "");
Error('\p{Block=/a/_	Limbu}');
Error('\P{Block=/a/_	Limbu}');
Expect(1, 6479, '\p{Block=limbu}', "");
Expect(0, 6479, '\p{^Block=limbu}', "");
Expect(0, 6479, '\P{Block=limbu}', "");
Expect(1, 6479, '\P{^Block=limbu}', "");
Expect(0, 6399, '\p{Block=limbu}', "");
Expect(1, 6399, '\p{^Block=limbu}', "");
Expect(1, 6399, '\P{Block=limbu}', "");
Expect(0, 6399, '\P{^Block=limbu}', "");
Expect(1, 6479, '\p{Block=- LIMBU}', "");
Expect(0, 6479, '\p{^Block=- LIMBU}', "");
Expect(0, 6479, '\P{Block=- LIMBU}', "");
Expect(1, 6479, '\P{^Block=- LIMBU}', "");
Expect(0, 6399, '\p{Block=- LIMBU}', "");
Expect(1, 6399, '\p{^Block=- LIMBU}', "");
Expect(1, 6399, '\P{Block=- LIMBU}', "");
Expect(0, 6399, '\P{^Block=- LIMBU}', "");
Error('\p{Blk=-:=Limbu}');
Error('\P{Blk=-:=Limbu}');
Expect(1, 6479, '\p{Blk=limbu}', "");
Expect(0, 6479, '\p{^Blk=limbu}', "");
Expect(0, 6479, '\P{Blk=limbu}', "");
Expect(1, 6479, '\P{^Blk=limbu}', "");
Expect(0, 6399, '\p{Blk=limbu}', "");
Expect(1, 6399, '\p{^Blk=limbu}', "");
Expect(1, 6399, '\P{Blk=limbu}', "");
Expect(0, 6399, '\P{^Blk=limbu}', "");
Expect(1, 6479, '\p{Blk=		Limbu}', "");
Expect(0, 6479, '\p{^Blk=		Limbu}', "");
Expect(0, 6479, '\P{Blk=		Limbu}', "");
Expect(1, 6479, '\P{^Blk=		Limbu}', "");
Expect(0, 6399, '\p{Blk=		Limbu}', "");
Expect(1, 6399, '\p{^Blk=		Limbu}', "");
Expect(1, 6399, '\P{Blk=		Limbu}', "");
Expect(0, 6399, '\P{^Blk=		Limbu}', "");
Error('\p{Is_Block=Limbu/a/}');
Error('\P{Is_Block=Limbu/a/}');
Expect(1, 6479, '\p{Is_Block=limbu}', "");
Expect(0, 6479, '\p{^Is_Block=limbu}', "");
Expect(0, 6479, '\P{Is_Block=limbu}', "");
Expect(1, 6479, '\P{^Is_Block=limbu}', "");
Expect(0, 6399, '\p{Is_Block=limbu}', "");
Expect(1, 6399, '\p{^Is_Block=limbu}', "");
Expect(1, 6399, '\P{Is_Block=limbu}', "");
Expect(0, 6399, '\P{^Is_Block=limbu}', "");
Expect(1, 6479, '\p{Is_Block= Limbu}', "");
Expect(0, 6479, '\p{^Is_Block= Limbu}', "");
Expect(0, 6479, '\P{Is_Block= Limbu}', "");
Expect(1, 6479, '\P{^Is_Block= Limbu}', "");
Expect(0, 6399, '\p{Is_Block= Limbu}', "");
Expect(1, 6399, '\p{^Is_Block= Limbu}', "");
Expect(1, 6399, '\P{Is_Block= Limbu}', "");
Expect(0, 6399, '\P{^Is_Block= Limbu}', "");
Error('\p{Is_Blk=  limbu:=}');
Error('\P{Is_Blk=  limbu:=}');
Expect(1, 6479, '\p{Is_Blk=limbu}', "");
Expect(0, 6479, '\p{^Is_Blk=limbu}', "");
Expect(0, 6479, '\P{Is_Blk=limbu}', "");
Expect(1, 6479, '\P{^Is_Blk=limbu}', "");
Expect(0, 6399, '\p{Is_Blk=limbu}', "");
Expect(1, 6399, '\p{^Is_Blk=limbu}', "");
Expect(1, 6399, '\P{Is_Blk=limbu}', "");
Expect(0, 6399, '\P{^Is_Blk=limbu}', "");
Expect(1, 6479, '\p{Is_Blk=-Limbu}', "");
Expect(0, 6479, '\p{^Is_Blk=-Limbu}', "");
Expect(0, 6479, '\P{Is_Blk=-Limbu}', "");
Expect(1, 6479, '\P{^Is_Blk=-Limbu}', "");
Expect(0, 6399, '\p{Is_Blk=-Limbu}', "");
Expect(1, 6399, '\p{^Is_Blk=-Limbu}', "");
Expect(1, 6399, '\P{Is_Blk=-Limbu}', "");
Expect(0, 6399, '\P{^Is_Blk=-Limbu}', "");
Error('\p{Block=currency/a/SYMBOLS}');
Error('\P{Block=currency/a/SYMBOLS}');
Expect(1, 8399, '\p{Block=currencysymbols}', "");
Expect(0, 8399, '\p{^Block=currencysymbols}', "");
Expect(0, 8399, '\P{Block=currencysymbols}', "");
Expect(1, 8399, '\P{^Block=currencysymbols}', "");
Expect(0, 8351, '\p{Block=currencysymbols}', "");
Expect(1, 8351, '\p{^Block=currencysymbols}', "");
Expect(1, 8351, '\P{Block=currencysymbols}', "");
Expect(0, 8351, '\P{^Block=currencysymbols}', "");
Expect(1, 8399, '\p{Block=_ CURRENCY	symbols}', "");
Expect(0, 8399, '\p{^Block=_ CURRENCY	symbols}', "");
Expect(0, 8399, '\P{Block=_ CURRENCY	symbols}', "");
Expect(1, 8399, '\P{^Block=_ CURRENCY	symbols}', "");
Expect(0, 8351, '\p{Block=_ CURRENCY	symbols}', "");
Expect(1, 8351, '\p{^Block=_ CURRENCY	symbols}', "");
Expect(1, 8351, '\P{Block=_ CURRENCY	symbols}', "");
Expect(0, 8351, '\P{^Block=_ CURRENCY	symbols}', "");
Error('\p{Blk=/a/	-Currency_symbols}');
Error('\P{Blk=/a/	-Currency_symbols}');
Expect(1, 8399, '\p{Blk=currencysymbols}', "");
Expect(0, 8399, '\p{^Blk=currencysymbols}', "");
Expect(0, 8399, '\P{Blk=currencysymbols}', "");
Expect(1, 8399, '\P{^Blk=currencysymbols}', "");
Expect(0, 8351, '\p{Blk=currencysymbols}', "");
Expect(1, 8351, '\p{^Blk=currencysymbols}', "");
Expect(1, 8351, '\P{Blk=currencysymbols}', "");
Expect(0, 8351, '\P{^Blk=currencysymbols}', "");
Expect(1, 8399, '\p{Blk= -CURRENCY	Symbols}', "");
Expect(0, 8399, '\p{^Blk= -CURRENCY	Symbols}', "");
Expect(0, 8399, '\P{Blk= -CURRENCY	Symbols}', "");
Expect(1, 8399, '\P{^Blk= -CURRENCY	Symbols}', "");
Expect(0, 8351, '\p{Blk= -CURRENCY	Symbols}', "");
Expect(1, 8351, '\p{^Blk= -CURRENCY	Symbols}', "");
Expect(1, 8351, '\P{Blk= -CURRENCY	Symbols}', "");
Expect(0, 8351, '\P{^Blk= -CURRENCY	Symbols}', "");
Error('\p{Is_Block=CURRENCY:=Symbols}');
Error('\P{Is_Block=CURRENCY:=Symbols}');
Expect(1, 8399, '\p{Is_Block=currencysymbols}', "");
Expect(0, 8399, '\p{^Is_Block=currencysymbols}', "");
Expect(0, 8399, '\P{Is_Block=currencysymbols}', "");
Expect(1, 8399, '\P{^Is_Block=currencysymbols}', "");
Expect(0, 8351, '\p{Is_Block=currencysymbols}', "");
Expect(1, 8351, '\p{^Is_Block=currencysymbols}', "");
Expect(1, 8351, '\P{Is_Block=currencysymbols}', "");
Expect(0, 8351, '\P{^Is_Block=currencysymbols}', "");
Expect(1, 8399, '\p{Is_Block: -	Currency-SYMBOLS}', "");
Expect(0, 8399, '\p{^Is_Block: -	Currency-SYMBOLS}', "");
Expect(0, 8399, '\P{Is_Block: -	Currency-SYMBOLS}', "");
Expect(1, 8399, '\P{^Is_Block: -	Currency-SYMBOLS}', "");
Expect(0, 8351, '\p{Is_Block: -	Currency-SYMBOLS}', "");
Expect(1, 8351, '\p{^Is_Block: -	Currency-SYMBOLS}', "");
Expect(1, 8351, '\P{Is_Block: -	Currency-SYMBOLS}', "");
Expect(0, 8351, '\P{^Is_Block: -	Currency-SYMBOLS}', "");
Error('\p{Is_Blk=	/a/currency symbols}');
Error('\P{Is_Blk=	/a/currency symbols}');
Expect(1, 8399, '\p{Is_Blk=currencysymbols}', "");
Expect(0, 8399, '\p{^Is_Blk=currencysymbols}', "");
Expect(0, 8399, '\P{Is_Blk=currencysymbols}', "");
Expect(1, 8399, '\P{^Is_Blk=currencysymbols}', "");
Expect(0, 8351, '\p{Is_Blk=currencysymbols}', "");
Expect(1, 8351, '\p{^Is_Blk=currencysymbols}', "");
Expect(1, 8351, '\P{Is_Blk=currencysymbols}', "");
Expect(0, 8351, '\P{^Is_Blk=currencysymbols}', "");
Expect(1, 8399, '\p{Is_Blk=-	CURRENCY-Symbols}', "");
Expect(0, 8399, '\p{^Is_Blk=-	CURRENCY-Symbols}', "");
Expect(0, 8399, '\P{Is_Blk=-	CURRENCY-Symbols}', "");
Expect(1, 8399, '\P{^Is_Blk=-	CURRENCY-Symbols}', "");
Expect(0, 8351, '\p{Is_Blk=-	CURRENCY-Symbols}', "");
Expect(1, 8351, '\p{^Is_Blk=-	CURRENCY-Symbols}', "");
Expect(1, 8351, '\P{Is_Blk=-	CURRENCY-Symbols}', "");
Expect(0, 8351, '\P{^Is_Blk=-	CURRENCY-Symbols}', "");
Error('\p{Block=/a/_spacing Modifier-Letters}');
Error('\P{Block=/a/_spacing Modifier-Letters}');
Expect(1, 767, '\p{Block=spacingmodifierletters}', "");
Expect(0, 767, '\p{^Block=spacingmodifierletters}', "");
Expect(0, 767, '\P{Block=spacingmodifierletters}', "");
Expect(1, 767, '\P{^Block=spacingmodifierletters}', "");
Expect(0, 687, '\p{Block=spacingmodifierletters}', "");
Expect(1, 687, '\p{^Block=spacingmodifierletters}', "");
Expect(1, 687, '\P{Block=spacingmodifierletters}', "");
Expect(0, 687, '\P{^Block=spacingmodifierletters}', "");
Expect(1, 767, '\p{Block=_SPACING Modifier Letters}', "");
Expect(0, 767, '\p{^Block=_SPACING Modifier Letters}', "");
Expect(0, 767, '\P{Block=_SPACING Modifier Letters}', "");
Expect(1, 767, '\P{^Block=_SPACING Modifier Letters}', "");
Expect(0, 687, '\p{Block=_SPACING Modifier Letters}', "");
Expect(1, 687, '\p{^Block=_SPACING Modifier Letters}', "");
Expect(1, 687, '\P{Block=_SPACING Modifier Letters}', "");
Expect(0, 687, '\P{^Block=_SPACING Modifier Letters}', "");
Error('\p{Blk= :=Spacingmodifier	LETTERS}');
Error('\P{Blk= :=Spacingmodifier	LETTERS}');
Expect(1, 767, '\p{Blk=spacingmodifierletters}', "");
Expect(0, 767, '\p{^Blk=spacingmodifierletters}', "");
Expect(0, 767, '\P{Blk=spacingmodifierletters}', "");
Expect(1, 767, '\P{^Blk=spacingmodifierletters}', "");
Expect(0, 687, '\p{Blk=spacingmodifierletters}', "");
Expect(1, 687, '\p{^Blk=spacingmodifierletters}', "");
Expect(1, 687, '\P{Blk=spacingmodifierletters}', "");
Expect(0, 687, '\P{^Blk=spacingmodifierletters}', "");
Expect(1, 767, '\p{Blk= Spacing_MODIFIER-LETTERS}', "");
Expect(0, 767, '\p{^Blk= Spacing_MODIFIER-LETTERS}', "");
Expect(0, 767, '\P{Blk= Spacing_MODIFIER-LETTERS}', "");
Expect(1, 767, '\P{^Blk= Spacing_MODIFIER-LETTERS}', "");
Expect(0, 687, '\p{Blk= Spacing_MODIFIER-LETTERS}', "");
Expect(1, 687, '\p{^Blk= Spacing_MODIFIER-LETTERS}', "");
Expect(1, 687, '\P{Blk= Spacing_MODIFIER-LETTERS}', "");
Expect(0, 687, '\P{^Blk= Spacing_MODIFIER-LETTERS}', "");
Error('\p{Is_Block=_SPACING-modifier	Letters:=}');
Error('\P{Is_Block=_SPACING-modifier	Letters:=}');
Expect(1, 767, '\p{Is_Block=spacingmodifierletters}', "");
Expect(0, 767, '\p{^Is_Block=spacingmodifierletters}', "");
Expect(0, 767, '\P{Is_Block=spacingmodifierletters}', "");
Expect(1, 767, '\P{^Is_Block=spacingmodifierletters}', "");
Expect(0, 687, '\p{Is_Block=spacingmodifierletters}', "");
Expect(1, 687, '\p{^Is_Block=spacingmodifierletters}', "");
Expect(1, 687, '\P{Is_Block=spacingmodifierletters}', "");
Expect(0, 687, '\P{^Is_Block=spacingmodifierletters}', "");
Expect(1, 767, '\p{Is_Block=-	Spacingmodifier LETTERS}', "");
Expect(0, 767, '\p{^Is_Block=-	Spacingmodifier LETTERS}', "");
Expect(0, 767, '\P{Is_Block=-	Spacingmodifier LETTERS}', "");
Expect(1, 767, '\P{^Is_Block=-	Spacingmodifier LETTERS}', "");
Expect(0, 687, '\p{Is_Block=-	Spacingmodifier LETTERS}', "");
Expect(1, 687, '\p{^Is_Block=-	Spacingmodifier LETTERS}', "");
Expect(1, 687, '\P{Is_Block=-	Spacingmodifier LETTERS}', "");
Expect(0, 687, '\P{^Is_Block=-	Spacingmodifier LETTERS}', "");
Error('\p{Is_Blk: :=Spacing	MODIFIER	LETTERS}');
Error('\P{Is_Blk: :=Spacing	MODIFIER	LETTERS}');
Expect(1, 767, '\p{Is_Blk=spacingmodifierletters}', "");
Expect(0, 767, '\p{^Is_Blk=spacingmodifierletters}', "");
Expect(0, 767, '\P{Is_Blk=spacingmodifierletters}', "");
Expect(1, 767, '\P{^Is_Blk=spacingmodifierletters}', "");
Expect(0, 687, '\p{Is_Blk=spacingmodifierletters}', "");
Expect(1, 687, '\p{^Is_Blk=spacingmodifierletters}', "");
Expect(1, 687, '\P{Is_Blk=spacingmodifierletters}', "");
Expect(0, 687, '\P{^Is_Blk=spacingmodifierletters}', "");
Expect(1, 767, '\p{Is_Blk=  Spacing_modifier Letters}', "");
Expect(0, 767, '\p{^Is_Blk=  Spacing_modifier Letters}', "");
Expect(0, 767, '\P{Is_Blk=  Spacing_modifier Letters}', "");
Expect(1, 767, '\P{^Is_Blk=  Spacing_modifier Letters}', "");
Expect(0, 687, '\p{Is_Blk=  Spacing_modifier Letters}', "");
Expect(1, 687, '\p{^Is_Blk=  Spacing_modifier Letters}', "");
Expect(1, 687, '\P{Is_Blk=  Spacing_modifier Letters}', "");
Expect(0, 687, '\P{^Is_Blk=  Spacing_modifier Letters}', "");
Error('\p{Block=:=	box	DRAWING}');
Error('\P{Block=:=	box	DRAWING}');
Expect(1, 9599, '\p{Block:boxdrawing}', "");
Expect(0, 9599, '\p{^Block:boxdrawing}', "");
Expect(0, 9599, '\P{Block:boxdrawing}', "");
Expect(1, 9599, '\P{^Block:boxdrawing}', "");
Expect(0, 9471, '\p{Block:boxdrawing}', "");
Expect(1, 9471, '\p{^Block:boxdrawing}', "");
Expect(1, 9471, '\P{Block:boxdrawing}', "");
Expect(0, 9471, '\P{^Block:boxdrawing}', "");
Expect(1, 9599, '\p{Block=	 BOX_Drawing}', "");
Expect(0, 9599, '\p{^Block=	 BOX_Drawing}', "");
Expect(0, 9599, '\P{Block=	 BOX_Drawing}', "");
Expect(1, 9599, '\P{^Block=	 BOX_Drawing}', "");
Expect(0, 9471, '\p{Block=	 BOX_Drawing}', "");
Expect(1, 9471, '\p{^Block=	 BOX_Drawing}', "");
Expect(1, 9471, '\P{Block=	 BOX_Drawing}', "");
Expect(0, 9471, '\P{^Block=	 BOX_Drawing}', "");
Error('\p{Blk:	 -Box:=drawing}');
Error('\P{Blk:	 -Box:=drawing}');
Expect(1, 9599, '\p{Blk=boxdrawing}', "");
Expect(0, 9599, '\p{^Blk=boxdrawing}', "");
Expect(0, 9599, '\P{Blk=boxdrawing}', "");
Expect(1, 9599, '\P{^Blk=boxdrawing}', "");
Expect(0, 9471, '\p{Blk=boxdrawing}', "");
Expect(1, 9471, '\p{^Blk=boxdrawing}', "");
Expect(1, 9471, '\P{Blk=boxdrawing}', "");
Expect(0, 9471, '\P{^Blk=boxdrawing}', "");
Expect(1, 9599, '\p{Blk:   _BoxDrawing}', "");
Expect(0, 9599, '\p{^Blk:   _BoxDrawing}', "");
Expect(0, 9599, '\P{Blk:   _BoxDrawing}', "");
Expect(1, 9599, '\P{^Blk:   _BoxDrawing}', "");
Expect(0, 9471, '\p{Blk:   _BoxDrawing}', "");
Expect(1, 9471, '\p{^Blk:   _BoxDrawing}', "");
Expect(1, 9471, '\P{Blk:   _BoxDrawing}', "");
Expect(0, 9471, '\P{^Blk:   _BoxDrawing}', "");
Error('\p{Is_Block:	/a/-	boxDrawing}');
Error('\P{Is_Block:	/a/-	boxDrawing}');
Expect(1, 9599, '\p{Is_Block=boxdrawing}', "");
Expect(0, 9599, '\p{^Is_Block=boxdrawing}', "");
Expect(0, 9599, '\P{Is_Block=boxdrawing}', "");
Expect(1, 9599, '\P{^Is_Block=boxdrawing}', "");
Expect(0, 9471, '\p{Is_Block=boxdrawing}', "");
Expect(1, 9471, '\p{^Is_Block=boxdrawing}', "");
Expect(1, 9471, '\P{Is_Block=boxdrawing}', "");
Expect(0, 9471, '\P{^Is_Block=boxdrawing}', "");
Expect(1, 9599, '\p{Is_Block=_BOX_drawing}', "");
Expect(0, 9599, '\p{^Is_Block=_BOX_drawing}', "");
Expect(0, 9599, '\P{Is_Block=_BOX_drawing}', "");
Expect(1, 9599, '\P{^Is_Block=_BOX_drawing}', "");
Expect(0, 9471, '\p{Is_Block=_BOX_drawing}', "");
Expect(1, 9471, '\p{^Is_Block=_BOX_drawing}', "");
Expect(1, 9471, '\P{Is_Block=_BOX_drawing}', "");
Expect(0, 9471, '\P{^Is_Block=_BOX_drawing}', "");
Error('\p{Is_Blk=_:=Box	Drawing}');
Error('\P{Is_Blk=_:=Box	Drawing}');
Expect(1, 9599, '\p{Is_Blk=boxdrawing}', "");
Expect(0, 9599, '\p{^Is_Blk=boxdrawing}', "");
Expect(0, 9599, '\P{Is_Blk=boxdrawing}', "");
Expect(1, 9599, '\P{^Is_Blk=boxdrawing}', "");
Expect(0, 9471, '\p{Is_Blk=boxdrawing}', "");
Expect(1, 9471, '\p{^Is_Blk=boxdrawing}', "");
Expect(1, 9471, '\P{Is_Blk=boxdrawing}', "");
Expect(0, 9471, '\P{^Is_Blk=boxdrawing}', "");
Expect(1, 9599, '\p{Is_Blk=	BOX Drawing}', "");
Expect(0, 9599, '\p{^Is_Blk=	BOX Drawing}', "");
Expect(0, 9599, '\P{Is_Blk=	BOX Drawing}', "");
Expect(1, 9599, '\P{^Is_Blk=	BOX Drawing}', "");
Expect(0, 9471, '\p{Is_Blk=	BOX Drawing}', "");
Expect(1, 9471, '\p{^Is_Blk=	BOX Drawing}', "");
Expect(1, 9471, '\P{Is_Blk=	BOX Drawing}', "");
Expect(0, 9471, '\P{^Is_Blk=	BOX Drawing}', "");
Error('\p{Block=-Superscripts AND:=Subscripts}');
Error('\P{Block=-Superscripts AND:=Subscripts}');
Expect(1, 8351, '\p{Block=superscriptsandsubscripts}', "");
Expect(0, 8351, '\p{^Block=superscriptsandsubscripts}', "");
Expect(0, 8351, '\P{Block=superscriptsandsubscripts}', "");
Expect(1, 8351, '\P{^Block=superscriptsandsubscripts}', "");
Expect(0, 8303, '\p{Block=superscriptsandsubscripts}', "");
Expect(1, 8303, '\p{^Block=superscriptsandsubscripts}', "");
Expect(1, 8303, '\P{Block=superscriptsandsubscripts}', "");
Expect(0, 8303, '\P{^Block=superscriptsandsubscripts}', "");
Expect(1, 8351, '\p{Block=__Superscripts-And-subscripts}', "");
Expect(0, 8351, '\p{^Block=__Superscripts-And-subscripts}', "");
Expect(0, 8351, '\P{Block=__Superscripts-And-subscripts}', "");
Expect(1, 8351, '\P{^Block=__Superscripts-And-subscripts}', "");
Expect(0, 8303, '\p{Block=__Superscripts-And-subscripts}', "");
Expect(1, 8303, '\p{^Block=__Superscripts-And-subscripts}', "");
Expect(1, 8303, '\P{Block=__Superscripts-And-subscripts}', "");
Expect(0, 8303, '\P{^Block=__Superscripts-And-subscripts}', "");
Error('\p{Blk=-	SUPERSCRIPTSAND:=Subscripts}');
Error('\P{Blk=-	SUPERSCRIPTSAND:=Subscripts}');
Expect(1, 8351, '\p{Blk: superscriptsandsubscripts}', "");
Expect(0, 8351, '\p{^Blk: superscriptsandsubscripts}', "");
Expect(0, 8351, '\P{Blk: superscriptsandsubscripts}', "");
Expect(1, 8351, '\P{^Blk: superscriptsandsubscripts}', "");
Expect(0, 8303, '\p{Blk: superscriptsandsubscripts}', "");
Expect(1, 8303, '\p{^Blk: superscriptsandsubscripts}', "");
Expect(1, 8303, '\P{Blk: superscriptsandsubscripts}', "");
Expect(0, 8303, '\P{^Blk: superscriptsandsubscripts}', "");
Expect(1, 8351, '\p{Blk=-Superscripts	AND	Subscripts}', "");
Expect(0, 8351, '\p{^Blk=-Superscripts	AND	Subscripts}', "");
Expect(0, 8351, '\P{Blk=-Superscripts	AND	Subscripts}', "");
Expect(1, 8351, '\P{^Blk=-Superscripts	AND	Subscripts}', "");
Expect(0, 8303, '\p{Blk=-Superscripts	AND	Subscripts}', "");
Expect(1, 8303, '\p{^Blk=-Superscripts	AND	Subscripts}', "");
Expect(1, 8303, '\P{Blk=-Superscripts	AND	Subscripts}', "");
Expect(0, 8303, '\P{^Blk=-Superscripts	AND	Subscripts}', "");
Error('\p{Is_Block:	 Superscripts	AND	SUBSCRIPTS:=}');
Error('\P{Is_Block:	 Superscripts	AND	SUBSCRIPTS:=}');
Expect(1, 8351, '\p{Is_Block=superscriptsandsubscripts}', "");
Expect(0, 8351, '\p{^Is_Block=superscriptsandsubscripts}', "");
Expect(0, 8351, '\P{Is_Block=superscriptsandsubscripts}', "");
Expect(1, 8351, '\P{^Is_Block=superscriptsandsubscripts}', "");
Expect(0, 8303, '\p{Is_Block=superscriptsandsubscripts}', "");
Expect(1, 8303, '\p{^Is_Block=superscriptsandsubscripts}', "");
Expect(1, 8303, '\P{Is_Block=superscriptsandsubscripts}', "");
Expect(0, 8303, '\P{^Is_Block=superscriptsandsubscripts}', "");
Expect(1, 8351, '\p{Is_Block=- Superscripts	And	SUBSCRIPTS}', "");
Expect(0, 8351, '\p{^Is_Block=- Superscripts	And	SUBSCRIPTS}', "");
Expect(0, 8351, '\P{Is_Block=- Superscripts	And	SUBSCRIPTS}', "");
Expect(1, 8351, '\P{^Is_Block=- Superscripts	And	SUBSCRIPTS}', "");
Expect(0, 8303, '\p{Is_Block=- Superscripts	And	SUBSCRIPTS}', "");
Expect(1, 8303, '\p{^Is_Block=- Superscripts	And	SUBSCRIPTS}', "");
Expect(1, 8303, '\P{Is_Block=- Superscripts	And	SUBSCRIPTS}', "");
Expect(0, 8303, '\P{^Is_Block=- Superscripts	And	SUBSCRIPTS}', "");
Error('\p{Is_Blk=SUPERSCRIPTS/a/AndSubscripts}');
Error('\P{Is_Blk=SUPERSCRIPTS/a/AndSubscripts}');
Expect(1, 8351, '\p{Is_Blk=superscriptsandsubscripts}', "");
Expect(0, 8351, '\p{^Is_Blk=superscriptsandsubscripts}', "");
Expect(0, 8351, '\P{Is_Blk=superscriptsandsubscripts}', "");
Expect(1, 8351, '\P{^Is_Blk=superscriptsandsubscripts}', "");
Expect(0, 8303, '\p{Is_Blk=superscriptsandsubscripts}', "");
Expect(1, 8303, '\p{^Is_Blk=superscriptsandsubscripts}', "");
Expect(1, 8303, '\P{Is_Blk=superscriptsandsubscripts}', "");
Expect(0, 8303, '\P{^Is_Blk=superscriptsandsubscripts}', "");
Expect(1, 8351, '\p{Is_Blk=-_Superscripts	AND	SUBSCRIPTS}', "");
Expect(0, 8351, '\p{^Is_Blk=-_Superscripts	AND	SUBSCRIPTS}', "");
Expect(0, 8351, '\P{Is_Blk=-_Superscripts	AND	SUBSCRIPTS}', "");
Expect(1, 8351, '\P{^Is_Blk=-_Superscripts	AND	SUBSCRIPTS}', "");
Expect(0, 8303, '\p{Is_Blk=-_Superscripts	AND	SUBSCRIPTS}', "");
Expect(1, 8303, '\p{^Is_Blk=-_Superscripts	AND	SUBSCRIPTS}', "");
Expect(1, 8303, '\P{Is_Blk=-_Superscripts	AND	SUBSCRIPTS}', "");
Expect(0, 8303, '\P{^Is_Blk=-_Superscripts	AND	SUBSCRIPTS}', "");
Error('\p{Block:	 -ancient greek Numbers:=}');
Error('\P{Block:	 -ancient greek Numbers:=}');
Expect(1, 65935, '\p{Block=ancientgreeknumbers}', "");
Expect(0, 65935, '\p{^Block=ancientgreeknumbers}', "");
Expect(0, 65935, '\P{Block=ancientgreeknumbers}', "");
Expect(1, 65935, '\P{^Block=ancientgreeknumbers}', "");
Expect(0, 65855, '\p{Block=ancientgreeknumbers}', "");
Expect(1, 65855, '\p{^Block=ancientgreeknumbers}', "");
Expect(1, 65855, '\P{Block=ancientgreeknumbers}', "");
Expect(0, 65855, '\P{^Block=ancientgreeknumbers}', "");
Expect(1, 65935, '\p{Block:- ancientGreek_numbers}', "");
Expect(0, 65935, '\p{^Block:- ancientGreek_numbers}', "");
Expect(0, 65935, '\P{Block:- ancientGreek_numbers}', "");
Expect(1, 65935, '\P{^Block:- ancientGreek_numbers}', "");
Expect(0, 65855, '\p{Block:- ancientGreek_numbers}', "");
Expect(1, 65855, '\p{^Block:- ancientGreek_numbers}', "");
Expect(1, 65855, '\P{Block:- ancientGreek_numbers}', "");
Expect(0, 65855, '\P{^Block:- ancientGreek_numbers}', "");
Error('\p{Blk=__Ancient-GREEK:=Numbers}');
Error('\P{Blk=__Ancient-GREEK:=Numbers}');
Expect(1, 65935, '\p{Blk=ancientgreeknumbers}', "");
Expect(0, 65935, '\p{^Blk=ancientgreeknumbers}', "");
Expect(0, 65935, '\P{Blk=ancientgreeknumbers}', "");
Expect(1, 65935, '\P{^Blk=ancientgreeknumbers}', "");
Expect(0, 65855, '\p{Blk=ancientgreeknumbers}', "");
Expect(1, 65855, '\p{^Blk=ancientgreeknumbers}', "");
Expect(1, 65855, '\P{Blk=ancientgreeknumbers}', "");
Expect(0, 65855, '\P{^Blk=ancientgreeknumbers}', "");
Expect(1, 65935, '\p{Blk=--ANCIENT-GREEK Numbers}', "");
Expect(0, 65935, '\p{^Blk=--ANCIENT-GREEK Numbers}', "");
Expect(0, 65935, '\P{Blk=--ANCIENT-GREEK Numbers}', "");
Expect(1, 65935, '\P{^Blk=--ANCIENT-GREEK Numbers}', "");
Expect(0, 65855, '\p{Blk=--ANCIENT-GREEK Numbers}', "");
Expect(1, 65855, '\p{^Blk=--ANCIENT-GREEK Numbers}', "");
Expect(1, 65855, '\P{Blk=--ANCIENT-GREEK Numbers}', "");
Expect(0, 65855, '\P{^Blk=--ANCIENT-GREEK Numbers}', "");
Error('\p{Is_Block=/a/	Ancient	greekNUMBERS}');
Error('\P{Is_Block=/a/	Ancient	greekNUMBERS}');
Expect(1, 65935, '\p{Is_Block=ancientgreeknumbers}', "");
Expect(0, 65935, '\p{^Is_Block=ancientgreeknumbers}', "");
Expect(0, 65935, '\P{Is_Block=ancientgreeknumbers}', "");
Expect(1, 65935, '\P{^Is_Block=ancientgreeknumbers}', "");
Expect(0, 65855, '\p{Is_Block=ancientgreeknumbers}', "");
Expect(1, 65855, '\p{^Is_Block=ancientgreeknumbers}', "");
Expect(1, 65855, '\P{Is_Block=ancientgreeknumbers}', "");
Expect(0, 65855, '\P{^Is_Block=ancientgreeknumbers}', "");
Expect(1, 65935, '\p{Is_Block:		Ancient_GREEK Numbers}', "");
Expect(0, 65935, '\p{^Is_Block:		Ancient_GREEK Numbers}', "");
Expect(0, 65935, '\P{Is_Block:		Ancient_GREEK Numbers}', "");
Expect(1, 65935, '\P{^Is_Block:		Ancient_GREEK Numbers}', "");
Expect(0, 65855, '\p{Is_Block:		Ancient_GREEK Numbers}', "");
Expect(1, 65855, '\p{^Is_Block:		Ancient_GREEK Numbers}', "");
Expect(1, 65855, '\P{Is_Block:		Ancient_GREEK Numbers}', "");
Expect(0, 65855, '\P{^Is_Block:		Ancient_GREEK Numbers}', "");
Error('\p{Is_Blk= /a/ancient greek-Numbers}');
Error('\P{Is_Blk= /a/ancient greek-Numbers}');
Expect(1, 65935, '\p{Is_Blk=ancientgreeknumbers}', "");
Expect(0, 65935, '\p{^Is_Blk=ancientgreeknumbers}', "");
Expect(0, 65935, '\P{Is_Blk=ancientgreeknumbers}', "");
Expect(1, 65935, '\P{^Is_Blk=ancientgreeknumbers}', "");
Expect(0, 65855, '\p{Is_Blk=ancientgreeknumbers}', "");
Expect(1, 65855, '\p{^Is_Blk=ancientgreeknumbers}', "");
Expect(1, 65855, '\P{Is_Blk=ancientgreeknumbers}', "");
Expect(0, 65855, '\P{^Is_Blk=ancientgreeknumbers}', "");
Expect(1, 65935, '\p{Is_Blk=_Ancient_greek-Numbers}', "");
Expect(0, 65935, '\p{^Is_Blk=_Ancient_greek-Numbers}', "");
Expect(0, 65935, '\P{Is_Blk=_Ancient_greek-Numbers}', "");
Expect(1, 65935, '\P{^Is_Blk=_Ancient_greek-Numbers}', "");
Expect(0, 65855, '\p{Is_Blk=_Ancient_greek-Numbers}', "");
Expect(1, 65855, '\p{^Is_Blk=_Ancient_greek-Numbers}', "");
Expect(1, 65855, '\P{Is_Blk=_Ancient_greek-Numbers}', "");
Expect(0, 65855, '\P{^Is_Blk=_Ancient_greek-Numbers}', "");
Error('\p{Block=	/a/NKO}');
Error('\P{Block=	/a/NKO}');
Expect(1, 2047, '\p{Block=nko}', "");
Expect(0, 2047, '\p{^Block=nko}', "");
Expect(0, 2047, '\P{Block=nko}', "");
Expect(1, 2047, '\P{^Block=nko}', "");
Expect(0, 1983, '\p{Block=nko}', "");
Expect(1, 1983, '\p{^Block=nko}', "");
Expect(1, 1983, '\P{Block=nko}', "");
Expect(0, 1983, '\P{^Block=nko}', "");
Expect(1, 2047, '\p{Block=	 NKo}', "");
Expect(0, 2047, '\p{^Block=	 NKo}', "");
Expect(0, 2047, '\P{Block=	 NKo}', "");
Expect(1, 2047, '\P{^Block=	 NKo}', "");
Expect(0, 1983, '\p{Block=	 NKo}', "");
Expect(1, 1983, '\p{^Block=	 NKo}', "");
Expect(1, 1983, '\P{Block=	 NKo}', "");
Expect(0, 1983, '\P{^Block=	 NKo}', "");
Error('\p{Blk=/a/	_NKO}');
Error('\P{Blk=/a/	_NKO}');
Expect(1, 2047, '\p{Blk=nko}', "");
Expect(0, 2047, '\p{^Blk=nko}', "");
Expect(0, 2047, '\P{Blk=nko}', "");
Expect(1, 2047, '\P{^Blk=nko}', "");
Expect(0, 1983, '\p{Blk=nko}', "");
Expect(1, 1983, '\p{^Blk=nko}', "");
Expect(1, 1983, '\P{Blk=nko}', "");
Expect(0, 1983, '\P{^Blk=nko}', "");
Expect(1, 2047, '\p{Blk= NKo}', "");
Expect(0, 2047, '\p{^Blk= NKo}', "");
Expect(0, 2047, '\P{Blk= NKo}', "");
Expect(1, 2047, '\P{^Blk= NKo}', "");
Expect(0, 1983, '\p{Blk= NKo}', "");
Expect(1, 1983, '\p{^Blk= NKo}', "");
Expect(1, 1983, '\P{Blk= NKo}', "");
Expect(0, 1983, '\P{^Blk= NKo}', "");
Error('\p{Is_Block:   :=  NKo}');
Error('\P{Is_Block:   :=  NKo}');
Expect(1, 2047, '\p{Is_Block=nko}', "");
Expect(0, 2047, '\p{^Is_Block=nko}', "");
Expect(0, 2047, '\P{Is_Block=nko}', "");
Expect(1, 2047, '\P{^Is_Block=nko}', "");
Expect(0, 1983, '\p{Is_Block=nko}', "");
Expect(1, 1983, '\p{^Is_Block=nko}', "");
Expect(1, 1983, '\P{Is_Block=nko}', "");
Expect(0, 1983, '\P{^Is_Block=nko}', "");
Expect(1, 2047, '\p{Is_Block= _NKo}', "");
Expect(0, 2047, '\p{^Is_Block= _NKo}', "");
Expect(0, 2047, '\P{Is_Block= _NKo}', "");
Expect(1, 2047, '\P{^Is_Block= _NKo}', "");
Expect(0, 1983, '\p{Is_Block= _NKo}', "");
Expect(1, 1983, '\p{^Is_Block= _NKo}', "");
Expect(1, 1983, '\P{Is_Block= _NKo}', "");
Expect(0, 1983, '\P{^Is_Block= _NKo}', "");
Error('\p{Is_Blk:   /a/  nko}');
Error('\P{Is_Blk:   /a/  nko}');
Expect(1, 2047, '\p{Is_Blk=nko}', "");
Expect(0, 2047, '\p{^Is_Blk=nko}', "");
Expect(0, 2047, '\P{Is_Blk=nko}', "");
Expect(1, 2047, '\P{^Is_Blk=nko}', "");
Expect(0, 1983, '\p{Is_Blk=nko}', "");
Expect(1, 1983, '\p{^Is_Blk=nko}', "");
Expect(1, 1983, '\P{Is_Blk=nko}', "");
Expect(0, 1983, '\P{^Is_Blk=nko}', "");
Expect(1, 2047, '\p{Is_Blk= -NKo}', "");
Expect(0, 2047, '\p{^Is_Blk= -NKo}', "");
Expect(0, 2047, '\P{Is_Blk= -NKo}', "");
Expect(1, 2047, '\P{^Is_Blk= -NKo}', "");
Expect(0, 1983, '\p{Is_Blk= -NKo}', "");
Expect(1, 1983, '\p{^Is_Blk= -NKo}', "");
Expect(1, 1983, '\P{Is_Blk= -NKo}', "");
Expect(0, 1983, '\P{^Is_Blk= -NKo}', "");
Error('\p{Block=-Musical_Symbols/a/}');
Error('\P{Block=-Musical_Symbols/a/}');
Expect(1, 119295, '\p{Block=musicalsymbols}', "");
Expect(0, 119295, '\p{^Block=musicalsymbols}', "");
Expect(0, 119295, '\P{Block=musicalsymbols}', "");
Expect(1, 119295, '\P{^Block=musicalsymbols}', "");
Expect(0, 119039, '\p{Block=musicalsymbols}', "");
Expect(1, 119039, '\p{^Block=musicalsymbols}', "");
Expect(1, 119039, '\P{Block=musicalsymbols}', "");
Expect(0, 119039, '\P{^Block=musicalsymbols}', "");
Expect(1, 119295, '\p{Block=_ Musical	Symbols}', "");
Expect(0, 119295, '\p{^Block=_ Musical	Symbols}', "");
Expect(0, 119295, '\P{Block=_ Musical	Symbols}', "");
Expect(1, 119295, '\P{^Block=_ Musical	Symbols}', "");
Expect(0, 119039, '\p{Block=_ Musical	Symbols}', "");
Expect(1, 119039, '\p{^Block=_ Musical	Symbols}', "");
Expect(1, 119039, '\P{Block=_ Musical	Symbols}', "");
Expect(0, 119039, '\P{^Block=_ Musical	Symbols}', "");
Error('\p{Blk=:= Musical-SYMBOLS}');
Error('\P{Blk=:= Musical-SYMBOLS}');
Expect(1, 119295, '\p{Blk=musicalsymbols}', "");
Expect(0, 119295, '\p{^Blk=musicalsymbols}', "");
Expect(0, 119295, '\P{Blk=musicalsymbols}', "");
Expect(1, 119295, '\P{^Blk=musicalsymbols}', "");
Expect(0, 119039, '\p{Blk=musicalsymbols}', "");
Expect(1, 119039, '\p{^Blk=musicalsymbols}', "");
Expect(1, 119039, '\P{Blk=musicalsymbols}', "");
Expect(0, 119039, '\P{^Blk=musicalsymbols}', "");
Expect(1, 119295, '\p{Blk=-_MUSICAL Symbols}', "");
Expect(0, 119295, '\p{^Blk=-_MUSICAL Symbols}', "");
Expect(0, 119295, '\P{Blk=-_MUSICAL Symbols}', "");
Expect(1, 119295, '\P{^Blk=-_MUSICAL Symbols}', "");
Expect(0, 119039, '\p{Blk=-_MUSICAL Symbols}', "");
Expect(1, 119039, '\p{^Blk=-_MUSICAL Symbols}', "");
Expect(1, 119039, '\P{Blk=-_MUSICAL Symbols}', "");
Expect(0, 119039, '\P{^Blk=-_MUSICAL Symbols}', "");
Error('\p{Is_Block=/a/-MUSICAL_SYMBOLS}');
Error('\P{Is_Block=/a/-MUSICAL_SYMBOLS}');
Expect(1, 119295, '\p{Is_Block: musicalsymbols}', "");
Expect(0, 119295, '\p{^Is_Block: musicalsymbols}', "");
Expect(0, 119295, '\P{Is_Block: musicalsymbols}', "");
Expect(1, 119295, '\P{^Is_Block: musicalsymbols}', "");
Expect(0, 119039, '\p{Is_Block: musicalsymbols}', "");
Expect(1, 119039, '\p{^Is_Block: musicalsymbols}', "");
Expect(1, 119039, '\P{Is_Block: musicalsymbols}', "");
Expect(0, 119039, '\P{^Is_Block: musicalsymbols}', "");
Expect(1, 119295, '\p{Is_Block:   - musical SYMBOLS}', "");
Expect(0, 119295, '\p{^Is_Block:   - musical SYMBOLS}', "");
Expect(0, 119295, '\P{Is_Block:   - musical SYMBOLS}', "");
Expect(1, 119295, '\P{^Is_Block:   - musical SYMBOLS}', "");
Expect(0, 119039, '\p{Is_Block:   - musical SYMBOLS}', "");
Expect(1, 119039, '\p{^Is_Block:   - musical SYMBOLS}', "");
Expect(1, 119039, '\P{Is_Block:   - musical SYMBOLS}', "");
Expect(0, 119039, '\P{^Is_Block:   - musical SYMBOLS}', "");
Error('\p{Is_Blk=	/a/Musical	symbols}');
Error('\P{Is_Blk=	/a/Musical	symbols}');
Expect(1, 119295, '\p{Is_Blk=musicalsymbols}', "");
Expect(0, 119295, '\p{^Is_Blk=musicalsymbols}', "");
Expect(0, 119295, '\P{Is_Blk=musicalsymbols}', "");
Expect(1, 119295, '\P{^Is_Blk=musicalsymbols}', "");
Expect(0, 119039, '\p{Is_Blk=musicalsymbols}', "");
Expect(1, 119039, '\p{^Is_Blk=musicalsymbols}', "");
Expect(1, 119039, '\P{Is_Blk=musicalsymbols}', "");
Expect(0, 119039, '\P{^Is_Blk=musicalsymbols}', "");
Expect(1, 119295, '\p{Is_Blk=_	musical_symbols}', "");
Expect(0, 119295, '\p{^Is_Blk=_	musical_symbols}', "");
Expect(0, 119295, '\P{Is_Blk=_	musical_symbols}', "");
Expect(1, 119295, '\P{^Is_Blk=_	musical_symbols}', "");
Expect(0, 119039, '\p{Is_Blk=_	musical_symbols}', "");
Expect(1, 119039, '\p{^Is_Blk=_	musical_symbols}', "");
Expect(1, 119039, '\P{Is_Blk=_	musical_symbols}', "");
Expect(0, 119039, '\P{^Is_Blk=_	musical_symbols}', "");
Error('\p{Block: /a/ _Miscellaneous-SYMBOLS}');
Error('\P{Block: /a/ _Miscellaneous-SYMBOLS}');
Expect(1, 9983, '\p{Block=miscellaneoussymbols}', "");
Expect(0, 9983, '\p{^Block=miscellaneoussymbols}', "");
Expect(0, 9983, '\P{Block=miscellaneoussymbols}', "");
Expect(1, 9983, '\P{^Block=miscellaneoussymbols}', "");
Expect(0, 9727, '\p{Block=miscellaneoussymbols}', "");
Expect(1, 9727, '\p{^Block=miscellaneoussymbols}', "");
Expect(1, 9727, '\P{Block=miscellaneoussymbols}', "");
Expect(0, 9727, '\P{^Block=miscellaneoussymbols}', "");
Expect(1, 9983, '\p{Block=_Miscellaneous-Symbols}', "");
Expect(0, 9983, '\p{^Block=_Miscellaneous-Symbols}', "");
Expect(0, 9983, '\P{Block=_Miscellaneous-Symbols}', "");
Expect(1, 9983, '\P{^Block=_Miscellaneous-Symbols}', "");
Expect(0, 9727, '\p{Block=_Miscellaneous-Symbols}', "");
Expect(1, 9727, '\p{^Block=_Miscellaneous-Symbols}', "");
Expect(1, 9727, '\P{Block=_Miscellaneous-Symbols}', "");
Expect(0, 9727, '\P{^Block=_Miscellaneous-Symbols}', "");
Error('\p{Blk=:=- miscellaneous_SYMBOLS}');
Error('\P{Blk=:=- miscellaneous_SYMBOLS}');
Expect(1, 9983, '\p{Blk: miscellaneoussymbols}', "");
Expect(0, 9983, '\p{^Blk: miscellaneoussymbols}', "");
Expect(0, 9983, '\P{Blk: miscellaneoussymbols}', "");
Expect(1, 9983, '\P{^Blk: miscellaneoussymbols}', "");
Expect(0, 9727, '\p{Blk: miscellaneoussymbols}', "");
Expect(1, 9727, '\p{^Blk: miscellaneoussymbols}', "");
Expect(1, 9727, '\P{Blk: miscellaneoussymbols}', "");
Expect(0, 9727, '\P{^Blk: miscellaneoussymbols}', "");
Expect(1, 9983, '\p{Blk=-_miscellaneous	Symbols}', "");
Expect(0, 9983, '\p{^Blk=-_miscellaneous	Symbols}', "");
Expect(0, 9983, '\P{Blk=-_miscellaneous	Symbols}', "");
Expect(1, 9983, '\P{^Blk=-_miscellaneous	Symbols}', "");
Expect(0, 9727, '\p{Blk=-_miscellaneous	Symbols}', "");
Expect(1, 9727, '\p{^Blk=-_miscellaneous	Symbols}', "");
Expect(1, 9727, '\P{Blk=-_miscellaneous	Symbols}', "");
Expect(0, 9727, '\P{^Blk=-_miscellaneous	Symbols}', "");
Error('\p{Is_Block: -_Miscellaneous/a/symbols}');
Error('\P{Is_Block: -_Miscellaneous/a/symbols}');
Expect(1, 9983, '\p{Is_Block=miscellaneoussymbols}', "");
Expect(0, 9983, '\p{^Is_Block=miscellaneoussymbols}', "");
Expect(0, 9983, '\P{Is_Block=miscellaneoussymbols}', "");
Expect(1, 9983, '\P{^Is_Block=miscellaneoussymbols}', "");
Expect(0, 9727, '\p{Is_Block=miscellaneoussymbols}', "");
Expect(1, 9727, '\p{^Is_Block=miscellaneoussymbols}', "");
Expect(1, 9727, '\P{Is_Block=miscellaneoussymbols}', "");
Expect(0, 9727, '\P{^Is_Block=miscellaneoussymbols}', "");
Expect(1, 9983, '\p{Is_Block=  MISCELLANEOUSSYMBOLS}', "");
Expect(0, 9983, '\p{^Is_Block=  MISCELLANEOUSSYMBOLS}', "");
Expect(0, 9983, '\P{Is_Block=  MISCELLANEOUSSYMBOLS}', "");
Expect(1, 9983, '\P{^Is_Block=  MISCELLANEOUSSYMBOLS}', "");
Expect(0, 9727, '\p{Is_Block=  MISCELLANEOUSSYMBOLS}', "");
Expect(1, 9727, '\p{^Is_Block=  MISCELLANEOUSSYMBOLS}', "");
Expect(1, 9727, '\P{Is_Block=  MISCELLANEOUSSYMBOLS}', "");
Expect(0, 9727, '\P{^Is_Block=  MISCELLANEOUSSYMBOLS}', "");
Error('\p{Is_Blk=- Miscellaneoussymbols:=}');
Error('\P{Is_Blk=- Miscellaneoussymbols:=}');
Expect(1, 9983, '\p{Is_Blk:miscellaneoussymbols}', "");
Expect(0, 9983, '\p{^Is_Blk:miscellaneoussymbols}', "");
Expect(0, 9983, '\P{Is_Blk:miscellaneoussymbols}', "");
Expect(1, 9983, '\P{^Is_Blk:miscellaneoussymbols}', "");
Expect(0, 9727, '\p{Is_Blk:miscellaneoussymbols}', "");
Expect(1, 9727, '\p{^Is_Blk:miscellaneoussymbols}', "");
Expect(1, 9727, '\P{Is_Blk:miscellaneoussymbols}', "");
Expect(0, 9727, '\P{^Is_Blk:miscellaneoussymbols}', "");
Expect(1, 9983, '\p{Is_Blk=- MISCELLANEOUS-Symbols}', "");
Expect(0, 9983, '\p{^Is_Blk=- MISCELLANEOUS-Symbols}', "");
Expect(0, 9983, '\P{Is_Blk=- MISCELLANEOUS-Symbols}', "");
Expect(1, 9983, '\P{^Is_Blk=- MISCELLANEOUS-Symbols}', "");
Expect(0, 9727, '\p{Is_Blk=- MISCELLANEOUS-Symbols}', "");
Expect(1, 9727, '\p{^Is_Blk=- MISCELLANEOUS-Symbols}', "");
Expect(1, 9727, '\P{Is_Blk=- MISCELLANEOUS-Symbols}', "");
Expect(0, 9727, '\P{^Is_Blk=- MISCELLANEOUS-Symbols}', "");
Error('\p{Block=/a/	Runic}');
Error('\P{Block=/a/	Runic}');
Expect(1, 5887, '\p{Block=runic}', "");
Expect(0, 5887, '\p{^Block=runic}', "");
Expect(0, 5887, '\P{Block=runic}', "");
Expect(1, 5887, '\P{^Block=runic}', "");
Expect(0, 5791, '\p{Block=runic}', "");
Expect(1, 5791, '\p{^Block=runic}', "");
Expect(1, 5791, '\P{Block=runic}', "");
Expect(0, 5791, '\P{^Block=runic}', "");
Expect(1, 5887, '\p{Block= Runic}', "");
Expect(0, 5887, '\p{^Block= Runic}', "");
Expect(0, 5887, '\P{Block= Runic}', "");
Expect(1, 5887, '\P{^Block= Runic}', "");
Expect(0, 5791, '\p{Block= Runic}', "");
Expect(1, 5791, '\p{^Block= Runic}', "");
Expect(1, 5791, '\P{Block= Runic}', "");
Expect(0, 5791, '\P{^Block= Runic}', "");
Error('\p{Blk: _:=Runic}');
Error('\P{Blk: _:=Runic}');
Expect(1, 5887, '\p{Blk=runic}', "");
Expect(0, 5887, '\p{^Blk=runic}', "");
Expect(0, 5887, '\P{Blk=runic}', "");
Expect(1, 5887, '\P{^Blk=runic}', "");
Expect(0, 5791, '\p{Blk=runic}', "");
Expect(1, 5791, '\p{^Blk=runic}', "");
Expect(1, 5791, '\P{Blk=runic}', "");
Expect(0, 5791, '\P{^Blk=runic}', "");
Expect(1, 5887, '\p{Blk=-Runic}', "");
Expect(0, 5887, '\p{^Blk=-Runic}', "");
Expect(0, 5887, '\P{Blk=-Runic}', "");
Expect(1, 5887, '\P{^Blk=-Runic}', "");
Expect(0, 5791, '\p{Blk=-Runic}', "");
Expect(1, 5791, '\p{^Blk=-Runic}', "");
Expect(1, 5791, '\P{Blk=-Runic}', "");
Expect(0, 5791, '\P{^Blk=-Runic}', "");
Error('\p{Is_Block=- runic/a/}');
Error('\P{Is_Block=- runic/a/}');
Expect(1, 5887, '\p{Is_Block=runic}', "");
Expect(0, 5887, '\p{^Is_Block=runic}', "");
Expect(0, 5887, '\P{Is_Block=runic}', "");
Expect(1, 5887, '\P{^Is_Block=runic}', "");
Expect(0, 5791, '\p{Is_Block=runic}', "");
Expect(1, 5791, '\p{^Is_Block=runic}', "");
Expect(1, 5791, '\P{Is_Block=runic}', "");
Expect(0, 5791, '\P{^Is_Block=runic}', "");
Expect(1, 5887, '\p{Is_Block: 	_Runic}', "");
Expect(0, 5887, '\p{^Is_Block: 	_Runic}', "");
Expect(0, 5887, '\P{Is_Block: 	_Runic}', "");
Expect(1, 5887, '\P{^Is_Block: 	_Runic}', "");
Expect(0, 5791, '\p{Is_Block: 	_Runic}', "");
Expect(1, 5791, '\p{^Is_Block: 	_Runic}', "");
Expect(1, 5791, '\P{Is_Block: 	_Runic}', "");
Expect(0, 5791, '\P{^Is_Block: 	_Runic}', "");
Error('\p{Is_Blk:	/a/runic}');
Error('\P{Is_Blk:	/a/runic}');
Expect(1, 5887, '\p{Is_Blk=runic}', "");
Expect(0, 5887, '\p{^Is_Blk=runic}', "");
Expect(0, 5887, '\P{Is_Blk=runic}', "");
Expect(1, 5887, '\P{^Is_Blk=runic}', "");
Expect(0, 5791, '\p{Is_Blk=runic}', "");
Expect(1, 5791, '\p{^Is_Blk=runic}', "");
Expect(1, 5791, '\P{Is_Blk=runic}', "");
Expect(0, 5791, '\P{^Is_Blk=runic}', "");
Expect(1, 5887, '\p{Is_Blk=-Runic}', "");
Expect(0, 5887, '\p{^Is_Blk=-Runic}', "");
Expect(0, 5887, '\P{Is_Blk=-Runic}', "");
Expect(1, 5887, '\P{^Is_Blk=-Runic}', "");
Expect(0, 5791, '\p{Is_Blk=-Runic}', "");
Expect(1, 5791, '\p{^Is_Blk=-Runic}', "");
Expect(1, 5791, '\P{Is_Blk=-Runic}', "");
Expect(0, 5791, '\P{^Is_Blk=-Runic}', "");
Error('\p{Block=/a/-	lycian}');
Error('\P{Block=/a/-	lycian}');
Expect(1, 66207, '\p{Block=lycian}', "");
Expect(0, 66207, '\p{^Block=lycian}', "");
Expect(0, 66207, '\P{Block=lycian}', "");
Expect(1, 66207, '\P{^Block=lycian}', "");
Expect(0, 66175, '\p{Block=lycian}', "");
Expect(1, 66175, '\p{^Block=lycian}', "");
Expect(1, 66175, '\P{Block=lycian}', "");
Expect(0, 66175, '\P{^Block=lycian}', "");
Expect(1, 66207, '\p{Block= lycian}', "");
Expect(0, 66207, '\p{^Block= lycian}', "");
Expect(0, 66207, '\P{Block= lycian}', "");
Expect(1, 66207, '\P{^Block= lycian}', "");
Expect(0, 66175, '\p{Block= lycian}', "");
Expect(1, 66175, '\p{^Block= lycian}', "");
Expect(1, 66175, '\P{Block= lycian}', "");
Expect(0, 66175, '\P{^Block= lycian}', "");
Error('\p{Blk= :=Lycian}');
Error('\P{Blk= :=Lycian}');
Expect(1, 66207, '\p{Blk=lycian}', "");
Expect(0, 66207, '\p{^Blk=lycian}', "");
Expect(0, 66207, '\P{Blk=lycian}', "");
Expect(1, 66207, '\P{^Blk=lycian}', "");
Expect(0, 66175, '\p{Blk=lycian}', "");
Expect(1, 66175, '\p{^Blk=lycian}', "");
Expect(1, 66175, '\P{Blk=lycian}', "");
Expect(0, 66175, '\P{^Blk=lycian}', "");
Expect(1, 66207, '\p{Blk=-_Lycian}', "");
Expect(0, 66207, '\p{^Blk=-_Lycian}', "");
Expect(0, 66207, '\P{Blk=-_Lycian}', "");
Expect(1, 66207, '\P{^Blk=-_Lycian}', "");
Expect(0, 66175, '\p{Blk=-_Lycian}', "");
Expect(1, 66175, '\p{^Blk=-_Lycian}', "");
Expect(1, 66175, '\P{Blk=-_Lycian}', "");
Expect(0, 66175, '\P{^Blk=-_Lycian}', "");
Error('\p{Is_Block=/a/--Lycian}');
Error('\P{Is_Block=/a/--Lycian}');
Expect(1, 66207, '\p{Is_Block=lycian}', "");
Expect(0, 66207, '\p{^Is_Block=lycian}', "");
Expect(0, 66207, '\P{Is_Block=lycian}', "");
Expect(1, 66207, '\P{^Is_Block=lycian}', "");
Expect(0, 66175, '\p{Is_Block=lycian}', "");
Expect(1, 66175, '\p{^Is_Block=lycian}', "");
Expect(1, 66175, '\P{Is_Block=lycian}', "");
Expect(0, 66175, '\P{^Is_Block=lycian}', "");
Expect(1, 66207, '\p{Is_Block=	LYCIAN}', "");
Expect(0, 66207, '\p{^Is_Block=	LYCIAN}', "");
Expect(0, 66207, '\P{Is_Block=	LYCIAN}', "");
Expect(1, 66207, '\P{^Is_Block=	LYCIAN}', "");
Expect(0, 66175, '\p{Is_Block=	LYCIAN}', "");
Expect(1, 66175, '\p{^Is_Block=	LYCIAN}', "");
Expect(1, 66175, '\P{Is_Block=	LYCIAN}', "");
Expect(0, 66175, '\P{^Is_Block=	LYCIAN}', "");
Error('\p{Is_Blk= -Lycian/a/}');
Error('\P{Is_Blk= -Lycian/a/}');
Expect(1, 66207, '\p{Is_Blk=lycian}', "");
Expect(0, 66207, '\p{^Is_Blk=lycian}', "");
Expect(0, 66207, '\P{Is_Blk=lycian}', "");
Expect(1, 66207, '\P{^Is_Blk=lycian}', "");
Expect(0, 66175, '\p{Is_Blk=lycian}', "");
Expect(1, 66175, '\p{^Is_Blk=lycian}', "");
Expect(1, 66175, '\P{Is_Blk=lycian}', "");
Expect(0, 66175, '\P{^Is_Blk=lycian}', "");
Expect(1, 66207, '\p{Is_Blk=_Lycian}', "");
Expect(0, 66207, '\p{^Is_Blk=_Lycian}', "");
Expect(0, 66207, '\P{Is_Blk=_Lycian}', "");
Expect(1, 66207, '\P{^Is_Blk=_Lycian}', "");
Expect(0, 66175, '\p{Is_Blk=_Lycian}', "");
Expect(1, 66175, '\p{^Is_Blk=_Lycian}', "");
Expect(1, 66175, '\P{Is_Blk=_Lycian}', "");
Expect(0, 66175, '\P{^Is_Blk=_Lycian}', "");
Error('\p{Block: :=_-GEOMETRIC-Shapes}');
Error('\P{Block: :=_-GEOMETRIC-Shapes}');
Expect(1, 9727, '\p{Block=geometricshapes}', "");
Expect(0, 9727, '\p{^Block=geometricshapes}', "");
Expect(0, 9727, '\P{Block=geometricshapes}', "");
Expect(1, 9727, '\P{^Block=geometricshapes}', "");
Expect(0, 9631, '\p{Block=geometricshapes}', "");
Expect(1, 9631, '\p{^Block=geometricshapes}', "");
Expect(1, 9631, '\P{Block=geometricshapes}', "");
Expect(0, 9631, '\P{^Block=geometricshapes}', "");
Expect(1, 9727, '\p{Block=	GeometricShapes}', "");
Expect(0, 9727, '\p{^Block=	GeometricShapes}', "");
Expect(0, 9727, '\P{Block=	GeometricShapes}', "");
Expect(1, 9727, '\P{^Block=	GeometricShapes}', "");
Expect(0, 9631, '\p{Block=	GeometricShapes}', "");
Expect(1, 9631, '\p{^Block=	GeometricShapes}', "");
Expect(1, 9631, '\P{Block=	GeometricShapes}', "");
Expect(0, 9631, '\P{^Block=	GeometricShapes}', "");
Error('\p{Blk=-_Geometric:=SHAPES}');
Error('\P{Blk=-_Geometric:=SHAPES}');
Expect(1, 9727, '\p{Blk=geometricshapes}', "");
Expect(0, 9727, '\p{^Blk=geometricshapes}', "");
Expect(0, 9727, '\P{Blk=geometricshapes}', "");
Expect(1, 9727, '\P{^Blk=geometricshapes}', "");
Expect(0, 9631, '\p{Blk=geometricshapes}', "");
Expect(1, 9631, '\p{^Blk=geometricshapes}', "");
Expect(1, 9631, '\P{Blk=geometricshapes}', "");
Expect(0, 9631, '\P{^Blk=geometricshapes}', "");
Expect(1, 9727, '\p{Blk:			GeometricShapes}', "");
Expect(0, 9727, '\p{^Blk:			GeometricShapes}', "");
Expect(0, 9727, '\P{Blk:			GeometricShapes}', "");
Expect(1, 9727, '\P{^Blk:			GeometricShapes}', "");
Expect(0, 9631, '\p{Blk:			GeometricShapes}', "");
Expect(1, 9631, '\p{^Blk:			GeometricShapes}', "");
Expect(1, 9631, '\P{Blk:			GeometricShapes}', "");
Expect(0, 9631, '\P{^Blk:			GeometricShapes}', "");
Error('\p{Is_Block:   _-GEOMETRIC:=SHAPES}');
Error('\P{Is_Block:   _-GEOMETRIC:=SHAPES}');
Expect(1, 9727, '\p{Is_Block=geometricshapes}', "");
Expect(0, 9727, '\p{^Is_Block=geometricshapes}', "");
Expect(0, 9727, '\P{Is_Block=geometricshapes}', "");
Expect(1, 9727, '\P{^Is_Block=geometricshapes}', "");
Expect(0, 9631, '\p{Is_Block=geometricshapes}', "");
Expect(1, 9631, '\p{^Is_Block=geometricshapes}', "");
Expect(1, 9631, '\P{Is_Block=geometricshapes}', "");
Expect(0, 9631, '\P{^Is_Block=geometricshapes}', "");
Expect(1, 9727, '\p{Is_Block=		GEOMETRIC_shapes}', "");
Expect(0, 9727, '\p{^Is_Block=		GEOMETRIC_shapes}', "");
Expect(0, 9727, '\P{Is_Block=		GEOMETRIC_shapes}', "");
Expect(1, 9727, '\P{^Is_Block=		GEOMETRIC_shapes}', "");
Expect(0, 9631, '\p{Is_Block=		GEOMETRIC_shapes}', "");
Expect(1, 9631, '\p{^Is_Block=		GEOMETRIC_shapes}', "");
Expect(1, 9631, '\P{Is_Block=		GEOMETRIC_shapes}', "");
Expect(0, 9631, '\P{^Is_Block=		GEOMETRIC_shapes}', "");
Error('\p{Is_Blk=-/a/Geometric Shapes}');
Error('\P{Is_Blk=-/a/Geometric Shapes}');
Expect(1, 9727, '\p{Is_Blk=geometricshapes}', "");
Expect(0, 9727, '\p{^Is_Blk=geometricshapes}', "");
Expect(0, 9727, '\P{Is_Blk=geometricshapes}', "");
Expect(1, 9727, '\P{^Is_Blk=geometricshapes}', "");
Expect(0, 9631, '\p{Is_Blk=geometricshapes}', "");
Expect(1, 9631, '\p{^Is_Blk=geometricshapes}', "");
Expect(1, 9631, '\P{Is_Blk=geometricshapes}', "");
Expect(0, 9631, '\P{^Is_Blk=geometricshapes}', "");
Expect(1, 9727, '\p{Is_Blk=_	GEOMETRIC-shapes}', "");
Expect(0, 9727, '\p{^Is_Blk=_	GEOMETRIC-shapes}', "");
Expect(0, 9727, '\P{Is_Blk=_	GEOMETRIC-shapes}', "");
Expect(1, 9727, '\P{^Is_Blk=_	GEOMETRIC-shapes}', "");
Expect(0, 9631, '\p{Is_Blk=_	GEOMETRIC-shapes}', "");
Expect(1, 9631, '\p{^Is_Blk=_	GEOMETRIC-shapes}', "");
Expect(1, 9631, '\P{Is_Blk=_	GEOMETRIC-shapes}', "");
Expect(0, 9631, '\P{^Is_Blk=_	GEOMETRIC-shapes}', "");
Error('\p{Block:  Aegean_Numbers:=}');
Error('\P{Block:  Aegean_Numbers:=}');
Expect(1, 65855, '\p{Block=aegeannumbers}', "");
Expect(0, 65855, '\p{^Block=aegeannumbers}', "");
Expect(0, 65855, '\P{Block=aegeannumbers}', "");
Expect(1, 65855, '\P{^Block=aegeannumbers}', "");
Expect(0, 65791, '\p{Block=aegeannumbers}', "");
Expect(1, 65791, '\p{^Block=aegeannumbers}', "");
Expect(1, 65791, '\P{Block=aegeannumbers}', "");
Expect(0, 65791, '\P{^Block=aegeannumbers}', "");
Expect(1, 65855, '\p{Block=	_aegean	Numbers}', "");
Expect(0, 65855, '\p{^Block=	_aegean	Numbers}', "");
Expect(0, 65855, '\P{Block=	_aegean	Numbers}', "");
Expect(1, 65855, '\P{^Block=	_aegean	Numbers}', "");
Expect(0, 65791, '\p{Block=	_aegean	Numbers}', "");
Expect(1, 65791, '\p{^Block=	_aegean	Numbers}', "");
Expect(1, 65791, '\P{Block=	_aegean	Numbers}', "");
Expect(0, 65791, '\P{^Block=	_aegean	Numbers}', "");
Error('\p{Blk=_:=Aegean-Numbers}');
Error('\P{Blk=_:=Aegean-Numbers}');
Expect(1, 65855, '\p{Blk=aegeannumbers}', "");
Expect(0, 65855, '\p{^Blk=aegeannumbers}', "");
Expect(0, 65855, '\P{Blk=aegeannumbers}', "");
Expect(1, 65855, '\P{^Blk=aegeannumbers}', "");
Expect(0, 65791, '\p{Blk=aegeannumbers}', "");
Expect(1, 65791, '\p{^Blk=aegeannumbers}', "");
Expect(1, 65791, '\P{Blk=aegeannumbers}', "");
Expect(0, 65791, '\P{^Blk=aegeannumbers}', "");
Expect(1, 65855, '\p{Blk=  AEGEAN_Numbers}', "");
Expect(0, 65855, '\p{^Blk=  AEGEAN_Numbers}', "");
Expect(0, 65855, '\P{Blk=  AEGEAN_Numbers}', "");
Expect(1, 65855, '\P{^Blk=  AEGEAN_Numbers}', "");
Expect(0, 65791, '\p{Blk=  AEGEAN_Numbers}', "");
Expect(1, 65791, '\p{^Blk=  AEGEAN_Numbers}', "");
Expect(1, 65791, '\P{Blk=  AEGEAN_Numbers}', "");
Expect(0, 65791, '\P{^Blk=  AEGEAN_Numbers}', "");
Error('\p{Is_Block= /a/AEGEANNumbers}');
Error('\P{Is_Block= /a/AEGEANNumbers}');
Expect(1, 65855, '\p{Is_Block=aegeannumbers}', "");
Expect(0, 65855, '\p{^Is_Block=aegeannumbers}', "");
Expect(0, 65855, '\P{Is_Block=aegeannumbers}', "");
Expect(1, 65855, '\P{^Is_Block=aegeannumbers}', "");
Expect(0, 65791, '\p{Is_Block=aegeannumbers}', "");
Expect(1, 65791, '\p{^Is_Block=aegeannumbers}', "");
Expect(1, 65791, '\P{Is_Block=aegeannumbers}', "");
Expect(0, 65791, '\P{^Is_Block=aegeannumbers}', "");
Expect(1, 65855, '\p{Is_Block=	_Aegean_NUMBERS}', "");
Expect(0, 65855, '\p{^Is_Block=	_Aegean_NUMBERS}', "");
Expect(0, 65855, '\P{Is_Block=	_Aegean_NUMBERS}', "");
Expect(1, 65855, '\P{^Is_Block=	_Aegean_NUMBERS}', "");
Expect(0, 65791, '\p{Is_Block=	_Aegean_NUMBERS}', "");
Expect(1, 65791, '\p{^Is_Block=	_Aegean_NUMBERS}', "");
Expect(1, 65791, '\P{Is_Block=	_Aegean_NUMBERS}', "");
Expect(0, 65791, '\P{^Is_Block=	_Aegean_NUMBERS}', "");
Error('\p{Is_Blk= AEGEAN:=NUMBERS}');
Error('\P{Is_Blk= AEGEAN:=NUMBERS}');
Expect(1, 65855, '\p{Is_Blk=aegeannumbers}', "");
Expect(0, 65855, '\p{^Is_Blk=aegeannumbers}', "");
Expect(0, 65855, '\P{Is_Blk=aegeannumbers}', "");
Expect(1, 65855, '\P{^Is_Blk=aegeannumbers}', "");
Expect(0, 65791, '\p{Is_Blk=aegeannumbers}', "");
Expect(1, 65791, '\p{^Is_Blk=aegeannumbers}', "");
Expect(1, 65791, '\P{Is_Blk=aegeannumbers}', "");
Expect(0, 65791, '\P{^Is_Blk=aegeannumbers}', "");
Expect(1, 65855, '\p{Is_Blk=		Aegean-NUMBERS}', "");
Expect(0, 65855, '\p{^Is_Blk=		Aegean-NUMBERS}', "");
Expect(0, 65855, '\P{Is_Blk=		Aegean-NUMBERS}', "");
Expect(1, 65855, '\P{^Is_Blk=		Aegean-NUMBERS}', "");
Expect(0, 65791, '\p{Is_Blk=		Aegean-NUMBERS}', "");
Expect(1, 65791, '\p{^Is_Blk=		Aegean-NUMBERS}', "");
Expect(1, 65791, '\P{Is_Blk=		Aegean-NUMBERS}', "");
Expect(0, 65791, '\P{^Is_Blk=		Aegean-NUMBERS}', "");
Error('\p{Block:   :=KHMER Symbols}');
Error('\P{Block:   :=KHMER Symbols}');
Expect(1, 6655, '\p{Block:   khmersymbols}', "");
Expect(0, 6655, '\p{^Block:   khmersymbols}', "");
Expect(0, 6655, '\P{Block:   khmersymbols}', "");
Expect(1, 6655, '\P{^Block:   khmersymbols}', "");
Expect(0, 6623, '\p{Block:   khmersymbols}', "");
Expect(1, 6623, '\p{^Block:   khmersymbols}', "");
Expect(1, 6623, '\P{Block:   khmersymbols}', "");
Expect(0, 6623, '\P{^Block:   khmersymbols}', "");
Expect(1, 6655, '\p{Block=	-Khmer SYMBOLS}', "");
Expect(0, 6655, '\p{^Block=	-Khmer SYMBOLS}', "");
Expect(0, 6655, '\P{Block=	-Khmer SYMBOLS}', "");
Expect(1, 6655, '\P{^Block=	-Khmer SYMBOLS}', "");
Expect(0, 6623, '\p{Block=	-Khmer SYMBOLS}', "");
Expect(1, 6623, '\p{^Block=	-Khmer SYMBOLS}', "");
Expect(1, 6623, '\P{Block=	-Khmer SYMBOLS}', "");
Expect(0, 6623, '\P{^Block=	-Khmer SYMBOLS}', "");
Error('\p{Blk=	_khmer:=Symbols}');
Error('\P{Blk=	_khmer:=Symbols}');
Expect(1, 6655, '\p{Blk=khmersymbols}', "");
Expect(0, 6655, '\p{^Blk=khmersymbols}', "");
Expect(0, 6655, '\P{Blk=khmersymbols}', "");
Expect(1, 6655, '\P{^Blk=khmersymbols}', "");
Expect(0, 6623, '\p{Blk=khmersymbols}', "");
Expect(1, 6623, '\p{^Blk=khmersymbols}', "");
Expect(1, 6623, '\P{Blk=khmersymbols}', "");
Expect(0, 6623, '\P{^Blk=khmersymbols}', "");
Expect(1, 6655, '\p{Blk=_	KHMER_Symbols}', "");
Expect(0, 6655, '\p{^Blk=_	KHMER_Symbols}', "");
Expect(0, 6655, '\P{Blk=_	KHMER_Symbols}', "");
Expect(1, 6655, '\P{^Blk=_	KHMER_Symbols}', "");
Expect(0, 6623, '\p{Blk=_	KHMER_Symbols}', "");
Expect(1, 6623, '\p{^Blk=_	KHMER_Symbols}', "");
Expect(1, 6623, '\P{Blk=_	KHMER_Symbols}', "");
Expect(0, 6623, '\P{^Blk=_	KHMER_Symbols}', "");
Error('\p{Is_Block=- Khmer/a/Symbols}');
Error('\P{Is_Block=- Khmer/a/Symbols}');
Expect(1, 6655, '\p{Is_Block=khmersymbols}', "");
Expect(0, 6655, '\p{^Is_Block=khmersymbols}', "");
Expect(0, 6655, '\P{Is_Block=khmersymbols}', "");
Expect(1, 6655, '\P{^Is_Block=khmersymbols}', "");
Expect(0, 6623, '\p{Is_Block=khmersymbols}', "");
Expect(1, 6623, '\p{^Is_Block=khmersymbols}', "");
Expect(1, 6623, '\P{Is_Block=khmersymbols}', "");
Expect(0, 6623, '\P{^Is_Block=khmersymbols}', "");
Expect(1, 6655, '\p{Is_Block=	 KHMER	Symbols}', "");
Expect(0, 6655, '\p{^Is_Block=	 KHMER	Symbols}', "");
Expect(0, 6655, '\P{Is_Block=	 KHMER	Symbols}', "");
Expect(1, 6655, '\P{^Is_Block=	 KHMER	Symbols}', "");
Expect(0, 6623, '\p{Is_Block=	 KHMER	Symbols}', "");
Expect(1, 6623, '\p{^Is_Block=	 KHMER	Symbols}', "");
Expect(1, 6623, '\P{Is_Block=	 KHMER	Symbols}', "");
Expect(0, 6623, '\P{^Is_Block=	 KHMER	Symbols}', "");
Error('\p{Is_Blk=_/a/Khmer_Symbols}');
Error('\P{Is_Blk=_/a/Khmer_Symbols}');
Expect(1, 6655, '\p{Is_Blk=khmersymbols}', "");
Expect(0, 6655, '\p{^Is_Blk=khmersymbols}', "");
Expect(0, 6655, '\P{Is_Blk=khmersymbols}', "");
Expect(1, 6655, '\P{^Is_Blk=khmersymbols}', "");
Expect(0, 6623, '\p{Is_Blk=khmersymbols}', "");
Expect(1, 6623, '\p{^Is_Blk=khmersymbols}', "");
Expect(1, 6623, '\P{Is_Blk=khmersymbols}', "");
Expect(0, 6623, '\P{^Is_Blk=khmersymbols}', "");
Expect(1, 6655, '\p{Is_Blk=	KHMER	Symbols}', "");
Expect(0, 6655, '\p{^Is_Blk=	KHMER	Symbols}', "");
Expect(0, 6655, '\P{Is_Blk=	KHMER	Symbols}', "");
Expect(1, 6655, '\P{^Is_Blk=	KHMER	Symbols}', "");
Expect(0, 6623, '\p{Is_Blk=	KHMER	Symbols}', "");
Expect(1, 6623, '\p{^Is_Blk=	KHMER	Symbols}', "");
Expect(1, 6623, '\P{Is_Blk=	KHMER	Symbols}', "");
Expect(0, 6623, '\P{^Is_Blk=	KHMER	Symbols}', "");
Error('\p{Block=	cjk/a/RADICALS-Supplement}');
Error('\P{Block=	cjk/a/RADICALS-Supplement}');
Expect(1, 12031, '\p{Block=cjkradicalssupplement}', "");
Expect(0, 12031, '\p{^Block=cjkradicalssupplement}', "");
Expect(0, 12031, '\P{Block=cjkradicalssupplement}', "");
Expect(1, 12031, '\P{^Block=cjkradicalssupplement}', "");
Expect(0, 11903, '\p{Block=cjkradicalssupplement}', "");
Expect(1, 11903, '\p{^Block=cjkradicalssupplement}', "");
Expect(1, 11903, '\P{Block=cjkradicalssupplement}', "");
Expect(0, 11903, '\P{^Block=cjkradicalssupplement}', "");
Expect(1, 12031, '\p{Block=_CJK_Radicals SUPPLEMENT}', "");
Expect(0, 12031, '\p{^Block=_CJK_Radicals SUPPLEMENT}', "");
Expect(0, 12031, '\P{Block=_CJK_Radicals SUPPLEMENT}', "");
Expect(1, 12031, '\P{^Block=_CJK_Radicals SUPPLEMENT}', "");
Expect(0, 11903, '\p{Block=_CJK_Radicals SUPPLEMENT}', "");
Expect(1, 11903, '\p{^Block=_CJK_Radicals SUPPLEMENT}', "");
Expect(1, 11903, '\P{Block=_CJK_Radicals SUPPLEMENT}', "");
Expect(0, 11903, '\P{^Block=_CJK_Radicals SUPPLEMENT}', "");
Error('\p{Blk=/a/ 	CJK_radicals	Supplement}');
Error('\P{Blk=/a/ 	CJK_radicals	Supplement}');
Expect(1, 12031, '\p{Blk=cjkradicalssupplement}', "");
Expect(0, 12031, '\p{^Blk=cjkradicalssupplement}', "");
Expect(0, 12031, '\P{Blk=cjkradicalssupplement}', "");
Expect(1, 12031, '\P{^Blk=cjkradicalssupplement}', "");
Expect(0, 11903, '\p{Blk=cjkradicalssupplement}', "");
Expect(1, 11903, '\p{^Blk=cjkradicalssupplement}', "");
Expect(1, 11903, '\P{Blk=cjkradicalssupplement}', "");
Expect(0, 11903, '\P{^Blk=cjkradicalssupplement}', "");
Expect(1, 12031, '\p{Blk=_	cjkradicals	Supplement}', "");
Expect(0, 12031, '\p{^Blk=_	cjkradicals	Supplement}', "");
Expect(0, 12031, '\P{Blk=_	cjkradicals	Supplement}', "");
Expect(1, 12031, '\P{^Blk=_	cjkradicals	Supplement}', "");
Expect(0, 11903, '\p{Blk=_	cjkradicals	Supplement}', "");
Expect(1, 11903, '\p{^Blk=_	cjkradicals	Supplement}', "");
Expect(1, 11903, '\P{Blk=_	cjkradicals	Supplement}', "");
Expect(0, 11903, '\P{^Blk=_	cjkradicals	Supplement}', "");
Error('\p{Is_Block=:= CJK_radicalssupplement}');
Error('\P{Is_Block=:= CJK_radicalssupplement}');
Expect(1, 12031, '\p{Is_Block=cjkradicalssupplement}', "");
Expect(0, 12031, '\p{^Is_Block=cjkradicalssupplement}', "");
Expect(0, 12031, '\P{Is_Block=cjkradicalssupplement}', "");
Expect(1, 12031, '\P{^Is_Block=cjkradicalssupplement}', "");
Expect(0, 11903, '\p{Is_Block=cjkradicalssupplement}', "");
Expect(1, 11903, '\p{^Is_Block=cjkradicalssupplement}', "");
Expect(1, 11903, '\P{Is_Block=cjkradicalssupplement}', "");
Expect(0, 11903, '\P{^Is_Block=cjkradicalssupplement}', "");
Expect(1, 12031, '\p{Is_Block= 	CJK	radicals-SUPPLEMENT}', "");
Expect(0, 12031, '\p{^Is_Block= 	CJK	radicals-SUPPLEMENT}', "");
Expect(0, 12031, '\P{Is_Block= 	CJK	radicals-SUPPLEMENT}', "");
Expect(1, 12031, '\P{^Is_Block= 	CJK	radicals-SUPPLEMENT}', "");
Expect(0, 11903, '\p{Is_Block= 	CJK	radicals-SUPPLEMENT}', "");
Expect(1, 11903, '\p{^Is_Block= 	CJK	radicals-SUPPLEMENT}', "");
Expect(1, 11903, '\P{Is_Block= 	CJK	radicals-SUPPLEMENT}', "");
Expect(0, 11903, '\P{^Is_Block= 	CJK	radicals-SUPPLEMENT}', "");
Error('\p{Is_Blk=:=_CJK Radicals	SUPPLEMENT}');
Error('\P{Is_Blk=:=_CJK Radicals	SUPPLEMENT}');
Expect(1, 12031, '\p{Is_Blk=cjkradicalssupplement}', "");
Expect(0, 12031, '\p{^Is_Blk=cjkradicalssupplement}', "");
Expect(0, 12031, '\P{Is_Blk=cjkradicalssupplement}', "");
Expect(1, 12031, '\P{^Is_Blk=cjkradicalssupplement}', "");
Expect(0, 11903, '\p{Is_Blk=cjkradicalssupplement}', "");
Expect(1, 11903, '\p{^Is_Blk=cjkradicalssupplement}', "");
Expect(1, 11903, '\P{Is_Blk=cjkradicalssupplement}', "");
Expect(0, 11903, '\P{^Is_Blk=cjkradicalssupplement}', "");
Expect(1, 12031, '\p{Is_Blk=_CJK	radicals	Supplement}', "");
Expect(0, 12031, '\p{^Is_Blk=_CJK	radicals	Supplement}', "");
Expect(0, 12031, '\P{Is_Blk=_CJK	radicals	Supplement}', "");
Expect(1, 12031, '\P{^Is_Blk=_CJK	radicals	Supplement}', "");
Expect(0, 11903, '\p{Is_Blk=_CJK	radicals	Supplement}', "");
Expect(1, 11903, '\p{^Is_Blk=_CJK	radicals	Supplement}', "");
Expect(1, 11903, '\P{Is_Blk=_CJK	radicals	Supplement}', "");
Expect(0, 11903, '\P{^Is_Blk=_CJK	radicals	Supplement}', "");
Error('\p{Block=-/a/ALCHEMICAL-SYMBOLS}');
Error('\P{Block=-/a/ALCHEMICAL-SYMBOLS}');
Expect(1, 128895, '\p{Block=alchemicalsymbols}', "");
Expect(0, 128895, '\p{^Block=alchemicalsymbols}', "");
Expect(0, 128895, '\P{Block=alchemicalsymbols}', "");
Expect(1, 128895, '\P{^Block=alchemicalsymbols}', "");
Expect(0, 128767, '\p{Block=alchemicalsymbols}', "");
Expect(1, 128767, '\p{^Block=alchemicalsymbols}', "");
Expect(1, 128767, '\P{Block=alchemicalsymbols}', "");
Expect(0, 128767, '\P{^Block=alchemicalsymbols}', "");
Expect(1, 128895, '\p{Block=	 ALCHEMICAL_Symbols}', "");
Expect(0, 128895, '\p{^Block=	 ALCHEMICAL_Symbols}', "");
Expect(0, 128895, '\P{Block=	 ALCHEMICAL_Symbols}', "");
Expect(1, 128895, '\P{^Block=	 ALCHEMICAL_Symbols}', "");
Expect(0, 128767, '\p{Block=	 ALCHEMICAL_Symbols}', "");
Expect(1, 128767, '\p{^Block=	 ALCHEMICAL_Symbols}', "");
Expect(1, 128767, '\P{Block=	 ALCHEMICAL_Symbols}', "");
Expect(0, 128767, '\P{^Block=	 ALCHEMICAL_Symbols}', "");
Error('\p{Blk=-/a/alchemicalsymbols}');
Error('\P{Blk=-/a/alchemicalsymbols}');
Expect(1, 128895, '\p{Blk=alchemicalsymbols}', "");
Expect(0, 128895, '\p{^Blk=alchemicalsymbols}', "");
Expect(0, 128895, '\P{Blk=alchemicalsymbols}', "");
Expect(1, 128895, '\P{^Blk=alchemicalsymbols}', "");
Expect(0, 128767, '\p{Blk=alchemicalsymbols}', "");
Expect(1, 128767, '\p{^Blk=alchemicalsymbols}', "");
Expect(1, 128767, '\P{Blk=alchemicalsymbols}', "");
Expect(0, 128767, '\P{^Blk=alchemicalsymbols}', "");
Expect(1, 128895, '\p{Blk=-_alchemical-Symbols}', "");
Expect(0, 128895, '\p{^Blk=-_alchemical-Symbols}', "");
Expect(0, 128895, '\P{Blk=-_alchemical-Symbols}', "");
Expect(1, 128895, '\P{^Blk=-_alchemical-Symbols}', "");
Expect(0, 128767, '\p{Blk=-_alchemical-Symbols}', "");
Expect(1, 128767, '\p{^Blk=-_alchemical-Symbols}', "");
Expect(1, 128767, '\P{Blk=-_alchemical-Symbols}', "");
Expect(0, 128767, '\P{^Blk=-_alchemical-Symbols}', "");
Error('\p{Is_Block=-:=alchemical_SYMBOLS}');
Error('\P{Is_Block=-:=alchemical_SYMBOLS}');
Expect(1, 128895, '\p{Is_Block=alchemicalsymbols}', "");
Expect(0, 128895, '\p{^Is_Block=alchemicalsymbols}', "");
Expect(0, 128895, '\P{Is_Block=alchemicalsymbols}', "");
Expect(1, 128895, '\P{^Is_Block=alchemicalsymbols}', "");
Expect(0, 128767, '\p{Is_Block=alchemicalsymbols}', "");
Expect(1, 128767, '\p{^Is_Block=alchemicalsymbols}', "");
Expect(1, 128767, '\P{Is_Block=alchemicalsymbols}', "");
Expect(0, 128767, '\P{^Is_Block=alchemicalsymbols}', "");
Expect(1, 128895, '\p{Is_Block= 	Alchemical_Symbols}', "");
Expect(0, 128895, '\p{^Is_Block= 	Alchemical_Symbols}', "");
Expect(0, 128895, '\P{Is_Block= 	Alchemical_Symbols}', "");
Expect(1, 128895, '\P{^Is_Block= 	Alchemical_Symbols}', "");
Expect(0, 128767, '\p{Is_Block= 	Alchemical_Symbols}', "");
Expect(1, 128767, '\p{^Is_Block= 	Alchemical_Symbols}', "");
Expect(1, 128767, '\P{Is_Block= 	Alchemical_Symbols}', "");
Expect(0, 128767, '\P{^Is_Block= 	Alchemical_Symbols}', "");
Error('\p{Is_Blk=-ALCHEMICAL:=Symbols}');
Error('\P{Is_Blk=-ALCHEMICAL:=Symbols}');
Expect(1, 128895, '\p{Is_Blk=alchemicalsymbols}', "");
Expect(0, 128895, '\p{^Is_Blk=alchemicalsymbols}', "");
Expect(0, 128895, '\P{Is_Blk=alchemicalsymbols}', "");
Expect(1, 128895, '\P{^Is_Blk=alchemicalsymbols}', "");
Expect(0, 128767, '\p{Is_Blk=alchemicalsymbols}', "");
Expect(1, 128767, '\p{^Is_Blk=alchemicalsymbols}', "");
Expect(1, 128767, '\P{Is_Blk=alchemicalsymbols}', "");
Expect(0, 128767, '\P{^Is_Blk=alchemicalsymbols}', "");
Expect(1, 128895, '\p{Is_Blk=_alchemical-SYMBOLS}', "");
Expect(0, 128895, '\p{^Is_Blk=_alchemical-SYMBOLS}', "");
Expect(0, 128895, '\P{Is_Blk=_alchemical-SYMBOLS}', "");
Expect(1, 128895, '\P{^Is_Blk=_alchemical-SYMBOLS}', "");
Expect(0, 128767, '\p{Is_Blk=_alchemical-SYMBOLS}', "");
Expect(1, 128767, '\p{^Is_Blk=_alchemical-SYMBOLS}', "");
Expect(1, 128767, '\P{Is_Blk=_alchemical-SYMBOLS}', "");
Expect(0, 128767, '\P{^Is_Blk=_alchemical-SYMBOLS}', "");
Error('\p{Block=/a/Imperial-Aramaic}');
Error('\P{Block=/a/Imperial-Aramaic}');
Expect(1, 67679, '\p{Block=imperialaramaic}', "");
Expect(0, 67679, '\p{^Block=imperialaramaic}', "");
Expect(0, 67679, '\P{Block=imperialaramaic}', "");
Expect(1, 67679, '\P{^Block=imperialaramaic}', "");
Expect(0, 67647, '\p{Block=imperialaramaic}', "");
Expect(1, 67647, '\p{^Block=imperialaramaic}', "");
Expect(1, 67647, '\P{Block=imperialaramaic}', "");
Expect(0, 67647, '\P{^Block=imperialaramaic}', "");
Expect(1, 67679, '\p{Block=--Imperial	Aramaic}', "");
Expect(0, 67679, '\p{^Block=--Imperial	Aramaic}', "");
Expect(0, 67679, '\P{Block=--Imperial	Aramaic}', "");
Expect(1, 67679, '\P{^Block=--Imperial	Aramaic}', "");
Expect(0, 67647, '\p{Block=--Imperial	Aramaic}', "");
Expect(1, 67647, '\p{^Block=--Imperial	Aramaic}', "");
Expect(1, 67647, '\P{Block=--Imperial	Aramaic}', "");
Expect(0, 67647, '\P{^Block=--Imperial	Aramaic}', "");
Error('\p{Blk=-	Imperial:=ARAMAIC}');
Error('\P{Blk=-	Imperial:=ARAMAIC}');
Expect(1, 67679, '\p{Blk=imperialaramaic}', "");
Expect(0, 67679, '\p{^Blk=imperialaramaic}', "");
Expect(0, 67679, '\P{Blk=imperialaramaic}', "");
Expect(1, 67679, '\P{^Blk=imperialaramaic}', "");
Expect(0, 67647, '\p{Blk=imperialaramaic}', "");
Expect(1, 67647, '\p{^Blk=imperialaramaic}', "");
Expect(1, 67647, '\P{Blk=imperialaramaic}', "");
Expect(0, 67647, '\P{^Blk=imperialaramaic}', "");
Expect(1, 67679, '\p{Blk= _Imperialaramaic}', "");
Expect(0, 67679, '\p{^Blk= _Imperialaramaic}', "");
Expect(0, 67679, '\P{Blk= _Imperialaramaic}', "");
Expect(1, 67679, '\P{^Blk= _Imperialaramaic}', "");
Expect(0, 67647, '\p{Blk= _Imperialaramaic}', "");
Expect(1, 67647, '\p{^Blk= _Imperialaramaic}', "");
Expect(1, 67647, '\P{Blk= _Imperialaramaic}', "");
Expect(0, 67647, '\P{^Blk= _Imperialaramaic}', "");
Error('\p{Is_Block=_IMPERIAL/a/Aramaic}');
Error('\P{Is_Block=_IMPERIAL/a/Aramaic}');
Expect(1, 67679, '\p{Is_Block=imperialaramaic}', "");
Expect(0, 67679, '\p{^Is_Block=imperialaramaic}', "");
Expect(0, 67679, '\P{Is_Block=imperialaramaic}', "");
Expect(1, 67679, '\P{^Is_Block=imperialaramaic}', "");
Expect(0, 67647, '\p{Is_Block=imperialaramaic}', "");
Expect(1, 67647, '\p{^Is_Block=imperialaramaic}', "");
Expect(1, 67647, '\P{Is_Block=imperialaramaic}', "");
Expect(0, 67647, '\P{^Is_Block=imperialaramaic}', "");
Expect(1, 67679, '\p{Is_Block=-IMPERIAL-Aramaic}', "");
Expect(0, 67679, '\p{^Is_Block=-IMPERIAL-Aramaic}', "");
Expect(0, 67679, '\P{Is_Block=-IMPERIAL-Aramaic}', "");
Expect(1, 67679, '\P{^Is_Block=-IMPERIAL-Aramaic}', "");
Expect(0, 67647, '\p{Is_Block=-IMPERIAL-Aramaic}', "");
Expect(1, 67647, '\p{^Is_Block=-IMPERIAL-Aramaic}', "");
Expect(1, 67647, '\P{Is_Block=-IMPERIAL-Aramaic}', "");
Expect(0, 67647, '\P{^Is_Block=-IMPERIAL-Aramaic}', "");
Error('\p{Is_Blk=/a/Imperial_ARAMAIC}');
Error('\P{Is_Blk=/a/Imperial_ARAMAIC}');
Expect(1, 67679, '\p{Is_Blk=imperialaramaic}', "");
Expect(0, 67679, '\p{^Is_Blk=imperialaramaic}', "");
Expect(0, 67679, '\P{Is_Blk=imperialaramaic}', "");
Expect(1, 67679, '\P{^Is_Blk=imperialaramaic}', "");
Expect(0, 67647, '\p{Is_Blk=imperialaramaic}', "");
Expect(1, 67647, '\p{^Is_Blk=imperialaramaic}', "");
Expect(1, 67647, '\P{Is_Blk=imperialaramaic}', "");
Expect(0, 67647, '\P{^Is_Blk=imperialaramaic}', "");
Expect(1, 67679, '\p{Is_Blk= IMPERIAL_Aramaic}', "");
Expect(0, 67679, '\p{^Is_Blk= IMPERIAL_Aramaic}', "");
Expect(0, 67679, '\P{Is_Blk= IMPERIAL_Aramaic}', "");
Expect(1, 67679, '\P{^Is_Blk= IMPERIAL_Aramaic}', "");
Expect(0, 67647, '\p{Is_Blk= IMPERIAL_Aramaic}', "");
Expect(1, 67647, '\p{^Is_Blk= IMPERIAL_Aramaic}', "");
Expect(1, 67647, '\P{Is_Blk= IMPERIAL_Aramaic}', "");
Expect(0, 67647, '\P{^Is_Blk= IMPERIAL_Aramaic}', "");
Error('\p{Block:-Mathematical	OPERATORS/a/}');
Error('\P{Block:-Mathematical	OPERATORS/a/}');
Expect(1, 8959, '\p{Block=mathematicaloperators}', "");
Expect(0, 8959, '\p{^Block=mathematicaloperators}', "");
Expect(0, 8959, '\P{Block=mathematicaloperators}', "");
Expect(1, 8959, '\P{^Block=mathematicaloperators}', "");
Expect(0, 8703, '\p{Block=mathematicaloperators}', "");
Expect(1, 8703, '\p{^Block=mathematicaloperators}', "");
Expect(1, 8703, '\P{Block=mathematicaloperators}', "");
Expect(0, 8703, '\P{^Block=mathematicaloperators}', "");
Expect(1, 8959, '\p{Block: 	 mathematical Operators}', "");
Expect(0, 8959, '\p{^Block: 	 mathematical Operators}', "");
Expect(0, 8959, '\P{Block: 	 mathematical Operators}', "");
Expect(1, 8959, '\P{^Block: 	 mathematical Operators}', "");
Expect(0, 8703, '\p{Block: 	 mathematical Operators}', "");
Expect(1, 8703, '\p{^Block: 	 mathematical Operators}', "");
Expect(1, 8703, '\P{Block: 	 mathematical Operators}', "");
Expect(0, 8703, '\P{^Block: 	 mathematical Operators}', "");
Error('\p{Blk=:=_mathematical-Operators}');
Error('\P{Blk=:=_mathematical-Operators}');
Expect(1, 8959, '\p{Blk=mathematicaloperators}', "");
Expect(0, 8959, '\p{^Blk=mathematicaloperators}', "");
Expect(0, 8959, '\P{Blk=mathematicaloperators}', "");
Expect(1, 8959, '\P{^Blk=mathematicaloperators}', "");
Expect(0, 8703, '\p{Blk=mathematicaloperators}', "");
Expect(1, 8703, '\p{^Blk=mathematicaloperators}', "");
Expect(1, 8703, '\P{Blk=mathematicaloperators}', "");
Expect(0, 8703, '\P{^Blk=mathematicaloperators}', "");
Expect(1, 8959, '\p{Blk=	-mathematicalOperators}', "");
Expect(0, 8959, '\p{^Blk=	-mathematicalOperators}', "");
Expect(0, 8959, '\P{Blk=	-mathematicalOperators}', "");
Expect(1, 8959, '\P{^Blk=	-mathematicalOperators}', "");
Expect(0, 8703, '\p{Blk=	-mathematicalOperators}', "");
Expect(1, 8703, '\p{^Blk=	-mathematicalOperators}', "");
Expect(1, 8703, '\P{Blk=	-mathematicalOperators}', "");
Expect(0, 8703, '\P{^Blk=	-mathematicalOperators}', "");
Error('\p{Is_Block: --MATHEMATICAL:=OPERATORS}');
Error('\P{Is_Block: --MATHEMATICAL:=OPERATORS}');
Expect(1, 8959, '\p{Is_Block: mathematicaloperators}', "");
Expect(0, 8959, '\p{^Is_Block: mathematicaloperators}', "");
Expect(0, 8959, '\P{Is_Block: mathematicaloperators}', "");
Expect(1, 8959, '\P{^Is_Block: mathematicaloperators}', "");
Expect(0, 8703, '\p{Is_Block: mathematicaloperators}', "");
Expect(1, 8703, '\p{^Is_Block: mathematicaloperators}', "");
Expect(1, 8703, '\P{Is_Block: mathematicaloperators}', "");
Expect(0, 8703, '\P{^Is_Block: mathematicaloperators}', "");
Expect(1, 8959, '\p{Is_Block=_-MathematicalOPERATORS}', "");
Expect(0, 8959, '\p{^Is_Block=_-MathematicalOPERATORS}', "");
Expect(0, 8959, '\P{Is_Block=_-MathematicalOPERATORS}', "");
Expect(1, 8959, '\P{^Is_Block=_-MathematicalOPERATORS}', "");
Expect(0, 8703, '\p{Is_Block=_-MathematicalOPERATORS}', "");
Expect(1, 8703, '\p{^Is_Block=_-MathematicalOPERATORS}', "");
Expect(1, 8703, '\P{Is_Block=_-MathematicalOPERATORS}', "");
Expect(0, 8703, '\P{^Is_Block=_-MathematicalOPERATORS}', "");
Error('\p{Is_Blk:		 Mathematical/a/Operators}');
Error('\P{Is_Blk:		 Mathematical/a/Operators}');
Expect(1, 8959, '\p{Is_Blk=mathematicaloperators}', "");
Expect(0, 8959, '\p{^Is_Blk=mathematicaloperators}', "");
Expect(0, 8959, '\P{Is_Blk=mathematicaloperators}', "");
Expect(1, 8959, '\P{^Is_Blk=mathematicaloperators}', "");
Expect(0, 8703, '\p{Is_Blk=mathematicaloperators}', "");
Expect(1, 8703, '\p{^Is_Blk=mathematicaloperators}', "");
Expect(1, 8703, '\P{Is_Blk=mathematicaloperators}', "");
Expect(0, 8703, '\P{^Is_Blk=mathematicaloperators}', "");
Expect(1, 8959, '\p{Is_Blk=-mathematical_Operators}', "");
Expect(0, 8959, '\p{^Is_Blk=-mathematical_Operators}', "");
Expect(0, 8959, '\P{Is_Blk=-mathematical_Operators}', "");
Expect(1, 8959, '\P{^Is_Blk=-mathematical_Operators}', "");
Expect(0, 8703, '\p{Is_Blk=-mathematical_Operators}', "");
Expect(1, 8703, '\p{^Is_Blk=-mathematical_Operators}', "");
Expect(1, 8703, '\P{Is_Blk=-mathematical_Operators}', "");
Expect(0, 8703, '\P{^Is_Blk=-mathematical_Operators}', "");
Error('\p{Block=__CJK/a/Compatibility	Ideographs-supplement}');
Error('\P{Block=__CJK/a/Compatibility	Ideographs-supplement}');
Expect(1, 195103, '\p{Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 195103, '\p{^Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 195103, '\P{Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 195103, '\P{^Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 194559, '\p{Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 194559, '\p{^Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 194559, '\P{Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 194559, '\P{^Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 195103, '\p{Block=  CJK_compatibility_ideographs_supplement}', "");
Expect(0, 195103, '\p{^Block=  CJK_compatibility_ideographs_supplement}', "");
Expect(0, 195103, '\P{Block=  CJK_compatibility_ideographs_supplement}', "");
Expect(1, 195103, '\P{^Block=  CJK_compatibility_ideographs_supplement}', "");
Expect(0, 194559, '\p{Block=  CJK_compatibility_ideographs_supplement}', "");
Expect(1, 194559, '\p{^Block=  CJK_compatibility_ideographs_supplement}', "");
Expect(1, 194559, '\P{Block=  CJK_compatibility_ideographs_supplement}', "");
Expect(0, 194559, '\P{^Block=  CJK_compatibility_ideographs_supplement}', "");
Error('\p{Blk=-	CJK/a/Compatibility	Ideographs-Supplement}');
Error('\P{Blk=-	CJK/a/Compatibility	Ideographs-Supplement}');
Expect(1, 195103, '\p{Blk=cjkcompatibilityideographssupplement}', "");
Expect(0, 195103, '\p{^Blk=cjkcompatibilityideographssupplement}', "");
Expect(0, 195103, '\P{Blk=cjkcompatibilityideographssupplement}', "");
Expect(1, 195103, '\P{^Blk=cjkcompatibilityideographssupplement}', "");
Expect(0, 194559, '\p{Blk=cjkcompatibilityideographssupplement}', "");
Expect(1, 194559, '\p{^Blk=cjkcompatibilityideographssupplement}', "");
Expect(1, 194559, '\P{Blk=cjkcompatibilityideographssupplement}', "");
Expect(0, 194559, '\P{^Blk=cjkcompatibilityideographssupplement}', "");
Expect(1, 195103, '\p{Blk=-	CJK	Compatibility	Ideographs	SUPPLEMENT}', "");
Expect(0, 195103, '\p{^Blk=-	CJK	Compatibility	Ideographs	SUPPLEMENT}', "");
Expect(0, 195103, '\P{Blk=-	CJK	Compatibility	Ideographs	SUPPLEMENT}', "");
Expect(1, 195103, '\P{^Blk=-	CJK	Compatibility	Ideographs	SUPPLEMENT}', "");
Expect(0, 194559, '\p{Blk=-	CJK	Compatibility	Ideographs	SUPPLEMENT}', "");
Expect(1, 194559, '\p{^Blk=-	CJK	Compatibility	Ideographs	SUPPLEMENT}', "");
Expect(1, 194559, '\P{Blk=-	CJK	Compatibility	Ideographs	SUPPLEMENT}', "");
Expect(0, 194559, '\P{^Blk=-	CJK	Compatibility	Ideographs	SUPPLEMENT}', "");
Error('\p{Is_Block=/a/	 CJK_COMPATIBILITYIdeographs Supplement}');
Error('\P{Is_Block=/a/	 CJK_COMPATIBILITYIdeographs Supplement}');
Expect(1, 195103, '\p{Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 195103, '\p{^Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 195103, '\P{Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 195103, '\P{^Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 194559, '\p{Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 194559, '\p{^Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 194559, '\P{Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(0, 194559, '\P{^Is_Block=cjkcompatibilityideographssupplement}', "");
Expect(1, 195103, '\p{Is_Block= CJKcompatibilityIdeographsSupplement}', "");
Expect(0, 195103, '\p{^Is_Block= CJKcompatibilityIdeographsSupplement}', "");
Expect(0, 195103, '\P{Is_Block= CJKcompatibilityIdeographsSupplement}', "");
Expect(1, 195103, '\P{^Is_Block= CJKcompatibilityIdeographsSupplement}', "");
Expect(0, 194559, '\p{Is_Block= CJKcompatibilityIdeographsSupplement}', "");
Expect(1, 194559, '\p{^Is_Block= CJKcompatibilityIdeographsSupplement}', "");
Expect(1, 194559, '\P{Is_Block= CJKcompatibilityIdeographsSupplement}', "");
Expect(0, 194559, '\P{^Is_Block= CJKcompatibilityIdeographsSupplement}', "");
Error('\p{Is_Blk=__CJK_Compatibility Ideographs:=Supplement}');
Error('\P{Is_Blk=__CJK_Compatibility Ideographs:=Supplement}');
Expect(1, 195103, '\p{Is_Blk:cjkcompatibilityideographssupplement}', "");
Expect(0, 195103, '\p{^Is_Blk:cjkcompatibilityideographssupplement}', "");
Expect(0, 195103, '\P{Is_Blk:cjkcompatibilityideographssupplement}', "");
Expect(1, 195103, '\P{^Is_Blk:cjkcompatibilityideographssupplement}', "");
Expect(0, 194559, '\p{Is_Blk:cjkcompatibilityideographssupplement}', "");
Expect(1, 194559, '\p{^Is_Blk:cjkcompatibilityideographssupplement}', "");
Expect(1, 194559, '\P{Is_Blk:cjkcompatibilityideographssupplement}', "");
Expect(0, 194559, '\P{^Is_Blk:cjkcompatibilityideographssupplement}', "");
Expect(1, 195103, '\p{Is_Blk=cjk_Compatibility	IDEOGRAPHS	SUPPLEMENT}', "");
Expect(0, 195103, '\p{^Is_Blk=cjk_Compatibility	IDEOGRAPHS	SUPPLEMENT}', "");
Expect(0, 195103, '\P{Is_Blk=cjk_Compatibility	IDEOGRAPHS	SUPPLEMENT}', "");
Expect(1, 195103, '\P{^Is_Blk=cjk_Compatibility	IDEOGRAPHS	SUPPLEMENT}', "");
Expect(0, 194559, '\p{Is_Blk=cjk_Compatibility	IDEOGRAPHS	SUPPLEMENT}', "");
Expect(1, 194559, '\p{^Is_Blk=cjk_Compatibility	IDEOGRAPHS	SUPPLEMENT}', "");
Expect(1, 194559, '\P{Is_Blk=cjk_Compatibility	IDEOGRAPHS	SUPPLEMENT}', "");
Expect(0, 194559, '\P{^Is_Blk=cjk_Compatibility	IDEOGRAPHS	SUPPLEMENT}', "");
Error('\p{Block= _IPA	EXTENSIONS/a/}');
Error('\P{Block= _IPA	EXTENSIONS/a/}');
Expect(1, 687, '\p{Block=ipaextensions}', "");
Expect(0, 687, '\p{^Block=ipaextensions}', "");
Expect(0, 687, '\P{Block=ipaextensions}', "");
Expect(1, 687, '\P{^Block=ipaextensions}', "");
Expect(0, 591, '\p{Block=ipaextensions}', "");
Expect(1, 591, '\p{^Block=ipaextensions}', "");
Expect(1, 591, '\P{Block=ipaextensions}', "");
Expect(0, 591, '\P{^Block=ipaextensions}', "");
Expect(1, 687, '\p{Block:	- IPA Extensions}', "");
Expect(0, 687, '\p{^Block:	- IPA Extensions}', "");
Expect(0, 687, '\P{Block:	- IPA Extensions}', "");
Expect(1, 687, '\P{^Block:	- IPA Extensions}', "");
Expect(0, 591, '\p{Block:	- IPA Extensions}', "");
Expect(1, 591, '\p{^Block:	- IPA Extensions}', "");
Expect(1, 591, '\P{Block:	- IPA Extensions}', "");
Expect(0, 591, '\P{^Block:	- IPA Extensions}', "");
Error('\p{Blk=--IPA_Extensions/a/}');
Error('\P{Blk=--IPA_Extensions/a/}');
Expect(1, 687, '\p{Blk=ipaextensions}', "");
Expect(0, 687, '\p{^Blk=ipaextensions}', "");
Expect(0, 687, '\P{Blk=ipaextensions}', "");
Expect(1, 687, '\P{^Blk=ipaextensions}', "");
Expect(0, 591, '\p{Blk=ipaextensions}', "");
Expect(1, 591, '\p{^Blk=ipaextensions}', "");
Expect(1, 591, '\P{Blk=ipaextensions}', "");
Expect(0, 591, '\P{^Blk=ipaextensions}', "");
Expect(1, 687, '\p{Blk=	 IPA_EXTENSIONS}', "");
Expect(0, 687, '\p{^Blk=	 IPA_EXTENSIONS}', "");
Expect(0, 687, '\P{Blk=	 IPA_EXTENSIONS}', "");
Expect(1, 687, '\P{^Blk=	 IPA_EXTENSIONS}', "");
Expect(0, 591, '\p{Blk=	 IPA_EXTENSIONS}', "");
Expect(1, 591, '\p{^Blk=	 IPA_EXTENSIONS}', "");
Expect(1, 591, '\P{Blk=	 IPA_EXTENSIONS}', "");
Expect(0, 591, '\P{^Blk=	 IPA_EXTENSIONS}', "");
Error('\p{Is_Block=--ipa extensions:=}');
Error('\P{Is_Block=--ipa extensions:=}');
Expect(1, 687, '\p{Is_Block=ipaextensions}', "");
Expect(0, 687, '\p{^Is_Block=ipaextensions}', "");
Expect(0, 687, '\P{Is_Block=ipaextensions}', "");
Expect(1, 687, '\P{^Is_Block=ipaextensions}', "");
Expect(0, 591, '\p{Is_Block=ipaextensions}', "");
Expect(1, 591, '\p{^Is_Block=ipaextensions}', "");
Expect(1, 591, '\P{Is_Block=ipaextensions}', "");
Expect(0, 591, '\P{^Is_Block=ipaextensions}', "");
Expect(1, 687, '\p{Is_Block:--IPA-Extensions}', "");
Expect(0, 687, '\p{^Is_Block:--IPA-Extensions}', "");
Expect(0, 687, '\P{Is_Block:--IPA-Extensions}', "");
Expect(1, 687, '\P{^Is_Block:--IPA-Extensions}', "");
Expect(0, 591, '\p{Is_Block:--IPA-Extensions}', "");
Expect(1, 591, '\p{^Is_Block:--IPA-Extensions}', "");
Expect(1, 591, '\P{Is_Block:--IPA-Extensions}', "");
Expect(0, 591, '\P{^Is_Block:--IPA-Extensions}', "");
Error('\p{Is_Blk=/a/-IPAextensions}');
Error('\P{Is_Blk=/a/-IPAextensions}');
Expect(1, 687, '\p{Is_Blk=ipaextensions}', "");
Expect(0, 687, '\p{^Is_Blk=ipaextensions}', "");
Expect(0, 687, '\P{Is_Blk=ipaextensions}', "");
Expect(1, 687, '\P{^Is_Blk=ipaextensions}', "");
Expect(0, 591, '\p{Is_Blk=ipaextensions}', "");
Expect(1, 591, '\p{^Is_Blk=ipaextensions}', "");
Expect(1, 591, '\P{Is_Blk=ipaextensions}', "");
Expect(0, 591, '\P{^Is_Blk=ipaextensions}', "");
Expect(1, 687, '\p{Is_Blk= -IPA EXTENSIONS}', "");
Expect(0, 687, '\p{^Is_Blk= -IPA EXTENSIONS}', "");
Expect(0, 687, '\P{Is_Blk= -IPA EXTENSIONS}', "");
Expect(1, 687, '\P{^Is_Blk= -IPA EXTENSIONS}', "");
Expect(0, 591, '\p{Is_Blk= -IPA EXTENSIONS}', "");
Expect(1, 591, '\p{^Is_Blk= -IPA EXTENSIONS}', "");
Expect(1, 591, '\P{Is_Blk= -IPA EXTENSIONS}', "");
Expect(0, 591, '\P{^Is_Blk= -IPA EXTENSIONS}', "");
Error('\p{Block=_Kannada/a/}');
Error('\P{Block=_Kannada/a/}');
Expect(1, 3327, '\p{Block=kannada}', "");
Expect(0, 3327, '\p{^Block=kannada}', "");
Expect(0, 3327, '\P{Block=kannada}', "");
Expect(1, 3327, '\P{^Block=kannada}', "");
Expect(0, 3199, '\p{Block=kannada}', "");
Expect(1, 3199, '\p{^Block=kannada}', "");
Expect(1, 3199, '\P{Block=kannada}', "");
Expect(0, 3199, '\P{^Block=kannada}', "");
Expect(1, 3327, '\p{Block=	kannada}', "");
Expect(0, 3327, '\p{^Block=	kannada}', "");
Expect(0, 3327, '\P{Block=	kannada}', "");
Expect(1, 3327, '\P{^Block=	kannada}', "");
Expect(0, 3199, '\p{Block=	kannada}', "");
Expect(1, 3199, '\p{^Block=	kannada}', "");
Expect(1, 3199, '\P{Block=	kannada}', "");
Expect(0, 3199, '\P{^Block=	kannada}', "");
Error('\p{Blk:	:=Kannada}');
Error('\P{Blk:	:=Kannada}');
Expect(1, 3327, '\p{Blk=kannada}', "");
Expect(0, 3327, '\p{^Blk=kannada}', "");
Expect(0, 3327, '\P{Blk=kannada}', "");
Expect(1, 3327, '\P{^Blk=kannada}', "");
Expect(0, 3199, '\p{Blk=kannada}', "");
Expect(1, 3199, '\p{^Blk=kannada}', "");
Expect(1, 3199, '\P{Blk=kannada}', "");
Expect(0, 3199, '\P{^Blk=kannada}', "");
Expect(1, 3327, '\p{Blk: -kannada}', "");
Expect(0, 3327, '\p{^Blk: -kannada}', "");
Expect(0, 3327, '\P{Blk: -kannada}', "");
Expect(1, 3327, '\P{^Blk: -kannada}', "");
Expect(0, 3199, '\p{Blk: -kannada}', "");
Expect(1, 3199, '\p{^Blk: -kannada}', "");
Expect(1, 3199, '\P{Blk: -kannada}', "");
Expect(0, 3199, '\P{^Blk: -kannada}', "");
Error('\p{Is_Block=:=	Kannada}');
Error('\P{Is_Block=:=	Kannada}');
Expect(1, 3327, '\p{Is_Block=kannada}', "");
Expect(0, 3327, '\p{^Is_Block=kannada}', "");
Expect(0, 3327, '\P{Is_Block=kannada}', "");
Expect(1, 3327, '\P{^Is_Block=kannada}', "");
Expect(0, 3199, '\p{Is_Block=kannada}', "");
Expect(1, 3199, '\p{^Is_Block=kannada}', "");
Expect(1, 3199, '\P{Is_Block=kannada}', "");
Expect(0, 3199, '\P{^Is_Block=kannada}', "");
Expect(1, 3327, '\p{Is_Block=	 KANNADA}', "");
Expect(0, 3327, '\p{^Is_Block=	 KANNADA}', "");
Expect(0, 3327, '\P{Is_Block=	 KANNADA}', "");
Expect(1, 3327, '\P{^Is_Block=	 KANNADA}', "");
Expect(0, 3199, '\p{Is_Block=	 KANNADA}', "");
Expect(1, 3199, '\p{^Is_Block=	 KANNADA}', "");
Expect(1, 3199, '\P{Is_Block=	 KANNADA}', "");
Expect(0, 3199, '\P{^Is_Block=	 KANNADA}', "");
Error('\p{Is_Blk= 	Kannada/a/}');
Error('\P{Is_Blk= 	Kannada/a/}');
Expect(1, 3327, '\p{Is_Blk=kannada}', "");
Expect(0, 3327, '\p{^Is_Blk=kannada}', "");
Expect(0, 3327, '\P{Is_Blk=kannada}', "");
Expect(1, 3327, '\P{^Is_Blk=kannada}', "");
Expect(0, 3199, '\p{Is_Blk=kannada}', "");
Expect(1, 3199, '\p{^Is_Blk=kannada}', "");
Expect(1, 3199, '\P{Is_Blk=kannada}', "");
Expect(0, 3199, '\P{^Is_Blk=kannada}', "");
Expect(1, 3327, '\p{Is_Blk=	_Kannada}', "");
Expect(0, 3327, '\p{^Is_Blk=	_Kannada}', "");
Expect(0, 3327, '\P{Is_Blk=	_Kannada}', "");
Expect(1, 3327, '\P{^Is_Blk=	_Kannada}', "");
Expect(0, 3199, '\p{Is_Blk=	_Kannada}', "");
Expect(1, 3199, '\p{^Is_Blk=	_Kannada}', "");
Expect(1, 3199, '\P{Is_Blk=	_Kannada}', "");
Expect(0, 3199, '\P{^Is_Blk=	_Kannada}', "");
Error('\p{Block=	_LatinExtended/a/ADDITIONAL}');
Error('\P{Block=	_LatinExtended/a/ADDITIONAL}');
Expect(1, 7935, '\p{Block=latinextendedadditional}', "");
Expect(0, 7935, '\p{^Block=latinextendedadditional}', "");
Expect(0, 7935, '\P{Block=latinextendedadditional}', "");
Expect(1, 7935, '\P{^Block=latinextendedadditional}', "");
Expect(0, 7679, '\p{Block=latinextendedadditional}', "");
Expect(1, 7679, '\p{^Block=latinextendedadditional}', "");
Expect(1, 7679, '\P{Block=latinextendedadditional}', "");
Expect(0, 7679, '\P{^Block=latinextendedadditional}', "");
Expect(1, 7935, '\p{Block=-_LATIN_ExtendedADDITIONAL}', "");
Expect(0, 7935, '\p{^Block=-_LATIN_ExtendedADDITIONAL}', "");
Expect(0, 7935, '\P{Block=-_LATIN_ExtendedADDITIONAL}', "");
Expect(1, 7935, '\P{^Block=-_LATIN_ExtendedADDITIONAL}', "");
Expect(0, 7679, '\p{Block=-_LATIN_ExtendedADDITIONAL}', "");
Expect(1, 7679, '\p{^Block=-_LATIN_ExtendedADDITIONAL}', "");
Expect(1, 7679, '\P{Block=-_LATIN_ExtendedADDITIONAL}', "");
Expect(0, 7679, '\P{^Block=-_LATIN_ExtendedADDITIONAL}', "");
Error('\p{Blk=_:=LATIN	EXTENDED_additional}');
Error('\P{Blk=_:=LATIN	EXTENDED_additional}');
Expect(1, 7935, '\p{Blk=latinextendedadditional}', "");
Expect(0, 7935, '\p{^Blk=latinextendedadditional}', "");
Expect(0, 7935, '\P{Blk=latinextendedadditional}', "");
Expect(1, 7935, '\P{^Blk=latinextendedadditional}', "");
Expect(0, 7679, '\p{Blk=latinextendedadditional}', "");
Expect(1, 7679, '\p{^Blk=latinextendedadditional}', "");
Expect(1, 7679, '\P{Blk=latinextendedadditional}', "");
Expect(0, 7679, '\P{^Blk=latinextendedadditional}', "");
Expect(1, 7935, '\p{Blk:   	latinExtended	Additional}', "");
Expect(0, 7935, '\p{^Blk:   	latinExtended	Additional}', "");
Expect(0, 7935, '\P{Blk:   	latinExtended	Additional}', "");
Expect(1, 7935, '\P{^Blk:   	latinExtended	Additional}', "");
Expect(0, 7679, '\p{Blk:   	latinExtended	Additional}', "");
Expect(1, 7679, '\p{^Blk:   	latinExtended	Additional}', "");
Expect(1, 7679, '\P{Blk:   	latinExtended	Additional}', "");
Expect(0, 7679, '\P{^Blk:   	latinExtended	Additional}', "");
Error('\p{Is_Block:   :=LATIN	Extended_Additional}');
Error('\P{Is_Block:   :=LATIN	Extended_Additional}');
Expect(1, 7935, '\p{Is_Block=latinextendedadditional}', "");
Expect(0, 7935, '\p{^Is_Block=latinextendedadditional}', "");
Expect(0, 7935, '\P{Is_Block=latinextendedadditional}', "");
Expect(1, 7935, '\P{^Is_Block=latinextendedadditional}', "");
Expect(0, 7679, '\p{Is_Block=latinextendedadditional}', "");
Expect(1, 7679, '\p{^Is_Block=latinextendedadditional}', "");
Expect(1, 7679, '\P{Is_Block=latinextendedadditional}', "");
Expect(0, 7679, '\P{^Is_Block=latinextendedadditional}', "");
Expect(1, 7935, '\p{Is_Block: 	latinextended Additional}', "");
Expect(0, 7935, '\p{^Is_Block: 	latinextended Additional}', "");
Expect(0, 7935, '\P{Is_Block: 	latinextended Additional}', "");
Expect(1, 7935, '\P{^Is_Block: 	latinextended Additional}', "");
Expect(0, 7679, '\p{Is_Block: 	latinextended Additional}', "");
Expect(1, 7679, '\p{^Is_Block: 	latinextended Additional}', "");
Expect(1, 7679, '\P{Is_Block: 	latinextended Additional}', "");
Expect(0, 7679, '\P{^Is_Block: 	latinextended Additional}', "");
Error('\p{Is_Blk:	LatinEXTENDED/a/Additional}');
Error('\P{Is_Blk:	LatinEXTENDED/a/Additional}');
Expect(1, 7935, '\p{Is_Blk=latinextendedadditional}', "");
Expect(0, 7935, '\p{^Is_Blk=latinextendedadditional}', "");
Expect(0, 7935, '\P{Is_Blk=latinextendedadditional}', "");
Expect(1, 7935, '\P{^Is_Blk=latinextendedadditional}', "");
Expect(0, 7679, '\p{Is_Blk=latinextendedadditional}', "");
Expect(1, 7679, '\p{^Is_Blk=latinextendedadditional}', "");
Expect(1, 7679, '\P{Is_Blk=latinextendedadditional}', "");
Expect(0, 7679, '\P{^Is_Blk=latinextendedadditional}', "");
Expect(1, 7935, '\p{Is_Blk= 	latin	Extended_Additional}', "");
Expect(0, 7935, '\p{^Is_Blk= 	latin	Extended_Additional}', "");
Expect(0, 7935, '\P{Is_Blk= 	latin	Extended_Additional}', "");
Expect(1, 7935, '\P{^Is_Blk= 	latin	Extended_Additional}', "");
Expect(0, 7679, '\p{Is_Blk= 	latin	Extended_Additional}', "");
Expect(1, 7679, '\p{^Is_Blk= 	latin	Extended_Additional}', "");
Expect(1, 7679, '\P{Is_Blk= 	latin	Extended_Additional}', "");
Expect(0, 7679, '\P{^Is_Blk= 	latin	Extended_Additional}', "");
Error('\p{Block=:=- Bamum_supplement}');
Error('\P{Block=:=- Bamum_supplement}');
Expect(1, 92735, '\p{Block=bamumsupplement}', "");
Expect(0, 92735, '\p{^Block=bamumsupplement}', "");
Expect(0, 92735, '\P{Block=bamumsupplement}', "");
Expect(1, 92735, '\P{^Block=bamumsupplement}', "");
Expect(0, 92159, '\p{Block=bamumsupplement}', "");
Expect(1, 92159, '\p{^Block=bamumsupplement}', "");
Expect(1, 92159, '\P{Block=bamumsupplement}', "");
Expect(0, 92159, '\P{^Block=bamumsupplement}', "");
Expect(1, 92735, '\p{Block=__Bamum Supplement}', "");
Expect(0, 92735, '\p{^Block=__Bamum Supplement}', "");
Expect(0, 92735, '\P{Block=__Bamum Supplement}', "");
Expect(1, 92735, '\P{^Block=__Bamum Supplement}', "");
Expect(0, 92159, '\p{Block=__Bamum Supplement}', "");
Expect(1, 92159, '\p{^Block=__Bamum Supplement}', "");
Expect(1, 92159, '\P{Block=__Bamum Supplement}', "");
Expect(0, 92159, '\P{^Block=__Bamum Supplement}', "");
Error('\p{Blk=	-Bamum/a/SUPPLEMENT}');
Error('\P{Blk=	-Bamum/a/SUPPLEMENT}');
Expect(1, 92735, '\p{Blk=bamumsupplement}', "");
Expect(0, 92735, '\p{^Blk=bamumsupplement}', "");
Expect(0, 92735, '\P{Blk=bamumsupplement}', "");
Expect(1, 92735, '\P{^Blk=bamumsupplement}', "");
Expect(0, 92159, '\p{Blk=bamumsupplement}', "");
Expect(1, 92159, '\p{^Blk=bamumsupplement}', "");
Expect(1, 92159, '\P{Blk=bamumsupplement}', "");
Expect(0, 92159, '\P{^Blk=bamumsupplement}', "");
Expect(1, 92735, '\p{Blk=	Bamum Supplement}', "");
Expect(0, 92735, '\p{^Blk=	Bamum Supplement}', "");
Expect(0, 92735, '\P{Blk=	Bamum Supplement}', "");
Expect(1, 92735, '\P{^Blk=	Bamum Supplement}', "");
Expect(0, 92159, '\p{Blk=	Bamum Supplement}', "");
Expect(1, 92159, '\p{^Blk=	Bamum Supplement}', "");
Expect(1, 92159, '\P{Blk=	Bamum Supplement}', "");
Expect(0, 92159, '\P{^Blk=	Bamum Supplement}', "");
Error('\p{Is_Block=:=		Bamum	supplement}');
Error('\P{Is_Block=:=		Bamum	supplement}');
Expect(1, 92735, '\p{Is_Block:bamumsupplement}', "");
Expect(0, 92735, '\p{^Is_Block:bamumsupplement}', "");
Expect(0, 92735, '\P{Is_Block:bamumsupplement}', "");
Expect(1, 92735, '\P{^Is_Block:bamumsupplement}', "");
Expect(0, 92159, '\p{Is_Block:bamumsupplement}', "");
Expect(1, 92159, '\p{^Is_Block:bamumsupplement}', "");
Expect(1, 92159, '\P{Is_Block:bamumsupplement}', "");
Expect(0, 92159, '\P{^Is_Block:bamumsupplement}', "");
Expect(1, 92735, '\p{Is_Block=_bamum-SUPPLEMENT}', "");
Expect(0, 92735, '\p{^Is_Block=_bamum-SUPPLEMENT}', "");
Expect(0, 92735, '\P{Is_Block=_bamum-SUPPLEMENT}', "");
Expect(1, 92735, '\P{^Is_Block=_bamum-SUPPLEMENT}', "");
Expect(0, 92159, '\p{Is_Block=_bamum-SUPPLEMENT}', "");
Expect(1, 92159, '\p{^Is_Block=_bamum-SUPPLEMENT}', "");
Expect(1, 92159, '\P{Is_Block=_bamum-SUPPLEMENT}', "");
Expect(0, 92159, '\P{^Is_Block=_bamum-SUPPLEMENT}', "");
Error('\p{Is_Blk= -Bamum_supplement:=}');
Error('\P{Is_Blk= -Bamum_supplement:=}');
Expect(1, 92735, '\p{Is_Blk=bamumsupplement}', "");
Expect(0, 92735, '\p{^Is_Blk=bamumsupplement}', "");
Expect(0, 92735, '\P{Is_Blk=bamumsupplement}', "");
Expect(1, 92735, '\P{^Is_Blk=bamumsupplement}', "");
Expect(0, 92159, '\p{Is_Blk=bamumsupplement}', "");
Expect(1, 92159, '\p{^Is_Blk=bamumsupplement}', "");
Expect(1, 92159, '\P{Is_Blk=bamumsupplement}', "");
Expect(0, 92159, '\P{^Is_Blk=bamumsupplement}', "");
Expect(1, 92735, '\p{Is_Blk=	BAMUM-SUPPLEMENT}', "");
Expect(0, 92735, '\p{^Is_Blk=	BAMUM-SUPPLEMENT}', "");
Expect(0, 92735, '\P{Is_Blk=	BAMUM-SUPPLEMENT}', "");
Expect(1, 92735, '\P{^Is_Blk=	BAMUM-SUPPLEMENT}', "");
Expect(0, 92159, '\p{Is_Blk=	BAMUM-SUPPLEMENT}', "");
Expect(1, 92159, '\p{^Is_Blk=	BAMUM-SUPPLEMENT}', "");
Expect(1, 92159, '\P{Is_Blk=	BAMUM-SUPPLEMENT}', "");
Expect(0, 92159, '\P{^Is_Blk=	BAMUM-SUPPLEMENT}', "");
Error('\p{Block=-CHAM/a/}');
Error('\P{Block=-CHAM/a/}');
Expect(1, 43615, '\p{Block=cham}', "");
Expect(0, 43615, '\p{^Block=cham}', "");
Expect(0, 43615, '\P{Block=cham}', "");
Expect(1, 43615, '\P{^Block=cham}', "");
Expect(0, 43519, '\p{Block=cham}', "");
Expect(1, 43519, '\p{^Block=cham}', "");
Expect(1, 43519, '\P{Block=cham}', "");
Expect(0, 43519, '\P{^Block=cham}', "");
Expect(1, 43615, '\p{Block=	cham}', "");
Expect(0, 43615, '\p{^Block=	cham}', "");
Expect(0, 43615, '\P{Block=	cham}', "");
Expect(1, 43615, '\P{^Block=	cham}', "");
Expect(0, 43519, '\p{Block=	cham}', "");
Expect(1, 43519, '\p{^Block=	cham}', "");
Expect(1, 43519, '\P{Block=	cham}', "");
Expect(0, 43519, '\P{^Block=	cham}', "");
Error('\p{Blk: _:=Cham}');
Error('\P{Blk: _:=Cham}');
Expect(1, 43615, '\p{Blk=cham}', "");
Expect(0, 43615, '\p{^Blk=cham}', "");
Expect(0, 43615, '\P{Blk=cham}', "");
Expect(1, 43615, '\P{^Blk=cham}', "");
Expect(0, 43519, '\p{Blk=cham}', "");
Expect(1, 43519, '\p{^Blk=cham}', "");
Expect(1, 43519, '\P{Blk=cham}', "");
Expect(0, 43519, '\P{^Blk=cham}', "");
Expect(1, 43615, '\p{Blk= Cham}', "");
Expect(0, 43615, '\p{^Blk= Cham}', "");
Expect(0, 43615, '\P{Blk= Cham}', "");
Expect(1, 43615, '\P{^Blk= Cham}', "");
Expect(0, 43519, '\p{Blk= Cham}', "");
Expect(1, 43519, '\p{^Blk= Cham}', "");
Expect(1, 43519, '\P{Blk= Cham}', "");
Expect(0, 43519, '\P{^Blk= Cham}', "");
Error('\p{Is_Block:   _/a/Cham}');
Error('\P{Is_Block:   _/a/Cham}');
Expect(1, 43615, '\p{Is_Block=cham}', "");
Expect(0, 43615, '\p{^Is_Block=cham}', "");
Expect(0, 43615, '\P{Is_Block=cham}', "");
Expect(1, 43615, '\P{^Is_Block=cham}', "");
Expect(0, 43519, '\p{Is_Block=cham}', "");
Expect(1, 43519, '\p{^Is_Block=cham}', "");
Expect(1, 43519, '\P{Is_Block=cham}', "");
Expect(0, 43519, '\P{^Is_Block=cham}', "");
Expect(1, 43615, '\p{Is_Block=-CHAM}', "");
Expect(0, 43615, '\p{^Is_Block=-CHAM}', "");
Expect(0, 43615, '\P{Is_Block=-CHAM}', "");
Expect(1, 43615, '\P{^Is_Block=-CHAM}', "");
Expect(0, 43519, '\p{Is_Block=-CHAM}', "");
Expect(1, 43519, '\p{^Is_Block=-CHAM}', "");
Expect(1, 43519, '\P{Is_Block=-CHAM}', "");
Expect(0, 43519, '\P{^Is_Block=-CHAM}', "");
Error('\p{Is_Blk=/a/--Cham}');
Error('\P{Is_Blk=/a/--Cham}');
Expect(1, 43615, '\p{Is_Blk=cham}', "");
Expect(0, 43615, '\p{^Is_Blk=cham}', "");
Expect(0, 43615, '\P{Is_Blk=cham}', "");
Expect(1, 43615, '\P{^Is_Blk=cham}', "");
Expect(0, 43519, '\p{Is_Blk=cham}', "");
Expect(1, 43519, '\p{^Is_Blk=cham}', "");
Expect(1, 43519, '\P{Is_Blk=cham}', "");
Expect(0, 43519, '\P{^Is_Blk=cham}', "");
Expect(1, 43615, '\p{Is_Blk=-	CHAM}', "");
Expect(0, 43615, '\p{^Is_Blk=-	CHAM}', "");
Expect(0, 43615, '\P{Is_Blk=-	CHAM}', "");
Expect(1, 43615, '\P{^Is_Blk=-	CHAM}', "");
Expect(0, 43519, '\p{Is_Blk=-	CHAM}', "");
Expect(1, 43519, '\p{^Is_Blk=-	CHAM}', "");
Expect(1, 43519, '\P{Is_Blk=-	CHAM}', "");
Expect(0, 43519, '\P{^Is_Blk=-	CHAM}', "");
Error('\p{Block= Cyrillic supplement/a/}');
Error('\P{Block= Cyrillic supplement/a/}');
Expect(1, 1327, '\p{Block=cyrillicsupplement}', "");
Expect(0, 1327, '\p{^Block=cyrillicsupplement}', "");
Expect(0, 1327, '\P{Block=cyrillicsupplement}', "");
Expect(1, 1327, '\P{^Block=cyrillicsupplement}', "");
Expect(0, 1279, '\p{Block=cyrillicsupplement}', "");
Expect(1, 1279, '\p{^Block=cyrillicsupplement}', "");
Expect(1, 1279, '\P{Block=cyrillicsupplement}', "");
Expect(0, 1279, '\P{^Block=cyrillicsupplement}', "");
Expect(1, 1327, '\p{Block:		_Cyrillic	supplement}', "");
Expect(0, 1327, '\p{^Block:		_Cyrillic	supplement}', "");
Expect(0, 1327, '\P{Block:		_Cyrillic	supplement}', "");
Expect(1, 1327, '\P{^Block:		_Cyrillic	supplement}', "");
Expect(0, 1279, '\p{Block:		_Cyrillic	supplement}', "");
Expect(1, 1279, '\p{^Block:		_Cyrillic	supplement}', "");
Expect(1, 1279, '\P{Block:		_Cyrillic	supplement}', "");
Expect(0, 1279, '\P{^Block:		_Cyrillic	supplement}', "");
Error('\p{Blk=/a/ -Cyrillic SUPPLEMENTARY}');
Error('\P{Blk=/a/ -Cyrillic SUPPLEMENTARY}');
Expect(1, 1327, '\p{Blk:cyrillicsupplementary}', "");
Expect(0, 1327, '\p{^Blk:cyrillicsupplementary}', "");
Expect(0, 1327, '\P{Blk:cyrillicsupplementary}', "");
Expect(1, 1327, '\P{^Blk:cyrillicsupplementary}', "");
Expect(0, 1279, '\p{Blk:cyrillicsupplementary}', "");
Expect(1, 1279, '\p{^Blk:cyrillicsupplementary}', "");
Expect(1, 1279, '\P{Blk:cyrillicsupplementary}', "");
Expect(0, 1279, '\P{^Blk:cyrillicsupplementary}', "");
Expect(1, 1327, '\p{Blk:__CYRILLICSUPPLEMENTARY}', "");
Expect(0, 1327, '\p{^Blk:__CYRILLICSUPPLEMENTARY}', "");
Expect(0, 1327, '\P{Blk:__CYRILLICSUPPLEMENTARY}', "");
Expect(1, 1327, '\P{^Blk:__CYRILLICSUPPLEMENTARY}', "");
Expect(0, 1279, '\p{Blk:__CYRILLICSUPPLEMENTARY}', "");
Expect(1, 1279, '\p{^Blk:__CYRILLICSUPPLEMENTARY}', "");
Expect(1, 1279, '\P{Blk:__CYRILLICSUPPLEMENTARY}', "");
Expect(0, 1279, '\P{^Blk:__CYRILLICSUPPLEMENTARY}', "");
Error('\p{Is_Block=/a/CYRILLIC-supplement}');
Error('\P{Is_Block=/a/CYRILLIC-supplement}');
Expect(1, 1327, '\p{Is_Block=cyrillicsupplement}', "");
Expect(0, 1327, '\p{^Is_Block=cyrillicsupplement}', "");
Expect(0, 1327, '\P{Is_Block=cyrillicsupplement}', "");
Expect(1, 1327, '\P{^Is_Block=cyrillicsupplement}', "");
Expect(0, 1279, '\p{Is_Block=cyrillicsupplement}', "");
Expect(1, 1279, '\p{^Is_Block=cyrillicsupplement}', "");
Expect(1, 1279, '\P{Is_Block=cyrillicsupplement}', "");
Expect(0, 1279, '\P{^Is_Block=cyrillicsupplement}', "");
Expect(1, 1327, '\p{Is_Block=-_CYRILLICSUPPLEMENT}', "");
Expect(0, 1327, '\p{^Is_Block=-_CYRILLICSUPPLEMENT}', "");
Expect(0, 1327, '\P{Is_Block=-_CYRILLICSUPPLEMENT}', "");
Expect(1, 1327, '\P{^Is_Block=-_CYRILLICSUPPLEMENT}', "");
Expect(0, 1279, '\p{Is_Block=-_CYRILLICSUPPLEMENT}', "");
Expect(1, 1279, '\p{^Is_Block=-_CYRILLICSUPPLEMENT}', "");
Expect(1, 1279, '\P{Is_Block=-_CYRILLICSUPPLEMENT}', "");
Expect(0, 1279, '\P{^Is_Block=-_CYRILLICSUPPLEMENT}', "");
Error('\p{Is_Blk=_Cyrillic	supplementary/a/}');
Error('\P{Is_Blk=_Cyrillic	supplementary/a/}');
Expect(1, 1327, '\p{Is_Blk=cyrillicsupplementary}', "");
Expect(0, 1327, '\p{^Is_Blk=cyrillicsupplementary}', "");
Expect(0, 1327, '\P{Is_Blk=cyrillicsupplementary}', "");
Expect(1, 1327, '\P{^Is_Blk=cyrillicsupplementary}', "");
Expect(0, 1279, '\p{Is_Blk=cyrillicsupplementary}', "");
Expect(1, 1279, '\p{^Is_Blk=cyrillicsupplementary}', "");
Expect(1, 1279, '\P{Is_Blk=cyrillicsupplementary}', "");
Expect(0, 1279, '\P{^Is_Blk=cyrillicsupplementary}', "");
Expect(1, 1327, '\p{Is_Blk= cyrillic-supplementary}', "");
Expect(0, 1327, '\p{^Is_Blk= cyrillic-supplementary}', "");
Expect(0, 1327, '\P{Is_Blk= cyrillic-supplementary}', "");
Expect(1, 1327, '\P{^Is_Blk= cyrillic-supplementary}', "");
Expect(0, 1279, '\p{Is_Blk= cyrillic-supplementary}', "");
Expect(1, 1279, '\p{^Is_Blk= cyrillic-supplementary}', "");
Expect(1, 1279, '\P{Is_Blk= cyrillic-supplementary}', "");
Expect(0, 1279, '\P{^Is_Blk= cyrillic-supplementary}', "");
Error('\p{Block=_ greekAND/a/Coptic}');
Error('\P{Block=_ greekAND/a/Coptic}');
Expect(1, 1023, '\p{Block=greekandcoptic}', "");
Expect(0, 1023, '\p{^Block=greekandcoptic}', "");
Expect(0, 1023, '\P{Block=greekandcoptic}', "");
Expect(1, 1023, '\P{^Block=greekandcoptic}', "");
Expect(0, 879, '\p{Block=greekandcoptic}', "");
Expect(1, 879, '\p{^Block=greekandcoptic}', "");
Expect(1, 879, '\P{Block=greekandcoptic}', "");
Expect(0, 879, '\P{^Block=greekandcoptic}', "");
Expect(1, 1023, '\p{Block= Greek And_COPTIC}', "");
Expect(0, 1023, '\p{^Block= Greek And_COPTIC}', "");
Expect(0, 1023, '\P{Block= Greek And_COPTIC}', "");
Expect(1, 1023, '\P{^Block= Greek And_COPTIC}', "");
Expect(0, 879, '\p{Block= Greek And_COPTIC}', "");
Expect(1, 879, '\p{^Block= Greek And_COPTIC}', "");
Expect(1, 879, '\P{Block= Greek And_COPTIC}', "");
Expect(0, 879, '\P{^Block= Greek And_COPTIC}', "");
Error('\p{Blk=:=-	Greek}');
Error('\P{Blk=:=-	Greek}');
Expect(1, 1023, '\p{Blk=greek}', "");
Expect(0, 1023, '\p{^Blk=greek}', "");
Expect(0, 1023, '\P{Blk=greek}', "");
Expect(1, 1023, '\P{^Blk=greek}', "");
Expect(0, 879, '\p{Blk=greek}', "");
Expect(1, 879, '\p{^Blk=greek}', "");
Expect(1, 879, '\P{Blk=greek}', "");
Expect(0, 879, '\P{^Blk=greek}', "");
Expect(1, 1023, '\p{Blk=__greek}', "");
Expect(0, 1023, '\p{^Blk=__greek}', "");
Expect(0, 1023, '\P{Blk=__greek}', "");
Expect(1, 1023, '\P{^Blk=__greek}', "");
Expect(0, 879, '\p{Blk=__greek}', "");
Expect(1, 879, '\p{^Blk=__greek}', "");
Expect(1, 879, '\P{Blk=__greek}', "");
Expect(0, 879, '\P{^Blk=__greek}', "");
Error('\p{Is_Block=	Greek/a/And	Coptic}');
Error('\P{Is_Block=	Greek/a/And	Coptic}');
Expect(1, 1023, '\p{Is_Block=greekandcoptic}', "");
Expect(0, 1023, '\p{^Is_Block=greekandcoptic}', "");
Expect(0, 1023, '\P{Is_Block=greekandcoptic}', "");
Expect(1, 1023, '\P{^Is_Block=greekandcoptic}', "");
Expect(0, 879, '\p{Is_Block=greekandcoptic}', "");
Expect(1, 879, '\p{^Is_Block=greekandcoptic}', "");
Expect(1, 879, '\P{Is_Block=greekandcoptic}', "");
Expect(0, 879, '\P{^Is_Block=greekandcoptic}', "");
Expect(1, 1023, '\p{Is_Block=__GreekandCoptic}', "");
Expect(0, 1023, '\p{^Is_Block=__GreekandCoptic}', "");
Expect(0, 1023, '\P{Is_Block=__GreekandCoptic}', "");
Expect(1, 1023, '\P{^Is_Block=__GreekandCoptic}', "");
Expect(0, 879, '\p{Is_Block=__GreekandCoptic}', "");
Expect(1, 879, '\p{^Is_Block=__GreekandCoptic}', "");
Expect(1, 879, '\P{Is_Block=__GreekandCoptic}', "");
Expect(0, 879, '\P{^Is_Block=__GreekandCoptic}', "");
Error('\p{Is_Blk=-/a/Greek}');
Error('\P{Is_Blk=-/a/Greek}');
Expect(1, 1023, '\p{Is_Blk=greek}', "");
Expect(0, 1023, '\p{^Is_Blk=greek}', "");
Expect(0, 1023, '\P{Is_Blk=greek}', "");
Expect(1, 1023, '\P{^Is_Blk=greek}', "");
Expect(0, 879, '\p{Is_Blk=greek}', "");
Expect(1, 879, '\p{^Is_Blk=greek}', "");
Expect(1, 879, '\P{Is_Blk=greek}', "");
Expect(0, 879, '\P{^Is_Blk=greek}', "");
Expect(1, 1023, '\p{Is_Blk=- GREEK}', "");
Expect(0, 1023, '\p{^Is_Blk=- GREEK}', "");
Expect(0, 1023, '\P{Is_Blk=- GREEK}', "");
Expect(1, 1023, '\P{^Is_Blk=- GREEK}', "");
Expect(0, 879, '\p{Is_Blk=- GREEK}', "");
Expect(1, 879, '\p{^Is_Blk=- GREEK}', "");
Expect(1, 879, '\P{Is_Blk=- GREEK}', "");
Expect(0, 879, '\P{^Is_Blk=- GREEK}', "");
Error('\p{Block: _:=MISCELLANEOUS-MATHEMATICAL	SYMBOLS B}');
Error('\P{Block: _:=MISCELLANEOUS-MATHEMATICAL	SYMBOLS B}');
Expect(1, 10751, '\p{Block=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10751, '\p{^Block=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10751, '\P{Block=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10751, '\P{^Block=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10623, '\p{Block=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10623, '\p{^Block=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10623, '\P{Block=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10623, '\P{^Block=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10751, '\p{Block=		miscellaneousMATHEMATICAL	SYMBOLS	B}', "");
Expect(0, 10751, '\p{^Block=		miscellaneousMATHEMATICAL	SYMBOLS	B}', "");
Expect(0, 10751, '\P{Block=		miscellaneousMATHEMATICAL	SYMBOLS	B}', "");
Expect(1, 10751, '\P{^Block=		miscellaneousMATHEMATICAL	SYMBOLS	B}', "");
Expect(0, 10623, '\p{Block=		miscellaneousMATHEMATICAL	SYMBOLS	B}', "");
Expect(1, 10623, '\p{^Block=		miscellaneousMATHEMATICAL	SYMBOLS	B}', "");
Expect(1, 10623, '\P{Block=		miscellaneousMATHEMATICAL	SYMBOLS	B}', "");
Expect(0, 10623, '\P{^Block=		miscellaneousMATHEMATICAL	SYMBOLS	B}', "");
Error('\p{Blk=	-Miscellaneous MATHEMATICAL Symbols	B:=}');
Error('\P{Blk=	-Miscellaneous MATHEMATICAL Symbols	B:=}');
Expect(1, 10751, '\p{Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10751, '\p{^Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10751, '\P{Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10751, '\P{^Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10623, '\p{Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10623, '\p{^Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10623, '\P{Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10623, '\P{^Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10751, '\p{Blk= 	MISCELLANEOUS	Mathematical-SYMBOLS B}', "");
Expect(0, 10751, '\p{^Blk= 	MISCELLANEOUS	Mathematical-SYMBOLS B}', "");
Expect(0, 10751, '\P{Blk= 	MISCELLANEOUS	Mathematical-SYMBOLS B}', "");
Expect(1, 10751, '\P{^Blk= 	MISCELLANEOUS	Mathematical-SYMBOLS B}', "");
Expect(0, 10623, '\p{Blk= 	MISCELLANEOUS	Mathematical-SYMBOLS B}', "");
Expect(1, 10623, '\p{^Blk= 	MISCELLANEOUS	Mathematical-SYMBOLS B}', "");
Expect(1, 10623, '\P{Blk= 	MISCELLANEOUS	Mathematical-SYMBOLS B}', "");
Expect(0, 10623, '\P{^Blk= 	MISCELLANEOUS	Mathematical-SYMBOLS B}', "");
Error('\p{Is_Block=:=Miscellaneous_Mathematical_symbols b}');
Error('\P{Is_Block=:=Miscellaneous_Mathematical_symbols b}');
Expect(1, 10751, '\p{Is_Block=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10751, '\p{^Is_Block=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10751, '\P{Is_Block=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10751, '\P{^Is_Block=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10623, '\p{Is_Block=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10623, '\p{^Is_Block=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10623, '\P{Is_Block=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10623, '\P{^Is_Block=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10751, '\p{Is_Block= _Miscellaneous	MATHEMATICALsymbols b}', "");
Expect(0, 10751, '\p{^Is_Block= _Miscellaneous	MATHEMATICALsymbols b}', "");
Expect(0, 10751, '\P{Is_Block= _Miscellaneous	MATHEMATICALsymbols b}', "");
Expect(1, 10751, '\P{^Is_Block= _Miscellaneous	MATHEMATICALsymbols b}', "");
Expect(0, 10623, '\p{Is_Block= _Miscellaneous	MATHEMATICALsymbols b}', "");
Expect(1, 10623, '\p{^Is_Block= _Miscellaneous	MATHEMATICALsymbols b}', "");
Expect(1, 10623, '\P{Is_Block= _Miscellaneous	MATHEMATICALsymbols b}', "");
Expect(0, 10623, '\P{^Is_Block= _Miscellaneous	MATHEMATICALsymbols b}', "");
Error('\p{Is_Blk=:=	MISCELLANEOUS mathematical-Symbols_B}');
Error('\P{Is_Blk=:=	MISCELLANEOUS mathematical-Symbols_B}');
Expect(1, 10751, '\p{Is_Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10751, '\p{^Is_Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10751, '\P{Is_Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10751, '\P{^Is_Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10623, '\p{Is_Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10623, '\p{^Is_Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10623, '\P{Is_Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(0, 10623, '\P{^Is_Blk=miscellaneousmathematicalsymbolsb}', "");
Expect(1, 10751, '\p{Is_Blk=	miscellaneousMathematicalSymbols B}', "");
Expect(0, 10751, '\p{^Is_Blk=	miscellaneousMathematicalSymbols B}', "");
Expect(0, 10751, '\P{Is_Blk=	miscellaneousMathematicalSymbols B}', "");
Expect(1, 10751, '\P{^Is_Blk=	miscellaneousMathematicalSymbols B}', "");
Expect(0, 10623, '\p{Is_Blk=	miscellaneousMathematicalSymbols B}', "");
Expect(1, 10623, '\p{^Is_Blk=	miscellaneousMathematicalSymbols B}', "");
Expect(1, 10623, '\P{Is_Blk=	miscellaneousMathematicalSymbols B}', "");
Expect(0, 10623, '\P{^Is_Blk=	miscellaneousMathematicalSymbols B}', "");
Error('\p{Block=:= bengali}');
Error('\P{Block=:= bengali}');
Expect(1, 2559, '\p{Block=bengali}', "");
Expect(0, 2559, '\p{^Block=bengali}', "");
Expect(0, 2559, '\P{Block=bengali}', "");
Expect(1, 2559, '\P{^Block=bengali}', "");
Expect(0, 2431, '\p{Block=bengali}', "");
Expect(1, 2431, '\p{^Block=bengali}', "");
Expect(1, 2431, '\P{Block=bengali}', "");
Expect(0, 2431, '\P{^Block=bengali}', "");
Error('\p{Blk=-/a/Bengali}');
Error('\P{Blk=-/a/Bengali}');
Expect(1, 2559, '\p{Blk=bengali}', "");
Expect(0, 2559, '\p{^Blk=bengali}', "");
Expect(0, 2559, '\P{Blk=bengali}', "");
Expect(1, 2559, '\P{^Blk=bengali}', "");
Expect(0, 2431, '\p{Blk=bengali}', "");
Expect(1, 2431, '\p{^Blk=bengali}', "");
Expect(1, 2431, '\P{Blk=bengali}', "");
Expect(0, 2431, '\P{^Blk=bengali}', "");
Expect(1, 2559, '\p{Blk=- BENGALI}', "");
Expect(0, 2559, '\p{^Blk=- BENGALI}', "");
Expect(0, 2559, '\P{Blk=- BENGALI}', "");
Expect(1, 2559, '\P{^Blk=- BENGALI}', "");
Expect(0, 2431, '\p{Blk=- BENGALI}', "");
Expect(1, 2431, '\p{^Blk=- BENGALI}', "");
Expect(1, 2431, '\P{Blk=- BENGALI}', "");
Expect(0, 2431, '\P{^Blk=- BENGALI}', "");
Error('\p{Is_Block=:= Bengali}');
Error('\P{Is_Block=:= Bengali}');
Expect(1, 2559, '\p{Is_Block=bengali}', "");
Expect(0, 2559, '\p{^Is_Block=bengali}', "");
Expect(0, 2559, '\P{Is_Block=bengali}', "");
Expect(1, 2559, '\P{^Is_Block=bengali}', "");
Expect(0, 2431, '\p{Is_Block=bengali}', "");
Expect(1, 2431, '\p{^Is_Block=bengali}', "");
Expect(1, 2431, '\P{Is_Block=bengali}', "");
Expect(0, 2431, '\P{^Is_Block=bengali}', "");
Expect(1, 2559, '\p{Is_Block=	_BENGALI}', "");
Expect(0, 2559, '\p{^Is_Block=	_BENGALI}', "");
Expect(0, 2559, '\P{Is_Block=	_BENGALI}', "");
Expect(1, 2559, '\P{^Is_Block=	_BENGALI}', "");
Expect(0, 2431, '\p{Is_Block=	_BENGALI}', "");
Expect(1, 2431, '\p{^Is_Block=	_BENGALI}', "");
Expect(1, 2431, '\P{Is_Block=	_BENGALI}', "");
Expect(0, 2431, '\P{^Is_Block=	_BENGALI}', "");
Error('\p{Is_Blk=:=-BENGALI}');
Error('\P{Is_Blk=:=-BENGALI}');
Expect(1, 2559, '\p{Is_Blk=bengali}', "");
Expect(0, 2559, '\p{^Is_Blk=bengali}', "");
Expect(0, 2559, '\P{Is_Blk=bengali}', "");
Expect(1, 2559, '\P{^Is_Blk=bengali}', "");
Expect(0, 2431, '\p{Is_Blk=bengali}', "");
Expect(1, 2431, '\p{^Is_Blk=bengali}', "");
Expect(1, 2431, '\P{Is_Blk=bengali}', "");
Expect(0, 2431, '\P{^Is_Blk=bengali}', "");
Expect(1, 2559, '\p{Is_Blk= 	Bengali}', "");
Expect(0, 2559, '\p{^Is_Blk= 	Bengali}', "");
Expect(0, 2559, '\P{Is_Blk= 	Bengali}', "");
Expect(1, 2559, '\P{^Is_Blk= 	Bengali}', "");
Expect(0, 2431, '\p{Is_Blk= 	Bengali}', "");
Expect(1, 2431, '\p{^Is_Blk= 	Bengali}', "");
Expect(1, 2431, '\P{Is_Blk= 	Bengali}', "");
Expect(0, 2431, '\P{^Is_Blk= 	Bengali}', "");
Error('\p{Block= :=cypriot-SYLLABARY}');
Error('\P{Block= :=cypriot-SYLLABARY}');
Expect(1, 67647, '\p{Block:	cypriotsyllabary}', "");
Expect(0, 67647, '\p{^Block:	cypriotsyllabary}', "");
Expect(0, 67647, '\P{Block:	cypriotsyllabary}', "");
Expect(1, 67647, '\P{^Block:	cypriotsyllabary}', "");
Expect(0, 67583, '\p{Block:	cypriotsyllabary}', "");
Expect(1, 67583, '\p{^Block:	cypriotsyllabary}', "");
Expect(1, 67583, '\P{Block:	cypriotsyllabary}', "");
Expect(0, 67583, '\P{^Block:	cypriotsyllabary}', "");
Expect(1, 67647, '\p{Block=-cypriot	SYLLABARY}', "");
Expect(0, 67647, '\p{^Block=-cypriot	SYLLABARY}', "");
Expect(0, 67647, '\P{Block=-cypriot	SYLLABARY}', "");
Expect(1, 67647, '\P{^Block=-cypriot	SYLLABARY}', "");
Expect(0, 67583, '\p{Block=-cypriot	SYLLABARY}', "");
Expect(1, 67583, '\p{^Block=-cypriot	SYLLABARY}', "");
Expect(1, 67583, '\P{Block=-cypriot	SYLLABARY}', "");
Expect(0, 67583, '\P{^Block=-cypriot	SYLLABARY}', "");
Error('\p{Blk=-Cypriot:=syllabary}');
Error('\P{Blk=-Cypriot:=syllabary}');
Expect(1, 67647, '\p{Blk=cypriotsyllabary}', "");
Expect(0, 67647, '\p{^Blk=cypriotsyllabary}', "");
Expect(0, 67647, '\P{Blk=cypriotsyllabary}', "");
Expect(1, 67647, '\P{^Blk=cypriotsyllabary}', "");
Expect(0, 67583, '\p{Blk=cypriotsyllabary}', "");
Expect(1, 67583, '\p{^Blk=cypriotsyllabary}', "");
Expect(1, 67583, '\P{Blk=cypriotsyllabary}', "");
Expect(0, 67583, '\P{^Blk=cypriotsyllabary}', "");
Expect(1, 67647, '\p{Blk=_Cypriot-SYLLABARY}', "");
Expect(0, 67647, '\p{^Blk=_Cypriot-SYLLABARY}', "");
Expect(0, 67647, '\P{Blk=_Cypriot-SYLLABARY}', "");
Expect(1, 67647, '\P{^Blk=_Cypriot-SYLLABARY}', "");
Expect(0, 67583, '\p{Blk=_Cypriot-SYLLABARY}', "");
Expect(1, 67583, '\p{^Blk=_Cypriot-SYLLABARY}', "");
Expect(1, 67583, '\P{Blk=_Cypriot-SYLLABARY}', "");
Expect(0, 67583, '\P{^Blk=_Cypriot-SYLLABARY}', "");
Error('\p{Is_Block=/a/_	cypriot Syllabary}');
Error('\P{Is_Block=/a/_	cypriot Syllabary}');
Expect(1, 67647, '\p{Is_Block=cypriotsyllabary}', "");
Expect(0, 67647, '\p{^Is_Block=cypriotsyllabary}', "");
Expect(0, 67647, '\P{Is_Block=cypriotsyllabary}', "");
Expect(1, 67647, '\P{^Is_Block=cypriotsyllabary}', "");
Expect(0, 67583, '\p{Is_Block=cypriotsyllabary}', "");
Expect(1, 67583, '\p{^Is_Block=cypriotsyllabary}', "");
Expect(1, 67583, '\P{Is_Block=cypriotsyllabary}', "");
Expect(0, 67583, '\P{^Is_Block=cypriotsyllabary}', "");
Expect(1, 67647, '\p{Is_Block=-CypriotSyllabary}', "");
Expect(0, 67647, '\p{^Is_Block=-CypriotSyllabary}', "");
Expect(0, 67647, '\P{Is_Block=-CypriotSyllabary}', "");
Expect(1, 67647, '\P{^Is_Block=-CypriotSyllabary}', "");
Expect(0, 67583, '\p{Is_Block=-CypriotSyllabary}', "");
Expect(1, 67583, '\p{^Is_Block=-CypriotSyllabary}', "");
Expect(1, 67583, '\P{Is_Block=-CypriotSyllabary}', "");
Expect(0, 67583, '\P{^Is_Block=-CypriotSyllabary}', "");
Error('\p{Is_Blk=	/a/CypriotSYLLABARY}');
Error('\P{Is_Blk=	/a/CypriotSYLLABARY}');
Expect(1, 67647, '\p{Is_Blk=cypriotsyllabary}', "");
Expect(0, 67647, '\p{^Is_Blk=cypriotsyllabary}', "");
Expect(0, 67647, '\P{Is_Blk=cypriotsyllabary}', "");
Expect(1, 67647, '\P{^Is_Blk=cypriotsyllabary}', "");
Expect(0, 67583, '\p{Is_Blk=cypriotsyllabary}', "");
Expect(1, 67583, '\p{^Is_Blk=cypriotsyllabary}', "");
Expect(1, 67583, '\P{Is_Blk=cypriotsyllabary}', "");
Expect(0, 67583, '\P{^Is_Blk=cypriotsyllabary}', "");
Expect(1, 67647, '\p{Is_Blk=--CYPRIOT Syllabary}', "");
Expect(0, 67647, '\p{^Is_Blk=--CYPRIOT Syllabary}', "");
Expect(0, 67647, '\P{Is_Blk=--CYPRIOT Syllabary}', "");
Expect(1, 67647, '\P{^Is_Blk=--CYPRIOT Syllabary}', "");
Expect(0, 67583, '\p{Is_Blk=--CYPRIOT Syllabary}', "");
Expect(1, 67583, '\p{^Is_Blk=--CYPRIOT Syllabary}', "");
Expect(1, 67583, '\P{Is_Blk=--CYPRIOT Syllabary}', "");
Expect(0, 67583, '\P{^Is_Blk=--CYPRIOT Syllabary}', "");
Error('\p{Block=:=	supplemental	punctuation}');
Error('\P{Block=:=	supplemental	punctuation}');
Expect(1, 11903, '\p{Block=supplementalpunctuation}', "");
Expect(0, 11903, '\p{^Block=supplementalpunctuation}', "");
Expect(0, 11903, '\P{Block=supplementalpunctuation}', "");
Expect(1, 11903, '\P{^Block=supplementalpunctuation}', "");
Expect(0, 11775, '\p{Block=supplementalpunctuation}', "");
Expect(1, 11775, '\p{^Block=supplementalpunctuation}', "");
Expect(1, 11775, '\P{Block=supplementalpunctuation}', "");
Expect(0, 11775, '\P{^Block=supplementalpunctuation}', "");
Expect(1, 11903, '\p{Block: -Supplemental_Punctuation}', "");
Expect(0, 11903, '\p{^Block: -Supplemental_Punctuation}', "");
Expect(0, 11903, '\P{Block: -Supplemental_Punctuation}', "");
Expect(1, 11903, '\P{^Block: -Supplemental_Punctuation}', "");
Expect(0, 11775, '\p{Block: -Supplemental_Punctuation}', "");
Expect(1, 11775, '\p{^Block: -Supplemental_Punctuation}', "");
Expect(1, 11775, '\P{Block: -Supplemental_Punctuation}', "");
Expect(0, 11775, '\P{^Block: -Supplemental_Punctuation}', "");
Error('\p{Blk=	Supplemental:=PUNCTUATION}');
Error('\P{Blk=	Supplemental:=PUNCTUATION}');
Expect(1, 11903, '\p{Blk=supplementalpunctuation}', "");
Expect(0, 11903, '\p{^Blk=supplementalpunctuation}', "");
Expect(0, 11903, '\P{Blk=supplementalpunctuation}', "");
Expect(1, 11903, '\P{^Blk=supplementalpunctuation}', "");
Expect(0, 11775, '\p{Blk=supplementalpunctuation}', "");
Expect(1, 11775, '\p{^Blk=supplementalpunctuation}', "");
Expect(1, 11775, '\P{Blk=supplementalpunctuation}', "");
Expect(0, 11775, '\P{^Blk=supplementalpunctuation}', "");
Expect(1, 11903, '\p{Blk= Supplemental_PUNCTUATION}', "");
Expect(0, 11903, '\p{^Blk= Supplemental_PUNCTUATION}', "");
Expect(0, 11903, '\P{Blk= Supplemental_PUNCTUATION}', "");
Expect(1, 11903, '\P{^Blk= Supplemental_PUNCTUATION}', "");
Expect(0, 11775, '\p{Blk= Supplemental_PUNCTUATION}', "");
Expect(1, 11775, '\p{^Blk= Supplemental_PUNCTUATION}', "");
Expect(1, 11775, '\P{Blk= Supplemental_PUNCTUATION}', "");
Expect(0, 11775, '\P{^Blk= Supplemental_PUNCTUATION}', "");
Error('\p{Is_Block= 	supplemental	punctuation/a/}');
Error('\P{Is_Block= 	supplemental	punctuation/a/}');
Expect(1, 11903, '\p{Is_Block=supplementalpunctuation}', "");
Expect(0, 11903, '\p{^Is_Block=supplementalpunctuation}', "");
Expect(0, 11903, '\P{Is_Block=supplementalpunctuation}', "");
Expect(1, 11903, '\P{^Is_Block=supplementalpunctuation}', "");
Expect(0, 11775, '\p{Is_Block=supplementalpunctuation}', "");
Expect(1, 11775, '\p{^Is_Block=supplementalpunctuation}', "");
Expect(1, 11775, '\P{Is_Block=supplementalpunctuation}', "");
Expect(0, 11775, '\P{^Is_Block=supplementalpunctuation}', "");
Expect(1, 11903, '\p{Is_Block=_-SUPPLEMENTALPUNCTUATION}', "");
Expect(0, 11903, '\p{^Is_Block=_-SUPPLEMENTALPUNCTUATION}', "");
Expect(0, 11903, '\P{Is_Block=_-SUPPLEMENTALPUNCTUATION}', "");
Expect(1, 11903, '\P{^Is_Block=_-SUPPLEMENTALPUNCTUATION}', "");
Expect(0, 11775, '\p{Is_Block=_-SUPPLEMENTALPUNCTUATION}', "");
Expect(1, 11775, '\p{^Is_Block=_-SUPPLEMENTALPUNCTUATION}', "");
Expect(1, 11775, '\P{Is_Block=_-SUPPLEMENTALPUNCTUATION}', "");
Expect(0, 11775, '\P{^Is_Block=_-SUPPLEMENTALPUNCTUATION}', "");
Error('\p{Is_Blk=/a/_SUPPLEMENTAL Punctuation}');
Error('\P{Is_Blk=/a/_SUPPLEMENTAL Punctuation}');
Expect(1, 11903, '\p{Is_Blk=supplementalpunctuation}', "");
Expect(0, 11903, '\p{^Is_Blk=supplementalpunctuation}', "");
Expect(0, 11903, '\P{Is_Blk=supplementalpunctuation}', "");
Expect(1, 11903, '\P{^Is_Blk=supplementalpunctuation}', "");
Expect(0, 11775, '\p{Is_Blk=supplementalpunctuation}', "");
Expect(1, 11775, '\p{^Is_Blk=supplementalpunctuation}', "");
Expect(1, 11775, '\P{Is_Blk=supplementalpunctuation}', "");
Expect(0, 11775, '\P{^Is_Blk=supplementalpunctuation}', "");
Expect(1, 11903, '\p{Is_Blk=_	supplemental Punctuation}', "");
Expect(0, 11903, '\p{^Is_Blk=_	supplemental Punctuation}', "");
Expect(0, 11903, '\P{Is_Blk=_	supplemental Punctuation}', "");
Expect(1, 11903, '\P{^Is_Blk=_	supplemental Punctuation}', "");
Expect(0, 11775, '\p{Is_Blk=_	supplemental Punctuation}', "");
Expect(1, 11775, '\p{^Is_Blk=_	supplemental Punctuation}', "");
Expect(1, 11775, '\P{Is_Blk=_	supplemental Punctuation}', "");
Expect(0, 11775, '\P{^Is_Blk=_	supplemental Punctuation}', "");
Error('\p{Block=:=Coptic}');
Error('\P{Block=:=Coptic}');
Expect(1, 11519, '\p{Block=coptic}', "");
Expect(0, 11519, '\p{^Block=coptic}', "");
Expect(0, 11519, '\P{Block=coptic}', "");
Expect(1, 11519, '\P{^Block=coptic}', "");
Expect(0, 11391, '\p{Block=coptic}', "");
Expect(1, 11391, '\p{^Block=coptic}', "");
Expect(1, 11391, '\P{Block=coptic}', "");
Expect(0, 11391, '\P{^Block=coptic}', "");
Expect(1, 11519, '\p{Block=	-Coptic}', "");
Expect(0, 11519, '\p{^Block=	-Coptic}', "");
Expect(0, 11519, '\P{Block=	-Coptic}', "");
Expect(1, 11519, '\P{^Block=	-Coptic}', "");
Expect(0, 11391, '\p{Block=	-Coptic}', "");
Expect(1, 11391, '\p{^Block=	-Coptic}', "");
Expect(1, 11391, '\P{Block=	-Coptic}', "");
Expect(0, 11391, '\P{^Block=	-Coptic}', "");
Error('\p{Blk=:=-_coptic}');
Error('\P{Blk=:=-_coptic}');
Expect(1, 11519, '\p{Blk:	coptic}', "");
Expect(0, 11519, '\p{^Blk:	coptic}', "");
Expect(0, 11519, '\P{Blk:	coptic}', "");
Expect(1, 11519, '\P{^Blk:	coptic}', "");
Expect(0, 11391, '\p{Blk:	coptic}', "");
Expect(1, 11391, '\p{^Blk:	coptic}', "");
Expect(1, 11391, '\P{Blk:	coptic}', "");
Expect(0, 11391, '\P{^Blk:	coptic}', "");
Expect(1, 11519, '\p{Blk=COPTIC}', "");
Expect(0, 11519, '\p{^Blk=COPTIC}', "");
Expect(0, 11519, '\P{Blk=COPTIC}', "");
Expect(1, 11519, '\P{^Blk=COPTIC}', "");
Expect(0, 11391, '\p{Blk=COPTIC}', "");
Expect(1, 11391, '\p{^Blk=COPTIC}', "");
Expect(1, 11391, '\P{Blk=COPTIC}', "");
Expect(0, 11391, '\P{^Blk=COPTIC}', "");
Error('\p{Is_Block=/a/	_COPTIC}');
Error('\P{Is_Block=/a/	_COPTIC}');
Expect(1, 11519, '\p{Is_Block=coptic}', "");
Expect(0, 11519, '\p{^Is_Block=coptic}', "");
Expect(0, 11519, '\P{Is_Block=coptic}', "");
Expect(1, 11519, '\P{^Is_Block=coptic}', "");
Expect(0, 11391, '\p{Is_Block=coptic}', "");
Expect(1, 11391, '\p{^Is_Block=coptic}', "");
Expect(1, 11391, '\P{Is_Block=coptic}', "");
Expect(0, 11391, '\P{^Is_Block=coptic}', "");
Expect(1, 11519, '\p{Is_Block=	coptic}', "");
Expect(0, 11519, '\p{^Is_Block=	coptic}', "");
Expect(0, 11519, '\P{Is_Block=	coptic}', "");
Expect(1, 11519, '\P{^Is_Block=	coptic}', "");
Expect(0, 11391, '\p{Is_Block=	coptic}', "");
Expect(1, 11391, '\p{^Is_Block=	coptic}', "");
Expect(1, 11391, '\P{Is_Block=	coptic}', "");
Expect(0, 11391, '\P{^Is_Block=	coptic}', "");
Error('\p{Is_Blk: __Coptic:=}');
Error('\P{Is_Blk: __Coptic:=}');
Expect(1, 11519, '\p{Is_Blk=coptic}', "");
Expect(0, 11519, '\p{^Is_Blk=coptic}', "");
Expect(0, 11519, '\P{Is_Blk=coptic}', "");
Expect(1, 11519, '\P{^Is_Blk=coptic}', "");
Expect(0, 11391, '\p{Is_Blk=coptic}', "");
Expect(1, 11391, '\p{^Is_Blk=coptic}', "");
Expect(1, 11391, '\P{Is_Blk=coptic}', "");
Expect(0, 11391, '\P{^Is_Blk=coptic}', "");
Expect(1, 11519, '\p{Is_Blk:	-coptic}', "");
Expect(0, 11519, '\p{^Is_Blk:	-coptic}', "");
Expect(0, 11519, '\P{Is_Blk:	-coptic}', "");
Expect(1, 11519, '\P{^Is_Blk:	-coptic}', "");
Expect(0, 11391, '\p{Is_Blk:	-coptic}', "");
Expect(1, 11391, '\p{^Is_Blk:	-coptic}', "");
Expect(1, 11391, '\P{Is_Blk:	-coptic}', "");
Expect(0, 11391, '\P{^Is_Blk:	-coptic}', "");
Error('\p{Block=		Supplemental:=arrows B}');
Error('\P{Block=		Supplemental:=arrows B}');
Expect(1, 10623, '\p{Block=supplementalarrowsb}', "");
Expect(0, 10623, '\p{^Block=supplementalarrowsb}', "");
Expect(0, 10623, '\P{Block=supplementalarrowsb}', "");
Expect(1, 10623, '\P{^Block=supplementalarrowsb}', "");
Expect(0, 10495, '\p{Block=supplementalarrowsb}', "");
Expect(1, 10495, '\p{^Block=supplementalarrowsb}', "");
Expect(1, 10495, '\P{Block=supplementalarrowsb}', "");
Expect(0, 10495, '\P{^Block=supplementalarrowsb}', "");
Expect(1, 10623, '\p{Block=  Supplemental	ArrowsB}', "");
Expect(0, 10623, '\p{^Block=  Supplemental	ArrowsB}', "");
Expect(0, 10623, '\P{Block=  Supplemental	ArrowsB}', "");
Expect(1, 10623, '\P{^Block=  Supplemental	ArrowsB}', "");
Expect(0, 10495, '\p{Block=  Supplemental	ArrowsB}', "");
Expect(1, 10495, '\p{^Block=  Supplemental	ArrowsB}', "");
Expect(1, 10495, '\P{Block=  Supplemental	ArrowsB}', "");
Expect(0, 10495, '\P{^Block=  Supplemental	ArrowsB}', "");
Error('\p{Blk=-SUPPLEMENTAL:=Arrows	B}');
Error('\P{Blk=-SUPPLEMENTAL:=Arrows	B}');
Expect(1, 10623, '\p{Blk=supplementalarrowsb}', "");
Expect(0, 10623, '\p{^Blk=supplementalarrowsb}', "");
Expect(0, 10623, '\P{Blk=supplementalarrowsb}', "");
Expect(1, 10623, '\P{^Blk=supplementalarrowsb}', "");
Expect(0, 10495, '\p{Blk=supplementalarrowsb}', "");
Expect(1, 10495, '\p{^Blk=supplementalarrowsb}', "");
Expect(1, 10495, '\P{Blk=supplementalarrowsb}', "");
Expect(0, 10495, '\P{^Blk=supplementalarrowsb}', "");
Expect(1, 10623, '\p{Blk= Supplemental	arrows	B}', "");
Expect(0, 10623, '\p{^Blk= Supplemental	arrows	B}', "");
Expect(0, 10623, '\P{Blk= Supplemental	arrows	B}', "");
Expect(1, 10623, '\P{^Blk= Supplemental	arrows	B}', "");
Expect(0, 10495, '\p{Blk= Supplemental	arrows	B}', "");
Expect(1, 10495, '\p{^Blk= Supplemental	arrows	B}', "");
Expect(1, 10495, '\P{Blk= Supplemental	arrows	B}', "");
Expect(0, 10495, '\P{^Blk= Supplemental	arrows	B}', "");
Error('\p{Is_Block=_ Supplemental	arrows-B/a/}');
Error('\P{Is_Block=_ Supplemental	arrows-B/a/}');
Expect(1, 10623, '\p{Is_Block=supplementalarrowsb}', "");
Expect(0, 10623, '\p{^Is_Block=supplementalarrowsb}', "");
Expect(0, 10623, '\P{Is_Block=supplementalarrowsb}', "");
Expect(1, 10623, '\P{^Is_Block=supplementalarrowsb}', "");
Expect(0, 10495, '\p{Is_Block=supplementalarrowsb}', "");
Expect(1, 10495, '\p{^Is_Block=supplementalarrowsb}', "");
Expect(1, 10495, '\P{Is_Block=supplementalarrowsb}', "");
Expect(0, 10495, '\P{^Is_Block=supplementalarrowsb}', "");
Expect(1, 10623, '\p{Is_Block=  supplementalarrows-B}', "");
Expect(0, 10623, '\p{^Is_Block=  supplementalarrows-B}', "");
Expect(0, 10623, '\P{Is_Block=  supplementalarrows-B}', "");
Expect(1, 10623, '\P{^Is_Block=  supplementalarrows-B}', "");
Expect(0, 10495, '\p{Is_Block=  supplementalarrows-B}', "");
Expect(1, 10495, '\p{^Is_Block=  supplementalarrows-B}', "");
Expect(1, 10495, '\P{Is_Block=  supplementalarrows-B}', "");
Expect(0, 10495, '\P{^Is_Block=  supplementalarrows-B}', "");
Error('\p{Is_Blk=/a/Supplemental-arrows_B}');
Error('\P{Is_Blk=/a/Supplemental-arrows_B}');
Expect(1, 10623, '\p{Is_Blk: supplementalarrowsb}', "");
Expect(0, 10623, '\p{^Is_Blk: supplementalarrowsb}', "");
Expect(0, 10623, '\P{Is_Blk: supplementalarrowsb}', "");
Expect(1, 10623, '\P{^Is_Blk: supplementalarrowsb}', "");
Expect(0, 10495, '\p{Is_Blk: supplementalarrowsb}', "");
Expect(1, 10495, '\p{^Is_Blk: supplementalarrowsb}', "");
Expect(1, 10495, '\P{Is_Blk: supplementalarrowsb}', "");
Expect(0, 10495, '\P{^Is_Blk: supplementalarrowsb}', "");
Expect(1, 10623, '\p{Is_Blk=-SUPPLEMENTAL-Arrows-b}', "");
Expect(0, 10623, '\p{^Is_Blk=-SUPPLEMENTAL-Arrows-b}', "");
Expect(0, 10623, '\P{Is_Blk=-SUPPLEMENTAL-Arrows-b}', "");
Expect(1, 10623, '\P{^Is_Blk=-SUPPLEMENTAL-Arrows-b}', "");
Expect(0, 10495, '\p{Is_Blk=-SUPPLEMENTAL-Arrows-b}', "");
Expect(1, 10495, '\p{^Is_Blk=-SUPPLEMENTAL-Arrows-b}', "");
Expect(1, 10495, '\P{Is_Blk=-SUPPLEMENTAL-Arrows-b}', "");
Expect(0, 10495, '\P{^Is_Blk=-SUPPLEMENTAL-Arrows-b}', "");
Error('\p{Block:/a/_TIBETAN}');
Error('\P{Block:/a/_TIBETAN}');
Expect(1, 4095, '\p{Block=tibetan}', "");
Expect(0, 4095, '\p{^Block=tibetan}', "");
Expect(0, 4095, '\P{Block=tibetan}', "");
Expect(1, 4095, '\P{^Block=tibetan}', "");
Expect(0, 3839, '\p{Block=tibetan}', "");
Expect(1, 3839, '\p{^Block=tibetan}', "");
Expect(1, 3839, '\P{Block=tibetan}', "");
Expect(0, 3839, '\P{^Block=tibetan}', "");
Error('\p{Blk=	_Tibetan:=}');
Error('\P{Blk=	_Tibetan:=}');
Expect(1, 4095, '\p{Blk=tibetan}', "");
Expect(0, 4095, '\p{^Blk=tibetan}', "");
Expect(0, 4095, '\P{Blk=tibetan}', "");
Expect(1, 4095, '\P{^Blk=tibetan}', "");
Expect(0, 3839, '\p{Blk=tibetan}', "");
Expect(1, 3839, '\p{^Blk=tibetan}', "");
Expect(1, 3839, '\P{Blk=tibetan}', "");
Expect(0, 3839, '\P{^Blk=tibetan}', "");
Expect(1, 4095, '\p{Blk=_Tibetan}', "");
Expect(0, 4095, '\p{^Blk=_Tibetan}', "");
Expect(0, 4095, '\P{Blk=_Tibetan}', "");
Expect(1, 4095, '\P{^Blk=_Tibetan}', "");
Expect(0, 3839, '\p{Blk=_Tibetan}', "");
Expect(1, 3839, '\p{^Blk=_Tibetan}', "");
Expect(1, 3839, '\P{Blk=_Tibetan}', "");
Expect(0, 3839, '\P{^Blk=_Tibetan}', "");
Error('\p{Is_Block=:= -Tibetan}');
Error('\P{Is_Block=:= -Tibetan}');
Expect(1, 4095, '\p{Is_Block:   tibetan}', "");
Expect(0, 4095, '\p{^Is_Block:   tibetan}', "");
Expect(0, 4095, '\P{Is_Block:   tibetan}', "");
Expect(1, 4095, '\P{^Is_Block:   tibetan}', "");
Expect(0, 3839, '\p{Is_Block:   tibetan}', "");
Expect(1, 3839, '\p{^Is_Block:   tibetan}', "");
Expect(1, 3839, '\P{Is_Block:   tibetan}', "");
Expect(0, 3839, '\P{^Is_Block:   tibetan}', "");
Expect(1, 4095, '\p{Is_Block=-TIBETAN}', "");
Expect(0, 4095, '\p{^Is_Block=-TIBETAN}', "");
Expect(0, 4095, '\P{Is_Block=-TIBETAN}', "");
Expect(1, 4095, '\P{^Is_Block=-TIBETAN}', "");
Expect(0, 3839, '\p{Is_Block=-TIBETAN}', "");
Expect(1, 3839, '\p{^Is_Block=-TIBETAN}', "");
Expect(1, 3839, '\P{Is_Block=-TIBETAN}', "");
Expect(0, 3839, '\P{^Is_Block=-TIBETAN}', "");
Error('\p{Is_Blk=/a/Tibetan}');
Error('\P{Is_Blk=/a/Tibetan}');
Expect(1, 4095, '\p{Is_Blk=tibetan}', "");
Expect(0, 4095, '\p{^Is_Blk=tibetan}', "");
Expect(0, 4095, '\P{Is_Blk=tibetan}', "");
Expect(1, 4095, '\P{^Is_Blk=tibetan}', "");
Expect(0, 3839, '\p{Is_Blk=tibetan}', "");
Expect(1, 3839, '\p{^Is_Blk=tibetan}', "");
Expect(1, 3839, '\P{Is_Blk=tibetan}', "");
Expect(0, 3839, '\P{^Is_Blk=tibetan}', "");
Expect(1, 4095, '\p{Is_Blk=	tibetan}', "");
Expect(0, 4095, '\p{^Is_Blk=	tibetan}', "");
Expect(0, 4095, '\P{Is_Blk=	tibetan}', "");
Expect(1, 4095, '\P{^Is_Blk=	tibetan}', "");
Expect(0, 3839, '\p{Is_Blk=	tibetan}', "");
Expect(1, 3839, '\p{^Is_Blk=	tibetan}', "");
Expect(1, 3839, '\P{Is_Blk=	tibetan}', "");
Expect(0, 3839, '\P{^Is_Blk=	tibetan}', "");
Error('\p{Block=-DominoTiles/a/}');
Error('\P{Block=-DominoTiles/a/}');
Expect(1, 127135, '\p{Block=dominotiles}', "");
Expect(0, 127135, '\p{^Block=dominotiles}', "");
Expect(0, 127135, '\P{Block=dominotiles}', "");
Expect(1, 127135, '\P{^Block=dominotiles}', "");
Expect(0, 127023, '\p{Block=dominotiles}', "");
Expect(1, 127023, '\p{^Block=dominotiles}', "");
Expect(1, 127023, '\P{Block=dominotiles}', "");
Expect(0, 127023, '\P{^Block=dominotiles}', "");
Expect(1, 127135, '\p{Block=	-Domino-TILES}', "");
Expect(0, 127135, '\p{^Block=	-Domino-TILES}', "");
Expect(0, 127135, '\P{Block=	-Domino-TILES}', "");
Expect(1, 127135, '\P{^Block=	-Domino-TILES}', "");
Expect(0, 127023, '\p{Block=	-Domino-TILES}', "");
Expect(1, 127023, '\p{^Block=	-Domino-TILES}', "");
Expect(1, 127023, '\P{Block=	-Domino-TILES}', "");
Expect(0, 127023, '\P{^Block=	-Domino-TILES}', "");
Error('\p{Blk= Domino_TILES/a/}');
Error('\P{Blk= Domino_TILES/a/}');
Expect(1, 127135, '\p{Blk: dominotiles}', "");
Expect(0, 127135, '\p{^Blk: dominotiles}', "");
Expect(0, 127135, '\P{Blk: dominotiles}', "");
Expect(1, 127135, '\P{^Blk: dominotiles}', "");
Expect(0, 127023, '\p{Blk: dominotiles}', "");
Expect(1, 127023, '\p{^Blk: dominotiles}', "");
Expect(1, 127023, '\P{Blk: dominotiles}', "");
Expect(0, 127023, '\P{^Blk: dominotiles}', "");
Expect(1, 127135, '\p{Blk=_-DominoTiles}', "");
Expect(0, 127135, '\p{^Blk=_-DominoTiles}', "");
Expect(0, 127135, '\P{Blk=_-DominoTiles}', "");
Expect(1, 127135, '\P{^Blk=_-DominoTiles}', "");
Expect(0, 127023, '\p{Blk=_-DominoTiles}', "");
Expect(1, 127023, '\p{^Blk=_-DominoTiles}', "");
Expect(1, 127023, '\P{Blk=_-DominoTiles}', "");
Expect(0, 127023, '\P{^Blk=_-DominoTiles}', "");
Error('\p{Is_Block=	 Domino_tiles/a/}');
Error('\P{Is_Block=	 Domino_tiles/a/}');
Expect(1, 127135, '\p{Is_Block=dominotiles}', "");
Expect(0, 127135, '\p{^Is_Block=dominotiles}', "");
Expect(0, 127135, '\P{Is_Block=dominotiles}', "");
Expect(1, 127135, '\P{^Is_Block=dominotiles}', "");
Expect(0, 127023, '\p{Is_Block=dominotiles}', "");
Expect(1, 127023, '\p{^Is_Block=dominotiles}', "");
Expect(1, 127023, '\P{Is_Block=dominotiles}', "");
Expect(0, 127023, '\P{^Is_Block=dominotiles}', "");
Expect(1, 127135, '\p{Is_Block=	 Domino	Tiles}', "");
Expect(0, 127135, '\p{^Is_Block=	 Domino	Tiles}', "");
Expect(0, 127135, '\P{Is_Block=	 Domino	Tiles}', "");
Expect(1, 127135, '\P{^Is_Block=	 Domino	Tiles}', "");
Expect(0, 127023, '\p{Is_Block=	 Domino	Tiles}', "");
Expect(1, 127023, '\p{^Is_Block=	 Domino	Tiles}', "");
Expect(1, 127023, '\P{Is_Block=	 Domino	Tiles}', "");
Expect(0, 127023, '\P{^Is_Block=	 Domino	Tiles}', "");
Error('\p{Is_Blk=  Domino/a/tiles}');
Error('\P{Is_Blk=  Domino/a/tiles}');
Expect(1, 127135, '\p{Is_Blk=dominotiles}', "");
Expect(0, 127135, '\p{^Is_Blk=dominotiles}', "");
Expect(0, 127135, '\P{Is_Blk=dominotiles}', "");
Expect(1, 127135, '\P{^Is_Blk=dominotiles}', "");
Expect(0, 127023, '\p{Is_Blk=dominotiles}', "");
Expect(1, 127023, '\p{^Is_Blk=dominotiles}', "");
Expect(1, 127023, '\P{Is_Blk=dominotiles}', "");
Expect(0, 127023, '\P{^Is_Blk=dominotiles}', "");
Expect(1, 127135, '\p{Is_Blk=-	Domino_Tiles}', "");
Expect(0, 127135, '\p{^Is_Blk=-	Domino_Tiles}', "");
Expect(0, 127135, '\P{Is_Blk=-	Domino_Tiles}', "");
Expect(1, 127135, '\P{^Is_Blk=-	Domino_Tiles}', "");
Expect(0, 127023, '\p{Is_Blk=-	Domino_Tiles}', "");
Expect(1, 127023, '\p{^Is_Blk=-	Domino_Tiles}', "");
Expect(1, 127023, '\P{Is_Blk=-	Domino_Tiles}', "");
Expect(0, 127023, '\P{^Is_Blk=-	Domino_Tiles}', "");
Error('\p{Block:	--CJK_Symbols-and/a/Punctuation}');
Error('\P{Block:	--CJK_Symbols-and/a/Punctuation}');
Expect(1, 12351, '\p{Block=cjksymbolsandpunctuation}', "");
Expect(0, 12351, '\p{^Block=cjksymbolsandpunctuation}', "");
Expect(0, 12351, '\P{Block=cjksymbolsandpunctuation}', "");
Expect(1, 12351, '\P{^Block=cjksymbolsandpunctuation}', "");
Expect(0, 12287, '\p{Block=cjksymbolsandpunctuation}', "");
Expect(1, 12287, '\p{^Block=cjksymbolsandpunctuation}', "");
Expect(1, 12287, '\P{Block=cjksymbolsandpunctuation}', "");
Expect(0, 12287, '\P{^Block=cjksymbolsandpunctuation}', "");
Expect(1, 12351, '\p{Block=-_CJK-Symbols AND	Punctuation}', "");
Expect(0, 12351, '\p{^Block=-_CJK-Symbols AND	Punctuation}', "");
Expect(0, 12351, '\P{Block=-_CJK-Symbols AND	Punctuation}', "");
Expect(1, 12351, '\P{^Block=-_CJK-Symbols AND	Punctuation}', "");
Expect(0, 12287, '\p{Block=-_CJK-Symbols AND	Punctuation}', "");
Expect(1, 12287, '\p{^Block=-_CJK-Symbols AND	Punctuation}', "");
Expect(1, 12287, '\P{Block=-_CJK-Symbols AND	Punctuation}', "");
Expect(0, 12287, '\P{^Block=-_CJK-Symbols AND	Punctuation}', "");
Error('\p{Blk=:=-_CJK_Symbols-and Punctuation}');
Error('\P{Blk=:=-_CJK_Symbols-and Punctuation}');
Expect(1, 12351, '\p{Blk=cjksymbolsandpunctuation}', "");
Expect(0, 12351, '\p{^Blk=cjksymbolsandpunctuation}', "");
Expect(0, 12351, '\P{Blk=cjksymbolsandpunctuation}', "");
Expect(1, 12351, '\P{^Blk=cjksymbolsandpunctuation}', "");
Expect(0, 12287, '\p{Blk=cjksymbolsandpunctuation}', "");
Expect(1, 12287, '\p{^Blk=cjksymbolsandpunctuation}', "");
Expect(1, 12287, '\P{Blk=cjksymbolsandpunctuation}', "");
Expect(0, 12287, '\P{^Blk=cjksymbolsandpunctuation}', "");
Expect(1, 12351, '\p{Blk=	CJK_SymbolsAND_Punctuation}', "");
Expect(0, 12351, '\p{^Blk=	CJK_SymbolsAND_Punctuation}', "");
Expect(0, 12351, '\P{Blk=	CJK_SymbolsAND_Punctuation}', "");
Expect(1, 12351, '\P{^Blk=	CJK_SymbolsAND_Punctuation}', "");
Expect(0, 12287, '\p{Blk=	CJK_SymbolsAND_Punctuation}', "");
Expect(1, 12287, '\p{^Blk=	CJK_SymbolsAND_Punctuation}', "");
Expect(1, 12287, '\P{Blk=	CJK_SymbolsAND_Punctuation}', "");
Expect(0, 12287, '\P{^Blk=	CJK_SymbolsAND_Punctuation}', "");
Error('\p{Is_Block:	_ CJK-Symbols and/a/Punctuation}');
Error('\P{Is_Block:	_ CJK-Symbols and/a/Punctuation}');
Expect(1, 12351, '\p{Is_Block=cjksymbolsandpunctuation}', "");
Expect(0, 12351, '\p{^Is_Block=cjksymbolsandpunctuation}', "");
Expect(0, 12351, '\P{Is_Block=cjksymbolsandpunctuation}', "");
Expect(1, 12351, '\P{^Is_Block=cjksymbolsandpunctuation}', "");
Expect(0, 12287, '\p{Is_Block=cjksymbolsandpunctuation}', "");
Expect(1, 12287, '\p{^Is_Block=cjksymbolsandpunctuation}', "");
Expect(1, 12287, '\P{Is_Block=cjksymbolsandpunctuation}', "");
Expect(0, 12287, '\P{^Is_Block=cjksymbolsandpunctuation}', "");
Expect(1, 12351, '\p{Is_Block=cjk	SYMBOLS-and_punctuation}', "");
Expect(0, 12351, '\p{^Is_Block=cjk	SYMBOLS-and_punctuation}', "");
Expect(0, 12351, '\P{Is_Block=cjk	SYMBOLS-and_punctuation}', "");
Expect(1, 12351, '\P{^Is_Block=cjk	SYMBOLS-and_punctuation}', "");
Expect(0, 12287, '\p{Is_Block=cjk	SYMBOLS-and_punctuation}', "");
Expect(1, 12287, '\p{^Is_Block=cjk	SYMBOLS-and_punctuation}', "");
Expect(1, 12287, '\P{Is_Block=cjk	SYMBOLS-and_punctuation}', "");
Expect(0, 12287, '\P{^Is_Block=cjk	SYMBOLS-and_punctuation}', "");
Error('\p{Is_Blk=:=cjk	SYMBOLS	and Punctuation}');
Error('\P{Is_Blk=:=cjk	SYMBOLS	and Punctuation}');
Expect(1, 12351, '\p{Is_Blk=cjksymbolsandpunctuation}', "");
Expect(0, 12351, '\p{^Is_Blk=cjksymbolsandpunctuation}', "");
Expect(0, 12351, '\P{Is_Blk=cjksymbolsandpunctuation}', "");
Expect(1, 12351, '\P{^Is_Blk=cjksymbolsandpunctuation}', "");
Expect(0, 12287, '\p{Is_Blk=cjksymbolsandpunctuation}', "");
Expect(1, 12287, '\p{^Is_Blk=cjksymbolsandpunctuation}', "");
Expect(1, 12287, '\P{Is_Blk=cjksymbolsandpunctuation}', "");
Expect(0, 12287, '\P{^Is_Blk=cjksymbolsandpunctuation}', "");
Expect(1, 12351, '\p{Is_Blk=-	cjk	SYMBOLS And	Punctuation}', "");
Expect(0, 12351, '\p{^Is_Blk=-	cjk	SYMBOLS And	Punctuation}', "");
Expect(0, 12351, '\P{Is_Blk=-	cjk	SYMBOLS And	Punctuation}', "");
Expect(1, 12351, '\P{^Is_Blk=-	cjk	SYMBOLS And	Punctuation}', "");
Expect(0, 12287, '\p{Is_Blk=-	cjk	SYMBOLS And	Punctuation}', "");
Expect(1, 12287, '\p{^Is_Blk=-	cjk	SYMBOLS And	Punctuation}', "");
Expect(1, 12287, '\P{Is_Blk=-	cjk	SYMBOLS And	Punctuation}', "");
Expect(0, 12287, '\P{^Is_Blk=-	cjk	SYMBOLS And	Punctuation}', "");
Error('\p{Block: _/a/Hangul Compatibility JAMO}');
Error('\P{Block: _/a/Hangul Compatibility JAMO}');
Expect(1, 12687, '\p{Block:	hangulcompatibilityjamo}', "");
Expect(0, 12687, '\p{^Block:	hangulcompatibilityjamo}', "");
Expect(0, 12687, '\P{Block:	hangulcompatibilityjamo}', "");
Expect(1, 12687, '\P{^Block:	hangulcompatibilityjamo}', "");
Expect(0, 12591, '\p{Block:	hangulcompatibilityjamo}', "");
Expect(1, 12591, '\p{^Block:	hangulcompatibilityjamo}', "");
Expect(1, 12591, '\P{Block:	hangulcompatibilityjamo}', "");
Expect(0, 12591, '\P{^Block:	hangulcompatibilityjamo}', "");
Expect(1, 12687, '\p{Block=	 HANGUL-COMPATIBILITY	Jamo}', "");
Expect(0, 12687, '\p{^Block=	 HANGUL-COMPATIBILITY	Jamo}', "");
Expect(0, 12687, '\P{Block=	 HANGUL-COMPATIBILITY	Jamo}', "");
Expect(1, 12687, '\P{^Block=	 HANGUL-COMPATIBILITY	Jamo}', "");
Expect(0, 12591, '\p{Block=	 HANGUL-COMPATIBILITY	Jamo}', "");
Expect(1, 12591, '\p{^Block=	 HANGUL-COMPATIBILITY	Jamo}', "");
Expect(1, 12591, '\P{Block=	 HANGUL-COMPATIBILITY	Jamo}', "");
Expect(0, 12591, '\P{^Block=	 HANGUL-COMPATIBILITY	Jamo}', "");
Error('\p{Blk: -Hangul_COMPATIBILITY	jamo:=}');
Error('\P{Blk: -Hangul_COMPATIBILITY	jamo:=}');
Expect(1, 12687, '\p{Blk=hangulcompatibilityjamo}', "");
Expect(0, 12687, '\p{^Blk=hangulcompatibilityjamo}', "");
Expect(0, 12687, '\P{Blk=hangulcompatibilityjamo}', "");
Expect(1, 12687, '\P{^Blk=hangulcompatibilityjamo}', "");
Expect(0, 12591, '\p{Blk=hangulcompatibilityjamo}', "");
Expect(1, 12591, '\p{^Blk=hangulcompatibilityjamo}', "");
Expect(1, 12591, '\P{Blk=hangulcompatibilityjamo}', "");
Expect(0, 12591, '\P{^Blk=hangulcompatibilityjamo}', "");
Expect(1, 12687, '\p{Blk=-_Hangul-Compatibility-jamo}', "");
Expect(0, 12687, '\p{^Blk=-_Hangul-Compatibility-jamo}', "");
Expect(0, 12687, '\P{Blk=-_Hangul-Compatibility-jamo}', "");
Expect(1, 12687, '\P{^Blk=-_Hangul-Compatibility-jamo}', "");
Expect(0, 12591, '\p{Blk=-_Hangul-Compatibility-jamo}', "");
Expect(1, 12591, '\p{^Blk=-_Hangul-Compatibility-jamo}', "");
Expect(1, 12591, '\P{Blk=-_Hangul-Compatibility-jamo}', "");
Expect(0, 12591, '\P{^Blk=-_Hangul-Compatibility-jamo}', "");
Error('\p{Is_Block=-hangul/a/Compatibility-Jamo}');
Error('\P{Is_Block=-hangul/a/Compatibility-Jamo}');
Expect(1, 12687, '\p{Is_Block=hangulcompatibilityjamo}', "");
Expect(0, 12687, '\p{^Is_Block=hangulcompatibilityjamo}', "");
Expect(0, 12687, '\P{Is_Block=hangulcompatibilityjamo}', "");
Expect(1, 12687, '\P{^Is_Block=hangulcompatibilityjamo}', "");
Expect(0, 12591, '\p{Is_Block=hangulcompatibilityjamo}', "");
Expect(1, 12591, '\p{^Is_Block=hangulcompatibilityjamo}', "");
Expect(1, 12591, '\P{Is_Block=hangulcompatibilityjamo}', "");
Expect(0, 12591, '\P{^Is_Block=hangulcompatibilityjamo}', "");
Expect(1, 12687, '\p{Is_Block=_ Hangul COMPATIBILITYJAMO}', "");
Expect(0, 12687, '\p{^Is_Block=_ Hangul COMPATIBILITYJAMO}', "");
Expect(0, 12687, '\P{Is_Block=_ Hangul COMPATIBILITYJAMO}', "");
Expect(1, 12687, '\P{^Is_Block=_ Hangul COMPATIBILITYJAMO}', "");
Expect(0, 12591, '\p{Is_Block=_ Hangul COMPATIBILITYJAMO}', "");
Expect(1, 12591, '\p{^Is_Block=_ Hangul COMPATIBILITYJAMO}', "");
Expect(1, 12591, '\P{Is_Block=_ Hangul COMPATIBILITYJAMO}', "");
Expect(0, 12591, '\P{^Is_Block=_ Hangul COMPATIBILITYJAMO}', "");
Error('\p{Is_Blk:	_:=Hangul	Compatibility_Jamo}');
Error('\P{Is_Blk:	_:=Hangul	Compatibility_Jamo}');
Expect(1, 12687, '\p{Is_Blk=hangulcompatibilityjamo}', "");
Expect(0, 12687, '\p{^Is_Blk=hangulcompatibilityjamo}', "");
Expect(0, 12687, '\P{Is_Blk=hangulcompatibilityjamo}', "");
Expect(1, 12687, '\P{^Is_Blk=hangulcompatibilityjamo}', "");
Expect(0, 12591, '\p{Is_Blk=hangulcompatibilityjamo}', "");
Expect(1, 12591, '\p{^Is_Blk=hangulcompatibilityjamo}', "");
Expect(1, 12591, '\P{Is_Blk=hangulcompatibilityjamo}', "");
Expect(0, 12591, '\P{^Is_Blk=hangulcompatibilityjamo}', "");
Expect(1, 12687, '\p{Is_Blk=_HANGULCompatibility Jamo}', "");
Expect(0, 12687, '\p{^Is_Blk=_HANGULCompatibility Jamo}', "");
Expect(0, 12687, '\P{Is_Blk=_HANGULCompatibility Jamo}', "");
Expect(1, 12687, '\P{^Is_Blk=_HANGULCompatibility Jamo}', "");
Expect(0, 12591, '\p{Is_Blk=_HANGULCompatibility Jamo}', "");
Expect(1, 12591, '\p{^Is_Blk=_HANGULCompatibility Jamo}', "");
Expect(1, 12591, '\P{Is_Blk=_HANGULCompatibility Jamo}', "");
Expect(0, 12591, '\P{^Is_Blk=_HANGULCompatibility Jamo}', "");
Error('\p{Block= 	old Persian:=}');
Error('\P{Block= 	old Persian:=}');
Expect(1, 66527, '\p{Block=oldpersian}', "");
Expect(0, 66527, '\p{^Block=oldpersian}', "");
Expect(0, 66527, '\P{Block=oldpersian}', "");
Expect(1, 66527, '\P{^Block=oldpersian}', "");
Expect(0, 66463, '\p{Block=oldpersian}', "");
Expect(1, 66463, '\p{^Block=oldpersian}', "");
Expect(1, 66463, '\P{Block=oldpersian}', "");
Expect(0, 66463, '\P{^Block=oldpersian}', "");
Expect(1, 66527, '\p{Block=-Old Persian}', "");
Expect(0, 66527, '\p{^Block=-Old Persian}', "");
Expect(0, 66527, '\P{Block=-Old Persian}', "");
Expect(1, 66527, '\P{^Block=-Old Persian}', "");
Expect(0, 66463, '\p{Block=-Old Persian}', "");
Expect(1, 66463, '\p{^Block=-Old Persian}', "");
Expect(1, 66463, '\P{Block=-Old Persian}', "");
Expect(0, 66463, '\P{^Block=-Old Persian}', "");
Error('\p{Blk=:=Oldpersian}');
Error('\P{Blk=:=Oldpersian}');
Expect(1, 66527, '\p{Blk=oldpersian}', "");
Expect(0, 66527, '\p{^Blk=oldpersian}', "");
Expect(0, 66527, '\P{Blk=oldpersian}', "");
Expect(1, 66527, '\P{^Blk=oldpersian}', "");
Expect(0, 66463, '\p{Blk=oldpersian}', "");
Expect(1, 66463, '\p{^Blk=oldpersian}', "");
Expect(1, 66463, '\P{Blk=oldpersian}', "");
Expect(0, 66463, '\P{^Blk=oldpersian}', "");
Expect(1, 66527, '\p{Blk=	 OLD	persian}', "");
Expect(0, 66527, '\p{^Blk=	 OLD	persian}', "");
Expect(0, 66527, '\P{Blk=	 OLD	persian}', "");
Expect(1, 66527, '\P{^Blk=	 OLD	persian}', "");
Expect(0, 66463, '\p{Blk=	 OLD	persian}', "");
Expect(1, 66463, '\p{^Blk=	 OLD	persian}', "");
Expect(1, 66463, '\P{Blk=	 OLD	persian}', "");
Expect(0, 66463, '\P{^Blk=	 OLD	persian}', "");
Error('\p{Is_Block= 	Old-Persian:=}');
Error('\P{Is_Block= 	Old-Persian:=}');
Expect(1, 66527, '\p{Is_Block=oldpersian}', "");
Expect(0, 66527, '\p{^Is_Block=oldpersian}', "");
Expect(0, 66527, '\P{Is_Block=oldpersian}', "");
Expect(1, 66527, '\P{^Is_Block=oldpersian}', "");
Expect(0, 66463, '\p{Is_Block=oldpersian}', "");
Expect(1, 66463, '\p{^Is_Block=oldpersian}', "");
Expect(1, 66463, '\P{Is_Block=oldpersian}', "");
Expect(0, 66463, '\P{^Is_Block=oldpersian}', "");
Expect(1, 66527, '\p{Is_Block=_old_persian}', "");
Expect(0, 66527, '\p{^Is_Block=_old_persian}', "");
Expect(0, 66527, '\P{Is_Block=_old_persian}', "");
Expect(1, 66527, '\P{^Is_Block=_old_persian}', "");
Expect(0, 66463, '\p{Is_Block=_old_persian}', "");
Expect(1, 66463, '\p{^Is_Block=_old_persian}', "");
Expect(1, 66463, '\P{Is_Block=_old_persian}', "");
Expect(0, 66463, '\P{^Is_Block=_old_persian}', "");
Error('\p{Is_Blk= _Old:=persian}');
Error('\P{Is_Blk= _Old:=persian}');
Expect(1, 66527, '\p{Is_Blk=oldpersian}', "");
Expect(0, 66527, '\p{^Is_Blk=oldpersian}', "");
Expect(0, 66527, '\P{Is_Blk=oldpersian}', "");
Expect(1, 66527, '\P{^Is_Blk=oldpersian}', "");
Expect(0, 66463, '\p{Is_Blk=oldpersian}', "");
Expect(1, 66463, '\p{^Is_Blk=oldpersian}', "");
Expect(1, 66463, '\P{Is_Blk=oldpersian}', "");
Expect(0, 66463, '\P{^Is_Blk=oldpersian}', "");
Expect(1, 66527, '\p{Is_Blk=	-OLD-persian}', "");
Expect(0, 66527, '\p{^Is_Blk=	-OLD-persian}', "");
Expect(0, 66527, '\P{Is_Blk=	-OLD-persian}', "");
Expect(1, 66527, '\P{^Is_Blk=	-OLD-persian}', "");
Expect(0, 66463, '\p{Is_Blk=	-OLD-persian}', "");
Expect(1, 66463, '\p{^Is_Blk=	-OLD-persian}', "");
Expect(1, 66463, '\P{Is_Blk=	-OLD-persian}', "");
Expect(0, 66463, '\P{^Is_Blk=	-OLD-persian}', "");
Error('\p{Block=	Modifier/a/Tone-LETTERS}');
Error('\P{Block=	Modifier/a/Tone-LETTERS}');
Expect(1, 42783, '\p{Block=modifiertoneletters}', "");
Expect(0, 42783, '\p{^Block=modifiertoneletters}', "");
Expect(0, 42783, '\P{Block=modifiertoneletters}', "");
Expect(1, 42783, '\P{^Block=modifiertoneletters}', "");
Expect(0, 42751, '\p{Block=modifiertoneletters}', "");
Expect(1, 42751, '\p{^Block=modifiertoneletters}', "");
Expect(1, 42751, '\P{Block=modifiertoneletters}', "");
Expect(0, 42751, '\P{^Block=modifiertoneletters}', "");
Expect(1, 42783, '\p{Block=	Modifier	Tone Letters}', "");
Expect(0, 42783, '\p{^Block=	Modifier	Tone Letters}', "");
Expect(0, 42783, '\P{Block=	Modifier	Tone Letters}', "");
Expect(1, 42783, '\P{^Block=	Modifier	Tone Letters}', "");
Expect(0, 42751, '\p{Block=	Modifier	Tone Letters}', "");
Expect(1, 42751, '\p{^Block=	Modifier	Tone Letters}', "");
Expect(1, 42751, '\P{Block=	Modifier	Tone Letters}', "");
Expect(0, 42751, '\P{^Block=	Modifier	Tone Letters}', "");
Error('\p{Blk= -modifier TONE	LETTERS/a/}');
Error('\P{Blk= -modifier TONE	LETTERS/a/}');
Expect(1, 42783, '\p{Blk=modifiertoneletters}', "");
Expect(0, 42783, '\p{^Blk=modifiertoneletters}', "");
Expect(0, 42783, '\P{Blk=modifiertoneletters}', "");
Expect(1, 42783, '\P{^Blk=modifiertoneletters}', "");
Expect(0, 42751, '\p{Blk=modifiertoneletters}', "");
Expect(1, 42751, '\p{^Blk=modifiertoneletters}', "");
Expect(1, 42751, '\P{Blk=modifiertoneletters}', "");
Expect(0, 42751, '\P{^Blk=modifiertoneletters}', "");
Expect(1, 42783, '\p{Blk=_	Modifier tone_letters}', "");
Expect(0, 42783, '\p{^Blk=_	Modifier tone_letters}', "");
Expect(0, 42783, '\P{Blk=_	Modifier tone_letters}', "");
Expect(1, 42783, '\P{^Blk=_	Modifier tone_letters}', "");
Expect(0, 42751, '\p{Blk=_	Modifier tone_letters}', "");
Expect(1, 42751, '\p{^Blk=_	Modifier tone_letters}', "");
Expect(1, 42751, '\P{Blk=_	Modifier tone_letters}', "");
Expect(0, 42751, '\P{^Blk=_	Modifier tone_letters}', "");
Error('\p{Is_Block=_modifier:=tone letters}');
Error('\P{Is_Block=_modifier:=tone letters}');
Expect(1, 42783, '\p{Is_Block: modifiertoneletters}', "");
Expect(0, 42783, '\p{^Is_Block: modifiertoneletters}', "");
Expect(0, 42783, '\P{Is_Block: modifiertoneletters}', "");
Expect(1, 42783, '\P{^Is_Block: modifiertoneletters}', "");
Expect(0, 42751, '\p{Is_Block: modifiertoneletters}', "");
Expect(1, 42751, '\p{^Is_Block: modifiertoneletters}', "");
Expect(1, 42751, '\P{Is_Block: modifiertoneletters}', "");
Expect(0, 42751, '\P{^Is_Block: modifiertoneletters}', "");
Expect(1, 42783, '\p{Is_Block=__MODIFIERTone-Letters}', "");
Expect(0, 42783, '\p{^Is_Block=__MODIFIERTone-Letters}', "");
Expect(0, 42783, '\P{Is_Block=__MODIFIERTone-Letters}', "");
Expect(1, 42783, '\P{^Is_Block=__MODIFIERTone-Letters}', "");
Expect(0, 42751, '\p{Is_Block=__MODIFIERTone-Letters}', "");
Expect(1, 42751, '\p{^Is_Block=__MODIFIERTone-Letters}', "");
Expect(1, 42751, '\P{Is_Block=__MODIFIERTone-Letters}', "");
Expect(0, 42751, '\P{^Is_Block=__MODIFIERTone-Letters}', "");
Error('\p{Is_Blk=		MODIFIER-TONE	Letters:=}');
Error('\P{Is_Blk=		MODIFIER-TONE	Letters:=}');
Expect(1, 42783, '\p{Is_Blk=modifiertoneletters}', "");
Expect(0, 42783, '\p{^Is_Blk=modifiertoneletters}', "");
Expect(0, 42783, '\P{Is_Blk=modifiertoneletters}', "");
Expect(1, 42783, '\P{^Is_Blk=modifiertoneletters}', "");
Expect(0, 42751, '\p{Is_Blk=modifiertoneletters}', "");
Expect(1, 42751, '\p{^Is_Blk=modifiertoneletters}', "");
Expect(1, 42751, '\P{Is_Blk=modifiertoneletters}', "");
Expect(0, 42751, '\P{^Is_Blk=modifiertoneletters}', "");
Expect(1, 42783, '\p{Is_Blk=		modifier_tone_Letters}', "");
Expect(0, 42783, '\p{^Is_Blk=		modifier_tone_Letters}', "");
Expect(0, 42783, '\P{Is_Blk=		modifier_tone_Letters}', "");
Expect(1, 42783, '\P{^Is_Blk=		modifier_tone_Letters}', "");
Expect(0, 42751, '\p{Is_Blk=		modifier_tone_Letters}', "");
Expect(1, 42751, '\p{^Is_Blk=		modifier_tone_Letters}', "");
Expect(1, 42751, '\P{Is_Blk=		modifier_tone_Letters}', "");
Expect(0, 42751, '\P{^Is_Blk=		modifier_tone_Letters}', "");
Error('\p{Block=/a/__ETHIOPIC_supplement}');
Error('\P{Block=/a/__ETHIOPIC_supplement}');
Expect(1, 5023, '\p{Block=ethiopicsupplement}', "");
Expect(0, 5023, '\p{^Block=ethiopicsupplement}', "");
Expect(0, 5023, '\P{Block=ethiopicsupplement}', "");
Expect(1, 5023, '\P{^Block=ethiopicsupplement}', "");
Expect(0, 4991, '\p{Block=ethiopicsupplement}', "");
Expect(1, 4991, '\p{^Block=ethiopicsupplement}', "");
Expect(1, 4991, '\P{Block=ethiopicsupplement}', "");
Expect(0, 4991, '\P{^Block=ethiopicsupplement}', "");
Expect(1, 5023, '\p{Block:			ethiopic-SUPPLEMENT}', "");
Expect(0, 5023, '\p{^Block:			ethiopic-SUPPLEMENT}', "");
Expect(0, 5023, '\P{Block:			ethiopic-SUPPLEMENT}', "");
Expect(1, 5023, '\P{^Block:			ethiopic-SUPPLEMENT}', "");
Expect(0, 4991, '\p{Block:			ethiopic-SUPPLEMENT}', "");
Expect(1, 4991, '\p{^Block:			ethiopic-SUPPLEMENT}', "");
Expect(1, 4991, '\P{Block:			ethiopic-SUPPLEMENT}', "");
Expect(0, 4991, '\P{^Block:			ethiopic-SUPPLEMENT}', "");
Error('\p{Blk=:=-Ethiopic-Supplement}');
Error('\P{Blk=:=-Ethiopic-Supplement}');
Expect(1, 5023, '\p{Blk=ethiopicsupplement}', "");
Expect(0, 5023, '\p{^Blk=ethiopicsupplement}', "");
Expect(0, 5023, '\P{Blk=ethiopicsupplement}', "");
Expect(1, 5023, '\P{^Blk=ethiopicsupplement}', "");
Expect(0, 4991, '\p{Blk=ethiopicsupplement}', "");
Expect(1, 4991, '\p{^Blk=ethiopicsupplement}', "");
Expect(1, 4991, '\P{Blk=ethiopicsupplement}', "");
Expect(0, 4991, '\P{^Blk=ethiopicsupplement}', "");
Expect(1, 5023, '\p{Blk=_ EthiopicSupplement}', "");
Expect(0, 5023, '\p{^Blk=_ EthiopicSupplement}', "");
Expect(0, 5023, '\P{Blk=_ EthiopicSupplement}', "");
Expect(1, 5023, '\P{^Blk=_ EthiopicSupplement}', "");
Expect(0, 4991, '\p{Blk=_ EthiopicSupplement}', "");
Expect(1, 4991, '\p{^Blk=_ EthiopicSupplement}', "");
Expect(1, 4991, '\P{Blk=_ EthiopicSupplement}', "");
Expect(0, 4991, '\P{^Blk=_ EthiopicSupplement}', "");
Error('\p{Is_Block=	-Ethiopic:=SUPPLEMENT}');
Error('\P{Is_Block=	-Ethiopic:=SUPPLEMENT}');
Expect(1, 5023, '\p{Is_Block=ethiopicsupplement}', "");
Expect(0, 5023, '\p{^Is_Block=ethiopicsupplement}', "");
Expect(0, 5023, '\P{Is_Block=ethiopicsupplement}', "");
Expect(1, 5023, '\P{^Is_Block=ethiopicsupplement}', "");
Expect(0, 4991, '\p{Is_Block=ethiopicsupplement}', "");
Expect(1, 4991, '\p{^Is_Block=ethiopicsupplement}', "");
Expect(1, 4991, '\P{Is_Block=ethiopicsupplement}', "");
Expect(0, 4991, '\P{^Is_Block=ethiopicsupplement}', "");
Expect(1, 5023, '\p{Is_Block:_ethiopic supplement}', "");
Expect(0, 5023, '\p{^Is_Block:_ethiopic supplement}', "");
Expect(0, 5023, '\P{Is_Block:_ethiopic supplement}', "");
Expect(1, 5023, '\P{^Is_Block:_ethiopic supplement}', "");
Expect(0, 4991, '\p{Is_Block:_ethiopic supplement}', "");
Expect(1, 4991, '\p{^Is_Block:_ethiopic supplement}', "");
Expect(1, 4991, '\P{Is_Block:_ethiopic supplement}', "");
Expect(0, 4991, '\P{^Is_Block:_ethiopic supplement}', "");
Error('\p{Is_Blk:	Ethiopic:=SUPPLEMENT}');
Error('\P{Is_Blk:	Ethiopic:=SUPPLEMENT}');
Expect(1, 5023, '\p{Is_Blk=ethiopicsupplement}', "");
Expect(0, 5023, '\p{^Is_Blk=ethiopicsupplement}', "");
Expect(0, 5023, '\P{Is_Blk=ethiopicsupplement}', "");
Expect(1, 5023, '\P{^Is_Blk=ethiopicsupplement}', "");
Expect(0, 4991, '\p{Is_Blk=ethiopicsupplement}', "");
Expect(1, 4991, '\p{^Is_Blk=ethiopicsupplement}', "");
Expect(1, 4991, '\P{Is_Blk=ethiopicsupplement}', "");
Expect(0, 4991, '\P{^Is_Blk=ethiopicsupplement}', "");
Expect(1, 5023, '\p{Is_Blk=_	ethiopic_Supplement}', "");
Expect(0, 5023, '\p{^Is_Blk=_	ethiopic_Supplement}', "");
Expect(0, 5023, '\P{Is_Blk=_	ethiopic_Supplement}', "");
Expect(1, 5023, '\P{^Is_Blk=_	ethiopic_Supplement}', "");
Expect(0, 4991, '\p{Is_Blk=_	ethiopic_Supplement}', "");
Expect(1, 4991, '\p{^Is_Blk=_	ethiopic_Supplement}', "");
Expect(1, 4991, '\P{Is_Blk=_	ethiopic_Supplement}', "");
Expect(0, 4991, '\P{^Is_Blk=_	ethiopic_Supplement}', "");
Error('\p{Block=	 Linear-b Syllabary:=}');
Error('\P{Block=	 Linear-b Syllabary:=}');
Expect(1, 65663, '\p{Block=linearbsyllabary}', "");
Expect(0, 65663, '\p{^Block=linearbsyllabary}', "");
Expect(0, 65663, '\P{Block=linearbsyllabary}', "");
Expect(1, 65663, '\P{^Block=linearbsyllabary}', "");
Expect(0, 1, '\p{Block=linearbsyllabary}', "");
Expect(1, 1, '\p{^Block=linearbsyllabary}', "");
Expect(1, 1, '\P{Block=linearbsyllabary}', "");
Expect(0, 1, '\P{^Block=linearbsyllabary}', "");
Expect(1, 65663, '\p{Block=-Linear-B_Syllabary}', "");
Expect(0, 65663, '\p{^Block=-Linear-B_Syllabary}', "");
Expect(0, 65663, '\P{Block=-Linear-B_Syllabary}', "");
Expect(1, 65663, '\P{^Block=-Linear-B_Syllabary}', "");
Expect(0, 1, '\p{Block=-Linear-B_Syllabary}', "");
Expect(1, 1, '\p{^Block=-Linear-B_Syllabary}', "");
Expect(1, 1, '\P{Block=-Linear-B_Syllabary}', "");
Expect(0, 1, '\P{^Block=-Linear-B_Syllabary}', "");
Error('\p{Blk=-Linear-B:=Syllabary}');
Error('\P{Blk=-Linear-B:=Syllabary}');
Expect(1, 65663, '\p{Blk=linearbsyllabary}', "");
Expect(0, 65663, '\p{^Blk=linearbsyllabary}', "");
Expect(0, 65663, '\P{Blk=linearbsyllabary}', "");
Expect(1, 65663, '\P{^Blk=linearbsyllabary}', "");
Expect(0, 1, '\p{Blk=linearbsyllabary}', "");
Expect(1, 1, '\p{^Blk=linearbsyllabary}', "");
Expect(1, 1, '\P{Blk=linearbsyllabary}', "");
Expect(0, 1, '\P{^Blk=linearbsyllabary}', "");
Expect(1, 65663, '\p{Blk=-Linear-B_syllabary}', "");
Expect(0, 65663, '\p{^Blk=-Linear-B_syllabary}', "");
Expect(0, 65663, '\P{Blk=-Linear-B_syllabary}', "");
Expect(1, 65663, '\P{^Blk=-Linear-B_syllabary}', "");
Expect(0, 1, '\p{Blk=-Linear-B_syllabary}', "");
Expect(1, 1, '\p{^Blk=-Linear-B_syllabary}', "");
Expect(1, 1, '\P{Blk=-Linear-B_syllabary}', "");
Expect(0, 1, '\P{^Blk=-Linear-B_syllabary}', "");
Error('\p{Is_Block=/a/		LinearB	syllabary}');
Error('\P{Is_Block=/a/		LinearB	syllabary}');
Expect(1, 65663, '\p{Is_Block=linearbsyllabary}', "");
Expect(0, 65663, '\p{^Is_Block=linearbsyllabary}', "");
Expect(0, 65663, '\P{Is_Block=linearbsyllabary}', "");
Expect(1, 65663, '\P{^Is_Block=linearbsyllabary}', "");
Expect(0, 1, '\p{Is_Block=linearbsyllabary}', "");
Expect(1, 1, '\p{^Is_Block=linearbsyllabary}', "");
Expect(1, 1, '\P{Is_Block=linearbsyllabary}', "");
Expect(0, 1, '\P{^Is_Block=linearbsyllabary}', "");
Expect(1, 65663, '\p{Is_Block=_ LINEAR B_SYLLABARY}', "");
Expect(0, 65663, '\p{^Is_Block=_ LINEAR B_SYLLABARY}', "");
Expect(0, 65663, '\P{Is_Block=_ LINEAR B_SYLLABARY}', "");
Expect(1, 65663, '\P{^Is_Block=_ LINEAR B_SYLLABARY}', "");
Expect(0, 1, '\p{Is_Block=_ LINEAR B_SYLLABARY}', "");
Expect(1, 1, '\p{^Is_Block=_ LINEAR B_SYLLABARY}', "");
Expect(1, 1, '\P{Is_Block=_ LINEAR B_SYLLABARY}', "");
Expect(0, 1, '\P{^Is_Block=_ LINEAR B_SYLLABARY}', "");
Error('\p{Is_Blk=_Linear-b_Syllabary/a/}');
Error('\P{Is_Blk=_Linear-b_Syllabary/a/}');
Expect(1, 65663, '\p{Is_Blk=linearbsyllabary}', "");
Expect(0, 65663, '\p{^Is_Blk=linearbsyllabary}', "");
Expect(0, 65663, '\P{Is_Blk=linearbsyllabary}', "");
Expect(1, 65663, '\P{^Is_Blk=linearbsyllabary}', "");
Expect(0, 1, '\p{Is_Blk=linearbsyllabary}', "");
Expect(1, 1, '\p{^Is_Blk=linearbsyllabary}', "");
Expect(1, 1, '\P{Is_Blk=linearbsyllabary}', "");
Expect(0, 1, '\P{^Is_Blk=linearbsyllabary}', "");
Expect(1, 65663, '\p{Is_Blk=	linear_BSyllabary}', "");
Expect(0, 65663, '\p{^Is_Blk=	linear_BSyllabary}', "");
Expect(0, 65663, '\P{Is_Blk=	linear_BSyllabary}', "");
Expect(1, 65663, '\P{^Is_Blk=	linear_BSyllabary}', "");
Expect(0, 1, '\p{Is_Blk=	linear_BSyllabary}', "");
Expect(1, 1, '\p{^Is_Blk=	linear_BSyllabary}', "");
Expect(1, 1, '\P{Is_Blk=	linear_BSyllabary}', "");
Expect(0, 1, '\P{^Is_Blk=	linear_BSyllabary}', "");
Error('\p{Block= /a/enclosed-cjk LETTERSAnd_MONTHS}');
Error('\P{Block= /a/enclosed-cjk LETTERSAnd_MONTHS}');
Expect(1, 13055, '\p{Block=enclosedcjklettersandmonths}', "");
Expect(0, 13055, '\p{^Block=enclosedcjklettersandmonths}', "");
Expect(0, 13055, '\P{Block=enclosedcjklettersandmonths}', "");
Expect(1, 13055, '\P{^Block=enclosedcjklettersandmonths}', "");
Expect(0, 12799, '\p{Block=enclosedcjklettersandmonths}', "");
Expect(1, 12799, '\p{^Block=enclosedcjklettersandmonths}', "");
Expect(1, 12799, '\P{Block=enclosedcjklettersandmonths}', "");
Expect(0, 12799, '\P{^Block=enclosedcjklettersandmonths}', "");
Expect(1, 13055, '\p{Block= 	ENCLOSEDCJK	Letters-and	MONTHS}', "");
Expect(0, 13055, '\p{^Block= 	ENCLOSEDCJK	Letters-and	MONTHS}', "");
Expect(0, 13055, '\P{Block= 	ENCLOSEDCJK	Letters-and	MONTHS}', "");
Expect(1, 13055, '\P{^Block= 	ENCLOSEDCJK	Letters-and	MONTHS}', "");
Expect(0, 12799, '\p{Block= 	ENCLOSEDCJK	Letters-and	MONTHS}', "");
Expect(1, 12799, '\p{^Block= 	ENCLOSEDCJK	Letters-and	MONTHS}', "");
Expect(1, 12799, '\P{Block= 	ENCLOSEDCJK	Letters-and	MONTHS}', "");
Expect(0, 12799, '\P{^Block= 	ENCLOSEDCJK	Letters-and	MONTHS}', "");
Error('\p{Blk=	ENCLOSEDCJK_Letters/a/And-MONTHS}');
Error('\P{Blk=	ENCLOSEDCJK_Letters/a/And-MONTHS}');
Expect(1, 13055, '\p{Blk=enclosedcjklettersandmonths}', "");
Expect(0, 13055, '\p{^Blk=enclosedcjklettersandmonths}', "");
Expect(0, 13055, '\P{Blk=enclosedcjklettersandmonths}', "");
Expect(1, 13055, '\P{^Blk=enclosedcjklettersandmonths}', "");
Expect(0, 12799, '\p{Blk=enclosedcjklettersandmonths}', "");
Expect(1, 12799, '\p{^Blk=enclosedcjklettersandmonths}', "");
Expect(1, 12799, '\P{Blk=enclosedcjklettersandmonths}', "");
Expect(0, 12799, '\P{^Blk=enclosedcjklettersandmonths}', "");
Expect(1, 13055, '\p{Blk=_ENCLOSED_CJK-LETTERS ANDmonths}', "");
Expect(0, 13055, '\p{^Blk=_ENCLOSED_CJK-LETTERS ANDmonths}', "");
Expect(0, 13055, '\P{Blk=_ENCLOSED_CJK-LETTERS ANDmonths}', "");
Expect(1, 13055, '\P{^Blk=_ENCLOSED_CJK-LETTERS ANDmonths}', "");
Expect(0, 12799, '\p{Blk=_ENCLOSED_CJK-LETTERS ANDmonths}', "");
Expect(1, 12799, '\p{^Blk=_ENCLOSED_CJK-LETTERS ANDmonths}', "");
Expect(1, 12799, '\P{Blk=_ENCLOSED_CJK-LETTERS ANDmonths}', "");
Expect(0, 12799, '\P{^Blk=_ENCLOSED_CJK-LETTERS ANDmonths}', "");
Error('\p{Is_Block= -ENCLOSED	CJKLETTERS AND:=MONTHS}');
Error('\P{Is_Block= -ENCLOSED	CJKLETTERS AND:=MONTHS}');
Expect(1, 13055, '\p{Is_Block=enclosedcjklettersandmonths}', "");
Expect(0, 13055, '\p{^Is_Block=enclosedcjklettersandmonths}', "");
Expect(0, 13055, '\P{Is_Block=enclosedcjklettersandmonths}', "");
Expect(1, 13055, '\P{^Is_Block=enclosedcjklettersandmonths}', "");
Expect(0, 12799, '\p{Is_Block=enclosedcjklettersandmonths}', "");
Expect(1, 12799, '\p{^Is_Block=enclosedcjklettersandmonths}', "");
Expect(1, 12799, '\P{Is_Block=enclosedcjklettersandmonths}', "");
Expect(0, 12799, '\P{^Is_Block=enclosedcjklettersandmonths}', "");
Expect(1, 13055, '\p{Is_Block=__enclosedCJK_Letters-and-Months}', "");
Expect(0, 13055, '\p{^Is_Block=__enclosedCJK_Letters-and-Months}', "");
Expect(0, 13055, '\P{Is_Block=__enclosedCJK_Letters-and-Months}', "");
Expect(1, 13055, '\P{^Is_Block=__enclosedCJK_Letters-and-Months}', "");
Expect(0, 12799, '\p{Is_Block=__enclosedCJK_Letters-and-Months}', "");
Expect(1, 12799, '\p{^Is_Block=__enclosedCJK_Letters-and-Months}', "");
Expect(1, 12799, '\P{Is_Block=__enclosedCJK_Letters-and-Months}', "");
Expect(0, 12799, '\P{^Is_Block=__enclosedCJK_Letters-and-Months}', "");
Error('\p{Is_Blk=_ENCLOSEDcjk Letters/a/ANDMONTHS}');
Error('\P{Is_Blk=_ENCLOSEDcjk Letters/a/ANDMONTHS}');
Expect(1, 13055, '\p{Is_Blk=enclosedcjklettersandmonths}', "");
Expect(0, 13055, '\p{^Is_Blk=enclosedcjklettersandmonths}', "");
Expect(0, 13055, '\P{Is_Blk=enclosedcjklettersandmonths}', "");
Expect(1, 13055, '\P{^Is_Blk=enclosedcjklettersandmonths}', "");
Expect(0, 12799, '\p{Is_Blk=enclosedcjklettersandmonths}', "");
Expect(1, 12799, '\p{^Is_Blk=enclosedcjklettersandmonths}', "");
Expect(1, 12799, '\P{Is_Blk=enclosedcjklettersandmonths}', "");
Expect(0, 12799, '\P{^Is_Blk=enclosedcjklettersandmonths}', "");
Expect(1, 13055, '\p{Is_Blk= 	ENCLOSED cjk	LETTERS And_Months}', "");
Expect(0, 13055, '\p{^Is_Blk= 	ENCLOSED cjk	LETTERS And_Months}', "");
Expect(0, 13055, '\P{Is_Blk= 	ENCLOSED cjk	LETTERS And_Months}', "");
Expect(1, 13055, '\P{^Is_Blk= 	ENCLOSED cjk	LETTERS And_Months}', "");
Expect(0, 12799, '\p{Is_Blk= 	ENCLOSED cjk	LETTERS And_Months}', "");
Expect(1, 12799, '\p{^Is_Blk= 	ENCLOSED cjk	LETTERS And_Months}', "");
Expect(1, 12799, '\P{Is_Blk= 	ENCLOSED cjk	LETTERS And_Months}', "");
Expect(0, 12799, '\P{^Is_Blk= 	ENCLOSED cjk	LETTERS And_Months}', "");
Error('\p{Block=__Combining:=diacritical_marks-Supplement}');
Error('\P{Block=__Combining:=diacritical_marks-Supplement}');
Expect(1, 7679, '\p{Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7679, '\p{^Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7679, '\P{Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7679, '\P{^Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7615, '\p{Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7615, '\p{^Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7615, '\P{Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7615, '\P{^Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7679, '\p{Block=-_Combining Diacritical Marks_SUPPLEMENT}', "");
Expect(0, 7679, '\p{^Block=-_Combining Diacritical Marks_SUPPLEMENT}', "");
Expect(0, 7679, '\P{Block=-_Combining Diacritical Marks_SUPPLEMENT}', "");
Expect(1, 7679, '\P{^Block=-_Combining Diacritical Marks_SUPPLEMENT}', "");
Expect(0, 7615, '\p{Block=-_Combining Diacritical Marks_SUPPLEMENT}', "");
Expect(1, 7615, '\p{^Block=-_Combining Diacritical Marks_SUPPLEMENT}', "");
Expect(1, 7615, '\P{Block=-_Combining Diacritical Marks_SUPPLEMENT}', "");
Expect(0, 7615, '\P{^Block=-_Combining Diacritical Marks_SUPPLEMENT}', "");
Error('\p{Blk=_ Combining-Diacritical/a/Marks_Supplement}');
Error('\P{Blk=_ Combining-Diacritical/a/Marks_Supplement}');
Expect(1, 7679, '\p{Blk=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7679, '\p{^Blk=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7679, '\P{Blk=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7679, '\P{^Blk=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7615, '\p{Blk=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7615, '\p{^Blk=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7615, '\P{Blk=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7615, '\P{^Blk=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7679, '\p{Blk=_combiningdiacriticalMARKS Supplement}', "");
Expect(0, 7679, '\p{^Blk=_combiningdiacriticalMARKS Supplement}', "");
Expect(0, 7679, '\P{Blk=_combiningdiacriticalMARKS Supplement}', "");
Expect(1, 7679, '\P{^Blk=_combiningdiacriticalMARKS Supplement}', "");
Expect(0, 7615, '\p{Blk=_combiningdiacriticalMARKS Supplement}', "");
Expect(1, 7615, '\p{^Blk=_combiningdiacriticalMARKS Supplement}', "");
Expect(1, 7615, '\P{Blk=_combiningdiacriticalMARKS Supplement}', "");
Expect(0, 7615, '\P{^Blk=_combiningdiacriticalMARKS Supplement}', "");
Error('\p{Is_Block:	 /a/combining_Diacritical_Marks supplement}');
Error('\P{Is_Block:	 /a/combining_Diacritical_Marks supplement}');
Expect(1, 7679, '\p{Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7679, '\p{^Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7679, '\P{Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7679, '\P{^Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7615, '\p{Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7615, '\p{^Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7615, '\P{Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7615, '\P{^Is_Block=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7679, '\p{Is_Block=  Combining-diacriticalMARKS_Supplement}', "");
Expect(0, 7679, '\p{^Is_Block=  Combining-diacriticalMARKS_Supplement}', "");
Expect(0, 7679, '\P{Is_Block=  Combining-diacriticalMARKS_Supplement}', "");
Expect(1, 7679, '\P{^Is_Block=  Combining-diacriticalMARKS_Supplement}', "");
Expect(0, 7615, '\p{Is_Block=  Combining-diacriticalMARKS_Supplement}', "");
Expect(1, 7615, '\p{^Is_Block=  Combining-diacriticalMARKS_Supplement}', "");
Expect(1, 7615, '\P{Is_Block=  Combining-diacriticalMARKS_Supplement}', "");
Expect(0, 7615, '\P{^Is_Block=  Combining-diacriticalMARKS_Supplement}', "");
Error('\p{Is_Blk=_/a/COMBINING-diacritical_MARKS SUPPLEMENT}');
Error('\P{Is_Blk=_/a/COMBINING-diacritical_MARKS SUPPLEMENT}');
Expect(1, 7679, '\p{Is_Blk=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7679, '\p{^Is_Blk=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7679, '\P{Is_Blk=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7679, '\P{^Is_Blk=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7615, '\p{Is_Blk=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7615, '\p{^Is_Blk=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7615, '\P{Is_Blk=combiningdiacriticalmarkssupplement}', "");
Expect(0, 7615, '\P{^Is_Blk=combiningdiacriticalmarkssupplement}', "");
Expect(1, 7679, '\p{Is_Blk=		Combining DIACRITICALmarks supplement}', "");
Expect(0, 7679, '\p{^Is_Blk=		Combining DIACRITICALmarks supplement}', "");
Expect(0, 7679, '\P{Is_Blk=		Combining DIACRITICALmarks supplement}', "");
Expect(1, 7679, '\P{^Is_Blk=		Combining DIACRITICALmarks supplement}', "");
Expect(0, 7615, '\p{Is_Blk=		Combining DIACRITICALmarks supplement}', "");
Expect(1, 7615, '\p{^Is_Blk=		Combining DIACRITICALmarks supplement}', "");
Expect(1, 7615, '\P{Is_Blk=		Combining DIACRITICALmarks supplement}', "");
Expect(0, 7615, '\P{^Is_Blk=		Combining DIACRITICALmarks supplement}', "");
Error('\p{Block=-_Bopomofo:=}');
Error('\P{Block=-_Bopomofo:=}');
Expect(1, 12591, '\p{Block=bopomofo}', "");
Expect(0, 12591, '\p{^Block=bopomofo}', "");
Expect(0, 12591, '\P{Block=bopomofo}', "");
Expect(1, 12591, '\P{^Block=bopomofo}', "");
Expect(0, 12543, '\p{Block=bopomofo}', "");
Expect(1, 12543, '\p{^Block=bopomofo}', "");
Expect(1, 12543, '\P{Block=bopomofo}', "");
Expect(0, 12543, '\P{^Block=bopomofo}', "");
Expect(1, 12591, '\p{Block=	-BOPOMOFO}', "");
Expect(0, 12591, '\p{^Block=	-BOPOMOFO}', "");
Expect(0, 12591, '\P{Block=	-BOPOMOFO}', "");
Expect(1, 12591, '\P{^Block=	-BOPOMOFO}', "");
Expect(0, 12543, '\p{Block=	-BOPOMOFO}', "");
Expect(1, 12543, '\p{^Block=	-BOPOMOFO}', "");
Expect(1, 12543, '\P{Block=	-BOPOMOFO}', "");
Expect(0, 12543, '\P{^Block=	-BOPOMOFO}', "");
Error('\p{Blk=-_Bopomofo:=}');
Error('\P{Blk=-_Bopomofo:=}');
Expect(1, 12591, '\p{Blk=bopomofo}', "");
Expect(0, 12591, '\p{^Blk=bopomofo}', "");
Expect(0, 12591, '\P{Blk=bopomofo}', "");
Expect(1, 12591, '\P{^Blk=bopomofo}', "");
Expect(0, 12543, '\p{Blk=bopomofo}', "");
Expect(1, 12543, '\p{^Blk=bopomofo}', "");
Expect(1, 12543, '\P{Blk=bopomofo}', "");
Expect(0, 12543, '\P{^Blk=bopomofo}', "");
Expect(1, 12591, '\p{Blk=-	Bopomofo}', "");
Expect(0, 12591, '\p{^Blk=-	Bopomofo}', "");
Expect(0, 12591, '\P{Blk=-	Bopomofo}', "");
Expect(1, 12591, '\P{^Blk=-	Bopomofo}', "");
Expect(0, 12543, '\p{Blk=-	Bopomofo}', "");
Expect(1, 12543, '\p{^Blk=-	Bopomofo}', "");
Expect(1, 12543, '\P{Blk=-	Bopomofo}', "");
Expect(0, 12543, '\P{^Blk=-	Bopomofo}', "");
Error('\p{Is_Block:	-Bopomofo:=}');
Error('\P{Is_Block:	-Bopomofo:=}');
Expect(1, 12591, '\p{Is_Block=bopomofo}', "");
Expect(0, 12591, '\p{^Is_Block=bopomofo}', "");
Expect(0, 12591, '\P{Is_Block=bopomofo}', "");
Expect(1, 12591, '\P{^Is_Block=bopomofo}', "");
Expect(0, 12543, '\p{Is_Block=bopomofo}', "");
Expect(1, 12543, '\p{^Is_Block=bopomofo}', "");
Expect(1, 12543, '\P{Is_Block=bopomofo}', "");
Expect(0, 12543, '\P{^Is_Block=bopomofo}', "");
Expect(1, 12591, '\p{Is_Block= _BOPOMOFO}', "");
Expect(0, 12591, '\p{^Is_Block= _BOPOMOFO}', "");
Expect(0, 12591, '\P{Is_Block= _BOPOMOFO}', "");
Expect(1, 12591, '\P{^Is_Block= _BOPOMOFO}', "");
Expect(0, 12543, '\p{Is_Block= _BOPOMOFO}', "");
Expect(1, 12543, '\p{^Is_Block= _BOPOMOFO}', "");
Expect(1, 12543, '\P{Is_Block= _BOPOMOFO}', "");
Expect(0, 12543, '\P{^Is_Block= _BOPOMOFO}', "");
Error('\p{Is_Blk::=- Bopomofo}');
Error('\P{Is_Blk::=- Bopomofo}');
Expect(1, 12591, '\p{Is_Blk=bopomofo}', "");
Expect(0, 12591, '\p{^Is_Blk=bopomofo}', "");
Expect(0, 12591, '\P{Is_Blk=bopomofo}', "");
Expect(1, 12591, '\P{^Is_Blk=bopomofo}', "");
Expect(0, 12543, '\p{Is_Blk=bopomofo}', "");
Expect(1, 12543, '\p{^Is_Blk=bopomofo}', "");
Expect(1, 12543, '\P{Is_Blk=bopomofo}', "");
Expect(0, 12543, '\P{^Is_Blk=bopomofo}', "");
Expect(1, 12591, '\p{Is_Blk=	 BOPOMOFO}', "");
Expect(0, 12591, '\p{^Is_Blk=	 BOPOMOFO}', "");
Expect(0, 12591, '\P{Is_Blk=	 BOPOMOFO}', "");
Expect(1, 12591, '\P{^Is_Blk=	 BOPOMOFO}', "");
Expect(0, 12543, '\p{Is_Blk=	 BOPOMOFO}', "");
Expect(1, 12543, '\p{^Is_Blk=	 BOPOMOFO}', "");
Expect(1, 12543, '\P{Is_Blk=	 BOPOMOFO}', "");
Expect(0, 12543, '\P{^Is_Blk=	 BOPOMOFO}', "");
Error('\p{Block= Miscellaneous:=technical}');
Error('\P{Block= Miscellaneous:=technical}');
Expect(1, 9215, '\p{Block=miscellaneoustechnical}', "");
Expect(0, 9215, '\p{^Block=miscellaneoustechnical}', "");
Expect(0, 9215, '\P{Block=miscellaneoustechnical}', "");
Expect(1, 9215, '\P{^Block=miscellaneoustechnical}', "");
Expect(0, 8959, '\p{Block=miscellaneoustechnical}', "");
Expect(1, 8959, '\p{^Block=miscellaneoustechnical}', "");
Expect(1, 8959, '\P{Block=miscellaneoustechnical}', "");
Expect(0, 8959, '\P{^Block=miscellaneoustechnical}', "");
Expect(1, 9215, '\p{Block= MiscellaneousTechnical}', "");
Expect(0, 9215, '\p{^Block= MiscellaneousTechnical}', "");
Expect(0, 9215, '\P{Block= MiscellaneousTechnical}', "");
Expect(1, 9215, '\P{^Block= MiscellaneousTechnical}', "");
Expect(0, 8959, '\p{Block= MiscellaneousTechnical}', "");
Expect(1, 8959, '\p{^Block= MiscellaneousTechnical}', "");
Expect(1, 8959, '\P{Block= MiscellaneousTechnical}', "");
Expect(0, 8959, '\P{^Block= MiscellaneousTechnical}', "");
Error('\p{Blk= _Miscellaneous_Technical:=}');
Error('\P{Blk= _Miscellaneous_Technical:=}');
Expect(1, 9215, '\p{Blk=miscellaneoustechnical}', "");
Expect(0, 9215, '\p{^Blk=miscellaneoustechnical}', "");
Expect(0, 9215, '\P{Blk=miscellaneoustechnical}', "");
Expect(1, 9215, '\P{^Blk=miscellaneoustechnical}', "");
Expect(0, 8959, '\p{Blk=miscellaneoustechnical}', "");
Expect(1, 8959, '\p{^Blk=miscellaneoustechnical}', "");
Expect(1, 8959, '\P{Blk=miscellaneoustechnical}', "");
Expect(0, 8959, '\P{^Blk=miscellaneoustechnical}', "");
Expect(1, 9215, '\p{Blk=	MISCELLANEOUS_Technical}', "");
Expect(0, 9215, '\p{^Blk=	MISCELLANEOUS_Technical}', "");
Expect(0, 9215, '\P{Blk=	MISCELLANEOUS_Technical}', "");
Expect(1, 9215, '\P{^Blk=	MISCELLANEOUS_Technical}', "");
Expect(0, 8959, '\p{Blk=	MISCELLANEOUS_Technical}', "");
Expect(1, 8959, '\p{^Blk=	MISCELLANEOUS_Technical}', "");
Expect(1, 8959, '\P{Blk=	MISCELLANEOUS_Technical}', "");
Expect(0, 8959, '\P{^Blk=	MISCELLANEOUS_Technical}', "");
Error('\p{Is_Block=		Miscellaneous_TECHNICAL:=}');
Error('\P{Is_Block=		Miscellaneous_TECHNICAL:=}');
Expect(1, 9215, '\p{Is_Block:   miscellaneoustechnical}', "");
Expect(0, 9215, '\p{^Is_Block:   miscellaneoustechnical}', "");
Expect(0, 9215, '\P{Is_Block:   miscellaneoustechnical}', "");
Expect(1, 9215, '\P{^Is_Block:   miscellaneoustechnical}', "");
Expect(0, 8959, '\p{Is_Block:   miscellaneoustechnical}', "");
Expect(1, 8959, '\p{^Is_Block:   miscellaneoustechnical}', "");
Expect(1, 8959, '\P{Is_Block:   miscellaneoustechnical}', "");
Expect(0, 8959, '\P{^Is_Block:   miscellaneoustechnical}', "");
Expect(1, 9215, '\p{Is_Block=_ Miscellaneous	technical}', "");
Expect(0, 9215, '\p{^Is_Block=_ Miscellaneous	technical}', "");
Expect(0, 9215, '\P{Is_Block=_ Miscellaneous	technical}', "");
Expect(1, 9215, '\P{^Is_Block=_ Miscellaneous	technical}', "");
Expect(0, 8959, '\p{Is_Block=_ Miscellaneous	technical}', "");
Expect(1, 8959, '\p{^Is_Block=_ Miscellaneous	technical}', "");
Expect(1, 8959, '\P{Is_Block=_ Miscellaneous	technical}', "");
Expect(0, 8959, '\P{^Is_Block=_ Miscellaneous	technical}', "");
Error('\p{Is_Blk=/a/Miscellaneous-technical}');
Error('\P{Is_Blk=/a/Miscellaneous-technical}');
Expect(1, 9215, '\p{Is_Blk: miscellaneoustechnical}', "");
Expect(0, 9215, '\p{^Is_Blk: miscellaneoustechnical}', "");
Expect(0, 9215, '\P{Is_Blk: miscellaneoustechnical}', "");
Expect(1, 9215, '\P{^Is_Blk: miscellaneoustechnical}', "");
Expect(0, 8959, '\p{Is_Blk: miscellaneoustechnical}', "");
Expect(1, 8959, '\p{^Is_Blk: miscellaneoustechnical}', "");
Expect(1, 8959, '\P{Is_Blk: miscellaneoustechnical}', "");
Expect(0, 8959, '\P{^Is_Blk: miscellaneoustechnical}', "");
Expect(1, 9215, '\p{Is_Blk=_-Miscellaneous Technical}', "");
Expect(0, 9215, '\p{^Is_Blk=_-Miscellaneous Technical}', "");
Expect(0, 9215, '\P{Is_Blk=_-Miscellaneous Technical}', "");
Expect(1, 9215, '\P{^Is_Blk=_-Miscellaneous Technical}', "");
Expect(0, 8959, '\p{Is_Blk=_-Miscellaneous Technical}', "");
Expect(1, 8959, '\p{^Is_Blk=_-Miscellaneous Technical}', "");
Expect(1, 8959, '\P{Is_Blk=_-Miscellaneous Technical}', "");
Expect(0, 8959, '\P{^Is_Blk=_-Miscellaneous Technical}', "");
Error('\p{Block=_:=Osmanya}');
Error('\P{Block=_:=Osmanya}');
Expect(1, 66735, '\p{Block=osmanya}', "");
Expect(0, 66735, '\p{^Block=osmanya}', "");
Expect(0, 66735, '\P{Block=osmanya}', "");
Expect(1, 66735, '\P{^Block=osmanya}', "");
Expect(0, 66687, '\p{Block=osmanya}', "");
Expect(1, 66687, '\p{^Block=osmanya}', "");
Expect(1, 66687, '\P{Block=osmanya}', "");
Expect(0, 66687, '\P{^Block=osmanya}', "");
Expect(1, 66735, '\p{Block= Osmanya}', "");
Expect(0, 66735, '\p{^Block= Osmanya}', "");
Expect(0, 66735, '\P{Block= Osmanya}', "");
Expect(1, 66735, '\P{^Block= Osmanya}', "");
Expect(0, 66687, '\p{Block= Osmanya}', "");
Expect(1, 66687, '\p{^Block= Osmanya}', "");
Expect(1, 66687, '\P{Block= Osmanya}', "");
Expect(0, 66687, '\P{^Block= Osmanya}', "");
Error('\p{Blk:   -_osmanya:=}');
Error('\P{Blk:   -_osmanya:=}');
Expect(1, 66735, '\p{Blk: osmanya}', "");
Expect(0, 66735, '\p{^Blk: osmanya}', "");
Expect(0, 66735, '\P{Blk: osmanya}', "");
Expect(1, 66735, '\P{^Blk: osmanya}', "");
Expect(0, 66687, '\p{Blk: osmanya}', "");
Expect(1, 66687, '\p{^Blk: osmanya}', "");
Expect(1, 66687, '\P{Blk: osmanya}', "");
Expect(0, 66687, '\P{^Blk: osmanya}', "");
Expect(1, 66735, '\p{Blk=_-Osmanya}', "");
Expect(0, 66735, '\p{^Blk=_-Osmanya}', "");
Expect(0, 66735, '\P{Blk=_-Osmanya}', "");
Expect(1, 66735, '\P{^Blk=_-Osmanya}', "");
Expect(0, 66687, '\p{Blk=_-Osmanya}', "");
Expect(1, 66687, '\p{^Blk=_-Osmanya}', "");
Expect(1, 66687, '\P{Blk=_-Osmanya}', "");
Expect(0, 66687, '\P{^Blk=_-Osmanya}', "");
Error('\p{Is_Block=:= _Osmanya}');
Error('\P{Is_Block=:= _Osmanya}');
Expect(1, 66735, '\p{Is_Block=osmanya}', "");
Expect(0, 66735, '\p{^Is_Block=osmanya}', "");
Expect(0, 66735, '\P{Is_Block=osmanya}', "");
Expect(1, 66735, '\P{^Is_Block=osmanya}', "");
Expect(0, 66687, '\p{Is_Block=osmanya}', "");
Expect(1, 66687, '\p{^Is_Block=osmanya}', "");
Expect(1, 66687, '\P{Is_Block=osmanya}', "");
Expect(0, 66687, '\P{^Is_Block=osmanya}', "");
Expect(1, 66735, '\p{Is_Block=_osmanya}', "");
Expect(0, 66735, '\p{^Is_Block=_osmanya}', "");
Expect(0, 66735, '\P{Is_Block=_osmanya}', "");
Expect(1, 66735, '\P{^Is_Block=_osmanya}', "");
Expect(0, 66687, '\p{Is_Block=_osmanya}', "");
Expect(1, 66687, '\p{^Is_Block=_osmanya}', "");
Expect(1, 66687, '\P{Is_Block=_osmanya}', "");
Expect(0, 66687, '\P{^Is_Block=_osmanya}', "");
Error('\p{Is_Blk=OSMANYA/a/}');
Error('\P{Is_Blk=OSMANYA/a/}');
Expect(1, 66735, '\p{Is_Blk=osmanya}', "");
Expect(0, 66735, '\p{^Is_Blk=osmanya}', "");
Expect(0, 66735, '\P{Is_Blk=osmanya}', "");
Expect(1, 66735, '\P{^Is_Blk=osmanya}', "");
Expect(0, 66687, '\p{Is_Blk=osmanya}', "");
Expect(1, 66687, '\p{^Is_Blk=osmanya}', "");
Expect(1, 66687, '\P{Is_Blk=osmanya}', "");
Expect(0, 66687, '\P{^Is_Blk=osmanya}', "");
Expect(1, 66735, '\p{Is_Blk=_	Osmanya}', "");
Expect(0, 66735, '\p{^Is_Blk=_	Osmanya}', "");
Expect(0, 66735, '\P{Is_Blk=_	Osmanya}', "");
Expect(1, 66735, '\P{^Is_Blk=_	Osmanya}', "");
Expect(0, 66687, '\p{Is_Blk=_	Osmanya}', "");
Expect(1, 66687, '\p{^Is_Blk=_	Osmanya}', "");
Expect(1, 66687, '\P{Is_Blk=_	Osmanya}', "");
Expect(0, 66687, '\P{^Is_Blk=_	Osmanya}', "");
Error('\p{Block=-Mahjong Tiles:=}');
Error('\P{Block=-Mahjong Tiles:=}');
Expect(1, 127023, '\p{Block=mahjongtiles}', "");
Expect(0, 127023, '\p{^Block=mahjongtiles}', "");
Expect(0, 127023, '\P{Block=mahjongtiles}', "");
Expect(1, 127023, '\P{^Block=mahjongtiles}', "");
Expect(0, 126975, '\p{Block=mahjongtiles}', "");
Expect(1, 126975, '\p{^Block=mahjongtiles}', "");
Expect(1, 126975, '\P{Block=mahjongtiles}', "");
Expect(0, 126975, '\P{^Block=mahjongtiles}', "");
Expect(1, 127023, '\p{Block=_-Mahjong-tiles}', "");
Expect(0, 127023, '\p{^Block=_-Mahjong-tiles}', "");
Expect(0, 127023, '\P{Block=_-Mahjong-tiles}', "");
Expect(1, 127023, '\P{^Block=_-Mahjong-tiles}', "");
Expect(0, 126975, '\p{Block=_-Mahjong-tiles}', "");
Expect(1, 126975, '\p{^Block=_-Mahjong-tiles}', "");
Expect(1, 126975, '\P{Block=_-Mahjong-tiles}', "");
Expect(0, 126975, '\P{^Block=_-Mahjong-tiles}', "");
Error('\p{Blk=-_MAHJONG:=tiles}');
Error('\P{Blk=-_MAHJONG:=tiles}');
Expect(1, 127023, '\p{Blk=mahjongtiles}', "");
Expect(0, 127023, '\p{^Blk=mahjongtiles}', "");
Expect(0, 127023, '\P{Blk=mahjongtiles}', "");
Expect(1, 127023, '\P{^Blk=mahjongtiles}', "");
Expect(0, 126975, '\p{Blk=mahjongtiles}', "");
Expect(1, 126975, '\p{^Blk=mahjongtiles}', "");
Expect(1, 126975, '\P{Blk=mahjongtiles}', "");
Expect(0, 126975, '\P{^Blk=mahjongtiles}', "");
Expect(1, 127023, '\p{Blk=- mahjongtiles}', "");
Expect(0, 127023, '\p{^Blk=- mahjongtiles}', "");
Expect(0, 127023, '\P{Blk=- mahjongtiles}', "");
Expect(1, 127023, '\P{^Blk=- mahjongtiles}', "");
Expect(0, 126975, '\p{Blk=- mahjongtiles}', "");
Expect(1, 126975, '\p{^Blk=- mahjongtiles}', "");
Expect(1, 126975, '\P{Blk=- mahjongtiles}', "");
Expect(0, 126975, '\P{^Blk=- mahjongtiles}', "");
Error('\p{Is_Block=	/a/mahjong-tiles}');
Error('\P{Is_Block=	/a/mahjong-tiles}');
Expect(1, 127023, '\p{Is_Block:mahjongtiles}', "");
Expect(0, 127023, '\p{^Is_Block:mahjongtiles}', "");
Expect(0, 127023, '\P{Is_Block:mahjongtiles}', "");
Expect(1, 127023, '\P{^Is_Block:mahjongtiles}', "");
Expect(0, 126975, '\p{Is_Block:mahjongtiles}', "");
Expect(1, 126975, '\p{^Is_Block:mahjongtiles}', "");
Expect(1, 126975, '\P{Is_Block:mahjongtiles}', "");
Expect(0, 126975, '\P{^Is_Block:mahjongtiles}', "");
Expect(1, 127023, '\p{Is_Block=	MAHJONGTiles}', "");
Expect(0, 127023, '\p{^Is_Block=	MAHJONGTiles}', "");
Expect(0, 127023, '\P{Is_Block=	MAHJONGTiles}', "");
Expect(1, 127023, '\P{^Is_Block=	MAHJONGTiles}', "");
Expect(0, 126975, '\p{Is_Block=	MAHJONGTiles}', "");
Expect(1, 126975, '\p{^Is_Block=	MAHJONGTiles}', "");
Expect(1, 126975, '\P{Is_Block=	MAHJONGTiles}', "");
Expect(0, 126975, '\P{^Is_Block=	MAHJONGTiles}', "");
Error('\p{Is_Blk=:=_ Mahjong_Tiles}');
Error('\P{Is_Blk=:=_ Mahjong_Tiles}');
Expect(1, 127023, '\p{Is_Blk=mahjongtiles}', "");
Expect(0, 127023, '\p{^Is_Blk=mahjongtiles}', "");
Expect(0, 127023, '\P{Is_Blk=mahjongtiles}', "");
Expect(1, 127023, '\P{^Is_Blk=mahjongtiles}', "");
Expect(0, 126975, '\p{Is_Blk=mahjongtiles}', "");
Expect(1, 126975, '\p{^Is_Blk=mahjongtiles}', "");
Expect(1, 126975, '\P{Is_Blk=mahjongtiles}', "");
Expect(0, 126975, '\P{^Is_Blk=mahjongtiles}', "");
Expect(1, 127023, '\p{Is_Blk= _Mahjong	Tiles}', "");
Expect(0, 127023, '\p{^Is_Blk= _Mahjong	Tiles}', "");
Expect(0, 127023, '\P{Is_Blk= _Mahjong	Tiles}', "");
Expect(1, 127023, '\P{^Is_Blk= _Mahjong	Tiles}', "");
Expect(0, 126975, '\p{Is_Blk= _Mahjong	Tiles}', "");
Expect(1, 126975, '\p{^Is_Blk= _Mahjong	Tiles}', "");
Expect(1, 126975, '\P{Is_Blk= _Mahjong	Tiles}', "");
Expect(0, 126975, '\P{^Is_Blk= _Mahjong	Tiles}', "");
Error('\p{Block=-arrows/a/}');
Error('\P{Block=-arrows/a/}');
Expect(1, 8703, '\p{Block=arrows}', "");
Expect(0, 8703, '\p{^Block=arrows}', "");
Expect(0, 8703, '\P{Block=arrows}', "");
Expect(1, 8703, '\P{^Block=arrows}', "");
Expect(0, 8591, '\p{Block=arrows}', "");
Expect(1, 8591, '\p{^Block=arrows}', "");
Expect(1, 8591, '\P{Block=arrows}', "");
Expect(0, 8591, '\P{^Block=arrows}', "");
Expect(1, 8703, '\p{Block=--Arrows}', "");
Expect(0, 8703, '\p{^Block=--Arrows}', "");
Expect(0, 8703, '\P{Block=--Arrows}', "");
Expect(1, 8703, '\P{^Block=--Arrows}', "");
Expect(0, 8591, '\p{Block=--Arrows}', "");
Expect(1, 8591, '\p{^Block=--Arrows}', "");
Expect(1, 8591, '\P{Block=--Arrows}', "");
Expect(0, 8591, '\P{^Block=--Arrows}', "");
Error('\p{Blk=	 Arrows/a/}');
Error('\P{Blk=	 Arrows/a/}');
Expect(1, 8703, '\p{Blk=arrows}', "");
Expect(0, 8703, '\p{^Blk=arrows}', "");
Expect(0, 8703, '\P{Blk=arrows}', "");
Expect(1, 8703, '\P{^Blk=arrows}', "");
Expect(0, 8591, '\p{Blk=arrows}', "");
Expect(1, 8591, '\p{^Blk=arrows}', "");
Expect(1, 8591, '\P{Blk=arrows}', "");
Expect(0, 8591, '\P{^Blk=arrows}', "");
Expect(1, 8703, '\p{Blk:  Arrows}', "");
Expect(0, 8703, '\p{^Blk:  Arrows}', "");
Expect(0, 8703, '\P{Blk:  Arrows}', "");
Expect(1, 8703, '\P{^Blk:  Arrows}', "");
Expect(0, 8591, '\p{Blk:  Arrows}', "");
Expect(1, 8591, '\p{^Blk:  Arrows}', "");
Expect(1, 8591, '\P{Blk:  Arrows}', "");
Expect(0, 8591, '\P{^Blk:  Arrows}', "");
Error('\p{Is_Block= -arrows/a/}');
Error('\P{Is_Block= -arrows/a/}');
Expect(1, 8703, '\p{Is_Block=arrows}', "");
Expect(0, 8703, '\p{^Is_Block=arrows}', "");
Expect(0, 8703, '\P{Is_Block=arrows}', "");
Expect(1, 8703, '\P{^Is_Block=arrows}', "");
Expect(0, 8591, '\p{Is_Block=arrows}', "");
Expect(1, 8591, '\p{^Is_Block=arrows}', "");
Expect(1, 8591, '\P{Is_Block=arrows}', "");
Expect(0, 8591, '\P{^Is_Block=arrows}', "");
Expect(1, 8703, '\p{Is_Block=	_Arrows}', "");
Expect(0, 8703, '\p{^Is_Block=	_Arrows}', "");
Expect(0, 8703, '\P{Is_Block=	_Arrows}', "");
Expect(1, 8703, '\P{^Is_Block=	_Arrows}', "");
Expect(0, 8591, '\p{Is_Block=	_Arrows}', "");
Expect(1, 8591, '\p{^Is_Block=	_Arrows}', "");
Expect(1, 8591, '\P{Is_Block=	_Arrows}', "");
Expect(0, 8591, '\P{^Is_Block=	_Arrows}', "");
Error('\p{Is_Blk=_:=Arrows}');
Error('\P{Is_Blk=_:=Arrows}');
Expect(1, 8703, '\p{Is_Blk=arrows}', "");
Expect(0, 8703, '\p{^Is_Blk=arrows}', "");
Expect(0, 8703, '\P{Is_Blk=arrows}', "");
Expect(1, 8703, '\P{^Is_Blk=arrows}', "");
Expect(0, 8591, '\p{Is_Blk=arrows}', "");
Expect(1, 8591, '\p{^Is_Blk=arrows}', "");
Expect(1, 8591, '\P{Is_Blk=arrows}', "");
Expect(0, 8591, '\P{^Is_Blk=arrows}', "");
Expect(1, 8703, '\p{Is_Blk=-_ARROWS}', "");
Expect(0, 8703, '\p{^Is_Blk=-_ARROWS}', "");
Expect(0, 8703, '\P{Is_Blk=-_ARROWS}', "");
Expect(1, 8703, '\P{^Is_Blk=-_ARROWS}', "");
Expect(0, 8591, '\p{Is_Blk=-_ARROWS}', "");
Expect(1, 8591, '\p{^Is_Blk=-_ARROWS}', "");
Expect(1, 8591, '\P{Is_Blk=-_ARROWS}', "");
Expect(0, 8591, '\P{^Is_Blk=-_ARROWS}', "");
Error('\p{Block=	:=Block Elements}');
Error('\P{Block=	:=Block Elements}');
Expect(1, 9631, '\p{Block=blockelements}', "");
Expect(0, 9631, '\p{^Block=blockelements}', "");
Expect(0, 9631, '\P{Block=blockelements}', "");
Expect(1, 9631, '\P{^Block=blockelements}', "");
Expect(0, 9599, '\p{Block=blockelements}', "");
Expect(1, 9599, '\p{^Block=blockelements}', "");
Expect(1, 9599, '\P{Block=blockelements}', "");
Expect(0, 9599, '\P{^Block=blockelements}', "");
Expect(1, 9631, '\p{Block=_-Block-Elements}', "");
Expect(0, 9631, '\p{^Block=_-Block-Elements}', "");
Expect(0, 9631, '\P{Block=_-Block-Elements}', "");
Expect(1, 9631, '\P{^Block=_-Block-Elements}', "");
Expect(0, 9599, '\p{Block=_-Block-Elements}', "");
Expect(1, 9599, '\p{^Block=_-Block-Elements}', "");
Expect(1, 9599, '\P{Block=_-Block-Elements}', "");
Expect(0, 9599, '\P{^Block=_-Block-Elements}', "");
Error('\p{Blk=/a/		BLOCK	elements}');
Error('\P{Blk=/a/		BLOCK	elements}');
Expect(1, 9631, '\p{Blk:	blockelements}', "");
Expect(0, 9631, '\p{^Blk:	blockelements}', "");
Expect(0, 9631, '\P{Blk:	blockelements}', "");
Expect(1, 9631, '\P{^Blk:	blockelements}', "");
Expect(0, 9599, '\p{Blk:	blockelements}', "");
Expect(1, 9599, '\p{^Blk:	blockelements}', "");
Expect(1, 9599, '\P{Blk:	blockelements}', "");
Expect(0, 9599, '\P{^Blk:	blockelements}', "");
Expect(1, 9631, '\p{Blk= _Block-Elements}', "");
Expect(0, 9631, '\p{^Blk= _Block-Elements}', "");
Expect(0, 9631, '\P{Blk= _Block-Elements}', "");
Expect(1, 9631, '\P{^Blk= _Block-Elements}', "");
Expect(0, 9599, '\p{Blk= _Block-Elements}', "");
Expect(1, 9599, '\p{^Blk= _Block-Elements}', "");
Expect(1, 9599, '\P{Blk= _Block-Elements}', "");
Expect(0, 9599, '\P{^Blk= _Block-Elements}', "");
Error('\p{Is_Block=/a/_ BLOCK Elements}');
Error('\P{Is_Block=/a/_ BLOCK Elements}');
Expect(1, 9631, '\p{Is_Block=blockelements}', "");
Expect(0, 9631, '\p{^Is_Block=blockelements}', "");
Expect(0, 9631, '\P{Is_Block=blockelements}', "");
Expect(1, 9631, '\P{^Is_Block=blockelements}', "");
Expect(0, 9599, '\p{Is_Block=blockelements}', "");
Expect(1, 9599, '\p{^Is_Block=blockelements}', "");
Expect(1, 9599, '\P{Is_Block=blockelements}', "");
Expect(0, 9599, '\P{^Is_Block=blockelements}', "");
Expect(1, 9631, '\p{Is_Block: -_BLOCKelements}', "");
Expect(0, 9631, '\p{^Is_Block: -_BLOCKelements}', "");
Expect(0, 9631, '\P{Is_Block: -_BLOCKelements}', "");
Expect(1, 9631, '\P{^Is_Block: -_BLOCKelements}', "");
Expect(0, 9599, '\p{Is_Block: -_BLOCKelements}', "");
Expect(1, 9599, '\p{^Is_Block: -_BLOCKelements}', "");
Expect(1, 9599, '\P{Is_Block: -_BLOCKelements}', "");
Expect(0, 9599, '\P{^Is_Block: -_BLOCKelements}', "");
Error('\p{Is_Blk=- block:=elements}');
Error('\P{Is_Blk=- block:=elements}');
Expect(1, 9631, '\p{Is_Blk=blockelements}', "");
Expect(0, 9631, '\p{^Is_Blk=blockelements}', "");
Expect(0, 9631, '\P{Is_Blk=blockelements}', "");
Expect(1, 9631, '\P{^Is_Blk=blockelements}', "");
Expect(0, 9599, '\p{Is_Blk=blockelements}', "");
Expect(1, 9599, '\p{^Is_Blk=blockelements}', "");
Expect(1, 9599, '\P{Is_Blk=blockelements}', "");
Expect(0, 9599, '\P{^Is_Blk=blockelements}', "");
Expect(1, 9631, '\p{Is_Blk=-_block ELEMENTS}', "");
Expect(0, 9631, '\p{^Is_Blk=-_block ELEMENTS}', "");
Expect(0, 9631, '\P{Is_Blk=-_block ELEMENTS}', "");
Expect(1, 9631, '\P{^Is_Blk=-_block ELEMENTS}', "");
Expect(0, 9599, '\p{Is_Blk=-_block ELEMENTS}', "");
Expect(1, 9599, '\p{^Is_Blk=-_block ELEMENTS}', "");
Expect(1, 9599, '\P{Is_Blk=-_block ELEMENTS}', "");
Expect(0, 9599, '\P{^Is_Blk=-_block ELEMENTS}', "");
Error('\p{Block=:=		CJKstrokes}');
Error('\P{Block=:=		CJKstrokes}');
Expect(1, 12783, '\p{Block=cjkstrokes}', "");
Expect(0, 12783, '\p{^Block=cjkstrokes}', "");
Expect(0, 12783, '\P{Block=cjkstrokes}', "");
Expect(1, 12783, '\P{^Block=cjkstrokes}', "");
Expect(0, 12735, '\p{Block=cjkstrokes}', "");
Expect(1, 12735, '\p{^Block=cjkstrokes}', "");
Expect(1, 12735, '\P{Block=cjkstrokes}', "");
Expect(0, 12735, '\P{^Block=cjkstrokes}', "");
Expect(1, 12783, '\p{Block= CJKSTROKES}', "");
Expect(0, 12783, '\p{^Block= CJKSTROKES}', "");
Expect(0, 12783, '\P{Block= CJKSTROKES}', "");
Expect(1, 12783, '\P{^Block= CJKSTROKES}', "");
Expect(0, 12735, '\p{Block= CJKSTROKES}', "");
Expect(1, 12735, '\p{^Block= CJKSTROKES}', "");
Expect(1, 12735, '\P{Block= CJKSTROKES}', "");
Expect(0, 12735, '\P{^Block= CJKSTROKES}', "");
Error('\p{Blk=_CJK:=STROKES}');
Error('\P{Blk=_CJK:=STROKES}');
Expect(1, 12783, '\p{Blk=cjkstrokes}', "");
Expect(0, 12783, '\p{^Blk=cjkstrokes}', "");
Expect(0, 12783, '\P{Blk=cjkstrokes}', "");
Expect(1, 12783, '\P{^Blk=cjkstrokes}', "");
Expect(0, 12735, '\p{Blk=cjkstrokes}', "");
Expect(1, 12735, '\p{^Blk=cjkstrokes}', "");
Expect(1, 12735, '\P{Blk=cjkstrokes}', "");
Expect(0, 12735, '\P{^Blk=cjkstrokes}', "");
Expect(1, 12783, '\p{Blk=	-cjkSTROKES}', "");
Expect(0, 12783, '\p{^Blk=	-cjkSTROKES}', "");
Expect(0, 12783, '\P{Blk=	-cjkSTROKES}', "");
Expect(1, 12783, '\P{^Blk=	-cjkSTROKES}', "");
Expect(0, 12735, '\p{Blk=	-cjkSTROKES}', "");
Expect(1, 12735, '\p{^Blk=	-cjkSTROKES}', "");
Expect(1, 12735, '\P{Blk=	-cjkSTROKES}', "");
Expect(0, 12735, '\P{^Blk=	-cjkSTROKES}', "");
Error('\p{Is_Block=_ CJK_Strokes:=}');
Error('\P{Is_Block=_ CJK_Strokes:=}');
Expect(1, 12783, '\p{Is_Block=cjkstrokes}', "");
Expect(0, 12783, '\p{^Is_Block=cjkstrokes}', "");
Expect(0, 12783, '\P{Is_Block=cjkstrokes}', "");
Expect(1, 12783, '\P{^Is_Block=cjkstrokes}', "");
Expect(0, 12735, '\p{Is_Block=cjkstrokes}', "");
Expect(1, 12735, '\p{^Is_Block=cjkstrokes}', "");
Expect(1, 12735, '\P{Is_Block=cjkstrokes}', "");
Expect(0, 12735, '\P{^Is_Block=cjkstrokes}', "");
Expect(1, 12783, '\p{Is_Block= CJK	Strokes}', "");
Expect(0, 12783, '\p{^Is_Block= CJK	Strokes}', "");
Expect(0, 12783, '\P{Is_Block= CJK	Strokes}', "");
Expect(1, 12783, '\P{^Is_Block= CJK	Strokes}', "");
Expect(0, 12735, '\p{Is_Block= CJK	Strokes}', "");
Expect(1, 12735, '\p{^Is_Block= CJK	Strokes}', "");
Expect(1, 12735, '\P{Is_Block= CJK	Strokes}', "");
Expect(0, 12735, '\P{^Is_Block= CJK	Strokes}', "");
Error('\p{Is_Blk=_/a/CJK_STROKES}');
Error('\P{Is_Blk=_/a/CJK_STROKES}');
Expect(1, 12783, '\p{Is_Blk=cjkstrokes}', "");
Expect(0, 12783, '\p{^Is_Blk=cjkstrokes}', "");
Expect(0, 12783, '\P{Is_Blk=cjkstrokes}', "");
Expect(1, 12783, '\P{^Is_Blk=cjkstrokes}', "");
Expect(0, 12735, '\p{Is_Blk=cjkstrokes}', "");
Expect(1, 12735, '\p{^Is_Blk=cjkstrokes}', "");
Expect(1, 12735, '\P{Is_Blk=cjkstrokes}', "");
Expect(0, 12735, '\P{^Is_Blk=cjkstrokes}', "");
Expect(1, 12783, '\p{Is_Blk=_	CJKstrokes}', "");
Expect(0, 12783, '\p{^Is_Blk=_	CJKstrokes}', "");
Expect(0, 12783, '\P{Is_Blk=_	CJKstrokes}', "");
Expect(1, 12783, '\P{^Is_Blk=_	CJKstrokes}', "");
Expect(0, 12735, '\p{Is_Blk=_	CJKstrokes}', "");
Expect(1, 12735, '\p{^Is_Blk=_	CJKstrokes}', "");
Expect(1, 12735, '\P{Is_Blk=_	CJKstrokes}', "");
Expect(0, 12735, '\P{^Is_Blk=_	CJKstrokes}', "");
Error('\p{Block=- noBlock:=}');
Error('\P{Block=- noBlock:=}');
Expect(1, 2303, '\p{Block=noblock}', "");
Expect(0, 2303, '\p{^Block=noblock}', "");
Expect(0, 2303, '\P{Block=noblock}', "");
Expect(1, 2303, '\P{^Block=noblock}', "");
Expect(0, 2143, '\p{Block=noblock}', "");
Expect(1, 2143, '\p{^Block=noblock}', "");
Expect(1, 2143, '\P{Block=noblock}', "");
Expect(0, 2143, '\P{^Block=noblock}', "");
Expect(1, 2303, '\p{Block= NO	Block}', "");
Expect(0, 2303, '\p{^Block= NO	Block}', "");
Expect(0, 2303, '\P{Block= NO	Block}', "");
Expect(1, 2303, '\P{^Block= NO	Block}', "");
Expect(0, 2143, '\p{Block= NO	Block}', "");
Expect(1, 2143, '\p{^Block= NO	Block}', "");
Expect(1, 2143, '\P{Block= NO	Block}', "");
Expect(0, 2143, '\P{^Block= NO	Block}', "");
Error('\p{Blk= noBlock:=}');
Error('\P{Blk= noBlock:=}');
Expect(1, 2303, '\p{Blk: noblock}', "");
Expect(0, 2303, '\p{^Blk: noblock}', "");
Expect(0, 2303, '\P{Blk: noblock}', "");
Expect(1, 2303, '\P{^Blk: noblock}', "");
Expect(0, 2143, '\p{Blk: noblock}', "");
Expect(1, 2143, '\p{^Blk: noblock}', "");
Expect(1, 2143, '\P{Blk: noblock}', "");
Expect(0, 2143, '\P{^Blk: noblock}', "");
Expect(1, 2303, '\p{Blk=-noBlock}', "");
Expect(0, 2303, '\p{^Blk=-noBlock}', "");
Expect(0, 2303, '\P{Blk=-noBlock}', "");
Expect(1, 2303, '\P{^Blk=-noBlock}', "");
Expect(0, 2143, '\p{Blk=-noBlock}', "");
Expect(1, 2143, '\p{^Blk=-noBlock}', "");
Expect(1, 2143, '\P{Blk=-noBlock}', "");
Expect(0, 2143, '\P{^Blk=-noBlock}', "");
Error('\p{Is_Block:	NO/a/block}');
Error('\P{Is_Block:	NO/a/block}');
Expect(1, 2303, '\p{Is_Block:   noblock}', "");
Expect(0, 2303, '\p{^Is_Block:   noblock}', "");
Expect(0, 2303, '\P{Is_Block:   noblock}', "");
Expect(1, 2303, '\P{^Is_Block:   noblock}', "");
Expect(0, 2143, '\p{Is_Block:   noblock}', "");
Expect(1, 2143, '\p{^Is_Block:   noblock}', "");
Expect(1, 2143, '\P{Is_Block:   noblock}', "");
Expect(0, 2143, '\P{^Is_Block:   noblock}', "");
Expect(1, 2303, '\p{Is_Block=-_NO	Block}', "");
Expect(0, 2303, '\p{^Is_Block=-_NO	Block}', "");
Expect(0, 2303, '\P{Is_Block=-_NO	Block}', "");
Expect(1, 2303, '\P{^Is_Block=-_NO	Block}', "");
Expect(0, 2143, '\p{Is_Block=-_NO	Block}', "");
Expect(1, 2143, '\p{^Is_Block=-_NO	Block}', "");
Expect(1, 2143, '\P{Is_Block=-_NO	Block}', "");
Expect(0, 2143, '\P{^Is_Block=-_NO	Block}', "");
Error('\p{Is_Blk=/a/_No Block}');
Error('\P{Is_Blk=/a/_No Block}');
Expect(1, 2303, '\p{Is_Blk=noblock}', "");
Expect(0, 2303, '\p{^Is_Blk=noblock}', "");
Expect(0, 2303, '\P{Is_Blk=noblock}', "");
Expect(1, 2303, '\P{^Is_Blk=noblock}', "");
Expect(0, 2143, '\p{Is_Blk=noblock}', "");
Expect(1, 2143, '\p{^Is_Blk=noblock}', "");
Expect(1, 2143, '\P{Is_Blk=noblock}', "");
Expect(0, 2143, '\P{^Is_Blk=noblock}', "");
Expect(1, 2303, '\p{Is_Blk=_-NO_BLOCK}', "");
Expect(0, 2303, '\p{^Is_Blk=_-NO_BLOCK}', "");
Expect(0, 2303, '\P{Is_Blk=_-NO_BLOCK}', "");
Expect(1, 2303, '\P{^Is_Blk=_-NO_BLOCK}', "");
Expect(0, 2143, '\p{Is_Blk=_-NO_BLOCK}', "");
Expect(1, 2143, '\p{^Is_Blk=_-NO_BLOCK}', "");
Expect(1, 2143, '\P{Is_Blk=_-NO_BLOCK}', "");
Expect(0, 2143, '\P{^Is_Blk=_-NO_BLOCK}', "");
Error('\p{Block= -cjk	Compatibility/a/}');
Error('\P{Block= -cjk	Compatibility/a/}');
Expect(1, 13311, '\p{Block:cjkcompatibility}', "");
Expect(0, 13311, '\p{^Block:cjkcompatibility}', "");
Expect(0, 13311, '\P{Block:cjkcompatibility}', "");
Expect(1, 13311, '\P{^Block:cjkcompatibility}', "");
Expect(0, 13055, '\p{Block:cjkcompatibility}', "");
Expect(1, 13055, '\p{^Block:cjkcompatibility}', "");
Expect(1, 13055, '\P{Block:cjkcompatibility}', "");
Expect(0, 13055, '\P{^Block:cjkcompatibility}', "");
Expect(1, 13311, '\p{Block=	CJK_compatibility}', "");
Expect(0, 13311, '\p{^Block=	CJK_compatibility}', "");
Expect(0, 13311, '\P{Block=	CJK_compatibility}', "");
Expect(1, 13311, '\P{^Block=	CJK_compatibility}', "");
Expect(0, 13055, '\p{Block=	CJK_compatibility}', "");
Expect(1, 13055, '\p{^Block=	CJK_compatibility}', "");
Expect(1, 13055, '\P{Block=	CJK_compatibility}', "");
Expect(0, 13055, '\P{^Block=	CJK_compatibility}', "");
Error('\p{Blk= :=CJK-COMPATIBILITY}');
Error('\P{Blk= :=CJK-COMPATIBILITY}');
Expect(1, 13311, '\p{Blk=cjkcompatibility}', "");
Expect(0, 13311, '\p{^Blk=cjkcompatibility}', "");
Expect(0, 13311, '\P{Blk=cjkcompatibility}', "");
Expect(1, 13311, '\P{^Blk=cjkcompatibility}', "");
Expect(0, 13055, '\p{Blk=cjkcompatibility}', "");
Expect(1, 13055, '\p{^Blk=cjkcompatibility}', "");
Expect(1, 13055, '\P{Blk=cjkcompatibility}', "");
Expect(0, 13055, '\P{^Blk=cjkcompatibility}', "");
Expect(1, 13311, '\p{Blk=	 CJK-COMPATIBILITY}', "");
Expect(0, 13311, '\p{^Blk=	 CJK-COMPATIBILITY}', "");
Expect(0, 13311, '\P{Blk=	 CJK-COMPATIBILITY}', "");
Expect(1, 13311, '\P{^Blk=	 CJK-COMPATIBILITY}', "");
Expect(0, 13055, '\p{Blk=	 CJK-COMPATIBILITY}', "");
Expect(1, 13055, '\p{^Blk=	 CJK-COMPATIBILITY}', "");
Expect(1, 13055, '\P{Blk=	 CJK-COMPATIBILITY}', "");
Expect(0, 13055, '\P{^Blk=	 CJK-COMPATIBILITY}', "");
Error('\p{Is_Block=-_CJK:=Compatibility}');
Error('\P{Is_Block=-_CJK:=Compatibility}');
Expect(1, 13311, '\p{Is_Block: cjkcompatibility}', "");
Expect(0, 13311, '\p{^Is_Block: cjkcompatibility}', "");
Expect(0, 13311, '\P{Is_Block: cjkcompatibility}', "");
Expect(1, 13311, '\P{^Is_Block: cjkcompatibility}', "");
Expect(0, 13055, '\p{Is_Block: cjkcompatibility}', "");
Expect(1, 13055, '\p{^Is_Block: cjkcompatibility}', "");
Expect(1, 13055, '\P{Is_Block: cjkcompatibility}', "");
Expect(0, 13055, '\P{^Is_Block: cjkcompatibility}', "");
Expect(1, 13311, '\p{Is_Block=__cjkCOMPATIBILITY}', "");
Expect(0, 13311, '\p{^Is_Block=__cjkCOMPATIBILITY}', "");
Expect(0, 13311, '\P{Is_Block=__cjkCOMPATIBILITY}', "");
Expect(1, 13311, '\P{^Is_Block=__cjkCOMPATIBILITY}', "");
Expect(0, 13055, '\p{Is_Block=__cjkCOMPATIBILITY}', "");
Expect(1, 13055, '\p{^Is_Block=__cjkCOMPATIBILITY}', "");
Expect(1, 13055, '\P{Is_Block=__cjkCOMPATIBILITY}', "");
Expect(0, 13055, '\P{^Is_Block=__cjkCOMPATIBILITY}', "");
Error('\p{Is_Blk=	-CJK/a/Compatibility}');
Error('\P{Is_Blk=	-CJK/a/Compatibility}');
Expect(1, 13311, '\p{Is_Blk=cjkcompatibility}', "");
Expect(0, 13311, '\p{^Is_Blk=cjkcompatibility}', "");
Expect(0, 13311, '\P{Is_Blk=cjkcompatibility}', "");
Expect(1, 13311, '\P{^Is_Blk=cjkcompatibility}', "");
Expect(0, 13055, '\p{Is_Blk=cjkcompatibility}', "");
Expect(1, 13055, '\p{^Is_Blk=cjkcompatibility}', "");
Expect(1, 13055, '\P{Is_Blk=cjkcompatibility}', "");
Expect(0, 13055, '\P{^Is_Blk=cjkcompatibility}', "");
Expect(1, 13311, '\p{Is_Blk=__CJK	compatibility}', "");
Expect(0, 13311, '\p{^Is_Blk=__CJK	compatibility}', "");
Expect(0, 13311, '\P{Is_Blk=__CJK	compatibility}', "");
Expect(1, 13311, '\P{^Is_Blk=__CJK	compatibility}', "");
Expect(0, 13055, '\p{Is_Blk=__CJK	compatibility}', "");
Expect(1, 13055, '\p{^Is_Blk=__CJK	compatibility}', "");
Expect(1, 13055, '\P{Is_Blk=__CJK	compatibility}', "");
Expect(0, 13055, '\P{^Is_Blk=__CJK	compatibility}', "");
Error('\p{Block:   	_Vedic:=Extensions}');
Error('\P{Block:   	_Vedic:=Extensions}');
Expect(1, 7423, '\p{Block=vedicextensions}', "");
Expect(0, 7423, '\p{^Block=vedicextensions}', "");
Expect(0, 7423, '\P{Block=vedicextensions}', "");
Expect(1, 7423, '\P{^Block=vedicextensions}', "");
Expect(0, 7375, '\p{Block=vedicextensions}', "");
Expect(1, 7375, '\p{^Block=vedicextensions}', "");
Expect(1, 7375, '\P{Block=vedicextensions}', "");
Expect(0, 7375, '\P{^Block=vedicextensions}', "");
Expect(1, 7423, '\p{Block= Vedic	Extensions}', "");
Expect(0, 7423, '\p{^Block= Vedic	Extensions}', "");
Expect(0, 7423, '\P{Block= Vedic	Extensions}', "");
Expect(1, 7423, '\P{^Block= Vedic	Extensions}', "");
Expect(0, 7375, '\p{Block= Vedic	Extensions}', "");
Expect(1, 7375, '\p{^Block= Vedic	Extensions}', "");
Expect(1, 7375, '\P{Block= Vedic	Extensions}', "");
Expect(0, 7375, '\P{^Block= Vedic	Extensions}', "");
Error('\p{Blk=--VEDIC	EXTENSIONS:=}');
Error('\P{Blk=--VEDIC	EXTENSIONS:=}');
Expect(1, 7423, '\p{Blk=vedicextensions}', "");
Expect(0, 7423, '\p{^Blk=vedicextensions}', "");
Expect(0, 7423, '\P{Blk=vedicextensions}', "");
Expect(1, 7423, '\P{^Blk=vedicextensions}', "");
Expect(0, 7375, '\p{Blk=vedicextensions}', "");
Expect(1, 7375, '\p{^Blk=vedicextensions}', "");
Expect(1, 7375, '\P{Blk=vedicextensions}', "");
Expect(0, 7375, '\P{^Blk=vedicextensions}', "");
Expect(1, 7423, '\p{Blk=	_VEDIC_extensions}', "");
Expect(0, 7423, '\p{^Blk=	_VEDIC_extensions}', "");
Expect(0, 7423, '\P{Blk=	_VEDIC_extensions}', "");
Expect(1, 7423, '\P{^Blk=	_VEDIC_extensions}', "");
Expect(0, 7375, '\p{Blk=	_VEDIC_extensions}', "");
Expect(1, 7375, '\p{^Blk=	_VEDIC_extensions}', "");
Expect(1, 7375, '\P{Blk=	_VEDIC_extensions}', "");
Expect(0, 7375, '\P{^Blk=	_VEDIC_extensions}', "");
Error('\p{Is_Block=	:=VEDICextensions}');
Error('\P{Is_Block=	:=VEDICextensions}');
Expect(1, 7423, '\p{Is_Block=vedicextensions}', "");
Expect(0, 7423, '\p{^Is_Block=vedicextensions}', "");
Expect(0, 7423, '\P{Is_Block=vedicextensions}', "");
Expect(1, 7423, '\P{^Is_Block=vedicextensions}', "");
Expect(0, 7375, '\p{Is_Block=vedicextensions}', "");
Expect(1, 7375, '\p{^Is_Block=vedicextensions}', "");
Expect(1, 7375, '\P{Is_Block=vedicextensions}', "");
Expect(0, 7375, '\P{^Is_Block=vedicextensions}', "");
Expect(1, 7423, '\p{Is_Block=_-Vedic extensions}', "");
Expect(0, 7423, '\p{^Is_Block=_-Vedic extensions}', "");
Expect(0, 7423, '\P{Is_Block=_-Vedic extensions}', "");
Expect(1, 7423, '\P{^Is_Block=_-Vedic extensions}', "");
Expect(0, 7375, '\p{Is_Block=_-Vedic extensions}', "");
Expect(1, 7375, '\p{^Is_Block=_-Vedic extensions}', "");
Expect(1, 7375, '\P{Is_Block=_-Vedic extensions}', "");
Expect(0, 7375, '\P{^Is_Block=_-Vedic extensions}', "");
Error('\p{Is_Blk=--Vedic-EXTENSIONS/a/}');
Error('\P{Is_Blk=--Vedic-EXTENSIONS/a/}');
Expect(1, 7423, '\p{Is_Blk=vedicextensions}', "");
Expect(0, 7423, '\p{^Is_Blk=vedicextensions}', "");
Expect(0, 7423, '\P{Is_Blk=vedicextensions}', "");
Expect(1, 7423, '\P{^Is_Blk=vedicextensions}', "");
Expect(0, 7375, '\p{Is_Blk=vedicextensions}', "");
Expect(1, 7375, '\p{^Is_Blk=vedicextensions}', "");
Expect(1, 7375, '\P{Is_Blk=vedicextensions}', "");
Expect(0, 7375, '\P{^Is_Blk=vedicextensions}', "");
Expect(1, 7423, '\p{Is_Blk=	 VedicEXTENSIONS}', "");
Expect(0, 7423, '\p{^Is_Blk=	 VedicEXTENSIONS}', "");
Expect(0, 7423, '\P{Is_Blk=	 VedicEXTENSIONS}', "");
Expect(1, 7423, '\P{^Is_Blk=	 VedicEXTENSIONS}', "");
Expect(0, 7375, '\p{Is_Blk=	 VedicEXTENSIONS}', "");
Expect(1, 7375, '\p{^Is_Blk=	 VedicEXTENSIONS}', "");
Expect(1, 7375, '\P{Is_Blk=	 VedicEXTENSIONS}', "");
Expect(0, 7375, '\P{^Is_Blk=	 VedicEXTENSIONS}', "");
Error('\p{Block:	/a/ tai viet}');
Error('\P{Block:	/a/ tai viet}');
Expect(1, 43743, '\p{Block=taiviet}', "");
Expect(0, 43743, '\p{^Block=taiviet}', "");
Expect(0, 43743, '\P{Block=taiviet}', "");
Expect(1, 43743, '\P{^Block=taiviet}', "");
Expect(0, 43647, '\p{Block=taiviet}', "");
Expect(1, 43647, '\p{^Block=taiviet}', "");
Expect(1, 43647, '\P{Block=taiviet}', "");
Expect(0, 43647, '\P{^Block=taiviet}', "");
Expect(1, 43743, '\p{Block=_TAI viet}', "");
Expect(0, 43743, '\p{^Block=_TAI viet}', "");
Expect(0, 43743, '\P{Block=_TAI viet}', "");
Expect(1, 43743, '\P{^Block=_TAI viet}', "");
Expect(0, 43647, '\p{Block=_TAI viet}', "");
Expect(1, 43647, '\p{^Block=_TAI viet}', "");
Expect(1, 43647, '\P{Block=_TAI viet}', "");
Expect(0, 43647, '\P{^Block=_TAI viet}', "");
Error('\p{Blk=_	Tai/a/VIET}');
Error('\P{Blk=_	Tai/a/VIET}');
Expect(1, 43743, '\p{Blk=taiviet}', "");
Expect(0, 43743, '\p{^Blk=taiviet}', "");
Expect(0, 43743, '\P{Blk=taiviet}', "");
Expect(1, 43743, '\P{^Blk=taiviet}', "");
Expect(0, 43647, '\p{Blk=taiviet}', "");
Expect(1, 43647, '\p{^Blk=taiviet}', "");
Expect(1, 43647, '\P{Blk=taiviet}', "");
Expect(0, 43647, '\P{^Blk=taiviet}', "");
Expect(1, 43743, '\p{Blk:	-Tai	viet}', "");
Expect(0, 43743, '\p{^Blk:	-Tai	viet}', "");
Expect(0, 43743, '\P{Blk:	-Tai	viet}', "");
Expect(1, 43743, '\P{^Blk:	-Tai	viet}', "");
Expect(0, 43647, '\p{Blk:	-Tai	viet}', "");
Expect(1, 43647, '\p{^Blk:	-Tai	viet}', "");
Expect(1, 43647, '\P{Blk:	-Tai	viet}', "");
Expect(0, 43647, '\P{^Blk:	-Tai	viet}', "");
Error('\p{Is_Block=_ TAI:=Viet}');
Error('\P{Is_Block=_ TAI:=Viet}');
Expect(1, 43743, '\p{Is_Block=taiviet}', "");
Expect(0, 43743, '\p{^Is_Block=taiviet}', "");
Expect(0, 43743, '\P{Is_Block=taiviet}', "");
Expect(1, 43743, '\P{^Is_Block=taiviet}', "");
Expect(0, 43647, '\p{Is_Block=taiviet}', "");
Expect(1, 43647, '\p{^Is_Block=taiviet}', "");
Expect(1, 43647, '\P{Is_Block=taiviet}', "");
Expect(0, 43647, '\P{^Is_Block=taiviet}', "");
Expect(1, 43743, '\p{Is_Block= TAI viet}', "");
Expect(0, 43743, '\p{^Is_Block= TAI viet}', "");
Expect(0, 43743, '\P{Is_Block= TAI viet}', "");
Expect(1, 43743, '\P{^Is_Block= TAI viet}', "");
Expect(0, 43647, '\p{Is_Block= TAI viet}', "");
Expect(1, 43647, '\p{^Is_Block= TAI viet}', "");
Expect(1, 43647, '\P{Is_Block= TAI viet}', "");
Expect(0, 43647, '\P{^Is_Block= TAI viet}', "");
Error('\p{Is_Blk=__TAI Viet:=}');
Error('\P{Is_Blk=__TAI Viet:=}');
Expect(1, 43743, '\p{Is_Blk=taiviet}', "");
Expect(0, 43743, '\p{^Is_Blk=taiviet}', "");
Expect(0, 43743, '\P{Is_Blk=taiviet}', "");
Expect(1, 43743, '\P{^Is_Blk=taiviet}', "");
Expect(0, 43647, '\p{Is_Blk=taiviet}', "");
Expect(1, 43647, '\p{^Is_Blk=taiviet}', "");
Expect(1, 43647, '\P{Is_Blk=taiviet}', "");
Expect(0, 43647, '\P{^Is_Blk=taiviet}', "");
Expect(1, 43743, '\p{Is_Blk=  Tai	viet}', "");
Expect(0, 43743, '\p{^Is_Blk=  Tai	viet}', "");
Expect(0, 43743, '\P{Is_Blk=  Tai	viet}', "");
Expect(1, 43743, '\P{^Is_Blk=  Tai	viet}', "");
Expect(0, 43647, '\p{Is_Blk=  Tai	viet}', "");
Expect(1, 43647, '\p{^Is_Blk=  Tai	viet}', "");
Expect(1, 43647, '\P{Is_Blk=  Tai	viet}', "");
Expect(0, 43647, '\P{^Is_Blk=  Tai	viet}', "");
Error('\p{Block:  /a/CYRILLICextendedB}');
Error('\P{Block:  /a/CYRILLICextendedB}');
Expect(1, 42655, '\p{Block=cyrillicextendedb}', "");
Expect(0, 42655, '\p{^Block=cyrillicextendedb}', "");
Expect(0, 42655, '\P{Block=cyrillicextendedb}', "");
Expect(1, 42655, '\P{^Block=cyrillicextendedb}', "");
Expect(0, 42559, '\p{Block=cyrillicextendedb}', "");
Expect(1, 42559, '\p{^Block=cyrillicextendedb}', "");
Expect(1, 42559, '\P{Block=cyrillicextendedb}', "");
Expect(0, 42559, '\P{^Block=cyrillicextendedb}', "");
Expect(1, 42655, '\p{Block=_CYRILLIC	extended-B}', "");
Expect(0, 42655, '\p{^Block=_CYRILLIC	extended-B}', "");
Expect(0, 42655, '\P{Block=_CYRILLIC	extended-B}', "");
Expect(1, 42655, '\P{^Block=_CYRILLIC	extended-B}', "");
Expect(0, 42559, '\p{Block=_CYRILLIC	extended-B}', "");
Expect(1, 42559, '\p{^Block=_CYRILLIC	extended-B}', "");
Expect(1, 42559, '\P{Block=_CYRILLIC	extended-B}', "");
Expect(0, 42559, '\P{^Block=_CYRILLIC	extended-B}', "");
Error('\p{Blk=/a/ cyrillic Extended-B}');
Error('\P{Blk=/a/ cyrillic Extended-B}');
Expect(1, 42655, '\p{Blk=cyrillicextendedb}', "");
Expect(0, 42655, '\p{^Blk=cyrillicextendedb}', "");
Expect(0, 42655, '\P{Blk=cyrillicextendedb}', "");
Expect(1, 42655, '\P{^Blk=cyrillicextendedb}', "");
Expect(0, 42559, '\p{Blk=cyrillicextendedb}', "");
Expect(1, 42559, '\p{^Blk=cyrillicextendedb}', "");
Expect(1, 42559, '\P{Blk=cyrillicextendedb}', "");
Expect(0, 42559, '\P{^Blk=cyrillicextendedb}', "");
Expect(1, 42655, '\p{Blk=_-CYRILLIC_Extended-B}', "");
Expect(0, 42655, '\p{^Blk=_-CYRILLIC_Extended-B}', "");
Expect(0, 42655, '\P{Blk=_-CYRILLIC_Extended-B}', "");
Expect(1, 42655, '\P{^Blk=_-CYRILLIC_Extended-B}', "");
Expect(0, 42559, '\p{Blk=_-CYRILLIC_Extended-B}', "");
Expect(1, 42559, '\p{^Blk=_-CYRILLIC_Extended-B}', "");
Expect(1, 42559, '\P{Blk=_-CYRILLIC_Extended-B}', "");
Expect(0, 42559, '\P{^Blk=_-CYRILLIC_Extended-B}', "");
Error('\p{Is_Block= 	cyrillicExtended/a/B}');
Error('\P{Is_Block= 	cyrillicExtended/a/B}');
Expect(1, 42655, '\p{Is_Block=cyrillicextendedb}', "");
Expect(0, 42655, '\p{^Is_Block=cyrillicextendedb}', "");
Expect(0, 42655, '\P{Is_Block=cyrillicextendedb}', "");
Expect(1, 42655, '\P{^Is_Block=cyrillicextendedb}', "");
Expect(0, 42559, '\p{Is_Block=cyrillicextendedb}', "");
Expect(1, 42559, '\p{^Is_Block=cyrillicextendedb}', "");
Expect(1, 42559, '\P{Is_Block=cyrillicextendedb}', "");
Expect(0, 42559, '\P{^Is_Block=cyrillicextendedb}', "");
Expect(1, 42655, '\p{Is_Block=	Cyrillic-Extended	B}', "");
Expect(0, 42655, '\p{^Is_Block=	Cyrillic-Extended	B}', "");
Expect(0, 42655, '\P{Is_Block=	Cyrillic-Extended	B}', "");
Expect(1, 42655, '\P{^Is_Block=	Cyrillic-Extended	B}', "");
Expect(0, 42559, '\p{Is_Block=	Cyrillic-Extended	B}', "");
Expect(1, 42559, '\p{^Is_Block=	Cyrillic-Extended	B}', "");
Expect(1, 42559, '\P{Is_Block=	Cyrillic-Extended	B}', "");
Expect(0, 42559, '\P{^Is_Block=	Cyrillic-Extended	B}', "");
Error('\p{Is_Blk=:=CYRILLIC-Extended-B}');
Error('\P{Is_Blk=:=CYRILLIC-Extended-B}');
Expect(1, 42655, '\p{Is_Blk=cyrillicextendedb}', "");
Expect(0, 42655, '\p{^Is_Blk=cyrillicextendedb}', "");
Expect(0, 42655, '\P{Is_Blk=cyrillicextendedb}', "");
Expect(1, 42655, '\P{^Is_Blk=cyrillicextendedb}', "");
Expect(0, 42559, '\p{Is_Blk=cyrillicextendedb}', "");
Expect(1, 42559, '\p{^Is_Blk=cyrillicextendedb}', "");
Expect(1, 42559, '\P{Is_Blk=cyrillicextendedb}', "");
Expect(0, 42559, '\P{^Is_Blk=cyrillicextendedb}', "");
Expect(1, 42655, '\p{Is_Blk=_ Cyrillic_Extended-b}', "");
Expect(0, 42655, '\p{^Is_Blk=_ Cyrillic_Extended-b}', "");
Expect(0, 42655, '\P{Is_Blk=_ Cyrillic_Extended-b}', "");
Expect(1, 42655, '\P{^Is_Blk=_ Cyrillic_Extended-b}', "");
Expect(0, 42559, '\p{Is_Blk=_ Cyrillic_Extended-b}', "");
Expect(1, 42559, '\p{^Is_Blk=_ Cyrillic_Extended-b}', "");
Expect(1, 42559, '\P{Is_Blk=_ Cyrillic_Extended-b}', "");
Expect(0, 42559, '\P{^Is_Blk=_ Cyrillic_Extended-b}', "");
Error('\p{Block=:= Phonetic-EXTENSIONS}');
Error('\P{Block=:= Phonetic-EXTENSIONS}');
Expect(1, 7551, '\p{Block=phoneticextensions}', "");
Expect(0, 7551, '\p{^Block=phoneticextensions}', "");
Expect(0, 7551, '\P{Block=phoneticextensions}', "");
Expect(1, 7551, '\P{^Block=phoneticextensions}', "");
Expect(0, 7423, '\p{Block=phoneticextensions}', "");
Expect(1, 7423, '\p{^Block=phoneticextensions}', "");
Expect(1, 7423, '\P{Block=phoneticextensions}', "");
Expect(0, 7423, '\P{^Block=phoneticextensions}', "");
Expect(1, 7551, '\p{Block=	_Phonetic extensions}', "");
Expect(0, 7551, '\p{^Block=	_Phonetic extensions}', "");
Expect(0, 7551, '\P{Block=	_Phonetic extensions}', "");
Expect(1, 7551, '\P{^Block=	_Phonetic extensions}', "");
Expect(0, 7423, '\p{Block=	_Phonetic extensions}', "");
Expect(1, 7423, '\p{^Block=	_Phonetic extensions}', "");
Expect(1, 7423, '\P{Block=	_Phonetic extensions}', "");
Expect(0, 7423, '\P{^Block=	_Phonetic extensions}', "");
Error('\p{Blk= Phonetic/a/EXTENSIONS}');
Error('\P{Blk= Phonetic/a/EXTENSIONS}');
Expect(1, 7551, '\p{Blk=phoneticextensions}', "");
Expect(0, 7551, '\p{^Blk=phoneticextensions}', "");
Expect(0, 7551, '\P{Blk=phoneticextensions}', "");
Expect(1, 7551, '\P{^Blk=phoneticextensions}', "");
Expect(0, 7423, '\p{Blk=phoneticextensions}', "");
Expect(1, 7423, '\p{^Blk=phoneticextensions}', "");
Expect(1, 7423, '\P{Blk=phoneticextensions}', "");
Expect(0, 7423, '\P{^Blk=phoneticextensions}', "");
Expect(1, 7551, '\p{Blk=-	phonetic	EXTENSIONS}', "");
Expect(0, 7551, '\p{^Blk=-	phonetic	EXTENSIONS}', "");
Expect(0, 7551, '\P{Blk=-	phonetic	EXTENSIONS}', "");
Expect(1, 7551, '\P{^Blk=-	phonetic	EXTENSIONS}', "");
Expect(0, 7423, '\p{Blk=-	phonetic	EXTENSIONS}', "");
Expect(1, 7423, '\p{^Blk=-	phonetic	EXTENSIONS}', "");
Expect(1, 7423, '\P{Blk=-	phonetic	EXTENSIONS}', "");
Expect(0, 7423, '\P{^Blk=-	phonetic	EXTENSIONS}', "");
Error('\p{Is_Block:	:= -phonetic Extensions}');
Error('\P{Is_Block:	:= -phonetic Extensions}');
Expect(1, 7551, '\p{Is_Block=phoneticextensions}', "");
Expect(0, 7551, '\p{^Is_Block=phoneticextensions}', "");
Expect(0, 7551, '\P{Is_Block=phoneticextensions}', "");
Expect(1, 7551, '\P{^Is_Block=phoneticextensions}', "");
Expect(0, 7423, '\p{Is_Block=phoneticextensions}', "");
Expect(1, 7423, '\p{^Is_Block=phoneticextensions}', "");
Expect(1, 7423, '\P{Is_Block=phoneticextensions}', "");
Expect(0, 7423, '\P{^Is_Block=phoneticextensions}', "");
Expect(1, 7551, '\p{Is_Block:   -_PHONETICExtensions}', "");
Expect(0, 7551, '\p{^Is_Block:   -_PHONETICExtensions}', "");
Expect(0, 7551, '\P{Is_Block:   -_PHONETICExtensions}', "");
Expect(1, 7551, '\P{^Is_Block:   -_PHONETICExtensions}', "");
Expect(0, 7423, '\p{Is_Block:   -_PHONETICExtensions}', "");
Expect(1, 7423, '\p{^Is_Block:   -_PHONETICExtensions}', "");
Expect(1, 7423, '\P{Is_Block:   -_PHONETICExtensions}', "");
Expect(0, 7423, '\P{^Is_Block:   -_PHONETICExtensions}', "");
Error('\p{Is_Blk=	:=Phonetic_Extensions}');
Error('\P{Is_Blk=	:=Phonetic_Extensions}');
Expect(1, 7551, '\p{Is_Blk=phoneticextensions}', "");
Expect(0, 7551, '\p{^Is_Blk=phoneticextensions}', "");
Expect(0, 7551, '\P{Is_Blk=phoneticextensions}', "");
Expect(1, 7551, '\P{^Is_Blk=phoneticextensions}', "");
Expect(0, 7423, '\p{Is_Blk=phoneticextensions}', "");
Expect(1, 7423, '\p{^Is_Blk=phoneticextensions}', "");
Expect(1, 7423, '\P{Is_Blk=phoneticextensions}', "");
Expect(0, 7423, '\P{^Is_Blk=phoneticextensions}', "");
Expect(1, 7551, '\p{Is_Blk=	Phonetic	Extensions}', "");
Expect(0, 7551, '\p{^Is_Blk=	Phonetic	Extensions}', "");
Expect(0, 7551, '\P{Is_Blk=	Phonetic	Extensions}', "");
Expect(1, 7551, '\P{^Is_Blk=	Phonetic	Extensions}', "");
Expect(0, 7423, '\p{Is_Blk=	Phonetic	Extensions}', "");
Expect(1, 7423, '\p{^Is_Blk=	Phonetic	Extensions}', "");
Expect(1, 7423, '\P{Is_Blk=	Phonetic	Extensions}', "");
Expect(0, 7423, '\P{^Is_Blk=	Phonetic	Extensions}', "");
Error('\p{Block=	-Tai_XUAN_JING Symbols/a/}');
Error('\P{Block=	-Tai_XUAN_JING Symbols/a/}');
Expect(1, 119647, '\p{Block=taixuanjingsymbols}', "");
Expect(0, 119647, '\p{^Block=taixuanjingsymbols}', "");
Expect(0, 119647, '\P{Block=taixuanjingsymbols}', "");
Expect(1, 119647, '\P{^Block=taixuanjingsymbols}', "");
Expect(0, 119551, '\p{Block=taixuanjingsymbols}', "");
Expect(1, 119551, '\p{^Block=taixuanjingsymbols}', "");
Expect(1, 119551, '\P{Block=taixuanjingsymbols}', "");
Expect(0, 119551, '\P{^Block=taixuanjingsymbols}', "");
Expect(1, 119647, '\p{Block=_-TAI-xuan_Jing symbols}', "");
Expect(0, 119647, '\p{^Block=_-TAI-xuan_Jing symbols}', "");
Expect(0, 119647, '\P{Block=_-TAI-xuan_Jing symbols}', "");
Expect(1, 119647, '\P{^Block=_-TAI-xuan_Jing symbols}', "");
Expect(0, 119551, '\p{Block=_-TAI-xuan_Jing symbols}', "");
Expect(1, 119551, '\p{^Block=_-TAI-xuan_Jing symbols}', "");
Expect(1, 119551, '\P{Block=_-TAI-xuan_Jing symbols}', "");
Expect(0, 119551, '\P{^Block=_-TAI-xuan_Jing symbols}', "");
Error('\p{Blk=- Tai/a/Xuan-Jing-Symbols}');
Error('\P{Blk=- Tai/a/Xuan-Jing-Symbols}');
Expect(1, 119647, '\p{Blk=taixuanjingsymbols}', "");
Expect(0, 119647, '\p{^Blk=taixuanjingsymbols}', "");
Expect(0, 119647, '\P{Blk=taixuanjingsymbols}', "");
Expect(1, 119647, '\P{^Blk=taixuanjingsymbols}', "");
Expect(0, 119551, '\p{Blk=taixuanjingsymbols}', "");
Expect(1, 119551, '\p{^Blk=taixuanjingsymbols}', "");
Expect(1, 119551, '\P{Blk=taixuanjingsymbols}', "");
Expect(0, 119551, '\P{^Blk=taixuanjingsymbols}', "");
Expect(1, 119647, '\p{Blk=Tai Xuan JING SYMBOLS}', "");
Expect(0, 119647, '\p{^Blk=Tai Xuan JING SYMBOLS}', "");
Expect(0, 119647, '\P{Blk=Tai Xuan JING SYMBOLS}', "");
Expect(1, 119647, '\P{^Blk=Tai Xuan JING SYMBOLS}', "");
Expect(0, 119551, '\p{Blk=Tai Xuan JING SYMBOLS}', "");
Expect(1, 119551, '\p{^Blk=Tai Xuan JING SYMBOLS}', "");
Expect(1, 119551, '\P{Blk=Tai Xuan JING SYMBOLS}', "");
Expect(0, 119551, '\P{^Blk=Tai Xuan JING SYMBOLS}', "");
Error('\p{Is_Block=:=	_TAI-XuanJing_Symbols}');
Error('\P{Is_Block=:=	_TAI-XuanJing_Symbols}');
Expect(1, 119647, '\p{Is_Block=taixuanjingsymbols}', "");
Expect(0, 119647, '\p{^Is_Block=taixuanjingsymbols}', "");
Expect(0, 119647, '\P{Is_Block=taixuanjingsymbols}', "");
Expect(1, 119647, '\P{^Is_Block=taixuanjingsymbols}', "");
Expect(0, 119551, '\p{Is_Block=taixuanjingsymbols}', "");
Expect(1, 119551, '\p{^Is_Block=taixuanjingsymbols}', "");
Expect(1, 119551, '\P{Is_Block=taixuanjingsymbols}', "");
Expect(0, 119551, '\P{^Is_Block=taixuanjingsymbols}', "");
Expect(1, 119647, '\p{Is_Block=-Tai-Xuan	jing_Symbols}', "");
Expect(0, 119647, '\p{^Is_Block=-Tai-Xuan	jing_Symbols}', "");
Expect(0, 119647, '\P{Is_Block=-Tai-Xuan	jing_Symbols}', "");
Expect(1, 119647, '\P{^Is_Block=-Tai-Xuan	jing_Symbols}', "");
Expect(0, 119551, '\p{Is_Block=-Tai-Xuan	jing_Symbols}', "");
Expect(1, 119551, '\p{^Is_Block=-Tai-Xuan	jing_Symbols}', "");
Expect(1, 119551, '\P{Is_Block=-Tai-Xuan	jing_Symbols}', "");
Expect(0, 119551, '\P{^Is_Block=-Tai-Xuan	jing_Symbols}', "");
Error('\p{Is_Blk:-	tai-Xuan/a/jingSYMBOLS}');
Error('\P{Is_Blk:-	tai-Xuan/a/jingSYMBOLS}');
Expect(1, 119647, '\p{Is_Blk=taixuanjingsymbols}', "");
Expect(0, 119647, '\p{^Is_Blk=taixuanjingsymbols}', "");
Expect(0, 119647, '\P{Is_Blk=taixuanjingsymbols}', "");
Expect(1, 119647, '\P{^Is_Blk=taixuanjingsymbols}', "");
Expect(0, 119551, '\p{Is_Blk=taixuanjingsymbols}', "");
Expect(1, 119551, '\p{^Is_Blk=taixuanjingsymbols}', "");
Expect(1, 119551, '\P{Is_Blk=taixuanjingsymbols}', "");
Expect(0, 119551, '\P{^Is_Blk=taixuanjingsymbols}', "");
Expect(1, 119647, '\p{Is_Blk=--TAI_XUAN_JING_symbols}', "");
Expect(0, 119647, '\p{^Is_Blk=--TAI_XUAN_JING_symbols}', "");
Expect(0, 119647, '\P{Is_Blk=--TAI_XUAN_JING_symbols}', "");
Expect(1, 119647, '\P{^Is_Blk=--TAI_XUAN_JING_symbols}', "");
Expect(0, 119551, '\p{Is_Blk=--TAI_XUAN_JING_symbols}', "");
Expect(1, 119551, '\p{^Is_Blk=--TAI_XUAN_JING_symbols}', "");
Expect(1, 119551, '\P{Is_Blk=--TAI_XUAN_JING_symbols}', "");
Expect(0, 119551, '\P{^Is_Blk=--TAI_XUAN_JING_symbols}', "");
Error('\p{Block= -MALAYALAM:=}');
Error('\P{Block= -MALAYALAM:=}');
Expect(1, 3455, '\p{Block=malayalam}', "");
Expect(0, 3455, '\p{^Block=malayalam}', "");
Expect(0, 3455, '\P{Block=malayalam}', "");
Expect(1, 3455, '\P{^Block=malayalam}', "");
Expect(0, 3327, '\p{Block=malayalam}', "");
Expect(1, 3327, '\p{^Block=malayalam}', "");
Expect(1, 3327, '\P{Block=malayalam}', "");
Expect(0, 3327, '\P{^Block=malayalam}', "");
Expect(1, 3455, '\p{Block=	_Malayalam}', "");
Expect(0, 3455, '\p{^Block=	_Malayalam}', "");
Expect(0, 3455, '\P{Block=	_Malayalam}', "");
Expect(1, 3455, '\P{^Block=	_Malayalam}', "");
Expect(0, 3327, '\p{Block=	_Malayalam}', "");
Expect(1, 3327, '\p{^Block=	_Malayalam}', "");
Expect(1, 3327, '\P{Block=	_Malayalam}', "");
Expect(0, 3327, '\P{^Block=	_Malayalam}', "");
Error('\p{Blk=:=_ Malayalam}');
Error('\P{Blk=:=_ Malayalam}');
Expect(1, 3455, '\p{Blk=malayalam}', "");
Expect(0, 3455, '\p{^Blk=malayalam}', "");
Expect(0, 3455, '\P{Blk=malayalam}', "");
Expect(1, 3455, '\P{^Blk=malayalam}', "");
Expect(0, 3327, '\p{Blk=malayalam}', "");
Expect(1, 3327, '\p{^Blk=malayalam}', "");
Expect(1, 3327, '\P{Blk=malayalam}', "");
Expect(0, 3327, '\P{^Blk=malayalam}', "");
Expect(1, 3455, '\p{Blk= Malayalam}', "");
Expect(0, 3455, '\p{^Blk= Malayalam}', "");
Expect(0, 3455, '\P{Blk= Malayalam}', "");
Expect(1, 3455, '\P{^Blk= Malayalam}', "");
Expect(0, 3327, '\p{Blk= Malayalam}', "");
Expect(1, 3327, '\p{^Blk= Malayalam}', "");
Expect(1, 3327, '\P{Blk= Malayalam}', "");
Expect(0, 3327, '\P{^Blk= Malayalam}', "");
Error('\p{Is_Block=/a/-Malayalam}');
Error('\P{Is_Block=/a/-Malayalam}');
Expect(1, 3455, '\p{Is_Block=malayalam}', "");
Expect(0, 3455, '\p{^Is_Block=malayalam}', "");
Expect(0, 3455, '\P{Is_Block=malayalam}', "");
Expect(1, 3455, '\P{^Is_Block=malayalam}', "");
Expect(0, 3327, '\p{Is_Block=malayalam}', "");
Expect(1, 3327, '\p{^Is_Block=malayalam}', "");
Expect(1, 3327, '\P{Is_Block=malayalam}', "");
Expect(0, 3327, '\P{^Is_Block=malayalam}', "");
Expect(1, 3455, '\p{Is_Block=__MALAYALAM}', "");
Expect(0, 3455, '\p{^Is_Block=__MALAYALAM}', "");
Expect(0, 3455, '\P{Is_Block=__MALAYALAM}', "");
Expect(1, 3455, '\P{^Is_Block=__MALAYALAM}', "");
Expect(0, 3327, '\p{Is_Block=__MALAYALAM}', "");
Expect(1, 3327, '\p{^Is_Block=__MALAYALAM}', "");
Expect(1, 3327, '\P{Is_Block=__MALAYALAM}', "");
Expect(0, 3327, '\P{^Is_Block=__MALAYALAM}', "");
Error('\p{Is_Blk=--MALAYALAM:=}');
Error('\P{Is_Blk=--MALAYALAM:=}');
Expect(1, 3455, '\p{Is_Blk=malayalam}', "");
Expect(0, 3455, '\p{^Is_Blk=malayalam}', "");
Expect(0, 3455, '\P{Is_Blk=malayalam}', "");
Expect(1, 3455, '\P{^Is_Blk=malayalam}', "");
Expect(0, 3327, '\p{Is_Blk=malayalam}', "");
Expect(1, 3327, '\p{^Is_Blk=malayalam}', "");
Expect(1, 3327, '\P{Is_Blk=malayalam}', "");
Expect(0, 3327, '\P{^Is_Blk=malayalam}', "");
Expect(1, 3455, '\p{Is_Blk=	MALAYALAM}', "");
Expect(0, 3455, '\p{^Is_Blk=	MALAYALAM}', "");
Expect(0, 3455, '\P{Is_Blk=	MALAYALAM}', "");
Expect(1, 3455, '\P{^Is_Blk=	MALAYALAM}', "");
Expect(0, 3327, '\p{Is_Blk=	MALAYALAM}', "");
Expect(1, 3327, '\p{^Is_Blk=	MALAYALAM}', "");
Expect(1, 3327, '\P{Is_Blk=	MALAYALAM}', "");
Expect(0, 3327, '\P{^Is_Blk=	MALAYALAM}', "");
Error('\p{Block=	-Lisu:=}');
Error('\P{Block=	-Lisu:=}');
Expect(1, 42239, '\p{Block:	lisu}', "");
Expect(0, 42239, '\p{^Block:	lisu}', "");
Expect(0, 42239, '\P{Block:	lisu}', "");
Expect(1, 42239, '\P{^Block:	lisu}', "");
Expect(0, 42191, '\p{Block:	lisu}', "");
Expect(1, 42191, '\p{^Block:	lisu}', "");
Expect(1, 42191, '\P{Block:	lisu}', "");
Expect(0, 42191, '\P{^Block:	lisu}', "");
Expect(1, 42239, '\p{Block=	-LISU}', "");
Expect(0, 42239, '\p{^Block=	-LISU}', "");
Expect(0, 42239, '\P{Block=	-LISU}', "");
Expect(1, 42239, '\P{^Block=	-LISU}', "");
Expect(0, 42191, '\p{Block=	-LISU}', "");
Expect(1, 42191, '\p{^Block=	-LISU}', "");
Expect(1, 42191, '\P{Block=	-LISU}', "");
Expect(0, 42191, '\P{^Block=	-LISU}', "");
Error('\p{Blk=:=Lisu}');
Error('\P{Blk=:=Lisu}');
Expect(1, 42239, '\p{Blk=lisu}', "");
Expect(0, 42239, '\p{^Blk=lisu}', "");
Expect(0, 42239, '\P{Blk=lisu}', "");
Expect(1, 42239, '\P{^Blk=lisu}', "");
Expect(0, 42191, '\p{Blk=lisu}', "");
Expect(1, 42191, '\p{^Blk=lisu}', "");
Expect(1, 42191, '\P{Blk=lisu}', "");
Expect(0, 42191, '\P{^Blk=lisu}', "");
Expect(1, 42239, '\p{Blk= Lisu}', "");
Expect(0, 42239, '\p{^Blk= Lisu}', "");
Expect(0, 42239, '\P{Blk= Lisu}', "");
Expect(1, 42239, '\P{^Blk= Lisu}', "");
Expect(0, 42191, '\p{Blk= Lisu}', "");
Expect(1, 42191, '\p{^Blk= Lisu}', "");
Expect(1, 42191, '\P{Blk= Lisu}', "");
Expect(0, 42191, '\P{^Blk= Lisu}', "");
Error('\p{Is_Block=-/a/Lisu}');
Error('\P{Is_Block=-/a/Lisu}');
Expect(1, 42239, '\p{Is_Block=lisu}', "");
Expect(0, 42239, '\p{^Is_Block=lisu}', "");
Expect(0, 42239, '\P{Is_Block=lisu}', "");
Expect(1, 42239, '\P{^Is_Block=lisu}', "");
Expect(0, 42191, '\p{Is_Block=lisu}', "");
Expect(1, 42191, '\p{^Is_Block=lisu}', "");
Expect(1, 42191, '\P{Is_Block=lisu}', "");
Expect(0, 42191, '\P{^Is_Block=lisu}', "");
Expect(1, 42239, '\p{Is_Block= -lisu}', "");
Expect(0, 42239, '\p{^Is_Block= -lisu}', "");
Expect(0, 42239, '\P{Is_Block= -lisu}', "");
Expect(1, 42239, '\P{^Is_Block= -lisu}', "");
Expect(0, 42191, '\p{Is_Block= -lisu}', "");
Expect(1, 42191, '\p{^Is_Block= -lisu}', "");
Expect(1, 42191, '\P{Is_Block= -lisu}', "");
Expect(0, 42191, '\P{^Is_Block= -lisu}', "");
Error('\p{Is_Blk=-/a/Lisu}');
Error('\P{Is_Blk=-/a/Lisu}');
Expect(1, 42239, '\p{Is_Blk=lisu}', "");
Expect(0, 42239, '\p{^Is_Blk=lisu}', "");
Expect(0, 42239, '\P{Is_Blk=lisu}', "");
Expect(1, 42239, '\P{^Is_Blk=lisu}', "");
Expect(0, 42191, '\p{Is_Blk=lisu}', "");
Expect(1, 42191, '\p{^Is_Blk=lisu}', "");
Expect(1, 42191, '\P{Is_Blk=lisu}', "");
Expect(0, 42191, '\P{^Is_Blk=lisu}', "");
Expect(1, 42239, '\p{Is_Blk=__Lisu}', "");
Expect(0, 42239, '\p{^Is_Blk=__Lisu}', "");
Expect(0, 42239, '\P{Is_Blk=__Lisu}', "");
Expect(1, 42239, '\P{^Is_Blk=__Lisu}', "");
Expect(0, 42191, '\p{Is_Blk=__Lisu}', "");
Expect(1, 42191, '\p{^Is_Blk=__Lisu}', "");
Expect(1, 42191, '\P{Is_Blk=__Lisu}', "");
Expect(0, 42191, '\P{^Is_Blk=__Lisu}', "");
Error('\p{Block=		ARABIC	PresentationForms/a/A}');
Error('\P{Block=		ARABIC	PresentationForms/a/A}');
Expect(1, 65023, '\p{Block=arabicpresentationformsa}', "");
Expect(0, 65023, '\p{^Block=arabicpresentationformsa}', "");
Expect(0, 65023, '\P{Block=arabicpresentationformsa}', "");
Expect(1, 65023, '\P{^Block=arabicpresentationformsa}', "");
Expect(0, 64335, '\p{Block=arabicpresentationformsa}', "");
Expect(1, 64335, '\p{^Block=arabicpresentationformsa}', "");
Expect(1, 64335, '\P{Block=arabicpresentationformsa}', "");
Expect(0, 64335, '\P{^Block=arabicpresentationformsa}', "");
Expect(1, 65023, '\p{Block= Arabic_presentation_forms-a}', "");
Expect(0, 65023, '\p{^Block= Arabic_presentation_forms-a}', "");
Expect(0, 65023, '\P{Block= Arabic_presentation_forms-a}', "");
Expect(1, 65023, '\P{^Block= Arabic_presentation_forms-a}', "");
Expect(0, 64335, '\p{Block= Arabic_presentation_forms-a}', "");
Expect(1, 64335, '\p{^Block= Arabic_presentation_forms-a}', "");
Expect(1, 64335, '\P{Block= Arabic_presentation_forms-a}', "");
Expect(0, 64335, '\P{^Block= Arabic_presentation_forms-a}', "");
Error('\p{Blk=- Arabic/a/Presentation_FORMS A}');
Error('\P{Blk=- Arabic/a/Presentation_FORMS A}');
Expect(1, 65023, '\p{Blk=arabicpresentationformsa}', "");
Expect(0, 65023, '\p{^Blk=arabicpresentationformsa}', "");
Expect(0, 65023, '\P{Blk=arabicpresentationformsa}', "");
Expect(1, 65023, '\P{^Blk=arabicpresentationformsa}', "");
Expect(0, 64335, '\p{Blk=arabicpresentationformsa}', "");
Expect(1, 64335, '\p{^Blk=arabicpresentationformsa}', "");
Expect(1, 64335, '\P{Blk=arabicpresentationformsa}', "");
Expect(0, 64335, '\P{^Blk=arabicpresentationformsa}', "");
Expect(1, 65023, '\p{Blk=	_Arabic	presentation Formsa}', "");
Expect(0, 65023, '\p{^Blk=	_Arabic	presentation Formsa}', "");
Expect(0, 65023, '\P{Blk=	_Arabic	presentation Formsa}', "");
Expect(1, 65023, '\P{^Blk=	_Arabic	presentation Formsa}', "");
Expect(0, 64335, '\p{Blk=	_Arabic	presentation Formsa}', "");
Expect(1, 64335, '\p{^Blk=	_Arabic	presentation Formsa}', "");
Expect(1, 64335, '\P{Blk=	_Arabic	presentation Formsa}', "");
Expect(0, 64335, '\P{^Blk=	_Arabic	presentation Formsa}', "");
Error('\p{Is_Block:	_-arabic-Presentation-forms:=A}');
Error('\P{Is_Block:	_-arabic-Presentation-forms:=A}');
Expect(1, 65023, '\p{Is_Block=arabicpresentationformsa}', "");
Expect(0, 65023, '\p{^Is_Block=arabicpresentationformsa}', "");
Expect(0, 65023, '\P{Is_Block=arabicpresentationformsa}', "");
Expect(1, 65023, '\P{^Is_Block=arabicpresentationformsa}', "");
Expect(0, 64335, '\p{Is_Block=arabicpresentationformsa}', "");
Expect(1, 64335, '\p{^Is_Block=arabicpresentationformsa}', "");
Expect(1, 64335, '\P{Is_Block=arabicpresentationformsa}', "");
Expect(0, 64335, '\P{^Is_Block=arabicpresentationformsa}', "");
Expect(1, 65023, '\p{Is_Block=-Arabic PRESENTATION forms A}', "");
Expect(0, 65023, '\p{^Is_Block=-Arabic PRESENTATION forms A}', "");
Expect(0, 65023, '\P{Is_Block=-Arabic PRESENTATION forms A}', "");
Expect(1, 65023, '\P{^Is_Block=-Arabic PRESENTATION forms A}', "");
Expect(0, 64335, '\p{Is_Block=-Arabic PRESENTATION forms A}', "");
Expect(1, 64335, '\p{^Is_Block=-Arabic PRESENTATION forms A}', "");
Expect(1, 64335, '\P{Is_Block=-Arabic PRESENTATION forms A}', "");
Expect(0, 64335, '\P{^Is_Block=-Arabic PRESENTATION forms A}', "");
Error('\p{Is_Blk=-Arabic_Presentation/a/forms A}');
Error('\P{Is_Blk=-Arabic_Presentation/a/forms A}');
Expect(1, 65023, '\p{Is_Blk=arabicpresentationformsa}', "");
Expect(0, 65023, '\p{^Is_Blk=arabicpresentationformsa}', "");
Expect(0, 65023, '\P{Is_Blk=arabicpresentationformsa}', "");
Expect(1, 65023, '\P{^Is_Blk=arabicpresentationformsa}', "");
Expect(0, 64335, '\p{Is_Blk=arabicpresentationformsa}', "");
Expect(1, 64335, '\p{^Is_Blk=arabicpresentationformsa}', "");
Expect(1, 64335, '\P{Is_Blk=arabicpresentationformsa}', "");
Expect(0, 64335, '\P{^Is_Blk=arabicpresentationformsa}', "");
Expect(1, 65023, '\p{Is_Blk=	arabic_PRESENTATION	Forms_A}', "");
Expect(0, 65023, '\p{^Is_Blk=	arabic_PRESENTATION	Forms_A}', "");
Expect(0, 65023, '\P{Is_Blk=	arabic_PRESENTATION	Forms_A}', "");
Expect(1, 65023, '\P{^Is_Blk=	arabic_PRESENTATION	Forms_A}', "");
Expect(0, 64335, '\p{Is_Blk=	arabic_PRESENTATION	Forms_A}', "");
Expect(1, 64335, '\p{^Is_Blk=	arabic_PRESENTATION	Forms_A}', "");
Expect(1, 64335, '\P{Is_Blk=	arabic_PRESENTATION	Forms_A}', "");
Expect(0, 64335, '\P{^Is_Blk=	arabic_PRESENTATION	Forms_A}', "");
Error('\p{Block:--taiLE:=}');
Error('\P{Block:--taiLE:=}');
Expect(1, 6527, '\p{Block=taile}', "");
Expect(0, 6527, '\p{^Block=taile}', "");
Expect(0, 6527, '\P{Block=taile}', "");
Expect(1, 6527, '\P{^Block=taile}', "");
Expect(0, 6479, '\p{Block=taile}', "");
Expect(1, 6479, '\p{^Block=taile}', "");
Expect(1, 6479, '\P{Block=taile}', "");
Expect(0, 6479, '\P{^Block=taile}', "");
Expect(1, 6527, '\p{Block=_tai_LE}', "");
Expect(0, 6527, '\p{^Block=_tai_LE}', "");
Expect(0, 6527, '\P{Block=_tai_LE}', "");
Expect(1, 6527, '\P{^Block=_tai_LE}', "");
Expect(0, 6479, '\p{Block=_tai_LE}', "");
Expect(1, 6479, '\p{^Block=_tai_LE}', "");
Expect(1, 6479, '\P{Block=_tai_LE}', "");
Expect(0, 6479, '\P{^Block=_tai_LE}', "");
Error('\p{Blk=_:=TaiLe}');
Error('\P{Blk=_:=TaiLe}');
Expect(1, 6527, '\p{Blk=taile}', "");
Expect(0, 6527, '\p{^Blk=taile}', "");
Expect(0, 6527, '\P{Blk=taile}', "");
Expect(1, 6527, '\P{^Blk=taile}', "");
Expect(0, 6479, '\p{Blk=taile}', "");
Expect(1, 6479, '\p{^Blk=taile}', "");
Expect(1, 6479, '\P{Blk=taile}', "");
Expect(0, 6479, '\P{^Blk=taile}', "");
Expect(1, 6527, '\p{Blk=	_TAI	le}', "");
Expect(0, 6527, '\p{^Blk=	_TAI	le}', "");
Expect(0, 6527, '\P{Blk=	_TAI	le}', "");
Expect(1, 6527, '\P{^Blk=	_TAI	le}', "");
Expect(0, 6479, '\p{Blk=	_TAI	le}', "");
Expect(1, 6479, '\p{^Blk=	_TAI	le}', "");
Expect(1, 6479, '\P{Blk=	_TAI	le}', "");
Expect(0, 6479, '\P{^Blk=	_TAI	le}', "");
Error('\p{Is_Block=_tai/a/Le}');
Error('\P{Is_Block=_tai/a/Le}');
Expect(1, 6527, '\p{Is_Block=taile}', "");
Expect(0, 6527, '\p{^Is_Block=taile}', "");
Expect(0, 6527, '\P{Is_Block=taile}', "");
Expect(1, 6527, '\P{^Is_Block=taile}', "");
Expect(0, 6479, '\p{Is_Block=taile}', "");
Expect(1, 6479, '\p{^Is_Block=taile}', "");
Expect(1, 6479, '\P{Is_Block=taile}', "");
Expect(0, 6479, '\P{^Is_Block=taile}', "");
Expect(1, 6527, '\p{Is_Block=--TAI	Le}', "");
Expect(0, 6527, '\p{^Is_Block=--TAI	Le}', "");
Expect(0, 6527, '\P{Is_Block=--TAI	Le}', "");
Expect(1, 6527, '\P{^Is_Block=--TAI	Le}', "");
Expect(0, 6479, '\p{Is_Block=--TAI	Le}', "");
Expect(1, 6479, '\p{^Is_Block=--TAI	Le}', "");
Expect(1, 6479, '\P{Is_Block=--TAI	Le}', "");
Expect(0, 6479, '\P{^Is_Block=--TAI	Le}', "");
Error('\p{Is_Blk=	TAIle/a/}');
Error('\P{Is_Blk=	TAIle/a/}');
Expect(1, 6527, '\p{Is_Blk=taile}', "");
Expect(0, 6527, '\p{^Is_Blk=taile}', "");
Expect(0, 6527, '\P{Is_Blk=taile}', "");
Expect(1, 6527, '\P{^Is_Blk=taile}', "");
Expect(0, 6479, '\p{Is_Blk=taile}', "");
Expect(1, 6479, '\p{^Is_Blk=taile}', "");
Expect(1, 6479, '\P{Is_Blk=taile}', "");
Expect(0, 6479, '\P{^Is_Blk=taile}', "");
Expect(1, 6527, '\p{Is_Blk=_-Tai Le}', "");
Expect(0, 6527, '\p{^Is_Blk=_-Tai Le}', "");
Expect(0, 6527, '\P{Is_Blk=_-Tai Le}', "");
Expect(1, 6527, '\P{^Is_Blk=_-Tai Le}', "");
Expect(0, 6479, '\p{Is_Blk=_-Tai Le}', "");
Expect(1, 6479, '\p{^Is_Blk=_-Tai Le}', "");
Expect(1, 6479, '\P{Is_Blk=_-Tai Le}', "");
Expect(0, 6479, '\P{^Is_Blk=_-Tai Le}', "");
Error('\p{Block=_:=Combiningdiacritical_Marks}');
Error('\P{Block=_:=Combiningdiacritical_Marks}');
Expect(1, 879, '\p{Block=combiningdiacriticalmarks}', "");
Expect(0, 879, '\p{^Block=combiningdiacriticalmarks}', "");
Expect(0, 879, '\P{Block=combiningdiacriticalmarks}', "");
Expect(1, 879, '\P{^Block=combiningdiacriticalmarks}', "");
Expect(0, 767, '\p{Block=combiningdiacriticalmarks}', "");
Expect(1, 767, '\p{^Block=combiningdiacriticalmarks}', "");
Expect(1, 767, '\P{Block=combiningdiacriticalmarks}', "");
Expect(0, 767, '\P{^Block=combiningdiacriticalmarks}', "");
Expect(1, 879, '\p{Block=_Combining Diacritical_Marks}', "");
Expect(0, 879, '\p{^Block=_Combining Diacritical_Marks}', "");
Expect(0, 879, '\P{Block=_Combining Diacritical_Marks}', "");
Expect(1, 879, '\P{^Block=_Combining Diacritical_Marks}', "");
Expect(0, 767, '\p{Block=_Combining Diacritical_Marks}', "");
Expect(1, 767, '\p{^Block=_Combining Diacritical_Marks}', "");
Expect(1, 767, '\P{Block=_Combining Diacritical_Marks}', "");
Expect(0, 767, '\P{^Block=_Combining Diacritical_Marks}', "");
Error('\p{Blk=_:=Combining-DIACRITICAL marks}');
Error('\P{Blk=_:=Combining-DIACRITICAL marks}');
Expect(1, 879, '\p{Blk=combiningdiacriticalmarks}', "");
Expect(0, 879, '\p{^Blk=combiningdiacriticalmarks}', "");
Expect(0, 879, '\P{Blk=combiningdiacriticalmarks}', "");
Expect(1, 879, '\P{^Blk=combiningdiacriticalmarks}', "");
Expect(0, 767, '\p{Blk=combiningdiacriticalmarks}', "");
Expect(1, 767, '\p{^Blk=combiningdiacriticalmarks}', "");
Expect(1, 767, '\P{Blk=combiningdiacriticalmarks}', "");
Expect(0, 767, '\P{^Blk=combiningdiacriticalmarks}', "");
Expect(1, 879, '\p{Blk=_ combiningdiacritical	Marks}', "");
Expect(0, 879, '\p{^Blk=_ combiningdiacritical	Marks}', "");
Expect(0, 879, '\P{Blk=_ combiningdiacritical	Marks}', "");
Expect(1, 879, '\P{^Blk=_ combiningdiacritical	Marks}', "");
Expect(0, 767, '\p{Blk=_ combiningdiacritical	Marks}', "");
Expect(1, 767, '\p{^Blk=_ combiningdiacritical	Marks}', "");
Expect(1, 767, '\P{Blk=_ combiningdiacritical	Marks}', "");
Expect(0, 767, '\P{^Blk=_ combiningdiacritical	Marks}', "");
Error('\p{Is_Block=:=_-combining	Diacritical_MARKS}');
Error('\P{Is_Block=:=_-combining	Diacritical_MARKS}');
Expect(1, 879, '\p{Is_Block=combiningdiacriticalmarks}', "");
Expect(0, 879, '\p{^Is_Block=combiningdiacriticalmarks}', "");
Expect(0, 879, '\P{Is_Block=combiningdiacriticalmarks}', "");
Expect(1, 879, '\P{^Is_Block=combiningdiacriticalmarks}', "");
Expect(0, 767, '\p{Is_Block=combiningdiacriticalmarks}', "");
Expect(1, 767, '\p{^Is_Block=combiningdiacriticalmarks}', "");
Expect(1, 767, '\P{Is_Block=combiningdiacriticalmarks}', "");
Expect(0, 767, '\P{^Is_Block=combiningdiacriticalmarks}', "");
Expect(1, 879, '\p{Is_Block:			COMBINING-Diacritical	MARKS}', "");
Expect(0, 879, '\p{^Is_Block:			COMBINING-Diacritical	MARKS}', "");
Expect(0, 879, '\P{Is_Block:			COMBINING-Diacritical	MARKS}', "");
Expect(1, 879, '\P{^Is_Block:			COMBINING-Diacritical	MARKS}', "");
Expect(0, 767, '\p{Is_Block:			COMBINING-Diacritical	MARKS}', "");
Expect(1, 767, '\p{^Is_Block:			COMBINING-Diacritical	MARKS}', "");
Expect(1, 767, '\P{Is_Block:			COMBINING-Diacritical	MARKS}', "");
Expect(0, 767, '\P{^Is_Block:			COMBINING-Diacritical	MARKS}', "");
Error('\p{Is_Blk=:=- Combining DIACRITICAL_MARKS}');
Error('\P{Is_Blk=:=- Combining DIACRITICAL_MARKS}');
Expect(1, 879, '\p{Is_Blk:	combiningdiacriticalmarks}', "");
Expect(0, 879, '\p{^Is_Blk:	combiningdiacriticalmarks}', "");
Expect(0, 879, '\P{Is_Blk:	combiningdiacriticalmarks}', "");
Expect(1, 879, '\P{^Is_Blk:	combiningdiacriticalmarks}', "");
Expect(0, 767, '\p{Is_Blk:	combiningdiacriticalmarks}', "");
Expect(1, 767, '\p{^Is_Blk:	combiningdiacriticalmarks}', "");
Expect(1, 767, '\P{Is_Blk:	combiningdiacriticalmarks}', "");
Expect(0, 767, '\P{^Is_Blk:	combiningdiacriticalmarks}', "");
Expect(1, 879, '\p{Is_Blk=CombiningDiacritical	Marks}', "");
Expect(0, 879, '\p{^Is_Blk=CombiningDiacritical	Marks}', "");
Expect(0, 879, '\P{Is_Blk=CombiningDiacritical	Marks}', "");
Expect(1, 879, '\P{^Is_Blk=CombiningDiacritical	Marks}', "");
Expect(0, 767, '\p{Is_Blk=CombiningDiacritical	Marks}', "");
Expect(1, 767, '\p{^Is_Blk=CombiningDiacritical	Marks}', "");
Expect(1, 767, '\P{Is_Blk=CombiningDiacritical	Marks}', "");
Expect(0, 767, '\P{^Is_Blk=CombiningDiacritical	Marks}', "");
Error('\p{Block: _LATIN/a/Extended b}');
Error('\P{Block: _LATIN/a/Extended b}');
Expect(1, 591, '\p{Block=latinextendedb}', "");
Expect(0, 591, '\p{^Block=latinextendedb}', "");
Expect(0, 591, '\P{Block=latinextendedb}', "");
Expect(1, 591, '\P{^Block=latinextendedb}', "");
Expect(0, 383, '\p{Block=latinextendedb}', "");
Expect(1, 383, '\p{^Block=latinextendedb}', "");
Expect(1, 383, '\P{Block=latinextendedb}', "");
Expect(0, 383, '\P{^Block=latinextendedb}', "");
Expect(1, 591, '\p{Block= 	Latin-EXTENDED_b}', "");
Expect(0, 591, '\p{^Block= 	Latin-EXTENDED_b}', "");
Expect(0, 591, '\P{Block= 	Latin-EXTENDED_b}', "");
Expect(1, 591, '\P{^Block= 	Latin-EXTENDED_b}', "");
Expect(0, 383, '\p{Block= 	Latin-EXTENDED_b}', "");
Expect(1, 383, '\p{^Block= 	Latin-EXTENDED_b}', "");
Expect(1, 383, '\P{Block= 	Latin-EXTENDED_b}', "");
Expect(0, 383, '\P{^Block= 	Latin-EXTENDED_b}', "");
Error('\p{Blk=- Latin_EXTENDED/a/B}');
Error('\P{Blk=- Latin_EXTENDED/a/B}');
Expect(1, 591, '\p{Blk=latinextendedb}', "");
Expect(0, 591, '\p{^Blk=latinextendedb}', "");
Expect(0, 591, '\P{Blk=latinextendedb}', "");
Expect(1, 591, '\P{^Blk=latinextendedb}', "");
Expect(0, 383, '\p{Blk=latinextendedb}', "");
Expect(1, 383, '\p{^Blk=latinextendedb}', "");
Expect(1, 383, '\P{Blk=latinextendedb}', "");
Expect(0, 383, '\P{^Blk=latinextendedb}', "");
Expect(1, 591, '\p{Blk:   LATIN_extended-B}', "");
Expect(0, 591, '\p{^Blk:   LATIN_extended-B}', "");
Expect(0, 591, '\P{Blk:   LATIN_extended-B}', "");
Expect(1, 591, '\P{^Blk:   LATIN_extended-B}', "");
Expect(0, 383, '\p{Blk:   LATIN_extended-B}', "");
Expect(1, 383, '\p{^Blk:   LATIN_extended-B}', "");
Expect(1, 383, '\P{Blk:   LATIN_extended-B}', "");
Expect(0, 383, '\P{^Blk:   LATIN_extended-B}', "");
Error('\p{Is_Block=_Latin/a/Extended-b}');
Error('\P{Is_Block=_Latin/a/Extended-b}');
Expect(1, 591, '\p{Is_Block=latinextendedb}', "");
Expect(0, 591, '\p{^Is_Block=latinextendedb}', "");
Expect(0, 591, '\P{Is_Block=latinextendedb}', "");
Expect(1, 591, '\P{^Is_Block=latinextendedb}', "");
Expect(0, 383, '\p{Is_Block=latinextendedb}', "");
Expect(1, 383, '\p{^Is_Block=latinextendedb}', "");
Expect(1, 383, '\P{Is_Block=latinextendedb}', "");
Expect(0, 383, '\P{^Is_Block=latinextendedb}', "");
Expect(1, 591, '\p{Is_Block:   Latin-extended_b}', "");
Expect(0, 591, '\p{^Is_Block:   Latin-extended_b}', "");
Expect(0, 591, '\P{Is_Block:   Latin-extended_b}', "");
Expect(1, 591, '\P{^Is_Block:   Latin-extended_b}', "");
Expect(0, 383, '\p{Is_Block:   Latin-extended_b}', "");
Expect(1, 383, '\p{^Is_Block:   Latin-extended_b}', "");
Expect(1, 383, '\P{Is_Block:   Latin-extended_b}', "");
Expect(0, 383, '\P{^Is_Block:   Latin-extended_b}', "");
Error('\p{Is_Blk=_:=latin_EXTENDEDB}');
Error('\P{Is_Blk=_:=latin_EXTENDEDB}');
Expect(1, 591, '\p{Is_Blk=latinextendedb}', "");
Expect(0, 591, '\p{^Is_Blk=latinextendedb}', "");
Expect(0, 591, '\P{Is_Blk=latinextendedb}', "");
Expect(1, 591, '\P{^Is_Blk=latinextendedb}', "");
Expect(0, 383, '\p{Is_Blk=latinextendedb}', "");
Expect(1, 383, '\p{^Is_Blk=latinextendedb}', "");
Expect(1, 383, '\P{Is_Blk=latinextendedb}', "");
Expect(0, 383, '\P{^Is_Blk=latinextendedb}', "");
Expect(1, 591, '\p{Is_Blk=	-latin	extendedb}', "");
Expect(0, 591, '\p{^Is_Blk=	-latin	extendedb}', "");
Expect(0, 591, '\P{Is_Blk=	-latin	extendedb}', "");
Expect(1, 591, '\P{^Is_Blk=	-latin	extendedb}', "");
Expect(0, 383, '\p{Is_Blk=	-latin	extendedb}', "");
Expect(1, 383, '\p{^Is_Blk=	-latin	extendedb}', "");
Expect(1, 383, '\P{Is_Blk=	-latin	extendedb}', "");
Expect(0, 383, '\P{^Is_Blk=	-latin	extendedb}', "");
Error('\p{Block=_playing/a/CARDS}');
Error('\P{Block=_playing/a/CARDS}');
Expect(1, 127231, '\p{Block=playingcards}', "");
Expect(0, 127231, '\p{^Block=playingcards}', "");
Expect(0, 127231, '\P{Block=playingcards}', "");
Expect(1, 127231, '\P{^Block=playingcards}', "");
Expect(0, 127135, '\p{Block=playingcards}', "");
Expect(1, 127135, '\p{^Block=playingcards}', "");
Expect(1, 127135, '\P{Block=playingcards}', "");
Expect(0, 127135, '\P{^Block=playingcards}', "");
Expect(1, 127231, '\p{Block= Playing-Cards}', "");
Expect(0, 127231, '\p{^Block= Playing-Cards}', "");
Expect(0, 127231, '\P{Block= Playing-Cards}', "");
Expect(1, 127231, '\P{^Block= Playing-Cards}', "");
Expect(0, 127135, '\p{Block= Playing-Cards}', "");
Expect(1, 127135, '\p{^Block= Playing-Cards}', "");
Expect(1, 127135, '\P{Block= Playing-Cards}', "");
Expect(0, 127135, '\P{^Block= Playing-Cards}', "");
Error('\p{Blk=/a/PLAYINGCARDS}');
Error('\P{Blk=/a/PLAYINGCARDS}');
Expect(1, 127231, '\p{Blk=playingcards}', "");
Expect(0, 127231, '\p{^Blk=playingcards}', "");
Expect(0, 127231, '\P{Blk=playingcards}', "");
Expect(1, 127231, '\P{^Blk=playingcards}', "");
Expect(0, 127135, '\p{Blk=playingcards}', "");
Expect(1, 127135, '\p{^Blk=playingcards}', "");
Expect(1, 127135, '\P{Blk=playingcards}', "");
Expect(0, 127135, '\P{^Blk=playingcards}', "");
Expect(1, 127231, '\p{Blk=	-Playing_Cards}', "");
Expect(0, 127231, '\p{^Blk=	-Playing_Cards}', "");
Expect(0, 127231, '\P{Blk=	-Playing_Cards}', "");
Expect(1, 127231, '\P{^Blk=	-Playing_Cards}', "");
Expect(0, 127135, '\p{Blk=	-Playing_Cards}', "");
Expect(1, 127135, '\p{^Blk=	-Playing_Cards}', "");
Expect(1, 127135, '\P{Blk=	-Playing_Cards}', "");
Expect(0, 127135, '\P{^Blk=	-Playing_Cards}', "");
Error('\p{Is_Block=:=PlayingCARDS}');
Error('\P{Is_Block=:=PlayingCARDS}');
Expect(1, 127231, '\p{Is_Block=playingcards}', "");
Expect(0, 127231, '\p{^Is_Block=playingcards}', "");
Expect(0, 127231, '\P{Is_Block=playingcards}', "");
Expect(1, 127231, '\P{^Is_Block=playingcards}', "");
Expect(0, 127135, '\p{Is_Block=playingcards}', "");
Expect(1, 127135, '\p{^Is_Block=playingcards}', "");
Expect(1, 127135, '\P{Is_Block=playingcards}', "");
Expect(0, 127135, '\P{^Is_Block=playingcards}', "");
Expect(1, 127231, '\p{Is_Block= PLAYING	CARDS}', "");
Expect(0, 127231, '\p{^Is_Block= PLAYING	CARDS}', "");
Expect(0, 127231, '\P{Is_Block= PLAYING	CARDS}', "");
Expect(1, 127231, '\P{^Is_Block= PLAYING	CARDS}', "");
Expect(0, 127135, '\p{Is_Block= PLAYING	CARDS}', "");
Expect(1, 127135, '\p{^Is_Block= PLAYING	CARDS}', "");
Expect(1, 127135, '\P{Is_Block= PLAYING	CARDS}', "");
Expect(0, 127135, '\P{^Is_Block= PLAYING	CARDS}', "");
Error('\p{Is_Blk= /a/Playing-Cards}');
Error('\P{Is_Blk= /a/Playing-Cards}');
Expect(1, 127231, '\p{Is_Blk=playingcards}', "");
Expect(0, 127231, '\p{^Is_Blk=playingcards}', "");
Expect(0, 127231, '\P{Is_Blk=playingcards}', "");
Expect(1, 127231, '\P{^Is_Blk=playingcards}', "");
Expect(0, 127135, '\p{Is_Blk=playingcards}', "");
Expect(1, 127135, '\p{^Is_Blk=playingcards}', "");
Expect(1, 127135, '\P{Is_Blk=playingcards}', "");
Expect(0, 127135, '\P{^Is_Blk=playingcards}', "");
Expect(1, 127231, '\p{Is_Blk=	_playing Cards}', "");
Expect(0, 127231, '\p{^Is_Blk=	_playing Cards}', "");
Expect(0, 127231, '\P{Is_Blk=	_playing Cards}', "");
Expect(1, 127231, '\P{^Is_Blk=	_playing Cards}', "");
Expect(0, 127135, '\p{Is_Blk=	_playing Cards}', "");
Expect(1, 127135, '\p{^Is_Blk=	_playing Cards}', "");
Expect(1, 127135, '\P{Is_Blk=	_playing Cards}', "");
Expect(0, 127135, '\P{^Is_Blk=	_playing Cards}', "");
Error('\p{Block=:= _supplementalArrowsA}');
Error('\P{Block=:= _supplementalArrowsA}');
Expect(1, 10239, '\p{Block=supplementalarrowsa}', "");
Expect(0, 10239, '\p{^Block=supplementalarrowsa}', "");
Expect(0, 10239, '\P{Block=supplementalarrowsa}', "");
Expect(1, 10239, '\P{^Block=supplementalarrowsa}', "");
Expect(0, 10223, '\p{Block=supplementalarrowsa}', "");
Expect(1, 10223, '\p{^Block=supplementalarrowsa}', "");
Expect(1, 10223, '\P{Block=supplementalarrowsa}', "");
Expect(0, 10223, '\P{^Block=supplementalarrowsa}', "");
Expect(1, 10239, '\p{Block=	SupplementalArrows_A}', "");
Expect(0, 10239, '\p{^Block=	SupplementalArrows_A}', "");
Expect(0, 10239, '\P{Block=	SupplementalArrows_A}', "");
Expect(1, 10239, '\P{^Block=	SupplementalArrows_A}', "");
Expect(0, 10223, '\p{Block=	SupplementalArrows_A}', "");
Expect(1, 10223, '\p{^Block=	SupplementalArrows_A}', "");
Expect(1, 10223, '\P{Block=	SupplementalArrows_A}', "");
Expect(0, 10223, '\P{^Block=	SupplementalArrows_A}', "");
Error('\p{Blk=/a/__SUPPLEMENTAL	Arrows	A}');
Error('\P{Blk=/a/__SUPPLEMENTAL	Arrows	A}');
Expect(1, 10239, '\p{Blk=supplementalarrowsa}', "");
Expect(0, 10239, '\p{^Blk=supplementalarrowsa}', "");
Expect(0, 10239, '\P{Blk=supplementalarrowsa}', "");
Expect(1, 10239, '\P{^Blk=supplementalarrowsa}', "");
Expect(0, 10223, '\p{Blk=supplementalarrowsa}', "");
Expect(1, 10223, '\p{^Blk=supplementalarrowsa}', "");
Expect(1, 10223, '\P{Blk=supplementalarrowsa}', "");
Expect(0, 10223, '\P{^Blk=supplementalarrowsa}', "");
Expect(1, 10239, '\p{Blk=-Supplemental arrows A}', "");
Expect(0, 10239, '\p{^Blk=-Supplemental arrows A}', "");
Expect(0, 10239, '\P{Blk=-Supplemental arrows A}', "");
Expect(1, 10239, '\P{^Blk=-Supplemental arrows A}', "");
Expect(0, 10223, '\p{Blk=-Supplemental arrows A}', "");
Expect(1, 10223, '\p{^Blk=-Supplemental arrows A}', "");
Expect(1, 10223, '\P{Blk=-Supplemental arrows A}', "");
Expect(0, 10223, '\P{^Blk=-Supplemental arrows A}', "");
Error('\p{Is_Block=/a/ SUPPLEMENTAL_arrows-A}');
Error('\P{Is_Block=/a/ SUPPLEMENTAL_arrows-A}');
Expect(1, 10239, '\p{Is_Block=supplementalarrowsa}', "");
Expect(0, 10239, '\p{^Is_Block=supplementalarrowsa}', "");
Expect(0, 10239, '\P{Is_Block=supplementalarrowsa}', "");
Expect(1, 10239, '\P{^Is_Block=supplementalarrowsa}', "");
Expect(0, 10223, '\p{Is_Block=supplementalarrowsa}', "");
Expect(1, 10223, '\p{^Is_Block=supplementalarrowsa}', "");
Expect(1, 10223, '\P{Is_Block=supplementalarrowsa}', "");
Expect(0, 10223, '\P{^Is_Block=supplementalarrowsa}', "");
Expect(1, 10239, '\p{Is_Block= _Supplemental Arrows-A}', "");
Expect(0, 10239, '\p{^Is_Block= _Supplemental Arrows-A}', "");
Expect(0, 10239, '\P{Is_Block= _Supplemental Arrows-A}', "");
Expect(1, 10239, '\P{^Is_Block= _Supplemental Arrows-A}', "");
Expect(0, 10223, '\p{Is_Block= _Supplemental Arrows-A}', "");
Expect(1, 10223, '\p{^Is_Block= _Supplemental Arrows-A}', "");
Expect(1, 10223, '\P{Is_Block= _Supplemental Arrows-A}', "");
Expect(0, 10223, '\P{^Is_Block= _Supplemental Arrows-A}', "");
Error('\p{Is_Blk=--supplemental	Arrows:=A}');
Error('\P{Is_Blk=--supplemental	Arrows:=A}');
Expect(1, 10239, '\p{Is_Blk=supplementalarrowsa}', "");
Expect(0, 10239, '\p{^Is_Blk=supplementalarrowsa}', "");
Expect(0, 10239, '\P{Is_Blk=supplementalarrowsa}', "");
Expect(1, 10239, '\P{^Is_Blk=supplementalarrowsa}', "");
Expect(0, 10223, '\p{Is_Blk=supplementalarrowsa}', "");
Expect(1, 10223, '\p{^Is_Blk=supplementalarrowsa}', "");
Expect(1, 10223, '\P{Is_Blk=supplementalarrowsa}', "");
Expect(0, 10223, '\P{^Is_Blk=supplementalarrowsa}', "");
Expect(1, 10239, '\p{Is_Blk=  SUPPLEMENTALarrows-A}', "");
Expect(0, 10239, '\p{^Is_Blk=  SUPPLEMENTALarrows-A}', "");
Expect(0, 10239, '\P{Is_Blk=  SUPPLEMENTALarrows-A}', "");
Expect(1, 10239, '\P{^Is_Blk=  SUPPLEMENTALarrows-A}', "");
Expect(0, 10223, '\p{Is_Blk=  SUPPLEMENTALarrows-A}', "");
Expect(1, 10223, '\p{^Is_Blk=  SUPPLEMENTALarrows-A}', "");
Expect(1, 10223, '\P{Is_Blk=  SUPPLEMENTALarrows-A}', "");
Expect(0, 10223, '\P{^Is_Blk=  SUPPLEMENTALarrows-A}', "");
Error('\p{Block=/a/-	Javanese}');
Error('\P{Block=/a/-	Javanese}');
Expect(1, 43487, '\p{Block=javanese}', "");
Expect(0, 43487, '\p{^Block=javanese}', "");
Expect(0, 43487, '\P{Block=javanese}', "");
Expect(1, 43487, '\P{^Block=javanese}', "");
Expect(0, 43391, '\p{Block=javanese}', "");
Expect(1, 43391, '\p{^Block=javanese}', "");
Expect(1, 43391, '\P{Block=javanese}', "");
Expect(0, 43391, '\P{^Block=javanese}', "");
Expect(1, 43487, '\p{Block=-JAVANESE}', "");
Expect(0, 43487, '\p{^Block=-JAVANESE}', "");
Expect(0, 43487, '\P{Block=-JAVANESE}', "");
Expect(1, 43487, '\P{^Block=-JAVANESE}', "");
Expect(0, 43391, '\p{Block=-JAVANESE}', "");
Expect(1, 43391, '\p{^Block=-JAVANESE}', "");
Expect(1, 43391, '\P{Block=-JAVANESE}', "");
Expect(0, 43391, '\P{^Block=-JAVANESE}', "");
Error('\p{Blk=/a/-	JAVANESE}');
Error('\P{Blk=/a/-	JAVANESE}');
Expect(1, 43487, '\p{Blk=javanese}', "");
Expect(0, 43487, '\p{^Blk=javanese}', "");
Expect(0, 43487, '\P{Blk=javanese}', "");
Expect(1, 43487, '\P{^Blk=javanese}', "");
Expect(0, 43391, '\p{Blk=javanese}', "");
Expect(1, 43391, '\p{^Blk=javanese}', "");
Expect(1, 43391, '\P{Blk=javanese}', "");
Expect(0, 43391, '\P{^Blk=javanese}', "");
Expect(1, 43487, '\p{Blk=-	JAVANESE}', "");
Expect(0, 43487, '\p{^Blk=-	JAVANESE}', "");
Expect(0, 43487, '\P{Blk=-	JAVANESE}', "");
Expect(1, 43487, '\P{^Blk=-	JAVANESE}', "");
Expect(0, 43391, '\p{Blk=-	JAVANESE}', "");
Expect(1, 43391, '\p{^Blk=-	JAVANESE}', "");
Expect(1, 43391, '\P{Blk=-	JAVANESE}', "");
Expect(0, 43391, '\P{^Blk=-	JAVANESE}', "");
Error('\p{Is_Block=-:=javanese}');
Error('\P{Is_Block=-:=javanese}');
Expect(1, 43487, '\p{Is_Block=javanese}', "");
Expect(0, 43487, '\p{^Is_Block=javanese}', "");
Expect(0, 43487, '\P{Is_Block=javanese}', "");
Expect(1, 43487, '\P{^Is_Block=javanese}', "");
Expect(0, 43391, '\p{Is_Block=javanese}', "");
Expect(1, 43391, '\p{^Is_Block=javanese}', "");
Expect(1, 43391, '\P{Is_Block=javanese}', "");
Expect(0, 43391, '\P{^Is_Block=javanese}', "");
Expect(1, 43487, '\p{Is_Block=_javanese}', "");
Expect(0, 43487, '\p{^Is_Block=_javanese}', "");
Expect(0, 43487, '\P{Is_Block=_javanese}', "");
Expect(1, 43487, '\P{^Is_Block=_javanese}', "");
Expect(0, 43391, '\p{Is_Block=_javanese}', "");
Expect(1, 43391, '\p{^Is_Block=_javanese}', "");
Expect(1, 43391, '\P{Is_Block=_javanese}', "");
Expect(0, 43391, '\P{^Is_Block=_javanese}', "");
Error('\p{Is_Blk=	JAVANESE:=}');
Error('\P{Is_Blk=	JAVANESE:=}');
Expect(1, 43487, '\p{Is_Blk=javanese}', "");
Expect(0, 43487, '\p{^Is_Blk=javanese}', "");
Expect(0, 43487, '\P{Is_Blk=javanese}', "");
Expect(1, 43487, '\P{^Is_Blk=javanese}', "");
Expect(0, 43391, '\p{Is_Blk=javanese}', "");
Expect(1, 43391, '\p{^Is_Blk=javanese}', "");
Expect(1, 43391, '\P{Is_Blk=javanese}', "");
Expect(0, 43391, '\P{^Is_Blk=javanese}', "");
Expect(1, 43487, '\p{Is_Blk:  Javanese}', "");
Expect(0, 43487, '\p{^Is_Blk:  Javanese}', "");
Expect(0, 43487, '\P{Is_Blk:  Javanese}', "");
Expect(1, 43487, '\P{^Is_Blk:  Javanese}', "");
Expect(0, 43391, '\p{Is_Blk:  Javanese}', "");
Expect(1, 43391, '\p{^Is_Blk:  Javanese}', "");
Expect(1, 43391, '\P{Is_Blk:  Javanese}', "");
Expect(0, 43391, '\P{^Is_Blk:  Javanese}', "");
Error('\p{Block= :=BRAHMI}');
Error('\P{Block= :=BRAHMI}');
Expect(1, 69759, '\p{Block=brahmi}', "");
Expect(0, 69759, '\p{^Block=brahmi}', "");
Expect(0, 69759, '\P{Block=brahmi}', "");
Expect(1, 69759, '\P{^Block=brahmi}', "");
Expect(0, 69631, '\p{Block=brahmi}', "");
Expect(1, 69631, '\p{^Block=brahmi}', "");
Expect(1, 69631, '\P{Block=brahmi}', "");
Expect(0, 69631, '\P{^Block=brahmi}', "");
Expect(1, 69759, '\p{Block=-BRAHMI}', "");
Expect(0, 69759, '\p{^Block=-BRAHMI}', "");
Expect(0, 69759, '\P{Block=-BRAHMI}', "");
Expect(1, 69759, '\P{^Block=-BRAHMI}', "");
Expect(0, 69631, '\p{Block=-BRAHMI}', "");
Expect(1, 69631, '\p{^Block=-BRAHMI}', "");
Expect(1, 69631, '\P{Block=-BRAHMI}', "");
Expect(0, 69631, '\P{^Block=-BRAHMI}', "");
Error('\p{Blk=/a/__BRAHMI}');
Error('\P{Blk=/a/__BRAHMI}');
Expect(1, 69759, '\p{Blk=brahmi}', "");
Expect(0, 69759, '\p{^Blk=brahmi}', "");
Expect(0, 69759, '\P{Blk=brahmi}', "");
Expect(1, 69759, '\P{^Blk=brahmi}', "");
Expect(0, 69631, '\p{Blk=brahmi}', "");
Expect(1, 69631, '\p{^Blk=brahmi}', "");
Expect(1, 69631, '\P{Blk=brahmi}', "");
Expect(0, 69631, '\P{^Blk=brahmi}', "");
Expect(1, 69759, '\p{Blk=--Brahmi}', "");
Expect(0, 69759, '\p{^Blk=--Brahmi}', "");
Expect(0, 69759, '\P{Blk=--Brahmi}', "");
Expect(1, 69759, '\P{^Blk=--Brahmi}', "");
Expect(0, 69631, '\p{Blk=--Brahmi}', "");
Expect(1, 69631, '\p{^Blk=--Brahmi}', "");
Expect(1, 69631, '\P{Blk=--Brahmi}', "");
Expect(0, 69631, '\P{^Blk=--Brahmi}', "");
Error('\p{Is_Block=--Brahmi/a/}');
Error('\P{Is_Block=--Brahmi/a/}');
Expect(1, 69759, '\p{Is_Block=brahmi}', "");
Expect(0, 69759, '\p{^Is_Block=brahmi}', "");
Expect(0, 69759, '\P{Is_Block=brahmi}', "");
Expect(1, 69759, '\P{^Is_Block=brahmi}', "");
Expect(0, 69631, '\p{Is_Block=brahmi}', "");
Expect(1, 69631, '\p{^Is_Block=brahmi}', "");
Expect(1, 69631, '\P{Is_Block=brahmi}', "");
Expect(0, 69631, '\P{^Is_Block=brahmi}', "");
Error('\p{Is_Blk= Brahmi:=}');
Error('\P{Is_Blk= Brahmi:=}');
Expect(1, 69759, '\p{Is_Blk=brahmi}', "");
Expect(0, 69759, '\p{^Is_Blk=brahmi}', "");
Expect(0, 69759, '\P{Is_Blk=brahmi}', "");
Expect(1, 69759, '\P{^Is_Blk=brahmi}', "");
Expect(0, 69631, '\p{Is_Blk=brahmi}', "");
Expect(1, 69631, '\p{^Is_Blk=brahmi}', "");
Expect(1, 69631, '\P{Is_Blk=brahmi}', "");
Expect(0, 69631, '\P{^Is_Blk=brahmi}', "");
Expect(1, 69759, '\p{Is_Blk= _Brahmi}', "");
Expect(0, 69759, '\p{^Is_Blk= _Brahmi}', "");
Expect(0, 69759, '\P{Is_Blk= _Brahmi}', "");
Expect(1, 69759, '\P{^Is_Blk= _Brahmi}', "");
Expect(0, 69631, '\p{Is_Blk= _Brahmi}', "");
Expect(1, 69631, '\p{^Is_Blk= _Brahmi}', "");
Expect(1, 69631, '\P{Is_Blk= _Brahmi}', "");
Expect(0, 69631, '\P{^Is_Blk= _Brahmi}', "");
Error('\p{Block= SupplementaryPrivate/a/USEArea	a}');
Error('\P{Block= SupplementaryPrivate/a/USEArea	a}');
Expect(1, 983040, '\p{Block=supplementaryprivateuseareaa}', "");
Expect(0, 983040, '\p{^Block=supplementaryprivateuseareaa}', "");
Expect(0, 983040, '\P{Block=supplementaryprivateuseareaa}', "");
Expect(1, 983040, '\P{^Block=supplementaryprivateuseareaa}', "");
Expect(0, 1, '\p{Block=supplementaryprivateuseareaa}', "");
Expect(1, 1, '\p{^Block=supplementaryprivateuseareaa}', "");
Expect(1, 1, '\P{Block=supplementaryprivateuseareaa}', "");
Expect(0, 1, '\P{^Block=supplementaryprivateuseareaa}', "");
Expect(1, 983040, '\p{Block=__SUPPLEMENTARYPRIVATE_Use area-A}', "");
Expect(0, 983040, '\p{^Block=__SUPPLEMENTARYPRIVATE_Use area-A}', "");
Expect(0, 983040, '\P{Block=__SUPPLEMENTARYPRIVATE_Use area-A}', "");
Expect(1, 983040, '\P{^Block=__SUPPLEMENTARYPRIVATE_Use area-A}', "");
Expect(0, 1, '\p{Block=__SUPPLEMENTARYPRIVATE_Use area-A}', "");
Expect(1, 1, '\p{^Block=__SUPPLEMENTARYPRIVATE_Use area-A}', "");
Expect(1, 1, '\P{Block=__SUPPLEMENTARYPRIVATE_Use area-A}', "");
Expect(0, 1, '\P{^Block=__SUPPLEMENTARYPRIVATE_Use area-A}', "");
Error('\p{Blk= 	SUPPLEMENTARY Private:=USE_Area	A}');
Error('\P{Blk= 	SUPPLEMENTARY Private:=USE_Area	A}');
Expect(1, 983040, '\p{Blk=supplementaryprivateuseareaa}', "");
Expect(0, 983040, '\p{^Blk=supplementaryprivateuseareaa}', "");
Expect(0, 983040, '\P{Blk=supplementaryprivateuseareaa}', "");
Expect(1, 983040, '\P{^Blk=supplementaryprivateuseareaa}', "");
Expect(0, 1, '\p{Blk=supplementaryprivateuseareaa}', "");
Expect(1, 1, '\p{^Blk=supplementaryprivateuseareaa}', "");
Expect(1, 1, '\P{Blk=supplementaryprivateuseareaa}', "");
Expect(0, 1, '\P{^Blk=supplementaryprivateuseareaa}', "");
Expect(1, 983040, '\p{Blk=	-supplementary_PRIVATE_Use Area_A}', "");
Expect(0, 983040, '\p{^Blk=	-supplementary_PRIVATE_Use Area_A}', "");
Expect(0, 983040, '\P{Blk=	-supplementary_PRIVATE_Use Area_A}', "");
Expect(1, 983040, '\P{^Blk=	-supplementary_PRIVATE_Use Area_A}', "");
Expect(0, 1, '\p{Blk=	-supplementary_PRIVATE_Use Area_A}', "");
Expect(1, 1, '\p{^Blk=	-supplementary_PRIVATE_Use Area_A}', "");
Expect(1, 1, '\P{Blk=	-supplementary_PRIVATE_Use Area_A}', "");
Expect(0, 1, '\P{^Blk=	-supplementary_PRIVATE_Use Area_A}', "");
Error('\p{Is_Block= 	Supplementary	Private Use-area	A:=}');
Error('\P{Is_Block= 	Supplementary	Private Use-area	A:=}');
Expect(1, 983040, '\p{Is_Block=supplementaryprivateuseareaa}', "");
Expect(0, 983040, '\p{^Is_Block=supplementaryprivateuseareaa}', "");
Expect(0, 983040, '\P{Is_Block=supplementaryprivateuseareaa}', "");
Expect(1, 983040, '\P{^Is_Block=supplementaryprivateuseareaa}', "");
Expect(0, 1, '\p{Is_Block=supplementaryprivateuseareaa}', "");
Expect(1, 1, '\p{^Is_Block=supplementaryprivateuseareaa}', "");
Expect(1, 1, '\P{Is_Block=supplementaryprivateuseareaa}', "");
Expect(0, 1, '\P{^Is_Block=supplementaryprivateuseareaa}', "");
Expect(1, 983040, '\p{Is_Block=	 SUPPLEMENTARY_PRIVATE-Use_Area_A}', "");
Expect(0, 983040, '\p{^Is_Block=	 SUPPLEMENTARY_PRIVATE-Use_Area_A}', "");
Expect(0, 983040, '\P{Is_Block=	 SUPPLEMENTARY_PRIVATE-Use_Area_A}', "");
Expect(1, 983040, '\P{^Is_Block=	 SUPPLEMENTARY_PRIVATE-Use_Area_A}', "");
Expect(0, 1, '\p{Is_Block=	 SUPPLEMENTARY_PRIVATE-Use_Area_A}', "");
Expect(1, 1, '\p{^Is_Block=	 SUPPLEMENTARY_PRIVATE-Use_Area_A}', "");
Expect(1, 1, '\P{Is_Block=	 SUPPLEMENTARY_PRIVATE-Use_Area_A}', "");
Expect(0, 1, '\P{^Is_Block=	 SUPPLEMENTARY_PRIVATE-Use_Area_A}', "");
Error('\p{Is_Blk= SUPPLEMENTARY-Private usearea-A/a/}');
Error('\P{Is_Blk= SUPPLEMENTARY-Private usearea-A/a/}');
Expect(1, 983040, '\p{Is_Blk=supplementaryprivateuseareaa}', "");
Expect(0, 983040, '\p{^Is_Blk=supplementaryprivateuseareaa}', "");
Expect(0, 983040, '\P{Is_Blk=supplementaryprivateuseareaa}', "");
Expect(1, 983040, '\P{^Is_Blk=supplementaryprivateuseareaa}', "");
Expect(0, 1, '\p{Is_Blk=supplementaryprivateuseareaa}', "");
Expect(1, 1, '\p{^Is_Blk=supplementaryprivateuseareaa}', "");
Expect(1, 1, '\P{Is_Blk=supplementaryprivateuseareaa}', "");
Expect(0, 1, '\P{^Is_Blk=supplementaryprivateuseareaa}', "");
Expect(1, 983040, '\p{Is_Blk=  supplementary	private	use-Area-A}', "");
Expect(0, 983040, '\p{^Is_Blk=  supplementary	private	use-Area-A}', "");
Expect(0, 983040, '\P{Is_Blk=  supplementary	private	use-Area-A}', "");
Expect(1, 983040, '\P{^Is_Blk=  supplementary	private	use-Area-A}', "");
Expect(0, 1, '\p{Is_Blk=  supplementary	private	use-Area-A}', "");
Expect(1, 1, '\p{^Is_Blk=  supplementary	private	use-Area-A}', "");
Expect(1, 1, '\P{Is_Blk=  supplementary	private	use-Area-A}', "");
Expect(0, 1, '\P{^Is_Blk=  supplementary	private	use-Area-A}', "");
Error('\p{Block=/a/HANUNOO}');
Error('\P{Block=/a/HANUNOO}');
Expect(1, 5951, '\p{Block=hanunoo}', "");
Expect(0, 5951, '\p{^Block=hanunoo}', "");
Expect(0, 5951, '\P{Block=hanunoo}', "");
Expect(1, 5951, '\P{^Block=hanunoo}', "");
Expect(0, 5919, '\p{Block=hanunoo}', "");
Expect(1, 5919, '\p{^Block=hanunoo}', "");
Expect(1, 5919, '\P{Block=hanunoo}', "");
Expect(0, 5919, '\P{^Block=hanunoo}', "");
Expect(1, 5951, '\p{Block= Hanunoo}', "");
Expect(0, 5951, '\p{^Block= Hanunoo}', "");
Expect(0, 5951, '\P{Block= Hanunoo}', "");
Expect(1, 5951, '\P{^Block= Hanunoo}', "");
Expect(0, 5919, '\p{Block= Hanunoo}', "");
Expect(1, 5919, '\p{^Block= Hanunoo}', "");
Expect(1, 5919, '\P{Block= Hanunoo}', "");
Expect(0, 5919, '\P{^Block= Hanunoo}', "");
Error('\p{Blk=:=hanunoo}');
Error('\P{Blk=:=hanunoo}');
Expect(1, 5951, '\p{Blk=hanunoo}', "");
Expect(0, 5951, '\p{^Blk=hanunoo}', "");
Expect(0, 5951, '\P{Blk=hanunoo}', "");
Expect(1, 5951, '\P{^Blk=hanunoo}', "");
Expect(0, 5919, '\p{Blk=hanunoo}', "");
Expect(1, 5919, '\p{^Blk=hanunoo}', "");
Expect(1, 5919, '\P{Blk=hanunoo}', "");
Expect(0, 5919, '\P{^Blk=hanunoo}', "");
Expect(1, 5951, '\p{Blk=- hanunoo}', "");
Expect(0, 5951, '\p{^Blk=- hanunoo}', "");
Expect(0, 5951, '\P{Blk=- hanunoo}', "");
Expect(1, 5951, '\P{^Blk=- hanunoo}', "");
Expect(0, 5919, '\p{Blk=- hanunoo}', "");
Expect(1, 5919, '\p{^Blk=- hanunoo}', "");
Expect(1, 5919, '\P{Blk=- hanunoo}', "");
Expect(0, 5919, '\P{^Blk=- hanunoo}', "");
Error('\p{Is_Block=	:=Hanunoo}');
Error('\P{Is_Block=	:=Hanunoo}');
Expect(1, 5951, '\p{Is_Block=hanunoo}', "");
Expect(0, 5951, '\p{^Is_Block=hanunoo}', "");
Expect(0, 5951, '\P{Is_Block=hanunoo}', "");
Expect(1, 5951, '\P{^Is_Block=hanunoo}', "");
Expect(0, 5919, '\p{Is_Block=hanunoo}', "");
Expect(1, 5919, '\p{^Is_Block=hanunoo}', "");
Expect(1, 5919, '\P{Is_Block=hanunoo}', "");
Expect(0, 5919, '\P{^Is_Block=hanunoo}', "");
Expect(1, 5951, '\p{Is_Block=_	Hanunoo}', "");
Expect(0, 5951, '\p{^Is_Block=_	Hanunoo}', "");
Expect(0, 5951, '\P{Is_Block=_	Hanunoo}', "");
Expect(1, 5951, '\P{^Is_Block=_	Hanunoo}', "");
Expect(0, 5919, '\p{Is_Block=_	Hanunoo}', "");
Expect(1, 5919, '\p{^Is_Block=_	Hanunoo}', "");
Expect(1, 5919, '\P{Is_Block=_	Hanunoo}', "");
Expect(0, 5919, '\P{^Is_Block=_	Hanunoo}', "");
Error('\p{Is_Blk=- HANUNOO/a/}');
Error('\P{Is_Blk=- HANUNOO/a/}');
Expect(1, 5951, '\p{Is_Blk:hanunoo}', "");
Expect(0, 5951, '\p{^Is_Blk:hanunoo}', "");
Expect(0, 5951, '\P{Is_Blk:hanunoo}', "");
Expect(1, 5951, '\P{^Is_Blk:hanunoo}', "");
Expect(0, 5919, '\p{Is_Blk:hanunoo}', "");
Expect(1, 5919, '\p{^Is_Blk:hanunoo}', "");
Expect(1, 5919, '\P{Is_Blk:hanunoo}', "");
Expect(0, 5919, '\P{^Is_Blk:hanunoo}', "");
Expect(1, 5951, '\p{Is_Blk=-HANUNOO}', "");
Expect(0, 5951, '\p{^Is_Blk=-HANUNOO}', "");
Expect(0, 5951, '\P{Is_Blk=-HANUNOO}', "");
Expect(1, 5951, '\P{^Is_Blk=-HANUNOO}', "");
Expect(0, 5919, '\p{Is_Blk=-HANUNOO}', "");
Expect(1, 5919, '\p{^Is_Blk=-HANUNOO}', "");
Expect(1, 5919, '\P{Is_Blk=-HANUNOO}', "");
Expect(0, 5919, '\P{^Is_Blk=-HANUNOO}', "");
Error('\p{Block:	 -Low:=surrogates}');
Error('\P{Block:	 -Low:=surrogates}');
Expect(1, 57343, '\p{Block=lowsurrogates}', "");
Expect(0, 57343, '\p{^Block=lowsurrogates}', "");
Expect(0, 57343, '\P{Block=lowsurrogates}', "");
Expect(1, 57343, '\P{^Block=lowsurrogates}', "");
Expect(0, 1, '\p{Block=lowsurrogates}', "");
Expect(1, 1, '\p{^Block=lowsurrogates}', "");
Expect(1, 1, '\P{Block=lowsurrogates}', "");
Expect(0, 1, '\P{^Block=lowsurrogates}', "");
Expect(1, 57343, '\p{Block=_	LOW	Surrogates}', "");
Expect(0, 57343, '\p{^Block=_	LOW	Surrogates}', "");
Expect(0, 57343, '\P{Block=_	LOW	Surrogates}', "");
Expect(1, 57343, '\P{^Block=_	LOW	Surrogates}', "");
Expect(0, 1, '\p{Block=_	LOW	Surrogates}', "");
Expect(1, 1, '\p{^Block=_	LOW	Surrogates}', "");
Expect(1, 1, '\P{Block=_	LOW	Surrogates}', "");
Expect(0, 1, '\P{^Block=_	LOW	Surrogates}', "");
Error('\p{Blk=:=-_Low	SURROGATES}');
Error('\P{Blk=:=-_Low	SURROGATES}');
Expect(1, 57343, '\p{Blk=lowsurrogates}', "");
Expect(0, 57343, '\p{^Blk=lowsurrogates}', "");
Expect(0, 57343, '\P{Blk=lowsurrogates}', "");
Expect(1, 57343, '\P{^Blk=lowsurrogates}', "");
Expect(0, 1, '\p{Blk=lowsurrogates}', "");
Expect(1, 1, '\p{^Blk=lowsurrogates}', "");
Expect(1, 1, '\P{Blk=lowsurrogates}', "");
Expect(0, 1, '\P{^Blk=lowsurrogates}', "");
Expect(1, 57343, '\p{Blk=		Low-Surrogates}', "");
Expect(0, 57343, '\p{^Blk=		Low-Surrogates}', "");
Expect(0, 57343, '\P{Blk=		Low-Surrogates}', "");
Expect(1, 57343, '\P{^Blk=		Low-Surrogates}', "");
Expect(0, 1, '\p{Blk=		Low-Surrogates}', "");
Expect(1, 1, '\p{^Blk=		Low-Surrogates}', "");
Expect(1, 1, '\P{Blk=		Low-Surrogates}', "");
Expect(0, 1, '\P{^Blk=		Low-Surrogates}', "");
Error('\p{Is_Block=:= -LowSURROGATES}');
Error('\P{Is_Block=:= -LowSURROGATES}');
Expect(1, 57343, '\p{Is_Block:   lowsurrogates}', "");
Expect(0, 57343, '\p{^Is_Block:   lowsurrogates}', "");
Expect(0, 57343, '\P{Is_Block:   lowsurrogates}', "");
Expect(1, 57343, '\P{^Is_Block:   lowsurrogates}', "");
Expect(0, 1, '\p{Is_Block:   lowsurrogates}', "");
Expect(1, 1, '\p{^Is_Block:   lowsurrogates}', "");
Expect(1, 1, '\P{Is_Block:   lowsurrogates}', "");
Expect(0, 1, '\P{^Is_Block:   lowsurrogates}', "");
Expect(1, 57343, '\p{Is_Block= 	low	Surrogates}', "");
Expect(0, 57343, '\p{^Is_Block= 	low	Surrogates}', "");
Expect(0, 57343, '\P{Is_Block= 	low	Surrogates}', "");
Expect(1, 57343, '\P{^Is_Block= 	low	Surrogates}', "");
Expect(0, 1, '\p{Is_Block= 	low	Surrogates}', "");
Expect(1, 1, '\p{^Is_Block= 	low	Surrogates}', "");
Expect(1, 1, '\P{Is_Block= 	low	Surrogates}', "");
Expect(0, 1, '\P{^Is_Block= 	low	Surrogates}', "");
Error('\p{Is_Blk=-_Low:=Surrogates}');
Error('\P{Is_Blk=-_Low:=Surrogates}');
Expect(1, 57343, '\p{Is_Blk=lowsurrogates}', "");
Expect(0, 57343, '\p{^Is_Blk=lowsurrogates}', "");
Expect(0, 57343, '\P{Is_Blk=lowsurrogates}', "");
Expect(1, 57343, '\P{^Is_Blk=lowsurrogates}', "");
Expect(0, 1, '\p{Is_Blk=lowsurrogates}', "");
Expect(1, 1, '\p{^Is_Blk=lowsurrogates}', "");
Expect(1, 1, '\P{Is_Blk=lowsurrogates}', "");
Expect(0, 1, '\P{^Is_Blk=lowsurrogates}', "");
Expect(1, 57343, '\p{Is_Blk=Low	Surrogates}', "");
Expect(0, 57343, '\p{^Is_Blk=Low	Surrogates}', "");
Expect(0, 57343, '\P{Is_Blk=Low	Surrogates}', "");
Expect(1, 57343, '\P{^Is_Blk=Low	Surrogates}', "");
Expect(0, 1, '\p{Is_Blk=Low	Surrogates}', "");
Expect(1, 1, '\p{^Is_Blk=Low	Surrogates}', "");
Expect(1, 1, '\P{Is_Blk=Low	Surrogates}', "");
Expect(0, 1, '\P{^Is_Blk=Low	Surrogates}', "");
Error('\p{Block=/a/	_Katakana}');
Error('\P{Block=/a/	_Katakana}');
Expect(1, 12543, '\p{Block=katakana}', "");
Expect(0, 12543, '\p{^Block=katakana}', "");
Expect(0, 12543, '\P{Block=katakana}', "");
Expect(1, 12543, '\P{^Block=katakana}', "");
Expect(0, 12447, '\p{Block=katakana}', "");
Expect(1, 12447, '\p{^Block=katakana}', "");
Expect(1, 12447, '\P{Block=katakana}', "");
Expect(0, 12447, '\P{^Block=katakana}', "");
Expect(1, 12543, '\p{Block= katakana}', "");
Expect(0, 12543, '\p{^Block= katakana}', "");
Expect(0, 12543, '\P{Block= katakana}', "");
Expect(1, 12543, '\P{^Block= katakana}', "");
Expect(0, 12447, '\p{Block= katakana}', "");
Expect(1, 12447, '\p{^Block= katakana}', "");
Expect(1, 12447, '\P{Block= katakana}', "");
Expect(0, 12447, '\P{^Block= katakana}', "");
Error('\p{Blk=_:=katakana}');
Error('\P{Blk=_:=katakana}');
Expect(1, 12543, '\p{Blk=katakana}', "");
Expect(0, 12543, '\p{^Blk=katakana}', "");
Expect(0, 12543, '\P{Blk=katakana}', "");
Expect(1, 12543, '\P{^Blk=katakana}', "");
Expect(0, 12447, '\p{Blk=katakana}', "");
Expect(1, 12447, '\p{^Blk=katakana}', "");
Expect(1, 12447, '\P{Blk=katakana}', "");
Expect(0, 12447, '\P{^Blk=katakana}', "");
Expect(1, 12543, '\p{Blk:   --Katakana}', "");
Expect(0, 12543, '\p{^Blk:   --Katakana}', "");
Expect(0, 12543, '\P{Blk:   --Katakana}', "");
Expect(1, 12543, '\P{^Blk:   --Katakana}', "");
Expect(0, 12447, '\p{Blk:   --Katakana}', "");
Expect(1, 12447, '\p{^Blk:   --Katakana}', "");
Expect(1, 12447, '\P{Blk:   --Katakana}', "");
Expect(0, 12447, '\P{^Blk:   --Katakana}', "");
Error('\p{Is_Block:	:=-	KATAKANA}');
Error('\P{Is_Block:	:=-	KATAKANA}');
Expect(1, 12543, '\p{Is_Block=katakana}', "");
Expect(0, 12543, '\p{^Is_Block=katakana}', "");
Expect(0, 12543, '\P{Is_Block=katakana}', "");
Expect(1, 12543, '\P{^Is_Block=katakana}', "");
Expect(0, 12447, '\p{Is_Block=katakana}', "");
Expect(1, 12447, '\p{^Is_Block=katakana}', "");
Expect(1, 12447, '\P{Is_Block=katakana}', "");
Expect(0, 12447, '\P{^Is_Block=katakana}', "");
Expect(1, 12543, '\p{Is_Block:   __Katakana}', "");
Expect(0, 12543, '\p{^Is_Block:   __Katakana}', "");
Expect(0, 12543, '\P{Is_Block:   __Katakana}', "");
Expect(1, 12543, '\P{^Is_Block:   __Katakana}', "");
Expect(0, 12447, '\p{Is_Block:   __Katakana}', "");
Expect(1, 12447, '\p{^Is_Block:   __Katakana}', "");
Expect(1, 12447, '\P{Is_Block:   __Katakana}', "");
Expect(0, 12447, '\P{^Is_Block:   __Katakana}', "");
Error('\p{Is_Blk=_:=katakana}');
Error('\P{Is_Blk=_:=katakana}');
Expect(1, 12543, '\p{Is_Blk=katakana}', "");
Expect(0, 12543, '\p{^Is_Blk=katakana}', "");
Expect(0, 12543, '\P{Is_Blk=katakana}', "");
Expect(1, 12543, '\P{^Is_Blk=katakana}', "");
Expect(0, 12447, '\p{Is_Blk=katakana}', "");
Expect(1, 12447, '\p{^Is_Blk=katakana}', "");
Expect(1, 12447, '\P{Is_Blk=katakana}', "");
Expect(0, 12447, '\P{^Is_Blk=katakana}', "");
Expect(1, 12543, '\p{Is_Blk: 	katakana}', "");
Expect(0, 12543, '\p{^Is_Blk: 	katakana}', "");
Expect(0, 12543, '\P{Is_Blk: 	katakana}', "");
Expect(1, 12543, '\P{^Is_Blk: 	katakana}', "");
Expect(0, 12447, '\p{Is_Blk: 	katakana}', "");
Expect(1, 12447, '\p{^Is_Blk: 	katakana}', "");
Expect(1, 12447, '\P{Is_Blk: 	katakana}', "");
Expect(0, 12447, '\P{^Is_Blk: 	katakana}', "");
Error('\p{Block=/a/_ CUNEIFORM}');
Error('\P{Block=/a/_ CUNEIFORM}');
Expect(1, 74751, '\p{Block=cuneiform}', "");
Expect(0, 74751, '\p{^Block=cuneiform}', "");
Expect(0, 74751, '\P{Block=cuneiform}', "");
Expect(1, 74751, '\P{^Block=cuneiform}', "");
Expect(0, 73727, '\p{Block=cuneiform}', "");
Expect(1, 73727, '\p{^Block=cuneiform}', "");
Expect(1, 73727, '\P{Block=cuneiform}', "");
Expect(0, 73727, '\P{^Block=cuneiform}', "");
Expect(1, 74751, '\p{Block= -cuneiform}', "");
Expect(0, 74751, '\p{^Block= -cuneiform}', "");
Expect(0, 74751, '\P{Block= -cuneiform}', "");
Expect(1, 74751, '\P{^Block= -cuneiform}', "");
Expect(0, 73727, '\p{Block= -cuneiform}', "");
Expect(1, 73727, '\p{^Block= -cuneiform}', "");
Expect(1, 73727, '\P{Block= -cuneiform}', "");
Expect(0, 73727, '\P{^Block= -cuneiform}', "");
Error('\p{Blk=--cuneiform/a/}');
Error('\P{Blk=--cuneiform/a/}');
Expect(1, 74751, '\p{Blk=cuneiform}', "");
Expect(0, 74751, '\p{^Blk=cuneiform}', "");
Expect(0, 74751, '\P{Blk=cuneiform}', "");
Expect(1, 74751, '\P{^Blk=cuneiform}', "");
Expect(0, 73727, '\p{Blk=cuneiform}', "");
Expect(1, 73727, '\p{^Blk=cuneiform}', "");
Expect(1, 73727, '\P{Blk=cuneiform}', "");
Expect(0, 73727, '\P{^Blk=cuneiform}', "");
Expect(1, 74751, '\p{Blk:   __cuneiform}', "");
Expect(0, 74751, '\p{^Blk:   __cuneiform}', "");
Expect(0, 74751, '\P{Blk:   __cuneiform}', "");
Expect(1, 74751, '\P{^Blk:   __cuneiform}', "");
Expect(0, 73727, '\p{Blk:   __cuneiform}', "");
Expect(1, 73727, '\p{^Blk:   __cuneiform}', "");
Expect(1, 73727, '\P{Blk:   __cuneiform}', "");
Expect(0, 73727, '\P{^Blk:   __cuneiform}', "");
Error('\p{Is_Block=  Cuneiform/a/}');
Error('\P{Is_Block=  Cuneiform/a/}');
Expect(1, 74751, '\p{Is_Block=cuneiform}', "");
Expect(0, 74751, '\p{^Is_Block=cuneiform}', "");
Expect(0, 74751, '\P{Is_Block=cuneiform}', "");
Expect(1, 74751, '\P{^Is_Block=cuneiform}', "");
Expect(0, 73727, '\p{Is_Block=cuneiform}', "");
Expect(1, 73727, '\p{^Is_Block=cuneiform}', "");
Expect(1, 73727, '\P{Is_Block=cuneiform}', "");
Expect(0, 73727, '\P{^Is_Block=cuneiform}', "");
Expect(1, 74751, '\p{Is_Block:	__Cuneiform}', "");
Expect(0, 74751, '\p{^Is_Block:	__Cuneiform}', "");
Expect(0, 74751, '\P{Is_Block:	__Cuneiform}', "");
Expect(1, 74751, '\P{^Is_Block:	__Cuneiform}', "");
Expect(0, 73727, '\p{Is_Block:	__Cuneiform}', "");
Expect(1, 73727, '\p{^Is_Block:	__Cuneiform}', "");
Expect(1, 73727, '\P{Is_Block:	__Cuneiform}', "");
Expect(0, 73727, '\P{^Is_Block:	__Cuneiform}', "");
Error('\p{Is_Blk=_Cuneiform:=}');
Error('\P{Is_Blk=_Cuneiform:=}');
Expect(1, 74751, '\p{Is_Blk=cuneiform}', "");
Expect(0, 74751, '\p{^Is_Blk=cuneiform}', "");
Expect(0, 74751, '\P{Is_Blk=cuneiform}', "");
Expect(1, 74751, '\P{^Is_Blk=cuneiform}', "");
Expect(0, 73727, '\p{Is_Blk=cuneiform}', "");
Expect(1, 73727, '\p{^Is_Blk=cuneiform}', "");
Expect(1, 73727, '\P{Is_Blk=cuneiform}', "");
Expect(0, 73727, '\P{^Is_Blk=cuneiform}', "");
Expect(1, 74751, '\p{Is_Blk=		CUNEIFORM}', "");
Expect(0, 74751, '\p{^Is_Blk=		CUNEIFORM}', "");
Expect(0, 74751, '\P{Is_Blk=		CUNEIFORM}', "");
Expect(1, 74751, '\P{^Is_Blk=		CUNEIFORM}', "");
Expect(0, 73727, '\p{Is_Blk=		CUNEIFORM}', "");
Expect(1, 73727, '\p{^Is_Blk=		CUNEIFORM}', "");
Expect(1, 73727, '\P{Is_Blk=		CUNEIFORM}', "");
Expect(0, 73727, '\P{^Is_Blk=		CUNEIFORM}', "");
Error('\p{Block=:=- high_SURROGATES}');
Error('\P{Block=:=- high_SURROGATES}');
Expect(1, 56191, '\p{Block=highsurrogates}', "");
Expect(0, 56191, '\p{^Block=highsurrogates}', "");
Expect(0, 56191, '\P{Block=highsurrogates}', "");
Expect(1, 56191, '\P{^Block=highsurrogates}', "");
Expect(0, 55295, '\p{Block=highsurrogates}', "");
Expect(1, 55295, '\p{^Block=highsurrogates}', "");
Expect(1, 55295, '\P{Block=highsurrogates}', "");
Expect(0, 55295, '\P{^Block=highsurrogates}', "");
Expect(1, 56191, '\p{Block=	-high	SURROGATES}', "");
Expect(0, 56191, '\p{^Block=	-high	SURROGATES}', "");
Expect(0, 56191, '\P{Block=	-high	SURROGATES}', "");
Expect(1, 56191, '\P{^Block=	-high	SURROGATES}', "");
Expect(0, 55295, '\p{Block=	-high	SURROGATES}', "");
Expect(1, 55295, '\p{^Block=	-high	SURROGATES}', "");
Expect(1, 55295, '\P{Block=	-high	SURROGATES}', "");
Expect(0, 55295, '\P{^Block=	-high	SURROGATES}', "");
Error('\p{Blk=-_HIGH/a/SURROGATES}');
Error('\P{Blk=-_HIGH/a/SURROGATES}');
Expect(1, 56191, '\p{Blk=highsurrogates}', "");
Expect(0, 56191, '\p{^Blk=highsurrogates}', "");
Expect(0, 56191, '\P{Blk=highsurrogates}', "");
Expect(1, 56191, '\P{^Blk=highsurrogates}', "");
Expect(0, 55295, '\p{Blk=highsurrogates}', "");
Expect(1, 55295, '\p{^Blk=highsurrogates}', "");
Expect(1, 55295, '\P{Blk=highsurrogates}', "");
Expect(0, 55295, '\P{^Blk=highsurrogates}', "");
Expect(1, 56191, '\p{Blk=- high Surrogates}', "");
Expect(0, 56191, '\p{^Blk=- high Surrogates}', "");
Expect(0, 56191, '\P{Blk=- high Surrogates}', "");
Expect(1, 56191, '\P{^Blk=- high Surrogates}', "");
Expect(0, 55295, '\p{Blk=- high Surrogates}', "");
Expect(1, 55295, '\p{^Blk=- high Surrogates}', "");
Expect(1, 55295, '\P{Blk=- high Surrogates}', "");
Expect(0, 55295, '\P{^Blk=- high Surrogates}', "");
Error('\p{Is_Block=-/a/HIGH-surrogates}');
Error('\P{Is_Block=-/a/HIGH-surrogates}');
Expect(1, 56191, '\p{Is_Block=highsurrogates}', "");
Expect(0, 56191, '\p{^Is_Block=highsurrogates}', "");
Expect(0, 56191, '\P{Is_Block=highsurrogates}', "");
Expect(1, 56191, '\P{^Is_Block=highsurrogates}', "");
Expect(0, 55295, '\p{Is_Block=highsurrogates}', "");
Expect(1, 55295, '\p{^Is_Block=highsurrogates}', "");
Expect(1, 55295, '\P{Is_Block=highsurrogates}', "");
Expect(0, 55295, '\P{^Is_Block=highsurrogates}', "");
Expect(1, 56191, '\p{Is_Block:	HighSurrogates}', "");
Expect(0, 56191, '\p{^Is_Block:	HighSurrogates}', "");
Expect(0, 56191, '\P{Is_Block:	HighSurrogates}', "");
Expect(1, 56191, '\P{^Is_Block:	HighSurrogates}', "");
Expect(0, 55295, '\p{Is_Block:	HighSurrogates}', "");
Expect(1, 55295, '\p{^Is_Block:	HighSurrogates}', "");
Expect(1, 55295, '\P{Is_Block:	HighSurrogates}', "");
Expect(0, 55295, '\P{^Is_Block:	HighSurrogates}', "");
Error('\p{Is_Blk=:=_-High surrogates}');
Error('\P{Is_Blk=:=_-High surrogates}');
Expect(1, 56191, '\p{Is_Blk=highsurrogates}', "");
Expect(0, 56191, '\p{^Is_Blk=highsurrogates}', "");
Expect(0, 56191, '\P{Is_Blk=highsurrogates}', "");
Expect(1, 56191, '\P{^Is_Blk=highsurrogates}', "");
Expect(0, 55295, '\p{Is_Blk=highsurrogates}', "");
Expect(1, 55295, '\p{^Is_Blk=highsurrogates}', "");
Expect(1, 55295, '\P{Is_Blk=highsurrogates}', "");
Expect(0, 55295, '\P{^Is_Blk=highsurrogates}', "");
Expect(1, 56191, '\p{Is_Blk=	_HighSurrogates}', "");
Expect(0, 56191, '\p{^Is_Blk=	_HighSurrogates}', "");
Expect(0, 56191, '\P{Is_Blk=	_HighSurrogates}', "");
Expect(1, 56191, '\P{^Is_Blk=	_HighSurrogates}', "");
Expect(0, 55295, '\p{Is_Blk=	_HighSurrogates}', "");
Expect(1, 55295, '\p{^Is_Blk=	_HighSurrogates}', "");
Expect(1, 55295, '\P{Is_Blk=	_HighSurrogates}', "");
Expect(0, 55295, '\P{^Is_Blk=	_HighSurrogates}', "");
Error('\p{Block=	 ruminumeral	symbols/a/}');
Error('\P{Block=	 ruminumeral	symbols/a/}');
Expect(1, 69247, '\p{Block:   ruminumeralsymbols}', "");
Expect(0, 69247, '\p{^Block:   ruminumeralsymbols}', "");
Expect(0, 69247, '\P{Block:   ruminumeralsymbols}', "");
Expect(1, 69247, '\P{^Block:   ruminumeralsymbols}', "");
Expect(0, 69215, '\p{Block:   ruminumeralsymbols}', "");
Expect(1, 69215, '\p{^Block:   ruminumeralsymbols}', "");
Expect(1, 69215, '\P{Block:   ruminumeralsymbols}', "");
Expect(0, 69215, '\P{^Block:   ruminumeralsymbols}', "");
Expect(1, 69247, '\p{Block=Rumi_NUMERAL symbols}', "");
Expect(0, 69247, '\p{^Block=Rumi_NUMERAL symbols}', "");
Expect(0, 69247, '\P{Block=Rumi_NUMERAL symbols}', "");
Expect(1, 69247, '\P{^Block=Rumi_NUMERAL symbols}', "");
Expect(0, 69215, '\p{Block=Rumi_NUMERAL symbols}', "");
Expect(1, 69215, '\p{^Block=Rumi_NUMERAL symbols}', "");
Expect(1, 69215, '\P{Block=Rumi_NUMERAL symbols}', "");
Expect(0, 69215, '\P{^Block=Rumi_NUMERAL symbols}', "");
Error('\p{Blk=:= Rumi-NUMERAL_symbols}');
Error('\P{Blk=:= Rumi-NUMERAL_symbols}');
Expect(1, 69247, '\p{Blk=ruminumeralsymbols}', "");
Expect(0, 69247, '\p{^Blk=ruminumeralsymbols}', "");
Expect(0, 69247, '\P{Blk=ruminumeralsymbols}', "");
Expect(1, 69247, '\P{^Blk=ruminumeralsymbols}', "");
Expect(0, 69215, '\p{Blk=ruminumeralsymbols}', "");
Expect(1, 69215, '\p{^Blk=ruminumeralsymbols}', "");
Expect(1, 69215, '\P{Blk=ruminumeralsymbols}', "");
Expect(0, 69215, '\P{^Blk=ruminumeralsymbols}', "");
Expect(1, 69247, '\p{Blk: -_RUMINumeral_SYMBOLS}', "");
Expect(0, 69247, '\p{^Blk: -_RUMINumeral_SYMBOLS}', "");
Expect(0, 69247, '\P{Blk: -_RUMINumeral_SYMBOLS}', "");
Expect(1, 69247, '\P{^Blk: -_RUMINumeral_SYMBOLS}', "");
Expect(0, 69215, '\p{Blk: -_RUMINumeral_SYMBOLS}', "");
Expect(1, 69215, '\p{^Blk: -_RUMINumeral_SYMBOLS}', "");
Expect(1, 69215, '\P{Blk: -_RUMINumeral_SYMBOLS}', "");
Expect(0, 69215, '\P{^Blk: -_RUMINumeral_SYMBOLS}', "");
Error('\p{Is_Block= /a/RumiNumeral Symbols}');
Error('\P{Is_Block= /a/RumiNumeral Symbols}');
Expect(1, 69247, '\p{Is_Block=ruminumeralsymbols}', "");
Expect(0, 69247, '\p{^Is_Block=ruminumeralsymbols}', "");
Expect(0, 69247, '\P{Is_Block=ruminumeralsymbols}', "");
Expect(1, 69247, '\P{^Is_Block=ruminumeralsymbols}', "");
Expect(0, 69215, '\p{Is_Block=ruminumeralsymbols}', "");
Expect(1, 69215, '\p{^Is_Block=ruminumeralsymbols}', "");
Expect(1, 69215, '\P{Is_Block=ruminumeralsymbols}', "");
Expect(0, 69215, '\P{^Is_Block=ruminumeralsymbols}', "");
Expect(1, 69247, '\p{Is_Block=_-Rumi_Numeral SYMBOLS}', "");
Expect(0, 69247, '\p{^Is_Block=_-Rumi_Numeral SYMBOLS}', "");
Expect(0, 69247, '\P{Is_Block=_-Rumi_Numeral SYMBOLS}', "");
Expect(1, 69247, '\P{^Is_Block=_-Rumi_Numeral SYMBOLS}', "");
Expect(0, 69215, '\p{Is_Block=_-Rumi_Numeral SYMBOLS}', "");
Expect(1, 69215, '\p{^Is_Block=_-Rumi_Numeral SYMBOLS}', "");
Expect(1, 69215, '\P{Is_Block=_-Rumi_Numeral SYMBOLS}', "");
Expect(0, 69215, '\P{^Is_Block=_-Rumi_Numeral SYMBOLS}', "");
Error('\p{Is_Blk=:= RUMI Numeral	Symbols}');
Error('\P{Is_Blk=:= RUMI Numeral	Symbols}');
Expect(1, 69247, '\p{Is_Blk=ruminumeralsymbols}', "");
Expect(0, 69247, '\p{^Is_Blk=ruminumeralsymbols}', "");
Expect(0, 69247, '\P{Is_Blk=ruminumeralsymbols}', "");
Expect(1, 69247, '\P{^Is_Blk=ruminumeralsymbols}', "");
Expect(0, 69215, '\p{Is_Blk=ruminumeralsymbols}', "");
Expect(1, 69215, '\p{^Is_Blk=ruminumeralsymbols}', "");
Expect(1, 69215, '\P{Is_Blk=ruminumeralsymbols}', "");
Expect(0, 69215, '\P{^Is_Blk=ruminumeralsymbols}', "");
Expect(1, 69247, '\p{Is_Blk=		Rumi NUMERAL-symbols}', "");
Expect(0, 69247, '\p{^Is_Blk=		Rumi NUMERAL-symbols}', "");
Expect(0, 69247, '\P{Is_Blk=		Rumi NUMERAL-symbols}', "");
Expect(1, 69247, '\P{^Is_Blk=		Rumi NUMERAL-symbols}', "");
Expect(0, 69215, '\p{Is_Blk=		Rumi NUMERAL-symbols}', "");
Expect(1, 69215, '\p{^Is_Blk=		Rumi NUMERAL-symbols}', "");
Expect(1, 69215, '\P{Is_Blk=		Rumi NUMERAL-symbols}', "");
Expect(0, 69215, '\P{^Is_Blk=		Rumi NUMERAL-symbols}', "");
Error('\p{Block=__Variation_Selectors:=supplement}');
Error('\P{Block=__Variation_Selectors:=supplement}');
Expect(1, 917999, '\p{Block=variationselectorssupplement}', "");
Expect(0, 917999, '\p{^Block=variationselectorssupplement}', "");
Expect(0, 917999, '\P{Block=variationselectorssupplement}', "");
Expect(1, 917999, '\P{^Block=variationselectorssupplement}', "");
Expect(0, 917759, '\p{Block=variationselectorssupplement}', "");
Expect(1, 917759, '\p{^Block=variationselectorssupplement}', "");
Expect(1, 917759, '\P{Block=variationselectorssupplement}', "");
Expect(0, 917759, '\P{^Block=variationselectorssupplement}', "");
Expect(1, 917999, '\p{Block= Variation selectors	supplement}', "");
Expect(0, 917999, '\p{^Block= Variation selectors	supplement}', "");
Expect(0, 917999, '\P{Block= Variation selectors	supplement}', "");
Expect(1, 917999, '\P{^Block= Variation selectors	supplement}', "");
Expect(0, 917759, '\p{Block= Variation selectors	supplement}', "");
Expect(1, 917759, '\p{^Block= Variation selectors	supplement}', "");
Expect(1, 917759, '\P{Block= Variation selectors	supplement}', "");
Expect(0, 917759, '\P{^Block= Variation selectors	supplement}', "");
Error('\p{Blk=:=_	VARIATION_Selectors-Supplement}');
Error('\P{Blk=:=_	VARIATION_Selectors-Supplement}');
Expect(1, 917999, '\p{Blk=variationselectorssupplement}', "");
Expect(0, 917999, '\p{^Blk=variationselectorssupplement}', "");
Expect(0, 917999, '\P{Blk=variationselectorssupplement}', "");
Expect(1, 917999, '\P{^Blk=variationselectorssupplement}', "");
Expect(0, 917759, '\p{Blk=variationselectorssupplement}', "");
Expect(1, 917759, '\p{^Blk=variationselectorssupplement}', "");
Expect(1, 917759, '\P{Blk=variationselectorssupplement}', "");
Expect(0, 917759, '\P{^Blk=variationselectorssupplement}', "");
Expect(1, 917999, '\p{Blk= -variationSELECTORS	Supplement}', "");
Expect(0, 917999, '\p{^Blk= -variationSELECTORS	Supplement}', "");
Expect(0, 917999, '\P{Blk= -variationSELECTORS	Supplement}', "");
Expect(1, 917999, '\P{^Blk= -variationSELECTORS	Supplement}', "");
Expect(0, 917759, '\p{Blk= -variationSELECTORS	Supplement}', "");
Expect(1, 917759, '\p{^Blk= -variationSELECTORS	Supplement}', "");
Expect(1, 917759, '\P{Blk= -variationSELECTORS	Supplement}', "");
Expect(0, 917759, '\P{^Blk= -variationSELECTORS	Supplement}', "");
Error('\p{Is_Block=variation	Selectors/a/supplement}');
Error('\P{Is_Block=variation	Selectors/a/supplement}');
Expect(1, 917999, '\p{Is_Block: variationselectorssupplement}', "");
Expect(0, 917999, '\p{^Is_Block: variationselectorssupplement}', "");
Expect(0, 917999, '\P{Is_Block: variationselectorssupplement}', "");
Expect(1, 917999, '\P{^Is_Block: variationselectorssupplement}', "");
Expect(0, 917759, '\p{Is_Block: variationselectorssupplement}', "");
Expect(1, 917759, '\p{^Is_Block: variationselectorssupplement}', "");
Expect(1, 917759, '\P{Is_Block: variationselectorssupplement}', "");
Expect(0, 917759, '\P{^Is_Block: variationselectorssupplement}', "");
Expect(1, 917999, '\p{Is_Block= variation	selectors	supplement}', "");
Expect(0, 917999, '\p{^Is_Block= variation	selectors	supplement}', "");
Expect(0, 917999, '\P{Is_Block= variation	selectors	supplement}', "");
Expect(1, 917999, '\P{^Is_Block= variation	selectors	supplement}', "");
Expect(0, 917759, '\p{Is_Block= variation	selectors	supplement}', "");
Expect(1, 917759, '\p{^Is_Block= variation	selectors	supplement}', "");
Expect(1, 917759, '\P{Is_Block= variation	selectors	supplement}', "");
Expect(0, 917759, '\P{^Is_Block= variation	selectors	supplement}', "");
Error('\p{Is_Blk=_	variation-SELECTORS:=SUPPLEMENT}');
Error('\P{Is_Blk=_	variation-SELECTORS:=SUPPLEMENT}');
Expect(1, 917999, '\p{Is_Blk=variationselectorssupplement}', "");
Expect(0, 917999, '\p{^Is_Blk=variationselectorssupplement}', "");
Expect(0, 917999, '\P{Is_Blk=variationselectorssupplement}', "");
Expect(1, 917999, '\P{^Is_Blk=variationselectorssupplement}', "");
Expect(0, 917759, '\p{Is_Blk=variationselectorssupplement}', "");
Expect(1, 917759, '\p{^Is_Blk=variationselectorssupplement}', "");
Expect(1, 917759, '\P{Is_Blk=variationselectorssupplement}', "");
Expect(0, 917759, '\P{^Is_Blk=variationselectorssupplement}', "");
Expect(1, 917999, '\p{Is_Blk= 	Variation	SELECTORS Supplement}', "");
Expect(0, 917999, '\p{^Is_Blk= 	Variation	SELECTORS Supplement}', "");
Expect(0, 917999, '\P{Is_Blk= 	Variation	SELECTORS Supplement}', "");
Expect(1, 917999, '\P{^Is_Blk= 	Variation	SELECTORS Supplement}', "");
Expect(0, 917759, '\p{Is_Blk= 	Variation	SELECTORS Supplement}', "");
Expect(1, 917759, '\p{^Is_Blk= 	Variation	SELECTORS Supplement}', "");
Expect(1, 917759, '\P{Is_Blk= 	Variation	SELECTORS Supplement}', "");
Expect(0, 917759, '\P{^Is_Blk= 	Variation	SELECTORS Supplement}', "");
Error('\p{Block=:=_Georgian	SUPPLEMENT}');
Error('\P{Block=:=_Georgian	SUPPLEMENT}');
Expect(1, 11567, '\p{Block=georgiansupplement}', "");
Expect(0, 11567, '\p{^Block=georgiansupplement}', "");
Expect(0, 11567, '\P{Block=georgiansupplement}', "");
Expect(1, 11567, '\P{^Block=georgiansupplement}', "");
Expect(0, 11519, '\p{Block=georgiansupplement}', "");
Expect(1, 11519, '\p{^Block=georgiansupplement}', "");
Expect(1, 11519, '\P{Block=georgiansupplement}', "");
Expect(0, 11519, '\P{^Block=georgiansupplement}', "");
Expect(1, 11567, '\p{Block=_ Georgian_supplement}', "");
Expect(0, 11567, '\p{^Block=_ Georgian_supplement}', "");
Expect(0, 11567, '\P{Block=_ Georgian_supplement}', "");
Expect(1, 11567, '\P{^Block=_ Georgian_supplement}', "");
Expect(0, 11519, '\p{Block=_ Georgian_supplement}', "");
Expect(1, 11519, '\p{^Block=_ Georgian_supplement}', "");
Expect(1, 11519, '\P{Block=_ Georgian_supplement}', "");
Expect(0, 11519, '\P{^Block=_ Georgian_supplement}', "");
Error('\p{Blk=/a/ _GeorgianSUPPLEMENT}');
Error('\P{Blk=/a/ _GeorgianSUPPLEMENT}');
Expect(1, 11567, '\p{Blk=georgiansupplement}', "");
Expect(0, 11567, '\p{^Blk=georgiansupplement}', "");
Expect(0, 11567, '\P{Blk=georgiansupplement}', "");
Expect(1, 11567, '\P{^Blk=georgiansupplement}', "");
Expect(0, 11519, '\p{Blk=georgiansupplement}', "");
Expect(1, 11519, '\p{^Blk=georgiansupplement}', "");
Expect(1, 11519, '\P{Blk=georgiansupplement}', "");
Expect(0, 11519, '\P{^Blk=georgiansupplement}', "");
Expect(1, 11567, '\p{Blk=-	georgian	supplement}', "");
Expect(0, 11567, '\p{^Blk=-	georgian	supplement}', "");
Expect(0, 11567, '\P{Blk=-	georgian	supplement}', "");
Expect(1, 11567, '\P{^Blk=-	georgian	supplement}', "");
Expect(0, 11519, '\p{Blk=-	georgian	supplement}', "");
Expect(1, 11519, '\p{^Blk=-	georgian	supplement}', "");
Expect(1, 11519, '\P{Blk=-	georgian	supplement}', "");
Expect(0, 11519, '\P{^Blk=-	georgian	supplement}', "");
Error('\p{Is_Block=	:=Georgiansupplement}');
Error('\P{Is_Block=	:=Georgiansupplement}');
Expect(1, 11567, '\p{Is_Block=georgiansupplement}', "");
Expect(0, 11567, '\p{^Is_Block=georgiansupplement}', "");
Expect(0, 11567, '\P{Is_Block=georgiansupplement}', "");
Expect(1, 11567, '\P{^Is_Block=georgiansupplement}', "");
Expect(0, 11519, '\p{Is_Block=georgiansupplement}', "");
Expect(1, 11519, '\p{^Is_Block=georgiansupplement}', "");
Expect(1, 11519, '\P{Is_Block=georgiansupplement}', "");
Expect(0, 11519, '\P{^Is_Block=georgiansupplement}', "");
Expect(1, 11567, '\p{Is_Block=	Georgian	SUPPLEMENT}', "");
Expect(0, 11567, '\p{^Is_Block=	Georgian	SUPPLEMENT}', "");
Expect(0, 11567, '\P{Is_Block=	Georgian	SUPPLEMENT}', "");
Expect(1, 11567, '\P{^Is_Block=	Georgian	SUPPLEMENT}', "");
Expect(0, 11519, '\p{Is_Block=	Georgian	SUPPLEMENT}', "");
Expect(1, 11519, '\p{^Is_Block=	Georgian	SUPPLEMENT}', "");
Expect(1, 11519, '\P{Is_Block=	Georgian	SUPPLEMENT}', "");
Expect(0, 11519, '\P{^Is_Block=	Georgian	SUPPLEMENT}', "");
Error('\p{Is_Blk= Georgian	supplement/a/}');
Error('\P{Is_Blk= Georgian	supplement/a/}');
Expect(1, 11567, '\p{Is_Blk=georgiansupplement}', "");
Expect(0, 11567, '\p{^Is_Blk=georgiansupplement}', "");
Expect(0, 11567, '\P{Is_Blk=georgiansupplement}', "");
Expect(1, 11567, '\P{^Is_Blk=georgiansupplement}', "");
Expect(0, 11519, '\p{Is_Blk=georgiansupplement}', "");
Expect(1, 11519, '\p{^Is_Blk=georgiansupplement}', "");
Expect(1, 11519, '\P{Is_Blk=georgiansupplement}', "");
Expect(0, 11519, '\P{^Is_Blk=georgiansupplement}', "");
Expect(1, 11567, '\p{Is_Blk= _georgiansupplement}', "");
Expect(0, 11567, '\p{^Is_Blk= _georgiansupplement}', "");
Expect(0, 11567, '\P{Is_Blk= _georgiansupplement}', "");
Expect(1, 11567, '\P{^Is_Blk= _georgiansupplement}', "");
Expect(0, 11519, '\p{Is_Blk= _georgiansupplement}', "");
Expect(1, 11519, '\p{^Is_Blk= _georgiansupplement}', "");
Expect(1, 11519, '\P{Is_Blk= _georgiansupplement}', "");
Expect(0, 11519, '\P{^Is_Blk= _georgiansupplement}', "");
Error('\p{Block=	_CJK Unified-Ideographs-EXTENSION-D/a/}');
Error('\P{Block=	_CJK Unified-Ideographs-EXTENSION-D/a/}');
Expect(1, 178207, '\p{Block:   cjkunifiedideographsextensiond}', "");
Expect(0, 178207, '\p{^Block:   cjkunifiedideographsextensiond}', "");
Expect(0, 178207, '\P{Block:   cjkunifiedideographsextensiond}', "");
Expect(1, 178207, '\P{^Block:   cjkunifiedideographsextensiond}', "");
Expect(0, 177983, '\p{Block:   cjkunifiedideographsextensiond}', "");
Expect(1, 177983, '\p{^Block:   cjkunifiedideographsextensiond}', "");
Expect(1, 177983, '\P{Block:   cjkunifiedideographsextensiond}', "");
Expect(0, 177983, '\P{^Block:   cjkunifiedideographsextensiond}', "");
Expect(1, 178207, '\p{Block=		CJK	unified_IDEOGRAPHSextension	D}', "");
Expect(0, 178207, '\p{^Block=		CJK	unified_IDEOGRAPHSextension	D}', "");
Expect(0, 178207, '\P{Block=		CJK	unified_IDEOGRAPHSextension	D}', "");
Expect(1, 178207, '\P{^Block=		CJK	unified_IDEOGRAPHSextension	D}', "");
Expect(0, 177983, '\p{Block=		CJK	unified_IDEOGRAPHSextension	D}', "");
Expect(1, 177983, '\p{^Block=		CJK	unified_IDEOGRAPHSextension	D}', "");
Expect(1, 177983, '\P{Block=		CJK	unified_IDEOGRAPHSextension	D}', "");
Expect(0, 177983, '\P{^Block=		CJK	unified_IDEOGRAPHSextension	D}', "");
Error('\p{Blk=_CJK	unified Ideographs_Extension:=D}');
Error('\P{Blk=_CJK	unified Ideographs_Extension:=D}');
Expect(1, 178207, '\p{Blk=cjkunifiedideographsextensiond}', "");
Expect(0, 178207, '\p{^Blk=cjkunifiedideographsextensiond}', "");
Expect(0, 178207, '\P{Blk=cjkunifiedideographsextensiond}', "");
Expect(1, 178207, '\P{^Blk=cjkunifiedideographsextensiond}', "");
Expect(0, 177983, '\p{Blk=cjkunifiedideographsextensiond}', "");
Expect(1, 177983, '\p{^Blk=cjkunifiedideographsextensiond}', "");
Expect(1, 177983, '\P{Blk=cjkunifiedideographsextensiond}', "");
Expect(0, 177983, '\P{^Blk=cjkunifiedideographsextensiond}', "");
Expect(1, 178207, '\p{Blk= cjkUNIFIED Ideographs	extensionD}', "");
Expect(0, 178207, '\p{^Blk= cjkUNIFIED Ideographs	extensionD}', "");
Expect(0, 178207, '\P{Blk= cjkUNIFIED Ideographs	extensionD}', "");
Expect(1, 178207, '\P{^Blk= cjkUNIFIED Ideographs	extensionD}', "");
Expect(0, 177983, '\p{Blk= cjkUNIFIED Ideographs	extensionD}', "");
Expect(1, 177983, '\p{^Blk= cjkUNIFIED Ideographs	extensionD}', "");
Expect(1, 177983, '\P{Blk= cjkUNIFIED Ideographs	extensionD}', "");
Expect(0, 177983, '\P{^Blk= cjkUNIFIED Ideographs	extensionD}', "");
Error('\p{Is_Block= :=CJK UNIFIEDideographs ExtensionD}');
Error('\P{Is_Block= :=CJK UNIFIEDideographs ExtensionD}');
Expect(1, 178207, '\p{Is_Block=cjkunifiedideographsextensiond}', "");
Expect(0, 178207, '\p{^Is_Block=cjkunifiedideographsextensiond}', "");
Expect(0, 178207, '\P{Is_Block=cjkunifiedideographsextensiond}', "");
Expect(1, 178207, '\P{^Is_Block=cjkunifiedideographsextensiond}', "");
Expect(0, 177983, '\p{Is_Block=cjkunifiedideographsextensiond}', "");
Expect(1, 177983, '\p{^Is_Block=cjkunifiedideographsextensiond}', "");
Expect(1, 177983, '\P{Is_Block=cjkunifiedideographsextensiond}', "");
Expect(0, 177983, '\P{^Is_Block=cjkunifiedideographsextensiond}', "");
Expect(1, 178207, '\p{Is_Block=--cjk UNIFIED IDEOGRAPHS Extension d}', "");
Expect(0, 178207, '\p{^Is_Block=--cjk UNIFIED IDEOGRAPHS Extension d}', "");
Expect(0, 178207, '\P{Is_Block=--cjk UNIFIED IDEOGRAPHS Extension d}', "");
Expect(1, 178207, '\P{^Is_Block=--cjk UNIFIED IDEOGRAPHS Extension d}', "");
Expect(0, 177983, '\p{Is_Block=--cjk UNIFIED IDEOGRAPHS Extension d}', "");
Expect(1, 177983, '\p{^Is_Block=--cjk UNIFIED IDEOGRAPHS Extension d}', "");
Expect(1, 177983, '\P{Is_Block=--cjk UNIFIED IDEOGRAPHS Extension d}', "");
Expect(0, 177983, '\P{^Is_Block=--cjk UNIFIED IDEOGRAPHS Extension d}', "");
Error('\p{Is_Blk= _CJK-UNIFIED_ideographsextension:=d}');
Error('\P{Is_Blk= _CJK-UNIFIED_ideographsextension:=d}');
Expect(1, 178207, '\p{Is_Blk=cjkunifiedideographsextensiond}', "");
Expect(0, 178207, '\p{^Is_Blk=cjkunifiedideographsextensiond}', "");
Expect(0, 178207, '\P{Is_Blk=cjkunifiedideographsextensiond}', "");
Expect(1, 178207, '\P{^Is_Blk=cjkunifiedideographsextensiond}', "");
Expect(0, 177983, '\p{Is_Blk=cjkunifiedideographsextensiond}', "");
Expect(1, 177983, '\p{^Is_Blk=cjkunifiedideographsextensiond}', "");
Expect(1, 177983, '\P{Is_Blk=cjkunifiedideographsextensiond}', "");
Expect(0, 177983, '\P{^Is_Blk=cjkunifiedideographsextensiond}', "");
Expect(1, 178207, '\p{Is_Blk=--CJK Unified-ideographsExtensionD}', "");
Expect(0, 178207, '\p{^Is_Blk=--CJK Unified-ideographsExtensionD}', "");
Expect(0, 178207, '\P{Is_Blk=--CJK Unified-ideographsExtensionD}', "");
Expect(1, 178207, '\P{^Is_Blk=--CJK Unified-ideographsExtensionD}', "");
Expect(0, 177983, '\p{Is_Blk=--CJK Unified-ideographsExtensionD}', "");
Expect(1, 177983, '\p{^Is_Blk=--CJK Unified-ideographsExtensionD}', "");
Expect(1, 177983, '\P{Is_Blk=--CJK Unified-ideographsExtensionD}', "");
Expect(0, 177983, '\P{^Is_Blk=--CJK Unified-ideographsExtensionD}', "");
Error('\p{Block=-/a/Myanmar}');
Error('\P{Block=-/a/Myanmar}');
Expect(1, 4255, '\p{Block=myanmar}', "");
Expect(0, 4255, '\p{^Block=myanmar}', "");
Expect(0, 4255, '\P{Block=myanmar}', "");
Expect(1, 4255, '\P{^Block=myanmar}', "");
Expect(0, 4095, '\p{Block=myanmar}', "");
Expect(1, 4095, '\p{^Block=myanmar}', "");
Expect(1, 4095, '\P{Block=myanmar}', "");
Expect(0, 4095, '\P{^Block=myanmar}', "");
Expect(1, 4255, '\p{Block=-	Myanmar}', "");
Expect(0, 4255, '\p{^Block=-	Myanmar}', "");
Expect(0, 4255, '\P{Block=-	Myanmar}', "");
Expect(1, 4255, '\P{^Block=-	Myanmar}', "");
Expect(0, 4095, '\p{Block=-	Myanmar}', "");
Expect(1, 4095, '\p{^Block=-	Myanmar}', "");
Expect(1, 4095, '\P{Block=-	Myanmar}', "");
Expect(0, 4095, '\P{^Block=-	Myanmar}', "");
Error('\p{Blk=:=_Myanmar}');
Error('\P{Blk=:=_Myanmar}');
Expect(1, 4255, '\p{Blk=myanmar}', "");
Expect(0, 4255, '\p{^Blk=myanmar}', "");
Expect(0, 4255, '\P{Blk=myanmar}', "");
Expect(1, 4255, '\P{^Blk=myanmar}', "");
Expect(0, 4095, '\p{Blk=myanmar}', "");
Expect(1, 4095, '\p{^Blk=myanmar}', "");
Expect(1, 4095, '\P{Blk=myanmar}', "");
Expect(0, 4095, '\P{^Blk=myanmar}', "");
Expect(1, 4255, '\p{Blk=-	Myanmar}', "");
Expect(0, 4255, '\p{^Blk=-	Myanmar}', "");
Expect(0, 4255, '\P{Blk=-	Myanmar}', "");
Expect(1, 4255, '\P{^Blk=-	Myanmar}', "");
Expect(0, 4095, '\p{Blk=-	Myanmar}', "");
Expect(1, 4095, '\p{^Blk=-	Myanmar}', "");
Expect(1, 4095, '\P{Blk=-	Myanmar}', "");
Expect(0, 4095, '\P{^Blk=-	Myanmar}', "");
Error('\p{Is_Block= _myanmar:=}');
Error('\P{Is_Block= _myanmar:=}');
Expect(1, 4255, '\p{Is_Block=myanmar}', "");
Expect(0, 4255, '\p{^Is_Block=myanmar}', "");
Expect(0, 4255, '\P{Is_Block=myanmar}', "");
Expect(1, 4255, '\P{^Is_Block=myanmar}', "");
Expect(0, 4095, '\p{Is_Block=myanmar}', "");
Expect(1, 4095, '\p{^Is_Block=myanmar}', "");
Expect(1, 4095, '\P{Is_Block=myanmar}', "");
Expect(0, 4095, '\P{^Is_Block=myanmar}', "");
Expect(1, 4255, '\p{Is_Block=__Myanmar}', "");
Expect(0, 4255, '\p{^Is_Block=__Myanmar}', "");
Expect(0, 4255, '\P{Is_Block=__Myanmar}', "");
Expect(1, 4255, '\P{^Is_Block=__Myanmar}', "");
Expect(0, 4095, '\p{Is_Block=__Myanmar}', "");
Expect(1, 4095, '\p{^Is_Block=__Myanmar}', "");
Expect(1, 4095, '\P{Is_Block=__Myanmar}', "");
Expect(0, 4095, '\P{^Is_Block=__Myanmar}', "");
Error('\p{Is_Blk= _Myanmar:=}');
Error('\P{Is_Blk= _Myanmar:=}');
Expect(1, 4255, '\p{Is_Blk=myanmar}', "");
Expect(0, 4255, '\p{^Is_Blk=myanmar}', "");
Expect(0, 4255, '\P{Is_Blk=myanmar}', "");
Expect(1, 4255, '\P{^Is_Blk=myanmar}', "");
Expect(0, 4095, '\p{Is_Blk=myanmar}', "");
Expect(1, 4095, '\p{^Is_Blk=myanmar}', "");
Expect(1, 4095, '\P{Is_Blk=myanmar}', "");
Expect(0, 4095, '\P{^Is_Blk=myanmar}', "");
Expect(1, 4255, '\p{Is_Blk=--MYANMAR}', "");
Expect(0, 4255, '\p{^Is_Blk=--MYANMAR}', "");
Expect(0, 4255, '\P{Is_Blk=--MYANMAR}', "");
Expect(1, 4255, '\P{^Is_Blk=--MYANMAR}', "");
Expect(0, 4095, '\p{Is_Blk=--MYANMAR}', "");
Expect(1, 4095, '\p{^Is_Blk=--MYANMAR}', "");
Expect(1, 4095, '\P{Is_Blk=--MYANMAR}', "");
Expect(0, 4095, '\P{^Is_Blk=--MYANMAR}', "");
Error('\p{Block=:=-_VERTICAL	forms}');
Error('\P{Block=:=-_VERTICAL	forms}');
Expect(1, 65055, '\p{Block=verticalforms}', "");
Expect(0, 65055, '\p{^Block=verticalforms}', "");
Expect(0, 65055, '\P{Block=verticalforms}', "");
Expect(1, 65055, '\P{^Block=verticalforms}', "");
Expect(0, 65039, '\p{Block=verticalforms}', "");
Expect(1, 65039, '\p{^Block=verticalforms}', "");
Expect(1, 65039, '\P{Block=verticalforms}', "");
Expect(0, 65039, '\P{^Block=verticalforms}', "");
Expect(1, 65055, '\p{Block=	 VerticalFORMS}', "");
Expect(0, 65055, '\p{^Block=	 VerticalFORMS}', "");
Expect(0, 65055, '\P{Block=	 VerticalFORMS}', "");
Expect(1, 65055, '\P{^Block=	 VerticalFORMS}', "");
Expect(0, 65039, '\p{Block=	 VerticalFORMS}', "");
Expect(1, 65039, '\p{^Block=	 VerticalFORMS}', "");
Expect(1, 65039, '\P{Block=	 VerticalFORMS}', "");
Expect(0, 65039, '\P{^Block=	 VerticalFORMS}', "");
Error('\p{Blk= :=VERTICAL-FORMS}');
Error('\P{Blk= :=VERTICAL-FORMS}');
Expect(1, 65055, '\p{Blk=verticalforms}', "");
Expect(0, 65055, '\p{^Blk=verticalforms}', "");
Expect(0, 65055, '\P{Blk=verticalforms}', "");
Expect(1, 65055, '\P{^Blk=verticalforms}', "");
Expect(0, 65039, '\p{Blk=verticalforms}', "");
Expect(1, 65039, '\p{^Blk=verticalforms}', "");
Expect(1, 65039, '\P{Blk=verticalforms}', "");
Expect(0, 65039, '\P{^Blk=verticalforms}', "");
Expect(1, 65055, '\p{Blk=  Vertical	Forms}', "");
Expect(0, 65055, '\p{^Blk=  Vertical	Forms}', "");
Expect(0, 65055, '\P{Blk=  Vertical	Forms}', "");
Expect(1, 65055, '\P{^Blk=  Vertical	Forms}', "");
Expect(0, 65039, '\p{Blk=  Vertical	Forms}', "");
Expect(1, 65039, '\p{^Blk=  Vertical	Forms}', "");
Expect(1, 65039, '\P{Blk=  Vertical	Forms}', "");
Expect(0, 65039, '\P{^Blk=  Vertical	Forms}', "");
Error('\p{Is_Block= 	Vertical-Forms:=}');
Error('\P{Is_Block= 	Vertical-Forms:=}');
Expect(1, 65055, '\p{Is_Block=verticalforms}', "");
Expect(0, 65055, '\p{^Is_Block=verticalforms}', "");
Expect(0, 65055, '\P{Is_Block=verticalforms}', "");
Expect(1, 65055, '\P{^Is_Block=verticalforms}', "");
Expect(0, 65039, '\p{Is_Block=verticalforms}', "");
Expect(1, 65039, '\p{^Is_Block=verticalforms}', "");
Expect(1, 65039, '\P{Is_Block=verticalforms}', "");
Expect(0, 65039, '\P{^Is_Block=verticalforms}', "");
Expect(1, 65055, '\p{Is_Block=_Vertical-forms}', "");
Expect(0, 65055, '\p{^Is_Block=_Vertical-forms}', "");
Expect(0, 65055, '\P{Is_Block=_Vertical-forms}', "");
Expect(1, 65055, '\P{^Is_Block=_Vertical-forms}', "");
Expect(0, 65039, '\p{Is_Block=_Vertical-forms}', "");
Expect(1, 65039, '\p{^Is_Block=_Vertical-forms}', "");
Expect(1, 65039, '\P{Is_Block=_Vertical-forms}', "");
Expect(0, 65039, '\P{^Is_Block=_Vertical-forms}', "");
Error('\p{Is_Blk=  Vertical_forms:=}');
Error('\P{Is_Blk=  Vertical_forms:=}');
Expect(1, 65055, '\p{Is_Blk=verticalforms}', "");
Expect(0, 65055, '\p{^Is_Blk=verticalforms}', "");
Expect(0, 65055, '\P{Is_Blk=verticalforms}', "");
Expect(1, 65055, '\P{^Is_Blk=verticalforms}', "");
Expect(0, 65039, '\p{Is_Blk=verticalforms}', "");
Expect(1, 65039, '\p{^Is_Blk=verticalforms}', "");
Expect(1, 65039, '\P{Is_Blk=verticalforms}', "");
Expect(0, 65039, '\P{^Is_Blk=verticalforms}', "");
Expect(1, 65055, '\p{Is_Blk= -Vertical forms}', "");
Expect(0, 65055, '\p{^Is_Blk= -Vertical forms}', "");
Expect(0, 65055, '\P{Is_Blk= -Vertical forms}', "");
Expect(1, 65055, '\P{^Is_Blk= -Vertical forms}', "");
Expect(0, 65039, '\p{Is_Blk= -Vertical forms}', "");
Expect(1, 65039, '\p{^Is_Blk= -Vertical forms}', "");
Expect(1, 65039, '\P{Is_Blk= -Vertical forms}', "");
Expect(0, 65039, '\P{^Is_Blk= -Vertical forms}', "");
Error('\p{Block=/a/-_kaithi}');
Error('\P{Block=/a/-_kaithi}');
Expect(1, 69839, '\p{Block=kaithi}', "");
Expect(0, 69839, '\p{^Block=kaithi}', "");
Expect(0, 69839, '\P{Block=kaithi}', "");
Expect(1, 69839, '\P{^Block=kaithi}', "");
Expect(0, 69759, '\p{Block=kaithi}', "");
Expect(1, 69759, '\p{^Block=kaithi}', "");
Expect(1, 69759, '\P{Block=kaithi}', "");
Expect(0, 69759, '\P{^Block=kaithi}', "");
Expect(1, 69839, '\p{Block=	_KAITHI}', "");
Expect(0, 69839, '\p{^Block=	_KAITHI}', "");
Expect(0, 69839, '\P{Block=	_KAITHI}', "");
Expect(1, 69839, '\P{^Block=	_KAITHI}', "");
Expect(0, 69759, '\p{Block=	_KAITHI}', "");
Expect(1, 69759, '\p{^Block=	_KAITHI}', "");
Expect(1, 69759, '\P{Block=	_KAITHI}', "");
Expect(0, 69759, '\P{^Block=	_KAITHI}', "");
Error('\p{Blk=/a/Kaithi}');
Error('\P{Blk=/a/Kaithi}');
Expect(1, 69839, '\p{Blk=kaithi}', "");
Expect(0, 69839, '\p{^Blk=kaithi}', "");
Expect(0, 69839, '\P{Blk=kaithi}', "");
Expect(1, 69839, '\P{^Blk=kaithi}', "");
Expect(0, 69759, '\p{Blk=kaithi}', "");
Expect(1, 69759, '\p{^Blk=kaithi}', "");
Expect(1, 69759, '\P{Blk=kaithi}', "");
Expect(0, 69759, '\P{^Blk=kaithi}', "");
Expect(1, 69839, '\p{Blk=-Kaithi}', "");
Expect(0, 69839, '\p{^Blk=-Kaithi}', "");
Expect(0, 69839, '\P{Blk=-Kaithi}', "");
Expect(1, 69839, '\P{^Blk=-Kaithi}', "");
Expect(0, 69759, '\p{Blk=-Kaithi}', "");
Expect(1, 69759, '\p{^Blk=-Kaithi}', "");
Expect(1, 69759, '\P{Blk=-Kaithi}', "");
Expect(0, 69759, '\P{^Blk=-Kaithi}', "");
Error('\p{Is_Block=-:=Kaithi}');
Error('\P{Is_Block=-:=Kaithi}');
Expect(1, 69839, '\p{Is_Block:kaithi}', "");
Expect(0, 69839, '\p{^Is_Block:kaithi}', "");
Expect(0, 69839, '\P{Is_Block:kaithi}', "");
Expect(1, 69839, '\P{^Is_Block:kaithi}', "");
Expect(0, 69759, '\p{Is_Block:kaithi}', "");
Expect(1, 69759, '\p{^Is_Block:kaithi}', "");
Expect(1, 69759, '\P{Is_Block:kaithi}', "");
Expect(0, 69759, '\P{^Is_Block:kaithi}', "");
Expect(1, 69839, '\p{Is_Block= 	KAITHI}', "");
Expect(0, 69839, '\p{^Is_Block= 	KAITHI}', "");
Expect(0, 69839, '\P{Is_Block= 	KAITHI}', "");
Expect(1, 69839, '\P{^Is_Block= 	KAITHI}', "");
Expect(0, 69759, '\p{Is_Block= 	KAITHI}', "");
Expect(1, 69759, '\p{^Is_Block= 	KAITHI}', "");
Expect(1, 69759, '\P{Is_Block= 	KAITHI}', "");
Expect(0, 69759, '\P{^Is_Block= 	KAITHI}', "");
Error('\p{Is_Blk: Kaithi:=}');
Error('\P{Is_Blk: Kaithi:=}');
Expect(1, 69839, '\p{Is_Blk=kaithi}', "");
Expect(0, 69839, '\p{^Is_Blk=kaithi}', "");
Expect(0, 69839, '\P{Is_Blk=kaithi}', "");
Expect(1, 69839, '\P{^Is_Blk=kaithi}', "");
Expect(0, 69759, '\p{Is_Blk=kaithi}', "");
Expect(1, 69759, '\p{^Is_Blk=kaithi}', "");
Expect(1, 69759, '\P{Is_Blk=kaithi}', "");
Expect(0, 69759, '\P{^Is_Blk=kaithi}', "");
Expect(1, 69839, '\p{Is_Blk=-_Kaithi}', "");
Expect(0, 69839, '\p{^Is_Blk=-_Kaithi}', "");
Expect(0, 69839, '\P{Is_Blk=-_Kaithi}', "");
Expect(1, 69839, '\P{^Is_Blk=-_Kaithi}', "");
Expect(0, 69759, '\p{Is_Blk=-_Kaithi}', "");
Expect(1, 69759, '\p{^Is_Blk=-_Kaithi}', "");
Expect(1, 69759, '\P{Is_Blk=-_Kaithi}', "");
Expect(0, 69759, '\P{^Is_Blk=-_Kaithi}', "");
Error('\p{Block=_-Sinhala:=}');
Error('\P{Block=_-Sinhala:=}');
Expect(1, 3583, '\p{Block=sinhala}', "");
Expect(0, 3583, '\p{^Block=sinhala}', "");
Expect(0, 3583, '\P{Block=sinhala}', "");
Expect(1, 3583, '\P{^Block=sinhala}', "");
Expect(0, 3455, '\p{Block=sinhala}', "");
Expect(1, 3455, '\p{^Block=sinhala}', "");
Expect(1, 3455, '\P{Block=sinhala}', "");
Expect(0, 3455, '\P{^Block=sinhala}', "");
Expect(1, 3583, '\p{Block=_Sinhala}', "");
Expect(0, 3583, '\p{^Block=_Sinhala}', "");
Expect(0, 3583, '\P{Block=_Sinhala}', "");
Expect(1, 3583, '\P{^Block=_Sinhala}', "");
Expect(0, 3455, '\p{Block=_Sinhala}', "");
Expect(1, 3455, '\p{^Block=_Sinhala}', "");
Expect(1, 3455, '\P{Block=_Sinhala}', "");
Expect(0, 3455, '\P{^Block=_Sinhala}', "");
Error('\p{Blk=--SINHALA/a/}');
Error('\P{Blk=--SINHALA/a/}');
Expect(1, 3583, '\p{Blk=sinhala}', "");
Expect(0, 3583, '\p{^Blk=sinhala}', "");
Expect(0, 3583, '\P{Blk=sinhala}', "");
Expect(1, 3583, '\P{^Blk=sinhala}', "");
Expect(0, 3455, '\p{Blk=sinhala}', "");
Expect(1, 3455, '\p{^Blk=sinhala}', "");
Expect(1, 3455, '\P{Blk=sinhala}', "");
Expect(0, 3455, '\P{^Blk=sinhala}', "");
Expect(1, 3583, '\p{Blk=  Sinhala}', "");
Expect(0, 3583, '\p{^Blk=  Sinhala}', "");
Expect(0, 3583, '\P{Blk=  Sinhala}', "");
Expect(1, 3583, '\P{^Blk=  Sinhala}', "");
Expect(0, 3455, '\p{Blk=  Sinhala}', "");
Expect(1, 3455, '\p{^Blk=  Sinhala}', "");
Expect(1, 3455, '\P{Blk=  Sinhala}', "");
Expect(0, 3455, '\P{^Blk=  Sinhala}', "");
Error('\p{Is_Block:	_	sinhala:=}');
Error('\P{Is_Block:	_	sinhala:=}');
Expect(1, 3583, '\p{Is_Block=sinhala}', "");
Expect(0, 3583, '\p{^Is_Block=sinhala}', "");
Expect(0, 3583, '\P{Is_Block=sinhala}', "");
Expect(1, 3583, '\P{^Is_Block=sinhala}', "");
Expect(0, 3455, '\p{Is_Block=sinhala}', "");
Expect(1, 3455, '\p{^Is_Block=sinhala}', "");
Expect(1, 3455, '\P{Is_Block=sinhala}', "");
Expect(0, 3455, '\P{^Is_Block=sinhala}', "");
Expect(1, 3583, '\p{Is_Block= _sinhala}', "");
Expect(0, 3583, '\p{^Is_Block= _sinhala}', "");
Expect(0, 3583, '\P{Is_Block= _sinhala}', "");
Expect(1, 3583, '\P{^Is_Block= _sinhala}', "");
Expect(0, 3455, '\p{Is_Block= _sinhala}', "");
Expect(1, 3455, '\p{^Is_Block= _sinhala}', "");
Expect(1, 3455, '\P{Is_Block= _sinhala}', "");
Expect(0, 3455, '\P{^Is_Block= _sinhala}', "");
Error('\p{Is_Blk=	Sinhala/a/}');
Error('\P{Is_Blk=	Sinhala/a/}');
Expect(1, 3583, '\p{Is_Blk=sinhala}', "");
Expect(0, 3583, '\p{^Is_Blk=sinhala}', "");
Expect(0, 3583, '\P{Is_Blk=sinhala}', "");
Expect(1, 3583, '\P{^Is_Blk=sinhala}', "");
Expect(0, 3455, '\p{Is_Blk=sinhala}', "");
Expect(1, 3455, '\p{^Is_Blk=sinhala}', "");
Expect(1, 3455, '\P{Is_Blk=sinhala}', "");
Expect(0, 3455, '\P{^Is_Blk=sinhala}', "");
Expect(1, 3583, '\p{Is_Blk=_Sinhala}', "");
Expect(0, 3583, '\p{^Is_Blk=_Sinhala}', "");
Expect(0, 3583, '\P{Is_Blk=_Sinhala}', "");
Expect(1, 3583, '\P{^Is_Blk=_Sinhala}', "");
Expect(0, 3455, '\p{Is_Blk=_Sinhala}', "");
Expect(1, 3455, '\p{^Is_Blk=_Sinhala}', "");
Expect(1, 3455, '\P{Is_Blk=_Sinhala}', "");
Expect(0, 3455, '\P{^Is_Blk=_Sinhala}', "");
Error('\p{Block=	_Glagolitic/a/}');
Error('\P{Block=	_Glagolitic/a/}');
Expect(1, 11359, '\p{Block=glagolitic}', "");
Expect(0, 11359, '\p{^Block=glagolitic}', "");
Expect(0, 11359, '\P{Block=glagolitic}', "");
Expect(1, 11359, '\P{^Block=glagolitic}', "");
Expect(0, 11263, '\p{Block=glagolitic}', "");
Expect(1, 11263, '\p{^Block=glagolitic}', "");
Expect(1, 11263, '\P{Block=glagolitic}', "");
Expect(0, 11263, '\P{^Block=glagolitic}', "");
Expect(1, 11359, '\p{Block=-GLAGOLITIC}', "");
Expect(0, 11359, '\p{^Block=-GLAGOLITIC}', "");
Expect(0, 11359, '\P{Block=-GLAGOLITIC}', "");
Expect(1, 11359, '\P{^Block=-GLAGOLITIC}', "");
Expect(0, 11263, '\p{Block=-GLAGOLITIC}', "");
Expect(1, 11263, '\p{^Block=-GLAGOLITIC}', "");
Expect(1, 11263, '\P{Block=-GLAGOLITIC}', "");
Expect(0, 11263, '\P{^Block=-GLAGOLITIC}', "");
Error('\p{Blk= glagolitic/a/}');
Error('\P{Blk= glagolitic/a/}');
Expect(1, 11359, '\p{Blk=glagolitic}', "");
Expect(0, 11359, '\p{^Blk=glagolitic}', "");
Expect(0, 11359, '\P{Blk=glagolitic}', "");
Expect(1, 11359, '\P{^Blk=glagolitic}', "");
Expect(0, 11263, '\p{Blk=glagolitic}', "");
Expect(1, 11263, '\p{^Blk=glagolitic}', "");
Expect(1, 11263, '\P{Blk=glagolitic}', "");
Expect(0, 11263, '\P{^Blk=glagolitic}', "");
Expect(1, 11359, '\p{Blk:   	-Glagolitic}', "");
Expect(0, 11359, '\p{^Blk:   	-Glagolitic}', "");
Expect(0, 11359, '\P{Blk:   	-Glagolitic}', "");
Expect(1, 11359, '\P{^Blk:   	-Glagolitic}', "");
Expect(0, 11263, '\p{Blk:   	-Glagolitic}', "");
Expect(1, 11263, '\p{^Blk:   	-Glagolitic}', "");
Expect(1, 11263, '\P{Blk:   	-Glagolitic}', "");
Expect(0, 11263, '\P{^Blk:   	-Glagolitic}', "");
Error('\p{Is_Block: :=  glagolitic}');
Error('\P{Is_Block: :=  glagolitic}');
Expect(1, 11359, '\p{Is_Block=glagolitic}', "");
Expect(0, 11359, '\p{^Is_Block=glagolitic}', "");
Expect(0, 11359, '\P{Is_Block=glagolitic}', "");
Expect(1, 11359, '\P{^Is_Block=glagolitic}', "");
Expect(0, 11263, '\p{Is_Block=glagolitic}', "");
Expect(1, 11263, '\p{^Is_Block=glagolitic}', "");
Expect(1, 11263, '\P{Is_Block=glagolitic}', "");
Expect(0, 11263, '\P{^Is_Block=glagolitic}', "");
Expect(1, 11359, '\p{Is_Block=--Glagolitic}', "");
Expect(0, 11359, '\p{^Is_Block=--Glagolitic}', "");
Expect(0, 11359, '\P{Is_Block=--Glagolitic}', "");
Expect(1, 11359, '\P{^Is_Block=--Glagolitic}', "");
Expect(0, 11263, '\p{Is_Block=--Glagolitic}', "");
Expect(1, 11263, '\p{^Is_Block=--Glagolitic}', "");
Expect(1, 11263, '\P{Is_Block=--Glagolitic}', "");
Expect(0, 11263, '\P{^Is_Block=--Glagolitic}', "");
Error('\p{Is_Blk=/a/  GLAGOLITIC}');
Error('\P{Is_Blk=/a/  GLAGOLITIC}');
Expect(1, 11359, '\p{Is_Blk=glagolitic}', "");
Expect(0, 11359, '\p{^Is_Blk=glagolitic}', "");
Expect(0, 11359, '\P{Is_Blk=glagolitic}', "");
Expect(1, 11359, '\P{^Is_Blk=glagolitic}', "");
Expect(0, 11263, '\p{Is_Blk=glagolitic}', "");
Expect(1, 11263, '\p{^Is_Blk=glagolitic}', "");
Expect(1, 11263, '\P{Is_Blk=glagolitic}', "");
Expect(0, 11263, '\P{^Is_Blk=glagolitic}', "");
Expect(1, 11359, '\p{Is_Blk=_ GLAGOLITIC}', "");
Expect(0, 11359, '\p{^Is_Blk=_ GLAGOLITIC}', "");
Expect(0, 11359, '\P{Is_Blk=_ GLAGOLITIC}', "");
Expect(1, 11359, '\P{^Is_Blk=_ GLAGOLITIC}', "");
Expect(0, 11263, '\p{Is_Blk=_ GLAGOLITIC}', "");
Expect(1, 11263, '\p{^Is_Blk=_ GLAGOLITIC}', "");
Expect(1, 11263, '\P{Is_Blk=_ GLAGOLITIC}', "");
Expect(0, 11263, '\P{^Is_Blk=_ GLAGOLITIC}', "");
Error('\p{Block=/a/tagalog}');
Error('\P{Block=/a/tagalog}');
Expect(1, 5919, '\p{Block=tagalog}', "");
Expect(0, 5919, '\p{^Block=tagalog}', "");
Expect(0, 5919, '\P{Block=tagalog}', "");
Expect(1, 5919, '\P{^Block=tagalog}', "");
Expect(0, 5887, '\p{Block=tagalog}', "");
Expect(1, 5887, '\p{^Block=tagalog}', "");
Expect(1, 5887, '\P{Block=tagalog}', "");
Expect(0, 5887, '\P{^Block=tagalog}', "");
Expect(1, 5919, '\p{Block=_ Tagalog}', "");
Expect(0, 5919, '\p{^Block=_ Tagalog}', "");
Expect(0, 5919, '\P{Block=_ Tagalog}', "");
Expect(1, 5919, '\P{^Block=_ Tagalog}', "");
Expect(0, 5887, '\p{Block=_ Tagalog}', "");
Expect(1, 5887, '\p{^Block=_ Tagalog}', "");
Expect(1, 5887, '\P{Block=_ Tagalog}', "");
Expect(0, 5887, '\P{^Block=_ Tagalog}', "");
Error('\p{Blk=_:=TAGALOG}');
Error('\P{Blk=_:=TAGALOG}');
Expect(1, 5919, '\p{Blk=tagalog}', "");
Expect(0, 5919, '\p{^Blk=tagalog}', "");
Expect(0, 5919, '\P{Blk=tagalog}', "");
Expect(1, 5919, '\P{^Blk=tagalog}', "");
Expect(0, 5887, '\p{Blk=tagalog}', "");
Expect(1, 5887, '\p{^Blk=tagalog}', "");
Expect(1, 5887, '\P{Blk=tagalog}', "");
Expect(0, 5887, '\P{^Blk=tagalog}', "");
Expect(1, 5919, '\p{Blk=	tagalog}', "");
Expect(0, 5919, '\p{^Blk=	tagalog}', "");
Expect(0, 5919, '\P{Blk=	tagalog}', "");
Expect(1, 5919, '\P{^Blk=	tagalog}', "");
Expect(0, 5887, '\p{Blk=	tagalog}', "");
Expect(1, 5887, '\p{^Blk=	tagalog}', "");
Expect(1, 5887, '\P{Blk=	tagalog}', "");
Expect(0, 5887, '\P{^Blk=	tagalog}', "");
Error('\p{Is_Block=	/a/TAGALOG}');
Error('\P{Is_Block=	/a/TAGALOG}');
Expect(1, 5919, '\p{Is_Block=tagalog}', "");
Expect(0, 5919, '\p{^Is_Block=tagalog}', "");
Expect(0, 5919, '\P{Is_Block=tagalog}', "");
Expect(1, 5919, '\P{^Is_Block=tagalog}', "");
Expect(0, 5887, '\p{Is_Block=tagalog}', "");
Expect(1, 5887, '\p{^Is_Block=tagalog}', "");
Expect(1, 5887, '\P{Is_Block=tagalog}', "");
Expect(0, 5887, '\P{^Is_Block=tagalog}', "");
Expect(1, 5919, '\p{Is_Block=_ TAGALOG}', "");
Expect(0, 5919, '\p{^Is_Block=_ TAGALOG}', "");
Expect(0, 5919, '\P{Is_Block=_ TAGALOG}', "");
Expect(1, 5919, '\P{^Is_Block=_ TAGALOG}', "");
Expect(0, 5887, '\p{Is_Block=_ TAGALOG}', "");
Expect(1, 5887, '\p{^Is_Block=_ TAGALOG}', "");
Expect(1, 5887, '\P{Is_Block=_ TAGALOG}', "");
Expect(0, 5887, '\P{^Is_Block=_ TAGALOG}', "");
Error('\p{Is_Blk=-/a/Tagalog}');
Error('\P{Is_Blk=-/a/Tagalog}');
Expect(1, 5919, '\p{Is_Blk=tagalog}', "");
Expect(0, 5919, '\p{^Is_Blk=tagalog}', "");
Expect(0, 5919, '\P{Is_Blk=tagalog}', "");
Expect(1, 5919, '\P{^Is_Blk=tagalog}', "");
Expect(0, 5887, '\p{Is_Blk=tagalog}', "");
Expect(1, 5887, '\p{^Is_Blk=tagalog}', "");
Expect(1, 5887, '\P{Is_Blk=tagalog}', "");
Expect(0, 5887, '\P{^Is_Blk=tagalog}', "");
Expect(1, 5919, '\p{Is_Blk:    _tagalog}', "");
Expect(0, 5919, '\p{^Is_Blk:    _tagalog}', "");
Expect(0, 5919, '\P{Is_Blk:    _tagalog}', "");
Expect(1, 5919, '\P{^Is_Blk:    _tagalog}', "");
Expect(0, 5887, '\p{Is_Blk:    _tagalog}', "");
Expect(1, 5887, '\p{^Is_Blk:    _tagalog}', "");
Expect(1, 5887, '\P{Is_Blk:    _tagalog}', "");
Expect(0, 5887, '\P{^Is_Blk:    _tagalog}', "");
Error('\p{Block=:=	_Kharoshthi}');
Error('\P{Block=:=	_Kharoshthi}');
Expect(1, 68191, '\p{Block=kharoshthi}', "");
Expect(0, 68191, '\p{^Block=kharoshthi}', "");
Expect(0, 68191, '\P{Block=kharoshthi}', "");
Expect(1, 68191, '\P{^Block=kharoshthi}', "");
Expect(0, 68095, '\p{Block=kharoshthi}', "");
Expect(1, 68095, '\p{^Block=kharoshthi}', "");
Expect(1, 68095, '\P{Block=kharoshthi}', "");
Expect(0, 68095, '\P{^Block=kharoshthi}', "");
Expect(1, 68191, '\p{Block:__KHAROSHTHI}', "");
Expect(0, 68191, '\p{^Block:__KHAROSHTHI}', "");
Expect(0, 68191, '\P{Block:__KHAROSHTHI}', "");
Expect(1, 68191, '\P{^Block:__KHAROSHTHI}', "");
Expect(0, 68095, '\p{Block:__KHAROSHTHI}', "");
Expect(1, 68095, '\p{^Block:__KHAROSHTHI}', "");
Expect(1, 68095, '\P{Block:__KHAROSHTHI}', "");
Expect(0, 68095, '\P{^Block:__KHAROSHTHI}', "");
Error('\p{Blk=/a/ 	KHAROSHTHI}');
Error('\P{Blk=/a/ 	KHAROSHTHI}');
Expect(1, 68191, '\p{Blk=kharoshthi}', "");
Expect(0, 68191, '\p{^Blk=kharoshthi}', "");
Expect(0, 68191, '\P{Blk=kharoshthi}', "");
Expect(1, 68191, '\P{^Blk=kharoshthi}', "");
Expect(0, 68095, '\p{Blk=kharoshthi}', "");
Expect(1, 68095, '\p{^Blk=kharoshthi}', "");
Expect(1, 68095, '\P{Blk=kharoshthi}', "");
Expect(0, 68095, '\P{^Blk=kharoshthi}', "");
Expect(1, 68191, '\p{Blk= KHAROSHTHI}', "");
Expect(0, 68191, '\p{^Blk= KHAROSHTHI}', "");
Expect(0, 68191, '\P{Blk= KHAROSHTHI}', "");
Expect(1, 68191, '\P{^Blk= KHAROSHTHI}', "");
Expect(0, 68095, '\p{Blk= KHAROSHTHI}', "");
Expect(1, 68095, '\p{^Blk= KHAROSHTHI}', "");
Expect(1, 68095, '\P{Blk= KHAROSHTHI}', "");
Expect(0, 68095, '\P{^Blk= KHAROSHTHI}', "");
Error('\p{Is_Block=/a/	_KHAROSHTHI}');
Error('\P{Is_Block=/a/	_KHAROSHTHI}');
Expect(1, 68191, '\p{Is_Block=kharoshthi}', "");
Expect(0, 68191, '\p{^Is_Block=kharoshthi}', "");
Expect(0, 68191, '\P{Is_Block=kharoshthi}', "");
Expect(1, 68191, '\P{^Is_Block=kharoshthi}', "");
Expect(0, 68095, '\p{Is_Block=kharoshthi}', "");
Expect(1, 68095, '\p{^Is_Block=kharoshthi}', "");
Expect(1, 68095, '\P{Is_Block=kharoshthi}', "");
Expect(0, 68095, '\P{^Is_Block=kharoshthi}', "");
Expect(1, 68191, '\p{Is_Block=_	KHAROSHTHI}', "");
Expect(0, 68191, '\p{^Is_Block=_	KHAROSHTHI}', "");
Expect(0, 68191, '\P{Is_Block=_	KHAROSHTHI}', "");
Expect(1, 68191, '\P{^Is_Block=_	KHAROSHTHI}', "");
Expect(0, 68095, '\p{Is_Block=_	KHAROSHTHI}', "");
Expect(1, 68095, '\p{^Is_Block=_	KHAROSHTHI}', "");
Expect(1, 68095, '\P{Is_Block=_	KHAROSHTHI}', "");
Expect(0, 68095, '\P{^Is_Block=_	KHAROSHTHI}', "");
Error('\p{Is_Blk=:=_-KHAROSHTHI}');
Error('\P{Is_Blk=:=_-KHAROSHTHI}');
Expect(1, 68191, '\p{Is_Blk=kharoshthi}', "");
Expect(0, 68191, '\p{^Is_Blk=kharoshthi}', "");
Expect(0, 68191, '\P{Is_Blk=kharoshthi}', "");
Expect(1, 68191, '\P{^Is_Blk=kharoshthi}', "");
Expect(0, 68095, '\p{Is_Blk=kharoshthi}', "");
Expect(1, 68095, '\p{^Is_Blk=kharoshthi}', "");
Expect(1, 68095, '\P{Is_Blk=kharoshthi}', "");
Expect(0, 68095, '\P{^Is_Blk=kharoshthi}', "");
Expect(1, 68191, '\p{Is_Blk=-Kharoshthi}', "");
Expect(0, 68191, '\p{^Is_Blk=-Kharoshthi}', "");
Expect(0, 68191, '\P{Is_Blk=-Kharoshthi}', "");
Expect(1, 68191, '\P{^Is_Blk=-Kharoshthi}', "");
Expect(0, 68095, '\p{Is_Blk=-Kharoshthi}', "");
Expect(1, 68095, '\p{^Is_Blk=-Kharoshthi}', "");
Expect(1, 68095, '\P{Is_Blk=-Kharoshthi}', "");
Expect(0, 68095, '\P{^Is_Blk=-Kharoshthi}', "");
Error('\p{Block:	:=_Arabic supplement}');
Error('\P{Block:	:=_Arabic supplement}');
Expect(1, 1919, '\p{Block=arabicsupplement}', "");
Expect(0, 1919, '\p{^Block=arabicsupplement}', "");
Expect(0, 1919, '\P{Block=arabicsupplement}', "");
Expect(1, 1919, '\P{^Block=arabicsupplement}', "");
Expect(0, 1871, '\p{Block=arabicsupplement}', "");
Expect(1, 1871, '\p{^Block=arabicsupplement}', "");
Expect(1, 1871, '\P{Block=arabicsupplement}', "");
Expect(0, 1871, '\P{^Block=arabicsupplement}', "");
Expect(1, 1919, '\p{Block=	arabic-SUPPLEMENT}', "");
Expect(0, 1919, '\p{^Block=	arabic-SUPPLEMENT}', "");
Expect(0, 1919, '\P{Block=	arabic-SUPPLEMENT}', "");
Expect(1, 1919, '\P{^Block=	arabic-SUPPLEMENT}', "");
Expect(0, 1871, '\p{Block=	arabic-SUPPLEMENT}', "");
Expect(1, 1871, '\p{^Block=	arabic-SUPPLEMENT}', "");
Expect(1, 1871, '\P{Block=	arabic-SUPPLEMENT}', "");
Expect(0, 1871, '\P{^Block=	arabic-SUPPLEMENT}', "");
Error('\p{Blk=:=ARABIC Supplement}');
Error('\P{Blk=:=ARABIC Supplement}');
Expect(1, 1919, '\p{Blk=arabicsupplement}', "");
Expect(0, 1919, '\p{^Blk=arabicsupplement}', "");
Expect(0, 1919, '\P{Blk=arabicsupplement}', "");
Expect(1, 1919, '\P{^Blk=arabicsupplement}', "");
Expect(0, 1871, '\p{Blk=arabicsupplement}', "");
Expect(1, 1871, '\p{^Blk=arabicsupplement}', "");
Expect(1, 1871, '\P{Blk=arabicsupplement}', "");
Expect(0, 1871, '\P{^Blk=arabicsupplement}', "");
Expect(1, 1919, '\p{Blk:  _arabic_Supplement}', "");
Expect(0, 1919, '\p{^Blk:  _arabic_Supplement}', "");
Expect(0, 1919, '\P{Blk:  _arabic_Supplement}', "");
Expect(1, 1919, '\P{^Blk:  _arabic_Supplement}', "");
Expect(0, 1871, '\p{Blk:  _arabic_Supplement}', "");
Expect(1, 1871, '\p{^Blk:  _arabic_Supplement}', "");
Expect(1, 1871, '\P{Blk:  _arabic_Supplement}', "");
Expect(0, 1871, '\P{^Blk:  _arabic_Supplement}', "");
Error('\p{Is_Block=Arabic supplement:=}');
Error('\P{Is_Block=Arabic supplement:=}');
Expect(1, 1919, '\p{Is_Block:arabicsupplement}', "");
Expect(0, 1919, '\p{^Is_Block:arabicsupplement}', "");
Expect(0, 1919, '\P{Is_Block:arabicsupplement}', "");
Expect(1, 1919, '\P{^Is_Block:arabicsupplement}', "");
Expect(0, 1871, '\p{Is_Block:arabicsupplement}', "");
Expect(1, 1871, '\p{^Is_Block:arabicsupplement}', "");
Expect(1, 1871, '\P{Is_Block:arabicsupplement}', "");
Expect(0, 1871, '\P{^Is_Block:arabicsupplement}', "");
Expect(1, 1919, '\p{Is_Block=-_arabic supplement}', "");
Expect(0, 1919, '\p{^Is_Block=-_arabic supplement}', "");
Expect(0, 1919, '\P{Is_Block=-_arabic supplement}', "");
Expect(1, 1919, '\P{^Is_Block=-_arabic supplement}', "");
Expect(0, 1871, '\p{Is_Block=-_arabic supplement}', "");
Expect(1, 1871, '\p{^Is_Block=-_arabic supplement}', "");
Expect(1, 1871, '\P{Is_Block=-_arabic supplement}', "");
Expect(0, 1871, '\P{^Is_Block=-_arabic supplement}', "");
Error('\p{Is_Blk=-	ARABIC	SUPPLEMENT/a/}');
Error('\P{Is_Blk=-	ARABIC	SUPPLEMENT/a/}');
Expect(1, 1919, '\p{Is_Blk=arabicsupplement}', "");
Expect(0, 1919, '\p{^Is_Blk=arabicsupplement}', "");
Expect(0, 1919, '\P{Is_Blk=arabicsupplement}', "");
Expect(1, 1919, '\P{^Is_Blk=arabicsupplement}', "");
Expect(0, 1871, '\p{Is_Blk=arabicsupplement}', "");
Expect(1, 1871, '\p{^Is_Blk=arabicsupplement}', "");
Expect(1, 1871, '\P{Is_Blk=arabicsupplement}', "");
Expect(0, 1871, '\P{^Is_Blk=arabicsupplement}', "");
Expect(1, 1919, '\p{Is_Blk=- Arabic supplement}', "");
Expect(0, 1919, '\p{^Is_Blk=- Arabic supplement}', "");
Expect(0, 1919, '\P{Is_Blk=- Arabic supplement}', "");
Expect(1, 1919, '\P{^Is_Blk=- Arabic supplement}', "");
Expect(0, 1871, '\p{Is_Blk=- Arabic supplement}', "");
Expect(1, 1871, '\p{^Is_Blk=- Arabic supplement}', "");
Expect(1, 1871, '\P{Is_Blk=- Arabic supplement}', "");
Expect(0, 1871, '\P{^Is_Blk=- Arabic supplement}', "");
Error('\p{Block=-	Cuneiform_NUMBERS/a/And Punctuation}');
Error('\P{Block=-	Cuneiform_NUMBERS/a/And Punctuation}');
Expect(1, 74879, '\p{Block=cuneiformnumbersandpunctuation}', "");
Expect(0, 74879, '\p{^Block=cuneiformnumbersandpunctuation}', "");
Expect(0, 74879, '\P{Block=cuneiformnumbersandpunctuation}', "");
Expect(1, 74879, '\P{^Block=cuneiformnumbersandpunctuation}', "");
Expect(0, 74751, '\p{Block=cuneiformnumbersandpunctuation}', "");
Expect(1, 74751, '\p{^Block=cuneiformnumbersandpunctuation}', "");
Expect(1, 74751, '\P{Block=cuneiformnumbersandpunctuation}', "");
Expect(0, 74751, '\P{^Block=cuneiformnumbersandpunctuation}', "");
Expect(1, 74879, '\p{Block=	 Cuneiform	Numbers	And_Punctuation}', "");
Expect(0, 74879, '\p{^Block=	 Cuneiform	Numbers	And_Punctuation}', "");
Expect(0, 74879, '\P{Block=	 Cuneiform	Numbers	And_Punctuation}', "");
Expect(1, 74879, '\P{^Block=	 Cuneiform	Numbers	And_Punctuation}', "");
Expect(0, 74751, '\p{Block=	 Cuneiform	Numbers	And_Punctuation}', "");
Expect(1, 74751, '\p{^Block=	 Cuneiform	Numbers	And_Punctuation}', "");
Expect(1, 74751, '\P{Block=	 Cuneiform	Numbers	And_Punctuation}', "");
Expect(0, 74751, '\P{^Block=	 Cuneiform	Numbers	And_Punctuation}', "");
Error('\p{Blk=	-CUNEIFORM:=Numbers	AND	Punctuation}');
Error('\P{Blk=	-CUNEIFORM:=Numbers	AND	Punctuation}');
Expect(1, 74879, '\p{Blk=cuneiformnumbersandpunctuation}', "");
Expect(0, 74879, '\p{^Blk=cuneiformnumbersandpunctuation}', "");
Expect(0, 74879, '\P{Blk=cuneiformnumbersandpunctuation}', "");
Expect(1, 74879, '\P{^Blk=cuneiformnumbersandpunctuation}', "");
Expect(0, 74751, '\p{Blk=cuneiformnumbersandpunctuation}', "");
Expect(1, 74751, '\p{^Blk=cuneiformnumbersandpunctuation}', "");
Expect(1, 74751, '\P{Blk=cuneiformnumbersandpunctuation}', "");
Expect(0, 74751, '\P{^Blk=cuneiformnumbersandpunctuation}', "");
Expect(1, 74879, '\p{Blk=__Cuneiform	NumbersAnd_Punctuation}', "");
Expect(0, 74879, '\p{^Blk=__Cuneiform	NumbersAnd_Punctuation}', "");
Expect(0, 74879, '\P{Blk=__Cuneiform	NumbersAnd_Punctuation}', "");
Expect(1, 74879, '\P{^Blk=__Cuneiform	NumbersAnd_Punctuation}', "");
Expect(0, 74751, '\p{Blk=__Cuneiform	NumbersAnd_Punctuation}', "");
Expect(1, 74751, '\p{^Blk=__Cuneiform	NumbersAnd_Punctuation}', "");
Expect(1, 74751, '\P{Blk=__Cuneiform	NumbersAnd_Punctuation}', "");
Expect(0, 74751, '\P{^Blk=__Cuneiform	NumbersAnd_Punctuation}', "");
Error('\p{Is_Block=- cuneiform Numbers_and_PUNCTUATION:=}');
Error('\P{Is_Block=- cuneiform Numbers_and_PUNCTUATION:=}');
Expect(1, 74879, '\p{Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(0, 74879, '\p{^Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(0, 74879, '\P{Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(1, 74879, '\P{^Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(0, 74751, '\p{Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(1, 74751, '\p{^Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(1, 74751, '\P{Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(0, 74751, '\P{^Is_Block=cuneiformnumbersandpunctuation}', "");
Expect(1, 74879, '\p{Is_Block: 	-Cuneiform-Numbers-and	PUNCTUATION}', "");
Expect(0, 74879, '\p{^Is_Block: 	-Cuneiform-Numbers-and	PUNCTUATION}', "");
Expect(0, 74879, '\P{Is_Block: 	-Cuneiform-Numbers-and	PUNCTUATION}', "");
Expect(1, 74879, '\P{^Is_Block: 	-Cuneiform-Numbers-and	PUNCTUATION}', "");
Expect(0, 74751, '\p{Is_Block: 	-Cuneiform-Numbers-and	PUNCTUATION}', "");
Expect(1, 74751, '\p{^Is_Block: 	-Cuneiform-Numbers-and	PUNCTUATION}', "");
Expect(1, 74751, '\P{Is_Block: 	-Cuneiform-Numbers-and	PUNCTUATION}', "");
Expect(0, 74751, '\P{^Is_Block: 	-Cuneiform-Numbers-and	PUNCTUATION}', "");
Error('\p{Is_Blk=-:=Cuneiform_numbers	and-PUNCTUATION}');
Error('\P{Is_Blk=-:=Cuneiform_numbers	and-PUNCTUATION}');
Expect(1, 74879, '\p{Is_Blk: cuneiformnumbersandpunctuation}', "");
Expect(0, 74879, '\p{^Is_Blk: cuneiformnumbersandpunctuation}', "");
Expect(0, 74879, '\P{Is_Blk: cuneiformnumbersandpunctuation}', "");
Expect(1, 74879, '\P{^Is_Blk: cuneiformnumbersandpunctuation}', "");
Expect(0, 74751, '\p{Is_Blk: cuneiformnumbersandpunctuation}', "");
Expect(1, 74751, '\p{^Is_Blk: cuneiformnumbersandpunctuation}', "");
Expect(1, 74751, '\P{Is_Blk: cuneiformnumbersandpunctuation}', "");
Expect(0, 74751, '\P{^Is_Blk: cuneiformnumbersandpunctuation}', "");
Expect(1, 74879, '\p{Is_Blk= _CuneiformNumbersAndPunctuation}', "");
Expect(0, 74879, '\p{^Is_Blk= _CuneiformNumbersAndPunctuation}', "");
Expect(0, 74879, '\P{Is_Blk= _CuneiformNumbersAndPunctuation}', "");
Expect(1, 74879, '\P{^Is_Blk= _CuneiformNumbersAndPunctuation}', "");
Expect(0, 74751, '\p{Is_Blk= _CuneiformNumbersAndPunctuation}', "");
Expect(1, 74751, '\p{^Is_Blk= _CuneiformNumbersAndPunctuation}', "");
Expect(1, 74751, '\P{Is_Blk= _CuneiformNumbersAndPunctuation}', "");
Expect(0, 74751, '\P{^Is_Blk= _CuneiformNumbersAndPunctuation}', "");
Error('\p{Block=-/a/DevanagariExtended}');
Error('\P{Block=-/a/DevanagariExtended}');
Expect(1, 43263, '\p{Block=devanagariextended}', "");
Expect(0, 43263, '\p{^Block=devanagariextended}', "");
Expect(0, 43263, '\P{Block=devanagariextended}', "");
Expect(1, 43263, '\P{^Block=devanagariextended}', "");
Expect(0, 43231, '\p{Block=devanagariextended}', "");
Expect(1, 43231, '\p{^Block=devanagariextended}', "");
Expect(1, 43231, '\P{Block=devanagariextended}', "");
Expect(0, 43231, '\P{^Block=devanagariextended}', "");
Expect(1, 43263, '\p{Block=_DevanagariExtended}', "");
Expect(0, 43263, '\p{^Block=_DevanagariExtended}', "");
Expect(0, 43263, '\P{Block=_DevanagariExtended}', "");
Expect(1, 43263, '\P{^Block=_DevanagariExtended}', "");
Expect(0, 43231, '\p{Block=_DevanagariExtended}', "");
Expect(1, 43231, '\p{^Block=_DevanagariExtended}', "");
Expect(1, 43231, '\P{Block=_DevanagariExtended}', "");
Expect(0, 43231, '\P{^Block=_DevanagariExtended}', "");
Error('\p{Blk=/a/	Devanagari-Extended}');
Error('\P{Blk=/a/	Devanagari-Extended}');
Expect(1, 43263, '\p{Blk=devanagariextended}', "");
Expect(0, 43263, '\p{^Blk=devanagariextended}', "");
Expect(0, 43263, '\P{Blk=devanagariextended}', "");
Expect(1, 43263, '\P{^Blk=devanagariextended}', "");
Expect(0, 43231, '\p{Blk=devanagariextended}', "");
Expect(1, 43231, '\p{^Blk=devanagariextended}', "");
Expect(1, 43231, '\P{Blk=devanagariextended}', "");
Expect(0, 43231, '\P{^Blk=devanagariextended}', "");
Expect(1, 43263, '\p{Blk=--Devanagari_Extended}', "");
Expect(0, 43263, '\p{^Blk=--Devanagari_Extended}', "");
Expect(0, 43263, '\P{Blk=--Devanagari_Extended}', "");
Expect(1, 43263, '\P{^Blk=--Devanagari_Extended}', "");
Expect(0, 43231, '\p{Blk=--Devanagari_Extended}', "");
Expect(1, 43231, '\p{^Blk=--Devanagari_Extended}', "");
Expect(1, 43231, '\P{Blk=--Devanagari_Extended}', "");
Expect(0, 43231, '\P{^Blk=--Devanagari_Extended}', "");
Error('\p{Is_Block=:=-	Devanagari	Extended}');
Error('\P{Is_Block=:=-	Devanagari	Extended}');
Expect(1, 43263, '\p{Is_Block=devanagariextended}', "");
Expect(0, 43263, '\p{^Is_Block=devanagariextended}', "");
Expect(0, 43263, '\P{Is_Block=devanagariextended}', "");
Expect(1, 43263, '\P{^Is_Block=devanagariextended}', "");
Expect(0, 43231, '\p{Is_Block=devanagariextended}', "");
Expect(1, 43231, '\p{^Is_Block=devanagariextended}', "");
Expect(1, 43231, '\P{Is_Block=devanagariextended}', "");
Expect(0, 43231, '\P{^Is_Block=devanagariextended}', "");
Expect(1, 43263, '\p{Is_Block=_-Devanagari-extended}', "");
Expect(0, 43263, '\p{^Is_Block=_-Devanagari-extended}', "");
Expect(0, 43263, '\P{Is_Block=_-Devanagari-extended}', "");
Expect(1, 43263, '\P{^Is_Block=_-Devanagari-extended}', "");
Expect(0, 43231, '\p{Is_Block=_-Devanagari-extended}', "");
Expect(1, 43231, '\p{^Is_Block=_-Devanagari-extended}', "");
Expect(1, 43231, '\P{Is_Block=_-Devanagari-extended}', "");
Expect(0, 43231, '\P{^Is_Block=_-Devanagari-extended}', "");
Error('\p{Is_Blk=/a/	 Devanagari_EXTENDED}');
Error('\P{Is_Blk=/a/	 Devanagari_EXTENDED}');
Expect(1, 43263, '\p{Is_Blk=devanagariextended}', "");
Expect(0, 43263, '\p{^Is_Blk=devanagariextended}', "");
Expect(0, 43263, '\P{Is_Blk=devanagariextended}', "");
Expect(1, 43263, '\P{^Is_Blk=devanagariextended}', "");
Expect(0, 43231, '\p{Is_Blk=devanagariextended}', "");
Expect(1, 43231, '\p{^Is_Blk=devanagariextended}', "");
Expect(1, 43231, '\P{Is_Blk=devanagariextended}', "");
Expect(0, 43231, '\P{^Is_Blk=devanagariextended}', "");
Expect(1, 43263, '\p{Is_Blk=-	Devanagari	extended}', "");
Expect(0, 43263, '\p{^Is_Blk=-	Devanagari	extended}', "");
Expect(0, 43263, '\P{Is_Blk=-	Devanagari	extended}', "");
Expect(1, 43263, '\P{^Is_Blk=-	Devanagari	extended}', "");
Expect(0, 43231, '\p{Is_Blk=-	Devanagari	extended}', "");
Expect(1, 43231, '\p{^Is_Blk=-	Devanagari	extended}', "");
Expect(1, 43231, '\P{Is_Blk=-	Devanagari	extended}', "");
Expect(0, 43231, '\P{^Is_Blk=-	Devanagari	extended}', "");
Error('\p{Block=/a/	-yi_SYLLABLES}');
Error('\P{Block=/a/	-yi_SYLLABLES}');
Expect(1, 42127, '\p{Block=yisyllables}', "");
Expect(0, 42127, '\p{^Block=yisyllables}', "");
Expect(0, 42127, '\P{Block=yisyllables}', "");
Expect(1, 42127, '\P{^Block=yisyllables}', "");
Expect(0, 40959, '\p{Block=yisyllables}', "");
Expect(1, 40959, '\p{^Block=yisyllables}', "");
Expect(1, 40959, '\P{Block=yisyllables}', "");
Expect(0, 40959, '\P{^Block=yisyllables}', "");
Expect(1, 42127, '\p{Block=__Yi SYLLABLES}', "");
Expect(0, 42127, '\p{^Block=__Yi SYLLABLES}', "");
Expect(0, 42127, '\P{Block=__Yi SYLLABLES}', "");
Expect(1, 42127, '\P{^Block=__Yi SYLLABLES}', "");
Expect(0, 40959, '\p{Block=__Yi SYLLABLES}', "");
Expect(1, 40959, '\p{^Block=__Yi SYLLABLES}', "");
Expect(1, 40959, '\P{Block=__Yi SYLLABLES}', "");
Expect(0, 40959, '\P{^Block=__Yi SYLLABLES}', "");
Error('\p{Blk:    Yi/a/syllables}');
Error('\P{Blk:    Yi/a/syllables}');
Expect(1, 42127, '\p{Blk:   yisyllables}', "");
Expect(0, 42127, '\p{^Blk:   yisyllables}', "");
Expect(0, 42127, '\P{Blk:   yisyllables}', "");
Expect(1, 42127, '\P{^Blk:   yisyllables}', "");
Expect(0, 40959, '\p{Blk:   yisyllables}', "");
Expect(1, 40959, '\p{^Blk:   yisyllables}', "");
Expect(1, 40959, '\P{Blk:   yisyllables}', "");
Expect(0, 40959, '\P{^Blk:   yisyllables}', "");
Expect(1, 42127, '\p{Blk=Yi_Syllables}', "");
Expect(0, 42127, '\p{^Blk=Yi_Syllables}', "");
Expect(0, 42127, '\P{Blk=Yi_Syllables}', "");
Expect(1, 42127, '\P{^Blk=Yi_Syllables}', "");
Expect(0, 40959, '\p{Blk=Yi_Syllables}', "");
Expect(1, 40959, '\p{^Blk=Yi_Syllables}', "");
Expect(1, 40959, '\P{Blk=Yi_Syllables}', "");
Expect(0, 40959, '\P{^Blk=Yi_Syllables}', "");
Error('\p{Is_Block=/a/_YI-SYLLABLES}');
Error('\P{Is_Block=/a/_YI-SYLLABLES}');
Expect(1, 42127, '\p{Is_Block=yisyllables}', "");
Expect(0, 42127, '\p{^Is_Block=yisyllables}', "");
Expect(0, 42127, '\P{Is_Block=yisyllables}', "");
Expect(1, 42127, '\P{^Is_Block=yisyllables}', "");
Expect(0, 40959, '\p{Is_Block=yisyllables}', "");
Expect(1, 40959, '\p{^Is_Block=yisyllables}', "");
Expect(1, 40959, '\P{Is_Block=yisyllables}', "");
Expect(0, 40959, '\P{^Is_Block=yisyllables}', "");
Expect(1, 42127, '\p{Is_Block:   		YiSyllables}', "");
Expect(0, 42127, '\p{^Is_Block:   		YiSyllables}', "");
Expect(0, 42127, '\P{Is_Block:   		YiSyllables}', "");
Expect(1, 42127, '\P{^Is_Block:   		YiSyllables}', "");
Expect(0, 40959, '\p{Is_Block:   		YiSyllables}', "");
Expect(1, 40959, '\p{^Is_Block:   		YiSyllables}', "");
Expect(1, 40959, '\P{Is_Block:   		YiSyllables}', "");
Expect(0, 40959, '\P{^Is_Block:   		YiSyllables}', "");
Error('\p{Is_Blk=		YiSyllables/a/}');
Error('\P{Is_Blk=		YiSyllables/a/}');
Expect(1, 42127, '\p{Is_Blk=yisyllables}', "");
Expect(0, 42127, '\p{^Is_Blk=yisyllables}', "");
Expect(0, 42127, '\P{Is_Blk=yisyllables}', "");
Expect(1, 42127, '\P{^Is_Blk=yisyllables}', "");
Expect(0, 40959, '\p{Is_Blk=yisyllables}', "");
Expect(1, 40959, '\p{^Is_Blk=yisyllables}', "");
Expect(1, 40959, '\P{Is_Blk=yisyllables}', "");
Expect(0, 40959, '\P{^Is_Blk=yisyllables}', "");
Expect(1, 42127, '\p{Is_Blk=_-Yi SYLLABLES}', "");
Expect(0, 42127, '\p{^Is_Blk=_-Yi SYLLABLES}', "");
Expect(0, 42127, '\P{Is_Blk=_-Yi SYLLABLES}', "");
Expect(1, 42127, '\P{^Is_Blk=_-Yi SYLLABLES}', "");
Expect(0, 40959, '\p{Is_Blk=_-Yi SYLLABLES}', "");
Expect(1, 40959, '\p{^Is_Blk=_-Yi SYLLABLES}', "");
Expect(1, 40959, '\P{Is_Blk=_-Yi SYLLABLES}', "");
Expect(0, 40959, '\P{^Is_Blk=_-Yi SYLLABLES}', "");
Error('\p{Block=-/a/CombiningHalf-Marks}');
Error('\P{Block=-/a/CombiningHalf-Marks}');
Expect(1, 65071, '\p{Block=combininghalfmarks}', "");
Expect(0, 65071, '\p{^Block=combininghalfmarks}', "");
Expect(0, 65071, '\P{Block=combininghalfmarks}', "");
Expect(1, 65071, '\P{^Block=combininghalfmarks}', "");
Expect(0, 65055, '\p{Block=combininghalfmarks}', "");
Expect(1, 65055, '\p{^Block=combininghalfmarks}', "");
Expect(1, 65055, '\P{Block=combininghalfmarks}', "");
Expect(0, 65055, '\P{^Block=combininghalfmarks}', "");
Expect(1, 65071, '\p{Block=		COMBININGHalf_Marks}', "");
Expect(0, 65071, '\p{^Block=		COMBININGHalf_Marks}', "");
Expect(0, 65071, '\P{Block=		COMBININGHalf_Marks}', "");
Expect(1, 65071, '\P{^Block=		COMBININGHalf_Marks}', "");
Expect(0, 65055, '\p{Block=		COMBININGHalf_Marks}', "");
Expect(1, 65055, '\p{^Block=		COMBININGHalf_Marks}', "");
Expect(1, 65055, '\P{Block=		COMBININGHalf_Marks}', "");
Expect(0, 65055, '\P{^Block=		COMBININGHalf_Marks}', "");
Error('\p{Blk= Combining	Half/a/marks}');
Error('\P{Blk= Combining	Half/a/marks}');
Expect(1, 65071, '\p{Blk=combininghalfmarks}', "");
Expect(0, 65071, '\p{^Blk=combininghalfmarks}', "");
Expect(0, 65071, '\P{Blk=combininghalfmarks}', "");
Expect(1, 65071, '\P{^Blk=combininghalfmarks}', "");
Expect(0, 65055, '\p{Blk=combininghalfmarks}', "");
Expect(1, 65055, '\p{^Blk=combininghalfmarks}', "");
Expect(1, 65055, '\P{Blk=combininghalfmarks}', "");
Expect(0, 65055, '\P{^Blk=combininghalfmarks}', "");
Expect(1, 65071, '\p{Blk:-COMBININGHALF-Marks}', "");
Expect(0, 65071, '\p{^Blk:-COMBININGHALF-Marks}', "");
Expect(0, 65071, '\P{Blk:-COMBININGHALF-Marks}', "");
Expect(1, 65071, '\P{^Blk:-COMBININGHALF-Marks}', "");
Expect(0, 65055, '\p{Blk:-COMBININGHALF-Marks}', "");
Expect(1, 65055, '\p{^Blk:-COMBININGHALF-Marks}', "");
Expect(1, 65055, '\P{Blk:-COMBININGHALF-Marks}', "");
Expect(0, 65055, '\P{^Blk:-COMBININGHALF-Marks}', "");
Error('\p{Is_Block= 	combining/a/HALF_Marks}');
Error('\P{Is_Block= 	combining/a/HALF_Marks}');
Expect(1, 65071, '\p{Is_Block=combininghalfmarks}', "");
Expect(0, 65071, '\p{^Is_Block=combininghalfmarks}', "");
Expect(0, 65071, '\P{Is_Block=combininghalfmarks}', "");
Expect(1, 65071, '\P{^Is_Block=combininghalfmarks}', "");
Expect(0, 65055, '\p{Is_Block=combininghalfmarks}', "");
Expect(1, 65055, '\p{^Is_Block=combininghalfmarks}', "");
Expect(1, 65055, '\P{Is_Block=combininghalfmarks}', "");
Expect(0, 65055, '\P{^Is_Block=combininghalfmarks}', "");
Expect(1, 65071, '\p{Is_Block=	-combining Half Marks}', "");
Expect(0, 65071, '\p{^Is_Block=	-combining Half Marks}', "");
Expect(0, 65071, '\P{Is_Block=	-combining Half Marks}', "");
Expect(1, 65071, '\P{^Is_Block=	-combining Half Marks}', "");
Expect(0, 65055, '\p{Is_Block=	-combining Half Marks}', "");
Expect(1, 65055, '\p{^Is_Block=	-combining Half Marks}', "");
Expect(1, 65055, '\P{Is_Block=	-combining Half Marks}', "");
Expect(0, 65055, '\P{^Is_Block=	-combining Half Marks}', "");
Error('\p{Is_Blk=Combining/a/Half marks}');
Error('\P{Is_Blk=Combining/a/Half marks}');
Expect(1, 65071, '\p{Is_Blk=combininghalfmarks}', "");
Expect(0, 65071, '\p{^Is_Blk=combininghalfmarks}', "");
Expect(0, 65071, '\P{Is_Blk=combininghalfmarks}', "");
Expect(1, 65071, '\P{^Is_Blk=combininghalfmarks}', "");
Expect(0, 65055, '\p{Is_Blk=combininghalfmarks}', "");
Expect(1, 65055, '\p{^Is_Blk=combininghalfmarks}', "");
Expect(1, 65055, '\P{Is_Blk=combininghalfmarks}', "");
Expect(0, 65055, '\P{^Is_Blk=combininghalfmarks}', "");
Expect(1, 65071, '\p{Is_Blk= 	COMBINING-Half Marks}', "");
Expect(0, 65071, '\p{^Is_Blk= 	COMBINING-Half Marks}', "");
Expect(0, 65071, '\P{Is_Blk= 	COMBINING-Half Marks}', "");
Expect(1, 65071, '\P{^Is_Blk= 	COMBINING-Half Marks}', "");
Expect(0, 65055, '\p{Is_Blk= 	COMBINING-Half Marks}', "");
Expect(1, 65055, '\p{^Is_Blk= 	COMBINING-Half Marks}', "");
Expect(1, 65055, '\P{Is_Blk= 	COMBINING-Half Marks}', "");
Expect(0, 65055, '\P{^Is_Blk= 	COMBINING-Half Marks}', "");
Error('\p{Block=-:=miscellaneous_SYMBOLSand	arrows}');
Error('\P{Block=-:=miscellaneous_SYMBOLSand	arrows}');
Expect(1, 11263, '\p{Block=miscellaneoussymbolsandarrows}', "");
Expect(0, 11263, '\p{^Block=miscellaneoussymbolsandarrows}', "");
Expect(0, 11263, '\P{Block=miscellaneoussymbolsandarrows}', "");
Expect(1, 11263, '\P{^Block=miscellaneoussymbolsandarrows}', "");
Expect(0, 11007, '\p{Block=miscellaneoussymbolsandarrows}', "");
Expect(1, 11007, '\p{^Block=miscellaneoussymbolsandarrows}', "");
Expect(1, 11007, '\P{Block=miscellaneoussymbolsandarrows}', "");
Expect(0, 11007, '\P{^Block=miscellaneoussymbolsandarrows}', "");
Expect(1, 11263, '\p{Block=	_Miscellaneous	SYMBOLS-And ARROWS}', "");
Expect(0, 11263, '\p{^Block=	_Miscellaneous	SYMBOLS-And ARROWS}', "");
Expect(0, 11263, '\P{Block=	_Miscellaneous	SYMBOLS-And ARROWS}', "");
Expect(1, 11263, '\P{^Block=	_Miscellaneous	SYMBOLS-And ARROWS}', "");
Expect(0, 11007, '\p{Block=	_Miscellaneous	SYMBOLS-And ARROWS}', "");
Expect(1, 11007, '\p{^Block=	_Miscellaneous	SYMBOLS-And ARROWS}', "");
Expect(1, 11007, '\P{Block=	_Miscellaneous	SYMBOLS-And ARROWS}', "");
Expect(0, 11007, '\P{^Block=	_Miscellaneous	SYMBOLS-And ARROWS}', "");
Error('\p{Blk=	 Miscellaneous	SYMBOLS:=AndArrows}');
Error('\P{Blk=	 Miscellaneous	SYMBOLS:=AndArrows}');
Expect(1, 11263, '\p{Blk: miscellaneoussymbolsandarrows}', "");
Expect(0, 11263, '\p{^Blk: miscellaneoussymbolsandarrows}', "");
Expect(0, 11263, '\P{Blk: miscellaneoussymbolsandarrows}', "");
Expect(1, 11263, '\P{^Blk: miscellaneoussymbolsandarrows}', "");
Expect(0, 11007, '\p{Blk: miscellaneoussymbolsandarrows}', "");
Expect(1, 11007, '\p{^Blk: miscellaneoussymbolsandarrows}', "");
Expect(1, 11007, '\P{Blk: miscellaneoussymbolsandarrows}', "");
Expect(0, 11007, '\P{^Blk: miscellaneoussymbolsandarrows}', "");
Expect(1, 11263, '\p{Blk=		MISCELLANEOUS-symbols	And_arrows}', "");
Expect(0, 11263, '\p{^Blk=		MISCELLANEOUS-symbols	And_arrows}', "");
Expect(0, 11263, '\P{Blk=		MISCELLANEOUS-symbols	And_arrows}', "");
Expect(1, 11263, '\P{^Blk=		MISCELLANEOUS-symbols	And_arrows}', "");
Expect(0, 11007, '\p{Blk=		MISCELLANEOUS-symbols	And_arrows}', "");
Expect(1, 11007, '\p{^Blk=		MISCELLANEOUS-symbols	And_arrows}', "");
Expect(1, 11007, '\P{Blk=		MISCELLANEOUS-symbols	And_arrows}', "");
Expect(0, 11007, '\P{^Blk=		MISCELLANEOUS-symbols	And_arrows}', "");
Error('\p{Is_Block= /a/MISCELLANEOUS-SymbolsAnd_ARROWS}');
Error('\P{Is_Block= /a/MISCELLANEOUS-SymbolsAnd_ARROWS}');
Expect(1, 11263, '\p{Is_Block=miscellaneoussymbolsandarrows}', "");
Expect(0, 11263, '\p{^Is_Block=miscellaneoussymbolsandarrows}', "");
Expect(0, 11263, '\P{Is_Block=miscellaneoussymbolsandarrows}', "");
Expect(1, 11263, '\P{^Is_Block=miscellaneoussymbolsandarrows}', "");
Expect(0, 11007, '\p{Is_Block=miscellaneoussymbolsandarrows}', "");
Expect(1, 11007, '\p{^Is_Block=miscellaneoussymbolsandarrows}', "");
Expect(1, 11007, '\P{Is_Block=miscellaneoussymbolsandarrows}', "");
Expect(0, 11007, '\P{^Is_Block=miscellaneoussymbolsandarrows}', "");
Expect(1, 11263, '\p{Is_Block:-MISCELLANEOUS symbolsAnd	ARROWS}', "");
Expect(0, 11263, '\p{^Is_Block:-MISCELLANEOUS symbolsAnd	ARROWS}', "");
Expect(0, 11263, '\P{Is_Block:-MISCELLANEOUS symbolsAnd	ARROWS}', "");
Expect(1, 11263, '\P{^Is_Block:-MISCELLANEOUS symbolsAnd	ARROWS}', "");
Expect(0, 11007, '\p{Is_Block:-MISCELLANEOUS symbolsAnd	ARROWS}', "");
Expect(1, 11007, '\p{^Is_Block:-MISCELLANEOUS symbolsAnd	ARROWS}', "");
Expect(1, 11007, '\P{Is_Block:-MISCELLANEOUS symbolsAnd	ARROWS}', "");
Expect(0, 11007, '\P{^Is_Block:-MISCELLANEOUS symbolsAnd	ARROWS}', "");
Error('\p{Is_Blk=  miscellaneous Symbols and/a/Arrows}');
Error('\P{Is_Blk=  miscellaneous Symbols and/a/Arrows}');
Expect(1, 11263, '\p{Is_Blk=miscellaneoussymbolsandarrows}', "");
Expect(0, 11263, '\p{^Is_Blk=miscellaneoussymbolsandarrows}', "");
Expect(0, 11263, '\P{Is_Blk=miscellaneoussymbolsandarrows}', "");
Expect(1, 11263, '\P{^Is_Blk=miscellaneoussymbolsandarrows}', "");
Expect(0, 11007, '\p{Is_Blk=miscellaneoussymbolsandarrows}', "");
Expect(1, 11007, '\p{^Is_Blk=miscellaneoussymbolsandarrows}', "");
Expect(1, 11007, '\P{Is_Blk=miscellaneoussymbolsandarrows}', "");
Expect(0, 11007, '\P{^Is_Blk=miscellaneoussymbolsandarrows}', "");
Expect(1, 11263, '\p{Is_Blk= miscellaneous Symbols	And	Arrows}', "");
Expect(0, 11263, '\p{^Is_Blk= miscellaneous Symbols	And	Arrows}', "");
Expect(0, 11263, '\P{Is_Blk= miscellaneous Symbols	And	Arrows}', "");
Expect(1, 11263, '\P{^Is_Blk= miscellaneous Symbols	And	Arrows}', "");
Expect(0, 11007, '\p{Is_Blk= miscellaneous Symbols	And	Arrows}', "");
Expect(1, 11007, '\p{^Is_Blk= miscellaneous Symbols	And	Arrows}', "");
Expect(1, 11007, '\P{Is_Blk= miscellaneous Symbols	And	Arrows}', "");
Expect(0, 11007, '\P{^Is_Blk= miscellaneous Symbols	And	Arrows}', "");
Error('\p{Block:-	halfwidth-and:=FULLWIDTH	Forms}');
Error('\P{Block:-	halfwidth-and:=FULLWIDTH	Forms}');
Expect(1, 65519, '\p{Block=halfwidthandfullwidthforms}', "");
Expect(0, 65519, '\p{^Block=halfwidthandfullwidthforms}', "");
Expect(0, 65519, '\P{Block=halfwidthandfullwidthforms}', "");
Expect(1, 65519, '\P{^Block=halfwidthandfullwidthforms}', "");
Expect(0, 65279, '\p{Block=halfwidthandfullwidthforms}', "");
Expect(1, 65279, '\p{^Block=halfwidthandfullwidthforms}', "");
Expect(1, 65279, '\P{Block=halfwidthandfullwidthforms}', "");
Expect(0, 65279, '\P{^Block=halfwidthandfullwidthforms}', "");
Expect(1, 65519, '\p{Block=	 Halfwidthand_fullwidth-FORMS}', "");
Expect(0, 65519, '\p{^Block=	 Halfwidthand_fullwidth-FORMS}', "");
Expect(0, 65519, '\P{Block=	 Halfwidthand_fullwidth-FORMS}', "");
Expect(1, 65519, '\P{^Block=	 Halfwidthand_fullwidth-FORMS}', "");
Expect(0, 65279, '\p{Block=	 Halfwidthand_fullwidth-FORMS}', "");
Expect(1, 65279, '\p{^Block=	 Halfwidthand_fullwidth-FORMS}', "");
Expect(1, 65279, '\P{Block=	 Halfwidthand_fullwidth-FORMS}', "");
Expect(0, 65279, '\P{^Block=	 Halfwidthand_fullwidth-FORMS}', "");
Error('\p{Blk=  HALFWIDTH/a/AND_Fullwidth	Forms}');
Error('\P{Blk=  HALFWIDTH/a/AND_Fullwidth	Forms}');
Expect(1, 65519, '\p{Blk=halfwidthandfullwidthforms}', "");
Expect(0, 65519, '\p{^Blk=halfwidthandfullwidthforms}', "");
Expect(0, 65519, '\P{Blk=halfwidthandfullwidthforms}', "");
Expect(1, 65519, '\P{^Blk=halfwidthandfullwidthforms}', "");
Expect(0, 65279, '\p{Blk=halfwidthandfullwidthforms}', "");
Expect(1, 65279, '\p{^Blk=halfwidthandfullwidthforms}', "");
Expect(1, 65279, '\P{Blk=halfwidthandfullwidthforms}', "");
Expect(0, 65279, '\P{^Blk=halfwidthandfullwidthforms}', "");
Expect(1, 65519, '\p{Blk=_ HALFWIDTH_AND_Fullwidth	Forms}', "");
Expect(0, 65519, '\p{^Blk=_ HALFWIDTH_AND_Fullwidth	Forms}', "");
Expect(0, 65519, '\P{Blk=_ HALFWIDTH_AND_Fullwidth	Forms}', "");
Expect(1, 65519, '\P{^Blk=_ HALFWIDTH_AND_Fullwidth	Forms}', "");
Expect(0, 65279, '\p{Blk=_ HALFWIDTH_AND_Fullwidth	Forms}', "");
Expect(1, 65279, '\p{^Blk=_ HALFWIDTH_AND_Fullwidth	Forms}', "");
Expect(1, 65279, '\P{Blk=_ HALFWIDTH_AND_Fullwidth	Forms}', "");
Expect(0, 65279, '\P{^Blk=_ HALFWIDTH_AND_Fullwidth	Forms}', "");
Error('\p{Is_Block=_:=HALFWIDTH-AND_fullwidth	FORMS}');
Error('\P{Is_Block=_:=HALFWIDTH-AND_fullwidth	FORMS}');
Expect(1, 65519, '\p{Is_Block=halfwidthandfullwidthforms}', "");
Expect(0, 65519, '\p{^Is_Block=halfwidthandfullwidthforms}', "");
Expect(0, 65519, '\P{Is_Block=halfwidthandfullwidthforms}', "");
Expect(1, 65519, '\P{^Is_Block=halfwidthandfullwidthforms}', "");
Expect(0, 65279, '\p{Is_Block=halfwidthandfullwidthforms}', "");
Expect(1, 65279, '\p{^Is_Block=halfwidthandfullwidthforms}', "");
Expect(1, 65279, '\P{Is_Block=halfwidthandfullwidthforms}', "");
Expect(0, 65279, '\P{^Is_Block=halfwidthandfullwidthforms}', "");
Expect(1, 65519, '\p{Is_Block=	-Halfwidth and	FULLWIDTH_Forms}', "");
Expect(0, 65519, '\p{^Is_Block=	-Halfwidth and	FULLWIDTH_Forms}', "");
Expect(0, 65519, '\P{Is_Block=	-Halfwidth and	FULLWIDTH_Forms}', "");
Expect(1, 65519, '\P{^Is_Block=	-Halfwidth and	FULLWIDTH_Forms}', "");
Expect(0, 65279, '\p{Is_Block=	-Halfwidth and	FULLWIDTH_Forms}', "");
Expect(1, 65279, '\p{^Is_Block=	-Halfwidth and	FULLWIDTH_Forms}', "");
Expect(1, 65279, '\P{Is_Block=	-Halfwidth and	FULLWIDTH_Forms}', "");
Expect(0, 65279, '\P{^Is_Block=	-Halfwidth and	FULLWIDTH_Forms}', "");
Error('\p{Is_Blk=- HALFWIDTH_And	Fullwidth Forms/a/}');
Error('\P{Is_Blk=- HALFWIDTH_And	Fullwidth Forms/a/}');
Expect(1, 65519, '\p{Is_Blk=halfwidthandfullwidthforms}', "");
Expect(0, 65519, '\p{^Is_Blk=halfwidthandfullwidthforms}', "");
Expect(0, 65519, '\P{Is_Blk=halfwidthandfullwidthforms}', "");
Expect(1, 65519, '\P{^Is_Blk=halfwidthandfullwidthforms}', "");
Expect(0, 65279, '\p{Is_Blk=halfwidthandfullwidthforms}', "");
Expect(1, 65279, '\p{^Is_Blk=halfwidthandfullwidthforms}', "");
Expect(1, 65279, '\P{Is_Blk=halfwidthandfullwidthforms}', "");
Expect(0, 65279, '\P{^Is_Blk=halfwidthandfullwidthforms}', "");
Expect(1, 65519, '\p{Is_Blk=--HALFWIDTH-AND Fullwidth Forms}', "");
Expect(0, 65519, '\p{^Is_Blk=--HALFWIDTH-AND Fullwidth Forms}', "");
Expect(0, 65519, '\P{Is_Blk=--HALFWIDTH-AND Fullwidth Forms}', "");
Expect(1, 65519, '\P{^Is_Blk=--HALFWIDTH-AND Fullwidth Forms}', "");
Expect(0, 65279, '\p{Is_Blk=--HALFWIDTH-AND Fullwidth Forms}', "");
Expect(1, 65279, '\p{^Is_Blk=--HALFWIDTH-AND Fullwidth Forms}', "");
Expect(1, 65279, '\P{Is_Blk=--HALFWIDTH-AND Fullwidth Forms}', "");
Expect(0, 65279, '\P{^Is_Blk=--HALFWIDTH-AND Fullwidth Forms}', "");
Error('\p{Block=-	supplementary Private_USE area/a/B}');
Error('\P{Block=-	supplementary Private_USE area/a/B}');
Expect(1, 1048576, '\p{Block=supplementaryprivateuseareab}', "");
Expect(0, 1048576, '\p{^Block=supplementaryprivateuseareab}', "");
Expect(0, 1048576, '\P{Block=supplementaryprivateuseareab}', "");
Expect(1, 1048576, '\P{^Block=supplementaryprivateuseareab}', "");
Expect(0, 1, '\p{Block=supplementaryprivateuseareab}', "");
Expect(1, 1, '\p{^Block=supplementaryprivateuseareab}', "");
Expect(1, 1, '\P{Block=supplementaryprivateuseareab}', "");
Expect(0, 1, '\P{^Block=supplementaryprivateuseareab}', "");
Expect(1, 1048576, '\p{Block=-	Supplementary	private_Use_Area B}', "");
Expect(0, 1048576, '\p{^Block=-	Supplementary	private_Use_Area B}', "");
Expect(0, 1048576, '\P{Block=-	Supplementary	private_Use_Area B}', "");
Expect(1, 1048576, '\P{^Block=-	Supplementary	private_Use_Area B}', "");
Expect(0, 1, '\p{Block=-	Supplementary	private_Use_Area B}', "");
Expect(1, 1, '\p{^Block=-	Supplementary	private_Use_Area B}', "");
Expect(1, 1, '\P{Block=-	Supplementary	private_Use_Area B}', "");
Expect(0, 1, '\P{^Block=-	Supplementary	private_Use_Area B}', "");
Error('\p{Blk=:=	 Supplementary-Private-Use AREAB}');
Error('\P{Blk=:=	 Supplementary-Private-Use AREAB}');
Expect(1, 1048576, '\p{Blk=supplementaryprivateuseareab}', "");
Expect(0, 1048576, '\p{^Blk=supplementaryprivateuseareab}', "");
Expect(0, 1048576, '\P{Blk=supplementaryprivateuseareab}', "");
Expect(1, 1048576, '\P{^Blk=supplementaryprivateuseareab}', "");
Expect(0, 1, '\p{Blk=supplementaryprivateuseareab}', "");
Expect(1, 1, '\p{^Blk=supplementaryprivateuseareab}', "");
Expect(1, 1, '\P{Blk=supplementaryprivateuseareab}', "");
Expect(0, 1, '\P{^Blk=supplementaryprivateuseareab}', "");
Expect(1, 1048576, '\p{Blk= Supplementary-Private	USE areaB}', "");
Expect(0, 1048576, '\p{^Blk= Supplementary-Private	USE areaB}', "");
Expect(0, 1048576, '\P{Blk= Supplementary-Private	USE areaB}', "");
Expect(1, 1048576, '\P{^Blk= Supplementary-Private	USE areaB}', "");
Expect(0, 1, '\p{Blk= Supplementary-Private	USE areaB}', "");
Expect(1, 1, '\p{^Blk= Supplementary-Private	USE areaB}', "");
Expect(1, 1, '\P{Blk= Supplementary-Private	USE areaB}', "");
Expect(0, 1, '\P{^Blk= Supplementary-Private	USE areaB}', "");
Error('\p{Is_Block=__supplementary Private:=Use_Area	B}');
Error('\P{Is_Block=__supplementary Private:=Use_Area	B}');
Expect(1, 1048576, '\p{Is_Block=supplementaryprivateuseareab}', "");
Expect(0, 1048576, '\p{^Is_Block=supplementaryprivateuseareab}', "");
Expect(0, 1048576, '\P{Is_Block=supplementaryprivateuseareab}', "");
Expect(1, 1048576, '\P{^Is_Block=supplementaryprivateuseareab}', "");
Expect(0, 1, '\p{Is_Block=supplementaryprivateuseareab}', "");
Expect(1, 1, '\p{^Is_Block=supplementaryprivateuseareab}', "");
Expect(1, 1, '\P{Is_Block=supplementaryprivateuseareab}', "");
Expect(0, 1, '\P{^Is_Block=supplementaryprivateuseareab}', "");
Expect(1, 1048576, '\p{Is_Block=		supplementary-Private USE-area	B}', "");
Expect(0, 1048576, '\p{^Is_Block=		supplementary-Private USE-area	B}', "");
Expect(0, 1048576, '\P{Is_Block=		supplementary-Private USE-area	B}', "");
Expect(1, 1048576, '\P{^Is_Block=		supplementary-Private USE-area	B}', "");
Expect(0, 1, '\p{Is_Block=		supplementary-Private USE-area	B}', "");
Expect(1, 1, '\p{^Is_Block=		supplementary-Private USE-area	B}', "");
Expect(1, 1, '\P{Is_Block=		supplementary-Private USE-area	B}', "");
Expect(0, 1, '\P{^Is_Block=		supplementary-Private USE-area	B}', "");
Error('\p{Is_Blk: :=Supplementary_private	use-AreaB}');
Error('\P{Is_Blk: :=Supplementary_private	use-AreaB}');
Expect(1, 1048576, '\p{Is_Blk=supplementaryprivateuseareab}', "");
Expect(0, 1048576, '\p{^Is_Blk=supplementaryprivateuseareab}', "");
Expect(0, 1048576, '\P{Is_Blk=supplementaryprivateuseareab}', "");
Expect(1, 1048576, '\P{^Is_Blk=supplementaryprivateuseareab}', "");
Expect(0, 1, '\p{Is_Blk=supplementaryprivateuseareab}', "");
Expect(1, 1, '\p{^Is_Blk=supplementaryprivateuseareab}', "");
Expect(1, 1, '\P{Is_Blk=supplementaryprivateuseareab}', "");
Expect(0, 1, '\P{^Is_Blk=supplementaryprivateuseareab}', "");
Expect(1, 1048576, '\p{Is_Blk=- supplementary_PrivateUse	AREAB}', "");
Expect(0, 1048576, '\p{^Is_Blk=- supplementary_PrivateUse	AREAB}', "");
Expect(0, 1048576, '\P{Is_Blk=- supplementary_PrivateUse	AREAB}', "");
Expect(1, 1048576, '\P{^Is_Blk=- supplementary_PrivateUse	AREAB}', "");
Expect(0, 1, '\p{Is_Blk=- supplementary_PrivateUse	AREAB}', "");
Expect(1, 1, '\p{^Is_Blk=- supplementary_PrivateUse	AREAB}', "");
Expect(1, 1, '\P{Is_Blk=- supplementary_PrivateUse	AREAB}', "");
Expect(0, 1, '\P{^Is_Blk=- supplementary_PrivateUse	AREAB}', "");
Error('\p{Block=-	SUNDANESE:=}');
Error('\P{Block=-	SUNDANESE:=}');
Expect(1, 7103, '\p{Block=sundanese}', "");
Expect(0, 7103, '\p{^Block=sundanese}', "");
Expect(0, 7103, '\P{Block=sundanese}', "");
Expect(1, 7103, '\P{^Block=sundanese}', "");
Expect(0, 7039, '\p{Block=sundanese}', "");
Expect(1, 7039, '\p{^Block=sundanese}', "");
Expect(1, 7039, '\P{Block=sundanese}', "");
Expect(0, 7039, '\P{^Block=sundanese}', "");
Expect(1, 7103, '\p{Block=	 Sundanese}', "");
Expect(0, 7103, '\p{^Block=	 Sundanese}', "");
Expect(0, 7103, '\P{Block=	 Sundanese}', "");
Expect(1, 7103, '\P{^Block=	 Sundanese}', "");
Expect(0, 7039, '\p{Block=	 Sundanese}', "");
Expect(1, 7039, '\p{^Block=	 Sundanese}', "");
Expect(1, 7039, '\P{Block=	 Sundanese}', "");
Expect(0, 7039, '\P{^Block=	 Sundanese}', "");
Error('\p{Blk=/a/	-sundanese}');
Error('\P{Blk=/a/	-sundanese}');
Expect(1, 7103, '\p{Blk=sundanese}', "");
Expect(0, 7103, '\p{^Blk=sundanese}', "");
Expect(0, 7103, '\P{Blk=sundanese}', "");
Expect(1, 7103, '\P{^Blk=sundanese}', "");
Expect(0, 7039, '\p{Blk=sundanese}', "");
Expect(1, 7039, '\p{^Blk=sundanese}', "");
Expect(1, 7039, '\P{Blk=sundanese}', "");
Expect(0, 7039, '\P{^Blk=sundanese}', "");
Expect(1, 7103, '\p{Blk=-Sundanese}', "");
Expect(0, 7103, '\p{^Blk=-Sundanese}', "");
Expect(0, 7103, '\P{Blk=-Sundanese}', "");
Expect(1, 7103, '\P{^Blk=-Sundanese}', "");
Expect(0, 7039, '\p{Blk=-Sundanese}', "");
Expect(1, 7039, '\p{^Blk=-Sundanese}', "");
Expect(1, 7039, '\P{Blk=-Sundanese}', "");
Expect(0, 7039, '\P{^Blk=-Sundanese}', "");
Error('\p{Is_Block= sundanese/a/}');
Error('\P{Is_Block= sundanese/a/}');
Expect(1, 7103, '\p{Is_Block:	sundanese}', "");
Expect(0, 7103, '\p{^Is_Block:	sundanese}', "");
Expect(0, 7103, '\P{Is_Block:	sundanese}', "");
Expect(1, 7103, '\P{^Is_Block:	sundanese}', "");
Expect(0, 7039, '\p{Is_Block:	sundanese}', "");
Expect(1, 7039, '\p{^Is_Block:	sundanese}', "");
Expect(1, 7039, '\P{Is_Block:	sundanese}', "");
Expect(0, 7039, '\P{^Is_Block:	sundanese}', "");
Expect(1, 7103, '\p{Is_Block=--SUNDANESE}', "");
Expect(0, 7103, '\p{^Is_Block=--SUNDANESE}', "");
Expect(0, 7103, '\P{Is_Block=--SUNDANESE}', "");
Expect(1, 7103, '\P{^Is_Block=--SUNDANESE}', "");
Expect(0, 7039, '\p{Is_Block=--SUNDANESE}', "");
Expect(1, 7039, '\p{^Is_Block=--SUNDANESE}', "");
Expect(1, 7039, '\P{Is_Block=--SUNDANESE}', "");
Expect(0, 7039, '\P{^Is_Block=--SUNDANESE}', "");
Error('\p{Is_Blk=-Sundanese/a/}');
Error('\P{Is_Blk=-Sundanese/a/}');
Expect(1, 7103, '\p{Is_Blk=sundanese}', "");
Expect(0, 7103, '\p{^Is_Blk=sundanese}', "");
Expect(0, 7103, '\P{Is_Blk=sundanese}', "");
Expect(1, 7103, '\P{^Is_Blk=sundanese}', "");
Expect(0, 7039, '\p{Is_Blk=sundanese}', "");
Expect(1, 7039, '\p{^Is_Blk=sundanese}', "");
Expect(1, 7039, '\P{Is_Blk=sundanese}', "");
Expect(0, 7039, '\P{^Is_Blk=sundanese}', "");
Expect(1, 7103, '\p{Is_Blk=_-Sundanese}', "");
Expect(0, 7103, '\p{^Is_Blk=_-Sundanese}', "");
Expect(0, 7103, '\P{Is_Blk=_-Sundanese}', "");
Expect(1, 7103, '\P{^Is_Blk=_-Sundanese}', "");
Expect(0, 7039, '\p{Is_Blk=_-Sundanese}', "");
Expect(1, 7039, '\p{^Is_Blk=_-Sundanese}', "");
Expect(1, 7039, '\P{Is_Blk=_-Sundanese}', "");
Expect(0, 7039, '\P{^Is_Blk=_-Sundanese}', "");
Error('\p{Block=	-enclosedAlphanumerics:=}');
Error('\P{Block=	-enclosedAlphanumerics:=}');
Expect(1, 9471, '\p{Block=enclosedalphanumerics}', "");
Expect(0, 9471, '\p{^Block=enclosedalphanumerics}', "");
Expect(0, 9471, '\P{Block=enclosedalphanumerics}', "");
Expect(1, 9471, '\P{^Block=enclosedalphanumerics}', "");
Expect(0, 9311, '\p{Block=enclosedalphanumerics}', "");
Expect(1, 9311, '\p{^Block=enclosedalphanumerics}', "");
Expect(1, 9311, '\P{Block=enclosedalphanumerics}', "");
Expect(0, 9311, '\P{^Block=enclosedalphanumerics}', "");
Expect(1, 9471, '\p{Block=_-Enclosed-Alphanumerics}', "");
Expect(0, 9471, '\p{^Block=_-Enclosed-Alphanumerics}', "");
Expect(0, 9471, '\P{Block=_-Enclosed-Alphanumerics}', "");
Expect(1, 9471, '\P{^Block=_-Enclosed-Alphanumerics}', "");
Expect(0, 9311, '\p{Block=_-Enclosed-Alphanumerics}', "");
Expect(1, 9311, '\p{^Block=_-Enclosed-Alphanumerics}', "");
Expect(1, 9311, '\P{Block=_-Enclosed-Alphanumerics}', "");
Expect(0, 9311, '\P{^Block=_-Enclosed-Alphanumerics}', "");
Error('\p{Blk=/a/Enclosed_ALPHANUMERICS}');
Error('\P{Blk=/a/Enclosed_ALPHANUMERICS}');
Expect(1, 9471, '\p{Blk=enclosedalphanumerics}', "");
Expect(0, 9471, '\p{^Blk=enclosedalphanumerics}', "");
Expect(0, 9471, '\P{Blk=enclosedalphanumerics}', "");
Expect(1, 9471, '\P{^Blk=enclosedalphanumerics}', "");
Expect(0, 9311, '\p{Blk=enclosedalphanumerics}', "");
Expect(1, 9311, '\p{^Blk=enclosedalphanumerics}', "");
Expect(1, 9311, '\P{Blk=enclosedalphanumerics}', "");
Expect(0, 9311, '\P{^Blk=enclosedalphanumerics}', "");
Expect(1, 9471, '\p{Blk: 		Enclosed	Alphanumerics}', "");
Expect(0, 9471, '\p{^Blk: 		Enclosed	Alphanumerics}', "");
Expect(0, 9471, '\P{Blk: 		Enclosed	Alphanumerics}', "");
Expect(1, 9471, '\P{^Blk: 		Enclosed	Alphanumerics}', "");
Expect(0, 9311, '\p{Blk: 		Enclosed	Alphanumerics}', "");
Expect(1, 9311, '\p{^Blk: 		Enclosed	Alphanumerics}', "");
Expect(1, 9311, '\P{Blk: 		Enclosed	Alphanumerics}', "");
Expect(0, 9311, '\P{^Blk: 		Enclosed	Alphanumerics}', "");
Error('\p{Is_Block:- EnclosedAlphanumerics/a/}');
Error('\P{Is_Block:- EnclosedAlphanumerics/a/}');
Expect(1, 9471, '\p{Is_Block=enclosedalphanumerics}', "");
Expect(0, 9471, '\p{^Is_Block=enclosedalphanumerics}', "");
Expect(0, 9471, '\P{Is_Block=enclosedalphanumerics}', "");
Expect(1, 9471, '\P{^Is_Block=enclosedalphanumerics}', "");
Expect(0, 9311, '\p{Is_Block=enclosedalphanumerics}', "");
Expect(1, 9311, '\p{^Is_Block=enclosedalphanumerics}', "");
Expect(1, 9311, '\P{Is_Block=enclosedalphanumerics}', "");
Expect(0, 9311, '\P{^Is_Block=enclosedalphanumerics}', "");
Expect(1, 9471, '\p{Is_Block=	 ENCLOSED-alphanumerics}', "");
Expect(0, 9471, '\p{^Is_Block=	 ENCLOSED-alphanumerics}', "");
Expect(0, 9471, '\P{Is_Block=	 ENCLOSED-alphanumerics}', "");
Expect(1, 9471, '\P{^Is_Block=	 ENCLOSED-alphanumerics}', "");
Expect(0, 9311, '\p{Is_Block=	 ENCLOSED-alphanumerics}', "");
Expect(1, 9311, '\p{^Is_Block=	 ENCLOSED-alphanumerics}', "");
Expect(1, 9311, '\P{Is_Block=	 ENCLOSED-alphanumerics}', "");
Expect(0, 9311, '\P{^Is_Block=	 ENCLOSED-alphanumerics}', "");
Error('\p{Is_Blk:    /a/Enclosed Alphanumerics}');
Error('\P{Is_Blk:    /a/Enclosed Alphanumerics}');
Expect(1, 9471, '\p{Is_Blk:	enclosedalphanumerics}', "");
Expect(0, 9471, '\p{^Is_Blk:	enclosedalphanumerics}', "");
Expect(0, 9471, '\P{Is_Blk:	enclosedalphanumerics}', "");
Expect(1, 9471, '\P{^Is_Blk:	enclosedalphanumerics}', "");
Expect(0, 9311, '\p{Is_Blk:	enclosedalphanumerics}', "");
Expect(1, 9311, '\p{^Is_Blk:	enclosedalphanumerics}', "");
Expect(1, 9311, '\P{Is_Blk:	enclosedalphanumerics}', "");
Expect(0, 9311, '\P{^Is_Blk:	enclosedalphanumerics}', "");
Expect(1, 9471, '\p{Is_Blk=-	enclosed Alphanumerics}', "");
Expect(0, 9471, '\p{^Is_Blk=-	enclosed Alphanumerics}', "");
Expect(0, 9471, '\P{Is_Blk=-	enclosed Alphanumerics}', "");
Expect(1, 9471, '\P{^Is_Blk=-	enclosed Alphanumerics}', "");
Expect(0, 9311, '\p{Is_Blk=-	enclosed Alphanumerics}', "");
Expect(1, 9311, '\p{^Is_Blk=-	enclosed Alphanumerics}', "");
Expect(1, 9311, '\P{Is_Blk=-	enclosed Alphanumerics}', "");
Expect(0, 9311, '\P{^Is_Blk=-	enclosed Alphanumerics}', "");
Error('\p{Block=__Old_South	ARABIAN:=}');
Error('\P{Block=__Old_South	ARABIAN:=}');
Expect(1, 68223, '\p{Block=oldsoutharabian}', "");
Expect(0, 68223, '\p{^Block=oldsoutharabian}', "");
Expect(0, 68223, '\P{Block=oldsoutharabian}', "");
Expect(1, 68223, '\P{^Block=oldsoutharabian}', "");
Expect(0, 68191, '\p{Block=oldsoutharabian}', "");
Expect(1, 68191, '\p{^Block=oldsoutharabian}', "");
Expect(1, 68191, '\P{Block=oldsoutharabian}', "");
Expect(0, 68191, '\P{^Block=oldsoutharabian}', "");
Expect(1, 68223, '\p{Block:   Old_South	Arabian}', "");
Expect(0, 68223, '\p{^Block:   Old_South	Arabian}', "");
Expect(0, 68223, '\P{Block:   Old_South	Arabian}', "");
Expect(1, 68223, '\P{^Block:   Old_South	Arabian}', "");
Expect(0, 68191, '\p{Block:   Old_South	Arabian}', "");
Expect(1, 68191, '\p{^Block:   Old_South	Arabian}', "");
Expect(1, 68191, '\P{Block:   Old_South	Arabian}', "");
Expect(0, 68191, '\P{^Block:   Old_South	Arabian}', "");
Error('\p{Blk=:=Old_South_ARABIAN}');
Error('\P{Blk=:=Old_South_ARABIAN}');
Expect(1, 68223, '\p{Blk=oldsoutharabian}', "");
Expect(0, 68223, '\p{^Blk=oldsoutharabian}', "");
Expect(0, 68223, '\P{Blk=oldsoutharabian}', "");
Expect(1, 68223, '\P{^Blk=oldsoutharabian}', "");
Expect(0, 68191, '\p{Blk=oldsoutharabian}', "");
Expect(1, 68191, '\p{^Blk=oldsoutharabian}', "");
Expect(1, 68191, '\P{Blk=oldsoutharabian}', "");
Expect(0, 68191, '\P{^Blk=oldsoutharabian}', "");
Expect(1, 68223, '\p{Blk= Old	SOUTHArabian}', "");
Expect(0, 68223, '\p{^Blk= Old	SOUTHArabian}', "");
Expect(0, 68223, '\P{Blk= Old	SOUTHArabian}', "");
Expect(1, 68223, '\P{^Blk= Old	SOUTHArabian}', "");
Expect(0, 68191, '\p{Blk= Old	SOUTHArabian}', "");
Expect(1, 68191, '\p{^Blk= Old	SOUTHArabian}', "");
Expect(1, 68191, '\P{Blk= Old	SOUTHArabian}', "");
Expect(0, 68191, '\P{^Blk= Old	SOUTHArabian}', "");
Error('\p{Is_Block=_/a/OldSouthArabian}');
Error('\P{Is_Block=_/a/OldSouthArabian}');
Expect(1, 68223, '\p{Is_Block=oldsoutharabian}', "");
Expect(0, 68223, '\p{^Is_Block=oldsoutharabian}', "");
Expect(0, 68223, '\P{Is_Block=oldsoutharabian}', "");
Expect(1, 68223, '\P{^Is_Block=oldsoutharabian}', "");
Expect(0, 68191, '\p{Is_Block=oldsoutharabian}', "");
Expect(1, 68191, '\p{^Is_Block=oldsoutharabian}', "");
Expect(1, 68191, '\P{Is_Block=oldsoutharabian}', "");
Expect(0, 68191, '\P{^Is_Block=oldsoutharabian}', "");
Expect(1, 68223, '\p{Is_Block=- Old_south_ARABIAN}', "");
Expect(0, 68223, '\p{^Is_Block=- Old_south_ARABIAN}', "");
Expect(0, 68223, '\P{Is_Block=- Old_south_ARABIAN}', "");
Expect(1, 68223, '\P{^Is_Block=- Old_south_ARABIAN}', "");
Expect(0, 68191, '\p{Is_Block=- Old_south_ARABIAN}', "");
Expect(1, 68191, '\p{^Is_Block=- Old_south_ARABIAN}', "");
Expect(1, 68191, '\P{Is_Block=- Old_south_ARABIAN}', "");
Expect(0, 68191, '\P{^Is_Block=- Old_south_ARABIAN}', "");
Error('\p{Is_Blk:	 _old:=southArabian}');
Error('\P{Is_Blk:	 _old:=southArabian}');
Expect(1, 68223, '\p{Is_Blk=oldsoutharabian}', "");
Expect(0, 68223, '\p{^Is_Blk=oldsoutharabian}', "");
Expect(0, 68223, '\P{Is_Blk=oldsoutharabian}', "");
Expect(1, 68223, '\P{^Is_Blk=oldsoutharabian}', "");
Expect(0, 68191, '\p{Is_Blk=oldsoutharabian}', "");
Expect(1, 68191, '\p{^Is_Blk=oldsoutharabian}', "");
Expect(1, 68191, '\P{Is_Blk=oldsoutharabian}', "");
Expect(0, 68191, '\P{^Is_Blk=oldsoutharabian}', "");
Expect(1, 68223, '\p{Is_Blk=_Old South_Arabian}', "");
Expect(0, 68223, '\p{^Is_Blk=_Old South_Arabian}', "");
Expect(0, 68223, '\P{Is_Blk=_Old South_Arabian}', "");
Expect(1, 68223, '\P{^Is_Blk=_Old South_Arabian}', "");
Expect(0, 68191, '\p{Is_Blk=_Old South_Arabian}', "");
Expect(1, 68191, '\p{^Is_Blk=_Old South_Arabian}', "");
Expect(1, 68191, '\P{Is_Blk=_Old South_Arabian}', "");
Expect(0, 68191, '\P{^Is_Blk=_Old South_Arabian}', "");
Error('\p{Block: tags:=}');
Error('\P{Block: tags:=}');
Expect(1, 917631, '\p{Block=tags}', "");
Expect(0, 917631, '\p{^Block=tags}', "");
Expect(0, 917631, '\P{Block=tags}', "");
Expect(1, 917631, '\P{^Block=tags}', "");
Expect(0, 1, '\p{Block=tags}', "");
Expect(1, 1, '\p{^Block=tags}', "");
Expect(1, 1, '\P{Block=tags}', "");
Expect(0, 1, '\P{^Block=tags}', "");
Expect(1, 917631, '\p{Block: 	tags}', "");
Expect(0, 917631, '\p{^Block: 	tags}', "");
Expect(0, 917631, '\P{Block: 	tags}', "");
Expect(1, 917631, '\P{^Block: 	tags}', "");
Expect(0, 1, '\p{Block: 	tags}', "");
Expect(1, 1, '\p{^Block: 	tags}', "");
Expect(1, 1, '\P{Block: 	tags}', "");
Expect(0, 1, '\P{^Block: 	tags}', "");
Error('\p{Blk=:=-	Tags}');
Error('\P{Blk=:=-	Tags}');
Expect(1, 917631, '\p{Blk=tags}', "");
Expect(0, 917631, '\p{^Blk=tags}', "");
Expect(0, 917631, '\P{Blk=tags}', "");
Expect(1, 917631, '\P{^Blk=tags}', "");
Expect(0, 1, '\p{Blk=tags}', "");
Expect(1, 1, '\p{^Blk=tags}', "");
Expect(1, 1, '\P{Blk=tags}', "");
Expect(0, 1, '\P{^Blk=tags}', "");
Expect(1, 917631, '\p{Blk: _TAGS}', "");
Expect(0, 917631, '\p{^Blk: _TAGS}', "");
Expect(0, 917631, '\P{Blk: _TAGS}', "");
Expect(1, 917631, '\P{^Blk: _TAGS}', "");
Expect(0, 1, '\p{Blk: _TAGS}', "");
Expect(1, 1, '\p{^Blk: _TAGS}', "");
Expect(1, 1, '\P{Blk: _TAGS}', "");
Expect(0, 1, '\P{^Blk: _TAGS}', "");
Error('\p{Is_Block=-:=TAGS}');
Error('\P{Is_Block=-:=TAGS}');
Expect(1, 917631, '\p{Is_Block=tags}', "");
Expect(0, 917631, '\p{^Is_Block=tags}', "");
Expect(0, 917631, '\P{Is_Block=tags}', "");
Expect(1, 917631, '\P{^Is_Block=tags}', "");
Expect(0, 1, '\p{Is_Block=tags}', "");
Expect(1, 1, '\p{^Is_Block=tags}', "");
Expect(1, 1, '\P{Is_Block=tags}', "");
Expect(0, 1, '\P{^Is_Block=tags}', "");
Expect(1, 917631, '\p{Is_Block=_tags}', "");
Expect(0, 917631, '\p{^Is_Block=_tags}', "");
Expect(0, 917631, '\P{Is_Block=_tags}', "");
Expect(1, 917631, '\P{^Is_Block=_tags}', "");
Expect(0, 1, '\p{Is_Block=_tags}', "");
Expect(1, 1, '\p{^Is_Block=_tags}', "");
Expect(1, 1, '\P{Is_Block=_tags}', "");
Expect(0, 1, '\P{^Is_Block=_tags}', "");
Error('\p{Is_Blk= /a/Tags}');
Error('\P{Is_Blk= /a/Tags}');
Expect(1, 917631, '\p{Is_Blk=tags}', "");
Expect(0, 917631, '\p{^Is_Blk=tags}', "");
Expect(0, 917631, '\P{Is_Blk=tags}', "");
Expect(1, 917631, '\P{^Is_Blk=tags}', "");
Expect(0, 1, '\p{Is_Blk=tags}', "");
Expect(1, 1, '\p{^Is_Blk=tags}', "");
Expect(1, 1, '\P{Is_Blk=tags}', "");
Expect(0, 1, '\P{^Is_Blk=tags}', "");
Expect(1, 917631, '\p{Is_Blk=  TAGS}', "");
Expect(0, 917631, '\p{^Is_Blk=  TAGS}', "");
Expect(0, 917631, '\P{Is_Blk=  TAGS}', "");
Expect(1, 917631, '\P{^Is_Blk=  TAGS}', "");
Expect(0, 1, '\p{Is_Blk=  TAGS}', "");
Expect(1, 1, '\p{^Is_Blk=  TAGS}', "");
Expect(1, 1, '\P{Is_Blk=  TAGS}', "");
Expect(0, 1, '\P{^Is_Blk=  TAGS}', "");
Error('\p{Block:   --miscellaneous:=mathematical Symbols	A}');
Error('\P{Block:   --miscellaneous:=mathematical Symbols	A}');
Expect(1, 10223, '\p{Block=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10223, '\p{^Block=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10223, '\P{Block=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10223, '\P{^Block=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10175, '\p{Block=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10175, '\p{^Block=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10175, '\P{Block=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10175, '\P{^Block=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10223, '\p{Block: -_Miscellaneous-Mathematical Symbols_A}', "");
Expect(0, 10223, '\p{^Block: -_Miscellaneous-Mathematical Symbols_A}', "");
Expect(0, 10223, '\P{Block: -_Miscellaneous-Mathematical Symbols_A}', "");
Expect(1, 10223, '\P{^Block: -_Miscellaneous-Mathematical Symbols_A}', "");
Expect(0, 10175, '\p{Block: -_Miscellaneous-Mathematical Symbols_A}', "");
Expect(1, 10175, '\p{^Block: -_Miscellaneous-Mathematical Symbols_A}', "");
Expect(1, 10175, '\P{Block: -_Miscellaneous-Mathematical Symbols_A}', "");
Expect(0, 10175, '\P{^Block: -_Miscellaneous-Mathematical Symbols_A}', "");
Error('\p{Blk:   :=--miscellaneous	mathematicalSYMBOLSA}');
Error('\P{Blk:   :=--miscellaneous	mathematicalSYMBOLSA}');
Expect(1, 10223, '\p{Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10223, '\p{^Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10223, '\P{Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10223, '\P{^Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10175, '\p{Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10175, '\p{^Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10175, '\P{Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10175, '\P{^Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10223, '\p{Blk=_miscellaneous	MATHEMATICAL	symbols_A}', "");
Expect(0, 10223, '\p{^Blk=_miscellaneous	MATHEMATICAL	symbols_A}', "");
Expect(0, 10223, '\P{Blk=_miscellaneous	MATHEMATICAL	symbols_A}', "");
Expect(1, 10223, '\P{^Blk=_miscellaneous	MATHEMATICAL	symbols_A}', "");
Expect(0, 10175, '\p{Blk=_miscellaneous	MATHEMATICAL	symbols_A}', "");
Expect(1, 10175, '\p{^Blk=_miscellaneous	MATHEMATICAL	symbols_A}', "");
Expect(1, 10175, '\P{Blk=_miscellaneous	MATHEMATICAL	symbols_A}', "");
Expect(0, 10175, '\P{^Blk=_miscellaneous	MATHEMATICAL	symbols_A}', "");
Error('\p{Is_Block=:=  Miscellaneous_MATHEMATICALSYMBOLS	A}');
Error('\P{Is_Block=:=  Miscellaneous_MATHEMATICALSYMBOLS	A}');
Expect(1, 10223, '\p{Is_Block=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10223, '\p{^Is_Block=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10223, '\P{Is_Block=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10223, '\P{^Is_Block=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10175, '\p{Is_Block=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10175, '\p{^Is_Block=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10175, '\P{Is_Block=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10175, '\P{^Is_Block=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10223, '\p{Is_Block=-	Miscellaneousmathematicalsymbols_a}', "");
Expect(0, 10223, '\p{^Is_Block=-	Miscellaneousmathematicalsymbols_a}', "");
Expect(0, 10223, '\P{Is_Block=-	Miscellaneousmathematicalsymbols_a}', "");
Expect(1, 10223, '\P{^Is_Block=-	Miscellaneousmathematicalsymbols_a}', "");
Expect(0, 10175, '\p{Is_Block=-	Miscellaneousmathematicalsymbols_a}', "");
Expect(1, 10175, '\p{^Is_Block=-	Miscellaneousmathematicalsymbols_a}', "");
Expect(1, 10175, '\P{Is_Block=-	Miscellaneousmathematicalsymbols_a}', "");
Expect(0, 10175, '\P{^Is_Block=-	Miscellaneousmathematicalsymbols_a}', "");
Error('\p{Is_Blk:	/a/		Miscellaneous	MATHEMATICAL_Symbols_A}');
Error('\P{Is_Blk:	/a/		Miscellaneous	MATHEMATICAL_Symbols_A}');
Expect(1, 10223, '\p{Is_Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10223, '\p{^Is_Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10223, '\P{Is_Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10223, '\P{^Is_Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10175, '\p{Is_Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10175, '\p{^Is_Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10175, '\P{Is_Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(0, 10175, '\P{^Is_Blk=miscellaneousmathematicalsymbolsa}', "");
Expect(1, 10223, '\p{Is_Blk=_-miscellaneousmathematical_Symbols A}', "");
Expect(0, 10223, '\p{^Is_Blk=_-miscellaneousmathematical_Symbols A}', "");
Expect(0, 10223, '\P{Is_Blk=_-miscellaneousmathematical_Symbols A}', "");
Expect(1, 10223, '\P{^Is_Blk=_-miscellaneousmathematical_Symbols A}', "");
Expect(0, 10175, '\p{Is_Blk=_-miscellaneousmathematical_Symbols A}', "");
Expect(1, 10175, '\p{^Is_Blk=_-miscellaneousmathematical_Symbols A}', "");
Expect(1, 10175, '\P{Is_Blk=_-miscellaneousmathematical_Symbols A}', "");
Expect(0, 10175, '\P{^Is_Blk=_-miscellaneousmathematical_Symbols A}', "");
Error('\p{Block=	-BraillePatterns/a/}');
Error('\P{Block=	-BraillePatterns/a/}');
Expect(1, 10495, '\p{Block=braillepatterns}', "");
Expect(0, 10495, '\p{^Block=braillepatterns}', "");
Expect(0, 10495, '\P{Block=braillepatterns}', "");
Expect(1, 10495, '\P{^Block=braillepatterns}', "");
Expect(0, 10239, '\p{Block=braillepatterns}', "");
Expect(1, 10239, '\p{^Block=braillepatterns}', "");
Expect(1, 10239, '\P{Block=braillepatterns}', "");
Expect(0, 10239, '\P{^Block=braillepatterns}', "");
Expect(1, 10495, '\p{Block=__Braille_Patterns}', "");
Expect(0, 10495, '\p{^Block=__Braille_Patterns}', "");
Expect(0, 10495, '\P{Block=__Braille_Patterns}', "");
Expect(1, 10495, '\P{^Block=__Braille_Patterns}', "");
Expect(0, 10239, '\p{Block=__Braille_Patterns}', "");
Expect(1, 10239, '\p{^Block=__Braille_Patterns}', "");
Expect(1, 10239, '\P{Block=__Braille_Patterns}', "");
Expect(0, 10239, '\P{^Block=__Braille_Patterns}', "");
Error('\p{Blk= :=BraillePATTERNS}');
Error('\P{Blk= :=BraillePATTERNS}');
Expect(1, 10495, '\p{Blk=braillepatterns}', "");
Expect(0, 10495, '\p{^Blk=braillepatterns}', "");
Expect(0, 10495, '\P{Blk=braillepatterns}', "");
Expect(1, 10495, '\P{^Blk=braillepatterns}', "");
Expect(0, 10239, '\p{Blk=braillepatterns}', "");
Expect(1, 10239, '\p{^Blk=braillepatterns}', "");
Expect(1, 10239, '\P{Blk=braillepatterns}', "");
Expect(0, 10239, '\P{^Blk=braillepatterns}', "");
Expect(1, 10495, '\p{Blk=	braille	Patterns}', "");
Expect(0, 10495, '\p{^Blk=	braille	Patterns}', "");
Expect(0, 10495, '\P{Blk=	braille	Patterns}', "");
Expect(1, 10495, '\P{^Blk=	braille	Patterns}', "");
Expect(0, 10239, '\p{Blk=	braille	Patterns}', "");
Expect(1, 10239, '\p{^Blk=	braille	Patterns}', "");
Expect(1, 10239, '\P{Blk=	braille	Patterns}', "");
Expect(0, 10239, '\P{^Blk=	braille	Patterns}', "");
Error('\p{Is_Block=_	Braille:=patterns}');
Error('\P{Is_Block=_	Braille:=patterns}');
Expect(1, 10495, '\p{Is_Block=braillepatterns}', "");
Expect(0, 10495, '\p{^Is_Block=braillepatterns}', "");
Expect(0, 10495, '\P{Is_Block=braillepatterns}', "");
Expect(1, 10495, '\P{^Is_Block=braillepatterns}', "");
Expect(0, 10239, '\p{Is_Block=braillepatterns}', "");
Expect(1, 10239, '\p{^Is_Block=braillepatterns}', "");
Expect(1, 10239, '\P{Is_Block=braillepatterns}', "");
Expect(0, 10239, '\P{^Is_Block=braillepatterns}', "");
Expect(1, 10495, '\p{Is_Block= Braille	Patterns}', "");
Expect(0, 10495, '\p{^Is_Block= Braille	Patterns}', "");
Expect(0, 10495, '\P{Is_Block= Braille	Patterns}', "");
Expect(1, 10495, '\P{^Is_Block= Braille	Patterns}', "");
Expect(0, 10239, '\p{Is_Block= Braille	Patterns}', "");
Expect(1, 10239, '\p{^Is_Block= Braille	Patterns}', "");
Expect(1, 10239, '\P{Is_Block= Braille	Patterns}', "");
Expect(0, 10239, '\P{^Is_Block= Braille	Patterns}', "");
Error('\p{Is_Blk=/a/-BRAILLEPatterns}');
Error('\P{Is_Blk=/a/-BRAILLEPatterns}');
Expect(1, 10495, '\p{Is_Blk=braillepatterns}', "");
Expect(0, 10495, '\p{^Is_Blk=braillepatterns}', "");
Expect(0, 10495, '\P{Is_Blk=braillepatterns}', "");
Expect(1, 10495, '\P{^Is_Blk=braillepatterns}', "");
Expect(0, 10239, '\p{Is_Blk=braillepatterns}', "");
Expect(1, 10239, '\p{^Is_Blk=braillepatterns}', "");
Expect(1, 10239, '\P{Is_Blk=braillepatterns}', "");
Expect(0, 10239, '\P{^Is_Blk=braillepatterns}', "");
Expect(1, 10495, '\p{Is_Blk=	BRAILLEPATTERNS}', "");
Expect(0, 10495, '\p{^Is_Blk=	BRAILLEPATTERNS}', "");
Expect(0, 10495, '\P{Is_Blk=	BRAILLEPATTERNS}', "");
Expect(1, 10495, '\P{^Is_Blk=	BRAILLEPATTERNS}', "");
Expect(0, 10239, '\p{Is_Blk=	BRAILLEPATTERNS}', "");
Expect(1, 10239, '\p{^Is_Blk=	BRAILLEPATTERNS}', "");
Expect(1, 10239, '\P{Is_Blk=	BRAILLEPATTERNS}', "");
Expect(0, 10239, '\P{^Is_Blk=	BRAILLEPATTERNS}', "");
Error('\p{Block=:=__oriya}');
Error('\P{Block=:=__oriya}');
Expect(1, 2943, '\p{Block=oriya}', "");
Expect(0, 2943, '\p{^Block=oriya}', "");
Expect(0, 2943, '\P{Block=oriya}', "");
Expect(1, 2943, '\P{^Block=oriya}', "");
Expect(0, 2815, '\p{Block=oriya}', "");
Expect(1, 2815, '\p{^Block=oriya}', "");
Expect(1, 2815, '\P{Block=oriya}', "");
Expect(0, 2815, '\P{^Block=oriya}', "");
Expect(1, 2943, '\p{Block: 	Oriya}', "");
Expect(0, 2943, '\p{^Block: 	Oriya}', "");
Expect(0, 2943, '\P{Block: 	Oriya}', "");
Expect(1, 2943, '\P{^Block: 	Oriya}', "");
Expect(0, 2815, '\p{Block: 	Oriya}', "");
Expect(1, 2815, '\p{^Block: 	Oriya}', "");
Expect(1, 2815, '\P{Block: 	Oriya}', "");
Expect(0, 2815, '\P{^Block: 	Oriya}', "");
Error('\p{Blk=:=- oriya}');
Error('\P{Blk=:=- oriya}');
Expect(1, 2943, '\p{Blk=oriya}', "");
Expect(0, 2943, '\p{^Blk=oriya}', "");
Expect(0, 2943, '\P{Blk=oriya}', "");
Expect(1, 2943, '\P{^Blk=oriya}', "");
Expect(0, 2815, '\p{Blk=oriya}', "");
Expect(1, 2815, '\p{^Blk=oriya}', "");
Expect(1, 2815, '\P{Blk=oriya}', "");
Expect(0, 2815, '\P{^Blk=oriya}', "");
Expect(1, 2943, '\p{Blk= Oriya}', "");
Expect(0, 2943, '\p{^Blk= Oriya}', "");
Expect(0, 2943, '\P{Blk= Oriya}', "");
Expect(1, 2943, '\P{^Blk= Oriya}', "");
Expect(0, 2815, '\p{Blk= Oriya}', "");
Expect(1, 2815, '\p{^Blk= Oriya}', "");
Expect(1, 2815, '\P{Blk= Oriya}', "");
Expect(0, 2815, '\P{^Blk= Oriya}', "");
Error('\p{Is_Block=-:=ORIYA}');
Error('\P{Is_Block=-:=ORIYA}');
Expect(1, 2943, '\p{Is_Block=oriya}', "");
Expect(0, 2943, '\p{^Is_Block=oriya}', "");
Expect(0, 2943, '\P{Is_Block=oriya}', "");
Expect(1, 2943, '\P{^Is_Block=oriya}', "");
Expect(0, 2815, '\p{Is_Block=oriya}', "");
Expect(1, 2815, '\p{^Is_Block=oriya}', "");
Expect(1, 2815, '\P{Is_Block=oriya}', "");
Expect(0, 2815, '\P{^Is_Block=oriya}', "");
Expect(1, 2943, '\p{Is_Block=-_Oriya}', "");
Expect(0, 2943, '\p{^Is_Block=-_Oriya}', "");
Expect(0, 2943, '\P{Is_Block=-_Oriya}', "");
Expect(1, 2943, '\P{^Is_Block=-_Oriya}', "");
Expect(0, 2815, '\p{Is_Block=-_Oriya}', "");
Expect(1, 2815, '\p{^Is_Block=-_Oriya}', "");
Expect(1, 2815, '\P{Is_Block=-_Oriya}', "");
Expect(0, 2815, '\P{^Is_Block=-_Oriya}', "");
Error('\p{Is_Blk=	/a/Oriya}');
Error('\P{Is_Blk=	/a/Oriya}');
Expect(1, 2943, '\p{Is_Blk=oriya}', "");
Expect(0, 2943, '\p{^Is_Blk=oriya}', "");
Expect(0, 2943, '\P{Is_Blk=oriya}', "");
Expect(1, 2943, '\P{^Is_Blk=oriya}', "");
Expect(0, 2815, '\p{Is_Blk=oriya}', "");
Expect(1, 2815, '\p{^Is_Blk=oriya}', "");
Expect(1, 2815, '\P{Is_Blk=oriya}', "");
Expect(0, 2815, '\P{^Is_Blk=oriya}', "");
Expect(1, 2943, '\p{Is_Blk=  Oriya}', "");
Expect(0, 2943, '\p{^Is_Blk=  Oriya}', "");
Expect(0, 2943, '\P{Is_Blk=  Oriya}', "");
Expect(1, 2943, '\P{^Is_Blk=  Oriya}', "");
Expect(0, 2815, '\p{Is_Blk=  Oriya}', "");
Expect(1, 2815, '\p{^Is_Blk=  Oriya}', "");
Expect(1, 2815, '\P{Is_Blk=  Oriya}', "");
Expect(0, 2815, '\P{^Is_Blk=  Oriya}', "");
Error('\p{Block=_Linear:=B	IDEOGRAMS}');
Error('\P{Block=_Linear:=B	IDEOGRAMS}');
Expect(1, 65791, '\p{Block=linearbideograms}', "");
Expect(0, 65791, '\p{^Block=linearbideograms}', "");
Expect(0, 65791, '\P{Block=linearbideograms}', "");
Expect(1, 65791, '\P{^Block=linearbideograms}', "");
Expect(0, 65663, '\p{Block=linearbideograms}', "");
Expect(1, 65663, '\p{^Block=linearbideograms}', "");
Expect(1, 65663, '\P{Block=linearbideograms}', "");
Expect(0, 65663, '\P{^Block=linearbideograms}', "");
Expect(1, 65791, '\p{Block=-_LinearBideograms}', "");
Expect(0, 65791, '\p{^Block=-_LinearBideograms}', "");
Expect(0, 65791, '\P{Block=-_LinearBideograms}', "");
Expect(1, 65791, '\P{^Block=-_LinearBideograms}', "");
Expect(0, 65663, '\p{Block=-_LinearBideograms}', "");
Expect(1, 65663, '\p{^Block=-_LinearBideograms}', "");
Expect(1, 65663, '\P{Block=-_LinearBideograms}', "");
Expect(0, 65663, '\P{^Block=-_LinearBideograms}', "");
Error('\p{Blk=_:=LINEAR BIdeograms}');
Error('\P{Blk=_:=LINEAR BIdeograms}');
Expect(1, 65791, '\p{Blk=linearbideograms}', "");
Expect(0, 65791, '\p{^Blk=linearbideograms}', "");
Expect(0, 65791, '\P{Blk=linearbideograms}', "");
Expect(1, 65791, '\P{^Blk=linearbideograms}', "");
Expect(0, 65663, '\p{Blk=linearbideograms}', "");
Expect(1, 65663, '\p{^Blk=linearbideograms}', "");
Expect(1, 65663, '\P{Blk=linearbideograms}', "");
Expect(0, 65663, '\P{^Blk=linearbideograms}', "");
Expect(1, 65791, '\p{Blk:   	 Linear_B_ideograms}', "");
Expect(0, 65791, '\p{^Blk:   	 Linear_B_ideograms}', "");
Expect(0, 65791, '\P{Blk:   	 Linear_B_ideograms}', "");
Expect(1, 65791, '\P{^Blk:   	 Linear_B_ideograms}', "");
Expect(0, 65663, '\p{Blk:   	 Linear_B_ideograms}', "");
Expect(1, 65663, '\p{^Blk:   	 Linear_B_ideograms}', "");
Expect(1, 65663, '\P{Blk:   	 Linear_B_ideograms}', "");
Expect(0, 65663, '\P{^Blk:   	 Linear_B_ideograms}', "");
Error('\p{Is_Block:	 	linear:=b	ideograms}');
Error('\P{Is_Block:	 	linear:=b	ideograms}');
Expect(1, 65791, '\p{Is_Block=linearbideograms}', "");
Expect(0, 65791, '\p{^Is_Block=linearbideograms}', "");
Expect(0, 65791, '\P{Is_Block=linearbideograms}', "");
Expect(1, 65791, '\P{^Is_Block=linearbideograms}', "");
Expect(0, 65663, '\p{Is_Block=linearbideograms}', "");
Expect(1, 65663, '\p{^Is_Block=linearbideograms}', "");
Expect(1, 65663, '\P{Is_Block=linearbideograms}', "");
Expect(0, 65663, '\P{^Is_Block=linearbideograms}', "");
Expect(1, 65791, '\p{Is_Block=	Linear_B-ideograms}', "");
Expect(0, 65791, '\p{^Is_Block=	Linear_B-ideograms}', "");
Expect(0, 65791, '\P{Is_Block=	Linear_B-ideograms}', "");
Expect(1, 65791, '\P{^Is_Block=	Linear_B-ideograms}', "");
Expect(0, 65663, '\p{Is_Block=	Linear_B-ideograms}', "");
Expect(1, 65663, '\p{^Is_Block=	Linear_B-ideograms}', "");
Expect(1, 65663, '\P{Is_Block=	Linear_B-ideograms}', "");
Expect(0, 65663, '\P{^Is_Block=	Linear_B-ideograms}', "");
Error('\p{Is_Blk=_/a/Linear_B	IDEOGRAMS}');
Error('\P{Is_Blk=_/a/Linear_B	IDEOGRAMS}');
Expect(1, 65791, '\p{Is_Blk: linearbideograms}', "");
Expect(0, 65791, '\p{^Is_Blk: linearbideograms}', "");
Expect(0, 65791, '\P{Is_Blk: linearbideograms}', "");
Expect(1, 65791, '\P{^Is_Blk: linearbideograms}', "");
Expect(0, 65663, '\p{Is_Blk: linearbideograms}', "");
Expect(1, 65663, '\p{^Is_Blk: linearbideograms}', "");
Expect(1, 65663, '\P{Is_Blk: linearbideograms}', "");
Expect(0, 65663, '\P{^Is_Blk: linearbideograms}', "");
Expect(1, 65791, '\p{Is_Blk=  LinearB-Ideograms}', "");
Expect(0, 65791, '\p{^Is_Blk=  LinearB-Ideograms}', "");
Expect(0, 65791, '\P{Is_Blk=  LinearB-Ideograms}', "");
Expect(1, 65791, '\P{^Is_Blk=  LinearB-Ideograms}', "");
Expect(0, 65663, '\p{Is_Blk=  LinearB-Ideograms}', "");
Expect(1, 65663, '\p{^Is_Blk=  LinearB-Ideograms}', "");
Expect(1, 65663, '\P{Is_Blk=  LinearB-Ideograms}', "");
Expect(0, 65663, '\P{^Is_Blk=  LinearB-Ideograms}', "");
Error('\p{Block=:=_	Cherokee}');
Error('\P{Block=:=_	Cherokee}');
Expect(1, 5119, '\p{Block=cherokee}', "");
Expect(0, 5119, '\p{^Block=cherokee}', "");
Expect(0, 5119, '\P{Block=cherokee}', "");
Expect(1, 5119, '\P{^Block=cherokee}', "");
Expect(0, 5023, '\p{Block=cherokee}', "");
Expect(1, 5023, '\p{^Block=cherokee}', "");
Expect(1, 5023, '\P{Block=cherokee}', "");
Expect(0, 5023, '\P{^Block=cherokee}', "");
Expect(1, 5119, '\p{Block=	cherokee}', "");
Expect(0, 5119, '\p{^Block=	cherokee}', "");
Expect(0, 5119, '\P{Block=	cherokee}', "");
Expect(1, 5119, '\P{^Block=	cherokee}', "");
Expect(0, 5023, '\p{Block=	cherokee}', "");
Expect(1, 5023, '\p{^Block=	cherokee}', "");
Expect(1, 5023, '\P{Block=	cherokee}', "");
Expect(0, 5023, '\P{^Block=	cherokee}', "");
Error('\p{Blk=:= 	Cherokee}');
Error('\P{Blk=:= 	Cherokee}');
Expect(1, 5119, '\p{Blk=cherokee}', "");
Expect(0, 5119, '\p{^Blk=cherokee}', "");
Expect(0, 5119, '\P{Blk=cherokee}', "");
Expect(1, 5119, '\P{^Blk=cherokee}', "");
Expect(0, 5023, '\p{Blk=cherokee}', "");
Expect(1, 5023, '\p{^Blk=cherokee}', "");
Expect(1, 5023, '\P{Blk=cherokee}', "");
Expect(0, 5023, '\P{^Blk=cherokee}', "");
Expect(1, 5119, '\p{Blk=__CHEROKEE}', "");
Expect(0, 5119, '\p{^Blk=__CHEROKEE}', "");
Expect(0, 5119, '\P{Blk=__CHEROKEE}', "");
Expect(1, 5119, '\P{^Blk=__CHEROKEE}', "");
Expect(0, 5023, '\p{Blk=__CHEROKEE}', "");
Expect(1, 5023, '\p{^Blk=__CHEROKEE}', "");
Expect(1, 5023, '\P{Blk=__CHEROKEE}', "");
Expect(0, 5023, '\P{^Blk=__CHEROKEE}', "");
Error('\p{Is_Block=/a/	 Cherokee}');
Error('\P{Is_Block=/a/	 Cherokee}');
Expect(1, 5119, '\p{Is_Block=cherokee}', "");
Expect(0, 5119, '\p{^Is_Block=cherokee}', "");
Expect(0, 5119, '\P{Is_Block=cherokee}', "");
Expect(1, 5119, '\P{^Is_Block=cherokee}', "");
Expect(0, 5023, '\p{Is_Block=cherokee}', "");
Expect(1, 5023, '\p{^Is_Block=cherokee}', "");
Expect(1, 5023, '\P{Is_Block=cherokee}', "");
Expect(0, 5023, '\P{^Is_Block=cherokee}', "");
Expect(1, 5119, '\p{Is_Block:	Cherokee}', "");
Expect(0, 5119, '\p{^Is_Block:	Cherokee}', "");
Expect(0, 5119, '\P{Is_Block:	Cherokee}', "");
Expect(1, 5119, '\P{^Is_Block:	Cherokee}', "");
Expect(0, 5023, '\p{Is_Block:	Cherokee}', "");
Expect(1, 5023, '\p{^Is_Block:	Cherokee}', "");
Expect(1, 5023, '\P{Is_Block:	Cherokee}', "");
Expect(0, 5023, '\P{^Is_Block:	Cherokee}', "");
Error('\p{Is_Blk=_cherokee:=}');
Error('\P{Is_Blk=_cherokee:=}');
Expect(1, 5119, '\p{Is_Blk=cherokee}', "");
Expect(0, 5119, '\p{^Is_Blk=cherokee}', "");
Expect(0, 5119, '\P{Is_Blk=cherokee}', "");
Expect(1, 5119, '\P{^Is_Blk=cherokee}', "");
Expect(0, 5023, '\p{Is_Blk=cherokee}', "");
Expect(1, 5023, '\p{^Is_Blk=cherokee}', "");
Expect(1, 5023, '\P{Is_Blk=cherokee}', "");
Expect(0, 5023, '\P{^Is_Blk=cherokee}', "");
Expect(1, 5119, '\p{Is_Blk=_-cherokee}', "");
Expect(0, 5119, '\p{^Is_Blk=_-cherokee}', "");
Expect(0, 5119, '\P{Is_Blk=_-cherokee}', "");
Expect(1, 5119, '\P{^Is_Blk=_-cherokee}', "");
Expect(0, 5023, '\p{Is_Blk=_-cherokee}', "");
Expect(1, 5023, '\p{^Is_Blk=_-cherokee}', "");
Expect(1, 5023, '\P{Is_Blk=_-cherokee}', "");
Expect(0, 5023, '\P{^Is_Blk=_-cherokee}', "");
Error('\p{Block= :=Bopomofo-Extended}');
Error('\P{Block= :=Bopomofo-Extended}');
Expect(1, 12735, '\p{Block=bopomofoextended}', "");
Expect(0, 12735, '\p{^Block=bopomofoextended}', "");
Expect(0, 12735, '\P{Block=bopomofoextended}', "");
Expect(1, 12735, '\P{^Block=bopomofoextended}', "");
Expect(0, 12703, '\p{Block=bopomofoextended}', "");
Expect(1, 12703, '\p{^Block=bopomofoextended}', "");
Expect(1, 12703, '\P{Block=bopomofoextended}', "");
Expect(0, 12703, '\P{^Block=bopomofoextended}', "");
Expect(1, 12735, '\p{Block=- bopomofo-extended}', "");
Expect(0, 12735, '\p{^Block=- bopomofo-extended}', "");
Expect(0, 12735, '\P{Block=- bopomofo-extended}', "");
Expect(1, 12735, '\P{^Block=- bopomofo-extended}', "");
Expect(0, 12703, '\p{Block=- bopomofo-extended}', "");
Expect(1, 12703, '\p{^Block=- bopomofo-extended}', "");
Expect(1, 12703, '\P{Block=- bopomofo-extended}', "");
Expect(0, 12703, '\P{^Block=- bopomofo-extended}', "");
Error('\p{Blk:   Bopomofo_extended/a/}');
Error('\P{Blk:   Bopomofo_extended/a/}');
Expect(1, 12735, '\p{Blk=bopomofoextended}', "");
Expect(0, 12735, '\p{^Blk=bopomofoextended}', "");
Expect(0, 12735, '\P{Blk=bopomofoextended}', "");
Expect(1, 12735, '\P{^Blk=bopomofoextended}', "");
Expect(0, 12703, '\p{Blk=bopomofoextended}', "");
Expect(1, 12703, '\p{^Blk=bopomofoextended}', "");
Expect(1, 12703, '\P{Blk=bopomofoextended}', "");
Expect(0, 12703, '\P{^Blk=bopomofoextended}', "");
Expect(1, 12735, '\p{Blk=Bopomofo-Extended}', "");
Expect(0, 12735, '\p{^Blk=Bopomofo-Extended}', "");
Expect(0, 12735, '\P{Blk=Bopomofo-Extended}', "");
Expect(1, 12735, '\P{^Blk=Bopomofo-Extended}', "");
Expect(0, 12703, '\p{Blk=Bopomofo-Extended}', "");
Expect(1, 12703, '\p{^Blk=Bopomofo-Extended}', "");
Expect(1, 12703, '\P{Blk=Bopomofo-Extended}', "");
Expect(0, 12703, '\P{^Blk=Bopomofo-Extended}', "");
Error('\p{Is_Block=-:=Bopomofo	extended}');
Error('\P{Is_Block=-:=Bopomofo	extended}');
Expect(1, 12735, '\p{Is_Block=bopomofoextended}', "");
Expect(0, 12735, '\p{^Is_Block=bopomofoextended}', "");
Expect(0, 12735, '\P{Is_Block=bopomofoextended}', "");
Expect(1, 12735, '\P{^Is_Block=bopomofoextended}', "");
Expect(0, 12703, '\p{Is_Block=bopomofoextended}', "");
Expect(1, 12703, '\p{^Is_Block=bopomofoextended}', "");
Expect(1, 12703, '\P{Is_Block=bopomofoextended}', "");
Expect(0, 12703, '\P{^Is_Block=bopomofoextended}', "");
Expect(1, 12735, '\p{Is_Block:  	bopomofoextended}', "");
Expect(0, 12735, '\p{^Is_Block:  	bopomofoextended}', "");
Expect(0, 12735, '\P{Is_Block:  	bopomofoextended}', "");
Expect(1, 12735, '\P{^Is_Block:  	bopomofoextended}', "");
Expect(0, 12703, '\p{Is_Block:  	bopomofoextended}', "");
Expect(1, 12703, '\p{^Is_Block:  	bopomofoextended}', "");
Expect(1, 12703, '\P{Is_Block:  	bopomofoextended}', "");
Expect(0, 12703, '\P{^Is_Block:  	bopomofoextended}', "");
Error('\p{Is_Blk=- BOPOMOFO_extended:=}');
Error('\P{Is_Blk=- BOPOMOFO_extended:=}');
Expect(1, 12735, '\p{Is_Blk: bopomofoextended}', "");
Expect(0, 12735, '\p{^Is_Blk: bopomofoextended}', "");
Expect(0, 12735, '\P{Is_Blk: bopomofoextended}', "");
Expect(1, 12735, '\P{^Is_Blk: bopomofoextended}', "");
Expect(0, 12703, '\p{Is_Blk: bopomofoextended}', "");
Expect(1, 12703, '\p{^Is_Blk: bopomofoextended}', "");
Expect(1, 12703, '\P{Is_Blk: bopomofoextended}', "");
Expect(0, 12703, '\P{^Is_Blk: bopomofoextended}', "");
Expect(1, 12735, '\p{Is_Blk=	_Bopomofo-extended}', "");
Expect(0, 12735, '\p{^Is_Blk=	_Bopomofo-extended}', "");
Expect(0, 12735, '\P{Is_Blk=	_Bopomofo-extended}', "");
Expect(1, 12735, '\P{^Is_Blk=	_Bopomofo-extended}', "");
Expect(0, 12703, '\p{Is_Blk=	_Bopomofo-extended}', "");
Expect(1, 12703, '\p{^Is_Blk=	_Bopomofo-extended}', "");
Expect(1, 12703, '\P{Is_Blk=	_Bopomofo-extended}', "");
Expect(0, 12703, '\P{^Is_Blk=	_Bopomofo-extended}', "");
Error('\p{Block:   	Ogham:=}');
Error('\P{Block:   	Ogham:=}');
Expect(1, 5791, '\p{Block=ogham}', "");
Expect(0, 5791, '\p{^Block=ogham}', "");
Expect(0, 5791, '\P{Block=ogham}', "");
Expect(1, 5791, '\P{^Block=ogham}', "");
Expect(0, 5759, '\p{Block=ogham}', "");
Expect(1, 5759, '\p{^Block=ogham}', "");
Expect(1, 5759, '\P{Block=ogham}', "");
Expect(0, 5759, '\P{^Block=ogham}', "");
Expect(1, 5791, '\p{Block= Ogham}', "");
Expect(0, 5791, '\p{^Block= Ogham}', "");
Expect(0, 5791, '\P{Block= Ogham}', "");
Expect(1, 5791, '\P{^Block= Ogham}', "");
Expect(0, 5759, '\p{Block= Ogham}', "");
Expect(1, 5759, '\p{^Block= Ogham}', "");
Expect(1, 5759, '\P{Block= Ogham}', "");
Expect(0, 5759, '\P{^Block= Ogham}', "");
Error('\p{Blk:OGHAM/a/}');
Error('\P{Blk:OGHAM/a/}');
Expect(1, 5791, '\p{Blk:	ogham}', "");
Expect(0, 5791, '\p{^Blk:	ogham}', "");
Expect(0, 5791, '\P{Blk:	ogham}', "");
Expect(1, 5791, '\P{^Blk:	ogham}', "");
Expect(0, 5759, '\p{Blk:	ogham}', "");
Expect(1, 5759, '\p{^Blk:	ogham}', "");
Expect(1, 5759, '\P{Blk:	ogham}', "");
Expect(0, 5759, '\P{^Blk:	ogham}', "");
Expect(1, 5791, '\p{Blk= -ogham}', "");
Expect(0, 5791, '\p{^Blk= -ogham}', "");
Expect(0, 5791, '\P{Blk= -ogham}', "");
Expect(1, 5791, '\P{^Blk= -ogham}', "");
Expect(0, 5759, '\p{Blk= -ogham}', "");
Expect(1, 5759, '\p{^Blk= -ogham}', "");
Expect(1, 5759, '\P{Blk= -ogham}', "");
Expect(0, 5759, '\P{^Blk= -ogham}', "");
Error('\p{Is_Block=-:=OGHAM}');
Error('\P{Is_Block=-:=OGHAM}');
Expect(1, 5791, '\p{Is_Block=ogham}', "");
Expect(0, 5791, '\p{^Is_Block=ogham}', "");
Expect(0, 5791, '\P{Is_Block=ogham}', "");
Expect(1, 5791, '\P{^Is_Block=ogham}', "");
Expect(0, 5759, '\p{Is_Block=ogham}', "");
Expect(1, 5759, '\p{^Is_Block=ogham}', "");
Expect(1, 5759, '\P{Is_Block=ogham}', "");
Expect(0, 5759, '\P{^Is_Block=ogham}', "");
Expect(1, 5791, '\p{Is_Block= 	ogham}', "");
Expect(0, 5791, '\p{^Is_Block= 	ogham}', "");
Expect(0, 5791, '\P{Is_Block= 	ogham}', "");
Expect(1, 5791, '\P{^Is_Block= 	ogham}', "");
Expect(0, 5759, '\p{Is_Block= 	ogham}', "");
Expect(1, 5759, '\p{^Is_Block= 	ogham}', "");
Expect(1, 5759, '\P{Is_Block= 	ogham}', "");
Expect(0, 5759, '\P{^Is_Block= 	ogham}', "");
Error('\p{Is_Blk= :=Ogham}');
Error('\P{Is_Blk= :=Ogham}');
Expect(1, 5791, '\p{Is_Blk=ogham}', "");
Expect(0, 5791, '\p{^Is_Blk=ogham}', "");
Expect(0, 5791, '\P{Is_Blk=ogham}', "");
Expect(1, 5791, '\P{^Is_Blk=ogham}', "");
Expect(0, 5759, '\p{Is_Blk=ogham}', "");
Expect(1, 5759, '\p{^Is_Blk=ogham}', "");
Expect(1, 5759, '\P{Is_Blk=ogham}', "");
Expect(0, 5759, '\P{^Is_Blk=ogham}', "");
Expect(1, 5791, '\p{Is_Blk=  OGHAM}', "");
Expect(0, 5791, '\p{^Is_Blk=  OGHAM}', "");
Expect(0, 5791, '\P{Is_Blk=  OGHAM}', "");
Expect(1, 5791, '\P{^Is_Blk=  OGHAM}', "");
Expect(0, 5759, '\p{Is_Blk=  OGHAM}', "");
Expect(1, 5759, '\p{^Is_Blk=  OGHAM}', "");
Expect(1, 5759, '\P{Is_Blk=  OGHAM}', "");
Expect(0, 5759, '\P{^Is_Blk=  OGHAM}', "");
Error('\p{Block=	/a/SAMARITAN}');
Error('\P{Block=	/a/SAMARITAN}');
Expect(1, 2111, '\p{Block=samaritan}', "");
Expect(0, 2111, '\p{^Block=samaritan}', "");
Expect(0, 2111, '\P{Block=samaritan}', "");
Expect(1, 2111, '\P{^Block=samaritan}', "");
Expect(0, 2047, '\p{Block=samaritan}', "");
Expect(1, 2047, '\p{^Block=samaritan}', "");
Expect(1, 2047, '\P{Block=samaritan}', "");
Expect(0, 2047, '\P{^Block=samaritan}', "");
Expect(1, 2111, '\p{Block=	-Samaritan}', "");
Expect(0, 2111, '\p{^Block=	-Samaritan}', "");
Expect(0, 2111, '\P{Block=	-Samaritan}', "");
Expect(1, 2111, '\P{^Block=	-Samaritan}', "");
Expect(0, 2047, '\p{Block=	-Samaritan}', "");
Expect(1, 2047, '\p{^Block=	-Samaritan}', "");
Expect(1, 2047, '\P{Block=	-Samaritan}', "");
Expect(0, 2047, '\P{^Block=	-Samaritan}', "");
Error('\p{Blk= /a/SAMARITAN}');
Error('\P{Blk= /a/SAMARITAN}');
Expect(1, 2111, '\p{Blk=samaritan}', "");
Expect(0, 2111, '\p{^Blk=samaritan}', "");
Expect(0, 2111, '\P{Blk=samaritan}', "");
Expect(1, 2111, '\P{^Blk=samaritan}', "");
Expect(0, 2047, '\p{Blk=samaritan}', "");
Expect(1, 2047, '\p{^Blk=samaritan}', "");
Expect(1, 2047, '\P{Blk=samaritan}', "");
Expect(0, 2047, '\P{^Blk=samaritan}', "");
Expect(1, 2111, '\p{Blk= Samaritan}', "");
Expect(0, 2111, '\p{^Blk= Samaritan}', "");
Expect(0, 2111, '\P{Blk= Samaritan}', "");
Expect(1, 2111, '\P{^Blk= Samaritan}', "");
Expect(0, 2047, '\p{Blk= Samaritan}', "");
Expect(1, 2047, '\p{^Blk= Samaritan}', "");
Expect(1, 2047, '\P{Blk= Samaritan}', "");
Expect(0, 2047, '\P{^Blk= Samaritan}', "");
Error('\p{Is_Block=-Samaritan:=}');
Error('\P{Is_Block=-Samaritan:=}');
Expect(1, 2111, '\p{Is_Block=samaritan}', "");
Expect(0, 2111, '\p{^Is_Block=samaritan}', "");
Expect(0, 2111, '\P{Is_Block=samaritan}', "");
Expect(1, 2111, '\P{^Is_Block=samaritan}', "");
Expect(0, 2047, '\p{Is_Block=samaritan}', "");
Expect(1, 2047, '\p{^Is_Block=samaritan}', "");
Expect(1, 2047, '\P{Is_Block=samaritan}', "");
Expect(0, 2047, '\P{^Is_Block=samaritan}', "");
Expect(1, 2111, '\p{Is_Block=_samaritan}', "");
Expect(0, 2111, '\p{^Is_Block=_samaritan}', "");
Expect(0, 2111, '\P{Is_Block=_samaritan}', "");
Expect(1, 2111, '\P{^Is_Block=_samaritan}', "");
Expect(0, 2047, '\p{Is_Block=_samaritan}', "");
Expect(1, 2047, '\p{^Is_Block=_samaritan}', "");
Expect(1, 2047, '\P{Is_Block=_samaritan}', "");
Expect(0, 2047, '\P{^Is_Block=_samaritan}', "");
Error('\p{Is_Blk=	 Samaritan/a/}');
Error('\P{Is_Blk=	 Samaritan/a/}');
Expect(1, 2111, '\p{Is_Blk:samaritan}', "");
Expect(0, 2111, '\p{^Is_Blk:samaritan}', "");
Expect(0, 2111, '\P{Is_Blk:samaritan}', "");
Expect(1, 2111, '\P{^Is_Blk:samaritan}', "");
Expect(0, 2047, '\p{Is_Blk:samaritan}', "");
Expect(1, 2047, '\p{^Is_Blk:samaritan}', "");
Expect(1, 2047, '\P{Is_Blk:samaritan}', "");
Expect(0, 2047, '\P{^Is_Blk:samaritan}', "");
Expect(1, 2111, '\p{Is_Blk=	_Samaritan}', "");
Expect(0, 2111, '\p{^Is_Blk=	_Samaritan}', "");
Expect(0, 2111, '\P{Is_Blk=	_Samaritan}', "");
Expect(1, 2111, '\P{^Is_Blk=	_Samaritan}', "");
Expect(0, 2047, '\p{Is_Blk=	_Samaritan}', "");
Expect(1, 2047, '\p{^Is_Blk=	_Samaritan}', "");
Expect(1, 2047, '\P{Is_Blk=	_Samaritan}', "");
Expect(0, 2047, '\P{^Is_Blk=	_Samaritan}', "");
Error('\p{Block=:= 	vai}');
Error('\P{Block=:= 	vai}');
Expect(1, 42559, '\p{Block=vai}', "");
Expect(0, 42559, '\p{^Block=vai}', "");
Expect(0, 42559, '\P{Block=vai}', "");
Expect(1, 42559, '\P{^Block=vai}', "");
Expect(0, 42239, '\p{Block=vai}', "");
Expect(1, 42239, '\p{^Block=vai}', "");
Expect(1, 42239, '\P{Block=vai}', "");
Expect(0, 42239, '\P{^Block=vai}', "");
Expect(1, 42559, '\p{Block=	Vai}', "");
Expect(0, 42559, '\p{^Block=	Vai}', "");
Expect(0, 42559, '\P{Block=	Vai}', "");
Expect(1, 42559, '\P{^Block=	Vai}', "");
Expect(0, 42239, '\p{Block=	Vai}', "");
Expect(1, 42239, '\p{^Block=	Vai}', "");
Expect(1, 42239, '\P{Block=	Vai}', "");
Expect(0, 42239, '\P{^Block=	Vai}', "");
Error('\p{Blk=_/a/vai}');
Error('\P{Blk=_/a/vai}');
Expect(1, 42559, '\p{Blk=vai}', "");
Expect(0, 42559, '\p{^Blk=vai}', "");
Expect(0, 42559, '\P{Blk=vai}', "");
Expect(1, 42559, '\P{^Blk=vai}', "");
Expect(0, 42239, '\p{Blk=vai}', "");
Expect(1, 42239, '\p{^Blk=vai}', "");
Expect(1, 42239, '\P{Blk=vai}', "");
Expect(0, 42239, '\P{^Blk=vai}', "");
Expect(1, 42559, '\p{Blk=--vai}', "");
Expect(0, 42559, '\p{^Blk=--vai}', "");
Expect(0, 42559, '\P{Blk=--vai}', "");
Expect(1, 42559, '\P{^Blk=--vai}', "");
Expect(0, 42239, '\p{Blk=--vai}', "");
Expect(1, 42239, '\p{^Blk=--vai}', "");
Expect(1, 42239, '\P{Blk=--vai}', "");
Expect(0, 42239, '\P{^Blk=--vai}', "");
Error('\p{Is_Block=/a/ _VAI}');
Error('\P{Is_Block=/a/ _VAI}');
Expect(1, 42559, '\p{Is_Block=vai}', "");
Expect(0, 42559, '\p{^Is_Block=vai}', "");
Expect(0, 42559, '\P{Is_Block=vai}', "");
Expect(1, 42559, '\P{^Is_Block=vai}', "");
Expect(0, 42239, '\p{Is_Block=vai}', "");
Expect(1, 42239, '\p{^Is_Block=vai}', "");
Expect(1, 42239, '\P{Is_Block=vai}', "");
Expect(0, 42239, '\P{^Is_Block=vai}', "");
Expect(1, 42559, '\p{Is_Block=	vai}', "");
Expect(0, 42559, '\p{^Is_Block=	vai}', "");
Expect(0, 42559, '\P{Is_Block=	vai}', "");
Expect(1, 42559, '\P{^Is_Block=	vai}', "");
Expect(0, 42239, '\p{Is_Block=	vai}', "");
Expect(1, 42239, '\p{^Is_Block=	vai}', "");
Expect(1, 42239, '\P{Is_Block=	vai}', "");
Expect(0, 42239, '\P{^Is_Block=	vai}', "");
Error('\p{Is_Blk=-/a/Vai}');
Error('\P{Is_Blk=-/a/Vai}');
Expect(1, 42559, '\p{Is_Blk=vai}', "");
Expect(0, 42559, '\p{^Is_Blk=vai}', "");
Expect(0, 42559, '\P{Is_Blk=vai}', "");
Expect(1, 42559, '\P{^Is_Blk=vai}', "");
Expect(0, 42239, '\p{Is_Blk=vai}', "");
Expect(1, 42239, '\p{^Is_Blk=vai}', "");
Expect(1, 42239, '\P{Is_Blk=vai}', "");
Expect(0, 42239, '\P{^Is_Blk=vai}', "");
Expect(1, 42559, '\p{Is_Blk=__Vai}', "");
Expect(0, 42559, '\p{^Is_Blk=__Vai}', "");
Expect(0, 42559, '\P{Is_Blk=__Vai}', "");
Expect(1, 42559, '\P{^Is_Blk=__Vai}', "");
Expect(0, 42239, '\p{Is_Blk=__Vai}', "");
Expect(1, 42239, '\p{^Is_Blk=__Vai}', "");
Expect(1, 42239, '\P{Is_Blk=__Vai}', "");
Expect(0, 42239, '\P{^Is_Blk=__Vai}', "");
Error('\p{Block=-:=generalPunctuation}');
Error('\P{Block=-:=generalPunctuation}');
Expect(1, 8303, '\p{Block=generalpunctuation}', "");
Expect(0, 8303, '\p{^Block=generalpunctuation}', "");
Expect(0, 8303, '\P{Block=generalpunctuation}', "");
Expect(1, 8303, '\P{^Block=generalpunctuation}', "");
Expect(0, 8191, '\p{Block=generalpunctuation}', "");
Expect(1, 8191, '\p{^Block=generalpunctuation}', "");
Expect(1, 8191, '\P{Block=generalpunctuation}', "");
Expect(0, 8191, '\P{^Block=generalpunctuation}', "");
Expect(1, 8303, '\p{Block=	General-Punctuation}', "");
Expect(0, 8303, '\p{^Block=	General-Punctuation}', "");
Expect(0, 8303, '\P{Block=	General-Punctuation}', "");
Expect(1, 8303, '\P{^Block=	General-Punctuation}', "");
Expect(0, 8191, '\p{Block=	General-Punctuation}', "");
Expect(1, 8191, '\p{^Block=	General-Punctuation}', "");
Expect(1, 8191, '\P{Block=	General-Punctuation}', "");
Expect(0, 8191, '\P{^Block=	General-Punctuation}', "");
Error('\p{Blk=/a/_ General_punctuation}');
Error('\P{Blk=/a/_ General_punctuation}');
Expect(1, 8303, '\p{Blk=generalpunctuation}', "");
Expect(0, 8303, '\p{^Blk=generalpunctuation}', "");
Expect(0, 8303, '\P{Blk=generalpunctuation}', "");
Expect(1, 8303, '\P{^Blk=generalpunctuation}', "");
Expect(0, 8191, '\p{Blk=generalpunctuation}', "");
Expect(1, 8191, '\p{^Blk=generalpunctuation}', "");
Expect(1, 8191, '\P{Blk=generalpunctuation}', "");
Expect(0, 8191, '\P{^Blk=generalpunctuation}', "");
Expect(1, 8303, '\p{Blk=-	GENERAL_PUNCTUATION}', "");
Expect(0, 8303, '\p{^Blk=-	GENERAL_PUNCTUATION}', "");
Expect(0, 8303, '\P{Blk=-	GENERAL_PUNCTUATION}', "");
Expect(1, 8303, '\P{^Blk=-	GENERAL_PUNCTUATION}', "");
Expect(0, 8191, '\p{Blk=-	GENERAL_PUNCTUATION}', "");
Expect(1, 8191, '\p{^Blk=-	GENERAL_PUNCTUATION}', "");
Expect(1, 8191, '\P{Blk=-	GENERAL_PUNCTUATION}', "");
Expect(0, 8191, '\P{^Blk=-	GENERAL_PUNCTUATION}', "");
Error('\p{Is_Block:   :=	 general	punctuation}');
Error('\P{Is_Block:   :=	 general	punctuation}');
Expect(1, 8303, '\p{Is_Block=generalpunctuation}', "");
Expect(0, 8303, '\p{^Is_Block=generalpunctuation}', "");
Expect(0, 8303, '\P{Is_Block=generalpunctuation}', "");
Expect(1, 8303, '\P{^Is_Block=generalpunctuation}', "");
Expect(0, 8191, '\p{Is_Block=generalpunctuation}', "");
Expect(1, 8191, '\p{^Is_Block=generalpunctuation}', "");
Expect(1, 8191, '\P{Is_Block=generalpunctuation}', "");
Expect(0, 8191, '\P{^Is_Block=generalpunctuation}', "");
Expect(1, 8303, '\p{Is_Block=_general PUNCTUATION}', "");
Expect(0, 8303, '\p{^Is_Block=_general PUNCTUATION}', "");
Expect(0, 8303, '\P{Is_Block=_general PUNCTUATION}', "");
Expect(1, 8303, '\P{^Is_Block=_general PUNCTUATION}', "");
Expect(0, 8191, '\p{Is_Block=_general PUNCTUATION}', "");
Expect(1, 8191, '\p{^Is_Block=_general PUNCTUATION}', "");
Expect(1, 8191, '\P{Is_Block=_general PUNCTUATION}', "");
Expect(0, 8191, '\P{^Is_Block=_general PUNCTUATION}', "");
Error('\p{Is_Blk=/a/-_Generalpunctuation}');
Error('\P{Is_Blk=/a/-_Generalpunctuation}');
Expect(1, 8303, '\p{Is_Blk=generalpunctuation}', "");
Expect(0, 8303, '\p{^Is_Blk=generalpunctuation}', "");
Expect(0, 8303, '\P{Is_Blk=generalpunctuation}', "");
Expect(1, 8303, '\P{^Is_Blk=generalpunctuation}', "");
Expect(0, 8191, '\p{Is_Blk=generalpunctuation}', "");
Expect(1, 8191, '\p{^Is_Blk=generalpunctuation}', "");
Expect(1, 8191, '\P{Is_Blk=generalpunctuation}', "");
Expect(0, 8191, '\P{^Is_Blk=generalpunctuation}', "");
Expect(1, 8303, '\p{Is_Blk=	-GENERAL	Punctuation}', "");
Expect(0, 8303, '\p{^Is_Blk=	-GENERAL	Punctuation}', "");
Expect(0, 8303, '\P{Is_Blk=	-GENERAL	Punctuation}', "");
Expect(1, 8303, '\P{^Is_Blk=	-GENERAL	Punctuation}', "");
Expect(0, 8191, '\p{Is_Blk=	-GENERAL	Punctuation}', "");
Expect(1, 8191, '\p{^Is_Blk=	-GENERAL	Punctuation}', "");
Expect(1, 8191, '\P{Is_Blk=	-GENERAL	Punctuation}', "");
Expect(0, 8191, '\P{^Is_Blk=	-GENERAL	Punctuation}', "");
Error('\p{Block=	:=Rejang}');
Error('\P{Block=	:=Rejang}');
Expect(1, 43359, '\p{Block=rejang}', "");
Expect(0, 43359, '\p{^Block=rejang}', "");
Expect(0, 43359, '\P{Block=rejang}', "");
Expect(1, 43359, '\P{^Block=rejang}', "");
Expect(0, 43311, '\p{Block=rejang}', "");
Expect(1, 43311, '\p{^Block=rejang}', "");
Expect(1, 43311, '\P{Block=rejang}', "");
Expect(0, 43311, '\P{^Block=rejang}', "");
Expect(1, 43359, '\p{Block= -REJANG}', "");
Expect(0, 43359, '\p{^Block= -REJANG}', "");
Expect(0, 43359, '\P{Block= -REJANG}', "");
Expect(1, 43359, '\P{^Block= -REJANG}', "");
Expect(0, 43311, '\p{Block= -REJANG}', "");
Expect(1, 43311, '\p{^Block= -REJANG}', "");
Expect(1, 43311, '\P{Block= -REJANG}', "");
Expect(0, 43311, '\P{^Block= -REJANG}', "");
Error('\p{Blk=:=	 Rejang}');
Error('\P{Blk=:=	 Rejang}');
Expect(1, 43359, '\p{Blk=rejang}', "");
Expect(0, 43359, '\p{^Blk=rejang}', "");
Expect(0, 43359, '\P{Blk=rejang}', "");
Expect(1, 43359, '\P{^Blk=rejang}', "");
Expect(0, 43311, '\p{Blk=rejang}', "");
Expect(1, 43311, '\p{^Blk=rejang}', "");
Expect(1, 43311, '\P{Blk=rejang}', "");
Expect(0, 43311, '\P{^Blk=rejang}', "");
Expect(1, 43359, '\p{Blk= -rejang}', "");
Expect(0, 43359, '\p{^Blk= -rejang}', "");
Expect(0, 43359, '\P{Blk= -rejang}', "");
Expect(1, 43359, '\P{^Blk= -rejang}', "");
Expect(0, 43311, '\p{Blk= -rejang}', "");
Expect(1, 43311, '\p{^Blk= -rejang}', "");
Expect(1, 43311, '\P{Blk= -rejang}', "");
Expect(0, 43311, '\P{^Blk= -rejang}', "");
Error('\p{Is_Block=:=_	rejang}');
Error('\P{Is_Block=:=_	rejang}');
Expect(1, 43359, '\p{Is_Block=rejang}', "");
Expect(0, 43359, '\p{^Is_Block=rejang}', "");
Expect(0, 43359, '\P{Is_Block=rejang}', "");
Expect(1, 43359, '\P{^Is_Block=rejang}', "");
Expect(0, 43311, '\p{Is_Block=rejang}', "");
Expect(1, 43311, '\p{^Is_Block=rejang}', "");
Expect(1, 43311, '\P{Is_Block=rejang}', "");
Expect(0, 43311, '\P{^Is_Block=rejang}', "");
Expect(1, 43359, '\p{Is_Block= rejang}', "");
Expect(0, 43359, '\p{^Is_Block= rejang}', "");
Expect(0, 43359, '\P{Is_Block= rejang}', "");
Expect(1, 43359, '\P{^Is_Block= rejang}', "");
Expect(0, 43311, '\p{Is_Block= rejang}', "");
Expect(1, 43311, '\p{^Is_Block= rejang}', "");
Expect(1, 43311, '\P{Is_Block= rejang}', "");
Expect(0, 43311, '\P{^Is_Block= rejang}', "");
Error('\p{Is_Blk=:= _Rejang}');
Error('\P{Is_Blk=:= _Rejang}');
Expect(1, 43359, '\p{Is_Blk=rejang}', "");
Expect(0, 43359, '\p{^Is_Blk=rejang}', "");
Expect(0, 43359, '\P{Is_Blk=rejang}', "");
Expect(1, 43359, '\P{^Is_Blk=rejang}', "");
Expect(0, 43311, '\p{Is_Blk=rejang}', "");
Expect(1, 43311, '\p{^Is_Blk=rejang}', "");
Expect(1, 43311, '\P{Is_Blk=rejang}', "");
Expect(0, 43311, '\P{^Is_Blk=rejang}', "");
Expect(1, 43359, '\p{Is_Blk=-rejang}', "");
Expect(0, 43359, '\p{^Is_Blk=-rejang}', "");
Expect(0, 43359, '\P{Is_Blk=-rejang}', "");
Expect(1, 43359, '\P{^Is_Blk=-rejang}', "");
Expect(0, 43311, '\p{Is_Blk=-rejang}', "");
Expect(1, 43311, '\p{^Is_Blk=-rejang}', "");
Expect(1, 43311, '\P{Is_Blk=-rejang}', "");
Expect(0, 43311, '\P{^Is_Blk=-rejang}', "");
Error('\p{Block=:=- OLD turkic}');
Error('\P{Block=:=- OLD turkic}');
Expect(1, 68687, '\p{Block=oldturkic}', "");
Expect(0, 68687, '\p{^Block=oldturkic}', "");
Expect(0, 68687, '\P{Block=oldturkic}', "");
Expect(1, 68687, '\P{^Block=oldturkic}', "");
Expect(0, 68607, '\p{Block=oldturkic}', "");
Expect(1, 68607, '\p{^Block=oldturkic}', "");
Expect(1, 68607, '\P{Block=oldturkic}', "");
Expect(0, 68607, '\P{^Block=oldturkic}', "");
Expect(1, 68687, '\p{Block=_ OLD_Turkic}', "");
Expect(0, 68687, '\p{^Block=_ OLD_Turkic}', "");
Expect(0, 68687, '\P{Block=_ OLD_Turkic}', "");
Expect(1, 68687, '\P{^Block=_ OLD_Turkic}', "");
Expect(0, 68607, '\p{Block=_ OLD_Turkic}', "");
Expect(1, 68607, '\p{^Block=_ OLD_Turkic}', "");
Expect(1, 68607, '\P{Block=_ OLD_Turkic}', "");
Expect(0, 68607, '\P{^Block=_ OLD_Turkic}', "");
Error('\p{Blk=--Old:=Turkic}');
Error('\P{Blk=--Old:=Turkic}');
Expect(1, 68687, '\p{Blk: oldturkic}', "");
Expect(0, 68687, '\p{^Blk: oldturkic}', "");
Expect(0, 68687, '\P{Blk: oldturkic}', "");
Expect(1, 68687, '\P{^Blk: oldturkic}', "");
Expect(0, 68607, '\p{Blk: oldturkic}', "");
Expect(1, 68607, '\p{^Blk: oldturkic}', "");
Expect(1, 68607, '\P{Blk: oldturkic}', "");
Expect(0, 68607, '\P{^Blk: oldturkic}', "");
Expect(1, 68687, '\p{Blk=	Old-Turkic}', "");
Expect(0, 68687, '\p{^Blk=	Old-Turkic}', "");
Expect(0, 68687, '\P{Blk=	Old-Turkic}', "");
Expect(1, 68687, '\P{^Blk=	Old-Turkic}', "");
Expect(0, 68607, '\p{Blk=	Old-Turkic}', "");
Expect(1, 68607, '\p{^Blk=	Old-Turkic}', "");
Expect(1, 68607, '\P{Blk=	Old-Turkic}', "");
Expect(0, 68607, '\P{^Blk=	Old-Turkic}', "");
Error('\p{Is_Block=/a/-	oldTurkic}');
Error('\P{Is_Block=/a/-	oldTurkic}');
Expect(1, 68687, '\p{Is_Block:oldturkic}', "");
Expect(0, 68687, '\p{^Is_Block:oldturkic}', "");
Expect(0, 68687, '\P{Is_Block:oldturkic}', "");
Expect(1, 68687, '\P{^Is_Block:oldturkic}', "");
Expect(0, 68607, '\p{Is_Block:oldturkic}', "");
Expect(1, 68607, '\p{^Is_Block:oldturkic}', "");
Expect(1, 68607, '\P{Is_Block:oldturkic}', "");
Expect(0, 68607, '\P{^Is_Block:oldturkic}', "");
Expect(1, 68687, '\p{Is_Block=-_Old-turkic}', "");
Expect(0, 68687, '\p{^Is_Block=-_Old-turkic}', "");
Expect(0, 68687, '\P{Is_Block=-_Old-turkic}', "");
Expect(1, 68687, '\P{^Is_Block=-_Old-turkic}', "");
Expect(0, 68607, '\p{Is_Block=-_Old-turkic}', "");
Expect(1, 68607, '\p{^Is_Block=-_Old-turkic}', "");
Expect(1, 68607, '\P{Is_Block=-_Old-turkic}', "");
Expect(0, 68607, '\P{^Is_Block=-_Old-turkic}', "");
Error('\p{Is_Blk=	_Old/a/turkic}');
Error('\P{Is_Blk=	_Old/a/turkic}');
Expect(1, 68687, '\p{Is_Blk=oldturkic}', "");
Expect(0, 68687, '\p{^Is_Blk=oldturkic}', "");
Expect(0, 68687, '\P{Is_Blk=oldturkic}', "");
Expect(1, 68687, '\P{^Is_Blk=oldturkic}', "");
Expect(0, 68607, '\p{Is_Blk=oldturkic}', "");
Expect(1, 68607, '\p{^Is_Blk=oldturkic}', "");
Expect(1, 68607, '\P{Is_Blk=oldturkic}', "");
Expect(0, 68607, '\P{^Is_Blk=oldturkic}', "");
Expect(1, 68687, '\p{Is_Blk=  Old_Turkic}', "");
Expect(0, 68687, '\p{^Is_Blk=  Old_Turkic}', "");
Expect(0, 68687, '\P{Is_Blk=  Old_Turkic}', "");
Expect(1, 68687, '\P{^Is_Blk=  Old_Turkic}', "");
Expect(0, 68607, '\p{Is_Blk=  Old_Turkic}', "");
Expect(1, 68607, '\p{^Is_Blk=  Old_Turkic}', "");
Expect(1, 68607, '\P{Is_Blk=  Old_Turkic}', "");
Expect(0, 68607, '\P{^Is_Blk=  Old_Turkic}', "");
Error('\p{Block= _OlChiki/a/}');
Error('\P{Block= _OlChiki/a/}');
Expect(1, 7295, '\p{Block=olchiki}', "");
Expect(0, 7295, '\p{^Block=olchiki}', "");
Expect(0, 7295, '\P{Block=olchiki}', "");
Expect(1, 7295, '\P{^Block=olchiki}', "");
Expect(0, 7247, '\p{Block=olchiki}', "");
Expect(1, 7247, '\p{^Block=olchiki}', "");
Expect(1, 7247, '\P{Block=olchiki}', "");
Expect(0, 7247, '\P{^Block=olchiki}', "");
Expect(1, 7295, '\p{Block=_-olCHIKI}', "");
Expect(0, 7295, '\p{^Block=_-olCHIKI}', "");
Expect(0, 7295, '\P{Block=_-olCHIKI}', "");
Expect(1, 7295, '\P{^Block=_-olCHIKI}', "");
Expect(0, 7247, '\p{Block=_-olCHIKI}', "");
Expect(1, 7247, '\p{^Block=_-olCHIKI}', "");
Expect(1, 7247, '\P{Block=_-olCHIKI}', "");
Expect(0, 7247, '\P{^Block=_-olCHIKI}', "");
Error('\p{Blk=	OL/a/chiki}');
Error('\P{Blk=	OL/a/chiki}');
Expect(1, 7295, '\p{Blk=olchiki}', "");
Expect(0, 7295, '\p{^Blk=olchiki}', "");
Expect(0, 7295, '\P{Blk=olchiki}', "");
Expect(1, 7295, '\P{^Blk=olchiki}', "");
Expect(0, 7247, '\p{Blk=olchiki}', "");
Expect(1, 7247, '\p{^Blk=olchiki}', "");
Expect(1, 7247, '\P{Blk=olchiki}', "");
Expect(0, 7247, '\P{^Blk=olchiki}', "");
Expect(1, 7295, '\p{Blk=-Ol-chiki}', "");
Expect(0, 7295, '\p{^Blk=-Ol-chiki}', "");
Expect(0, 7295, '\P{Blk=-Ol-chiki}', "");
Expect(1, 7295, '\P{^Blk=-Ol-chiki}', "");
Expect(0, 7247, '\p{Blk=-Ol-chiki}', "");
Expect(1, 7247, '\p{^Blk=-Ol-chiki}', "");
Expect(1, 7247, '\P{Blk=-Ol-chiki}', "");
Expect(0, 7247, '\P{^Blk=-Ol-chiki}', "");
Error('\p{Is_Block=_:=Ol-chiki}');
Error('\P{Is_Block=_:=Ol-chiki}');
Expect(1, 7295, '\p{Is_Block=olchiki}', "");
Expect(0, 7295, '\p{^Is_Block=olchiki}', "");
Expect(0, 7295, '\P{Is_Block=olchiki}', "");
Expect(1, 7295, '\P{^Is_Block=olchiki}', "");
Expect(0, 7247, '\p{Is_Block=olchiki}', "");
Expect(1, 7247, '\p{^Is_Block=olchiki}', "");
Expect(1, 7247, '\P{Is_Block=olchiki}', "");
Expect(0, 7247, '\P{^Is_Block=olchiki}', "");
Expect(1, 7295, '\p{Is_Block=_ Olchiki}', "");
Expect(0, 7295, '\p{^Is_Block=_ Olchiki}', "");
Expect(0, 7295, '\P{Is_Block=_ Olchiki}', "");
Expect(1, 7295, '\P{^Is_Block=_ Olchiki}', "");
Expect(0, 7247, '\p{Is_Block=_ Olchiki}', "");
Expect(1, 7247, '\p{^Is_Block=_ Olchiki}', "");
Expect(1, 7247, '\P{Is_Block=_ Olchiki}', "");
Expect(0, 7247, '\P{^Is_Block=_ Olchiki}', "");
Error('\p{Is_Blk=-_Ol	chiki/a/}');
Error('\P{Is_Blk=-_Ol	chiki/a/}');
Expect(1, 7295, '\p{Is_Blk=olchiki}', "");
Expect(0, 7295, '\p{^Is_Blk=olchiki}', "");
Expect(0, 7295, '\P{Is_Blk=olchiki}', "");
Expect(1, 7295, '\P{^Is_Blk=olchiki}', "");
Expect(0, 7247, '\p{Is_Blk=olchiki}', "");
Expect(1, 7247, '\p{^Is_Blk=olchiki}', "");
Expect(1, 7247, '\P{Is_Blk=olchiki}', "");
Expect(0, 7247, '\P{^Is_Blk=olchiki}', "");
Expect(1, 7295, '\p{Is_Blk=-Ol-Chiki}', "");
Expect(0, 7295, '\p{^Is_Blk=-Ol-Chiki}', "");
Expect(0, 7295, '\P{Is_Blk=-Ol-Chiki}', "");
Expect(1, 7295, '\P{^Is_Blk=-Ol-Chiki}', "");
Expect(0, 7247, '\p{Is_Blk=-Ol-Chiki}', "");
Expect(1, 7247, '\p{^Is_Blk=-Ol-Chiki}', "");
Expect(1, 7247, '\P{Is_Blk=-Ol-Chiki}', "");
Expect(0, 7247, '\P{^Is_Blk=-Ol-Chiki}', "");
Error('\p{Block:	:=--ethiopic-extendedA}');
Error('\P{Block:	:=--ethiopic-extendedA}');
Expect(1, 43823, '\p{Block:   ethiopicextendeda}', "");
Expect(0, 43823, '\p{^Block:   ethiopicextendeda}', "");
Expect(0, 43823, '\P{Block:   ethiopicextendeda}', "");
Expect(1, 43823, '\P{^Block:   ethiopicextendeda}', "");
Expect(0, 43775, '\p{Block:   ethiopicextendeda}', "");
Expect(1, 43775, '\p{^Block:   ethiopicextendeda}', "");
Expect(1, 43775, '\P{Block:   ethiopicextendeda}', "");
Expect(0, 43775, '\P{^Block:   ethiopicextendeda}', "");
Expect(1, 43823, '\p{Block=_	Ethiopic_Extended_a}', "");
Expect(0, 43823, '\p{^Block=_	Ethiopic_Extended_a}', "");
Expect(0, 43823, '\P{Block=_	Ethiopic_Extended_a}', "");
Expect(1, 43823, '\P{^Block=_	Ethiopic_Extended_a}', "");
Expect(0, 43775, '\p{Block=_	Ethiopic_Extended_a}', "");
Expect(1, 43775, '\p{^Block=_	Ethiopic_Extended_a}', "");
Expect(1, 43775, '\P{Block=_	Ethiopic_Extended_a}', "");
Expect(0, 43775, '\P{^Block=_	Ethiopic_Extended_a}', "");
Error('\p{Blk=_ETHIOPIC:=extended	A}');
Error('\P{Blk=_ETHIOPIC:=extended	A}');
Expect(1, 43823, '\p{Blk=ethiopicextendeda}', "");
Expect(0, 43823, '\p{^Blk=ethiopicextendeda}', "");
Expect(0, 43823, '\P{Blk=ethiopicextendeda}', "");
Expect(1, 43823, '\P{^Blk=ethiopicextendeda}', "");
Expect(0, 43775, '\p{Blk=ethiopicextendeda}', "");
Expect(1, 43775, '\p{^Blk=ethiopicextendeda}', "");
Expect(1, 43775, '\P{Blk=ethiopicextendeda}', "");
Expect(0, 43775, '\P{^Blk=ethiopicextendeda}', "");
Expect(1, 43823, '\p{Blk=	 Ethiopic-extended-A}', "");
Expect(0, 43823, '\p{^Blk=	 Ethiopic-extended-A}', "");
Expect(0, 43823, '\P{Blk=	 Ethiopic-extended-A}', "");
Expect(1, 43823, '\P{^Blk=	 Ethiopic-extended-A}', "");
Expect(0, 43775, '\p{Blk=	 Ethiopic-extended-A}', "");
Expect(1, 43775, '\p{^Blk=	 Ethiopic-extended-A}', "");
Expect(1, 43775, '\P{Blk=	 Ethiopic-extended-A}', "");
Expect(0, 43775, '\P{^Blk=	 Ethiopic-extended-A}', "");
Error('\p{Is_Block=/a/Ethiopic	EXTENDEDA}');
Error('\P{Is_Block=/a/Ethiopic	EXTENDEDA}');
Expect(1, 43823, '\p{Is_Block=ethiopicextendeda}', "");
Expect(0, 43823, '\p{^Is_Block=ethiopicextendeda}', "");
Expect(0, 43823, '\P{Is_Block=ethiopicextendeda}', "");
Expect(1, 43823, '\P{^Is_Block=ethiopicextendeda}', "");
Expect(0, 43775, '\p{Is_Block=ethiopicextendeda}', "");
Expect(1, 43775, '\p{^Is_Block=ethiopicextendeda}', "");
Expect(1, 43775, '\P{Is_Block=ethiopicextendeda}', "");
Expect(0, 43775, '\P{^Is_Block=ethiopicextendeda}', "");
Expect(1, 43823, '\p{Is_Block=_ethiopic_Extended_A}', "");
Expect(0, 43823, '\p{^Is_Block=_ethiopic_Extended_A}', "");
Expect(0, 43823, '\P{Is_Block=_ethiopic_Extended_A}', "");
Expect(1, 43823, '\P{^Is_Block=_ethiopic_Extended_A}', "");
Expect(0, 43775, '\p{Is_Block=_ethiopic_Extended_A}', "");
Expect(1, 43775, '\p{^Is_Block=_ethiopic_Extended_A}', "");
Expect(1, 43775, '\P{Is_Block=_ethiopic_Extended_A}', "");
Expect(0, 43775, '\P{^Is_Block=_ethiopic_Extended_A}', "");
Error('\p{Is_Blk:    Ethiopic_EXTENDED:=a}');
Error('\P{Is_Blk:    Ethiopic_EXTENDED:=a}');
Expect(1, 43823, '\p{Is_Blk=ethiopicextendeda}', "");
Expect(0, 43823, '\p{^Is_Blk=ethiopicextendeda}', "");
Expect(0, 43823, '\P{Is_Blk=ethiopicextendeda}', "");
Expect(1, 43823, '\P{^Is_Blk=ethiopicextendeda}', "");
Expect(0, 43775, '\p{Is_Blk=ethiopicextendeda}', "");
Expect(1, 43775, '\p{^Is_Blk=ethiopicextendeda}', "");
Expect(1, 43775, '\P{Is_Blk=ethiopicextendeda}', "");
Expect(0, 43775, '\P{^Is_Blk=ethiopicextendeda}', "");
Expect(1, 43823, '\p{Is_Blk=--ETHIOPIC_Extended A}', "");
Expect(0, 43823, '\p{^Is_Blk=--ETHIOPIC_Extended A}', "");
Expect(0, 43823, '\P{Is_Blk=--ETHIOPIC_Extended A}', "");
Expect(1, 43823, '\P{^Is_Blk=--ETHIOPIC_Extended A}', "");
Expect(0, 43775, '\p{Is_Blk=--ETHIOPIC_Extended A}', "");
Expect(1, 43775, '\p{^Is_Blk=--ETHIOPIC_Extended A}', "");
Expect(1, 43775, '\P{Is_Blk=--ETHIOPIC_Extended A}', "");
Expect(0, 43775, '\P{^Is_Blk=--ETHIOPIC_Extended A}', "");
Error('\p{Block=-_Kanbun:=}');
Error('\P{Block=-_Kanbun:=}');
Expect(1, 12703, '\p{Block=kanbun}', "");
Expect(0, 12703, '\p{^Block=kanbun}', "");
Expect(0, 12703, '\P{Block=kanbun}', "");
Expect(1, 12703, '\P{^Block=kanbun}', "");
Expect(0, 12687, '\p{Block=kanbun}', "");
Expect(1, 12687, '\p{^Block=kanbun}', "");
Expect(1, 12687, '\P{Block=kanbun}', "");
Expect(0, 12687, '\P{^Block=kanbun}', "");
Expect(1, 12703, '\p{Block=	_Kanbun}', "");
Expect(0, 12703, '\p{^Block=	_Kanbun}', "");
Expect(0, 12703, '\P{Block=	_Kanbun}', "");
Expect(1, 12703, '\P{^Block=	_Kanbun}', "");
Expect(0, 12687, '\p{Block=	_Kanbun}', "");
Expect(1, 12687, '\p{^Block=	_Kanbun}', "");
Expect(1, 12687, '\P{Block=	_Kanbun}', "");
Expect(0, 12687, '\P{^Block=	_Kanbun}', "");
Error('\p{Blk=	/a/Kanbun}');
Error('\P{Blk=	/a/Kanbun}');
Expect(1, 12703, '\p{Blk=kanbun}', "");
Expect(0, 12703, '\p{^Blk=kanbun}', "");
Expect(0, 12703, '\P{Blk=kanbun}', "");
Expect(1, 12703, '\P{^Blk=kanbun}', "");
Expect(0, 12687, '\p{Blk=kanbun}', "");
Expect(1, 12687, '\p{^Blk=kanbun}', "");
Expect(1, 12687, '\P{Blk=kanbun}', "");
Expect(0, 12687, '\P{^Blk=kanbun}', "");
Expect(1, 12703, '\p{Blk= kanbun}', "");
Expect(0, 12703, '\p{^Blk= kanbun}', "");
Expect(0, 12703, '\P{Blk= kanbun}', "");
Expect(1, 12703, '\P{^Blk= kanbun}', "");
Expect(0, 12687, '\p{Blk= kanbun}', "");
Expect(1, 12687, '\p{^Blk= kanbun}', "");
Expect(1, 12687, '\P{Blk= kanbun}', "");
Expect(0, 12687, '\P{^Blk= kanbun}', "");
Error('\p{Is_Block= Kanbun/a/}');
Error('\P{Is_Block= Kanbun/a/}');
Expect(1, 12703, '\p{Is_Block=kanbun}', "");
Expect(0, 12703, '\p{^Is_Block=kanbun}', "");
Expect(0, 12703, '\P{Is_Block=kanbun}', "");
Expect(1, 12703, '\P{^Is_Block=kanbun}', "");
Expect(0, 12687, '\p{Is_Block=kanbun}', "");
Expect(1, 12687, '\p{^Is_Block=kanbun}', "");
Expect(1, 12687, '\P{Is_Block=kanbun}', "");
Expect(0, 12687, '\P{^Is_Block=kanbun}', "");
Expect(1, 12703, '\p{Is_Block= 	KANBUN}', "");
Expect(0, 12703, '\p{^Is_Block= 	KANBUN}', "");
Expect(0, 12703, '\P{Is_Block= 	KANBUN}', "");
Expect(1, 12703, '\P{^Is_Block= 	KANBUN}', "");
Expect(0, 12687, '\p{Is_Block= 	KANBUN}', "");
Expect(1, 12687, '\p{^Is_Block= 	KANBUN}', "");
Expect(1, 12687, '\P{Is_Block= 	KANBUN}', "");
Expect(0, 12687, '\P{^Is_Block= 	KANBUN}', "");
Error('\p{Is_Blk=:= _Kanbun}');
Error('\P{Is_Blk=:= _Kanbun}');
Expect(1, 12703, '\p{Is_Blk=kanbun}', "");
Expect(0, 12703, '\p{^Is_Blk=kanbun}', "");
Expect(0, 12703, '\P{Is_Blk=kanbun}', "");
Expect(1, 12703, '\P{^Is_Blk=kanbun}', "");
Expect(0, 12687, '\p{Is_Blk=kanbun}', "");
Expect(1, 12687, '\p{^Is_Blk=kanbun}', "");
Expect(1, 12687, '\P{Is_Blk=kanbun}', "");
Expect(0, 12687, '\P{^Is_Blk=kanbun}', "");
Expect(1, 12703, '\p{Is_Blk:	__KANBUN}', "");
Expect(0, 12703, '\p{^Is_Blk:	__KANBUN}', "");
Expect(0, 12703, '\P{Is_Blk:	__KANBUN}', "");
Expect(1, 12703, '\P{^Is_Blk:	__KANBUN}', "");
Expect(0, 12687, '\p{Is_Blk:	__KANBUN}', "");
Expect(1, 12687, '\p{^Is_Blk:	__KANBUN}', "");
Expect(1, 12687, '\P{Is_Blk:	__KANBUN}', "");
Expect(0, 12687, '\P{^Is_Blk:	__KANBUN}', "");
Error('\p{Block=/a/	 CYRILLIC-ExtendedA}');
Error('\P{Block=/a/	 CYRILLIC-ExtendedA}');
Expect(1, 11775, '\p{Block=cyrillicextendeda}', "");
Expect(0, 11775, '\p{^Block=cyrillicextendeda}', "");
Expect(0, 11775, '\P{Block=cyrillicextendeda}', "");
Expect(1, 11775, '\P{^Block=cyrillicextendeda}', "");
Expect(0, 11743, '\p{Block=cyrillicextendeda}', "");
Expect(1, 11743, '\p{^Block=cyrillicextendeda}', "");
Expect(1, 11743, '\P{Block=cyrillicextendeda}', "");
Expect(0, 11743, '\P{^Block=cyrillicextendeda}', "");
Expect(1, 11775, '\p{Block=_	CyrillicEXTENDED A}', "");
Expect(0, 11775, '\p{^Block=_	CyrillicEXTENDED A}', "");
Expect(0, 11775, '\P{Block=_	CyrillicEXTENDED A}', "");
Expect(1, 11775, '\P{^Block=_	CyrillicEXTENDED A}', "");
Expect(0, 11743, '\p{Block=_	CyrillicEXTENDED A}', "");
Expect(1, 11743, '\p{^Block=_	CyrillicEXTENDED A}', "");
Expect(1, 11743, '\P{Block=_	CyrillicEXTENDED A}', "");
Expect(0, 11743, '\P{^Block=_	CyrillicEXTENDED A}', "");
Error('\p{Blk=--CYRILLIC:=extended-A}');
Error('\P{Blk=--CYRILLIC:=extended-A}');
Expect(1, 11775, '\p{Blk=cyrillicextendeda}', "");
Expect(0, 11775, '\p{^Blk=cyrillicextendeda}', "");
Expect(0, 11775, '\P{Blk=cyrillicextendeda}', "");
Expect(1, 11775, '\P{^Blk=cyrillicextendeda}', "");
Expect(0, 11743, '\p{Blk=cyrillicextendeda}', "");
Expect(1, 11743, '\p{^Blk=cyrillicextendeda}', "");
Expect(1, 11743, '\P{Blk=cyrillicextendeda}', "");
Expect(0, 11743, '\P{^Blk=cyrillicextendeda}', "");
Expect(1, 11775, '\p{Blk=	-CYRILLIC	extendedA}', "");
Expect(0, 11775, '\p{^Blk=	-CYRILLIC	extendedA}', "");
Expect(0, 11775, '\P{Blk=	-CYRILLIC	extendedA}', "");
Expect(1, 11775, '\P{^Blk=	-CYRILLIC	extendedA}', "");
Expect(0, 11743, '\p{Blk=	-CYRILLIC	extendedA}', "");
Expect(1, 11743, '\p{^Blk=	-CYRILLIC	extendedA}', "");
Expect(1, 11743, '\P{Blk=	-CYRILLIC	extendedA}', "");
Expect(0, 11743, '\P{^Blk=	-CYRILLIC	extendedA}', "");
Error('\p{Is_Block=	 CYRILLIC_EXTENDED:=a}');
Error('\P{Is_Block=	 CYRILLIC_EXTENDED:=a}');
Expect(1, 11775, '\p{Is_Block=cyrillicextendeda}', "");
Expect(0, 11775, '\p{^Is_Block=cyrillicextendeda}', "");
Expect(0, 11775, '\P{Is_Block=cyrillicextendeda}', "");
Expect(1, 11775, '\P{^Is_Block=cyrillicextendeda}', "");
Expect(0, 11743, '\p{Is_Block=cyrillicextendeda}', "");
Expect(1, 11743, '\p{^Is_Block=cyrillicextendeda}', "");
Expect(1, 11743, '\P{Is_Block=cyrillicextendeda}', "");
Expect(0, 11743, '\P{^Is_Block=cyrillicextendeda}', "");
Expect(1, 11775, '\p{Is_Block: -CYRILLICExtended	A}', "");
Expect(0, 11775, '\p{^Is_Block: -CYRILLICExtended	A}', "");
Expect(0, 11775, '\P{Is_Block: -CYRILLICExtended	A}', "");
Expect(1, 11775, '\P{^Is_Block: -CYRILLICExtended	A}', "");
Expect(0, 11743, '\p{Is_Block: -CYRILLICExtended	A}', "");
Expect(1, 11743, '\p{^Is_Block: -CYRILLICExtended	A}', "");
Expect(1, 11743, '\P{Is_Block: -CYRILLICExtended	A}', "");
Expect(0, 11743, '\P{^Is_Block: -CYRILLICExtended	A}', "");
Error('\p{Is_Blk= -CYRILLIC EXTENDED:=A}');
Error('\P{Is_Blk= -CYRILLIC EXTENDED:=A}');
Expect(1, 11775, '\p{Is_Blk: cyrillicextendeda}', "");
Expect(0, 11775, '\p{^Is_Blk: cyrillicextendeda}', "");
Expect(0, 11775, '\P{Is_Blk: cyrillicextendeda}', "");
Expect(1, 11775, '\P{^Is_Blk: cyrillicextendeda}', "");
Expect(0, 11743, '\p{Is_Blk: cyrillicextendeda}', "");
Expect(1, 11743, '\p{^Is_Blk: cyrillicextendeda}', "");
Expect(1, 11743, '\P{Is_Blk: cyrillicextendeda}', "");
Expect(0, 11743, '\P{^Is_Blk: cyrillicextendeda}', "");
Expect(1, 11775, '\p{Is_Blk=-_cyrillicEXTENDED A}', "");
Expect(0, 11775, '\p{^Is_Blk=-_cyrillicEXTENDED A}', "");
Expect(0, 11775, '\P{Is_Blk=-_cyrillicEXTENDED A}', "");
Expect(1, 11775, '\P{^Is_Blk=-_cyrillicEXTENDED A}', "");
Expect(0, 11743, '\p{Is_Blk=-_cyrillicEXTENDED A}', "");
Expect(1, 11743, '\p{^Is_Blk=-_cyrillicEXTENDED A}', "");
Expect(1, 11743, '\P{Is_Blk=-_cyrillicEXTENDED A}', "");
Expect(0, 11743, '\P{^Is_Blk=-_cyrillicEXTENDED A}', "");
Error('\p{Block=-:=HEBREW}');
Error('\P{Block=-:=HEBREW}');
Expect(1, 1535, '\p{Block=hebrew}', "");
Expect(0, 1535, '\p{^Block=hebrew}', "");
Expect(0, 1535, '\P{Block=hebrew}', "");
Expect(1, 1535, '\P{^Block=hebrew}', "");
Expect(0, 1423, '\p{Block=hebrew}', "");
Expect(1, 1423, '\p{^Block=hebrew}', "");
Expect(1, 1423, '\P{Block=hebrew}', "");
Expect(0, 1423, '\P{^Block=hebrew}', "");
Expect(1, 1535, '\p{Block=  HEBREW}', "");
Expect(0, 1535, '\p{^Block=  HEBREW}', "");
Expect(0, 1535, '\P{Block=  HEBREW}', "");
Expect(1, 1535, '\P{^Block=  HEBREW}', "");
Expect(0, 1423, '\p{Block=  HEBREW}', "");
Expect(1, 1423, '\p{^Block=  HEBREW}', "");
Expect(1, 1423, '\P{Block=  HEBREW}', "");
Expect(0, 1423, '\P{^Block=  HEBREW}', "");
Error('\p{Blk=/a/ -HEBREW}');
Error('\P{Blk=/a/ -HEBREW}');
Expect(1, 1535, '\p{Blk=hebrew}', "");
Expect(0, 1535, '\p{^Blk=hebrew}', "");
Expect(0, 1535, '\P{Blk=hebrew}', "");
Expect(1, 1535, '\P{^Blk=hebrew}', "");
Expect(0, 1423, '\p{Blk=hebrew}', "");
Expect(1, 1423, '\p{^Blk=hebrew}', "");
Expect(1, 1423, '\P{Blk=hebrew}', "");
Expect(0, 1423, '\P{^Blk=hebrew}', "");
Expect(1, 1535, '\p{Blk=-	Hebrew}', "");
Expect(0, 1535, '\p{^Blk=-	Hebrew}', "");
Expect(0, 1535, '\P{Blk=-	Hebrew}', "");
Expect(1, 1535, '\P{^Blk=-	Hebrew}', "");
Expect(0, 1423, '\p{Blk=-	Hebrew}', "");
Expect(1, 1423, '\p{^Blk=-	Hebrew}', "");
Expect(1, 1423, '\P{Blk=-	Hebrew}', "");
Expect(0, 1423, '\P{^Blk=-	Hebrew}', "");
Error('\p{Is_Block=_Hebrew:=}');
Error('\P{Is_Block=_Hebrew:=}');
Expect(1, 1535, '\p{Is_Block=hebrew}', "");
Expect(0, 1535, '\p{^Is_Block=hebrew}', "");
Expect(0, 1535, '\P{Is_Block=hebrew}', "");
Expect(1, 1535, '\P{^Is_Block=hebrew}', "");
Expect(0, 1423, '\p{Is_Block=hebrew}', "");
Expect(1, 1423, '\p{^Is_Block=hebrew}', "");
Expect(1, 1423, '\P{Is_Block=hebrew}', "");
Expect(0, 1423, '\P{^Is_Block=hebrew}', "");
Expect(1, 1535, '\p{Is_Block:	 -hebrew}', "");
Expect(0, 1535, '\p{^Is_Block:	 -hebrew}', "");
Expect(0, 1535, '\P{Is_Block:	 -hebrew}', "");
Expect(1, 1535, '\P{^Is_Block:	 -hebrew}', "");
Expect(0, 1423, '\p{Is_Block:	 -hebrew}', "");
Expect(1, 1423, '\p{^Is_Block:	 -hebrew}', "");
Expect(1, 1423, '\P{Is_Block:	 -hebrew}', "");
Expect(0, 1423, '\P{^Is_Block:	 -hebrew}', "");
Error('\p{Is_Blk=-Hebrew/a/}');
Error('\P{Is_Blk=-Hebrew/a/}');
Expect(1, 1535, '\p{Is_Blk=hebrew}', "");
Expect(0, 1535, '\p{^Is_Blk=hebrew}', "");
Expect(0, 1535, '\P{Is_Blk=hebrew}', "");
Expect(1, 1535, '\P{^Is_Blk=hebrew}', "");
Expect(0, 1423, '\p{Is_Blk=hebrew}', "");
Expect(1, 1423, '\p{^Is_Blk=hebrew}', "");
Expect(1, 1423, '\P{Is_Blk=hebrew}', "");
Expect(0, 1423, '\P{^Is_Blk=hebrew}', "");
Expect(1, 1535, '\p{Is_Blk=_ hebrew}', "");
Expect(0, 1535, '\p{^Is_Blk=_ hebrew}', "");
Expect(0, 1535, '\P{Is_Blk=_ hebrew}', "");
Expect(1, 1535, '\P{^Is_Blk=_ hebrew}', "");
Expect(0, 1423, '\p{Is_Blk=_ hebrew}', "");
Expect(1, 1423, '\p{^Is_Blk=_ hebrew}', "");
Expect(1, 1423, '\P{Is_Blk=_ hebrew}', "");
Expect(0, 1423, '\P{^Is_Blk=_ hebrew}', "");
Error('\p{Block=/a/ 	Cyrillic}');
Error('\P{Block=/a/ 	Cyrillic}');
Expect(1, 1279, '\p{Block=cyrillic}', "");
Expect(0, 1279, '\p{^Block=cyrillic}', "");
Expect(0, 1279, '\P{Block=cyrillic}', "");
Expect(1, 1279, '\P{^Block=cyrillic}', "");
Expect(0, 1023, '\p{Block=cyrillic}', "");
Expect(1, 1023, '\p{^Block=cyrillic}', "");
Expect(1, 1023, '\P{Block=cyrillic}', "");
Expect(0, 1023, '\P{^Block=cyrillic}', "");
Expect(1, 1279, '\p{Block=- CYRILLIC}', "");
Expect(0, 1279, '\p{^Block=- CYRILLIC}', "");
Expect(0, 1279, '\P{Block=- CYRILLIC}', "");
Expect(1, 1279, '\P{^Block=- CYRILLIC}', "");
Expect(0, 1023, '\p{Block=- CYRILLIC}', "");
Expect(1, 1023, '\p{^Block=- CYRILLIC}', "");
Expect(1, 1023, '\P{Block=- CYRILLIC}', "");
Expect(0, 1023, '\P{^Block=- CYRILLIC}', "");
Error('\p{Blk=/a/- Cyrillic}');
Error('\P{Blk=/a/- Cyrillic}');
Expect(1, 1279, '\p{Blk=cyrillic}', "");
Expect(0, 1279, '\p{^Blk=cyrillic}', "");
Expect(0, 1279, '\P{Blk=cyrillic}', "");
Expect(1, 1279, '\P{^Blk=cyrillic}', "");
Expect(0, 1023, '\p{Blk=cyrillic}', "");
Expect(1, 1023, '\p{^Blk=cyrillic}', "");
Expect(1, 1023, '\P{Blk=cyrillic}', "");
Expect(0, 1023, '\P{^Blk=cyrillic}', "");
Expect(1, 1279, '\p{Blk=	 cyrillic}', "");
Expect(0, 1279, '\p{^Blk=	 cyrillic}', "");
Expect(0, 1279, '\P{Blk=	 cyrillic}', "");
Expect(1, 1279, '\P{^Blk=	 cyrillic}', "");
Expect(0, 1023, '\p{Blk=	 cyrillic}', "");
Expect(1, 1023, '\p{^Blk=	 cyrillic}', "");
Expect(1, 1023, '\P{Blk=	 cyrillic}', "");
Expect(0, 1023, '\P{^Blk=	 cyrillic}', "");
Error('\p{Is_Block=:=__CYRILLIC}');
Error('\P{Is_Block=:=__CYRILLIC}');
Expect(1, 1279, '\p{Is_Block=cyrillic}', "");
Expect(0, 1279, '\p{^Is_Block=cyrillic}', "");
Expect(0, 1279, '\P{Is_Block=cyrillic}', "");
Expect(1, 1279, '\P{^Is_Block=cyrillic}', "");
Expect(0, 1023, '\p{Is_Block=cyrillic}', "");
Expect(1, 1023, '\p{^Is_Block=cyrillic}', "");
Expect(1, 1023, '\P{Is_Block=cyrillic}', "");
Expect(0, 1023, '\P{^Is_Block=cyrillic}', "");
Expect(1, 1279, '\p{Is_Block: 		cyrillic}', "");
Expect(0, 1279, '\p{^Is_Block: 		cyrillic}', "");
Expect(0, 1279, '\P{Is_Block: 		cyrillic}', "");
Expect(1, 1279, '\P{^Is_Block: 		cyrillic}', "");
Expect(0, 1023, '\p{Is_Block: 		cyrillic}', "");
Expect(1, 1023, '\p{^Is_Block: 		cyrillic}', "");
Expect(1, 1023, '\P{Is_Block: 		cyrillic}', "");
Expect(0, 1023, '\P{^Is_Block: 		cyrillic}', "");
Error('\p{Is_Blk=	/a/cyrillic}');
Error('\P{Is_Blk=	/a/cyrillic}');
Expect(1, 1279, '\p{Is_Blk=cyrillic}', "");
Expect(0, 1279, '\p{^Is_Blk=cyrillic}', "");
Expect(0, 1279, '\P{Is_Blk=cyrillic}', "");
Expect(1, 1279, '\P{^Is_Blk=cyrillic}', "");
Expect(0, 1023, '\p{Is_Blk=cyrillic}', "");
Expect(1, 1023, '\p{^Is_Blk=cyrillic}', "");
Expect(1, 1023, '\P{Is_Blk=cyrillic}', "");
Expect(0, 1023, '\P{^Is_Blk=cyrillic}', "");
Expect(1, 1279, '\p{Is_Blk:   -_Cyrillic}', "");
Expect(0, 1279, '\p{^Is_Blk:   -_Cyrillic}', "");
Expect(0, 1279, '\P{Is_Blk:   -_Cyrillic}', "");
Expect(1, 1279, '\P{^Is_Blk:   -_Cyrillic}', "");
Expect(0, 1023, '\p{Is_Blk:   -_Cyrillic}', "");
Expect(1, 1023, '\p{^Is_Blk:   -_Cyrillic}', "");
Expect(1, 1023, '\P{Is_Blk:   -_Cyrillic}', "");
Expect(0, 1023, '\P{^Is_Blk:   -_Cyrillic}', "");
Error('\p{Block=_-latin-Extended:=a}');
Error('\P{Block=_-latin-Extended:=a}');
Expect(1, 383, '\p{Block=latinextendeda}', "");
Expect(0, 383, '\p{^Block=latinextendeda}', "");
Expect(0, 383, '\P{Block=latinextendeda}', "");
Expect(1, 383, '\P{^Block=latinextendeda}', "");
Expect(0, 255, '\p{Block=latinextendeda}', "");
Expect(1, 255, '\p{^Block=latinextendeda}', "");
Expect(1, 255, '\P{Block=latinextendeda}', "");
Expect(0, 255, '\P{^Block=latinextendeda}', "");
Expect(1, 383, '\p{Block: LATIN_EXTENDEDA}', "");
Expect(0, 383, '\p{^Block: LATIN_EXTENDEDA}', "");
Expect(0, 383, '\P{Block: LATIN_EXTENDEDA}', "");
Expect(1, 383, '\P{^Block: LATIN_EXTENDEDA}', "");
Expect(0, 255, '\p{Block: LATIN_EXTENDEDA}', "");
Expect(1, 255, '\p{^Block: LATIN_EXTENDEDA}', "");
Expect(1, 255, '\P{Block: LATIN_EXTENDEDA}', "");
Expect(0, 255, '\P{^Block: LATIN_EXTENDEDA}', "");
Error('\p{Blk=		Latin-extended/a/A}');
Error('\P{Blk=		Latin-extended/a/A}');
Expect(1, 383, '\p{Blk=latinextendeda}', "");
Expect(0, 383, '\p{^Blk=latinextendeda}', "");
Expect(0, 383, '\P{Blk=latinextendeda}', "");
Expect(1, 383, '\P{^Blk=latinextendeda}', "");
Expect(0, 255, '\p{Blk=latinextendeda}', "");
Expect(1, 255, '\p{^Blk=latinextendeda}', "");
Expect(1, 255, '\P{Blk=latinextendeda}', "");
Expect(0, 255, '\P{^Blk=latinextendeda}', "");
Expect(1, 383, '\p{Blk:   _Latin	Extended A}', "");
Expect(0, 383, '\p{^Blk:   _Latin	Extended A}', "");
Expect(0, 383, '\P{Blk:   _Latin	Extended A}', "");
Expect(1, 383, '\P{^Blk:   _Latin	Extended A}', "");
Expect(0, 255, '\p{Blk:   _Latin	Extended A}', "");
Expect(1, 255, '\p{^Blk:   _Latin	Extended A}', "");
Expect(1, 255, '\P{Blk:   _Latin	Extended A}', "");
Expect(0, 255, '\P{^Blk:   _Latin	Extended A}', "");
Error('\p{Is_Block=/a/LATIN-ExtendedA}');
Error('\P{Is_Block=/a/LATIN-ExtendedA}');
Expect(1, 383, '\p{Is_Block=latinextendeda}', "");
Expect(0, 383, '\p{^Is_Block=latinextendeda}', "");
Expect(0, 383, '\P{Is_Block=latinextendeda}', "");
Expect(1, 383, '\P{^Is_Block=latinextendeda}', "");
Expect(0, 255, '\p{Is_Block=latinextendeda}', "");
Expect(1, 255, '\p{^Is_Block=latinextendeda}', "");
Expect(1, 255, '\P{Is_Block=latinextendeda}', "");
Expect(0, 255, '\P{^Is_Block=latinextendeda}', "");
Expect(1, 383, '\p{Is_Block=-Latinextended_A}', "");
Expect(0, 383, '\p{^Is_Block=-Latinextended_A}', "");
Expect(0, 383, '\P{Is_Block=-Latinextended_A}', "");
Expect(1, 383, '\P{^Is_Block=-Latinextended_A}', "");
Expect(0, 255, '\p{Is_Block=-Latinextended_A}', "");
Expect(1, 255, '\p{^Is_Block=-Latinextended_A}', "");
Expect(1, 255, '\P{Is_Block=-Latinextended_A}', "");
Expect(0, 255, '\P{^Is_Block=-Latinextended_A}', "");
Error('\p{Is_Blk=-Latin/a/Extended	A}');
Error('\P{Is_Blk=-Latin/a/Extended	A}');
Expect(1, 383, '\p{Is_Blk=latinextendeda}', "");
Expect(0, 383, '\p{^Is_Blk=latinextendeda}', "");
Expect(0, 383, '\P{Is_Blk=latinextendeda}', "");
Expect(1, 383, '\P{^Is_Blk=latinextendeda}', "");
Expect(0, 255, '\p{Is_Blk=latinextendeda}', "");
Expect(1, 255, '\p{^Is_Blk=latinextendeda}', "");
Expect(1, 255, '\P{Is_Blk=latinextendeda}', "");
Expect(0, 255, '\P{^Is_Blk=latinextendeda}', "");
Expect(1, 383, '\p{Is_Blk= _LATIN-extended a}', "");
Expect(0, 383, '\p{^Is_Blk= _LATIN-extended a}', "");
Expect(0, 383, '\P{Is_Blk= _LATIN-extended a}', "");
Expect(1, 383, '\P{^Is_Blk= _LATIN-extended a}', "");
Expect(0, 255, '\p{Is_Blk= _LATIN-extended a}', "");
Expect(1, 255, '\p{^Is_Blk= _LATIN-extended a}', "");
Expect(1, 255, '\P{Is_Blk= _LATIN-extended a}', "");
Expect(0, 255, '\P{^Is_Blk= _LATIN-extended a}', "");
Error('\p{Block= :=Balinese}');
Error('\P{Block= :=Balinese}');
Expect(1, 7039, '\p{Block=balinese}', "");
Expect(0, 7039, '\p{^Block=balinese}', "");
Expect(0, 7039, '\P{Block=balinese}', "");
Expect(1, 7039, '\P{^Block=balinese}', "");
Expect(0, 6911, '\p{Block=balinese}', "");
Expect(1, 6911, '\p{^Block=balinese}', "");
Expect(1, 6911, '\P{Block=balinese}', "");
Expect(0, 6911, '\P{^Block=balinese}', "");
Expect(1, 7039, '\p{Block=-_BALINESE}', "");
Expect(0, 7039, '\p{^Block=-_BALINESE}', "");
Expect(0, 7039, '\P{Block=-_BALINESE}', "");
Expect(1, 7039, '\P{^Block=-_BALINESE}', "");
Expect(0, 6911, '\p{Block=-_BALINESE}', "");
Expect(1, 6911, '\p{^Block=-_BALINESE}', "");
Expect(1, 6911, '\P{Block=-_BALINESE}', "");
Expect(0, 6911, '\P{^Block=-_BALINESE}', "");
Error('\p{Blk: /a/Balinese}');
Error('\P{Blk: /a/Balinese}');
Expect(1, 7039, '\p{Blk=balinese}', "");
Expect(0, 7039, '\p{^Blk=balinese}', "");
Expect(0, 7039, '\P{Blk=balinese}', "");
Expect(1, 7039, '\P{^Blk=balinese}', "");
Expect(0, 6911, '\p{Blk=balinese}', "");
Expect(1, 6911, '\p{^Blk=balinese}', "");
Expect(1, 6911, '\P{Blk=balinese}', "");
Expect(0, 6911, '\P{^Blk=balinese}', "");
Expect(1, 7039, '\p{Blk:   		BALINESE}', "");
Expect(0, 7039, '\p{^Blk:   		BALINESE}', "");
Expect(0, 7039, '\P{Blk:   		BALINESE}', "");
Expect(1, 7039, '\P{^Blk:   		BALINESE}', "");
Expect(0, 6911, '\p{Blk:   		BALINESE}', "");
Expect(1, 6911, '\p{^Blk:   		BALINESE}', "");
Expect(1, 6911, '\P{Blk:   		BALINESE}', "");
Expect(0, 6911, '\P{^Blk:   		BALINESE}', "");
Error('\p{Is_Block=/a/-	balinese}');
Error('\P{Is_Block=/a/-	balinese}');
Expect(1, 7039, '\p{Is_Block=balinese}', "");
Expect(0, 7039, '\p{^Is_Block=balinese}', "");
Expect(0, 7039, '\P{Is_Block=balinese}', "");
Expect(1, 7039, '\P{^Is_Block=balinese}', "");
Expect(0, 6911, '\p{Is_Block=balinese}', "");
Expect(1, 6911, '\p{^Is_Block=balinese}', "");
Expect(1, 6911, '\P{Is_Block=balinese}', "");
Expect(0, 6911, '\P{^Is_Block=balinese}', "");
Expect(1, 7039, '\p{Is_Block=-BALINESE}', "");
Expect(0, 7039, '\p{^Is_Block=-BALINESE}', "");
Expect(0, 7039, '\P{Is_Block=-BALINESE}', "");
Expect(1, 7039, '\P{^Is_Block=-BALINESE}', "");
Expect(0, 6911, '\p{Is_Block=-BALINESE}', "");
Expect(1, 6911, '\p{^Is_Block=-BALINESE}', "");
Expect(1, 6911, '\P{Is_Block=-BALINESE}', "");
Expect(0, 6911, '\P{^Is_Block=-BALINESE}', "");
Error('\p{Is_Blk=:=	balinese}');
Error('\P{Is_Blk=:=	balinese}');
Expect(1, 7039, '\p{Is_Blk=balinese}', "");
Expect(0, 7039, '\p{^Is_Blk=balinese}', "");
Expect(0, 7039, '\P{Is_Blk=balinese}', "");
Expect(1, 7039, '\P{^Is_Blk=balinese}', "");
Expect(0, 6911, '\p{Is_Blk=balinese}', "");
Expect(1, 6911, '\p{^Is_Blk=balinese}', "");
Expect(1, 6911, '\P{Is_Blk=balinese}', "");
Expect(0, 6911, '\P{^Is_Blk=balinese}', "");
Expect(1, 7039, '\p{Is_Blk=_-Balinese}', "");
Expect(0, 7039, '\p{^Is_Blk=_-Balinese}', "");
Expect(0, 7039, '\P{Is_Blk=_-Balinese}', "");
Expect(1, 7039, '\P{^Is_Blk=_-Balinese}', "");
Expect(0, 6911, '\p{Is_Blk=_-Balinese}', "");
Expect(1, 6911, '\p{^Is_Blk=_-Balinese}', "");
Expect(1, 6911, '\P{Is_Blk=_-Balinese}', "");
Expect(0, 6911, '\P{^Is_Blk=_-Balinese}', "");
Error('\p{Block:	:=	avestan}');
Error('\P{Block:	:=	avestan}');
Expect(1, 68415, '\p{Block=avestan}', "");
Expect(0, 68415, '\p{^Block=avestan}', "");
Expect(0, 68415, '\P{Block=avestan}', "");
Expect(1, 68415, '\P{^Block=avestan}', "");
Expect(0, 68351, '\p{Block=avestan}', "");
Expect(1, 68351, '\p{^Block=avestan}', "");
Expect(1, 68351, '\P{Block=avestan}', "");
Expect(0, 68351, '\P{^Block=avestan}', "");
Expect(1, 68415, '\p{Block=_avestan}', "");
Expect(0, 68415, '\p{^Block=_avestan}', "");
Expect(0, 68415, '\P{Block=_avestan}', "");
Expect(1, 68415, '\P{^Block=_avestan}', "");
Expect(0, 68351, '\p{Block=_avestan}', "");
Expect(1, 68351, '\p{^Block=_avestan}', "");
Expect(1, 68351, '\P{Block=_avestan}', "");
Expect(0, 68351, '\P{^Block=_avestan}', "");
Error('\p{Blk= -avestan/a/}');
Error('\P{Blk= -avestan/a/}');
Expect(1, 68415, '\p{Blk=avestan}', "");
Expect(0, 68415, '\p{^Blk=avestan}', "");
Expect(0, 68415, '\P{Blk=avestan}', "");
Expect(1, 68415, '\P{^Blk=avestan}', "");
Expect(0, 68351, '\p{Blk=avestan}', "");
Expect(1, 68351, '\p{^Blk=avestan}', "");
Expect(1, 68351, '\P{Blk=avestan}', "");
Expect(0, 68351, '\P{^Blk=avestan}', "");
Expect(1, 68415, '\p{Blk= _Avestan}', "");
Expect(0, 68415, '\p{^Blk= _Avestan}', "");
Expect(0, 68415, '\P{Blk= _Avestan}', "");
Expect(1, 68415, '\P{^Blk= _Avestan}', "");
Expect(0, 68351, '\p{Blk= _Avestan}', "");
Expect(1, 68351, '\p{^Blk= _Avestan}', "");
Expect(1, 68351, '\P{Blk= _Avestan}', "");
Expect(0, 68351, '\P{^Blk= _Avestan}', "");
Error('\p{Is_Block=/a/ 	Avestan}');
Error('\P{Is_Block=/a/ 	Avestan}');
Expect(1, 68415, '\p{Is_Block=avestan}', "");
Expect(0, 68415, '\p{^Is_Block=avestan}', "");
Expect(0, 68415, '\P{Is_Block=avestan}', "");
Expect(1, 68415, '\P{^Is_Block=avestan}', "");
Expect(0, 68351, '\p{Is_Block=avestan}', "");
Expect(1, 68351, '\p{^Is_Block=avestan}', "");
Expect(1, 68351, '\P{Is_Block=avestan}', "");
Expect(0, 68351, '\P{^Is_Block=avestan}', "");
Expect(1, 68415, '\p{Is_Block=_avestan}', "");
Expect(0, 68415, '\p{^Is_Block=_avestan}', "");
Expect(0, 68415, '\P{Is_Block=_avestan}', "");
Expect(1, 68415, '\P{^Is_Block=_avestan}', "");
Expect(0, 68351, '\p{Is_Block=_avestan}', "");
Expect(1, 68351, '\p{^Is_Block=_avestan}', "");
Expect(1, 68351, '\P{Is_Block=_avestan}', "");
Expect(0, 68351, '\P{^Is_Block=_avestan}', "");
Error('\p{Is_Blk: _AVESTAN:=}');
Error('\P{Is_Blk: _AVESTAN:=}');
Expect(1, 68415, '\p{Is_Blk=avestan}', "");
Expect(0, 68415, '\p{^Is_Blk=avestan}', "");
Expect(0, 68415, '\P{Is_Blk=avestan}', "");
Expect(1, 68415, '\P{^Is_Blk=avestan}', "");
Expect(0, 68351, '\p{Is_Blk=avestan}', "");
Expect(1, 68351, '\p{^Is_Blk=avestan}', "");
Expect(1, 68351, '\P{Is_Blk=avestan}', "");
Expect(0, 68351, '\P{^Is_Blk=avestan}', "");
Expect(1, 68415, '\p{Is_Blk= avestan}', "");
Expect(0, 68415, '\p{^Is_Blk= avestan}', "");
Expect(0, 68415, '\P{Is_Blk= avestan}', "");
Expect(1, 68415, '\P{^Is_Blk= avestan}', "");
Expect(0, 68351, '\p{Is_Blk= avestan}', "");
Expect(1, 68351, '\p{^Is_Blk= avestan}', "");
Expect(1, 68351, '\P{Is_Blk= avestan}', "");
Expect(0, 68351, '\P{^Is_Blk= avestan}', "");
Error('\p{Block=/a/  myanmarEXTENDED-A}');
Error('\P{Block=/a/  myanmarEXTENDED-A}');
Expect(1, 43647, '\p{Block=myanmarextendeda}', "");
Expect(0, 43647, '\p{^Block=myanmarextendeda}', "");
Expect(0, 43647, '\P{Block=myanmarextendeda}', "");
Expect(1, 43647, '\P{^Block=myanmarextendeda}', "");
Expect(0, 43615, '\p{Block=myanmarextendeda}', "");
Expect(1, 43615, '\p{^Block=myanmarextendeda}', "");
Expect(1, 43615, '\P{Block=myanmarextendeda}', "");
Expect(0, 43615, '\P{^Block=myanmarextendeda}', "");
Expect(1, 43647, '\p{Block=	 MYANMAR Extended-A}', "");
Expect(0, 43647, '\p{^Block=	 MYANMAR Extended-A}', "");
Expect(0, 43647, '\P{Block=	 MYANMAR Extended-A}', "");
Expect(1, 43647, '\P{^Block=	 MYANMAR Extended-A}', "");
Expect(0, 43615, '\p{Block=	 MYANMAR Extended-A}', "");
Expect(1, 43615, '\p{^Block=	 MYANMAR Extended-A}', "");
Expect(1, 43615, '\P{Block=	 MYANMAR Extended-A}', "");
Expect(0, 43615, '\P{^Block=	 MYANMAR Extended-A}', "");
Error('\p{Blk=-myanmar_extended-a:=}');
Error('\P{Blk=-myanmar_extended-a:=}');
Expect(1, 43647, '\p{Blk=myanmarextendeda}', "");
Expect(0, 43647, '\p{^Blk=myanmarextendeda}', "");
Expect(0, 43647, '\P{Blk=myanmarextendeda}', "");
Expect(1, 43647, '\P{^Blk=myanmarextendeda}', "");
Expect(0, 43615, '\p{Blk=myanmarextendeda}', "");
Expect(1, 43615, '\p{^Blk=myanmarextendeda}', "");
Expect(1, 43615, '\P{Blk=myanmarextendeda}', "");
Expect(0, 43615, '\P{^Blk=myanmarextendeda}', "");
Expect(1, 43647, '\p{Blk:_MYANMAR_EXTENDED-A}', "");
Expect(0, 43647, '\p{^Blk:_MYANMAR_EXTENDED-A}', "");
Expect(0, 43647, '\P{Blk:_MYANMAR_EXTENDED-A}', "");
Expect(1, 43647, '\P{^Blk:_MYANMAR_EXTENDED-A}', "");
Expect(0, 43615, '\p{Blk:_MYANMAR_EXTENDED-A}', "");
Expect(1, 43615, '\p{^Blk:_MYANMAR_EXTENDED-A}', "");
Expect(1, 43615, '\P{Blk:_MYANMAR_EXTENDED-A}', "");
Expect(0, 43615, '\P{^Blk:_MYANMAR_EXTENDED-A}', "");
Error('\p{Is_Block: -:=Myanmar-Extended-a}');
Error('\P{Is_Block: -:=Myanmar-Extended-a}');
Expect(1, 43647, '\p{Is_Block:	myanmarextendeda}', "");
Expect(0, 43647, '\p{^Is_Block:	myanmarextendeda}', "");
Expect(0, 43647, '\P{Is_Block:	myanmarextendeda}', "");
Expect(1, 43647, '\P{^Is_Block:	myanmarextendeda}', "");
Expect(0, 43615, '\p{Is_Block:	myanmarextendeda}', "");
Expect(1, 43615, '\p{^Is_Block:	myanmarextendeda}', "");
Expect(1, 43615, '\P{Is_Block:	myanmarextendeda}', "");
Expect(0, 43615, '\P{^Is_Block:	myanmarextendeda}', "");
Expect(1, 43647, '\p{Is_Block=-Myanmar-EXTENDED-A}', "");
Expect(0, 43647, '\p{^Is_Block=-Myanmar-EXTENDED-A}', "");
Expect(0, 43647, '\P{Is_Block=-Myanmar-EXTENDED-A}', "");
Expect(1, 43647, '\P{^Is_Block=-Myanmar-EXTENDED-A}', "");
Expect(0, 43615, '\p{Is_Block=-Myanmar-EXTENDED-A}', "");
Expect(1, 43615, '\p{^Is_Block=-Myanmar-EXTENDED-A}', "");
Expect(1, 43615, '\P{Is_Block=-Myanmar-EXTENDED-A}', "");
Expect(0, 43615, '\P{^Is_Block=-Myanmar-EXTENDED-A}', "");
Error('\p{Is_Blk=  MYANMAR/a/extended-a}');
Error('\P{Is_Blk=  MYANMAR/a/extended-a}');
Expect(1, 43647, '\p{Is_Blk=myanmarextendeda}', "");
Expect(0, 43647, '\p{^Is_Blk=myanmarextendeda}', "");
Expect(0, 43647, '\P{Is_Blk=myanmarextendeda}', "");
Expect(1, 43647, '\P{^Is_Blk=myanmarextendeda}', "");
Expect(0, 43615, '\p{Is_Blk=myanmarextendeda}', "");
Expect(1, 43615, '\p{^Is_Blk=myanmarextendeda}', "");
Expect(1, 43615, '\P{Is_Blk=myanmarextendeda}', "");
Expect(0, 43615, '\P{^Is_Blk=myanmarextendeda}', "");
Expect(1, 43647, '\p{Is_Blk=-_MYANMARExtendedA}', "");
Expect(0, 43647, '\p{^Is_Blk=-_MYANMARExtendedA}', "");
Expect(0, 43647, '\P{Is_Blk=-_MYANMARExtendedA}', "");
Expect(1, 43647, '\P{^Is_Blk=-_MYANMARExtendedA}', "");
Expect(0, 43615, '\p{Is_Blk=-_MYANMARExtendedA}', "");
Expect(1, 43615, '\p{^Is_Blk=-_MYANMARExtendedA}', "");
Expect(1, 43615, '\P{Is_Blk=-_MYANMARExtendedA}', "");
Expect(0, 43615, '\P{^Is_Blk=-_MYANMARExtendedA}', "");
Error('\p{Block=	COUNTING_Rod:=Numerals}');
Error('\P{Block=	COUNTING_Rod:=Numerals}');
Expect(1, 119679, '\p{Block=countingrodnumerals}', "");
Expect(0, 119679, '\p{^Block=countingrodnumerals}', "");
Expect(0, 119679, '\P{Block=countingrodnumerals}', "");
Expect(1, 119679, '\P{^Block=countingrodnumerals}', "");
Expect(0, 119647, '\p{Block=countingrodnumerals}', "");
Expect(1, 119647, '\p{^Block=countingrodnumerals}', "");
Expect(1, 119647, '\P{Block=countingrodnumerals}', "");
Expect(0, 119647, '\P{^Block=countingrodnumerals}', "");
Expect(1, 119679, '\p{Block=	COUNTING rod	NUMERALS}', "");
Expect(0, 119679, '\p{^Block=	COUNTING rod	NUMERALS}', "");
Expect(0, 119679, '\P{Block=	COUNTING rod	NUMERALS}', "");
Expect(1, 119679, '\P{^Block=	COUNTING rod	NUMERALS}', "");
Expect(0, 119647, '\p{Block=	COUNTING rod	NUMERALS}', "");
Expect(1, 119647, '\p{^Block=	COUNTING rod	NUMERALS}', "");
Expect(1, 119647, '\P{Block=	COUNTING rod	NUMERALS}', "");
Expect(0, 119647, '\P{^Block=	COUNTING rod	NUMERALS}', "");
Error('\p{Blk=/a/-COUNTING rod Numerals}');
Error('\P{Blk=/a/-COUNTING rod Numerals}');
Expect(1, 119679, '\p{Blk=countingrodnumerals}', "");
Expect(0, 119679, '\p{^Blk=countingrodnumerals}', "");
Expect(0, 119679, '\P{Blk=countingrodnumerals}', "");
Expect(1, 119679, '\P{^Blk=countingrodnumerals}', "");
Expect(0, 119647, '\p{Blk=countingrodnumerals}', "");
Expect(1, 119647, '\p{^Blk=countingrodnumerals}', "");
Expect(1, 119647, '\P{Blk=countingrodnumerals}', "");
Expect(0, 119647, '\P{^Blk=countingrodnumerals}', "");
Expect(1, 119679, '\p{Blk=	-COUNTINGRod	Numerals}', "");
Expect(0, 119679, '\p{^Blk=	-COUNTINGRod	Numerals}', "");
Expect(0, 119679, '\P{Blk=	-COUNTINGRod	Numerals}', "");
Expect(1, 119679, '\P{^Blk=	-COUNTINGRod	Numerals}', "");
Expect(0, 119647, '\p{Blk=	-COUNTINGRod	Numerals}', "");
Expect(1, 119647, '\p{^Blk=	-COUNTINGRod	Numerals}', "");
Expect(1, 119647, '\P{Blk=	-COUNTINGRod	Numerals}', "");
Expect(0, 119647, '\P{^Blk=	-COUNTINGRod	Numerals}', "");
Error('\p{Is_Block=/a/COUNTING	rod_NUMERALS}');
Error('\P{Is_Block=/a/COUNTING	rod_NUMERALS}');
Expect(1, 119679, '\p{Is_Block=countingrodnumerals}', "");
Expect(0, 119679, '\p{^Is_Block=countingrodnumerals}', "");
Expect(0, 119679, '\P{Is_Block=countingrodnumerals}', "");
Expect(1, 119679, '\P{^Is_Block=countingrodnumerals}', "");
Expect(0, 119647, '\p{Is_Block=countingrodnumerals}', "");
Expect(1, 119647, '\p{^Is_Block=countingrodnumerals}', "");
Expect(1, 119647, '\P{Is_Block=countingrodnumerals}', "");
Expect(0, 119647, '\P{^Is_Block=countingrodnumerals}', "");
Expect(1, 119679, '\p{Is_Block: 	Counting	Rod	Numerals}', "");
Expect(0, 119679, '\p{^Is_Block: 	Counting	Rod	Numerals}', "");
Expect(0, 119679, '\P{Is_Block: 	Counting	Rod	Numerals}', "");
Expect(1, 119679, '\P{^Is_Block: 	Counting	Rod	Numerals}', "");
Expect(0, 119647, '\p{Is_Block: 	Counting	Rod	Numerals}', "");
Expect(1, 119647, '\p{^Is_Block: 	Counting	Rod	Numerals}', "");
Expect(1, 119647, '\P{Is_Block: 	Counting	Rod	Numerals}', "");
Expect(0, 119647, '\P{^Is_Block: 	Counting	Rod	Numerals}', "");
Error('\p{Is_Blk= 	Counting_rod/a/Numerals}');
Error('\P{Is_Blk= 	Counting_rod/a/Numerals}');
Expect(1, 119679, '\p{Is_Blk=countingrodnumerals}', "");
Expect(0, 119679, '\p{^Is_Blk=countingrodnumerals}', "");
Expect(0, 119679, '\P{Is_Blk=countingrodnumerals}', "");
Expect(1, 119679, '\P{^Is_Blk=countingrodnumerals}', "");
Expect(0, 119647, '\p{Is_Blk=countingrodnumerals}', "");
Expect(1, 119647, '\p{^Is_Blk=countingrodnumerals}', "");
Expect(1, 119647, '\P{Is_Blk=countingrodnumerals}', "");
Expect(0, 119647, '\P{^Is_Blk=countingrodnumerals}', "");
Expect(1, 119679, '\p{Is_Blk=- Counting_RodNumerals}', "");
Expect(0, 119679, '\p{^Is_Blk=- Counting_RodNumerals}', "");
Expect(0, 119679, '\P{Is_Blk=- Counting_RodNumerals}', "");
Expect(1, 119679, '\P{^Is_Blk=- Counting_RodNumerals}', "");
Expect(0, 119647, '\p{Is_Blk=- Counting_RodNumerals}', "");
Expect(1, 119647, '\p{^Is_Blk=- Counting_RodNumerals}', "");
Expect(1, 119647, '\P{Is_Blk=- Counting_RodNumerals}', "");
Expect(0, 119647, '\P{^Is_Blk=- Counting_RodNumerals}', "");
Error('\p{Block=_/a/ByzantinemusicalSYMBOLS}');
Error('\P{Block=_/a/ByzantinemusicalSYMBOLS}');
Expect(1, 119039, '\p{Block=byzantinemusicalsymbols}', "");
Expect(0, 119039, '\p{^Block=byzantinemusicalsymbols}', "");
Expect(0, 119039, '\P{Block=byzantinemusicalsymbols}', "");
Expect(1, 119039, '\P{^Block=byzantinemusicalsymbols}', "");
Expect(0, 118783, '\p{Block=byzantinemusicalsymbols}', "");
Expect(1, 118783, '\p{^Block=byzantinemusicalsymbols}', "");
Expect(1, 118783, '\P{Block=byzantinemusicalsymbols}', "");
Expect(0, 118783, '\P{^Block=byzantinemusicalsymbols}', "");
Expect(1, 119039, '\p{Block=-_Byzantine MUSICAL-SYMBOLS}', "");
Expect(0, 119039, '\p{^Block=-_Byzantine MUSICAL-SYMBOLS}', "");
Expect(0, 119039, '\P{Block=-_Byzantine MUSICAL-SYMBOLS}', "");
Expect(1, 119039, '\P{^Block=-_Byzantine MUSICAL-SYMBOLS}', "");
Expect(0, 118783, '\p{Block=-_Byzantine MUSICAL-SYMBOLS}', "");
Expect(1, 118783, '\p{^Block=-_Byzantine MUSICAL-SYMBOLS}', "");
Expect(1, 118783, '\P{Block=-_Byzantine MUSICAL-SYMBOLS}', "");
Expect(0, 118783, '\P{^Block=-_Byzantine MUSICAL-SYMBOLS}', "");
Error('\p{Blk=		Byzantine Musical/a/SYMBOLS}');
Error('\P{Blk=		Byzantine Musical/a/SYMBOLS}');
Expect(1, 119039, '\p{Blk=byzantinemusicalsymbols}', "");
Expect(0, 119039, '\p{^Blk=byzantinemusicalsymbols}', "");
Expect(0, 119039, '\P{Blk=byzantinemusicalsymbols}', "");
Expect(1, 119039, '\P{^Blk=byzantinemusicalsymbols}', "");
Expect(0, 118783, '\p{Blk=byzantinemusicalsymbols}', "");
Expect(1, 118783, '\p{^Blk=byzantinemusicalsymbols}', "");
Expect(1, 118783, '\P{Blk=byzantinemusicalsymbols}', "");
Expect(0, 118783, '\P{^Blk=byzantinemusicalsymbols}', "");
Expect(1, 119039, '\p{Blk=  byzantine-musicalsymbols}', "");
Expect(0, 119039, '\p{^Blk=  byzantine-musicalsymbols}', "");
Expect(0, 119039, '\P{Blk=  byzantine-musicalsymbols}', "");
Expect(1, 119039, '\P{^Blk=  byzantine-musicalsymbols}', "");
Expect(0, 118783, '\p{Blk=  byzantine-musicalsymbols}', "");
Expect(1, 118783, '\p{^Blk=  byzantine-musicalsymbols}', "");
Expect(1, 118783, '\P{Blk=  byzantine-musicalsymbols}', "");
Expect(0, 118783, '\P{^Blk=  byzantine-musicalsymbols}', "");
Error('\p{Is_Block=	/a/byzantine	MUSICAL-SYMBOLS}');
Error('\P{Is_Block=	/a/byzantine	MUSICAL-SYMBOLS}');
Expect(1, 119039, '\p{Is_Block:byzantinemusicalsymbols}', "");
Expect(0, 119039, '\p{^Is_Block:byzantinemusicalsymbols}', "");
Expect(0, 119039, '\P{Is_Block:byzantinemusicalsymbols}', "");
Expect(1, 119039, '\P{^Is_Block:byzantinemusicalsymbols}', "");
Expect(0, 118783, '\p{Is_Block:byzantinemusicalsymbols}', "");
Expect(1, 118783, '\p{^Is_Block:byzantinemusicalsymbols}', "");
Expect(1, 118783, '\P{Is_Block:byzantinemusicalsymbols}', "");
Expect(0, 118783, '\P{^Is_Block:byzantinemusicalsymbols}', "");
Expect(1, 119039, '\p{Is_Block=	 BYZANTINE Musical-SYMBOLS}', "");
Expect(0, 119039, '\p{^Is_Block=	 BYZANTINE Musical-SYMBOLS}', "");
Expect(0, 119039, '\P{Is_Block=	 BYZANTINE Musical-SYMBOLS}', "");
Expect(1, 119039, '\P{^Is_Block=	 BYZANTINE Musical-SYMBOLS}', "");
Expect(0, 118783, '\p{Is_Block=	 BYZANTINE Musical-SYMBOLS}', "");
Expect(1, 118783, '\p{^Is_Block=	 BYZANTINE Musical-SYMBOLS}', "");
Expect(1, 118783, '\P{Is_Block=	 BYZANTINE Musical-SYMBOLS}', "");
Expect(0, 118783, '\P{^Is_Block=	 BYZANTINE Musical-SYMBOLS}', "");
Error('\p{Is_Blk=-	ByzantineMusical:=symbols}');
Error('\P{Is_Blk=-	ByzantineMusical:=symbols}');
Expect(1, 119039, '\p{Is_Blk=byzantinemusicalsymbols}', "");
Expect(0, 119039, '\p{^Is_Blk=byzantinemusicalsymbols}', "");
Expect(0, 119039, '\P{Is_Blk=byzantinemusicalsymbols}', "");
Expect(1, 119039, '\P{^Is_Blk=byzantinemusicalsymbols}', "");
Expect(0, 118783, '\p{Is_Blk=byzantinemusicalsymbols}', "");
Expect(1, 118783, '\p{^Is_Blk=byzantinemusicalsymbols}', "");
Expect(1, 118783, '\P{Is_Blk=byzantinemusicalsymbols}', "");
Expect(0, 118783, '\P{^Is_Blk=byzantinemusicalsymbols}', "");
Expect(1, 119039, '\p{Is_Blk=_-BYZANTINEMusical symbols}', "");
Expect(0, 119039, '\p{^Is_Blk=_-BYZANTINEMusical symbols}', "");
Expect(0, 119039, '\P{Is_Blk=_-BYZANTINEMusical symbols}', "");
Expect(1, 119039, '\P{^Is_Blk=_-BYZANTINEMusical symbols}', "");
Expect(0, 118783, '\p{Is_Blk=_-BYZANTINEMusical symbols}', "");
Expect(1, 118783, '\p{^Is_Blk=_-BYZANTINEMusical symbols}', "");
Expect(1, 118783, '\P{Is_Blk=_-BYZANTINEMusical symbols}', "");
Expect(0, 118783, '\P{^Is_Blk=_-BYZANTINEMusical symbols}', "");
Error('\p{Block=tamil:=}');
Error('\P{Block=tamil:=}');
Expect(1, 3071, '\p{Block=tamil}', "");
Expect(0, 3071, '\p{^Block=tamil}', "");
Expect(0, 3071, '\P{Block=tamil}', "");
Expect(1, 3071, '\P{^Block=tamil}', "");
Expect(0, 2943, '\p{Block=tamil}', "");
Expect(1, 2943, '\p{^Block=tamil}', "");
Expect(1, 2943, '\P{Block=tamil}', "");
Expect(0, 2943, '\P{^Block=tamil}', "");
Expect(1, 3071, '\p{Block=-	Tamil}', "");
Expect(0, 3071, '\p{^Block=-	Tamil}', "");
Expect(0, 3071, '\P{Block=-	Tamil}', "");
Expect(1, 3071, '\P{^Block=-	Tamil}', "");
Expect(0, 2943, '\p{Block=-	Tamil}', "");
Expect(1, 2943, '\p{^Block=-	Tamil}', "");
Expect(1, 2943, '\P{Block=-	Tamil}', "");
Expect(0, 2943, '\P{^Block=-	Tamil}', "");
Error('\p{Blk=-/a/TAMIL}');
Error('\P{Blk=-/a/TAMIL}');
Expect(1, 3071, '\p{Blk:tamil}', "");
Expect(0, 3071, '\p{^Blk:tamil}', "");
Expect(0, 3071, '\P{Blk:tamil}', "");
Expect(1, 3071, '\P{^Blk:tamil}', "");
Expect(0, 2943, '\p{Blk:tamil}', "");
Expect(1, 2943, '\p{^Blk:tamil}', "");
Expect(1, 2943, '\P{Blk:tamil}', "");
Expect(0, 2943, '\P{^Blk:tamil}', "");
Expect(1, 3071, '\p{Blk=-TAMIL}', "");
Expect(0, 3071, '\p{^Blk=-TAMIL}', "");
Expect(0, 3071, '\P{Blk=-TAMIL}', "");
Expect(1, 3071, '\P{^Blk=-TAMIL}', "");
Expect(0, 2943, '\p{Blk=-TAMIL}', "");
Expect(1, 2943, '\p{^Blk=-TAMIL}', "");
Expect(1, 2943, '\P{Blk=-TAMIL}', "");
Expect(0, 2943, '\P{^Blk=-TAMIL}', "");
Error('\p{Is_Block=Tamil:=}');
Error('\P{Is_Block=Tamil:=}');
Expect(1, 3071, '\p{Is_Block=tamil}', "");
Expect(0, 3071, '\p{^Is_Block=tamil}', "");
Expect(0, 3071, '\P{Is_Block=tamil}', "");
Expect(1, 3071, '\P{^Is_Block=tamil}', "");
Expect(0, 2943, '\p{Is_Block=tamil}', "");
Expect(1, 2943, '\p{^Is_Block=tamil}', "");
Expect(1, 2943, '\P{Is_Block=tamil}', "");
Expect(0, 2943, '\P{^Is_Block=tamil}', "");
Expect(1, 3071, '\p{Is_Block= 	tamil}', "");
Expect(0, 3071, '\p{^Is_Block= 	tamil}', "");
Expect(0, 3071, '\P{Is_Block= 	tamil}', "");
Expect(1, 3071, '\P{^Is_Block= 	tamil}', "");
Expect(0, 2943, '\p{Is_Block= 	tamil}', "");
Expect(1, 2943, '\p{^Is_Block= 	tamil}', "");
Expect(1, 2943, '\P{Is_Block= 	tamil}', "");
Expect(0, 2943, '\P{^Is_Block= 	tamil}', "");
Error('\p{Is_Blk=-Tamil/a/}');
Error('\P{Is_Blk=-Tamil/a/}');
Expect(1, 3071, '\p{Is_Blk=tamil}', "");
Expect(0, 3071, '\p{^Is_Blk=tamil}', "");
Expect(0, 3071, '\P{Is_Blk=tamil}', "");
Expect(1, 3071, '\P{^Is_Blk=tamil}', "");
Expect(0, 2943, '\p{Is_Blk=tamil}', "");
Expect(1, 2943, '\p{^Is_Blk=tamil}', "");
Expect(1, 2943, '\P{Is_Blk=tamil}', "");
Expect(0, 2943, '\P{^Is_Blk=tamil}', "");
Expect(1, 3071, '\p{Is_Blk= 	tamil}', "");
Expect(0, 3071, '\p{^Is_Blk= 	tamil}', "");
Expect(0, 3071, '\P{Is_Blk= 	tamil}', "");
Expect(1, 3071, '\P{^Is_Blk= 	tamil}', "");
Expect(0, 2943, '\p{Is_Blk= 	tamil}', "");
Expect(1, 2943, '\p{^Is_Blk= 	tamil}', "");
Expect(1, 2943, '\P{Is_Blk= 	tamil}', "");
Expect(0, 2943, '\P{^Is_Blk= 	tamil}', "");
Error('\p{Block=/a/specials}');
Error('\P{Block=/a/specials}');
Expect(1, 65520, '\p{Block=specials}', "");
Expect(0, 65520, '\p{^Block=specials}', "");
Expect(0, 65520, '\P{Block=specials}', "");
Expect(1, 65520, '\P{^Block=specials}', "");
Expect(0, 65519, '\p{Block=specials}', "");
Expect(1, 65519, '\p{^Block=specials}', "");
Expect(1, 65519, '\P{Block=specials}', "");
Expect(0, 65519, '\P{^Block=specials}', "");
Expect(1, 65520, '\p{Block=		Specials}', "");
Expect(0, 65520, '\p{^Block=		Specials}', "");
Expect(0, 65520, '\P{Block=		Specials}', "");
Expect(1, 65520, '\P{^Block=		Specials}', "");
Expect(0, 65519, '\p{Block=		Specials}', "");
Expect(1, 65519, '\p{^Block=		Specials}', "");
Expect(1, 65519, '\P{Block=		Specials}', "");
Expect(0, 65519, '\P{^Block=		Specials}', "");
Error('\p{Blk=	:=specials}');
Error('\P{Blk=	:=specials}');
Expect(1, 65520, '\p{Blk=specials}', "");
Expect(0, 65520, '\p{^Blk=specials}', "");
Expect(0, 65520, '\P{Blk=specials}', "");
Expect(1, 65520, '\P{^Blk=specials}', "");
Expect(0, 65519, '\p{Blk=specials}', "");
Expect(1, 65519, '\p{^Blk=specials}', "");
Expect(1, 65519, '\P{Blk=specials}', "");
Expect(0, 65519, '\P{^Blk=specials}', "");
Expect(1, 65520, '\p{Blk=_specials}', "");
Expect(0, 65520, '\p{^Blk=_specials}', "");
Expect(0, 65520, '\P{Blk=_specials}', "");
Expect(1, 65520, '\P{^Blk=_specials}', "");
Expect(0, 65519, '\p{Blk=_specials}', "");
Expect(1, 65519, '\p{^Blk=_specials}', "");
Expect(1, 65519, '\P{Blk=_specials}', "");
Expect(0, 65519, '\P{^Blk=_specials}', "");
Error('\p{Is_Block=	 specials:=}');
Error('\P{Is_Block=	 specials:=}');
Expect(1, 65520, '\p{Is_Block=specials}', "");
Expect(0, 65520, '\p{^Is_Block=specials}', "");
Expect(0, 65520, '\P{Is_Block=specials}', "");
Expect(1, 65520, '\P{^Is_Block=specials}', "");
Expect(0, 65519, '\p{Is_Block=specials}', "");
Expect(1, 65519, '\p{^Is_Block=specials}', "");
Expect(1, 65519, '\P{Is_Block=specials}', "");
Expect(0, 65519, '\P{^Is_Block=specials}', "");
Expect(1, 65520, '\p{Is_Block: _-Specials}', "");
Expect(0, 65520, '\p{^Is_Block: _-Specials}', "");
Expect(0, 65520, '\P{Is_Block: _-Specials}', "");
Expect(1, 65520, '\P{^Is_Block: _-Specials}', "");
Expect(0, 65519, '\p{Is_Block: _-Specials}', "");
Expect(1, 65519, '\p{^Is_Block: _-Specials}', "");
Expect(1, 65519, '\P{Is_Block: _-Specials}', "");
Expect(0, 65519, '\P{^Is_Block: _-Specials}', "");
Error('\p{Is_Blk=		Specials:=}');
Error('\P{Is_Blk=		Specials:=}');
Expect(1, 65520, '\p{Is_Blk=specials}', "");
Expect(0, 65520, '\p{^Is_Blk=specials}', "");
Expect(0, 65520, '\P{Is_Blk=specials}', "");
Expect(1, 65520, '\P{^Is_Blk=specials}', "");
Expect(0, 65519, '\p{Is_Blk=specials}', "");
Expect(1, 65519, '\p{^Is_Blk=specials}', "");
Expect(1, 65519, '\P{Is_Blk=specials}', "");
Expect(0, 65519, '\P{^Is_Blk=specials}', "");
Expect(1, 65520, '\p{Is_Blk=-SPECIALS}', "");
Expect(0, 65520, '\p{^Is_Blk=-SPECIALS}', "");
Expect(0, 65520, '\P{Is_Blk=-SPECIALS}', "");
Expect(1, 65520, '\P{^Is_Blk=-SPECIALS}', "");
Expect(0, 65519, '\p{Is_Blk=-SPECIALS}', "");
Expect(1, 65519, '\p{^Is_Blk=-SPECIALS}', "");
Expect(1, 65519, '\P{Is_Blk=-SPECIALS}', "");
Expect(0, 65519, '\P{^Is_Blk=-SPECIALS}', "");
Error('\p{Block=- georgian/a/}');
Error('\P{Block=- georgian/a/}');
Expect(1, 4351, '\p{Block=georgian}', "");
Expect(0, 4351, '\p{^Block=georgian}', "");
Expect(0, 4351, '\P{Block=georgian}', "");
Expect(1, 4351, '\P{^Block=georgian}', "");
Expect(0, 4255, '\p{Block=georgian}', "");
Expect(1, 4255, '\p{^Block=georgian}', "");
Expect(1, 4255, '\P{Block=georgian}', "");
Expect(0, 4255, '\P{^Block=georgian}', "");
Expect(1, 4351, '\p{Block=	 georgian}', "");
Expect(0, 4351, '\p{^Block=	 georgian}', "");
Expect(0, 4351, '\P{Block=	 georgian}', "");
Expect(1, 4351, '\P{^Block=	 georgian}', "");
Expect(0, 4255, '\p{Block=	 georgian}', "");
Expect(1, 4255, '\p{^Block=	 georgian}', "");
Expect(1, 4255, '\P{Block=	 georgian}', "");
Expect(0, 4255, '\P{^Block=	 georgian}', "");
Error('\p{Blk=/a/georgian}');
Error('\P{Blk=/a/georgian}');
Expect(1, 4351, '\p{Blk=georgian}', "");
Expect(0, 4351, '\p{^Blk=georgian}', "");
Expect(0, 4351, '\P{Blk=georgian}', "");
Expect(1, 4351, '\P{^Blk=georgian}', "");
Expect(0, 4255, '\p{Blk=georgian}', "");
Expect(1, 4255, '\p{^Blk=georgian}', "");
Expect(1, 4255, '\P{Blk=georgian}', "");
Expect(0, 4255, '\P{^Blk=georgian}', "");
Expect(1, 4351, '\p{Blk=--georgian}', "");
Expect(0, 4351, '\p{^Blk=--georgian}', "");
Expect(0, 4351, '\P{Blk=--georgian}', "");
Expect(1, 4351, '\P{^Blk=--georgian}', "");
Expect(0, 4255, '\p{Blk=--georgian}', "");
Expect(1, 4255, '\p{^Blk=--georgian}', "");
Expect(1, 4255, '\P{Blk=--georgian}', "");
Expect(0, 4255, '\P{^Blk=--georgian}', "");
Error('\p{Is_Block= :=Georgian}');
Error('\P{Is_Block= :=Georgian}');
Expect(1, 4351, '\p{Is_Block=georgian}', "");
Expect(0, 4351, '\p{^Is_Block=georgian}', "");
Expect(0, 4351, '\P{Is_Block=georgian}', "");
Expect(1, 4351, '\P{^Is_Block=georgian}', "");
Expect(0, 4255, '\p{Is_Block=georgian}', "");
Expect(1, 4255, '\p{^Is_Block=georgian}', "");
Expect(1, 4255, '\P{Is_Block=georgian}', "");
Expect(0, 4255, '\P{^Is_Block=georgian}', "");
Expect(1, 4351, '\p{Is_Block=_-Georgian}', "");
Expect(0, 4351, '\p{^Is_Block=_-Georgian}', "");
Expect(0, 4351, '\P{Is_Block=_-Georgian}', "");
Expect(1, 4351, '\P{^Is_Block=_-Georgian}', "");
Expect(0, 4255, '\p{Is_Block=_-Georgian}', "");
Expect(1, 4255, '\p{^Is_Block=_-Georgian}', "");
Expect(1, 4255, '\P{Is_Block=_-Georgian}', "");
Expect(0, 4255, '\P{^Is_Block=_-Georgian}', "");
Error('\p{Is_Blk=:=-	georgian}');
Error('\P{Is_Blk=:=-	georgian}');
Expect(1, 4351, '\p{Is_Blk=georgian}', "");
Expect(0, 4351, '\p{^Is_Blk=georgian}', "");
Expect(0, 4351, '\P{Is_Blk=georgian}', "");
Expect(1, 4351, '\P{^Is_Blk=georgian}', "");
Expect(0, 4255, '\p{Is_Blk=georgian}', "");
Expect(1, 4255, '\p{^Is_Blk=georgian}', "");
Expect(1, 4255, '\P{Is_Blk=georgian}', "");
Expect(0, 4255, '\P{^Is_Blk=georgian}', "");
Expect(1, 4351, '\p{Is_Blk=_Georgian}', "");
Expect(0, 4351, '\p{^Is_Blk=_Georgian}', "");
Expect(0, 4351, '\P{Is_Blk=_Georgian}', "");
Expect(1, 4351, '\P{^Is_Blk=_Georgian}', "");
Expect(0, 4255, '\p{Is_Blk=_Georgian}', "");
Expect(1, 4255, '\p{^Is_Blk=_Georgian}', "");
Expect(1, 4255, '\P{Is_Blk=_Georgian}', "");
Expect(0, 4255, '\P{^Is_Blk=_Georgian}', "");
Error('\p{Block=- emoticons:=}');
Error('\P{Block=- emoticons:=}');
Expect(1, 128591, '\p{Block=emoticons}', "");
Expect(0, 128591, '\p{^Block=emoticons}', "");
Expect(0, 128591, '\P{Block=emoticons}', "");
Expect(1, 128591, '\P{^Block=emoticons}', "");
Expect(0, 128511, '\p{Block=emoticons}', "");
Expect(1, 128511, '\p{^Block=emoticons}', "");
Expect(1, 128511, '\P{Block=emoticons}', "");
Expect(0, 128511, '\P{^Block=emoticons}', "");
Expect(1, 128591, '\p{Block=-	emoticons}', "");
Expect(0, 128591, '\p{^Block=-	emoticons}', "");
Expect(0, 128591, '\P{Block=-	emoticons}', "");
Expect(1, 128591, '\P{^Block=-	emoticons}', "");
Expect(0, 128511, '\p{Block=-	emoticons}', "");
Expect(1, 128511, '\p{^Block=-	emoticons}', "");
Expect(1, 128511, '\P{Block=-	emoticons}', "");
Expect(0, 128511, '\P{^Block=-	emoticons}', "");
Error('\p{Blk:	--EMOTICONS:=}');
Error('\P{Blk:	--EMOTICONS:=}');
Expect(1, 128591, '\p{Blk=emoticons}', "");
Expect(0, 128591, '\p{^Blk=emoticons}', "");
Expect(0, 128591, '\P{Blk=emoticons}', "");
Expect(1, 128591, '\P{^Blk=emoticons}', "");
Expect(0, 128511, '\p{Blk=emoticons}', "");
Expect(1, 128511, '\p{^Blk=emoticons}', "");
Expect(1, 128511, '\P{Blk=emoticons}', "");
Expect(0, 128511, '\P{^Blk=emoticons}', "");
Expect(1, 128591, '\p{Blk=	EMOTICONS}', "");
Expect(0, 128591, '\p{^Blk=	EMOTICONS}', "");
Expect(0, 128591, '\P{Blk=	EMOTICONS}', "");
Expect(1, 128591, '\P{^Blk=	EMOTICONS}', "");
Expect(0, 128511, '\p{Blk=	EMOTICONS}', "");
Expect(1, 128511, '\p{^Blk=	EMOTICONS}', "");
Expect(1, 128511, '\P{Blk=	EMOTICONS}', "");
Expect(0, 128511, '\P{^Blk=	EMOTICONS}', "");
Error('\p{Is_Block=	 Emoticons/a/}');
Error('\P{Is_Block=	 Emoticons/a/}');
Expect(1, 128591, '\p{Is_Block=emoticons}', "");
Expect(0, 128591, '\p{^Is_Block=emoticons}', "");
Expect(0, 128591, '\P{Is_Block=emoticons}', "");
Expect(1, 128591, '\P{^Is_Block=emoticons}', "");
Expect(0, 128511, '\p{Is_Block=emoticons}', "");
Expect(1, 128511, '\p{^Is_Block=emoticons}', "");
Expect(1, 128511, '\P{Is_Block=emoticons}', "");
Expect(0, 128511, '\P{^Is_Block=emoticons}', "");
Expect(1, 128591, '\p{Is_Block=__Emoticons}', "");
Expect(0, 128591, '\p{^Is_Block=__Emoticons}', "");
Expect(0, 128591, '\P{Is_Block=__Emoticons}', "");
Expect(1, 128591, '\P{^Is_Block=__Emoticons}', "");
Expect(0, 128511, '\p{Is_Block=__Emoticons}', "");
Expect(1, 128511, '\p{^Is_Block=__Emoticons}', "");
Expect(1, 128511, '\P{Is_Block=__Emoticons}', "");
Expect(0, 128511, '\P{^Is_Block=__Emoticons}', "");
Error('\p{Is_Blk:	-:=EMOTICONS}');
Error('\P{Is_Blk:	-:=EMOTICONS}');
Expect(1, 128591, '\p{Is_Blk=emoticons}', "");
Expect(0, 128591, '\p{^Is_Blk=emoticons}', "");
Expect(0, 128591, '\P{Is_Blk=emoticons}', "");
Expect(1, 128591, '\P{^Is_Blk=emoticons}', "");
Expect(0, 128511, '\p{Is_Blk=emoticons}', "");
Expect(1, 128511, '\p{^Is_Blk=emoticons}', "");
Expect(1, 128511, '\P{Is_Blk=emoticons}', "");
Expect(0, 128511, '\P{^Is_Blk=emoticons}', "");
Expect(1, 128591, '\p{Is_Blk= 	EMOTICONS}', "");
Expect(0, 128591, '\p{^Is_Blk= 	EMOTICONS}', "");
Expect(0, 128591, '\P{Is_Blk= 	EMOTICONS}', "");
Expect(1, 128591, '\P{^Is_Blk= 	EMOTICONS}', "");
Expect(0, 128511, '\p{Is_Blk= 	EMOTICONS}', "");
Expect(1, 128511, '\p{^Is_Blk= 	EMOTICONS}', "");
Expect(1, 128511, '\P{Is_Blk= 	EMOTICONS}', "");
Expect(0, 128511, '\P{^Is_Blk= 	EMOTICONS}', "");
Error('\p{Block=:=transport_andmapSymbols}');
Error('\P{Block=:=transport_andmapSymbols}');
Expect(1, 128767, '\p{Block: transportandmapsymbols}', "");
Expect(0, 128767, '\p{^Block: transportandmapsymbols}', "");
Expect(0, 128767, '\P{Block: transportandmapsymbols}', "");
Expect(1, 128767, '\P{^Block: transportandmapsymbols}', "");
Expect(0, 128639, '\p{Block: transportandmapsymbols}', "");
Expect(1, 128639, '\p{^Block: transportandmapsymbols}', "");
Expect(1, 128639, '\P{Block: transportandmapsymbols}', "");
Expect(0, 128639, '\P{^Block: transportandmapsymbols}', "");
Expect(1, 128767, '\p{Block=-Transport-AndMAP-Symbols}', "");
Expect(0, 128767, '\p{^Block=-Transport-AndMAP-Symbols}', "");
Expect(0, 128767, '\P{Block=-Transport-AndMAP-Symbols}', "");
Expect(1, 128767, '\P{^Block=-Transport-AndMAP-Symbols}', "");
Expect(0, 128639, '\p{Block=-Transport-AndMAP-Symbols}', "");
Expect(1, 128639, '\p{^Block=-Transport-AndMAP-Symbols}', "");
Expect(1, 128639, '\P{Block=-Transport-AndMAP-Symbols}', "");
Expect(0, 128639, '\P{^Block=-Transport-AndMAP-Symbols}', "");
Error('\p{Blk:     Transport-And/a/map	Symbols}');
Error('\P{Blk:     Transport-And/a/map	Symbols}');
Expect(1, 128767, '\p{Blk:transportandmapsymbols}', "");
Expect(0, 128767, '\p{^Blk:transportandmapsymbols}', "");
Expect(0, 128767, '\P{Blk:transportandmapsymbols}', "");
Expect(1, 128767, '\P{^Blk:transportandmapsymbols}', "");
Expect(0, 128639, '\p{Blk:transportandmapsymbols}', "");
Expect(1, 128639, '\p{^Blk:transportandmapsymbols}', "");
Expect(1, 128639, '\P{Blk:transportandmapsymbols}', "");
Expect(0, 128639, '\P{^Blk:transportandmapsymbols}', "");
Expect(1, 128767, '\p{Blk:-	Transport-And_Mapsymbols}', "");
Expect(0, 128767, '\p{^Blk:-	Transport-And_Mapsymbols}', "");
Expect(0, 128767, '\P{Blk:-	Transport-And_Mapsymbols}', "");
Expect(1, 128767, '\P{^Blk:-	Transport-And_Mapsymbols}', "");
Expect(0, 128639, '\p{Blk:-	Transport-And_Mapsymbols}', "");
Expect(1, 128639, '\p{^Blk:-	Transport-And_Mapsymbols}', "");
Expect(1, 128639, '\P{Blk:-	Transport-And_Mapsymbols}', "");
Expect(0, 128639, '\P{^Blk:-	Transport-And_Mapsymbols}', "");
Error('\p{Is_Block=	/a/transport_and-MAP_Symbols}');
Error('\P{Is_Block=	/a/transport_and-MAP_Symbols}');
Expect(1, 128767, '\p{Is_Block=transportandmapsymbols}', "");
Expect(0, 128767, '\p{^Is_Block=transportandmapsymbols}', "");
Expect(0, 128767, '\P{Is_Block=transportandmapsymbols}', "");
Expect(1, 128767, '\P{^Is_Block=transportandmapsymbols}', "");
Expect(0, 128639, '\p{Is_Block=transportandmapsymbols}', "");
Expect(1, 128639, '\p{^Is_Block=transportandmapsymbols}', "");
Expect(1, 128639, '\P{Is_Block=transportandmapsymbols}', "");
Expect(0, 128639, '\P{^Is_Block=transportandmapsymbols}', "");
Expect(1, 128767, '\p{Is_Block=	_TRANSPORT-and map-Symbols}', "");
Expect(0, 128767, '\p{^Is_Block=	_TRANSPORT-and map-Symbols}', "");
Expect(0, 128767, '\P{Is_Block=	_TRANSPORT-and map-Symbols}', "");
Expect(1, 128767, '\P{^Is_Block=	_TRANSPORT-and map-Symbols}', "");
Expect(0, 128639, '\p{Is_Block=	_TRANSPORT-and map-Symbols}', "");
Expect(1, 128639, '\p{^Is_Block=	_TRANSPORT-and map-Symbols}', "");
Expect(1, 128639, '\P{Is_Block=	_TRANSPORT-and map-Symbols}', "");
Expect(0, 128639, '\P{^Is_Block=	_TRANSPORT-and map-Symbols}', "");
Error('\p{Is_Blk=-	transport and/a/map-Symbols}');
Error('\P{Is_Blk=-	transport and/a/map-Symbols}');
Expect(1, 128767, '\p{Is_Blk=transportandmapsymbols}', "");
Expect(0, 128767, '\p{^Is_Blk=transportandmapsymbols}', "");
Expect(0, 128767, '\P{Is_Blk=transportandmapsymbols}', "");
Expect(1, 128767, '\P{^Is_Blk=transportandmapsymbols}', "");
Expect(0, 128639, '\p{Is_Blk=transportandmapsymbols}', "");
Expect(1, 128639, '\p{^Is_Blk=transportandmapsymbols}', "");
Expect(1, 128639, '\P{Is_Blk=transportandmapsymbols}', "");
Expect(0, 128639, '\P{^Is_Blk=transportandmapsymbols}', "");
Expect(1, 128767, '\p{Is_Blk= _transport_AND	map_Symbols}', "");
Expect(0, 128767, '\p{^Is_Blk= _transport_AND	map_Symbols}', "");
Expect(0, 128767, '\P{Is_Blk= _transport_AND	map_Symbols}', "");
Expect(1, 128767, '\P{^Is_Blk= _transport_AND	map_Symbols}', "");
Expect(0, 128639, '\p{Is_Blk= _transport_AND	map_Symbols}', "");
Expect(1, 128639, '\p{^Is_Blk= _transport_AND	map_Symbols}', "");
Expect(1, 128639, '\P{Is_Blk= _transport_AND	map_Symbols}', "");
Expect(0, 128639, '\P{^Is_Blk= _transport_AND	map_Symbols}', "");
Error('\p{Block:	:=kayahLi}');
Error('\P{Block:	:=kayahLi}');
Expect(1, 43311, '\p{Block=kayahli}', "");
Expect(0, 43311, '\p{^Block=kayahli}', "");
Expect(0, 43311, '\P{Block=kayahli}', "");
Expect(1, 43311, '\P{^Block=kayahli}', "");
Expect(0, 43263, '\p{Block=kayahli}', "");
Expect(1, 43263, '\p{^Block=kayahli}', "");
Expect(1, 43263, '\P{Block=kayahli}', "");
Expect(0, 43263, '\P{^Block=kayahli}', "");
Expect(1, 43311, '\p{Block=-KAYAH	li}', "");
Expect(0, 43311, '\p{^Block=-KAYAH	li}', "");
Expect(0, 43311, '\P{Block=-KAYAH	li}', "");
Expect(1, 43311, '\P{^Block=-KAYAH	li}', "");
Expect(0, 43263, '\p{Block=-KAYAH	li}', "");
Expect(1, 43263, '\p{^Block=-KAYAH	li}', "");
Expect(1, 43263, '\P{Block=-KAYAH	li}', "");
Expect(0, 43263, '\P{^Block=-KAYAH	li}', "");
Error('\p{Blk=/a/-_kayah-li}');
Error('\P{Blk=/a/-_kayah-li}');
Expect(1, 43311, '\p{Blk=kayahli}', "");
Expect(0, 43311, '\p{^Blk=kayahli}', "");
Expect(0, 43311, '\P{Blk=kayahli}', "");
Expect(1, 43311, '\P{^Blk=kayahli}', "");
Expect(0, 43263, '\p{Blk=kayahli}', "");
Expect(1, 43263, '\p{^Blk=kayahli}', "");
Expect(1, 43263, '\P{Blk=kayahli}', "");
Expect(0, 43263, '\P{^Blk=kayahli}', "");
Expect(1, 43311, '\p{Blk=__Kayah-Li}', "");
Expect(0, 43311, '\p{^Blk=__Kayah-Li}', "");
Expect(0, 43311, '\P{Blk=__Kayah-Li}', "");
Expect(1, 43311, '\P{^Blk=__Kayah-Li}', "");
Expect(0, 43263, '\p{Blk=__Kayah-Li}', "");
Expect(1, 43263, '\p{^Blk=__Kayah-Li}', "");
Expect(1, 43263, '\P{Blk=__Kayah-Li}', "");
Expect(0, 43263, '\P{^Blk=__Kayah-Li}', "");
Error('