#!/usr/bin/env python
# vim:fileencoding=utf-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011, Karl Weckworth <kweckwor@gmail.com>'
__docformat__ = 'restructuredtext en'

# This file contains the actual implementation of the classes required to
# implement the ManageSonyX50BookList plugin, with any supporting classes it requires

SONY_VENDOR_ID = [0x054c]
SONY_X50_PRODUCT_ID = [0x031e]

CONFIG_BOOKS_LIST_KEY = 'booksList'
CONFIG_UNREAD_LIST_KEY = 'unreadList'
CONFIG_PROFILES_KEY = 'profiles'
CONFIG_NAME_KEY = 'name'
CONFIG_BOOKS_KEY = 'books'
CONFIG_BOOKS_VALUE_SELECTION = 'selection'
CONFIG_BOOKS_COLLECTIONS_KEY = 'collections'
CONFIG_BOOKS_QUERY_KEY = 'query'
CONFIG_STRATEGY_KEY = 'strategy'
CONFIG_STRATEGY_VALUE_ONE = 'one'
CONFIG_STRATEGY_VALUE_NEWEST = 'newest'
CONFIG_STRATEGY_VALUE_OLDEST = 'oldest'
CONFIG_STRATEGY_VALUE_RANDOM = 'random'
CONFIG_ORDER_KEY = 'order'
CONFIG_ORDER_VALUE_DATE = 'date'
CONFIG_ORDER_VALUE_AUTHOR = 'author'
CONFIG_ORDER_VALUE_TITLE = 'title'
CONFIG_POS_KEY = 'pos'

DEFAULT_CONFIG_BOOKS_LIST_QUERY = _('tags:=reading')
DEFAULT_CONFIG_UNREAD_LIST_QUERY = _('not (tags:=read or tags:=reading)')
DEFAULT_CONFIG_BOOKS_LIST_BOOKS = CONFIG_BOOKS_VALUE_SELECTION
DEFAULT_CONFIG_UNREAD_LIST_BOOKS = {CONFIG_BOOKS_QUERY_KEY: DEFAULT_CONFIG_UNREAD_LIST_QUERY}
DEFAULT_CONFIG_STRATEGY = CONFIG_STRATEGY_VALUE_NEWEST
DEFAULT_CONFIG_ORDER = CONFIG_ORDER_VALUE_DATE

DEFAULT_CONFIG_PROFILE_KEY = _('Default')
DEFAULT_CONFIG_BOOKS_LIST = {CONFIG_PROFILES_KEY: {DEFAULT_CONFIG_PROFILE_KEY: {CONFIG_POS_KEY: 0}}}
DEFAULT_CONFIG_UNREAD_LIST = {CONFIG_PROFILES_KEY: {DEFAULT_CONFIG_PROFILE_KEY: {CONFIG_POS_KEY: 0}}}

MAIN_INDEX = 0
CARD_A_INDEX = 1
CARD_B_INDEX = 2
CACHE_INDEX = 0
CACHE_EXT_INDEX = 1
DEVICE_KEY_PREFIXES = ['Main:', 'A:', 'B:']
DEVICE_KEY_INDICES = {'Main': MAIN_INDEX, 'A': CARD_A_INDEX, 'B': CARD_B_INDEX}
SONY_MEDIA_XML = 'database/cache/media.xml'
SONY_CACHE_XML = 'Sony Reader/database/cache.xml'
SONY_MAIN_CACHE_EXT_XML = 'database/cache/cacheExt.xml'
SONY_CARD_CACHE_EXT_XML = 'Sony Reader/database/cacheExt.xml'
SONY_CACHE_FILE = [[SONY_MEDIA_XML, SONY_CACHE_XML, SONY_CACHE_XML],
                   [SONY_MAIN_CACHE_EXT_XML, SONY_CARD_CACHE_EXT_XML, SONY_CARD_CACHE_EXT_XML]]
READER_DRIVE_LABEL = [_('Main'), _('Card A'), _('Card B')]

MAX_BOOK_LIST_LENGTH = 3

ICON_MANAGE_BOOKS = 'images/managebooks.png'
ICON_BOOKS_LIST = 'images/bookslist.png'
ICON_MARK_READ = 'images/markread.png'
ICON_MARK_UNREAD = 'images/markunread.png'
ICON_SET_UNREAD = 'images/setunread.png'
ICON_CONFIG = 'config.png'

ACTION_TITLE_MANAGE_BOOKS = _('Book Lists')
ACTION_TOOLTIP_MANAGE_BOOKS = _('Manage Sony x50 reader book lists')
ACTION_TITLE_BOOKS_LIST = _('Set Books List...')
ACTION_TOOLTIP_BOOKS_LIST = _('Manage reader home screen books list')
ACTION_TITLE_MANAGE_UNREAD = _('Manage Unread Collection')
ACTION_TOOLTIP_MANAGE_UNREAD = _('Manage reader unread collection')
ACTION_TITLE_MARK_READ = _('Mark Selected Read')
ACTION_TOOLTIP_MARK_READ = _('Remove selected books from unread collection') 
ACTION_TITLE_MARK_UNREAD = _('Mark Selected Unread')
ACTION_TOOLTIP_MARK_UNREAD = _('Add selected books to unread collection') 
ACTION_TITLE_SET_UNREAD = _('Set Unread Collection...')
ACTION_TOOLTIP_SET_UNREAD = _('Set unread collection based on metadata criteria')
ACTION_TITLE_CONFIG = _('Customize Plugin...')
ACTION_TOOLTIP_CONFIG = _('Launch the plugin configuration dialog')

from calibre_plugins.ManageSonyX50BookList.__init__ import PLUGIN_NAME
from calibre.utils.config import JSONConfig

prefs = JSONConfig('plugins/ManageSonyX50BookList')
prefs.defaults[CONFIG_BOOKS_LIST_KEY] = DEFAULT_CONFIG_BOOKS_LIST
prefs.defaults[CONFIG_UNREAD_LIST_KEY] = DEFAULT_CONFIG_UNREAD_LIST

def getPluginCustomization(pluginNamedObject):
    import copy
    from calibre.customize.ui import plugin_customization
    
    pluginCustomization = plugin_customization(pluginNamedObject)
    
    if pluginCustomization is not None and len(pluginCustomization.strip()) > 0:
        migratePreferences_v10(pluginNamedObject, pluginCustomization)
    
    if prefs.get(CONFIG_PROFILES_KEY) is not None:
        migratePreferences_v11();
    
    return copy.deepcopy({CONFIG_BOOKS_LIST_KEY: prefs.get(CONFIG_BOOKS_LIST_KEY), CONFIG_UNREAD_LIST_KEY: prefs.get(CONFIG_UNREAD_LIST_KEY)})

def migratePreferences_v10(pluginNamedObject, pluginCustomization):
    import json
    from calibre.customize.ui import customize_plugin
    
    legacyConfigString = pluginCustomization.strip()
    legacyConfig = json.loads(legacyConfigString)
    
    prefs[CONFIG_BOOKS_LIST_KEY] = {CONFIG_PROFILES_KEY: {}}
    profiles = prefs[CONFIG_BOOKS_LIST_KEY][CONFIG_PROFILES_KEY]
    
    for index, profileName in enumerate(legacyConfig.keys()):
        profile = legacyConfig[profileName]
        
        if CONFIG_BOOKS_KEY in profile.keys() and profile[CONFIG_BOOKS_KEY] != CONFIG_BOOKS_VALUE_SELECTION:
            booksCollections = profile[CONFIG_BOOKS_KEY]
            del profile[CONFIG_BOOKS_KEY]
            profile[CONFIG_BOOKS_KEY] = {CONFIG_BOOKS_COLLECTIONS_KEY: booksCollections}
        
        profile[CONFIG_POS_KEY] = index
        profiles[profileName] = profile
    
    customize_plugin(pluginNamedObject, '')
    prefs.commit()

def migratePreferences_v11():
    profiles = prefs[CONFIG_PROFILES_KEY]
    del prefs[CONFIG_PROFILES_KEY]
    
    for profileName in profiles.keys():
        profile = profiles[profileName]
        
        if CONFIG_BOOKS_KEY in profile.keys() and profile[CONFIG_BOOKS_KEY] != CONFIG_BOOKS_VALUE_SELECTION:
            booksCollections = profile[CONFIG_BOOKS_KEY]
            del profile[CONFIG_BOOKS_KEY]
            profile[CONFIG_BOOKS_KEY] = {CONFIG_BOOKS_COLLECTIONS_KEY: booksCollections}
            
    prefs[CONFIG_BOOKS_LIST_KEY] = {CONFIG_PROFILES_KEY: profiles}
    prefs.commit()

from calibre.gui2.actions import InterfaceAction
from PyQt4.Qt import QToolButton

class ManageSonyX50BookListAction(InterfaceAction):
    action_type = 'current'
    popup_type = QToolButton.InstantPopup
    action_spec = (ACTION_TITLE_MANAGE_BOOKS, None, ACTION_TOOLTIP_MANAGE_BOOKS, None)
    dont_add_to = frozenset(['toolbar', 'toolbar-child', 'menubar', 'context-menu'])
    #DEBUG
    #dont_add_to = frozenset(['context-menu', 'context-menu-device'])
    
    def __init__(self, gui, site_customization):
        InterfaceAction.__init__(self, gui, site_customization)
        self.setBooksListAction = SetBooksListAction(self.gui)
        self.markReadAction = MarkReadAction(self.gui)
        self.markUnreadAction = MarkUnreadAction(self.gui)
        self.markReadAction = MarkReadAction(self.gui)
        self.setUnreadListAction = SetUnreadListAction(self.gui)
        
    def genesis(self):
        from calibre_plugins.ManageSonyX50BookList.view import createMenu, createAction, createIcon
        
        iconNames = [ICON_MANAGE_BOOKS, ICON_BOOKS_LIST, ICON_MARK_READ, ICON_MARK_UNREAD, ICON_SET_UNREAD]
        iconResources = self.load_resources(iconNames)
        
        unreadMenu = createMenu(self)
        createAction(self, unreadMenu, ACTION_TITLE_MARK_READ, ACTION_TOOLTIP_MARK_READ, createIcon(iconResource=iconResources[ICON_MARK_READ]), self.markRead)
        createAction(self, unreadMenu, ACTION_TITLE_MARK_UNREAD, ACTION_TOOLTIP_MARK_UNREAD, createIcon(iconResource=iconResources[ICON_MARK_UNREAD]), self.markUnread)
        createAction(self, unreadMenu, ACTION_TITLE_SET_UNREAD, ACTION_TOOLTIP_SET_UNREAD, createIcon(iconResource=iconResources[ICON_SET_UNREAD]), self.setUnreadList)
        
        menu = createMenu(self)
        createAction(self, menu, ACTION_TITLE_BOOKS_LIST, ACTION_TOOLTIP_BOOKS_LIST, createIcon(iconResource=iconResources[ICON_BOOKS_LIST]), self.setBooksList)
        createAction(self, menu, ACTION_TITLE_MANAGE_UNREAD, ACTION_TOOLTIP_MANAGE_UNREAD).setMenu(unreadMenu)
        menu.addSeparator()
        createAction(self, menu, ACTION_TITLE_CONFIG, ACTION_TOOLTIP_CONFIG, createIcon(iconName=ICON_CONFIG), self.showConfiguration)
        
        self.qaction.setMenu(menu)
        self.qaction.setIcon(createIcon(iconResource=iconResources[ICON_MANAGE_BOOKS]))

    def showConfiguration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)
    
    def setBooksList(self):
        self.setBooksListAction.manageBooks()
    
    def markRead(self):
        self.markReadAction.manageBooks()
        
    def markUnread(self):
        self.markUnreadAction.manageBooks()
    
    def setUnreadList(self):
        self.setUnreadListAction.manageBooks()
    
class ManageBookListsAction():
    def __init__(self, gui, loadCacheExt=False):
        self.gui = gui
        self.loadCacheExt = loadCacheExt
                
    def manageBooks(self):
        from calibre.gui2 import warning_dialog
        
        okToContinue = True
        devInfo = SonyX50Reader(self.gui)
        devInfo.detect()
        
        if devInfo.isConnected is False:
            okToContinue = False
        
        if okToContinue:
            okToContinue = self.handleUI(devInfo)
        
        if okToContinue:
            if self.gui.job_manager.has_device_jobs():
                warning_dialog(self.gui, PLUGIN_NAME, _('A background device job is running. Wait until that completes and try again.'), show=True)
                okToContinue = False
        
        if okToContinue:
            cache = SonyX50ReaderCache(self.gui, devInfo)
            cache.load(self.loadCacheExt)
            
            if cache.cacheDOM is None:
                okToContinue = False
        
        if okToContinue:
            bookList = BookList(devInfo)
            bookList.build(cache.cacheDOM)
            
            if bookList.books is None:
                okToContinue = False
            
        if okToContinue:
            #DEBUG
            keys = bookList.books.keys()
            keys.sort(key=lambda key: bookList.books[key]['dateSorter'])
            print '********manageBooks********'
            for id in keys:
                print '*** bookList.books[' + str(id) + ']: tz=' + str(bookList.books[id]['tz']) + '; date=' + str(bookList.books[id]['date']) + ' (' + str(bookList.books[id]['dateSorter']) + '); title=' + str(bookList.books[id]['title'])
            print '********manageBooks********'
            print '*** latestDateTimeStamp=' + str(bookList.latestDateTimeStamp)
            print '********manageBooks********'
            #DEBUG
            self.handleBackend(devInfo, cache, bookList)

class SetBooksListAction(ManageBookListsAction):
    name = PLUGIN_NAME
    
    def handleUI(self, devInfo):
        from calibre_plugins.ManageSonyX50BookList.view import ManageBookListView
        from calibre_plugins.ManageSonyX50BookList.view import ManageBookListController
        
        okToContinue = True
        config = Config(getPluginCustomization(self))
        dialog = ManageBookListView(self.gui)
        controller = ManageBookListController(config.config[CONFIG_BOOKS_LIST_KEY][CONFIG_PROFILES_KEY], dialog)
        
        if dialog.exec_() == dialog.Accepted:
            self.profile = config.getBooksListProfile(dialog.getProfileName())
        else:
            okToContinue = False
            
        return okToContinue
    
    def handleBackend(self, devInfo, cache, bookList):
        bookSelection = BookSelection(self.gui)
        libraryBookList = LibraryBookList(self.gui)
        
        if CONFIG_BOOKS_VALUE_SELECTION in self.profile[CONFIG_BOOKS_KEY]:
            bookSelection.getSelection(bookList.pathIdMap)
        elif CONFIG_BOOKS_QUERY_KEY in self.profile[CONFIG_BOOKS_KEY].keys():
            libraryBookList.build(self.profile[CONFIG_BOOKS_KEY][CONFIG_BOOKS_QUERY_KEY])
            
        if bookSelection.books is not None and libraryBookList.books is not None:
            self.handleProfile(cache, bookList, bookSelection, libraryBookList)
        
    def handleProfile(self, cache, bookList, bookSelection, libraryBookList):
        shortBookList = ShortBookList(self.profile, bookList, bookSelection, libraryBookList)
        shortBookList.extract()
        
        if shortBookList.books is not None:
            self.handleShortBookList(cache, shortBookList, bookList.latestDateTimeStamp)
    
    def handleShortBookList(self, cache, shortBookList, latestDateTimeStamp):
        from calibre.gui2 import info_dialog
        
        if len(shortBookList.books) is 0:
            info_dialog(self.gui, PLUGIN_NAME, 'No books satisfied the selection criteria for this profile.', show=True)
        
        else:
            cache.updateTimestamp(shortBookList.books, latestDateTimeStamp)
            info_dialog(self.gui, PLUGIN_NAME, self.formatConfirmation(shortBookList), show=True)
    
    def formatConfirmation(self, shortBookList):
        confirmation = _('The Sony Reader books list was successfully updated with the following books:') + '<ul>'
        
        for book in shortBookList.books:
            bookLabel = _('{0} by {1}').format(book['title'], book['author'])
            confirmation = confirmation + '<li>' + bookLabel + '</li>'
        
        confirmation = confirmation + '</ul>'
        
        return confirmation
        
class MarkReadAction(ManageBookListsAction):
    def __init__(self, gui):
        ManageBookListsAction.__init__(self, gui, True)
        
    def handleUI(self, devInfo):
        # Nothing to do here since this action uses Calibre device book list for selection.
        return True
    
    def handleBackend(self, devInfo, cache, bookList):
        from calibre.gui2 import info_dialog
        
        bookSelection = BookSelection(self.gui)
        bookSelection.getSelection(bookList.pathIdMap)
            
        if bookSelection.books is not None:
            cache.updateOpenedState(bookList, bookSelection.books, None)
            info_dialog(self.gui, PLUGIN_NAME, self.formatConfirmation(len(bookSelection.books)), show=True)
    
    def formatConfirmation(self, numBooks):
        confirmation = _('The selected books were successfully removed from the Sony Reader Unread Books collection. The number of books removed was {0}.')
        confirmation = confirmation.format(numBooks)
        
        return confirmation
        
class MarkUnreadAction(ManageBookListsAction):
    def __init__(self, gui):
        ManageBookListsAction.__init__(self, gui, True)
        
    def handleUI(self, devInfo):
        # Nothing to do here since this action uses Calibre device book list for selection.
        return True
    
    def handleBackend(self, devInfo, cache, bookList):
        from calibre.gui2 import info_dialog
        
        bookSelection = BookSelection(self.gui)
        bookSelection.getSelection(bookList.pathIdMap)
            
        if bookSelection.books is not None:
            cache.updateOpenedState(bookList, None, bookSelection.books)
            info_dialog(self.gui, PLUGIN_NAME, self.formatConfirmation(len(bookSelection.books)), show=True)
    
    def formatConfirmation(self, numBooks):
        confirmation = _('The selected books were successfully added to the Sony Reader Unread Books collection. The number of books added was {0}.')
        confirmation = confirmation.format(numBooks)
        
        return confirmation

class SetUnreadListAction(ManageBookListsAction):
    name = PLUGIN_NAME
    
    def __init__(self, gui):
        ManageBookListsAction.__init__(self, gui, True)
        
    def handleUI(self, devInfo):
        from calibre_plugins.ManageSonyX50BookList.view import ManageUnreadListView
        from calibre_plugins.ManageSonyX50BookList.view import ManageUnreadListController
        
        okToContinue = True
        config = Config(getPluginCustomization(self))
        dialog = ManageUnreadListView(self.gui)
        controller = ManageUnreadListController(config.config[CONFIG_UNREAD_LIST_KEY][CONFIG_PROFILES_KEY], dialog)
        
        if dialog.exec_() == dialog.Accepted:
            self.profile = config.getUnreadListProfile(dialog.getProfileName())
        else:
            okToContinue = False
            
        return okToContinue
    
    def handleBackend(self, devInfo, cache, bookList):
        from calibre.gui2 import info_dialog
        
        libraryBookList = LibraryBookList(self.gui)
        
        if CONFIG_BOOKS_QUERY_KEY in self.profile[CONFIG_BOOKS_KEY].keys():
            libraryBookList.build(self.profile[CONFIG_BOOKS_KEY][CONFIG_BOOKS_QUERY_KEY])
            
        if libraryBookList.books is not None:
            splitBookList = SplitBookList(self.profile, bookList, libraryBookList)
            splitBookList.split()
            
            cache.updateOpenedState(bookList, splitBookList.openedIds, splitBookList.unopenedIds)
            info_dialog(self.gui, PLUGIN_NAME, self.formatConfirmation(splitBookList), show=True)
    
    def formatConfirmation(self, splitBookList):
        confirmation = _('The Sony Reader Unread Books collection was successfully synchronized with the profile criteria. The results are:') + '<ul>'
        
        countLabel = _('Number of unread books: {0}').format(str(len(splitBookList.unopenedIds)))
        confirmation = confirmation + '<li>' + countLabel + '</li>'
        
        countLabel = _('Number of read books: {0}').format(str(len(splitBookList.openedIds)))
        confirmation = confirmation + '<li>' + countLabel + '</li>'
        
        confirmation = confirmation + '</ul>'
        
        return confirmation
        
class Config():
    def __init__(self, config):
        self.config = config
        booksListProfiles = self.config[CONFIG_BOOKS_LIST_KEY][CONFIG_PROFILES_KEY]
        unreadListProfiles = self.config[CONFIG_UNREAD_LIST_KEY][CONFIG_PROFILES_KEY]
        
        for name in booksListProfiles.keys():
            self.normalizeBooksList(name, booksListProfiles[name])
    
        for name in unreadListProfiles.keys():
            self.normalizeUnreadList(name, unreadListProfiles[name])
    
    def normalizeBooksList(self, profileName, profile):
        keys = profile.keys()
        
        profile[CONFIG_NAME_KEY] = profileName
        
        if CONFIG_BOOKS_KEY not in keys:
            profile[CONFIG_BOOKS_KEY] = DEFAULT_CONFIG_BOOKS_LIST_BOOKS
        
        if CONFIG_STRATEGY_KEY not in keys:
            profile[CONFIG_STRATEGY_KEY] = [DEFAULT_CONFIG_STRATEGY]
        
        if CONFIG_STRATEGY_VALUE_NEWEST not in profile[CONFIG_STRATEGY_KEY] and \
           CONFIG_STRATEGY_VALUE_OLDEST not in profile[CONFIG_STRATEGY_KEY] and \
           CONFIG_STRATEGY_VALUE_RANDOM not in profile[CONFIG_STRATEGY_KEY]:
            profile[CONFIG_STRATEGY_KEY].append(DEFAULT_CONFIG_STRATEGY)
        
        if CONFIG_ORDER_KEY not in keys:
            profile[CONFIG_ORDER_KEY] = DEFAULT_CONFIG_ORDER
    
    def normalizeUnreadList(self, profileName, profile):
        keys = profile.keys()
        
        profile[CONFIG_NAME_KEY] = profileName
        
        if CONFIG_BOOKS_KEY not in keys:
            profile[CONFIG_BOOKS_KEY] = DEFAULT_CONFIG_UNREAD_LIST_BOOKS
            
    def getBooksListProfile(self, profileName):
        return self.config[CONFIG_BOOKS_LIST_KEY][CONFIG_PROFILES_KEY][profileName]
    
    def getUnreadListProfile(self, profileName):
        return self.config[CONFIG_UNREAD_LIST_KEY][CONFIG_PROFILES_KEY][profileName]
    
class SonyX50Reader():
    def __init__(self, gui):
        self.gui = gui
        self.initState()
    
    def initState(self):
        self.isConnected = True
        self.rootPath = [None, None, None] # [main, card A, card B]
        self.vendorId = None
        self.productId = None
    
    def detect(self):
        from calibre.gui2 import error_dialog
        
        self.initState()
        
        # Confirm Sony is connected
        if self.gui.device_manager.connected_device is not None:
            self.rootPath[MAIN_INDEX] = self.gui.device_manager.connected_device._main_prefix
            self.rootPath[CARD_A_INDEX] = self.gui.device_manager.connected_device._card_a_prefix
            self.rootPath[CARD_B_INDEX] = self.gui.device_manager.connected_device._card_b_prefix
            self.vendorId = self.gui.device_manager.connected_device.VENDOR_ID
            self.productId = self.gui.device_manager.connected_device.PRODUCT_ID
            
        else:
            error_dialog(self.gui, PLUGIN_NAME, _('Sony reader not detected.'), show=True)
            #self.isConnected = False
            
            # DEBUG - comment out lines above and uncomment those below to access test data
            #self.rootPath[MAIN_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.authors/'
            #self.rootPath[MAIN_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.author_sort/'
            #self.rootPath[MAIN_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.empty/'
            #self.rootPath[MAIN_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.2/'
            #self.rootPath[MAIN_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.main/'
            #self.rootPath[CARD_A_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.a/'
            #self.rootPath[CARD_B_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.b/'
            self.rootPath[MAIN_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.debug.main/'
            self.vendorId = SONY_VENDOR_ID
            self.productId = SONY_X50_PRODUCT_ID
            
        if self.isConnected and ((self.vendorId != SONY_VENDOR_ID) or (self.productId != SONY_X50_PRODUCT_ID)):
            error_dialog(self.gui, PLUGIN_NAME, _('Reader does not appear to be a Sony x50 model.'), show=True)
            self.isConnected = False
        
class SonyX50ReaderCache():
    def __init__(self, gui, devInfo):
        self.gui = gui
        self.devInfo = devInfo
        self.initState()
    
    def initState(self):
        self.cacheDOM = [[None, None, None], [None, None, None]] # [cache/cacheExt][main, card a, card b]
    
    def load(self, loadExt):
        from calibre.gui2 import error_dialog
        
        self.initState()
        
        if self.devInfo.rootPath[MAIN_INDEX] is not None:
            self.cacheDOM[CACHE_INDEX][MAIN_INDEX] = self.loadCache(CACHE_INDEX, MAIN_INDEX)
            if loadExt:
                self.cacheDOM[CACHE_EXT_INDEX][MAIN_INDEX] = self.loadCache(CACHE_EXT_INDEX, MAIN_INDEX)
            
        if self.devInfo.rootPath[CARD_A_INDEX] is not None:
            self.cacheDOM[CACHE_INDEX][CARD_A_INDEX] = self.loadCache(CACHE_INDEX, CARD_A_INDEX)
            if loadExt:
               self.cacheDOM[CACHE_EXT_INDEX][CARD_A_INDEX] = self.loadCache(CACHE_EXT_INDEX, CARD_A_INDEX)
            
        if self.devInfo.rootPath[CARD_B_INDEX] is not None:
            self.cacheDOM[CACHE_INDEX][CARD_B_INDEX] = self.loadCache(CACHE_INDEX, CARD_B_INDEX)
            if loadExt:
               self.cacheDOM[CACHE_EXT_INDEX][CARD_B_INDEX] = self.loadCache(CACHE_EXT_INDEX, CARD_B_INDEX)
        
        if (self.cacheDOM[CACHE_INDEX][MAIN_INDEX] is None) and (self.cacheDOM[CACHE_INDEX][CARD_A_INDEX] is None) and (self.cacheDOM[CACHE_INDEX][CARD_B_INDEX] is None):
            error_dialog(self.gui, PLUGIN_NAME, _('The SONY reader does not have any books loaded.'), show=True)
            self.cacheDOM = None
    
    def updateTimestamp(self, bookList, latestDateTimeStamp):
        import os, time, copy
        from calibre.devices.prs505.sony_cache import strftime
        
        updatedDOMs = [None, None, None] # [main, card a, card b]
        
        books = copy.copy(bookList)
        books.reverse()
        baseTime = latestDateTimeStamp['date']
        #DEBUG
        print '********updateTimeStamp********'
        for book in books:
            print '*** shortBookList.books[' + str(book['id']) + ']: tz=' + str(book['tz']) + '; date=' + str(book['date']) + ' (' + str(book['dateSorter']) + '); title=' + str(book['title'])
        print '********updateTimeStamp********'
        #DEBUG
        
        for index, book in enumerate(books):
            #DEBUG
            print '********updateTimeStamp********'
            print '*** id=' + str(book['id'] + '; date=' + str(book['date']) + '; tz=' + str(book['tz']))
            #DEBUG
            path = book['path']
            modTime = baseTime + ((index + 1) * 2) # With one second intervals, I sometimes see duplicate times
            os.utime(path, (modTime, modTime))
            
            mtime = os.path.getmtime(path)
            book['date'] = strftime(mtime, zone=time.gmtime)
            book['tz'] = latestDateTimeStamp['tz']
            #DEBUG
            print '*** id=' + str(book['id'] + '; *date*=' + str(book['date']) + '; tz=' + str(book['tz']))
            print '********updateTimeStamp********'
            #DEBUG
            
            deviceIndex = DEVICE_KEY_INDICES[book['id'].split(':')[0]]
            cacheDOM = self.cacheDOM[CACHE_INDEX][deviceIndex]
            updatedDOMs[deviceIndex] = cacheDOM
            
            self.updateCacheTimestamp(book)
        
        self.writeUpdatedDOMs(updatedDOMs, CACHE_INDEX)
    
    def updateOpenedState(self, bookList, openedBookIds, unopenedBookIds):
        updatedDOMs = [None, None, None] # [main, card a, card b]
        
        if openedBookIds is not None:
            for id in openedBookIds:
                self.updateBookOpenState(updatedDOMs, id, bookList, CACHE_EXT_INDEX, True)
            
        if unopenedBookIds is not None:
            for id in unopenedBookIds:
                self.updateBookOpenState(updatedDOMs, id, bookList, CACHE_EXT_INDEX, False)
        
        self.writeUpdatedDOMs(updatedDOMs, CACHE_EXT_INDEX)
    
    def updateBookOpenState(self, updatedDOMs, id, bookList, cacheIndex, setOpen):
        book = bookList.books[id]
        oldOpenedState = book['DOMExtRecord'].get('opened')
        
        if (setOpen and not oldOpenedState) or (oldOpenedState and not setOpen):
            self.updateCacheOpenedState(book, setOpen)
        
            deviceIndex = DEVICE_KEY_INDICES[id.split(':')[0]]
            cacheDOM = self.cacheDOM[cacheIndex][deviceIndex]
            updatedDOMs[deviceIndex] = cacheDOM
    
    def writeUpdatedDOMs(self, updatedDOMs, cacheIndex):
        #DEBUG
        print '********writeUpdatedDOMs********'
        #DEBUG
        for index, updatedDOM in enumerate(updatedDOMs):
            if updatedDOM is not None:
                self.writeCache(cacheIndex, index, updatedDOM)
            
    def loadCache(self, cacheIndex, deviceIndex):
        import os
        from lxml import etree
        from calibre.constants import DEBUG
        from calibre.ebooks.chardet import xml_to_unicode
        from calibre.gui2 import error_dialog
        
        cache = {}
        path = os.path.join(self.devInfo.rootPath[deviceIndex], SONY_CACHE_FILE[cacheIndex][deviceIndex])
        
        if not os.path.exists(path):
            message = _('The SONY XML cache[{0}] ({1}) is missing. Any books on that drive will not be detected.')
            error_dialog(self.gui, PLUGIN_NAME, message.format(str(cacheIndex), READER_DRIVE_LABEL[deviceIndex]), show=True)
            cache = None

        if cache is not None:
            cache['mtime'] = os.path.getmtime(path)
            raw = None
            with open(path, 'rb') as f:
                raw = f.read()
            
            parser = etree.XMLParser(recover=True)
            roots = etree.fromstring(xml_to_unicode(raw, strip_encoding_pats=True, 
                                         assume_utf8=True,
                                         verbose=DEBUG)[0],
                                     parser=parser)
                                 
            if roots is None:
                cache = None
            else:
                cache['roots'] = roots
        
        if cache is not None:
            cache['books'] = roots.xpath('//*[local-name()="text"]')
            
            if not cache['books']:
                cache = None
                
        if cache is not None and cacheIndex is CACHE_INDEX:
            cache['collections'] = roots.xpath('//*[local-name()="playlist"]')
               
        return cache
    
    def updateCacheTimestamp(self, book):
        DOMBook = book['DOMRecord']
        DOMBook.set('date', book['date'])
        
        if book['tz'] is None:
            if 'tz' in DOMBook.attrib.keys():
                del DOMBook.attrib['tz']
        else:
            DOMBook.set('tz', book['tz'])
    
    def updateCacheOpenedState(self, book, setOpen):
        DOMExtBook = book['DOMExtRecord']
        
        if setOpen:
            DOMExtBook.set('opened', 'true')
        else:
            del DOMExtBook.attrib['opened']
    
    def writeCache(self, cacheIndex, deviceIndex, cacheDOM):
        import os
        from lxml import etree
        from calibre.gui2 import error_dialog
        
        path = os.path.join(self.devInfo.rootPath[deviceIndex], SONY_CACHE_FILE[cacheIndex][deviceIndex])
        #DEBUG
        print '********writeCache********'
        print '*** path=' + str(path)
        #DEBUG
        raw = etree.tostring(cacheDOM['roots'], encoding='UTF-8', xml_declaration=True)
        raw = raw.replace("<?xml version='1.0' encoding='UTF-8'?>", '<?xml version="1.0" encoding="UTF-8"?>')
        
        # Check that cache is still consistent
        if os.path.getmtime(path) != cacheDOM['mtime']:
            message = _('The SONY XML cache ({0}) has been updated outside of this plugin. To avoid corruption, the final write will be bypassed. Please invoke the action again.')
            error_dialog(self.gui, PLUGIN_NAME, message.format(READER_DRIVE_LABEL[deviceIndex]), show=True)
        else:
            with open(path, 'wb') as f:
                f.write(raw)
        #DEBUG
        print '********writeCache********'
        #DEBUG

class BookList():
    def __init__(self, devInfo):
        self.devInfo = devInfo
        self.initState()
    
    def initState(self):
        self.books = {}
        self.collections = {}
        self.pathIdMap = {}
        self.latestDateTimeStamp = {'date': None}
        
    def build(self, cacheDOM):
        self.initState()
        
        if cacheDOM[CACHE_INDEX][MAIN_INDEX] is not None:
            self.buildBookList(cacheDOM[CACHE_INDEX][MAIN_INDEX], cacheDOM[CACHE_EXT_INDEX][MAIN_INDEX], MAIN_INDEX)
            
        if cacheDOM[CACHE_INDEX][CARD_A_INDEX] is not None:
            self.buildBookList(cacheDOM[CACHE_INDEX][CARD_A_INDEX], cacheDOM[CACHE_EXT_INDEX][CARD_A_INDEX], CARD_A_INDEX)
            
        if cacheDOM[CACHE_INDEX][CARD_B_INDEX] is not None:
            self.buildBookList(cacheDOM[CACHE_INDEX][CARD_B_INDEX], cacheDOM[CACHE_EXT_INDEX][CARD_B_INDEX], CARD_B_INDEX)
    
    def buildBookList(self, cacheDOM, cacheExtDOM, deviceIndex):
        import os, calendar
        from calibre.devices.prs505.sony_cache import strptime
        
        for book in cacheDOM['books']:
            id = DEVICE_KEY_PREFIXES[deviceIndex] + str(book.get('id'))
            self.books[id] = {}
            self.books[id]['DOMRecord'] = book
            self.books[id]['id'] = id
            self.books[id]['author'] = book.get('author')
            self.books[id]['date'] = book.get('date')
            self.books[id]['dateSorter'] = calendar.timegm(strptime(self.books[id]['date']))
            self.books[id]['tz'] = book.get('tz')
            self.books[id]['title'] = book.get('title')
            self.books[id]['titleSorter'] = book.get('titleSorter')
            
            path = os.path.normpath(os.path.join(self.devInfo.rootPath[deviceIndex], book.get('path')))
            self.books[id]['path'] = path
            self.pathIdMap[path] = id
            
            if self.latestDateTimeStamp['date'] is None or self.books[id]['dateSorter'] > self.latestDateTimeStamp['date']:
                self.latestDateTimeStamp['date'] = self.books[id]['dateSorter']
                self.latestDateTimeStamp['tz'] = self.books[id]['tz']
                #DEBUG
                self.latestDateTimeStamp['dateString'] = self.books[id]['date']
                #DEBUG
        
        if cacheExtDOM is not None:
            for book in cacheExtDOM['books']:
                path = os.path.normpath(os.path.join(self.devInfo.rootPath[deviceIndex], book.get('path')))
                id = self.pathIdMap[path]
                self.books[id]['DOMExtRecord'] = book
                self.books[id]['opened'] = book.get('opened')
        
        names = self.collections.keys()
        for collection in cacheDOM['collections']:
            title = collection.get('title')
            
            if title not in names:
                self.collections[title] = []
            
            for item in collection.xpath('*[local-name()="item"]'):
                self.collections[title].append(DEVICE_KEY_PREFIXES[deviceIndex] + str(item.get('id')))

class LibraryBookList():
    def __init__(self, gui):
        self.gui = gui
        self.initState()
    
    def initState(self):
        self.books = {}
        
    def build(self, query):
        from calibre.gui2 import warning_dialog, info_dialog
        
        self.initState()
        ids = []
        
        try:
            ids = self.gui.current_db.search(query, True)
        except:
            message = _('The library query "{0}" failed. Check that the query expression is valid.')
            warning_dialog(self.gui, PLUGIN_NAME, message.format(query), show=True)
            self.books = None
            
        if self.books is not None:
            for id in ids:
                self.books[id] = {'id': id}
                self.books[id]['title'] = self.gui.current_db.title(id, True)
                authors = self.gui.current_db.authors(id, True).replace(',', ' & ').replace('|', ',')
                self.books[id]['authors'] = authors
                self.books[id]['author_sort'] = self.gui.current_db.author_sort(id, True)
    
    def matchBookListToLibrary(self, bookList):
        # First, match by 'title' and 'authors'
        matchedBooks = {}
        unmatchedBooks = {}
        ids = self.sortBooks(bookList, 'author')
        libraryIds = self.sortBooks(self.books, 'authors')
        self.match(ids, libraryIds, 'authors', bookList, matchedBooks, unmatchedBooks)
        
        # Next, match by 'title' and 'author_sort'
        ids = self.sortBooks(unmatchedBooks, 'author')
        unmatchedBooks = {}
        libraryIds = self.sortBooks(self.books, 'author_sort')
        self.match(ids, libraryIds, 'author_sort', bookList, matchedBooks, unmatchedBooks)
        
        return {'matched': matchedBooks.keys(), 'unmatched': unmatchedBooks.keys()}
    
    def match(self, ids, libraryIds, sortKey, bookList, matchedBooks, unmatchedBooks):
        pos = 0
        
        for libraryId in libraryIds:
            libraryBook = self.books[libraryId]
            
            while pos < len(ids):
                id = ids[pos]
                book = bookList[id]
                cmp1 = cmp(book['author'], libraryBook[sortKey])
               
                if cmp1 < 0:
                    unmatchedBooks[id] = book
                elif cmp1 == 0:
                    cmp2 = cmp(book['title'], libraryBook['title'])
                  
                    if cmp2 < 0:
                        unmatchedBooks[id] = book
                    elif cmp2 == 0:
                        matchedBooks[id] = book
                    else:
                        break
                else:
                    break
               
                pos = pos + 1
        
        for index in range(pos, len(ids)):
            id = ids[index]
            unmatchedBooks[id] = bookList[id]
            
    def sortBooks(self, booksToSort, sortKey):
        keys = booksToSort.keys()
        keys.sort(key=lambda key: booksToSort[key]['title'])
        keys.sort(key=lambda key: booksToSort[key][sortKey])
        
        return keys
   
class BookSelection():
    def __init__(self, gui):
        self.gui = gui
        self.initState()
    
    def initState(self):
        self.books = []
    
    def getSelection(self, pathIdMap):
        import os
        from calibre.gui2 import warning_dialog
        
        self.initState()
        
        if self.gui.current_view() is self.gui.library_view:
            warning_dialog(self.gui, PLUGIN_NAME, _('You must select books from the device book lists.'), show=True)
            self.books = None
            
        if self.books is not None:
            rows = self.gui.current_view().selectionModel().selectedRows();
        
            if len(rows) is 0:
                warning_dialog(self.gui, PLUGIN_NAME, _('No books are selected.'), show=True)
                self.books = None
           
        if self.books is not None:
            paths = self.gui.current_view().model().paths(rows)
        
            for path in paths:
               self.books.append(pathIdMap[os.path.normpath(path)])

class ShortBookList():
    def __init__(self, profile, bookList, bookSelection, libraryBookList):
        self.profile = profile
        self.bookList = bookList
        self.bookSelection = bookSelection
        self.libraryBookList = libraryBookList
        self.initState()
    
    def initState(self):
        self.books = []
    
    def extract(self):
        self.initState()
        
        collectionNames = self.bookList.collections.keys()
        idSets = []
        
        if len(self.bookSelection.books) is not 0:
            idSets.append(self.bookSelection.books)
        elif CONFIG_BOOKS_QUERY_KEY in self.profile[CONFIG_BOOKS_KEY].keys():
            matchedIds = self.libraryBookList.matchBookListToLibrary(self.bookList.books)['matched']
            idSets.append(matchedIds)
        else:
            for collectionName in self.profile[CONFIG_BOOKS_KEY][CONFIG_BOOKS_COLLECTIONS_KEY]:
                if collectionName in collectionNames:
                    idSets.append(self.bookList.collections[collectionName])
                
        bookSets = []
        for idSet in idSets:
            bookSet = []
            for bookId in idSet:
                bookSet.append(self.bookList.books[bookId])
            
            if len(bookSet) is not 0:
                bookSets.append(bookSet)
        
        if len(bookSets) is not 0:
            self.handleBookSets(bookSets)
    
    def handleBookSets(self, bookSets):
        maxRequired = MAX_BOOK_LIST_LENGTH
            
        for bookSet in bookSets:
            bookSet.sort(key=lambda book: book['dateSorter'])
            self.books.extend(self.chooseBooks(bookSet, maxRequired))
            maxRequired = MAX_BOOK_LIST_LENGTH - len(self.books)
            
            if len(self.books) is MAX_BOOK_LIST_LENGTH:
                break
        
        self.sortBooks()
    
    def chooseBooks(self, bookSet, maxRequired):
        import random
        
        strategy = self.profile[CONFIG_STRATEGY_KEY]
        
        if CONFIG_STRATEGY_VALUE_ONE in strategy:
            maxLen = 1
        else:
            maxLen = maxRequired
        
        books = []
            
        # Select newest first
        if CONFIG_STRATEGY_VALUE_NEWEST in strategy:
            for index, book in enumerate(bookSet):
                book = bookSet[len(bookSet) - index - 1]
                
                if (book not in self.books):
                    books.append(book)
                    
                    if len(books) is maxLen:
                        break
        
        # Select oldest first
        elif CONFIG_STRATEGY_VALUE_OLDEST in strategy:
            for index, book in enumerate(bookSet):
                book = bookSet[index]
                
                if book not in self.books:
                    books.append(book)
                    
                    if len(books) is maxLen:
                        break
        
        # Select randomly
        else:
            while len(books) is not maxLen:
                book = bookSet[random.randint(0, len(bookSet) - 1)]
                
                if (book in books) or (book in self.books):
                    bookSet.remove(book)
                else:
                    books.append(book)
                
                if len(bookSet) is 0:
                    break
        
        return books
    
    def sortBooks(self):
        order = self.profile[CONFIG_ORDER_KEY]
        
        if order == CONFIG_ORDER_VALUE_DATE:
            self.books.sort(key=lambda book: book['dateSorter'], reverse=True)
        elif order == CONFIG_ORDER_VALUE_AUTHOR:
            self.books.sort(key=lambda book: book['author'], reverse=False)
        else:
            self.books.sort(key=lambda book: book['titleSorter'], reverse=False)

class SplitBookList():
    def __init__(self, profile, bookList, libraryBookList):
        self.profile = profile
        self.bookList = bookList
        self.libraryBookList = libraryBookList
        self.initState()
    
    def initState(self):
        self.openedIds = []
        self.unopenedIds = []
    
    def split(self):
        self.initState()
        
        collectionNames = self.bookList.collections.keys()
        
        if CONFIG_BOOKS_QUERY_KEY in self.profile[CONFIG_BOOKS_KEY].keys():
            matchResults = self.libraryBookList.matchBookListToLibrary(self.bookList.books)
            self.openedIds = matchResults['unmatched']
            self.unopenedIds = matchResults['matched']
        else:
            collectionIds = []
            
            for collectionName in self.profile[CONFIG_BOOKS_KEY][CONFIG_BOOKS_COLLECTIONS_KEY]:
                if collectionName in collectionNames:
                    collectionIds.extend(self.bookList.collections[collectionName])
            
            self.unopenedIds = list(set(collectionIds))
            self.openedIds = list(set(self.bookList.books.keys()).difference(set(self.unopenedIds)))
                