#!/usr/bin/env python
# vim:fileencoding=utf-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011, Karl Weckworth <kweckwor@gmail.com>'
__docformat__ = 'restructuredtext en'

def createMenu(interfaceAction):
    from PyQt4.Qt import QMenu
    
    return QMenu(interfaceAction.gui)

def createAction(interfaceAction, parentMenu, actionTitle, actionTooltip, actionIcon=None, actionTrigger=None):
    action = interfaceAction.create_action(spec=(actionTitle, None, actionTooltip, None), attr=actionTitle)
    
    if actionIcon is not None:
        action.setIcon(actionIcon)
    
    if actionTrigger is not None:
        action.triggered.connect(actionTrigger)
        
    parentMenu.addAction(action)
    
    return action

def createIcon(iconName=None, iconResource=None):
    from PyQt4.Qt import QIcon, QPixmap
    
    icon = None
    
    if iconName is not None:
        icon = QIcon(I(iconName))
    
    elif iconResource is not None:
        pixmap = QPixmap()
        pixmap.loadFromData(iconResource)
        icon = QIcon(pixmap)
    
    return icon
    
def getUniqueName(name, existingNames):
    namePrefix = name
    tryName = name
    index = 1
        
    while tryName in existingNames:
        tryName = namePrefix + ':' + str(index)
        index = index + 1
        
    return tryName
 
CONFIG_LABEL_BOOKS_LIST_SELECT_TITLE = _('Set Books List')
CONFIG_LABEL_UNREAD_LIST_SELECT_TITLE = _('Set Unread List')
CONFIG_LABEL_BOOKS_LIST_TAB = _('Books List')
CONFIG_LABEL_UNREAD_LIST_TAB = _('Unread List')
CONFIG_LABEL_BOOKS_LIST = _('The Books List profile configuration determines how books for the Books List are chosen and how thay will appear on your reader.')
CONFIG_LABEL_UNREAD_LIST = _('The Unread List profile configuration determines how books for the Unread List are chosen.')
CONFIG_LABEL_CHOOSE_PROFILE = _('Choose Profile:')
CONFIG_LABEL_PROFILE = _('Profile:')
CONFIG_LABEL_PROFILES = _('Profiles:')
CONFIG_LABEL_DETAILS = _('Details:')
CONFIG_LABEL_SELECTION = _('Choose books from Device/Card A/Card B book list selections.')
CONFIG_LABEL_COLLECTIONS = _('Choose books from device collections: ')
CONFIG_LABEL_QUERY = _('Choose books using library query: ')
CONFIG_LABEL_CHOOSE_ONE = _('Choose one book from each source.')
CONFIG_LABEL_CHOOSE_NEWEST = _('Choose books by newest first.')
CONFIG_LABEL_CHOOSE_OLDEST = _('Choose books by oldest first.')
CONFIG_LABEL_CHOOSE_RANDOM = _('Choose books randomly.')
CONFIG_LABEL_ORDER_DATE = _('Order books list by date.')
CONFIG_LABEL_ORDER_AUTHOR = _('Order books list by author.')
CONFIG_LABEL_ORDER_TITLE = _('Order books list by title.')
CONFIG_LABEL_CHOOSE_GROUP = _('Choose Books From:')
CONFIG_LABEL_SELECTION_OPTION = _('Device/Card A/Card B book list selections')
CONFIG_LABEL_QUERY_OPTION = _('Library Query:')
CONFIG_LABEL_COLLECTIONS_OPTION = _('Device Collections:')
CONFIG_LABEL_STRATEGY_GROUP = _('Choose Books By:')
CONFIG_LABEL_ONE_OPTION = _('One book from each source')
CONFIG_LABEL_NEWEST_OPTION = _('Newest first')
CONFIG_LABEL_OLDEST_OPTION = _('Oldest first')
CONFIG_LABEL_RANDOM_OPTION = _('Randomly')
CONFIG_LABEL_ORDER_GROUP = _('Order Books List By:')
CONFIG_LABEL_DATE_OPTION = _('Date')
CONFIG_LABEL_AUTHOR_OPTION = _('Author')
CONFIG_LABEL_TITLE_OPTION = _('Title')
CONFIG_LABEL_ADD = _('Add')
CONFIG_LABEL_DELETE = _('Delete')

DEFAULT_PROFILE_NAME = _('New Profile')
DEFAULT_COLLECTION_NAME = _('New Collection')

from calibre_plugins.ManageSonyX50BookList.action import CONFIG_BOOKS_LIST_KEY
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_UNREAD_LIST_KEY
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_PROFILES_KEY
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_NAME_KEY
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_BOOKS_KEY
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_BOOKS_VALUE_SELECTION
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_BOOKS_COLLECTIONS_KEY
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_BOOKS_QUERY_KEY
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_STRATEGY_KEY
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_STRATEGY_VALUE_ONE
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_STRATEGY_VALUE_NEWEST
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_STRATEGY_VALUE_OLDEST
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_STRATEGY_VALUE_RANDOM
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_ORDER_KEY
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_ORDER_VALUE_DATE
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_ORDER_VALUE_AUTHOR
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_ORDER_VALUE_TITLE
from calibre_plugins.ManageSonyX50BookList.action import CONFIG_POS_KEY

from calibre_plugins.ManageSonyX50BookList.action import DEFAULT_CONFIG_BOOKS_LIST_QUERY
from calibre_plugins.ManageSonyX50BookList.action import DEFAULT_CONFIG_UNREAD_LIST_QUERY

from PyQt4.Qt import QDialog

class ManageBookListView(QDialog):
    def __init__(self, parent):
        from PyQt4 import QtCore
        from PyQt4.Qt import QGridLayout, QLabel, QDialogButtonBox
        
        QDialog.__init__(self, parent)
        
        self.setWindowTitle(CONFIG_LABEL_BOOKS_LIST_SELECT_TITLE)
        
        self.dialogLayout = QGridLayout(self)
        
        self.setBookListText = QLabel(CONFIG_LABEL_BOOKS_LIST)
        self.setBookListText.setWordWrap(True)
        self.dialogLayout.addWidget(self.setBookListText, 0, 0, 1, 2)

        self.profileLabel = QLabel()
        self.dialogLayout.addWidget(self.profileLabel, 1, 0, 1, 1, QtCore.Qt.AlignLeft)
                
        self.infoLabel = QLabel('<b>' + CONFIG_LABEL_DETAILS + '</b>')
        self.dialogLayout.addWidget(self.infoLabel, 2, 0, 1, 2, QtCore.Qt.AlignLeft | QtCore.Qt.AlignTop)

        self.profileDetails = QLabel()
        self.profileDetails.setWordWrap(True)
        self.dialogLayout.addWidget(self.profileDetails, 3, 0, 1, 2)
        
        buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        buttonBox.accepted.connect(self.accept)
        buttonBox.rejected.connect(self.reject)
        self.dialogLayout.addWidget(buttonBox, 4, 0, 1, 2)
        
        self.dialogLayout.setColumnStretch(1, 1)
        self.dialogLayout.setColumnMinimumWidth(1, 200)
    
    def getProfileName(self):
        from PyQt4.Qt import QComboBox
        
        if isinstance(self.profileValue, QComboBox):
            profileName = unicode(self.profileValue.itemText(self.profileValue.currentIndex()), 'utf-8')
        else:
            profileName = unicode(self.profileValue.text(), 'utf-8')
            
        return profileName
    
class ManageBookListController():
    def __init__(self, model, view):
        from PyQt4 import QtCore
        from PyQt4.Qt import QComboBox, QLabel
        
        self.model = model
        self.view = view
        
        profileNames = self.model.keys()
        
        if len(profileNames) > 1:
            profileNames.sort(key=lambda profileName: self.model[profileName][CONFIG_POS_KEY])
            
            self.view.profileLabel.setText('<b>' + CONFIG_LABEL_CHOOSE_PROFILE + '</b>')
            self.view.profileValue = QComboBox()
            
            for profileName in profileNames:
                self.view.profileValue.addItem(profileName)
        else:
            self.view.profileLabel.setText('<b>' + CONFIG_LABEL_PROFILE + '</b>')
            self.view.profileValue = QLabel(profileNames[0])
            
        self.view.dialogLayout.addWidget(self.view.profileValue, 1, 1, 1, 1, QtCore.Qt.AlignLeft)
            
        self.view.connect(self.view.profileValue, QtCore.SIGNAL('activated(QString)'), self.onProfileActivated)
        self.setProfile(profileNames[0])
    
    def onProfileActivated(self, text):
        self.setProfile(unicode(text, 'utf-8'))
    
    def setProfile(self, profileName):
        profile = self.model[profileName]
        booksValue = profile[CONFIG_BOOKS_KEY]
        strategyValue = profile[CONFIG_STRATEGY_KEY]
        orderValue = profile[CONFIG_ORDER_KEY]
        
        details = '<ul>'
        
        if CONFIG_BOOKS_VALUE_SELECTION in booksValue:
            details = details + '<li>' + CONFIG_LABEL_SELECTION + '</li>'
        elif CONFIG_BOOKS_QUERY_KEY in booksValue.keys():
            details = details + '<li>' + CONFIG_LABEL_QUERY
            details = details + '"' + booksValue[CONFIG_BOOKS_QUERY_KEY] + '"'
            details = details + '</li>'
        else:
            details = details + '<li>' + CONFIG_LABEL_COLLECTIONS
            for index, value in enumerate(booksValue[CONFIG_BOOKS_COLLECTIONS_KEY]):
                if index is not 0:
                    details = details + ', '
                details = details + '"' + value + '"'
            details = details + '</li>'
            
        if CONFIG_STRATEGY_VALUE_ONE in strategyValue:
            details = details + '<li>' + CONFIG_LABEL_CHOOSE_ONE + '</li>'
            
        if CONFIG_STRATEGY_VALUE_NEWEST in strategyValue:
            details = details + '<li>' + CONFIG_LABEL_CHOOSE_NEWEST + '</li>'
        elif CONFIG_STRATEGY_VALUE_OLDEST in strategyValue:
            details = details + '<li>' + CONFIG_LABEL_CHOOSE_OLDEST + '</li>'
        else:
            details = details + '<li>' + CONFIG_LABEL_CHOOSE_RANDOM + '</li>'
            
        if CONFIG_ORDER_VALUE_DATE in orderValue:
            details = details + '<li>' + CONFIG_LABEL_ORDER_DATE + '</li>'
        elif CONFIG_ORDER_VALUE_AUTHOR in orderValue:
            details = details + '<li>' + CONFIG_LABEL_ORDER_AUTHOR + '</li>'
        else:
            details = details + '<li>' + CONFIG_LABEL_ORDER_TITLE + '</li>'
            
        details = details + '</ul>'
        
        self.view.profileDetails.setText(details)
        self.view.resize(self.view.sizeHint())
 
class ManageUnreadListView(QDialog):
    def __init__(self, parent):
        from PyQt4 import QtCore
        from PyQt4.Qt import QGridLayout, QLabel, QDialogButtonBox
        
        QDialog.__init__(self, parent)
        
        self.setWindowTitle(CONFIG_LABEL_UNREAD_LIST_SELECT_TITLE)
        
        self.dialogLayout = QGridLayout(self)
        
        self.setUnreadListText = QLabel(CONFIG_LABEL_UNREAD_LIST)
        self.setUnreadListText.setWordWrap(True)
        self.dialogLayout.addWidget(self.setUnreadListText, 0, 0, 1, 2)

        self.profileLabel = QLabel()
        self.dialogLayout.addWidget(self.profileLabel, 1, 0, 1, 1, QtCore.Qt.AlignLeft)
                
        self.infoLabel = QLabel('<b>' + CONFIG_LABEL_DETAILS + '</b>')
        self.dialogLayout.addWidget(self.infoLabel, 2, 0, 1, 2, QtCore.Qt.AlignLeft | QtCore.Qt.AlignTop)

        self.profileDetails = QLabel()
        self.profileDetails.setWordWrap(True)
        self.dialogLayout.addWidget(self.profileDetails, 3, 0, 1, 2)
        
        buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        buttonBox.accepted.connect(self.accept)
        buttonBox.rejected.connect(self.reject)
        self.dialogLayout.addWidget(buttonBox, 4, 0, 1, 2)
        
        self.dialogLayout.setColumnStretch(1, 1)
        self.dialogLayout.setColumnMinimumWidth(1, 200)
    
    def getProfileName(self):
        from PyQt4.Qt import QComboBox
        
        if isinstance(self.profileValue, QComboBox):
            profileName = unicode(self.profileValue.itemText(self.profileValue.currentIndex()), 'utf-8')
        else:
            profileName = unicode(self.profileValue.text(), 'utf-8')
            
        return profileName
    
class ManageUnreadListController():
    def __init__(self, model, view):
        from PyQt4 import QtCore
        from PyQt4.Qt import QComboBox, QLabel
        
        self.model = model
        self.view = view
        
        profileNames = self.model.keys()
        
        if len(profileNames) > 1:
            profileNames.sort(key=lambda profileName: self.model[profileName][CONFIG_POS_KEY])
            
            self.view.profileLabel.setText('<b>' + CONFIG_LABEL_CHOOSE_PROFILE + '</b>')
            self.view.profileValue = QComboBox()
            
            for profileName in profileNames:
                self.view.profileValue.addItem(profileName)
        else:
            self.view.profileLabel.setText('<b>' + CONFIG_LABEL_PROFILE + '</b>')
            self.view.profileValue = QLabel(profileNames[0])
            
        self.view.dialogLayout.addWidget(self.view.profileValue, 1, 1, 1, 1, QtCore.Qt.AlignLeft)
            
        self.view.connect(self.view.profileValue, QtCore.SIGNAL('activated(QString)'), self.onProfileActivated)
        self.setProfile(profileNames[0])
    
    def onProfileActivated(self, text):
        self.setProfile(unicode(text, 'utf-8'))
    
    def setProfile(self, profileName):
        profile = self.model[profileName]
        booksValue = profile[CONFIG_BOOKS_KEY]
        
        details = '<ul>'
        
        if CONFIG_BOOKS_QUERY_KEY in booksValue.keys():
            details = details + '<li>' + CONFIG_LABEL_QUERY
            details = details + '"' + booksValue[CONFIG_BOOKS_QUERY_KEY] + '"'
            details = details + '</li>'
        else:
            details = details + '<li>' + CONFIG_LABEL_COLLECTIONS
            for index, value in enumerate(booksValue[CONFIG_BOOKS_COLLECTIONS_KEY]):
                if index is not 0:
                    details = details + ', '
                details = details + '"' + value + '"'
            details = details + '</li>'
            
        details = details + '</ul>'
        
        self.view.profileDetails.setText(details)
        self.view.resize(self.view.sizeHint())
 
class ManageBookListConfigView(QDialog):
    def __init__(self):
        from PyQt4 import QtCore
        from PyQt4.Qt import QGridLayout, QTabWidget
        
        QDialog.__init__(self)
        
        self.dialogLayout = QGridLayout(self)
        
        self.tabWidget = QTabWidget(self)
        
        self.booksListConfig = ManageBooksListConfigView()
        self.unreadListConfig = ManageUnreadListConfigView()
        
        self.tabWidget.addTab(self.booksListConfig, CONFIG_LABEL_BOOKS_LIST_TAB)
        self.tabWidget.addTab(self.unreadListConfig, CONFIG_LABEL_UNREAD_LIST_TAB)
        
        self.dialogLayout.addWidget(self.tabWidget, 0, 0, 1, 1)
        self.controller = None
    
    def setController(self, controller):
        self.controller = controller
             
class ManageBookListConfigController():
    def __init__(self, model, view, modelNormalizer):
        self.model = model
        self.view = view
        
        self.booksListConfigController = ManageBooksListConfigController(self.model[CONFIG_BOOKS_LIST_KEY], self.view.booksListConfig, modelNormalizer.normalizeBooksList)
        self.unreadListConfigController = ManageUnreadListConfigController(self.model[CONFIG_UNREAD_LIST_KEY], self.view.unreadListConfig, modelNormalizer.normalizeUnreadList)
        
        self.view.tabWidget.setCurrentIndex(0)
        
    def getBooksListConfig(self):
        return self.booksListConfigController.getBooksListConfig()
    
    def getUnreadListConfig(self):
        return self.unreadListConfigController.getUnreadListConfig()

from PyQt4.Qt import QWidget

class ManageBooksListConfigView(QWidget):
    def __init__(self):
        from PyQt4 import QtCore
        from PyQt4.Qt import QLabel, QGridLayout, QGroupBox, QCheckBox, QRadioButton, QVBoxLayout
        
        QWidget.__init__(self)
        
        self.booksListLayout = QGridLayout(self)
        
        self.setBookListText = QLabel(CONFIG_LABEL_BOOKS_LIST)
        self.setBookListText.setWordWrap(True)
        self.booksListLayout.addWidget(self.setBookListText, 0, 0, 1, 3)
        
        self.profileGroup = ProfileListView()
        self.booksListLayout.addWidget(self.profileGroup, 1, 0, 2, 1)
        
        self.chooseGroup = BookChoiceCriteriaView()
        self.booksListLayout.addWidget(self.chooseGroup, 1, 1, 2, 1)
        
        self.strategyGroup = QGroupBox(CONFIG_LABEL_STRATEGY_GROUP)
        self.oneOption = QCheckBox(CONFIG_LABEL_ONE_OPTION)
        self.newestOption = QRadioButton(CONFIG_LABEL_NEWEST_OPTION)
        self.oldestOption = QRadioButton(CONFIG_LABEL_OLDEST_OPTION)
        self.randomOption = QRadioButton(CONFIG_LABEL_RANDOM_OPTION)
        self.strategyLayout = QVBoxLayout(self.strategyGroup)
        self.strategyLayout.addWidget(self.oneOption)
        self.strategyLayout.addWidget(self.newestOption)
        self.strategyLayout.addWidget(self.oldestOption)
        self.strategyLayout.addWidget(self.randomOption, 1, QtCore.Qt.AlignTop)
        self.strategyGroup.setLayout(self.strategyLayout)
        self.booksListLayout.addWidget(self.strategyGroup, 1, 2, 1, 1)
        
        self.orderGroup = QGroupBox(CONFIG_LABEL_ORDER_GROUP)
        self.dateOption = QRadioButton(CONFIG_LABEL_DATE_OPTION)
        self.authorOption = QRadioButton(CONFIG_LABEL_AUTHOR_OPTION)
        self.titleOption = QRadioButton(CONFIG_LABEL_TITLE_OPTION)
        self.orderLayout = QVBoxLayout(self.orderGroup)
        self.orderLayout.addWidget(self.dateOption)
        self.orderLayout.addWidget(self.authorOption)
        self.orderLayout.addWidget(self.titleOption, 1, QtCore.Qt.AlignTop)
        self.orderGroup.setLayout(self.orderLayout)
        self.booksListLayout.addWidget(self.orderGroup, 2, 2, 1, 1)
        
        self.booksListLayout.setColumnStretch(0, 1)
        self.booksListLayout.setRowStretch(1, 1)
        self.booksListLayout.setRowStretch(2, 1)
        
class ManageBooksListConfigController():
    def __init__(self, model, view, normalizeFunc):
        from PyQt4 import QtCore
        
        self.model = model
        self.view = view
        
        profiles = self.model[CONFIG_PROFILES_KEY]
        self.profileListController = ProfileListController(profiles, view.profileGroup, normalizeFunc)
        self.bookChoiceCriteriaController = BookChoiceCriteriaController(profiles, view.chooseGroup, DEFAULT_CONFIG_BOOKS_LIST_QUERY)
        
        self.view.connect(self.view.oneOption, QtCore.SIGNAL('toggled(bool)'), self.onOneButtonToggled)
        self.view.connect(self.view.newestOption, QtCore.SIGNAL('toggled(bool)'), self.onNewestButtonToggled)
        self.view.connect(self.view.oldestOption, QtCore.SIGNAL('toggled(bool)'), self.onOldestButtonToggled)
        self.view.connect(self.view.randomOption, QtCore.SIGNAL('toggled(bool)'), self.onRandomButtonToggled)
        self.view.connect(self.view.dateOption, QtCore.SIGNAL('toggled(bool)'), self.onDateButtonToggled)
        self.view.connect(self.view.authorOption, QtCore.SIGNAL('toggled(bool)'), self.onAuthorButtonToggled)
        self.view.connect(self.view.titleOption, QtCore.SIGNAL('toggled(bool)'), self.onTitleButtonToggled)
        
        self.profileListController.currentProfileChanged.connect(self.setCurrentProfile)
        self.profileListController.currentProfileChanged.connect(self.bookChoiceCriteriaController.setCurrentProfile)
        
        self.profileListController.initialize()
        
    def getBooksListConfig(self):
        profiles = self.model[CONFIG_PROFILES_KEY]
        
        for index in range(self.view.profileGroup.profileList.count()):
            item = self.view.profileGroup.profileList.item(index)
            profile = profiles[unicode(item.text(), 'utf-8')]
            profile[CONFIG_POS_KEY] = index
        
        return self.model
    
    def getCurrentProfile(self):
        return self.profileListController.getCurrentProfile()
    
    def setCurrentProfile(self, oldProfileName, newProfileName):
        profiles = self.model[CONFIG_PROFILES_KEY]
        newProfile = profiles[unicode(newProfileName, 'utf-8')]
        strategyValue = newProfile[CONFIG_STRATEGY_KEY]
        orderValue = newProfile[CONFIG_ORDER_KEY]

        self.view.oneOption.setChecked(CONFIG_STRATEGY_VALUE_ONE in strategyValue)
        
        if CONFIG_STRATEGY_VALUE_NEWEST in strategyValue:
            self.view.newestOption.setChecked(True)
        elif CONFIG_STRATEGY_VALUE_OLDEST in strategyValue:
            self.view.oldestOption.setChecked(True)
        else:
            self.view.randomOption.setChecked(True)
        
        if CONFIG_ORDER_VALUE_DATE in orderValue:
            self.view.dateOption.setChecked(True)
        elif CONFIG_ORDER_VALUE_AUTHOR in orderValue:
            self.view.authorOption.setChecked(True)
        else:
            self.view.titleOption.setChecked(True)
    
    def onOneButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            if CONFIG_STRATEGY_VALUE_ONE not in profile[CONFIG_STRATEGY_KEY]:
                profile[CONFIG_STRATEGY_KEY].insert(0, CONFIG_STRATEGY_VALUE_ONE)
        else:
            if CONFIG_STRATEGY_VALUE_ONE in profile[CONFIG_STRATEGY_KEY]:
                profile[CONFIG_STRATEGY_KEY].remove(CONFIG_STRATEGY_VALUE_ONE)
    
    def onNewestButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            if CONFIG_STRATEGY_VALUE_ONE in profile[CONFIG_STRATEGY_KEY]:
                profile[CONFIG_STRATEGY_KEY] = [CONFIG_STRATEGY_VALUE_ONE, CONFIG_STRATEGY_VALUE_NEWEST]
            else:
                profile[CONFIG_STRATEGY_KEY] = [CONFIG_STRATEGY_VALUE_NEWEST]
    
    def onOldestButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            if CONFIG_STRATEGY_VALUE_ONE in profile[CONFIG_STRATEGY_KEY]:
                profile[CONFIG_STRATEGY_KEY] = [CONFIG_STRATEGY_VALUE_ONE, CONFIG_STRATEGY_VALUE_OLDEST]
            else:
                profile[CONFIG_STRATEGY_KEY] = [CONFIG_STRATEGY_VALUE_OLDEST]
    
    def onRandomButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            if CONFIG_STRATEGY_VALUE_ONE in profile[CONFIG_STRATEGY_KEY]:
                profile[CONFIG_STRATEGY_KEY] = [CONFIG_STRATEGY_VALUE_ONE, CONFIG_STRATEGY_VALUE_RANDOM]
            else:
                profile[CONFIG_STRATEGY_KEY] = [CONFIG_STRATEGY_VALUE_RANDOM]
    
    def onDateButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            profile[CONFIG_ORDER_KEY] = CONFIG_ORDER_VALUE_DATE
    
    def onAuthorButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            profile[CONFIG_ORDER_KEY] = CONFIG_ORDER_VALUE_AUTHOR
    
    def onTitleButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            profile[CONFIG_ORDER_KEY] = CONFIG_ORDER_VALUE_TITLE
        
class ManageUnreadListConfigView(QWidget):
    def __init__(self):
        from PyQt4 import QtCore
        from PyQt4.Qt import QLabel, QGridLayout
        
        QWidget.__init__(self)
        
        self.unreadListLayout = QGridLayout(self)
        
        self.setUnreadListText = QLabel(CONFIG_LABEL_UNREAD_LIST)
        self.setUnreadListText.setWordWrap(True)
        self.unreadListLayout.addWidget(self.setUnreadListText, 0, 0, 1, 2)
        
        self.profileGroup = ProfileListView()
        self.unreadListLayout.addWidget(self.profileGroup, 1, 0, 1, 1)
        
        self.chooseGroup = BookChoiceCriteriaView(False)
        self.unreadListLayout.addWidget(self.chooseGroup, 1, 1, 1, 1)
        
        self.unreadListLayout.setRowStretch(1, 1)
        
class ManageUnreadListConfigController():
    def __init__(self, model, view, normalizeFunc):
        self.model = model
        self.view = view
        
        profiles = self.model[CONFIG_PROFILES_KEY]
        self.profileListController = ProfileListController(profiles, view.profileGroup, normalizeFunc)
        self.bookChoiceCriteriaController = BookChoiceCriteriaController(profiles, view.chooseGroup, DEFAULT_CONFIG_UNREAD_LIST_QUERY)
        
        self.profileListController.currentProfileChanged.connect(self.bookChoiceCriteriaController.setCurrentProfile)
        
        self.profileListController.initialize()
        
    def getUnreadListConfig(self):
        profiles = self.model[CONFIG_PROFILES_KEY]
        
        for index in range(self.view.profileGroup.profileList.count()):
            item = self.view.profileGroup.profileList.item(index)
            profile = profiles[unicode(item.text(), 'utf-8')]
            profile[CONFIG_POS_KEY] = index
        
        return self.model
        
from PyQt4.Qt import QGroupBox

class BookChoiceCriteriaView(QGroupBox):
    def __init__(self, includeSelection=True):
        from PyQt4 import QtCore
        from PyQt4.Qt import QRadioButton, QLineEdit, QListWidget, QPushButton, QGridLayout, QIcon
        
        QGroupBox.__init__(self)
        
        self.setTitle(CONFIG_LABEL_CHOOSE_GROUP)
        
        if includeSelection:
            self.selectionOption = QRadioButton(CONFIG_LABEL_SELECTION_OPTION)
        else:
            self.selectionOption = None
            
        self.queryOption = QRadioButton(CONFIG_LABEL_QUERY_OPTION)
        self.queryInput = QLineEdit()
        
        self.collectionsOption = QRadioButton(CONFIG_LABEL_COLLECTIONS_OPTION)
        self.collectionsList = QListWidget()
        self.collectionAddButton = QPushButton(CONFIG_LABEL_ADD)
        self.collectionDeleteButton = QPushButton(CONFIG_LABEL_DELETE)
        self.collectionUpButton = QPushButton(QIcon(I('arrow-up.png')), '')
        self.collectionDownButton = QPushButton(QIcon(I('arrow-down.png')), '')
        
        self.chooseLayout = QGridLayout(self)
        
        if self.selectionOption is not None:
            self.chooseLayout.addWidget(self.selectionOption, 0, 0, 1, 2)
            
        self.chooseLayout.addWidget(self.queryOption, 1, 0, 1, 2)
        self.chooseLayout.addWidget(self.queryInput, 2, 0, 1, 2)
        self.chooseLayout.addWidget(self.collectionsOption, 3, 0, 1, 2)
        self.chooseLayout.addWidget(self.collectionsList, 4, 0, 2, 2)
        self.chooseLayout.addWidget(self.collectionUpButton, 4, 2, 1, 1, QtCore.Qt.AlignBottom)
        self.chooseLayout.addWidget(self.collectionDownButton, 5, 2, 1, 1, QtCore.Qt.AlignTop)
        self.chooseLayout.addWidget(self.collectionAddButton, 6, 0, 1, 1)
        self.chooseLayout.addWidget(self.collectionDeleteButton, 6, 1, 1, 1)
             
class BookChoiceCriteriaController():
    def __init__(self, model, view, defaultQuery):
        from PyQt4 import QtCore
        
        self.model = model
        self.view = view
        self.defaultQuery = defaultQuery
        self.currentProfile = None
        
        if self.view.selectionOption is not None:
            self.view.connect(self.view.selectionOption, QtCore.SIGNAL('toggled(bool)'), self.onSelectionButtonToggled)

        self.view.connect(self.view.collectionsList, QtCore.SIGNAL('currentItemChanged(QListWidgetItem *, QListWidgetItem *)'), self.onCollectionsCurrentItemChanged)
        self.view.connect(self.view.collectionsList, QtCore.SIGNAL('itemDoubleClicked(QListWidgetItem *)'), self.onCollectionsItemDoubleClicked)
        self.view.connect(self.view.collectionsList, QtCore.SIGNAL('itemChanged(QListWidgetItem *)'), self.onCollectionsItemChanged)
        self.view.connect(self.view.queryOption, QtCore.SIGNAL('toggled(bool)'), self.onQueryButtonToggled)
        self.view.connect(self.view.queryInput, QtCore.SIGNAL('editingFinished()'), self.onQueryEditingFinished)
        self.view.connect(self.view.collectionsOption, QtCore.SIGNAL('toggled(bool)'), self.onCollectionsButtonToggled)
        self.view.connect(self.view.collectionDeleteButton, QtCore.SIGNAL('clicked(bool)'), self.onCollectionDeleteButtonClicked)
        self.view.connect(self.view.collectionAddButton, QtCore.SIGNAL('clicked(bool)'), self.onCollectionAddButtonClicked)
        self.view.connect(self.view.collectionUpButton, QtCore.SIGNAL('clicked(bool)'), self.onCollectionUpButtonClicked)
        self.view.connect(self.view.collectionDownButton, QtCore.SIGNAL('clicked(bool)'), self.onCollectionDownButtonClicked)
              
    def getCurrentCollection(self):
        return unicode(self.view.collectionsList.item(self.view.collectionsList.currentRow()).text(), 'utf-8')
    
    def setCurrentProfile(self, oldProfileName, newProfileName):
        self.currentProfile = self.model[unicode(newProfileName, 'utf-8')]
        booksValue = self.currentProfile[CONFIG_BOOKS_KEY]

        if CONFIG_BOOKS_VALUE_SELECTION in booksValue:
            self.view.selectionOption.setChecked(True)
        elif CONFIG_BOOKS_QUERY_KEY in booksValue.keys():
            if self.view.queryOption.isChecked():
                self.onQueryButtonToggled(True)
            else:
                self.view.queryOption.setChecked(True)
        else:
            if self.view.collectionsOption.isChecked():
                self.onCollectionsButtonToggled(True)
            else:
                self.view.collectionsOption.setChecked(True)
            
    def onCollectionsCurrentItemChanged(self, newItem, previousItem):
        if newItem is not None:
            self.view.collectionDeleteButton.setEnabled(self.view.collectionsList.count() > 1)
            self.view.collectionUpButton.setEnabled(self.view.collectionsList.count() > 1 and self.view.collectionsList.currentRow() > 0)
            self.view.collectionDownButton.setEnabled(self.view.collectionsList.count() > 1 and self.view.collectionsList.currentRow() < self.view.collectionsList.count() - 1)
    
    def onCollectionsItemDoubleClicked(self, item):
        self.view.collectionsList.editItem(item)
    
    def onSelectionButtonToggled(self, on):
        if on:
            self.currentProfile[CONFIG_BOOKS_KEY] = CONFIG_BOOKS_VALUE_SELECTION

            self.view.queryInput.clear()
            self.view.queryInput.setDisabled(True)
            
            self.view.collectionAddButton.setDisabled(True)
            self.view.collectionDeleteButton.setDisabled(True)
            self.view.collectionUpButton.setDisabled(True)
            self.view.collectionDownButton.setDisabled(True)
            self.view.collectionsList.clear()
            self.view.collectionsList.setDisabled(True)
    
    def onQueryButtonToggled(self, on):
        if on:
            booksValue = self.currentProfile[CONFIG_BOOKS_KEY]

            if (CONFIG_BOOKS_VALUE_SELECTION in booksValue) or (CONFIG_BOOKS_COLLECTIONS_KEY in booksValue.keys()):
                self.currentProfile[CONFIG_BOOKS_KEY] = {CONFIG_BOOKS_QUERY_KEY: self.defaultQuery}
                booksValue = self.currentProfile[CONFIG_BOOKS_KEY]
            
            self.view.queryInput.setDisabled(False)
            self.view.queryInput.setText(booksValue[CONFIG_BOOKS_QUERY_KEY])
                
            self.view.collectionAddButton.setDisabled(True)
            self.view.collectionDeleteButton.setDisabled(True)
            self.view.collectionUpButton.setDisabled(True)
            self.view.collectionDownButton.setDisabled(True)
            self.view.collectionsList.clear()
            self.view.collectionsList.setDisabled(True)
                
    def onCollectionsButtonToggled(self, on):
        from PyQt4 import QtCore
        
        if on:
            autoEdit = False
            booksValue = self.currentProfile[CONFIG_BOOKS_KEY]
            
            if (CONFIG_BOOKS_VALUE_SELECTION in booksValue) or (CONFIG_BOOKS_QUERY_KEY in booksValue.keys()):
                self.currentProfile[CONFIG_BOOKS_KEY] = {CONFIG_BOOKS_COLLECTIONS_KEY: [DEFAULT_COLLECTION_NAME]}
                booksValue = self.currentProfile[CONFIG_BOOKS_KEY]
                autoEdit = True
            
            self.view.collectionDeleteButton.setDisabled(True)
            self.view.collectionUpButton.setDisabled(True)
            self.view.collectionDownButton.setDisabled(True)
            self.view.collectionsList.clear()
            self.view.collectionsList.setDisabled(False)
            
            for collection in booksValue[CONFIG_BOOKS_COLLECTIONS_KEY]:
                self.view.collectionsList.addItem(collection)
                item = self.view.collectionsList.item(self.view.collectionsList.count() - 1)
                item.setFlags(item.flags() | QtCore.Qt.ItemIsEditable)
                
            if autoEdit:
                self.view.collectionsList.editItem(self.view.collectionsList.item(0))
                
            self.view.collectionAddButton.setEnabled(True)
    
            self.view.queryInput.clear()
            self.view.queryInput.setDisabled(True)
    
    def onQueryEditingFinished(self):
        queryString = unicode(self.view.queryInput.text(), 'utf-8').strip()
        
        if len(queryString) is 0:
            queryString = self.defaultQuery
            self.view.queryInput.setText(queryString)
        
        self.currentProfile[CONFIG_BOOKS_KEY][CONFIG_BOOKS_QUERY_KEY] = queryString
            
    def onCollectionDeleteButtonClicked(self):
        currentRow = self.view.collectionsList.currentRow()
        previousCollection = self.view.collectionsList.takeItem(currentRow)
        newCollection = self.view.collectionsList.item(self.view.collectionsList.currentRow())
        self.onCollectionsCurrentItemChanged(newCollection, previousCollection)
        
        collections = self.currentProfile[CONFIG_BOOKS_KEY][CONFIG_BOOKS_COLLECTIONS_KEY]
        collections.remove(unicode(previousCollection.text(), 'utf-8'))
        
        del previousCollection
    
    def onCollectionUpButtonClicked(self):
        currentRow = self.view.collectionsList.currentRow()
        collectionItem = self.view.collectionsList.takeItem(currentRow)
        self.view.collectionsList.insertItem(currentRow - 1, collectionItem)
        self.view.collectionsList.setCurrentRow(currentRow - 1)
        
        collection = self.getCurrentCollection()
        collections = self.currentProfile[CONFIG_BOOKS_KEY][CONFIG_BOOKS_COLLECTIONS_KEY]
        index = collections.index(collection)
        collections.pop(index)
        collections.insert(index - 1, collection)
    
    def onCollectionDownButtonClicked(self):
        currentRow = self.view.collectionsList.currentRow()
        collectionItem = self.view.collectionsList.takeItem(currentRow)
        self.view.collectionsList.insertItem(currentRow + 1, collectionItem)
        self.view.collectionsList.setCurrentRow(currentRow + 1)
        
        collection = self.getCurrentCollection()
        collections = self.currentProfile[CONFIG_BOOKS_KEY][CONFIG_BOOKS_COLLECTIONS_KEY]
        index = collections.index(collection)
        collections.pop(index)
        collections.insert(index + 1, collection)
    
    def onCollectionAddButtonClicked(self):
        from PyQt4 import QtCore
        
        collections = self.currentProfile[CONFIG_BOOKS_KEY][CONFIG_BOOKS_COLLECTIONS_KEY]
        name = getUniqueName(DEFAULT_COLLECTION_NAME, collections)
        collections.append(name)
        
        self.view.collectionsList.addItem(name)
        lastRow = self.view.collectionsList.count() - 1
        self.view.collectionsList.setCurrentRow(lastRow)
        item = self.view.collectionsList.item(lastRow)
        item.setFlags(item.flags() | QtCore.Qt.ItemIsEditable)
        self.view.collectionsList.editItem(item)
    
    def onCollectionsItemChanged(self, item):
        from PyQt4.Qt import QString
        
        collections = self.currentProfile[CONFIG_BOOKS_KEY][CONFIG_BOOKS_COLLECTIONS_KEY]
        row = self.view.collectionsList.row(item)
        oldName = collections[row]
        newName = unicode(item.text(), 'utf-8').strip()
        
        if newName != unicode(item.text(), 'utf-8'): # Ignore added leading and trailing spaces
            item.setText(QString(newName))
        
        if newName is None or len(newName) is 0:
            newName = oldName
            item.setText(QString(oldName))
        
        if newName != oldName:
            uniqueName = getUniqueName(newName, collections)
            collections[row] = uniqueName
            item.setText(QString(uniqueName))
    
class ProfileListView(QGroupBox):
    def __init__(self):
        from PyQt4 import QtCore
        from PyQt4.Qt import QListWidget, QPushButton, QGridLayout, QIcon
        
        QGroupBox.__init__(self)
        
        self.setTitle(CONFIG_LABEL_PROFILES)
        
        self.profileList = QListWidget()
        self.profileAddButton = QPushButton(CONFIG_LABEL_ADD)
        self.profileDeleteButton = QPushButton(CONFIG_LABEL_DELETE)
        self.profileUpButton = QPushButton(QIcon(I('arrow-up.png')), '')
        self.profileDownButton = QPushButton(QIcon(I('arrow-down.png')), '')
        self.profileLayout = QGridLayout(self)
        self.profileLayout.addWidget(self.profileList, 0, 0, 2, 2)
        self.profileLayout.addWidget(self.profileAddButton, 2, 0, 1, 1)
        self.profileLayout.addWidget(self.profileDeleteButton, 2, 1, 1, 1)
        self.profileLayout.addWidget(self.profileUpButton, 0, 2, 1, 1, QtCore.Qt.AlignBottom)
        self.profileLayout.addWidget(self.profileDownButton, 1, 2, 1, 1, QtCore.Qt.AlignTop)
    
from PyQt4.QtCore import QObject, pyqtSignal
             
class ProfileListController(QObject):
    currentProfileChanged = pyqtSignal(unicode, unicode)
    
    def __init__(self, model, view, normalizeFunc):
        from PyQt4 import QtCore
        
        QObject.__init__(self)
        
        self.model = model
        self.normalizeFunc = normalizeFunc
        self.view = view
        
        self.currentModelProfile = None
    
        profiles = self.model
        profileNames = profiles.keys()
        profileNames.sort(key=lambda profileName: profiles[profileName][CONFIG_POS_KEY])
        
        for profileName in profileNames:
            self.view.profileList.addItem(profileName)
            item = self.view.profileList.item(self.view.profileList.count() - 1)
            item.setFlags(item.flags() | QtCore.Qt.ItemIsEditable)
        
        self.view.connect(self.view.profileList, QtCore.SIGNAL('currentItemChanged(QListWidgetItem *, QListWidgetItem *)'), self.onProfileCurrentItemChanged)
        self.view.connect(self.view.profileList, QtCore.SIGNAL('itemDoubleClicked(QListWidgetItem *)'), self.onProfileItemDoubleClicked)
        self.view.connect(self.view.profileList, QtCore.SIGNAL('itemChanged(QListWidgetItem *)'), self.onProfileItemChanged)
        self.view.connect(self.view.profileDeleteButton, QtCore.SIGNAL('clicked(bool)'), self.onProfileDeleteButtonClicked)
        self.view.connect(self.view.profileAddButton, QtCore.SIGNAL('clicked(bool)'), self.onProfileAddButtonClicked)
        self.view.connect(self.view.profileUpButton, QtCore.SIGNAL('clicked(bool)'), self.onProfileUpButtonClicked)
        self.view.connect(self.view.profileDownButton, QtCore.SIGNAL('clicked(bool)'), self.onProfileDownButtonClicked)
        
    def initialize(self):
        self.view.profileList.setCurrentRow(0)
            
    def getCurrentProfile(self):
        profileName = unicode(self.view.profileList.item(self.view.profileList.currentRow()).text(), 'utf-8')
        
        return self.model[profileName]
    
    def setCurrentProfile(self, oldProfileName, newProfileName):
        self.currentModelProfile = self.model[newProfileName]
            
        self.currentProfileChanged.emit(oldProfileName, newProfileName)
    
    def onProfileCurrentItemChanged(self, newItem, previousItem):
        if newItem is not None:
            if previousItem is None:
                oldProfileName = unicode('', 'utf-8')
            else:
                oldProfileName = unicode(previousItem.text(), 'utf-8')
                
            newProfileName = unicode(newItem.text(), 'utf-8')
                
            self.setCurrentProfile(oldProfileName, newProfileName)
            
            self.view.profileDeleteButton.setEnabled(self.view.profileList.count() > 1)
            self.view.profileUpButton.setEnabled(self.view.profileList.count() > 1 and self.view.profileList.currentRow() > 0)
            self.view.profileDownButton.setEnabled(self.view.profileList.count() > 1 and self.view.profileList.currentRow() < self.view.profileList.count() - 1)
    
    def onProfileItemDoubleClicked(self, item):
        self.view.profileList.editItem(item)
    
    def onProfileDeleteButtonClicked(self):
        profile = self.getCurrentProfile()
        currentRow = self.view.profileList.currentRow()
        previousProfile = self.view.profileList.takeItem(currentRow)
        newProfile = self.view.profileList.item(self.view.profileList.currentRow())
        self.onProfileCurrentItemChanged(newProfile, previousProfile)
        
        del self.model[unicode(previousProfile.text(), 'utf-8')]
        
        del previousProfile
    
    def onProfileUpButtonClicked(self):
        currentRow = self.view.profileList.currentRow()
        profileItem = self.view.profileList.takeItem(currentRow)
        self.view.profileList.insertItem(currentRow - 1, profileItem)
        self.view.profileList.setCurrentRow(currentRow - 1)
    
    def onProfileDownButtonClicked(self):
        currentRow = self.view.profileList.currentRow()
        profileItem = self.view.profileList.takeItem(currentRow)
        self.view.profileList.insertItem(currentRow + 1, profileItem)
        self.view.profileList.setCurrentRow(currentRow + 1)
    
    def onProfileAddButtonClicked(self):
        from PyQt4 import QtCore
        
        profiles = self.model
        name = getUniqueName(DEFAULT_PROFILE_NAME, profiles.keys())
        profile = {}
        self.normalizeFunc(name, profile)
        profiles[name] = profile
        
        self.view.profileList.addItem(name)
        lastRow = self.view.profileList.count() - 1
        self.view.profileList.setCurrentRow(lastRow)
        item = self.view.profileList.item(lastRow)
        item.setFlags(item.flags() | QtCore.Qt.ItemIsEditable)
        self.view.profileList.editItem(item)
    
    def onProfileItemChanged(self, item):
        import copy
        from PyQt4.Qt import QString
        
        profile = self.currentModelProfile
        oldName = profile[CONFIG_NAME_KEY]
        newName = unicode(item.text(), 'utf-8').strip()
        
        if newName != unicode(item.text(), 'utf-8'): # Ignore added leading and trailing spaces
            item.setText(QString(newName))
        
        if newName is None or len(newName) is 0:
            newName = oldName
            item.setText(QString(oldName))
        
        if newName != oldName:
            profiles = self.model
            uniqueName = getUniqueName(newName, profiles.keys())
            newProfile = copy.copy(profile)
            self.normalizeFunc(uniqueName, newProfile)
            profiles[uniqueName] = newProfile
            self.setCurrentProfile(oldName, uniqueName)
            del profiles[oldName]
            item.setText(QString(uniqueName))
        