#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import with_statement

__license__   = 'GPL v3'
__copyright__ = '2011, Eddie Lau; 2009, Kovid Goyal <kovid@kovidgoyal.net>'
__docformat__ = 'restructuredtext en'

from cStringIO import StringIO

from calibre.customize.conversion import OutputFormatPlugin
from calibre.customize.conversion import OptionRecommendation

# import to obtain calibre version
from calibre.constants import numeric_version

# import from oeb output
import os, re
import tempfile
from lxml import etree

from calibre.customize.conversion import OutputFormatPlugin
from calibre import CurrentDir
from calibre.customize.conversion import OptionRecommendation

from urllib import unquote

import CalibreKindlegenHelper

# import for calling external kindlegen
import random
from subprocess import call

# import from kindlestrip by Paul Durrant
from kindlestrip import SectionStripper 

class MobiError(Exception):
    pass
    
class AZWOutput(OutputFormatPlugin):

    name = 'AZW Output'
    description = 'An output plugin that output azw-format ebooks. It uses the original Calibre routine to generate typical ebooks, but uses external Kindlegen to produce periodicals. Before use, please put the kindlegen executable into this plugin zip file.'
    author = 'Eddie Lau (modified from code by Kovid Goyal)'
    supported_platforms = ['windows', 'linux']
    version = (1, 0, 4)
    minimum_calibre_version = (0, 7, 53)
    file_type = 'azw'

    options = set([
        OptionRecommendation(name='rescale_images', recommended_value=False,
            help=_('Modify images to meet Palm device size limitations.')
        ),
        OptionRecommendation(name='prefer_author_sort',
            recommended_value=False, level=OptionRecommendation.LOW,
            help=_('When present, use author sort field as author.')
        ),
        OptionRecommendation(name='no_inline_toc',
            recommended_value=False, level=OptionRecommendation.LOW,
            help=_('Don\'t add Table of Contents to end of book. Useful if '
                'the book has its own table of contents.')),
        OptionRecommendation(name='toc_title', recommended_value=None,
            help=_('Title for any generated in-line table of contents.')
        ),
        OptionRecommendation(name='dont_compress',
            recommended_value=False, level=OptionRecommendation.LOW,
            help=_('Disable compression of the file contents.')
        ),
        OptionRecommendation(name='personal_doc', recommended_value='[PDOC]',
            help=_('Tag marking book to be filed with Personal Docs')
        ),
        OptionRecommendation(name='mobi_ignore_margins',
            recommended_value=False,
            help=_('Ignore margins in the input document. If False, then '
                'the MOBI output plugin will try to convert margins specified'
                ' in the input document, otherwise it will ignore them.')
        ),
    ])
    
    # from oeb output
    recommendations = set([('pretty_print', True, OptionRecommendation.HIGH)])
    
    # from RemoveNavBar
    exception = 'index.html'
                       
    def check_for_periodical(self):
        if self.is_periodical:
            self.periodicalize_toc()
            self.check_for_masthead()
            self.opts.mobi_periodical = True
        else:
            self.opts.mobi_periodical = False

    def check_for_masthead(self):
        found = 'masthead' in self.oeb.guide
        if not found:
            self.oeb.log.debug('No masthead found in manifest, generating default mastheadImage...')
            try:
                from PIL import Image as PILImage
                PILImage
            except ImportError:
                import Image as PILImage

            raw = open(P('content_server/calibre_banner.png'), 'rb')
            im = PILImage.open(raw)
            of = StringIO()
            im.save(of, 'GIF')
            raw = of.getvalue()
            id, href = self.oeb.manifest.generate('masthead', 'masthead')
            self.oeb.manifest.add(id, href, 'image/gif', data=raw)
            self.oeb.guide.add('masthead', 'Masthead Image', href)
        else:
            self.oeb.log.debug('Using mastheadImage supplied in manifest...')


    def dump_toc(self, toc) :
        self.log( "\n         >>> TOC contents <<<")
        self.log( "     toc.title: %s" % toc.title)
        self.log( "      toc.href: %s" % toc.href)
        for periodical in toc.nodes :
            self.log( "\tperiodical title: %s" % periodical.title)
            self.log( "\t            href: %s" % periodical.href)
            for section in periodical :
                self.log( "\t\tsection title: %s" % section.title)
                self.log( "\t\tfirst article: %s" % section.href)
                for article in section :
                    self.log( "\t\t\tarticle title: %s" % repr(article.title))
                    self.log( "\t\t\t         href: %s" % article.href)

    def dump_manifest(self) :
        self.log( "\n         >>> Manifest entries <<<")
        for href in self.oeb.manifest.hrefs :
            self.log ("\t%s" % href)

    def periodicalize_toc(self):
        from calibre.ebooks.oeb.base import TOC
        toc = self.oeb.toc
        if not toc or len(self.oeb.spine) < 3:
            return
        if toc and toc[0].klass != 'periodical':
            one, two = self.oeb.spine[0], self.oeb.spine[1]
            self.log('Converting TOC for MOBI periodical indexing...')

            articles = {}
            if toc.depth() < 3:
                # single section periodical
                self.oeb.manifest.remove(one)
                self.oeb.manifest.remove(two)
                sections = [TOC(klass='section', title=_('All articles'),
                    href=self.oeb.spine[0].href)]
                for x in toc:
                    sections[0].nodes.append(x)
            else:
                # multi-section periodical
                self.oeb.manifest.remove(one)
                sections = list(toc)
                for i,x in enumerate(sections):
                    x.klass = 'section'
                    articles_ = list(x)
                    if articles_:
                        self.oeb.manifest.remove(self.oeb.manifest.hrefs[x.href])
                        x.href = articles_[0].href


            for sec in sections:
                articles[id(sec)] = []
                for a in list(sec):
                    a.klass = 'article'
                    articles[id(sec)].append(a)
                    sec.nodes.remove(a)

            root = TOC(klass='periodical', href=self.oeb.spine[0].href,
                    title=unicode(self.oeb.metadata.title[0]))

            for s in sections:
                if articles[id(s)]:
                    for a in articles[id(s)]:
                        s.nodes.append(a)
                    root.nodes.append(s)

            for x in list(toc.nodes):
                toc.nodes.remove(x)

            toc.nodes.append(root)

            # Fix up the periodical href to point to first section href
            toc.nodes[0].href = toc.nodes[0].nodes[0].href

            # GR diagnostics
            if self.opts.verbose > 3:
                self.dump_toc(toc)
                self.dump_manifest()

    # from oeb output
    def workaround_nook_cover_bug(self, root): # {{{
        cov = root.xpath('//*[local-name() = "meta" and @name="cover" and'
                ' @content != "cover"]')

        def manifest_items_with_id(id_):
            return root.xpath('//*[local-name() = "manifest"]/*[local-name() = "item" '
                ' and @id="%s"]'%id_)

        if len(cov) == 1:
            cov = cov[0]
            covid = cov.get('content', '')

            if covid:
                manifest_item = manifest_items_with_id(covid)
                if len(manifest_item) == 1 and \
                        manifest_item[0].get('media-type',
                                '').startswith('image/'):
                    self.log.warn('The cover image has an id != "cover". Renaming'
                            ' to work around bug in Nook Color')

                    import uuid
                    newid = str(uuid.uuid4())

                    for item in manifest_items_with_id('cover'):
                        item.set('id', newid)

                    for x in root.xpath('//*[@idref="cover"]'):
                        x.set('idref', newid)

                    manifest_item = manifest_item[0]
                    manifest_item.set('id', 'cover')
                    cov.set('content', 'cover')
    # }}}

    def workaround_pocketbook_cover_bug(self, root): # {{{
        m = root.xpath('//*[local-name() = "manifest"]/*[local-name() = "item" '
                ' and @id="cover"]')
        if len(m) == 1:
            m = m[0]
            p = m.getparent()
            p.remove(m)
            p.insert(0, m)
    # }}}
    
    def convert(self, oeb, output_path, input_plugin, opts, log):
        self.log, self.opts, self.oeb = log, opts, oeb
        from calibre.ebooks.mobi.writer import PALM_MAX_IMAGE_SIZE, \
                    MobiWriter, PALMDOC, UNCOMPRESSED
        from calibre.ebooks.mobi.mobiml import MobiMLizer
        from calibre.ebooks.oeb.transforms.manglecase import CaseMangler
        from calibre.ebooks.oeb.transforms.rasterize import SVGRasterizer, Unavailable
        from calibre.ebooks.oeb.transforms.htmltoc import HTMLTOCAdder
        from calibre.customize.ui import plugin_for_input_format
        # change: if it is not a periodical, use typical routine
        if not self.is_periodical:
            imagemax = PALM_MAX_IMAGE_SIZE if opts.rescale_images else None
            if not opts.no_inline_toc:
                tocadder = HTMLTOCAdder(title=opts.toc_title)
                tocadder(oeb, opts)
            mangler = CaseMangler()
            mangler(oeb, opts)
            try:
                rasterizer = SVGRasterizer()
                rasterizer(oeb, opts)
            except Unavailable:
                self.log.warn('SVG rasterizer unavailable, SVG will not be converted')
            mobimlizer = MobiMLizer(ignore_tables=opts.linearize_tables)
            mobimlizer(oeb, opts)
            self.check_for_periodical()
        
            write_page_breaks_after_item = not input_plugin is plugin_for_input_format('cbz')
            if numeric_version[0] < 0 or numeric_version[0] == 0 and numeric_version[1] < 8 or numeric_version[0] == 0 and numeric_version[1] == 8 and numeric_version[2] < 11:   
                writer = MobiWriter(opts, imagemax=imagemax,
                        compression=UNCOMPRESSED if opts.dont_compress else PALMDOC,
                                    prefer_author_sort=opts.prefer_author_sort,
                                    write_page_breaks_after_item=write_page_breaks_after_item)
            else:
                from calibre.utils.config import tweaks
                if tweaks.get('new_mobi_writer', False):
                    from calibre.ebooks.mobi.writer2.main import MobiWriter
                    MobiWriter
                else:
                    from calibre.ebooks.mobi.writer import MobiWriter
                    writer = MobiWriter(opts,
                             write_page_breaks_after_item=write_page_breaks_after_item)
			writer(oeb, output_path)
        else:
            # overall routine:
            # 1) remove navbar, use oeb code to generate tmp oeb files
            # 2) use my code to remove nav bar and add necessary tags
            # 3) use kindlegen to generate the final file
            # no navbar        
            self.log, self.opts = log, opts
            imagemax = PALM_MAX_IMAGE_SIZE if opts.rescale_images else None
            if not opts.no_inline_toc:
                tocadder = HTMLTOCAdder(title=opts.toc_title)
                tocadder(oeb, opts)
            mangler = CaseMangler()
            mangler(oeb, opts)
            try:
                rasterizer = SVGRasterizer()
                rasterizer(oeb, opts)
            except Unavailable:
                self.log.warn('SVG rasterizer unavailable, SVG will not be converted')
                
            from calibre.ebooks.oeb.base import XPath
            for item in oeb.spine:
                for div in XPath('//h:div[contains(@class, "calibrenavbar")]')(item.data):
                    div.getparent().remove(div)
            
            # generate a unique validtemppath
            validtemppath = False
            temppath = os.path.join(tempfile.gettempdir(), 'tmp' + str(random.random()))
            while not validtemppath:
                if not os.path.exists(temppath):
                    os.makedirs(temppath)
                    validtemppath = True
                  
            # use MobiMLizer to format the content of the periodical better
            # currently it removes all the article title and all its figures
            mobimlizer = MobiMLizer(ignore_tables=opts.linearize_tables)
            mobimlizer(oeb, opts)
            
            from calibre.ebooks.oeb.base import OPF_MIME, NCX_MIME, PAGE_MAP_MIME
            with CurrentDir(temppath):
                results = oeb.to_opf2(page_map=True)
                for key in (OPF_MIME, NCX_MIME, PAGE_MAP_MIME):
                    href, root = results.pop(key, [None, None])
                    if root is not None:
                        if key == OPF_MIME:
                            try:
                                self.workaround_nook_cover_bug(root)
                            except:
                                self.log.exception('Something went wrong while trying to'
                                        ' workaround Nook cover bug, ignoring')
                            try:
                                self.workaround_pocketbook_cover_bug(root)
                            except:
                                self.log.exception('Something went wrong while trying to'
                                        ' workaround Pocketbook cover bug, ignoring')
                        raw = etree.tostring(root, pretty_print=True,
                                encoding='utf-8', xml_declaration=True)
                        if key == OPF_MIME:
                            # Needed as I can't get lxml to output opf:role and
                            # not output <opf:metadata> as well
                            raw = re.sub(r'(<[/]{0,1})opf:', r'\1', raw)
                        with open(href, 'wb') as f:
                            f.write(raw)

                for item in oeb.manifest:
                    path = os.path.abspath(unquote(item.href))
                    dir = os.path.dirname(path)
                    if not os.path.exists(dir):
                        os.makedirs(dir)
                    with open(path, 'wb') as f:
                        f.write(str(item))
                    item.unload_data_from_memory(memory=path)
            # remove navbar
            # only deal with folders of the given folder
            #CalibreKindlegenHelper.FuncRemoveNavBar(temppath)
              
            # compute new opf and nox file
            CalibreKindlegenHelper.FuncConvertKG(os.path.join(temppath, 'content.opf'), os.path.join(temppath, 'toc.ncx'), os.path.join(temppath, 'content.opf'), os.path.join(temppath, 'toc.ncx'))
            # use external kindlegen:
            # copy the kindlegen exe to the temp folder, then run it
            if os.name == 'posix':
                # Linux stuff
                kindlegenbin = get_resources('kindlegen')
                f = open(os.path.join(temppath, 'kindlegen'), 'wb')
                f.write(kindlegenbin)
                f.close()
                os.system('chmod 755 ' + os.path.join(temppath, 'kindlegen'))
                os.system(os.path.join(temppath, 'kindlegen') + ' ' + os.path.join(temppath, 'content.opf'))
                os.system('mv ' + os.path.join(temppath, 'content.mobi') + ' ' + output_path)
                os.system('rm -rf ' + temppath)
            elif os.name == 'nt':
                # Windows stuff
                kindlegenbin = get_resources('kindlegen.exe')
                f = open(os.path.join(temppath, 'kindlegen.exe'), 'wb')
                f.write(kindlegenbin)
                f.close()
                os.system(os.path.join(temppath, 'kindlegen.exe') + ' ' + os.path.join(temppath, 'content.opf'))
                os.system('move ' + os.path.join(temppath, 'content.mobi') + ' ' + output_path)
                os.system('rmdir /Q /S ' + temppath)

            # strip unnecessary source files
            infile = file(output_path, 'rb')
            data_file = infile.read()
            infile.close()
            try:
                strippedFile = SectionStripper(data_file)
                outfile = file(output_path, 'wb')
                outfile.write(strippedFile.getResult())
                outfile.close()
            except StripException, e:
			    print "Error %s" % e
