//////////////////////////////////////////////////////////////////////
// metadata: Manage books metadata for iRex DR series eReaders
//
// Copyright (C) 2011  Iñigo Serna <inigoserna@gmail.com>
// Released under GPL v3+
//
// Time-stamp: <2011-07-10 20:42:58 inigo>
//////////////////////////////////////////////////////////////////////


using GLib;
using Gdk;
using Sqlite;
using Poppler;
using Posix;

using MyUtils;


//////////////////////////////////////////////////////////////////////
///// Constants
const string SDCARD = "/media/mmcblk0p1";
const string DBFILE = "global.db";
const string THUMB_FS_FILE_FMT = "System/.covers/%s/%.2x/%d.png";
const string THUMB_FS_FILE_FMT2 = "System/.covers/%s/%.2x";
const int SIZE_SMALL  = 60;
const int SIZE_MEDIUM = 120;
const string TEMP_IMG = "_tmpimg_";
const string TEMP_DIR = "_tmpdir_";
const string[] IMGS_MIMETYPE = {"image/jpg", "image/jpeg", "image/png", "image/bmp"};
const string[] IMGS_ID = {"cover", "cover.jpg", "cover.jpeg"};
enum FILETYPE {ALL, DESKTOP, PDF, FB2, EPUB}
const string[] FILETYPES = {"all", "desktop", "pdf", "fb2", "epub"};
enum ACTION {REPORT, ADD, ADD_METADATA, ADD_THUMBS,
             OVERWRITE, OVERWRITE_METADATA, OVERWRITE_THUMBS,
             UPDATE, UPDATE_METADATA, UPDATE_THUMBS,
             DELETE, DELETE_THUMBS, VACUUM}
const string[] ACTIONS = {"report", "add", "add-metadata", "add-thumbs",
                          "overwrite", "overwrite-metadata", "overwrite-thumbs",
                          "update", "update-metadata", "update-thumbs",
                          "delete", "delete-thumbs", "vacuum"};

errordomain MyError {
    FILE,
    POPPLER,
    INVALID_FILE,
    NO_ICON,
    DB_ERROR,
    FS_ERROR,
    FAILED
}


//////////////////////////////////////////////////////////////////////
///// Global Variables
Config config;


//////////////////////////////////////////////////////////////////////
///// Utils
void pr1(string fmt, ...) {
    var l = va_list();
    string buf = fmt.vprintf(l);
    if (config.verbose)
        GLib.stdout.printf(buf);
    if (config.log) {
        config.log_file.puts(buf);
        config.log_file.flush();
    }
}


void pr2(string fmt, ...) {
    var l = va_list();
    string buf = fmt.vprintf(l);
    GLib.stdout.printf(buf);
    if (config.log) {
        config.log_file.puts(buf);
        config.log_file.flush();
    }
}


void prerr(string fmt, ...) {
    var l = va_list();
    string buf = fmt.vprintf(l);
    GLib.stderr.printf(buf);
    if (config.log) {
        config.log_file.puts(buf);
        config.log_file.flush();
    }
}


string normalize_path(string path) {
    var newpath = path.replace("//", "/");
    if (newpath.has_suffix("/"))
        newpath = newpath[0:-1];
    return newpath;
}


void pixbuf2pngs(Gdk.Pixbuf pb, out uint8[]? png_small,
                 out uint8[]? png_medium) throws GLib.Error {
    Gdk.Pixbuf pb1, pb2;
    int h1, w1, h2, w2;
    if (pb.width > pb.height) {
        w1 = SIZE_MEDIUM;
        h1 = (int) w1 * pb.height/pb.width;
        w2 = SIZE_SMALL;
        h2 = (int) w2 * pb.height/pb.width;
    } else {
        h1 = SIZE_MEDIUM;
        w1 = (int) h1 * pb.width/pb.height;
        h2 = SIZE_SMALL;
        w2 = (int) h2 * pb.width/pb.height;
    }
    // pb1 = pb.scale_simple(w1, h1, Gdk.InterpType.BILINEAR);
    pb1 = pb.scale_simple(w1, h1, Gdk.InterpType.TILES);
    pb1.save_to_buffer(out png_medium, "png");
    pb2 = pb1.scale_simple(w2, h2, Gdk.InterpType.TILES);
    pb2.save_to_buffer(out png_small, "png");
}


void imgdata2pngs(uint8[] imgdata, out uint8[]? png_small,
                  out uint8[]? png_medium) throws GLib.Error {
    Gdk.Pixbuf pb;
    #if (GDKPIXBUF_HIGHERTHAN_212)
        var imgstream = new GLib.MemoryInputStream.from_data(imgdata, null);
        pb = new Gdk.Pixbuf.from_stream(imgstream);
    #else
        string tmpfile = TEMP_IMG;
        FileUtils.set_data(tmpfile, imgdata);
        pb = new Gdk.Pixbuf.from_file(tmpfile);
        File.new_for_path(tmpfile).delete();
    #endif
    pixbuf2pngs(pb, out png_small, out png_medium);
}


//////////////////////////////////////////////////////////////////////
///// Database
class DBGlobal : GLib.Object {

    ///// Constants
    const string STMT_CHECK_ENTRY     = "SELECT file_id FROM file_metadata WHERE filename=? AND directory_path=?";
    const string STMT_CHECK_THUMBS    = "SELECT file_id FROM thumbnails WHERE file_id=?";
    const string STMT_ADD_METADATA    = "INSERT INTO file_metadata('filename', 'directory_path', 'file_type', 'file_size', 'file_time_added', 'file_time_modified', 'title', 'author', 'number_of_pages') VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";
    const string STMT_ADD_THUMBS      = "INSERT INTO thumbnails('file_id', 'thumb_data_small', 'thumb_data_medium') VALUES (?, ?, ?)";
    const string STMT_UPDATE_METADATA = "UPDATE file_metadata SET title=?, author=? WHERE file_id=?";
    const string STMT_UPDATE_THUMBS   = "UPDATE thumbnails SET thumb_data_small=?, thumb_data_medium=? WHERE file_id=?";
    const string STMT_DELETE_METADATA = "DELETE FROM file_metadata WHERE file_id=?";
    const string STMT_DELETE_THUMBS   = "DELETE FROM thumbnails WHERE file_id=?";
    const string STMT_DO_VACUUM       = "VACUUM;";


    ///// Variables
    Sqlite.Database db;
    Sqlite.Statement stmt;
    int rc;


    ///// Constructor
    public DBGlobal() {
        var dbfile = Path.build_filename(config.sdcard, DBFILE);
        rc = 0;
        if ((rc=Database.open(dbfile, out db)) != Sqlite.OK) {
            prerr("can't open database: %s", db.errmsg());
        }
    }


    ///// Helpers
    private void _build_query(string query, ...) throws MyError {
        var l = va_list();
        rc = 0;
        if ((rc=db.prepare_v2(query, -1, out stmt, null)) != Sqlite.OK)
            throw new MyError.DB_ERROR("SQL prepare error (%d): %s", rc, db.errmsg());
        int i = 1;
        while (true) {
            GLib.Value? v = l.arg();
            if (v == null)
                break;  // end of the list
            switch (v.type_name()) {
                case "gchararray":
                    if ((rc=stmt.bind_text(i, (string) v)) != Sqlite.OK)
                        throw new MyError.DB_ERROR("SQL bind text (%d): %s", rc, db.errmsg());
                    break;
                case "gint":
                case "guint":
                case "gshort":
                case "gushort":
                    if ((rc=stmt.bind_int(i, (int) v)) != Sqlite.OK)
                        throw new MyError.DB_ERROR("SQL bind int (%d): %s", rc, db.errmsg());
                    break;
                case "gint64":
                case "glong":
                case "gulong":
                    if ((rc=stmt.bind_int64(i, (int64) v)) != Sqlite.OK)
                        throw new MyError.DB_ERROR("SQL bind int (%d): %s", rc, db.errmsg());
                    break;
                default:
                    throw new MyError.DB_ERROR("SQL: bad type passed to prepare_query: %s", v.type_name());
            }
            i++;
        }
    }


    private void _check_fileid(BookBase b) throws MyError {
        GLib.Value v1 = b.filename, v2 = b.path;
        _build_query(STMT_CHECK_ENTRY, v1, v2);
        if ((rc=stmt.step()) == Sqlite.DONE)
            return;
        else if (rc == Sqlite.ROW)
            b.id = stmt.column_int(0);
        else
            throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
    }


    private void _check_thumb(BookBase b) throws MyError {
        if (config.thumbs_in_fs) {
            var fthumb_s = Path.build_filename(config.sdcard,
                                               THUMB_FS_FILE_FMT.printf("small",
                                                                        b.id % 256,
                                                                        b.id));
            if (FileUtils.test(fthumb_s, FileTest.IS_REGULAR))
                b.thumb_stored = true;
        } else {
            GLib.Value v1 = b.id;
            _build_query(STMT_CHECK_THUMBS, v1);
            if ((rc=stmt.step()) == Sqlite.DONE)
                return;
            else if (rc == Sqlite.ROW)
                b.thumb_stored = true;
            else
                throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
        }
    }


    private void _do_fs_thumbs(BookBase b) throws MyError {
        try {
            var dir_s = Path.build_filename(config.sdcard,
                                            THUMB_FS_FILE_FMT2.printf("small", b.id % 256));
            var dir_m = Path.build_filename(config.sdcard,
                                            THUMB_FS_FILE_FMT2.printf("medium", b.id % 256));
            if (!FileUtils.test(dir_s, FileTest.IS_DIR))
                DirUtils.create_with_parents(dir_s, 0755);
            if (!FileUtils.test(dir_m, FileTest.IS_DIR))
                DirUtils.create_with_parents(dir_m, 0755);
            var imgfile_s = Path.build_filename(dir_s, "%d.png".printf(b.id));
            var imgfile_m = Path.build_filename(dir_m, "%d.png".printf(b.id));
            FileUtils.set_data(imgfile_s, b.png_s);
            FileUtils.set_data(imgfile_m, b.png_m);
            } catch (GLib.FileError e) {
                throw new MyError.FAILED("could't store thumbnails in file system: %s", e.message);
            }
    }


    public void check_for_entry(BookBase b) throws MyError {
        _check_fileid(b);
        _check_thumb(b);
    }


    ///// add entry only metadata (no thumbs)
    public void add_entry_metadata(BookBase b) throws MyError {
        if (b.id != -1)
            throw new MyError.FAILED("book already exists in database");
        int64 now = time_t();
        GLib.Value v1 = b.filename, v2 = b.path, v3 = b.type, v4 = b.file_size,
            v5 = now, v6 = now, v7 = b.title, v8 = b.author, v9 = b.num_pages;
        _build_query(STMT_ADD_METADATA, v1, v2, v3, v4, v5, v6, v7, v8, v9);
        if ((rc=stmt.step()) != Sqlite.DONE)
            throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
        _check_fileid(b); // update file_id
    }


    ///// add entry missing thumbnails
    public void add_entry_missing_thumbs(BookBase b) throws MyError {
        if (b.id == -1)
            throw new MyError.FAILED("book does not exist in database");
        // if (b.thumb_stored)
        //     throw new MyError.FAILED("book already has thumbnails");
        if (config.thumbs_in_fs) {
            _do_fs_thumbs(b);
        } else {
            if ((rc=db.prepare_v2(STMT_ADD_THUMBS, -1, out stmt, null)) != Sqlite.OK)
                throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
            if ((rc=stmt.bind_int(1, b.id)) != Sqlite.OK)
                throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
            if ((rc=stmt.bind_blob(2, b.png_s, b.png_s.length)) != Sqlite.OK)
                throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
            if ((rc=stmt.bind_blob(3, b.png_m, b.png_m.length)) != Sqlite.OK)
                throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
            if ((rc=stmt.step()) != Sqlite.DONE)
                throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
        }
        _check_thumb(b);
    }


    ///// update entry only metadata (no thumbs)
    public void update_entry_metadata(BookBase b) throws MyError {
        if (b.id == -1)
            throw new MyError.FAILED("book does not exist in database");
        GLib.Value v1 = b.title, v2 = b.author, v3 = b.id;
        _build_query(STMT_UPDATE_METADATA, v1, v2, v3);
        if ((rc=stmt.step()) != Sqlite.DONE)
            throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
    }


    ///// udpate entry thumbnails
    public void update_entry_thumbs(BookBase b) throws MyError {
        if (b.id == -1)
            throw new MyError.FAILED("book does not exist in database");
        if (config.thumbs_in_fs) {
            _do_fs_thumbs(b);
        } else {
            if ((rc=db.prepare_v2(STMT_UPDATE_THUMBS, -1, out stmt, null)) != Sqlite.OK)
                throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
            if ((rc=stmt.bind_blob(1, b.png_s, b.png_s.length)) != Sqlite.OK)
                throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
            if ((rc=stmt.bind_blob(2, b.png_m, b.png_m.length)) != Sqlite.OK)
                throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
            if ((rc=stmt.bind_int(3, b.id)) != Sqlite.OK)
                throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
            if ((rc=stmt.step()) != Sqlite.DONE)
                throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
        }
        _check_thumb(b);
    }


    ///// delete entry metadata
    public void delete_metadata(BookBase b) throws MyError {
        if (b.id == -1)
            throw new MyError.FAILED("book does not exist in database");
        GLib.Value v1 = b.id;
        _build_query(STMT_DELETE_METADATA, v1);
        if ((rc=stmt.step()) != Sqlite.DONE)
            throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
    }


    ///// delete entry thumbnails
    public void delete_thumbs(BookBase b) throws MyError {
        if (b.id == -1)
            throw new MyError.FAILED("book does not exist in database");
        if (!b.thumb_stored)
            throw new MyError.FAILED("book has not thumbnails stored");
        if (config.thumbs_in_fs) {
            var fthumb_s = Path.build_filename(config.sdcard,
                                               THUMB_FS_FILE_FMT.printf("small",
                                                                        b.id % 256,
                                                                        b.id));
            var fthumb_m = Path.build_filename(config.sdcard,
                                               THUMB_FS_FILE_FMT.printf("medium",
                                                                        b.id % 256,
                                                                        b.id));
            try {
                File.new_for_path(fthumb_s).delete();
                File.new_for_path(fthumb_m).delete();
            } catch (GLib.Error e) {
                throw new MyError.FS_ERROR("couldn't delete thumbs from file system: %s", e.message);
            }
        } else {
            GLib.Value v1 = b.id;
            _build_query(STMT_DELETE_THUMBS, v1);
            if ((rc=stmt.step()) != Sqlite.DONE)
                throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
        }
    }


    ///// vacuum database
    public void vacuum() throws MyError {
        _build_query(STMT_DO_VACUUM);
        if ((rc=stmt.step()) != Sqlite.DONE)
            throw new MyError.DB_ERROR("SQL (%d): %s", rc, db.errmsg());
    }
}


//////////////////////////////////////////////////////////////////////
///// BookBase
class BookBase : GLib.Object {
    public string fs_fullpath;
    public string type;

    public int id = -1;
    public string path;
    public string filename;

    public string? title;
    public string? author;
    public uint8[]? png_s;
    public uint8[]? png_m;

    public int num_pages = -1;
    public int64 file_size = -1;
    public string[]? tags;
    public bool thumb_stored = false;

    public BookBase(string fullpath, string ext="") throws MyError {
        if (!fullpath.down().has_suffix(ext.down()))
            throw new MyError.FILE("invalid type of file");
        var fp = fullpath;
        if (!Path.is_absolute(fullpath))
            fp = Path.build_filename(config.curdir, fullpath);
        this.fs_fullpath = fp;
        if (fp.has_prefix(config.del_prefix))
            fp = fullpath.replace(config.del_prefix, "");
        fp = normalize_path(Path.build_filename(config.add_prefix, fp));
        this.path = Path.get_dirname(fp);
        this.filename = Path.get_basename(fp);
        this.type = ext[1:ext.length].down();
    }

    public virtual void get_metadata() throws MyError {
    }

    public string repr_metadata() {
        string t = title==null ? "<none>" : title;
        string a = author==null ? "<none>" : author;
        string ts = tags==null ? "<none>" : "".join(", ", tags);
        string tfile = png_s==null ? "no" : "yes";
        string tdb = thumb_stored ? "yes" : "no";
        return @"==> $fs_fullpath\n$path/$filename (id: $id, $type, $file_size bytes)\nTitle: $t\nAuthor: $a\nNumber of pages: $num_pages\nTags: $ts\nCover: $tdb in database or file system, $tfile in book\n";
    }

    public void print_metadata() {
        pr1(repr_metadata());
    }

    public void print_shortdata() {
        pr1("==> [%5d] %s\n", id, fs_fullpath);
    }

    protected void fill_defaults() {
        if (title==null || title =="" || title.down()=="untitled")
            title = filename[0:filename.length-(type.length+1)];
        if (author==null || author=="")
            author = Path.get_basename(path);
        if (file_size == -1) {
            try {
                File file = File.new_for_path(fs_fullpath);
                file_size = file.query_info("*", FileQueryInfoFlags.NONE).get_size();
            } catch (GLib.Error e) {
                file_size = -1;
            }
        }
    }
}


//////////////////////////////////////////////////////////////////////
///// Application files
class Application : BookBase {
    const string EXT = ".desktop";

    public Application(string fullpath) throws MyError {
        base(fullpath, EXT);
    }

    public override void get_metadata() throws MyError {
        string contents, icon_path=null;
        try {
            FileUtils.get_contents(fs_fullpath, out contents);
            file_size = contents.length;
            MatchInfo mi = null;
            Regex regex_name = /^\s*Name\s*=\s*(.+)\s*$/;
            Regex regex_icon = /^\s*Icon\s*=\s*(.+\.png)\s*$/;
            foreach(string line in contents.split("\n")) {
                if (regex_name.match(line, 0, out mi))
                    title = mi.fetch(1);
                else if (regex_icon.match(line, 0, out mi))
                    icon_path = mi.fetch(1);
            }
            fill_defaults();
        } catch (GLib.Error e) {
            throw new MyError.FILE("problem with file: %s", e.message);
        }
        if (icon_path != null)
            try {
                uint8[] buffer = new uint8[100000];
                FileUtils.get_data(icon_path, out buffer);
                imgdata2pngs(buffer, out png_s, out png_m);
            } catch (GLib.Error e) {
                prerr("WARNING: can't get thumbnail for file '%s': %s\n", fs_fullpath, e.message);
                return;
            }
    }
}


//////////////////////////////////////////////////////////////////////
///// PDF
class Pdf : BookBase {
    const string EXT = ".pdf";

    public Pdf(string fullpath) throws MyError {
        base(fullpath, EXT);
    }

    public override void get_metadata() throws MyError {
        Gdk.Pixbuf pixbuf;
        double w, h;
        try {
            var doc = new Poppler.Document.from_file(Filename.to_uri(fs_fullpath), "");
            // title, author, tags, num_pages
            author = doc.author==null ? "" : doc.author.strip();
            title = doc.title==null ? "" : doc.title.strip();
            if (doc.keywords != null)
                tags = doc.keywords.split(" ");
            num_pages = doc.get_n_pages();
            fill_defaults();
            // cover
            Poppler.Page page = doc.get_page(0);
            page.get_size(out w, out h);
            pixbuf = new Gdk.Pixbuf(Gdk.Colorspace.RGB, false, 8, (int) w, (int) h);
            page.render_to_pixbuf(0, 0, (int) w, (int) h, 1.0, 0, pixbuf);
            pixbuf2pngs(pixbuf, out png_s, out png_m);
        } catch (GLib.Error e) {
            throw new MyError.FILE("problem with file: %s", e.message);
        }
    }
}


//////////////////////////////////////////////////////////////////////
///// FB2
struct Binary {
    public string id;
    public string contenttype;
    public uint8[] data;
}


class FB2Parser: GLib.Object {

    MarkupParseContext context;
    const MarkupParser parser = { // It's a structure, not an object
        start,// when an element opens
        end,  // when an element closes
        text, // when text is found
        null, // when comments are found
        null  // when errors occur
    };

    bool in_coverpage = false;
    bool in_titleinfo = false;
    bool in_booktitle = false;
    bool in_author    = false;
    bool in_author_firstname  = false;
    bool in_author_middlename = false;
    bool in_author_lastname   = false;
    string? cover_id;
    Binary? bin;
    public Binary? cover;
    public string? title;
    public string? author_firstname;
    public string? author_middlename;
    public string? author_lastname;

    construct {
        context = new MarkupParseContext(
            parser, // the structure with the callbacks
            0,      // MarkupParseFlags
            this,   // extra argument for the callbacks, methods in this case
            destroy // when the parsing ends
        );
    }

    void destroy() {
        // pr1("Releasing any allocated resource\n");
    }

    public bool parse(string content) throws MarkupError {
        return context.parse(content, -1);
    }

    void start(MarkupParseContext context, string name,
               string[] attr_names, string[] attr_values) throws MarkupError {
        if (in_coverpage && name=="image") {
            for (int i=0; i<attr_names.length; i++) {
                if (attr_names[i] == "l:href") {
                    cover_id = attr_values[i];
                    if (cover_id[0] == 35) // "#"
                        cover_id = cover_id[1:cover_id.length];
                }
            }
        }
        if (name == "coverpage") {
            in_coverpage = true;
        } else if (name == "title-info") {
            in_titleinfo = true;
        } else if (in_titleinfo && name == "book-title") {
            in_booktitle = true;
        } else if (in_titleinfo && name == "author") {
            in_author = true;
        } else if (in_titleinfo && in_author && name == "first-name") {
            in_author_firstname = true;
        } else if (in_titleinfo && in_author && name == "middle-name") {
            in_author_middlename = true;
        } else if (in_titleinfo && in_author && name == "last-name") {
            in_author_lastname = true;
        } else if (name == "binary") {
            bin = Binary();
            for (int i=0; i<attr_names.length; i++) {
                if (attr_names[i] == "id")
                    bin.id = attr_values[i];
                else if (attr_names[i] == "content-type")
                    bin.contenttype = attr_values[i];
            }
            if (!(bin.contenttype in IMGS_MIMETYPE))
                bin = null;
        }
    }

    void end(MarkupParseContext context, string name) throws MarkupError {
        if (name == "coverpage") {
            in_coverpage = false;
        } else if (name == "title-info") {
            in_titleinfo = false;
        } else if (in_titleinfo && name == "book-title") {
            in_booktitle = false;
        } else if (in_titleinfo && name == "author") {
            in_author = false;
        } else if (in_titleinfo && in_author && name == "first-name") {
            in_author_firstname = false;
        } else if (in_titleinfo && in_author && name == "middle-name") {
            in_author_middlename = false;
        } else if (in_titleinfo && in_author && name == "last-name") {
            in_author_lastname = false;
        } else if (name == "binary" && bin != null) {
            if (bin.id==cover_id || (cover_id==null && cover==null)) {
                cover = bin;
            }
            bin = null;
        }
    }

    void text(MarkupParseContext context,
               string text, size_t text_len) throws MarkupError {
        if (bin != null)
            bin.data = GLib.Base64.decode(text);
        if (in_titleinfo && in_booktitle)
            title = text;
        else if (in_titleinfo && in_author && in_author_firstname)
            author_firstname = text;
        else if (in_titleinfo && in_author && in_author_middlename)
            author_middlename = text;
        else if (in_titleinfo && in_author && in_author_lastname)
            author_lastname = text;
    }
}


class Fb2 : BookBase {
    const string EXT = ".fb2";

    public Fb2(string fullpath) throws MyError {
        base(fullpath, EXT);
    }

    public override void get_metadata() throws MyError {
        FB2Parser parser;
        try {
            string data;
            FileUtils.get_contents(fs_fullpath, out data);
            file_size = data.length;
            parser = new FB2Parser();
            parser.parse(data);
            title = parser.title;
            author = _build_author(parser.author_firstname,
                                   parser.author_middlename,
                                   parser.author_lastname);
            fill_defaults();
        } catch (GLib.Error e) {
            throw new MyError.FILE("problem with file: %s", e.message);
        }
        if (parser.cover != null) {
            try {
                imgdata2pngs(parser.cover.data, out png_s, out png_m);
            } catch (GLib.Error e) {
                prerr("WARNING: can't get thumbnail for file '%s': %s\n", fs_fullpath, e.message);
                return;
            }
        }
    }

    private string _build_author(string? first, string? middle, string? last) {
        var author = first==null ? "" : first;
        author += middle==null ? "" : " "+middle;
        author += last==null ? "" : " "+last;
        return author ;
    }
}


//////////////////////////////////////////////////////////////////////
///// EPUB
class Epub : BookBase {
    const string EXT = ".epub";

    public Epub(string fullpath) throws MyError {
        base(fullpath, EXT);
    }

    public override void get_metadata() throws MyError {
        string cmd, out0, err0;
        string contents;
        string tmpdir = TEMP_DIR;
        MatchInfo mi = null;
        Regex regex_opf = /^\s*<rootfile full-path="(.+)"\s+media-type.+>\s*$/;
        Regex regex_author = /^\s*<dc:[cC]reator.*>(.+)<\/dc:[cC]reator>\s*$/;
        Regex regex_title = /^\s*<dc:[tT]itle.*>(.+)<\/dc:[tT]itle>\s*$/;
        Regex regex_cover1 = /^\s*<item\s+href="(.+)"\s+id="(.+)"\s+media-type="(.+)"\s*\/>\s*$/;
        Regex regex_cover2 = /^\s*<item\s+id="(.+)"\s+href="(.+)"\s+media-type="(.)"\s*\/>\s*$/;
        string opf_file = null;
        string imgfile = null;

        try {
            // uncompress epub to a temp directory
            DirUtils.create_with_parents(tmpdir, 0755);
            cmd = "unzip -qq -o %s -d %s".printf(Shell.quote(fs_fullpath), tmpdir);
            Process.spawn_command_line_sync(cmd, null, out err0, null);
            if (err0 != null && err0 != "")
                throw new SpawnError.FAILED(err0);
            // read META-INF/container.xml
            FileUtils.get_contents(tmpdir+"/META-INF/container.xml", out contents);
            // get content.opf path and read contents
            foreach(string line in contents.split("\n")) {
                if (regex_opf.match(line, 0, out mi)) {
                    opf_file = Path.build_filename(tmpdir, mi.fetch(1));
                    break;
                }
            }
            if (!FileUtils.test(opf_file, FileTest.IS_REGULAR))
                throw new IOError.NOT_FOUND("can't find content.opf in epub file");
            FileUtils.get_contents(opf_file, out contents);
        } catch (GLib.Error e) {
            try {
                Process.spawn_command_line_sync("rm -rf " + tmpdir, null, null, null);
            } catch (GLib.Error e) {}
            throw new MyError.FILE("problem with file: %s", e.message);
        }

        // parse and get dc:creator, get dc:Title, cover image file
        string href = null;
        foreach(string line in contents.split("\n")) {
            if (regex_author.match(line, 0, out mi))
                author = mi.fetch(1);
            else if (regex_title.match(line, 0, out mi))
                title = mi.fetch(1);
            else if (regex_cover1.match(line, 0, out mi))
                if ((mi.fetch(3) in IMGS_MIMETYPE) && (mi.fetch(2) in IMGS_ID))
                    href = mi.fetch(1);
            else if (regex_cover2.match(line, 0, out mi))
                if ((mi.fetch(3) in IMGS_MIMETYPE) && (mi.fetch(1) in IMGS_ID))
                    href = mi.fetch(2);
        }
        fill_defaults();

        // and now the cover
        try {
            if (href == null) {
                // if no cover found => get any image from .epub
                cmd = "find " + tmpdir + " -name '*.png' -o -name '*.jpg' -o -name '*.jpeg' -o -name '*.bmp'";
                Process.spawn_command_line_sync(cmd, out out0, out err0, null);
                if (err0 != null && err0 != "")
                    throw new SpawnError.FAILED(err0);
                foreach (string line in out0.split("\n")) {
                    if ("cover" in line)
                        break;
                    if (line != "")
                        imgfile = line;
                }
            } else {
                imgfile = Path.build_filename(Path.get_dirname(opf_file), href);
            }
            if (imgfile != null) {
                uint8[] buffer = new uint8[100000];
                FileUtils.get_data(imgfile, out buffer);
                imgdata2pngs(buffer, out png_s, out png_m);
            }
        } catch (GLib.Error e) {
            prerr("WARNING: can't get thumbnail for file '%s': %s\n", fs_fullpath, e.message);
            try {
                Process.spawn_command_line_sync("rm -rf " + tmpdir, null, null, null);
            } catch (GLib.Error e) { return; }
            return;
        }

        // delete temp directory
        try {
            Process.spawn_command_line_sync("rm -rf " + tmpdir, null, null, null);
        } catch (GLib.Error e) { return; }
    }
}


//////////////////////////////////////////////////////////////////////
///// Configuration
class FileType {
    public bool active = false;
    public string ext  = null; // without dot
    public int ntotal  = 0;
    public int nok     = 0;

    public FileType(string ext) {
        this.ext = ext;
    }
}

class Config {
    public bool help         = false;
    public bool verbose      = false;
    public bool log          = false;
    public bool thumbs_in_fs = false;
    public Posix.FILE log_file = null;
    public string curdir;
    public string sdcard;
    public string del_prefix;
    public string add_prefix;
    public int action;
    public FileType[] fts = new FileType[FILETYPES.length];

    public Config() {
        this.curdir = Environment.get_current_dir();
        this.sdcard = this.curdir;
        this.del_prefix = this.curdir;
        this.add_prefix = this.curdir;
        this.action = ACTION.REPORT;
        for (int i=0; i < FILETYPES.length; i++) {
            this.fts[i] = new FileType(FILETYPES[i]);
        }
    }
}


//////////////////////////////////////////////////////////////////////
///// MyApp
class MyApp {

    ///// Constants
    const string HELP = """
Manage books metadata for iRex DR series eReaders.
Copyright (C) 2011  Iñigo Serna <inigoserna@gmail.com>
Released under GPL v3+

Usage: %s [options] directory1 [directory2] [...]

    -h, --help            Display this help
    -v, --verbose         Show additional information on screen
    -l, --log             Write a log in file metadata-{timestamp}.log

    -fs, --thumbs-in-fs   Whether thumbs are stored on file system (default false, in global.db)
                          This option is needed if you are using DR800+ version 4 or higher

    --sdcard=/path/to     Path where to find SD card 'global.db' file and 'System' folder (default current directory, then /media/mmcblk0p1)
    --del-prefix=/path    Prefix path to be removed from the start of book path in database (default current directory)
    --add-prefix=/path    Prefix path to be added to the start of book path in database (default current directory)

    --types=TYPES         File types to analyze (default all: desktop, pdf, fb2, epub)
                          all
                          item1,item2,item3 (no spaces allowed between items)
                          item

    --action=ACTION       Action to perform (default report)
                          report: show book information
                          add: add new books to database
                          add-metadata: add new books (without thumbnails) to database
                          add-thumbs: add missing thumbs to database
                          overwrite: force update books metadata to database
                          overwrite-metadata: force update books metadata (no thumbnails) to database
                          overwrite-thumbs: force update books thumbs to database
                          update: add if new book or update
                          update-metadata: add if new book or update metadata
                          update-thumbs: add if new book or update thumbs
                          delete: remove entries from database
                          delete-thumbs: remove thumbs from database
                          vacuum: clean database

""";

    ///// Variables
    Config cfg;
    DBGlobal db;

    ///// Methods
    public MyApp() {
        this.cfg = new Config();
    }


    public void print_help(string progname) {
        GLib.stderr.printf(HELP, progname);
    }


    public List<string>? parse_args(string[] args) {
        var dirs = new List<string>();
        foreach(string arg in args) {
            if (arg=="") {
                continue;
            } else if (arg=="--help" || arg=="-h") {
                cfg.help = true;
            } else if (arg=="--verbose" || arg=="-v") {
                cfg.verbose = true;
            } else if (arg=="--log" || arg=="-l") {
                cfg.log = true;
                var fname = "./metadata-" + MyUtils.now2strftime("%Y%m%d%H%M%S") + ".log";
                cfg.log_file = Posix.FILE.open(fname, "w");
            } else if (arg=="--thumbs-in-fs" || arg=="-fs") {
                cfg.thumbs_in_fs = true;
            } else if (arg.length>=9 && arg[0:9]=="--sdcard=") {
                cfg.sdcard = normalize_path(arg.replace("--sdcard=", ""));
            } else if (arg.length>=13 && arg[0:13]=="--del-prefix=") {
                cfg.del_prefix = normalize_path(arg.replace("--del-prefix=", ""));
                if (!Path.is_absolute(cfg.del_prefix)) {
                    GLib.stderr.printf("Error: '%s' passed to --del-prefix should be an absolute path\n", cfg.del_prefix);
                    return null;
                }
            } else if (arg.length>=13 && arg[0:13]=="--add-prefix=") {
                cfg.add_prefix = normalize_path(arg.replace("--add-prefix=", ""));
                if (!Path.is_absolute(cfg.add_prefix)) {
                    GLib.stderr.printf("Error: '%s' passed to --add-prefix should be an absolute path\n", cfg.add_prefix);
                    return null;
                }
            } else if (arg.length>=8 && arg[0:8]=="--types=") {
                var fts = arg.replace("--types=", "").split(",");
                foreach (string ft in fts) {
                    if (ft == "all") {
                        foreach (FileType ft2 in cfg.fts) {
                            ft2.active = true;
                        }
                    } else {
                        var found = false;
                        for (int i=0; i<FILETYPES.length; i++) {
                            if (FILETYPES[i] == ft) {
                                cfg.fts[i].active = true;
                                found = true;
                                break;
                            }
                        }
                        if (!found) {
                            GLib.stderr.printf("Error: '%s' is not a valid file type\n", ft);
                            return null;
                        }
                    }
                }
            } else if (arg.length>=9 && arg[0:9]=="--action=") {
                var action = arg.replace("--action=", "");
                if (!(action in ACTIONS)) {
                    GLib.stderr.printf("Error: '%s' is not a valid action\n", action);
                    return null;
                }
                for (int i=0; i<ACTIONS.length; i++) {
                    if (ACTIONS[i] == action) {
                        cfg.action = i;
                        break;
                    }
                }
            } else if (arg.length>=1 && arg[0:1]=="-") {
                GLib.stderr.printf("Error: invalid argument <%s>\n", arg);
                return null;
            } else {
                if (!FileUtils.test(arg, FileTest.IS_DIR))
                    GLib.stderr.printf("Error: '%s' is not a directory\n", arg);
                else
                    dirs.append(normalize_path(arg));
            }
        }
        return dirs;
    }


    private void _check_filetypes() {
        bool any_filetype = false;
        foreach (FileType ft in cfg.fts) {
            any_filetype |= ft.active;
        }
        if (!any_filetype) {
            foreach (FileType ft in cfg.fts) {
                ft.active = true;
            }
        }
    }


    private string _build_find_fts_args() {
        string ret = "";
        for (int i=1; i<cfg.fts.length; i++) {
            if (cfg.fts[i].active)
                ret += (ret=="" ? "" : " -o") + " -name '*." + cfg.fts[i].ext + "'";
        }
        return ret;
    }


    private string _build_fts_list() {
        string ret = "";
        for (int i=1; i<cfg.fts.length; i++) {
            if (cfg.fts[i].active)
                ret += (ret=="" ? "": ", ") + cfg.fts[i].ext;
        }
        return ret;
    }


    private string _build_fts_results() {
        string ret = "";
        for (int i=1; i<cfg.fts.length; i++) {
            if (cfg.fts[i].active) {
                if (ret != "")
                    ret += ", ";
                ret += "%s: %d/%d".printf(cfg.fts[i].ext, cfg.fts[i].nok, cfg.fts[i].ntotal);
            }
        }
        return ret;
    }


    public void vacuum() {
        pr1("###########################################################################\n");
        pr1("########## METADATA  -  %s\n", MyUtils.now2strftime("%a, %d %B %Y  %X"));
        pr2("##### Vacuum database in '%s'\n", Path.build_filename(cfg.sdcard, DBFILE));
        try {
            db.vacuum();
        } catch (MyError e) {
            prerr("ERROR: vacuum: %s\n", e.message);
        }
        pr1("###########################################################################\n");
    }


    public void process(List<string> dirs) {
        pr1("###########################################################################\n");
        pr1("########## METADATA  -  %s\n", MyUtils.now2strftime("%a, %d %B %Y  %X"));
        pr1("###########################################################################\n");
        Timer timer = new GLib.Timer();
        var ntotal = 0;
        string cmd, out0, err0;
        string fts_args = _build_find_fts_args();
        foreach (string dir in dirs) {
            pr1("###########################################################################\n");
            pr2("##### Processing folder '%s'\n", dir);
            // cmd = "find " + Shell.quote(dir) + " -name '*.desktop' -o -name '*.pdf' -o -name '*.fb2' -o -name '*.epub'";
            cmd = "find " + Shell.quote(dir) + fts_args;
            try {
                Process.spawn_command_line_sync(cmd, out out0, out err0, null);
            } catch (GLib.SpawnError e) {
                prerr("ERROR: find <%s>: %s\n", dir, e.message);
                continue;
            }
            if (err0 != null && err0 != "") {
                prerr("ERROR: find <%s>: %s\n", dir, err0);
                continue;
            }
            string[] lines = out0.split("\n");
            pr2("##### action: %s for %d files (%s)\n", ACTIONS[cfg.action],
                lines.length, _build_fts_list());
            pr1("###########################################################################\n");
            foreach (string line in lines) {
                FileType ft;
                var ext = line[line.last_index_of(".")+1:line.length];
                if (line=="" || !(ext in FILETYPES))
                    continue;
                ntotal++;
                try {
                    BookBase item;
                    if (ext == FILETYPES[FILETYPE.DESKTOP]) {
                        ft = cfg.fts[FILETYPE.DESKTOP];
                        item = new Application(line);
                    } else if (ext == FILETYPES[FILETYPE.PDF]) {
                        ft = cfg.fts[FILETYPE.PDF];
                        item = new Pdf(line);
                    } else if (ext == FILETYPES[FILETYPE.FB2]) {
                        ft = cfg.fts[FILETYPE.FB2];
                        item = new Fb2(line);
                    } else if (ext == FILETYPES[FILETYPE.EPUB]) {
                        ft = cfg.fts[FILETYPE.EPUB];
                        item = new Epub(line);
                    } else
                        throw new MyError.INVALID_FILE("file type unknown");
                    ft.ntotal++;
                    db.check_for_entry(item);
                    if (cfg.log)
                        print("%2d%% ", 100*ntotal/lines.length);
                    item.print_shortdata();
                    if (cfg.action!=ACTION.DELETE && cfg.action!=ACTION.DELETE_THUMBS)
                        item.get_metadata();
                    switch (cfg.action) {
                        case ACTION.REPORT:
                            item.print_metadata();
                            break;
                        case ACTION.ADD:
                            db.add_entry_metadata(item);
                            try {
                                db.add_entry_missing_thumbs(item);
                            } catch (MyError e) {}
                            break;
                        case ACTION.ADD_METADATA:
                            db.add_entry_metadata(item);
                            break;
                        case ACTION.ADD_THUMBS:
                            db.add_entry_missing_thumbs(item);
                            break;
                        case ACTION.OVERWRITE:
                            db.update_entry_metadata(item);
                            try {
                                db.update_entry_thumbs(item);
                            } catch (MyError e) {}
                            break;
                        case ACTION.OVERWRITE_METADATA:
                            db.update_entry_metadata(item);
                            break;
                        case ACTION.OVERWRITE_THUMBS:
                            db.update_entry_thumbs(item);
                            break;
                        case ACTION.UPDATE:
                            try {
                                db.add_entry_metadata(item);
                            } catch (MyError e) {
                                db.update_entry_metadata(item);
                            }
                            try {
                                db.add_entry_missing_thumbs(item);
                            } catch (MyError e) {
                                try {
                                    db.update_entry_thumbs(item);
                                } catch (MyError e) {}
                            }
                            break;
                        case ACTION.UPDATE_METADATA:
                            try {
                                db.add_entry_metadata(item);
                            } catch (MyError e) {
                                db.update_entry_metadata(item);
                            }
                            break;
                        case ACTION.UPDATE_THUMBS:
                            try {
                                db.add_entry_missing_thumbs(item);
                            } catch (MyError e) {
                                db.update_entry_thumbs(item);
                            }
                            break;
                        case ACTION.DELETE:
                            try {
                                db.delete_thumbs(item);
                            } catch (MyError e) {}
                            db.delete_metadata(item);
                            break;
                        case ACTION.DELETE_THUMBS:
                            db.delete_thumbs(item);
                            break;
                        default:
                            throw new MyError.FAILED("invalid action");
                    }
                    ft.nok++;
                } catch (MyError e) {
                    var msg = (e.message.length>100) ? e.message[0:100] : e.message;
                    prerr("ERROR: %s: %s\n", line, msg.strip());
                }
            }
        }

        // finish
        timer.stop();
        pr1("###########################################################################\n");
        pr2("##### Processed %d entries (%s) in %.2f secs\n",
            ntotal, _build_fts_results(), timer.elapsed());
        pr1("###########################################################################\n");
    }


    public static int main(string[] args) {
        var app = new MyApp();
        var dirs = app.parse_args(args[1:args.length]);
        if (app.cfg.help || args.length==1) {
            app.print_help(args[0]);
            return 0;
        }
        var dbfile = Path.build_filename(app.cfg.sdcard, DBFILE);
        if (!FileUtils.test(dbfile, FileTest.IS_REGULAR)) {
            dbfile = Path.build_filename(SDCARD, DBFILE);
            if (!FileUtils.test(dbfile, FileTest.IS_REGULAR)) {
                GLib.stderr.printf("Error: Can't find global.db database in '%s' or in '%s'\n",
                                   app.cfg.sdcard, SDCARD);
                GLib.stderr.printf("Maybe you should specify --sdcard=/path/to option\n");
                app.print_help(args[0]);
                return -1;
            }
        }
        config = app.cfg; // make it global
        app.db = new DBGlobal();

        if (app.cfg.action == ACTION.VACUUM) {
            app.vacuum();
        } else {
            if (dirs==null || dirs.length()==0) {
                app.print_help(args[0]);
                return -1;
            }
            app._check_filetypes();
            app.process(dirs);
        }
        return 0;
    }
}


//////////////////////////////////////////////////////////////////////
/*
  TODO:

  IDEAS:
  - replace mdbindex?
    . we should manage every registered file and dirs
    . delete entries and dirs present in db but not in file system
  - a GUI tool
  - allow a different source to specify book metadata (title, author, file_size, tags)
  - add Mackx's "no-update" tag or many tags features to DR800+
  - handle more than one tag

  ISSUES & BUGS:
  - overwrite-metadata: update any other field?
  - does sort by last added work with books added with this program?
  - .epub and .fb2 books without images -> no thumbnail will be generated
  - .epub with error message "unzip: inflate error": unzip program in DR is old and can't uncompress the file
  - .fb2 books not saved with UTF8 encoding fail
 */
