#!/usr/bin/env python

__license__   = 'GPL v3'
__copyright__ = '2011, BurbleBurble <mobireads_forum> <NAMLEHMIARFE>'
__environment__ = 'calibre'

import os
import os.path
import shutil
import tempfile
import zipfile
import codecs
import sys
from PyQt4 import QtCore, QtGui, QtWebKit, Qt
import time
import re
from lxml import etree
import lxml.html

try:
    from calibre_plugins.ebook_cleaner import htmlz
except ImportError:
    import htmlz


def icon(name):
    if __environment__ == 'calibre':
        icon_ = get_icons('icons/' + name)
        return icon_
    elif __environment__ == 'pyscripter':
        icon_ = QtGui.QIcon(os.path.join('D:\\ECleaner\\Plugin\\icons', name))
        return icon_

def pyobject(qtobject):
    if __environment__ == 'calibre':
        if type(qtobject) == QtCore.QVariant:
            return str(qtobject.toString())
        else:
            return str(qtobject)
    elif __environment__ == 'pyscripter':
        return qtobject

class Main(QtGui.QDialog):
    def __init__(self, parent=None):
        #objects
        super(Main, self).__init__(parent)
        self.parent = parent
        self.setWindowTitle('Cleaner')
        self.setWindowIcon(icon('cleaner.png'))
        self.text = Text(self)
        self.navigation = Navigation(self)
        self.tabs = QtGui.QTabWidget(self)
        self.tabs.setIconSize(QtCore.QSize(64, 64))
        self.file = File(self)
        self.settings = Settings(self)
        self.structure = Structure(self)
        self.javascript = Javascript(self)
        #layout
        self.setLayout(QtGui.QGridLayout())
        self.layout().addWidget(self.navigation, 0, 0)
        self.layout().setColumnMinimumWidth(0, 150)
        self.layout().addWidget(self.text, 0, 1)
        self.layout().setRowMinimumHeight(0, 500)
        self.layout().setColumnMinimumWidth(1, 500)
        self.layout().addWidget(self.tabs, 1, 0, 1, 2)
        self.tabs.addTab(self.file, icon('file.png'), 'Open/Save')
        self.tabs.addTab(self.settings, icon('settings.png'), 'Settings')
        self.tabs.addTab(self.structure, icon('structure.png'), 'Structure')
        self.tabs.addTab(self.javascript, icon('javascript.png'), 'Javascript/JQuery')

class Text(QtWebKit.QWebView):
    def __init__(self, parent=None):
        super(Text, self).__init__(parent)
        self.page().mainFrame().javaScriptWindowObjectCleared.connect(self.slotLoadBasicJavaScript)

    def slotLoadBasicJavaScript(self):
        #javascript/python bridge
        self.page().mainFrame().addToJavaScriptWindowObject("jspyBridge", self)
        #jquery
        if __environment__ == 'calibre':
            self.page().mainFrame().evaluateJavaScript(P('content_server/jquery.js', data=True))
        else:
            with codecs.open('D:\\ECleaner\\Plugin\\jquery.js', 'r', 'utf-8') as jquery:
                self.page().mainFrame().evaluateJavaScript(jquery.read())

class Navigation(QtGui.QTreeView):
    def __init__(self, parent=None):
        #object
        super(Navigation, self).__init__(parent)
        self.parent = parent
        self.setRootIsDecorated(False)
        self.setAlternatingRowColors(True)
        self.setSortingEnabled(True)
        self.setModel(self.Model(self.parent.text))
        #signals
        self.clicked.connect(self.slotClicked)
        self.parent.text.loadFinished.connect(self.model().update)
        self.parent.text.page().contentsChanged.connect(self.model().update)

    def slotClicked(self, clickedIndex):
        clickedIndex.internalPointer().setAttribute('id', 'clicked')
        script ='''
                //make selection
                clicked = document.getElementById("clicked")
                clicked.removeAttribute("id")
                clickedrange = document.createRange()
                clickedrange.selectNode(clicked)
                selection = window.getSelection()
                selection.removeAllRanges()
                selection.addRange(clickedrange)
                //go to selection
                window.scrollTo(0, clicked.offsetTop - 200)
                true
                '''
        self.parent.text.page().mainFrame().evaluateJavaScript(script)

    class Model(QtCore.QAbstractItemModel):
        def __init__(self, text):
            super(Navigation.Model, self).__init__()
            self.text = text
            self.items = []
            self.columns = []
            self.showClasses = False
            self.showPatterns = True
            self.showStyles = False

        def columnCount(self, parentIndex):
            return len(self.columns)

        def data(self, itemIndex, role):
            if not itemIndex.isValid():
                return None
            elif role == QtCore.Qt.DisplayRole:
                return self.items[itemIndex.row()].attribute(self.columns[itemIndex.column()].lower())
            else:
                return None

        def headerData(self, section, orientation, role):
            if role == QtCore.Qt.DisplayRole:
                if orientation == QtCore.Qt.Horizontal:
                    return self.columns[section]
                else:
                    return None
            else:
                return None

        def index(self, row, column, parentIndex):
            if not self.hasIndex(row, column, parentIndex):
                return QtCore.QModelIndex()
            else:
                return self.createIndex(row, column, self.items[row])

        def parent(self, childIndex):
            #just a list, so all items are at the root
            return QtCore.QModelIndex()

        def rowCount(self, parentIndex):
            if not parentIndex.isValid():
                return len(self.items)
            else:
                #this is just a list, so there are no children
                return 0

        def update(self):
            selector = []
            self.columns = []
            if self.showClasses:
                selector.append('*[class]')
                self.columns.append('Class')
            if self.showPatterns:
                selector.append('*[pattern]')
                self.columns.append('Pattern')
            if self.showStyles:
                selector.append('*[style]')
                self.columns.append('Style')
            if selector:
                self.items = list(self.text.page().mainFrame().documentElement().findAll(','.join(selector)))
            else:
                self.items = []
            self.beginResetModel()
            self.endResetModel()

class Javascript(QtGui.QWidget):
    def __init__(self, parent=None):
        #objects
        super(Javascript, self).__init__(parent)
        self.parent = parent
        self.scriptGroup = QtGui.QGroupBox('Javascript/Jquery', self)
        self.script = QtGui.QTextEdit()
        self.script.setWordWrapMode(QtGui.QTextOption.NoWrap)
        self.runScript = QtGui.QPushButton('Run')
        self.resultGroup = QtGui.QGroupBox('Result')
        self.result = QtGui.QTextBrowser()
        self.result.setWordWrapMode(QtGui.QTextOption.NoWrap)
        #layout
        self.setLayout(QtGui.QHBoxLayout())
        self.layout().addWidget(self.scriptGroup)
        self.layout().addWidget(self.resultGroup)
        self.scriptGroup.setLayout(QtGui.QVBoxLayout())
        self.scriptGroup.layout().addWidget(self.script)
        self.scriptGroup.layout().addWidget(self.runScript)
        self.resultGroup.setLayout(QtGui.QVBoxLayout())
        self.resultGroup.layout().addWidget(self.result)
        #signals
        self.runScript.clicked.connect(self.slotRunScript)

    def slotRunScript(self):
        result = self.parent.text.page().mainFrame().evaluateJavaScript(self.script.toPlainText())
        if type(result) is bool or type(result) is float:
            self.result.setPlainText(str(result))
        else:
            self.result.setPlainText(pyobject(result))


class Settings(QtGui.QWidget):
    def __init__(self, parent=None):
        #objects
        super(Settings, self).__init__(parent)
        self.parent = parent
        self.textGroup = QtGui.QGroupBox('Text')
        self.editModeLabel = QtGui.QLabel(
            '<p>When automatic updating of the navigation view, etc., is activated, the editing speed slows to a painful pace. '
            'Therefore, to edit by hand, toggle the following option. It will turn on/off automatic updating and editing.</p>')
        self.editModeLabel.setWordWrap(True)
        self.editMode = QtGui.QPushButton('Edit Mode')
        self.editMode.setCheckable(True)
        self.navigationGroup = QtGui.QGroupBox('Navigation')
        self.navigationShowLabel = QtGui.QLabel('<p>Show:</p>')
        self.navigationShowGroup = QtGui.QButtonGroup()
        self.showClasses = QtGui.QRadioButton('Classes')
        self.showPatterns = QtGui.QRadioButton('Patterns')
        self.showPatterns.setChecked(True)
        self.showStyles = QtGui.QRadioButton('Styles')
        self.navigationShowGroup.addButton(self.showClasses)
        self.navigationShowGroup.addButton(self.showPatterns)
        self.navigationShowGroup.addButton(self.showStyles)
        #layout
        self.setLayout(QtGui.QHBoxLayout())
        self.layout().addWidget(self.textGroup)
        self.layout().addWidget(self.navigationGroup)
        self.textGroup.setLayout(QtGui.QVBoxLayout())
        self.textGroup.layout().addWidget(self.editModeLabel)
        self.textGroup.layout().addWidget(self.editMode)
        self.navigationGroup.setLayout(QtGui.QVBoxLayout())
        self.navigationGroup.layout().addWidget(self.navigationShowLabel)
        self.navigationGroup.layout().addWidget(self.showPatterns)
        self.navigationGroup.layout().addWidget(self.showClasses)
        self.navigationGroup.layout().addWidget(self.showStyles)
        #signals
        self.editMode.toggled.connect(self.slotEditMode)
        self.showClasses.toggled.connect(self.slotShowClasses)
        self.showPatterns.toggled.connect(self.slotShowPatterns)
        self.showStyles.toggled.connect(self.slotShowStyles)

    def slotEditMode(self, isChecked):
        if isChecked:
            self.parent.text.page().setContentEditable(True)
            self.parent.text.page().contentsChanged.disconnect(self.parent.navigation.model().update)
        else:
            self.parent.text.page().setContentEditable(False)
            self.parent.text.page().contentsChanged.connect(self.parent.navigation.model().update)
            self.parent.navigation.model().update()

    def slotShowClasses(self, isChecked):
        if isChecked:
            self.parent.navigation.model().showClasses = True
        else:
            self.parent.navigation.model().showClasses = False
        self.parent.navigation.model().update()

    def slotShowPatterns(self, isChecked):
        if isChecked:
            self.parent.navigation.model().showPatterns = True
        else:
            self.parent.navigation.model().showPatterns = False
        self.parent.navigation.model().update()

    def slotShowStyles(self, isChecked):
        if isChecked:
            self.parent.navigation.model().showStyles = True
        else:
            self.parent.navigation.model().showStyles = False
        self.parent.navigation.model().update()


class File(QtGui.QWidget):
    def __init__(self, parent=None):
        #objects
        super(File, self).__init__(parent)
        self.parent = parent
        self.tempdir = tempfile.mkdtemp(suffix = 'EbookCleaner', prefix='tmp')
        self.htmlz_object = None
        self.path_to_htmlz = ''
        self.openGroup = QtGui.QGroupBox('Open')
        self.openHtmlz = QtGui.QPushButton('Open Htmlz')
        self.openOptionsLabel = QtGui.QLabel(
            '<p>Note: This plugin\'s functionality was designed for htmlz\'s run through its own basic cleaning.'
            'Therefore, if this htmlz was not previously run through this plugin\'s basic cleaning, it is strongly recommended. '
            'Be aware though, it pays to eyeball the results before saving!</p>')
        self.openOptionsLabel.setWordWrap(True)
        self.openOptionBasicCleaning = QtGui.QCheckBox('Basic Cleaning')
        self.openOptionBasicCleaning.setChecked(True)
        self.openOptionDiscoverPatterns = QtGui.QCheckBox('Discover Patterns')
        self.openOptionDiscoverPatterns.setChecked(True)
        self.saveGroup = QtGui.QGroupBox('Save')
        self.saveHtmlzLabel = QtGui.QLabel(
            '<p>This will save the edited ebook to the same directory as the orignal, titled "cleaned.htmlz".'
            'It will the open the folder for your viewing.</p>')
        self.saveHtmlzLabel.setWordWrap(True)
        self.saveHtmlz = QtGui.QPushButton('Save Htmlz')
        self.saveEpub = QtGui.QPushButton('Save Epub')
        self.informationGroup = QtGui.QGroupBox('Information')
        self.information = QtGui.QTextBrowser()
        self.information.setWordWrapMode(QtGui.QTextOption.NoWrap)
        self.progressbar = QtGui.QProgressBar()
        self.progressbar.setTextVisible(False)
        self.progressbarLabel = QtGui.QLabel()
        self.progressbarLabel.setWordWrap(True)
        #layout
        self.setLayout(QtGui.QHBoxLayout())
        self.layout().addWidget(self.openGroup)
        self.layout().addWidget(self.saveGroup)
        self.layout().addWidget(self.informationGroup)
        self.openGroup.setLayout(QtGui.QVBoxLayout())
        self.openGroup.layout().addWidget(self.openOptionsLabel)
        self.openGroup.layout().addWidget(self.openHtmlz)
        self.openGroup.layout().addWidget(self.openOptionBasicCleaning)
        self.openGroup.layout().addWidget(self.openOptionDiscoverPatterns)
        self.saveGroup.setLayout(QtGui.QVBoxLayout())
        self.saveGroup.layout().addWidget(self.saveHtmlzLabel)
        self.saveGroup.layout().addWidget(self.saveHtmlz)
        self.saveGroup.layout().addWidget(self.saveEpub)
        self.informationGroup.setLayout(QtGui.QVBoxLayout())
        self.informationGroup.layout().addWidget(self.information)
        self.informationGroup.layout().addWidget(self.progressbar)
        self.informationGroup.layout().addWidget(self.progressbarLabel)
        #signals
        self.openHtmlz.clicked.connect(self.slotOpenHtmlz)
        self.saveHtmlz.clicked.connect(self.slotSaveHtmlz)
        self.saveEpub.clicked.connect(self.slotSaveEpub)
        #enable/disable
        self.saveHtmlz.setEnabled(False)
        self.saveEpub.setEnabled(False)

    def closeEvent(self, event):
        #cleanup tempdir
        shutil.rmtree(self.tempdir)

    def slotOpenHtmlz(self):
        #activate progressbar
        self.progressbar.setRange(0, 10)
        #get path to htmlz
        self.progressbarLabel.setText('<p>Retrieving path to htmlz...<p>')
        self.progressbar.setValue(1)
        if __environment__ == 'calibre':
            book_id = self.parent.parent.library_view.model().id(self.parent.parent.library_view.currentIndex())
            self.path_to_htmlz = self.parent.parent.library_view.model().db.format_abspath(book_id, 'HTMLZ', index_is_id=True)
        elif __environment__ == 'pyscripter':
            self.path_to_htmlz = 'D:\\ECleaner\\Test Files\\TestB.htmlz'
        #read htmlz
        self.progressbarLabel.setText('<p>Reading htmlz...<p>')
        self.progressbar.setValue(2)
        reader = htmlz.Reader()
        self.htmlz_object = reader.read(self.path_to_htmlz, self.tempdir)
        #basic cleaning
        if self.openOptionBasicCleaning.isChecked():
            tools = htmlz.Tools(self.information)
            self.progressbarLabel.setText('<p>Parsing html...<p>')
            self.progressbar.setValue(3)
            etree.parse(os.path.join(self.htmlz_object.basepath, 'index.html'), etree.HTMLParser(target=tools))
            self.progressbarLabel.setText('<p>Removing extra whitespace...<p>')
            self.progressbar.setValue(4)
            tools.whitespace()
            self.progressbarLabel.setText('<p>Merging spans with same styles...<p>')
            self.progressbar.setValue(5)
            tools.merge()
            self.progressbarLabel.setText('<p>Upgrading block wide styles to the block level...<p>')
            self.progressbar.setValue(6)
            tools.upgrade()
            if self.openOptionDiscoverPatterns.isChecked():
                self.progressbarLabel.setText('<p>Discovering patterns...<p>')
                self.progressbar.setValue(7)
                tools.pattern()
            self.progressbarLabel.setText('<p>Rewriting html...<p>')
            self.progressbar.setValue(8)
            html = tools.write()
        else:
            self.progressbarLabel.setText('<p>Parsing html...<p>')
            self.progressbar.setValue(3)
            with codecs.open(os.path.join(self.htmlz_object.basepath, 'index.html'), 'r', 'utf-8') as html_file:
                html = html_file.read()
        #display ebook
        self.progressbarLabel.setText('<p>Displaying ebook...<p>')
        self.progressbar.setValue(9)
        self.parent.text.setContent(html.encode(), 'text/html', QtCore.QUrl().fromLocalFile(os.path.join(self.htmlz_object.basepath, 'index.html')))
        #enable/disable
        self.saveHtmlz.setEnabled(True)
        self.saveEpub.setEnabled(False)
        self.openHtmlz.setEnabled(False)
        #deactivate progressbar
        self.progressbarLabel.setText('')
        self.progressbar.reset()

    def slotSaveHtmlz(self):
        with codecs.open(os.path.join(self.htmlz_object.basepath, 'index.html'), 'w', 'utf-8') as html_file:
            html_file.write(pyobject(self.parent.text.page().mainFrame().toHtml()))
        writer = htmlz.Writer()
        writer.write(self.htmlz_object, os.path.join(os.path.dirname(self.path_to_htmlz), 'cleaned.htmlz'))
        os.startfile(os.path.dirname(self.path_to_htmlz))

    def slotSaveEpub(self):
        pass

class Structure(QtGui.QWidget):
    def __init__(self, parent=None):
        #objects
        super(Structure, self).__init__(parent)
        self.parent = parent
        self.replaceGroup = QtGui.QGroupBox('Replace')
        self.selectedAttributeLabel = QtGui.QLabel("<p>Attribute:</p>")
        self.selectedAttributeLabel.setWordWrap(True)
        self.selectedAttribute = QtGui.QLineEdit()
        self.selectedValueLabel = QtGui.QLabel("<p>Value:</p>")
        self.selectedValueLabel.setWordWrap(True)
        self.selectedValue = QtGui.QLineEdit()
        self.replacementAttributeLabel = QtGui.QLabel("<p>Replacment:</p>")
        self.replacementAttributeLabel.setWordWrap(True)
        self.replacementAttribute = QtGui.QComboBox()
        self.replacementAttribute.addItems(['', 'class', 'style'])
        self.replacementValueLabel = QtGui.QLabel("<p>Replacement:</p>")
        self.replacementValueLabel.setWordWrap(True)
        self.replacementValue = QtGui.QComboBox()
        self.replacementValue.setEditable(True)
        self.replaceSelected = QtGui.QPushButton('Replace Selected')
        self.replaceForward = QtGui.QPushButton('Replace Forward')
        self.replaceAll = QtGui.QPushButton('Replace All')
        #layout
        self.setLayout(QtGui.QHBoxLayout())
        self.layout().addWidget(self.replaceGroup)
        self.replaceGroup.setLayout(QtGui.QGridLayout())
        self.replaceGroup.layout().addWidget(self.selectedAttributeLabel, 0, 0)
        self.replaceGroup.layout().addWidget(self.selectedAttribute, 1, 0)
        self.replaceGroup.layout().addWidget(self.selectedValueLabel, 2, 0)
        self.replaceGroup.layout().addWidget(self.selectedValue, 3, 0)
        self.replaceGroup.layout().addWidget(self.replacementAttributeLabel, 0, 1)
        self.replaceGroup.layout().addWidget(self.replacementAttribute, 1, 1)
        self.replaceGroup.layout().addWidget(self.replacementValueLabel, 2, 1)
        self.replaceGroup.layout().addWidget(self.replacementValue, 3, 1)
        self.replaceGroup.layout().addWidget(self.replaceSelected, 4, 0, 1, 2)
        self.replaceGroup.layout().addWidget(self.replaceForward, 5, 0, 1, 2)
        self.replaceGroup.layout().addWidget(self.replaceAll, 6, 0, 1, 2)
        #signals
        self.parent.navigation.clicked.connect(self.slotNavigationClicked)
        self.replaceSelected.clicked.connect(self.slotReplaceSelected)
        self.replaceForward.clicked.connect(self.slotReplaceForward)
        self.replaceAll.clicked.connect(self.slotReplaceAll)

    def slotNavigationClicked(self, clickedIndex):
        attribute = self.parent.navigation.model().columns[0].lower()
        value = clickedIndex.data()
        self.selectedAttribute.setText(pyobject(attribute))
        self.selectedValue.setText(pyobject(value))

    def slotReplaceAll(self):
        selectedAttribute = pyobject(self.selectedAttribute.text())
        selectedValue = pyobject(self.selectedValue.text())
        replacementAttribute = pyobject(self.replacementAttribute.currentText())
        replacementValue = pyobject(self.replacementValue.currentText())
        if not replacementAttribute or not replacementValue:
            script ='''
                    selectedAttribute = '%s'
                    selectedValue = '%s'
                    $('['+selectedAttribute+'="'+selectedValue+'"]').removeAttr(selectedAttribute)
                    true
                    ''' %(selectedAttribute,selectedValue)
        elif not selectedAttribute == replacementAttribute:
            script ='''
                    selectedAttribute = '%s'
                    selectedValue = '%s'
                    replacementAttribute = '%s'
                    replacementValue = '%s'
                    $matches = $('['+selectedAttribute+'="'+selectedValue+'"]')
                    $matches.removeAttr(selectedAttribute)
                    $matches.attr(replacementAttribute,replacementValue)
                    true
                    ''' %(selectedAttribute,selectedValue,replacementAttribute,replacementValue)
        else:
            script ='''
                    selectedAttribute = '%s'
                    selectedValue = '%s'
                    replacementValue = '%s'
                    $('['+selectedAttribute+'="'+selectedValue+'"]').attr(selectedAttribute,replacementValue)
                    true
                    ''' %(selectedAttribute,selectedValue,replacementValue)
        self.parent.text.page().mainFrame().evaluateJavaScript(script)
        self.parent.text.page().contentsChanged.emit()

    def slotReplaceForward(self):
        currentIndex = self.parent.navigation.currentIndex()
        currentIndex.internalPointer().setAttribute('id', 'target')
        selectedAttribute = pyobject(self.selectedAttribute.text())
        selectedValue = pyobject(self.selectedValue.text())
        replacementAttribute = pyobject(self.replacementAttribute.currentText())
        replacementValue = pyobject(self.replacementValue.currentText())
        if not replacementAttribute or not replacementValue:
            script ='''
                    start = document.getElementById("target")
                    start.removeAttribute("id")
                    selectedAttribute = '%s'
                    selectedValue = '%s'
                    $matches = $('['+selectedAttribute+'="'+selectedValue+'"]')
                    $matches.slice($matches.index(start)).removeAttr(selectedAttribute)
                    true
                    ''' %(selectedAttribute,selectedValue)
        elif not selectedAttribute == replacementAttribute:
            script ='''
                    start = document.getElementById("target")
                    start.removeAttribute("id")
                    selectedAttribute = '%s'
                    selectedValue = '%s'
                    replacementAttribute = '%s'
                    replacementValue = '%s'
                    $matches = $('['+selectedAttribute+'="'+selectedValue+'"]')
                    $matches = $matches.slice($matches.index(start))
                    $matches.removeAttr(selectedAttribute)
                    $matches.attr(replacementAttribute,replacementValue)
                    true
                    ''' %(selectedAttribute,selectedValue,replacementAttribute,replacementValue)
        else:
            script ='''
                    start = document.getElementById("target")
                    start.removeAttribute("id")
                    selectedAttribute = '%s'
                    selectedValue = '%s'
                    replacementValue = '%s'
                    $matches = $('['+selectedAttribute+'="'+selectedValue+'"]')
                    $matches.slice($matches.index(start)).attr(selectedAttribute,replacementValue)
                    true
                    ''' %(selectedAttribute,selectedValue,replacementValue)
        self.parent.text.page().mainFrame().evaluateJavaScript(script)
        self.parent.text.page().contentsChanged.emit()

    def slotReplaceSelected(self):
        currentIndex = self.parent.navigation.currentIndex()
        currentIndex.internalPointer().setAttribute('id', 'target')
        selectedAttribute = pyobject(self.selectedAttribute.text())
        selectedValue = pyobject(self.selectedValue.text())
        replacementAttribute = pyobject(self.replacementAttribute.currentText())
        replacementValue = pyobject(self.replacementValue.currentText())
        if not replacementAttribute or not replacementValue:
            script ='''
                    start = document.getElementById("target")
                    start.removeAttribute("id")
                    selectedAttribute = '%s'
                    $(start).removeAttr(selectedAttribute)
                    true
                    ''' %(selectedAttribute)
        elif not selectedAttribute == replacementAttribute:
            script ='''
                    start = document.getElementById("target")
                    start.removeAttribute("id")
                    selectedAttribute = '%s'
                    replacementAttribute = '%s'
                    replacementValue = '%s'
                    $match = $(start)
                    $match.removeAttr(selectedAttribute)
                    $match.attr(replacementAttribute,replacementValue)
                    true
                    ''' %(selectedAttribute,replacementAttribute,replacementValue)
        else:
            script ='''
                    start = document.getElementById("target")
                    start.removeAttribute("id")
                    selectedAttribute = '%s'
                    replacementValue = '%s'
                    $(start).attr(selectedAttribute,replacementValue)
                    true
                    ''' %(selectedAttribute,replacementValue)
        self.parent.text.page().mainFrame().evaluateJavaScript(script)
        self.parent.text.page().contentsChanged.emit()


def main():
    #set environment
    global __environment__
    __environment__ = 'pyscripter'
    #launch plugin
    app = QtGui.QApplication(sys.argv)
    root = Main()
    root.show()
    sys.exit(app.exec_())

if __name__ == '__main__':
    main()
