#!/usr/bin/env python

__license__   = 'GPL v3'
__copyright__ = '2011, BurbleBurble <mobireads_forum> <NAMLEHMIARFE>'

import os
import os.path
import shutil
import tempfile
import zipfile
import codecs
import sys
from PyQt4 import QtCore, QtGui, QtWebKit, Qt
import time
import re
from lxml import etree
import lxml.html

try:
    #calibre environment
    from calibre_plugins.ebook_cleaner import wizards
    from calibre_plugins.ebook_cleaner import epub
except ImportError:
    #python environment
    import epub
    import wizards


class Configuration():

    #standalone configuration
    standalone_jquery = 'D:\\ECleaner\\Plugin\\jquery.js'
    standalone_icons = 'D:\\ECleaner\\Plugin\\icons'

    #tempdir
    tempdir = None



    style_group = ['Font', 'Margin', 'Text']

    #-Template: '':{'values':[], 'group':'', 'level':[], 'default':''}
    #values: given in regex form (for decimal, type DECIMAL, for positive only, type PDECIMAL)     (  [-]{0,1}[0-9]*[.]{0,1}[0-9]+   )
    #group: what group to place it in, in the advanced formatting display
    #level: what element level it can exist on (ex: align only works for p and larger; assumes spans are always smaller than p)
    #default: default value rendered when no value specified; based on 'http://w3schools.com/CSS/css_reference.asp'; is ignored in parser when not negating an inherited value
    #(regex_values: created by parser at time of initiation)
    #(regex_default: created by parser at time of initiation)
    #friendly_name: name to be displayed in advanced css_selector
    style_names = {

                'font-size':        {'values':['', 'XX-Small', 'X-Small','Small','Medium','Large','X-Large','XX-Large','Smaller','Larger', '#', '#cm', '#em', '#pt', '#px', '#%'],
                                     'group':'Font',
                                     'level':['p','span'],
                                     'friendly_name':'Size',
                                     'javascript_name':'fontSize',
                                     'default':['', 'Medium', '1.0em']},
                'font-style':       {'values':['', 'Normal', 'Italic', 'Oblique'],
                                     'group':'Font',
                                     'level':['p','span'],
                                     'friendly_name':'Style',
                                     'javascript_name':'fontStyle',
                                     'default':['Normal', '']},
                'font-variant':     {'values':['', 'Normal', 'Small-Caps'],
                                     'group':'Font',
                                     'level':['p','span'],
                                     'friendly_name':'Variant',
                                     'javascript_name':'fontVariant',
                                     'default':['Normal', '']},
                'font-weight':      {'values':['', 'Normal','Bold','Bolder','Lighter','100','200','300','400','500','600','700','800','900'], #400=Normal, 700=Bold
                                     'group':'Font',
                                     'level':['p','span'],
                                     'friendly_name':'Weight',
                                     'javascript_name':'fontWeight',
                                     'default':['Normal', '']},
                'text-indent':      {'values':['', '#', '#cm', '#em', '#pt', '#px', '#%'],
                                     'group':'Text',
                                     'level':['p'],
                                     'friendly_name':'Indent',
                                     'javascript_name':'textIndent',
                                     'default':['0.0', '']},
                'text-align':       {'values':['', 'Left', 'Right', 'Center', 'Justify'],
                                     'group':'Text',
                                     'level':['p'],
                                     'friendly_name':'Align',
                                     'javascript_name':'textAlign',
                                     'default':['']},
                'text-decoration':  {'values':['', 'None', 'Underline', 'Overline', 'Line-Through'],
                                     'group':'Text',
                                     'level':['p','span'],
                                     'friendly_name':'Decoration',
                                     'javascript_name':'textDecoration',
                                     'default':['None', '']},
                'text-transform':   {'values':['', 'None', 'Capitalize', 'Uppercase', 'Lowercase'],
                                     'group':'Text',
                                     'level':['p','span'],
                                     'friendly_name':'Transform',
                                     'javascript_name':'textTransform',
                                     'default':['None', '']},
                'margin-left':      {'values':['', 'Auto', '#', '#cm', '#em', '#pt', '#px', '#%'],
                                     'group':'Margin',
                                     'level':['p'],
                                     'friendly_name':'Left',
                                     'javascript_name':'marginLeft',
                                     'default':['0.0','0.0px', '']},
                'margin-right':     {'values':['', 'Auto', '#', '#cm', '#em', '#pt', '#px', '#%'],
                                     'group':'Margin',
                                     'level':['p'],
                                     'friendly_name':'Right',
                                     'javascript_name':'marginRight',
                                     'default':['0.0','0.0px', '']},
                'margin-top':       {'values':['', 'Auto', '#', '#cm', '#em', '#pt', '#px', '#%'],
                                     'group':'Margin',
                                     'level':['p'],
                                     'friendly_name':'Top',
                                     'javascript_name':'marginTop',
                                     'default':['0.0','0.0px', '']},
                'margin-bottom':    {'values':['', 'Auto', '#', '#cm', '#em', '#pt', '#px', '#%'],
                                     'group':'Margin',
                                     'level':['p'],
                                     'friendly_name':'Bottom',
                                     'javascript_name':'marginBottom',
                                     'default':['0.0','0.0px', '']}
                }

    #-Template: ('', ''):{'in_basic':''},
    #in_basic: translate for basic formatting display
    #kindle: translate for kindle.....
    style_values = {
                     ('font-size', 'XX-Small'):     {'in_basic':''},
                     ('font-size', 'X-Small'):      {'in_basic':''},
                     ('font-size', 'Small'):        {'in_basic':''},
                     ('font-size', 'Medium'):       {'in_basic':''},
                     ('font-size', 'Large'):        {'in_basic':''},
                     ('font-size', 'X-Large'):      {'in_basic':''},
                     ('font-size', 'XX-Large'):     {'in_basic':''},
                     ('font-size', 'Smaller'):      {'in_basic':''},
                     ('font-size', 'Larger'):       {'in_basic':''},
                     ('font-size', '#'):      {'in_basic':''},
                     ('font-size', '#cm'):    {'in_basic':''},
                     ('font-size', '#em'):    {'in_basic':''},
                     ('font-size', '#pt'):    {'in_basic':''},
                     ('font-size', '#px'):    {'in_basic':''},
                     ('font-size', '#%'):     {'in_basic':''},

                     ('font-style', 'Normal'):  {'in_basic':''},
                     ('font-style', 'Italic'):  {'in_basic':''},
                     ('font-style', 'Oblique'): {'in_basic':''},

                     ('font-variant', 'Normal'):        {'in_basic':''},
                     ('font-variant', 'Small-Caps'):    {'in_basic':''},

                     ('font-weight', 'Normal'):     {'in_basic':''},
                     ('font-weight', 'Bold'):       {'in_basic':''},
                     ('font-weight', 'Bolder'):     {'in_basic':''},
                     ('font-weight', 'Lighter'):    {'in_basic':''},
                     ('font-weight', '100'):        {'in_basic':''},
                     ('font-weight', '200'):        {'in_basic':''},
                     ('font-weight', '300'):        {'in_basic':''},
                     ('font-weight', '400'):        {'in_basic':''},
                     ('font-weight', '500'):        {'in_basic':''},
                     ('font-weight', '600'):        {'in_basic':''},
                     ('font-weight', '700'):        {'in_basic':''},
                     ('font-weight', '800'):        {'in_basic':''},
                     ('font-weight', '900'):        {'in_basic':''},


                     ('text-indent', '#'):    {'in_basic':''},
                     ('text-indent', '#cm'):  {'in_basic':''},
                     ('text-indent', '#em'):  {'in_basic':''},
                     ('text-indent', '#pt'):  {'in_basic':''},
                     ('text-indent', '#px'):  {'in_basic':''},
                     ('text-indent', '#%'):   {'in_basic':''},

                     ('text-align', 'Left'):    {'in_basic':''},
                     ('text-align', 'Right'):   {'in_basic':''},
                     ('text-align', 'Center'):  {'in_basic':''},
                     ('text-align', 'Justify'): {'in_basic':''},

                     ('text-decoration', 'None'):           {'in_basic':''},
                     ('text-decoration', 'Underline'):      {'in_basic':''},
                     ('text-decoration', 'Overline'):       {'in_basic':''},
                     ('text-decoration', 'Line-Through'):   {'in_basic':''},

                     ('text-transform', 'None'):        {'in_basic':''},
                     ('text-transform', 'Capitalize'):  {'in_basic':''},
                     ('text-transform', 'Uppercase'):   {'in_basic':''},
                     ('text-transform', 'Lowercase'):   {'in_basic':''},

                     ('margin-left', 'Auto'):       {'in_basic':''},
                     ('margin-left', '#'):    {'in_basic':''},
                     ('margin-left', '#cm'):  {'in_basic':''},
                     ('margin-left', '#em'):  {'in_basic':''},
                     ('margin-left', '#pt'):  {'in_basic':''},
                     ('margin-left', '#px'):  {'in_basic':''},
                     ('margin-left', '#%'):   {'in_basic':''},

                     ('margin-right', 'Auto'):       {'in_basic':''},
                     ('margin-right', '#'):    {'in_basic':''},
                     ('margin-right', '#cm'):  {'in_basic':''},
                     ('margin-right', '#em'):  {'in_basic':''},
                     ('margin-right', '#pt'):  {'in_basic':''},
                     ('margin-right', '#px'):  {'in_basic':''},
                     ('margin-right', '#%'):   {'in_basic':''},

                     ('margin-top', 'Auto'):       {'in_basic':''},
                     ('margin-top', '#'):    {'in_basic':''},
                     ('margin-top', '#cm'):  {'in_basic':''},
                     ('margin-top', '#em'):  {'in_basic':''},
                     ('margin-top', '#pt'):  {'in_basic':''},
                     ('margin-top', '#px'):  {'in_basic':''},
                     ('margin-top', '#%'):   {'in_basic':''},

                     ('margin-bottom', 'Auto'):       {'in_basic':''},
                     ('margin-bottom', '#'):    {'in_basic':''},
                     ('margin-bottom', '#cm'):  {'in_basic':''},
                     ('margin-bottom', '#em'):  {'in_basic':''},
                     ('margin-bottom', '#pt'):  {'in_basic':''},
                     ('margin-bottom', '#px'):  {'in_basic':''},
                     ('margin-bottom', '#%'):   {'in_basic':''},
                     }

    #used to translate tags into css styles
    #Template: '':{'property':'', 'value':''}
    tags = {'a':{'transform':None, 'isStyle':False},
                     'b':{'transform':None, 'isStyle':True, 'styleName':'font-weight', 'styleValue':'bold'},
                     'blockquote':{'transform':'p', 'isStyle':False,},
                     'body':{'transform':None, 'isStyle':False},
                     'br':{'transform':None, 'isStyle':False},  #ends last block and starts new one...
                     'div':{'transform':None, 'isStyle':False},
                     'h1':{'transform':'p', 'isStyle':False},
                     'h2':{'transform':'p', 'isStyle':False},
                     'h3':{'transform':'p', 'isStyle':False},
                     'h4':{'transform':'p', 'isStyle':False},
                     'h5':{'transform':'p', 'isStyle':False},
                     'h6':{'transform':'p', 'isStyle':False},
                     'head':{'transform':None, 'isStyle':False},
                     'html':{'transform':None, 'isStyle':False},
                     'i':{'transform':None, 'isStyle':True, 'styleName':'font-style', 'styleValue':'italic'},
                     'img':{'transform':None, 'isStyle':False},
                     #'li':{'transform':'li', 'isStyle':False},
                     'link':{'transform':None, 'isStyle':False},
                     'meta':{'transform':None, 'isStyle':False},
                     #'ol':{'transform':'ol', 'isStyle':False},
                     'p':{'transform':'p', 'isStyle':False},
                     'span':{'transform':None, 'isStyle':False},
                     'style':{'transform':None, 'isStyle':False},
                     #'sub':{'transform':'sub', 'isStyle':False},
                     #'sup':{'transform':'sup', 'isStyle':False},
                     #'table':{'transform':'table', 'isStyle':False},
                     #'tbody':{'transform':'tbody', 'isStyle':False},
                     'title':{'transform':None, 'isStyle':False},
                     #'tr':{'transform':'tr', 'isStyle':False},
                     #'td':{'transform':'td', 'isStyle':False},
                     'u':{'transform':None, 'isStyle':True, 'styleName':'text-decoration', 'styleValue':'underline'},
                     #'ul':{'transform':'ul', 'isStyle':False},
                     }

class Plugboard():
    '''
    Provides interface for interactions that are (potentially) dependent on the enviroment.
    '''
    __environment__ = None
    __path_to_tempdir__ = None

    def __init__(self):
        try:
            import calibre
            self.__environment__ = 'calibre'
        except ImportError:
            self.__environment__ = 'python'

    def icon(self, icon):
        if self.__environment__ == 'calibre':
            icon = get_icons('icons/' + icon)
            return icon
        else:
            icon = QtGui.QIcon(os.path.join(Configuration.standalone_icons, icon))
            return icon

    def jquery(self):
        if self.__environment__ == 'calibre':
            jquery = P('content_server/jquery.js', data=True)
            return jquery
        else:
            with codecs.open(Configuration.standalone_jquery, 'r', 'utf-8') as jquery:
                return jquery.read()

    def python(self, object_, python_):
        '''
        Calibre (last checked v0.8.5) doesn't implicitly convert QtObjects to python.
        Standalone enviroment with latest Python\PyQt often does...
        '''
        if type(object_) == QtCore.QVariant:
            if python_ == str:
                return str(object_.toString())
        #latest version of pyqt has no QString
        elif hasattr(QtCore, 'QString') and type(object_) == QtCore.QString:
            if python_ == str:
                return str(object_)
        elif type(object_) == str and python_ == str:
            return object_


plugboard = Plugboard()

class Debugger(QtGui.QWidget):
    def __init__(self, parent=None):
        #self
        super(Debugger, self).__init__(parent)
        self.setLayout(QtGui.QGridLayout())
        #messages group box
        self.messagesGroup = QtGui.QGroupBox('Message Log', self)
        self.messagesGroup.setLayout(QtGui.QGridLayout())
        self.layout().addWidget(self.messagesGroup, 0, 0)
        #messages text
        self.messages = QtGui.QTextBrowser(self.messagesGroup)
        self.messages.setWordWrapMode(QtGui.QTextOption.NoWrap)
        self.messagesGroup.layout().addWidget(self.messages, 0, 0)
        #messages cursor
        self.messagesCursor = self.messages.textCursor()
        self.messagesCursor.movePosition(QtGui.QTextCursor.Start)

    def reportDetail(self, sender, detail_category, detail):
        self.messagesCursor.insertText('[' + detail_category+ '] ' + detail + '\n')


class Main(QtGui.QDialog):
    def __init__(self, parent=None, path_to_epub=None):
        #configure
        self.configure()
        #self
        super(Main, self).__init__(parent)
        self.setWindowTitle('Ebook Cleaner')
        self.setWindowIcon(plugboard.icon('icon.png'))
        self.setLayout(QtGui.QGridLayout())
        #text
        self.text = Text(self)
        self.layout().addWidget(self.text, 0, 1)
        self.layout().setColumnMinimumWidth(1, 500)
        self.layout().setRowMinimumHeight(0, 500)
        #tree
        self.tree = Tree(self.text, self)
        self.layout().addWidget(self.tree, 0, 0)
        self.layout().setColumnMinimumWidth(0, 150)
        #debugger
        self.debugger = debugger
        #format
        self.format = Format(self.text, self)
        #javascript
        self.javascript = Javascript(self.text, self)
        #OpenAndSave
        self.openAndSave = OpenAndSave(self.text, path_to_epub, self)
        #structure
        self.structure = Structure(self.text, self)
        #wizards
        self.wizards = QtGui.QTabWidget(self)
        self.wizards.setIconSize(QtCore.QSize(64, 64))
        self.wizards.addTab(self.openAndSave, plugboard.icon('file.png'), 'Open/Save')
        self.wizards.addTab(self.structure, plugboard.icon('find_replace.png'), 'Structure')
        self.wizards.addTab(self.format, plugboard.icon('format.png'), 'Format')
        self.wizards.addTab(self.javascript, plugboard.icon('javascript.png'), 'Javascript/Jquery')
        self.wizards.addTab(self.debugger, plugboard.icon('debugger.png'), 'Debugger')
        self.layout().addWidget(self.wizards, 1, 0, 1, 2)

    def closeEvent(self, event):
        #cleanup tempdir
        debugger.reportDetail(self, 'LOG', 'Performing cleanup')
        shutil.rmtree(Configuration.tempdir)

    def configure(self):
        #global
        #global plugboard
        global debugger
        #plugboard = Plugboard()
        debugger = Debugger()

        #tempdir
        Configuration.tempdir = tempfile.mkdtemp(suffix = 'EbookCleaner', prefix='tmp')

        #setup regex_values
        for name in Configuration.style_names:
            Configuration.style_names[name]['regex_values'] = '|'.join( '(?:' + (  #non saving match
                                                    '[-]{0,1}[0-9]*[.]{0,1}[0-9]+' + value[1:] if value.startswith('#')  #decimal based match
                                                     else value #normal match
                                                     )+'$)' for value in Configuration.style_names[name]['values']).lower() #end of line...
        #setup regex_default
        for name in Configuration.style_names:
            Configuration.style_names[name]['regex_default'] = '|'.join( '(?:' + (  #non saving match
                                                     value #normal match
                                                     )+'$)' for value in Configuration.style_names[name]['default']).lower() #end of line...


class Text(QtWebKit.QWebView):
    def __init__(self, parent=None):
        #self
        super(Text, self).__init__(parent)
        #page
        self.page().setContentEditable(True)
        #main frame
        self.page().mainFrame().javaScriptWindowObjectCleared.connect(self.slotLoadBasicJavaScript)
        #epub
        self.path_to_epub = None

    @Qt.pyqtSignature('')
    def signalHighlightChanged(self):
        self.emit(QtCore.SIGNAL("signalHighlightChanged()"))

    def slotLoadBasicJavaScript(self):
        #report
        debugger.reportDetail(self, 'LOG', 'Loading  javascript/python bridge...')
        #javascript/python bridge
        self.page().mainFrame().addToJavaScriptWindowObject("jspyBridge", self)
        #report
        debugger.reportDetail(self, 'LOG', 'Loading jquery...')
        #jquery
        self.page().mainFrame().evaluateJavaScript(plugboard.jquery())
        #report
        debugger.reportDetail(self, 'LOG', 'Loading basic javasript...')
        #script to move highlight
        script = '''function moveHighlightTo(next)
                 {
                 if(next)
                    {
                     current = document.getElementById("highlight")
                     if(current)
                     {
                        current.removeAttribute("style")
                        current.removeAttribute("id")
                     }
                     next.setAttribute("id", "highlight")
                     next.setAttribute("style", "background:#9c9;")
                     window.scrollTo(0, next.offsetTop - 200)
                     jspyBridge.signalHighlightChanged()
                    }
                 }
                 true
                 '''
        self.page().mainFrame().evaluateJavaScript(script)
        #script to highlight clicked element
        script = '''document.onclick = function highlightClickedElement(event)
                 {
                 if(event.target.tagName != "BODY" && event.target.tagName != "HTML")
                    {
                    moveHighlightTo(event.target)
                    }
                 }
                 true
                 '''
        self.page().mainFrame().evaluateJavaScript(script)


class Tree(QtGui.QTreeView):
    def __init__(self, text,  parent=None):
        #self
        super(Tree, self).__init__(parent)
        self.setAlternatingRowColors(True)
        self.setModel(TreeModel(text))
        self.just_clicked= False
        self.clicked.connect(self.slotClicked)
        #text
        self.text = text
        self.text.connect(self.text, QtCore.SIGNAL("signalHighlightChanged()"), self.slotHighlightChanged)
        self.text.connect(self.text, QtCore.SIGNAL("loadFinished(bool)"), self.model().update)
        #page
        self.text.page().connect(self.text.page(), QtCore.SIGNAL("contentsChanged()"), self.model().update)

    def slotClicked(self, clickedIndex):
        #clicked
        clickedIndex.internalPointer().element.setAttribute('id', 'target')
        #so not to run slotHighlightChanged because of this change:
        self.just_clicked = True
        #highlight
        script = '''
                 next = document.getElementById("target")
                 next.removeAttribute("id")
                 moveHighlightTo(next)
                 true
                 '''
        self.text.page().mainFrame().evaluateJavaScript(script)

    def slotHighlightChanged(self):
        if self.just_clicked:
            self.just_clicked = False
        else:
            #if highlighted element is in structureview, set as current item
            highlight = self.text.page().mainFrame().documentElement().findFirst('*[id="highlight"]')
            if highlight in self.model().indexMap:
                self.setCurrentIndex(self.model().indexMap[self.model().indexMap.index(highlight) + 1])

class TreeModel(QtCore.QAbstractItemModel):
    def __init__(self, text, parent=None):
        super(TreeModel, self).__init__(parent)
        self.text = text
        self.rootItems = []
        self.indexMap = []

    def columnCount(self, parentIndex):
        return 1

    def data(self, index, role):
        if not index.isValid():
            return None

        if role == QtCore.Qt.DisplayRole:
            return index.internalPointer().displayRole()
        else:
            return None

    def headerData(self, section, orientation, role):
        if orientation == QtCore.Qt.Horizontal and role == QtCore.Qt.DisplayRole:
            return 'Structure'

        return None

    def index(self, row, column, parentIndex):
        if not self.hasIndex(row, column, parentIndex):
            return QtCore.QModelIndex()

        if not parentIndex.isValid():
            childItem = self.rootItems[row]
        else:
            childItem = parentIndex.internalPointer().childItems[row]

        childIndex = self.createIndex(row, column, childItem)
        self.indexMap.append(childItem.element)
        self.indexMap.append(childIndex)
        return childIndex

    def parent(self, childIndex):
        if not childIndex.isValid():
            return QtCore.QModelIndex()

        parentItem = childIndex.internalPointer().parentItem

        if parentItem == None:
            return QtCore.QModelIndex()
        else:
            grandparentItem = parentItem.parentItem
            if grandparentItem:
                parentRow = grandparentItem.childItems.index(parentItem)
            else:
                parentRow = self.rootItems.index(parentItem)

            parentIndex = self.createIndex(parentRow, 0, parentItem)
            self.indexMap.append(parentItem.element)
            self.indexMap.append(parentIndex)
            return parentIndex

    def rowCount(self, parentIndex):
        if parentIndex.isValid():
            return len(parentIndex.internalPointer().childItems)
        else:
            return len(self.rootItems)

    def update(self):
        #items
        self.rootItems = []
        self.indexMap = []
        body = self.text.page().mainFrame().documentElement().findFirst('BODY')
        child = body.firstChild()
        if not child.isNull():
            while True:
                if child.attribute('class'):
                    self.rootItems.append(TreeNode(child))
                child = child.nextSibling()
                if child.isNull():
                    break
        #reset
        self.beginResetModel()
        self.endResetModel()

class TreeNode():
    def __init__(self, element, parentItem=None):
        self.element = element

        self.parentItem = parentItem
        self.childItems = []

        child = element.firstChild()
        if not child.isNull():
            while True:
                if child.attribute('class'):
                    self.childItems.append(TreeNode(child, self))
                child = child.nextSibling()
                if child.isNull():
                    break

    def displayRole(self):
        return self.element.attribute('class')

class Javascript(QtGui.QWidget):
    def __init__(self, text, parent=None):
        #self
        super(Javascript, self).__init__(parent)
        self.setLayout(QtGui.QGridLayout())
        #text
        self.text = text
        #javascript group box
        self.javascriptGroup = QtGui.QGroupBox('Javascript/Jquery', self)
        self.javascriptGroup.setLayout(QtGui.QGridLayout())
        self.layout().addWidget(self.javascriptGroup, 0, 0)
        #javascript text
        self.javascript = QtGui.QTextEdit(self.javascriptGroup)
        self.javascript.setWordWrapMode(QtGui.QTextOption.NoWrap)
        self.javascriptGroup.layout().addWidget(self.javascript, 0, 0)
        #javascript run button
        self.runJavascript= QtGui.QPushButton('Run Script', self.javascriptGroup)
        self.runJavascript.clicked.connect(self.slotRunJavascript)
        self.javascriptGroup.layout().addWidget(self.runJavascript, 1, 0)
        #result group box
        self.resultGroup = QtGui.QGroupBox('Result', self)
        self.resultGroup.setLayout(QtGui.QGridLayout())
        self.layout().addWidget(self.resultGroup, 0, 1)
        #result text
        self.result = QtGui.QTextBrowser(self.resultGroup)
        self.result.setWordWrapMode(QtGui.QTextOption.NoWrap)
        self.resultGroup.layout().addWidget(self.result, 0, 0)

    def slotRunJavascript(self):
        #report
        debugger.reportDetail(self, 'LOG', 'Running javascript...')
        #read script
        script = self.javascript.toPlainText()
        #run script
        result = plugboard.python(self.text.page().mainFrame().evaluateJavaScript(script), str)
        #write result
        self.result.clear()
        cursor = self.result.textCursor()
        cursor.movePosition(QtGui.QTextCursor.Start)
        cursor.insertText(result)
        #report
        debugger.reportDetail(self, 'LOG', 'Finished running javascript.')


class Format(QtGui.QWidget):
    def __init__(self, text, parent=None):
        #self
        super(Format, self).__init__(parent)
        self.setLayout(QtGui.QGridLayout())
        #text
        self.text = text
        #selector group
        self.selectorGroup = QtGui.QGroupBox('Selector', self)
        self.selectorGroup.setLayout(QtGui.QGridLayout())
        self.layout().addWidget(self.selectorGroup, 0, 0)
        #selectors list
        self.selectors = QtGui.QComboBox(self)
        self.text.page().connect(self.text.page(), QtCore.SIGNAL("contentsChanged()"), self.slotUpdateSelectors)
        self.text.connect(self.text, QtCore.SIGNAL("loadFinished(bool)"), self.slotUpdateSelectors)
        self.text.connect(self.text, QtCore.SIGNAL("signalHighlightChanged()"), self.slotHighlightChanged)
        self.text.page().connect(self.text.page(), QtCore.SIGNAL("contentsChanged()"), self.slotHighlightChanged) #in case replaced selector in structure wizard
        self.selectorGroup.layout().addWidget(self.selectors, 0, 0)

        #advanced css
        self.advancedCss = AdvancedCss(self.text, self)
        self.advancedCss.connect(self.advancedCss, QtCore.SIGNAL('signalFormatChanged(QString, QString)'), self.slotFormatChanged)
        self.selectors.connect(self.selectors, QtCore.SIGNAL('currentIndexChanged(QString)'), self.advancedCss.slotUpdateFormattingValues)
        self.layout().addWidget(self.advancedCss, 0, 1)

    def slotFormatChanged(self, name, value):
        #get variables
        value = plugboard.python(value, str) #for calibre
        name = plugboard.python(name, str) #for calibre
        name = Configuration.style_names[name]['javascript_name']
        selector = self.selectors.currentText()
        #update stylesheet rules
        script = '''
                 //variables
                 selector = "%s"
                 stylesheet = document.styleSheets[0]
                 rule = null; for(i=0;i<stylesheet.cssRules.length;i++){if(stylesheet.cssRules[i].selectorText==selector){rule=stylesheet.cssRules[i]}}
                 //update rule
                 rule.style.%s = "%s"
                 true
                 ''' %(selector, name, value)
        self.text.page().mainFrame().evaluateJavaScript(script)

    def slotHighlightChanged(self):
        highlight_selector = '.' + str(self.text.page().mainFrame().documentElement().findFirst('*[id="highlight"]').attribute('class')).lower()
        if not highlight_selector == '.':
            self.selectors.setCurrentIndex(self.selectors.findText(highlight_selector))

    def slotUpdateSelectors(self):
        #get list of selectors/remove unused ones
        script = '''
                 selectors = ''
                 stylesheet = document.styleSheets[0]
                 i=0
                 while(i<stylesheet.cssRules.length)
                 {
                    selector = stylesheet.cssRules[i].selectorText

                    if($(selector).size() == 0)
                    {
                        stylesheet.deleteRule(i)
                    }
                    else
                    {
                        selectors += '{' + selector
                        i++
                    }
                 }
                 selectors
                 '''
        self.selectors.clear()
        selectors = plugboard.python(self.text.page().mainFrame().evaluateJavaScript(script), str) #for calibre
        selectors = selectors.split('{')[1:]
        self.selectors.addItems(selectors)

#class CleanAndOpenWizard(QtGui.)

class OpenAndSave(QtGui.QWidget):
    def __init__(self, text, path_to_epub, parent=None):
        #self
        super(OpenAndSave, self).__init__(parent)
        self.setLayout(QtGui.QGridLayout())
        self.epub = None
        #epub
        self.path_to_epub = path_to_epub
        #text
        self.text = text
        #open group box
        self.openGroup = QtGui.QGroupBox('Open', self)
        self.openGroup.setLayout(QtGui.QGridLayout())
        self.layout().addWidget(self.openGroup, 0, 0)
        #open epub label
        self.openEpubLabel = QtGui.QLabel('Cleans and opens the epub.'
                                            '\nNote: As of now, you are much better off'
                                            '\nusing an epub already run through calibre\'s conversion process!'
                                            '\nPlease be patient while it loads.'
                                            '\nIt will not affect the original.'
                                            '\n\nNote: As of now, there are no options for the cleaning,'
                                            '\nand there is no report of errors/issues in the cleaning.'
                                            '\nThis generally shouldn\'t matter, but it is worth quickly scanning the result :)',
                                             self.openGroup)
        self.openGroup.layout().addWidget(self.openEpubLabel, 0, 0)
        #open epub button
        self.openEpub = QtGui.QPushButton('Open Epub', self.openGroup)
        self.openEpub.clicked.connect(self.slotOpenEpub)
        self.openGroup.layout().addWidget(self.openEpub, 1, 0)
        #save group box
        self.saveGroup = QtGui.QGroupBox('Save', self)
        self.saveGroup.setLayout(QtGui.QGridLayout())
        #self.saveGroup.layout().setSizeConstraint(QtGui.QLayout.SetMinimumSize)
        self.layout().addWidget(self.saveGroup, 0, 1)
        #save xhtml label
        self.saveXhtmlLabel = QtGui.QLabel('Opens the temp directory containing a saved copy in xhtml.'
                                            '\nThis action is mainly to simplify debugging.'
                                            '\nNote: You must copy it to a location of your choice before exiting.'
                                            '\nOtherwise the tempdir is deleted on exit.',
                                             self.saveGroup)
        self.saveGroup.layout().addWidget(self.saveXhtmlLabel, 0, 0)
        #save xhtml button
        self.saveXhtml = QtGui.QPushButton('Save as Xhtml', self.saveGroup)
        self.saveXhtml.setEnabled(False)
        self.saveXhtml.clicked.connect(self.slotSaveXhtml)
        self.saveGroup.layout().addWidget(self.saveXhtml, 1, 0)
        #save epub label
        self.saveEpubLabel = QtGui.QLabel('Saves epub to tempdir and opens the temp directory containing the epub.'
                                            '\nNote: You must copy it to a location of your choice before exiting.'
                                            '\nOtherwise the tempdir is deleted on exit.'
                                            , self.saveGroup)
        self.saveGroup.layout().addWidget(self.saveEpubLabel, 2, 0)
        #save epub button
        self.saveEpub = QtGui.QPushButton('Save as Epub', self.saveGroup)
        self.saveEpub.setEnabled(False)
        self.saveEpub.clicked.connect(self.slotSaveEpub)
        self.saveGroup.layout().addWidget(self.saveEpub, 3, 0)

    def slotOpenEpub(self):
        '''
        wizard = wizards.OpenWizard(self.text, self.path_to_epub, Configuration, self)
        self.layout().addWidget(wizard, 1, 0)
        '''


        #read
        debugger.reportDetail(self, 'LOG', 'Reading...')
        epub_reader = epub.EpubReader()
        self.epub = epub_reader.read(self.path_to_epub, Configuration.tempdir)
        #clean
        debugger.reportDetail(self, 'LOG', 'Cleaning...')
        cleaner = epub.EpubXhtmlXmlCleaner()
        cleaner.clean(self.epub, Configuration.tags, Configuration.style_names, Configuration.style_values)
        #open
        debugger.reportDetail(self, 'LOG', 'Opening...')
        url = self.epub.getXhtmlXml()[0]
        self.text.setUrl(QtCore.QUrl(url))

        #disable/enable
        self.openEpub.setEnabled(False)
        self.saveXhtml.setEnabled(True)
        self.saveEpub.setEnabled(True)

    def slotSaveXhtml(self):
        #remove highlight
        script = '''current = document.getElementById("highlight")
                     if(current)
                     {
                        current.removeAttribute("style")
                        current.removeAttribute("id")
                     }
                     $("*", document.body).removeAttr("new")
                     true
                     '''
        self.text.page().mainFrame().evaluateJavaScript(script)
        #path to html
        path_to_html = self.epub.getXhtmlXml()[0]
        with open(path_to_html, 'w+b') as file_to_save_to:
            #xhtml; complexity because we want xhtml, and webkit delivers xhtml!
            html = plugboard.python(self.text.page().mainFrame().toHtml(), str)
            #write
            file_to_save_to.write(etree.tostring(lxml.html.document_fromstring(html), encoding='utf-8', method='xml', xml_declaration=True, pretty_print=True))
        #open directory
        os.startfile(self.epub.basepath)

    def slotSaveEpub(self):
        #report
        debugger.reportDetail(self, 'LOG', 'Saving epub in same directory as epub, as cleaned*...')
        #remove highlight
        script = '''current = document.getElementById("highlight")
                     if(current)
                     {
                        current.removeAttribute("style")
                        current.removeAttribute("id")
                     }
                     $("*", document.body).removeAttr("new")
                     true
                     '''
        self.text.page().mainFrame().evaluateJavaScript(script)
        #path to html
        path_to_html = self.epub.getXhtmlXml()[0]
        with open(path_to_html, 'w+b') as file_to_save_to:
            #xhtml; complexity because we want xhtml, and webkit delivers xhtml!
            html = plugboard.python(self.text.page().mainFrame().toHtml(), str)
            #write
            file_to_save_to.write(etree.tostring(lxml.html.document_fromstring(html), encoding='utf-8', method='xml', xml_declaration=True, pretty_print=True))
        #write
        epubWriter = epub.EpubWriter()
        epubWriter.write(self.epub, os.path.join(self.epub.basepath, os.path.splitext(os.path.basename(self.path_to_epub))[0]) + '.epub')
        #open directory
        os.startfile(self.epub.basepath)


class Structure(QtGui.QWidget):
    def __init__(self, text, parent=None):
        #self
        super(Structure, self).__init__(parent)
        self.setLayout(QtGui.QGridLayout())

        #text
        self.text = text
        #find group
        self.findGroup = QtGui.QGroupBox('Find', self)
        self.findGroup.setLayout(QtGui.QGridLayout())
        self.layout().addWidget(self.findGroup, 0, 0)
        #findNextOccurence button
        self.findNextOccurence = QtGui.QPushButton('Next Occurence', self.findGroup)
        self.findGroup.layout().addWidget(self.findNextOccurence, 0, 0)
        self.findNextOccurence.clicked.connect(self.slotFindNextOccurence)
        #findNextPatternOccurence button
        self.findNextPatternOccurence = QtGui.QPushButton('Next Pattern Occurence', self.findGroup)
        self.findGroup.layout().addWidget(self.findNextPatternOccurence, 1, 0)
        self.findNextPatternOccurence.clicked.connect(self.slotFindNextPatternOccurence)
        self.findNextPatternOccurence.setEnabled(False)
        #replace group
        self.replaceGroup = QtGui.QGroupBox('Replace', self)
        self.replaceGroup.setLayout(QtGui.QGridLayout())
        self.layout().addWidget(self.replaceGroup, 0, 1)
        #replacements list
        self.replacements = QtGui.QComboBox(self.replaceGroup)
        self.replacements.setEditable(True)
        self.replacements.addItems([''])
        self.replaceGroup.layout().addWidget(self.replacements, 0, 0)
        #replaceAll button
        self.replaceAll = QtGui.QPushButton('Replace All', self.replaceGroup)
        self.replaceGroup.layout().addWidget(self.replaceAll, 1, 0)
        self.replaceAll.clicked.connect(self.slotReplaceAll)
        #replaceForward button
        self.replaceForward = QtGui.QPushButton('Replace Forward', self.replaceGroup)
        self.replaceGroup.layout().addWidget(self.replaceForward, 2, 0)
        self.replaceForward.clicked.connect(self.slotReplaceForward)
        self.replaceForward.setEnabled(False)
        #replaceSelected button
        self.replaceSelected = QtGui.QPushButton('Replace Selected', self.replaceGroup)
        self.replaceGroup.layout().addWidget(self.replaceSelected, 3, 0)
        self.replaceSelected.clicked.connect(self.slotReplaceSelected)
        self.replaceSelected.setEnabled(False)


    def slotFindNextOccurence(self):
        #find
        script = '''
                 current =  document.getElementById("highlight")
                 if(current && current.className){
                    $all = $("." + current.className)
                    index =  $all.index(current)
                    $forward = $all.slice(index + 1)
                    next = null
                    if($forward.size()>0){next = $forward[0]}
                    else if(index>0){next = $all[0]}
                    moveHighlightTo(next)
                    true
                 }'''
        self.text.page().mainFrame().evaluateJavaScript(script)

    def slotFindNextPatternOccurence(self):
        #find
        script = '''
                 current =  document.getElementById("highlight")
                 if(current){
                    $all = $("[class^=Pattern]").add(current)
                    index =  $all.index(current)
                    $forward = $all.slice(index + 1)
                    next = null
                    if($forward.size()>0){next = $forward[0]}
                    else if(index>0){next = $all[0]}
                    moveHighlightTo(next)
                    true
                 }'''
        self.text.page().mainFrame().evaluateJavaScript(script)

    def slotReplaceAll(self):
        #get replacement
        replacement = str(self.replacements.currentText()) #for calibre
        #break down: new, ____, of, ... (some are divs, some ps, some spans........???????)
        levels = [level for level in re.split(' of ', replacement) if level]
        levels.reverse() #work top down (? important?)
        for level in levels:
            new = False
            if level.startswith('new'):
                new = True
                level = level[3:]
            #update css
            script = '''
                     //variables
                     replacement = "%s"
                     stylesheet = document.styleSheets[0]
                     //check if replacement already exists in stylesheet
                     for(i=0;i<stylesheet.cssRules.length;i++){if(stylesheet.cssRules[i].selectorText == "." + replacement){return}}
                     //add replacement to stylesheet
                     stylesheet.insertRule("."+replacement+"{}",document.styleSheets[0].cssRules.length)
                     true
                     ''' %level.lower()
            self.text.page().mainFrame().evaluateJavaScript(script)
            #insert
            script = '''
                    original = document.getElementById("highlight").className
                    if(original){
                        newdiv = %s
                        replacement = "%s"
                        if(newdiv){
                            wrapper = "<div class=\\""+replacement+"\\" new=\\"true\\" />"
                            $("." + original).each(function(){$(this).add($(this).nextUntil("[class]")).wrapAll(wrapper)
                                //combine with next if same type and not new
                                if($(this).parent().next().attr('new')=="false" && $(this).parent().next().attr('class') == replacement){
                                    $(this).parent().next().contents().appendTo($(this).parent())
                                    $(this).parent().next().remove()
                                }
                            })
                        }
                        else{
                            wrapper = "<div class=\\""+replacement+"\\" new=\\"false\\" />"
                            $("." + original).each(function(){
                                if($(this).prev().size() > 0 && $(this).prev()[0].className == replacement){
                                    $(this).add($(this).nextUntil("[class]")).appendTo($(this).prev())
                                }
                                else{
                                    $(this).add($(this).nextUntil("[class]")).wrapAll(wrapper)
                                }
                                //combine with next if same type and not new
                                if($(this).parent().next().attr('new')=="false" && $(this).parent().next().attr('class') == replacement){
                                    $(this).parent().next().contents().appendTo($(this).parent())
                                    $(this).parent().next().remove()
                                }
                            })
                        }
                    }
                    true
                    ''' %(str(new).lower(), level.lower())
            self.text.page().mainFrame().evaluateJavaScript(script)
        #remove original
        script = '''
                 original =  document.getElementById("highlight").className
                 if(original){
                    $all = $("." + original)
                    $all.each(function(){this.removeAttribute('class')})
                 }
                 true
                 '''
        self.text.page().mainFrame().evaluateJavaScript(script)

        #signal
        self.text.page().contentsChanged.emit()

    def slotReplaceForward(self):
        #get replacement
        replacement = self.replacements.currentText()
        #update css
        script = '''
                 //variables
                 replacement = "%s"
                 stylesheet = document.styleSheets[0]
                 //check if replacement already exists in stylesheet
                 for(i=0;i<stylesheet.cssRules.length;i++){if(stylesheet.cssRules[i].selectorText == "." + replacement){return}}
                 //add replacement to stylesheet
                 stylesheet.insertRule("."+replacement+"{}",document.styleSheets[0].cssRules.length)
                 true
                 ''' %replacement
        self.text.page().mainFrame().evaluateJavaScript(script)
        #replace
        script = '''
                 original =  document.getElementById("highlight").className
                 if(original){
                 replacement = "%s"
                 $all = $("." + original)
                 $forward = $all.slice($all.index(document.getElementById("highlight")))
                 $forward.each(function(){this.className=replacement})
                 true
                 }
                 ''' %replacement
        self.text.page().mainFrame().evaluateJavaScript(script)
        #signal
        self.text.page().contentsChanged.emit()

    def slotReplaceSelected(self):
        #get replacement
        replacement = self.replacements.currentText()
        #update css
        script = '''
                 //variables
                 replacement = "%s"
                 stylesheet = document.styleSheets[0]
                 //check if replacement already exists in stylesheet
                 for(i=0;i<stylesheet.cssRules.length;i++){if(stylesheet.cssRules[i].selectorText == "." + replacement){return}}
                 //add replacement to stylesheet
                 stylesheet.insertRule("."+replacement+"{}",document.styleSheets[0].cssRules.length)
                 true
                 ''' %replacement
        self.text.page().mainFrame().evaluateJavaScript(script)
        #replace
        script = '''
                 selected =  document.getElementById("highlight")
                 if(selected){
                 replacement = "%s"
                 selected.className = replacement
                 true
                 }
                 ''' %replacement
        self.text.page().mainFrame().evaluateJavaScript(script)
        #signal
        self.text.page().contentsChanged.emit()

class StructureDeprecated(QtGui.QWidget):
    def __init__(self, text, parent=None):
        #self
        super(Structure, self).__init__(parent)
        self.setLayout(QtGui.QGridLayout())

        #text
        self.text = text
        #find group
        self.findGroup = QtGui.QGroupBox('Find', self)
        self.findGroup.setLayout(QtGui.QGridLayout())
        self.layout().addWidget(self.findGroup, 0, 0)
        #findNextOccurence button
        self.findNextOccurence = QtGui.QPushButton('Next Occurence', self.findGroup)
        self.findGroup.layout().addWidget(self.findNextOccurence, 0, 0)
        self.findNextOccurence.clicked.connect(self.slotFindNextOccurence)
        #findNextPatternOccurence button
        self.findNextPatternOccurence = QtGui.QPushButton('Next Pattern Occurence', self.findGroup)
        self.findGroup.layout().addWidget(self.findNextPatternOccurence, 1, 0)
        self.findNextPatternOccurence.clicked.connect(self.slotFindNextPatternOccurence)
        #replace group
        self.replaceGroup = QtGui.QGroupBox('Replace', self)
        self.replaceGroup.setLayout(QtGui.QGridLayout())
        self.layout().addWidget(self.replaceGroup, 0, 1)
        #replacements list
        self.replacements = QtGui.QComboBox(self.replaceGroup)
        self.replacements.setEditable(True)
        self.replacements.addItems(['', 'TitleOfChapter', 'NameOfChapter', 'StartOfChapter', 'StartOfScene', 'Regular', 'StartOfVerse', 'MiddleOfVerse'])
        self.replaceGroup.layout().addWidget(self.replacements, 0, 0)
        #replaceAll button
        self.replaceAll = QtGui.QPushButton('Replace All', self.replaceGroup)
        self.replaceGroup.layout().addWidget(self.replaceAll, 1, 0)
        self.replaceAll.clicked.connect(self.slotReplaceAll)
        #replaceForward button
        self.replaceForward = QtGui.QPushButton('Replace Forward', self.replaceGroup)
        self.replaceGroup.layout().addWidget(self.replaceForward, 2, 0)
        self.replaceForward.clicked.connect(self.slotReplaceForward)
        #replaceSelected button
        self.replaceSelected = QtGui.QPushButton('Replace Selected', self.replaceGroup)
        self.replaceGroup.layout().addWidget(self.replaceSelected, 3, 0)
        self.replaceSelected.clicked.connect(self.slotReplaceSelected)

        #testreplace group
        self.testreplaceGroup = QtGui.QGroupBox('Test Replace', self)
        self.testreplaceGroup.setLayout(QtGui.QGridLayout())
        self.layout().addWidget(self.testreplaceGroup, 0, 2)
        #testreplacements list
        self.testreplacements = QtGui.QComboBox(self.testreplaceGroup)
        self.testreplacements.setEditable(True)
        self.testreplacements.addItems(['', 'TitleOfChapter', 'NameOfChapter', 'StartOfChapter', 'StartOfScene', 'Regular', 'StartOfVerse', 'MiddleOfVerse'])
        self.testreplaceGroup.layout().addWidget(self.testreplacements, 0, 0)
        #testreplaceAll button
        self.testreplaceAll = QtGui.QPushButton('Test Replace All', self.testreplaceGroup)
        self.testreplaceGroup.layout().addWidget(self.testreplaceAll, 1, 0)
        self.testreplaceAll.clicked.connect(self.testslotReplaceAll)


    def testslotReplaceAll(self):
        #get replacement
        debugger.reportDetail(self, 'LOG', 'testslotReplaceAll')
        hierarchy = {'chapter':{'title':None,'name':None,'scene':None}}
        self.testslotRecursiveHierarchy(hierarchy)
        #signal
        self.text.page().contentsChanged.emit()

    def testslotRecursiveHierarchy(self, hierarchy, parent=None):
        debugger.reportDetail(self, 'LOG', 'testslotRecursiveHierarchy')
        for item in hierarchy:
            self.testslotRestructure(item, parent, [key for key in hierarchy if not key == item])
        for item in hierarchy:
            if hierarchy[item]:
               self.testslotRecursiveHierarchy(hierarchy[item], item)

    def testslotRestructure(self, target, parent, siblings):
        debugger.reportDetail(self, 'LOG', 'testslotRestructure')
        next_ = '$(".%s")' %target
        next_ += '.add($(".%s"))' % parent if parent else ''
        next_ += ''.join(['.add($(".%s"))' % sibling for sibling in siblings])

        script ='''
                target =  "%s"
                wrapper = "<div class=\\"" + target + "\\" />"
                $next = %s
                $("*", document.body).first().each(function(){$(this).add($(this).nextUntil($next)).wrapAll(wrapper)})
                $("." + target).each(function(){$(this).add($(this).nextUntil($next)).wrapAll(wrapper)})
                true
                ''' %(target, next_)
        self.text.page().mainFrame().evaluateJavaScript(script)


    def slotFindNextOccurence(self):
        #find
        script = '''
                 current =  document.getElementById("highlight")
                 if(current && current.className){
                    $all = $("." + current.className)
                    index =  $all.index(current)
                    $forward = $all.slice(index + 1)
                    next = null
                    if($forward.size()>0){next = $forward[0]}
                    else if(index>0){next = $all[0]}
                    moveHighlightTo(next)
                    true
                 }'''
        self.text.page().mainFrame().evaluateJavaScript(script)

    def slotFindNextPatternOccurence(self):
        #find
        script = '''
                 current =  document.getElementById("highlight")
                 if(current){
                    $all = $("[class^=Pattern]").add(current)
                    index =  $all.index(current)
                    $forward = $all.slice(index + 1)
                    next = null
                    if($forward.size()>0){next = $forward[0]}
                    else if(index>0){next = $all[0]}
                    moveHighlightTo(next)
                    true
                 }'''
        self.text.page().mainFrame().evaluateJavaScript(script)

    def slotReplaceAll(self):
        #get replacement
        replacement = str(self.replacements.currentText()) #for calibre
        #update css
        script = '''
                 //variables
                 replacement = "%s"
                 stylesheet = document.styleSheets[0]
                 //check if replacement already exists in stylesheet
                 for(i=0;i<stylesheet.cssRules.length;i++){if(stylesheet.cssRules[i].selectorText == "." + replacement){return}}
                 //add replacement to stylesheet
                 stylesheet.insertRule("."+replacement+"{}",document.styleSheets[0].cssRules.length)
                 true
                 ''' %replacement.lower()
        self.text.page().mainFrame().evaluateJavaScript(script)
        #replace
        script = '''
                 original =  document.getElementById("highlight").className
                 if(original){
                    replacement = "%s"
                    $all = $("." + original)
                    if(replacement){
                        $all.each(function(){this.className=replacement})
                    }
                    else{
                        $all.each(function(){this.removeAttribute('class')})
                    }
                 }
                 true
                 ''' %replacement
        self.text.page().mainFrame().evaluateJavaScript(script)
        #signal
        self.text.page().contentsChanged.emit()

    def slotReplaceForward(self):
        #get replacement
        replacement = self.replacements.currentText()
        #update css
        script = '''
                 //variables
                 replacement = "%s"
                 stylesheet = document.styleSheets[0]
                 //check if replacement already exists in stylesheet
                 for(i=0;i<stylesheet.cssRules.length;i++){if(stylesheet.cssRules[i].selectorText == "." + replacement){return}}
                 //add replacement to stylesheet
                 stylesheet.insertRule("."+replacement+"{}",document.styleSheets[0].cssRules.length)
                 true
                 ''' %replacement
        self.text.page().mainFrame().evaluateJavaScript(script)
        #replace
        script = '''
                 original =  document.getElementById("highlight").className
                 if(original){
                 replacement = "%s"
                 $all = $("." + original)
                 $forward = $all.slice($all.index(document.getElementById("highlight")))
                 $forward.each(function(){this.className=replacement})
                 true
                 }
                 ''' %replacement
        self.text.page().mainFrame().evaluateJavaScript(script)
        #signal
        self.text.page().contentsChanged.emit()

    def slotReplaceSelected(self):
        #get replacement
        replacement = self.replacements.currentText()
        #update css
        script = '''
                 //variables
                 replacement = "%s"
                 stylesheet = document.styleSheets[0]
                 //check if replacement already exists in stylesheet
                 for(i=0;i<stylesheet.cssRules.length;i++){if(stylesheet.cssRules[i].selectorText == "." + replacement){return}}
                 //add replacement to stylesheet
                 stylesheet.insertRule("."+replacement+"{}",document.styleSheets[0].cssRules.length)
                 true
                 ''' %replacement
        self.text.page().mainFrame().evaluateJavaScript(script)
        #replace
        script = '''
                 selected =  document.getElementById("highlight")
                 if(selected){
                 replacement = "%s"
                 selected.className = replacement
                 true
                 }
                 ''' %replacement
        self.text.page().mainFrame().evaluateJavaScript(script)
        #signal
        self.text.page().contentsChanged.emit()


class AdvancedCss(QtGui.QWidget):

    def __init__(self, text, parent=None):
        #self
        super(AdvancedCss, self).__init__(parent)
        self.setLayout(QtGui.QGridLayout())
        self.layout().setSizeConstraint(QtGui.QLayout.SetFixedSize)
        self.input_map = {}
        self.text = text

        #frames
        frame_count = 0
        for group in Configuration.style_group:
            #create frame
            frame = QtGui.QGroupBox(group, self)
            frame.setLayout(QtGui.QGridLayout())
            #frame.layout().setSizeConstraint(QtGui.QLayout.SetFixedSize)
            self.layout().addWidget(frame, 0, frame_count)
            #update count
            frame_count += 1
            #inputs
            input_count = 0
            names = [name for name in Configuration.style_names if Configuration.style_names[name]['group'] == group]
            for name in names:
                #create objects
                label = QtGui.QLabel(Configuration.style_names[name]['friendly_name'], frame)
                combo = QtGui.QComboBox(frame)
                combo.setObjectName(name)
                combo.addItems(Configuration.style_names[name]['values'])
                spin = QtGui.QDoubleSpinBox(frame)
                spin.setObjectName(name)
                spin.hide()
                #map objects
                self.input_map[name] = {}
                self.input_map[name]['frame'] = frame
                self.input_map[name]['label'] = label
                self.input_map[name]['combo'] = combo
                self.input_map[name]['spin'] = spin
                self.input_map[name]['row'] = input_count
                #layout objects
                frame.layout().addWidget(label, input_count, 0)
                frame.layout().addWidget(combo, input_count, 1, 1, 2)
                #connect signals
                spin.connect(spin, QtCore.SIGNAL('valueChanged(QString)'), self.signalFormatChanged)
                combo.connect(combo, QtCore.SIGNAL('currentIndexChanged(QString)'), self.signalFormatChanged)
                combo.connect(combo, QtCore.SIGNAL('currentIndexChanged(QString)'), self.slotToggleSpinBox)
                combo.connect(combo, QtCore.SIGNAL('currentIndexChanged(QString)'), self.slotToggleSunkenLabel)
                #update count
                input_count += 1

    @Qt.pyqtSignature('QString, QString')
    def signalFormatChanged(self, value):
        sender = self.sender()
        name = str(sender.objectName())
        value = str(value) #for calibre
        if type(sender) == QtGui.QDoubleSpinBox:
            value = str(self.input_map[name]['combo'].currentText())
        if value.startswith('#'):
            value = str(self.input_map[name]['spin'].value()) + value[1:]
        self.emit(QtCore.SIGNAL("signalFormatChanged(QString, QString)"), name, value)


    def slotToggleSpinBox(self, value, name=None):
        if not name:
            name = plugboard.python(self.sender().objectName(), str) #for calibre, convert
        value = plugboard.python(value, str) #for calibre
        if value.startswith('#'):
            if not self.input_map[name]['spin'].isVisible():
                #gather data
                frame = self.input_map[name]['frame']
                combo = self.input_map[name]['combo']
                spin = self.input_map[name]['spin']
                row = self.input_map[name]['row']
                #shrink combo
                frame.layout().removeWidget(combo)
                frame.layout().addWidget(combo, row, 2, 1, 1)
                #show spin
                frame.layout().addWidget(spin, row, 1)
                spin.show()
        else:
            if self.input_map[name]['spin'].isVisible():
                #gather data
                frame = self.input_map[name]['frame']
                combo = self.input_map[name]['combo']
                spin = self.input_map[name]['spin']
                row = self.input_map[name]['row']
                #hide spin
                frame.layout().removeWidget(spin)
                spin.hide()
                #expand combo
                frame.layout().removeWidget(combo)
                frame.layout().addWidget(combo, row, 1, 1, 2)

    def slotToggleSunkenLabel(self, value, name=None):
        if not name:
            name = plugboard.python(self.sender().objectName(), str) #for calibre, convert
        value = plugboard.python(value, str) #for calibre
        if value == '':
            #unsink label
            label = self.input_map[name]['label']
            label.setFrameShape(QtGui.QFrame.NoFrame)
            label.setStyleSheet('QLabel{ background-color:None;}')
        else:
            #sink label
            label = self.input_map[name]['label']
            label.setFrameShape(QtGui.QFrame.Panel)
            label.setFrameShadow(QtGui.QFrame.Sunken)
            label.setStyleSheet('QLabel{background-color:#9c9;}')

    def slotUpdateFormattingValues(self, selector):
        selector = plugboard.python(selector, str) #for calibre
        if selector:
            #get stylesheet rule
            script = '''
                     selector = "%s"
                     stylesheet = document.styleSheets[0]
                     rule = null
                     for(i=0;i<stylesheet.cssRules.length;i++)
                     {
                        if(stylesheet.cssRules[i].selectorText==selector)
                        {
                            rule = stylesheet.cssRules[i].cssText
                            break
                        }
                     }
                     rule
                     ''' %selector
            rule = plugboard.python(self.text.page().mainFrame().evaluateJavaScript(script), str) #for calibre, convert
            #rule -> style dict
            tempParser = epub.EpubXhtmlXmlCleaner()
            tempParser.style_names = Configuration.style_names
            tempParser.style_values = Configuration.style_values
            style = tempParser.style(re.split('[{}]', rule)[1])
            #disconnect input fields while setting (already the rule has the values being set...)
            for name in self.input_map:
                combo =  self.input_map[name]['combo']
                spin = self.input_map[name]['spin']
                combo.disconnect(combo, QtCore.SIGNAL('currentIndexChanged(QString)'), self.signalFormatChanged)
                spin.disconnect(spin, QtCore.SIGNAL('valueChanged(QString)'), self.signalFormatChanged)
            #update input fields
            for name in Configuration.style_names:
                #if a property of the current rule
                if name in style:
                    value = style[name]
                    combo = self.input_map[name]['combo']
                    #if # based
                    if value[0].isdigit() or value[0] == '-':
                        fragments = [fragment for fragment in re.split('([-]{0,1}[0-9]*[.]{0,1}[0-9]+)', value) if fragment]
                        #if only #
                        if len(fragments) == 1:
                            value = '#'
                            number = float(fragments[0])
                        #if # and measurement (no error checking though...)
                        else:
                            value = '#' + fragments[1]
                            number = float(fragments[0])
                        #set spin
                        spin = self.input_map[name]['spin']
                        spin.setValue(number)
                        #set combo
                        combo.setCurrentIndex(combo.findText(value))
                    #if not # based
                    else:
                        #set combo
                        combo.setCurrentIndex(combo.findText(value.title())) #cant share with # based, since there dont want titlecase (ex. Px vs px)
                #if not a property of the current rule
                else:
                    #clear combo
                    combo = self.input_map[name]['combo']
                    combo.setCurrentIndex(combo.findText(''))
                    #clear spin
                    spin = self.input_map[name]['spin']
                    spin.setValue(0.0)
            #reconnect input fields/toggle spin box
            for name in self.input_map:
                combo =  self.input_map[name]['combo']
                spin = self.input_map[name]['spin']
                combo.connect(combo, QtCore.SIGNAL('currentIndexChanged(QString)'), self.signalFormatChanged)
                spin.connect(spin, QtCore.SIGNAL('valueChanged(QString)'), self.signalFormatChanged)



def main():
    app = QtGui.QApplication(sys.argv)
    root = Main(path_to_epub = 'D:\\ECleaner\\Test Files\\TestA.epub')
    root.show()
    sys.exit(app.exec_())

if __name__ == '__main__':
    main()
